/***************************************************************************
  $RCSfile: libchipcard.c,v $
                             -------------------
    cvs         : $Id: libchipcard.c,v 1.41 2003/06/13 17:14:21 aquamaniac Exp $
    begin       : Sun Mar 10 2002
    copyright   : (C) 2002 by Martin Preuss
    email       : martin@libchipcard.de


 ***************************************************************************
 *                                                                         *
 *   This library is free software; you can redistribute it and/or         *
 *   modify it under the terms of the GNU Lesser General Public            *
 *   License as published by the Free Software Foundation; either          *
 *   version 2.1 of the License, or (at your option) any later version.    *
 *                                                                         *
 *   This library is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU     *
 *   Lesser General Public License for more details.                       *
 *                                                                         *
 *   You should have received a copy of the GNU Lesser General Public      *
 *   License along with this library; if not, write to the Free Software   *
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston,                 *
 *   MA  02111-1307  USA                                                   *
 *                                                                         *
 ***************************************************************************/

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif

#ifdef __declspec
# if BUILDING_CHIPCARD_DLL
#  define CHIPCARD_API __declspec (dllexport)
# else /* Not BUILDING_CHIPCARD_DLL */
#  define CHIPCARD_API __declspec (dllimport)
# endif /* Not BUILDING_CHIPCARD_DLL */
#else
# define CHIPCARD_API
#endif


#include "libchipcard.h"
#include <engine/chameleon/logger.h>
#include <engine/chameleon/debug.h>
#include <engine/chameleon/chameleon.h>
#include <engine/chameleon/conf.h>
#include <engine/chameleon/directory.h>
#include <engine/chameleon/error.h>
#include <engine/chameleon/inetsocket.h>
#include <engine/command/command.h>
#include <engine/service/ctclient.h>
#include <readerclient.h>
#include <ctversion.h>

#include <stdlib.h>
#include <assert.h>
#include <string.h>
#include <stdarg.h>


#define CHIPCARD__MAX_SUBREQUESTS 32
#define CHIPCARD__MAX_FINDREADERS 32
#define CHIPCARD__SUPERREQUEST_OFFSET 0x1000000

#define CHIPCARD__DEFAULT_MAXMSG 10
#define CHIPCARD__DEFAULT_TIMEOUT 750


typedef struct CHIPCARDSUPERREQUESTSTRUCT CHIPCARDSUPERREQUEST;


static CTCLIENTDATA *LibChipCard_ClientData=0;
static int LibChipCard_NextSuperRequestId=CHIPCARD__SUPERREQUEST_OFFSET;
static CHIPCARDSUPERREQUEST *LibChipCard_SuperRequests=0;
static CONFIGGROUP *LibChipCard_Config=0;
static CONFIGGROUP *LibChipCard_Commands=0;
static int LibChipCard_Timeout=CHIPCARD__DEFAULT_TIMEOUT;
static int LibChipCard_MaxMsgPerLoop=CHIPCARD__DEFAULT_MAXMSG;


struct CHIPCARDSUPERREQUESTSTRUCT {
  CHIPCARDSUPERREQUEST *next;
  int id;
  int waitForAll;
  int startNext;
  int subRequestCount;
  int subRequests[CHIPCARD__MAX_SUBREQUESTS];
  int serviceIds[CHIPCARD__MAX_SUBREQUESTS];
  void *requestData;
};


CHIPCARD_READERDESCR *ChipCard__ReaderDescr_new() {
  CHIPCARD_READERDESCR *rd;

  rd=(CHIPCARD_READERDESCR *)malloc(sizeof(CHIPCARD_READERDESCR));
  assert(rd);
  memset(rd,0,sizeof(CHIPCARD_READERDESCR));
  return rd;
}


void ChipCard_ReaderDescr_free(CHIPCARD_READERDESCR *rd) {
  if (rd)
    free(rd);
}


int ChipCard__xlresult(int result) {
  int rv;

  switch(result) {
  case CTSERVICE_SUCCESS: rv=CHIPCARD_SUCCESS; break;
  case CTSERVICE_ERROR_DRIVER: rv=CHIPCARD_ERROR_DRIVER; break;
  case CTSERVICE_ERROR_INVALID: rv=CHIPCARD_ERROR_INVALID; break;
  case CTSERVICE_ERROR_BUFFER: rv=CHIPCARD_ERROR_BUFFER; break;
  case CTSERVICE_ERROR_CARD_REMOVED:
    rv=CHIPCARD_ERROR_CARD_REMOVED;
    break;
  case CTSERVICE_ERROR_NO_REQUEST: rv=CHIPCARD_ERROR_NO_REQUEST; break;
  case CTSERVICE_ERROR_NO_MESSAGE: rv=CHIPCARD_ERROR_NO_MESSAGE; break;
  case CTSERVICE_ERROR_BAD_CHANNEL_STATUS:
    rv=CHIPCARD_ERROR_BAD_CHANNEL_STATUS;
    break;
  case CTSERVICE_ERROR_NO_COMMANDS: rv=CHIPCARD_ERROR_NO_COMMANDS; break;
  case CTSERVICE_ERROR_NO_CONFIG: rv=CHIPCARD_ERROR_NO_CONFIG; break;
  case CTSERVICE_ERROR_UNREACHABLE: rv=CHIPCARD_ERROR_UNREACHABLE; break;
  case CTSERVICE_ERROR_NO_TRANSPORT_LAYER: rv=CHIPCARD_ERROR_NO_TRANSPORT; break;
  default: rv=CHIPCARD_ERROR_INTERNAL;break;
  } /* switch */

  return rv;
}


int ChipCard__xlerr(ERRORCODE err) {
  int rv;

  if (!Error_IsOk(err)) {
    if (Error_GetType(err)!=Error_FindType(CTSERVICE_ERROR_TYPE)) {
      if ((Error_GetType(err)==Error_FindType(SOCKET_ERROR_TYPE)) &&
	  Error_GetCode(err)==SOCKET_ERROR_INTERRUPTED)
	rv=CHIPCARD_ERROR_INTERRUPTED;
      else {
	DBG_ERROR("Internal error:");
	DBG_ERROR_ERR(err);
	rv=CHIPCARD_ERROR_INTERNAL;
      }
    }
    else
      rv=ChipCard__xlresult(Error_GetCode(err));
  }
  else
    rv=CHIPCARD_SUCCESS;
  return rv;
}


CHIPCARDSUPERREQUEST *ChipCard__SuperRequest_new() {
  CHIPCARDSUPERREQUEST *sr;

  sr=(CHIPCARDSUPERREQUEST *)malloc(sizeof(CHIPCARDSUPERREQUEST));
  assert(sr);
  memset(sr, 0, sizeof(CHIPCARDSUPERREQUEST));
  sr->id=++LibChipCard_NextSuperRequestId;
  return sr;
}


void ChipCard__SuperRequest_free(CHIPCARDSUPERREQUEST *sr) {
  if (sr) {
    if (sr->requestData)
      free(sr->requestData);
    free(sr);
  }
}


void ChipCard__AddSuperRequest(CHIPCARDSUPERREQUEST *sr) {
  CHIPCARDSUPERREQUEST *curr;

  DBG_ENTER;
  assert(sr);

  curr=LibChipCard_SuperRequests;
  if (!curr) {
    LibChipCard_SuperRequests=sr;
  }
  else {
    /* find last */
    while(curr->next) {
      curr=curr->next;
    } /* while */
    curr->next=sr;
  }
  DBG_LEAVE;
}


void ChipCard__RemoveSuperRequest(CHIPCARDSUPERREQUEST *sr) {
  CHIPCARDSUPERREQUEST *curr;

  DBG_ENTER;
  assert(sr);

  curr=LibChipCard_SuperRequests;
  if (curr) {
    if (curr==sr) {
      LibChipCard_SuperRequests=curr->next;
    }
    else {
      /* find predecessor */
      while(curr->next!=sr) {
	curr=curr->next;
      } /* while */
      if (curr)
	curr->next=sr->next;
    }
  }
  DBG_LEAVE;
}


CHIPCARDSUPERREQUEST *ChipCard__FindSuperRequest(int id) {
  CHIPCARDSUPERREQUEST *curr;

  DBG_ENTER;

  curr=LibChipCard_SuperRequests;
  while(curr) {
    if (curr->id==id)
      return curr;
    curr=curr->next;
  } /* while */
  return 0;
  DBG_LEAVE;
}



ERRORCODE ChipCard__ReadCommands(const char *configdir) {
  DIRECTORYDATA *d;
  char buffer[512];
  int bufferlen;
  int goon=1;

  LibChipCard_Commands=Config_new();
  d=Directory_new();
  if (Directory_Open(d,configdir)) {
    DBG_ERROR("Could not open directory \"%s\"", configdir);
    Directory_free(d);
    return Error_New(0,
		     ERROR_SEVERITY_ERR,
		     Error_FindType(CTSERVICE_ERROR_TYPE),
		     CTSERVICE_ERROR_NO_CONFIG);
  }

  DBG_INFO("Opened dir \"%s\", reading.", configdir);

  do {
    int rv;

    bufferlen=sizeof(buffer);
    buffer[0]=0;
    if (Directory_Read(d,buffer,bufferlen)) {
      DBG_DEBUG("No file left");
      goon=0;
    }
    else {
      CONFIGGROUP *cmdgroup;
      int i;
      char fnbuffer[256];

      goon=1;
      DBG_INFO("Found file \"%s\"",buffer);
      i=strlen(buffer);
      if (i>4) {
	if (strcmp(buffer+(i-4),".cmd")==0) {
	  if (strlen(configdir+i+2)>sizeof(fnbuffer)) {
	    DBG_ERROR("Path too long");
	  }
	  else {
	    strcpy(fnbuffer, configdir);
            strcat(fnbuffer, "/");
	    strcat(fnbuffer,buffer);
	    DBG_DEBUG("Really reading file \"%s\"",fnbuffer);
	    cmdgroup=Config_new();
	    if (Config_ReadFile(cmdgroup,
				fnbuffer,
				CONFIGMODE_REMOVE_STARTING_BLANKS |
				CONFIGMODE_REMOVE_TRAILING_BLANKS |
				CONFIGMODE_REMOVE_QUOTES|
				CONFIGMODE_ALLOW_GROUPS)) {
	      DBG_ERROR("Could not read file \"%s\"",fnbuffer);
	    }
	    else {
	      DBG_INFO("Adding commands from file \"%s\"",fnbuffer);
	      rv=CTCommand_Add(LibChipCard_Commands,
			       cmdgroup);
	      if (rv!=CTCOMMAND_RESULT_OK) {
		DBG_ERROR("Error adding commands from file  \"%s\"",fnbuffer);
	      }
	    }
	    Config_free(cmdgroup);
	  }
	}
      }
    }
  }
  while(goon);

  if (Directory_Close(d)) {
    DBG_WARN("Error closing directory \"%s\"",configdir);
  }
  Directory_free(d);

#if DEBUGMODE>10
  Config_DumpGroup(stdout, LibChipCard_Commands, 2);
#endif
  return 0;
}





int ChipCard_Init(const char *configfile,
		  const char *section){
  ERRORCODE err;
  CONFIGGROUP *group;
  CONFIGGROUP *group2;
  const char *p;
  int sid;

  DBG_ENTER;
  if (LibChipCard_ClientData!=0) {
    DBG_DEBUG("Already initialized");
    return 0;
  }

  if (!configfile) {
    configfile=CHIPCARDC_CFGFILE;
  }

  /* init all modules */
  DBG_NOTICE("Starting libchipcard client %s with config file \"%s\"",
	     k_CHIPCARD_VERSION_COMPLETE_STRING,
	     configfile);
  DBG_DEBUG("Initializing Chameleon");
  err=Chameleon_Init();
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    DBG_LEAVE;
    return ChipCard__xlerr(err);
  }

  /* load configuration file */
  LibChipCard_Config=Config_new();
  DBG_DEBUG("Reading configuration file \"%s\"",
	    configfile);
  if (Config_ReadFile(LibChipCard_Config,
		      configfile,
		      CONFIGMODE_REMOVE_STARTING_BLANKS |
		      CONFIGMODE_REMOVE_TRAILING_BLANKS |
		      CONFIGMODE_REMOVE_QUOTES|
		      CONFIGMODE_ALLOW_GROUPS)) {
    DBG_ERROR("Could not load configuration file \"%s\"",configfile);
    return CHIPCARD_ERROR_NO_CONFIG;
  }

  /* initialize CTService module */
  DBG_DEBUG("Initializing service module");
  err=CTService_ModuleInit();
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    DBG_LEAVE;
    return ChipCard__xlerr(err);
  }

  /* create client */
  LibChipCard_ClientData=CTClient_new();
  DBG_DEBUG("Initializing Client");
  err=CTClient_Init(LibChipCard_ClientData);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    CTClient_free(LibChipCard_ClientData);
    Config_free(LibChipCard_Config);
    DBG_LEAVE;
    return ChipCard__xlerr(err);
  }

  /* set client id */
  CTClient_SetClientIdString(LibChipCard_ClientData,
			     "Libchipcard "
			     k_CHIPCARD_VERSION_COMPLETE_STRING"; "
			     OS_NAME);

  /* add servers from config file */
  if (section) {
    DBG_DEBUG("Using section \"%s\"",
              section);
    group=Config_GetGroup(LibChipCard_Config,
			  section,
			  CONFIGMODE_NAMEMUSTEXIST);
    if (!group)
      group=LibChipCard_Config;
  }
  else {
    DBG_DEBUG("Using root configuration");
    group=LibChipCard_Config;
  }

  if (!group) {
    DBG_ERROR("Empty configuration file \"%s\"",configfile);
    return CHIPCARD_ERROR_NO_CONFIG;
  }

  DBG_DEBUG("Reading service");
  /* read the services */
  group2=Config_GetGroup(group,
		       "service",
		       CONFIGMODE_NAMEMUSTEXIST);
  if (!group2) {
    DBG_WARN("No servers in section \"%s\" of configuration file \"%s\"",
	     section, configfile);
  }
  else {
    group2=group2->groups;
    while(group2) {
      const char *addr;
      int port;
      const char *tp;
      CONFIGGROUP *currgroup;

      currgroup=group2;
      group2=group2->next;

      tp=Config_GetValue(currgroup,
			 "type",
			 "net",
			 0);
      addr=Config_GetValue(currgroup,
			   "address",
			   0,
			   0);
      if (!addr) {
	DBG_WARN("Bad address in entry \"%s\" in section \"%s\"",
		 currgroup->name, section);
      }
      else {
	if (strcasecmp(tp,"net")==0) {
	  port=Config_GetIntValue(currgroup,
				  "port", CTSERVICE_DEFAULT_PORT,0);
	}
	else if (strcasecmp(tp,"local")==0) {
#ifdef OS_WIN32
	  DBG_ERROR("Local type not supported on this plattform");
	  continue;
#else
	  port=-1;
#endif
	}
	else {
	  DBG_WARN("Unknown type in service entry \"%s\" in section \"%s\"",
		   currgroup->name, section);
	  continue;
	}
	DBG_INFO("Adding server %s:%d", addr, port);
	err=CTClient_AddServer(LibChipCard_ClientData, addr, port, &sid);
	if (!Error_IsOk(err)) {
	  DBG_WARN_ERR(err);
	  DBG_WARN("Will skip this service");
	}
      }
    } /* while */
  }

  /* read command file */
  p=Config_GetValue(group,
		    "commands",
                    CHIPCARD_COMMANDS,
		    0);
  if (!p) {
    DBG_WARN("No command directory given !");
  }
  else {
    DBG_INFO("Reading command files from \"%s\"",p);
    err=ChipCard__ReadCommands(p);
    if (!Error_IsOk(err)) {
      DBG_WARN_ERR(err);
    }
  }


  /* read some global vars */
  LibChipCard_Timeout=Config_GetIntValue(group,
					 "timeout",
					 CHIPCARD__DEFAULT_TIMEOUT,0);
  LibChipCard_MaxMsgPerLoop=Config_GetIntValue(group,
					       "maxmsg",
					       CHIPCARD__DEFAULT_MAXMSG,0);


  DBG_LEAVE;
  return 0;
}


void ChipCard_Fini(){
  ERRORCODE err;

  DBG_ENTER;
  DBG_NOTICE("Chipcard_Fini");
  if (LibChipCard_ClientData==0)
    return;

  /* free all super requests */
  if (LibChipCard_SuperRequests) {
    CHIPCARDSUPERREQUEST *curr, *next;

    DBG_WARN("Some requests still enlisted");
    curr=LibChipCard_SuperRequests;
    while(curr) {
      next=curr->next;
      ChipCard__SuperRequest_free(curr);
      curr=next;
    } /* while */
    LibChipCard_SuperRequests=0;
  }

  /* free configuration */
  DBG_INFO("Releasing configuration.");
  Config_free(LibChipCard_Config);
  LibChipCard_Config=0;

  /* free commands */
  DBG_INFO("Releasing commands.");
  Config_free(LibChipCard_Commands);
  LibChipCard_Commands=0;

  /* deinitialize CTClient */
  err=CTClient_Fini(LibChipCard_ClientData);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
  }

  CTClient_free(LibChipCard_ClientData);
  LibChipCard_ClientData=0;

  /* deinit all modules */
  err=CTService_ModuleFini();
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
  }
  err=Chameleon_Fini();
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
  }

  DBG_LEAVE;
}


int ChipCard_AddServer(const char *addr,
		       int port){
  int sid;
  return ChipCard__xlerr(CTClient_AddServer(LibChipCard_ClientData,
					    addr, port, &sid));
}


int ChipCard_Work(){
  return ChipCard__xlerr(CTClient_Work(LibChipCard_ClientData,
				       LibChipCard_Timeout,
				       LibChipCard_MaxMsgPerLoop));
}


int ChipCard_CheckResponse(int requestid){

  if (requestid>=CHIPCARD__SUPERREQUEST_OFFSET) {
    int i;
    CHIPCARDSUPERREQUEST *sr;
    ERRORCODE err;
    int requests;
    int lastErr;
    int pos;

    sr=ChipCard__FindSuperRequest(requestid);
    if (!sr)
      return CHIPCARD_ERROR_NO_REQUEST;

    if (!(sr->waitForAll)) {
      /* check all subrequests until one reveals a response */
      requests=0;
      lastErr=0;
      if (sr->startNext>=sr->subRequestCount)
	sr->startNext=0;
      pos=sr->startNext++;
      DBG_DEBUG("Starting now with pos %d\n",pos);
      for (i=0; i<sr->subRequestCount; i++) {
	if (pos>=sr->subRequestCount)
	  pos=0;
	if (sr->subRequests[pos]!=0) {
	  requests++;
	  err=CTClient_CheckResponse(LibChipCard_ClientData,
				     sr->subRequests[pos]);
	  if (!Error_IsOk(err)) {
	    if (Error_GetType(err)!=Error_FindType(CTSERVICE_ERROR_TYPE) ||
		Error_GetCode(err)!=CTSERVICE_ERROR_NO_MESSAGE) {
	      DBG_ERROR_ERR(err);
	      lastErr=ChipCard__xlerr(err);
	      if (lastErr==CHIPCARD_ERROR_UNREACHABLE) {
		DBG_INFO("Withdrawing request at pos %d\n",pos);
		CTClient_WithdrawRequest(LibChipCard_ClientData,
					 sr->subRequests[pos]);
		sr->subRequests[pos]=0;
	      }
	    }
	  }
	  else {
	    /* we have a response ;-) */
	    return 0;
	  } /* have response */
	} /* have request */
        pos++;
      } /* for */
      if (!requests)
	return CHIPCARD_ERROR_NO_REQUEST;
      return CHIPCARD_ERROR_NO_MESSAGE;
    }
    else {
      /* wait for all */
      requests=0;
      lastErr=0;
      for (i=0; i<sr->subRequestCount; i++) {
	if (sr->subRequests[i]!=0) {
	  requests++;
	  err=CTClient_CheckResponse(LibChipCard_ClientData,
				     sr->subRequests[i]);
	  if (!Error_IsOk(err)) {
	    if (Error_GetType(err)!=Error_FindType(CTSERVICE_ERROR_TYPE) ||
		Error_GetCode(err)!=CTSERVICE_ERROR_NO_MESSAGE) {
	      DBG_ERROR_ERR(err);
	      lastErr=ChipCard__xlerr(err);
	      if (lastErr==CHIPCARD_ERROR_UNREACHABLE) {
		CTClient_WithdrawRequest(LibChipCard_ClientData,
					 sr->subRequests[i]);
		sr->subRequests[i]=0;
		requests--;
	      }
	    } /* if not NOMESSAGE */
	    else {
	      /* there is still a request that has no answer */
              DBG_DEBUG("Still an unanswered request");
	      return CHIPCARD_ERROR_NO_MESSAGE;
	    }
	  }
	} /* have request */
      } /* for */
      if (!requests)
	return CHIPCARD_ERROR_NO_REQUEST;
      /* ok, all requests are answered */
      DBG_DEBUG("All requests answered.\n");
      return CHIPCARD_SUCCESS;
    } /* if wait for all */
  }
  else
    return
      ChipCard__xlerr(CTClient_CheckResponse(LibChipCard_ClientData,
					     requestid));
}


int ChipCard_RequestPing(int *requestid,
			 int serviceid){
  return
    ChipCard__xlerr(ReaderClient_RequestPing(LibChipCard_ClientData,
					 requestid,
					 serviceid));
}


int ChipCard_CheckPing(int requestid){
  return ChipCard__xlerr(ReaderClient_CheckPing(LibChipCard_ClientData,
					    requestid));
}


int ChipCard_RequestAllocReader(int *requestid,
				unsigned int tid){
  return
    ChipCard__xlerr(ReaderClient_RequestAllocReader(LibChipCard_ClientData,
						requestid,
						(tid>>16) & 0xffff,
						tid & 0xffff));
}


int ChipCard_CheckAllocReader(int requestid,
			      int *tid,
			      CHIPCARD_READERDESCR **crd){
  int localtid;
  int serviceid;
  ERRORCODE err;
  READERCLIENT_READERDESCR *rd;
  CHIPCARD_READERDESCR *lrd;

  err=ReaderClient_CheckAllocReader(LibChipCard_ClientData,
				requestid,
				&localtid,
				&serviceid,
				&rd);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    return ChipCard__xlerr(err);
  }

  /* copy data */
  *tid=(serviceid << 16)+(localtid & 0xffff);

  assert(rd);
  lrd=ChipCard__ReaderDescr_new();
  if (strlen(rd->name)+1>sizeof(lrd->name) ||
      strlen(rd->type)+1>sizeof(lrd->type)) {
    ChipCard_ReaderDescr_free(lrd);
    ReaderClient_ReaderDescr_free(rd);
    DBG_ERROR("Description: name or type string too long");
    return CHIPCARD_ERROR_BUFFER;
  }
  strcpy(lrd->name, rd->name);
  strcpy(lrd->type, rd->type);

  lrd->flags=rd->flags;
  ReaderClient_ReaderDescr_free(rd);

  *crd=lrd;
  return CHIPCARD_SUCCESS;
}


int ChipCard_RequestReleaseReader(int *requestid,
				  unsigned int tid){
  return
    ChipCard__xlerr(ReaderClient_RequestReleaseReader(LibChipCard_ClientData,
						  requestid,
						  (tid >> 16) & 0xffff,
						  tid & 0xffff));
}


int ChipCard_CheckReleaseReader(int requestid){
  return
    ChipCard__xlerr(ReaderClient_CheckReleaseReader(LibChipCard_ClientData,
						requestid));
}


int ChipCard_RequestConnect(int *requestid,
			    unsigned int tid,
			    int cardId,
			    int waitForIt){
  return
    ChipCard__xlerr(ReaderClient_RequestConnectReader(LibChipCard_ClientData,
						      requestid,
						      (tid >> 16) & 0xffff,
						      tid & 0xffff,
						      cardId,
						      waitForIt));
}


int ChipCard_CheckConnect(int requestid,
			  int *result,
			  char *atrbuffer,
			  int *atrbufferlength){
  int err;

  err=
    ChipCard__xlerr(ReaderClient_CheckConnectReader(LibChipCard_ClientData,
						    requestid,
						    result,
						    atrbuffer,
						    atrbufferlength));
  if (err==CHIPCARD_SUCCESS) {
    *result=ChipCard__xlresult(*result);
  }
  return err;
}


int ChipCard_RequestStopConnect(int *requestid,
				unsigned int tid,
				int prevRequest){
  return
    ChipCard__xlerr(ReaderClient_RequestStopOpenReader(LibChipCard_ClientData,
						       requestid,
						       (tid >> 16) & 0xffff,
						       prevRequest));
}


int ChipCard_CheckStopConnect(int requestid,
			      int *result){
  int err;

  err=
    ChipCard__xlerr(ReaderClient_CheckStopOpenReader(LibChipCard_ClientData,
						     requestid,
						     result));
  if (err==CHIPCARD_SUCCESS) {
    *result=ChipCard__xlresult(*result);
  }
  return err;
}




int ChipCard_RequestDisconnect(int *requestid,
			       unsigned int tid){
  return
    ChipCard__xlerr(ReaderClient_RequestDisconnectReader(LibChipCard_ClientData,
						     requestid,
						     (tid >> 16) & 0xffff,
						     tid & 0xffff));
}


int ChipCard_CheckDisconnect(int requestid,
			     int *result){
  int err;

  err=
    ChipCard__xlerr(ReaderClient_CheckDisconnectReader(LibChipCard_ClientData,
						       requestid,
						       result));
  if (err==CHIPCARD_SUCCESS) {
    *result=ChipCard__xlresult(*result);
  }
  return err;
}


int ChipCard_RequestCommand(int *requestid,
			    unsigned int tid,
			    const char *sendBuffer,
			    int sendBufferLength){
  return
    ChipCard__xlerr(ReaderClient_RequestCommandReader(LibChipCard_ClientData,
						  requestid,
						  (tid >> 16) & 0xffff,
						  tid & 0xffff,
						  sendBuffer,
						  sendBufferLength));
}


int ChipCard_CheckCommand(int requestid,
			  int *result,
			  char *recvBuffer,
			  int *recvBufferLength){
  int err;

  err=
    ChipCard__xlerr(ReaderClient_CheckCommandReader(LibChipCard_ClientData,
						    requestid,
						    result,
						    recvBuffer,
						    recvBufferLength));
  if (err==CHIPCARD_SUCCESS) {
    *result=ChipCard__xlresult(*result);
  }
  return err;
}


int ChipCard_RequestFindReader(int *requestid,
			       const char *readerType,
			       unsigned int readerFlags,
			       unsigned int readerFlagsMask){
  CHIPCARDSUPERREQUEST *sr;
  ERRORCODE err;
  IPCMESSAGELAYER *ml;
  int reqid;
  int i;

  i=0;
  sr=ChipCard__SuperRequest_new();
  sr->waitForAll=1;

  /* create a request for each server */
  ml=LibChipCard_ClientData->service->messageLayers;
  while(sr->subRequestCount<CHIPCARD__MAX_SUBREQUESTS && ml) {
    err=ReaderClient_RequestFindReader(LibChipCard_ClientData,
				       &reqid,
				       IPCMessageLayer_GetId(ml),
				       readerType,
				       readerFlags,
				       readerFlagsMask);
    if (!Error_IsOk(err)) {
      DBG_ERROR_ERR(err);
    }
    else {
      DBG_DEBUG("Added request");
      sr->subRequests[sr->subRequestCount]=reqid;
      sr->serviceIds[sr->subRequestCount++]=IPCMessageLayer_GetId(ml);
      i++;
    }
    ml=ml->next;
  } /* while */

  if (!i) {
    DBG_ERROR("No request created, maybe \"chipcardd\" is not running ?");
    ChipCard__SuperRequest_free(sr);
    return CHIPCARD_ERROR_NO_REQUEST;
  }

  /* enqueue super request */
  ChipCard__AddSuperRequest(sr);
  *requestid=sr->id;

  return CHIPCARD_SUCCESS;
}


int ChipCard_CheckFindReader(int requestid,
			     unsigned int *readerbuffer,
			     int *readerBufferLength){
  int i,j;
  CHIPCARDSUPERREQUEST *sr;
  ERRORCODE err;
  int tids[CHIPCARD__MAX_FINDREADERS];
  int tidcount;
  int found;
  int requests;
  int messages;

  found=0;
  messages=0;
  sr=ChipCard__FindSuperRequest(requestid);
  if (!sr)
    return CHIPCARD_ERROR_NO_REQUEST;

  /* check all subrequests */
  for (i=0; i<sr->subRequestCount; i++) {
    if (sr->subRequests[i]!=0) {
      if (found>=*readerBufferLength)
	break;

      err=CTClient_CheckResponse(LibChipCard_ClientData,
				 sr->subRequests[i]);
      if (!Error_IsOk(err)) {
	if (Error_GetType(err)!=Error_FindType(CTSERVICE_ERROR_TYPE) ||
	    Error_GetCode(err)!=CTSERVICE_ERROR_NO_MESSAGE) {
	  DBG_ERROR_ERR(err);
	  if (Error_GetType(err)==Error_FindType(CTSERVICE_ERROR_TYPE) &&
	      Error_GetCode(err)==CTSERVICE_ERROR_UNREACHABLE) {
	    DBG_WARN("Service is unreachable, dequeuing request");
	    CTClient_WithdrawRequest(LibChipCard_ClientData,
				     sr->subRequests[i]);
	    sr->subRequests[i]=0;
	  }
	}
      }
      else {
	/* have a response, so handle it */
	messages++;
	tidcount=*readerBufferLength-found;
	err=ReaderClient_CheckFindReader(LibChipCard_ClientData,
				     sr->subRequests[i],
				     tids,
				     &tidcount);
	if (!Error_IsOk(err)) {
	  DBG_ERROR_ERR(err);
	}
	else {
          DBG_DEBUG("Found %d readers",tidcount);
	  /* copy new readers */
	  for (j=0; j<tidcount; j++) {
	    readerbuffer[j+found]=
	      ((sr->serviceIds[i]) << 16)+(tids[j] & 0xffff);
	  } /* for */
	  found+=tidcount;
	}
	/* mark subrequest free */
	sr->subRequests[i]=0;
      } /* have response */
    }
  } /* for */

  *readerBufferLength=found;

  /* check whether there still were requests */
  requests=0;
  for (i=0; i<sr->subRequestCount; i++) {
    if (sr->subRequests[i])
      requests++;
  } /* for */
  /* there were none, so free this super request */
  if (!requests) {
    ChipCard__RemoveSuperRequest(sr);
    ChipCard__SuperRequest_free(sr);
    DBG_DEBUG("Super request handled completely\n");
    if (!found)
      // nothing found, so no reader seems to be available
      return CHIPCARD_ERROR_NO_READER;
  }

  if (messages==0)
    return CHIPCARD_ERROR_NO_MESSAGE;
  else {
    if (found)
      return 0;
    else
      return CHIPCARD_ERROR_NO_MESSAGE;
  }
}


int ChipCard_MakeAPDU(char *buffer, int *bufferlen,
		      const char *command,
		      int argc, ...){
  int sendBufferLength;
  int rv;
  va_list arguments;
  char **argv;
  int i;

  assert(command);
  if (!LibChipCard_Commands) {
    DBG_ERROR("No card commands loaded");
    return CHIPCARD_ERROR_NO_COMMANDS;
  }

  sendBufferLength=*bufferlen;

  va_start(arguments,argc);
  argv=(char**)malloc(sizeof(char*)*argc);
  assert(argv);
  for (i=0; i<argc; i++)
    argv[i]=va_arg(arguments, char*);
  rv=CTCommand_MakeAPDU(LibChipCard_Commands,
			command,
			argc,
			argv,
			buffer,
			&sendBufferLength);
  free(argv);
  va_end(arguments);
  if (rv!=CTCOMMAND_RESULT_OK) {
    DBG_ERROR("Error in request (Code %d)",rv);
    return ChipCard__xlerr(Error_New(0,
				     ERROR_SEVERITY_ERR,
				     Error_FindType(CTSERVICE_ERROR_TYPE),
				     rv));
  }
  *bufferlen=sendBufferLength;
  return CHIPCARD_SUCCESS;
}




void ChipCard_WithdrawRequest(int requestid){
  if (requestid>=CHIPCARD__SUPERREQUEST_OFFSET) {
    CHIPCARDSUPERREQUEST *sr;
    int i;

    sr=ChipCard__FindSuperRequest(requestid);
    if (!sr) {
      DBG_WARN("Superrequest not found");
      return;
    }
    for (i=0; i<sr->subRequestCount; i++) {
      if (sr->subRequests[i]!=0)
	CTClient_WithdrawRequest(LibChipCard_ClientData,
				 sr->subRequests[i]);
    }
    ChipCard__RemoveSuperRequest(sr);
    ChipCard__SuperRequest_free(sr);
  }
  else
    return CTClient_WithdrawRequest(LibChipCard_ClientData,
				    requestid);
}


void ChipCard_AbandonRequest(int requestid){
  if (requestid>=CHIPCARD__SUPERREQUEST_OFFSET) {
    CHIPCARDSUPERREQUEST *sr;
    int i;

    sr=ChipCard__FindSuperRequest(requestid);
    if (!sr) {
      DBG_WARN("Superrequest not found");
      return;
    }
    for (i=0; i<sr->subRequestCount; i++) {
      if (sr->subRequests[i]!=0)
	CTClient_AbandonRequest(LibChipCard_ClientData,
				sr->subRequests[i]);
    }
    ChipCard__RemoveSuperRequest(sr);
    ChipCard__SuperRequest_free(sr);
  }
  else
    return CTClient_AbandonRequest(LibChipCard_ClientData,
				   requestid);
}


int ChipCard_LocateCommand(const char *command,
			   const char *readertype,
			   const char *cardtype,
			   char *buffer,
			   int bufferlen){
  int rv;

  rv=CTCommand_Locate(LibChipCard_Commands,
		      readertype,
		      cardtype,
		      command,
		      buffer,
		      bufferlen);
  if (rv!=CTCOMMAND_RESULT_OK)
    return CHIPCARD_ERROR_COMMAND_NOT_FOUND;
  return CHIPCARD_SUCCESS;
}


int ChipCard_ExistsCommand(const char *command){
  int rv;

  rv=CTCommand_Check(LibChipCard_Commands,
		     command);
  if (rv!=CTCOMMAND_RESULT_OK)
    return CHIPCARD_ERROR_COMMAND_NOT_FOUND;
  return CHIPCARD_SUCCESS;
}


int ChipCard_RequestStatReader(int *requestid,
			       int tid){
  return
    ChipCard__xlerr(ReaderClient_RequestStatReader(LibChipCard_ClientData,
						   requestid,
						   (tid >> 16) & 0xffff,
						   tid & 0xffff));
}


int ChipCard_CheckStatReader(int requestid,
			     int *result,
			     unsigned int *status,
			     char *atrbuffer,
			     int *atrlen){
  int err;

  err=
    ChipCard__xlerr(ReaderClient_CheckStatReader(LibChipCard_ClientData,
						 requestid,
						 result,
						 status,
						 atrbuffer,
						 atrlen));
  if (err==CHIPCARD_SUCCESS) {
    *result=ChipCard__xlresult(*result);
  }
  return err;
}


int ChipCard_RequestWaitReader(int *requestid,
			       int mustChange,
			       const char *readerType,
			       unsigned int readerFlags,
			       unsigned int readerFlagsMask,
			       unsigned int status,
			       unsigned int statusMask,
			       unsigned int statusDelta){
  CHIPCARDSUPERREQUEST *sr;
  ERRORCODE err;
  IPCMESSAGELAYER *ml;
  int reqid;
  int i;

  i=0;
  sr=ChipCard__SuperRequest_new();

  /* create a request for each server */
  ml=LibChipCard_ClientData->service->messageLayers;
  while(sr->subRequestCount<CHIPCARD__MAX_SUBREQUESTS && ml) {
    err=ReaderClient_RequestWaitReader(LibChipCard_ClientData,
				       &reqid,
				       IPCMessageLayer_GetId(ml),
				       mustChange,
				       readerType,
				       readerFlags,
				       readerFlagsMask,
				       status,
				       statusMask,
				       statusDelta);
    if (!Error_IsOk(err)) {
      DBG_NOTICE_ERR(err);
    }
    else {
      DBG_DEBUG("Added request");
      sr->subRequests[sr->subRequestCount]=reqid;
      sr->serviceIds[sr->subRequestCount++]=IPCMessageLayer_GetId(ml);
      i++;
    }
    ml=ml->next;
  } /* while */

  if (!i) {
    DBG_NOTICE("No request created, maybe \"chipcardd\" is not running ?");
    ChipCard__SuperRequest_free(sr);
    return CHIPCARD_ERROR_NO_REQUEST;
  }

  /* enqueue super request */
  ChipCard__AddSuperRequest(sr);
  *requestid=sr->id;

  return CHIPCARD_SUCCESS;
}



int ChipCard_CheckWaitReader(int requestid,
			     int *tid,
			     unsigned int *status,
			     unsigned int *readerflags,
			     unsigned int *cardId){
  int i;
  CHIPCARDSUPERREQUEST *sr;
  ERRORCODE err;
  int requests;
  int ltid;

  sr=ChipCard__FindSuperRequest(requestid);
  if (!sr)
    return CHIPCARD_ERROR_NO_REQUEST;

  /* check all subrequests until one reveals a response */
  requests=0;
  for (i=0; i<sr->subRequestCount; i++) {
    if (sr->subRequests[i]!=0) {
      err=ReaderClient_CheckWaitReader(LibChipCard_ClientData,
				       sr->subRequests[i],
				       &ltid,
				       status,
				       readerflags,
				       cardId);
      if (!Error_IsOk(err)) {
	if (Error_GetType(err)!=Error_FindType(CTSERVICE_ERROR_TYPE) ||
	    Error_GetCode(err)!=CTSERVICE_ERROR_NO_MESSAGE) {
	  DBG_ERROR_ERR(err);
	  return ChipCard__xlerr(err);
	}
      }
      else {
	*tid=((sr->serviceIds[i]) << 16)+(ltid & 0xffff);
	return CHIPCARD_SUCCESS;
      } /* have response */
    } /* have request */
  } /* for */
  return CHIPCARD_ERROR_NO_MESSAGE;
}


int ChipCard_StopWaitReader(int prevRequest){
  int i;
  CHIPCARDSUPERREQUEST *sr;
  ERRORCODE err;
  int requests;
  int requestid;

  sr=ChipCard__FindSuperRequest(prevRequest);
  if (!sr)
    return CHIPCARD_ERROR_NO_REQUEST;

  /* send a StopWait request to all servers in the superrequest */
  requests=0;
  for (i=0; i<sr->subRequestCount; i++) {
    if (sr->subRequests[i]!=0) {
      err=ReaderClient_RequestStopWaitReader(LibChipCard_ClientData,
					 &requestid,
					 sr->serviceIds[i],
					 sr->subRequests[i]);
      if (!Error_IsOk(err)) {
	DBG_NOTICE_ERR(err);
      }
      CTClient_AbandonRequest(LibChipCard_ClientData,sr->subRequests[i]);
      sr->subRequests[i]=0;
    } /* have request */
  } /* for */

  DBG_DEBUG("Dequeuing WaitReader requests");
  ChipCard__RemoveSuperRequest(sr);
  ChipCard__SuperRequest_free(sr);

  return CHIPCARD_SUCCESS;

}


