/* 
 * $Id: ctktable.c,v 1.23 2000/07/12 22:47:15 terpstra Exp $
 *
 * CTK - Console Toolkit
 *
 * Copyright (C) 1998-2000 Stormix Technologies Inc.
 *
 * License: LGPL
 *
 * Authors: Kevin Lindsay, Wesley Terpstra
 *  
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License as published by the Free Software Foundation; either
 *    version 2 of the License, or (at your option) any later version.
 *    
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *    
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 */

#include <stdio.h>
#include <stdlib.h>
#include <glib.h>
#include <string.h>

#include "ctkcolor.h"
#include "ctk.h"

// FIXME: I need a destructor for heterogeneous

static gint compareWidgetX(gconstpointer a, gconstpointer b)
{
	gint ra = ((CtkTableChild*)a)->right_attach;
	gint rb = ((CtkTableChild*)b)->right_attach;
	
	if (rb != ra)
	{ /* Major sort key by end of attach */
		return ra - rb;
	}
	else
	{ /* Minor sort key by ptr (work around for duplicate keys) */
		return a - b;
	}
}

static gint compareWidgetY(gconstpointer a, gconstpointer b)
{
	gint ba = ((CtkTableChild*)a)->bottom_attach;
	gint bb = ((CtkTableChild*)b)->bottom_attach;
	
	if (bb != ba)
	{ /* Major sort key by end of attach */
		return ba - bb;
	}
	else
	{ /* Minor sort key by ptr (work around for duplicate keys) */
		return a - b;
	}
}

/* Initialize the table */
void ctk_table_init(CtkTable *table, guint rows, guint cols, gboolean homogeneous)
{
	ctk_container_init(&table->container);
	
	table->row_spacing = 0;
	table->col_spacing = 0;

	((CtkWidget*)table)->width  = 1;
	((CtkWidget*)table)->height = 1;
	
	((CtkWidget*)table)->min_width  = 0;
	((CtkWidget*)table)->min_height = 0;
	
	((CtkWidget*)table)->orig_width  = 0;
	((CtkWidget*)table)->orig_height = 0;
	
	((CtkWidget *)table)->main_col       = ctk_calculate_palette(CTK_COLOR_BLACK,CTK_COLOR_LIGHT_GRAY);
	((CtkWidget *)table)->title_col      = ctk_calculate_palette(CTK_COLOR_LIGHT_GRAY,CTK_COLOR_BLACK);
	((CtkWidget *)table)->inverse_col    = ctk_calculate_palette(CTK_COLOR_WHITE,CTK_COLOR_RED);
	((CtkWidget *)table)->selected_col   = ctk_calculate_palette(CTK_COLOR_WHITE,CTK_COLOR_BLUE);
	
	memset(&table->layout, 0, sizeof(table->layout));
	
	((CtkWidget*)table)->set_min_size  = &ctk_table_min_size;
	((CtkWidget*)table)->set_real_size = &ctk_table_real_size;

	((CtkObject*)table)->type = CtkTypeTable;

	((CtkTable*) table)->homogeneous = homogeneous;
	((CtkTable*) table)->rows        = rows;
	((CtkTable*) table)->cols        = cols;
	
	if (!homogeneous)
	{
		((CtkTable*)table)->layout.hetero.xstop = g_tree_new(&compareWidgetX);
		((CtkTable*)table)->layout.hetero.ystop = g_tree_new(&compareWidgetY);
		((CtkTable*)table)->layout.hetero.x = (CtkTableAxis*)calloc(cols+1, sizeof(CtkTableAxis));
		((CtkTable*)table)->layout.hetero.y = (CtkTableAxis*)calloc(rows+1, sizeof(CtkTableAxis));
	}
}

CtkWidget* ctk_table_new(guint rows, guint cols, gboolean homogeneous)
{
	CtkTable *table;
	
	table = g_malloc(sizeof(CtkTable));
	
	ctk_table_init(table, rows, cols, homogeneous);
	
    return (CtkWidget*)table;
}

void ctk_table_attach(CtkTable *table,
		      CtkWidget *child,
		      guint left_attach,
		      guint right_attach,
		      guint top_attach,
		      guint bottom_attach,
		      CtkAttachOptions xoptions,
		      CtkAttachOptions yoptions,
		      guint xpadding,
		      guint ypadding)
{
      CtkWidget* tablechild;
      gint       i;
      
      if (!table || !child)
	    return;
	    
	  ctk_assert(right_attach  >  left_attach, "Attempt to attach X inversed widget to table");
	  ctk_assert(bottom_attach >  top_attach,  "Attempt to attach Y inversed widget to table");
	  ctk_assert(left_attach   >= 0,           "Attempt to attach widget before first col");
	  ctk_assert(top_attach    >= 0,           "Attempt to attach widget before first row");
	  ctk_assert(right_attach  <= table->cols, "Attempt to attach widget after last col");
	  ctk_assert(bottom_attach <= table->rows, "Attempt to attach widget after last row");
	  
      tablechild = ctk_table_child_new(left_attach,
				       right_attach,
				       top_attach,
				       bottom_attach);
      
      ctk_widget_show(tablechild); /* Always show the container */
      ctk_container_add(CTK_CONTAINER(table),tablechild);
      ctk_container_add(CTK_CONTAINER(tablechild),child);
      
      child->xexpand  = (xoptions & CTK_EXPAND) != 0;
      child->xshrink  = (xoptions & CTK_SHRINK) != 0;
      child->xfill    = (xoptions & CTK_FILL) != 0;
      child->xpadding = xpadding;
      child->yexpand  = (yoptions & CTK_EXPAND) != 0;
      child->yshrink  = (yoptions & CTK_SHRINK) != 0;
      child->yfill    = (yoptions & CTK_FILL) != 0;
      child->ypadding = ypadding;
      
      /* Not homogeneous; stick it in the list */
      if (!table->homogeneous)
      {
           if (child->xexpand)
           {
                 for (i = left_attach; i < right_attach; i++)
                 {
                       table->layout.hetero.x[i].canGrow = TRUE;
                 }
           }
           if (child->yexpand)
           {
                 for (i = top_attach; i < bottom_attach; i++)
                 {
                       table->layout.hetero.y[i].canGrow = TRUE;
                 }
           }
           
           g_tree_insert(table->layout.hetero.xstop, tablechild, NULL);
           g_tree_insert(table->layout.hetero.ystop, tablechild, NULL);
      }
}

void ctk_table_detach(CtkTable* table, CtkWidget* child)
{
	CtkTableChild* tablechild = CTK_TABLE_CHILD(child->node->parent->data);
	ctk_container_remove(CTK_CONTAINER(table), CTK_WIDGET(tablechild));
	ctk_container_remove(CTK_CONTAINER(tablechild), child);
	
	if (!table->homogeneous)
	{
		g_tree_remove(table->layout.hetero.ystop, tablechild);
		g_tree_remove(table->layout.hetero.xstop, tablechild);
	}
	
	/* Destroy the tablechild, but not the widget itself */
	ctk_widget_destroy(CTK_WIDGET(tablechild));
}

void ctk_table_delete_row(CtkTable* table, gint row)
{
	GNode*     scan;
	GNode*     next;
	
	/* Correct attachments */
	for (scan = ((CtkWidget*)table)->node->children; scan; scan = next)
	{
		CtkTableChild* child = CTK_TABLE_CHILD(scan->data);
		
		next = scan->next;
		
		if (child->top_attach <= row && child->bottom_attach > row)
		{
			CtkWidget* victim = CTK_WIDGET(child);
			victim = CTK_WIDGET(victim->node->children->data);
			ctk_table_detach(table, victim);
			ctk_widget_destroy(victim);
		}
		else
		{
			if (child->top_attach    > row) child->top_attach--;
			if (child->bottom_attach > row) child->bottom_attach--;
		}
	}
	
	table->rows--;	
	
	if (!table->homogeneous)
	{
		memmove(&table->layout.hetero.y[row], 
				&table->layout.hetero.y[row+1],
				sizeof(CtkTableAxis) * (table->rows - row));
	}
	
	ctk_size_mark_changed(CTK_WIDGET(table));
}

void ctk_table_delete_below_row(CtkTable* table, gint row)
{
	GNode* scan;
	GNode* next;

	for (scan = ((CtkWidget*)table)->node->children; scan; scan = next)
	{
		CtkTableChild* child = CTK_TABLE_CHILD(scan->data);
		
		next = scan->next;
		
		if (child->top_attach >= row)
		{
			CtkWidget* victim = CTK_WIDGET(child);
			victim = CTK_WIDGET(victim->node->children->data);
			ctk_table_detach(table, victim);
			ctk_widget_destroy(victim);
		}
	}
	
	table->rows = row;
	ctk_size_mark_changed(CTK_WIDGET(table));
}

/* Find a widget based on the specified coordinates */
CtkWidget* ctk_table_find_child(CtkTable *table,
				guint left_attach,
				guint right_attach,
				guint top_attach,
				guint bottom_attach)
{
      GNode *child_node;
      CtkTableChild *tablechild;
      
      for (child_node = ((CtkWidget *)table)->node->children;
	   child_node;
	   child_node = child_node->next)
      {
	    tablechild = (CtkTableChild *)child_node->data;
	    
		if (left_attach >= tablechild->left_attach &&
		    right_attach <= tablechild->right_attach &&
		    top_attach >= tablechild->top_attach &&
		    bottom_attach <= tablechild->bottom_attach)
		{
		      return CTK_WIDGET(((CtkWidget*)tablechild)->node->children->data);
		}
      }
      
      return NULL;
}

/* Set row spacings */
void ctk_table_set_row_spacings(CtkTable* table, gint spacings)
{
	if (!table)
		return;
      
	table->row_spacing = spacings;
	ctk_size_mark_changed(CTK_WIDGET(table));
}

/* Set col spacings */
void ctk_table_set_col_spacings(CtkTable* table, gint spacings)
{
	if (!table)
	    return;
	
	table->col_spacing = spacings;
	ctk_size_mark_changed(CTK_WIDGET(table));
}

static int divide_roundup(int a, int b)
{
      return (a + (b - 1)) / b;
}

static void ctk_table_get_min_size_homogeneous(CtkTable* table)
{
      CtkWidget*     widget;
      GNode*         scan;
      CtkWidget*     which;
      CtkTableChild* child;
      gint           needX;
      gint           needY;
      gint           cols;
      gint           rows;

      widget = CTK_WIDGET(table);
      
      table->layout.homo.min_col_size = 0;
      table->layout.homo.min_row_size = 0;
      
      for (scan = widget->node->children;
	   scan;
	   scan = scan->next)
      {
	    which = CTK_WIDGET(scan->data);
	    child = CTK_TABLE_CHILD(which);
	    
	    if (!which->show) continue;
	    
	    cols = child->right_attach  - child->left_attach;
	    rows = child->bottom_attach - child->top_attach;
	    
	    /* Absorb padding into multi-cell widgets */
	    needX = divide_roundup(which->min_width  - 
				   table->col_spacing * (cols - 1), cols);
	    needY = divide_roundup(which->min_height - 
				   table->row_spacing * (rows - 1), rows);
	    
	    if (needX > table->layout.homo.min_col_size)
		  table->layout.homo.min_col_size = needX;
	    if (needY > table->layout.homo.min_row_size)
		  table->layout.homo.min_row_size = needY;
      }
      
      /* Minimum size including padding */
      widget->min_width  =  table->cols *
	    (table->layout.homo.min_col_size + table->col_spacing) -
	    table->col_spacing;
      widget->min_height = table->rows *
	    (table->layout.homo.min_row_size + table->row_spacing) -
	    table->row_spacing;
}

static gint residue_x(CtkTable* table, CtkTableChild* child)
{
	/* This is space not accounted for that the widget needs */
	/* Assumption: all columns but the last have been touched */
	
	gint spacingUsed;
	gint prevColUsage;
	
	spacingUsed = (child->right_attach - child->left_attach - 1) *
					table->col_spacing;
	prevColUsage = table->layout.hetero.x[child->right_attach - 1].accumSize -
					table->layout.hetero.x[child->left_attach].accumSize;
	
	return ((CtkWidget*)child)->min_width - spacingUsed - prevColUsage;
}

// FIXME hard_size is buggy if mixed with not hard stuff
static void fill_empty_cols(CtkTable* table, gint col)
{
	gint tracer;
	
	for (tracer = col; table->layout.hetero.x[tracer].accumSize == -1; tracer--)
	{
		// noop
	}
	
	while (tracer < col)
	{
		table->layout.hetero.x[tracer+1].accumSize = 
			table->layout.hetero.x[tracer].accumSize +
			table->layout.hetero.x[tracer].hard_size;
		tracer++;
	}
}

static gint compress_x_helper(gpointer key, gpointer value, gpointer data)
{
	CtkTableChild* child = CTK_TABLE_CHILD(key);
	CtkTable*      table = CTK_TABLE(data);
	gint           residue_accumulated;
	
	fill_empty_cols(table, child->right_attach);
	
	if (table->layout.hetero.x[child->right_attach - 1].hard_size)
		return FALSE;
	
	residue_accumulated = residue_x(table, child) +
					table->layout.hetero.x[child->right_attach - 1].accumSize;
	
	if (residue_accumulated > 
		table->layout.hetero.x[child->right_attach].accumSize)
	{
		table->layout.hetero.x[child->right_attach].accumSize = residue_accumulated;
	}
	
	return FALSE;
}

static void compress_x(CtkTable* table)
{
	gint i;
	for (i = 1; i <= table->cols; i++)
		table->layout.hetero.x[i].accumSize = -1;
	table->layout.hetero.x[0].accumSize = 0;
	
	g_tree_traverse(table->layout.hetero.xstop, 
		&compress_x_helper,
		G_IN_ORDER,
		table);
		
	fill_empty_cols(table, table->cols);
}

static gint residue_y(CtkTable* table, CtkTableChild* child)
{
	/* This is space not accounted for that the widget needs */
	/* Assumption: all columns but the last have been touched */
	
	gint spacingUsed;
	gint prevRowUsage;
	
	spacingUsed = (child->bottom_attach - child->top_attach - 1) *
					table->row_spacing;
	prevRowUsage = table->layout.hetero.y[child->bottom_attach - 1].accumSize -
					table->layout.hetero.y[child->top_attach].accumSize;
	
	return ((CtkWidget*)child)->min_height - spacingUsed - prevRowUsage;
}

// FIXME hard_size is buggy if mixed with not hard stuff
static void fill_empty_rows(CtkTable* table, gint row)
{
	gint tracer;
	
	for (tracer = row; table->layout.hetero.y[tracer].accumSize == -1; tracer--)
	{
		// noop
	}
	
	while (tracer < row)
	{
		table->layout.hetero.y[tracer+1].accumSize = 
			table->layout.hetero.y[tracer].accumSize +
			table->layout.hetero.y[tracer].hard_size;
		tracer++;
	}
}

static gint compress_y_helper(gpointer key, gpointer value, gpointer data)
{
	CtkTableChild* child = CTK_TABLE_CHILD(key);
	CtkTable*      table = CTK_TABLE(data);
	gint           residue_accumulated;
	
	fill_empty_rows(table, child->bottom_attach);
	
	if (table->layout.hetero.y[child->bottom_attach - 1].hard_size)
		return FALSE;
	
	residue_accumulated = residue_y(table, child) +
				table->layout.hetero.y[child->bottom_attach - 1].accumSize;
	
	if (residue_accumulated > 
		table->layout.hetero.y[child->bottom_attach].accumSize)
	{
		table->layout.hetero.y[child->bottom_attach].accumSize = residue_accumulated;
	}
	
	return FALSE;
}

static void compress_y(CtkTable* table)
{
	gint i;
	for (i = 1; i <= table->rows; i++)
		table->layout.hetero.y[i].accumSize = -1;
	table->layout.hetero.y[0].accumSize = 0;
	
	g_tree_traverse(table->layout.hetero.ystop, 
		&compress_y_helper,
		G_IN_ORDER,
		table);
		
	fill_empty_rows(table, table->rows);
}
static void ctk_table_get_min_size_heterogeneous(CtkTable* table)
{
	compress_x(table);
	compress_y(table);
	
	((CtkWidget*)table)->min_width  = 
		table->layout.hetero.x[table->cols].accumSize + 
			(table->cols - 1) * table->col_spacing;
	((CtkWidget*)table)->min_height = 
		table->layout.hetero.y[table->rows].accumSize +
			(table->rows - 1) * table->row_spacing;
}

void ctk_table_min_size(CtkWidget* widget)
{
	CtkTable*     table     = CTK_TABLE(widget);
	CtkContainer* container = CTK_CONTAINER(widget);
	
	if (table->homogeneous)
	{
		ctk_table_get_min_size_homogeneous(table);
	}
	else
	{
		ctk_table_get_min_size_heterogeneous(table);
	}

	widget->min_width  += container->border_width*2;
	widget->min_height += container->border_width*2;
}

static gint size_type_table_heterogeneous(GNode* node, CtkWidget* widget)
{
	/* We want to divide extra space evenly among the expanding columns */
	CtkTable*      table;
	CtkTableChild* tablechild;
	CtkWidget*     widgetchild;
	GNode*         scanner;
	gint           growAxis;
	gint           growSize;
	gint           growOverflow;
	gint           growAccum;
	gint           v1; /* temporaries */
	gint           v2;
	gint           i;  /* counter */
	
	if (!widget || !node)
	  return 0;
	
	table = CTK_TABLE(widget);

	/* Grow the vertical columns that are set to expand */
	growAxis = 0;
	for (i = 0; i < table->cols; i++)
		if (table->layout.hetero.x[i].canGrow) growAxis++;
	
	if (growAxis == 0)
	{
		growSize     = 0;
		growOverflow = 0;
	}
	else
	{
		growSize     = (widget->width - widget->min_width) / growAxis;
		growOverflow = (widget->width - widget->min_width) % growAxis;
	}
	
	growAxis = 0;
	growAccum = 0;
	for (i = 0; i < table->cols; i++)
	{
		if (table->layout.hetero.x[i].canGrow)
		{
			if (growAxis < growOverflow)
			  growAccum++;
			
			growAccum += growSize;
			table->layout.hetero.x[i+1].accumSize += growAccum;
			growAxis++;
		}
		else
		{
			table->layout.hetero.x[i+1].accumSize += growAccum;
		}		
	}
	
	/* Grow the vertical columns that are set to expand */
	growAxis = 0;
	for (i = 0; i < table->rows; i++)
		if (table->layout.hetero.y[i].canGrow) growAxis++;
	
	if (growAxis == 0)
	{
		growSize     = 0;
		growOverflow = 0;
	}
	else
	{
		growSize     = (widget->height - widget->min_height) / growAxis;
		growOverflow = (widget->height - widget->min_height) % growAxis;
	}
	
	growAxis = 0;
	growAccum = 0;
	for (i = 0; i < table->rows; i++)
	{
		if (table->layout.hetero.y[i].canGrow)
		{
			if (growAxis < growOverflow)
			  growAccum++;
			
			growAccum += growSize;
			table->layout.hetero.y[i+1].accumSize += growAccum;
			growAxis++;
		}
		else
		{
			table->layout.hetero.y[i+1].accumSize += growAccum;
		}
	}

	/* Now, lets place widgets - no worries about details like centering,
	 * visible, expand, etc. b/c tablechild does it for us
	 */
	for (scanner = node->children; scanner; scanner = scanner->next)
	{
		widgetchild = CTK_WIDGET(scanner->data);
		tablechild = CTK_TABLE_CHILD(widgetchild);
		
		v1 = table->layout.hetero.x[tablechild->left_attach].accumSize;
		v2 = table->layout.hetero.x[tablechild->right_attach].accumSize;
		
		widgetchild->width = v2 - v1 + table->col_spacing * 
			(tablechild->right_attach - tablechild->left_attach - 1);
		widgetchild->col   = widget->col + v1 + table->col_spacing * 
			tablechild->left_attach;
		
		v1 = table->layout.hetero.y[tablechild->top_attach].accumSize;
		v2 = table->layout.hetero.y[tablechild->bottom_attach].accumSize;
		
		widgetchild->height = v2 - v1 + table->row_spacing * 
			(tablechild->bottom_attach - tablechild->top_attach - 1);
		widgetchild->row    = widget->row + v1 + table->row_spacing *
			tablechild->top_attach;
	}
	
	return 0;
}

static gint size_type_table_homogeneous(GNode* node, CtkWidget* widget)
{
      /* This is the easy case :-) */
      /* We've already got our height, width, row, col set for us */
      /* Plan, give extra space to the earlier rows / cols */
      
      GNode*         scanner;
      CtkTable*      table;
      CtkTableChild* tablechild;
      CtkWidget*     widgetchild;
      gint           rowSize;
      gint           colSize;
      gint           rowOverflow;
      gint           colOverflow;
      gint           colSpan;
      gint           rowSpan;
      gint           rowSpace;
      gint           colSpace;

      if (!node || !widget)
	    return 0;

      table  = CTK_TABLE(widget);
      
      rowSpace    = widget->height - table->row_spacing * (table->rows - 1);
      if (table->rows)
      {
          rowSize     = rowSpace / table->rows;
          rowOverflow = rowSpace % table->rows;
      }
	  else
	  {
	      rowSize     = 0;
	      rowOverflow = 0;
	  }
      colSpace    = widget->width  - table->col_spacing * (table->cols - 1);
      if (table->cols)
      {
          colSize     = colSpace / table->cols;
          colOverflow = colSpace % table->cols;
      }
      else
      {
          colSize     = 0;
          colOverflow = 0;
      }

      for (scanner = node->children; scanner; scanner = scanner->next)
      {
	    widgetchild = CTK_WIDGET(scanner->data);
	    tablechild  = CTK_TABLE_CHILD(widgetchild);
	    /* Because tablechild is a real bin it does expand, visible,
	     * centering, and all that for us. :-)
	     */
	    colSpan = tablechild->right_attach  - tablechild->left_attach;
	    rowSpan = tablechild->bottom_attach - tablechild->top_attach;

	    widgetchild->row = widget->row + tablechild->top_attach  * 
		  (rowSize + table->row_spacing);
	    widgetchild->col = widget->col + tablechild->left_attach * 
		  (colSize + table->col_spacing);
	    widgetchild->height = rowSpan * (rowSize + table->row_spacing) -
		  table->row_spacing;
	    widgetchild->width  = colSpan * (colSize + table->col_spacing) -
		  table->col_spacing;

	    /* Three cases for the overflow: */
	    if (tablechild->left_attach < colOverflow)
	    {
		  if (tablechild->right_attach < colOverflow)
		  { /* Cell lies completely in overflow space */
			widgetchild->width += colSpan;
			widgetchild->col   += tablechild->left_attach;
		  }
		  else
		  { /* Cell lies on threshhold */
			widgetchild->width += colOverflow - 
			      tablechild->left_attach;
			widgetchild->col   += tablechild->left_attach;
		  }
	    }
	    else
	    { /* Cell lies completely past overflow space */
		  widgetchild->col += colOverflow;
	    }

	    /* Three cases for the overflow: */
	    if (tablechild->top_attach < rowOverflow)
	    {
		  if (tablechild->bottom_attach < rowOverflow)
		  { /* Cell lies completely in overflow space */
			widgetchild->height += rowSpan;
			widgetchild->row    += tablechild->top_attach;
		  }
		  else
		  { /* Cell lies on threshhold */
			widgetchild->height += rowOverflow - 
			      tablechild->top_attach;
			widgetchild->row    += tablechild->top_attach;
		  }
	    }
	    else
	    { /* Cell lies completely past overflow space */
		  widgetchild->row += rowOverflow;
	    }     
      }

      return 0;
}

void ctk_table_real_size(CtkWidget *widget)
{
	CtkTable*     table     = CTK_TABLE(widget);
	CtkContainer* container = CTK_CONTAINER(widget);
	
	widget->col += container->border_width;
	widget->row += container->border_width;
	widget->width  -= container->border_width*2;
	widget->height -= container->border_width*2;
	widget->min_width  -= container->border_width*2;
	widget->min_height -= container->border_width*2;
	
	if (table->homogeneous)
		size_type_table_homogeneous(widget->node, widget);
	else
		size_type_table_heterogeneous(widget->node, widget);

	widget->col -= container->border_width;
	widget->row -= container->border_width;
	widget->width  += container->border_width*2;
	widget->height += container->border_width*2;
	widget->min_width  += container->border_width*2;
	widget->min_height += container->border_width*2;
}

void ctk_table_insert_row(CtkTable* table, gint row)
{
	CtkWidget* self    = CTK_WIDGET(table);
	gint       oldRows = table->rows;
	GNode*     scan;
	gint       i;
	
	if (row < table->rows)
	{
		table->rows++;
	}
	else
	{ /* row >= table->rows */
		table->rows = row+1;
	}
	
	if (!table->homogeneous)
	{
		table->layout.hetero.y = (CtkTableAxis*)
			realloc(table->layout.hetero.y, sizeof(CtkTableAxis) * (table->rows+1));
		
		for (i = oldRows; i > row; i--)
		{
			table->layout.hetero.y[i].canGrow   = table->layout.hetero.y[i-1].canGrow;
			table->layout.hetero.y[i].hard_size = table->layout.hetero.y[i-1].hard_size;
		}
		
		for (i = oldRows+1; i < table->rows+1; i++)
		{
			table->layout.hetero.y[i].canGrow = FALSE;
			table->layout.hetero.y[i].hard_size = 0;
			table->layout.hetero.y[i].accumSize = 0;
		}

		table->layout.hetero.y[row].canGrow   = FALSE;
		table->layout.hetero.y[row].hard_size = 0;
		table->layout.hetero.y[row].accumSize = 0;
	}
	
	/* Now if we need to change attaches, do it. */
	if (table->rows - 1 != row)
	{
		for (scan = self->node->children; scan; scan = scan->next)
		{
			CtkTableChild* child       = CTK_TABLE_CHILD(scan->data);
			CtkWidget*     childwidget = CTK_WIDGET(scan->data);
			
			if (child->top_attach >= row) child->top_attach++;
			if (child->bottom_attach > row) child->bottom_attach++;
				
			if (child->top_attach <= row && child->bottom_attach > row 
				&& childwidget->yexpand && !table->homogeneous)
			{
				table->layout.hetero.y[row].canGrow = TRUE;
			}
		}
	}
	
	ctk_size_mark_changed(CTK_WIDGET(table));
}

void ctk_table_insert_col(CtkTable* table, gint col)
{
	CtkWidget* self    = CTK_WIDGET(table);
	gint       oldCols = table->cols;
	GNode*     scan;
	gint       i;
	
	if (col < table->cols)
	{
		table->cols++;
	}
	else
	{
		table->cols = col+1;
	}
	
	if (!table->homogeneous)
	{
		table->layout.hetero.x = (CtkTableAxis*)
			realloc(table->layout.hetero.x, sizeof(CtkTableAxis) * (table->cols+1));
		
		for (i = oldCols; i > col; i--)
		{
			table->layout.hetero.x[i].canGrow   = table->layout.hetero.x[i-1].canGrow;
			table->layout.hetero.x[i].hard_size = table->layout.hetero.x[i-1].hard_size;
		}
		
		for (i = oldCols+1; i < table->cols+1; i++)
		{
			table->layout.hetero.x[i].canGrow   = FALSE;
			table->layout.hetero.x[i].hard_size = 0;
			table->layout.hetero.x[i].accumSize = 0;
		}

		table->layout.hetero.x[col].canGrow = FALSE;
		table->layout.hetero.x[col].hard_size = 0;
		table->layout.hetero.x[col].accumSize = 0;
	}
	
	/* Now if we need to change attaches, do it. */
	if (table->cols - 1 != col)
	{
		for (scan = self->node->children; scan; scan = scan->next)
		{
			CtkTableChild* child       = CTK_TABLE_CHILD(scan->data);
			CtkWidget*     childwidget = CTK_WIDGET(scan->data);
			
			if (child->left_attach >= col) child->left_attach++;
			if (child->right_attach > col) child->right_attach++;
			
			if (child->left_attach <= col && child->right_attach > col 
				&& childwidget->xexpand && !table->homogeneous)
			{
				table->layout.hetero.x[col].canGrow = TRUE;
			}
		}
	}
	
	ctk_size_mark_changed(CTK_WIDGET(table));
}

void ctk_table_colour_row(CtkTable* table, gint row, gint colour)
{
	CtkWidget* self = CTK_WIDGET(table);
	GNode*     scan;
	
	for (scan = self->node->children; scan; scan = scan->next)
	{
		CtkTableChild* child = CTK_TABLE_CHILD(scan->data);
		
		if (child->top_attach <= row && child->bottom_attach > row)
		{
			((CtkWidget*)child)->main_col        = colour;
			((CtkWidget*)child)->insensitive_col = colour;
		}
	}
	
	ctk_draw_mark_changed(CTK_WIDGET(table));
}

void ctk_table_column_hard_size(CtkTable* table, gint col, gint size)
{
	if (table->homogeneous) return;
	
	table->layout.hetero.x[col].hard_size = size;
	ctk_size_mark_changed(CTK_WIDGET(table));
}

gint ctk_table_get_row_row(CtkTable* table, gint row)
{
	if (table->homogeneous)
	{
		return ((CtkWidget*)table)->row + 
			(row * ((CtkWidget*)table)->height) / table->rows;
	}
	else
	{
		return ((CtkWidget*)table)->row + table->layout.hetero.y[row].accumSize + 
			table->row_spacing * row;
	}
}

gint ctk_table_get_col_col(CtkTable* table, gint col)
{
	if (table->homogeneous)
	{
		return ((CtkWidget*)table)->col + 
			(col * ((CtkWidget*)table)->width) / table->cols;
	}
	else
	{
		return ((CtkWidget*)table)->col + table->layout.hetero.x[col].accumSize + 
			table->col_spacing * col;
	}
}

void ctk_table_get_click_cell(CtkTable* table, gint* row, gint* col, gint x, gint y)
{
	/* Let's make this super-fast.. :-) - we will do a binary search */
	gint left;
	gint right;
	gint middle;
	gint top;
	gint bottom;
	
	left  = 0;
	right = table->cols;
	
	while (left < right - 1)
	{
		middle = (left + right) / 2;
		if (ctk_table_get_col_col(table, middle) > x)
		{
			right = middle;
		}
		else
		{
			left = middle;
		}
	}
	
	*col = left;
	
	top    = 0;
	bottom = table->rows;
	
	while (top < bottom - 1)
	{
		middle = (top + bottom) / 2;
		if (ctk_table_get_row_row(table, middle) > y)
		{
			bottom = middle;
		}
		else
		{
			top = middle;
		}
	}
	
	*row = top;
}
