/*
 * CV2D.h
 * $Id: CV2D.h,v 1.4 2003/06/24 14:50:02 anxo Exp $
 *
 * Copyright (C) 1999, 2000 Markus Janich, Michael Meissner, Rainer Jaeger
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * As a special exception to the GPL, the QGLViewer authors (Markus
 * Janich, Michael Meissner, Richard Guenther, Alexander Buck and Thomas
 * Woerner) give permission to link this program with Qt (non-)commercial
 * edition, and distribute the resulting executable, without including
 * the source code for the Qt (non-)commercial edition in the source
 * distribution.
 *
 */



#ifndef __CV2D_H
#define __CV2D_H



// System
///////////
#include <math.h>
#ifdef _MSC_VER
#if _MSC_VER >= 1300
#include <iostream>
#endif
#else
#include <iostream.h>
#endif


// Own
///////////
//#include "CP2D.h"

// Forward declaration
////////////////////////
class CV3D;

using namespace std;

/** This class implements a vector class for 2D coordinates
  * including functionality to vector calculations.
  *
  * @author Markus Janich, Michael Meissner, Rainer Jaeger
  */
class CV2D {
public:
  static double epsilon;

  /** Default constructor.
    * The default value of the instantiated vector will be (0.0,0.0). */
  CV2D(void) { m_ard[0] = 0.0;
               m_ard[1] = 0.0; };

  /** Construct new vector.
    * The value of the vector will be (rdX, rdY). */
  CV2D(double rdX, double rdY) { m_ard[0] = rdX;
                                 m_ard[1] = rdY; };

  /** Copyconstructor.
    * Initializes the new vector with the vector passed in 'v'. */
  CV2D(const CV2D& Vector) { m_ard[0] = Vector.m_ard[0];
                             m_ard[1] = Vector.m_ard[1]; };

  /** Default destructor. */
  ~CV2D(void) {};



  //////////////////////////
  // OVERLOADED OPERATORS //
  //////////////////////////

  /** Cast operator to convert CV2D vectors to CV3D vectors.
    * Initializes the new vector with this vector.
    * The third component is set to 0. */
  operator CV3D() const;

  /** Assign one vector to another.*/
  const CV2D& operator=(const CV2D&);

  /** Compares to vectors.
    * The result will be 'true' if the two vector are indentically
    * in each coefficient. Otherwise 'false' will be returned. */ 
  bool operator==(const CV2D&) const;

  /** Compares to vectors.
    * Same as above. Only the result is negated.
    * @see operator==() */
  bool operator!=(const CV2D&) const;

  /** Adds another vector to this vector. */
  CV2D& operator+=(const CV2D&);

  /** Subtracts another vector from this vector. */
  CV2D& operator-=(const CV2D&);

  /** Multiplies a vector with a scalar. */
  CV2D& operator*=(double);

  /** Multiplies a vector with a scalar. */
  CV2D& operator/=(double);

  /** Adds two vectors. The sum will be returned. */
  CV2D operator+(const CV2D&) const;

  /** Subtracts two vectors. The difference will be returned. */
  CV2D operator-(const CV2D&) const;

  /** Negates the vector. */
  CV2D operator-(void) const;

  /** Scalar multiplication of two vectors. The result will be returned. */
  double operator*(const CV2D&) const;

  /** Multiplication of a vector with a scalar. */
  CV2D operator*(double) const;

  /** Division of a vector with a scalar. */
  CV2D operator/(double) const;

  /** Returns the i-th coefficient or the vector.
    * The index goes from 0 to 3, so 0 stands for the
    * x-coordinate, 1 for the y-coordinate and so on. */
  double& operator[](int i) { return m_ard[i]; };

  /** Same as above but for constant vectors. */
  double operator[](int i) const { return m_ard[i]; };

  /** Multiplication of a scalar with a vector. */
  friend CV2D operator*(double, const CV2D&); 



  /////////////
  // METHODS //
  /////////////

  /** Returns the value of the minimal point component. */
  double getMinComponent(void) const    { return m_ard[getMinComponentCoord()]; };
 
  /** Returns the value of the minimal point component. */
  double getAbsMinComponent(void) const { return m_ard[getAbsMinComponentCoord()]; };
 
  /** Returns the value of the maximal point component. */
  double getMaxComponent(void) const    { return m_ard[getMaxComponentCoord()]; };
 
  /** Returns the value of the maximal point component. */
  double getAbsMaxComponent(void) const { return m_ard[getAbsMaxComponentCoord()]; };
 
  /** Returns the coordinate index of the minial point component. */
  int getMinComponentCoord(void) const;

  /** Returns the coordinate index of the minial point component (using fabs). */
  int getAbsMinComponentCoord(void) const;

  /** Returns the coordinate index of the maximum point component. */
  int getMaxComponentCoord(void) const;

  /** Returns the coordinate index of the maximum point component (using fabs). */
  int getAbsMaxComponentCoord(void) const;

  /** Returns the x-coordinate of the vector. */
  double getX(void) const { return m_ard[0]; };
  
  /** Returns the y-coordinate of the vector. */
  double getY(void) const { return m_ard[1]; };
  
  /** Sets the x-coordinate of the vector to 'x'. */
  void setX(double rdX)   { m_ard[0] = rdX; };

  /** Sets the y-coordinate of the vector to 'y'. */
  void setY(double rdY)   { m_ard[1] = rdY; };

  /** Set the value of the vector.
    * The value of the vector will be (rdX, rdY, rdZ, rdW). */
  void setCoord(double rdX, double rdY) { m_ard[0] = rdX; 
                                          m_ard[1] = rdY; 
                                          return; };

  /** Returns the euclidian norm of the vector. */
  double getNorm(void) const { return sqrt(m_ard[0]*m_ard[0] + m_ard[1]*m_ard[1]); };

  /** Normalizes the vector. */
  void normalize(void);

  /** Returns the normalized vector. */
  CV2D getNormalized(void) const;

  /** Prints a vector to the standard output. */ 
  void print(void) const;
  
  /** Same as above. But more useful for streams. */
  friend ostream& operator<<(ostream&, const CV2D&); 

  /** Reads a vector from the given stream. */
  friend istream& operator>>(istream&, CV2D&); 


protected:
  double m_ard[2];

};



// Function   : operator=
// Parameters : const CP2D& p1
// Purpose    : assign another point to this point
// Comments   : 
inline const CV2D& CV2D::operator=(const CV2D& v)
/*******************************************************************/
{
  m_ard[0] = v[0];
  m_ard[1] = v[1];
  return *this;
}

#endif // _CV2D_H_
