//////////////////////////////////////////////////////////////////////////
//
//  Filename    : QLightSourceEditorImpl.cpp
//  Class       : QLightSourceEditorImpl
//
//  Author(s)   : Michael Meissner
// 
//  Description : GUI for editing light sources
//  Purpose     : 
//
//  Date        : September 2001
//
//  Updates     :    who    | date   | what
//                ----------+--------+-------------------------------------
//                          |        |
//
////////////////////////////////////////////////////////////////////////////


// QT
/////////
#include <qbuttongroup.h>
#include <qlineedit.h>
#include <qradiobutton.h>
#include <qcheckbox.h>


// QGLViewer
//////////////
#include <QMyListBox.h>
#include <QLightSourceEditorImpl.h>


// Own
////////


// Forward declarations
////////////////////////////


// Function  : QLightSourceEditorImpl
// Parameters:
// Purpose   : constructor
// Comments  : See docu for details.
QLightSourceEditorImpl::QLightSourceEditorImpl(QGLViewer *viewer, 
                                               QWidget* parent, 
                                               const char* name, 
                                               bool modal, 
                                               WFlags fl)
                       :QLightSourceEditor(parent, name, modal, fl)
//=============================================================
{
   m_pMyListBox->setSelectionMode(QListBox::Extended);                                                                                                                           
   m_pViewer       = viewer;
   m_pWorkingList  = new CList<CLightSource>;
   m_pOriginalList = new CList<CLightSource>;

   connect(m_pMyListBox, SIGNAL(selectionChanged()), this, SLOT(sltSelectionChanged()));
   updateGUI();
}



// Function  : ~QLightSourceEditorImpl
// Parameters:
// Purpose   : destructor
// Comments  : See docu for details.
QLightSourceEditorImpl::~QLightSourceEditorImpl()
//=============================================================
{
   // delete the dynamically generated lists
   ///////////////////////////////////////////
   if (m_pWorkingList != NULL) {
      m_pWorkingList->clear(1);
      delete m_pWorkingList;
   }
   if (m_pOriginalList != NULL) {
      m_pOriginalList->clear(1);
      delete m_pOriginalList;
   }
}



// Function  : setLightList
// Parameters: CList<CLightSource> &
// Purpose   : initialize the list of light source
// Comments  : See docu for details.
void QLightSourceEditorImpl::setLightList(CList<CLightSource> &list)
//=============================================================
{
   // delete items in QMyListBox
   ///////////////////////////////
   m_pMyListBox->clear();

   // delete the dynamically generated lists
   ///////////////////////////////////////////
   if (m_pOriginalList != NULL) {
      m_pOriginalList->clear(1);
      delete m_pOriginalList;
   }
   if (m_pWorkingList != NULL) {
      m_pWorkingList->clear(1);
      delete m_pWorkingList;
   }

   // copy the list
   //////////////////
   m_pOriginalList = list.getFullDuplicate();
   m_pWorkingList  = list.getFullDuplicate();

   // insert new list items in QMyListBox
   ////////////////////////////////////////
   QString string;
   if (m_pWorkingList) {
      for (int nIndex=0; nIndex<m_pWorkingList->getNumObjects(); nIndex++) {
         string = "Light source " + nIndex;
         m_pMyListBox->insertStringList(string);
      }
   }
   if (m_pWorkingList->getNumObjects() > 0)
      m_pMyListBox->setSelected(0, TRUE);

   // update GUI
   ///////////////
   updateGUI();

   return;
}



// Function  : sltTypeChanged
// Parameters:
// Purpose   : changes the type of light source
// Comments  : See docu for details.
void QLightSourceEditorImpl::sltTypeChanged(int)
//=============================================================
{
   int i;
   for (i=0; i<m_pWorkingList->getNumObjects(); i++) {
      if (m_pMyListBox->isSelected(i)) {
         if (m_pRadioButton0->isChecked()) {
            (*m_pWorkingList)[i].setType(CLightSource::POINT);
            //cout << "Setting type to CLightSource::POINT" << endl;
         }
         if (m_pRadioButton1->isChecked()) {
            (*m_pWorkingList)[i].setType(CLightSource::DIRECTIONAL);
            //cout << "Setting type to CLightSource::DIRECTIONAL" << endl;
         }
         if (m_pRadioButton2->isChecked()) {
            (*m_pWorkingList)[i].setType(CLightSource::CONE);
            //cout << "Setting type to CLightSource::CONE" << endl;
         }
         if (m_pRadioButton3->isChecked()) {
            (*m_pWorkingList)[i].setType(CLightSource::CONE_DIFFUSE);
            //cout << "Setting type to CLightSource::CONE_DIFFUSE" << endl;
         }
      }
   }

   updateGUI();
}



// Function  : sltStatusChanged
// Parameters:
// Purpose   : changes the =state of the light source
// Comments  : See docu for details.
void QLightSourceEditorImpl::sltStatusChanged(int)
//=============================================================
{
   int i;
   for (i=0; i<m_pWorkingList->getNumObjects(); i++) {
      if (m_pMyListBox->isSelected(i)) {
         if (m_pCheckBoxStatus->isChecked())
            (*m_pWorkingList)[i].setStatus(CLightSource::ON);
         else
            (*m_pWorkingList)[i].setStatus(CLightSource::OFF);
      }
   }

   updateGUI();
}



void QLightSourceEditorImpl::sltIntensityChanged()
//=============================================================
{
   int i;
   for (i=0; i<m_pWorkingList->getNumObjects(); i++)
      if (m_pMyListBox->isSelected(i))
         (*m_pWorkingList)[i].setIntensity(m_pLineEditIntensity->text().toFloat());

   updateGUI();
}



void QLightSourceEditorImpl::sltRedChanged()
//=============================================================
{
   int i;
   for (i=0; i<m_pWorkingList->getNumObjects(); i++) {
      if (m_pMyListBox->isSelected(i)) {
         float rfRed   = m_pLineEditRed->text().toFloat()/255.0;
         float rfGreen = m_pLineEditGreen->text().toFloat()/255.0;
         float rfBlue  = m_pLineEditBlue->text().toFloat()/255.0;
         if (rfRed < 0.0)
            rfRed = 0.0;
         if (rfRed > 1.0)
            rfRed = 1.0;
         if (rfGreen < 0.0)
            rfGreen = 0.0;
         if (rfGreen > 1.0)
            rfGreen = 1.0;
         if (rfBlue < 0.0)
            rfBlue = 0.0;
         if (rfBlue > 1.0)
            rfBlue = 1.0;
         (*m_pWorkingList)[i].setColor(rfRed, rfGreen, rfBlue);
      }
   }

   updateGUI();
}



void QLightSourceEditorImpl::sltGreenChanged()
//=============================================================
{
   int i;
   for (i=0; i<m_pWorkingList->getNumObjects(); i++) {
      if (m_pMyListBox->isSelected(i)) {
         float rfRed   = m_pLineEditRed->text().toFloat()/255.0;
         float rfGreen = m_pLineEditGreen->text().toFloat()/255.0;
         float rfBlue  = m_pLineEditBlue->text().toFloat()/255.0;
         if (rfRed < 0.0)
            rfRed = 0.0;
         if (rfRed > 1.0)
            rfRed = 1.0;
         if (rfGreen < 0.0)
            rfGreen = 0.0;
         if (rfGreen > 1.0)
            rfGreen = 1.0;
         if (rfBlue < 0.0)
            rfBlue = 0.0;
         if (rfBlue > 1.0)
            rfBlue = 1.0;
         (*m_pWorkingList)[i].setColor(rfRed, rfGreen, rfBlue);
      }
   }

   updateGUI();
}



void QLightSourceEditorImpl::sltBlueChanged()
//=============================================================
{
   int i;
   for (i=0; i<m_pWorkingList->getNumObjects(); i++) {
      if (m_pMyListBox->isSelected(i)) {
         float rfRed   = m_pLineEditRed->text().toFloat()/255.0;
         float rfGreen = m_pLineEditGreen->text().toFloat()/255.0;
         float rfBlue  = m_pLineEditBlue->text().toFloat()/255.0;
         if (rfRed < 0.0)
            rfRed = 0.0;
         if (rfRed > 1.0)
            rfRed = 1.0;
         if (rfGreen < 0.0)
            rfGreen = 0.0;
         if (rfGreen > 1.0)
            rfGreen = 1.0;
         if (rfBlue < 0.0)
            rfBlue = 0.0;
         if (rfBlue > 1.0)
            rfBlue = 1.0;
         (*m_pWorkingList)[i].setColor(rfRed, rfGreen, rfBlue);
      }
   }

   updateGUI();
}



void QLightSourceEditorImpl::sltPosXchanged()
//=============================================================
{
   int i;
   for (i=0; i<m_pWorkingList->getNumObjects(); i++) {
      if (m_pMyListBox->isSelected(i)) {
         CP3D Position = (*m_pWorkingList)[i].getPosition();
         Position[0] = m_pLineEditPosX->text().toFloat();
         (*m_pWorkingList)[i].setPosition(Position);
      }
   }

   updateGUI();
}



void QLightSourceEditorImpl::sltPosYchanged()
//=============================================================
{
   int i;
   for (i=0; i<m_pWorkingList->getNumObjects(); i++) {
      if (m_pMyListBox->isSelected(i)) {
         CP3D Position = (*m_pWorkingList)[i].getPosition();
         Position[1] = m_pLineEditPosY->text().toFloat();
         (*m_pWorkingList)[i].setPosition(Position);
      }
   }

   updateGUI();
}



void QLightSourceEditorImpl::sltPosZchanged()
//=============================================================
{
   int i;
   for (i=0; i<m_pWorkingList->getNumObjects(); i++) {
      if (m_pMyListBox->isSelected(i)) {
         CP3D Position = (*m_pWorkingList)[i].getPosition();
         Position[2] = m_pLineEditPosZ->text().toFloat();
         (*m_pWorkingList)[i].setPosition(Position);
      }
   }

   updateGUI();
}



void QLightSourceEditorImpl::sltDirXchanged()
//=============================================================
{
   int i;
   for (i=0; i<m_pWorkingList->getNumObjects(); i++) {
      if (m_pMyListBox->isSelected(i)) {
         CV3D Direction = (*m_pWorkingList)[i].getDirection();
         Direction[0] = m_pLineEditDirX->text().toFloat();
         (*m_pWorkingList)[i].setDirection(Direction);
      }
   }

   updateGUI();
}



void QLightSourceEditorImpl::sltDirYchanged()
//=============================================================
{
   int i;
   for (i=0; i<m_pWorkingList->getNumObjects(); i++) {
      if (m_pMyListBox->isSelected(i)) {
         CV3D Direction = (*m_pWorkingList)[i].getDirection();
         Direction[1] = m_pLineEditDirY->text().toFloat();
         (*m_pWorkingList)[i].setDirection(Direction);
      }
   }

   updateGUI();
}



void QLightSourceEditorImpl::sltDirZchanged()
//=============================================================
{
   int i;
   for (i=0; i<m_pWorkingList->getNumObjects(); i++) {
      if (m_pMyListBox->isSelected(i)) {
         CV3D Direction = (*m_pWorkingList)[i].getDirection();
         Direction[2] = m_pLineEditDirZ->text().toFloat();
         (*m_pWorkingList)[i].setDirection(Direction);
      }
   }

   updateGUI();
}



void QLightSourceEditorImpl::sltAngleChanged()
//=============================================================
{
   int i;
   for (i=0; i<m_pWorkingList->getNumObjects(); i++) 
      if (m_pMyListBox->isSelected(i)) {
         if (m_pLineEditAngle->text().toFloat() >= 0.0 && m_pLineEditAngle->text().toFloat() <= 360.0)
            (*m_pWorkingList)[i].setAngle(m_pLineEditAngle->text().toFloat());
         if (m_pLineEditAngle->text().toFloat() > 360.0)
            (*m_pWorkingList)[i].setAngle(360.0);
         if (m_pLineEditAngle->text().toFloat() < 0.0)
            (*m_pWorkingList)[i].setAngle(0.0);
      }

   updateGUI();
}



void QLightSourceEditorImpl::sltExponentChanged()
//=============================================================
{
   int i;
   for (i=0; i<m_pWorkingList->getNumObjects(); i++)
      if (m_pMyListBox->isSelected(i))
         (*m_pWorkingList)[i].setExponent(m_pLineEditExponent->text().toFloat());

   updateGUI();
}



void QLightSourceEditorImpl::sltConstantAttenuationChanged()
//=============================================================
{
   int i;
   for (i=0; i<m_pWorkingList->getNumObjects(); i++) 
      if (m_pMyListBox->isSelected(i))
         (*m_pWorkingList)[i].setConstantAttenuation(m_pLineEditConstantAttenuation->text().toFloat());

   updateGUI();
}



void QLightSourceEditorImpl::sltLinearAttenuationChanged()
//=============================================================
{
   int i;
   for (i=0; i<m_pWorkingList->getNumObjects(); i++)
      if (m_pMyListBox->isSelected(i))
         (*m_pWorkingList)[i].setLinearAttenuation(m_pLineEditLinearAttenuation->text().toFloat());

   updateGUI();
}



void QLightSourceEditorImpl::sltQuadraticAttenuationChanged()
//=============================================================
{
   int i;
   for (i=0; i<m_pWorkingList->getNumObjects(); i++)
      if (m_pMyListBox->isSelected(i))
         (*m_pWorkingList)[i].setQuadraticAttenuation(m_pLineEditQuadraticAttenuation->text().toFloat());

   updateGUI();
}



// Name      : eventFilter
// Parameters: QObject *o, QEvent *e
// Purpose   : handles events which come from the viewer.
// Comments  : See docu for details.
bool QLightSourceEditorImpl::eventFilter(QObject *o, QEvent *e)
//=============================================================
{
   if (e->type() == QEvent::KeyPress) {
      QKeyEvent *k = (QKeyEvent*)e;
      if (k->key() == Key_L) {
         setLight(m_pViewer->getCamera());
         return true;
      }
   }
   return QWidget::eventFilter(o, e);    // standard event processing
}



// Name      : setLight
// Parameters: CCamera &camera
// Purpose   : inserts a new light into the list
// Comments  : See docu for details.
void QLightSourceEditorImpl::setLight(CCamera &camera)
//=============================================================
{
   CLightSource *pLight = new CLightSource(CLightSource::DIRECTIONAL);
   pLight->setIntensity(1.0);
   pLight->setStatus(CLightSource::ON);
   pLight->setColor(1.0, 1.0, 1.0);
   pLight->setPosition(camera.getEyePos());
   pLight->setDirection(camera.getViewDir());
   pLight->setAngle(360.0);
   pLight->setExponent(1.0);
   pLight->setConstantAttenuation(1.0);
   pLight->setLinearAttenuation(0.0);
   pLight->setQuadraticAttenuation(0.0);
   m_pWorkingList->insertAsLast(pLight);

   QString string = "new Light source ";
   m_pMyListBox->insertStringList(string);

   int i;
   for (i=0; i<m_pWorkingList->getNumObjects(); i++)
      m_pMyListBox->setSelected(i, false);
   m_pMyListBox->setSelected(m_pWorkingList->getNumObjects()-1, true);

   updateGUI();
}



// Name      : sltDeleteLight
// Parameters: 
// Purpose   : delete selected light(s)
// Comments  : See docu for details.
void QLightSourceEditorImpl::sltDeleteLight()
//=============================================================
{
   int i, nIndex;
   for (i=m_pWorkingList->getNumObjects(); i>=0; i--) {
      if (m_pMyListBox->isSelected(i)) {
         delete (*m_pWorkingList)(i)->getObject();
         m_pWorkingList->remove((*m_pWorkingList)(i));
         m_pMyListBox->removeItem(i);
         nIndex = i;
      }
   }

   updateGUI();
}



// Name      : sltSelectionChanged
// Parameters: 
// Purpose   : is called whenever highlighted/selected items in MyListBox change
// Comments  : See docu for details.
void QLightSourceEditorImpl::sltSelectionChanged()
//=============================================================
{
   updateGUI();
}



// Name      : updateGUI
// Parameters: 
// Purpose   : 
// Comments  : See docu for details.
void QLightSourceEditorImpl::updateGUI()
//=============================================================
{
   bool fAnythingSelected = false;

   if (m_pWorkingList) {
      if (m_pWorkingList->getNumObjects() > 0) {
         int i;
         for (i=m_pWorkingList->getNumObjects(); i>=0; i--) {
            if (m_pMyListBox->isSelected(i)) {
               fAnythingSelected = true;
               if ((*m_pWorkingList)[i].getType() == CLightSource::POINT) {
                  m_pButtonGroupType->setEnabled(TRUE);
                  m_pRadioButton0->setChecked(TRUE);
                  m_pButtonGroupIntensity->setEnabled(TRUE);
                  m_pFrame->setEnabled(TRUE);
                  m_pButtonGroupColor->setEnabled(TRUE);
                  m_pButtonGroupPos->setEnabled(TRUE);
                  m_pButtonGroupAttenuation->setEnabled(TRUE);
               }
               if ((*m_pWorkingList)[i].getType() == CLightSource::DIRECTIONAL) {
                  m_pButtonGroupType->setEnabled(TRUE);
                  m_pRadioButton1->setChecked(TRUE);
                  m_pButtonGroupIntensity->setEnabled(TRUE);
                  m_pFrame->setEnabled(TRUE);
                  m_pButtonGroupColor->setEnabled(TRUE);
                  m_pButtonGroupPos->setEnabled(TRUE);
                  m_pButtonGroupDir->setEnabled(TRUE);
                  m_pButtonGroupAttenuation->setEnabled(TRUE);
               }
               if ((*m_pWorkingList)[i].getType() == CLightSource::CONE) {
                  m_pButtonGroupType->setEnabled(TRUE);
                  m_pRadioButton2->setChecked(TRUE);
                  m_pButtonGroupIntensity->setEnabled(TRUE);
                  m_pFrame->setEnabled(TRUE);
                  m_pButtonGroupColor->setEnabled(TRUE);
                  m_pButtonGroupPos->setEnabled(TRUE);
                  m_pButtonGroupDir->setEnabled(TRUE);
                  m_pButtonGroupAngle->setEnabled(TRUE);
                  m_pButtonGroupAttenuation->setEnabled(TRUE);
               }
               if ((*m_pWorkingList)[i].getType() == CLightSource::CONE_DIFFUSE) {
                  m_pButtonGroupType->setEnabled(TRUE);
                  m_pRadioButton3->setChecked(TRUE);
                  m_pButtonGroupIntensity->setEnabled(TRUE);
                  m_pFrame->setEnabled(TRUE);
                  m_pButtonGroupColor->setEnabled(TRUE);
                  m_pButtonGroupPos->setEnabled(TRUE);
                  m_pButtonGroupDir->setEnabled(TRUE);
                  m_pButtonGroupAngle->setEnabled(TRUE);
                  m_pButtonGroupExponent->setEnabled(TRUE);
                  m_pButtonGroupAttenuation->setEnabled(TRUE);
               }

               // set intensity
               QString string;
               string.setNum((*m_pWorkingList)[i].getIntensity(), 'g', 4);
               m_pLineEditIntensity->setText(string);

               // active light source?
               m_pCheckBoxStatus->setEnabled(TRUE);
               if ((*m_pWorkingList)[i].getStatus() == CLightSource::ON)
                  m_pCheckBoxStatus->setChecked(TRUE);
               else
                  m_pCheckBoxStatus->setChecked(FALSE);

               // set color
               const float *prfColor = (*m_pWorkingList)[i].getColor();
               string.setNum(255 * prfColor[0], 'g', 4);
               m_pLineEditRed->setText(string);
               string.setNum(255 * prfColor[1], 'g', 4);
               m_pLineEditGreen->setText(string);
               string.setNum(255 * prfColor[2], 'g', 4);
               m_pLineEditBlue->setText(string);

               // set position
               CP3D Position = (*m_pWorkingList)[i].getPosition();
               string.setNum(Position[0], 'g', 4);
               m_pLineEditPosX->setText(string);
               string.setNum(Position[1], 'g', 4);
               m_pLineEditPosY->setText(string);
               string.setNum(Position[2], 'g', 4);
               m_pLineEditPosZ->setText(string);

               // set direction
               CV3D Direction = (*m_pWorkingList)[i].getDirection();
               string.setNum(Direction[0], 'g', 4);
               m_pLineEditDirX->setText(string);
               string.setNum(Direction[1], 'g', 4);
               m_pLineEditDirY->setText(string);
               string.setNum(Direction[2], 'g', 4);
               m_pLineEditDirZ->setText(string);

               // set angle
               float rfTmp = (*m_pWorkingList)[i].getAngle();
               string.setNum(rfTmp, 'g', 4);
               m_pLineEditAngle->setText(string);

               // set exponent
               rfTmp = (*m_pWorkingList)[i].getExponent();
               string.setNum(rfTmp, 'g', 4);
               m_pLineEditExponent->setText(string);

               // set constant attenutation 
               rfTmp = (*m_pWorkingList)[i].getConstantAttenuation();
               string.setNum(rfTmp, 'g', 4);
               m_pLineEditConstantAttenuation->setText(string);

               // set constant attenutation 
               rfTmp = (*m_pWorkingList)[i].getLinearAttenuation();
               string.setNum(rfTmp, 'g', 4);
               m_pLineEditLinearAttenuation->setText(string);

               // set constant attenutation 
               rfTmp = (*m_pWorkingList)[i].getQuadraticAttenuation();
               string.setNum(rfTmp, 'g', 4);
               m_pLineEditQuadraticAttenuation->setText(string);

               break;
            }
         }
      }
   }
   if (fAnythingSelected == false) {
      m_pButtonGroupType->setEnabled(FALSE);
      m_pRadioButton0->setChecked(FALSE);
      m_pRadioButton1->setChecked(FALSE);
      m_pRadioButton2->setChecked(FALSE);
      m_pRadioButton3->setChecked(FALSE);
      m_pCheckBoxStatus->setEnabled(FALSE);
      m_pFrame->setEnabled(FALSE);
      m_pButtonGroupColor->setEnabled(FALSE);
      m_pButtonGroupPos->setEnabled(FALSE);
      m_pButtonGroupDir->setEnabled(FALSE);
      m_pButtonGroupAngle->setEnabled(FALSE);
      m_pButtonGroupExponent->setEnabled(FALSE);
      m_pButtonGroupAttenuation->setEnabled(FALSE);
   }

   return;
}



// Function  : sltButtonCancel
// Parameters: 
// Purpose   : sltButtonCancel changes
// Comments  : See docu for details.
void QLightSourceEditorImpl::sltButtonCancel()
//=============================================================
{
   // delete items in QMyListBox
   ///////////////////////////////
   m_pMyListBox->clear();

   // delete the dynamic working list
   ////////////////////////////////////
   if (m_pWorkingList != NULL) {
      m_pWorkingList->clear(1);
      delete m_pWorkingList;
   }

   // copy the list (restore)
   ////////////////////////////
   m_pWorkingList = m_pOriginalList->getFullDuplicate();

   // insert new list items in QMyListBox
   ////////////////////////////////////////
   QString string;
   if (m_pWorkingList) {
      for (int nIndex=0; nIndex<m_pWorkingList->getNumObjects(); nIndex++) {
         string = "Light source " + nIndex;
         m_pMyListBox->insertStringList(string);
      }
   }
   if (m_pWorkingList->getNumObjects() > 0)
      m_pMyListBox->setSelected(0, TRUE);

   // update GUI
   ///////////////
   updateGUI();

   hide();

   return;
}



// Function  : sltButtonApply
// Parameters: 
// Purpose   : 
// Comments  : See docu for details.
void QLightSourceEditorImpl::sltButtonApply()
//=============================================================
{
   // delete the original list 
   /////////////////////////////
   if (m_pOriginalList != NULL) {
      m_pOriginalList->clear(1);
      delete m_pOriginalList;
   }

   // copy the list
   //////////////////
   m_pOriginalList = m_pWorkingList->getFullDuplicate();

   // emit reference to new list to the outside world
   ////////////////////////////////////////////////////
   emit(sigLightSourceList(*m_pWorkingList));

   return;
}



// Function  : sltButtonOk
// Parameters: CList<CLightSource> &
// Purpose   : initialize the list of light source
// Comments  : See docu for details.
void QLightSourceEditorImpl::sltButtonOk()
//=============================================================
{
   sltButtonApply();

   hide();

   return;
}
