/***************************************************************************
                          arrow.cpp  -  description
                             -------------------
    begin                : Sun Nov 23 2003
    copyright            : (C) 2003 by Michael Margraf
    email                : michael.margraf@alumni.tu-berlin.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "arrow.h"
#include "arrowdialog.h"

#include <qlineedit.h>
#include <qpushbutton.h>

#include <math.h>


Arrow::Arrow()
{
  Name = "Arrow ";
  isSelected = false;
  Pen = QPen(QColor());
  cx = cy = 0;
  x1 = x2 = 0;
  y1 = y2 = 0;

  Height = 20.0;
  Width  =  8.0;
  beta   = atan2(double(Width), double(Height));
  Length = sqrt(Width*Width + Height*Height);
}

Arrow::~Arrow()
{
}

// --------------------------------------------------------------------------
void Arrow::paint(ViewPainter *p)
{
  if(isSelected) {
    p->Painter->setPen(QPen(QPen::darkGray,Pen.width()+5));
    p->drawLine(cx, cy, cx+x2, cy+y2);
    p->drawLine(cx+x2, cy+y2, cx+xp1, cy+yp1);
    p->drawLine(cx+x2, cy+y2, cx+xp2, cy+yp2);
    p->Painter->setPen(QPen(QPen::white, Pen.width(), Pen.style()));
    p->drawLine(cx, cy, cx+x2, cy+y2);
    p->drawLine(cx+x2, cy+y2, cx+xp1, cy+yp1);
    p->drawLine(cx+x2, cy+y2, cx+xp2, cy+yp2);

    p->Painter->setPen(QPen(QPen::darkRed,2));
    p->drawResizeRect(cx, cy);  // markers for changing the size
    p->drawResizeRect(cx+x2, cy+y2);
    return;
  }
  p->Painter->setPen(Pen);
  p->drawLine(cx, cy, cx+x2, cy+y2);
  p->drawLine(cx+x2, cy+y2, cx+xp1, cy+yp1);
  p->drawLine(cx+x2, cy+y2, cx+xp2, cy+yp2);
}

// --------------------------------------------------------------------------
void Arrow::paintScheme(QPainter *p)
{
  p->drawLine(cx, cy, cx+x2, cy+y2);
  p->drawLine(cx+x2, cy+y2, cx+xp1, cy+yp1);
  p->drawLine(cx+x2, cy+y2, cx+xp2, cy+yp2);
}

// --------------------------------------------------------------------------
void Arrow::getCenter(int& x, int &y)
{
  x = cx+(x2>>1);
  y = cy+(y2>>1);
}

// --------------------------------------------------------------------------
// Sets the center of the painting to x/y.
void Arrow::setCenter(int x, int y, bool relative)
{
  if(relative) { cx += x;  cy += y; }
  else { cx = x-(x2>>1);  cy = y-(y2>>1); }
}

// --------------------------------------------------------------------------
Painting* Arrow::newOne()
{
  return new Arrow();
}

// --------------------------------------------------------------------------
bool Arrow::load(const QString& s)
{
  bool ok;

  QString n;
  n  = s.section(' ',1,1);    // cx
  cx = n.toInt(&ok);
  if(!ok) return false;

  n  = s.section(' ',2,2);    // cy
  cy = n.toInt(&ok);
  if(!ok) return false;

  n  = s.section(' ',3,3);    // x2
  x2 = n.toInt(&ok);
  if(!ok) return false;

  n  = s.section(' ',4,4);    // y2
  y2 = n.toInt(&ok);
  if(!ok) return false;

  n  = s.section(' ',5,5);    // height
  Height = n.toDouble(&ok);
  if(!ok) return false;

  n  = s.section(' ',6,6);    // width
  Width = n.toDouble(&ok);
  if(!ok) return false;

  n  = s.section(' ',7,7);    // color
  QColor co;
  co.setNamedColor(n);
  Pen.setColor(co);
  if(!Pen.color().isValid()) return false;

  n  = s.section(' ',8,8);    // thickness
  Pen.setWidth(n.toInt(&ok));
  if(!ok) return false;

  n  = s.section(' ',9,9);    // line style
  Pen.setStyle((Qt::PenStyle)n.toInt(&ok));
  if(!ok) return false;

  beta   = atan2(double(Width), double(Height));
  Length = sqrt(Width*Width + Height*Height);
  calcArrowHead();
  return true;
}

// --------------------------------------------------------------------------
QString Arrow::save()
{
  QString s = Name+QString::number(cx)+" "+QString::number(cy)+" ";
  s += QString::number(x2)+" "+QString::number(y2)+" ";
  s += QString::number(int(Height))+" "+QString::number(int(Width))+" ";
  s += Pen.color().name()+" "+QString::number(Pen.width())+" ";
  s += QString::number(Pen.style());
  return s;
}

// --------------------------------------------------------------------------
// Checks if the resize area was clicked.
bool Arrow::ResizeTouched(int x, int y)
{
  if(x < cx+5) if(x > cx-5) if(y < cy+5) if(y > cy-5) {
    State = 1;
    return true;
  }

  if(x < cx+x2+5) if(x > cx+x2-5) if(y < cy+y2+5) if(y > cy+y2-5) {
    State = 2;
    return true;
  }

  State = 0;
  return false;
}

// --------------------------------------------------------------------------
// Mouse move action during resize.
void Arrow::MouseResizeMoving(int x, int y, QPainter *p)
{
  paintScheme(p);  // erase old painting
  if(State == 1) { x2 += cx-x; y2 += cy-y; cx = x; cy = y; } // moving shaft
  else { x2 = x-cx;  y2 = y-cy; }  // moving head

  calcArrowHead();
  paintScheme(p);  // paint new painting
}

// --------------------------------------------------------------------------
void Arrow::calcArrowHead()
{
  double phi  = atan2(double(y2), double(x2));

  double w = beta+phi;
  xp1 = x2-int(Length*cos(w));
  yp1 = y2-int(Length*sin(w));

  w = phi-beta;
  xp2 = x2-int(Length*cos(w));
  yp2 = y2-int(Length*sin(w));
}

// --------------------------------------------------------------------------
// fx/fy are the precise coordinates, gx/gy are the coordinates set on grid.
// x/y are coordinates without scaling.
void Arrow::MouseMoving(
	QPainter *paintScale, int, int, int gx, int gy,
	QPainter *p, int x, int y, bool drawn)
{
  if(State > 0) {
    if(State > 1) {
      calcArrowHead();
      paintScheme(paintScale);  // erase old painting
    }
    State++;
    x2 = gx-cx;
    y2 = gy-cy;
    calcArrowHead();
    paintScheme(paintScale);  // paint new painting
  }
  else { cx = gx; cy = gy; }


  p->setPen(Qt::SolidLine);
  if(drawn) {
    p->drawLine(x1+25, y1, x1+13, y1+12);  // erase old cursor symbol
    p->drawLine(x1+18, y1+2, x1+25, y1);
    p->drawLine(x1+23, y1+7, x1+25, y1);
  }
  x1 = x;
  y1 = y;
  p->drawLine(x1+25, y1, x1+13, y1+12);  // paint new cursor symbol
  p->drawLine(x1+18, y1+2, x1+25, y1);
  p->drawLine(x1+23, y1+7, x1+25, y1);
}

// --------------------------------------------------------------------------
bool Arrow::MousePressing()
{
  State++;
  if(State > 2) {
    x1 = y1 = 0;
    State = 0;

    calcArrowHead();
    return true;    // painting is ready
  }
  return false;
}

// --------------------------------------------------------------------------
// Checks if the coordinates x/y point to the painting.
// 5 is the precision the user must point onto the painting.
bool Arrow::getSelected(int x, int y)
{
  int A, xn, yn;
  // first check if coordinates match the arrow body
  x  -= cx;
  y  -= cy;

  if(x < -5) { if(x < x2-5) goto Head1; } // is point between x coordinates ?
  else { if(x > 5) if(x > x2+5) goto Head1; }
  if(y < -5) { if(y < y2-5) goto Head1; } // is point between y coordinates ?
  else { if(y > 5) if(y > y2+5) goto Head1; }

  A  = x2*y - x*y2;         // calculate the rectangle area spanned
  A *= A;                   // avoid the need for square root
  A -= 25*(x2*x2 + y2*y2);  // substract selectable area

  if(A <= 0)  return true;     // lies x/y onto the graph line ?

Head1:    // check if coordinates match the first arrow head line
  xn = xp1-x2;  x  -= x2;
  yn = yp1-y2;  y  -= y2;

  if(x < -5) { if(x < xn-5) goto Head2; } // is point between x coordinates ?
  else { if(x > 5) if(x > xn+5) goto Head2; }
  if(y < -5) { if(y < yn-5) goto Head2; } // is point between y coordinates ?
  else { if(y > 5) if(y > yn+5) goto Head2; }

  A  = xn*y - x*yn;         // calculate the rectangle area spanned
  A *= A;                   // avoid the need for square root
  A -= 25*(xn*xn + yn*yn);  // substract selectable area

  if(A <= 0)  return true;     // lies x/y onto the graph line ?

Head2:    // check if coordinates match the second arrow head line
  xn = xp2-x2;
  yn = yp2-y2;

  if(x < -5) { if(x < xn-5) return false; }// is point between x coordinates ?
  else { if(x > 5) if(x > xn+5) return false; }
  if(y < -5) { if(y < yn-5) return false; }// is point between y coordinates ?
  else { if(y > 5) if(y > yn+5) return false; }

  A  = xn*y - x*yn;         // calculate the rectangle area spanned
  A *= A;                   // avoid the need for square root
  A -= 25*(xn*xn + yn*yn);  // substract selectable area

  if(A <= 0)  return true;     // lies x/y onto the graph line ?

  return false;
}

// --------------------------------------------------------------------------
void Arrow::Bounding(int& _x1, int& _y1, int& _x2, int& _y2)
{
  if(x2 < 0) { _x1 = cx+x2; _x2 = cx; }
  else { _x1 = cx; _x2 = cx+x2; }

  if(y2 < 0) { _y1 = cy+y2; _y2 = cy; }
  else { _y1 = cy; _y2 = cy+y2; }
}

// --------------------------------------------------------------------------
// Rotates around the center.
void Arrow::rotate()
{
  cx += (x2>>1) - (y2>>1);
  cy += (x2>>1) + (y2>>1);

  int tmp = x2;
  x2  =  y2;
  y2  = -tmp;

  tmp =  xp1;
  xp1 =  yp1;
  yp1 = -tmp;

  tmp =  xp2;
  xp2 =  yp2;
  yp2 = -tmp;
}

// --------------------------------------------------------------------------
// Mirrors about center line.
void Arrow::mirrorX()
{
  yp1 = -yp1;
  yp2 = -yp2;
  cy +=  y2;   // change cy after the other changes !
  y2  = -y2;
}

// --------------------------------------------------------------------------
// Mirrors about center line.
void Arrow::mirrorY()
{
  xp1 = -xp1;
  xp2 = -xp2;
  cx +=  x2;   // change cx after the other changes !
  x2  = -x2;
}

// --------------------------------------------------------------------------
// Calls the property dialog for the painting and changes them accordingly.
// If there were changes, it returns 'true'.
bool Arrow::Dialog()
{
  bool changed = false;

  ArrowDialog *d = new ArrowDialog();
  d->HeadWidth->setText(QString::number(Width));
  d->HeadLength->setText(QString::number(Height));
  d->ColorButt->setPaletteBackgroundColor(Pen.color());
  d->LineWidth->setText(QString::number(Pen.width()));
  d->SetComboBox(Pen.style());

  if(d->exec() == QDialog::Rejected) {
    delete d;
    return false;
  }

  if(Width  != d->HeadWidth->text().toDouble()) {
    Width = d->HeadWidth->text().toDouble();
    changed = true;
  }
  if(Height  != d->HeadLength->text().toDouble()) {
    Height = d->HeadLength->text().toDouble();
    changed = true;
  }
  if(Pen.color() != d->ColorButt->paletteBackgroundColor()) {
    Pen.setColor(d->ColorButt->paletteBackgroundColor());
    changed = true;
  }
  if(Pen.width()  != d->LineWidth->text().toUInt()) {
    Pen.setWidth(d->LineWidth->text().toUInt());
    changed = true;
  }
  if(Pen.style()  != d->LineStyle) {
    Pen.setStyle(d->LineStyle);
    changed = true;
  }

  beta   = atan2(double(Width), double(Height));
  Length = sqrt(Width*Width + Height*Height);
  calcArrowHead();

  delete d;
  return changed;
}
