/* specter
 *
 * userspace logging daemon for the iptables ULOG target
 * of the linux 2.4/2.6 netfilter subsystem.
 *
 * (C) 2000-2003 by Harald Welte <laforge@gnumonks.org>
 * (C) 2004,2005 by Michal Kwiatkowski <ruby@joker.linuxstuff.pl>
 * 
 * Modifications:
 * 	14 Jun 2001 Martin Josefsson <gandalf@wlug.westbo.se>
 * 		- added SIGHUP handler for logfile cycling
 *
 * 	10 Feb 2002 Alessandro Bono <a.bono@libero.it>
 * 		- added support for non-fork mode
 * 		- added support for logging to stdout
 * 
 * 	20 Apr 2004 Nicolas Pougetoux <nicolas.pougetoux@edelweb.fr>
 * 		- added suppurt for seteuid()
 */

/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2
 *  as published by the Free Software Foundation
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <ctype.h>
#include <unistd.h>
#include <getopt.h>
#include <pwd.h>
#include <grp.h>
#include <time.h>
#include <signal.h>
#include <dlfcn.h>
#include <libipulog/libipulog.h>
#include <conffile/conffile.h>
#include <specter/specter.h>


#define SPECTER_VERSION	"1.4-pre2"



/***********************************************************************
 *   DEFAULTS
 ***********************************************************************/
/* Size of the socket recevive memory.  Should be at least the same size as the
 * 'nlbufsiz' module loadtime parameter of ipt_ULOG
 * If you have _big_ in-kernel queues, you may have to increase this number.
 * --qthreshold 100 * 1500 bytes/packet = 150kB  */
#define SPECTER_RMEM_DEFAULT  131071

/* Size of the receive buffer for the netlink socket.  Should be at least of
 * RMEM_DEFAULT size.  */
#define SPECTER_BUFSIZE_DEFAULT  150000

/* default config parameters, if not changed in configfile */
#ifndef SPECTER_LOGFILE_DEFAULT
#define SPECTER_LOGFILE_DEFAULT  "/var/log/specter.log"
#endif

/* where to look for the config file */
#ifndef SPECTER_CONFIGFILE_DEFAULT
#define SPECTER_CONFIGFILE_DEFAULT  "/etc/specter.conf"
#endif

/* how many groups we need */
#ifndef SPECTER_GROUPS_MAX
#define SPECTER_GROUPS_MAX	32
#endif


/***********************************************************************
 *   GLOBAL VARIABLES
 ***********************************************************************/
/* configuration directives of the main program */
static config_entry_t errignore_ce = {
	.next = NULL,
	.key = "errignore",
	.type = CONFIG_TYPE_BOOLEAN,
	.options = CONFIG_OPT_NONE,
	.u = { .value = 0 }
};

static config_entry_t logfile_ce = {
	.next = &errignore_ce,
	.key = "logfile",
	.type = CONFIG_TYPE_STRING,
	.options = CONFIG_OPT_NONE,
	.u = { .string = SPECTER_LOGFILE_DEFAULT }
};

static config_entry_t bufsiz_ce = {
	.next = &logfile_ce,
	.key = "bufsize",
	.type = CONFIG_TYPE_MEM,
	.options = CONFIG_OPT_NONE,
	.u = { .value = SPECTER_BUFSIZE_DEFAULT }
};

static config_entry_t loglevel_ce = {
	.next = &bufsiz_ce,
	.key = "loglevel",
	.type = CONFIG_TYPE_INT,
	.options = CONFIG_OPT_NONE,
	.u = { .value = SPECTER_INFO }
};

static config_entry_t rmem_ce = {
	.next = &loglevel_ce,
	.key = "rmem",
	.type = CONFIG_TYPE_MEM,
	.options = CONFIG_OPT_NONE,
	.u = { .value = SPECTER_RMEM_DEFAULT }
};

static config_entry_t nlgroup_ce = {
	.next = &rmem_ce,
	.key = "nlgroup",
	.type = CONFIG_TYPE_INT,
	.options = CONFIG_OPT_NONE,
	.u = { .value = 1 }
};

static config_entry_t grouping_ce = {
	.next = &nlgroup_ce,
	.key = "grouping",
	.type = CONFIG_TYPE_STRING,
	.options = CONFIG_OPT_NONE,
	.u = { .string = "netlink" }
};

/* should always point to first ce in list above */
static config_entry_t *global_ce = &grouping_ce;

/* the highest number of logical execution groups;
 * it will be recalculated in load_config() */
static int groups_top = SPECTER_GROUPS_MAX;

/* each group mode has its own config entries and options */
struct output_plugin_list {
	struct output_plugin_list *next;
	specter_output_t *plugin;
	config_entry_t *ce;
	void *data;
};

/* input plugins behave the same regardless of group current
 * packet arrived from */
struct input_plugin_list {
	struct input_plugin_list *next;
	specter_input_t *plugin;
	int called; /* was this plugin called for current packet? */
};

/* arrays allocated in main() */
static struct output_plugin_list **output_list;
static struct input_plugin_list **input_list;

/* used during plugin initialization */
static int last_group;

/* our libipulog handle */
static struct ipulog_handle *libulog_h;

/* the receive buffer */
static unsigned char *libulog_buf;

/* linked list of all keys registered by input plugins */
static specter_iret_t *iret_values;

/* used only by load_config_block() and load_config() */
struct plugin_tags {
	char tag[CONFIG_KEY_LEN];
	char path[PATH_MAX];
};

/***********************************************************************
 *   DEBUG FUNCTIONS
 ***********************************************************************/
#ifdef DEBUG

#define DEBUGP(format, args...) do {				\
		fprintf(stderr, "%s:%i:", __FILE__, __LINE__);	\
		fprintf(stderr, format, ## args);		\
	} while (0)

static void dump_ce(config_entry_t *ce)
{
	int ctr;

	if (!ce) {
		printf("ce[0]: EMPTY\n");
	}
	for (ctr = 0; ce; ce = ce->next, ctr++) {
		printf("* ce[%i]: 0x%08x\n"
			"\tkey:%s\n"
			"\ttype:%i\n"
			"\toptions:%i\n"
			"\thit:%i\n"
			"\tvalue:%i\n"
			"\tstring:\"%s\"\n",
			ctr,
			(int)ce,
			ce->key,
			ce->type,
			ce->options,
			ce->hit,
			(ce->type == CONFIG_TYPE_STRING) ? 0 : ce->u.value,
			(ce->type == CONFIG_TYPE_STRING) ? ce->u.string : "");

	}
	fflush(stdout);

	return;
}

static void dump_iret(specter_iret_t *iret)
{
	while (iret) {
		printf("name: \"%s\" type: %i.\n",
			iret->name, iret->type);
		iret = iret->next;
	}
	fflush(stdout);

	return;
}

static void dump_lists(void)
{
	int ctr;
	struct output_plugin_list *opl;
	struct input_plugin_list *ipl;

	for (ctr = 0; ctr < groups_top; ctr++) {
		if (output_list[ctr] == NULL) {
			printf("* output_list[%i] empty.\n", ctr);
			continue;
		}
		printf("* output_list[%i]:\n", ctr);
		for (opl = output_list[ctr]; opl; opl = opl->next) {
			printf("\tplugin: 0x%08x\n"
				"\tce: 0x%08x\n"
				"\tdata: 0x%08x\n"
				"\t\n",
				(int)opl->plugin,
				(int)opl->ce,
				(int)opl->data);
			if (opl->plugin)
				printf("\tplugin->name: \"%s\"\n"
					"\tplugin->path: \"%s\"\n"
					"\topl->plugin->ce_base: 0x%08x%s\n"
					"\topl->plugin->ce_num: %i\n"
					"\topl->plugin->init: 0x%08x\n"
					"\topl->plugin->fini: 0x%08x\n"
					"\topl->plugin->output: 0x%08x\n"
					"\topl->plugin->signal: 0x%08x\n"
					"\t---\n",
					opl->plugin->name,
					opl->plugin->path,
					(int)opl->plugin->ce_base,
					(opl->ce == opl->plugin->ce_base)?" ERROR":"",
					opl->plugin->ce_num,
					(int)opl->plugin->init,
					(int)opl->plugin->fini,
					(int)opl->plugin->output,
					(int)opl->plugin->signal);
		}

		if (input_list[ctr] == NULL) {
			printf("* input_list[%i] empty.\n", ctr);
			continue;
		}
		printf("* input_list[%i]:\n", ctr);
		for (ipl = input_list[ctr]; ipl; ipl = ipl->next) {
			printf("\tplugin: 0x%08x\n"
				"\tcalled: %i\n"
				"\t\n",
				(int)ipl->plugin,
				ipl->called);
			if (ipl->plugin)
				printf("\tipl->plugin->name: \"%s\"\n"
					"\tipl->plugin->path: \"%s\"\n"
					"\tipl->plugin->input: 0x%08x\n"
					"\t---\n",
					ipl->plugin->name,
					ipl->plugin->path,
					(int)ipl->plugin->input);
		}
	}
	fflush(stdout);

	return;
}


#else
#define DEBUGP(format, args...)

#endif /* DEBUG */


/***********************************************************************
 *   SUPPORT FUNCTIONS
 ***********************************************************************/

/*
 * register_iret()  - adds given iret structure to iret_values list
 *
 * Arguments:     iret    pointer to structure to add
 * Return value:  none
 */
static void register_iret(specter_iret_t *iret)
{
	specter_iret_t *p;

	if (iret_values) {
		for (p = iret_values; p->next; p = p->next) ;
		p->next = iret;
	} else {
		iret_values = iret;
	}

	return;
}


/*
 * alloc_config()  - copies given config entries converting
 *                   data from array into linked list
 *
 * Arguments:     ce_list   array of config entries to copy
 *                num       number of elements in array
 * Return value:            pointer to allocated linked list of config entries
 */
static config_entry_t *alloc_config(config_entry_t *ce_list, int num)
{
	config_entry_t *curr = NULL, *last = NULL;

	while (num--) {
		if ((curr = malloc(sizeof(config_entry_t))) == NULL) {
			specter_log(SPECTER_FATAL, "alloc_config(): out of memory.\n");
			return NULL;
		}
		memcpy(curr, &ce_list[num], sizeof(config_entry_t));
		if (last) {
			curr->next = last;
		}
		last = curr;
	}

	return curr;
}


/*
 * init_all_output()  - initializes all output plugins
 *
 * Arguments:     none
 * Return value:          0 if successful, -1 on error
 */
static int init_all_output(void)
{
	struct output_plugin_list *curr;
	int ctr;

	for (ctr = 0; ctr < groups_top; ctr++) {
		last_group = ctr;
		for (curr = output_list[ctr]; curr; curr = curr->next) {
			if (curr->plugin && curr->plugin->init) {
				if ((curr->data = curr->plugin->init(curr->ce)) == NULL)
					return -1;
			} else {
				curr->data = NULL;
			}
		}
	}

	return 0;
}


/*
 * isloaded()  - checks if input plugin is present in given group
 *
 * Arguments:     plugin    plugin to look for
 *                group     group list to search
 * Return value:            1 if plugin was found, 0 if not
 */
static int isloaded(specter_input_t *plugin, int group)
{
	struct input_plugin_list *p;

	for (p = input_list[group]; p; p = p->next) {
		if (p->plugin == plugin)
			return 1;
	}

	return 0;
}


/***********************************************************************
 *   PUBLIC INTERFACE
 ***********************************************************************/
#define REGISTER_AGAIN 1

/*
 * register_input()  - registers input plugin in local structures
 *
 * Arguments:     plugin  pointer to plugin description
 *                flags   special options, see below
 * Return value:          0 if successful, -1 on error
 *
 * Function is usually invoked by plugin during dlopen(), but
 * it could also be called in load_plugin() - REGISTER_AGAIN
 * flag is set in that case.
 */
int register_input(specter_input_t *plugin, specter_iret_t *iret, int num, int flags)
{
	struct input_plugin_list *curr;
	int ctr;

	if (plugin->input == NULL) {
		DEBUGP("register_input(): requested registering "
			"plugin with (plugin->input == NULL).\n");
		return -1;
	}

	specter_log(SPECTER_DEBUG, "Registering input plugin \"%s\" for group %u.\n",
			plugin->name, last_group+1);

	if (!(flags & REGISTER_AGAIN)) {
		for (ctr = 0; ctr < num; ctr++) {
			iret[ctr].plugin = plugin;
			register_iret(&iret[ctr]);
		}
	}

	if ((curr = malloc(sizeof(struct input_plugin_list))) == NULL) {
		specter_log(SPECTER_FATAL, "register_input(): out of memory.\n");
		return -1;
	}
	curr->plugin = plugin;
	curr->called = 0;

	/* prepend plugin to list, so that output_list will point to last
	 * registered plugin (needed in load_config() and load_plugin()) */
	if (input_list[last_group])
		curr->next = input_list[last_group];
	else
		curr->next = NULL;
	input_list[last_group] = curr;

	return 0;
}


/*
 * register_output()  - registers output plugin in local structures
 *
 * Arguments:     plugin  pointer to plugin description
 *                flags   special options, see below
 * Return value:          0 if successful, -1 on error
 *
 * Function is usually invoked by plugin during dlopen(), but
 * it could also be called in load_plugin() - REGISTER_AGAIN
 * flag is set in that case.
 */
int register_output(specter_output_t *plugin, int flags)
{
	struct output_plugin_list *curr;

	if (plugin->output == NULL) {
		DEBUGP("register_output(): requested registering "
			"plugin with (plugin->output == NULL).\n");
		return -1;
	}

	specter_log(SPECTER_DEBUG, "Registering output plugin \"%s\" for group %i.\n",
			plugin->name, last_group+1);

	if ((curr = malloc(sizeof(struct output_plugin_list))) == NULL) {
		specter_log(SPECTER_FATAL, "register_output(): out of memory.\n");
		return -1;
	}
	curr->plugin = plugin;

	/* prepend plugin to list, so that output_list will point to last
	 * registered plugin (needed in load_config() and load_plugin()) */
	if (output_list[last_group])
		curr->next = output_list[last_group];
	else
		curr->next = NULL;
	output_list[last_group] = curr;

	if (plugin->ce_num) {
		if ((curr->ce = alloc_config(plugin->ce_base, plugin->ce_num)) == NULL) {
			return -1;
		}
	} else {
		curr->ce = NULL;
	}

	return 0;
}


/*
 * find_iret()  - finds iret value with given name
 *
 * Arguments:     name   iret name to look for
 * Return value:         pointer to found iret, NULL on error
 *
 * If (name == "*"), full list of iret values is stored in p.
 */
specter_iret_t *find_iret(char *name)
{
	specter_iret_t *iret;

	if (!strcmp(name, "*"))
		return iret_values;

	for (iret = iret_values; iret; iret = iret->next) {
		if (!strncmp(name, iret->name, SPECTER_IRET_NAME_LEN)) {
			break;
		}
	}

	if (!iret) {
		return NULL;
	}

	if (!isloaded(iret->plugin, last_group)) {
		specter_log(SPECTER_INFO, "Plugin \"%s\" needed in block %u.\n",
				iret->plugin->name, last_group + 1);
		return NULL;
	}

	return iret;
}


/*
 * GET_CE()  - finds config entry of given number in a linked list
 *
 * Arguments:     ce   linked list of config entries to search
 *                x    number of iterations to perform
 * Return value:       pointer to found config entry
 *
 * No checks here, so be careful - if 'ce' don't have enough elements,
 * it will cause SIGSEGV.
 */
config_entry_t *GET_CE(config_entry_t *ce, int x)
{
	while (x--)
		ce = ce->next;
	return ce;
}


/***********************************************************************
 *   PLUGIN INVOKING
 ***********************************************************************/

/*
 * call_input()  - calls all input plugins for appropriate group
 *
 * Arguments:     group    group current packet arrived with
 * Return value:           0 when successful, -1 on error
 */
static int call_input(ulog_packet_msg_t *pkt, int group)
{
	struct input_plugin_list *p;

	if (input_list[group] == NULL) {
		specter_log(SPECTER_DEBUG, "No input plugins set for group %i.\n", group);
		return 0;
	}

	for (p = input_list[group]; p; p = p->next) {
		if (p->called)
			continue;
		p->called = 1;
		if (p->plugin && p->plugin->input(pkt) == -1) {
			if (!errignore_ce.u.value)
				return -1;
		}
	}

	return 0;
}


/*
 * call_output()  - calls all output plugins for appropriate group
 *
 * Arguments:     group    group current packet arrived with
 *                         (is in range 0-31)
 * Return value:           0 when successful, -1 on error
 */
static int call_output(int group)
{
	struct output_plugin_list *p;

	if (output_list[group] == NULL) {
		specter_log(SPECTER_INFO, "No output plugins set for group %i.\n", group);
		return 0;
	}

	for (p = output_list[group]; p; p = p->next) {
		if (p->plugin && p->plugin->output(p->ce, p->data) == -1) {
			if (!errignore_ce.u.value)
				return -1;
		}
	}

	return 0;
}


/*
 * clear_input()  - sets all iret values to their initial state
 *                  as well as input plugins statuses
 *
 * Arguments:     none
 * Return value:  none
 */
static void clear_input(void)
{
	struct input_plugin_list *p;
	specter_iret_t *iret;
	int ctr;

	for (ctr = 0; ctr < groups_top; ctr++)
		for (p = input_list[ctr]; p; p = p->next)
			p->called = 0;

	for (iret = iret_values; iret; iret = iret->next) {
		if (iret->flags & SPECTER_RETF_FREE)
			free(iret->value.ptr);
		iret->flags = 0x0;
	}
}


/***********************************************************************
 *   LOGGING FACILITY
 ***********************************************************************/
static FILE *logfile;

/*
 * specter_log()  - logs given message at given log level
 *
 * Arguments:     level   level of message
 *                file    __FILE__
 *                format  message with printf formatting
 *                ...     arguments for format
 * Return value:  none
 */
void __specter_log(int level, const char *file, const char *format, ...)
{
	static char buff[256];
	char *levelstr;
	char *timestr;
	char *pluginstr;
	va_list ap;
	time_t tm;

	if (!logfile) {
		DEBUGP("specter_log(): logfile == NULL.\n");
		logfile = stderr;
	}

	if (level < loglevel_ce.u.value)
		return;

	switch (level) {
		case SPECTER_DEBUG:
			levelstr = "DEBUG";
			break;
		case SPECTER_INFO:
			levelstr = "INFO";
			break;
		case SPECTER_NOTICE:
			levelstr = "NOTICE";
			break;
		case SPECTER_ERROR:
			levelstr = "ERROR";
			break;
		case SPECTER_FATAL:
			levelstr = "FATAL";
			break;
		default:
			DEBUGP("specter_log(): bad loglevel %i.\n", level);
			return;
	}

	/* if 'file' is a plugin filename, trim it */
	pluginstr = strncpy(buff, file, 255);
	if (!strncmp(pluginstr, "specter_", 8)) {
		char *p;
		if ((p = strrchr(pluginstr, '.')) != NULL)
			*p = '\0';
		pluginstr += 8;
	} else {
		pluginstr = "specter";
	}

	tm = time(NULL);
	timestr = ctime(&tm);
	timestr[strlen(timestr) - 1] = '\0';
	fprintf(logfile, "%s <%s> %s: ", timestr, levelstr, pluginstr);

	va_start(ap, format);
	vfprintf(logfile, format, ap);
	va_end(ap);

	fflush(logfile);

	return;
}


/*
 * open_logfile()  - opens logfile from given path or stdout
 *
 * Arguments:     path  path to log file
 * Return value:        0 when successful, -1 on error
 */
static int open_logfile(char *path)
{
	if (!strcmp(path, "stdout")) {
		logfile = stdout;
	} else if (!strcmp(path, "stderr")) {
		logfile = stderr;
	} else {
		if ((logfile = fopen(path, "a")) == NULL) {
			specter_log(SPECTER_FATAL, "Couldn't open logfile \"%s\": %s.\n",
					path, strerror(errno));
			return -1;
		}
	}

	return 0;
}


/***********************************************************************
 *   INITIALIZATION
 ***********************************************************************/

/*
 * load_plugin()  - loads a plugin or register it manually if
 *                  it was already loaded
 *
 * Arguments:     path  path to plugin
 * Return value:        0 when successful, -1 on error
 */
static int load_plugin(char *path)
{
	int ctr;
	struct output_plugin_list *opl;
	struct input_plugin_list *ipl;

	/* check if plugin was loaded */
	for (ctr = 0; ctr < groups_top; ctr++) {
		for (opl = output_list[ctr]; opl; opl = opl->next) {
			if (opl->plugin && !strcmp(opl->plugin->path, path)) {
				if (last_group == ctr) {
					specter_log(SPECTER_FATAL, "Output plugin \"%s\" "
							"already loaded.\n", opl->plugin->name);
					/* output plugins have different set of options
					 * we can't ignore multiple attempts to load them */
					return -1;
				}
				if (register_output(opl->plugin, REGISTER_AGAIN) == -1)
					return -1;
				return 0;
			}
		}
	}

	for (ctr = 0; ctr < groups_top; ctr++) {
		for (ipl = input_list[ctr]; ipl; ipl = ipl->next) {
			if (ipl->plugin && !strcmp(ipl->plugin->path, path)) {
				if (last_group == ctr) {
					specter_log(SPECTER_INFO, "Input plugin \"%s\" "
							"already loaded.\n", ipl->plugin->name);
					return 0;
				}
				if (register_input(ipl->plugin, NULL, 0, REGISTER_AGAIN) == -1)
					return -1;
				return 0;
			}
		}
	}

	if (dlopen(path, RTLD_NOW) == NULL) {
		specter_log(SPECTER_FATAL, "Couldn't load plugin: %s.\n", dlerror());
		return -1;
	}

	/* fill missing path element, as register_* invoked by plugin don't set it */
	if (output_list[last_group] != NULL
			&& output_list[last_group]->plugin->path[0] == '\0') {
		strncpy(output_list[last_group]->plugin->path, path, PATH_MAX-1);
	} else if (input_list[last_group] != NULL
			&& input_list[last_group]->plugin->path[0] == '\0') {
		strncpy(input_list[last_group]->plugin->path, path, PATH_MAX-1);
	}

	return 0;
}


/*
 * load_config_block()  - reads whole config block
 *
 * Arguments:     block     block name to be parsed
 *                pt        array of plugin_tags structures
 *                tags_nb   number of elements in pt[]
 * Return value:            0 when successful, -1 on error
 */
static int load_config_block(const char *block, struct plugin_tags *pt, int tags_nb)
{
	int ctr, ret;
	config_entry_t ce_dummy = { .type = CONFIG_TYPE_STRING, .options = CONFIG_OPT_ANY };

	/* check for 'include' statements */
	if (is_section(block, NULL)) {
		config_entry_t ce_include = {
			.key = "include",
			.type = CONFIG_TYPE_STRING,
			.options = CONFIG_OPT_MULTI
		};
		/* no need for recursion limits, as each 'include' statement gets
		 * read only once (CONFIG_READ_REWIND flag is not set) */
		while ((ret = config_read(block, NULL, &ce_include, 0)) == CONFIG_RET_MORE) {
			DEBUGP("Entering include statement \"%s\".\n",
					ce_include.u.string);
			if (!is_block(ce_include.u.string)) {
				specter_log(SPECTER_FATAL, "No such block \"%s\".\n",
						ce_include.u.string);
				return -1;
			}
			if (load_config_block(ce_include.u.string, pt, tags_nb) == -1)
				return -1;
		}
		if (ret == CONFIG_RET_ERROR)
			return -1;
	}

	for (ctr = 0; ctr < tags_nb; ctr++) {
		if (!is_section(block, pt[ctr].tag))
			continue;

		/* load plugin first */
		if (load_plugin(pt[ctr].path) == -1)
			return -1;

		/* now read its options */
		if (output_list[last_group] && output_list[last_group]->plugin
				&& !strncmp(pt[ctr].path, output_list[last_group]->plugin->path, PATH_MAX)) {
			if (config_read(block, pt[ctr].tag, output_list[last_group]->ce,
						CONFIG_READ_REWIND) != CONFIG_RET_OK)
			return -1;
		} else {
			/* input plugins don't need options */
			ret = config_read(block, pt[ctr].tag, &ce_dummy, CONFIG_READ_REWIND);
			if (ret == CONFIG_RET_MORE) {
				fprintf(stderr, "Config error in block \"%s\": "
						"Input plugins don't have options.\n",
						block);
				return -1;
			} else if (ret == CONFIG_RET_ERROR) {
				return -1;
			}
		}
	}

	return 0;
}


/*
 * load_config()  - reads config file and sets all related structures
 *
 * Arguments:     path    path to configuration file
 * Return value:          0 when successful, -1 on error
 */
static int load_config(char *path)
{
	int ctr, ret;
	struct plugin_tags pt[32]; /* currently we have 11 plugins, that should be enough ;) */
	int tags_nb = 0;
	config_entry_t ce_dummy = { .type = CONFIG_TYPE_STRING, .options = CONFIG_OPT_ANY };
	int highest_group = 0;

	if (!config_open(path)) {
		return -1;
	}

	/* load global options */
	ret = config_read("global", NULL, global_ce, CONFIG_READ_REWIND);
	if (ret == CONFIG_RET_ERROR || ret == CONFIG_RET_MORE) {
		return -1;
	} else if (ret == CONFIG_RET_MISSING) {
		specter_log(SPECTER_DEBUG, "\"global\" config block missing - using defaults.\n");
	}

	/* now we can open logfile */
	if (open_logfile(logfile_ce.u.string) == -1)
		return -1;

	/* load plugins tags */
	while (tags_nb < 32) {
		ret = config_read("plugins", NULL, &ce_dummy, 0);
		if (ret == CONFIG_RET_ERROR || ret == CONFIG_RET_MISSING) {
			specter_log(SPECTER_FATAL, "\"plugins\" config block missing.\n");
			return -1;
		} else if (ret == CONFIG_RET_OK) {
			break;
		} /* else if (ret == CONFIG_RET_MORE) */
		strncpy(pt[tags_nb].tag, ce_dummy.key, CONFIG_KEY_LEN);
		strncpy(pt[tags_nb].path, ce_dummy.u.string, CONFIG_VALUE_LEN);
		tags_nb++;
	}

	/* load options for each group mode */
	for (ctr = 0; ctr < groups_top; ctr++) {
		char ctrstr[3];

		snprintf(ctrstr, 3, "%i", ctr+1);
		last_group = ctr;

		if (!is_block(ctrstr)) {
			output_list[ctr] = NULL;
			input_list[ctr] = NULL;
			continue;
		}

		if (load_config_block(ctrstr, pt, tags_nb) == -1) {
			return -1;
		}

		if (!output_list[ctr]) {
			fprintf(stderr, "Block \"%s\" has no meaning "
					"(no output plugins defined).\n",
					ctrstr);
			return -1;
		}

		highest_group = ctr + 1;
	}

	groups_top = highest_group;
	DEBUGP("groups_top stripped to %i.\n", groups_top);

	if (config_close() == -1)
		return -1;

	if (init_all_output() == -1)
		return -1;

	return 0;
}


/***********************************************************************
 *   MAIN PROGRAM
 ***********************************************************************/
static void cleanup(void)
{
	struct output_plugin_list *p;
	int ctr;

	ipulog_destroy_handle(libulog_h);
	free(libulog_buf);

	if (logfile != stdout && logfile != stderr)
		fclose(logfile);

	for (ctr = 0; ctr < groups_top; ctr++) {
		for (p = output_list[ctr]; p; p = p->next) {
			if (p->plugin && p->plugin->fini)
				p->plugin->fini(p->ce, p->data);
		}
	}
}

static void sigterm_handler(int signal)
{
	if (signal == SIGTERM) {
		specter_log(SPECTER_FATAL, "Sigterm received. Exiting...\n");
		cleanup();
		exit(EXIT_SUCCESS);
	}
	if (signal == SIGINT) {
		specter_log(SPECTER_FATAL, "Sigint received. Exiting...\n");
		cleanup();
		exit(EXIT_SUCCESS);
	}
}


static void sighup_handler(int signal)
{
	struct output_plugin_list *p;
	int ctr;

	if (signal == SIGHUP)
		specter_log(SPECTER_NOTICE, "Sighup received, reopening logfiles.\n");

	if (logfile != stdout) {
		fclose(logfile);
		if ((logfile = fopen(logfile_ce.u.string, "a")) == NULL) {
			specter_log(SPECTER_FATAL, "Couldn't open logfile: %s.\n",
					strerror(errno));
			cleanup();
			exit(EXIT_FAILURE);
		}
	}

	for (ctr = 0; ctr < groups_top; ctr++)
		for (p = output_list[ctr]; p; p = p->next)
			if (p->plugin && p->plugin->signal
				&& p->plugin->signal(p->ce, p->data, SIGHUP) == -1) {
				/* ignore error but delete plugin from list */
				if (errignore_ce.u.value) {
					p->plugin = NULL;
				} else {
					specter_log(SPECTER_FATAL, "Module \"%s\" failed.\n",
							p->plugin->name);
					cleanup();
					exit(EXIT_FAILURE);
				}
			}
}


static void print_version(void)
{
	printf("specter %s\n", SPECTER_VERSION);
	printf("Copyright (C) 2000-2003 Harald Welte "
	       "<laforge@gnumonks.org>\n"
	       "          (C) 2004,2005 Michal Kwiatkowski "
	       "<ruby@joker.linuxstuff.pl>\n");
	printf("\nThis program is free software; you can redistribute it and/or modify\n"
		"it under the terms of the GNU General Public License version 2\n"
		"as published by the Free Software Foundation\n"
		"\n"
		"This program is distributed in the hope that it will be useful,\n"
		"but WITHOUT ANY WARRANTY; without even the implied warranty of\n"
		"MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the\n"
		"GNU General Public License for more details.\n"
		"\n"
		"You should have received a copy of the GNU General Public License\n"
		"along with this program; if not, write to the Free Software\n"
		"Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA\n");
}


static void print_usage(void)
{
	printf("Usage: specter [OPTION]...\n\n"
		"  -h, --help         display this help page\n"
		"  -V, --version      print version information\n"
		"  -d, --daemon       daemonize (fork into background)\n"
		"  -c, --configfile   use alternative configfile\n"
		"  -u, --uid          change uid\n"
		"  -g, --gid          change gid\n");
}


int main(int argc, char *argv[])
{
	struct option opts[] = {
		{ "version", 0, NULL, 'V' },
		{ "daemon", 0, NULL, 'd' },
		{ "help", 0, NULL, 'h' },
		{ "configfile", 1, NULL, 'c'},
		{ "uid", 1, NULL, 'u' },
		{ "gid", 1, NULL, 'g' },
		{ 0 }
	};
	int argch;
	int len;
	int ctr;
	char *configfile = SPECTER_CONFIGFILE_DEFAULT;
	ulog_packet_msg_t *upkt;
	u_int32_t nlmask = 0x0;
	int grouping;
	enum {
		GROUPING_NETLINK,
		GROUPING_NFMARK
	};

	int daemonize = 0;
	int change_uid = 0, change_gid = 0;

	char *user = NULL, *group = NULL;
	struct passwd *pw;
	struct group *gr;
	uid_t uid = 0;
	gid_t gid = 0;

	while ((argch = getopt_long(argc, argv, "c:dh::Vu:g:", opts, NULL)) != -1) {
		switch (argch) {
		case '?':
		case ':':
			print_usage();
			exit(EXIT_FAILURE);
			break;
		case 'h':
			print_usage();
			exit(EXIT_SUCCESS);
			break;
		case 'd':
			daemonize = 1;
			break;
		case 'V':
			print_version();
			exit(EXIT_SUCCESS);
			break;
		case 'c':
			configfile = optarg;
			break;
		case 'u':
			change_uid = 1;
			user = strdup(optarg);
			if ((pw = getpwnam(user)) == 0) {
				printf("Unknown user %s.\n", user);
				free(user);
				exit(EXIT_FAILURE);
			}
			uid = pw->pw_uid;
			break;
		case 'g':
			change_gid = 1;
			group = strdup(optarg);
			if ((gr = getgrnam(group)) == 0) {
				printf("Unknown group %s.\n", group);
				free(group);
				exit(EXIT_FAILURE);
			}
			gid = gr->gr_gid;
			break;
		default:
			if (isprint(optopt))
				fprintf(stderr, "Unknown option `-%c'.\n", optopt);
			else
				fprintf(stderr, "Unknown option character `\\x%x'.\n", optopt);
			print_usage();
			exit(EXIT_FAILURE);
			break;
		}
	}

	/* allocate output_list/input_list array of pointers */
	output_list = malloc(groups_top * sizeof(struct output_plugin_list *));
	input_list = malloc(groups_top * sizeof(struct input_plugin_list *));
	if (output_list == NULL || input_list == NULL) {
		specter_log(SPECTER_FATAL, "Unable to allocate plugins lists.\n");
		exit(EXIT_FAILURE);
	}
	memset(output_list, 0x0, groups_top * sizeof(struct input_plugin_list *));
	memset(input_list, 0x0, groups_top * sizeof(struct input_plugin_list *));

	/* load config, open logs and initialize plugins */
	if (load_config(configfile) == -1)
		exit(EXIT_FAILURE);

	/* free unused pointers */
	output_list = realloc(output_list, groups_top * sizeof(struct output_plugin_list *));
	input_list = realloc(input_list, groups_top * sizeof(struct input_plugin_list *));
	if (output_list == NULL || input_list == NULL) {
		specter_log(SPECTER_FATAL, "Unable to reallocate plugins lists.\n");
		exit(EXIT_FAILURE);
	}

	/* set grouping mode */
	if (!strcmp(grouping_ce.u.string, "netlink")) {
		grouping = GROUPING_NETLINK;
	} else if (!strcmp(grouping_ce.u.string, "nfmark")) {
		grouping = GROUPING_NFMARK;
	} else {
		specter_log(SPECTER_FATAL, "Grouping mode \"%s\" unknown.\n",
				grouping_ce.u.string);
		exit(EXIT_FAILURE);
	}

	/* allocate a receive buffer */
	libulog_buf = (unsigned char *) malloc(bufsiz_ce.u.value);
	if (libulog_buf == NULL) {
		specter_log(SPECTER_FATAL, "Unable to allocate receive buffer "
			  "of %d bytes.\n", bufsiz_ce.u.value);
		exit(EXIT_FAILURE);
	}

	if (grouping == GROUPING_NETLINK) {
		if (groups_top > 32) {
			specter_log(SPECTER_FATAL, "Netlink group value too high.\n");
			exit(EXIT_FAILURE);
		}
		if (nlgroup_ce.hit) {
			specter_log(SPECTER_FATAL, "With netlink grouping \"nlgroup\" "
					"option is invalid.\n");
			exit(EXIT_FAILURE);
		}
		/* calculate nlmask */
		for (ctr = 0; ctr < groups_top; ctr++) {
			if (output_list[ctr] != NULL)
				nlmask |= ipulog_group2gmask(ctr+1);
		}
	} else {
		if ((nlmask |= ipulog_group2gmask(nlgroup_ce.u.value)) == 0) {
			specter_log(SPECTER_FATAL, "Invalid netlink group value.\n");
			exit(EXIT_FAILURE);
		}
	}

	/* create ipulog handle */
	libulog_h = ipulog_create_handle(nlmask, rmem_ce.u.value);
	if (!libulog_h) {
		specter_log(SPECTER_FATAL, "Unable to create ipulog handle: %s.\n",
				ipulog_strerror(ipulog_errno));
		exit(EXIT_FAILURE);
	}

	/* change uid/gid as soon as possible */
	if (change_uid && change_gid) {
		if (initgroups(user, gid)) {
			specter_log(SPECTER_FATAL, "Can't set user secondary gid: %s.\n",
					strerror(errno));
			exit(EXIT_FAILURE);
		}
	}
	if (change_gid) {
		specter_log(SPECTER_NOTICE, "Setting gid=%i\n", gid);
		if (setgid(gid)) {
			specter_log(SPECTER_FATAL, "Can't set gid: %s.\n",
					strerror(errno));
			exit(EXIT_FAILURE);
		}
		if (setegid(gid)) {
			specter_log(SPECTER_FATAL, "Can't set effective gid: %s.\n",
					strerror(errno));
			exit(EXIT_FAILURE);
		}
	}
	if (change_uid) {
		specter_log(SPECTER_NOTICE, "Setting uid=%i\n", uid);
		if (setuid(uid)) {
			specter_log(SPECTER_FATAL, "Can't set uid: %s.\n",
					strerror(errno));
			exit(EXIT_FAILURE);
		}
		if (seteuid(uid)) {
			specter_log(SPECTER_FATAL, "Can't set effective uid: %s.\n",
					strerror(errno));
			exit(EXIT_FAILURE);
		}
	}
	if (user)
		free(user);
	if (group)
		free(group);

	/* move to background */
	if (daemonize) {
		int ret;
		ret = fork();
		if (ret == -1) {
			specter_log(SPECTER_FATAL, "Unable to fork(): %s.\n",
				strerror(errno));
			cleanup();
			exit(EXIT_FAILURE);
		}
		if (ret > 0) {
			exit(EXIT_SUCCESS);
		}
		if (logfile != stdout)
			fclose(stdout);
		if (logfile != stderr)
			fclose(stderr);
		fclose(stdin);
		setsid();
	}

	signal(SIGTERM, &sigterm_handler);
	signal(SIGINT, &sigterm_handler);
	signal(SIGHUP, &sighup_handler);

	specter_log(SPECTER_INFO, "specter %s starting...\n", SPECTER_VERSION);

	/* endless loop */
	while ((len = ipulog_read(libulog_h, libulog_buf, 
				bufsiz_ce.u.value, 1))) {
		if (len < 0) {
			/* this is not supposed to happen */
			specter_log(SPECTER_ERROR, "ipulog_read() error: %s : %s.\n",
				  ipulog_strerror(ipulog_errno), strerror(errno));
		} else {
			while ((upkt = ipulog_get_packet(libulog_h,
					       libulog_buf, len, &nlmask))) {

				DEBUGP("main() loop: packet with nlmask 0x%08x "
						"and nfmark %lu received.\n",
						nlmask, upkt->mark);

				if (grouping == GROUPING_NETLINK) {
					for (len = 0; len < 32; len++) {
						if (nlmask & (0x1 << len)) {
							if (call_input(upkt, len) == -1)
								exit(EXIT_FAILURE);
							if (call_output(len) == -1)
								exit(EXIT_FAILURE);
						}
					}
				} else if (grouping == GROUPING_NFMARK) {
					if (upkt->mark < 1 || upkt->mark > groups_top)
						continue;
					if (call_input(upkt, upkt->mark - 1) == -1)
						exit(EXIT_FAILURE);
					if (call_output(upkt->mark - 1) == -1)
						exit(EXIT_FAILURE);
				}

				clear_input();
			}
		}
	}

	specter_log(SPECTER_FATAL, "ipulog_read() returned 0.\n");
	cleanup();
	return EXIT_SUCCESS;
}

