/*  $Id: misc-types.h,v 1.3 2005/03/13 04:03:27 sarrazip Exp $
    misc-types.h - Miscellaneous types used by the dictionary class.

    verbiste - French conjugation system
    Copyright (C) 2003-2005 Pierre Sarrazin <http://sarrazip.com/>

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
    02111-1307, USA.
*/

#ifndef _H_misc_types
#define _H_misc_types

#include <verbiste/c-api.h>
#include <verbiste/HashMap.h>

#include <assert.h>
#include <vector>
#include <string>


/**
    Valid modes.
*/
enum Mode
{
    INVALID_MODE = VERBISTE_INVALID_MODE, 
    INFINITIVE_MODE = VERBISTE_INFINITIVE_MODE, 
    INDICATIVE_MODE = VERBISTE_INDICATIVE_MODE, 
    CONDITIONAL_MODE = VERBISTE_CONDITIONAL_MODE,
    SUBJUNCTIVE_MODE = VERBISTE_SUBJUNCTIVE_MODE, 
    IMPERATIVE_MODE = VERBISTE_IMPERATIVE_MODE, 
    PARTICIPLE_MODE = VERBISTE_PARTICIPLE_MODE
};


/**
    Valid tenses.
*/
enum Tense 
{
    INVALID_TENSE = VERBISTE_INVALID_TENSE, 
    PRESENT_TENSE = VERBISTE_PRESENT_TENSE, 
    PAST_TENSE = VERBISTE_PAST_TENSE, 
    IMPERFECT_TENSE = VERBISTE_IMPERFECT_TENSE, 
    FUTURE_TENSE = VERBISTE_FUTURE_TENSE
};



/**
    Description of a verb inflection.
    Gives the mode, tense and person of a conjugated verb.
    The person is 1, 2 or 3 for je/nous, tu/vous, il/ils, except in
    infinitive and participle mode, where it is always 0
    since it does not apply.
*/
class ModeTensePersonNumber
{
public:

    /** Mode (infinitive, indicative, etc). */
    Mode mode;

    /** Tense (present, past, etc). */
    Tense tense;

    /** Person (1, 2 or 3, or 0 in infinitive and participle modes). */
    unsigned char person;  // 1, 2 or 3, or 0 (invalid)

    /** Number (true for plural, false for singular). */
    bool plural;


    /**
	Constructs an object from optional arguments.
    */
    ModeTensePersonNumber(Mode m = INVALID_MODE,
		    Tense t = INVALID_TENSE,
		    unsigned char pers = 0,
		    bool plur = false)
      : mode(m), tense(t), person(pers), plural(plur)
    {
	assert(mode <= PARTICIPLE_MODE);
	assert(tense <= FUTURE_TENSE);
	assert(person <= 3);
    }


    /**
	Constructs an object from English mode and tense names.
	Call the set() method.
	@param	modeName	mode name known to method
				FrenchVerbDictionary::convertModeName()
	@param	tenseName	tense name known to method
				FrenchVerbDictionary::convertTenseName()
	@param	personNum	person "counter" (1--6, except for imperative
				mode, where it must be 1--3; ignored for
				infinitive and participle modes)
    */
    ModeTensePersonNumber(const char *modeName,
		    const char *tenseName,
		    int personNum)
    {
	set(modeName, tenseName, personNum);
    }


    /**
	Initializes an object from English mode and tense names.
	Call the set() method.
	@param	modeName	mode name known to method
				FrenchVerbDictionary::convertModeName()
	@param	tenseName	tense name known to method
				FrenchVerbDictionary::convertTenseName()
	@param	personNum	person "counter" (1--6, except for imperative
				mode, where it must be 1--3; ignored for
				infinitive and participle modes)
    */
    void set(const char *modeName, const char *tenseName, int personNum);


    /**
	Dumps the fields of this object into those of the destination object.
	@param	destination	object whose fields (mode, tense, person,
				number) are assigned with values taken
				from this object
    */
    void dump(Verbiste_ModeTensePersonNumber &destination) const;
};


/** List of inflections. */
typedef std::vector<std::string> PersonSpec;


/** List of persons (1, 3 or 6 persons depending on the mode and tense). */
typedef std::vector<PersonSpec> TenseSpec;


/** Function object that hashes a Tense value for use in a hash_map. */
class TenseHasher
{
public:
    size_t operator () (Tense t) const
    {
	return size_t(t) * 8191;  // multiply by a prime; result is < 32767
    }
};


/**
    Mode specification.
    Contains tense specifications indexed by Tense values.
*/
typedef HashMap<Tense, TenseSpec, TenseHasher> ModeSpec;


/** Function object that hashes a Mode value for use in a hash_map. */
struct ModeHasher
{
    size_t operator () (Mode m) const
    {
	return size_t(m) * 5449;  // multiply by a prime; result is < 32767
    }
};


/**
    Conjugation template specification.
    Contains mode specifications indexed by Mode values.
*/
typedef HashMap<Mode, ModeSpec, ModeHasher> TemplateSpec;


/**
    Conjugation system for the known verbs of a language.
    Contains conjugation templates indexed by template names.
*/
typedef HashMap<std::string, TemplateSpec> ConjugationSystem;


/**
    Table of template names indexed by verb infinitive.
    If the verb "abaisser" follows the "aim:er" conjugation template,
    then a VerbTable would contain an entry where the key is "abaisser"
    and the value is "aim:er".
*/
typedef HashMap<std::string, std::string> VerbTable;


/**
    Table that describes the mode, tense and person of a number of inflections.
    For example, in the "aim:er" conjugation template, the inflection
    (termination) "e" is associated with:

    - the 1st person singular of the indicative present;

    - the 3rd person singular of the indicative present;

    - the 1st person singular of the subjunctive present;

    - the 3rd person singular of the subjunctive present;

    - the 2nd person singular of the imperative present.

    Thus, in this table, the key "e" would be be associated with a vector
    of five objects representing these five mode-tense-person combinations.
*/
typedef HashMap<std::string, std::vector<ModeTensePersonNumber> >
					    TemplateInflectionTable;


/**
    Table of template inflection tables, indexed by template names.
    For example, a key "aim:er" would be associated with a template
    inflection table that gives all inflections (terminations) that are
    accepted by the "aim:er" conjugation template (e.g., "e", "es",
    "ons", etc).
*/
typedef HashMap<std::string, TemplateInflectionTable> InflectionTable;


/**
    Description of a conjugated verb's inflection.
*/
class InflectionDesc
{
public:

    /** Infinitive form of the conjugated verb (UTF-8). */
    std::string infinitive;

    /** Conjugated template used by the verb (e.g. "aim:er") (UTF-8). */
    std::string templateName;

    /** Mode, tense, person and number of the inflection. */
    ModeTensePersonNumber mtpn;

    /**
	Constructs an inflection description from optional arguments.
    */
    InflectionDesc(const std::string &inf = "",
		const std::string &tname = "",
		ModeTensePersonNumber m = ModeTensePersonNumber())
      : infinitive(inf),
	templateName(tname),
	mtpn(m)
    {
    }
};


#endif  /* _H_misc_types */
