/*
 * WallFire -- a comprehensive firewall administration tool.
 * 
 * Copyright (C) 2001 Herv Eychenne <rv@wallfire.org>
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * 
 */

using namespace std;

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif

#include <stdio.h>
#include <time.h>
#include <pwd.h>    /* for getpwuid() */
#include <unistd.h> /* for getuid() */

#include "text.h"
#include "list1.h"
#include "defs.h"

#include "preresolv.h"
#include "timediff.h"
#include "conf.h"

#include "wfipaddr.h"
#include "wfport.h"
#include "wfprotocol.h"

#define TIMESIZE 40


wf_outmodule_text::wf_outmodule_text() :
  wf_outmodule(),
  dns(NULL),
  whois(NULL)
{
  conf = new wf_outmodule_text_conf();
}

wf_outmodule_text::~wf_outmodule_text() {
  if (dns != NULL)
    delete dns;
  if (whois != NULL)
    delete whois;
  delete conf;
}


void
wf_outmodule_text::output_header(ostream& os) const {
  char nowstr[TIMESIZE];
  time_t now = time(NULL);
  strftime(nowstr, TIMESIZE, "%a %b %d %H:%M:%S %Z %Y", localtime(&now));
  
  os << conf->config["title"].tostr() << "\n\n";

  struct passwd* gen_user = getpwuid(getuid());
  if (gen_user != NULL)
    ostream_printf(os, _("Generated on %s by %s."), nowstr,
		   gen_user->pw_gecos[0] != '\0' ?
		   gen_user->pw_gecos : gen_user->pw_name);
  else
    ostream_printf(os, _("Generated on %s by an unknown user."), nowstr);
  os << "\n\n";
}

static void
output_footer(ostream& os) {
  os << '\n';
  ostream_printf(os, _("Generated by wflogs %s"), VERSION);
  os << " (C) Herve Eychenne " WALLFIRE_URL "." << endl;
}

bool
wf_outmodule_text::resolv_init(unsigned char verbose) {
  if (conf->config["whois_lookup"].toint() != WHOIS_NO) {
    if (whois == NULL) {
      whois = new wf_whois(verbose);
      if (whois == NULL)
	return false;
    }
    if (whois->isconnected() == false && whois->connect() == false)
      return false; /* cannot connect */
  }
  if (conf->config["resolve"].toint() == RESOLV_YES) {
    if (dns == NULL) {
      dns = new wf_dns(verbose);
      if (dns == NULL)
	return false;
    }
  }
  return true;
}

bool
wf_outmodule_text::print(const wf_logentry* entry, ostream& os) {
  char time[TIMESIZE];
  unsigned char verbose = conf->config["verbose"].toint();

  os << entry->count;

  if (conf->config["stimes"].tobool()) {
    strftime(time, TIMESIZE, "%b %d %H:%M:%S", localtime(&entry->start_time));
    os << ' ' << time;
  }

  if (conf->config["etimes"].tobool()) {
    os << ' ';
    if (entry->end_time != 0) {
      strftime(time, TIMESIZE, "%b %d %H:%M:%S", localtime(&entry->end_time));
      os << time;
    }
    else
      os << '-';
  }

  if (conf->config["duration"].tobool())
    os << ' ' << timediff_tostr(entry->start_time, entry->end_time);

  if (conf->config["loghost"].tobool())
    os << ' ' << entry->hostname;

  if (conf->config["chain"].tobool())
    os << ' ' << entry->chainlabel;

  if (conf->config["branch"].tobool())
    os << ' ' << entry->branchname;

  if (conf->config["interfaces"].tobool()) {
    os << ' ';
    if (entry->input_iface.empty())
      os << '-';
    else
      os << entry->input_iface;
  }

  if (conf->config["src_mac"].tobool() && entry->smacaddr.isdefined()) {
    os << " mac " << entry->smacaddr;
    if (conf->config["mac_vendor"].tobool())
      os << " (" << entry->smacaddr.vendor() << ')';
  }

  if (conf->config["interfaces"].tobool()) {
    os << ' ';
    if (entry->output_iface.empty())
      os << '-';
    else
      os << entry->output_iface;
  }

  if (conf->config["dst_mac"].tobool() && entry->dmacaddr.isdefined()) {
    os << " mac " << entry->dmacaddr;
    if (conf->config["mac_vendor"].tobool())
      os << " (" << entry->dmacaddr.vendor() << ')';
  }

  string proto; /* make it "global" because we may need it later */
  {
    wf_protocol protocol = wf_protocol(entry->protocol);
    proto = protocol.tostr();
  }

  if (conf->config["proto"].tobool())
    os << ' ' << proto;
  
  if (entry->protocol == IPPROTO_ICMP) {
    os << " type " << entry->sport;
    if (wf_protocol_icmp_type_hascode(entry->sport))
      os << " code " << entry->dport;
    
    if (conf->config["sresolve"].tobool())
      os << " (" <<
	wf_protocol_icmp_type_tostr(entry->sport, entry->dport) << ')';
  }
  
  if (conf->config["datalen"].tobool())
    os << ' ' << entry->datalen;

  if (conf->config["src_ip"].tobool()) {
    bool src_resolved = false;

    os << " from " << entry->sipaddr;
    
    bool resolv_ok = resolv_init(verbose);

    if (conf->config["resolve"].toint() != RESOLV_NO) {
      os << " (";
      if (resolv_ok) {
	wf_dns_entry* dnsentry = dns->resolv(entry->sipaddr);
	if (dnsentry != NULL && dnsentry->name.empty() == false) {
	  src_resolved = true;
	  os << dnsentry->name;
	}
	else
	  os << '-';
      }
      else
	os << '?';
      os << ')';
    }
    if (conf->config["whois_lookup"].toint() != WHOIS_NO) {
      os << " (";
      if (resolv_ok) {
	if (conf->config["whois_lookup"].toint() == WHOIS_YES ||
	    (conf->config["whois_lookup"].toint() == WHOIS_IFNODNS &&
	     src_resolved == false)) {
	  wf_whois_entry* we = whois->whois(entry->sipaddr);
	  if (we != NULL)
	    os << *we;
	  else
	    os << '-';
	}
      }
      else
	os << '?';
      os << ')';
    }
  }

  if (conf->config["src_port"].tobool()) {
    if (entry->protocol == IPPROTO_UDP || entry->protocol == IPPROTO_TCP) {
      os << " port " << entry->sport;

      if (conf->config["sresolve"].tobool()) {
	string service = wf_port_toservice(entry->sport, proto);
	os << " (";
	if (service.empty())
	  os << '-';
	else
	  os << service;
	os << ')';
      }
    }
  }

  if (conf->config["dst_ip"].tobool()) {
    bool dst_resolved = false;

    os << " to " << entry->dipaddr;

    bool resolv_ok = resolv_init(verbose);

    if (conf->config["resolve"].toint() != RESOLV_NO) {
      os << " (";
      if (resolv_ok) {
	wf_dns_entry* dnsentry = dns->resolv(entry->dipaddr);
	if (dnsentry != NULL && dnsentry->name.empty() == false) {
	  dst_resolved = true;
	  os << dnsentry->name;
	}
	else
	  os << '-';
      }
      else
	os << '?';
      os << ')';
    }

    if (conf->config["whois_lookup"].toint() != WHOIS_NO) {
      os << " (";
      if (resolv_ok) {
	if (conf->config["whois_lookup"].toint() == WHOIS_YES ||
	    (conf->config["whois_lookup"].toint() == WHOIS_IFNODNS &&
	     dst_resolved == false)) {
	  wf_whois_entry* we = whois->whois(entry->dipaddr);
	  if (we != NULL)
	    os << *we;
	  else
	    os << '-';
	}
      }
      else
	os << '?';
      os << ')';
    }
  }
  
  if (conf->config["dst_port"].tobool()) {
    if (entry->protocol == IPPROTO_UDP || entry->protocol == IPPROTO_TCP) {
      os << " port " << entry->dport;
      
      if (conf->config["sresolve"].tobool()) {
	string service = wf_port_toservice(entry->dport, proto);
	os << " (";
	if (service.empty())
	  os << '-';
	else
	  os << service;
	os << ')';
      }
    }
  }
  
  if (entry->protocol == IPPROTO_TCP && conf->config["tcpflags"].tobool())
    os << ' ' << entry->tcpflags_tostr_mini();

  return true;
}

bool
wf_outmodule_text::print(const wf_logentries& logentries, ostream& os) {
  unsigned char verbose = conf->config["verbose"].toint();
  int lines = conf->config["lines"].toint();

  wf_logentries* newlogentries;
  if (lines) { /* copy n first logentries */
    newlogentries = new wf_logentries();
    if (newlogentries == NULL)
      return false;
    list1_firstncopy(newlogentries->elems, logentries.elems, lines);
  }
  else /* do not copy logentries */
    newlogentries = (wf_logentries*)&logentries;

  if (resolv_init(verbose) == false) {
    if (lines)
      delete newlogentries;
    return false;
  }
  
  if (conf->config["resolve"].toint() == RESOLV_YES) {
    preresolv(*newlogentries, dns,
	      conf->config["src_ip"].tobool(), conf->config["dst_ip"].tobool(),
	      verbose);
  }

  if (conf->config["header"].tobool())
    output_header(os);
  if (wf_outmodule::print(*newlogentries, os) == false) {
    if (lines)
      delete newlogentries;
    return false;
  }
  if (conf->config["header"].tobool())
    output_footer(os);

  if (conf->config["whois_lookup"].toint() != WHOIS_NO)
    whois->close();
  if (lines)
    delete newlogentries;

  return true;
}

extern "C" wf_outmodule*
wf_outmodule_text_init() {
  return new wf_outmodule_text();
}
