/*
 * WallFire -- a comprehensive firewall administration tool.
 * 
 * Copyright (C) 2001 Herv Eychenne <rv@wallfire.org>
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * 
 */

using namespace std;

#include <stdio.h>

#include "wfroute.h"
#include "defs.h"

#ifdef linux

#define _PATH_PROCNET_ROUTE	"/proc/net/route"

/* Caller must free return string. */
static char*
proc_gen_fmt(char* name, int more, FILE* fh, ...) {
    char buf[512], format[512] = "";
    char* title, *head, *hdr;
    va_list ap;

    if (fgets(buf, (sizeof buf) - 1, fh) == NULL)
	return NULL;
    strcat(buf, " ");

    va_start(ap, fh);
    title = va_arg(ap, char*);
    for (hdr = buf; hdr;) {
	while (isspace(*hdr) || *hdr == '|')
	    hdr++;
	head = hdr;
	hdr = strpbrk(hdr, "| \t\n");
	if (hdr)
	    *hdr++ = 0;

	if (!strcmp(title, head)) {
	    strcat(format, va_arg(ap, char*));
	    title = va_arg(ap, char*);
	    if (title == NULL || head == NULL)
		break;
	} else {
	    strcat(format, "%*s");	/* XXX */
	}
	strcat(format, " ");
    }
    va_end(ap);

    if (more == 0 && title != NULL) {
	fprintf(stderr, "warning: %s does not contain required field %s\n",
		name, title);
	return NULL;
    }
    return strdup(format);
}

int
wf_route_defgateways_get_linux(list<pair<wf_ipaddr, string> >& lst) {
  FILE* fp = fopen(_PATH_PROCNET_ROUTE, "r");
  if (fp == NULL) {
    perror(_PATH_PROCNET_ROUTE);
    printf(_("INET (IPv4) not configured in this system.\n"));
    return 1;
  }

  char buff[1024], iface[16];
  unsigned int gate_addr, net_addr, mask_addr;
  int iflags, refcnt, use, metric, mss, window, irtt;

  char* fmt = proc_gen_fmt(_PATH_PROCNET_ROUTE, 0, fp,
		     "Iface", "%16s",
		     "Destination", "%X",
		     "Gateway", "%X",
		     "Flags", "%X",
		     "RefCnt", "%d",
		     "Use", "%d",
		     "Metric", "%d",
		     "Mask", "%X",
		     "MTU", "%d",
		     "Window", "%d",
		     "IRTT", "%d",
		     NULL);
  /* "%16s %X %X %X %d %d %d %X %d %d %d\n" */
  if (fmt == NULL) {
    fclose(fp);
    return 1;
  }

  lst.clear();
  while (fgets(buff, sizeof(buff) - 1, fp) != NULL) {
    int num = sscanf(buff, fmt,
		     iface, &net_addr, &gate_addr,
		     &iflags, &refcnt, &use, &metric, &mask_addr,
		     &mss, &window, &irtt);
    if (num < 10 || !(iflags & RTF_UP) || (iflags & RTF_REJECT))
      continue;

    wf_ipaddr net_target, net_gateway, net_mask;

    net_target.set(net_addr);
    net_gateway.set(gate_addr);
    net_mask.set(mask_addr);

    if (!(iflags & RTF_GATEWAY) || net_addr != 0)
      continue;

#if 0
    /* Decode the flags. */
    char flags[64];
    flags[0] = '\0';
    if (iflags & RTF_UP)
      strcat(flags, "U");
    if (iflags & RTF_GATEWAY)
      strcat(flags, "G");
    if (iflags & RTF_REJECT)
      strcpy(flags, "!");
    if (iflags & RTF_HOST)
      strcat(flags, "H");
    if (iflags & RTF_REINSTATE)
      strcat(flags, "R");
    if (iflags & RTF_DYNAMIC)
      strcat(flags, "D");
    if (iflags & RTF_MODIFIED)
      strcat(flags, "M");
    if (iflags & RTF_DEFAULT)
      strcat(flags, "d");
    if (iflags & RTF_ALLONLINK)
      strcat(flags, "a");
    if (iflags & RTF_ADDRCONF)
      strcat(flags, "c");
    if (iflags & RTF_NONEXTHOP)
      strcat(flags, "o");
    if (iflags & RTF_EXPIRES)
      strcat(flags, "e");
    if (iflags & RTF_CACHE)
      strcat(flags, "c");
    if (iflags & RTF_FLOW)
      strcat(flags, "f");
    if (iflags & RTF_POLICY)
      strcat(flags, "p");
    if (iflags & RTF_LOCAL)
      strcat(flags, "l");
    if (iflags & RTF_MTU)
      strcat(flags, "u");
    if (iflags & RTF_WINDOW)
      strcat(flags, "w");
    if (iflags & RTF_IRTT)
      strcat(flags, "i");
#endif

    // cout << net_target << " " << net_gateway << " " << net_mask << " " << flags << " " << iface << endl;
    lst.push_back(pair<wf_ipaddr, string>(net_gateway, iface));
  }

  free(fmt);
  fclose(fp);
  return 0;
}

#endif /* linux */

int
wf_route_defgateways_get(list<pair<wf_ipaddr, string> >& lst) {
#ifdef linux
  return wf_route_defgateways_get_linux(lst);
#else
  return 1;
#endif
}
