/*
 * WallFire -- a comprehensive firewall administration tool.
 * 
 * Copyright (C) 2001 Herv Eychenne <rv@wallfire.org>
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * 
 */

using namespace std;

#include <stdio.h>
#include <string.h>
#include <errno.h>

#include "wfnetobjs.h"
#include "output_common.h"
#include "defs.h"


static bool
iface_check(const wf_iface& iface,
	    const string& filename, unsigned int lineno) {
  if (iface.network.belong(iface.ipaddr) == false) {
    fprintf(stderr, _("Error: %s:%i: interface %s: address %s does not belong to network %s\n"),
	    filename.c_str(), lineno,
	    iface.name.c_str(), iface.ipaddr.tostr().c_str(),
	    iface.network.tostr().c_str());
    return false;
  }
  if (iface.broadcast.isdefined() &&
      iface.broadcast != iface.network.broadcast()) {
    fprintf(stderr, _("Warning: %s:%i: interface %s: broadcast address %s is different from the broadcast address (%s) of network %s\n"),
	    filename.c_str(), lineno,
	    iface.name.c_str(), iface.broadcast.tostr().c_str(),
	    iface.network.broadcast().tostr().c_str(),
	    iface.network.tostr().c_str());
  }
  return true;
}

/* parsing is really ugly ALL@@2 */
bool
debiface_file_tohost(wf_host& host, FILE* file, const string& filename) {
  if (file == NULL)
    return false;

  char line[1024];
  unsigned int lineno = 0, iface_lineno = 0;
  wf_iface* iface = NULL;

  while (fgets(line, sizeof(line), file) != NULL) {
    char *tmp = line, *comment, *word;

    lineno++;
    line[strlen(line) - 1] = '\0';

    for (comment = line; *comment != '\n'; comment++) {
      if (*comment == '#') {
	*comment = '\0';
	break;
      }
    }
    
    // cerr << '[' << line << ']' << '\n';
    while ((word = strsep(&tmp, "\t ")) != NULL) {
      char *name, *type;

      // cerr << '|' << word << '|' << '\n';
      if (*word == '\0')
	continue;

      if (!strcmp(word, "iface")) {
	if (iface != NULL) {
	  /* The interface is now complete: do some checking. */
	  if (iface_check(*iface, filename, iface_lineno) == false) {
	    /* Do some cleaning... RV@@6 */
	    return false;
	  }
	  host.iface_add(*iface);
	}

	iface = new wf_iface();
	if (iface == NULL)
	  return false; /* Do some cleaning... RV@@6 */

	iface_lineno = lineno;
	if ((name = strsep(&tmp, "\t ")) != NULL) {
	  //	  fprintf(stderr, "interface %s\n", name);
	  iface->name = name;
	}
	if ((type = strsep(&tmp, "\t ")) != NULL &&
	    !strcmp(type, "inet")) {
	  if ((type = strsep(&tmp, "\t ")) != NULL) {
	    if (!strcmp(type, "static"))
	      break;
	    else if (!strcmp(type, "loopback")) {
	      iface->ipaddr.set("127.0.0.1");
	      iface->network.network_set("127.0.0.0/8");
	    }
	  }
	}
      }
      else if (!strcmp(word, "address")) {
	char *addr;
	if ((addr = strsep(&tmp, "\t ")) != NULL)
	  iface->ipaddr.set(addr);
      }
      else if (!strcmp(word, "network")) {
	char *addr;
	if ((addr = strsep(&tmp, "\t ")) != NULL)
	  iface->network.network.set(addr);
      }
      else if (!strcmp(word, "netmask")) {
	char *addr;
	if ((addr = strsep(&tmp, "\t ")) != NULL)
	  iface->network.netmask.set(addr);
      }
      else if (!strcmp(word, "broadcast")) {
	char *addr;
	if ((addr = strsep(&tmp, "\t ")) != NULL)
	  iface->broadcast.set(addr);
      }
      else if (!strcmp(word, "gateway")) {
	/* ignored for the moment RV@@2 */
	break;
      }
      else if (!strcmp(word, "auto") ||
	       !strcmp(word, "pointopoint") ||  !strcmp(word, "media") ||
	       !strcmp(word, "hwaddress") ||  !strcmp(word, "mtu") ||
	       !strcmp(word, "up") || !strcmp(word, "down") ||
	       !strcmp(word, "pre-up") || !strcmp(word, "post-down"))
	break; /* ignored */
      else {
	fprintf(stderr, _("Error: %s:%i: unknown keywork `%s'\n"),
		filename.c_str(), lineno, word);
	return false;
      }
    }
  }

  if (iface != NULL) {
    /* The interface is now complete: do some checking. */
    if (iface_check(*iface, filename, iface_lineno) == false) {
      /* Do some cleaning... RV@@6 */
      return false;
    }
    host.iface_add(*iface);
  }

  return true;
}

bool
debiface_file_tohost(wf_host& host, const string& filename) {
  FILE* file = fopen(filename.c_str(), "r");
  if (file == NULL) {
    cerr << _("Error: ") << filename << ": " << strerror(errno) << '\n';
    return false;
  }

  if (debiface_file_tohost(host, file, filename) == false) {
    cerr << _("Error: ") << filename << ": " << _("parsing failed") << '\n';
    return false;
  }
  return true;
}

string
iface_todebiface_file(const wf_iface& iface) {
  string str;

  if (iface.name.empty()) {
    cerr << _("Error: interface has no name") << '\n';
    return str;
  }

  if (iface.name.rfind(':') == string::npos) {
    if (iface.network.name.empty())
      str += string_printf(_("# interface %s"), iface.name.c_str());
    else
      str += string_printf(_("# interface %s to network %s"),
			   iface.name.c_str(), iface.network.name.c_str());
  }
  else {
    if (iface.network.name.empty())
      str += string_printf(_("# interface %s (aliased)"),
			   iface.name.c_str());
    else
      str += string_printf(_("# interface %s (aliased) to network %s"),
			   iface.name.c_str(), iface.network.name.c_str());
  }
  str += '\n';
  
  str += wf_shell_comment(iface.comment) +=
    "iface " + iface.name + " inet ";

  if (iface.name == "lo") {
    str += "loopback\n";
    return str;
  }

  str += "static\n";

  if (iface.ipaddr.isdefined())
    str += "\taddress " + iface.ipaddr.tostr();
  else
    str += "#\taddress (unknown)";
  str += '\n';

  if (iface.network.network.isdefined())
    str += "\tnetwork " + iface.network.network.tostr();
  else
    str += "#\tnetwork (unknown)";
  str += '\n';

  if (iface.network.netmask.isdefined())
    str += "\tnetmask " + iface.network.netmask.tostr() + " # /" +
      string_printf("%i", iface.network.netmask_tobitmask());
  else
    str += "#\tnetmask (unknown)";
  str += '\n';

  if (iface.broadcast.isdefined()) {
    str += "\tbroadcast " + iface.broadcast.tostr();
    if (iface.broadcast != iface.network.broadcast()) {
      str += " # " +
	string_printf(_("different from %s (theorical network broadcast address)"),
		      iface.network.broadcast().tostr().c_str());
    }
  }
  else
    str += "#\tbroadcast " + iface.network.broadcast().tostr();
  str += '\n';

  return str;
}


string
host_todebiface_file(const wf_host& host) {
  string str;
  list<wf_iface>::const_iterator first = host.ifaces.begin(),
    last = host.ifaces.end();

  str += wf_shell_comment(host.comment) + '\n';

  for (; first != last; ++first)
    str += iface_todebiface_file(*first) + '\n';
  return str;
}
