/*
  libwftk - Worldforge Toolkit - a widget library
  Copyright (C) 2002 Malcolm Walker <malcolm@worldforge.org>
  Based on code copyright  (C) 1999-2002  Karsten Laux

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.
  
  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.
  
  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the
  Free Software Foundation, Inc., 59 Temple Place - Suite 330,
  Boston, MA  02111-1307, SA.
*/


#ifndef _APP_H
#define _APP_H

#include <string>
#include <exception>
#include <queue>
#include <map>

#if SIGC_MAJOR_VERSION == 1 && SIGC_MINOR_VERSION == 0
#include <sigc++/signal_system.h>
#else
#include <sigc++/signal.h>
#include <sigc++/object.h>
#endif

#include <SDL/SDL_types.h>
#if 0
#include <SDL/SDL_keysym.h> // for SDLK_LAST
#endif

#include <wftk/debug.h>

namespace wftk {

/// an exception thrown when the library experiences a fatal error
class Fatal : public std::exception
{
 public:
  /// constructor takes the reason for the fatal error
  Fatal(const std::string& reason);
  virtual ~Fatal() throw() {}

  /// the reason this error was thrown
  virtual const char* what() const throw() {return reason_.c_str();}
 private:
  std::string reason_;
};

/** Toplevel application class
 * 
 * Extend this class to create your application.
 **/
class Application : virtual public SigC::Object
{
 public:
  typedef std::map<std::string,Debug::Mask> DebugMap;

  /// Constructor.
  Application(int &argc, char **&argv,
	const DebugMap& other_flags = DebugMap(), Uint32 update = 50);
  /// Default destructor.
  ~Application();

  /// emitted after events are handled, but before drawing
  SigC::Signal0<void> update;
  /// signal to tell RootWindow to draw, nothing else should connect to this
  SigC::Signal0<void> draw;
  /// emitted in destructor before SDL shutdown
  SigC::Signal0<void> destroyed;
  
  /// run mainloop
  int exec();
  /// Halt execution and close window
  virtual void quit(int exitcode = 0);
  /// Return a slot which binds in the appropriate exit code
  SigC::Slot0<void> quitSlot(int exitcode = 0);
  /// Halt execution and close window, bypassing quit handler
  void abort();

  /// true during normal application processing
  bool running() {return running_;}

  /// pointer to this application (safe pointer)
  /**
   * If Application::instance() is non-null, SDL is guaranteed
   * to have been initialized. You still have to check particular
   * subsystems with SDL_WasInit().
   **/
  static Application* instance() {return instance_;}

  /// handle all events
  bool keepAlive();

  /// runs until var == wait_val
  void waitFor(bool& var, bool wait_val = true);
  /// runs until func returns wait_val
  void waitFor(SigC::Slot0<bool> func, bool wait_val = true);

  /// handle the next event in the queue
  void handleEvent(bool can_block = false);

  /** Set mainloop delay time.
   *
   * \param n delay time in milliseconds
   * 
   *  Set the maximum time to spend before querying SDL for events.
   */
  void setIdleTime(Uint32 n) {event_pump_ = n;}

  class Event
  {
   public:
    virtual ~Event() {}

    virtual void operator()() = 0;
  };

  // An event which is associated with data that may
  // be destroyed before it goes off. To use it,
  // your class C should have a currentEvent_ member
  // which it initializes to zero. Make FloatingEvent<C>
  // a friend of C, and call FloatingEvent::clear() in
  // your destructor if currentEvent_ is non-null.
  // Any particular piece of data can only have one of
  // these events at a time.
  template<class C>
  class FloatingEvent : public Event
  {
   public:
    FloatingEvent(C& c) : c_(&c) {c.currentEvent_ = this;}
    virtual ~FloatingEvent() {if(c_) c_->currentEvent_ = 0;}

    void clear() {c_ = 0;}

    C* data() const {return c_;}

   private:
    C* c_;
  };

  /// Takes an Event which it deletes after handling
  void pushEvent(Event* event) {if(event) queue_.push(event);}

 protected:

#if 0
  ///
  enum { 
    KEY_TABLE_SIZE = SDLK_LAST , 
    KEY_DELAY = 10, // rough 500ms
    KEY_REPEAT = -1 // 0 means 100ms, -1 means 50ms
  };
#endif

 private:

  typedef std::queue<Event*> EventQueue;
  EventQueue queue_;

  // list of event sources to poll
  enum Source {
    POLLING,
    TIMERS,
    GUI, // SDL events
    UPDATE,
    DRAW,
    NUM_SOURCES,
    FIRST_SOURCE = 0 // for initialization, looping
  } next_source_;

  // functions called in the constructor

  ///
  void parseArgs(int& argc, char**& argv, const DebugMap&);
  ///
  void loadResources();

  ///
  Uint32 event_pump_;
  ///
  int exitcode_;
  ///
  bool running_;

  static Application* instance_;
};

}
  
#endif // !_APP_H
  
