/* normalmode.cc
 * This file belongs to Worker, a filemanager for UNIX/X11.
 * Copyright (C) 2001-2004 Ralf Hoffmann.
 * You can contact me at: ralf@boomerangsworld.de
 *   or http://www.boomerangsworld.de/worker
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
/* $Id: normalmode.cc,v 1.273 2004/12/15 22:46:14 ralf Exp $ */

#include "normalmode.h"
#include "worker_locale.h"
#include "wconfig.h"
#include "worker.h"
#include <aguix/bevelbox.h>
#include "wpucontext.h"
#include "execlass.h"
#include "configheader.h"
#include "configparser.tab.h"
#include "worker_shm.h"
#include "nmrowdata.h"
#include "basic_actions.h"

const char *NormalMode::type="NormalMode";

int NormalMode::maxfilestrlen=1;
int NormalMode::maxdirstrlen=1;
int NormalMode::maxbytesstrlen=1;
int NormalMode::maxwbytesstrlen=1;
int NormalMode::maxnochanges=0;

#ifdef WANT_THREADS
void *NM_slavestart( void *arg )
{
  NormalMode *nm = (NormalMode*)arg;
  nm->slavehandler();
  return NULL;
}
#endif

NormalMode::NormalMode(Lister *tparent):ListerMode(tparent)
{
  x=0;
  y=0;
  w=1;
  h=1;
  ce=NULL;
  stored_cacheentry = -1;
  cache=new List();
  filters=new List();
  showHidden=true;
  req=new Requester(aguix);
  searchmodeon=false;
  searchmode_ignore_case = false;
  lastsearch=NULL;
  sortmode=SORT_NAME;
  filterlfdnr=0;
  lv=NULL;
  lastrow=-1;
  showfreespace=true;
  namestr=NULL;
  freespacestr = NULL;
  lastfsupdate=0;
  updatetime=10;
  parentlister->getWorker()->PS_setLifetime( (double)updatetime );

  lasteagain = false;
  
#ifdef WANT_THREADS
  // to preventing slave access
  reclist = new ft_rec_list();
  recreslist = new ft_recres_list();

  slave.filetypes = new List();
  slave.filetype_ex.lock();
  slave.running = 0;
  slave.status = THREAD_NOP;
  slave.order = THREAD_REINIT;
  if ( pthread_create( &slave.th, NULL, NM_slavestart, this ) == 0 ) {
    thread_usage = true;
  } else {
    fprintf( stderr, "Worker: Can't create thread, disabling thread-support!\n" );
    thread_usage = false;
  }

  if ( thread_usage == true ) {
    slave.filetype_ex.wait();
  }
  ft_list_update();
  slave.filetype_ex.signal();
  slave.filetype_ex.unlock();
#else
  thread_usage = false;
#endif
  
  finish_done = false;

  chownUserList = NULL;
  chownGroupList = NULL;

  visChanged = true;
  oldlvy = -1;
  
  busyFlag = 0;
  lastBusyFlagUpdate = time( NULL );
}

NormalMode::~NormalMode()
{
#ifdef WANT_THREADS
  switchOrder( THREAD_EXIT );
  if ( thread_usage == true ) {
    pthread_join( slave.th, NULL );
  }
  ft_list_clear();
  delete slave.filetypes;
  ft_rec_list_clear();
  ft_recres_list_clear();
  delete reclist;
  delete recreslist;
#endif

  if(searchmodeon==true) finishsearchmode();
  deleteCache();
  deleteFilters();
  delete cache;
  delete filters;
  delete req;
  if(namestr!=NULL) _freesafe(namestr);
  if ( freespacestr != NULL ) _freesafe( freespacestr );
}

void NormalMode::messageHandler(AGMessage *msg)
{
  int tx,ty,tw,th;
  bool ma=false;
  switch(msg->type) {
    case AG_STRINGGADGET_OK:
      if(searchmodeon==false) {
        if(msg->stringgadget.sg==sg) {
          enterDir(sg->getText());
        }
        break;
      }
    case AG_STRINGGADGET_CANCEL:
      if(searchmodeon==false) {
        if(msg->stringgadget.sg==sg) {
          if ( getCurrentDir() != NULL ) sg->setText( getCurrentDir() );
        }
        break;
      }
    case AG_STRINGGADGET_DEACTIVATE:
      if(searchmodeon==true) {
        finishsearchmode();
      }
      break;
    case AG_STRINGGADGET_CONTENTCHANGE:
      if(searchmodeon==true) {
        if(showfirstentry(sg->getText())==true) {
          if(lastsearch!=NULL) _freesafe(lastsearch);
          else debugmsg("oops1\n");
          lastsearch=dupstring(sg->getText());
        } else {
          sg->setText(lastsearch);
        }
      }
      break;
    case AG_SIZECHANGED:
      parentlister->getGeometry(&tx,&ty,&tw,&th);
      reconf(tx,ty,tw,th);
      break;
    case AG_STRINGGADGET_ACTIVATE:
      if(msg->stringgadget.sg==sg) ma=true;
      break;
    case AG_BUTTONCLICKED:
      if(msg->button.button==hb[0]) {
        showcache(-1);
        ma=true;
      } else if(msg->button.button==hb[1]) { 
        showcache(1);
        ma=true;
      } else if ( msg->button.button == parentb ) { 
        parent();
        ma = true;
      }
      break;
    case AG_FIELDLV_PRESSED:
      if(msg->fieldlv.lv==lv) ma=true;
      break;
    case AG_FIELDLV_ONESELECT:
      if(msg->fieldlv.lv==lv) {
        if(lastrow==-1) {
          lastrow=msg->fieldlv.row;
          gettimeofday(&lastclick,NULL);
        } else {
          if(msg->fieldlv.row==lastrow) {
            struct timeval acttime;
            gettimeofday(&acttime,NULL);
            if ( aguix->isDoubleClick( &acttime, &lastclick ) == true ) {
              //TODO: Wir haben einen Doppelclick
              // Aus lvc->data==fe->nr bekommen wir das fe (nr beschreibt Position)
              if(ce!=NULL) {
                ArrayList *f1=ce->verz->getFiles();
                FileEntry *tfe=(FileEntry*)f1->getElementAt( lv->getData( lastrow ) );
                startAction(tfe);
              }
              // when had a doubleclick the next click on the same should not be a
              // doubleclick again
              lastrow=-1;
            } else {
              lastrow=msg->fieldlv.row;
              gettimeofday(&lastclick,NULL);
            }
          } else {
            lastrow=msg->fieldlv.row;
            gettimeofday(&lastclick,NULL);
          }
        }
      }
      break;
    case AG_FIELDLV_HEADERCLICKED:
      if ( ( msg->fieldlv.lv == lv ) &&
           ( msg->fieldlv.button == Button1 ) ) {
        changeSortModeForField( msg->fieldlv.row );
      }
      break;
  }
  if(ma==true) parentlister->makeActive();
  showFreeSpace(true);
}

void NormalMode::on()
{
  parentlister->getGeometry(&x,&y,&w,&h);
  int side=parentlister->getSide();
  AGUIXFont *afont=aguix->getFont(wconfig->getFont(2+side));
  int sgh=afont->getCharHeight()+8;
  int hbw=afont->getCharWidth()+10;
  int m;

  parentb = new Button( aguix, x, y, 2 * afont->getCharWidth() + 10, "..", 1, 0, 5 );
  hb[0] = new Button( aguix, x, y, hbw, "<", 1, 0, 2 );
  hb[1] = new Button( aguix, x, y, hbw, ">", 1, 0, 3 );

  m = parentb->getWidth() + hb[0]->getWidth() + hb[1]->getWidth();
  if ( w < ( 10 + m ) ) w = 10 + m;
  if ( h < ( 10 + sgh ) ) h = 10 + sgh;

  lv=(NMFieldListViewDND*)parentawindow->add(new NMFieldListViewDND(aguix,x,y,w,h-sgh,1));
  lv->setAcceptFocus( false );
  lv->setMBG(wconfig->getListerBG());
  m=(wconfig->getHBarTop(side)==true)?1:2;
  lv->setHBarState(m);
  m=(wconfig->getVBarLeft(side)==true)?1:2;
  lv->setVBarState(m);
  m=wconfig->getHBarHeight(side);
  lv->setHBarHeight(m);
  m=wconfig->getVBarWidth(side);
  lv->setVBarWidth(m);
  lv->setShowHeader( wconfig->getShowHeader( side ) );
  lv->setFont(wconfig->getFont(2+side));
  lv->setHeaderFG( wconfig->getLVHeader( 0 ) );
  lv->setHeaderBG( wconfig->getLVHeader( 1 ) );
  
  parentb->move( x + ( ( side == 0 ) ? 0 : ( w - parentb->getWidth() ) ), y + h - sgh );
  parentb->resize( parentb->getWidth(), sgh );
  parentawindow->add( parentb );
  parentb->setAcceptFocus( false );
  parentb->setFont(wconfig->getFont(2+side));
  
  hb[0]->move( x + ( ( side == 0 ) ? parentb->getWidth() : 0 ), y + h - sgh );
  hb[0]->resize( hb[0]->getWidth(), sgh );
  parentawindow->add( hb[0] );
  hb[0]->setAcceptFocus( false );
  hb[0]->setFont(wconfig->getFont(2+side));
  
  hb[1]->resize( hb[1]->getWidth(), sgh );
  hb[1]->move( x + w - hb[1]->getWidth() - ( ( side == 0 ) ? 0 : parentb->getWidth() ), y + h - sgh );
  parentawindow->add( hb[1] );
  hb[1]->setAcceptFocus( false );
  hb[1]->setFont(wconfig->getFont(2+side));
  sg = (StringGadget*)parentawindow->add( new StringGadget( aguix,
                                                            hb[0]->getX() + hb[0]->getWidth(),
                                                            y + h - sgh,
                                                            hb[1]->getX() - hb[0]->getX() - hb[0]->getWidth(), "", 4 ) );
  sg->setAcceptFocus( false );
  sg->resize( sg->getWidth(), sgh );
  sg->setFont(wconfig->getFont(2+side));
  parentlister->setActiveMode(this);
  setName();
  lv->setSelectHandler(this);

  setupLVFields();
  
  if ( stored_cacheentry >= 0 )
    showCacheAbs( stored_cacheentry );
}

void NormalMode::off()
{
  finishsearchmode();

  // store the current pos for reactivating
  if ( ( ce != NULL ) && ( lv != NULL ) ) {
    ce->pos = lv->getYOffset();
    ce->xpos = lv->getXOffset();
    stored_cacheentry = cache->getIndex( ce );
  }
  ce = NULL;

  parentawindow->remove(lv);
  delete lv;
  lv = NULL;
  parentawindow->remove(hb[0]);
  delete hb[0];
  hb[0] = NULL;
  parentawindow->remove(hb[1]);
  delete hb[1];
  hb[1] = NULL;
  parentawindow->remove(sg);
  delete sg;
  sg = NULL;
  parentawindow->remove( parentb );
  delete parentb;
  parentb = NULL;
  parentlister->setActiveMode(NULL);
  parentlister->setName("");
  if ( parentlister->getFocus() == true )
    parentlister->getWorker()->setTitle(NULL);
  lv=NULL;
}

void NormalMode::reconf( int nx, int ny, int nw, int nh, bool force )
{
  if ( lv != NULL ) {
    if ( ( nw != w ) || ( nh != h ) || ( nx != x ) || ( ny != y ) || ( force == true ) ) {
      int side = parentlister->getSide();
      int sgh = sg->getHeight();

      w = nw;
      h = nh;
      x = nx;
      y = ny;
      if ( w < ( 10 + parentb->getWidth() + hb[0]->getWidth() + hb[1]->getWidth() ) )
        w = 10 + parentb->getWidth() + hb[0]->getWidth() + hb[1]->getWidth();
      if ( h < ( 10 + sgh ) ) h = 10 + sgh;
      lv->resize( w, h - sgh );
      lv->move( x, y );

      parentb->move( x + ( ( side == 0 ) ? 0 : ( w - parentb->getWidth() ) ), y + h - sgh );
      hb[0]->move( x + ( ( side == 0 ) ? parentb->getWidth() : 0 ), y + h - sgh );
      hb[1]->move( x + w - hb[1]->getWidth() - ( ( side == 0 ) ? 0 : parentb->getWidth() ), y + h - sgh );

      sg->move( hb[0]->getX() + hb[0]->getWidth(), y + h - sgh );
      sg->resize( hb[1]->getX() - hb[0]->getX() - hb[0]->getWidth(), sgh );
    }
  }
}

void NormalMode::activate()
{
  int row;
  ArrayList *l;
  if(ce!=NULL) {
    if(ce->activefe!=NULL) {
      l=ce->verz->getFiles();
      row = 0;
      while( row < lv->getElements() ) {
        if( lv->getData( row ) == ce->activefe->nr) break;
        row++;
      }
      if( row < lv->getElements() ) lv->setActiveRow( row );
    }
  }
  showCacheState();
  parentlister->getWorker()->setTitle(sg->getText());
}

void NormalMode::deactivate()
{
  int row=lv->getActiveRow();
  if ( row >= 0 ) {
    lv->setActiveRow( -1);
  }
  finishsearchmode();
  parentlister->getWorker()->setTitle(NULL);
}

void NormalMode::buildLister()
{
  if(ce==NULL) return;
  int side=parentlister->getSide();
  int id;
  const std::vector<WorkerTypes::listcol_t> *dis;

  if(lv==NULL) return;
  lv->setSize(ce->files[0]+ce->dirs[0]+1);
  lv->setSizeDNDText(ce->files[0]+ce->dirs[0]+1);
  ArrayList *al=ce->verz->getFiles();
  id=al->initEnum();
  FileEntry *fe=(FileEntry*)al->getFirstElement(id);
 
  lv->setActiveRowQ( -1 );
 
  int tpos;
 
  tpos = 0;
  dis = wconfig->getVisCols( side );
  while(fe!=NULL) {
    if(fe->use==true) {
      fe->setLVC_DND( lv, tpos, dis );
      lv->setData( tpos, fe->nr);
      if(fe->select==true) lv->setSelectQ( tpos, true ); else lv->setSelectQ( tpos, false );
      if(fe==ce->activefe) {
        // only make the lvc active if we are active
        if(parentlister->isActive()==true) lv->setActiveRowQ( tpos );
      }
      tpos++;
    }
    fe=(FileEntry*)al->getNextElement(id);
  }
  al->closeEnum(id);
  lv->redraw();
  
  visChanged = true;
}

void NormalMode::deleteCache()
{
  NMCacheEntry *tce=(NMCacheEntry*)cache->getFirstElement();
  while(tce!=NULL) {
    delete tce;
    tce=(NMCacheEntry*)cache->getNextElement();
  }
}

int NormalMode::enterDir(const char *dirstr)
{
  int maxs=wconfig->getCacheSize();
  int caches=cache->size();
  char *tstr;
  Verzeichnis *tverz=NULL;
  NMCacheEntry *tce=NULL;
  int exentry=-1;
  ArrayList *oldfiles,*newfiles;
  int oldid,newid;
  FileEntry *ofe,*nfe,*oldact;
  int comperg;
  bool reset;
  int returnvalue=0;  // 0 = OK
                      // 1 = Not found
                      // 2 = not found, but lv cleared (was actually)
#if 0
  char *textstr;
#endif
  std::string str1;

  if(lv==NULL) return 1;

  if ( dirstr == NULL ) return 1;
  if ( dirstr[0] != '/' ) {
    str1 = "./";
    str1 += dirstr;
    tstr = HandlePathExt( str1.c_str() );
  } else {
    tstr = HandlePath( dirstr );
  }
  
  if ( access( tstr, R_OK | X_OK ) != 0 ) {
    _freesafe( tstr );
    return 1;
  }

  parentlister->getWorker()->setWaitCursor();
  aguix->Flush();
#ifdef DEBUG
  printf("enterDir: %s\n",dirstr);
#endif
  
  finishsearchmode();
  
#ifdef WANT_THREADS
  // clear the reclist so no wrong pointers are in the list
  ft_rec_list_clear();
#endif
  
  if(caches>0) {
    int i=0;
    tce=(NMCacheEntry*)cache->getFirstElement();
    while(tce!=NULL) {
      if(strcmp(tce->verz->getDir(),tstr)==0) {
        exentry=i;
#ifdef DEBUG
        printf("enterDir: cache hit\n");
#endif
        break;
      }
      tce=(NMCacheEntry*)cache->getNextElement();
      i++;
    }
  }
  tverz=new Verzeichnis();
  if(tverz->readDir(tstr)==0) {
  } else {
    delete tverz;
    tverz=NULL;
  }
  if(tverz!=NULL) {

#if 0
    // at the moment I will get a requester for each invalid entry
    // directly in readDir but I don't remove this code here because
    // perhaps I need this code later again (one can never know)
    if(tverz->getInvalidEntries()->size()>0) {
      // show requester
      ArrayList *invalid_entries=tverz->getInvalidEntries();
      int id=invalid_entries->initEnum();
      FileEntry *tfe;
      
      tfe=(FileEntry*)invalid_entries->getFirstElement(id);
      while(tfe!=NULL) {
        textstr=(char*)_allocsafe(strlen(catalog.getLocale(203))+strlen(tfe->name)+1);
        sprintf(textstr,catalog.getLocale(203),tfe->name);
        req->request(catalog.getLocale(125),textstr,catalog.getLocale(11));
        _freesafe(textstr);
        tfe=(FileEntry*)invalid_entries->getNextElement(id);
      }
      invalid_entries->closeEnum(id);
    }
#endif

#ifdef DEBUG
    printf("enterDir: readdir ok\n");
#endif
    if(tce==NULL) {
      // no cache entry found
      if(cache->size()<maxs) {
        // da ist noch Platz im Cache
#ifdef DEBUG
        printf("enterDir: new cache entry\n");
#endif
      } else {
        tce=(NMCacheEntry*)cache->getElementAt(0);
        delete tce;
	if ( ce == tce ) ce = NULL;
        cache->removeElementAt(0);
#ifdef DEBUG
        printf("enterDir: cache is full\n");
#endif
      }
      tce=new NMCacheEntry();
      tce->verz=tverz;
      cache->addElement(tce);
      exentry=cache->size()-1;
      tce->restartcheck();
      newfiles=tce->verz->getFiles();
      if(newfiles!=NULL) tce->activefe=(FileEntry*)newfiles->getFirstElement();
      //^^^ should never happen, but to be sure
      tce->checkfordcd();
    } else {
#ifdef DEBUG
      printf("enterDir: using existing cache, updating values\n");
#endif
      // put tce at the end of the cache to realize Not-Recently-Used
      cache->removeElement(tce);
      cache->addElement(tce);
      exentry=cache->size()-1;
      // insert old values into to new tverz
      // first sort both for names so we can easily (and fast) get old values
      tverz->sort(SORT_NAME|SORT_DIRMIXED);
      tce->verz->sort(SORT_NAME|SORT_DIRMIXED);
      oldfiles=tce->verz->getFiles();
      newfiles=tverz->getFiles();
      oldid=oldfiles->initEnum();
      newid=newfiles->initEnum();
      ofe=(FileEntry*)oldfiles->getFirstElement(oldid);
      nfe=(FileEntry*)newfiles->getFirstElement(newid);
      oldact=tce->activefe;
      tce->activefe = nfe;
      while(nfe!=NULL) {
        if(oldact!=NULL)
          if(strcmp(nfe->fullname,oldact->fullname)==0)
            tce->activefe=nfe;
        while(ofe!=NULL) {
          comperg=strcmp(ofe->fullname,nfe->fullname);
          if(comperg>0) break;
          else if(comperg==0) {
            // found an entry fit to the actual
            if(ofe->select==true) nfe->select=true;
            else nfe->select=false;
            if(ofe->isDir()==true)
              if(ofe->dirsize>=0) nfe->dirsize=ofe->dirsize;
            if(ofe->filetype!=NULL) nfe->filetype=ofe->filetype;

	    if ( ofe->getCustomColors() == true ) {
	      nfe->setCustomColors( true );
	      nfe->setColor( 0, ofe->getColor( 0 ) );
	      nfe->setColor( 1, ofe->getColor( 1 ) );
	    }
          }
          ofe=(FileEntry*)oldfiles->getNextElement(oldid);
          if(strcmp(nfe->name,"..")==0) break; // the ".." is always first in new and old so stop here
                                               // because following elements can be "smaller" then ".."
                                               // f.i. "#343" < ".."
        }
        if(ofe==NULL) break; // there are no further old files => no need to continue
        nfe=(FileEntry*)newfiles->getNextElement(newid);
      }
      oldfiles->closeEnum(oldid);
      newfiles->closeEnum(newid);
      delete tce->verz;
      tce->verz=tverz;
#ifdef DEBUG
      printf("enterDir: update values finished\n");
#endif
    }
    if ( tce->activefe != NULL) {
      if ( tce->activefe->use == false ) {
        // this will never happen because the fallback is
        // the first entry in the filelist which is always ".."
        // but this doesn't hurt and when get my great ideas...
        tce->activefe = NULL;
      }
    }
    if(ce!=NULL) {
      ce->pos=lv->getYOffset();
      ce->xpos=lv->getXOffset();
    }
    ce=tce;
    tce->actsortmode = ~0;
    tce->actfilterlfdnr = filterlfdnr - 1;
    tce->actshowhidden = ! showHidden;
    checkRebuild( tce, true );
    lv->setYOffset(ce->pos);
    lv->setXOffset(ce->xpos);
    sg->setText(tstr);
#ifdef DEBUG
    printf("enterDir: create successfully finished\n");
#endif
  } else {
    // can't read the dir
#ifdef DEBUG
    printf("enterDir: readdir failed\n");
#endif
    returnvalue=1;
    reset=true;
    if(tce!=NULL) {
      //TODO: Verzeichnis gibt es nicht mehr, also aus Cache entfernen
      // klappt jetzt so
      if(tce==ce) {
#ifdef DEBUG
        printf("enterDir: remove current cache entry\n");
#endif
        // it's the current cacheentry
        reset=false;
        returnvalue=2;
        cache->removeElement(ce);
        delete ce;
        ce=NULL;
        lv->setSize(0);
        lv->redraw();
        sg->setText("");
      } else {
#ifdef DEBUG
        printf("enterDir: remove cache entry\n");
#endif
        cache->removeElement(tce);
        delete tce;
      }
    }
    if(reset==true) {
      // wiederherstellen
#ifdef DEBUG
      printf("enterDir: restore\n");
#endif
      if(ce!=NULL) {
        _freesafe(tstr);
        tstr=HandlePath(ce->verz->getDir());
        sg->setText(tstr);
      } else sg->setText("");
    }
  }
  _freesafe(tstr);
#ifdef DEBUG
  printf("enterDir: finished\n\n");
#endif
  if ( parentlister->getFocus() == true )
    parentlister->getWorker()->setTitle( sg->getText() );
  parentlister->getWorker()->unsetWaitCursor();
  return returnvalue;
}

void NormalMode::showcache(int direction)
{
  NMCacheEntry *tce;
  int cs=cache->size();
  int i=0,id;
  if(cs<1) return;

  finishsearchmode();
  
  if(ce!=NULL) {
    ce->pos=lv->getYOffset();
    ce->xpos=lv->getXOffset();
  }
  int t = cache->getIndex( ce );
  if ( t < 0 ) t = 0;
  
  id=cache->initEnum();
  do {
    i++;
    t+=direction;
    if(t>=cache->size()) t=0;
    else if(t<0) t=cache->size()-1;
    tce=(NMCacheEntry*)cache->getElementAt(id,t);
    if(tce!=NULL) {
      if(enterDir(tce->verz->getDir())==0) {
        break;
      }
    }
  } while(i<cs);
  cache->closeEnum(id);
}

NM_Filter::NM_Filter()
{
  pattern=NULL;
  check=0;
}

NM_Filter::~NM_Filter()
{
  if(pattern!=NULL) _freesafe(pattern);
}

void NM_Filter::setPattern(const char *np)
{
  if(pattern!=NULL) _freesafe(pattern);
  if(np!=NULL) {
    pattern=dupstring(np);
  } else {
    pattern=NULL;
  }
}

char *NM_Filter::getPattern()
{
  return pattern;
}

void NM_Filter::setCheck(int nc)
{
  if((nc<0)||(nc>2)) return;
  check=nc;
}

int NM_Filter::getCheck()
{
  return check;
}

void NormalMode::deleteFilters()
{
  NM_Filter *fil=(NM_Filter*)filters->getFirstElement();
  while(fil!=NULL) 
  {
    delete fil;
    fil=(NM_Filter*)filters->getNextElement();
  }
}

void NormalMode::checkFilters( NMCacheEntry *tce )
{
  int fid,id;
  
  if ( tce == NULL ) return;
  
  ArrayList *fl=tce->verz->getFiles();
  id=fl->initEnum();
  FileEntry *fe=(FileEntry*)fl->getElementAt(id,1);
  NM_Filter *fil;
  int ins=0,outs=0;
  bool isIn;
  fid=filters->initEnum();
  fil=(NM_Filter*)filters->getFirstElement(fid);
  while(fil!=NULL) {
    if(fil->getCheck()==1) ins++;
    if(fil->getCheck()==2) outs++;
    fil=(NM_Filter*)filters->getNextElement(fid);
  }
  while(fe!=NULL) {
    if((showHidden==false)&&(fe->isHidden==true)) {
      fe->use=false;
    } else {
      if((fe->isDir()==true)&&(fe->isCorrupt==false)) {
        // a dir is always visible
        fe->use=true;
      } else {
        if(ins>0) isIn=false; else isIn=true;
        fil=(NM_Filter*)filters->getFirstElement(fid);
        if(fil==NULL) isIn=true;
        else {
          while(fil!=NULL) {
            if(fil->getCheck()>0) {
              if(fnmatch(fil->getPattern(),fe->name,0)==0) {
                if(fil->getCheck()==1) isIn=true; else isIn=false;
                break;
              }
            }
            fil=(NM_Filter*)filters->getNextElement(fid);
          }
        }
        fe->inFilter=isIn;
        fe->use=isIn;
      }
    }
    if ( ( fe == tce->activefe ) && ( fe->use == false ) )
      tce->activefe = NULL;
    fe=(FileEntry*)fl->getNextElement(id);
  }
  filters->closeEnum(fid);
  fl->closeEnum(id);

  tce->recalcStats();
  tce->reset_checkfe();
}

NMCacheEntry::NMCacheEntry()
{
  int i;
  verz=NULL;
  activefe=NULL;
  pos=0;
  xpos=0;
  checkfe=NULL;
  lastfe = NULL;
  for(i=0;i<2;i++) {
    files[i]=-1;
    dirs[i]=-1;
  }
  for(i=0;i<2;i++) {
    files_s[i]=-1;
    dirs_s[i]=-1;
  }
  actsortmode=-1;
  actshowhidden=-1;
  actfilterlfdnr=-1;
  dontcheck=false;
  firstnullft = NULL;
}

NMCacheEntry::~NMCacheEntry()
{
  if(verz!=NULL) delete verz;
}

void NMCacheEntry::subentry(FileEntry *fe)
{
  if(fe->isDir()==true) {
    if(strcmp(fe->name,"..")!=0) {
      dirs[0]--;
      if ( fe->dirsize >= 0 ) dirs_s[0] -= fe->dirsize; else dirs_s[0] -= fe->size();
      if(fe->select==true) {
        dirs[1]--;
        if ( fe->dirsize >= 0 ) dirs_s[1] -= fe->dirsize; else dirs_s[1] -= fe->size();
      }
    }
  } else {
    files[0]--;
    files_s[0] -= fe->size();
    if(fe->select==true) {
      files[1]--;
      files_s[1] -= fe->size();
    }
  }
}

/*
 * NMCacheEntry::recalcStats
 *
 * recalculate the nr of bytes/files...
 * resets also activefe when it's not visible
 */
void NMCacheEntry::recalcStats()
{
  int i;
  if(verz==NULL) return;
  ArrayList *filelist=verz->getFiles();
  int id=filelist->initEnum();
  for(i=0;i<2;i++) {
    files[i]=0;
    dirs[i]=0;
  }
  for(i=0;i<2;i++) {
    files_s[i]=0;
    dirs_s[i]=0;
  }
  FileEntry *fe=(FileEntry*)filelist->getFirstElement(id);
  while(fe!=NULL) {
    if(fe->use==true) {
      if(fe->isDir()==true) {
        if(strcmp(fe->name,"..")!=0) {
          dirs[0]++;
          if ( fe->dirsize >= 0 ) dirs_s[0] += fe->dirsize; else dirs_s[0] += fe->size();
          if(fe->select==true) {
            dirs[1]++;
            if ( fe->dirsize >= 0 ) dirs_s[1] += fe->dirsize; else dirs_s[1] += fe->size();
          }
        }
      } else {
        files[0]++;
        files_s[0] += fe->size();
        if(fe->select==true) {
          files[1]++;
          files_s[1] += fe->size();
        }
      }
    } else {
      if(fe==activefe) activefe=NULL;
    }
    fe=(FileEntry*)filelist->getNextElement(id);
  }
  filelist->closeEnum(id);
}

void NormalMode::selhandler(FieldListView *tlv, int row)
{
  if(ce==NULL) return;
  if(ce->verz==NULL) return;
  FileEntry *fe;
  ArrayList *filelist=ce->verz->getFiles();
  fe=(FileEntry*)filelist->getElementAt(lv->getData( row ));

  if(fe==NULL) return;

  bool state=lv->getSelect( row );
  bool oldstate=fe->select;
  
  if(state!=oldstate) {
    // Aenderung
    
    if( row == 0 ) {
      lv->setSelect( row, false );
    } else {
      if(state==true) {
        // Aus -> An
        if((fe->isDir()==true)&&(fe->isCorrupt==false)) {
          ce->dirs[1]++;
          if ( fe->dirsize >= 0 ) ce->dirs_s[1] += fe->dirsize; else ce->dirs_s[1] += fe->size();
        } else {
          ce->files[1]++;
          ce->files_s[1] += fe->size();
        }
      } else {
        // An -> Aus
        if((fe->isDir()==true)&&(fe->isCorrupt==false)) {
          ce->dirs[1]--;
          if ( fe->dirsize >= 0 ) ce->dirs_s[1] -= fe->dirsize; else ce->dirs_s[1] -= fe->size();
        } else {
          ce->files[1]--;
          ce->files_s[1] -= fe->size();
        }
      }
      fe->select=state;
    }
  }
  showCacheState();
  if ( lv->getActiveRow() == row ) {
    ce->activefe=fe;
  }
}

NMFieldListViewDND::NMFieldListViewDND(AGUIX *taguix,int tx,int ty,int width,int height,int tdata):FieldListViewDND(taguix,tx,ty,width,height,tdata)
{
  nm=NULL;
}

NMFieldListViewDND::~NMFieldListViewDND()
{
}

void NMFieldListViewDND::setSelectHandler(NormalMode *nnm)
{
  nm=nnm;
}

void NMFieldListViewDND::runSelectHandler( int row )
{
  if(nm!=NULL) nm->selhandler(this,row );
}

void NormalMode::showCacheState()
{
  if(parentlister->isActive()==false) return;  // do nothing when we are not active
  if ( ce == NULL ) {
    // no directory but at least clear statebar
    parentlister->setStatebarText( "" );
    return;
  }
  const char *tstr1=catalog.getLocale(107),*tstr2=catalog.getLocale(108);
  const char *tstr3=catalog.getLocale(13);
  char *str;
  char buf1[A_BYTESFORNICENUMBER(loff_t)],
       buf2[A_BYTESFORNICENUMBER(loff_t)],
       buf3[sizeof(loff_t)*8+5],
       buf4[sizeof(loff_t)*8+5],
       bufx[sizeof(loff_t)*8];
  loff_t i1=-1,i2=-1;
  loff_t l1,l2;
  int l,tl;
  char *formatstr;
  
  l1=ce->files_s[1]+ce->dirs_s[1];
  l2=ce->files_s[0]+ce->dirs_s[0];
  if(l1>(10*1024*1024)) {
    // MB
    i1=l1/(1024*1024);
    
    bufx[WriteLong2Str(bufx,i1)]='\0';
    sprintf(buf3,"%s MB",bufx);
  } else if(l1>(10*1024)) {
    // KB
    i1=l1/1024;
    sprintf(buf3,"%ld KB",(long)i1);
  } else {
    sprintf(buf3,"%ld B",(long)l1);
  }
  if(l2>(10*1024*1024)) {
    // MB
    i2=l2/(1024*1024);

    bufx[WriteLong2Str(bufx,i2)]='\0';
    sprintf(buf4,"%s MB",bufx);
  } else if(l2>(10*1024)) {
    // KB
    i2=l2/1024;
    sprintf(buf4,"%ld KB",(long)i2);
  } else {
    sprintf(buf4,"%ld B",(long)l2);
  }
  buf1[MakeLong2NiceStr(buf1,l1)]='\0';
  buf2[MakeLong2NiceStr(buf2,l2)]='\0';
  
  // wenn mehr als 100 mal nichts an diesen Variablen
  // geaendert wurde, jeweils um 1 verringern, so dass
  // nicht ewig zu viel Platz verwendet wird, nur weil
  // einmal eine besonders grosse Datei da war
  if(maxnochanges>100) {
    maxnochanges=0;
    // auf >0 muss nicht geprft werden, da danach sowieso nochmal geprft
    // wird
    maxfilestrlen--;
    maxdirstrlen--;
    maxbytesstrlen--;
    maxwbytesstrlen--;
  }

  tl=0;
  l = (int)( log10( (double)ce->files[0] + 1 ) + 1.0 );
  if(l>maxfilestrlen) {
    maxfilestrlen=l;
    tl++;
  }
  l = (int)( log10( (double)ce->dirs[0] + 1 ) +1.0 );
  if(l>maxdirstrlen) {
    maxdirstrlen=l;
    tl++;
  }
  l=strlen(buf2);
  if(l>maxbytesstrlen) {
    maxbytesstrlen=l;
    tl++;
  }
  
  // we have to check for buf3 and buf4 because this is possibly:
  //  1200 KB / 100 MB
  // the left part takes mort space
  l=strlen(buf3);
  if(l>maxwbytesstrlen) {
    maxwbytesstrlen=l;
    tl++;
  }
  l=strlen(buf4);
  if(l>maxwbytesstrlen) {
    maxwbytesstrlen=l;
    tl++;
  }

  // this are the "double" format strings for the statebar
  // the first is when we will also print some nice byte-values (in B, KB or MB)
  // the second otherwise
  // they have to be same length and modifiers
  // from this strings the real format string is build according to the needed string/number lengths
  // NOTE: the two last %s in _SLIM will be replaced by spaces
  //       this is needed to keep the string at the same place in the statebar when switching between listers
#define STATEBAR_FORMAT      "[%%s %%%dld / %%%dld]   [%%s %%%dld / %%%dld]   [%%s %%%ds / %%%ds   (%%%ds / %%%ds) ]"
#define STATEBAR_FORMAT_SLIM "[%%s %%%dld / %%%dld]   [%%s %%%dld / %%%dld]   [%%s %%%ds / %%%ds]   %%%ds   %%%ds   "

  formatstr = (char*)_allocsafe( strlen( STATEBAR_FORMAT ) +
                                 8 * ( A_BYTESFORNUMBER( maxfilestrlen ) ) + 1 );
  // for a 4 byte value I need 11 decimals
  // there will be 8 replacements

  str = (char*)_allocsafe( sizeof( char ) *
                           ( strlen( STATEBAR_FORMAT ) +     // space of the format string
                             strlen( tstr1 ) +               // length of string for "files" (first %s)
                             strlen( tstr2 ) +               // length of string for "dirs" (second %s)
                             strlen( tstr3 ) +               // length of string for "bytes" (third %s)
                             2 * maxfilestrlen +             // space needed for "sel files / files"
                             2 * maxdirstrlen +              // space needed for "sel dirs / dirs"
                             2 * maxbytesstrlen +            // space needed for "sel bytes / bytes"
                             2 * maxwbytesstrlen +           // space needed for "sel bytes / bytes" (in B/KB/MB)
                             1 ) );                          // and not to forget the final zero
  //NOTE: this is more than we will need because the modifiers in the formatstring will be replaced
  //      but it's okay to alloc some more bytes than needed
  if(i2>=0) {
    // this only when the field is visible

    sprintf( formatstr, STATEBAR_FORMAT,
             maxfilestrlen,
             maxfilestrlen,
             maxdirstrlen,
             maxdirstrlen,
             maxbytesstrlen,
             maxbytesstrlen,
             maxwbytesstrlen,
             maxwbytesstrlen );
    sprintf( str, formatstr,
             tstr1,
             ce->files[1],
             ce->files[0],
             tstr2,
             ce->dirs[1],
             ce->dirs[0],
             tstr3,
             buf1,
             buf2,
             buf3,
             buf4 );
  } else {
    sprintf( formatstr, STATEBAR_FORMAT_SLIM,
             maxfilestrlen,
             maxfilestrlen,
             maxdirstrlen,
             maxdirstrlen,
             maxbytesstrlen,
             maxbytesstrlen,
             maxwbytesstrlen,
             maxwbytesstrlen );
    sprintf( str, formatstr,
             tstr1,
             ce->files[1],
             ce->files[0],
             tstr2,
             ce->dirs[1],
             ce->dirs[0],
             tstr3,
             buf1,
             buf2,
             "",    // print just spaces
             "" );  // print just spaces
  }
  _freesafe(formatstr);
  
  if(tl==0) // keine nderungen
    maxnochanges++;
  
  parentlister->setStatebarText(str);
  _freesafe(str);
#undef STATEBAR_FORMAT
#undef STATEBAR_FORMAT_SLIM
}

bool NormalMode::isType(const char *str)
{
  if(strcmp(str,type)==0) return true; else return false;
}

const char *NormalMode::getType()
{
  return type;
}

const char *NormalMode::getStaticType()
{
  return type;
}

int NormalMode::configure()
{
  Button *okb,*cb,*fb;
  AWindow *win;
  CycleButton *cyb[2];
  ChooseButton *chb,*ucb;
  Text *ttext;
  int tw,ttw,tth,ttx,tty,t1,t2;
  AGMessage *msg;
  int endmode=-1;
  char *tstr;
  bool tshowhidden;
  List *tfilters;
  int id;
  NM_Filter *fi,*tfi;
  StringGadget *tsg;
  int tut;
  BevelBox *bb1, *bb2;
  
  ttw=tth=10;
  ttx=tty=5;
  win=new AWindow(aguix);

  tstr = (char*)_allocsafe( strlen( catalog.getLocale( 293 ) ) + strlen( getLocaleName() ) + 1 );
  sprintf( tstr, catalog.getLocale( 293 ), getLocaleName() );
  win->create(NULL,10,10,ttw,tth,0,tstr);
  _freesafe(tstr);

  bb1 = (BevelBox*)win->add( new BevelBox( aguix, ttx, tty, 5, 5, 1 ) );
  ttx += 5;
  tty += 5;
  ttext=(Text*)win->add(new Text(aguix,ttx,tty,catalog.getLocale(158),1));
  ttx+=ttext->getWidth()+5;
  cyb[0]=(CycleButton*)win->add(new CycleButton(aguix,ttx,tty,100,1,0,0));
  cyb[0]->addOption(catalog.getLocale(160));
  cyb[0]->addOption(catalog.getLocale(161));
  cyb[0]->addOption(catalog.getLocale(162));
  cyb[0]->addOption(catalog.getLocale(163));
  cyb[0]->addOption(catalog.getLocale(164));
  cyb[0]->addOption(catalog.getLocale( 432 ));
  cyb[0]->addOption(catalog.getLocale( 433 ));
  cyb[0]->addOption(catalog.getLocale( 552 ));
  cyb[0]->addOption(catalog.getLocale( 553 ));
  cyb[0]->resize(cyb[0]->getMaxSize(),cyb[0]->getHeight());
  if((sortmode&0xff)==SORT_SIZE) cyb[0]->setOption(1);
  else if((sortmode&0xff)==SORT_ACCTIME) cyb[0]->setOption(2);
  else if((sortmode&0xff)==SORT_MODTIME) cyb[0]->setOption(3);
  else if((sortmode&0xff)==SORT_CHGTIME) cyb[0]->setOption(4);
  else if((sortmode&0xff)==SORT_TYPE) cyb[0]->setOption(5);
  else if((sortmode&0xff)==SORT_OWNER) cyb[0]->setOption(6);
  else if ( ( sortmode & 0xff ) == SORT_INODE ) cyb[0]->setOption( 7 );
  else if ( ( sortmode & 0xff ) == SORT_NLINK ) cyb[0]->setOption( 8 );
  else cyb[0]->setOption(0);

  tty+=cyb[0]->getHeight()+5;
  tw=ttx+cyb[0]->getWidth()+5+5;
  if(tw>ttw) ttw=tw;
  ttx=10;

  chb=(ChooseButton*)win->add(new ChooseButton(aguix,ttx,tty,
                                              ((sortmode&SORT_REVERSE)==SORT_REVERSE)?1:0,
                                              catalog.getLocale(165),LABEL_RIGHT,1,0));
  tty+=chb->getHeight()+5;

  ttext=(Text*)win->add(new Text(aguix,ttx,tty,catalog.getLocale(353),1));
  ttx+=ttext->getWidth()+5;
  cyb[1]=(CycleButton*)win->add(new CycleButton(aguix,ttx,tty,100,1,0,0));
  cyb[1]->addOption(catalog.getLocale(354));
  cyb[1]->addOption(catalog.getLocale(355));
  cyb[1]->addOption(catalog.getLocale(356));
  cyb[1]->resize(cyb[1]->getMaxSize(),cyb[1]->getHeight());
  if((sortmode&SORT_DIRLAST)==SORT_DIRLAST) cyb[1]->setOption(1);
  else if((sortmode&SORT_DIRMIXED)==SORT_DIRMIXED) cyb[1]->setOption(2);
  else cyb[1]->setOption(0);

  tty+=cyb[1]->getHeight()+5;
  tw=ttx+cyb[1]->getWidth()+5+5;
  if(tw>ttw) ttw=tw;
  ttx=5;
  bb1->resize( 10, tty - bb1->getY() );
  tty += 5;

  t1=(strlen(catalog.getLocale(159))+2);
  t1*=aguix->getCharWidth();
  fb=(Button*)win->add(new Button(aguix,ttx,tty,t1,catalog.getLocale(159),1,0,0));
  tw=t1+10;
  if(tw>ttw) ttw=tw;
  tty+=fb->getHeight()+5;

  bb2 = (BevelBox*)win->add( new BevelBox( aguix, ttx, tty, 5, 5, 1 ) );
  ttx += 5;
  tty += 5;
  ucb=(ChooseButton*)win->add(new ChooseButton(aguix,ttx,tty,
                                              (showfreespace==true)?1:0,
                                              catalog.getLocale(128),LABEL_RIGHT,1,0));
  tty+=ucb->getHeight()+5;
  ttext=(Text*)win->add(new Text(aguix,ttx,tty,catalog.getLocale(129),1));
  ttx+=ttext->getWidth()+5;
  tstr = (char*)_allocsafe( A_BYTESFORNUMBER( int ) );
  sprintf(tstr,"%d",updatetime);
  tsg=(StringGadget*)win->add(new StringGadget(aguix,ttx,tty,a_max( ttw-ttx-5, 5 * aguix->getCharWidth() ),tstr,0));
  _freesafe(tstr);
  tw = tsg->getX() + tsg->getWidth() + 5 + 5;
  if ( tw > ttw ) ttw = tw;
  tty+=tsg->getHeight()+5;
  bb2->resize( 10, tty - bb2->getY() );
  tty += 5;

  t1=(strlen(catalog.getLocale(11))+2);
  t1*=aguix->getCharWidth();
  t2=(strlen(catalog.getLocale(8))+2);
  t2*=aguix->getCharWidth();
  tw=5+t1+5+t2+5;
  if(tw>ttw) ttw=tw;
  okb=(Button*)win->add(new Button(aguix,
                                   5,
                                   tty,
                                   t1,
                                   catalog.getLocale(11),
                                   1,
                                   0,
                                   0));
  cb=(Button*)win->add(new Button(aguix,
                                  ttw-5-t2,
                                  tty,
                                  t2,
                                  catalog.getLocale(8),
                                  1,
                                  0,
                                  0));
  tty+=okb->getHeight()+5;
  
  tth=tty;
  
  tshowhidden=showHidden;
  tfilters=new List();
  id=filters->initEnum();
  fi=(NM_Filter*)filters->getFirstElement(id);
  while(fi!=NULL) {
    tfilters->addElement(fi->duplicate());
    fi=(NM_Filter*)filters->getNextElement(id);
  }
  filters->closeEnum(id);
  
  win->setDoTabCycling( true );
  win->resize(ttw,tth);
  win->setMaxSize(ttw,tth);
  win->setMinSize(ttw,tth);

  bb1->resize( ttw - 10, bb1->getHeight() );
  bb2->resize( ttw - 10, bb2->getHeight() );

  win->show();
  for(;endmode==-1;) {
    msg=aguix->WaitMessage(win);
    if(msg!=NULL) {
      switch(msg->type) {
        case AG_CLOSEWINDOW:
          if(msg->closewindow.window==win->getWindow()) endmode=1;
          break;
        case AG_BUTTONCLICKED:
          if(msg->button.button==okb) endmode=0;
          else if(msg->button.button==cb) endmode=1;
          else if(msg->button.button==fb) {
            configureFilters(&tshowhidden,tfilters);
          }
          break;
      }
      aguix->ReplyMessage(msg);
    }
  }
  
  if(endmode==0) {
    // ok
    int tsortmode=0;
    switch(cyb[0]->getSelectedOption()) {
      case 1:
        tsortmode=SORT_SIZE;
        break;
      case 2:
        tsortmode=SORT_ACCTIME;
        break;
      case 3:
        tsortmode=SORT_MODTIME;
        break;
      case 4:
        tsortmode=SORT_CHGTIME;
        break;
      case 5:
        tsortmode=SORT_TYPE;
        break;
      case 6:
        tsortmode=SORT_OWNER;
        break;
      case 7:
        tsortmode = SORT_INODE;
        break;
      case 8:
        tsortmode = SORT_NLINK;
        break;
      default:
        tsortmode=SORT_NAME;
        break;
    }
    if(chb->getState()==1) tsortmode|=SORT_REVERSE;
    switch(cyb[1]->getSelectedOption()) {
      case 1:
        tsortmode|=SORT_DIRLAST;
        break;
      case 2:
        tsortmode|=SORT_DIRMIXED;
        break;
      default:
        break;
    }
    setSortmode(tsortmode);
    setShowHiddenE(tshowhidden);
    // apply tfilters
    setFilters(tfilters);
    setShowFreeSpace((ucb->getState()==1)?true:false);
    tut=atoi(tsg->getText());
    setUpdatetime(tut);
  }
  id=tfilters->initEnum();
  tfi=(NM_Filter*)tfilters->getFirstElement(id);
  while(tfi!=NULL) {
    delete tfi;
    tfi=(NM_Filter*)tfilters->getNextElement(id);
  }
  tfilters->closeEnum(id);
  delete tfilters;
  
  win->close();
  delete win;

  return endmode;
}

void NormalMode::down()
{
  ArrayList *al1;
  int row;
  FileEntry *fe;
  int pos,id;
  if(ce==NULL) return;

  finishsearchmode();
  
  al1=ce->verz->getFiles();
  id=al1->initEnum();
  if(ce->activefe==NULL) {
    ce->activefe=(FileEntry*)al1->getFirstElement(id);
    lv->setActiveRow( 0 );
    lv->showActive();
  } else {
    pos=al1->getIndex(ce->activefe);
    do {
      pos++;
      fe=(FileEntry*)al1->getElementAt(id,pos);
      if(fe==NULL) break;
    } while(fe->use==false);
    if(fe!=NULL) {
      // Naechstes Element gefunden
      ce->activefe=fe;
      row = lv->getActiveRow();
      if( lv->isValidRow( row ) == true ) {
        lv->setActiveRow( row + 1 );
        lv->showActive();
      }
    }
  }
  al1->closeEnum(id);
  aguix->Flush();
}

void NormalMode::up()
{
  ArrayList *al1;
  int row;
  FileEntry *fe;
  int pos,id;
  if(ce==NULL) return;

  finishsearchmode();
  
  al1=ce->verz->getFiles();
  id=al1->initEnum();
  if(ce->activefe==NULL) {
    ce->activefe=(FileEntry*)al1->getFirstElement(id);
    lv->setActiveRow( 0 );
    lv->showActive();
  } else {
    pos=al1->getIndex(ce->activefe);
    do {
      pos--;
      fe=(FileEntry*)al1->getElementAt(id,pos);
      if(fe==NULL) break;
    } while(fe->use==false);
    if(fe!=NULL) {
      // Voriges Element gefunden
      ce->activefe=fe;

      row = lv->getActiveRow();
      if ( lv->isValidRow( row ) == true ) {
        lv->setActiveRow( row - 1 );
        lv->showActive();
      }
    }
  }
  al1->closeEnum(id);
  aguix->Flush();
}

void NormalMode::toggleHidden()
{
  setShowHidden((showHidden==true)?false:true);
}

void NormalMode::setShowHidden(bool nhf)
{
  showHidden=nhf;
  if ( ce != NULL ) checkFilters( ce );
  setName();
}

void NormalMode::toggleHiddenE()
{
  setShowHiddenE((showHidden==true)?false:true);
}

void NormalMode::setShowHiddenE(bool nhf)
{
  finishsearchmode();
  
  setShowHidden(nhf);
  buildLister();
  showCacheState();
  if(lv!=NULL) lv->showActive();
}

void NormalMode::startAction(FileEntry *tfe)
{
  WCFiletype *ft;
  bool fallbackEnterDir;
  if(tfe==NULL) return;
  if(tfe->isCorrupt==true) return; // Ziel existiert sowieso nicht, alles koennen wir nichts
                                   // machen

  finishsearchmode();
  
  fallbackEnterDir = false;
  ft = tfe->filetype;
  if ( ft == NULL ) {
    if ( tfe->isDir() == true ) {
      ft = wconfig->getdirtype();
    } else {
      ft = wconfig->getnotyettype();
    }
  }

  if ( ft != NULL ) {
    if ( tfe->isDir() == true ) {
      List *l = ft->getDoubleClickActions();
      if ( l != NULL ) {
	if ( l->size() < 1 ) fallbackEnterDir = true;
      }
    }
    if ( fallbackEnterDir == false ) {
      ActionMessage amsg( parentlister->getWorker() );
      amsg.flags=ft->getDoubleClickActionsGF();
      
      //amsg.mode=amsg.AM_MODE_ONLYACTIVE;
      amsg.mode=amsg.AM_MODE_SPECIAL;
      
      amsg.setFE( tfe );
      amsg.filetype = ft;
      amsg.getActionList = DoubleClickAction::getActionList;
      parentlister->getWorker()->interpret(ft->getDoubleClickActions(),&amsg);
    }
  } else {
    if ( tfe->isDir() == true ) {
      fallbackEnterDir = true;
    }
  }
  if ( fallbackEnterDir == true ) {
    // duplicate fullname because enterDir could delete
    // the directory and with it the fullname
    std::string s1( tfe->fullname );

    // fallback for directories
    enterDir( s1.c_str() );
  }
}

void NormalMode::cyclicfunc(cyclicfunc_mode_t mode)
{
  WCFiletype *ft;
  int trow;
  int side;
  const std::vector<WorkerTypes::listcol_t> *dis;
  bool finished;
  bool doflush;
  time_t now;

  if(mode==CYCLICFUNC_MODE_NORMAL) {
    if(ce!=NULL) {
      FileEntry *fe=ce->checkfe;

      side = parentlister->getSide();
      dis = wconfig->getVisCols( side );
  
      ft = NULL;

      finished = false;
      doflush = false;
      
      if ( visChanged == false ) {
        if ( lv->getYOffset() != oldlvy ) visChanged = true;
      }
#ifdef WANT_THREADS
      if ( thread_usage == true ) {
        if ( ce->firstnullft != NULL )
           finish_done = false;

        // first fill the slave buffer
        reclist->lock();
        
        if ( ( visChanged == true ) && ( reclist->isFull_locked() == false ) ) {
          FileEntry *topfe;
          int toppos, topdata;
          int useelements = a_min( lv->getMaxDisplayV(), lv->getElements() - lv->getYOffset() );
          
          toppos = lv->getYOffset();
          topdata = lv->getData( toppos );

          topfe = findFileEntryForData( topdata, toppos );
          while ( ( topfe != NULL ) && ( useelements > 0 ) && ( reclist->isFull_locked() == false ) ) {
            if ( ( topfe->reclistQueued == false ) &&
                 ( topfe->filetype == NULL ) &&
                 ( topfe->isDir() == false ) ) {
              // add topfe because:
              // 1.it's visible
              // 2.type is not checked
              // 3.it's not queued
              // 4.reclist is not full
              // 5.it's a file
              ft_rec_list::ft_rec_list_t *te = new ft_rec_list::ft_rec_list_t;
              te->name = topfe->fullname;
              te->fe = topfe;
              te->dontCheckContent = ce->dontcheck;

              reclist->put_locked( te );
            }
            useelements--;
            
            // find next visible entry
            toppos++;
            if ( lv->isValidRow( toppos ) ) {
              topdata = lv->getData( toppos );
              topfe = findFileEntryForData( topdata, toppos );
            } else topfe = NULL;
          }
          if (  useelements == 0 ) {
            // all visible are already known or we added them
            visChanged = false;
            oldlvy = lv->getYOffset();
          }
        }
        
        while ( ( fe != NULL ) && ( reclist->isFull_locked() == false ) ){
          if ( ( fe->reclistQueued == false ) && ( fe->filetype == NULL ) ) {
            // when fe was visible we already added it to the list
            // it's not wrong to still add it but it's just double work
            ft_rec_list::ft_rec_list_t *te = new ft_rec_list::ft_rec_list_t;
            te->name = fe->fullname;
            te->fe = fe;
            te->dontCheckContent = ce->dontcheck;
  
            reclist->put_locked( te );
          }
          ce->next_checkfe();
  
          if ( ce->checkfe == NULL )
            ce->lastfe = fe;
  
          fe = ce->checkfe;
        }
        reclist->signal();
        reclist->unlock();
      
        // now check for results
        trow = 0;
        
        recreslist->lock();
        ft_recres_list::ft_recres_list_t *rte;
        ft_rec_list::ft_rec_list_t *te;
        while ( recreslist->isEmpty_locked() == false ) {
          // alle Resultate entnehmen
          rte = recreslist->remove_locked();
          if ( rte != NULL ) {
            // mit oberstem Element von reclist vergleichen
            reclist->lock();
  
            te = reclist->gettop_locked();
            if ( te != NULL ) {
              if ( strcmp( te->name, rte->name ) == 0 ) {
                // correct answer
                te = reclist->remove_locked();
#ifdef DEBUG
		//                printf("type for %s:%d\n",te->name,rte->ft_index);
#endif
                if ( rte->ft_index != NULL ) {
		  ft = findFiletype( rte->ft_index );
                } else {
                  ft = NULL;
                }
                
                if ( ft != NULL ) {
                  // change fe and lvc to filetype
                  fe = te->fe;
                  fe->filetype = ft;
		  if ( rte->customcolors == true ) {
		    fe->setCustomColors( true );
		    fe->setColor( 0, rte->fg );
		    fe->setColor( 1, rte->bg );
		  }
  
                  while( lv->isValidRow( trow ) == true ) {
                    if ( lv->getData( trow ) == fe->nr ) break;
                    trow++;
                  }
                  if ( lv->isValidRow( trow ) == false ) {
                    // lvc not found
                    // because we start always at the last position
                    // let's search again from 0

#ifdef DEBUG
                    printf( "filetype lvc search rollover\n" );
#endif
  
                    trow = 0;
                    while( lv->isValidRow( trow ) == true ) {
                      if ( lv->getData( trow ) == fe->nr ) break;
                      trow++;
                    }
                  }
                  
                  if ( lv->isValidRow( trow ) == true ) {
                    fe->setLVC_DND( lv, trow, dis );
                    //lv->redraw( trow );
                    doflush = true;
                  } else {
                    // lvc not found
                    // this is not a problem because filters,... could be changed
                  
                    // reset pos for next element
                    trow = 0;
                  }
                  
                  if ( fe == ce->firstnullft ) {
                    // firstnullft is no longer the first element with null ft
                    ce->next_firstnullft();
                  }
                }
                
#if 0
                if ( ( te->fe == ce->lastfe ) && ( ce->checkfe == NULL ) ) {
                  // last fileentry recognized
                  finished = true;
                }
#endif
                
                delete te;
              } else {
#ifdef DEBUG
                printf( "wrong answer: %s\n", rte->name );
#endif
              }
            }
	    if ( rte->ft_index != NULL ) {
	      delete rte->ft_index;
	      rte->ft_index = NULL;
	    }
            delete rte;

            reclist->unlock();
          }
        }
        if ( ( ce->firstnullft == NULL ) && ( finish_done == false ) )
           finished = true;
        
        recreslist->unlock();
      } else {  // thread_usage
#endif
        if ( fe != NULL ) {
          ft = fe->checkFiletype( wconfig->getFiletypes(), ce->dontcheck, &condparser );
          if ( ft != NULL ) {
            trow = 0;
            while ( lv->isValidRow( trow ) == true ) {
              if ( lv->getData( trow ) == fe->nr ) break;
              trow++;
            }
            if ( lv->isValidRow( trow ) == true ) {
              fe->setLVC_DND( lv, trow, dis );
              //lv->redraw( trow );
              doflush = true;
            }
          }
          if ( fe == ce->firstnullft ) ce->next_firstnullft();
          ce->next_checkfe();
          if(ce->checkfe==NULL) {
            finished = true;
          }
        }
#ifdef WANT_THREADS
      }
#endif
      if ( finished == true ) {
        // last fe recognized
        // the following is no longer needed because I have FieldLV now
#if 0
        // Am Ende
        #ifdef DEBUG
        printf("lv->getYOffset\n");
        #endif
 
        int ty=lv->getYOffset();

        #ifdef DEBUG
        printf("buildLister\n");
        #endif

        buildLister();

        #ifdef DEBUG
        printf("lv->setYOffset\n");
        #endif

        lv->setYOffset(ty);
        doflush = true;
#endif
        finish_done = true;
      }
      
      if ( doflush == true ) {
        #ifdef DEBUG
        printf("Flush\n");
        #endif
        lv->cond_redraw();
      }
      if ( ce->firstnullft != NULL ) {
        // still some work to do so update busyFlag
        now = time( NULL );
        if ( ( lastBusyFlagUpdate < now ) || ( busyFlag == 0 ) ) {
          lastBusyFlagUpdate = now;
          busyFlag++;
          if ( busyFlag > 4 ) busyFlag = 1;
          setName();
        }
      } else {
        // recognition done so clear busyFlag
        if ( busyFlag > 0 ) {
          busyFlag = 0;
          setName();
        }
      }
      showFreeSpace(false);
    }
  } else if(mode==CYCLICFUNC_MODE_FIRSTNULLFT) {
    NMCacheEntry *tce;
    int id=cache->initEnum();
    tce=(NMCacheEntry*)cache->getFirstElement(id);
    while(tce!=NULL) {
      tce->reset_checkfe();
      tce=(NMCacheEntry*)cache->getNextElement(id);
    }
    cache->closeEnum(id);
  } else {
#ifdef WANT_THREADS
    // get slave in reinit state

    if ( thread_usage == true ) {
      // first wait for the slave to process previous order
      waitForNoOrder();
    }
    
    // put it in reinit mode
    switchOrder( THREAD_REINIT );

    slave.filetype_ex.lock();
    // update filetype list
    ft_list_update();
    
    // clear lists
    ft_rec_list_clear();
    ft_recres_list_clear();
    
    // wake up slave
    slave.filetype_ex.signal();
    slave.filetype_ex.unlock();
    
    if ( thread_usage == true ) {
      // wait for slave in wait mode
      // slave will go from reinit in wait mode after wake up
      waitForNoOrder();
      waitForStatus( THREAD_WAIT );
    }
    
    switchOrder( THREAD_RUN );
#endif

    setupLVFields();
  
    NMCacheEntry *tce;
    int id=cache->initEnum();
    tce=(NMCacheEntry*)cache->getFirstElement(id);
    while(tce!=NULL) {
      tce->restartcheck();
      tce->checkfordcd();
      tce=(NMCacheEntry*)cache->getNextElement(id);
    }
    cache->closeEnum(id);
    if(ce!=NULL) {
      int oldx,oldy;
      oldx=lv->getXOffset();
      oldy=lv->getYOffset();
      buildLister();
      lv->setYOffset(oldy);
      lv->setXOffset(oldx);
    }
  }
}

/*
 * finds next visible entry without filetype
 * effect: advance checkfe
 */
void NMCacheEntry::next_checkfe()
{
  if(checkfe==NULL) return;
  if(verz!=NULL) {
    ArrayList *fes=verz->getFiles();
    if(fes!=NULL) {
      int tpos=fes->getIndex(checkfe)+1;
      for(;;) {
        checkfe=(FileEntry*)fes->getElementAt(tpos++);
        if(checkfe!=NULL) {
          if(checkfe->use==true) {
            // Pruefen, ob Dateityp noch nicht erkannt wurde
            // TODO: Jetzt reicht noch auf NULL pruefen, spaeter muss auf den NotYetChecked geprueft
            // werden
            if(checkfe->filetype==NULL) {
              if((checkfe->isLink==true)&&(checkfe->isCorrupt==false)&&(checkfe->isDir()==false)) break;
              if(checkfe->isDir()==false) break;
            }
          }
        } else break;
      }
    } else checkfe=NULL;
  } else checkfe=NULL;
}

/*
 * reset all filetypes to NULL
 * effect: reset checkfe and firstnullft
 */
void NMCacheEntry::restartcheck()
{
  if(verz!=NULL) {
    ArrayList *fes=verz->getFiles();
    if(fes!=NULL) {
      int id=fes->initEnum();
      FileEntry *fe=(FileEntry*)fes->getFirstElement(id);
      while(fe!=NULL) {
        fe->filetype=NULL;
	fe->setCustomColors( false );
        fe=(FileEntry*)fes->getNextElement(id);
      }
      fes->closeEnum(id);
      checkfe=(FileEntry*)fes->getFirstElement();
    } else checkfe=NULL;
  } else checkfe=NULL;
  next_checkfe();
  firstnullft = checkfe;
}

/*
 * resets checkfe to first null filetype
 * effect: reset checkfe and firstnullft
 */
void NMCacheEntry::reset_checkfe()
{
  if(verz!=NULL) {
    ArrayList *fes=verz->getFiles();
    if(fes!=NULL) {
      checkfe=(FileEntry*)fes->getFirstElement();
    } else checkfe=NULL;
  } else checkfe=NULL;
  next_checkfe();
  firstnullft = checkfe;
}

/*
 * resets firstnullft to first null filetype
 * effect: advance firstnullft
 */
void NMCacheEntry::reset_firstnullft()
{
  if(verz!=NULL) {
    ArrayList *fes=verz->getFiles();
    if(fes!=NULL) {
      firstnullft=(FileEntry*)fes->getFirstElement();
      next_firstnullft();
    } else firstnullft=NULL;
  } else firstnullft=NULL;
}

/*
 * finds next visible entry without filetype
 * effect: advance firstnullft
 */
void NMCacheEntry::next_firstnullft()
{
  if(firstnullft==NULL) return;
  if(verz!=NULL) {
    ArrayList *fes=verz->getFiles();
    if(fes!=NULL) {
      int tpos=fes->getIndex(firstnullft)+1;
      for(;;) {
        firstnullft=(FileEntry*)fes->getElementAt(tpos++);
        if(firstnullft!=NULL) {
          if(firstnullft->use==true) {
            // Pruefen, ob Dateityp noch nicht erkannt wurde
            // TODO: Jetzt reicht noch auf NULL pruefen, spaeter muss auf den NotYetChecked geprueft
            // werden
            if(firstnullft->filetype==NULL) {
              if((firstnullft->isLink==true)&&(firstnullft->isCorrupt==false)&&(firstnullft->isDir()==false)) break;
              if(firstnullft->isDir()==false) break;
            }
          }
        } else break;
      }
    } else firstnullft=NULL;
  } else firstnullft=NULL;
}

const char* NormalMode::getLocaleName()
{
  return getStaticLocaleName();
}

const char* NormalMode::getStaticLocaleName()
{
  return catalog.getLocale(173);
}

int NormalMode::load()
{
  int id;
  int found_error = 0;
  int tsortmode;
  List *tfilters;
  NM_Filter *tfi;

  tsortmode = SORT_NAME;
  tfilters = new List();
  for (;;) {
    if ( worker_token == HIDDENFILES_WCP ) {
      readtoken();

      if ( worker_token != '=' ) {
        found_error = 1;
        break;
      }
      readtoken();

      if ( worker_token == SHOW_WCP ) {
        setShowHiddenE( true );
      } else if ( worker_token == HIDE_WCP ) {
        setShowHiddenE( false );
      } else {
        found_error = 1;
        break;
      }
      readtoken();

      if ( worker_token != ';' ) {
        found_error = 1;
        break;
      }
      readtoken();
    } else if ( worker_token == SORTBY_WCP ) {
      readtoken();

      if ( worker_token != '=' ) {
        found_error = 1;
        break;
      }
      readtoken();

      if ( worker_token == SIZE_WCP ) {
        tsortmode = SORT_SIZE | ( tsortmode & ~0xff );
      } else if ( worker_token == ACCTIME_WCP ) {
        tsortmode = SORT_ACCTIME | ( tsortmode & ~0xff );
      } else if ( worker_token == MODTIME_WCP ) {
        tsortmode = SORT_MODTIME | ( tsortmode & ~0xff );
      } else if ( worker_token == CHGTIME_WCP ) {
        tsortmode = SORT_CHGTIME | ( tsortmode & ~0xff );
      } else if ( worker_token == TYPE_WCP ) {
        tsortmode = SORT_TYPE | ( tsortmode & ~0xff );
      } else if ( worker_token == OWNER_WCP ) {
        tsortmode = SORT_OWNER | ( tsortmode & ~0xff );
      } else if ( worker_token == INODE_WCP ) {
        tsortmode = SORT_INODE | ( tsortmode & ~0xff );
      } else if ( worker_token == NLINK_WCP ) {
        tsortmode = SORT_NLINK | ( tsortmode & ~0xff );
      } else if ( worker_token == NAME_WCP ) {
        tsortmode = SORT_NAME | ( tsortmode & ~0xff );
      } else {
        found_error = 1;
        break;
      }
      readtoken();

      if ( worker_token != ';' ) {
        found_error = 1;
        break;
      }
      readtoken();
    } else if ( worker_token == SORTFLAG_WCP ) {
      readtoken();

      if ( worker_token != '=' ) {
        found_error = 1;
        break;
      }
      readtoken();

      if ( worker_token == REVERSE_WCP ) {
        tsortmode |= SORT_REVERSE;
      } else if ( worker_token == DIRLAST_WCP ) {
        tsortmode |= SORT_DIRLAST;
        tsortmode &= ~SORT_DIRMIXED;
      } else if ( worker_token == DIRMIXED_WCP ) {
        tsortmode |= SORT_DIRMIXED;
        tsortmode &= ~SORT_DIRLAST;
      } else {
        found_error = 1;
        break;
      }
      readtoken();

      if ( worker_token != ';' ) {
        found_error = 1;
        break;
      }
      readtoken();
    } else if ( worker_token == SHOWFREESPACE_WCP ) {
      readtoken();

      if ( worker_token != '=' ) {
        found_error = 1;
        break;
      }
      readtoken();

      if ( worker_token == YES_WCP ) {
        showfreespace = true;
      } else if ( worker_token == NO_WCP ) {
        showfreespace = false;
      } else {
        found_error = 1;
        break;
      }
      readtoken();

      if ( worker_token != ';' ) {
        found_error = 1;
        break;
      }
      readtoken();
    } else if ( worker_token == UPDATETIME_WCP ) {
      readtoken();

      if ( worker_token != '=' ) {
        found_error = 1;
        break;
      }
      readtoken();

      if ( worker_token == NUM_WCP ) {
        setUpdatetime( yylval.num );
      } else {
        found_error = 1;
        break;
      }
      readtoken();

      if ( worker_token != ';' ) {
        found_error = 1;
        break;
      }
      readtoken();
    } else if ( worker_token == FILTER_WCP ) {
      readtoken();

      if ( worker_token != LEFTBRACE_WCP ) {
        found_error = 1;
        break;
      }
      readtoken();
      
      tfi = new NM_Filter();
      if ( tfi->load() == 0 ) {
        tfilters->addElement( tfi );
      } else {
        delete tfi;
        found_error = 1;
        break;
      }

      if ( worker_token != RIGHTBRACE_WCP ) {
        found_error = 1;
        break;
      }
      readtoken();
    } else {
      break;
    }
  }
  setSortmode( tsortmode );
  setFilters( tfilters );
  id = tfilters->initEnum();
  tfi = (NM_Filter*)tfilters->getFirstElement( id );
  while ( tfi != NULL ) {
    delete tfi;
    tfi = (NM_Filter*)tfilters->getNextElement( id );
  }
  tfilters->closeEnum( id );
  delete tfilters;
  
  return found_error;
}

int NormalMode::loadBin(Datei *fh)
{
  int chunksize=fh->getInt();

  int cs=Datei::getUCharSize(),
      is=Datei::getIntSize();
  unsigned char c1;
  int size;
  int tsortmode=0;
  int id;
  bool tshowHidden=false;
  List *tfilters;
  NM_Filter *tfi;

  if(chunksize>=(4*cs+is)) {
    tshowHidden=(fh->getUChar()==1)?true:false;
    c1=fh->getUChar();
    switch(c1) {
      case 1:
        tsortmode=SORT_SIZE;
        break;
      case 2:
        tsortmode=SORT_ACCTIME;
        break;
      case 3:
        tsortmode=SORT_MODTIME;
        break;
      case 4:
        tsortmode=SORT_CHGTIME;
        break;
      case 5:
        tsortmode=SORT_TYPE;
        break;
      case 6:
        tsortmode=SORT_OWNER;
        break;
      default:
        tsortmode=SORT_NAME;
        break;
    }
    if(fh->getUChar()==1) tsortmode|=SORT_REVERSE;
    c1=fh->getUChar();
    switch(c1) {
      case 1:
        tsortmode|=SORT_DIRLAST;
        break;
      case 2:
        tsortmode=SORT_DIRMIXED;
        break;
      default:
        break;
    }
    size=fh->getInt();
    chunksize-=4*cs+is;
    
    tfilters=new List();
    for(int i=0;i<size;i++) {
      tfi=new NM_Filter();
      tfi->loadBin( fh );
      tfilters->addElement(tfi);
    }
    // apply new values
    setShowHiddenE(tshowHidden);
    setSortmode(tsortmode);
    setFilters(tfilters);
    id=tfilters->initEnum();
    tfi=(NM_Filter*)tfilters->getFirstElement(id);
    while(tfi!=NULL) {
      delete tfi;
      tfi=(NM_Filter*)tfilters->getNextElement(id);
    }
    tfilters->closeEnum(id);
    delete tfilters;
    // this is added from version 2.0.2 so for file from older version
    if(chunksize>=(cs+is)) {
      showfreespace=(fh->getUChar()==1)?true:false;
      chunksize-=cs;
      setUpdatetime(fh->getInt());
      chunksize-=is;
    }
  }
  while(chunksize>0) {
    fh->getUChar();
    chunksize--;
  }
  return 0;
}

bool NormalMode::save(Datei *fh)
{
  int id;

  if ( fh == NULL ) return false;
 
  WConfig::configPutPair( fh, "hiddenfiles" , ( showHidden == true ) ? "show" : "hide" );
  switch ( sortmode & 0xff ) {
    case SORT_SIZE:
      WConfig::configPutPair( fh, "sortby", "size" );
      break;
    case SORT_ACCTIME:
      WConfig::configPutPair( fh, "sortby", "acctime" );
      break;
    case SORT_MODTIME:
      WConfig::configPutPair( fh, "sortby", "modtime" );
      break;
    case SORT_CHGTIME:
      WConfig::configPutPair( fh, "sortby", "chgtime" );
      break;
    case SORT_TYPE:
      WConfig::configPutPair( fh, "sortby", "type" );
      break;
    case SORT_OWNER:
      WConfig::configPutPair( fh, "sortby", "owner" );
      break;
    case SORT_INODE:
      WConfig::configPutPair( fh, "sortby", "inode" );
      break;
    case SORT_NLINK:
      WConfig::configPutPair( fh, "sortby", "nlink" );
      break;
    default:
      WConfig::configPutPair( fh, "sortby", "name" );
      break;
  }
  if ( ( sortmode & SORT_REVERSE ) != 0 ) WConfig::configPutPair( fh, "sortflag", "reverse" );
  if ( ( sortmode & SORT_DIRLAST ) != 0 ) WConfig::configPutPair( fh, "sortflag", "dirlast" );
  else if ( ( sortmode & SORT_DIRMIXED ) != 0 ) WConfig::configPutPair( fh, "sortflag", "dirmixed" );
  
  WConfig::configPutPairBool( fh, "showfreespace", showfreespace );
  WConfig::configPutPairNum( fh, "updatetime", updatetime );

  id = filters->initEnum();
  NM_Filter *fi = (NM_Filter*)filters->getFirstElement( id );
  while ( fi != NULL ) {
    WConfig::configOpenSection( fh, "filter" );
    fi->save( fh );
    WConfig::configCloseSection( fh ); //filter
    fi = (NM_Filter*)filters->getNextElement( id );
  }
  filters->closeEnum( id );
  return false;
}

void NormalMode::top()
{
  ArrayList *al1;
  int id;
  if(ce==NULL) return;

  finishsearchmode();
  
  al1=ce->verz->getFiles();
  id=al1->initEnum();
  ce->activefe=(FileEntry*)al1->getFirstElement(id);
  lv->setActiveRow( 0 );
  lv->showActive();
  al1->closeEnum(id);
  aguix->Flush();
}

void NormalMode::last()
{
  ArrayList *al1;
  int row;
  FileEntry *fe;
  int pos,id;
  if(ce==NULL) return;

  finishsearchmode();
  
  al1=ce->verz->getFiles();
  id=al1->initEnum();
  pos=al1->size();
  do {
    pos--;
    fe=(FileEntry*)al1->getElementAt(id,pos);
    if(fe==NULL) break;
  } while(fe->use==false);
  if(fe!=NULL) {
    ce->activefe=fe;
    row = lv->getElements() - 1;
    lv->setActiveRow( row );
    lv->showActive();
  }
  al1->closeEnum(id);
  aguix->Flush();
}

void NormalMode::pageup()
{
  ArrayList *al1;
  int row;
  FileEntry *fe;
  int pos,id;
  if(ce==NULL) return;

  finishsearchmode();
  
  al1=ce->verz->getFiles();
  id=al1->initEnum();
  if(ce->activefe==NULL) {
    ce->activefe=(FileEntry*)al1->getFirstElement(id);
    lv->setActiveRow( 0 );
    lv->showActive();
  } else {
    int rpos=lv->getMaxDisplayV()-1;
    FileEntry *lasthit=ce->activefe;
    pos=al1->getIndex(ce->activefe);
    while(rpos>0) {
      do {
        pos--;
        fe=(FileEntry*)al1->getElementAt(id,pos);
        if(fe==NULL) break;
      } while(fe->use==false);
      if(fe==NULL) break;
      lasthit=fe;
      rpos--;
    };
    if(lasthit!=ce->activefe) {
      ce->activefe=lasthit;
      row = lv->getActiveRow();
      if( row >= 0 ) {
        row -= lv->getMaxDisplayV()-1;
        if(row<0) row=0;
        lv->setActiveRow( row);
        lv->showActive();
      }
    }
  }
  al1->closeEnum(id);
  aguix->Flush();
}

void NormalMode::pagedown()
{
  ArrayList *al1;
  int row;
  FileEntry *fe;
  int pos,id;
  if(ce==NULL) return;

  finishsearchmode();
  
  al1=ce->verz->getFiles();
  id=al1->initEnum();
  if(ce->activefe==NULL) {
    ce->activefe=(FileEntry*)al1->getFirstElement(id);
    lv->setActiveRow( 0 );
    lv->showActive();
  } else {
    int rpos=lv->getMaxDisplayV()-1;
    FileEntry *lasthit=ce->activefe;
    pos=al1->getIndex(ce->activefe);
    while(rpos>0) {
      do {
        pos++;
        fe=(FileEntry*)al1->getElementAt(id,pos);
        if(fe==NULL) break;
      } while(fe->use==false);
      if(fe==NULL) break;
      lasthit=fe;
      rpos--;
    };
    if(lasthit!=ce->activefe) {
      ce->activefe=lasthit;
      row = lv->getActiveRow();
      if ( row >= 0 ) {
        row += lv->getMaxDisplayV()-1;
        if(row>=lv->getElements()) row=lv->getElements()-1;
        lv->setActiveRow( row );
        lv->showActive();
      }
    }
  }
  al1->closeEnum(id);
  aguix->Flush();
}

void NormalMode::select()
{
  if(ce==NULL) return;
  if(ce->activefe==NULL) return;

  finishsearchmode();
  
  int row=lv->getActiveRow();
  if( row < 0 ) {
    printf("oops, this should never happen!\n");
    return;
  }
  if(lv->getSelect(row)==false) lv->setSelect(row, true); else lv->setSelect(row,false);
  selhandler(lv,row );
  down();
}

void NormalMode::selectall()
{
  if(ce==NULL) return;
  FileEntry *fe;
  ArrayList *filelist=ce->verz->getFiles();
  int row;
  
  finishsearchmode();
  
  /* LVCs auswaehlen */
  row = 1;
  while( row < lv->getElements() ) {
    lv->setSelect(row,true);
    row++;
  }
  /* FEs auswaehlen */
  int eid=filelist->initEnum();
  fe=(FileEntry*)filelist->getFirstElement(eid);
  while(fe!=NULL) {
    if((fe->use==true)&&(strcmp(fe->name,"..")!=0)) fe->select=true;
    fe=(FileEntry*)filelist->getNextElement(eid);
  }
  filelist->closeEnum(eid);
  ce->files[1]=ce->files[0];
  ce->files_s[1]=ce->files_s[0];
  ce->dirs[1]=ce->dirs[0];
  ce->dirs_s[1]=ce->dirs_s[0];
  lv->redraw();
  showCacheState();
}

void NormalMode::selectnone()
{
  if(ce==NULL) return;
  FileEntry *fe;
  ArrayList *filelist=ce->verz->getFiles();
  int row;
  
  finishsearchmode();
  
  /* LVCs deselecten */
  row = 0;
  while( row < lv->getElements()) {
    lv->setSelect(row,false);
    row++;
  }
  /* FEs deselecten */
  int eid=filelist->initEnum();
  fe=(FileEntry*)filelist->getFirstElement(eid);
  while(fe!=NULL) {
    if(fe->use==true) fe->select=false;
    fe=(FileEntry*)filelist->getNextElement(eid);
  }
  filelist->closeEnum(eid);
  ce->files[1]=0;
  ce->files_s[1]=0;
  ce->dirs[1]=0;
  ce->dirs_s[1]=0;
  lv->redraw();
  showCacheState();
}

void NormalMode::invertall()
{
  if(ce==NULL) return;
  FileEntry *fe;
  ArrayList *filelist=ce->verz->getFiles();
  int row;
  
  finishsearchmode();
  
  /* LVCs invertieren */
  row = 1;
  while(row<lv->getElements()) {
    lv->setSelect(row,lv->getSelect(row)==true?false:true);
    row++;
  }
  /* FEs invertieren */
  int eid=filelist->initEnum();
  fe=(FileEntry*)filelist->getFirstElement(eid);
  while(fe!=NULL) {
    if((fe->use==true)&&(strcmp(fe->name,"..")!=0)) fe->select=(fe->select==true?false:true);
    fe=(FileEntry*)filelist->getNextElement(eid);
  }
  filelist->closeEnum(eid);
  ce->files[1]=ce->files[0]-ce->files[1];
  ce->files_s[1]=ce->files_s[0]-ce->files_s[1];
  ce->dirs[1]=ce->dirs[0]-ce->dirs[1];
  ce->dirs_s[1]=ce->dirs_s[0]-ce->dirs_s[1];
  lv->redraw();
  showCacheState();
}

void NormalMode::parent()
{
  char oentry[128];
  FileEntry *fe;
  if(ce==NULL) return;
  ArrayList *filelist;
  int row;
  int eid;
  
  char *adir=ce->verz->getDir();
  char *ndir=ParentDir(adir,oentry,127);
  enterDir(ndir);
  /* oentry finden und aktivieren */
  filelist=ce->verz->getFiles();
  eid=filelist->initEnum();
  fe=(FileEntry*)filelist->getFirstElement(eid);
  while(fe!=NULL) {
    if(fe->use==true) if(strcmp(fe->name,oentry)==0) break;
    fe=(FileEntry*)filelist->getNextElement(eid);
  }
  if(fe!=NULL) {
    /* Eintrag gefunden */
    ce->activefe=fe;
    row = 0;
    while( row < lv->getElements() ) {
      if(lv->getData(row)==fe->nr) break;
      row++;
    }
    if( row < lv->getElements()) {
      lv->setActiveRow(row);
      lv->showActive();
      aguix->Flush();
    }
  }
  filelist->closeEnum(eid);
  _freesafe(ndir);
}

void NormalMode::enterDirActive()
{
  if(ce==NULL) return;
  if(ce->activefe!=NULL) {
    if(ce->activefe->isDir()==true) {
      // copy fullname because enterDir can delete directory
      std::string s1( ce->activefe->fullname );
      enterDir( s1.c_str() );
    }
  }
}

void NormalMode::filterselect(char *filter,int mode)
{
  /* Steht in filter[0] ein /, werden Verzeichnisse betrachtet, sonst Dateien */
  if(ce==NULL) return;
  FileEntry *fe;
  ArrayList *filelist=ce->verz->getFiles();
  int row;
  char *realfilter;
  bool tselect=(mode==0)?true:false;
  bool usedir;
  
  finishsearchmode();
  
  if(filter[0]=='/') {
    usedir=true;
    realfilter=&filter[1];
  } else {
    usedir=false;
    realfilter=filter;
  }
  
  int eid=filelist->initEnum();
  fe=(FileEntry*)filelist->getFirstElement(eid);
  while(fe!=NULL) {
    if((fe->use==true)&&(strcmp(fe->name,"..")!=0))
      if(fe->match(realfilter)==true) 
        if(fe->isDir()==usedir) fe->select=tselect;
    fe=(FileEntry*)filelist->getNextElement(eid);
  }

  row = 0;
  fe=(FileEntry*)filelist->getFirstElement(eid);
  while( row < lv->getElements()) {
    while(fe->nr!=lv->getData(row)) fe=(FileEntry*)filelist->getNextElement(eid);
    lv->setSelect(row,fe->select);
    row++;
  }
  filelist->closeEnum(eid);
  lv->redraw();
  ce->recalcStats();
  showCacheState();
}

char *NormalMode::getCurrentDir()
{
  if ( ce != NULL ) return ce->verz->getDir();
  return NULL;
}

void
NormalMode::copy(struct NM_copyorder *copyorder)
{
  Lister *olister=NULL;
  NormalMode *nm2=NULL,*updatenm=NULL;
  ListerMode *lm=NULL;
  Verzeichnis *verz;
  NM_specialsourceExt *sse;
  NM_specialsourceInt *ss1, *ss2, *tss;
  ArrayList *filelist;
  FileEntry *fe;
  const FileEntry *tfe;
  bool found;
  int row;
  bool enter;
  NM_CopyOp_Dir *cod1,*tcod;
  unsigned long files,dirs,gf,gd;
  loff_t bytes;
  bool skip,cancel=false;
  CopyOpWin *cowin;
  nm_copy_t ce1;
  bool nodatabase;
  int oldx,oldy;
  int erg;
  char *textstr,*tstr,*buttonstr;
  std::list<NM_specialsourceExt*>::iterator ite1;
  std::list<NM_specialsourceInt*> *copylist;
  std::list<NM_specialsourceInt*>::iterator iti1, iti2;
  std::list<NM_specialsourceInt*> *copy_lvc_list;
  std::list<const FileEntry*>::iterator itfe1;
  
  if(copyorder==NULL) return;
  if(copyorder->destdir==NULL) return;
  if ( getCurrentDir() == NULL ) {
    // no current dir => no files to copy
    return;
  }
  copyorder->overwrite=copyorder->NM_OVERWRITE_NORMAL;  // force to start in this mode

  finishsearchmode();
  
  if(Datei::fileExistsExt(copyorder->destdir)!=Datei::D_FE_DIR) {
    // no dir as destination
    textstr=(char*)_allocsafe(strlen(catalog.getLocale(67))+strlen(copyorder->destdir)+1);
    sprintf(textstr,catalog.getLocale(67),copyorder->destdir);
    buttonstr = dupstring( catalog.getLocale(11) );
    req->request(catalog.getLocale(125),textstr,buttonstr);
    _freesafe(textstr);
    _freesafe( buttonstr );
    return;
  }
  
#ifdef WANT_THREADS
  // clear the reclist so the slace doesn't block us because he is still reading
  // from disk
  ft_rec_list_clear();
#endif
  
  if ( strcmp( getCurrentDir(), copyorder->destdir ) == 0 ) {
    copyorder->do_rename=true;
    updatenm=this;
  } else {
    olister=parentlister->getWorker()->getOtherLister(parentlister);
    if(olister!=NULL) {
      lm=olister->getActiveMode();
      if(lm!=NULL) {
        if(lm->isType("NormalMode")==true) {
          nm2=(NormalMode*)lm;
          if ( nm2->getCurrentDir() != NULL ) {
            if ( strcmp( nm2->getCurrentDir(), copyorder->destdir ) == 0 ) {
              updatenm=nm2;
            }
          }
        }
      }
    }
  }
  verz=ce->verz;
  filelist=verz->getFiles();
  buf=_allocsafe(BUFSIZE);
  
  if(copyorder->copymode==copyorder->NM_COPYMODE_NODATABASE) nodatabase=true;
  else nodatabase=false;
  
  // no sense for follow_symlinks when moving
  if(copyorder->move==true) copyorder->follow_symlinks=false;

  /* now presupposition checked
     next search for the files to copy
     therefore collect entries to copy in a list together with the LVC (when on root-level)
     then for each entry do recursive call to copy it
     after success deactivate LVC */

  /* first: create List contains NM_specialsource
     when source==NM_SPECIAL then the given list but check for existing of the FEs
     else source==NM_ONLYACTIVE => take the activefe (from ce) in this list
     else take all selected entries (except the "..")
  */
  cowin=copyorder->cowin;
  if(cowin!=NULL) {
    cowin->open(nodatabase);
    cowin->setmessage(catalog.getLocale(122),0);
    cowin->redraw();
  }
  copylist = new std::list<NM_specialsourceInt*>;
  switch(copyorder->source) {
    case NM_copyorder::NM_SPECIAL:
      for ( ite1 = copyorder->sources->begin(); ite1 != copyorder->sources->end(); ite1++ ) {
	sse = *ite1;
        fe=(FileEntry*)filelist->getFirstElement();
        found=false;
        while(fe!=NULL) {
          if ( fe == sse->entry() ) {
            if(strcmp(fe->name,"..")!=0) {
              found=true;
              break;
            }
          }
          fe=(FileEntry*)filelist->getNextElement();
        }
        if(found==true) {
          // valid entry found
	  ss2 = new NM_specialsourceInt( fe );
          ss2->row=sse->row;
          ss2->cod=NULL;
          copylist->push_back(ss2);
        }
      }
      break;
    case NM_copyorder::NM_ONLYACTIVE:
      if(ce->activefe!=NULL) {
        row=lv->getActiveRow();
        if( row >= 0) {
          if(lv->getData(row)==ce->activefe->nr) {
            if(strcmp(ce->activefe->name,"..")!=0) {
	      ss2 = new NM_specialsourceInt( ce->activefe );
              ss2->row=row;
              ss2->cod=NULL;
              copylist->push_back(ss2);
            }
          }
        }
      }
      break;
    default:  // all selected entries
      fe=(FileEntry*)filelist->getFirstElement();
      row = 0;
      while(fe!=NULL) {
        if(fe->use==true) {
          if((strcmp(fe->name,"..")!=0)&&(fe->select==true)) {
	    ss2 = new NM_specialsourceInt( fe );
            ss2->row=row;
            ss2->cod=NULL;
            copylist->push_back(ss2);
          }
          row++;
        }
        fe=(FileEntry*)filelist->getNextElement();
      }
      if(copylist->size()==0) {
        // no selected entries => take the active
        if(ce->activefe!=NULL) {
          row=lv->getActiveRow();
          if(row >= 0) {
            if(lv->getData(row)==ce->activefe->nr) {
              if(strcmp(ce->activefe->name,"..")!=0) {
		ss2 = new NM_specialsourceInt( ce->activefe );
                ss2->row=row;
                ss2->cod=NULL;
                copylist->push_back(ss2);
              }
            }
          }
        }
      }
      break;
  }
  if(cowin!=NULL) cowin->redraw();

  // create the NM_CopyOp_Dir for each dir in copylist
  files=dirs=gf=gd=0;
  bytes = 0;
  
  if(nodatabase==false) {
    for ( iti1 = copylist->begin(); iti1 != copylist->end(); iti1++ ) {
      if ( cancel == true ) break;
      ss1 = *iti1;
      enter=false;
      if ( ss1->entry()->isDir() == true ) {
        // fe is a dir, check if it is a link and take it only when follow_symlinks==true
	// entry is a dir so it cannot be a corrupt link so no need to check
        if ( ss1->entry()->isLink == false ) enter = true;
        else if(copyorder->follow_symlinks==true) enter=true;
      }
      if(enter==true) {
        // fe is a dir so creating corresponding entry
        cod1 = new NM_CopyOp_Dir( ss1->entry() );
        if ( cod1->user_abort == false ) {
          // recursive call
          if(cod1->createSubDirs(copyorder,&gf,&gd)!=0) cancel=true;
        } else cancel = true;
        // add the values from this subdir to this dir
        files+=cod1->files;
        dirs+=cod1->dirs;
        bytes+=cod1->bytes;

        ss1->cod=cod1;
        // this is a dir so inc the counter
        dirs++;
        gd++;
        if(cowin!=NULL) {
          cowin->set_files_to_copy(gf);
          cowin->set_dirs_to_copy(gd);
          if(cowin->redraw()!=0) cancel=true;
        }
      } else {
        // is not dir (mostly a file but can also be links ...)
        files++;
        gf++;
        if ( ( ss1->entry()->isLink == true ) &&
             ( copyorder->follow_symlinks == true ) &&
             ( ss1->entry()->isCorrupt == false ) ) {
	  bytes += ss1->entry()->dsize();
        } else {
          bytes += ss1->entry()->size();
        }
        ss1->cod=NULL;
      }
    }
  }

  if(cowin!=NULL) {
    cowin->set_files_to_copy(files);
    cowin->set_dirs_to_copy(dirs);
    cowin->set_bytes_to_copy(bytes);
    cowin->starttimer();
  }
  // now database ready, can start copy-process

  copy_deletefe_list = new std::list<const FileEntry*>;
  copy_lvc_list = new std::list<NM_specialsourceInt*>;

  skip=false;

  for ( iti1 = copylist->begin(); ( iti1 != copylist->end() ) && ( cancel == false ); iti1++ ) {
    ss1 = *iti1;
    if(nodatabase==false) {
      tcod=ss1->cod;
    } else {
      enter=false;
      tcod=NULL;
      if ( ss1->entry()->isDir() == true ) {
	// entry is a dir so it cannot be a corrupt link so no need to check
        if ( ss1->entry()->isLink == false ) enter = true;
        else if(copyorder->follow_symlinks==true) enter=true;
      }
      if ( enter == true ) {
        tcod = new NM_CopyOp_Dir( ss1->entry() );
        if ( tcod->user_abort == true ) {
          delete tcod;
          tcod = NULL;
          cancel = true;
        }
      }
    }
    if ( cancel == true ) break;
    if ( ss1->row >= 0 ) {
      lv->setVisMark( ss1->row, true );
      lv->showRow( ss1->row );
    }
    if(tcod!=NULL) {
      ce1 = copydir( ss1->entry(), tcod, copyorder, true, copyorder->destdir );
      if((ce1==NM_COPY_OK)||(ce1==NM_COPY_NEED_DELETE)) {
        // success
      } else if(ce1==NM_COPY_CANCEL) {
        cancel=true;
      }
      if(copyorder->move==true) {
        if((ce1==NM_COPY_OK)&&(tcod->error_counter==0)) {
          if(ss1->row>=0) lv->showRow(ss1->row);
          removeEntry( ss1->entry(), ss1->row, true );
          fixSpecialSourceList( copylist, ss1->row );
        } else if(ce1==NM_COPY_NEED_DELETE) {
          if(ss1->row>=0) {
	    tss = new NM_specialsourceInt( ss1->entry() );
            tss->row=ss1->row;
            copy_lvc_list->push_back(tss);
          }
        }
      } else {
        if((tcod->error_counter==0)&&(cancel==false)) {
          // only deselect when dir totally copied
          if(ss1->row>=0) {
            lv->setSelect( ss1->row, false );
            lv->showRow(ss1->row);
          }
          ss1->entry()->select=false;
          ce->recalcStats();
          showCacheState();
        }
      }
      if(nodatabase==true) delete tcod;
    } else {
      ce1 = copyfile( ss1->entry(), copyorder, true, copyorder->destdir );
      if((ce1==NM_COPY_OK)||(ce1==NM_COPY_NEED_DELETE)) {
        // success
        if(copyorder->move==true) {
          if(ce1==NM_COPY_OK) {
            if(ss1->row>=0) lv->showRow(ss1->row);
            removeEntry( ss1->entry(), ss1->row, true );
            fixSpecialSourceList( copylist, ss1->row );
          } else {
            if(ss1->row>=0) {
	      tss = new NM_specialsourceInt( ss1->entry() );
              tss->row=ss1->row;
              copy_lvc_list->push_back(tss);
            }
          }
        } else {
          if(ss1->row>=0) {
            lv->setSelect(ss1->row,false);
            lv->showRow(ss1->row);
          }
          ss1->entry()->select = false;
          ce->recalcStats();
          showCacheState();
        }
      } else if(ce1==NM_COPY_CANCEL) {
        cancel=true;
      }
    }
    if ( ss1->row >= 0 ) {
      // this can and will changed the wrong row when the entry
      // was moved successfully
      // but for this case this doesn't matter because
      // the disabled visual mark is the default
      lv->setVisMark( ss1->row, false );
    }
    if(cowin!=NULL) {
      if(cowin->redraw()!=0) cancel=true;
    }
  }
  
  // now delete remaining files/dirs when moving
  if(copyorder->move==true) {
    bool delcancel; // additional cancel so we don't need to
                    // clear cancel for the special case when move
                    // has to copy&delete and was aborted somewhere
                    // I ask the user what to do with the correctly copied
                    // files
    std::string str1;
  
    if(cowin!=NULL) cowin->setmessage("",1);

    itfe1 = copy_deletefe_list->begin();

    if ( itfe1 != copy_deletefe_list->end() )
      tfe = *itfe1;
    else
      tfe = NULL;
    iti2 = copy_lvc_list->begin();
    if ( iti2 != copy_lvc_list->end() )
      tss = *iti2;
    else
      tss = NULL;
    
    delcancel = false;
    if ( ( cancel == true ) && ( tfe != NULL ) ) {
      // copy was canceled
      // ask the user what to do with the correctly moved files
      // (exactly what to do with the sources!!)
      str1 = catalog.getLocale( 11 );
      str1 += "|";
      str1 += catalog.getLocale( 8 );
      erg = request( cowin, catalog.getLocale( 123 ), catalog.getLocale( 528 ), str1.c_str() );
      if ( erg != 0 ) {
        delcancel = true;
      }
    }
    
    while ( ( tfe != NULL ) && ( delcancel == false ) ) {
      if(tfe->isDir()==true) {
        if(cowin!=NULL) {
          tstr=(char*)_allocsafe(strlen(catalog.getLocale(141))+strlen(tfe->fullname)+1);
          sprintf(tstr,catalog.getLocale(141),tfe->fullname);
          cowin->setmessage(tstr,0);
          _freesafe(tstr);
        }
      }
      // remove tfe
      if((tfe->isDir()==true)&&(tfe->isLink==false)) erg=rmdir(tfe->fullname);
      else erg=remove(tfe->fullname);
      if(erg==0) {
        // success
        if(tss!=NULL) {
          if ( tfe == tss->entry() ) {
            lv->showRow(tss->row);
            removeEntry( tss->entry(), tss->row, true );
            fixSpecialSourceList( copy_lvc_list, tss->row );
	    iti2++;
	    if ( iti2 != copy_lvc_list->end() )
	      tss = *iti2;
	    else
	      tss = NULL;
          }
        }
      } else {
        textstr=(char*)_allocsafe(strlen(catalog.getLocale(291))+strlen(tfe->fullname)+1);
        sprintf(textstr,catalog.getLocale(291),tfe->fullname);
        buttonstr=(char*)_allocsafe(strlen(catalog.getLocale(11))+1+
                                    strlen(catalog.getLocale(8))+1);
        sprintf(buttonstr,"%s|%s",catalog.getLocale(11),
                                  catalog.getLocale(8));
        erg = request( cowin, catalog.getLocale( 347 ), textstr, buttonstr );
        _freesafe(buttonstr);
        _freesafe(textstr);
        if ( erg == 1 ) delcancel = true;
      }
      if ( itfe1 != copy_deletefe_list->end() ) itfe1++;
      if ( itfe1 != copy_deletefe_list->end() )
	tfe = *itfe1;
      else
	tfe = NULL;
    }
    if ( delcancel == true ) cancel = true; // as I write this cancel is not used
                                            // but perhaps later I will use it and
                                            // so set it also to true in this case
  }
  
  if(cowin!=NULL) cowin->close();
  
  for ( iti1 = copylist->begin(); iti1 != copylist->end(); iti1++ ) {
    ss1 = *iti1;
    if(ss1->cod!=NULL) delete ss1->cod;
    delete ss1;
  }
  delete copylist;
  delete copy_deletefe_list;

  for ( iti1 = copy_lvc_list->begin(); iti1 != copy_lvc_list->end(); iti1++ ) {
    tss = *iti1;
    delete tss;
  } 

  delete copy_lvc_list;
  _freesafe(buf);
  if(copyorder->move==true) {
    // now rebuilt the "pos in filelist" <-> "fe->nr" <-> "lvc->data" relation
    oldx=lv->getXOffset();
    oldy=lv->getYOffset();
    removeEntry(NULL,-1,false);
    ce->restartcheck();
    buildLister();
    lv->setXOffset(oldx);
    lv->setYOffset(oldy);
    lv->redraw();
  }
  if(updatenm!=NULL) updatenm->update(false);
  aguix->Flush();
}

NormalMode::nm_newname_t NormalMode::getNewName(const FileEntry *oldfe,struct NM_copyorder *copyorder,char *dest,char **newname_return,bool requestDir,bool acceptrename)
{
  if(requestDir==true)
    return getNewName4Dir(oldfe,copyorder,dest,newname_return,acceptrename);
  else
    return getNewName4File(oldfe,copyorder,dest,newname_return,acceptrename);
}

NormalMode::nm_newname_t NormalMode::getNewName4Dir(const FileEntry *oldfe,struct NM_copyorder *copyorder,char *dest,char **newname_return,bool acceptrename)
{
  char *oldname=oldfe->name;
  char *newname=dupstring(oldname);
  bool nameok=false,skip=false,cancel=false;
  char *return_str=NULL,*buttonstr,*textstr,*newdest;
  int erg;
  Datei::d_fe_t se,lse;
  int round=1;
  nm_newname_t returnvalue=NM_NEWNAME_CANCEL;
  bool trydel,do_rename,strongdirs;
  char *title;
  
  title = (char*)_allocsafe( strlen( catalog.getLocale( 500 ) ) + strlen( oldfe->fullname ) + 1 );
  sprintf( title, catalog.getLocale( 500 ), oldfe->fullname );
  
  do_rename=false;
  if(copyorder->do_rename==false) do_rename=false;
  else if(acceptrename==true) do_rename=true;
  
  strongdirs=false;

  do {
    if(!((round==1)&&(do_rename==false))) {
      buttonstr=(char*)_allocsafe(strlen(catalog.getLocale(11))+1+
                                  strlen(catalog.getLocale(225))+1+
                                  strlen(catalog.getLocale(8))+1);
      sprintf(buttonstr,"%s|%s|%s",catalog.getLocale(11),
                                   catalog.getLocale(225),
                                   catalog.getLocale(8));
      textstr=(char*)_allocsafe(strlen(catalog.getLocale(148))+strlen(oldname)+1);
      sprintf(textstr,catalog.getLocale(148),oldname);
      erg = string_request( copyorder->cowin,
                            catalog.getLocale( 149 ),
                            textstr,
                            newname,
                            buttonstr,
                            &return_str,
                            Requester::REQUEST_SELECTALL );
      _freesafe(buttonstr);
      _freesafe(textstr);
      if(return_str!=NULL) {
        _freesafe(newname);
        newname=return_str;
      }
      if(erg==1) {
        skip=true;
        returnvalue=NM_NEWNAME_SKIP;
        break;
      } else if(erg==2) {
        cancel=true;
        returnvalue=NM_NEWNAME_CANCEL;
        break;
      }
    }
    
    if ( strlen( newname ) < 1 ) {
      // empty name->request
      round++;  // not needed since empty names should only come from user input
                // but perhaps there is (will be) a OS which allows empty names
      continue;
    }
    
    newdest=(char*)_allocsafe(strlen(dest)+1+strlen(newname)+1);
    strcpy(newdest,dest);
    if(strlen(dest)>1) strcat(newdest,"/");
    strcat(newdest,newname);
    se=Datei::fileExistsExt(newdest);
    lse=Datei::lfileExistsExt(newdest);

    switch(lse) {
      case Datei::D_FE_DIR:
        if(strongdirs==true) {
          textstr = (char*)_allocsafe( strlen( catalog.getLocale( 190 ) ) + strlen( newdest ) + 1 );
          sprintf( textstr, catalog.getLocale( 190 ), newdest );

          buttonstr=(char*)_allocsafe(strlen(catalog.getLocale(230))+1+
                                      strlen(catalog.getLocale(273))+1+
                                      strlen(catalog.getLocale(225))+1+
                                      strlen(catalog.getLocale(8))+1);
          sprintf(buttonstr,"%s|%s|%s|%s",catalog.getLocale(230),
                                          catalog.getLocale(273),
                                          catalog.getLocale(225),
                                          catalog.getLocale(8));

          erg = request( copyorder->cowin, title/*catalog.getLocale(123)*/, textstr, buttonstr );
          _freesafe(buttonstr);
          _freesafe(textstr);
          if(erg==1) {
            nameok=true;
            returnvalue=NM_NEWNAME_USE;
          } else if(erg==2) {
            skip=true;
            returnvalue=NM_NEWNAME_SKIP;
          } else if(erg==3) {
            cancel=true;
            returnvalue=NM_NEWNAME_CANCEL;
          }
        } else {
          // just use the dir
          nameok=true;
          returnvalue=NM_NEWNAME_USE;
        }
        break;
      case Datei::D_FE_LINK:
      case Datei::D_FE_FILE:
        if((lse==Datei::D_FE_LINK)&&(se==Datei::D_FE_DIR)) {
          // Symlink with dir as destination 
          if(strongdirs==true) {
//            textstr="There is already a symlink called %s, which points to a dir|You can use this link or delete it to|create a real directory";
//            buttonstr="Enter new name|Use this dir|Delete link|Skip|Cancel";

            textstr = (char*)_allocsafe( strlen( catalog.getLocale( 275 ) ) + strlen( newdest ) + 1 );
            sprintf( textstr, catalog.getLocale( 275 ), newdest );
            buttonstr=(char*)_allocsafe(strlen(catalog.getLocale(230))+1+
                                        strlen(catalog.getLocale(273))+1+
                                        strlen(catalog.getLocale(274))+1+
                                        strlen(catalog.getLocale(225))+1+
                                        strlen(catalog.getLocale(8))+1);
            sprintf(buttonstr,"%s|%s|%s|%s|%s",catalog.getLocale(230),
                                            catalog.getLocale(273),
                                            catalog.getLocale(274),
                                            catalog.getLocale(225),
                                            catalog.getLocale(8));

            erg = request( copyorder->cowin, title/*catalog.getLocale(123)*/, textstr, buttonstr );
            _freesafe(buttonstr);
            _freesafe(textstr);
            switch(erg) {
              case 1:
                nameok=true;
                returnvalue=NM_NEWNAME_USE;
                break;
              case 2:
                // try to remove the link
                // if success set nameok to true
                // in case of failure show a request and repeat the whole
                if(remove(newdest)==0) {
                  nameok=true;
                  returnvalue=NM_NEWNAME_OK;
                } else {
//                  textstr="Failed to remove this file|Please enter new name!";
//                  buttonstr="Ok|Skip|Cancel";

                  textstr = dupstring( catalog.getLocale(276) );
                  buttonstr=(char*)_allocsafe(strlen(catalog.getLocale(11))+1+
                                              strlen(catalog.getLocale(225))+1+
                                              strlen(catalog.getLocale(8))+1);
                  sprintf(buttonstr,"%s|%s|%s",catalog.getLocale(11),
                                               catalog.getLocale(225),
                                               catalog.getLocale(8));

                  erg = request( copyorder->cowin, catalog.getLocale( 347 ), textstr, buttonstr );
                  _freesafe(buttonstr);
                  _freesafe( textstr );
                  if(erg==1) {
                    skip=true;
                    returnvalue=NM_NEWNAME_SKIP;
                  } else if(erg==2) {
                    cancel=true;
                    returnvalue=NM_NEWNAME_CANCEL;
                  }
                }
                break;
              case 3:
                skip=true;
                returnvalue=NM_NEWNAME_SKIP;
                break;
              case 4:
                cancel=true;
                returnvalue=NM_NEWNAME_CANCEL;
                break;
            }
          } else {
            // just use the link to the dir
            nameok=true;
            returnvalue=NM_NEWNAME_USE;
          }
        } else {
          trydel=false;
          if((do_rename==false)&&(copyorder->overwrite==copyorder->NM_OVERWRITE_ALWAYS)) {
            trydel=true;
          } else if((do_rename==false)&&(copyorder->overwrite==copyorder->NM_OVERWRITE_NEVER)) {
            // skip
            skip=true;
            returnvalue=NM_NEWNAME_SKIP;
          } else {
            if(lse==Datei::D_FE_LINK) {
//              textstr="There is already a link named %s";
//              buttonstr="Enter new name|Delete link|Skip|Cancel";

              textstr = (char*)_allocsafe( strlen( catalog.getLocale( 278 ) ) + strlen( newdest ) + 1 );
              sprintf( textstr, catalog.getLocale( 278 ), newdest );
              buttonstr=(char*)_allocsafe(strlen(catalog.getLocale(230))+1+
                                          strlen(catalog.getLocale(274))+1+
                                          strlen(catalog.getLocale(225))+1+
                                          strlen(catalog.getLocale(8))+1);
              sprintf(buttonstr,"%s|%s|%s|%s",catalog.getLocale(230),
                                              catalog.getLocale(274),
                                              catalog.getLocale(225),
                                              catalog.getLocale(8));
            } else {
//              textstr="There is already a file named %s";
//              buttonstr="Enter new name|Delete file|Skip|Cancel";

              textstr = (char*)_allocsafe( strlen( catalog.getLocale( 279 ) ) + strlen( newdest ) + 1 );
              sprintf( textstr, catalog.getLocale( 279 ), newdest );
              buttonstr=(char*)_allocsafe(strlen(catalog.getLocale(230))+1+
                                          strlen(catalog.getLocale(277))+1+
                                          strlen(catalog.getLocale(225))+1+
                                          strlen(catalog.getLocale(8))+1);
              sprintf(buttonstr,"%s|%s|%s|%s",catalog.getLocale(230),
                                              catalog.getLocale(277),
                                              catalog.getLocale(225),
                                              catalog.getLocale(8));
            }
            erg = request( copyorder->cowin, title/*catalog.getLocale(123)*/, textstr, buttonstr );
            _freesafe(buttonstr);
            _freesafe(textstr);
            switch(erg) {
              case 1:
                trydel=true;
                break;
              case 2:
                skip=true;
                returnvalue=NM_NEWNAME_SKIP;
                break;
              case 3:
                cancel=true;
                returnvalue=NM_NEWNAME_CANCEL;
                break;
            }
          }
          if(trydel==true) {
            if(remove(newdest)==0) {
              nameok=true;
              returnvalue=NM_NEWNAME_OK;
            } else {
//              textstr="Failed to remove this file|Please enter new name!";
//              buttonstr="Ok|Skip|Cancel";

              textstr = dupstring( catalog.getLocale(276) );
              buttonstr=(char*)_allocsafe(strlen(catalog.getLocale(11))+1+
                                          strlen(catalog.getLocale(225))+1+
                                          strlen(catalog.getLocale(8))+1);
              sprintf(buttonstr,"%s|%s|%s",catalog.getLocale(11),
                                           catalog.getLocale(225),
                                           catalog.getLocale(8));

              erg = request( copyorder->cowin, catalog.getLocale( 347 ), textstr, buttonstr );
              _freesafe(buttonstr);
              _freesafe( textstr );
              if(erg==1) {
                skip=true;
                returnvalue=NM_NEWNAME_SKIP;
              } else if(erg==2) {
                cancel=true;
                returnvalue=NM_NEWNAME_CANCEL;
              }
            }
          }
        }
        break;
      default:
        nameok=true;
        returnvalue=NM_NEWNAME_OK;
        break;
    }
    _freesafe(newdest);
    round++;
  } while((nameok==false)&&(skip==false)&&(cancel==false));
  if((skip==true)||(cancel==true)) {
    _freesafe(newname);
    *newname_return=NULL;
  } else {
    *newname_return=newname;
  }
  _freesafe( title );
  return returnvalue;
}

NormalMode::nm_newname_t NormalMode::getNewName4File(const FileEntry *oldfe,struct NM_copyorder *copyorder,char *dest,char **newname_return,bool acceptrename)
{
  char *oldname=oldfe->name;
  char *newname=dupstring(oldname);
  bool nameok=false,skip=false,cancel=false;
  char *return_str=NULL,*buttonstr,*textstr,*newdest;
  int erg;
  Datei::d_fe_t se,lse;
  int round=1;
  nm_newname_t returnvalue=NM_NEWNAME_CANCEL;
  bool do_rename,trydel;
  char *newsize,*oldsize,*extrastr,*tstr,*newtime,*oldtime;
  int newsizelen,oldsizelen,maxsizelen;
  char *title;
  time_t tvar;
  
  title = (char*)_allocsafe( strlen( catalog.getLocale( 116 ) ) + strlen( oldfe->fullname ) + 1 );
  sprintf( title, catalog.getLocale( 116 ), oldfe->fullname );
  
  do_rename=false;
  if(copyorder->do_rename==false) do_rename=false;
  else if(acceptrename==true) do_rename=true;

  do {
    if(!((round==1)&&(do_rename==false))) {
//      buttonstr="Ok|Skip|Cancel";
//      textstr="Enter new name for ...";
      buttonstr=(char*)_allocsafe(strlen(catalog.getLocale(11))+1+
                                  strlen(catalog.getLocale(225))+1+
                                  strlen(catalog.getLocale(8))+1);
      sprintf(buttonstr,"%s|%s|%s",catalog.getLocale(11),
                                   catalog.getLocale(225),
                                   catalog.getLocale(8));
      textstr=(char*)_allocsafe(strlen(catalog.getLocale(148))+strlen(oldname)+1);
      sprintf(textstr,catalog.getLocale(148),oldname);
      erg = string_request( copyorder->cowin,
                            catalog.getLocale( 149 ),
                            textstr,
                            newname,
                            buttonstr,
                            &return_str,
                            Requester::REQUEST_SELECTALL );
      _freesafe(buttonstr);
      _freesafe(textstr);
      if(return_str!=NULL) {
        _freesafe(newname);
        newname=return_str;
      }
      if(erg==1) {
        skip=true;
        returnvalue=NM_NEWNAME_SKIP;
        break;
      } else if(erg==2) {
        cancel=true;
        returnvalue=NM_NEWNAME_CANCEL;
        break;
      }
    }
    
    if ( strlen( newname ) < 1 ) {
      // empty name->request
      round++;  // not needed since empty names should only come from user input
                // but perhaps there is (will be) a OS which allows empty names
      continue;
    }
    
    newdest=(char*)_allocsafe(strlen(dest)+1+strlen(newname)+1);
    strcpy(newdest,dest);
    if(strlen(dest)>1) strcat(newdest,"/");
    strcat(newdest,newname);
    se=Datei::fileExistsExt(newdest);
    lse=Datei::lfileExistsExt(newdest);

    switch(lse) {
      case Datei::D_FE_DIR:
//        textstr="there is already a dir called %s";
//        buttonstr="Enter new name|Skip|Cancel";

        textstr = (char*)_allocsafe( strlen( catalog.getLocale( 190 ) ) + strlen( newdest ) + 1 );
        sprintf( textstr, catalog.getLocale( 190 ), newdest );

        buttonstr=(char*)_allocsafe(strlen(catalog.getLocale(230))+1+
                                    strlen(catalog.getLocale(225))+1+
                                    strlen(catalog.getLocale(8))+1);
        sprintf(buttonstr,"%s|%s|%s",catalog.getLocale(230),
                                     catalog.getLocale(225),
                                     catalog.getLocale(8));

        erg = request( copyorder->cowin, title/*catalog.getLocale(123)*/, textstr, buttonstr );
        _freesafe(buttonstr);
        _freesafe(textstr);
        if(erg==1) {
          skip=true;
          returnvalue=NM_NEWNAME_SKIP;
        } else if(erg==2) {
          cancel=true;
          returnvalue=NM_NEWNAME_CANCEL;
        }
        break;
      case Datei::D_FE_LINK:
        trydel=false;
        if((do_rename==false)&&(copyorder->overwrite==copyorder->NM_OVERWRITE_ALWAYS)) {
          trydel=true;
        } else if((do_rename==false)&&(copyorder->overwrite==copyorder->NM_OVERWRITE_NEVER)) {
          // skip
          skip=true;
          returnvalue=NM_NEWNAME_SKIP;
        } else {
//          textstr="there is already a link named %s";
//          buttonstr="Enter new name|Delete link|Skip|Cancel";

          textstr = (char*)_allocsafe( strlen( catalog.getLocale( 278 ) ) + strlen( newdest ) + 1 );
          sprintf( textstr, catalog.getLocale( 278 ), newdest );
          buttonstr=(char*)_allocsafe(strlen(catalog.getLocale(230))+1+
                                      strlen(catalog.getLocale(274))+1+
                                      strlen(catalog.getLocale(225))+1+
                                      strlen(catalog.getLocale(8))+1);
          sprintf(buttonstr,"%s|%s|%s|%s",catalog.getLocale(230),
                                          catalog.getLocale(274),
                                          catalog.getLocale(225),
                                          catalog.getLocale(8));

          erg = request( copyorder->cowin, title/*catalog.getLocale(123)*/, textstr, buttonstr );
          _freesafe(buttonstr);
          _freesafe(textstr);
          switch(erg) {
            case 1:
              trydel=true;
              break;
            case 2:
              skip=true;
              returnvalue=NM_NEWNAME_SKIP;
              break;
            case 3:
              cancel=true;
              returnvalue=NM_NEWNAME_CANCEL;
              break;
          }
        }
        if(trydel==true) {
          if(remove(newdest)==0) {
            nameok=true;
            returnvalue=NM_NEWNAME_OK;
          } else {
//            textstr="Failed to remove this file|Please enter new name!";
//            buttonstr="Ok|Skip|Cancel";

            textstr = dupstring( catalog.getLocale(276) );
            buttonstr=(char*)_allocsafe(strlen(catalog.getLocale(11))+1+
                                        strlen(catalog.getLocale(225))+1+
                                        strlen(catalog.getLocale(8))+1);
            sprintf(buttonstr,"%s|%s|%s",catalog.getLocale(11),
                                         catalog.getLocale(225),
                                         catalog.getLocale(8));

            erg = request( copyorder->cowin, catalog.getLocale( 347 ), textstr, buttonstr );
            _freesafe(buttonstr);
            _freesafe( textstr );
            if(erg==1) {
              skip=true;
              returnvalue=NM_NEWNAME_SKIP;
            } else if(erg==2) {
              cancel=true;
              returnvalue=NM_NEWNAME_CANCEL;
            }
          }
        }
        break;
      case Datei::D_FE_FILE:
        if((do_rename==false)&&(copyorder->overwrite==copyorder->NM_OVERWRITE_ALWAYS)) {
          nameok=true;
          returnvalue=NM_NEWNAME_OVERWRITE;
        } else if((do_rename==false)&&(copyorder->overwrite==copyorder->NM_OVERWRITE_NEVER)) {
          skip=true;
          returnvalue=NM_NEWNAME_SKIP;
        } else {
          FileEntry *newfe=new FileEntry();
          if(newfe->name!=NULL) _freesafe(newfe->name);
          if(newfe->fullname!=NULL) _freesafe(newfe->fullname);
          newfe->name=dupstring(newname);
          newfe->fullname=dupstring(newdest);
          newfe->readInfos();
          
          newsizelen = NiceLongSize( newfe->size() );
          oldsizelen = NiceLongSize( oldfe->size() );
          maxsizelen=(newsizelen>oldsizelen)?newsizelen:oldsizelen;
          newsize=(char*)_allocsafe(maxsizelen+1);
          oldsize=(char*)_allocsafe(maxsizelen+1);

          memset(newsize,' ',maxsizelen);
          memset(oldsize,' ',maxsizelen);
          
          MakeLong2NiceStr( newsize + maxsizelen - newsizelen, newfe->size() );
          MakeLong2NiceStr( oldsize + maxsizelen - oldsizelen, oldfe->size() );
          
          newsize[maxsizelen]='\0';
          oldsize[maxsizelen]='\0';
          
          tvar = newfe->lastmod();
          newtime = dupstring( ctime( &( tvar ) ) );
          tvar = oldfe->lastmod();
          oldtime = dupstring( ctime( &( tvar ) ) );
          newtime[strlen(newtime)-1]='\0';  // remove return
          oldtime[strlen(oldtime)-1]='\0';  // remove return
          
          extrastr=(char*)_allocsafe(strlen(catalog.getLocale(137))+strlen(newsize)+strlen(oldsize)+
                                     strlen(newtime)+strlen(oldtime)+1);
          sprintf(extrastr,catalog.getLocale(137),newsize,oldsize,newtime,oldtime);
          
          _freesafe(newsize);
          _freesafe(oldsize);
          _freesafe(newtime);
          _freesafe(oldtime);
          delete newfe;
//          textstr="There is already a file named %s";
//          buttonstr="Enter new name|Overwrite|Overwrite all|Overwrite none|Skip|Cancel";

          textstr = (char*)_allocsafe( strlen( catalog.getLocale( 279 ) ) + strlen( newdest ) + 1 );
          sprintf( textstr, catalog.getLocale( 279 ), newdest );

          tstr=catstring(textstr,"|");
          _freesafe(textstr);
          textstr=catstring(tstr,extrastr);
          _freesafe(tstr);
          _freesafe(extrastr);

          buttonstr=(char*)_allocsafe(strlen(catalog.getLocale(230))+1+
                                      strlen(catalog.getLocale(280))+1+
                                      strlen(catalog.getLocale(135))+1+
                                      strlen(catalog.getLocale(136))+1+
                                      strlen(catalog.getLocale(225))+1+
                                      strlen(catalog.getLocale(8))+1);
          sprintf(buttonstr,"%s|%s|%s|%s|%s|%s",catalog.getLocale(230),
                                                catalog.getLocale(280),
                                                catalog.getLocale(135),
                                                catalog.getLocale(136),
                                                catalog.getLocale(225),
                                                catalog.getLocale(8));

          erg = request( copyorder->cowin, title/*catalog.getLocale(123)*/, textstr, buttonstr );
          _freesafe(buttonstr);
          _freesafe(textstr);
          switch(erg) {
            case 1:
              nameok=true;
              returnvalue=NM_NEWNAME_OVERWRITE;
              break;
            case 2:
              nameok=true;
              returnvalue=NM_NEWNAME_OVERWRITE;
              copyorder->overwrite=copyorder->NM_OVERWRITE_ALWAYS;
              break;
            case 3:
              skip=true;
              returnvalue=NM_NEWNAME_SKIP;
              copyorder->overwrite=copyorder->NM_OVERWRITE_NEVER;
              break;
            case 4:
              skip=true;
              returnvalue=NM_NEWNAME_SKIP;
              break;
            case 5:
              cancel=true;
              returnvalue=NM_NEWNAME_CANCEL;
              break;
          }
        }
        break;
      default:
        nameok=true;
        returnvalue=NM_NEWNAME_OK;
        break;
    }
    _freesafe(newdest);
    round++;
  } while((nameok==false)&&(skip==false)&&(cancel==false));
  if((skip==true)||(cancel==true)) {
    _freesafe(newname);
    *newname_return=NULL;
  } else {
    *newname_return=newname;
  }
  _freesafe( title );
  return returnvalue;
}

NormalMode::nm_copy_t
NormalMode::copydir(const FileEntry *fe,NM_CopyOp_Dir *cod,struct NM_copyorder *copyorder,bool acceptrename,char *destdir)
{
  ArrayList *filelist;
  NM_CopyOp_Dir *cod1=NULL,*tcod;
  bool nameok,createdir;
  nm_newname_t e1;
  char *destsubdir;
  Datei::d_fe_t de1;
  char *newname=NULL;
  int id1,id2;
  FileEntry *subfe;
  bool isdir,skip,cancel;
  char *buttonstr,*textstr;
  int erg;
  nm_copy_t ce1;
  CopyOpWin *cowin=copyorder->cowin;
  bool nodatabase;
  bool docopy,failedmove=false;
  struct utimbuf utb;

  if(copyorder->copymode==copyorder->NM_COPYMODE_NODATABASE) nodatabase=true;
  else nodatabase=false;

  if ( cowin != NULL ) {
    // call newfile without destination so the user
    // can see the filename in the copyopwin when he will
    // be asked for the new name
    cowin->newfile( fe->name, NULL );
    cowin->redraw();
  }

  skip=cancel=false;
  e1=getNewName(fe,copyorder,destdir,&newname,true,acceptrename);
  if((e1==NM_NEWNAME_OK)||(e1==NM_NEWNAME_USE)) {
    // build new name
    destsubdir=(char*)_allocsafe(strlen(destdir)+1+strlen(newname)+1);
    strcpy(destsubdir,destdir);
    if(strlen(destsubdir)>1) strcat(destsubdir,"/");
    strcat(destsubdir,newname);

    nameok=false;
    createdir=true;
    if(e1==NM_NEWNAME_USE) {
      // there is already such dir (or a symlink to a dir) -> use it
      nameok=true;
      createdir=false;
    } else if(e1==NM_NEWNAME_OK) {
      // check for fileexists (should not)
      // if nethertheless then skip this dir because the rename-func wasn't able to remove it
      de1=Datei::lfileExistsExt(destsubdir);
      if(de1==Datei::D_FE_NOFILE) {
        // everything is ok
        nameok=true;
      } else if(de1==Datei::D_FE_DIR) {
        // dest is a dir
        // should not happend but ok
        nameok=true;
        createdir=false;
        e1=NM_NEWNAME_USE; // for moving it's important to set this, so it doesn't try to
                           // move
      }
    }
    if(nameok==true) {
      docopy=true;
      failedmove=false;
      if(copyorder->move==true) {
        if(e1==NM_NEWNAME_USE) {
          // because of using the destination dir
          // we need to do as failed rename
          failedmove=true;
        } else {
          if(rename(fe->fullname,destsubdir)==0) {
            // success
            docopy=false;
          } else {
            // failure
            failedmove=true;
          }
        }
      }
      
      if(docopy==true) {
        if(cowin!=NULL) cowin->newfile(fe->name,destsubdir);
        if(createdir==true) {
          if ( mkdir( destsubdir, 0700 ) != 0 ) {
            // failed to create dir -> skip
//            textstr="Failed to create dir|I will skip this dir!";
//            buttonstr="Ok|Cancel";

            if(cowin!=NULL) cowin->stoptimer();
            textstr=(char*)_allocsafe(strlen(catalog.getLocale(126))+strlen(destsubdir)+1);
            sprintf(textstr,catalog.getLocale(126),destsubdir);
            buttonstr=(char*)_allocsafe(strlen(catalog.getLocale(11))+1+
                                        strlen(catalog.getLocale(8))+1);
            sprintf(buttonstr,"%s|%s",catalog.getLocale(11),
                                         catalog.getLocale(8));
            erg = request( copyorder->cowin, catalog.getLocale( 347 ), textstr, buttonstr );
            _freesafe(buttonstr);
            _freesafe(textstr);
            if(erg==1) cancel=true;
            nameok=false;
            if(cowin!=NULL) cowin->conttimer();
          }
        }
        if(nameok==true) {
          // first copy all subdirs
          // then copy all files 
          if(nodatabase==false) {
            id1=cod->subdirs->initEnum();
            cod1=(NM_CopyOp_Dir*)cod->subdirs->getFirstElement(id1);
            while(cod1!=NULL) {
              ce1=copydir(cod1->fileentry,cod1,copyorder,false,destsubdir);
              if((ce1==NM_COPY_OK)||(ce1==NM_COPY_NEED_DELETE)) {
                // success
                cod->error_counter+=cod1->error_counter;
              } else {
                cod->error_counter+=cod1->error_counter+1;
                if(ce1==NM_COPY_CANCEL) {
                  cancel=true;
                  break;
                }
              }
              cod1=(NM_CopyOp_Dir*)cod->subdirs->getNextElement(id1);
            }
            cod->subdirs->closeEnum(id1);
          }

          // next only if read this dir correctly
          if((cod->ok==true)&&(cancel==false)) {
            filelist=cod->verz->getFiles();
            id2=filelist->initEnum();
            subfe=(FileEntry*)filelist->getFirstElement(id2);
            while(subfe!=NULL) {
              if(strcmp(subfe->name,"..")!=0) {
                isdir=false;
                if(subfe->isDir()==true) {
                  if(subfe->isLink==false) isdir=true;
                  else if(copyorder->follow_symlinks==true) isdir=true;
                }
                if(isdir==false) {
                  ce1=copyfile(subfe,copyorder,false,destsubdir);
                  if((ce1==NM_COPY_OK)||(ce1==NM_COPY_NEED_DELETE)) {
                    // success
                  } else {
                    cod->error_counter++;
                    if(ce1==NM_COPY_CANCEL) {
                      cancel=true;
                      break;
                    }
                  }
                } else {
                  if(nodatabase==true) {
                    // no database so copy the dir
                    tcod=new NM_CopyOp_Dir(subfe);
                    if ( tcod->user_abort == false ) {
                      ce1=copydir(subfe,tcod,copyorder,false,destsubdir);
                      if(ce1==NM_COPY_OK) {
                        // success
                        cod->error_counter+=tcod->error_counter;
                      } else {
                        cod->error_counter+=tcod->error_counter+1;
                        if(ce1==NM_COPY_CANCEL) {
                          cancel=true;
                        }
                      }
                    } else cancel = true;
                    delete tcod;
                    if(cancel==true) break;
                  }
                }
              }
              subfe=(FileEntry*)filelist->getNextElement(id2);
            }
            filelist->closeEnum(id2);
          }
          // finally change the permissions of the dir
          if((createdir==true)&&(copyorder->preserve_attr==true)) {
	    // fe is a dir so if it is a link it is not corrupt
            chown( destsubdir, ( fe->isLink == true ) ? fe->duserid() : fe->userid(),
                               ( fe->isLink == true ) ? fe->dgroupid() : fe->groupid() );
            chmod( destsubdir, ( fe->isLink == true ) ? fe->dmode() : fe->mode() );
            utb.actime = ( fe->isLink == true ) ? fe->dlastaccess() : fe->lastaccess();
            utb.modtime = ( fe->isLink == true ) ? fe->dlastmod() : fe->lastmod();
            utime(destsubdir,&utb);
          }
        } else skip=true;
      }
    } else skip=true;
    _freesafe(destsubdir);
  } else {
    switch(e1) {
      case NM_NEWNAME_CANCEL:
        cancel=true;
      default:
        skip=true;
        break;
    }
  }
  if((skip==true)&&(cowin!=NULL)) {
    // dir skiped so dec CopyOpWin counter by the files/dirs skiped
    cowin->dec_file_counter(cod->files);
    cowin->dec_dir_counter(cod->dirs);
    cowin->dec_byte_counter(cod->bytes);
  }
  if ( cod->verz->getInvalidEntries() == NULL ) {
    skip = true;
  } else {
    if ( cod->verz->getInvalidEntries()->size() > 0 ) {
#if 0
      // same comment as in ::enterDir
      ArrayList *invalid_entries = cod->verz->getInvalidEntries();
      char *my_msg;
      
      my_msg = (char*)_allocsafe( strlen( catalog.getLocale( 530 ) ) +
                                  strlen( cod->verz->getDir() ) + 1 );
      sprintf( my_msg, catalog.getLocale( 530 ), cod->verz->getDir() );
      showInvalidList( invalid_entries, my_msg );
      _freesafe( my_msg );
#endif
      // when there are some invalid files mark this dir as skipped
      // But I do this after the test above because this is no user initiated
      // skip so all normal files are copied (if everything was okay)
      skip = true;
    }
  }
  if(cowin!=NULL) {
    cowin->dir_finished();
    if(cowin->redraw()!=0) cancel=true;
  }
  if(newname!=NULL) _freesafe(newname);
  if(cancel==true) return NM_COPY_CANCEL;
  else if(skip==true) {
    cod->error_counter++;
    return NM_COPY_SKIP;
  }
  if(copyorder->move==true) {
    if((cod->error_counter==0)&&(failedmove==true)) {
      // put fe in list

      // Man koennte das FE auch also Kopie benutzen
      // man muesste nur den Vergleich beim spaeteren Loeschen anpassen
      copy_deletefe_list->push_back( fe );
      return NM_COPY_NEED_DELETE;
    }
  }
  return NM_COPY_OK;
}

NormalMode::nm_copy_t
NormalMode::copyfile(const FileEntry *fe,struct NM_copyorder *copyorder,bool acceptrename,char *destdir)
{
  bool nameok;
  nm_newname_t e1;
  char *newfullname;
  Datei::d_fe_t de1;
  char *newname=NULL;
  int fdw,fdr;
  bool useregcopy, skip, cancel;
  char *textstr=NULL,*buttonstr;
  int erg;
  ssize_t readbytes,writebytes;
  CopyOpWin *cowin=copyorder->cowin;
  bool docopy,failedmove=false,opok=false;
  struct utimbuf utb;
  std::string str1;
  const char *myerrstr;
  loff_t bytesToCopy;
  mode_t modeCheck;
  bool useLinkDest;
  
  if ( ( fe->isLink == true ) && ( fe->isCorrupt == false ) )
    bytesToCopy = fe->dsize();
  else
    bytesToCopy = fe->size();

  if ( cowin != NULL ) {
    // call newfile without destination so the user
    // can see the filename in the copyopwin when he will
    // be asked for the new name
    cowin->newfile( fe->name, NULL );
    cowin->redraw();
  }

  skip=cancel=false;
  e1=getNewName(fe,copyorder,destdir,&newname,false,acceptrename);

  if((e1==NM_NEWNAME_OK)||(e1==NM_NEWNAME_OVERWRITE)) {
    // build new name
    newfullname=(char*)_allocsafe(strlen(destdir)+1+strlen(newname)+1);
    strcpy(newfullname,destdir);
    if(strlen(newfullname)>1) strcat(newfullname,"/");
    strcat(newfullname,newname);

    nameok=false;
    skip=false;
    if(e1==NM_NEWNAME_OVERWRITE) {
      nameok=true;
    } else if(e1==NM_NEWNAME_OK) {
      // check for fileexists (should not)
      de1=Datei::lfileExistsExt(newfullname);
      if(de1==Datei::D_FE_NOFILE) {
        // everything is ok
        nameok=true;
      } else if(de1==Datei::D_FE_FILE) {
        // dest is a dir
        // should not happend but ok try to overwrite this
        nameok=true;
      }
    }
    if(nameok==true) {
      if ( fe->isSameFile( newfullname, copyorder->follow_symlinks ) == false ) {
        // wenn move true, dann rename probieren
        //        klappt rename nicht, muss flag gesetzt werden und normal weitergemacht
        //        werden und bei korrekten kopieren FE in liste gepackt werden und NEED_DELETE
        //        zurueckgeliefert werden
        //      also: 1.flag docopy am Anfang auf true setzen
        //            2.bei move==true rename probieren
        //            2.1.bei Erfolg docopy auf false, Rueckgabewert ist automatisch OK
        //            2.2.bei Misslingen failedmove auf true setzen
        //            3.bei docopy==true normal kopieren
        //            3.1.bei Misslingen kommt halt skip zurueck
        //            3.2.Bei Erfolg und failedmove==true muss FE in liste gepackt werden
        //                und NEED_DELETE zurueckgeliefert werden
        docopy=true;
        failedmove=false;

        if ( cowin != NULL ) cowin->newfile( fe->name, newfullname );

        if(copyorder->move==true) {
          if(rename(fe->fullname,newfullname)==0) {
            // success
            docopy=false;
          } else {
            // failure
            failedmove=true;
          }
        }
      
        if(docopy==true) {
          // first try to open the file
          // in case of failure give a requester with choose "Delete"
          // then try to delete it and retry to open
          // if all fails, skip
          useregcopy = false;
          if ( S_ISREG( fe->mode() ) ) useregcopy = true;
          else if ( ( fe->isLink == true ) &&
                    ( copyorder->follow_symlinks == true ) &&
		    ( fe->isCorrupt == false ) &&
                    ( S_ISREG( fe->dmode() ) ) ) useregcopy = true;

          if ( useregcopy == true ) {
            fdr = worker_open( fe->fullname, O_RDONLY );
            if(fdr==-1) {
              // can't open inputfile
              if(cowin!=NULL) cowin->stoptimer();
              textstr=(char*)_allocsafe(strlen(catalog.getLocale(281))+strlen(fe->fullname)+1);
              sprintf(textstr,catalog.getLocale(281),fe->fullname);
              buttonstr=(char*)_allocsafe(strlen(catalog.getLocale(225))+1+
                                          strlen(catalog.getLocale(8))+1);
              sprintf(buttonstr,"%s|%s",catalog.getLocale(225),
                                        catalog.getLocale(8));

              erg = request( copyorder->cowin, catalog.getLocale( 347 ), textstr, buttonstr );
              _freesafe(buttonstr);
              _freesafe(textstr);
              if(erg==1) cancel=true;
              else skip=true;
              if(cowin!=NULL) cowin->conttimer();
            } else {
              if(cowin!=NULL) {
                cowin->set_bytes_to_copy_curfile( bytesToCopy );
              }

              for(int i=0;i<2;i++) {
		// fe is a regular file or a working symlink to a regular file so we can use dmode() without checking
		// for corrupt link
                fdw = worker_open( newfullname, O_CREAT | O_TRUNC | O_WRONLY, ( fe->isLink == true ) ? fe->dmode() : fe->mode() );
                if(fdw==-1) {
                  if(errno==ENOSPC) {
                    textstr = (char*)_allocsafe( strlen( catalog.getLocale( 545 ) ) +
                                                 strlen( newname ) +
                                                 strlen( destdir ) + 1 );
                    sprintf( textstr, catalog.getLocale( 545 ), newname, destdir );
                    buttonstr=(char*)_allocsafe(strlen(catalog.getLocale(225))+1+
                                                strlen(catalog.getLocale(8))+1);
                    sprintf(buttonstr,"%s|%s",catalog.getLocale(225),
                                              catalog.getLocale(8));

                    erg = request( copyorder->cowin, catalog.getLocale( 347 ), textstr, buttonstr );
                    _freesafe(buttonstr);
                    _freesafe(textstr);
                    if(erg==1) cancel=true;
                    else skip=true;
                  } else if(errno==EACCES) {
                    if(Datei::lfileExistsExt(newfullname)!=Datei::D_FE_NOFILE) {
                      // error
                      // give request
		      // textstr="Can't overwrite the file|Shall I try to delete it?";
		      // buttonstr="Delete file|Skip|Cancel";

                      if(cowin!=NULL) cowin->stoptimer();
                      textstr=(char*)_allocsafe(strlen(catalog.getLocale(282))+strlen(newfullname)+1);
                      sprintf(textstr,catalog.getLocale(282),newfullname);
                      buttonstr=(char*)_allocsafe(strlen(catalog.getLocale(277))+1+
                                                  strlen(catalog.getLocale(225))+1+
                                                  strlen(catalog.getLocale(8))+1);
                      sprintf(buttonstr,"%s|%s|%s",catalog.getLocale(277),
                                                   catalog.getLocale(225),
                                                   catalog.getLocale(8));

                      erg = request( copyorder->cowin, catalog.getLocale( 123 ), textstr, buttonstr );
                      _freesafe(buttonstr);
                      _freesafe(textstr);
                      if(erg==1) skip=true;
                      else if(erg==2) cancel=true;
                      else {
                        if(remove(newfullname)!=0) {
//                          textstr="Failed to remove this file|I will skip this file!";
//                          buttonstr="Ok|Cancel";

                          textstr=(char*)_allocsafe(strlen(catalog.getLocale(138))+strlen(newfullname)+1);
                          sprintf(textstr,catalog.getLocale(138),newfullname);
                          buttonstr=(char*)_allocsafe(strlen(catalog.getLocale(11))+1+
                                                      strlen(catalog.getLocale(8))+1);
                          sprintf(buttonstr,"%s|%s",catalog.getLocale(11),
                                                    catalog.getLocale(8));
                          erg = request( copyorder->cowin, catalog.getLocale( 347 ), textstr, buttonstr );
                          _freesafe(buttonstr);
                          _freesafe(textstr);
                          if(erg==1) cancel=true;
                          else skip = true;
                        }
                      }
                    } else {
                      // "Can't open dest file"
                      textstr=(char*)_allocsafe(strlen(catalog.getLocale(198))+strlen(newfullname)+1);
                      sprintf(textstr,catalog.getLocale(198),newfullname);
                      buttonstr=(char*)_allocsafe(strlen(catalog.getLocale(11))+1+
                                                  strlen(catalog.getLocale(8))+1);
                      sprintf(buttonstr,"%s|%s",catalog.getLocale(11),
                                                catalog.getLocale(8));

                      erg = request( copyorder->cowin, catalog.getLocale( 347 ), textstr, buttonstr );
                      _freesafe(buttonstr);
                      _freesafe(textstr);
                      if(erg==1) cancel=true;
                      else skip=true;
                    }
                  } else {
                    textstr=(char*)_allocsafe(strlen(catalog.getLocale(198))+strlen(newfullname)+1);
                    sprintf(textstr,catalog.getLocale(198),newfullname);
                    buttonstr=(char*)_allocsafe(strlen(catalog.getLocale(11))+1+
                                                strlen(catalog.getLocale(8))+1);
                    sprintf(buttonstr,"%s|%s",catalog.getLocale(11),
                                              catalog.getLocale(8));

                    erg = request( copyorder->cowin, catalog.getLocale( 347 ), textstr, buttonstr );
                    _freesafe(buttonstr);
                    _freesafe(textstr);
                    if(erg==1) cancel=true;
                    else skip=true;
                  }
                  if(cowin!=NULL) cowin->conttimer();
                } else break;
                if((skip==true)||(cancel==true)) break;
              }
              if(fdw!=-1) {
                writebytes = -1;
                do {
                  readbytes = worker_read( fdr, buf, BUFSIZE );
                  if ( readbytes < 0 ) {
                    // error while reading

                    if(cowin!=NULL) cowin->stoptimer();

                    myerrstr = strerror( errno );
                    textstr = (char*)_allocsafe( strlen( catalog.getLocale( 525 ) ) +
                                                 strlen( fe->fullname ) +
                                                 strlen( myerrstr ) + 1);
                    sprintf( textstr, catalog.getLocale( 525 ), fe->fullname, myerrstr );
                    str1 = catalog.getLocale( 225 );
                    str1 += "|";
                    str1 += catalog.getLocale( 8 );
                    erg = request( copyorder->cowin, catalog.getLocale( 347 ), textstr, str1.c_str() );
                    _freesafe( textstr );
                    if ( erg == 0 ) {
                      skip = true;
                    } else {
                      cancel = true;
                    }
                    if ( cowin != NULL ) cowin->conttimer();
                  } else if ( readbytes == 0 ) {
                    break;
                  } else {
                    writebytes = worker_write( fdw, buf, readbytes );
                    if(writebytes==readbytes) {
                      if(cowin!=NULL) {
                        cowin->add_curbytes_copied(writebytes);
                        if(writebytes==BUFSIZE) {
                          if(cowin->redraw()!=0) cancel=true;
                        }
                      }
                    } else {
                      // something gone wrong
                      // let the user choose to cancel or skip this file
                      // Ask to delete the incomplete destfile
                      // Attention: This method also moves files so be sure to
                      //   NOT delete the source!
                      if( ((writebytes>=0)&&(writebytes<readbytes)) ||
                          (errno==ENOSPC) ) {
                        // ENOSPC isn't always reported so assume ENOSPC
                        textstr = (char*)_allocsafe( strlen( catalog.getLocale( 545 ) ) +
                                                     strlen( newname ) +
                                                     strlen( destdir ) + 1 );
                        sprintf( textstr, catalog.getLocale( 545 ), newname, destdir );
                        buttonstr=(char*)_allocsafe(strlen(catalog.getLocale(225))+1+
                                                    strlen(catalog.getLocale(8))+1);
                        sprintf(buttonstr,"%s|%s",catalog.getLocale(225),
                                                  catalog.getLocale(8));
  
                        erg = request( copyorder->cowin, catalog.getLocale( 347 ), textstr, buttonstr );
                        _freesafe(buttonstr);
                        _freesafe(textstr);
                        if(erg==1) cancel=true;
                        else skip = true;
                      } else {
                        if ( writebytes < 0 ) {
                          myerrstr = strerror( errno );
                          textstr = (char*)_allocsafe( strlen( catalog.getLocale( 526 ) ) +
                                                       strlen( newfullname ) +
                                                       strlen( myerrstr ) + 1);
                          sprintf( textstr, catalog.getLocale( 526 ), newfullname, myerrstr );
                        } else {
                          textstr=(char*)_allocsafe(strlen(catalog.getLocale(359))+strlen(newfullname)+1);
                          sprintf(textstr,catalog.getLocale(359),newfullname);
                        }
                        buttonstr=(char*)_allocsafe(strlen(catalog.getLocale(225))+1+
                                                    strlen(catalog.getLocale(8))+1);
                        sprintf(buttonstr,"%s|%s",catalog.getLocale(225),
                                                  catalog.getLocale(8));
  
                        erg = request( copyorder->cowin, catalog.getLocale( 347 ), textstr, buttonstr );
                        _freesafe(buttonstr);
                        _freesafe(textstr);
                        if(erg==1) cancel=true;
                        else skip = true;
                      }
                    }
                  }
                } while ( ( readbytes > 0 ) &&
                          ( cancel == false ) &&
                          ( skip == false ) );
                worker_close( fdw );

                if ( ( cancel == true ) || ( skip == true ) ) {
                  // error while copying!
                  // ask to remove the incomplete destination file

                  if(cowin!=NULL) cowin->stoptimer();

                  textstr = (char*)_allocsafe( strlen( catalog.getLocale( 393 ) ) + strlen( newfullname ) + 1 );
                  sprintf( textstr, catalog.getLocale( 393 ), newfullname );
                  buttonstr = (char*)_allocsafe( strlen( catalog.getLocale( 394 ) ) + 1 +
                                                 strlen( catalog.getLocale( 395 ) ) + 1 );
                  sprintf( buttonstr, "%s|%s", catalog.getLocale( 394 ),
                                               catalog.getLocale( 395 ) );

                  erg = request( copyorder->cowin, catalog.getLocale( 123 ), textstr, buttonstr );
                  _freesafe( buttonstr );
                  _freesafe( textstr );
                  if(erg==1) {
                    if(remove(newfullname)!=0) {
                      // textstr="Failed to remove the destination file|This file is probably incomplete!";

                      textstr=(char*)_allocsafe(strlen(catalog.getLocale(139))+strlen(newfullname)+1);
                      sprintf(textstr,catalog.getLocale(139),newfullname);
                      buttonstr = dupstring( catalog.getLocale(11) );
                      erg = request( copyorder->cowin, catalog.getLocale( 347 ), textstr, buttonstr );
                      _freesafe(textstr);
                      _freesafe( buttonstr );
                    }
                  }
                  if(cowin!=NULL) cowin->conttimer();
                } else {
                  // copy complete
                  opok=true;
                }
              }
            }
            if ( fdr != -1 ) worker_close( fdr );
          } else {
            // no regular copy
            if ( e1 == NM_NEWNAME_OVERWRITE ) {
              // there is already a file but because we cannot overwrite
              // with special file I will delete it
              // remember the user already choose to overwrite in getNewName!
              if ( remove( newfullname ) != 0 ) {
                // remove failed => cannot continue

                if ( cowin != NULL ) cowin->stoptimer();
                myerrstr = strerror( errno );
                textstr = (char*)_allocsafe( strlen( catalog.getLocale( 516 ) ) +
                                             strlen( fe->fullname ) +
                                             strlen( newfullname ) +
                                             strlen( myerrstr ) + 1);
                sprintf( textstr, catalog.getLocale( 516 ), fe->fullname, newfullname, myerrstr );
                str1 = catalog.getLocale( 225 );
                str1 += "|";
                str1 += catalog.getLocale( 8 );
                erg = request( copyorder->cowin, catalog.getLocale( 347 ), textstr, str1.c_str() );
                _freesafe( textstr );
                if ( erg == 0 ) {
                  skip = true;
                } else {
                  cancel = true;
                }
                if ( cowin != NULL ) cowin->conttimer();
              }
            }
	    //TODO: mc will skip corrupt links when follow_symlinks is used
	    //      I could do this too if I set modeCheck = 0 so it will got to
	    //      the else part
            if ( ( fe->isLink == true ) &&
                 ( copyorder->follow_symlinks == true ) &&
		 ( fe->isCorrupt == false ) ) {
              useLinkDest = true;
              modeCheck = fe->dmode();
            } else {
              useLinkDest = false;
              modeCheck = fe->mode();
            }

            if ( ( skip == false ) && ( cancel == false ) ) {
              if ( ( S_ISCHR( modeCheck ) ) ||
#ifdef S_ISSOCK
                   ( S_ISSOCK( modeCheck ) ) ||
#endif
                   ( S_ISBLK( modeCheck ) ) ) {
                worker_struct_stat statbuf;

                if ( useLinkDest == true ) {
                  erg = worker_stat( fe->fullname, &statbuf );
                } else {
                  erg = worker_lstat( fe->fullname, &statbuf );
                }
                if ( erg == 0 ) {
                  // mknod
                  if ( mknod( newfullname, statbuf.st_mode, statbuf.st_rdev ) != 0 ) {
                    if ( cowin != NULL ) cowin->stoptimer();
                    myerrstr = strerror( errno );
                    textstr = (char*)_allocsafe( strlen( catalog.getLocale( 517 ) ) +
                                                 strlen( fe->fullname ) +
                                                 strlen( newfullname ) +
                                                 strlen( myerrstr ) + 1);
                    sprintf( textstr, catalog.getLocale( 517 ), fe->fullname, newfullname, myerrstr );
                    str1 = catalog.getLocale( 225 );
                    str1 += "|";
                    str1 += catalog.getLocale( 8 );
                    erg = request( copyorder->cowin, catalog.getLocale( 347 ), textstr, str1.c_str() );
                    _freesafe( textstr );
                    if ( erg == 0 ) {
                      skip = true;
                    } else {
                      cancel = true;
                    }
                    if ( cowin != NULL ) cowin->conttimer();
                  } else opok = true;
                } else {
                  if ( cowin != NULL ) cowin->stoptimer();
                  myerrstr = strerror( errno );
                  textstr = (char*)_allocsafe( strlen( catalog.getLocale( 518 ) ) +
                                               strlen( fe->fullname ) +
                                               strlen( newfullname ) +
                                               strlen( myerrstr ) + 1);
                  sprintf( textstr, catalog.getLocale( 518 ), fe->fullname, newfullname, myerrstr );
                  str1 = catalog.getLocale( 225 );
                  str1 += "|";
                  str1 += catalog.getLocale( 8 );
                  erg = request( copyorder->cowin, catalog.getLocale( 347 ), textstr, str1.c_str() );
                  _freesafe( textstr );
                  if ( erg == 0 ) {
                    skip = true;
                  } else {
                    cancel = true;
                  }
                  if ( cowin != NULL ) cowin->conttimer();
                }
              } else if ( S_ISFIFO( modeCheck ) ) {
                // mkfifo
                if ( mkfifo( newfullname, modeCheck ) != 0 ) {
                  if ( cowin != NULL ) cowin->stoptimer();
                  myerrstr = strerror( errno );
                  textstr = (char*)_allocsafe( strlen( catalog.getLocale( 517 ) ) +
                                               strlen( fe->fullname ) +
                                               strlen( newfullname ) +
                                               strlen( myerrstr ) + 1);
                  sprintf( textstr, catalog.getLocale( 517 ), fe->fullname, newfullname, myerrstr );
                  str1 = catalog.getLocale( 225 );
                  str1 += "|";
                  str1 += catalog.getLocale( 8 );
                  erg = request( copyorder->cowin, catalog.getLocale( 347 ), textstr, str1.c_str() );
                  _freesafe( textstr );
                  if ( erg == 0 ) {
                    skip = true;
                  } else {
                    cancel = true;
                  }
                  if ( cowin != NULL ) cowin->conttimer();
                } else opok = true;
              } else if ( S_ISLNK( modeCheck ) ) {
                // it's a link so create a new symlink which points to the same destination
                char *linkdest = fe->getDestination();
                if ( linkdest != NULL ) {
                  if ( symlink( linkdest, newfullname ) != 0 ) {
                    if ( cowin != NULL ) cowin->stoptimer();
                    myerrstr = strerror( errno );
                    textstr = (char*)_allocsafe( strlen( catalog.getLocale( 517 ) ) +
                                                 strlen( fe->fullname ) +
                                                 strlen( newfullname ) +
                                                 strlen( myerrstr ) + 1);
                    sprintf( textstr, catalog.getLocale( 517 ), fe->fullname, newfullname, myerrstr );
                    str1 = catalog.getLocale( 225 );
                    str1 += "|";
                    str1 += catalog.getLocale( 8 );
                    erg = request( copyorder->cowin, catalog.getLocale( 347 ), textstr, str1.c_str() );
                    _freesafe( textstr );
                    if ( erg == 0 ) {
                      skip = true;
                    } else {
                      cancel = true;
                    }
                    if ( cowin != NULL ) cowin->conttimer();
                  } else opok = true;
                  _freesafe( linkdest );
                } else {
                  if ( cowin != NULL ) cowin->stoptimer();
                  myerrstr = strerror( errno );
                  textstr = (char*)_allocsafe( strlen( catalog.getLocale( 519 ) ) +
                                               strlen( fe->fullname ) +
                                               strlen( newfullname ) +
                                               strlen( myerrstr ) + 1);
                  sprintf( textstr, catalog.getLocale( 519 ), fe->fullname, newfullname, myerrstr );
                  str1 = catalog.getLocale( 225 );
                  str1 += "|";
                  str1 += catalog.getLocale( 8 );
                  erg = request( copyorder->cowin, catalog.getLocale( 347 ), textstr, str1.c_str() );
                  _freesafe( textstr );
                  if ( erg == 0 ) {
                    skip = true;
                  } else {
                    cancel = true;
                  }
                  if ( cowin != NULL ) cowin->conttimer();
                }
              } else {
                if ( cowin != NULL ) cowin->stoptimer();
                myerrstr = strerror( errno );
                textstr = (char*)_allocsafe( strlen( catalog.getLocale( 520 ) ) +
                                             strlen( fe->fullname ) + 1);
                sprintf( textstr, catalog.getLocale( 520 ), fe->fullname );
                str1 = catalog.getLocale( 225 );
                str1 += "|";
                str1 += catalog.getLocale( 8 );
                erg = request( copyorder->cowin, catalog.getLocale( 347 ), textstr, str1.c_str() );
                _freesafe( textstr );
                if ( erg == 0 ) {
                  skip = true;
                } else {
                  cancel = true;
                }
                if ( cowin != NULL ) cowin->conttimer();
              }
            }
            //TODO: muss hier nicht eher add_curbytes_copied aufgerufen werden?
//            if(cowin!=NULL) cowin->set_bytes_to_copy_curfile(fe->size);
            if ( cowin != NULL ) cowin->add_curbytes_copied( fe->size() );
          }
        }
      } else {
        textstr=(char*)_allocsafe(strlen(catalog.getLocale(113))+strlen(fe->fullname)+1);
        sprintf(textstr,catalog.getLocale(113),fe->fullname);
        buttonstr = dupstring( catalog.getLocale(11) );

        request( copyorder->cowin, catalog.getLocale( 347 ), textstr, buttonstr );
        _freesafe(textstr);
        _freesafe( buttonstr );
      }
    } else skip=true;

    if ( opok == true ) {
      // try to change the mode because open modify it with umask
      // it doesn't matter if this fails 
      if(copyorder->preserve_attr==true) {
        //TODO:currently deactivated
        //     chown is necessary because link can have an owner
        //     but chmod could be optional
        // only apply new mode and time if it's not a link
        // doing this would not do anything wrong
        // but chown/chmod/utime would update the access/change time
        // of the destination file although we don't have to touch it
        //if ( ( fe->isLink == false ) || ( copyorder->follow_symlinks == true ) ) {
          tryApplyOwnerPerm( newfullname,
                             ( ( fe->isLink == true ) && ( fe->isCorrupt == false ) ) ? fe->duserid() : fe->userid(),
                             ( ( fe->isLink == true ) && ( fe->isCorrupt == false ) ) ? fe->dgroupid() : fe->groupid(),
                             ( ( fe->isLink == true ) && ( fe->isCorrupt == false ) ) ? fe->dmode() : fe->mode(),
                             copyorder );
          utb.actime = ( ( fe->isLink == true ) && ( fe->isCorrupt == false ) ) ? fe->dlastaccess() : fe->lastaccess();
          utb.modtime = ( ( fe->isLink == true ) && ( fe->isCorrupt == false ) ) ? fe->dlastmod() : fe->lastmod();
          utime(newfullname,&utb);
        //}
      }
    }

    _freesafe(newfullname);
  } else {
    switch(e1) {
      case NM_NEWNAME_CANCEL:
        cancel=true;
      default:
        skip=true;
        break;
    }
  }

  if(cowin!=NULL) {
    if ( skip == true ) {
      // the CopyOpWin need to substract bytesToCopy from global bytes sum
      // and already copied bytes from copyied bytes sum
      // if we called newfile() (which resets this counter)
      cowin->file_skipped( bytesToCopy, true );
    } else {
      cowin->file_finished();
    }
    if(cowin->redraw()!=0) cancel=true;
  }
  if(newname!=NULL) _freesafe(newname);
  if(cancel==true) return NM_COPY_CANCEL;
  else if(skip==true) return NM_COPY_SKIP;
  if(copyorder->move==true) {
    if((opok==true)&&(failedmove==true)) {
      // remember: failedmove is true when OS function rename failed
      //           in this case we have to copy the file and delete the source
      //           so add this entry only when copy is complete (opok) AND
      //           rename failed (failedmove)
      // put fe in list
      copy_deletefe_list->push_back( fe );
      return NM_COPY_NEED_DELETE;
    }
  }
  return NM_COPY_OK;
}

bool
NormalMode::isCorrectViewProg(const char *str)
{
  const char *pos;
  if(strlen(str)<1) return true;
  pos=strstr(str,"%s");
  if(pos!=NULL) return true;
  return false;
}

void
NormalMode::deletef(struct NM_deleteorder *delorder)
{
  Verzeichnis *verz;
  NM_specialsourceInt *ss1, *ss2;
  ArrayList *filelist;
  FileEntry *fe;
//  bool found;
  int row;
  bool enter;
  NM_CopyOp_Dir *cod1,*tcod;
  unsigned long files,dirs,gf,gd;
  bool skip,cancel=false;
  DeleteOpWin *dowin;
  nm_copy_t ce1;
  bool nodatabase;
  char *buttonstr,*textstr;
  int erg;
  bool empty;
  int oldx,oldy;
  int tnrfiles,tnrdirs;
  std::list<NM_specialsourceInt*> *dellist;
  std::list<NM_specialsourceInt*>::iterator iti1;
  
  if(delorder==NULL) return;
  if ( getCurrentDir() == NULL ) {
    // no current dir => no files to delete
    return;
  }

  tnrfiles=ce->files[1];
  tnrdirs=ce->dirs[1];

  if ( ( delorder->source == delorder->NM_ALLENTRIES ) &&
       ( ( tnrfiles + tnrdirs ) < 1 ) ) {
    if ( ce->activefe != NULL ) {
      if ( strcmp( ce->activefe->name, ".." ) != 0 ) {
        if ( ce->activefe->isDir() == true ) tnrdirs++;
        else tnrfiles++;
      }
    }
  }
  
  if ( ( tnrfiles + tnrdirs ) < 1 )
    return;

  finishsearchmode();
  
  verz=ce->verz;
  filelist=verz->getFiles();
  
  buttonstr=(char*)_allocsafe(strlen(catalog.getLocale(11))+1+
                              strlen(catalog.getLocale(8))+1);
  sprintf(buttonstr,"%s|%s",catalog.getLocale(11),
                            catalog.getLocale(8));
  textstr = (char*)_allocsafe( strlen( catalog.getLocale( 145 ) ) + NiceLongSize( tnrfiles ) +
                               NiceLongSize( tnrdirs ) + 1 );
  sprintf(textstr,catalog.getLocale(145),tnrfiles,tnrdirs);
  erg = request( catalog.getLocale( 123 ), textstr, buttonstr );
  _freesafe(buttonstr);
  _freesafe(textstr);
  if(erg!=0) return;

#ifdef WANT_THREADS
  // clear the reclist so the slace doesn't block us because he is still reading
  // from disk
  ft_rec_list_clear();
#endif
  
  if(delorder->delmode==delorder->NM_DELMODE_NODATABASE) nodatabase=true;
  else nodatabase=false;

  dowin=delorder->dowin;
  if(dowin!=NULL) {
    dowin->open();
    dowin->setmessage(catalog.getLocale(140));
    dowin->redraw();
  }
  dellist = new std::list<NM_specialsourceInt*>;
  switch(delorder->source) {
    case NM_deleteorder::NM_SPECIAL:
/*OPT:At the moment I doesn't allow specialsources
      ss1=(NM_specialsource*)delorder->sources->getFirstElement();
      while(ss1!=NULL) {
        fe=(FileEntry*)filelist->getFirstElement();
        found=false;
        while(fe!=NULL) {
          if ( fe == ss1->entry() ) {
            if(strcmp(fe->name,"..")!=0) {
              found=true;
              break;
            }
          }
          fe=(FileEntry*)filelist->getNextElement();
        }
        if(found==true) {
          // valid entry found
	  ss2 = new NM_specialsource( ss1->entry() );
          ss2->lvc=ss1->lvc;
          ss2->cod=NULL;
          dellist->addElement(ss2);
        }
        ss1=(NM_specialsource*)delorder->sources->getNextElement();
      }*/
      break;
    case NM_deleteorder::NM_ONLYACTIVE:
/*OPT:At the moment I doesn't allow onlyactive
      if(ce->activefe!=NULL) {
        lvc=lv->getActiveLVC();
        if(lvc!=NULL) {
          if(lvc->getData()==ce->activefe->nr) {
            if(strcmp(ce->activefe->name,"..")!=0) {
	      ss2 = new NM_specialsource( ce->activefe );
              ss2->lvc=lvc;
              ss2->cod=NULL;
              dellist->addElement(ss2);
            }
          }
        }
      }*/
      break;
    case NM_deleteorder::NM_ONLYSELECTED:
      fe=(FileEntry*)filelist->getFirstElement();
      row = 0;
      while(fe!=NULL) {
        if(fe->use==true) {
          if((strcmp(fe->name,"..")!=0)&&(fe->select==true)) {
	    ss2 = new NM_specialsourceInt( fe );
            ss2->row=row;
            ss2->cod=NULL;
            dellist->push_back(ss2);
          }
          row++;
        }
        fe=(FileEntry*)filelist->getNextElement();
      }
      break;
    default:  // all selected entries
      fe=(FileEntry*)filelist->getFirstElement();
      row = 0;
      while(fe!=NULL) {
        if(fe->use==true) {
          if((strcmp(fe->name,"..")!=0)&&(fe->select==true)) {
	    ss2 = new NM_specialsourceInt( fe );
            ss2->row=row;
            ss2->cod=NULL;
            dellist->push_back(ss2);
          }
          row++;
        }
        fe=(FileEntry*)filelist->getNextElement();
      }
      if(dellist->size()==0) {
        // no selected entries => take the active
        if(ce->activefe!=NULL) {
          row = lv->getActiveRow();
          if(row>=0) {
            if(lv->getData(row)==ce->activefe->nr) {
              if(strcmp(ce->activefe->name,"..")!=0) {
		ss2 = new NM_specialsourceInt( ce->activefe );
                ss2->row=row;
                ss2->cod=NULL;
                dellist->push_back(ss2);
              }
            }
          }
        }
      }
      break;
  }
  if(dowin!=NULL) dowin->redraw();

  // create the NM_CopyOp_Dir for each dir in dellist
  files=dirs=gf=gd=0;
  
  if(nodatabase==false) {
    for ( iti1 = dellist->begin(); iti1 != dellist->end(); iti1++ ) {
      if ( cancel == true ) break;
      ss1 = *iti1;
      enter=false;
      if ( ss1->entry()->isDir() == true ) {
        // fe is a dir, check if it is a link and take it only when follow_symlinks==true
        if ( ss1->entry()->isLink == false ) enter = true;
      }
      if(enter==true) {
        // fe is a dir so creating corresponding entry
        cod1 = new NM_CopyOp_Dir( ss1->entry() );
        if ( cod1->user_abort == false ) {
          // recursive call
          if(cod1->createSubDirs(delorder,&gf,&gd)!=0) cancel=true;
        } else cancel = true;
        // add the values from this subdir to this dir
        files+=cod1->files;
        dirs+=cod1->dirs;

        ss1->cod=cod1;
        // this is a dir so inc the counter
        dirs++;
        gd++;
        if(dowin!=NULL) {
          dowin->set_files_to_delete(gf);
          dowin->set_dirs_to_delete(gd);
          if(dowin->redraw()!=0) cancel=true;;
        }
      } else {
        // is not dir (mostly a file but can also be links ...)
        files++;
        gf++;
        ss1->cod=NULL;
      }
    }
  }

  if(dowin!=NULL) {
    dowin->set_files_to_delete(files);
    dowin->set_dirs_to_delete(dirs);
  }
  // now database ready, can start copy-process
  skip=false;
  for ( iti1 = dellist->begin(); ( iti1 != dellist->end() ) && ( cancel == false ); iti1++ ) {
    ss1 = *iti1;
    if(nodatabase==false) {
      tcod=ss1->cod;
    } else {
      enter=false;
      tcod=NULL;
      if ( ss1->entry()->isDir() == true ) {
        if ( ss1->entry()->isLink == false ) enter = true;
      }
      if ( enter == true ) {
        tcod = new NM_CopyOp_Dir( ss1->entry() );
        if ( tcod->user_abort == true ) {
          delete tcod;
          tcod = NULL;
          cancel = true;
        }
      }
    }
    if ( cancel == true ) break;
    if ( ss1->row >= 0 ) {
      lv->setVisMark( ss1->row, true );
      lv->showRow( ss1->row );
    }
    if(tcod!=NULL) {
      // determine if the dir is empty or not
      if(nodatabase==false) {
        if((tcod->files+tcod->dirs)>0) empty=false;
        else empty=true;
      } else {
        empty=false;
        if(tcod->ok==true) {
          if(tcod->verz->getFiles()->size()<2) empty=true;
        }
      }
      skip=false;
      // skip if the user doesn't want to delete non-empty dirs
      if((delorder->dirdel==delorder->NM_DIRDELETE_NONE)&&(empty==false)) skip=true;
      // if the user doesn't select ALL or NONE and the the dir isn't empty show a request
      if((delorder->dirdel==delorder->NM_DIRDELETE_NORMAL)&&(empty==false)) {
        textstr=(char*)_allocsafe(strlen(catalog.getLocale(146))+
                                  strlen( ss1->entry()->fullname ) + 1 );
        sprintf( textstr, catalog.getLocale( 146 ), ss1->entry()->fullname );
        buttonstr=(char*)_allocsafe(strlen(catalog.getLocale(11))+1+
                                    strlen(catalog.getLocale(287))+1+
                                    strlen(catalog.getLocale(288))+1+
                                    strlen(catalog.getLocale(225))+1+
                                    strlen(catalog.getLocale(8))+1);
        sprintf(buttonstr,"%s|%s|%s|%s|%s",catalog.getLocale(11),
                                           catalog.getLocale(287),
                                           catalog.getLocale(288),
                                           catalog.getLocale(225),
                                           catalog.getLocale(8));
        erg = request( delorder->dowin, catalog.getLocale( 123 ), textstr, buttonstr );
        _freesafe(buttonstr);
        _freesafe(textstr);
        switch(erg) {
          case 1:
            delorder->dirdel=delorder->NM_DIRDELETE_ALL;
            break;
          case 2:
            delorder->dirdel=delorder->NM_DIRDELETE_NONE;
            skip=true;
            break;
          case 3:
            skip=true;
            break;
          case 4:
            cancel=true;
            skip=true;
            break;
        }
      }
      if(skip==false) {
        ce1 = deldir( ss1->entry(), tcod, delorder );
        if(ce1==NM_COPY_OK) {
          // success
        } else if(ce1==NM_COPY_CANCEL) {
          cancel=true;
        }
        if((tcod->error_counter==0)&&(cancel==false)) {
          // successful delete this dir so remove it form the lv
          // first show this lvc
          if(ss1->row>=0) lv->showRow(ss1->row);
          removeEntry( ss1->entry(), ss1->row, true );
          fixSpecialSourceList( dellist, ss1->row );
        }
      }
      if(nodatabase==true) delete tcod;
    } else {
      if ( dowin != NULL ) dowin->setfilename( ss1->entry()->fullname );
      ce1 = delfile( ss1->entry(), delorder, false );
      if(ce1==NM_COPY_OK) {
        // success
        if(ss1->row>=0) lv->showRow(ss1->row);
        removeEntry( ss1->entry(), ss1->row, true );
        fixSpecialSourceList( dellist, ss1->row );
      } else if(ce1==NM_COPY_CANCEL) {
        cancel=true;
      }
    }
    if ( ss1->row >= 0 ) {
      // this can and will changed the wrong row when the entry
      // was deleted successfully
      // but for this case this doesn't matter because
      // the disabled visual mark is the default
      lv->setVisMark( ss1->row, false );
    }
    if(dowin!=NULL) {
      if(dowin->redraw()!=0) cancel=true;
    }
  }
  if(dowin!=NULL) dowin->close();
  
  for ( iti1 = dellist->begin(); iti1 != dellist->end(); iti1++ ) {
    ss1 = *iti1;
    if(ss1->cod!=NULL) delete ss1->cod;
    delete ss1;
  }
  delete dellist;

  // now rebuilt the "pos in filelist" <-> "fe->nr" <-> "lvc->data" relation
  oldx=lv->getXOffset();
  oldy=lv->getYOffset();
  removeEntry(NULL,-1,false);
  ce->restartcheck();
  buildLister();
  lv->setXOffset(oldx);
  lv->setYOffset(oldy);
  lv->redraw();
  aguix->Flush();
}

void
NormalMode::removeEntry(FileEntry *fe, int row,bool quick)
{
  ArrayList *filelist;
  int pos,tid;
  FileEntry *tfe;
  int trow;
  bool searchcheckfe, searchfirstnullft;
  
  if(ce==NULL) return;
  if(ce->verz==NULL) return;
  filelist=ce->verz->getFiles();
  if(filelist==NULL) return;
  
#ifdef WANT_THREADS
  // we will remove an entry so clear this list
  // so it won't contain wrong pointers
  ft_rec_list_clear();
#endif
  
  if(fe!=NULL) {
    // if it's the active one then move the active up by one
    if(fe==ce->activefe) {
      pos=filelist->getIndex(ce->activefe);
      tid=filelist->initEnum();
      do {
        pos--;
        tfe=(FileEntry*)filelist->getElementAt(tid,pos);
        if(tfe==NULL) break;
      } while(tfe->use==false);
      filelist->closeEnum(tid);
      if(tfe!=NULL) {
        // Voriges Element gefunden
        ce->activefe=tfe;
        trow = row - 1;
        if(trow>=0) {
        //TODO: activerow auf -1 setzen, wenn nicht gefunden?
          lv->setActiveRow( trow);
          lv->showActive();
        }
      }
    }
    if ( fe == ce->checkfe )
      searchcheckfe = true;
    else searchcheckfe = false;
    if ( fe == ce->firstnullft )
      searchfirstnullft = true;
    else searchfirstnullft = false;
    
    // the following destroy the relation between the pos in filelist and lvc->data
    // but at the end I will rebuilt it
    // remove the lvc
    lv->deleteRow(row);
    lv->redraw();
    // remove the FileEntry
    filelist->removeElement(fe);
    ce->subentry(fe);
    delete fe;

    if ( searchcheckfe == true )
      ce->reset_checkfe();
    if ( searchfirstnullft == true )
      ce->reset_firstnullft();

    // display it
    showCacheState();
  }
  if(quick==false) {
    // now fix the structures
    // now rebuilt the "pos in filelist" <-> "fe->nr" <-> "lvc->data" relation
    trow=0;
    tid=filelist->initEnum();
    tfe=(FileEntry*)filelist->getFirstElement(tid);
    pos=0;
    while(tfe!=NULL) {
      if(tfe->use==true) {
        if(tfe->nr==lv->getData(trow)) {
          lv->setData(trow,pos);
/*          if(tfe->select==true) tlvc->setSelectQ(true); else tlvc->setSelectQ(false);
          if(tfe==ce->activefe) {
             tlvc->setActiveQ(true);
          }*/
        } else {
          // this should never happen
          debugmsg("error1 in removeEntry\n");
        }
        trow++;
      }
      tfe->nr=pos;
      tfe=(FileEntry*)filelist->getNextElement(tid);
      pos++;
    }
    filelist->closeEnum(tid);
  }
}

NormalMode::nm_copy_t
NormalMode::deldir(const FileEntry *fe,NM_CopyOp_Dir *cod,struct NM_deleteorder *delorder)
{
  ArrayList *filelist;
  NM_CopyOp_Dir *cod1=NULL,*tcod;
  int id1,id2;
  FileEntry *subfe;
  bool isdir,skip,cancel;
//  char *buttonstr,*textstr;
//  int erg;
  nm_copy_t ce1;
  DeleteOpWin *dowin=delorder->dowin;
  bool nodatabase,delerror;

  if(delorder->delmode==delorder->NM_DELMODE_NODATABASE) nodatabase=true;
  else nodatabase=false;

  skip=cancel=false;

  if(dowin!=NULL) dowin->setfilename(fe->fullname);
  // first del all subdirs
  // then del all files 
  if(nodatabase==false) {
    id1=cod->subdirs->initEnum();
    cod1=(NM_CopyOp_Dir*)cod->subdirs->getFirstElement(id1);
    while(cod1!=NULL) {
      ce1=deldir(cod1->fileentry,cod1,delorder);
      if(ce1==NM_COPY_OK) {
        // success
        cod->error_counter+=cod1->error_counter;
      } else {
        cod->error_counter+=cod1->error_counter+1;
        if(ce1==NM_COPY_CANCEL) {
          cancel=true;
          break;
        }
      }
      cod1=(NM_CopyOp_Dir*)cod->subdirs->getNextElement(id1);
    }
    cod->subdirs->closeEnum(id1);
  }

  // next only if read this dir correctly
  if((cod->ok==true)&&(cancel==false)) {
    filelist=cod->verz->getFiles();
    id2=filelist->initEnum();
    subfe=(FileEntry*)filelist->getFirstElement(id2);
    while(subfe!=NULL) {
      if(strcmp(subfe->name,"..")!=0) {
        isdir=false;
        if(subfe->isDir()==true) {
          if(subfe->isLink==false) isdir=true;
        }
        if(isdir==false) {
          ce1=delfile(subfe,delorder,false);
          if(ce1==NM_COPY_OK) {
            // success
          } else {
            cod->error_counter++;
            if(ce1==NM_COPY_CANCEL) {
              cancel=true;
              break;
            }
          }
        } else {
          if(nodatabase==true) {
            // no database so copy the dir
            tcod=new NM_CopyOp_Dir(subfe);
            if ( tcod->user_abort == false ) {
              ce1=deldir(subfe,tcod,delorder);
              if(ce1==NM_COPY_OK) {
                // success
                cod->error_counter+=tcod->error_counter;
              } else {
                cod->error_counter+=tcod->error_counter+1;
                if(ce1==NM_COPY_CANCEL) {
                  cancel=true;
                }
              }
            } else cancel = true;
            delete tcod;  
            if(cancel==true) break;
          }
        }
      }
      subfe=(FileEntry*)filelist->getNextElement(id2);
    }
    filelist->closeEnum(id2);
  }
  if((skip==true)&&(dowin!=NULL)) {
    // dir skiped so dec CopyOpWin counter by the files/dirs skiped
    dowin->dec_file_counter(cod->files);
    dowin->dec_dir_counter(cod->dirs);
  }
  if ( cod->verz->getInvalidEntries() == NULL ) {
    // in difference to copydir don't skip but try to delete the directory
    // this case will happen when the directory isn't readable or executeable
    // if this unreadable dir is empty the delete will work
    //skip = true;
  } else {
    if ( cod->verz->getInvalidEntries()->size() > 0 ) {
      // when there are some invalid files mark this dir as skipped
      // But I do this after the test above because this is no user initiated
      // skip so all normal files are deleted (if everything was okay)
      skip = true;
    }
  }
  delerror=true;
  if((skip==false)&&(cancel==false)) {
    if(cod->error_counter==0) {
      ce1=delfile(fe,delorder,true);
      if(ce1==NM_COPY_OK) {
        // success
        delerror=false;
      } else {
        cod->error_counter++;
        if(ce1==NM_COPY_CANCEL) {
          cancel=true;
        }
      }
    }
  }
  if(delerror==true) {
    char *textstr=(char*)_allocsafe(strlen(catalog.getLocale(289))+strlen(fe->fullname)+1);
    sprintf(textstr,catalog.getLocale(289),fe->fullname);
    char *buttonstr=(char*)_allocsafe(strlen(catalog.getLocale(11))+1+
                                strlen(catalog.getLocale(8))+1);
    sprintf(buttonstr,"%s|%s",catalog.getLocale(11),
                              catalog.getLocale(8));
    int erg = request( delorder->dowin, catalog.getLocale( 125 ), textstr, buttonstr );
    _freesafe(buttonstr);
    _freesafe(textstr);
    if(erg==1) cancel=true;
    else skip=true;
  }
  if(cancel==true) return NM_COPY_CANCEL;
  else if(skip==true) {
    cod->error_counter++;
    return NM_COPY_SKIP;
  }
  return NM_COPY_OK;
}

NormalMode::nm_copy_t
NormalMode::delfile(const FileEntry *fe,struct NM_deleteorder *delorder,bool dir)
{
  bool skip,cancel;
  char *textstr,*buttonstr;
  int erg;
  DeleteOpWin *dowin=delorder->dowin;

  skip=cancel=false;

  if(dir==true) erg=rmdir(fe->fullname);
  else erg=remove(fe->fullname);
  if(erg!=0) {
//    textstr="Failed to remove this file!";
//    buttonstr="Ok|Cancel";

    if(dir==true) {
      textstr=(char*)_allocsafe(strlen(catalog.getLocale(147))+strlen(fe->fullname)+1);
      sprintf(textstr,catalog.getLocale(147),fe->fullname);
    } else { 
      textstr=(char*)_allocsafe(strlen(catalog.getLocale(286))+strlen(fe->fullname)+1);
      sprintf(textstr,catalog.getLocale(286),fe->fullname);
    }
    buttonstr=(char*)_allocsafe(strlen(catalog.getLocale(11))+1+
                                strlen(catalog.getLocale(8))+1);
    sprintf(buttonstr,"%s|%s",catalog.getLocale(11),
                              catalog.getLocale(8));
    erg = request( delorder->dowin, catalog.getLocale( 347 ), textstr, buttonstr );
    _freesafe(buttonstr);
    _freesafe(textstr);
    if(erg==1) cancel=true;
    else skip=true;
  }
  if(dowin!=NULL) {
    if(dir==true) {
      dowin->dir_finished();
      if(dowin->redraw()!=0) cancel=true;
    } else dowin->file_finished();
  }
  if(cancel==true) return NM_COPY_CANCEL;
  else if(skip==true) return NM_COPY_SKIP;
  return NM_COPY_OK;
}

void
NormalMode::update(bool reset_dirsizes)
{
  update( reset_dirsizes, false );
}

void
NormalMode::update( bool reset_dirsizes, bool keep_filetypes )
{
  if(ce==NULL) return;
  char *dir,*tstr;
  int oldx,oldy,e;
  
  finishsearchmode();
  
  oldx=lv->getXOffset();
  oldy=lv->getYOffset();
  dir=dupstring(ce->verz->getDir());
  for(;;) {
    e=enterDir(dir);
    if(e==0) break;  // read ok, so finish
    if(strcmp(dir,"/")==0) break;  // there is no dir we can display anymore, so finish
    tstr=ParentDir(dir,NULL);
    _freesafe(dir);
    dir=tstr;
  }
  _freesafe(dir);
  if(ce!=NULL) {
    if ( keep_filetypes == true ) {
      ce->reset_checkfe();
    } else {
      ce->restartcheck();
    }
    if(reset_dirsizes==true) ce->reset_dirsizes();
  }
  buildLister();
  lv->setXOffset(oldx);
  lv->setYOffset(oldy);
  lv->redraw();
}

void
NormalMode::makedir()
{
  if(ce==NULL) return;
  int id;
  ArrayList *filelist;
  FileEntry *tfe;
  int row;
  char *buttonstr,*textstr,*return_str,*destdir;
  int erg;
  
  finishsearchmode();
  
  textstr = dupstring( catalog.getLocale(152) );
  buttonstr=(char*)_allocsafe(strlen(catalog.getLocale(11))+1+
                              strlen(catalog.getLocale(8))+1);
  sprintf(buttonstr,"%s|%s",catalog.getLocale(11),
                            catalog.getLocale(8));
  erg=req->string_request(catalog.getLocaleCom(27),textstr,"",buttonstr,&return_str);
  _freesafe(buttonstr);
  _freesafe( textstr );
  if((erg==0)&&(strlen(return_str)>0)) {
    destdir=(char*)_allocsafe(strlen(ce->verz->getDir())+1+strlen(return_str)+1);
    strcpy(destdir,ce->verz->getDir());
    if(strlen(destdir)>1) strcat(destdir,"/");
    strcat(destdir,return_str);

    if(mkdir(destdir,S_IRWXU|S_IRGRP|S_IXGRP|S_IROTH|S_IXOTH)!=0) {
      textstr=(char*)_allocsafe(strlen(catalog.getLocale(127))+strlen(destdir)+1);
      sprintf(textstr,catalog.getLocale(127),destdir);
      buttonstr = dupstring( catalog.getLocale(11) );
      erg=req->request(catalog.getLocale(347),textstr,buttonstr);
      _freesafe(textstr);
      _freesafe( buttonstr );
    } else {
      // update the dir
      update(false);
      // find the new dir
      filelist=ce->verz->getFiles();
      id=filelist->initEnum();
      tfe=(FileEntry*)filelist->getFirstElement(id);
      while(tfe!=NULL) {
        if(strcmp(tfe->name,return_str)==0) {
          // found
          ce->activefe=tfe;
          row = 0;
          while( row < lv->getElements() ) {
            if(lv->getData(row)==tfe->nr) {
              // lvc found
              lv->setActiveRow(row);
              break;
            }
            row++;
          }
          break;
        }
        tfe=(FileEntry*)filelist->getNextElement(id);
      }
      filelist->closeEnum(id);
      lv->showActive();
      lv->redraw();
    }
    _freesafe(destdir);
  }
  _freesafe(return_str);
}

void
NormalMode::externprog(class NM_externorder *extorder)
{
  char *tmpname,*tmpoutput,*exestr,*tstr;
  Datei *fp;
  int lastsize[2];
  bool useint;
  bool cancel=false;
  WPUContext *wpu;
  bool oldrec, oldtd;
  std::string string1;
  bool removeTMPFiles;

  if(extorder==NULL) return;
  if(extorder->com_str==NULL) return;  // nothing to do
  if ( extorder->wpu == NULL ) return;
  wpu = extorder->wpu;

  oldrec = wpu->getRecursive();
  oldtd = wpu->getTakeDirs();
  wpu->setRecursive( extorder->recursive );
  wpu->setTakeDirs( extorder->take_dirs );

  finishsearchmode();

  tmpname=Datei::createTMPName();
  tmpoutput=Datei::createTMPName();
  // runCommand will remove the tmpfiles but if we do runCommand
  // (because of an error) remove them here
  removeTMPFiles = true;
  fp=new Datei();
  if(fp->open(tmpname,"w")==0) {
    fp->putString("#! /bin/sh\n");
    if ( wpu->getBaseDir() != NULL ) {
      tstr = AGUIX_catTrustedAndUnTrusted( "cd ", wpu->getBaseDir() );
      fp->putString( tstr );
      _freesafe( tstr );
      fp->putString( "\n" );
    }
    lastsize[0] = wpu->filelistRealSize( false );
    lastsize[1] = wpu->filelistRealSize( true );
    for(int loopnr=1;;loopnr++) {
      exestr = wpu->parse( extorder->com_str, EXE_STRING_LEN - strlen( tmpoutput ) - 3 );
      if(exestr!=NULL) {
        string1 = Worker_secureCommandForShell( exestr );
        if ( string1.length() > 0 ) {
          if(extorder->extstart==extorder->NM_EXT_SHOW_OUTPUT) {
            tstr=(char*)_allocsafe(strlen(string1.c_str())+3+strlen(tmpoutput)+1);
            if(loopnr==1) sprintf(tstr,"%s >%s",string1.c_str(),tmpoutput);
            else sprintf(tstr,"%s >>%s",string1.c_str(),tmpoutput);
            fp->putString( tstr);
            _freesafe(tstr);
          } else {
            fp->putString( string1.c_str() );
          }
          fp->putString("\n");
        }
        _freesafe(exestr);
        if(extorder->separate_each_entry==false) break;
        if ( wpu->filelistRealSize( false ) == 0 ) break;
        if ( wpu->filelistRealSize( false ) == lastsize[0] ) {
          // endless loop
          //TODO: requester
          break;
        }
      } else {
        cancel=true;
        break;
      }
      lastsize[0] = wpu->filelistRealSize( false );
    }
    if(cancel==false) {
      if((extorder->extstart==extorder->NM_EXT_START_IN_TERMINAL_AND_WAIT4KEY)||
         (extorder->extstart==extorder->NM_EXT_START_IN_TERMINAL)) {
        /* in terminal starten */
        if(extorder->extstart==extorder->NM_EXT_START_IN_TERMINAL_AND_WAIT4KEY) {
          fp->putStringExt("echo %s\n",catalog.getLocale(201));
          fp->putString("read x\n");
        }
        exestr=(char*)_allocsafe(strlen(wconfig->getTerminalStr())+strlen(tmpname)+1);
        sprintf(exestr,wconfig->getTerminalStr(),tmpname);
      } else {
        exestr=dupstring(tmpname);
      }
      if(extorder->extstart==extorder->NM_EXT_SHOW_OUTPUT) {
        useint=true;
        if(extorder->view_str!=NULL)
          if(strlen(extorder->view_str)>2)
            if(strstr(extorder->view_str,"%s")!=NULL) useint=false;
        if(useint==true) fp->putStringExt(OUTPUT_BIN,tmpoutput);
        else fp->putStringExt(extorder->view_str,tmpoutput);
        fp->putString("\n");
      }
      fp->close();
      chmod(tmpname,0700);
      removeTMPFiles = false;
      runCommand( exestr, tmpname, tmpoutput, extorder->inbackground );
      _freesafe(exestr);
    } else fp->close();
  }
  delete fp;
  if ( removeTMPFiles == true ) {
    remove( tmpname );
    remove( tmpoutput );
  }
  _freesafe(tmpname);
  _freesafe(tmpoutput);

  wpu->setRecursive( oldrec );
  wpu->setTakeDirs( oldtd );
}

int NormalMode::create_externfe_rec(List *extlist,char *fullname,bool rec,const FileEntry *fe,int row) const
{
  ArrayList *f1;
  int id;
  NM_extern_fe *efe;
  Verzeichnis *tverz;
  FileEntry *tfe;
  
  if((Datei::fileExistsExt(fullname)==Datei::D_FE_DIR)&&(rec==true)) {
    // create all subentries
    tverz=new Verzeichnis();
    if(tverz->readDir(fullname)==0) {
      f1=tverz->getFiles();
      if(f1!=NULL) {
        id=f1->initEnum();
        tfe=(FileEntry*)f1->getFirstElement(id);
        while(tfe!=NULL) {
          if(strcmp(tfe->name,"..")!=0) {
            create_externfe_rec(extlist,tfe->fullname,rec,NULL,-1);
          }
          tfe=(FileEntry*)f1->getNextElement(id);
        }
        f1->closeEnum(id);
      }
    }
    delete tverz;
    // now append the dir
    efe=new NM_extern_fe(fullname,fe,row);
    efe->setDirFinished(1);  // this indicates the end of the recursive dir
                             // so it can be deactivated in the LV
    extlist->addElement(efe);
  } else {
    efe=new NM_extern_fe(fullname,fe,row);
    extlist->addElement(efe);
  }
  return 0;
}

int NormalMode::createExtList( std::list<NM_specialsourceExt*> *splist, bool rec, List *extlist) const
{
  NM_specialsourceExt *ss1;
  std::list<NM_specialsourceExt*>::iterator ite1;

  if ( splist == NULL ) return -1;

  for ( ite1 = splist->begin(); ite1 != splist->end(); ite1++ ) {
    ss1 = *ite1;
    create_externfe_rec( extlist, ss1->entry()->fullname, rec, ss1->entry(), ss1->row );
  }
  return 0;
}

#define PARSECAT( str1 ) \
do { \
  if ( ( quote == true ) && ( force_noquoting == false ) ) { \
    dstr[dpos] = '\0'; \
    newdstr = AGUIX_catTrustedAndUnTrusted( dstr, ( str1 ) ); \
    snprintf( dstr, maxlen, "%s", newdstr ); \
    _freesafe( newdstr ); \
    dstr[maxlen] = '\0'; \
    dpos = strlen( dstr ); \
    if ( dpos >= maxlen ) ende = true; \
  } else { \
    tlen = strlen( ( str1 ) ); \
    if ( ( dpos + tlen ) < maxlen ) { \
      strcpy( &dstr[dpos], ( str1 ) ); \
      dpos += strlen( ( str1 ) ); \
    } else ende = true; \
  } \
} while ( 0 );

#define PUTCHARDSTR( ch ) \
do { \
  if ( dpos < maxlen ) { \
    dstr[dpos++] = ch; \
  } else ende = true; \
} while ( 0 );

char *NormalMode::parseComStr(char *sstr,class NM_externorder *extorder,int maxlen,NMExtList *entries[2], bool quote )
{
  char *dstr = NULL, *buf1, ch, *newdstr;
  int spos=0,dpos=0,bracketcount=0,bufpos=0;
  int mode=0,myside;
  bool otherside,nounselect,noext;
  int filemode,tpos,tmode;
  NormalMode *nm2,*tnm;
  Lister *l2;
  ListerMode *lm1;
  int tlen;
  bool ende,takefirst;
  char *tstr;
  NM_extern_fe *efe1;
  NMExtList *l;
  int id;
  bool cancel=false;
  int num;
  std::string str1;
  char buf2[ A_BYTESFORNUMBER( int ) + 2 ];
  bool efeskip;
  char *flagbuf;
  int flagbuf_size;
  bool force_noquoting;
  
  nm2=NULL;
  l2=parentlister->getWorker()->getOtherLister(parentlister);
  lm1=l2->getActiveMode();
  if(lm1!=NULL)
    if(lm1->isType("NormalMode")==true)
      nm2=(NormalMode*)lm1;
  
  myside=parentlister->getSide();
  dstr=(char*)_allocsafe(maxlen+1);
  flagbuf_size = strlen( sstr );
  flagbuf = (char*)_allocsafe( flagbuf_size + 1 );
  if(dstr!=NULL) {
    for(ende=false;ende==false;) {
      ch=sstr[spos++];
      if(ch=='\0') break;
      switch(mode) {
        case 1:
          // we are in a open bracket and waiting for close bracket
          if(ch=='{') {
            // other flag, perhaps useless, but perhaps it is in a String-requester
            // in any case just overtake it in the buffer
            if ( bufpos < flagbuf_size ) flagbuf[ bufpos++ ] = ch;
            bracketcount++;
          } else if(ch=='}') {
            // a closeing bracket
            if(bracketcount>1) {
              // this is a bracket in the bracket
              if ( bufpos < flagbuf_size ) flagbuf[ bufpos++ ] = ch;
            } else {
              // this is the awaited closebracket
              flagbuf[ bufpos ] = 0;
              // now flagbuf contains a flag which must be parsed
              mode=3;
            }
            bracketcount--;
          } else if(ch=='\\') {
            // backslash are only resolved on toplevel (bracketcount==0)
            // so overtake also this
            if ( bufpos < flagbuf_size ) flagbuf[ bufpos++ ] = ch;
            mode=4;
          } else {
            if ( bufpos < flagbuf_size ) flagbuf[ bufpos++ ] = ch;
          }
          break;
        case 2:
          PUTCHARDSTR( ch );
          mode=0;
          break;
        case 4:
          if ( bufpos < flagbuf_size ) flagbuf[ bufpos++ ] = ch;
          mode=1;
          break;
        default:
          // we are in no bracket
          if(ch=='\\') {
            // next character is protected and will be overtaken
            mode=2;
          } else if(ch!='{') {
            // a normal character
            PUTCHARDSTR( ch );
          } else {
            mode=1;
            bracketcount++;
            bufpos=0;
          }
          break;
      }
      if(mode==3) {
        if ( flagbuf[0] == '-' ) {
          force_noquoting = true;
          buf1 = flagbuf + 1;
        } else {
          force_noquoting = false;
          buf1 = flagbuf;
        }
        // parse buf1
        if ( strncmp( buf1, "Rs", 2 ) == 0 ) {
          // Stringrequest
          tstr = StringRequest( buf1, extorder, maxlen - dpos, entries );
          if(tstr!=NULL) {
            PARSECAT( tstr );
            _freesafe(tstr);
          } else {
            cancel=true;
            ende=true;
          }
        } else if ( strncmp( buf1, "scripts", 7 ) == 0 ) {
          // dir of the scripts
          str1 = Worker::getDataDir();
          str1 += "/scripts";
          PARSECAT( str1.c_str() );
        } else if ( strncmp( buf1, "pop", 3 ) == 0 ) {
          if ( strlen( buf1 ) > 3 ) {
            num = atoi( buf1 + 3 );
            if ( num < 0 ) num = 0;
          } else {
            num = 0;
          }
          if ( extorder->wpu != NULL ) {
            if ( extorder->wpu->size( num ) > 0 ) {
              str1 = extorder->wpu->pop( num );
              PARSECAT( str1.c_str() );
            }
          }
        } else if ( strncmp( buf1, "top", 3 ) == 0 ) {
          if ( strlen( buf1 ) > 3 ) {
            num = atoi( buf1 + 3 );
            if ( num < 0 ) num = 0;
          } else {
            num = 0;
          }
          if ( extorder->wpu != NULL ) {
            if ( extorder->wpu->size( num ) > 0 ) {
              str1 = extorder->wpu->top( num );
              PARSECAT( str1.c_str() );
            }
          }
        } else if ( strncmp( buf1, "size", 4 ) == 0 ) {
          if ( strlen( buf1 ) > 4 ) {
            num = atoi( buf1 + 4 );
            if ( num < 0 ) num = 0;
          } else {
            num = 0;
          }
          if ( extorder->wpu != NULL ) {
            sprintf( buf2, "%d", extorder->wpu->size( num ) );
            str1 = buf2;
            PARSECAT( str1.c_str() );
          }
        } else if ( strncmp( buf1, "p", 1 ) == 0 ) {
          // path
          tstr = getCurrentDir();
          if(tstr!=NULL) {
            PARSECAT( tstr );
          }
        } else if ( strncmp( buf1, "op", 2 ) == 0 ) {
          // other side
          if(nm2==NULL) tstr=NULL;
          else tstr = nm2->getCurrentDir();
          if(tstr!=NULL) {
            PARSECAT( tstr );
          }
        } else if ( strncmp( buf1, "lp", 2 ) == 0 ) {
          // left side
          if(myside==0) tnm=this;
          else tnm=nm2;
          if(tnm==NULL) tstr=NULL;
          else tstr = tnm->getCurrentDir();
          if(tstr!=NULL) {
            PARSECAT( tstr );
          }
        } else if ( strncmp( buf1, "rp", 2 ) == 0 ) {
          // right side
          if(myside==1) tnm=this;
          else tnm=nm2;
          if(tnm==NULL) tstr=NULL;
          else tstr = tnm->getCurrentDir();
          if(tstr!=NULL) {
            PARSECAT( tstr );
          }
        } else if ( strncmp( buf1, "dp", 2 ) == 0 ) {
          // destination path
          if(extorder->destmode==NULL) tstr=NULL;
          else tstr = extorder->destmode->getCurrentDir();
          if(tstr!=NULL) {
            PARSECAT( tstr );
          }
        } else if ( strncmp( buf1, "v", 1 ) == 0 ) {
          // env variable
          tstr = getenv( buf1 + 1 );
          if ( tstr != NULL ) {
            PARSECAT( tstr );
          }
        } else {
          // file flag or unknown
          filemode=-1;
          otherside=false;
          nounselect=false;
          noext=false;
          tpos=0;
          tmode=0;
          for(;(tmode<4)&&(tpos<4);) {
            ch = buf1[ tpos++ ];
            if(ch=='o') {
              if(tmode<1) {
                otherside=true;
                tmode=1;
              } else {
                // an "o" at the wrong position
                tmode=6;
              }
            } else if(ch=='u') {
              if(tmode<2) {
                nounselect=true;
                tmode=2;
              } else {
                tmode=6;
              }
            } else if((ch=='f')&&(tmode<3)) {
              filemode=0;
              tmode=3;
            } else if((ch=='a')&&(tmode<3)) {
              filemode=1;
              tmode=3;
            } else if((ch=='F')&&(tmode<3)) {
              filemode=2;
              tmode=3;
            } else if((ch=='A')&&(tmode<3)) {
              filemode=3;
              tmode=3;
            } else if((ch=='E')&&(tmode==3)) {
              noext=true;
              tmode=4;
            } else if(ch=='\0') tmode=4;
          }
          if((filemode>=0)&&(filemode<4)) {
            // we have a valid flag
            takefirst=true;
            if((filemode==1)||(filemode==3)) takefirst=false;
            if(otherside==true) l=entries[1];
            else l=entries[0];
            id=l->initEnum();
            efe1=(NM_extern_fe*)l->getFirstElement(id);
            for(int c=0;;) {
              if(efe1==NULL) break;
              
              efeskip = false;
              if ( ( efe1->getDirFinished() != 0 ) &&
                   ( extorder->take_dirs == false ) ) {
                efeskip = true;
              } else {
                // this is a valid entry
                if(filemode<2) tstr=efe1->getName(noext);
                else tstr=efe1->getFullname(noext);

                if ( tstr != NULL ) {
                  if ( c > 0 ) PUTCHARDSTR( ' ' );
                  if ( dpos >= maxlen ) ende = true;
                  PARSECAT( tstr );
                }               
                if(ende==true) break;
              }
              for (;;) {
                if(nounselect==false) {
                  // remove efe1;
                  l->removeElement( efe1 );
                  // now deactivate LVC if not NULL
                  if ( efe1->getFE() != NULL ) {
                    if(otherside==false) {
                      // in my list
		      deselect( efe1->getFE() );
                    } else {
		      if ( nm2 != NULL ) nm2->deselect( efe1->getFE() );
                    }
                  }
                  delete efe1;
                  efe1=(NM_extern_fe*)l->getFirstElement(id);
                } else {
                  efe1=(NM_extern_fe*)l->getNextElement(id);
                }
                if ( efe1 == NULL ) break;
                if ( ( efe1->getDirFinished() == 0 ) ||
                     ( extorder->take_dirs == true ) ) {
                  break;
                }
              }

              if ( efeskip == true ) continue;
              if((c==0)&&(takefirst==true)) break;
              c=1;
            }
            l->closeEnum(id);
          }
        }
        mode=0;
      }
    }
    dstr[dpos]=0;
  }
  _freesafe( flagbuf );
  if(cancel==true) {
    _freesafe(dstr);
    return NULL;
  }
  return dstr;
}

void NormalMode::deselect( FileEntry *fe, int row)
{
  if((lv->isValidRow(row)==false)||(fe==NULL)) return;
  if(ce==NULL) return;
  if(fe->select==false) return;
  if(lv->isValidRow(row)) {
    lv->setSelect(row,false);
    lv->showRow(row);
  }
  fe->select=false;
  ce->recalcStats();
  showCacheState();
}

char *NormalMode::StringRequest(char *sstr,class NM_externorder *extorder,int maxlen,NMExtList *entries[2])
{
  char *istr=NULL,ch;
  char *defstr=NULL;
  char *ristr=NULL,*rdefstr=NULL;
  int spos=0,bracketcount=0;
  int istart,iend,defstart,defend;
  int mode;
  char *buttonstr,*textstr,*return_str;
  int erg;
  
  spos=2; // buf1 starts with Rs
  istart=iend=defstart=defend=-1;
  mode=0;
  for(;mode<4;) {
    ch=sstr[spos++];
    if(ch=='\0') break;
    switch(mode) {
      case 0:
        // we are in no bracket
        if(ch=='{') {
          mode=1;
          bracketcount++;
          istart=spos;
        }
        break;
      case 1:
        // in the infotext
        if(ch=='{') bracketcount++;
        else if(ch=='}') {
          if(bracketcount==1) {
            iend=spos-2;
            mode=2;
          }
          bracketcount--;
        }
        break;
      case 2:
        // we are in no bracket
        if(ch=='{') {
          mode=3;
          bracketcount++;
          defstart=spos;
        }
        break;
      case 3:
        // in the defaulttext
        if(ch=='{') bracketcount++;
        else if(ch=='}') {
          if(bracketcount==1) {
            defend=spos-2;
            mode=4;
          }
          bracketcount--;
        }
        break;
      default:
        break;
    }
  }
  if((istart>=0)&&(iend>=istart)) {
    // we have an infotext
    istr=dupstring(&sstr[istart]);
    istr[iend-istart+1]=0;
    ristr=parseComStr(istr,extorder,maxlen,entries, false );
    _freesafe(istr);
  }
  if((defstart>=0)&&(defend>=istart)) {
    // we have a defaulttext
    defstr=dupstring(&sstr[defstart]);
    defstr[defend-defstart+1]=0;
    rdefstr=parseComStr(defstr,extorder,maxlen,entries, false );
    _freesafe(defstr);
  }
  
  textstr=NULL;
  if(ristr!=NULL) {
    if(strlen(ristr)>0) textstr=ristr;
    else _freesafe(ristr);
  }
  if(textstr==NULL) textstr=dupstring(catalog.getLocale(200));
  buttonstr=(char*)_allocsafe(strlen(catalog.getLocale(11))+1+
                              strlen(catalog.getLocale(8))+1);
  sprintf(buttonstr,"%s|%s",catalog.getLocale(11),
                            catalog.getLocale(8));
  erg=req->string_request(catalog.getLocale(123),textstr,
                          (rdefstr!=NULL)?rdefstr:"",
                          buttonstr,&return_str);
  _freesafe(buttonstr);
  _freesafe(textstr);
  if(rdefstr!=NULL) _freesafe(rdefstr);
  if(erg!=0) {
    _freesafe(return_str);
    return NULL;
  }
  return return_str;
}

int NormalMode::getEntriesRealSize(List*l)
{
  int id=l->initEnum(),rs;
  NM_extern_fe *efe1;
  efe1=(NM_extern_fe*)l->getFirstElement(id);
  rs=0;
  while(efe1!=NULL) {
    if(efe1->getDirFinished()==0) rs++;
    efe1=(NM_extern_fe*)l->getNextElement(id);
  }
  l->closeEnum(id);
  return rs;
}

void NormalMode::renamef(struct NM_renameorder *renorder)
{
  Verzeichnis *verz;
  NM_specialsourceExt *sse;
  NM_specialsourceInt *ss1, *ss2;
  ArrayList *filelist;
  FileEntry *fe;
  bool found;
  int row;
  int erg;
  bool skip,cancel,nameok;
  char *newname,*buttonstr,*textstr,*newfullname,*return_str;
  const std::vector<WorkerTypes::listcol_t> *dis;
  std::list<NM_specialsourceExt*>::iterator ite1;
  std::list<NM_specialsourceInt*> *renlist;
  std::list<NM_specialsourceInt*>::iterator iti1;
#if 0
  int oldx,oldy;
#endif

  if ( getCurrentDir() == NULL ) {
    // no current dir => no files to rename
    return;
  }

  finishsearchmode();
  
  verz=ce->verz;
  filelist=verz->getFiles();

  dis = wconfig->getVisCols( parentlister->getSide() );
  
  renlist = new std::list<NM_specialsourceInt*>;
  switch(renorder->source) {
    case NM_renameorder::NM_SPECIAL:
      for ( ite1 = renorder->sources->begin(); ite1 != renorder->sources->end(); ite1++ ) {
	sse = *ite1;
        fe=(FileEntry*)filelist->getFirstElement();
        found=false;
        while(fe!=NULL) {
          if ( fe == sse->entry() ) {
            if(strcmp(fe->name,"..")!=0) {
              found=true;
              break;
            }
          }
          fe=(FileEntry*)filelist->getNextElement();
        }
        if(found==true) {
          // valid entry found
	  ss2 = new NM_specialsourceInt( fe );
          ss2->row=sse->row;
          ss2->cod=NULL;
          renlist->push_back( ss2 );
        }
      }
      break;
    case NM_renameorder::NM_ONLYACTIVE:
      if(ce->activefe!=NULL) {
        row=lv->getActiveRow();
        if(lv->isValidRow(row)==true) {
          if(lv->getData(row)==ce->activefe->nr) {
            if(strcmp(ce->activefe->name,"..")!=0) {
	      ss2 = new NM_specialsourceInt( ce->activefe );
              ss2->row=row;
              ss2->cod=NULL;
              renlist->push_back(ss2);
            }
          }
        }
      }
      break;
    default:  // all selected entries
      fe=(FileEntry*)filelist->getFirstElement();
      row = 0;
      while(fe!=NULL) {
        if(fe->use==true) {
          if((strcmp(fe->name,"..")!=0)&&(fe->select==true)) {
	    ss2 = new NM_specialsourceInt( fe );
            ss2->row=row;
            ss2->cod=NULL;
            renlist->push_back(ss2);
          }
          row++;
        }
        fe=(FileEntry*)filelist->getNextElement();
      }
      if(renlist->size()==0) {
        // no selected entries => take the active
        if(ce->activefe!=NULL) {
          row=lv->getActiveRow();
          if(lv->isValidRow(row)==true) {
            if(lv->getData(row)==ce->activefe->nr) {
              if(strcmp(ce->activefe->name,"..")!=0) {
		ss2 = new NM_specialsourceInt( ce->activefe );
                ss2->row=row;
                ss2->cod=NULL;
                renlist->push_back(ss2);
              }
            }
          }
        }
      }
      break;
  }
  
  cancel=skip=false;

  for ( iti1 = renlist->begin(); iti1 != renlist->end(); iti1++ ) {
    if ( cancel == true ) break;
    ss1 = *iti1;
    // now rename ss1->entry
    if ( ss1->row >= 0 ) {
      lv->setVisMark( ss1->row, true );
      lv->showRow( ss1->row );
    }
    fe = ss1->entry();
    if(fe!=NULL) {
      newname=dupstring(fe->name);
      newfullname=NULL;
      for(nameok=false;nameok==false;) {
        buttonstr=(char*)_allocsafe(strlen(catalog.getLocale(11))+1+
                                    strlen(catalog.getLocale(225))+1+
                                    strlen(catalog.getLocale(8))+1);
        sprintf(buttonstr,"%s|%s|%s",catalog.getLocale(11),
                                     catalog.getLocale(225),
                                     catalog.getLocale(8));
        textstr=(char*)_allocsafe(strlen(catalog.getLocale(148))+strlen(fe->name)+1);
        sprintf(textstr,catalog.getLocale(148),fe->name);
        erg=req->string_request(catalog.getLocale(149),textstr,newname,buttonstr,&return_str, Requester::REQUEST_SELECTALL );
        _freesafe(buttonstr);
        _freesafe(textstr);
        _freesafe(newname);
        newname=return_str;
        if(erg==2) {
          cancel=true;
          break;
        } else if(erg==1) {
          break;
        } else {
          if(strcmp(newname,fe->name)==0) {
            // same name as orig
            // so skip this entry
            break;
          } else if ( strlen( newname ) < 1 ) {
            // empty string->repeat
            continue;
          }
          newfullname = (char*)_allocsafe( strlen( getCurrentDir() ) + 1 + strlen( newname ) + 1 );
          if ( strlen( getCurrentDir() ) > 1 ) sprintf( newfullname, "%s/%s", getCurrentDir(), newname );
          else sprintf( newfullname, "%s%s", getCurrentDir(), newname );
          if(Datei::lfileExistsExt(newfullname)!=Datei::D_FE_NOFILE) {
            textstr=(char*)_allocsafe(strlen(catalog.getLocale(279))+strlen(newname)+1);
            sprintf(textstr,catalog.getLocale(279),newname);
            buttonstr=(char*)_allocsafe(strlen(catalog.getLocale(230))+1+
                                        strlen(catalog.getLocale(225))+1+
                                        strlen(catalog.getLocale(8))+1);
            sprintf(buttonstr,"%s|%s|%s",catalog.getLocale(230),
                                         catalog.getLocale(225),
                                         catalog.getLocale(8));
            erg=req->request(catalog.getLocale(123),textstr,buttonstr);
            _freesafe(buttonstr);
            _freesafe(textstr);
            if(erg==1) break;
            else if(erg==2) {
              cancel=true;
              break;
            }
            _freesafe(newfullname);
            newfullname=NULL;
          } else {
            nameok=true;
          }
        }
      }
      if((nameok==true)&&(newfullname!=NULL)) {
        erg=rename(fe->fullname,newfullname);
        if(erg!=0) {
          textstr = dupstring( catalog.getLocale(151) );
          buttonstr=(char*)_allocsafe(strlen(catalog.getLocale(11))+1+
                                      strlen(catalog.getLocale(8))+1);
          sprintf(buttonstr,"%s|%s",catalog.getLocale(11),
                                    catalog.getLocale(8));
          erg=req->request(catalog.getLocale(347),textstr,buttonstr);
          _freesafe(buttonstr);
          _freesafe( textstr );
          if(erg==1) cancel=true;
        } else {
          // deselect and change lvc temp.
          if(lv->isValidRow(ss1->row)==true) deselect(fe,ss1->row);
	  //TODO: Man koennte eine helper-Funktion einfuehren, um
	  //      name/fullname zu setzen
          _freesafe(fe->name);
          _freesafe(fe->fullname);
          fe->name=newname;
          newname=NULL;
          if(fe->name[0]=='.') fe->isHidden=true;
          else fe->isHidden=false;          
          fe->fullname=newfullname;
          newfullname=NULL;
          if(lv->isValidRow(ss1->row)==true) {
            //TODO: Eigentlich muesste hier auch das DND neugesetzt werden, aber ich weiss nicht die Position
            // d.h. ich muesste erst suchen.
            // allerdings kann das auch so bleiben,da am Ende ohnehin buildLister gemacht wird
            fe->setLVC( lv, ss1->row, dis );
            lv->redraw();
            aguix->Flush();
          }
        }
      }
      if(newname!=NULL) _freesafe(newname);
      if(newfullname!=NULL) _freesafe(newfullname);
    }
    if ( ss1->row >= 0 ) {
      lv->setVisMark( ss1->row, false );
    }
  }
  
  for ( iti1 = renlist->begin(); iti1 != renlist->end(); iti1++ ) {
    ss1 = *iti1;
    delete ss1;
  }
  delete renlist;
  // now update
  
#if 0
  oldx=lv->getXOffset();
  oldy=lv->getYOffset();
  verz->sort(sortmode,true);
  ce->restartcheck();
  check(ce);
  buildLister();
  lv->setYOffset(oldy);
  lv->setXOffset(oldx);
  lv->showActive();
  lv->redraw();
#else
  update(false);
  lv->showActive();
#endif
}

void NormalMode::dirsize(struct NM_dirsizeorder *dsorder)
{
  Verzeichnis *verz;
  NM_specialsourceExt *sse;
  NM_specialsourceInt *ss1, *ss2;
  ArrayList *filelist;
  FileEntry *fe;
  bool found;
  int row;
  bool enter;
  const std::vector<WorkerTypes::listcol_t> *dis;
  int oldx,oldy;
  std::list<NM_specialsourceExt*>::iterator ite1;
  std::list<NM_specialsourceInt*> *dslist;
  std::list<NM_specialsourceInt*>::iterator iti1;
  
  finishsearchmode();
  
  oldx=lv->getXOffset();
  oldy=lv->getYOffset();

  if(dsorder==NULL) return;
  if ( getCurrentDir() == NULL ) return;
  verz=ce->verz;
  filelist=verz->getFiles();
  
  dis = wconfig->getVisCols( parentlister->getSide() );
  
  dslist = new std::list<NM_specialsourceInt*>;
  switch(dsorder->source) {
    case NM_dirsizeorder::NM_SPECIAL:
      for ( ite1 = dsorder->sources->begin(); ite1 != dsorder->sources->end(); ite1++ ) {
	sse = *ite1;
        fe=(FileEntry*)filelist->getFirstElement();
        found=false;
        while(fe!=NULL) {
          if ( fe == sse->entry() ) {
            if(strcmp(fe->name,"..")!=0) {
              found=true;
              break;
            }
          }
          fe=(FileEntry*)filelist->getNextElement();
        }
        if(found==true) {
          // valid entry found
	  ss2 = new NM_specialsourceInt( fe );
          ss2->row=sse->row;
          ss2->cod=NULL;
          dslist->push_back(ss2);
        }
      }
      break;
    case NM_dirsizeorder::NM_ONLYACTIVE:
      if(ce->activefe!=NULL) {
        row=lv->getActiveRow();
        if(lv->isValidRow(row)==true) {
          if(lv->getData(row)==ce->activefe->nr) {
            if(strcmp(ce->activefe->name,"..")!=0) {
	      ss2 = new NM_specialsourceInt( ce->activefe );
              ss2->row=row;
              ss2->cod=NULL;
              dslist->push_back(ss2);
            }
          }
        }
      }
      break;
    default:  // all selected entries
      fe=(FileEntry*)filelist->getFirstElement();
      row = 0;
      while(fe!=NULL) {
        if(fe->use==true) {
          if((strcmp(fe->name,"..")!=0)&&(fe->select==true)) {
	    ss2 = new NM_specialsourceInt( fe );
            ss2->row=row;
            ss2->cod=NULL;
            dslist->push_back(ss2);
          }
          row++;
        }
        fe=(FileEntry*)filelist->getNextElement();
      }
      if(dslist->size()==0) {
        // no selected entries => take the active
        if(ce->activefe!=NULL) {
          row=lv->getActiveRow();
          if(lv->isValidRow(row)==true) {
            if(lv->getData(row)==ce->activefe->nr) {
              if(strcmp(ce->activefe->name,"..")!=0) {
		ss2 = new NM_specialsourceInt( ce->activefe );
                ss2->row=row;
                ss2->cod=NULL;
                dslist->push_back(ss2);
              }
            }
          }
        }
      }
      break;
  }

  // create the NM_CopyOp_Dir for each dir in copylist
  
  for ( iti1 = dslist->begin(); iti1 != dslist->end(); iti1++ ) {
    ss1 = *iti1;
    enter=false;
    if ( ss1->entry()->isDir() == true ) {
      if ( ss1->entry()->isLink == false ) enter = true;
    }
    if(enter==true) {
      if ( getDirSize( ss1->entry(), ss1->entry()->dirsize ) == 0 ) {
        if(lv->isValidRow(ss1->row)==true) {
#if 1
          ss1->entry()->setLVC( lv, ss1->row, dis );
          lv->redraw();
#else
          buildLister();
          lv->setYOffset(oldy);
          lv->setXOffset(oldx);
          lv->redraw();
#endif
          aguix->Flush();
        }
      } else {
        ss1->entry()->dirsize = -1;
      }
    }
  }

  for ( iti1 = dslist->begin(); iti1 != dslist->end(); iti1++ ) {
    ss1 = *iti1;
    delete ss1;
  }
  delete dslist;
  // now update
  verz->sort(sortmode,true);
  ce->restartcheck();
  checkFilters( ce );
  buildLister();
  lv->setYOffset(oldy);
  lv->setXOffset(oldx);
  showCacheState();
  lv->redraw();
}

int
NormalMode::getDirSize( FileEntry*rootfe, loff_t &return_size )
{
  ArrayList *f1;
  int id;
  FileEntry *fe;
  bool enter;
  loff_t size,recsize;
  Verzeichnis *verz=new Verzeichnis();
  
  if(verz->readDir(rootfe->fullname)!=0) {
    delete verz;
    return -1;
  }
  
  f1=verz->getFiles();
  if(f1==NULL) {
    delete verz;
    return -1;
  }
  parentlister->setStatebarText(rootfe->fullname);
  id=f1->initEnum();
  fe=(FileEntry*)f1->getFirstElement(id);
  size=0;
  while(fe!=NULL) {
    if(strcmp(fe->name,"..")!=0) {
      enter=false;
      if(fe->isDir()==true) {
        if(fe->isLink==false) enter=true;
      }
      if(enter==true) {
        recsize = 0;
        if ( getDirSize( fe, recsize ) == 0 ) {
          if(recsize>0) size+=recsize;
        }
      } else {
        // is not dir (mostly a file but can also be links ...)
        size += fe->size();
      }
    }
    fe=(FileEntry*)f1->getNextElement(id);
  }
  f1->closeEnum(id);
  delete verz;
  return_size = size;
  return 0;
}

void NormalMode::simdd()
{
  if ( getCurrentDir() == NULL ) {
    return;
  }
  if(ce->activefe!=NULL) startAction(ce->activefe);
}

void
NormalMode::startprog(struct NM_startprogorder *sporder)
{
  FileEntry *fe;
  char *tmpname,*tmpoutput,*exestr, *tstr, *tstr2;
  char *textstr,*buttonstr,*return_str;
  Datei *fp;
  bool useint;
  int erg;
  std::string string1;

  if(sporder==NULL) return;

  finishsearchmode();
  
  fe = findFE( sporder->usefe );
  if(fe==NULL)
    if(ce!=NULL)
      if(ce->activefe!=NULL) fe=ce->activefe;
  if(fe!=NULL) {
    textstr = dupstring( catalog.getLocale( 522 ) );
    buttonstr=(char*)_allocsafe(strlen(catalog.getLocale(11))+1+
                                strlen(catalog.getLocale(8))+1);
    sprintf(buttonstr,"%s|%s",catalog.getLocale(11),
                              catalog.getLocale(8));
#if 0
    // this code convert the filename to singlequoted
    tstr = AGUIX_prepareForSingleQuote( fe->name );
    tstr2 = (char*)_allocsafe( strlen( tstr ) + 2 + 1 );
    sprintf( tstr2, "'%s'", tstr );
    _freesafe( tstr );
#else
    // this code just protect all necessary chars in filename
    tstr2 = AGUIX_catQuotedAndUnQuoted( "", fe->name );
#endif
    erg=req->string_request(catalog.getLocale(123),textstr,
                            tstr2,buttonstr,&return_str);
    _freesafe( tstr2 );
    _freesafe(buttonstr);
    _freesafe( textstr );
    if(erg==0) {
      tmpname=Datei::createTMPName();
      tmpoutput=Datei::createTMPName();
      fp=new Datei();
      if(fp->open(tmpname,"w")==0) {
        fp->putString("#! /bin/sh\n");
        if ( getCurrentDir() != NULL ) {
          tstr = AGUIX_catTrustedAndUnTrusted( "cd ", getCurrentDir() );
          fp->putString( tstr );
          _freesafe( tstr );
          fp->putString( "\n" );
        }
        if(sporder->global==false) fp->putString("./");

        string1 = Worker_secureCommandForShell( return_str );
        if ( string1.length() > 0 ) {
          if ( sporder->start == sporder->NM_SHOW_OUTPUT ) {
            tstr = (char*)_allocsafe( strlen( string1.c_str() ) + 3 + strlen( tmpoutput ) + 1 );
            sprintf( tstr, "%s >%s", string1.c_str(), tmpoutput );
            fp->putString( tstr );
            _freesafe( tstr );
          } else {
            fp->putString( string1.c_str() );
          }
          fp->putString("\n");
        }
        if((sporder->start==sporder->NM_START_IN_TERMINAL_AND_WAIT4KEY)||
           (sporder->start==sporder->NM_START_IN_TERMINAL)) {
          /* in terminal starten */
          if(sporder->start==sporder->NM_START_IN_TERMINAL_AND_WAIT4KEY) {
            fp->putStringExt("echo %s\n",catalog.getLocale(201));
            fp->putString("read x\n");
          }
          exestr=(char*)_allocsafe(strlen(wconfig->getTerminalStr())+strlen(tmpname)+1);
          sprintf(exestr,wconfig->getTerminalStr(),tmpname);
        } else {
          exestr=dupstring(tmpname);
        }
        if(sporder->start==sporder->NM_SHOW_OUTPUT) {
          useint=true;
          if(sporder->view_str!=NULL)
            if(strlen(sporder->view_str)>2)
              if(strstr(sporder->view_str,"%s")!=NULL) useint=false;
          if(useint==true) fp->putStringExt(OUTPUT_BIN,tmpoutput);
          else fp->putStringExt(sporder->view_str,tmpoutput);
          fp->putString("\n");
        }
        fp->close();
        chmod(tmpname,0700);
        runCommand( exestr, tmpname, tmpoutput, sporder->inbackground );
        _freesafe(exestr);
      }
      delete fp;
      _freesafe(tmpname);
      _freesafe(tmpoutput);
    }
    _freesafe(return_str);
  }
}

void NormalMode::searchentry()
{
  searchentry( false );
}

void NormalMode::searchentry( bool ignore_case )
{
  searchmode_ignore_case = ignore_case;
  if(searchmodeon==false) startsearchmode();
  else {
    shownextentry();
  }
}

void NormalMode::startsearchmode()
{
  if ( getCurrentDir() == NULL ) {
    return;
  }
  oldstr=dupstring(sg->getText());
  lastsearch=dupstring("");
  sg->setText("");
  sg->setForbidPosChange(true);
  sg->setStrongKeyCapture(false);
  sg->activate();
  searchmodeon=true;
}

void NormalMode::finishsearchmode()
{
  if(searchmodeon==false) return;
  sg->setForbidPosChange(false);
  sg->setStrongKeyCapture(true);
  sg->setText(oldstr);
  sg->deactivate();
  _freesafe(oldstr);
  if(lastsearch!=NULL) _freesafe(lastsearch);
  else debugmsg("oops2\n");
  lastsearch=NULL;
  searchmodeon=false;
}

bool NormalMode::showfirstentry(const char *str)
{
  Verzeichnis *verz;
  ArrayList *filelist;
  FileEntry *startfe;
  bool found;
  int pos;
  
  found=false;

  if ( getCurrentDir() == NULL ) {
    return found;
  }
  verz=ce->verz;
  filelist=verz->getFiles();
  
  startfe=ce->activefe;
  pos=0;
  if(startfe!=NULL) pos=filelist->getIndex(startfe);
  return shownextentry(str,pos);
}

bool NormalMode::shownextentry()
{
  Verzeichnis *verz;
  ArrayList *filelist;
  FileEntry *startfe;
  bool found;
  int pos;
  
  found=false;

  if ( getCurrentDir() == NULL ) {
    return found;
  }
  verz=ce->verz;
  filelist=verz->getFiles();
  
  startfe=ce->activefe;
  pos=0;
  if(startfe!=NULL) pos=filelist->getIndex(startfe)+1;
  return shownextentry( lastsearch, pos );
}

bool NormalMode::shownextentry( const char *str, int pos )
{
  Verzeichnis *verz;
  ArrayList *filelist;
  FileEntry *fe;
  bool found;
  int row;
  int id;
  char *realstr, *tstr1, *tstr2;
  
  found=false;

  if ( getCurrentDir() == NULL ) {
    return found;
  }
  verz=ce->verz;
  filelist=verz->getFiles();
  
  id=filelist->initEnum();
  
  if(pos>=0) {
    realstr=(char*)_allocsafe(strlen(str)+1+1);
    sprintf(realstr,"%s*",str);
    if ( searchmode_ignore_case == true ) {
      for ( tstr1 = realstr; *tstr1 != '\0'; tstr1++ ) {
        *tstr1 = tolower( *tstr1 );
      }
    }
    for(int i=0;(i<2)&&(found==false);i++) {
      fe=(FileEntry*)filelist->getElementAt(id,pos);
      while((fe!=NULL)&&(found==false)) {
        if(fe->use==true) {
          tstr1 = dupstring( fe->name );
          if ( searchmode_ignore_case == true ) {
            for ( tstr2 = tstr1; *tstr2 != '\0'; tstr2++ ) {
              *tstr2 = tolower( *tstr2 );
            }
          }
          if ( fnmatch( realstr, tstr1, 0 ) == 0 ) {
            // hit
            ce->activefe=fe;
            row = 0;
            while(lv->isValidRow(row)==true) {
              if(lv->getData(row)==fe->nr) {
                lv->setActiveRow( row);
                break;
              }
              row++;
            }
            found=true;
          }
          _freesafe( tstr1 );
        }
        fe=(FileEntry*)filelist->getNextElement(id);
      }
      pos=0; // if not found try from pos 0 again
    }
    _freesafe(realstr);
  }
  if(found==true) {
    lv->showActive();
    lv->redraw();
    aguix->Flush();
  }
  
  filelist->closeEnum(id);
  return found;
}

void NormalMode::enterpath()
{
  sg->activate();
}

void NormalMode::scrolllister(int dir)
{
  lv->setXOffset(lv->getXOffset()+dir);
  aguix->Flush();
}

void NormalMode::setActiveDir2Other()
{
  Lister *l1;
  ListerMode *lm1;
  NormalMode *nm1;

  if(ce==NULL) return;
  if(ce->activefe!=NULL) {
    if(ce->activefe->isDir()==true) {
      l1=parentlister->getWorker()->getOtherLister(parentlister);
      if(l1!=NULL) {
        l1->switch2Mode(0);
        lm1=l1->getActiveMode();
        if(lm1!=NULL) {
          if(lm1->isType("NormalMode")==true) {
	    std::string s1( ce->activefe->fullname );

            nm1=(NormalMode*)lm1;
            nm1->enterDir( s1.c_str() );
          }
        }
      }
    }
  }
}

void NormalMode::createsymlink(struct NM_createsymlinkorder *cslorder)
{
  Verzeichnis *verz;
  NM_specialsourceExt *sse;
  NM_specialsourceInt *ss1, *ss2;
  ArrayList *filelist;
  FileEntry *fe;
  bool found;
  int row;
  int erg;
  bool skip,cancel,nameok;
  char *newname,*buttonstr,*textstr,*newfullname,*return_str;
  char *sldest;
  Lister *olister=NULL;
  NormalMode *nm2=NULL,*updatenm=NULL;
  ListerMode *lm=NULL;
  std::list<NM_specialsourceExt*>::iterator ite1;
  std::list<NM_specialsourceInt*> *csllist;
  std::list<NM_specialsourceInt*>::iterator iti1;

  if ( getCurrentDir() == NULL ) {
    // no current dir => no files to rename
    return;
  }

  finishsearchmode();
  
  if ( strcmp( getCurrentDir(), cslorder->destdir ) == 0 ) {
    updatenm=this;
  }
  olister=parentlister->getWorker()->getOtherLister(parentlister);
  if(olister==NULL) return;
  lm=olister->getActiveMode();
  if(lm!=NULL) {
    if(lm->isType("NormalMode")==true) {
      nm2=(NormalMode*)lm;
      if ( nm2->getCurrentDir() != NULL ) {
        if ( strcmp( nm2->getCurrentDir(), cslorder->destdir ) == 0 ) {
          updatenm=nm2;
        }
      }
    }
  }

  verz=ce->verz;
  filelist=verz->getFiles();

  csllist = new std::list<NM_specialsourceInt*>;
  switch(cslorder->source) {
    case NM_createsymlinkorder::NM_SPECIAL:
      for ( ite1 = cslorder->sources->begin(); ite1 != cslorder->sources->end(); ite1++ ) {
	sse = *ite1;
        fe=(FileEntry*)filelist->getFirstElement();
        found=false;
        while(fe!=NULL) {
          if ( fe == sse->entry() ) {
            if(strcmp(fe->name,"..")!=0) {
              found=true;
              break;
            }
          }
          fe=(FileEntry*)filelist->getNextElement();
        }
        if(found==true) {
          // valid entry found
	  ss2 = new NM_specialsourceInt( fe );
          ss2->row=sse->row;
          ss2->cod=NULL;
          csllist->push_back(ss2);
        }
      }
      break;
    case NM_createsymlinkorder::NM_ONLYACTIVE:
      if(ce->activefe!=NULL) {
        row=lv->getActiveRow();
        if(lv->isValidRow(row)==true) {
          if(lv->getData(row)==ce->activefe->nr) {
            if(strcmp(ce->activefe->name,"..")!=0) {
	      ss2 = new NM_specialsourceInt( ce->activefe );
              ss2->row=row;
              ss2->cod=NULL;
              csllist->push_back(ss2);
            }
          }
        }
      }
      break;
    default:  // all selected entries
      fe=(FileEntry*)filelist->getFirstElement();
      row = 0;
      while(fe!=NULL) {
        if(fe->use==true) {
          if((strcmp(fe->name,"..")!=0)&&(fe->select==true)) {
	    ss2 = new NM_specialsourceInt( fe );
            ss2->row=row;
            ss2->cod=NULL;
            csllist->push_back(ss2);
          }
          row++;
        }
        fe=(FileEntry*)filelist->getNextElement();
      }
      if(csllist->size()==0) {
        // no selected entries => take the active
        if(ce->activefe!=NULL) {
          row=lv->getActiveRow();
          if(lv->isValidRow(row)==true) {
            if(lv->getData(row)==ce->activefe->nr) {
              if(strcmp(ce->activefe->name,"..")!=0) {
		ss2 = new NM_specialsourceInt( ce->activefe );
                ss2->row=row;
                ss2->cod=NULL;
                csllist->push_back(ss2);
              }
            }
          }
        }
      }
      break;
  }
  
  cancel=skip=false;
  for ( iti1 = csllist->begin(); iti1 != csllist->end(); iti1++ ) {
    if ( cancel == true ) break;
    ss1 = *iti1;
    fe = ss1->entry();
    if ( ss1->row >= 0 ) {
      lv->setVisMark( ss1->row, true );
      lv->showRow( ss1->row );
    }
    if(fe!=NULL) {
      newname=dupstring(fe->name);
      newfullname=NULL;
      for(nameok=false;nameok==false;) {
        buttonstr=(char*)_allocsafe(strlen(catalog.getLocale(11))+1+
                                    strlen(catalog.getLocale(225))+1+
                                    strlen(catalog.getLocale(8))+1);
        sprintf(buttonstr,"%s|%s|%s",catalog.getLocale(11),
                                     catalog.getLocale(225),
                                     catalog.getLocale(8));
        textstr=(char*)_allocsafe(strlen(catalog.getLocale(207))+strlen(fe->name)+1);
        sprintf(textstr,catalog.getLocale(207),fe->name);
        erg=req->string_request(catalog.getLocaleCom(36),textstr,newname,buttonstr,&return_str);
        _freesafe(buttonstr);
        _freesafe(textstr);
        _freesafe(newname);
        newname=return_str;
        if(erg==2) {
          cancel=true;
          break;
        } else if(erg==1) {
          break;
        } else {
          newfullname=(char*)_allocsafe(strlen(cslorder->destdir)+1+strlen(newname)+1);
          if(strlen(cslorder->destdir)>1) sprintf(newfullname,"%s/%s",cslorder->destdir,newname);
          else sprintf(newfullname,"%s%s",cslorder->destdir,newname);
          if(Datei::lfileExistsExt(newfullname)!=Datei::D_FE_NOFILE) {
            textstr=(char*)_allocsafe(strlen(catalog.getLocale(279))+strlen(newname)+1);
            sprintf(textstr,catalog.getLocale(279),newname);
            buttonstr=(char*)_allocsafe(strlen(catalog.getLocale(230))+1+
                                        strlen(catalog.getLocale(225))+1+
                                        strlen(catalog.getLocale(8))+1);
            sprintf(buttonstr,"%s|%s|%s",catalog.getLocale(230),
                                         catalog.getLocale(225),
                                         catalog.getLocale(8));
            erg=req->request(catalog.getLocale(123),textstr,buttonstr);
            _freesafe(buttonstr);
            _freesafe(textstr);
            if(erg==1) break;
            else if(erg==2) {
              cancel=true;
              break;
            }
            _freesafe(newfullname);
            newfullname=NULL;
          } else {
            nameok=true;
          }
        }
      }
      if((nameok==true)&&(newfullname!=NULL)) {
        // now create symlink newfullname which points to fe->fullname or fe->name
        if(cslorder->local==true) {
          sldest = Datei::getRelativePath( fe->fullname, cslorder->destdir );
          if ( sldest == NULL ) {
            //TODO: I currently can't imagine a situation but the call
            //      can return NULL (shouldn't happen because I give
            //      correct info) so in this case just use the fullname
            //      But give the user a requester?
            sldest = dupstring( fe->fullname );
          }
        } else sldest = dupstring( fe->fullname );
        erg=symlink(sldest,newfullname);
        _freesafe( sldest );
        if(erg!=0) {
          textstr=(char*)_allocsafe(strlen(catalog.getLocale(205))+strlen(newfullname)+1);
          sprintf(textstr,catalog.getLocale(205),newfullname);
          buttonstr=(char*)_allocsafe(strlen(catalog.getLocale(11))+1+
                                      strlen(catalog.getLocale(8))+1);
          sprintf(buttonstr,"%s|%s",catalog.getLocale(11),
                                    catalog.getLocale(8));
          erg=req->request(catalog.getLocale(347),textstr,buttonstr);
          _freesafe(buttonstr);
          _freesafe(textstr);
          if(erg==1) cancel=true;
        } else {
          // deselect
          if(lv->isValidRow(ss1->row)==true) deselect(fe,ss1->row);
        }
      }
      if(newname!=NULL) _freesafe(newname);
      if(newfullname!=NULL) _freesafe(newfullname);
    }
    if ( ss1->row >= 0 ) {
      lv->setVisMark( ss1->row, false );
    }
  }
  
  for ( iti1 = csllist->begin(); iti1 != csllist->end(); iti1++ ) {
    ss1 = *iti1;
    delete ss1;
  }
  delete csllist;
  // now update
  if(updatenm!=NULL) updatenm->update(false);
}

void NormalMode::changesymlink(struct NM_changesymlinkorder *chslorder)
{
  Verzeichnis *verz;
  NM_specialsourceExt *sse;
  NM_specialsourceInt *ss1, *ss2;
  ArrayList *filelist;
  FileEntry *fe;
  bool found;
  int row;
  int erg;
  bool cancel;
  char *buttonstr,*textstr,*return_str;
  char buffer[4096];
  int tx;
  std::list<NM_specialsourceExt*>::iterator ite1;
  std::list<NM_specialsourceInt*> *csllist;
  std::list<NM_specialsourceInt*>::iterator iti1;

  if ( getCurrentDir() == NULL ) {
    // no current dir => no files to rename
    return;
  }

  finishsearchmode();
  
  verz=ce->verz;
  filelist=verz->getFiles();

  csllist = new std::list<NM_specialsourceInt*>;
  switch(chslorder->source) {
    case NM_changesymlinkorder::NM_SPECIAL:
      for ( ite1 = chslorder->sources->begin(); ite1 != chslorder->sources->end(); ite1++ ) {
	sse = *ite1;
        fe=(FileEntry*)filelist->getFirstElement();
        found=false;
        while(fe!=NULL) {
          if ( fe == sse->entry() ) {
            if(strcmp(fe->name,"..")!=0) {
              found=true;
              break;
            }
          }
          fe=(FileEntry*)filelist->getNextElement();
        }
        if(found==true) {
          // valid entry found
	  ss2 = new NM_specialsourceInt( fe );
          ss2->row=sse->row;
          ss2->cod=NULL;
          csllist->push_back(ss2);
        }
      }
      break;
    case NM_changesymlinkorder::NM_ONLYACTIVE:
      if(ce->activefe!=NULL) {
        row=lv->getActiveRow();
        if(lv->isValidRow(row)==true) {
          if(lv->getData(row)==ce->activefe->nr) {
            if(strcmp(ce->activefe->name,"..")!=0) {
	      ss2 = new NM_specialsourceInt( ce->activefe );
              ss2->row=row;
              ss2->cod=NULL;
              csllist->push_back(ss2);
            }
          }
        }
      }
      break;
    default:  // all selected entries
      fe=(FileEntry*)filelist->getFirstElement();
      row = 0;
      while(fe!=NULL) {
        if(fe->use==true) {
          if((strcmp(fe->name,"..")!=0)&&(fe->select==true)) {
	    ss2 = new NM_specialsourceInt( fe );
            ss2->row=row;
            ss2->cod=NULL;
            csllist->push_back(ss2);
          }
          row++;
        }
        fe=(FileEntry*)filelist->getNextElement();
      }
      if(csllist->size()==0) {
        // no selected entries => take the active
        if(ce->activefe!=NULL) {
          row=lv->getActiveRow();
          if(lv->isValidRow(row)==true) {
            if(lv->getData(row)==ce->activefe->nr) {
              if(strcmp(ce->activefe->name,"..")!=0) {
		ss2 = new NM_specialsourceInt( ce->activefe );
                ss2->row=row;
                ss2->cod=NULL;
                csllist->push_back(ss2);
              }
            }
          }
        }
      }
      break;
  }
  
  cancel=false;

  for ( iti1 = csllist->begin(); iti1 != csllist->end(); iti1++ ) {
    if ( cancel == true ) break;
    ss1 = *iti1;
    fe = ss1->entry();
    if ( ss1->row >= 0 ) {
      lv->setVisMark( ss1->row, true );
      lv->showRow( ss1->row );
    }
    if(fe!=NULL) {
      if(fe->isLink==true) {
        tx=readlink(fe->fullname,buffer,4095);
        if(tx>=0) {
          buffer[tx]=0;
          buttonstr=(char*)_allocsafe(strlen(catalog.getLocale(11))+1+
                                      strlen(catalog.getLocale(225))+1+
                                      strlen(catalog.getLocale(8))+1);
          sprintf(buttonstr,"%s|%s|%s",catalog.getLocale(11),
                                       catalog.getLocale(225),
                                       catalog.getLocale(8));
          textstr=(char*)_allocsafe(strlen(catalog.getLocale(206))+strlen(fe->name)+1);
          sprintf(textstr,catalog.getLocale(206),fe->name);
          erg=req->string_request(catalog.getLocaleCom(37),textstr,buffer,buttonstr,&return_str);
          _freesafe(buttonstr);
          _freesafe(textstr);
          if(erg==2) {
            // cancel
            cancel=true;
          } else if(erg==0) {
            // ok
            if(strcmp(buffer,return_str)!=0) {
              // only if new value differs
              if(remove(fe->fullname)==0) {
                if(symlink(return_str,fe->fullname)==0) {
                  // deselect
                  if(lv->isValidRow(ss1->row)==true) deselect(fe,ss1->row);
                } else {
                  textstr=(char*)_allocsafe(strlen(catalog.getLocale(205))+strlen(fe->fullname)+1);
                  sprintf(textstr,catalog.getLocale(205),fe->fullname);
                  buttonstr=(char*)_allocsafe(strlen(catalog.getLocale(11))+1+
                                              strlen(catalog.getLocale(8))+1);
                  sprintf(buttonstr,"%s|%s",catalog.getLocale(11),
                                            catalog.getLocale(8));
                  erg=req->request(catalog.getLocale(347),textstr,buttonstr);
                  _freesafe(buttonstr);
                  _freesafe(textstr);
                  if(erg==1) cancel=true;
                }
              } else {
                textstr=(char*)_allocsafe(strlen(catalog.getLocale(291))+strlen(fe->fullname)+1);
                sprintf(textstr,catalog.getLocale(291),fe->fullname);
                buttonstr = dupstring( catalog.getLocale(11) );
                req->request(catalog.getLocale(347),textstr,buttonstr);
                _freesafe(textstr);
                _freesafe( buttonstr );
              }
            }
          }
          _freesafe(return_str);
        }
      }
    }
    if ( ss1->row >= 0 ) {
      lv->setVisMark( ss1->row, false );
    }
  }
  
  for ( iti1 = csllist->begin(); iti1 != csllist->end(); iti1++ ) {
    ss1 = *iti1;
    delete ss1;
  }
  delete csllist;
}

void NormalMode::chmodf(struct NM_chmodorder *cmorder)
{
  Verzeichnis *verz;
  NM_specialsourceExt *sse;
  NM_specialsourceInt *ss1, *ss2;
  ArrayList *filelist;
  FileEntry *fe;
  bool found;
  changemod_info_t *cminfo;
  std::list<NM_specialsourceExt*>::iterator ite1;
  std::list<NM_specialsourceInt*> *cmlist;
  std::list<NM_specialsourceInt*>::iterator iti1;
   
  if(cmorder==NULL) return;
  if ( getCurrentDir() == NULL ) {
    // no current dir => no files to copy
    return;
  }

  finishsearchmode();
  
  verz=ce->verz;
  filelist=verz->getFiles();
  
  cmlist = new std::list<NM_specialsourceInt*>;
  switch(cmorder->source) {
    case NM_chmodorder::NM_SPECIAL:
      for ( ite1 = cmorder->sources->begin(); ite1 != cmorder->sources->end(); ite1++ ) {
	sse = *ite1;
        fe=(FileEntry*)filelist->getFirstElement();
        found=false;
        while(fe!=NULL) {
          if ( fe == sse->entry() ) {
            if(strcmp(fe->name,"..")!=0) {
              found=true;
              break;
            }
          }
          fe=(FileEntry*)filelist->getNextElement();
        }
        if(found==true) {
          // valid entry found
	  ss2 = new NM_specialsourceInt( fe );
          ss2->row=sse->row;
          ss2->cod=NULL;
          cmlist->push_back(ss2);
        }
      }
      break;
    case NM_chmodorder::NM_ONLYACTIVE:
      getSelFiles( cmlist, NM_GETFILES_ONLYACTIVE, false );
      break;
    default:  // all selected entries
      getSelFiles( cmlist, NM_GETFILES_SELORACT, false );
      break;
  }

  cminfo = new changemod_info_t;

  for ( iti1 = cmlist->begin(); iti1 != cmlist->end(); iti1++ ) {
    ss1 = *iti1;
    if ( worker_changemod( ss1, cminfo, cmorder ) != 0 ) {
      break;
    }
  }
  for ( iti1 = cmlist->begin(); iti1 != cmlist->end(); iti1++ ) {
    ss1 = *iti1;
    delete ss1;
  }
  delete cminfo;
  delete cmlist;
  update(false);
}

int NormalMode::worker_changemod( const struct NM_specialsourceInt *ss1,
                                  changemod_info_t *cminfo,
                                  const NM_chmodorder *cmorder )
{
  bool enter, dontChange, skip, cancel, changemodFirst;
  int id, erg;
  mode_t newmode = 0700;
  FileEntry *subfe;
  NM_specialsourceInt *ss2;
  ArrayList *filelist;
  NM_CopyOp_Dir *cod;
  
  if ( ( ss1 == NULL ) || ( cminfo == NULL ) || ( cmorder == NULL ) ) return -1;

  // do we have to enter this entry?
  enter = false;
  if ( ( ss1->entry()->isDir() == true ) && ( cmorder->recursive == true ) ) {
    if ( ss1->entry()->isLink == false ) enter = true;
  }
  
  // check operation applies to this entry
  dontChange = false;
  if ( ( ss1->entry()->isDir() == true ) && ( cmorder->ondirs == false ) ) dontChange = true;
  else if ( ( ss1->entry()->isDir() == false ) && ( cmorder->onfiles == false ) ) dontChange = true;

  /* skip means skip entry AND all sub entries
   * cancel is clear
   * dontchange means to normally handle this entry
   *   but actually don't change mod
   */
  skip = cancel = false;
  if ( dontChange == false ) {
    // ask for new mode
    if ( cminfo->forAll == true ) {
      newmode = cminfo->newmode;
    } else {
      erg = requestNewMode( ss1->entry(), &newmode );
      if ( erg == 1 ) {
        cminfo->forAll = true;
        cminfo->newmode = newmode;
      } else if ( erg == 2 ) {
        skip = true;
      } else if ( erg == 3 ) {
        cancel = true;
      }
    }
  }
  if ( skip == true ) return 0;
  if ( cancel == true ) return 1;

  if ( ss1->entry()->checkAccess( R_OK|W_OK|X_OK ) != true ) {
    // cannot access directory to change mod so apply
    // newmode first
    changemodFirst = true;
  } else {
    changemodFirst = false;
  }
  if ( ( dontChange == false ) && ( changemodFirst == true ) ) {
    if ( applyNewMode( ss1, newmode ) != 0 ) cancel = true;
  }
  if ( ( enter == true ) && ( cancel == false ) ) {
    cod = new NM_CopyOp_Dir( ss1->entry() );
    if ( cod->user_abort == true ) {
      cancel = true;
    } else if ( cod->ok == true ) {
      filelist = cod->verz->getFiles();
      id = filelist->initEnum();
      subfe = (FileEntry*)filelist->getFirstElement( id );
      while ( ( subfe != NULL ) && ( cancel == false ) ) {
        if ( strcmp( subfe->name, ".." ) != 0 ) {
	  ss2 = new NM_specialsourceInt( subfe );
          ss2->row = -1;
          if ( worker_changemod( ss2, cminfo, cmorder ) != 0 ) {
            cancel = true;
          }
          delete ss2;
        }
        subfe = (FileEntry*)filelist->getNextElement( id );
      }
      filelist->closeEnum( id );
    }
    delete cod;
  }
  if ( ( dontChange == false ) && ( changemodFirst == false ) ) {
    if ( applyNewMode( ss1, newmode ) != 0 ) cancel = true;
  }
  return ( cancel == true ) ? 1 : 0;
}

int NormalMode::applyNewMode( const NM_specialsourceInt *ss1, mode_t newmode )
{
  char *textstr, *buttonstr;
  int erg;
  bool cancel = false;
  
  if ( ss1 == NULL ) return -1;

  if ( chmod( ss1->entry()->fullname, newmode ) != 0 ) {
    // error
    buttonstr = (char*)_allocsafe( strlen( catalog.getLocale( 11 ) ) + 1 +
                                   strlen( catalog.getLocale( 8 ) ) + 1 );
    sprintf( buttonstr, "%s|%s", catalog.getLocale( 11 ),
                                 catalog.getLocale( 8 ) );
    textstr = (char*)_allocsafe( strlen( catalog.getLocale( 226 ) ) + strlen( ss1->entry()->fullname ) + 1 );
    sprintf( textstr, catalog.getLocale( 226 ), ss1->entry()->fullname );
    erg = req->request( catalog.getLocale( 347 ), textstr, buttonstr );
    _freesafe( buttonstr );
    _freesafe( textstr );
    if ( erg == 1 ) cancel = true;
  } else {
    if ( lv->isValidRow( ss1->row ) == true ) deselect( ss1->entry(), ss1->row );
  }
  return ( cancel == true ) ? 1 : 0;
}

int NormalMode::requestNewMode(FileEntry *fe,mode_t *return_mode)
{
  mode_t tmode,newmode;
  Button *okb,*cb,*ok2allb,*skipb;
  AWindow *win;
  Text *ttext,*readtext,*writetext,*exetext;
  Text *owntext,*grouptext,*othtext,*spetext;
  Text *uidtext,*gidtext,*stickytext;
  ChooseButton *mcb[12];
  int mw[5],tx[3],maxw;
  int tw,ttw,tth,ttx,tty;
  AGMessage *msg;
  int endmode=-1;
  char *tstr;
  GUIElement *ba[4];
  
  tmode = ( ( fe->isLink == true ) && ( fe->isCorrupt == false ) ) ? fe->dmode() : fe->mode();
  ttw=tth=10;
  ttx=tty=5;
  win=new AWindow(aguix);
  win->create(NULL,10,10,ttw,tth,0,catalog.getLocaleCom(38));
  tstr=(char*)_allocsafe(strlen(catalog.getLocale(213))+strlen(fe->fullname)+1);
  sprintf(tstr,catalog.getLocale(213),fe->fullname);

  ttext=(Text*)win->add(new Text(aguix,ttx,tty,tstr,1));
  _freesafe(tstr);
  mw[4]=ttext->getWidth();
  
  tty+=ttext->getHeight()+5;
  
  readtext=(Text*)win->add(new Text(aguix,ttx,tty,catalog.getLocale(218),1));
  writetext=(Text*)win->add(new Text(aguix,ttx,tty,catalog.getLocale(219),1));
  exetext=(Text*)win->add(new Text(aguix,ttx,tty,catalog.getLocale(220),1));
  tty+=exetext->getHeight()+5;
  
  owntext=(Text*)win->add(new Text(aguix,ttx,tty,catalog.getLocale(214),1));
  grouptext=(Text*)win->add(new Text(aguix,ttx,tty,catalog.getLocale(215),1));
  othtext=(Text*)win->add(new Text(aguix,ttx,tty,catalog.getLocale(216),1));
  spetext=(Text*)win->add(new Text(aguix,ttx,tty,catalog.getLocale(217),1));
  maxw=owntext->getWidth();
  tw=grouptext->getWidth();
  if(tw>maxw) maxw=tw;
  tw=othtext->getWidth();
  if(tw>maxw) maxw=tw;
  tw=spetext->getWidth();
  if(tw>maxw) maxw=tw;
  mw[0]=maxw;
  
  uidtext=(Text*)win->add(new Text(aguix,ttx,tty,catalog.getLocale(221),1));
  gidtext=(Text*)win->add(new Text(aguix,ttx,tty,catalog.getLocale(222),1));
  stickytext=(Text*)win->add(new Text(aguix,ttx,tty,catalog.getLocale(223),1));
  
  maxw=uidtext->getWidth();
  tw=readtext->getWidth();
  if(tw>maxw) maxw=tw;
  mw[1]=maxw;
  maxw=gidtext->getWidth();
  tw=writetext->getWidth();
  if(tw>maxw) maxw=tw;
  mw[2]=maxw;
  maxw=stickytext->getWidth();
  tw=exetext->getWidth();
  if(tw>maxw) maxw=tw;
  mw[3]=maxw;
  
  // now move all elements
  owntext->move(ttx,tty);
  grouptext->move(ttx,owntext->getY()+owntext->getHeight()+5);
  othtext->move(ttx,grouptext->getY()+grouptext->getHeight()+5);
  spetext->move(ttx,othtext->getY()+othtext->getHeight()+5+uidtext->getHeight()+5);
  
  ttx+=mw[0]+5;
  tx[0]=ttx+mw[1]/2;  // center of first column
  readtext->move(tx[0]-readtext->getWidth()/2,readtext->getY());
  ttx+=mw[1]+5;
  tx[1]=ttx+mw[2]/2;
  writetext->move(tx[1]-writetext->getWidth()/2,writetext->getY());
  ttx+=mw[2]+5;
  tx[2]=ttx+mw[3]/2;
  exetext->move(tx[2]-exetext->getWidth()/2,exetext->getY());

  tty=othtext->getY()+othtext->getHeight()+5;
  
  // now create the ChooseButtons
  tty = readtext->getY() + readtext->getHeight() + 5;
  mcb[0]=(ChooseButton*)win->add(new ChooseButton(aguix,tx[0], tty,
                                ((tmode&S_IRUSR)==0)?0:1,
                                "",LABEL_RIGHT,0,0));
  mcb[1]=(ChooseButton*)win->add(new ChooseButton(aguix,tx[1], tty,
                                ((tmode&S_IWUSR)==0)?0:1,
                                "",LABEL_RIGHT,0,0));
  mcb[2]=(ChooseButton*)win->add(new ChooseButton(aguix,tx[2], tty,
                                ((tmode&S_IXUSR)==0)?0:1,
                                "",LABEL_RIGHT,0,0));
  AGUIX::centerElementsY( owntext, mcb[0] );
  mcb[0]->move( tx[0] - mcb[0]->getWidth() / 2, mcb[0]->getY() );
  mcb[1]->move( tx[1] - mcb[1]->getWidth() / 2, mcb[1]->getY() );
  mcb[2]->move( tx[2] - mcb[2]->getWidth() / 2, mcb[2]->getY() );
  tty = mcb[0]->getY() + mcb[0]->getHeight() + 5;

  mcb[3]=(ChooseButton*)win->add(new ChooseButton(aguix,tx[0], tty,
                                ((tmode&S_IRGRP)==0)?0:1,
                                "",LABEL_RIGHT,0,0));
  mcb[4]=(ChooseButton*)win->add(new ChooseButton(aguix,tx[1], tty,
                                ((tmode&S_IWGRP)==0)?0:1,
                                "",LABEL_RIGHT,0,0));
  mcb[5]=(ChooseButton*)win->add(new ChooseButton(aguix,tx[2], tty,
                                ((tmode&S_IXGRP)==0)?0:1,
                                "",LABEL_RIGHT,0,0));
  AGUIX::centerElementsY( grouptext, mcb[3] );
  mcb[3]->move( tx[0] - mcb[3]->getWidth() / 2, mcb[3]->getY() );
  mcb[4]->move( tx[1] - mcb[4]->getWidth() / 2, mcb[4]->getY() );
  mcb[5]->move( tx[2] - mcb[5]->getWidth() / 2, mcb[5]->getY() );
  tty = mcb[3]->getY() + mcb[3]->getHeight() + 5;

  mcb[6]=(ChooseButton*)win->add(new ChooseButton(aguix,tx[0], tty,
                                ((tmode&S_IROTH)==0)?0:1,
                                "",LABEL_RIGHT,0,0));
  mcb[7]=(ChooseButton*)win->add(new ChooseButton(aguix,tx[1], tty,
                                ((tmode&S_IWOTH)==0)?0:1,
                                "",LABEL_RIGHT,0,0));
  mcb[8]=(ChooseButton*)win->add(new ChooseButton(aguix,tx[2], tty,
                                ((tmode&S_IXOTH)==0)?0:1,
                                "",LABEL_RIGHT,0,0));
  AGUIX::centerElementsY( othtext, mcb[6] );
  mcb[6]->move( tx[0] - mcb[6]->getWidth() / 2, mcb[6]->getY() );
  mcb[7]->move( tx[1] - mcb[7]->getWidth() / 2, mcb[7]->getY() );
  mcb[8]->move( tx[2] - mcb[8]->getWidth() / 2, mcb[8]->getY() );
  tty = mcb[6]->getY() + mcb[6]->getHeight() + 5;

  uidtext->move( tx[0] - uidtext->getWidth() / 2, tty );
  gidtext->move( tx[1] - gidtext->getWidth() / 2, tty );
  stickytext->move( tx[2] - stickytext->getWidth() / 2, tty );
  tty += uidtext->getHeight() + 5;

  mcb[9]=(ChooseButton*)win->add(new ChooseButton(aguix,tx[0], tty,
                                ((tmode&S_ISUID)==0)?0:1,
                                "",LABEL_RIGHT,0,0));
  mcb[10]=(ChooseButton*)win->add(new ChooseButton(aguix,tx[1], tty,
                                ((tmode&S_ISGID)==0)?0:1,
                                "",LABEL_RIGHT,0,0));
  mcb[11]=(ChooseButton*)win->add(new ChooseButton(aguix,tx[2], tty,
                                ((tmode&S_ISVTX)==0)?0:1,
                                "",LABEL_RIGHT,0,0));
  AGUIX::centerElementsY( spetext, mcb[9] );
  mcb[9]->move( tx[0] - mcb[9]->getWidth() / 2, mcb[9]->getY() );
  mcb[10]->move( tx[1] - mcb[10]->getWidth() / 2, mcb[10]->getY() );
  mcb[11]->move( tx[2] - mcb[11]->getWidth() / 2, mcb[11]->getY() );
  tty = mcb[9]->getY() + mcb[9]->getHeight() + 10;
  
  win->maximizeX();
  w = win->getWidth();

  okb=(Button*)win->add(new Button(aguix,
                                   5,
                                   tty,
                                   catalog.getLocale(11),
                                   1,
                                   0,
                                   0));
  ok2allb=(Button*)win->add(new Button(aguix,
                                   0,
                                   tty,
                                   catalog.getLocale(224),
                                   1,
                                   0,
                                   0));
  skipb=(Button*)win->add(new Button(aguix,
                                   0,
                                   tty,
                                   catalog.getLocale(225),
                                   1,
                                   0,
                                   0));
  cb=(Button*)win->add(new Button(aguix,
                                  0,
                                  tty,
                                  catalog.getLocale(8),
                                  1,
                                  0,
                                  0));

  ba[0] = okb;
  ba[1] = ok2allb;
  ba[2] = skipb;
  ba[3] = cb;
  tw = AGUIX::scaleElementsW( w, 5, 5, -1, false, false, ba, NULL, 4 );
  if ( tw > w ) {
    w = tw;
    win->resize( w, win->getHeight() );
  }
  
  tty+=okb->getHeight()+5;
  
  tth=tty;
  okb->takeFocus();
  win->setDoTabCycling( true );
  win->resize( w, tth );
  win->setMaxSize( w, tth );
  win->setMinSize( w, tth );
  win->show();
  for(;endmode==-1;) {
    msg=aguix->WaitMessage(win);
    if(msg!=NULL) {
      switch(msg->type) {
        case AG_CLOSEWINDOW:
          if(msg->closewindow.window==win->getWindow()) endmode=3;
          break;
        case AG_BUTTONCLICKED:
          if(msg->button.button==okb) endmode=0;
          else if(msg->button.button==ok2allb) endmode=1;
          else if(msg->button.button==skipb) endmode=2;
          else if(msg->button.button==cb) endmode=3;
          break;
        case AG_KEYPRESSED:
          if(win->isParent(msg->key.window,false)==true) {
            switch(msg->key.key) {
              case XK_1:
              case XK_KP_1:
              case XK_KP_End:
                mcb[6]->setState((mcb[6]->getState()==1)?0:1);
                break;
              case XK_2:
              case XK_KP_2:
              case XK_KP_Down:
                mcb[7]->setState((mcb[7]->getState()==1)?0:1);
                break;
              case XK_3:
              case XK_KP_3:
              case XK_KP_Next:
                mcb[8]->setState((mcb[8]->getState()==1)?0:1);
                break;
              case XK_4:
              case XK_KP_4:
              case XK_KP_Left:
                mcb[3]->setState((mcb[3]->getState()==1)?0:1);
                break;
              case XK_5:
              case XK_KP_5:
              case XK_KP_Begin:
                mcb[4]->setState((mcb[4]->getState()==1)?0:1);
                break;
              case XK_6:
              case XK_KP_6:
              case XK_KP_Right:
                mcb[5]->setState((mcb[5]->getState()==1)?0:1);
                break;
              case XK_7:
              case XK_KP_7:
              case XK_KP_Home:
                mcb[0]->setState((mcb[0]->getState()==1)?0:1);
                break;
              case XK_8:
              case XK_KP_8:
              case XK_KP_Up:
                mcb[1]->setState((mcb[1]->getState()==1)?0:1);
                break;
              case XK_9:
              case XK_KP_9:
              case XK_KP_Prior:
                mcb[2]->setState((mcb[2]->getState()==1)?0:1);
                break;
              case XK_Return:
              case XK_KP_Enter:
                if ( ( ok2allb->hasFocus() == false ) &&
                     ( skipb->hasFocus() == false ) &&
                     ( cb->hasFocus() == false ) ) {
                  endmode=0;
                }
                break;
              case XK_F1:
                endmode=0;
                break;
              case XK_Escape:
              case XK_F4:
                endmode=3;
                break;
              case XK_F2:
                endmode=1;
                break;
              case XK_F3:
                endmode=2;
                break;
            }
          }
          break;
      }
      aguix->ReplyMessage(msg);
    }
  }
  
  if((endmode==0)||(endmode==1)) {
    // ok
    newmode=0;
    newmode|=(mcb[0]->getState()==1)?S_IRUSR:0;
    newmode|=(mcb[1]->getState()==1)?S_IWUSR:0;
    newmode|=(mcb[2]->getState()==1)?S_IXUSR:0;
    newmode|=(mcb[3]->getState()==1)?S_IRGRP:0;
    newmode|=(mcb[4]->getState()==1)?S_IWGRP:0;
    newmode|=(mcb[5]->getState()==1)?S_IXGRP:0;
    newmode|=(mcb[6]->getState()==1)?S_IROTH:0;
    newmode|=(mcb[7]->getState()==1)?S_IWOTH:0;
    newmode|=(mcb[8]->getState()==1)?S_IXOTH:0;
    newmode|=(mcb[9]->getState()==1)?S_ISUID:0;
    newmode|=(mcb[10]->getState()==1)?S_ISGID:0;
    newmode|=(mcb[11]->getState()==1)?S_ISVTX:0;
    *return_mode=newmode;
  }
  
  win->close();
  delete win;

  return endmode;
}

/*****************
 * NM_CopyOp_Dir *
 *****************/

NM_CopyOp_Dir::NM_CopyOp_Dir(const FileEntry *fe)
{
  int erg;

  fileentry=fe;
  subdirs=new List();
  verz=new Verzeichnis();
  files=dirs=error_counter=0;
  bytes = 0;
  ok=false;
  user_abort = false;
  
  erg = verz->readDir( fe->fullname );
  if ( erg == 0 ) ok = true;
  else if ( erg > 0 ) user_abort = true;
}

NM_CopyOp_Dir::~NM_CopyOp_Dir()
{
  int id=subdirs->initEnum();
  NM_CopyOp_Dir *cod=(NM_CopyOp_Dir*)subdirs->getFirstElement(id);
  while(cod!=NULL) {
    delete cod;
    cod=(NM_CopyOp_Dir*)subdirs->getNextElement(id);
  }
  subdirs->closeEnum( id );
  delete subdirs;
  delete verz;
}

int
NM_CopyOp_Dir::createSubDirs(struct NM_copyorder *copyorder,unsigned long *gf,unsigned long *gd)
{
  ArrayList *f1;
  int id;
  FileEntry *fe;
  NM_CopyOp_Dir *cod1;
  bool enter;
  bool cancel=false;
  
  /*TODO: vielleicht sollte der Fehler weitergereicht werden, aber cancel=true soll nicht entstehen
    daher erstmal 0 zurueckgeben */
  if(ok==false) return 0;
  f1=verz->getFiles();
  if(f1==NULL) return 0;
  id=f1->initEnum();
  fe=(FileEntry*)f1->getFirstElement(id);
/*TODO:Vorerst zu langsam, kann aber an MemSystem liegen, daher nochmal ohne
       das pruefen
  if(copyorder->cowin!=NULL) {
    copyorder->cowin->setmessage(verz->getDir(),1);
    copyorder->cowin->redraw();
  }*/
  while((fe!=NULL)&&(cancel==false)) {
    if(strcmp(fe->name,"..")!=0) {
      enter=false;
      if(fe->isDir()==true) {
        // fe is a dir, check if it is a link and take it only when follow_symlinks==true
	// entry is a dir so it cannot be a corrupt link so no need to check
        if(fe->isLink==false) enter=true;
        else if(copyorder->follow_symlinks==true) enter=true;
      }
      if(enter==true) {
        // fe is a dir so creating corresponding entry
        cod1=new NM_CopyOp_Dir(fe);
        if ( cod1->user_abort == false ) {
          // recursive call
          if(cod1->createSubDirs(copyorder,gf,gd)!=0) cancel=true;
        } else cancel = true;
        // add the values from this subdir to this dir
        files+=cod1->files;
        dirs+=cod1->dirs;
        bytes+=cod1->bytes;

        // add this subdir to the list
        subdirs->addElement(cod1);
        
        // this is a dir so inc the counter
        dirs++;
        (*gd)++;

        if(copyorder->cowin!=NULL) {
          copyorder->cowin->set_files_to_copy(*gf);
          copyorder->cowin->set_dirs_to_copy(*gd);
          if(copyorder->cowin->redraw()!=0) cancel=true;
        }
      } else {
        // is not dir (mostly a file but can also be links ...)
        files++;
        (*gf)++;
        if ( ( fe->isLink == true ) &&
             ( copyorder->follow_symlinks == true ) &&
             ( fe->isCorrupt == false ) ) {
	  bytes += fe->dsize();
        } else {
          bytes += fe->size();
        }
      }
    }
    fe=(FileEntry*)f1->getNextElement(id);
  }
  f1->closeEnum(id);
  return (cancel==true)?1:0;
}

int
NM_CopyOp_Dir::createSubDirs(struct NM_deleteorder *delorder,unsigned long *gf,unsigned long *gd)
{
  ArrayList *f1;
  int id;
  FileEntry *fe;
  NM_CopyOp_Dir *cod1;
  bool enter;
  bool cancel=false;
  
  if(ok==false) return 0;
  f1=verz->getFiles();
  if(f1==NULL) return 0;
  id=f1->initEnum();
  fe=(FileEntry*)f1->getFirstElement(id);
  while((fe!=NULL)&&(cancel==false)) {
    if(strcmp(fe->name,"..")!=0) {
      enter=false;
      if(fe->isDir()==true) {
        // fe is a dir, check if it is a link and take it only when follow_symlinks==true
        if(fe->isLink==false) enter=true;
      }
      if(enter==true) {
        // fe is a dir so creating corresponding entry
        cod1=new NM_CopyOp_Dir(fe);
        if ( cod1->user_abort == false ) {
          // recursive call
          if(cod1->createSubDirs(delorder,gf,gd)!=0) cancel=true;
        } else cancel = true;
        // add the values from this subdir to this dir
        files+=cod1->files;
        dirs+=cod1->dirs;
        bytes+=cod1->bytes;
        
        // add this subdir to the list
        subdirs->addElement(cod1);
        
        // this is a dir so inc the counter
        dirs++;
        (*gd)++;

        if(delorder->dowin!=NULL) {
          delorder->dowin->set_files_to_delete(*gf);
          delorder->dowin->set_dirs_to_delete(*gd);
          if(delorder->dowin->redraw()!=0) cancel=true;
        }
      } else {
        // is not dir (mostly a file but can also be links ...)
        files++;
        (*gf)++;
        // when deleting only size of file, not the dest matters
        bytes += fe->size();
      }
    }
    fe=(FileEntry*)f1->getNextElement(id);
  }
  f1->closeEnum(id);
  return (cancel==true)?1:0;
}

int
NM_CopyOp_Dir::createSubDirs()
{
  ArrayList *f1;
  int id;
  FileEntry *fe;
  NM_CopyOp_Dir *cod1;
  bool enter;
  
  if(ok==false) return 0;
  f1=verz->getFiles();
  if(f1==NULL) return 0;
  id=f1->initEnum();
  fe=(FileEntry*)f1->getFirstElement(id);
  while(fe!=NULL) {
    if(strcmp(fe->name,"..")!=0) {
      enter=false;
      if(fe->isDir()==true) {
        // fe is a dir, check if it is a link and take it only when follow_symlinks==true
        if(fe->isLink==false) enter=true;
      }
      if(enter==true) {
        // fe is a dir so creating corresponding entry
        cod1=new NM_CopyOp_Dir(fe);
        // recursive call
        cod1->createSubDirs();
        
        // add this subdir to the list
        subdirs->addElement(cod1);
      }
    }
    fe=(FileEntry*)f1->getNextElement(id);
  }
  f1->closeEnum(id);
  return 0;
}

NM_extern_fe::NM_extern_fe(char *nfn)
{
  fullname[0]=dupstring(nfn);
  fullname[1]=NULL;
  name[0]=NULL;
  name[1]=NULL;
  fe=NULL;
  row = -1;
  dirfinished=0;
}

NM_extern_fe::~NM_extern_fe()
{
  if(fullname[0]!=NULL) _freesafe(fullname[0]);
  if(fullname[1]!=NULL) _freesafe(fullname[1]);
  if(name[0]!=NULL) _freesafe(name[0]);
  if(name[1]!=NULL) _freesafe(name[1]);
  if ( fe != NULL ) delete fe;
}

char *NM_extern_fe::getFullname(bool noext)
{
  if(noext==false) return fullname[0];
  else {
    if(fullname[1]==NULL) fullname[1]=Datei::getNameWithoutExt(fullname[0]);
    return fullname[1];
  }
}

char *NM_extern_fe::getName(bool noext)
{
  if(noext==false) {
    if(name[0]==NULL) name[0]=Datei::getFilenameFromPath(fullname[0]);
    return name[0];
  } else {
    if(name[0]==NULL) name[0]=Datei::getFilenameFromPath(fullname[0]);
    if(name[1]==NULL) name[1]=Datei::getNameWithoutExt(name[0]);
    return name[1];
  }
}

NM_extern_fe::NM_extern_fe(char *nfn,const FileEntry *tfe, int trow)
{
  fullname[0]=dupstring(nfn);
  fullname[1]=NULL;
  name[0]=NULL;
  name[1]=NULL;
  if ( tfe != NULL ) {
    this->fe = new FileEntry( *tfe );
  } else {
    this->fe = NULL;
  }
  this->row=trow;
  dirfinished=0;
}

void NM_extern_fe::setDirFinished(int newv)
{
  dirfinished=newv;
}

int NM_extern_fe::getDirFinished()
{
  return dirfinished;
}

const FileEntry *NM_extern_fe::getFE()
{
  return fe;
}

int NM_extern_fe::getRow()
{
  return row;
}

int NormalMode::getSelFiles( std::list<NM_specialsourceExt*> *list, nm_getfiles_t selmode, bool unselect )
{
  int res;
  std::list<NM_specialsourceInt*> *tlist;
  std::list<NM_specialsourceInt*>::iterator iti1;
  NM_specialsourceInt *ss1;
  NM_specialsourceExt *sse1;

  if ( list == NULL ) return -1;

  tlist = new std::list<NM_specialsourceInt*>;
  res = getSelFiles( tlist, selmode, unselect );
  if ( res < 0 ) {
    delete tlist;
    return -1;
  }

  for ( iti1 = tlist->begin(); iti1 != tlist->end(); iti1++ ) {
    ss1 = *iti1;
    sse1 = new NM_specialsourceExt( ss1->entry() );
    sse1->row = ss1->row;
    
    list->push_back( sse1 );

    delete ss1;
  }
  delete tlist;
  return res;
}

int NM_Filter::load()
{
  int found_error = 0;

  setCheck( 0 );
  for (;;) {
    if ( worker_token == PATTERN_WCP ) {
      readtoken();

      if ( worker_token != '=' ) {
        found_error = 1;
        break;
      }      
      readtoken();

      if ( worker_token == STRING_WCP ) {
        if ( pattern != NULL ) _freesafe( pattern );
        pattern = dupstring( yylval.strptr );
      } else {
        found_error = 1;
        break;
      }      
      readtoken();

      if ( worker_token != ';' ) {
        found_error = 1;
        break;
      }
      readtoken();
    } else if ( worker_token == MODE_WCP ) {
      readtoken();

      if ( worker_token != '=' ) {
        found_error = 1;
        break;
      }      
      readtoken();

      if ( worker_token == INCLUDE_WCP ) {
        setCheck( 1 );
      } else if ( worker_token == EXCLUDE_WCP ) {
        setCheck( 2 );
      } else {
        found_error = 1;
        break;
      }      
      readtoken();

      if ( worker_token != ';' ) {
        found_error = 1;
        break;
      }
      readtoken();
    } else {
      break;
    }
  }
  
  return found_error;
}

int NM_Filter::loadBin(Datei *fh)
{
  int chunksize=fh->getInt();
  int len;
  char *str;
  if(chunksize>=(2*Datei::getIntSize())) {
    len=fh->getInt();
    chunksize-=Datei::getIntSize();
    str=fh->getString(len);
    if(pattern!=NULL) _freesafe(pattern);
    pattern=str;
    chunksize-=len;
    check=fh->getInt();
    chunksize-=Datei::getIntSize();
  }
  while(chunksize>0) {
    fh->getUChar();
    chunksize--;
  }
  return 0;
}

int NM_Filter::save(Datei *fh)
{
  if ( fh == NULL ) return 1;

  if ( pattern != NULL ) WConfig::configPutPairString( fh, "pattern", pattern );  
  switch ( check ) {
    case 1:
      WConfig::configPutPair( fh, "mode", "include" );
      break;
    case 2:
      WConfig::configPutPair( fh, "mode", "exclude" );
      break;
    default:
      break;
  }
  return 0;
}

void NormalMode::setSortmode(int nmode)
{
  if ( ISVALID_SORTMODE( nmode ) ) {
    nmode&=0xff|SORT_REVERSE|SORT_DIRLAST|SORT_DIRMIXED;
    sortmode=nmode;
    // aktualisieren
    checkRebuild( ce, true );
    if ( ce != NULL )
      if ( ce->activefe != NULL )
        lv->centerActive();
    setName();
  }
}

void NormalMode::setFilters(List *nfilters)
{
  int id;
  NM_Filter *fi;
  id=filters->initEnum();
  fi=(NM_Filter*)filters->getFirstElement(id);
  while(fi!=NULL) {
    delete fi;
    fi=(NM_Filter*)filters->getNextElement(id);
  }
  filters->closeEnum(id);
  filters->removeAllElements();
  id=nfilters->initEnum();
  fi=(NM_Filter*)nfilters->getFirstElement(id);
  while(fi!=NULL) {
    filters->addElement(fi->duplicate());
    fi=(NM_Filter*)nfilters->getNextElement(id);
  }
  nfilters->closeEnum(id);
  filterlfdnr++;
  // aktualisieren
  checkRebuild( ce, true );
  if ( ce != NULL )
    if ( ce->activefe != NULL )
      lv->centerActive();
  setName();
}

NM_Filter *NM_Filter::duplicate()
{
  NM_Filter *fi=new NM_Filter();
  fi->setCheck(check);
  fi->setPattern(pattern);
  return fi;
}

void NormalMode::configureFilters(bool *tshowhidden,List *tfilters)
{
  int tw,ttw,tth,ty;
  int t1,t2;
  int trow;
  int pos;
  List *usetf;
  int id;
  NM_Filter *fi;
  GUIElement *gl[3];
  int minw[3];

  usetf=new List();
  id=tfilters->initEnum();
  fi=(NM_Filter*)tfilters->getFirstElement(id);
  while(fi!=NULL) {
    usetf->addElement(fi->duplicate());
    fi=(NM_Filter*)tfilters->getNextElement(id);
  }
  tfilters->closeEnum(id);

  ttw=10;
  tth=50;
  AWindow *win=new AWindow(aguix);
  win->create(NULL,10,10,ttw,tth,0,catalog.getLocale(159));

  ty=5;
  ChooseButton *chb=(ChooseButton*)win->add(new ChooseButton(aguix,5,ty,
                                            (*tshowhidden==true)?1:0,
                                            catalog.getLocale(357),LABEL_RIGHT,1,0));
  ty+=chb->getHeight()+5;
  FieldListView *filv = (FieldListView*)win->add( new FieldListView( aguix,
                                                                     5,
                                                                     ty,
                                                                     ttw,
                                                                     10 * aguix->getCharHeight(),
                                                                     0 ) );
  filv->setHBarState(2);
  filv->setVBarState(2);
  filv->setNrOfFields( 3 );
  filv->setFieldWidth( 1, 1 );
  ty+=filv->getHeight()+5;
  t1=(strlen(catalog.getLocale(167))+2);
  t1*=aguix->getCharWidth();
  Button *newb=(Button*)win->add(new Button(aguix,5,ty,t1,catalog.getLocale(167),1,0,0));
  t1=(strlen(catalog.getLocale(169))+2);
  t1*=aguix->getCharWidth();
  Button *delb=(Button*)win->add(new Button(aguix,newb->getX()+newb->getWidth(),ty,t1,
                                            catalog.getLocale(169),1,0,0));
  t1=(strlen(catalog.getLocale(168))+2);
  t1*=aguix->getCharWidth();
  Button *editb=(Button*)win->add(new Button(aguix,delb->getX()+delb->getWidth(),ty,t1,
                                            catalog.getLocale(168),1,0,0));
  ty+=newb->getHeight();
  ttw=editb->getX()+editb->getWidth()+5;
  
  minw[0] = newb->getWidth();
  minw[1] = delb->getWidth();
  minw[2] = editb->getWidth();

  t1=(strlen(catalog.getLocale(194))+2);
  t1*=aguix->getCharWidth();
  if(t1<(ttw-10)) t1=ttw-10;
  Button *uallb=(Button*)win->add(new Button(aguix,5,ty,t1,
                                             catalog.getLocale(194),1,0,0));
  ty+=uallb->getHeight()+5;
  ttw=uallb->getX()+uallb->getWidth()+5;

  t1=(strlen(catalog.getLocale(11))+2);
  t1*=aguix->getCharWidth();
  t2=(strlen(catalog.getLocale(8))+2);
  t2*=aguix->getCharWidth();
  tw=5+t1+5+t2+5;
  if(tw>ttw) ttw=tw;
  Button *okb=(Button*)win->add(new Button(aguix,
                                             5,
                                             ty,
                                             t1,
                                             catalog.getLocale(11),
                                             1,
                                             0,
                                             0));
  Button *cancelb=(Button*)win->add(new Button(aguix,
                                                 ttw-5-t2,
                                                 ty,
                                                 t2,
                                                 catalog.getLocale(8),
                                                 1,
                                                 0,
                                                 0));
  filv->resize(ttw-10,filv->getHeight());
  editb->resize(ttw-5-editb->getX(),editb->getHeight());
  tth=okb->getY()+okb->getHeight()+5;
  
  id=usetf->initEnum();
  fi=(NM_Filter*)usetf->getFirstElement(id);
  pos=0;
  while(fi!=NULL) {
    trow = filv->addRow();
    setLVC4Filter( filv, trow, fi );
    filv->setPreColors( trow, FieldListView::PRECOLOR_ONLYSELECT );
    fi=(NM_Filter*)usetf->getNextElement(id);
  }
  usetf->closeEnum(id);
  filv->redraw();
  
  win->setDoTabCycling( true );
  win->resize(ttw,tth);
  //win->setMaxSize(ttw,tth);
  win->setMinSize(ttw,tth);
  win->show();
  AGMessage *msg;
  while((msg=aguix->GetMessage(NULL))!=NULL) aguix->ReplyMessage(msg);
  int ende=0;
  while(ende==0) {
    msg=aguix->WaitMessage(win);
    if(msg!=NULL) {
      if(msg->type==AG_CLOSEWINDOW) {
        if(msg->closewindow.window==win->getWindow()) ende=-1;
      } else if(msg->type==AG_BUTTONCLICKED) {
        if(msg->button.button==okb) ende=1;
        else if(msg->button.button==cancelb) ende=-1;
        else if(msg->button.button==newb) {
          NM_Filter *tfi=new NM_Filter();
          usetf->addElement(tfi);
          
          trow = filv->addRow();
          setLVC4Filter( filv, trow, tfi );
          filv->setPreColors( trow, FieldListView::PRECOLOR_ONLYSELECT );
          filv->setActiveRow( trow );
          filv->showActive();
          if(configureFilter(tfi)==0) {
            setLVC4Filter( filv, trow, tfi );
            filv->redraw();
          }
        } else if(msg->button.button==delb) {
          pos = 0;
          while ( filv->isValidRow( pos ) == true ) {
            if ( filv->getSelect( pos ) == true ) {
              fi = (NM_Filter*)usetf->getElementAt( pos );
              if ( fi != NULL ) {
                delete fi;
                usetf->removeElementAt( pos );
                filv->deleteRow( pos );
                filv->redraw();
                pos--;
              }
            }
            pos++;
          }
        } else if(msg->button.button==editb) {
          pos = 0;
          while ( filv->isValidRow( pos ) == true ) {
            if ( filv->getSelect( pos ) == true ) {
              fi = (NM_Filter*)usetf->getElementAt( pos );
              if ( fi != NULL ) {
                if ( configureFilter( fi ) == 0 ) {
                  setLVC4Filter( filv, pos, fi );
                  filv->redraw();
                }
              }
            }
            pos++;
          }
        } else if(msg->button.button==uallb) {
          pos = 0;
          id = usetf->initEnum();
          fi = (NM_Filter*)usetf->getFirstElement( id );
          while ( fi != NULL ) {
            fi->setCheck( 0 );
            if ( filv->isValidRow( pos ) == true ) {
              setLVC4Filter( filv, pos, fi );
              pos++;
            }
            fi = (NM_Filter*)usetf->getNextElement( id );
          }
          usetf->closeEnum( id );
          filv->redraw();
        }
      } else if ( msg->type == AG_SIZECHANGED ) {
        if ( msg->size.window == win->getWindow() ) {
          AGUIX::scaleElementsW( msg->size.neww, 5, -1, -1, true, true, (GUIElement**)&filv, NULL, 1 );
          gl[0] = newb;
          gl[1] = delb;
          gl[2] = editb;
          AGUIX::scaleElementsW( msg->size.neww, 5, 0, 0, true, true, gl, minw, 3 );
          AGUIX::scaleElementsW( msg->size.neww, 5, 0, 0, true, true, (GUIElement**)&uallb, NULL, 1 );
          gl[0] = okb;
          gl[1] = cancelb;
          AGUIX::scaleElementsW( msg->size.neww, 5, 10, -1, false, false, gl, NULL, 2 );
          okb->move( okb->getX(), msg->size.newh - 5 - okb->getHeight() );
          cancelb->move( cancelb->getX(), okb->getY() );
          uallb->move( uallb->getX(), okb->getY() - 5 - uallb->getHeight() );
          newb->move( newb->getX(), uallb->getY() - newb->getHeight() );
          delb->move( delb->getX(), newb->getY() );
          editb->move( editb->getX(), newb->getY() );
          filv->resize( filv->getWidth(), newb->getY() - 5 - filv->getY() );
        }
      }
      aguix->ReplyMessage(msg);
    }
  }
  if(ende==1) {
    // insert usetf in tfilters
    id=tfilters->initEnum();
    fi=(NM_Filter*)tfilters->getFirstElement(id);
    while(fi!=NULL) {
      delete fi;
      fi=(NM_Filter*)tfilters->getNextElement(id);
    }
    tfilters->closeEnum(id);
    tfilters->removeAllElements();

    id=usetf->initEnum();
    fi=(NM_Filter*)usetf->getFirstElement(id);
    while(fi!=NULL) {
      tfilters->addElement(fi->duplicate());
      fi=(NM_Filter*)usetf->getNextElement(id);
    }
    usetf->closeEnum(id);
    
    *tshowhidden=(chb->getState()==1)?true:false;
  }
  
  win->close();
  delete win;
  id=usetf->initEnum();
  fi=(NM_Filter*)usetf->getFirstElement(id);
  while(fi!=NULL) {
    delete fi;
    fi=(NM_Filter*)usetf->getNextElement(id);
  }
  usetf->closeEnum(id);
  usetf->removeAllElements();
  delete usetf;
}

void NormalMode::setLVC4Filter( FieldListView *filv, int row, NM_Filter *fi )
{
  if ( filv == NULL ) return;
  if ( fi == NULL ) return;
  char *p=fi->getPattern();
  
  filv->setText( row, 0, ( p != NULL ) ? p : "" );
  if ( fi->getCheck() == 1 ) filv->setText( row, 2, catalog.getLocale( 170 ) );
  else if ( fi->getCheck() == 2 ) filv->setText( row, 2, catalog.getLocale( 171 ) );
  else filv->setText( row, 2, "" );
}

int NormalMode::configureFilter(NM_Filter *fi)
{
  int tw,ttw,tth,tty,ttx;
  int t1,t2;
  char *tstr;

  ttw=10;
  tth=10;
  AWindow *win=new AWindow(aguix);
  win->create(NULL,10,10,ttw,tth,0,catalog.getLocale(172));

  tty=5;
  ttx=5;
  Text *ttext=(Text*)win->add(new Text(aguix,ttx,tty,catalog.getLocale(93),1));
  ttx+=ttext->getWidth()+5;
  tstr=fi->getPattern();
  StringGadget *tsg=(StringGadget*)win->add(new StringGadget(aguix,ttx,tty,100,
                                       (tstr!=NULL)?tstr:"",0));
  tty+=tsg->getHeight()+5;
  tw=ttx+tsg->getWidth()+5;
  if(tw>ttw) ttw=tw;

  ttx=5;
  CycleButton *cyb=(CycleButton*)win->add(new CycleButton(aguix,ttx,tty,100,1,0,0));
  cyb->addOption(catalog.getLocale(358));
  cyb->addOption(catalog.getLocale(170));
  cyb->addOption(catalog.getLocale(171));
  if(fi->getCheck()==1) cyb->setOption(1);
  else if(fi->getCheck()==2) cyb->setOption(2);
  else cyb->setOption(0);
  cyb->resize(cyb->getMaxSize(),cyb->getHeight());

  tty+=cyb->getHeight()+5;
  tw=ttx+cyb->getWidth()+5;
  if(tw>ttw) ttw=tw;
  ttx=5;

  t1=(strlen(catalog.getLocale(11))+2);
  t1*=aguix->getCharWidth();
  t2=(strlen(catalog.getLocale(8))+2);
  t2*=aguix->getCharWidth();
  tw=5+t1+5+t2+5;
  if(tw>ttw) ttw=tw;
  Button *okb=(Button*)win->add(new Button(aguix,
                                             5,
                                             tty,
                                             t1,
                                             catalog.getLocale(11),
                                             1,
                                             0,
                                             0));
  Button *cancelb=(Button*)win->add(new Button(aguix,
                                                 ttw-5-t2,
                                                 tty,
                                                 t2,
                                                 catalog.getLocale(8),
                                                 1,
                                                 0,
                                                 0));
  tth=okb->getY()+okb->getHeight()+5;
  
  win->setDoTabCycling( true );
  win->resize(ttw,tth);
  win->setMaxSize(ttw,tth);
  win->setMinSize(ttw,tth);
  win->show();
  AGMessage *msg;
  while((msg=aguix->GetMessage(NULL))!=NULL) aguix->ReplyMessage(msg);
  int ende=0;
  while(ende==0) {
    msg=aguix->WaitMessage(win);
    if(msg!=NULL) {
      if(msg->type==AG_CLOSEWINDOW) {
        if(msg->closewindow.window==win->getWindow()) ende=-1;
      } else if(msg->type==AG_BUTTONCLICKED) {
        if(msg->button.button==okb) ende=1;
        else if(msg->button.button==cancelb) ende=-1;
      }
      aguix->ReplyMessage(msg);
    }
  }
  if(ende==1) {
    switch(cyb->getSelectedOption()) {
      case 1:
        fi->setCheck(1);
        break;
      case 2:
        fi->setCheck(2);
        break;
      default:
        fi->setCheck(0);
        break;
    }
    fi->setPattern(tsg->getText());
  }
  
  win->close();
  delete win;
  return (ende==1)?0:1;
}

/*
 * NormalMode::checkRebuild
 *
 * checks for needed rebuild when filter/showhidden/sortmode changed
 * rebuild lister when needed or forced
 */
void NormalMode::checkRebuild( NMCacheEntry *tce, bool forcerebuild )
{
  int oldx, oldy;
  int sh = (showHidden == true ) ? 1 : 0;
  
  if ( tce != NULL ) {
    if ( ( tce->actfilterlfdnr != filterlfdnr ) || ( tce->actshowhidden != sh ) ) {
      checkFilters( tce );
      tce->actfilterlfdnr = filterlfdnr;
      tce->actshowhidden = sh;
      tce->reset_checkfe();
      forcerebuild = true;
    }
    if ( tce->actsortmode != sortmode ) {
      tce->verz->sort( sortmode );
      tce->actsortmode = sortmode;
      //TODO warum zu Hlle nicht einfach nur tce->firstnullftcheck?
      cyclicfunc( CYCLICFUNC_MODE_FIRSTNULLFT );
      forcerebuild = true;
    }
  }
  if ( forcerebuild == true ) {
    if ( lv != NULL ) {
      oldx = lv->getXOffset();
      oldy = lv->getYOffset();
      buildLister();
      lv->setYOffset( oldy );
      lv->setXOffset( oldx );
      showCacheState();
      lv->redraw();
    }
  }
  
  visChanged = true;
}

void NormalMode::setName()
{
  bool filteractive;
  int fid=filters->initEnum();
  char *tstr;
  NM_Filter *fil=(NM_Filter*)filters->getFirstElement(fid);
  bool doUpdate;
  
  filteractive=false;
  while(fil!=NULL) {
    if(fil->getCheck()!=0) {
      filteractive=true;
      break;
    }
    fil=(NM_Filter*)filters->getNextElement(fid);
  }
  filters->closeEnum(fid);
  tstr = (char*)_allocsafe( 2 + // busyFlag + space
                            2 + // sortmode+reverse
                            2 + // hidden+filter
                            3 + // spaces
                            strlen(catalog.getLocale( 173 ) ) + 1 );
  sprintf(tstr,"         %s",catalog.getLocale(173));
  switch ( busyFlag ) {
    case 1:
      tstr[0] = '|';
      break;
    case 2:
      tstr[0] = '/';
      break;
    case 3:
      tstr[0] = '-';
      break;
    case 4:
      tstr[0] = '\\';
      break;
    default:
      break;
  }
  switch(sortmode&0xff) {
    case SORT_NAME:
      tstr[2] = 'N';
      break;
    case SORT_SIZE:
      tstr[2] = 'S';
      break;
    case SORT_CHGTIME:
      tstr[2] = 'C';
      break;
    case SORT_MODTIME:
      tstr[2] = 'M';
      break;
    case SORT_ACCTIME:
      tstr[2] = 'A';
      break;
    case SORT_TYPE:
      tstr[2] = 'T';
      break;
    case SORT_OWNER:
      tstr[2] = 'O';
      break;
    case SORT_INODE:
      tstr[2] = 'I';
      break;
    case SORT_NLINK:
      tstr[2] = 'L';
      break;
    default:
      tstr[2] = '?';
      break;
  }
  if ( ( sortmode & SORT_REVERSE ) == SORT_REVERSE ) tstr[3] = 'R';
  if ( showHidden == false ) tstr[4] = 'H';
  if ( filteractive == true ) tstr[5] = '*';
  
  if ( namestr == NULL ) doUpdate = true;
  else {
    if ( strcmp( namestr, tstr ) != 0 ) doUpdate = true;
    else doUpdate = false;
  }
  
  if(namestr!=NULL) _freesafe(namestr);
  namestr=tstr;

  if ( doUpdate == true ) updateName();
}

/*
 * update lvb name with current name and freespace str
 */
void NormalMode::updateName()
{
  char *tstr;
  const char *t1, *t2;
  
  t1 = ( namestr != NULL ) ? namestr : "";
  t2 = ( freespacestr != NULL ) ? freespacestr : "";
  tstr = catstring( t1, t2 );
  //TODO: Vielleicht mit aktuellen String vergleichen, um redraw
  //      zu vermeiden?
  parentlister->setName( tstr );
  _freesafe( tstr );
}

void NormalMode::reconfig()
{
  int side = parentlister->getSide();
  AGUIXFont *afont = aguix->getFont( wconfig->getFont( 2 + side ) );
  int sgh = afont->getCharHeight() + 8;
  int hbw;

  lv->setMBG(wconfig->getListerBG());
  lv->setHBarState( ( wconfig->getHBarTop( side ) == true ) ? 1 : 2 );
  lv->setVBarState( ( wconfig->getVBarLeft( side ) == true ) ? 1 : 2 );
  lv->setHBarHeight( wconfig->getHBarHeight( side ) );
  lv->setVBarWidth( wconfig->getVBarWidth( side ) );
  lv->setShowHeader( wconfig->getShowHeader( side ) );
  lv->setFont( wconfig->getFont( 2 + side ) );
  sg->setFont( wconfig->getFont( 2 + side ) );
  hb[0]->setFont( wconfig->getFont( 2 + side ) );
  hb[1]->setFont( wconfig->getFont( 2 + side ) );
  parentb->setFont( wconfig->getFont( 2 + side ) );
  lv->setHeaderFG( wconfig->getLVHeader( 0 ) );
  lv->setHeaderBG( wconfig->getLVHeader( 1 ) );
  
  sg->resize( sg->getWidth(), sgh );
  hbw = afont->getCharWidth() + 10;
  hb[0]->resize( hbw, sgh );
  hb[1]->resize( hbw, sgh );
  parentb->resize( 2 * afont->getCharWidth() + 10, sgh );
  reconf( x, y, w, h, true );
}

void NormalMode::showFreeSpace(bool force)
{
  int erg;
  if ( force == true ) erg = updateFreeSpaceStr( 1 );
  else erg = updateFreeSpaceStr( updatetime );
  if ( erg != 0 ) updateName();
}

/*
 * method updates freespacestr
 * returns 0 for no change, 1 for change
 */
int NormalMode::updateFreeSpaceStr( int ti )
{
  char *tstr,buffer1[3],buffer2[3];
  char numbufs[A_BYTESFORNUMBER(loff_t)], numbuff[A_BYTESFORNUMBER(loff_t)];
  loff_t s1,f1;
  int len;
  time_t now;
  int erg, rv = 0;

  if(showfreespace==true) {
    now=time(NULL);
    if ( ( now - lastfsupdate >= ti ) || ( lasteagain == true ) ) {
      if ( getCurrentDir() != NULL ) {
        erg = parentlister->getWorker()->PS_readSpace( getCurrentDir() );
        if ( erg == 0 ) {
          tstr=buffer1;
          s1=parentlister->getWorker()->PS_getSpaceH(&tstr);
          tstr=buffer2;
          f1=parentlister->getWorker()->PS_getFreeSpaceH(&tstr);
          numbufs[WriteLong2Str( numbufs, s1 )] = '\0';
          numbuff[WriteLong2Str( numbuff, f1 )] = '\0';

          len = strlen( numbuff ) +
                strlen( numbufs ) +
                strlen( buffer1 ) +
                strlen( buffer2 ) +
                strlen( catalog.getLocale( 529 ) ) + 1;
          tstr=(char*)_allocsafe(len);
          sprintf( tstr, catalog.getLocale( 529 ), numbuff, buffer2, numbufs, buffer1 );
          if ( freespacestr != NULL ) _freesafe( freespacestr );
          freespacestr = tstr;
          lasteagain = false;
          rv = 1;
        } else if ( erg == EAGAIN ) {
          // no valid value so lets try again next call
          lasteagain = true;
        }
        lastfsupdate=now;
      }
    }
  } else {
    if ( freespacestr != NULL ) {
      _freesafe( freespacestr );
      freespacestr = NULL;
      rv = 1;
    }
  }
  return rv;
}

void NormalMode::setShowFreeSpace(bool v)
{
  showfreespace=v;
  setName();
  showFreeSpace(true);
}

void NormalMode::setUpdatetime(int nv)
{
  updatetime=nv;
  if(updatetime<1) updatetime=1;
  parentlister->getWorker()->PS_setLifetime( (double)updatetime );
}

bool NormalMode::startdnd(DNDMsg *dm)
{
  bool returnvalue=false;
  if(dm->getStart()->element==lv) {
    returnvalue=true;

    int row = dm->getStart()->specialinfo.value;
    if(lv->isValidRow(row)==true) {
      if(ce!=NULL) {
        if(ce->verz!=NULL) {
          FileEntry *fe=(FileEntry*)ce->verz->getFiles()->getElementAt(lv->getData(row));
          if(fe!=NULL) {
            // found entry for dnd
            // now use the type to call the dnd action

            WCFiletype *ft;
            ActionMessage amsg( parentlister->getWorker() );
            amsg.startLister=parentlister;
            amsg.mode=amsg.AM_MODE_DNDACTION;

            if(fe->isDir()==false) {
              ft=fe->filetype;
              if ( ft == NULL ) ft = wconfig->getnotyettype(); // not yet checked
            } else {
              ft = wconfig->getdirtype();
            }
            if ( ft == NULL ) ft = wconfig->getvoidtype();
            if(ft!=NULL) {
              if ( ft->getDNDActions()->size() < 1 ) ft = wconfig->getvoidtype();
              if(ft!=NULL) {
                amsg.dndmsg = dm;
                amsg.flags=ft->getDNDActionsGF();
		amsg.filetype = ft;
		amsg.getActionList = DNDAction::getActionList;
                parentlister->getWorker()->interpret(ft->getDNDActions(),&amsg);
              }
            }
          }
        }
      }
    }
  }
  return returnvalue;
}

bool NormalMode::isyours(GUIElement *elem)
{
  if(elem==lv) return true;
  return false;
}

FileEntry *NormalMode::getFE4DNDSTART( const AGDNDSTART *st )
{
  int row, id;
  FileEntry *fe=NULL;
  NMRowData *rdp;

  if ( st == NULL ) return NULL;
  if ( st->specialinfo.rowDataP == NULL ) return NULL;
  if( ( lv != NULL ) && ( st->element == lv ) ) {
    // lookup strategy:
    // 1.check if row (value) contains FE
    // 2.if not, find FE
    // 3.check fullname with found FE
    //TODO:search for fullname
    //     this is no bug but would improve it
    //     this would also only help users which
    //     try to trigger this
    row = st->specialinfo.value;
    if(lv->isValidRow(row)==true) {
      if(ce!=NULL) {
        if(ce->verz!=NULL) {
          fe=(FileEntry*)ce->verz->getFiles()->getElementAt(lv->getData(row));
          //TODO:for more secure cast
          //if ( typeid( *(st->specialinfo.rowDataP) ) == typeid( NMRowData ) );
          rdp = (NMRowData*)st->specialinfo.rowDataP;
          if ( fe->equals( rdp->getFE() ) == false ) {
            ArrayList *l1 = ce->verz->getFiles();
            id = l1->initEnum();
            fe = (FileEntry*)l1->getFirstElement( id );
            while ( fe != NULL ) {
              if ( fe->equals( rdp->getFE() ) == true ) break;
              fe = (FileEntry*)l1->getNextElement( id );
            }
            l1->closeEnum( id );
          }
          if ( fe != NULL ) {
	    // The idea for this test was that the DND msg contains a pointer to an old
	    // FileEntry which could be overwritten by another instance so I check the fullname
	    // As I now store copies of the FE inside the NMRowData and use equals() to find
	    // the corresponding FE this test is obsolete but doesn't hurt
            if ( strcmp( fe->fullname, rdp->getFullname() ) != 0 ) fe = NULL;
          }
        }
      }
    }
  }
  return fe;
}

void NormalMode::unsetAllFilters()
{
  int id;
  NM_Filter *fi;
  id=filters->initEnum();
  fi=(NM_Filter*)filters->getFirstElement(id);
  while(fi!=NULL) {
    fi->setCheck(0);
    fi=(NM_Filter*)filters->getNextElement(id);
  }
  filters->closeEnum(id);
  filterlfdnr++;
  // aktualisieren
  checkRebuild( ce, true );
  if ( ce != NULL )
    if ( ce->activefe != NULL )
      lv->centerActive();
  setName();
}

void NormalMode::setFilter(const char *filter,nm_filter_t mode)
{
  int newmode;
  int id;
  NM_Filter *fi;
  bool found;
  
  switch(mode) {
    case NM_FILTER_EXCLUDE:
      newmode=2;
      break;
    case NM_FILTER_UNSET:
      newmode=0;
      break;
    default:
      newmode=1;
      break;
  }
  id=filters->initEnum();
  found=false;
  fi=(NM_Filter*)filters->getFirstElement(id);
  while(fi!=NULL) {
    if(strcmp(fi->getPattern(),filter)==0) {
      found=true;
      break;
    }
    fi=(NM_Filter*)filters->getNextElement(id);
  }
  filters->closeEnum(id);
  if(found==true) {
    fi->setCheck(newmode);
  } else {
    fi=new NM_Filter();
    fi->setCheck(newmode);
    fi->setPattern(filter);
    filters->addElement(fi);
  }

  filterlfdnr++;
  // aktualisieren
  checkRebuild( ce, true );
  if ( ce != NULL )
    if ( ce->activefe != NULL )
      lv->centerActive();
  setName();
}

/*
 * this method will execute a given string
 * Worker will fork and wait for this child while the child also forks
 * and wait for his child
 *
 * This way Worker can sent disconnect signal and
 * the actual program can run even when Worker quits
 *
 * for detach I use 2 pipes with the following protocol:
 * 1.child will always wait for ack from Worker before
 *   he quits (to avoid broken pipes!)
 * 2.Worker will not access any pipes when he sent ack single
 *   (some reason)
 * 3.Worker will always wait for his child to quit
 *   (for the child the pipes are opened until he exits)
 * 4.Child will send Q when his child quits
 * 5.Worker can send D when user want to detach
 * 6.After D signal or receiving Q signal Worker
 *   send A
 *
 * it's complicated but avoids any problems with broken pipes (child
 * exits same time user want to detach->Worker would write to closed pipe)
 *
 * an other idea is to use shared memory segmentbut that's not so portable
 * although it's much nicer
 */
int NormalMode::runCommand( const char*exestr,
                            const char *tmpname,
                            const char *tmpoutput,
                            bool inbackground )
{
  pid_t child,ret,p2;
  int status,retval;
  
  parentlister->getWorker()->setWaitCursor();
  WorkerSHM::initSHM();
  if ( WorkerSHM::WSHMInit == true ) {
    *(WorkerSHM::WSHMPtr ) = ' ';
  }
  child=fork();
  if(child!=0) {
    // parent
    if(child==-1) {
      parentlister->getWorker()->unsetWaitCursor();
      return -2;
    }
    retval=0;
    // calling these functions will clear old values
    aguix->getLastKeyRelease();
    aguix->getLastMouseRelease();
    if ( inbackground == false ) {
      parentlister->getWorker()->setStatebarText( catalog.getLocale( 547 ) );
    }
    // after setting this flag Worker won't read/write from the pipe
    // because the child is expected to exit right after this
    for(;;) {
      // first answer all messages to be able to redraw
      aguix->doXMsgs( NULL, false );
      // check for events to disconnect from child
      if ( ( aguix->getLastKeyRelease() == XK_Escape ) ||
           ( aguix->getLastMouseRelease() == Button2 ) ) {
        if ( WorkerSHM::WSHMInit == true ) {
          *(WorkerSHM::WSHMPtr ) = 'Q';
        }
      }
      // now check for existence of the child
      ret=waitpid(child,&status,WNOHANG);
      if(ret==child) {
        if(WIFEXITED(status)) {
          // normal exit
#ifdef DEBUG
          printf( "child normally exited!\n" );
#endif
          retval=WEXITSTATUS(status);
          break;
        } else if(WIFSIGNALED(status)) {
#ifdef DEBUG
          printf("child exited by a signal!\n");
#endif
          retval=-1;
          break;
        }
      }
      // wait some time and the retry
      waittime(10);
    }
    parentlister->getWorker()->unsetWaitCursor();
    if ( inbackground == false ) {
      showCacheState();
    }
    return retval;
  }
  // child
  // wait for second child exit until there is something in the pipe

  // disconnect from parent
  setsid();
  chdir("/");

  p2 = fork();
  if ( p2 != 0 ) {
    // parent
    if ( p2 < 0 ) {
      // error
      exit(1);
    } else {
      // exit immediately when command should run in background
      bool doexit = inbackground;

      while ( doexit == false ) {
        ret = waitpid( p2, &status, WNOHANG );
        if ( ret == p2 ) {
          doexit = true;
          continue;
        }
        if ( WorkerSHM::WSHMInit == true ) {
          if ( *(WorkerSHM::WSHMPtr) == 'Q' ) {
            doexit = true;
            continue;
          }
        }
        // wait some time and retry
        waittime(10);
      }
      exit(0);
    }
  }
  setsid();
  // execute command
  // we could use exec* to call this
  //   but then we have to change exestr to give a char**argv
  //   (which is not so hard because it contains only the commands to execute
  //    tmpname)
  //   we also have to make another fork because exec* doesn't return
  //     and so we need an other process to remove the tmp-files after command finished
  system(exestr);
  
  // remove temp files
  remove(tmpname);
  remove(tmpoutput);
  
  // and exit
  exit(0);
  // just for some compiler to compile without warning
  return 0;
}

void NMCacheEntry::checkfordcd()
{
  List *dcd=wconfig->getDontCheckDirs();
  char *tstr;
  const char *dirstr;
  int id;

  dontcheck=false;
  if(verz!=NULL) {
    dirstr=verz->getDir();
    if(dcd!=NULL) {
      id=dcd->initEnum();
      tstr=(char*)dcd->getFirstElement(id);
      while(tstr!=NULL) {
        if(strncmp(tstr,dirstr,strlen(tstr))==0) {
          dontcheck=true;
          break;
        }
        tstr=(char*)dcd->getNextElement(id);
      }
      dcd->closeEnum(id);
    }
  }
}

void NMCacheEntry::reset_dirsizes()
{
  if(verz==NULL) return;
  ArrayList *filelist=verz->getFiles();
  int id=filelist->initEnum();
  FileEntry *fe=(FileEntry*)filelist->getFirstElement(id);
  while(fe!=NULL) {
    if(fe->isDir()==true) {
      fe->dirsize=-1;
    }
    fe=(FileEntry*)filelist->getNextElement(id);
  }
  filelist->closeEnum(id);
}

#ifdef WANT_THREADS
void NormalMode::slavehandler()
{
  const char *fname;
  char *filename;
  ft_recres_list::ft_recres_list_t *te;
  bool ende;
  int erg;
  bool dontCheckContent;
  
  // to get from REINIT directly to RUN
  bool firstrun = true;

  if ( slave.running != 0 ) {
    fprintf( stderr, "Worker: another thread already running!\n");
    return;
  }
  slave.running = 1;
  switchStatus( THREAD_REINIT );
  switchOrder( THREAD_NOP );

  slave.filetype_ex.lock();

#ifdef DEBUG
  printf("entering slave handler\n");
#endif

  for( ende = false; ende == false; ) {

    switch ( slave.status ) {
      case THREAD_RUN:
#ifdef DEBUG
        printf("waiting for element\n");
#endif

        reclist->lock();
        // wait for next element or stop
        while ( ( reclist->elemAvail_locked() == false ) && ( slave.order == THREAD_NOP ) )
          reclist->wait();

#ifdef DEBUG
        printf("wait finished\n");
#endif

        if ( slave.order != THREAD_NOP ) {
          // new command
          // switch mode for new command
          if ( slave.order == THREAD_EXIT ) {
            switchStatus( THREAD_EXIT );
            ende = true;
          } else if ( slave.order == THREAD_REINIT ) {
            switchStatus( THREAD_REINIT );
          } else if ( slave.order == THREAD_WAIT ) {
            switchStatus( THREAD_WAIT );
          }
          // mark this command read
          switchOrder( THREAD_NOP );
          reclist->unlock();
        } else {
          // an element to check
          erg = reclist->slaveread_locked( &fname, &dontCheckContent );
          filename = NULL;
          if ( erg == 0 ) {
            if ( fname != NULL ) {
              filename = dupstring( fname );
            }
          }
          reclist->unlock();

          if ( ( erg == 0 ) && ( filename != NULL ) ) {
	    checkFiletypeRes_t res;
#ifdef DEBUG
//            printf("erkenne %s\n",filename);
#endif
            res = slave_checkFiletype( filename, dontCheckContent );
            recreslist->lock();
    
            te = new ft_recres_list::ft_recres_list_t;
            te->name = filename;
            te->ft_index = res.v;
	    if ( res.customcolors == true ) {
	      te->customcolors = true;
	      te->fg = res.fg;
	      te->bg = res.bg;
	    }
    
            recreslist->put_locked( te );
            recreslist->unlock();
          }
        }
        break;
      case THREAD_WAIT:
        waitForNewOrder();
        if ( slave.order == THREAD_EXIT ) {
          switchStatus( THREAD_EXIT );
          ende = true;
        } else if ( slave.order == THREAD_REINIT ) {
          switchStatus( THREAD_REINIT );
        } else if ( slave.order == THREAD_RUN ) {
          switchStatus( THREAD_RUN );
        }
        switchOrder( THREAD_NOP );
        break;
      case THREAD_REINIT:
        // wait for change
        slave.filetype_ex.signal();
        slave.filetype_ex.wait();
        
        // change mode
        if ( firstrun == true ) {
          firstrun = false;
          switchStatus( THREAD_RUN );
        } else {
          switchStatus( THREAD_WAIT );
        }
        break;
      default:
        break;
    }
  }
    
  slave.filetype_ex.unlock();

#ifdef DEBUG
  printf("leaving slave handler\n");
#endif
}

int NormalMode::ft_rec_list::slaveread_locked( const char **name, bool *dontCheckContent )
{
  if ( ( name == NULL ) || ( dontCheckContent == NULL ) ) return 1;
  if ( readp != NULL ) {
    *name = readp->name;
    *dontCheckContent = readp->dontCheckContent;
    readp = readp->next;
  } else return 1;
  return 0;
}

bool NormalMode::ft_rec_list::isFull_locked()
{
  if ( elements >= size ) return true;
  return false;
}

bool NormalMode::ft_rec_list::isEmpty_locked()
{
  if ( elements < 1 ) return true;
  return false;
}

int NormalMode::ft_rec_list::put_locked( ft_rec_list_t *elem )
{
  int pos;

  if ( elem == NULL ) return -1;
  if ( isFull_locked() == true ) return -1;

  // don't accept any pointer in next
  elem->next = NULL;
  if ( tail != NULL ) {
    // add behind last
    tail->next = elem;
    tail = elem;
    // if readp is null, slave has read all elems, so set it to this new elem
    if ( readp == NULL )
      readp = tail;
  } else {
    head = tail = elem;
    readp = tail;
  }

  if ( elem->fe != NULL )
    elem->fe->reclistQueued = true;

  pos = elements++;
  return pos;
}

NormalMode::ft_rec_list::ft_rec_list_t *NormalMode::ft_rec_list::remove_locked()
{
  ft_rec_list_t *te;

  if ( elements == 0 ) return NULL;
  
  // readp could be the element to remove
  if ( readp == head )
    readp = readp->next;
  
  te = head;
  head = head->next;
  if ( head == NULL )
    tail = NULL;
  elements--;

  if ( te->fe != NULL )
    te->fe->reclistQueued = false;

  // leave no pointer in out list
  te->next = NULL;  
  return te;
}

NormalMode::ft_rec_list::ft_rec_list()
{
  size = 10;
  head = NULL;
  tail = NULL;
  readp = NULL;
  elements = 0;
}

NormalMode::ft_rec_list::~ft_rec_list()
{
  ft_rec_list_t *te;

  lock();
  while ( isEmpty_locked() == false ) {
    te = remove_locked();
    delete te;
  }
  unlock();
}

void NormalMode::ft_rec_list::lock()
{
  ex.lock();
}

void NormalMode::ft_rec_list::unlock()
{
  ex.unlock();
}

void NormalMode::ft_rec_list::wait()
{
  ex.wait();
}

void NormalMode::ft_rec_list::signal()
{
  ex.signal();
}

bool NormalMode::ft_rec_list::elemAvail_locked()
{
  if ( readp != NULL ) return true;
  return false;
}

NormalMode::ft_recres_list::ft_recres_list()
{
  head = tail = NULL;
  elements = 0;
}

NormalMode::ft_recres_list::~ft_recres_list()
{
  ft_recres_list_t *rte;

  lock();
  while ( isEmpty_locked() == false ) {
    rte = remove_locked();
    if ( rte != NULL ) {
      delete rte;
    }
  }
  unlock();
}

bool NormalMode::ft_recres_list::isEmpty_locked()
{
  if ( elements == 0 ) return true;
  return false;
}

int NormalMode::ft_recres_list::put_locked( ft_recres_list_t *elem )
{
  if ( elem == NULL ) return -1;

  // don't accept any pointer in next
  elem->next = NULL;
  if ( tail != NULL ) {
    // add behind last
    tail->next = elem;
    tail = elem;
  } else {
    head = tail = elem;
  }

  elements++;
  return 0;
}

NormalMode::ft_recres_list::ft_recres_list_t *NormalMode::ft_recres_list::remove_locked()
{
  ft_recres_list_t *te;

  if ( elements == 0 ) return NULL;
  
  te = head;
  head = head->next;
  if ( head == NULL )
    tail = NULL;
  elements--;

  // leave no pointer in out list
  te->next = NULL;  
  return te;
}

void NormalMode::ft_recres_list::lock()
{
  ex.lock();
}

void NormalMode::ft_recres_list::unlock()
{
  ex.unlock();
}


NormalMode::ft_rec_list::ft_rec_list_t *NormalMode::ft_rec_list::gettop_locked()
{
  return head;
}

void NormalMode::ft_rec_list_clear()
{
  ft_rec_list::ft_rec_list_t *te;
  NMCacheEntry *tce;
  int id;

  reclist->lock();
  while ( reclist->isEmpty_locked() == false ) {
    te = reclist->remove_locked();
    delete te;
  }
  // reset checkfe for all caches
  id = cache->initEnum();
  tce = (NMCacheEntry*)cache->getFirstElement( id );
  while ( tce != NULL ) {
    tce->checkfe = tce->firstnullft;
    tce = (NMCacheEntry*)cache->getNextElement( id );
  }
  cache->closeEnum( id );
  reclist->unlock();
  
  visChanged = true;
}

void NormalMode::ft_recres_list_clear()
{
  ft_recres_list::ft_recres_list_t *rte;

  recreslist->lock();
  while ( recreslist->isEmpty_locked() == false ) {
    // alle Resultate entnehmen
    rte = recreslist->remove_locked();
    if ( rte != NULL ) {
      delete rte;
    }
  }
  recreslist->unlock();
}

/*
 * ft_list_clear
 *
 * will delete all filetypes from list
 *
 * only allowed for master thread
 */
void NormalMode::ft_list_clear()
{
  WCFiletype *ft;
  int id;

  slave.filetype_ex.lock();
  
  id = slave.filetypes->initEnum();
  ft = (WCFiletype*)slave.filetypes->getFirstElement( id );
  while ( ft != NULL ) {
    delete ft;
    slave.filetypes->removeFirstElement();
    ft = (WCFiletype*)slave.filetypes->getFirstElement( id );
  }
  slave.filetypes->closeEnum( id );
  
  slave.filetype_ex.unlock();
}

/*
 * ft_list_update
 *
 * will update the filetype-list to the current config
 *
 * only allowed for master thread
 */
void NormalMode::ft_list_update()
{
  List *ftlist;
  WCFiletype *ft;
  int id;
  
  slave.filetype_ex.lock();

  ft_list_clear();

  ftlist = wconfig->getFiletypes();
  id = ftlist->initEnum();
  ft = (WCFiletype*)ftlist->getFirstElement( id );
  while ( ft != NULL ) {
    slave.filetypes->addElement( ft->duplicate() );
    ft = (WCFiletype*)ftlist->getNextElement( id );
  }
  ftlist->closeEnum( id );
  
  slave.filetype_ex.unlock();
}

NormalMode::checkFiletypeRes_t NormalMode::slave_checkFiletype( const char *fullname, bool dontCheckContent )
{
  FileEntry *fe1;
  int pos = -1;
  WCFiletype *ft, *parentft;
  std::vector<unsigned int> *v = NULL;
  checkFiletypeRes_t res;

  slave.filetype_ex.lock();

  fe1 = new FileEntry();
  fe1->fullname = dupstring( fullname );
  fe1->name = Datei::getFilenameFromPath( fullname );
  fe1->nr = 0;
  if( fe1->readInfos() == 0 ) {
    ft = fe1->checkFiletype( slave.filetypes, dontCheckContent, &condparser );
    if ( ft != NULL ) {
      // first get vector with filetype position for each child
      v = ft->getTypePos();
      // now find root filetype
      
      for ( parentft = ft; parentft->getParentType() != NULL; parentft = parentft->getParentType() );
      pos = slave.filetypes->getIndex( parentft );
      if ( pos >= 0 ) {
	if ( v != NULL ) v->push_back( (unsigned int)pos );
      } else {
	if ( v != NULL ) {
	  delete v;
	  v = NULL;
	}
      }
    }
  }
  res.v = v;

  if ( fe1->getCustomColors() == true ) {
    res.customcolors = true;
    res.fg = new int[4];
    res.bg = new int[4];
    for ( int i = 0; i < 4; i++ ) {
      res.fg[i] = fe1->getColor( 0, i );
      res.bg[i] = fe1->getColor( 1, i );
    }
  }
  delete fe1;

  slave.filetype_ex.unlock();
  return res;
}
#endif

void NormalMode::setupLVFields()
{
  int side = parentlister->getSide();
  const std::vector<WorkerTypes::listcol_t> *sets;
  int i;
  int used_fields, li;

  sets = wconfig->getVisCols( side );
  if ( sets == NULL ) return;
  
  used_fields = sets->size();

  if ( used_fields == 0 ) {
    // oops, what's this?
    lv->setNrOfFields( 1 );  // 0 fields are not supported, blame the author (me ;-) )
  } else {
    lv->setNrOfFields( used_fields * 2 - 1 );
  
    for( i = 0; i < (int)sets->size(); i++ ) {
      if ( i > 0 ) {
        lv->setFieldWidthQ( 2 * i - 1, 1 );
      }
      lv->setFieldTextMergedQ( 2 * i, true );
      li = WorkerTypes::getAvailListColEntry( (*sets)[i] );
      if ( li >= 0 ) {
        lv->setFieldTextQ( 2 * i, catalog.getLocale( WorkerTypes::availListCols[li].catalogid ) );
      }
      switch ( (*sets)[i] ) {
        case WorkerTypes::LISTCOL_NAME:
          lv->setFieldAlignQ( 2 * i, FieldListView::ALIGN_LEFT );
          lv->setFieldWidthQ( 2 * i, -1 );
          break;
        case WorkerTypes::LISTCOL_SIZE:
#ifdef LEFTJUSTIFY
          lv->setFieldAlignQ( 2 * i, FieldListView::ALIGN_LEFT );
#else
          lv->setFieldAlignQ( 2 * i, FieldListView::ALIGN_RIGHT );
#endif
          lv->setFieldWidthQ( 2 * i, -1 );
          break;
        case WorkerTypes::LISTCOL_TYPE:
          lv->setFieldAlignQ( 2 * i, FieldListView::ALIGN_LEFT );
          lv->setFieldWidthQ( 2 * i, -1 );
          break;
        case WorkerTypes::LISTCOL_PERM:
          lv->setFieldAlignQ( 2 * i, FieldListView::ALIGN_LEFT );
          lv->setFieldWidthQ( 2 * i, WORKER_PERMCHARS );
          break;
        case WorkerTypes::LISTCOL_OWNER:
          lv->setFieldAlignQ( 2 * i, FieldListView::ALIGN_LEFT );
          lv->setFieldWidthQ( 2 * i, -1 );
          break;
        case WorkerTypes::LISTCOL_DEST:
          lv->setFieldAlignQ( 2 * i, FieldListView::ALIGN_LEFT );
          lv->setFieldWidthQ( 2 * i, -1 );
          break;
        case WorkerTypes::LISTCOL_MOD:
          lv->setFieldAlignQ( 2 * i, FieldListView::ALIGN_LEFT );
          lv->setFieldWidthQ( 2 * i, -1 );
          break;
        case WorkerTypes::LISTCOL_ACC:
          lv->setFieldAlignQ( 2* i, FieldListView::ALIGN_LEFT );
          lv->setFieldWidthQ( 2 * i, -1 );
          break;
        case WorkerTypes::LISTCOL_CHANGE:
          lv->setFieldAlignQ( 2 * i, FieldListView::ALIGN_LEFT );
          lv->setFieldWidthQ( 2 * i, -1 );
          break;
        case WorkerTypes::LISTCOL_INODE:
          lv->setFieldAlignQ( 2 * i, FieldListView::ALIGN_RIGHT );
          lv->setFieldWidthQ( 2 * i, -1 );
          break;
        case WorkerTypes::LISTCOL_NLINK:
          lv->setFieldAlignQ( 2 * i, FieldListView::ALIGN_RIGHT );
          lv->setFieldWidthQ( 2 * i, -1 );
          break;
        case WorkerTypes::LISTCOL_BLOCKS:
          lv->setFieldAlignQ( 2 * i, FieldListView::ALIGN_RIGHT );
          lv->setFieldWidthQ( 2 * i, -1 );
          break;
        default:
          break;
      }
    }
    lv->redraw();
  }
}

void NormalMode::chownf(struct NM_chownorder *coorder)
{
  Verzeichnis *verz;
  NM_specialsourceExt *sse;
  NM_specialsourceInt *ss1, *ss2;
  ArrayList *filelist;
  FileEntry *fe;
  bool found;
  changeown_info_t *coinfo;
  std::list<NM_specialsourceExt*>::iterator ite1;
  std::list<NM_specialsourceInt*> *colist;
  std::list<NM_specialsourceInt*>::iterator iti1;
  
  if ( coorder == NULL ) return;
  if ( getCurrentDir() == NULL ) return;

  finishsearchmode();
  
  verz = ce->verz;
  filelist = verz->getFiles();
  
  colist = new std::list<NM_specialsourceInt*>;
  switch ( coorder->source ) {
    case NM_chownorder::NM_SPECIAL:
      for ( ite1 = coorder->sources->begin(); ite1 != coorder->sources->end(); ite1++ ) {
	sse = *ite1;
        fe = (FileEntry*)filelist->getFirstElement();
        found = false;
        while ( fe != NULL ) {
          if ( fe == sse->entry() ) {
            if ( strcmp( fe->name, ".." ) != 0 ) {
              found = true;
              break;
            }
          }
          fe = (FileEntry*)filelist->getNextElement();
        }
        if ( found == true ) {
          // valid entry found
	  ss2 = new NM_specialsourceInt( fe );
          ss2->row = sse->row;
          ss2->cod = NULL;
          colist->push_back( ss2 );
        }
      }
      break;
    case NM_chownorder::NM_ONLYACTIVE:
      getSelFiles( colist, NM_GETFILES_ONLYACTIVE, false );
      break;
    default:  // all selected entries
      getSelFiles( colist, NM_GETFILES_SELORACT, false );
      break;
  }

  buildOwnerRequestInfos();

  coinfo = new changeown_info_t;
  for ( iti1 = colist->begin(); iti1 != colist->end(); iti1++ ) {
    ss1 = *iti1;
    if ( worker_changeown( ss1, coinfo, coorder ) != 0 ) {
      break;
    }
  }

  for ( iti1 = colist->begin(); iti1 != colist->end(); iti1++ ) {
    ss1 = *iti1;
    delete ss1;
  }
  delete coinfo;
  delete colist;
  freeOwnerRequestInfos();
  update(false);
}

int NormalMode::worker_changeown( const struct NM_specialsourceInt *ss1,
                                  changeown_info_t *coinfo,
                                  const NM_chownorder *coorder )
{
  bool enter, dontChange, skip, cancel;
  int id, erg;
  uid_t newuid = (uid_t)-1;
  gid_t newgid = (gid_t)-1;
  FileEntry *subfe;
  NM_specialsourceInt *ss2;
  ArrayList *filelist;
  NM_CopyOp_Dir *cod;
  
  if ( ( ss1 == NULL ) || ( coinfo == NULL ) || ( coorder == NULL ) ) return -1;

  // do we have to enter this entry?
  enter = false;
  if ( ( ss1->entry()->isDir() == true ) && ( coorder->recursive == true ) ) {
    if ( ss1->entry()->isLink == false ) enter = true;
  }
  
  // check operation applies to this entry
  dontChange = false;
  if ( ( ss1->entry()->isDir() == true ) && ( coorder->ondirs == false ) ) dontChange = true;
  else if ( ( ss1->entry()->isDir() == false ) && ( coorder->onfiles == false ) ) dontChange = true;

  /* skip means skip entry AND all sub entries
   * cancel is clear
   * dontchange means to normally handle this entry
   *   but actually don't change mod
   */
  skip = cancel = false;
  if ( dontChange == false ) {
    // ask for new owner
    if ( coinfo->forAll == true ) {
      newuid = coinfo->newuid;
      newgid = coinfo->newgid;
    } else {
      erg = requestNewOwner( ss1->entry(), &newuid, &newgid );
      if ( erg == 1 ) {
        coinfo->forAll = true;
        coinfo->newuid = newuid;
        coinfo->newgid = newgid;
      } else if ( erg == 2 ) {
        skip = true;
      } else if ( erg == 3 ) {
        cancel = true;
      }
    }
  }
  if ( skip == true ) return 0;
  if ( cancel == true ) return 1;

  if ( applyNewOwner( ss1, newuid, newgid ) != 0 ) cancel = true;
  if ( ( enter == true ) && ( cancel == false ) ) {
    cod = new NM_CopyOp_Dir( ss1->entry() );
    if ( cod->user_abort == true ) {
      cancel = true;
    } else if ( cod->ok == true ) {
      filelist = cod->verz->getFiles();
      id = filelist->initEnum();
      subfe = (FileEntry*)filelist->getFirstElement( id );
      while ( ( subfe != NULL ) && ( cancel == false ) ) {
        if ( strcmp( subfe->name, ".." ) != 0 ) {
	  ss2 = new NM_specialsourceInt( subfe );
          ss2->row = -1;
          if ( worker_changeown( ss2, coinfo, coorder ) != 0 ) {
            cancel = true;
          }
          delete ss2;
        }
        subfe = (FileEntry*)filelist->getNextElement( id );
      }
      filelist->closeEnum( id );
    }
    delete cod;
  }
  return ( cancel == true ) ? 1 : 0;
}

int NormalMode::applyNewOwner( const NM_specialsourceInt *ss1, uid_t newuid, gid_t newgid )
{
  char *textstr, *buttonstr;
  int erg;
  bool cancel = false;
  
  if ( ss1 == NULL ) return -1;

  if ( chown( ss1->entry()->fullname, newuid, newgid ) != 0 ) {
    if ( errno == EPERM ) {
      buttonstr = (char*)_allocsafe( strlen( catalog.getLocale( 11 ) ) + 1 +
                                     strlen( catalog.getLocale( 8 ) ) + 1 );
      sprintf( buttonstr, "%s|%s", catalog.getLocale( 11 ),
                                   catalog.getLocale( 8 ) );
      textstr = (char*)_allocsafe( strlen( catalog.getLocale( 446 ) ) + strlen( ss1->entry()->fullname ) + 1 );
      sprintf( textstr, catalog.getLocale( 446 ), ss1->entry()->fullname );
      erg = req->request( catalog.getLocale( 347 ), textstr, buttonstr );
      _freesafe( buttonstr );
      _freesafe( textstr );
      if ( erg == 1 ) cancel = true;
    } else {
      // error
      buttonstr = (char*)_allocsafe( strlen( catalog.getLocale( 11 ) ) + 1 +
                                     strlen( catalog.getLocale( 8 ) ) + 1 );
      sprintf( buttonstr, "%s|%s", catalog.getLocale( 11 ),
                                   catalog.getLocale( 8 ) );
      textstr = (char*)_allocsafe( strlen( catalog.getLocale( 447 ) ) + strlen( ss1->entry()->fullname ) + 1 );
      sprintf( textstr, catalog.getLocale( 447 ), ss1->entry()->fullname );
      erg = req->request( catalog.getLocale( 347 ), textstr, buttonstr );
      _freesafe( buttonstr );
      _freesafe( textstr );
      if ( erg == 1 ) cancel = true;
    }
  } else {
    if ( lv->isValidRow( ss1->row ) == true ) deselect( ss1->entry(), ss1->row );
  }
  return ( cancel == true ) ? 1 : 0;
}

void NormalMode::freeOwnerRequestInfos()
{
  chownrequest_id_name_t *elem;

  if ( chownUserList != NULL ) {
    elem = (chownrequest_id_name_t*)chownUserList->getFirstElement();
    while ( elem != NULL ) {
      _freesafe( elem->name );
      _freesafe( elem );
      chownUserList->removeFirstElement();
      elem = (chownrequest_id_name_t*)chownUserList->getFirstElement();
    }
    delete chownUserList;
    chownUserList = NULL;
  }
  
  if ( chownGroupList != NULL ) {
    elem = (chownrequest_id_name_t*)chownGroupList->getFirstElement();
    while ( elem != NULL ) {
      _freesafe( elem->name );
      _freesafe( elem );
      chownGroupList->removeFirstElement();
      elem = (chownrequest_id_name_t*)chownGroupList->getFirstElement();
    }
    delete chownGroupList;
    chownGroupList = NULL;
  }
}

void NormalMode::buildOwnerRequestInfos()
{
  struct passwd *pwdp;
  struct group *grpp;
  chownrequest_id_name_t *elem;

  freeOwnerRequestInfos();

  chownUserList = new List();
  chownGroupList = new List();
  
  //setpwent();   // is this needed?
  pwdp = getpwent();
  while ( pwdp != NULL ) {
    elem = (chownrequest_id_name_t*)_allocsafe( sizeof( chownrequest_id_name_t ) );
    elem->name = dupstring( pwdp->pw_name );
    elem->id.uid = pwdp->pw_uid;
    chownUserList->addElement( elem );
    pwdp = getpwent();
  }
  endpwent();

  //setgrent();  // is this needed?
  grpp = getgrent();
  while ( grpp != NULL ) {
    elem = (chownrequest_id_name_t*)_allocsafe( sizeof( chownrequest_id_name_t ) );
    elem->name = dupstring( grpp->gr_name );
    elem->id.gid = grpp->gr_gid;
    chownGroupList->addElement( elem );
    grpp = getgrent();
  }
  endgrent();
}

int NormalMode::requestNewOwner( FileEntry *fe, uid_t *return_owner, gid_t *return_group )
{
  uid_t towner;
  gid_t tgroup;
  Button *okb, *cb, *ok2allb, *skipb;
  AWindow *win;
  Text *ttext, *utext, *gtext;
  int tw, ttw, tth, ttx, tty;
  AGMessage *msg;
  int endmode = -1;
  char *tstr;
  GUIElement *ba[4];
  FieldListView *lvu, *lvg;
  chownrequest_id_name_t *elem;
  int row, pos, i;
  
  if ( ( chownUserList == NULL ) || ( chownGroupList == NULL ) )
    buildOwnerRequestInfos();

  towner = ( ( fe->isLink == true ) && ( fe->isCorrupt == false ) ) ? fe->duserid() : fe->userid();
  tgroup = ( ( fe->isLink == true ) && ( fe->isCorrupt == false ) ) ? fe->dgroupid() : fe->groupid();

  ttw = tth = 10;
  ttx = tty = 5;
  win = new AWindow( aguix );
  win->create( NULL, 10, 10, ttw, tth, 0, catalog.getLocaleCom( 43 ) );
  tstr = (char*)_allocsafe( strlen( catalog.getLocale( 448 ) ) + strlen( fe->fullname ) + 1 );
  sprintf( tstr, catalog.getLocale( 448 ), fe->fullname );

  ttext = (Text*)win->add( new Text( aguix, ttx, tty, tstr, 1 ) );
  _freesafe( tstr );
  
  tty += ttext->getHeight() + 5;
  
  utext = (Text*)win->add( new Text( aguix, ttx, tty, catalog.getLocale( 214 ), 1 ) );
  gtext = (Text*)win->add( new Text( aguix, ttx, tty, catalog.getLocale( 215 ), 1 ) );
  tty += utext->getHeight() + 5;

  lvu = (FieldListView*)win->add( new FieldListView( aguix, ttx, tty, 50, 10 * aguix->getCharHeight(), 0 ) );
  lvg = (FieldListView*)win->add( new FieldListView( aguix, ttx, tty, 50, 10 * aguix->getCharHeight(), 0 ) );
  
  lvu->setNrOfFields( 1 );
  lvg->setNrOfFields( 1 );

  elem = (chownrequest_id_name_t*)chownUserList->getFirstElement();
  pos = 0;
  while ( elem != NULL ) {
    row = lvu->addRow();
    lvu->setText( row, 0, elem->name );
    lvu->setData( row, pos );
    lvu->setPreColors( row, FieldListView::PRECOLOR_ONLYACTIVE );
    if ( towner == elem->id.uid ) {
      lvu->setActiveRow( row );
    }
    elem = (chownrequest_id_name_t*)chownUserList->getNextElement();
    pos++;
  }
  elem = (chownrequest_id_name_t*)chownGroupList->getFirstElement();
  pos = 0;
  while ( elem != NULL ) {
    row = lvg->addRow();
    lvg->setText( row, 0, elem->name );
    lvg->setData( row, pos );
    lvg->setPreColors( row, FieldListView::PRECOLOR_ONLYACTIVE );
    if ( tgroup == elem->id.gid ) {
      lvg->setActiveRow( row );
    }
    elem = (chownrequest_id_name_t*)chownGroupList->getNextElement();
    pos++;
  }
  
  lvu->setVBarState( 2 );
  lvu->setHBarState( 0 );
  lvg->setVBarState( 2 );
  lvg->setHBarState( 0 );
  lvu->maximizeX();
  lvg->maximizeX();
  lvu->setDisplayFocus( true );
  lvg->setDisplayFocus( true );
  lvu->setAcceptFocus( true );
  lvg->setAcceptFocus( true );
  
  tw = a_max( utext->getWidth(), lvu->getWidth() );
  lvu->resize( a_max( tw, 20 * aguix->getCharWidth() ), lvu->getHeight() );
  tw = a_max( gtext->getWidth(), lvg->getWidth() );
  lvg->resize( a_max( tw, 20 * aguix->getCharWidth() ), lvg->getHeight() );
  lvg->move( lvu->getX() + lvu->getWidth() + 10, lvg->getY() );
  lvu->showActive();
  lvg->showActive();
  
  gtext->move( lvg->getX(), gtext->getY() );
  
  lvu->takeFocus();

  tty += lvu->getHeight() +5;
  
  win->maximizeX();
  w = win->getWidth();

  okb = (Button*)win->add( new Button( aguix,
                                       5,
                                       tty,
                                       catalog.getLocale( 11 ),
                                       1,
                                       0,
                                       0 ) );
  ok2allb = (Button*)win->add( new Button( aguix,
                                           0,
                                           tty,
                                           catalog.getLocale( 224 ),
                                           1,
                                           0,
                                           0 ) );
  skipb = (Button*)win->add( new Button( aguix,
                                         0,
                                         tty,
                                         catalog.getLocale( 225 ),
                                         1,
                                         0,
                                         0 ) );
  cb = (Button*)win->add( new Button( aguix,
                                      0,
                                      tty,
                                      catalog.getLocale( 8 ),
                                      1,
                                      0,
                                      0 ) );

  ba[0] = okb;
  ba[1] = ok2allb;
  ba[2] = skipb;
  ba[3] = cb;
  tw = AGUIX::scaleElementsW( w, 5, 5, -1, false, false, ba, NULL, 4 );
  if ( tw > w ) {
    w = tw;
    win->resize( w, win->getHeight() );
  }
  
  tty += okb->getHeight() + 5;
  
  for ( i = 0; i < 4; i++ ) {
    ba[i]->setAcceptFocus( true );
  }
  
  tth = tty;
  okb->takeFocus();
  win->setDoTabCycling( true );
  win->resize( w, tth );
  win->setMaxSize( w, tth );
  win->setMinSize( w, tth );
  win->show();
  for( ; endmode == -1; ) {
    msg = aguix->WaitMessage( win );
    if ( msg != NULL ) {
      switch ( msg->type ) {
        case AG_CLOSEWINDOW:
          if ( msg->closewindow.window == win->getWindow() ) endmode = 3;
          break;
        case AG_BUTTONCLICKED:
          if ( msg->button.button == okb ) endmode = 0;
          else if ( msg->button.button == ok2allb ) endmode = 1;
          else if ( msg->button.button == skipb ) endmode = 2;
          else if ( msg->button.button == cb ) endmode = 3;
          break;
        case AG_KEYPRESSED:
          if ( win->isParent( msg->key.window, false ) == true ) {
            switch ( msg->key.key ) {
              case XK_Return:
              case XK_KP_Enter:
                if ( ( ok2allb->hasFocus() == false ) &&
                     ( skipb->hasFocus() == false ) &&
                     ( cb->hasFocus() == false ) ) {
                  endmode = 0;
                }
                break;
              case XK_F1:
                endmode = 0;
                break;
              case XK_Escape:
              case XK_F4:
                endmode = 3;
                break;
              case XK_F2:
                endmode = 1;
                break;
              case XK_F3:
                endmode = 2;
                break;
            }
          }
          break;
        case AG_FIELDLV_PRESSED:
          if ( msg->fieldlv.lv == lvu ) {
            lvu->takeFocus();
          } else if ( msg->fieldlv.lv == lvg ) {
            lvg->takeFocus();
          }
      }
      aguix->ReplyMessage( msg );
    }
  }
  
  if ( ( endmode == 0 ) || ( endmode == 1 ) ) {
    // ok
    if ( return_owner != NULL )
      *return_owner = (uid_t)-1;  // -1 means no change for chown
    if ( return_group != NULL )
      *return_group = (gid_t)-1;

    row = lvu->getActiveRow();
    if ( lvu->isValidRow( row ) == true ) {
      elem = (chownrequest_id_name_t*)chownUserList->getElementAt( lvu->getData( row ) );
      if ( elem != NULL ) {
        if ( return_owner != NULL )
          *return_owner = elem->id.uid;
      }
    }

    row = lvg->getActiveRow();
    if ( lvg->isValidRow( row ) == true ) {
      elem = (chownrequest_id_name_t*)chownGroupList->getElementAt( lvg->getData( row ) );
      if ( elem != NULL ) {
        if ( return_group != NULL )
          *return_group = elem->id.gid;
      }
    }
  }
  
  win->close();
  delete win;

  return endmode;
}

void NormalMode::fixSpecialSourceList( const std::list<NM_specialsourceInt*> *l, int deleted_row )
{
  std::list<NM_specialsourceInt*>::const_iterator iti1;
  if ( l == NULL ) return;
  if ( deleted_row < 0 ) return;
  
  NM_specialsourceInt *ss1;
  
  for ( iti1 = l->begin(); iti1 != l->end(); iti1++ ) {
    ss1 = *iti1;
    if ( ss1->row == deleted_row ) {
      ss1->row = -1;
    } else if ( ss1->row > deleted_row ) {
      ss1->row--;
    }
  }
}

void NormalMode::showCacheAbs( int pos )
{
  NMCacheEntry *tce;
  int id;

  finishsearchmode();
  
  if ( ce != NULL ) {
    ce->pos = lv->getYOffset();
    ce->xpos = lv->getXOffset();
  }
  
  id = cache->initEnum();
  tce = (NMCacheEntry*)cache->getElementAt( id, pos );
  if ( tce != NULL ) {
    enterDir( tce->verz->getDir() );
  }
  cache->closeEnum( id );
}

/*
 * isColumnVisible
 *
 * checks if given column is currently visible
 */
bool NormalMode::isColumnVisible( const WorkerTypes::listcol_t c )
{
  int side = parentlister->getSide();
  const std::vector<WorkerTypes::listcol_t> *dis = wconfig->getVisCols( side );
  int i;
  bool found = false;
  
  if ( dis == NULL ) return false;
  for ( i = 0; i < (int)dis->size(); i++ ) {
    if ( (*dis)[i] == c ) {
      found = true;
      break;
    }
  }
  return found;
}

int NormalMode::createExtList( bool rec, List *extlist )
{
  std::list<NM_specialsourceExt*> *splist;

  if ( extlist == NULL ) return 1;
  splist = new std::list<NM_specialsourceExt*>;

  getSelFiles( splist, NM_GETFILES_SELORACT );
  createExtList( splist, rec, extlist );
  freeSelFiles( splist );

  return 0;
}

void NormalMode::freeExtList( List *extlist )
{
  NM_extern_fe *efe;

  if ( extlist == NULL ) return;
  efe = (NM_extern_fe*)extlist->getFirstElement();
  while ( efe != NULL ) {
    delete efe;
    efe = (NM_extern_fe*)extlist->getNextElement();
  }
}

char *NormalMode::parseComStrExt(char *sstr,class NM_externorder *extorder,int maxlen,NMExtList *entries[2], bool quote )
{
  char *tstr;
  
  tstr = parseComStr( sstr, extorder, maxlen, entries, quote );
  return tstr;
}

void NormalMode::deselect( const FileEntry *fe )
{
  FileEntry *tfe;
  int id;
  ArrayList *filelist;
  int row;

  if ( ce == NULL ) return;
  if ( ce->verz == NULL ) return;

  filelist = ce->verz->getFiles();
  
  id = filelist->initEnum();  
  tfe = (FileEntry*)filelist->getFirstElement( id );
  row = 0;
  while ( tfe != NULL ) {
    if ( tfe->use == true ) {
      if ( tfe->equals( *fe ) == true ) break;
      row++;
    }
    tfe = (FileEntry*)filelist->getNextElement( id );
  }
  filelist->closeEnum( id );
  if (tfe != NULL ) {
    /* found */
    deselect( tfe, row );
  }
}

/*
 * NormalMode::findFileEntryForData
 *
 * this search for the FE in ce with the nr "data" (usally from LV)
 * startpos >= 0 means begin the search at this position
 *
 * search is done by binary search so fe->nr needs to be ascending
 * but if it's not found, there will be a linear search so it will
 * work even when fe->nr is not ascending
 */
FileEntry *NormalMode::findFileEntryForData( int data, int startpos )
{
  FileEntry *tfe;
  int tpos;
  ArrayList *filelist;
  int id;
  int bs_start, bs_end, bs_mid;
  
  if ( ce == NULL ) return NULL;
  filelist = ce->verz->getFiles();
  id = filelist->initEnum();
  
  if ( startpos < 0 ) tpos = 0;
  else tpos = startpos;

  // normally fe->nr is ascending so we can do a binary
  // but if I change it in the future (or forgot to keep it
  // ascending) then I will do normal search if not found
  // so in no case it will be problem
  // because in many cases the topfe is near toppos
  // (when no filters are active it is toppos)
  // I first raise bs_end as needed and do then a search
  //NOTE: in many cases a linear search is faster because
  //      it's just a little loop
  //      the binary version is not much slower because
  //      bs_end is raised to the lowest possible value
  //      For big directories this is a real win
  //      so even it's more complicated I keep it

  bs_start = tpos;
  bs_end = bs_start;
  for (;;) {
    tfe = (FileEntry*)filelist->getElementAt( id, bs_end );
    if ( tfe == NULL ) break;
    if ( tfe->nr == data ) break;
    if ( tfe->nr > data ) break;
    bs_end = bs_start + ( bs_end - bs_start + 1 ) * 2;
    if ( bs_end >= filelist->size() ) {
      bs_end = filelist->size() - 1;
      break;
    }
  }
  tfe = NULL;
  do {
    bs_mid = ( bs_start + bs_end ) / 2;
    tfe = (FileEntry*)filelist->getElementAt( id, bs_mid );
    if ( tfe == NULL ) break;
    if ( tfe->nr == data ) {
      // found
      break;
    } else if ( tfe->nr < data ) {
      bs_start = bs_mid + 1;
    } else {
      bs_end = bs_mid - 1;
    }
  } while ( bs_start <= bs_end );
  if ( tfe == NULL ) {
    tfe = (FileEntry*)filelist->getElementAt( id, tpos );
    while ( tfe != NULL ) {
      if ( tfe->nr == data ) break;
      tfe = (FileEntry*)filelist->getElementAt( id, ++tpos );
    }
  }
  filelist->closeEnum( id );
  return tfe;
}

/*
 * NormalMode::createExtList
 *
 * will create a list of all involved entries depending on what type
 * of message is given.
 * At the moment it handles SPECIAL and DND
 */
int NormalMode::createExtList( ActionMessage *amsg, bool rec, List *extlist )
{
  std::list<NM_specialsourceExt*> *splist;
  NM_specialsourceExt *ss1;
  const DNDMsg *dm;
  std::list<NM_specialsourceExt*>::iterator ite1;

  if ( extlist == NULL ) return 1;
  splist = new std::list<NM_specialsourceExt*>;

  switch ( amsg->mode ) {
    case ActionMessage::AM_MODE_SPECIAL:
      ss1 = getSpecialsourceForFE( amsg->getFE() );
      if ( ss1 != NULL ) splist->push_back( ss1 );
      break;
    case ActionMessage::AM_MODE_DNDACTION:
      dm = amsg->dndmsg;
      if ( dm->getSourceLister() == parentlister ) {
        ss1 = getSpecialsourceForFE( dm->getFE() );
        if ( ss1 != NULL ) splist->push_back( ss1 );
      }
      break;
    case ActionMessage::AM_MODE_ONLYACTIVE:
      break;
    default:
      break;
  }

  createExtList( splist, rec, extlist );

  for ( ite1 = splist->begin(); ite1 != splist->end(); ite1++ ) {
    delete *ite1;
  }
  delete splist;

  return 0;
}

/*
 * getSpecialsourceForFE
 *
 * creates a NM_specialsource for the given fe if found!
 * can return NULL
 */
NM_specialsourceExt *NormalMode::getSpecialsourceForFE( const FileEntry *searchfe )
{
  ArrayList *filelist;
  int row;
  NM_specialsourceExt *ss2;
  FileEntry *fe;
  int id;

  if ( ce == NULL ) return NULL;
  if ( ce->verz == NULL ) return NULL;
  if ( searchfe == NULL ) return NULL;

  filelist = ce->verz->getFiles();
  id = filelist->initEnum();

  fe = (FileEntry*)filelist->getFirstElement( id );
  row = 0;
  ss2 = NULL;
  while ( fe != NULL ) {
    if ( fe->use == true ) {
      if ( fe->equals( searchfe ) == true ) {
	ss2 = new NM_specialsourceExt( fe );
        ss2->row = row;
        break;
      }
      row++;
    }
    fe = (FileEntry*)filelist->getNextElement( id );
  }
  filelist->closeEnum(id);

  return ss2;
}

NM_externorder::NM_externorder()
{
  separate_each_entry = false;
  recursive = false;
  inbackground = false;

  extstart = NM_EXT_START_NORMAL;

  com_str = NULL;
  view_str = NULL;

  source = NM_ALLENTRIES;
  sources = NULL;
  
  destmode = NULL;
  wpu = NULL;
  take_dirs = false;
}

NM_externorder::~NM_externorder()
{
  //TODO: at the moment I do nothing here but perhaps it's a good
  //      idea to free com_str/viewstr/sources... (it's done by the user of
  //      this class
}

NMExtList::NMExtList()
{
  extlist = new List();
  files = dirs = 0;
  take_dirs = false;
}

NMExtList::~NMExtList()
{
  NormalMode::freeExtList( extlist );
  delete extlist;
}

int NMExtList::getSize()
{
  return extlist->size();
}

int NMExtList::getRealSize()
{
  return ( files + ( ( take_dirs == true ) ? dirs : 0 ) );
}

void NMExtList::setTakeDirs( bool nv )
{
  take_dirs = nv;
}

int NMExtList::initEnum()
{
  return extlist->initEnum();
}

void NMExtList::closeEnum( int id )
{
  extlist->closeEnum( id );
}

NM_extern_fe *NMExtList::getFirstElement( int id )
{
  return (NM_extern_fe*)extlist->getFirstElement( id );
}

NM_extern_fe *NMExtList::getNextElement( int id )
{
  return (NM_extern_fe*)extlist->getNextElement( id );
}

int NMExtList::removeElement( NM_extern_fe *efe )
{
  if ( efe == NULL ) return 1;
  if ( efe->getDirFinished() != 0 ) dirs--;
  else files--;
  extlist->removeElement( efe );
  return 0;
}

void NMExtList::createExtList( class NormalMode *nm, bool rec )
{
  if ( nm != NULL ) {
    nm->createExtList( rec, extlist );
    files = NormalMode::getEntriesRealSize( extlist );
    dirs = extlist->size() - files;
  }
}

void NMExtList::createExtList( class NormalMode *nm, ActionMessage *msg, bool rec )
{
  if ( nm != NULL ) {
    nm->createExtList( msg, rec, extlist );
    files = NormalMode::getEntriesRealSize( extlist );
    dirs = extlist->size() - files;
  }
}

void NormalMode::lvbDoubleClicked()
{
  showCacheList();
}

int NormalMode::showCacheList()
{
  AWindow *win;
  int my_w, my_h, endmode, mw, mh, tw;
  AGMessage *msg;
  int ty, row, my_lastrow;
  FieldListView *cachelv;
  Button *cacheokcb[2];
  StringGadget *cachesg;
  Text *ttext;
  char *tstr1, *tstr2;
  int caches = cache->size();
  NMCacheEntry *tce = NULL;
  int id;
  int parents, cachebegin;
  struct timeval my_lastclick, acttime;
  const char *tstr;
  
  win = new AWindow( aguix );
  my_w = my_h = 20;
  win->create( NULL, 10, 10, my_w, my_h, 0, catalog.getLocale( 523 ) );
  ty = 5;
  
  ttext = (Text*)win->add( new Text( aguix, 5, ty, catalog.getLocale( 524 ), 1 ) );
  ty += ttext->getHeight() + 5;
  
  cachelv = (FieldListView*)win->add( new FieldListView( aguix, 5, ty, 50, 50 ,0 ) );
  cachelv->setHBarState( 2 );
  cachelv->setVBarState( 2 );
  ty += cachelv->getHeight();
  
  cachesg = (StringGadget*)win->add( new StringGadget( aguix, 5, ty, 50, "", 0 ) );
  ty += cachesg->getHeight() + 5;
  
  cacheokcb[0] = (Button*)win->add( new Button( aguix, 5, ty, catalog.getLocale( 11 ), 1, 0, 0 ) );
  cacheokcb[1] = (Button*)win->add( new Button( aguix, 5, ty, catalog.getLocale( 8 ), 1, 0, 0 ) );
  ty += cacheokcb[0]->getHeight() + 5;

  my_h = ty;

  tstr2 = getCurrentDir();
  if ( tstr2 != NULL ) {
    tstr1 = dupstring( tstr2 );
    for ( ;; ) {
      tstr2 = ParentDir( tstr1, NULL );
      if ( tstr2 == NULL ) break;
      
      // add tstr2 to the lv
      row = cachelv->addRow();
      cachelv->setText( row, 0 , tstr2 );
      cachelv->setPreColors( row, FieldListView::PRECOLOR_ONLYACTIVE );
      
      _freesafe( tstr1 );
      tstr1 = tstr2;
      if ( strcmp( tstr1, "/" ) == 0 ) break;
    }
    _freesafe( tstr1 );
  }
  
  parents = cachelv->getElements();

  tstr1 = (char*)_allocsafe( strlen( catalog.getLocale( 523 ) ) + 1 );
  memset( tstr1, '-', strlen( catalog.getLocale( 523 ) ) );
  tstr1[strlen( catalog.getLocale( 523 ) )] = '\0';
  row = cachelv->addRow();
  cachelv->setText( row, 0, tstr1 );
  cachelv->setPreColors( row, FieldListView::PRECOLOR_NOTSELORACT );
  row = cachelv->addRow();
  cachelv->setText( row, 0, catalog.getLocale( 523 ) );
  cachelv->setPreColors( row, FieldListView::PRECOLOR_NOTSELORACT );
  row = cachelv->addRow();
  cachelv->setText( row, 0, tstr1 );
  cachelv->setPreColors( row, FieldListView::PRECOLOR_NOTSELORACT );
  _freesafe( tstr1 );
  
  cachebegin = row + 1;

  if ( caches > 0 ) {
    id = cache->initEnum();
    tce = (NMCacheEntry*)cache->getFirstElement( id );
    while ( tce != NULL ) {
      row = cachelv->addRow();
      cachelv->setText( row, 0, tce->verz->getDir() );
      cachelv->setPreColors( row, FieldListView::PRECOLOR_ONLYACTIVE );
      tce = (NMCacheEntry*)cache->getNextElement( id );
    }
    cache->closeEnum( id );
  }

  cachelv->maximizeX();
  cachelv->maximizeY();
  cachesg->move( cachesg->getX(), cachelv->getY() + cachelv->getHeight() );
  cacheokcb[0]->move( 5, cachesg->getY() + cachesg->getHeight() + 5 );
  cacheokcb[1]->move( 5, cachesg->getY() + cachesg->getHeight() + 5 );
  win->maximizeX();
  win->maximizeY();
  my_w = win->getWidth();
  my_h = win->getHeight();
  mw = aguix->getRootWindowWidth() * 80 / 100;
  mh = aguix->getRootWindowHeight() * 80 / 100;
  if ( my_w > mw ) {
    my_w = mw;
  }
  tw = AGUIX::scaleElementsW( my_w, 5, 5, -1, false, false, (GUIElement**)cacheokcb, NULL, 2 );
  if ( tw > my_w ) {
    my_w = tw;
  }
  cachelv->resize( my_w - cachelv->getX() * 2, cachelv->getHeight() );
  cachesg->resize( cachelv->getWidth(), cachesg->getHeight() );
  if ( my_h > mh ) {
    cacheokcb[0]->move( cacheokcb[0]->getX(), mh - 5 -cacheokcb[0]->getHeight() );
    cacheokcb[1]->move( cacheokcb[1]->getX(), mh - 5 -cacheokcb[1]->getHeight() );
    cachesg->move( cachesg->getX(), cacheokcb[0]->getY() - 5 - cachesg->getHeight() );
    cachelv->resize( cachelv->getWidth(), cachesg->getY() - cachelv->getY() );
    my_h = mh;
  }

  win->resize( my_w, my_h );
  win->setMinSize( my_w, my_h );
  win->setDoTabCycling( true );
  cachesg->takeFocus();

  if ( parents > 0 ) {
    cachelv->setActiveRow( 0 );
    cachesg->setText( cachelv->getText( 0, 0 ).c_str() );
  } else if ( caches > 0 ) {
    cachelv->setActiveRow( cachebegin );
    cachesg->setText( cachelv->getText( cachebegin, 0 ).c_str() );
  }

  // catch msgs before mapping, this are mostly resize msgs
  while ( ( msg = aguix->GetMessage( win ) ) != NULL ) aguix->ReplyMessage( msg );

  win->show();

  my_lastrow = -1;
  timerclear( &my_lastclick );
  for( endmode = -1; endmode == -1; ) {
    msg = aguix->WaitMessage( win );
    if ( msg != NULL ) {
      switch ( msg->type ) {
        case AG_CLOSEWINDOW:
          if ( msg->closewindow.window == win->getWindow() ) endmode = 1;
          break;
        case AG_BUTTONCLICKED:
          if ( msg->button.button == cacheokcb[0] ) endmode = 0;
          else if ( msg->button.button == cacheokcb[1] ) endmode = 1;
          break;
        case AG_SIZECHANGED:
          if ( msg->size.window == win->getWindow() ) {
            cachelv->resize( msg->size.neww - 10, cachelv->getHeight() );
            cachesg->resize( cachelv->getWidth(), cachesg->getHeight() );
            AGUIX::scaleElementsW( msg->size.neww, 5, 5, -1, false, false, (GUIElement**)cacheokcb, NULL, 2 );
            cacheokcb[0]->move( cacheokcb[0]->getX(), msg->size.newh - 5 - cacheokcb[0]->getHeight() );
            cacheokcb[1]->move( cacheokcb[1]->getX(), msg->size.newh - 5 - cacheokcb[1]->getHeight() );
            cachesg->move( cachesg->getX(), cacheokcb[0]->getY() - 5 - cachesg->getHeight() );
            cachelv->resize( cachelv->getWidth(), cachesg->getY() - cachelv->getY() );
          }
          break;
        case AG_FIELDLV_ONESELECT:
        case AG_FIELDLV_MULTISELECT:
          if ( msg->fieldlv.lv == cachelv ) {
            row = cachelv->getActiveRow();
            if ( cachelv->isValidRow( row ) == true ) {
              if ( ( row < parents ) ||
                   ( row >= cachebegin ) ) {
                // this is one of the parents
                cachesg->setText( cachelv->getText( row, 0 ).c_str() );
                if ( my_lastrow == row ) {
                  gettimeofday( &acttime, NULL );
                  if ( aguix->isDoubleClick( &acttime, &my_lastclick ) == true ) {
                    endmode = 0;
                    timerclear( &my_lastclick );
                  } else {
                    my_lastclick.tv_sec = acttime.tv_sec;
                    my_lastclick.tv_usec = acttime.tv_usec;
                  }
                } else {
                  my_lastrow = row;
                  gettimeofday( &my_lastclick, NULL );
                }
              }
            }
          }
          break;
        case AG_STRINGGADGET_CONTENTCHANGE:
          if ( msg->stringgadget.sg == cachesg ) {
            // to avoid doubleclick
            timerclear( &my_lastclick );
            //TODO: U.U. merkt user ein Lag, deshalb knnte man ein Flag setzen und
            // auerhalb messen, wenn letzte Nachricht her ist und dann suchen
            // Aber dann muss ich getMessage machen
            tstr = cachesg->getText();
            for ( row = 0; row < cachelv->getElements(); row++ ) {
              if ( strcmp( tstr, cachelv->getText( row, 0 ).c_str() ) == 0 ) {
                if ( ( ( row >= 0 ) && ( row < parents ) ) ||
                     ( ( row >= cachebegin ) && ( row < ( cachebegin + caches ) ) ) ) {
                  cachelv->setActiveRow( row );
                  cachelv->showActive();
                }
                break;
              }
            }
          }
          break;
        case AG_STRINGGADGET_CANCEL:
          if ( msg->stringgadget.sg == cachesg ) endmode = 1;
          break;
        case AG_STRINGGADGET_OK:
          if ( msg->stringgadget.sg == cachesg ) endmode = 0;
          break;
        case AG_KEYPRESSED:
          if ( win->isParent( msg->key.window, false ) == true ) {
            switch ( msg->key.key ) {
              case XK_Up:
                row = cachelv->getActiveRow();
                if ( cachelv->isValidRow( row ) == true ) {
                  if ( ( row >= parents ) && ( row <= cachebegin ) ) {
                    row = parents - 1;
                  } else {
                    row--;
                  }
                  if ( row >= 0 ) {
                    cachelv->setActiveRow( row );
                    cachelv->showActive();
                    cachesg->setText( cachelv->getText( row, 0 ).c_str() );
                    my_lastrow = -1;
                  }
                }
                break;
              case XK_Down:
                row = cachelv->getActiveRow();
                if ( cachelv->isValidRow( row ) == true ) {
                  if ( ( row >= ( parents - 1 ) ) && ( row < cachebegin ) ) {
                    row = cachebegin;
                  } else {
                    row++;
                  }
                  if ( cachelv->isValidRow( row ) == true ) {
                    cachelv->setActiveRow( row );
                    cachelv->showActive();
                    cachesg->setText( cachelv->getText( row, 0 ).c_str() );
                    my_lastrow = -1;
                  }
                }
                break;
              case XK_Prior:
                row = cachelv->getActiveRow();
                if ( cachelv->isValidRow( row ) == true ) {
                  row -= cachelv->getMaxDisplayV() - 1;
                  if ( ( row >= parents ) && ( row < cachebegin ) ) row = parents - 1;
                  if ( row < 0 ) {
                    if ( parents > 0 ) row = 0;
                    else row = cachebegin;
                  }
                } else {
                  if ( parents > 0 ) row = 0;
                  else if ( caches > 0 ) row = cachebegin;
                }
                if ( cachelv->isValidRow( row ) == true ) {
                  cachelv->setActiveRow( row );
                  cachelv->showActive();
                  cachesg->setText( cachelv->getText( row, 0 ).c_str() );
                  my_lastrow = -1;
                }
                break;
              case XK_Next:
                row = cachelv->getActiveRow();
                if ( cachelv->isValidRow( row ) == true ) {
                  row += cachelv->getMaxDisplayV() - 1;
                  if ( ( row >= parents ) && ( row < cachebegin ) ) row = cachebegin;
                  if ( row >= ( cachebegin + caches ) ) {
                    if ( caches > 0 ) row = cachebegin + caches - 1;
                    else row = parents - 1;
                  }
                } else {
                  if ( parents > 0 ) row = 0;
                  else if ( caches > 0 ) row = cachebegin;
                }
                if ( cachelv->isValidRow( row ) == true ) {
                  cachelv->setActiveRow( row );
                  cachelv->showActive();
                  cachesg->setText( cachelv->getText( row, 0 ).c_str() );
                  my_lastrow = -1;
                }
                break;
              case XK_Home:
                row = -1;
                if ( parents > 0 ) row = 0;
                else if ( caches > 0 ) row = cachebegin;
                if ( cachelv->isValidRow( row ) == true ) {
                  cachelv->setActiveRow( row );
                  cachelv->showActive();
                  cachesg->setText( cachelv->getText( row, 0 ).c_str() );
                  my_lastrow = -1;
                }
                break;
              case XK_End:
                row = -1;
                if ( caches > 0 ) row = cachebegin + caches - 1;
                else if ( parents > 0 ) row = parents - 1;
                if ( cachelv->isValidRow( row ) == true ) {
                  cachelv->setActiveRow( row );
                  cachelv->showActive();
                  cachesg->setText( cachelv->getText( row, 0 ).c_str() );
                  my_lastrow = -1;
                }
                break;
              case XK_Return:
              case XK_KP_Enter:
                if ( cacheokcb[1]->hasFocus() == false ) {
                  endmode = 0;
                }
                break;
              case XK_Escape:
                endmode = 1;
                break;
            }
          }
          break;
      }
      aguix->ReplyMessage( msg );
    }
  }
  if ( endmode == 0 ) {
    // ok->take cachesg and enterdir
    enterDir( cachesg->getText() );
  }
  win->close();
  delete win;
  return endmode;
}

void NormalMode::showInvalidList( ArrayList *list, const char *descr )
{
  int id;
  FileEntry *tfe;
  int tw, th, tx, ty, endmode, row;
  AWindow *win;
  FieldListView *my_lv;
  Button *okayb;
  AGMessage *msg;

  if ( ( list == NULL ) || ( descr == NULL ) ) return;
  
  win = new AWindow( aguix );
  tw = th = 10;
  win->create( NULL, 10, 10, tw, th, 0, catalog.getLocale( 125 ) );
  tx = ty = 5;
  win->addTextFromString( descr, tx, ty, 5, NULL, NULL, &ty );
  ty += 5;
  my_lv = (FieldListView*)win->add( new FieldListView( aguix,
                                                       tx,
                                                       ty,
                                                       40 * aguix->getCharWidth(),
                                                       6 * aguix->getCharHeight(),
                                                       0 ) );
  ty += my_lv->getHeight() + 5;
  
  my_lv->setHBarState( 2 );
  my_lv->setVBarState( 2 );
  
  okayb = (Button*)win->add( new Button( aguix,
                                         0,
                                         ty,
                                         catalog.getLocale( 11 ),
                                         1,
                                         0,
                                         0 ) );
  ty += okayb->getHeight() + 5;
  win->maximizeX();

  tw = AGUIX::scaleElementsW( win->getWidth(), 5, 5, -1, false, false, (GUIElement**)&okayb, NULL, 1 );
  if ( tw < win->getWidth() ) tw = win->getWidth();

  win->resize( tw, ty );
  win->setMinSize( win->getWidth(), win->getHeight() );
  win->setDoTabCycling( true );
  
  my_lv->resize( win->getWidth() - 10, my_lv->getHeight() );
  
  win->show();
  
  id = list->initEnum();
  tfe = (FileEntry*)list->getFirstElement( id );
  while ( tfe != NULL ) {
    row = my_lv->addRow();
    my_lv->setText( row, 0, tfe->name );
    my_lv->setPreColors( row, FieldListView::PRECOLOR_NOTSELORACT );
    tfe = (FileEntry*)list->getNextElement( id );
  }
  list->closeEnum( id );
    
  for( endmode = -1; endmode == -1; ) {
    msg = aguix->WaitMessage( win );
    if ( msg != NULL ) {
      switch ( msg->type ) {
        case AG_CLOSEWINDOW:
          if ( msg->closewindow.window == win->getWindow() ) endmode = 1;
          break;
        case AG_BUTTONCLICKED:
          if ( msg->button.button == okayb ) endmode = 0;
          break;
        case AG_SIZECHANGED:
          if ( msg->size.window == win->getWindow() ) {
            AGUIX::scaleElementsW( msg->size.neww, 5, 5, -1, false, false, (GUIElement**)&okayb, NULL, 1 );
            okayb->move( okayb->getX(), msg->size.newh - 5 - okayb->getHeight() );
            my_lv->resize( msg->size.neww - 10, okayb->getY() - 5 - my_lv->getY() );
          }
          break;
        case AG_KEYPRESSED:
          if ( win->isParent( msg->key.window, false ) == true ) {
            switch ( msg->key.key ) {
              case XK_Return:
              case XK_KP_Enter:
                endmode = 0;
                break;
              case XK_Escape:
                endmode = 1;
                break;
            }
          }
          break;
      }
      aguix->ReplyMessage( msg );
    }
  }
  win->close();
  delete win;
  return;
}

int NormalMode::request( const char *title,
                         const char *text,
                         const char *buttons,
                         Requester::request_flags_t flags )
{
  return req->request( title, text, buttons, flags );
}

int NormalMode::string_request( const char *title,
                                const char *lines,
                                const char *default_str,
                                const char *buttons,
                                char **return_str,
                                Requester::request_flags_t flags )
{
  return req->string_request( title, lines, default_str, buttons, return_str, flags );
}

int NormalMode::request( CopyOpWin *cowin,
                         const char *title,
                         const char *text,
                         const char *buttons,
                         Requester::request_flags_t flags )
{
  if ( cowin != NULL ) {
    return cowin->request( title, text, buttons, flags );
  } else {
    return req->request( title, text, buttons, flags );
  }
}

int NormalMode::string_request( CopyOpWin *cowin,
                                const char *title,
                                const char *lines,
                                const char *default_str,
                                const char *buttons,
                                char **return_str,
                                Requester::request_flags_t flags )
{
  if ( cowin != NULL ) {
    return cowin->string_request( title, lines, default_str, buttons, return_str, flags );
  } else {
    return req->string_request( title, lines, default_str, buttons, return_str, flags );
  }
}

int NormalMode::request( DeleteOpWin *dowin,
                         const char *title,
                         const char *text,
                         const char *buttons,
                         Requester::request_flags_t flags )
{
  if ( dowin != NULL ) {
    return dowin->request( title, text, buttons, flags );
  } else {
    return req->request( title, text, buttons, flags );
  }
}

int NormalMode::string_request( DeleteOpWin *dowin,
                                const char *title,
                                const char *lines,
                                const char *default_str,
                                const char *buttons,
                                char **return_str,
                                Requester::request_flags_t flags )
{
  if ( dowin != NULL ) {
    return dowin->string_request( title, lines, default_str, buttons, return_str, flags );
  } else {
    return req->string_request( title, lines, default_str, buttons, return_str, flags );
  }
}

/*
 * tryApplyOwnerPerm will try to set owner and permission
 *   it will apply SUID/SGID only if owner or root (as root only
 *   if chown don't fail)
 * this function don't care if chmod/chown fails
 *
 * returnvalue:
 *   0 okay
 *   -1 wrong args
 *   Bit 0 set: SUID cleared
 *   Bit 1 set: SGID cleared
 *   Bit 2 set: chown failed
 *   Bit 3 set: chmod failed
 */
int NormalMode::tryApplyOwnerPerm( const char *filename,
                                   const uid_t wanted_user,
                                   const gid_t wanted_group,
                                   const mode_t wanted_mode,
                                   struct NM_copyorder *copyorder )
{
  mode_t use_mode;
  int erg = 0, chownres, chmodres;

  if ( filename == NULL ) return -1;
  
  chownres = chown( filename, wanted_user, wanted_group );
  if ( chownres != 0 ) {
    erg |= 1<<2;
  }
  use_mode = wanted_mode;
  if ( ( use_mode & S_ISUID ) != 0 ) {
    // SUID bit set, check if we still want it
    // root will apply it (when chown didn't failed and)
    // and euid/egid == ower
    if ( ! ( ( geteuid() == wanted_user ) ||
             ( ( geteuid() == 0 ) && ( chownres == 0 ) ) ) ) {
      use_mode &= ~S_ISUID;
      erg |= 1<<0;
    }
  }
  if ( ( use_mode & S_ISGID ) != 0 ) {
    // SGID bit set, check if we still want it
    // root will apply it (when chown didn't failed and)
    // and euid/egid == ower
    if ( ! ( ( getegid() == wanted_group ) ||
             ( ( geteuid() == 0 ) && ( chownres == 0 ) ) ) ) {
      use_mode &= ~S_ISGID;
      erg |= 1<<1;
    }
  }
  chmodres = chmod ( filename, use_mode );
  if ( chmodres != 0 ) {
    erg |= 1<<3;
  }
  if ( ( ( erg & 3 ) != 0 ) && ( copyorder != NULL ) ) {
    // removed SUID/SGUID bit, let the user know this
    if ( copyorder->ignoreLosedAttr != true ) {
      int choose;
      char *textstr;
      std::string str2;

      textstr = (char*)_allocsafe( strlen( catalog.getLocale( 536 ) ) + strlen( filename ) + 1 );
      sprintf( textstr, catalog.getLocale( 536 ), filename );
      str2 = catalog.getLocale( 11 );
      str2 += "|";
      str2 += catalog.getLocale( 537 );
      choose = request( copyorder->cowin,
                        catalog.getLocale( 125 ),
                        textstr,
                        str2.c_str(),
                        Requester::REQUEST_CANCELWITHLEFT );
      _freesafe( textstr );
      if ( choose == 1 ) {
        copyorder->ignoreLosedAttr = true;
      }
    }
  }
  return erg;
}

void NormalMode::changeSortModeForField( int field )
{
  int side = parentlister->getSide();
  int realfield;
  const std::vector<WorkerTypes::listcol_t> *sets;
  int used_fields;
  int newsortmode = sortmode;

  sets = wconfig->getVisCols( side );
  if ( sets == NULL ) return;
  
  used_fields = sets->size();
  
  realfield = field / 2;
  if ( ( realfield >= 0 ) && ( realfield < used_fields ) ) {
    switch ( (*sets)[realfield] ) {
      case WorkerTypes::LISTCOL_NAME:
        if ( ( newsortmode & 0xff ) == SORT_NAME ) {
          // toggle reverse
          newsortmode ^= SORT_REVERSE;
        } else {
          newsortmode = SORT_NAME | ( newsortmode & ~0xff );
          newsortmode &= ~SORT_REVERSE;
        }
        break;
      case WorkerTypes::LISTCOL_SIZE:
        if ( ( newsortmode & 0xff ) == SORT_SIZE ) {
          // toggle reverse
          newsortmode ^= SORT_REVERSE;
        } else {
          newsortmode = SORT_SIZE | ( newsortmode & ~0xff );
          newsortmode &= ~SORT_REVERSE;
        }
        break;
      case WorkerTypes::LISTCOL_TYPE:
        if ( ( newsortmode & 0xff ) == SORT_TYPE ) {
          // toggle reverse
          newsortmode ^= SORT_REVERSE;
        } else {
          newsortmode = SORT_TYPE | ( newsortmode & ~0xff );
          newsortmode &= ~SORT_REVERSE;
        }
        break;
      case WorkerTypes::LISTCOL_OWNER:
        if ( ( newsortmode & 0xff ) == SORT_OWNER ) {
          // toggle reverse
          newsortmode ^= SORT_REVERSE;
        } else {
          newsortmode = SORT_OWNER | ( newsortmode & ~0xff );
          newsortmode &= ~SORT_REVERSE;
        }
        break;
      case WorkerTypes::LISTCOL_MOD:
        if ( ( newsortmode & 0xff ) == SORT_MODTIME ) {
          // toggle reverse
          newsortmode ^= SORT_REVERSE;
        } else {
          newsortmode = SORT_MODTIME | ( newsortmode & ~0xff );
          newsortmode &= ~SORT_REVERSE;
        }
        break;
      case WorkerTypes::LISTCOL_ACC:
        if ( ( newsortmode & 0xff ) == SORT_ACCTIME ) {
          // toggle reverse
          newsortmode ^= SORT_REVERSE;
        } else {
          newsortmode = SORT_ACCTIME | ( newsortmode & ~0xff );
          newsortmode &= ~SORT_REVERSE;
        }
        break;
      case WorkerTypes::LISTCOL_CHANGE:
        if ( ( newsortmode & 0xff ) == SORT_CHGTIME ) {
          // toggle reverse
          newsortmode ^= SORT_REVERSE;
        } else {
          newsortmode = SORT_CHGTIME | ( newsortmode & ~0xff );
          newsortmode &= ~SORT_REVERSE;
        }
        break;
      case WorkerTypes::LISTCOL_INODE:
        if ( ( newsortmode & 0xff ) == SORT_INODE ) {
          // toggle reverse
          newsortmode ^= SORT_REVERSE;
        } else {
          newsortmode = SORT_INODE | ( newsortmode & ~0xff );
          newsortmode &= ~SORT_REVERSE;
        }
        break;
      case WorkerTypes::LISTCOL_NLINK:
        if ( ( newsortmode & 0xff ) == SORT_NLINK ) {
          // toggle reverse
          newsortmode ^= SORT_REVERSE;
        } else {
          newsortmode = SORT_NLINK | ( newsortmode & ~0xff );
          newsortmode &= ~SORT_REVERSE;
        }
        break;
      case WorkerTypes::LISTCOL_BLOCKS:
        //no sort available
        break;
      case WorkerTypes::LISTCOL_PERM:
        //no sort available
        break;
      case WorkerTypes::LISTCOL_DEST:
        //no sort available
        break;
      default:
        break;
    }
    if ( newsortmode != sortmode ) {
      setSortmode( newsortmode );
    }
  }
}

NormalMode::changemod_info::changemod_info()
{
  forAll = false;
  newmode = 0;
}

NormalMode::changeown_info::changeown_info()
{
  forAll = false;
  newuid = (uid_t)-1;
  newgid = (gid_t)-1;
}

#ifdef WANT_THREADS
int NormalMode::switchOrder( thread_order_t neworder )
{
  if ( slave.order != neworder ) {
    reclist->lock();
    slave.order = neworder;
    reclist->signal();
    reclist->unlock();
  }
  return 0;
}

int NormalMode::waitForNoOrder( void )
{
  reclist->lock();
  while ( slave.order != THREAD_NOP ) {
    reclist->wait();
  }
  reclist->unlock();
  return 0;
}

int NormalMode::waitForNewOrder( void )
{
  reclist->lock();
  while ( slave.order == THREAD_NOP ) {
    reclist->wait();
  }
  reclist->unlock();
  return 0;
}

int NormalMode::switchStatus( thread_order_t newstatus )
{
  if ( slave.status != newstatus ) {
    slave.statusvar.lock();
    slave.status = newstatus;
    slave.statusvar.signal();
    slave.statusvar.unlock();
  }
  return 0;
}

int NormalMode::waitForStatus( thread_order_t waitstatus )
{
  slave.statusvar.lock();
  while ( slave.status != waitstatus ) {
    slave.statusvar.wait();
  }
  slave.statusvar.unlock();
  return 0;
}
#endif

WCFiletype *NormalMode::findFiletype( std::vector<unsigned int> *v )
{
  unsigned int in1, p, tp;
  const std::list<WCFiletype*> *subtype;
  std::list<WCFiletype*>::const_iterator it1;
  WCFiletype *ft;
  
  ft = NULL;

  // check tree depth
  p = v->size();
  if ( p > 0 ) {
    // we need at least the root type
    in1 = (*v)[p - 1];  // get position
    
    ft =  (WCFiletype*)wconfig->getFiletypes()->getElementAt( in1 );
    // root type, no problem if it is NULL
    
    // now go for each tree depth
    p = v->size();
    if ( p > 1 ) {
      // we need one more depth at least
      
      p--; // skip root index
      
      do {
	// now for each depth

	p--; // jump to next level
	// p is correct entry in "v"

	if ( ft == NULL ) break;

	// get position in current tree depth
	in1 = (*v)[p];

	// now find subtype
	// because of std::list we need to iterate
	subtype = ft->getSubTypeList();
	if ( subtype == NULL ) ft = NULL;  // oops, index but no subtype
	else {
	  // iterate through subtype list and find "in1" position
	  for ( it1 = subtype->begin(), tp = 0; it1 != subtype->end(); it1++, tp++ ) {
	    if ( tp == in1 ) break;
	  }
	  
	  if ( it1 != subtype->end() ) {
	    // found subtype at index "in1"
	    ft = *it1;
	  } else {
	    ft = NULL;
	  }
	}
	// stop when we processed last depth
      } while ( p > 0 );
    }
  }
  return ft;
}

int NormalMode::getSelFiles( std::list<NM_specialsourceInt*> *list, nm_getfiles_t selmode, bool unselect )
{
  int added = 0;
  ArrayList *filelist;
  int row;
  NM_specialsourceInt *ss2;
  FileEntry *fe;
  int id;

  if ( ce == NULL ) return -1;
  if ( ce->verz == NULL ) return -1;
  if ( list == NULL ) return -1;

  filelist = ce->verz->getFiles();
  id = filelist->initEnum();

  if ( selmode != NM_GETFILES_ONLYACTIVE ) {
    fe = (FileEntry*)filelist->getFirstElement( id );
    row = 0;
    while ( fe != NULL ) {
      if ( fe->use == true ) {
	if ( ( strcmp( fe->name, ".." ) != 0 ) && ( fe->select == true ) ) {
	  ss2 = new NM_specialsourceInt( fe );
	  ss2->row = row;
	  ss2->cod = NULL;
	  list->push_back( ss2 );
	  added++;
	  if ( unselect == true ) deselect( fe, ss2->row );
	}
	row++;
      }
      fe = (FileEntry*)filelist->getNextElement( id );
    }
  }

  // for ONLYACTIVE always take the active
  // otherwise only if not ONLYSELECT and empty list
  if ( ( selmode == NM_GETFILES_ONLYACTIVE ) ||
       ( ( selmode != NM_GETFILES_ONLYSELECT ) && ( list->size() == 0 ) ) ) {
    if ( ce->activefe != NULL ) {
      row = lv->getActiveRow();
      if ( lv->isValidRow( row ) == true ) {
	if ( lv->getData( row ) == ce->activefe->nr ) {
	  if ( strcmp( ce->activefe->name, ".." ) != 0 ) {
	    ss2 = new NM_specialsourceInt( ce->activefe );
	    ss2->row = row;
	    ss2->cod = NULL;
	    list->push_back( ss2 );
	    added++;
	    if ( unselect == true ) deselect( ce->activefe, ss2->row );
	  }
	}
      }
    }
  }
  filelist->closeEnum( id );
  
  return added;
}

void NormalMode::freeSelFiles( std::list<NM_specialsourceExt*> *splist ) const
{
  std::list<NM_specialsourceExt*>::iterator it1;
  
  if ( splist == NULL ) return;
  for ( it1 = splist->begin(); it1 != splist->end(); it1++ ) {
    if ( *it1 != NULL ) delete *it1;
  }
  delete splist;
}

NM_specialsourceExt::NM_specialsourceExt( const FileEntry *tfe )
{
  if ( tfe == NULL ) {
    fe = NULL;
  } else {
    fe = new FileEntry( *tfe );
  }
  row = -1;
  //  cod = NULL;
}

NM_specialsourceExt::~NM_specialsourceExt()
{
  if ( fe != NULL ) delete fe;
}

NormalMode::NM_specialsourceInt::NM_specialsourceInt( FileEntry *tfe )
{
  fe = tfe;
  row = -1;
  cod = NULL;
}

NormalMode::NM_specialsourceInt::~NM_specialsourceInt()
{
}

FileEntry *NormalMode::findFE( const FileEntry *searchfe )
{
  ArrayList *filelist;
  FileEntry *fe;
  int id;

  if ( ce == NULL ) return NULL;
  if ( ce->verz == NULL ) return NULL;

  filelist = ce->verz->getFiles();
  id = filelist->initEnum();

  fe = (FileEntry*)filelist->getFirstElement( id );
  while ( fe != NULL ) {
    if ( fe->equals( searchfe ) == true ) {
      break;
    }
    fe = (FileEntry*)filelist->getNextElement( id );
  }
  filelist->closeEnum(id);

  return fe;
}
