% -*- LaTeX -*-
% $Id: CamPP.lhs,v 1.3 2004/08/08 12:05:32 berrueta Exp $
%
% Copyright (c) 2002,2003, Wolfgang Lux
% See LICENSE for the full license.
%
\codesubsection{Pretty-printing abstract machine code}
\begin{lstlisting}

> module CamPP where
> import Cam
> import Char
> import Pretty

> default(Int)

> blockIndent = 2

> ppModule :: Module -> Doc
> ppModule ds = vcat $ punctuate semi $ map ppDecl ds

> ppDecl :: Decl -> Doc
> ppDecl (ImportDecl m) = ppKW "import" <+> ppName m
> ppDecl (DataDecl t cs) =
>   ppKW "data" <+> ppName t
>               <+> sep (zipWith (<+>) (equals : repeat bar) (map ppConstr cs))
>   where ppConstr (ConstrDecl c n) = ppName c <> char '/' <> int n
> ppDecl (FunctionDecl f vs is) =
>   ppCode (ppKW "function" <+> ppName f <+> ppNames vs) is

> ppCode :: Doc -> Stmt -> Doc
> ppCode prefix = ppBlock prefix . ppStmt

> ppBlock :: Doc -> Doc -> Doc
> ppBlock prefix x = sep [prefix <+> lbrace,nest blockIndent x,rbrace]

> ppStmt :: Stmt -> Doc
> ppStmt (Return v) = ppKW "return" <+> ppName v
> ppStmt (Enter v) = ppKW "enter" <+> ppName v
> ppStmt (Exec f vs) = ppKW "exec" <+> ppName f <+> ppNames vs
> ppStmt (Lock v st) = ppKW "lock" <+> ppName v <> semi $$ ppStmt st
> ppStmt (Update v1 v2 st) =
>   ppKW "update" <+> ppName v1 <+> ppName v2 <> semi $$ ppStmt st
> ppStmt (Seq v st1 st2) =
>   (case st1 of
>      Let _ _ -> ppBlock (ppName v <+> text "<-") (ppStmt st1)
>      Seq _ _ _ -> ppBlock (ppName v <+> text "<-") (ppStmt st1)
>      _ -> ppName v <+> text "<-" <+> ppStmt st1) <> semi $$ ppStmt st2
> ppStmt (Let bds st) = ppKW "let" <+> ppBindings bds <> semi $$ ppStmt st
>   where ppBindings bds =
>           lbrace <+> vcat (punctuate semi (map ppBinding bds)) <+> rbrace
>         ppBinding (Bind v n) = ppName v <+> equals <+> ppExpr n
> ppStmt (Switch rf v cases) =
>   ppBlock (ppKW "switch" <+> ppName v <+> ppRF rf) (ppAlts ppCase cases)
>   where ppRF Rigid = ppKW "rigid"
>         ppRF Flex = ppKW "flex"
> ppStmt (Choices alts) = ppBlock (ppKW "choices") (ppAlts ppAlt alts)

> ppLiteral :: Literal -> Doc
> ppLiteral (Char c) = ppKW "char" <+> int (ord c)
> ppLiteral (Int i) = ppKW "int" <+> int i
> ppLiteral (Float f) = ppKW "float" <+> double f

> ppExpr :: Expr -> Doc
> ppExpr (Lit c) = ppLiteral c
> ppExpr (Constr c vs) = ppKW "data" <+> ppName c <+> ppNames vs
> ppExpr (Closure f vs) = ppKW "function" <+> ppName f <+> ppNames vs
> ppExpr (Suspend v) = ppKW "suspend" <+> ppName v
> ppExpr Free = ppKW "free"
> ppExpr (Ref v) = ppName v

> ppAlts :: (a -> Doc) -> [a] -> Doc
> ppAlts ppAlt = vcat . zipWith (<+>) (space : repeat bar) . map ppAlt

> ppCase :: Case -> Doc
> ppCase (Case t is) = ppCode (ppTag t <> colon) is
>   where ppTag (LitCase c) = ppLiteral c
>         ppTag (ConstrCase c vs) = ppKW "data" <+> ppName c <+> ppNames vs
>         ppTag DefaultCase = ppKW "default"

> ppAlt :: Stmt -> Doc
> ppAlt = ppCode empty

> ppKW :: String -> Doc
> ppKW kw = char '.' <> text kw

> ppName :: Name -> Doc
> ppName = text . show

> ppNames :: [Name] -> Doc
> ppNames = text . show

> bar :: Doc
> bar = char '|'

\end{lstlisting}
