/*
 *  Copyright 2001-2005 Internet2
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/*
-----------------------------------------
file	:	SAMLAuthorizationDecisionQuery.cpp
project	:	Policy Server
date	:	
author	:	andy
rights	:	(c) Parthenon Computing
notes	:	 extension to openSAML C++ library (based on Java class)
--------------------------------------------
*/

#include "internal.h"

using namespace saml;
using namespace std;

SAMLAuthorizationDecisionQuery::SAMLAuthorizationDecisionQuery(
    SAMLSubject* subject, const XMLCh* resource, const Iterator<SAMLAction*>& actions, SAMLEvidence* evidence
    ) : SAMLSubjectQuery(subject), m_resource(XML::assign(resource)), m_evidence(NULL)
{
    RTTI(SAMLAuthorizationDecisionQuery);

    while(actions.hasNext())
        m_actions.push_back(static_cast<SAMLAction*>(actions.next()->setParent(this)));

    if(evidence) {
        evidence->setParent(this);
        m_evidence = evidence;
    }
}

SAMLAuthorizationDecisionQuery::SAMLAuthorizationDecisionQuery(DOMElement* e)
    : SAMLSubjectQuery(e), m_resource(NULL), m_evidence(NULL)
{
    RTTI(SAMLAuthorizationDecisionQuery);
    fromDOM(e);
}

SAMLAuthorizationDecisionQuery::SAMLAuthorizationDecisionQuery(std::istream& in)
    : SAMLSubjectQuery(in), m_resource(NULL), m_evidence(NULL)
{
    RTTI(SAMLAuthorizationDecisionQuery);
    fromDOM(m_document->getDocumentElement());
}

SAMLAuthorizationDecisionQuery::~SAMLAuthorizationDecisionQuery()
{
    if(m_bOwnStrings)
        XMLString::release(&m_resource);
    for (vector<SAMLAction*>::const_iterator i=m_actions.begin(); i!=m_actions.end(); i++)
        delete (*i);
    delete m_evidence;
}

void SAMLAuthorizationDecisionQuery::ownStrings()
{
    if (!m_bOwnStrings) {
        m_resource=XML::assign(m_resource);
        m_bOwnStrings=true;
    }
}

void SAMLAuthorizationDecisionQuery::fromDOM(DOMElement* e)
{
    if (SAMLConfig::getConfig().strict_dom_checking) {
        if (XMLString::compareString(XML::SAMLP_NS,e->getNamespaceURI()))
            throw MalformedException(SAMLException::REQUESTER,"SAMLAuthorizationDecisionQuery::fromDOM() root element isn't in samlp namespace");
        if (XMLString::compareString(L(AuthorizationDecisionQuery),e->getLocalName())) {
            auto_ptr<saml::QName> type(saml::QName::getQNameAttribute(e,XML::XSI_NS,L(type)));
            if ((XMLString::compareString(L(Query),e->getLocalName()) && XMLString::compareString(L(SubjectQuery),e->getLocalName())) ||
                !type.get() || XMLString::compareString(XML::SAMLP_NS,type->getNamespaceURI()) ||
                XMLString::compareString(L(AuthorizationDecisionQueryType),type->getLocalName()))
                throw MalformedException(SAMLException::REQUESTER,"SAMLAuthorizationDecisionQuery::fromDOM() missing samlp:AuthorizationDecisionQuery element at root");
        }
    }

    m_resource = const_cast<XMLCh*>(e->getAttributeNS(NULL,L(Resource)));

    DOMElement* n=XML::getFirstChildElement(e);
    while(n) {
        // the top-level children are either actions (mandatory, unbounded) or evidence (optional)
        if (XML::isElementNamed(n,XML::SAML_NS,L(Action))) {
            SAMLAction* a=new SAMLAction(n);
            m_actions.push_back(static_cast<SAMLAction*>(a->setParent(this)));
        }
        else if (XML::isElementNamed(n,XML::SAML_NS,L(Evidence))) {
            m_evidence = new SAMLEvidence(n);
            m_evidence->setParent(this);
        }
        n=XML::getNextSiblingElement(n);
    } 

    checkValidity();
}

void SAMLAuthorizationDecisionQuery::setResource(const XMLCh* resource)
{
    if (XML::isEmpty(resource))
        throw SAMLException("resource cannot be null or empty");
        
    if (m_bOwnStrings)
        XMLString::release(&m_resource);
    else {
        m_resource=NULL;
        ownStrings();
    }
    m_resource=XML::assign(resource);
    setDirty();
}

void SAMLAuthorizationDecisionQuery::setActions(const Iterator<SAMLAction*>& actions)
{
    while (m_actions.size())
        removeAction(0);
    while (actions.hasNext())
        addAction(actions.next());
}

void SAMLAuthorizationDecisionQuery::addAction(SAMLAction* action)
{
    if (action) {
        action->setParent(this);
        m_actions.push_back(action);
        ownStrings();
        setDirty();
    }
    else
        throw SAMLException("action cannot be null");
}

void SAMLAuthorizationDecisionQuery::removeAction(unsigned long index)
{
    SAMLAction* kill=m_actions[index];
    m_actions.erase(m_actions.begin()+index);
    delete kill;
    ownStrings();
    setDirty();
}

void SAMLAuthorizationDecisionQuery::setEvidence(SAMLEvidence* evidence)
{
    delete m_evidence;
    m_evidence=NULL;
    if (evidence)
        m_evidence=static_cast<SAMLEvidence*>(evidence->setParent(this));
    ownStrings();
    setDirty();
}

DOMElement* SAMLAuthorizationDecisionQuery::buildRoot(DOMDocument* doc, bool xmlns) const
{
    DOMElement* q=doc->createElementNS(XML::SAMLP_NS,L(AuthorizationDecisionQuery));
    if (xmlns)
        q->setAttributeNS(XML::XMLNS_NS,L(xmlns),XML::SAMLP_NS);
    return q;
}

DOMNode* SAMLAuthorizationDecisionQuery::toDOM(DOMDocument* doc, bool xmlns) const
{
    SAMLSubjectQuery::toDOM(doc,xmlns);
    DOMElement* q=static_cast<DOMElement*>(m_root);
    doc=q->getOwnerDocument();

    if (m_bDirty) {
        q->setAttributeNS(NULL,L(Resource),m_resource);
    
        for (vector<SAMLAction*>::const_iterator i=m_actions.begin(); i!=m_actions.end(); i++)
            q->appendChild((*i)->toDOM(doc));
    
        if(m_evidence)
            q->appendChild(m_evidence->toDOM(doc));
    
        setClean();
    }
    else if (xmlns) {
        DECLARE_DEF_NAMESPACE(q,XML::SAMLP_NS);
    }

    return m_root;
}

void SAMLAuthorizationDecisionQuery::checkValidity() const
{
    SAMLSubjectQuery::checkValidity();
    if (XML::isEmpty(m_resource) || m_actions.empty())
        throw MalformedException(SAMLException::REQUESTER, "SAMLAuthorizationDecisionQuery requires a resource and at least one action");
}

SAMLObject* SAMLAuthorizationDecisionQuery::clone() const
{
    return new SAMLAuthorizationDecisionQuery(
        static_cast<SAMLSubject*>(m_subject->clone()), 
        m_resource, 
        getActions().clone(), 
        m_evidence ? static_cast<SAMLEvidence*>(m_evidence->clone()) : NULL
        );
}
