/*
 *  xfmedia - simple gtk2 media player based on xine
 *
 *  Copyright (c) 2004-2005 Brian Tarricone, <bjt23@cornell.edu>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; version 2 of the License ONLY.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#ifdef HAVE_STRING_H
#include <string.h>
#endif

#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif

#include <gdk/gdk.h>
#include <gdk/gdkx.h>
#include <gtk/gtkversion.h>
#ifdef GTK_DISABLE_DEPRECATED
#undef GTK_DISABLE_DEPRECATED
#endif
#include <gtk/gtk.h>

#include <libxfce4util/libxfce4util.h>
#include <libxfcegui4/libxfcegui4.h>

#define EXO_API_SUBJECT_TO_CHANGE
#include <exo/exo.h>

#include <xfmedia/xfmedia-settings.h>
#include "xfmedia-common.h"
#include "mainwin.h"
#include "xfmedia-keybindings.h"
#include <xfmedia/xfmedia-plugin.h>
#include "xfmedia-internal.h"

#define _NET_WM_STATE_REMOVE 0
#define _NET_WM_STATE_ADD    1
#define _NET_WM_STATE_TOGGLE 2

struct PConfData
{
    GtkWidget *xfx;
    GtkWidget *cfgwin;
    GtkTreeSelection *treesel;
    const gchar *title;
};

struct SDData
{
    GtkTooltips *ttips;
    struct PConfData *ipcdata;
    struct PConfData *opcdata;
};

enum
{
    PLUGINS_ENABLED = 0,
    PLUGINS_NAME,
    PLUGINS_PLUGIN,
    N_PLUGINS
};

static gboolean sd_running = FALSE;

static gboolean
xfmedia_xfmedia_xine_cfgedit_str_cb(GtkWidget *w, GdkEventFocus *evt,
        gpointer user_data)
{
    XfmediaMainwin *mwin = user_data;
    const gchar *key = g_object_get_data(G_OBJECT(w), "xfmedia-xinecfg-key");
    gchar *value;
    xine_cfg_entry_t cfg;
    
    g_return_val_if_fail(key && *key, FALSE);
    
    if(!xfmedia_xine_config_lookup_entry(mwin->xfx, key, &cfg))
        return FALSE;
    
    value = gtk_editable_get_chars(GTK_EDITABLE(w), 0, -1);
    if(value) {
        if(strcmp(value, cfg.str_value)) {
            cfg.str_value = value;
            xfmedia_xine_config_update_entry(mwin->xfx, &cfg);
        }
        g_free(value);
    }
    
    return FALSE;
}

static void
xfmedia_xfmedia_xine_cfgedit_enum_cb(GtkComboBox *combo, gpointer user_data)
{
    XfmediaMainwin *mwin = user_data;
    const gchar *key = g_object_get_data(G_OBJECT(combo), "xfmedia-xinecfg-key");
    gint value;
    xine_cfg_entry_t cfg;
    
    g_return_if_fail(key && *key);
    
    if(!xfmedia_xine_config_lookup_entry(mwin->xfx, key, &cfg))
        return;
    
    value = gtk_combo_box_get_active(combo);
    if(value != cfg.num_value) {
        cfg.num_value = value;
        xfmedia_xine_config_update_entry(mwin->xfx, &cfg);
        
        if(!strcmp(key, "audio.driver"))
            xfmedia_xine_set_audio_output_plugin(mwin->xfx, cfg.enum_values[value]);
        else if(!strcmp(key, "video.driver"))
            xfmedia_xine_set_video_output_plugin(mwin->xfx, cfg.enum_values[value]);
    }
}

static GtkWidget *
xfmedia_settings_create_devices_tab(XfmediaMainwin *mwin, GtkWidget *parent)
{
    GtkWidget *topvbox, *frame, *frame_bin, *vbox, *hbox, *lbl, *entry, *combo;
    GtkSizeGroup *sg;
    xine_cfg_entry_t cfg;
    GList *plugins, *l;
    
    topvbox = gtk_vbox_new(FALSE, BORDER/2);
    gtk_container_set_border_width(GTK_CONTAINER(topvbox), BORDER/2);
    gtk_widget_show(topvbox);
    
    frame = xfmedia_create_framebox(_("Optical Devices"), &frame_bin);
    gtk_widget_show(frame);
    gtk_box_pack_start(GTK_BOX(topvbox), frame, FALSE, FALSE, 0);
    
    vbox = gtk_vbox_new(FALSE, BORDER/2);
    gtk_widget_show(vbox);
    gtk_container_add(GTK_CONTAINER(frame_bin), vbox);
    
    sg = gtk_size_group_new(GTK_SIZE_GROUP_HORIZONTAL);
    
    hbox = gtk_hbox_new(FALSE, BORDER/2);
    gtk_widget_show(hbox);
    gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
    
    lbl = gtk_label_new_with_mnemonic(_("_CD Audio Device:"));
    gtk_misc_set_alignment(GTK_MISC(lbl), 0.0, 0.5);
    gtk_size_group_add_widget(sg, lbl);
    gtk_widget_show(lbl);
    gtk_box_pack_start(GTK_BOX(hbox), lbl, FALSE, FALSE, 0);
    
    entry = gtk_entry_new();
    if(xfmedia_xine_config_lookup_entry(mwin->xfx, "media.audio_cd.device", &cfg)) {
        gtk_entry_set_text(GTK_ENTRY(entry), cfg.str_value);
        g_object_set_data(G_OBJECT(entry), "xfmedia-xinecfg-key",
                "media.audio_cd.device");
        g_signal_connect(G_OBJECT(entry), "focus-out-event",
                G_CALLBACK(xfmedia_xfmedia_xine_cfgedit_str_cb), mwin);
    }
    gtk_widget_show(entry);
    gtk_box_pack_start(GTK_BOX(hbox), entry, TRUE, TRUE, 0);
    gtk_label_set_mnemonic_widget(GTK_LABEL(lbl), entry);
    
    hbox = gtk_hbox_new(FALSE, BORDER/2);
    gtk_widget_show(hbox);
    gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
    
    lbl = gtk_label_new_with_mnemonic(_("_DVD Video Device:"));
    gtk_misc_set_alignment(GTK_MISC(lbl), 0.0, 0.5);
    gtk_size_group_add_widget(sg, lbl);
    gtk_widget_show(lbl);
    gtk_box_pack_start(GTK_BOX(hbox), lbl, FALSE, FALSE, 0);
    
    entry = gtk_entry_new();
    if(xfmedia_xine_config_lookup_entry(mwin->xfx, "media.dvd.device", &cfg)) {
        gtk_entry_set_text(GTK_ENTRY(entry), cfg.str_value);
        g_object_set_data(G_OBJECT(entry), "xfmedia-xinecfg-key",
                "media.dvd.device");
        g_signal_connect(G_OBJECT(entry), "focus-out-event",
                G_CALLBACK(xfmedia_xfmedia_xine_cfgedit_str_cb), mwin);
    }
    gtk_widget_show(entry);
    gtk_box_pack_start(GTK_BOX(hbox), entry, TRUE, TRUE, 0);
    gtk_label_set_mnemonic_widget(GTK_LABEL(lbl), entry);
    
    hbox = gtk_hbox_new(FALSE, BORDER/2);
    gtk_widget_show(hbox);
    gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
    
    lbl = gtk_label_new_with_mnemonic(_("_Video CD Device:"));
    gtk_misc_set_alignment(GTK_MISC(lbl), 0.0, 0.5);
    gtk_size_group_add_widget(sg, lbl);
    gtk_widget_show(lbl);
    gtk_box_pack_start(GTK_BOX(hbox), lbl, FALSE, FALSE, 0);
    
    entry = gtk_entry_new();
    if(xfmedia_xine_config_lookup_entry(mwin->xfx, "media.vcd.device", &cfg)) {
        gtk_entry_set_text(GTK_ENTRY(entry), cfg.str_value);
        g_object_set_data(G_OBJECT(entry), "xfmedia-xinecfg-key",
                "media.vcd.device");
        g_signal_connect(G_OBJECT(entry), "focus-out-event",
                G_CALLBACK(xfmedia_xfmedia_xine_cfgedit_str_cb), mwin);
    }
    gtk_widget_show(entry);
    gtk_box_pack_start(GTK_BOX(hbox), entry, TRUE, TRUE, 0);
    gtk_label_set_mnemonic_widget(GTK_LABEL(lbl), entry);
    
    frame = xfmedia_create_framebox(_("Audio"), &frame_bin);
    gtk_widget_show(frame);
    gtk_box_pack_start(GTK_BOX(topvbox), frame, FALSE, FALSE, 0);
    
    vbox = gtk_vbox_new(FALSE, BORDER/2);
    gtk_widget_show(vbox);
    gtk_container_add(GTK_CONTAINER(frame_bin), vbox);
    
    hbox = gtk_hbox_new(FALSE, BORDER/2);
    gtk_widget_show(hbox);
    gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
    
    lbl = gtk_label_new_with_mnemonic(_("_Driver:"));
    gtk_misc_set_alignment(GTK_MISC(lbl), 0.0, 0.5);
    gtk_size_group_add_widget(sg, lbl);
    gtk_widget_show(lbl);
    gtk_box_pack_start(GTK_BOX(hbox), lbl, FALSE, FALSE, 0);
    
    combo = gtk_combo_box_new_text();
    plugins = xfmedia_xine_list_audio_output_plugins(mwin->xfx);
    xfmedia_xine_config_lookup_entry(mwin->xfx, "audio.driver", &cfg);
    if(!plugins) {
        gtk_combo_box_append_text(GTK_COMBO_BOX(combo), _("(no drivers)"));
        gtk_combo_box_set_active(GTK_COMBO_BOX(combo), 0);
    } else {
        gint i, selected = -1;
        for(l = plugins, i = 0; l; l = l->next, i++) {
            gtk_combo_box_append_text(GTK_COMBO_BOX(combo), l->data);
            if(!strcmp(l->data, cfg.enum_values[cfg.num_value]))
                selected = i;
        }
        if(selected != -1)
            gtk_combo_box_set_active(GTK_COMBO_BOX(combo), selected);
        g_object_set_data(G_OBJECT(combo), "xfmedia-xinecfg-key",
                "audio.driver");
        g_signal_connect(G_OBJECT(combo), "changed",
                G_CALLBACK(xfmedia_xfmedia_xine_cfgedit_enum_cb), mwin);
    }
    gtk_widget_show(combo);
    gtk_box_pack_start(GTK_BOX(hbox), combo, TRUE, TRUE, 0);
    gtk_label_set_mnemonic_widget(GTK_LABEL(lbl), combo);
    
    hbox = gtk_hbox_new(FALSE, BORDER/2);
    gtk_widget_show(hbox);
    gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
    
    lbl = gtk_label_new_with_mnemonic(_("_Speaker Layout:"));
    gtk_misc_set_alignment(GTK_MISC(lbl), 0.0, 0.5);
    gtk_size_group_add_widget(sg, lbl);
    gtk_widget_show(lbl);
    gtk_box_pack_start(GTK_BOX(hbox), lbl, FALSE, FALSE, 0);
    
    combo = gtk_combo_box_new_text();
    if(!xfmedia_xine_config_lookup_entry(mwin->xfx, "audio.output.speaker_arrangement", &cfg))
        gtk_combo_box_append_text(GTK_COMBO_BOX(combo), _("(option unavailable)"));
    else {
        gint i;
        for(i = 0; cfg.enum_values[i]; i++)
            gtk_combo_box_append_text(GTK_COMBO_BOX(combo), cfg.enum_values[i]);
        gtk_combo_box_set_active(GTK_COMBO_BOX(combo), cfg.num_value);
        g_object_set_data(G_OBJECT(combo), "xfmedia-xinecfg-key",
                "audio.output.speaker_arrangement");
        g_signal_connect(G_OBJECT(combo), "changed",
                G_CALLBACK(xfmedia_xfmedia_xine_cfgedit_enum_cb), mwin);
    }
    gtk_widget_show(combo);
    gtk_box_pack_start(GTK_BOX(hbox), combo, TRUE, TRUE, 0);
    gtk_label_set_mnemonic_widget(GTK_LABEL(lbl), combo);
    
    frame = xfmedia_create_framebox(_("Video"), &frame_bin);
    gtk_widget_show(frame);
    gtk_box_pack_start(GTK_BOX(topvbox), frame, FALSE, FALSE, 0);
    
    vbox = gtk_vbox_new(FALSE, BORDER/2);
    gtk_widget_show(vbox);
    gtk_container_add(GTK_CONTAINER(frame_bin), vbox);
    
    hbox = gtk_hbox_new(FALSE, BORDER/2);
    gtk_widget_show(hbox);
    gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
    
    lbl = gtk_label_new_with_mnemonic(_("D_river:"));
    gtk_misc_set_alignment(GTK_MISC(lbl), 0.0, 0.5);
    gtk_size_group_add_widget(sg, lbl);
    gtk_widget_show(lbl);
    gtk_box_pack_start(GTK_BOX(hbox), lbl, FALSE, FALSE, 0);
    
    combo = gtk_combo_box_new_text();
    plugins = xfmedia_xine_list_video_output_plugins(mwin->xfx);
    xfmedia_xine_config_lookup_entry(mwin->xfx, "video.driver", &cfg);
    if(!plugins) {
        gtk_combo_box_append_text(GTK_COMBO_BOX(combo), _("(no drivers)"));
        gtk_combo_box_set_active(GTK_COMBO_BOX(combo), 0);
    } else {
        gint i, selected = -1;
        for(l = plugins, i = 0; l; l = l->next, i++) {
            gtk_combo_box_append_text(GTK_COMBO_BOX(combo), l->data);
            if(!strcmp(l->data, cfg.enum_values[cfg.num_value]))
                selected = i;
        }
        if(selected != -1)
            gtk_combo_box_set_active(GTK_COMBO_BOX(combo), selected);
        g_object_set_data(G_OBJECT(combo), "xfmedia-xinecfg-key",
                "video.driver");
        g_signal_connect(G_OBJECT(combo), "changed",
                G_CALLBACK(xfmedia_xfmedia_xine_cfgedit_enum_cb), mwin);
    }
    gtk_widget_show(combo);
    gtk_box_pack_start(GTK_BOX(hbox), combo, TRUE, TRUE, 0);
    gtk_label_set_mnemonic_widget(GTK_LABEL(lbl), combo);
    
    return topvbox;
}

static gboolean
titledisp_cb(GtkWidget *w, GdkEventFocus *evt, gpointer user_data)
{
    XfmediaMainwin *mwin = user_data;
    gchar *fmt;
    gint i, tot;
    
    fmt = gtk_editable_get_chars(GTK_EDITABLE(w), 0, -1);
    xfmedia_settings_set_string("/xfmedia/playlist/playlist_format", fmt);
    g_free(fmt);
    
    /* invalidate all the playlist info fields (expensive?) */
    tot = xfmedia_playlist_get_n_entries(mwin->plist);
    for(i = 0; i < tot; i++)
        xfmedia_playlist_set_metadata_loaded(mwin->plist, i, FALSE);
    
    return FALSE;
}

static void
autoadvance_cb(GtkToggleButton *tb, gpointer user_data)
{
    xfmedia_settings_set_bool("/xfmedia/playlist/auto_advance",
            gtk_toggle_button_get_active(tb));
}

static void
allws_cb(GtkToggleButton *tb, gpointer user_data)
{
    XfmediaMainwin *mwin = user_data;
    gboolean set;
    
    set = gtk_toggle_button_get_active(tb);
    if(set)
        gtk_window_stick(GTK_WINDOW(mwin->window));
    else
        gtk_window_unstick(GTK_WINDOW(mwin->window));
    
    xfmedia_settings_set_bool("/xfmedia/general/window_sticky", set);
}

static void
alwaysontop_cb(GtkToggleButton *tb, gpointer user_data)
{
    XfmediaMainwin *mwin = user_data;
    gboolean set;
    
    set = gtk_toggle_button_get_active(tb);
    gtk_window_set_keep_above(GTK_WINDOW(mwin->window), set);
    xfmedia_settings_set_bool("/xfmedia/general/window_above", set);
}

static void
trayicon_cb(GtkToggleButton *tb, gpointer user_data)
{
    XfmediaMainwin *mwin = user_data;
    gboolean set;
    
    set = gtk_toggle_button_get_active(tb);
    
    gtk_widget_set_sensitive(mwin->tray_mini_chk, set);
    gtk_widget_set_sensitive(mwin->tray_middle_box, set);
    
    if(set) {
        xfmedia_mainwin_activate_systray(mwin);
        if(mwin->is_iconified
                && xfmedia_settings_get_bool("/xfmedia/general/minimize_to_tray"))
        {
            gtk_widget_hide(mwin->window);
        }
    } else {
        gtk_window_set_skip_taskbar_hint(GTK_WINDOW(mwin->window), FALSE);
        gtk_widget_show(mwin->window);
        xfmedia_mainwin_deactivate_systray(mwin);
    }
    xfmedia_settings_set_bool("/xfmedia/general/show_tray_icon", set);
}

static void
trayiconmini_cb(GtkToggleButton *tb, gpointer user_data)
{
    XfmediaMainwin *mwin = user_data;
    gboolean set;
    
    set = gtk_toggle_button_get_active(tb);
    if(!set)
        gtk_window_set_skip_taskbar_hint(GTK_WINDOW(mwin->window), FALSE);
    else if(set && mwin->is_iconified) {
        gtk_window_set_skip_pager_hint(GTK_WINDOW(mwin->window), TRUE);
        gtk_widget_hide(mwin->window);
    }
    xfmedia_settings_set_bool("/xfmedia/general/minimize_to_tray", set);
}

static void
tray_middle_combo_changed_cb(GtkComboBox *combo, gpointer user_data)
{
    xfmedia_settings_set_int("/xfmedia/general/tray_middle_click_action",
            gtk_combo_box_get_active(combo));
}

static void
sd_destroy(struct SDData *sddata)
{
    gtk_object_sink(GTK_OBJECT(sddata->ttips));
    if(sddata->ipcdata)
        g_free(sddata->ipcdata);
    if(sddata->opcdata)
        g_free(sddata->opcdata);
    g_free(sddata);
    sd_running = FALSE;
}

static void
soea_toggle_cb(GtkToggleButton *tb, gpointer user_data)
{
    xfmedia_settings_set_bool("/xfmedia/playlist/save_on_exit_always",
            !gtk_toggle_button_get_active(tb));
}

static void
dock_autohide_cb(GtkToggleButton *tb, gpointer user_data)
{
    XfmediaMainwin *mwin = user_data;
    gboolean set = gtk_toggle_button_get_active(tb);
    
    xfmedia_settings_set_bool("/xfmedia/general/dock_autohide", set);
    xfmedia_mainwin_set_autohide(mwin, set);
}

static void
metadata_load_change_cb(GtkToggleButton *tb, gpointer user_data)
{
    guint setting = GPOINTER_TO_UINT(user_data);
    
    if(gtk_toggle_button_get_active(tb))
        xfmedia_settings_set_int("/xfmedia/playlist/metadata_load_impetus", setting);
}

static void
audio_vis_changed_cb(GtkMenuItem *mi, gpointer user_data)
{
    XfmediaMainwin *mwin = user_data;
    GtkLabel *lbl = GTK_LABEL(gtk_bin_get_child(GTK_BIN(mi)));
    const gchar *vis_name = gtk_label_get_text(lbl);
    
    xfmedia_settings_set_string("/xfmedia/general/selected_vis", vis_name);
    
    if(xfmedia_settings_get_bool("/xfmedia/general/show_audio_vis")
            && xfmedia_xine_get_status(mwin->xfx) == XINE_STATUS_PLAY
            && !xfmedia_xine_get_stream_info(mwin->xfx, XINE_STREAM_INFO_HAS_VIDEO))
    {
        xfmedia_xine_set_vis(mwin->xfx, vis_name);
    }
}

#if 0

static gint
plugins_sort_alpha(gconstpointer a, gconstpointer b)
{
    XfmediaPlugin *pa = (XfmediaPlugin *)a, *pb = (XfmediaPlugin *)b;
    
    return g_ascii_strcasecmp(xfmedia_plugin_get_name(pa), xfmedia_plugin_get_name(pb));
}

static void
plugin_enabled_toggle_cb(GtkCellRendererToggle *crt, gchar *arg1,
        gpointer user_data)
{
    GtkTreeIter itr;
    
    
}

static GtkWidget *
add_plugins_tab(XfmediaMainwin *mwin)
{
    GtkWidget *topvbox, *hbox, *sw, *treeview, *btn;
    GtkListStore *ls;
    //GtkTreeSelection *sel;
    GtkTreeViewColumn *col;
    GtkCellRenderer *render;
    GList *plugins, *l;
    GtkTreeIter itr;
    
    topvbox = gtk_vbox_new(FALSE, BORDER);
    gtk_container_set_border_width(GTK_CONTAINER(topvbox), BORDER/2);
    gtk_widget_show(topvbox);
    
    sw = gtk_scrolled_window_new(NULL, NULL);
    gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(sw), GTK_POLICY_NEVER,
            GTK_POLICY_AUTOMATIC);
    gtk_scrolled_window_set_shadow_type(GTK_SCROLLED_WINDOW(sw), GTK_SHADOW_ETCHED_IN);
    gtk_widget_show(sw);
    gtk_box_pack_start(GTK_BOX(topvbox), sw, TRUE, TRUE, 0);
    
    ls = gtk_list_store_new(N_PLUGINS, G_TYPE_BOOLEAN, G_TYPE_STRING, G_TYPE_POINTER);
    plugins = xfmedia_plugins_enumerate();
    if(plugins)
        g_list_sort(plugins, plugins_sort_alpha);
    for(l = plugins; l; l = l->next) {
        XfmediaPlugin *plugin = l->data;
        
        gtk_list_store_append(ls, &itr);
        gtk_list_store_set(ls, &itr,
                PLUGINS_ENABLED, xfmedia_plugin_get_enabled(plugin),
                PLUGINS_NAME, xfmedia_plugin_get_name(plugin),
                PLUGINS_PLUGIN, plugin,
                -1);
    }
    
    treeview = gtk_tree_view_new_with_model(GTK_TREE_MODEL(ls));
    gtk_tree_view_set_headers_visible(GTK_TREE_VIEW(treeview), FALSE);
    
    render = gtk_cell_renderer_toggle_new();
    col = gtk_tree_view_column_new_with_attributes("enabled", render,
            "active", PLUGINS_ENABLED, NULL);
    gtk_tree_view_append_column(GTK_TREE_VIEW(treeview), col);
    g_signal_connect(G_OBJECT(render), "toggled",
            G_CALLBACK(plugin_enabled_toggle_cb), mwin);
    
    render = exo_cell_renderer_ellipsized_text_new();
    col = gtk_tree_view_column_new_with_attributes("name", render,
            "text", PLUGINS_NAME, NULL);
    g_object_set(G_OBJECT(render), "ellipsize", EXO_PANGO_ELLIPSIZE_END,
            "ellipsize-set", TRUE, NULL);
    gtk_tree_view_append_column(GTK_TREE_VIEW(treeview), col);
    
    gtk_widget_show(treeview);
    gtk_container_add(GTK_CONTAINER(sw), treeview);
    
    hbox = gtk_hbox_new(FALSE, BORDER);
    gtk_widget_show(hbox);
    gtk_box_pack_start(GTK_BOX(topvbox), hbox, FALSE, FALSE, 0);
    
    btn = gtk_button_new_from_stock(GTK_STOCK_PROPERTIES);
    gtk_widget_show(btn);
    gtk_box_pack_start(GTK_BOX(hbox), btn, FALSE, FALSE, 0);
    
    btn = gtk_button_new_from_stock(GTK_STOCK_ABOUT);
    gtk_widget_show(btn);
    gtk_box_pack_start(GTK_BOX(hbox), btn, FALSE, FALSE, 0);
    
    return topvbox;
}

#endif

void
xfmedia_settings_show_dialog(XfmediaMainwin *mwin)
{
    static GtkWidget *win = NULL;
    GtkWidget *topvbox, *vbox, *hbox, *btn, *frame, *frame_bin, *entry, *lbl,
              *combo, *notebook, *menu, *mi, *cb;
    GtkTooltips *ttips;
    struct SDData *sddata;
    GList *vis_plugins;
    gint selected_vis_index = 0;
    //GtkTreeView *treeview;
    //GtkTreeSelection *treesel;
    
    if(sd_running && win) {
        gtk_window_present(GTK_WINDOW(win));
        return;
    }
    
    sddata = g_new0(struct SDData, 1);
    sddata->ipcdata = g_new(struct PConfData, 1);
    sddata->opcdata = g_new(struct PConfData, 1);
    
    sddata->ttips = ttips = gtk_tooltips_new();
    
    win = gtk_dialog_new_with_buttons(_("Xfmedia Preferences"),
            GTK_WINDOW(mwin->window),
            GTK_DIALOG_DESTROY_WITH_PARENT|GTK_DIALOG_NO_SEPARATOR,
            GTK_STOCK_CLOSE, GTK_RESPONSE_CLOSE, NULL);
    gtk_dialog_set_default_response(GTK_DIALOG(win), GTK_RESPONSE_CLOSE);
    gtk_container_set_border_width(GTK_CONTAINER(win), BORDER);
    g_signal_connect_swapped(G_OBJECT(win), "destroy",
            G_CALLBACK(sd_destroy), sddata);
    g_signal_connect(G_OBJECT(win), "response",
            G_CALLBACK(gtk_widget_destroy), NULL);
    
    notebook = gtk_notebook_new();
    gtk_widget_show(notebook);
    gtk_container_add(GTK_CONTAINER(GTK_DIALOG(win)->vbox), notebook);
    
    
    /* options tab */
    
    topvbox = gtk_vbox_new(FALSE, BORDER);
    gtk_container_set_border_width(GTK_CONTAINER(topvbox), BORDER/2);
    gtk_widget_show(topvbox);
    
    frame = xfmedia_create_framebox(_("Playlist"), &frame_bin);
    gtk_widget_show(frame);
    gtk_box_pack_start(GTK_BOX(topvbox), frame, FALSE, FALSE, 0);
    
    vbox = gtk_vbox_new(FALSE, BORDER/2);
    gtk_container_set_border_width(GTK_CONTAINER(vbox), BORDER/2);
    gtk_widget_show(vbox);
    gtk_container_add(GTK_CONTAINER(frame_bin), vbox);
    
    hbox = gtk_hbox_new(FALSE, BORDER);
    gtk_widget_show(hbox);
    gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
    
    lbl = gtk_label_new_with_mnemonic(_("_Title Display:"));
    gtk_widget_show(lbl);
    gtk_box_pack_start(GTK_BOX(hbox), lbl, FALSE, FALSE, 0);
    
    entry = gtk_entry_new();
    gtk_entry_set_text(GTK_ENTRY(entry),
            xfmedia_settings_get_string("/xfmedia/playlist/playlist_format"));
    gtk_widget_show(entry);
    gtk_box_pack_start(GTK_BOX(hbox), entry, TRUE, TRUE, 0);
    gtk_tooltips_set_tip(ttips, entry,
            _("This is a printf-like format string controlling the display of" \
              " the playlist.  Xfmedia will replace the following symbols:\n" \
              "\t%p: Artist/Performer\n" \
              "\t%t: Track Title\n" \
              "\t%a: Album Name\n" \
              "\t%T: Track Number\n" \
              "\t%g: Genre\n" \
              "\t%y: Year\n" \
              "\t%f: Filename\n" \
              "\t%%: a percent sign"), NULL);
    g_signal_connect(G_OBJECT(entry), "focus-out-event",
            G_CALLBACK(titledisp_cb), mwin);
    gtk_label_set_mnemonic_widget(GTK_LABEL(lbl), entry);
    
    btn = gtk_check_button_new_with_mnemonic(_("_Advance playlist when track ends"));
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(btn),
            xfmedia_settings_get_bool("/xfmedia/playlist/auto_advance"));
    gtk_widget_show(btn);
    gtk_box_pack_start(GTK_BOX(vbox), btn, FALSE, FALSE, 0);
    g_signal_connect(G_OBJECT(btn), "toggled",
            G_CALLBACK(autoadvance_cb), mwin);
    gtk_tooltips_set_tip(ttips, btn,
            _("When a track ends, Xfmedia can automatically advance to the next track in the playlist and continue playback."), NULL);            
    
    btn = gtk_check_button_new_with_mnemonic(_("_Don't save playlist on exit for one-shot invocations"));
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(btn),
            !xfmedia_settings_get_bool("/xfmedia/playlist/save_on_exit_always"));
    gtk_widget_show(btn);
    gtk_box_pack_start(GTK_BOX(vbox), btn, FALSE, FALSE, 0);
    g_signal_connect(G_OBJECT(btn), "toggled",
            G_CALLBACK(soea_toggle_cb), mwin);
    gtk_tooltips_set_tip(ttips, btn,
            _("Only automatically save the playlist during interactive use; i.e., don't save the playlist when Xfmedia was invoked from the commandline with a short list of definite arguments.  This option needs to be named better."), NULL);
    
    hbox = gtk_hbox_new(FALSE, BORDER);
    gtk_widget_show(hbox);
    gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
    
    lbl = gtk_label_new(_("Load metadata:"));
    gtk_widget_show(lbl);
    gtk_box_pack_start(GTK_BOX(hbox), lbl, FALSE, FALSE, 0);
    
    btn = gtk_radio_button_new_with_mnemonic(NULL, _("On _scroll"));
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(btn),
            (xfmedia_settings_get_int("/xfmedia/playlist/metadata_load_impetus") == 0));
    gtk_widget_show(btn);
    gtk_box_pack_start(GTK_BOX(hbox), btn, FALSE, FALSE, 0);
    g_signal_connect(G_OBJECT(btn), "toggled",
            G_CALLBACK(metadata_load_change_cb), GUINT_TO_POINTER(0));
    gtk_tooltips_set_tip(ttips, btn,
            _("Load file metadata for entries in the playlist when they are visible."), NULL);
    
    btn = gtk_radio_button_new_with_mnemonic_from_widget(GTK_RADIO_BUTTON(btn),
            _("On pla_y"));
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(btn),
            (xfmedia_settings_get_int("/xfmedia/playlist/metadata_load_impetus") == 1));
    gtk_widget_show(btn);
    gtk_box_pack_start(GTK_BOX(hbox), btn, FALSE, FALSE, 0);
    g_signal_connect(G_OBJECT(btn), "toggled",
            G_CALLBACK(metadata_load_change_cb), GUINT_TO_POINTER(1));
    gtk_tooltips_set_tip(ttips, btn,
            _("Load file metadata for entries in the playlist only when playing them."), NULL);
    
    btn = gtk_radio_button_new_with_mnemonic_from_widget(GTK_RADIO_BUTTON(btn),
            _("_Never"));
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(btn),
            (xfmedia_settings_get_int("/xfmedia/playlist/metadata_load_impetus") == 2));
    gtk_widget_show(btn);
    gtk_box_pack_start(GTK_BOX(hbox), btn, FALSE, FALSE, 0);
    g_signal_connect(G_OBJECT(btn), "toggled",
            G_CALLBACK(metadata_load_change_cb), GUINT_TO_POINTER(2));
    gtk_tooltips_set_tip(ttips, btn,
            _("Never load file metadata for entries in the playlist."), NULL);
    
    frame = xfmedia_create_framebox(_("Window"), &frame_bin);
    gtk_widget_show(frame);
    gtk_box_pack_start(GTK_BOX(topvbox), frame, FALSE, FALSE, 0);
    
    vbox = gtk_vbox_new(FALSE, BORDER/2);
    gtk_container_set_border_width(GTK_CONTAINER(vbox), BORDER/2);
    gtk_widget_show(vbox);
    gtk_container_add(GTK_CONTAINER(frame_bin), vbox);
    
    btn = gtk_check_button_new_with_mnemonic(_("Show Xfmedia on all _workspaces"));
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(btn),
            xfmedia_settings_get_bool("/xfmedia/general/window_sticky"));
    gtk_widget_show(btn);
    gtk_box_pack_start(GTK_BOX(vbox), btn, FALSE, FALSE, 0);
    g_signal_connect(G_OBJECT(btn), "toggled", G_CALLBACK(allws_cb), mwin);
    gtk_tooltips_set_tip(ttips, btn, _("Sets Xfmedia \"sticky\" such that it appears on all workspaces (virtual desktops)."), NULL);
    
    btn = gtk_check_button_new_with_mnemonic(_("_Keep Xfmedia above other windows"));
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(btn),
            xfmedia_settings_get_bool("/xfmedia/general/window_above"));
    gtk_widget_show(btn);
    gtk_box_pack_start(GTK_BOX(vbox), btn, FALSE, FALSE, 0);
    g_signal_connect(G_OBJECT(btn), "toggled", G_CALLBACK(alwaysontop_cb), mwin);
    gtk_tooltips_set_tip(ttips, btn,
            _("Sets the Xfmedia window to \"Always-on-Top\" status."), NULL);
            
    btn = gtk_check_button_new_with_mnemonic(_("_Dock and auto-hide Xfmedia at screen edges"));
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(btn),
            xfmedia_settings_get_bool("/xfmedia/general/dock_autohide"));
    gtk_widget_show(btn);
    gtk_box_pack_start(GTK_BOX(vbox), btn, FALSE, FALSE, 0);
    g_signal_connect(G_OBJECT(btn), "toggled", G_CALLBACK(dock_autohide_cb), mwin);
    gtk_tooltips_set_tip(ttips, btn,
            _("When the Xfmedia window is dragged to a screen edge, it will auto-hide itself and reappear when you move the mouse to the edge of the screen."), NULL);
    
    frame = xfmedia_create_framebox(_("Tray Icon"), &frame_bin);
    gtk_widget_show(frame);
    gtk_box_pack_start(GTK_BOX(topvbox), frame, FALSE, FALSE, 0);
    
    vbox = gtk_vbox_new(FALSE, BORDER/2);
    gtk_container_set_border_width(GTK_CONTAINER(vbox), BORDER/2);
    gtk_widget_show(vbox);
    gtk_container_add(GTK_CONTAINER(frame_bin), vbox);
    
    btn = gtk_check_button_new_with_mnemonic(_("Show _tray icon"));
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(btn),
            xfmedia_settings_get_bool("/xfmedia/general/show_tray_icon"));
    gtk_widget_show(btn);
    gtk_box_pack_start(GTK_BOX(vbox), btn, FALSE, FALSE, 0);
    g_signal_connect(G_OBJECT(btn), "toggled", G_CALLBACK(trayicon_cb), mwin);
    gtk_tooltips_set_tip(ttips, btn, _("Displays an Xfmedia icon in the system notification area."), NULL);
    
    mwin->tray_mini_chk = btn = gtk_check_button_new_with_mnemonic(_("_Minimize Xfmedia to tray icon"));
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(btn),
            xfmedia_settings_get_bool("/xfmedia/general/minimize_to_tray"));
    gtk_widget_set_sensitive(btn,
            xfmedia_settings_get_bool("/xfmedia/general/show_tray_icon"));
    gtk_widget_show(btn);
    gtk_box_pack_start(GTK_BOX(vbox), btn, FALSE, FALSE, 0);
    g_signal_connect(G_OBJECT(btn), "toggled", G_CALLBACK(trayiconmini_cb), mwin);
    gtk_tooltips_set_tip(ttips, btn, _("Minimizes Xfmedia to the tray icon rather than to the taskbar or iconbox."), NULL);
    
    mwin->tray_middle_box = hbox = gtk_hbox_new(FALSE, BORDER/2);
    gtk_widget_show(hbox);
    gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
    
    lbl = gtk_label_new_with_mnemonic(_("On middle _click:"));
    gtk_widget_show(lbl);
    gtk_box_pack_start(GTK_BOX(hbox), lbl, FALSE, FALSE, 0);
    gtk_tooltips_set_tip(ttips, lbl, _("The tray icon can perform an action when middle-clicked.  Select the action in the dropdown box."), NULL);
    
    combo = gtk_combo_box_new_text();
    gtk_combo_box_append_text(GTK_COMBO_BOX(combo), _("Do nothing"));
    gtk_combo_box_append_text(GTK_COMBO_BOX(combo), _("Toggle play/pause"));
    gtk_combo_box_append_text(GTK_COMBO_BOX(combo), _("Next track"));
    gtk_combo_box_append_text(GTK_COMBO_BOX(combo), _("Previous track"));
    gtk_combo_box_set_active(GTK_COMBO_BOX(combo),
            xfmedia_settings_get_int("/xfmedia/general/tray_middle_click_action"));
    gtk_widget_show(combo);
    gtk_box_pack_start(GTK_BOX(hbox), combo, FALSE, FALSE, 0);
    g_signal_connect(G_OBJECT(combo), "changed",
            G_CALLBACK(tray_middle_combo_changed_cb), mwin);
    gtk_label_set_mnemonic_widget(GTK_LABEL(lbl), combo);
    gtk_tooltips_set_tip(ttips, combo, _("The tray icon can perform an action when middle-clicked.  Select the action in the dropdown box."), NULL);
    
    gtk_widget_set_sensitive(hbox,
            xfmedia_settings_get_bool("/xfmedia/general/show_tray_icon"));
    
    lbl = gtk_label_new_with_mnemonic(_("O_ptions"));
    gtk_widget_show(lbl);
    gtk_notebook_append_page(GTK_NOTEBOOK(notebook), topvbox, lbl);
    
    
    /* vis settings tab */
    
    topvbox = gtk_vbox_new(FALSE, BORDER);
    gtk_container_set_border_width(GTK_CONTAINER(topvbox), BORDER/2);
    gtk_widget_show(topvbox);
    
    frame = xfmedia_create_framebox(_("Selected Visualization"), &frame_bin);
    gtk_widget_show(frame);
    gtk_box_pack_start(GTK_BOX(topvbox), frame, FALSE, FALSE, 0);
    
    hbox = gtk_hbox_new(FALSE, BORDER/2);
    gtk_widget_show(hbox);
    gtk_container_add(GTK_CONTAINER(frame_bin), hbox);
    
    menu = gtk_menu_new();
    gtk_widget_show(menu);
    
    vis_plugins = xfmedia_xine_list_post_plugins_typed(mwin->xfx,
            XINE_POST_TYPE_AUDIO_VISUALIZATION);
    if(vis_plugins) {
        GList *l;
        const gchar *selected_vis =
                xfmedia_settings_get_string("/xfmedia/general/selected_vis");
        for(l = vis_plugins; l; l = l->next) {
            if(!g_ascii_strcasecmp((const gchar *)l->data, selected_vis))
                selected_vis_index = g_list_index(vis_plugins, l->data);
            mi = gtk_menu_item_new_with_label((const gchar *)l->data);
            gtk_widget_show(mi);
            gtk_menu_shell_append(GTK_MENU_SHELL(menu), mi);
            g_signal_connect(G_OBJECT(mi), "activate",
                    G_CALLBACK(audio_vis_changed_cb), mwin);
        }
        g_list_free(vis_plugins);
    } else {
        mi = gtk_menu_item_new_with_label(_("(none available)"));
        gtk_widget_show(mi);
        gtk_menu_shell_append(GTK_MENU_SHELL(menu), mi);
    }
    
    cb = gtk_option_menu_new();
    gtk_option_menu_set_menu(GTK_OPTION_MENU(cb), menu);
    gtk_option_menu_set_history(GTK_OPTION_MENU(cb), selected_vis_index);
    gtk_widget_show(cb);
    gtk_box_pack_start(GTK_BOX(hbox), cb, TRUE, TRUE, 0);
    
    frame = xfmedia_create_framebox(_("Visualization Options"), &frame_bin);
    gtk_widget_show(frame);
    gtk_box_pack_start(GTK_BOX(topvbox), frame, FALSE, FALSE, 0);
    
    vbox = gtk_vbox_new(FALSE, BORDER/2);
    gtk_container_set_border_width(GTK_CONTAINER(vbox), BORDER/2);
    gtk_widget_show(vbox);
    gtk_container_add(GTK_CONTAINER(frame_bin), vbox);
    
    lbl = gtk_label_new("(not yet implemented)");
    gtk_misc_set_alignment(GTK_MISC(lbl), 0.0, 0.5);
    gtk_widget_show(lbl);
    gtk_box_pack_start(GTK_BOX(vbox), lbl, FALSE, FALSE, 0);
    
    
    lbl = gtk_label_new_with_mnemonic(_("_Audio Visualization"));
    gtk_widget_show(lbl);
    gtk_notebook_append_page(GTK_NOTEBOOK(notebook), topvbox, lbl);
    
    /* devices tab */
    topvbox = xfmedia_settings_create_devices_tab(mwin, win);
    lbl = gtk_label_new_with_mnemonic(_("_Devices"));
    gtk_widget_show(lbl);
    gtk_notebook_append_page(GTK_NOTEBOOK(notebook), topvbox, lbl);
    
    /* keybindings editor */
    topvbox = xfmedia_keybindings_create_editor(mwin, win);
    lbl = gtk_label_new_with_mnemonic(_("_Keybindings"));
    gtk_widget_show(lbl);
    gtk_notebook_append_page(GTK_NOTEBOOK(notebook), topvbox, lbl);
    
#if 0
    /* xfmedia plugins tab */
    
    topvbox = add_plugins_tab(mwin);
    lbl = gtk_label_new_with_mnemonic(_("_Plugins"));
    gtk_widget_show(lbl);
    gtk_notebook_append_page(GTK_NOTEBOOK(notebook), topvbox, lbl);
#endif
    
    sd_running = TRUE;
    gtk_widget_show(win);

}
