/*
 *  xfmedia - simple gtk2 media player based on xine
 *
 *  Copyright (c) 2004-2005 Brian Tarricone, <bjt23@cornell.edu>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; version 2 of the License ONLY.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <gdk/gdkx.h>
#include <gtk/gtk.h>

#include <libxfcegui4/libxfcegui4.h>
#include <libxfcegui4/netk-trayicon.h>

#include "trayicon.h"
#include "xfmedia-common.h"
#include <xfmedia/xfmedia-settings.h>
#include "mainwin.h"

enum
{
    TRAY_ACTION_NONE = 0,
    TRAY_ACTION_PLAYPAUSE,
    TRAY_ACTION_NEXT,
    TRAY_ACTION_PREV,
};

struct _XfmediaTrayIcon
{
    GtkWidget *tray_icon;
    GtkWidget *event_box;
    GtkMenu *menu;
    GtkTooltips *tooltip;
    gchar *tooltip_str;
    XfmediaMainwin *mwin;
};

static GtkWidget *_xfmedia_tray_icon_create(GdkScreen *gscreen, gpointer cb_data);

static gboolean
xfmedia_tray_icon_button_press_cb(GtkWidget *w, GdkEventButton *evt,
        gpointer user_data)
{
    XfmediaTrayIcon *tray_icon = user_data;
    XfmediaMainwin *mwin = tray_icon->mwin;
    
    if(evt->button == 1) {
        if(mwin->is_iconified) {
            NetkWindow *nwin;
            NetkScreen *nscr;
            NetkWorkspace *nws;
            
            gtk_window_deiconify(GTK_WINDOW(mwin->window));
            
            /* this next line is lame and probably won't work on multihead, but
             * netk_window_get() will fail if we don't first force an update */
            netk_screen_force_update(netk_screen_get_default());
            nwin = netk_window_get(GDK_WINDOW_XID(mwin->window->window));
            if(nwin) {
                nscr = netk_window_get_screen(nwin);
                netk_screen_force_update(nscr);
                nws = netk_screen_get_active_workspace(nscr);
                netk_window_move_to_workspace(nwin, nws);
            }
        } else
            gtk_window_iconify(GTK_WINDOW(mwin->window));
        return TRUE;
    } else if(evt->button == 2) {
        gint action = xfmedia_settings_get_int("/xfmedia/general/tray_middle_click_action");
        
        switch(action) {
            case TRAY_ACTION_PLAYPAUSE:
                xfmedia_mainwin_toggle_playpause(mwin);
                break;
            
            case TRAY_ACTION_NEXT:
                xfmedia_mainwin_next(mwin);
                break;
            
            case TRAY_ACTION_PREV:
                xfmedia_mainwin_prev(mwin);
                break;
            
            case TRAY_ACTION_NONE:
            default:
                break;
        }
    } else if(evt->button == 3) {
        if(tray_icon->menu) {
            gtk_menu_popup(GTK_MENU(tray_icon->menu), NULL, NULL, NULL,
                    NULL, evt->button, evt->time);
        }
        return TRUE;
    }
    
    return FALSE;
}

static void
xfmedia_tray_icon_destroy_cb(GtkObject *object, gpointer user_data)
{
    XfmediaTrayIcon *tray_icon = user_data;
    GdkScreen *gscreen = gtk_widget_get_screen(GTK_WIDGET(object));
    
    tray_icon->tray_icon = _xfmedia_tray_icon_create(gscreen, tray_icon);
    if(tray_icon->tooltip_str) {
        gtk_tooltips_set_tip(tray_icon->tooltip, tray_icon->event_box,
                tray_icon->tooltip_str, NULL);
    }
}

static GtkWidget *
_xfmedia_tray_icon_create(GdkScreen *gscreen, gpointer cb_data)
{
    GtkWidget *tray_icon, *evtbox, *img;
    GdkPixbuf *pix, *tmp;
    gint iw, ih;
    
    tray_icon = netk_tray_icon_new(GDK_SCREEN_XSCREEN(gscreen));
    
    evtbox = gtk_event_box_new();
    gtk_widget_add_events(evtbox, GDK_BUTTON_PRESS);
    gtk_widget_show(evtbox);
    gtk_container_add(GTK_CONTAINER(tray_icon), evtbox);
    
    gtk_icon_size_lookup(GTK_ICON_SIZE_LARGE_TOOLBAR, &iw, &ih);
    pix = xfce_icon_theme_load(icon_theme, "xfmedia", iw);
    if(!pix) {
        tmp = gdk_pixbuf_from_pixdata(&xfmedia_icon, TRUE, NULL);
        if(tmp && gdk_pixbuf_get_width(tmp) == iw && gdk_pixbuf_get_height(tmp) == ih)
            pix = tmp;
        else if(tmp) {
            pix = gdk_pixbuf_scale_simple(tmp, iw, ih, GDK_INTERP_BILINEAR);
            g_object_unref(G_OBJECT(tmp));
        } else {
            pix = gdk_pixbuf_new(GDK_COLORSPACE_RGB, TRUE, 8, iw, ih);
            gdk_pixbuf_fill(pix, 0);
        }
    }
    
    img = gtk_image_new_from_pixbuf(pix);
    g_object_unref(G_OBJECT(pix));
    gtk_widget_show(img);
    gtk_container_add(GTK_CONTAINER(evtbox), img);
    
    g_signal_connect(G_OBJECT(evtbox), "button-press-event",
            G_CALLBACK(xfmedia_tray_icon_button_press_cb), cb_data);
    g_signal_connect(G_OBJECT(tray_icon), "destroy",
            G_CALLBACK(xfmedia_tray_icon_destroy_cb), cb_data);
    
    gtk_widget_show(tray_icon);
    
    return tray_icon;
}

XfmediaTrayIcon *
xfmedia_tray_icon_create(GdkScreen *gscreen, XfmediaMainwin *mwin,
        GtkMenu *menu)
{
    XfmediaTrayIcon *tray_icon;
    
    
    tray_icon = g_new0(XfmediaTrayIcon, 1);
    tray_icon->tray_icon = _xfmedia_tray_icon_create(gscreen, tray_icon);
    tray_icon->event_box = gtk_bin_get_child(GTK_BIN(tray_icon->tray_icon));
    tray_icon->mwin = mwin;
    tray_icon->menu = menu;
    tray_icon->tooltip = gtk_tooltips_new();
    g_object_ref(G_OBJECT(tray_icon->tooltip));
    gtk_object_sink(GTK_OBJECT(tray_icon->tooltip));
    
    return tray_icon;
}

void
xfmedia_tray_icon_set_tooltip(XfmediaTrayIcon *tray_icon, const gchar *tip)
{
    g_return_if_fail(tray_icon);
    
    if(tray_icon->tooltip_str)
        g_free(tray_icon->tooltip_str);
    tray_icon->tooltip_str = g_strdup(tip);
    
    gtk_tooltips_set_tip(tray_icon->tooltip, tray_icon->event_box, tip, NULL);
}

void
xfmedia_tray_icon_destroy(XfmediaTrayIcon *tray_icon)
{
    g_return_if_fail(tray_icon && NETK_IS_TRAY_ICON(tray_icon->tray_icon));
    
    g_signal_handlers_disconnect_by_func(G_OBJECT(tray_icon->tray_icon),
            xfmedia_tray_icon_destroy_cb, tray_icon);
    
    g_object_unref(G_OBJECT(tray_icon->tooltip));
    if(tray_icon->tooltip_str)
        g_free(tray_icon->tooltip_str);
    
    gtk_widget_destroy(tray_icon->tray_icon);
    g_free(tray_icon);
}
