/*
 * Worldvisions Weaver Software:
 *   Copyright (C) 1997, 1998 Worldvisions Computer Technology, Inc.
 */
#include "wvtcp.h"
#include "wvstreamlist.h"
#include "wvlog.h"
#include <time.h>
#include <stdlib.h>
#include <signal.h>

static __u16 ports[] = {8042,4242,3325,3380,1234,3334,3321,3323,3300,3310};
static const int numports = sizeof(ports) / sizeof(__u16);

static int failures = 0, tries = 0;

#define RANDOM ((char *)1)

// generic test stuff
static char *c1[] = {
    "A-1\n", "A-2\n", "A-3\n", "A-4\n", "A-5\n", RANDOM, "A-6\n",
    "A-7\n", "A-8\n", NULL
};

// more generic test stuff
static char *c2[] = {
    "B-1\n", "B-2\n", "B-3\n", "B-4\n", "B-5\n", "B-6\n",
    "B-7\n", "B-8\n", NULL
};

// assorted tunnel vision commands
static char *tvc[] = {
    "HELLO\n", "HELLO ", RANDOM, "!! ", RANDOM, "help\n", "!!\n",
    "xor 0\n", "X ", RANDOM, "rsa ", RANDOM, "blowfish 32\n", RANDOM,
    "password ", RANDOM, "ihave\n", "ihave ", RANDOM, "vpn\n", "quit\n",
    "hello\n!! foo\npassword scs\nihave 1.2.3.4/255.0.0.0\nvpn\n", "quit\n",
    NULL
};

// some web server stuff
static char *webc[] = {
    "GET /\n", "GET / HTTP/1.0\n", "GET / HTTP/1.1\n", "\n",
    "GET ", RANDOM, "GET / ", RANDOM, "GET / HTTP", RANDOM,
    "POST /\n", "POST / HTTP/1.0\n", "POST / HTTP/1.1\n", "\n",
    "POST ", RANDOM, "POST / ", RANDOM, "POST / HTTP", RANDOM,
    NULL
};

// Weaver remote control commands
static char *rcc[] = {
    "help\n", "auth ", RANDOM, "auth root scs\n", "\n", RANDOM,
    "debug 9\n", "getcfg ", RANDOM, "setcfg ", RANDOM, "dial ", RANDOM,
    "dial connect\n", "tunnel\n", "tunnel ", RANDOM, "softupdate\n",
    "softupdate index\n", "softupdate switch ", RANDOM,
    "softupdate download ", RANDOM, "netmap\n", 
    "debug ", RANDOM, "dial hangup\n", "quit\n", NULL
};


static char **conversations[] = {c1, c2, tvc, webc, rcc};
static const int numconv = sizeof(conversations) / sizeof(char **);

enum StressMode { Exact = 0, RandomOrder, RandomData, NUM_MODES };

class TCPStressor : public WvTCPConn
{
public:
    TCPStressor(const WvIPPortAddr &target, char **_conversation,
		StressMode _mode);
    virtual ~TCPStressor();
    
    virtual bool select_setup(SelectInfo &si);
    virtual void execute();
    
    void converse();
    void send_random();
    
    void greetme()
        { log("[PING]\n"); }
    
private:
    int num;
    bool was_connected;
    WvLog log;
    time_t last_rx, last_try;
    
    char **conversation, **convptr;
    int numwords;
    StressMode mode;
};


TCPStressor::TCPStressor(const WvIPPortAddr &target, char **_conversation,
			 StressMode _mode)
		: WvTCPConn(target), num(tries++),
		  log(WvString("TCP-#%s-p%s", num, target.port), WvLog::Info)
{
    was_connected = false;
    last_rx = last_try = time(NULL)-1;
    
    conversation = convptr = _conversation;
    mode = _mode;
    
    for (numwords = 0; conversation[numwords]; numwords++)
	;
    
    // log("Created! ", target);
}


TCPStressor::~TCPStressor()
{
    // log("Dead!\n");
    log("\n");
    failures++;
}


bool TCPStressor::select_setup(SelectInfo &si)
{
    time_t now = time(NULL);
    
    if (isconnected() && now - last_try > 0)
	return true;
    else if (!isconnected() && now - last_rx > 5)
    {
	log("[NO-ANSWER]\n");
	close();
    }
    
    return WvTCPConn::select_setup(si);
}
  


void TCPStressor::execute()
{
    char buf[50];
    size_t len;
    time_t now = time(NULL);
    
    last_try = now;
	
    if (isconnected())
    {
	if (!was_connected)
	{
	    // log("Connected! ");
	    was_connected = true;
	    if (isok() && select(0, true, true) && isok())
		failures = 0;
	}
    }
    else
    {
	log("[NO-CONNECTION]\n");
    }
    
    while (isok())
    {
	len = read(buf, sizeof(buf) - 1);
	
	if (len > 0)
	{
	    buf[len] = 0;
	    log("[READ-%s] ", len);
	    last_rx = now;
	    converse();
	}
	else
	{
	    converse();
	    break;
	}
    }
    
    if (now - last_rx > 10)
	close();
    
    // done conversation
    if (!*convptr)
	close();
}


void TCPStressor::converse()
{
    char **which;
    
    if (!*convptr)
	return;
    
    switch (mode)
    {
    case Exact:
	if (*convptr == RANDOM) // randomness
	    send_random();
	else
	{
	    log("[EW#%s] ", convptr - conversation);
	    print(*convptr);
	}
	convptr++;
	break;
	
    case RandomOrder:
	which = conversation + (random() % numwords);
	if (*which == RANDOM)
	    send_random();
	else
	{
	    log("[RW#%s] ", which - conversation);
	    print(*which);
	}
	convptr++;
	break;
	
    case RandomData:
	send_random();
	convptr++;
	break;
	
    case NUM_MODES:
	break;
    }
}


void TCPStressor::send_random()
{
    char buf[128];
    size_t count, size = random() % (sizeof(buf) - 1);
    
    for (count = 0; count < size; count++)
	buf[count] = random() % 256;
    
    log("[xW-%s] ", size);
    write(buf, size);
    write("\n", 1);
}


int main(int argc, char **argv)
{
    WvLog log("stresstest", WvLog::Info);
    WvString target_name(argc >= 2 ? argv[1] : "localhost");
    WvResolver dns;
    const WvIPAddr *target;
    WvStreamList l;
    unsigned int portc = 0;
    
    signal(SIGPIPE, SIG_IGN);
    
    while (dns.findaddr(-1, target_name, &target) < 0)
	;
    
    if (dns.findaddr(-1, target_name, NULL) == 0)
    {
	log(WvLog::Error, "No such name: %s\n", target_name);
	return 1;
    }
    
    srandom(time(NULL));

    while (l.isok() && failures < 50)
    {
	if (l.count() < 0)
	    l.append(new TCPStressor(WvIPPortAddr(*target, 3300),
				     conversations[1], Exact),
		     true);

	while (l.count() < 10)
	{
	    l.append(new TCPStressor(WvIPPortAddr(*target,
						  ports[portc++ % numports]),
				     conversations[random() % numconv],
				     (StressMode)(random() % NUM_MODES)),
		     true);
	}
	
	log("\n");
	
	if (l.select(1000))
	    l.callback();
	else
	{
	    log("All %s streams are idle!\n", l.count());
	    
	    WvStreamList::Iter i(l);
	    for (i.rewind(); i.next(); )
	    {
		TCPStressor &s = (TCPStressor &)i();
		s.greetme();
	    }
	}
    }
    
    if (failures >= 50)
    {
	log("Too many consecutive connect failures (%s) -- aborting.\n",
	    failures);
    }
    
    return 0;
}
