/*
 * Copyright (c) 1995 - 2000 Kungliga Tekniska Hgskolan
 * (Royal Institute of Technology, Stockholm, Sweden).
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 
 * 3. Neither the name of the Institute nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE INSTITUTE AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE INSTITUTE OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include <sys/errno.h>
#include <sys/file.h>
#include <sys/proc.h>
#include <sys/systm.h>

#include <xfs/xfs_common.h>
#include <xfs/xfs_message.h>
#include <xfs/xfs_msg_locl.h>
#include <xfs/xfs_dev.h>
#include <xfs/xfs_fs.h>
#include <xfs/xfs_deb.h>
#include <xfs/nxfs.h>

/*
 * Queues of xfs_links hold outbound messages and processes sleeping
 * for replies. The last field is used to return error to sleepers and
 * to keep record of memory to be deallocated when messages have been
 * delivered or dropped.
 */
struct xfs_link {
  struct xfs_link *prev, *next;
  struct xfs_message_header *message;
  u_int error_or_size;		/* error on sleepq and size on messageq */
};  

struct xfs_channel {
  struct xfs_link messageq;	/* Messages not yet read */
  struct xfs_link sleepq;	/* Waiting for reply message */
  u_int nsequence;
  struct proc *selecting_proc;
  struct xfs_message_header *message_buffer;
  int status;
#define CHANNEL_OPENED	0x1
};

static struct xfs_channel xfs_channel[NXFS];

#if defined(__STDC__)
static void xfs_initq(struct xfs_link *q)
#else
static void
xfs_initq(q)
     struct xfs_link *q;
#endif
{
  q->next = q;
  q->prev = q;
}

/* Is this queue empty? */
#define xfs_emptyq(q) ((q)->next == (q))

/* Is this link on any queue? Link *must* be inited! */
#define xfs_onq(link) ((link)->next != 0 || (link)->prev != 0)

/* Append q with p */
#if defined(__STDC__)
static void xfs_appendq(struct xfs_link *q, struct xfs_link *p)     
#else
static void
xfs_appendq(q, p)
     struct xfs_link *q, *p;
#endif
{
  p->next = q;
  p->prev = q->prev;
  p->prev->next = p;
  q->prev = p;
}

#if defined(__STDC__)
static void xfs_outq(struct xfs_link *p)     
#else
static void
xfs_outq(p)
     struct xfs_link *p;
#endif
{
  p->next->prev = p->prev;
  p->prev->next = p->next;
  p->next = p->prev = 0;
}

/*
 * Only allow one open.
 */
#if defined(__STDC__)
int xfs_devopen(dev_t dev, int flags)
#else
int
xfs_devopen(dev, flags)
     dev_t dev;
     int flags;
#endif
{
  struct xfs_channel *chan;

  XFSDEB(XDEBDEV, ("xfs_devopen dev = %d, flags = %d\n", dev, flags));

  if (minor(dev) < 0 || minor(dev) >= NXFS)
    return ENXIO;

  chan = &xfs_channel[minor(dev)];

  /* Only allow one reader/writer */
  if (chan->status & CHANNEL_OPENED)
    return EBUSY;
  else
    chan->status |= CHANNEL_OPENED;

  chan->message_buffer = xfs_alloc(MAX_XMSG_SIZE);

  return 0;
}

/*
 * Wakeup all sleepers and cleanup.
 */
#if defined(__STDC__)
int xfs_devclose(dev_t dev, int flags)
#else
int
xfs_devclose(dev, flags)
     dev_t dev;
     int flags;
#endif
{
  struct xfs_channel *chan = &xfs_channel[minor(dev)];
  struct xfs_link *first;

  XFSDEB(XDEBDEV, ("xfs_devclose dev = %d, flags = %d\n", dev, flags));

  /* Sanity check, paranoia? */
  if (!(chan->status & CHANNEL_OPENED))
    panic("xfs_devclose never opened?");

  chan->status &= ~CHANNEL_OPENED;

  /* No one is going to read those messages so empty queue! */
  while (!xfs_emptyq(&chan->messageq))
    {
      XFSDEB(XDEBDEV, ("before outq(messageq)\n"));
      first = chan->messageq.next;
      xfs_outq(first);
      if (first->error_or_size != 0)
	xfs_free(first, first->error_or_size);
      XFSDEB(XDEBDEV, ("after outq(messageq)\n"));
    }

  /* Wakeup those waiting for replies that will never arrive. */
  while (!xfs_emptyq(&chan->sleepq))
    {
      XFSDEB(XDEBDEV, ("before outq(sleepq)\n"));
      first = chan->sleepq.next;
      xfs_outq(first);
      first->error_or_size = ENODEV;
      wakeup((caddr_t) first);
      XFSDEB(XDEBDEV, ("after outq(sleepq)\n"));
    }
  
  if (chan->message_buffer)
    {
      xfs_free(chan->message_buffer, MAX_XMSG_SIZE);
      chan->message_buffer = 0;
    }
      
  /* Free all xfs_nodes. */
  free_all_xfs_nodes(&xfs[minor(dev)]);
  return 0;
}

/*
 * Move messages from kernel to user space.
 */
#if defined(__STDC__)
int xfs_devread(dev_t dev, struct uio *uiop)
#else
int
xfs_devread(dev, uiop)
     dev_t dev;
     struct uio *uiop;
#endif
{
  struct xfs_channel *chan = &xfs_channel[minor(dev)];
  struct xfs_link *first;
  int error = 0;

  XFSDEB(XDEBDEV, ("xfs_devread dev = %d\n", dev));

  while (!xfs_emptyq (&chan->messageq)) {
      /* Remove message */
      first = chan->messageq.next;

      if (first->message->size > uiop->uio_resid)
	  break;

      error = uiomove((caddr_t) first->message, first->message->size,
		      UIO_READ, uiop);
      if (error)
	  break;
      xfs_outq(first);

      if (first->error_or_size != 0)
	  xfs_free(first, first->error_or_size);
  }
  return error;
}

/*
 * Move messages from user space to kernel space,
 * wakeup sleepers, insert new data in VFS.
 */
#if defined(__STDC__)
int xfs_devwrite(dev_t dev, struct uio *uiop)
#else
int
xfs_devwrite(dev, uiop)
     dev_t dev;
     struct uio *uiop;
#endif
{
  struct xfs_channel *chan = &xfs_channel[minor(dev)];
  char *p;
  int error;
  u_int cnt;
  struct xfs_message_header *msg_buf;

  XFSDEB(XDEBDEV, ("xfs_devwrite dev = %d\n", dev));

  cnt = uiop->uio_resid;
  error = uiomove((caddr_t) chan->message_buffer, MAX_XMSG_SIZE,
		  UIO_WRITE, uiop);
  if (error != 0)
    return error;
  
  cnt -= uiop->uio_resid;

  /*
   * This thread handles the received message.
   */

  for (p = (char *)chan->message_buffer;
       cnt > 0;
       p += msg_buf->size, cnt -= msg_buf->size) {
      msg_buf = (struct xfs_message_header *)p;
      error = xfs_message_receive (minor(dev),
				   msg_buf,
				   msg_buf->size);
  }
  XFSDEB(XDEBDEV, ("xfs_devwrite error = %d\n", error));
  return error;
}

/*
 * Not used.
 */
#if defined(__STDC__)
int xfs_devioctl(dev_t dev, int cmd, caddr_t data, int flags)
#else
int
xfs_devioctl(dev, cmd, data, flags)
     dev_t dev;
     int cmd;
     caddr_t data;
     int flags;
#endif
{
  XFSDEB(XDEBDEV, ("xfs_devioctl dev = %d, flags = %d\n", dev, flags));
  return EINVAL;
}

/*
 * Are there any messages on this filesystem?
 */
#if defined(__STDC__)
int xfs_devselect(dev_t dev, int rw)
#else
int
xfs_devselect(dev, rw)
     dev_t dev;
     int rw;
#endif
{
  struct xfs_channel *chan = &xfs_channel[minor(dev)];

  XFSDEB(XDEBDEV, ("xfs_devselect dev = %d, rw = %d\n", dev, rw));

  if (rw != FREAD)
    return 0;

  if (!xfs_emptyq(&chan->messageq))
    return 1;			/* Something to read */

  /*
   * No need to handle a "collission" since we only allow one
   * concurrent open. */
  chan->selecting_proc = u.u_procp; 
  return 0;
}

/*
 * Send a message to user space.
 */
#if defined(__STDC__)
int xfs_message_send(int fd, struct xfs_message_header *message, u_int size)
#else
int
xfs_message_send(fd, message, size)
     int fd;
     struct xfs_message_header *message;
     u_int size;
#endif
{
  struct xfs_channel *chan = &xfs_channel[fd];
  struct {
    struct xfs_link this_message;
    struct xfs_message_header msg;
  } *t;

  XFSDEB(XDEBMSG, ("xfs_message_send opcode = %d\n", message->opcode));

  if (!(chan->status & CHANNEL_OPENED))	/* No receiver? */
    return ENODEV;
  
  /* Prepare message and copy it later */
  message->size = size;
  message->sequence_num = chan->nsequence++;

  t = xfs_alloc(sizeof(t->this_message) + size);
  t->this_message.error_or_size = sizeof(t->this_message) + size;
  bcopy(message, &t->msg, size);

  t->this_message.message = &t->msg;
  xfs_appendq(&chan->messageq, &t->this_message);
  if (   chan->selecting_proc != 0
      && chan->selecting_proc->p_wchan == (caddr_t) &selwait)
    {
      selwakeup(chan->selecting_proc, 0); /* There is only one to wakeup */
      chan->selecting_proc = 0;
    }
  return 0;
}

/*
 * Send a message to user space and wait for reply.
 */
#if defined(__STDC__)
int xfs_message_rpc(int fd, struct xfs_message_header *message, u_int size)
#else
int
xfs_message_rpc(fd, message, size)
     int fd;
     struct xfs_message_header *message;
     u_int size;
#endif
{
  struct xfs_channel *chan = &xfs_channel[fd];
  struct xfs_link this_message;
  struct xfs_link this_process;

  XFSDEB(XDEBMSG, ("xfs_message_rpc opcode = %d\n", message->opcode));

  if (!(chan->status & CHANNEL_OPENED))	/* No receiver? */
    return ENODEV;
  
  if (size < sizeof(struct xfs_message_wakeup))
    {
      printf("XFS PANIC Error: Message to small to receive wakeup, opcode = %d\n", message->opcode);
      return ENOMEM;
    }

  message->size = size;
  message->sequence_num = chan->nsequence++;
  this_message.error_or_size = 0;
  this_message.message = message;
  this_process.message = message;
  xfs_appendq(&chan->messageq, &this_message);
  xfs_appendq(&chan->sleepq, &this_process);
  if (   chan->selecting_proc != 0
      && chan->selecting_proc->p_wchan == (caddr_t) &selwait)
    {
      selwakeup(chan->selecting_proc, 0); /* There is only one to wakeup */
      chan->selecting_proc = 0;
    }
  this_process.error_or_size = 0;
  if (sleep((caddr_t) &this_process, (PZERO + 1)|PCATCH))
    {
      XFSDEB(XDEBMSG, ("caught signal\n"));
      this_process.error_or_size = EINTR;
    }
  /*
   * Caught signal, got reply message or device was closed.
   * Need to clean up both messageq and sleepq.
   */
  if (xfs_onq(&this_message))
    {
      xfs_outq(&this_message);
    }
  if (xfs_onq(&this_process))
    {
      xfs_outq(&this_process);
    }
  return this_process.error_or_size;
}

/*
 * For each message type there is a message handler
 * that implements its action, xfs_message_receive
 * invokes the correct function.
 */
#if defined(__STDC__)
int xfs_message_receive(int fd, struct xfs_message_header *message, u_int size)
#else
int
xfs_message_receive(fd, message, size)
     int fd;
     struct xfs_message_header *message;
     u_int size;
#endif
{
  XFSDEB(XDEBMSG, ("xfs_message_receive opcode = %d\n", message->opcode));

  /* Dispatch and coerce message type */
  switch (message->opcode) {
  case XFS_MSG_WAKEUP:
    return xfs_message_wakeup(fd, (struct xfs_message_wakeup *) message, message->size);
  case XFS_MSG_WAKEUP_DATA:
    return xfs_message_wakeup_data(fd, (struct xfs_message_wakeup_data *) message, message->size);
  case XFS_MSG_INSTALLROOT:
    return xfs_message_installroot(fd, (struct xfs_message_installroot *) message, message->size);
  case XFS_MSG_INSTALLNODE:
    return xfs_message_installnode(fd, (struct xfs_message_installnode *) message, message->size);
  case XFS_MSG_INSTALLATTR:
    return xfs_message_installattr(fd, (struct xfs_message_installattr *) message, message->size);
  case XFS_MSG_INSTALLDATA:
    return xfs_message_installdata(fd, (struct xfs_message_installdata *) message, message->size);
  case XFS_MSG_INVALIDNODE:
    return xfs_message_invalidnode(fd, (struct xfs_message_invalidnode *) message, message->size);
  default:
    printf("XFS PANIC Warning xfs_dev: Unknown message opcode == %d\n", message->opcode);
    return EINVAL;
  }
}

#if defined(__STDC__)
int xfs_message_wakeup(int fd, struct xfs_message_wakeup *message, u_int size)
#else
int
xfs_message_wakeup(fd, message, size)
     int fd;
     struct xfs_message_wakeup *message;
     u_int size;
#endif
{
  struct xfs_channel *chan = &xfs_channel[fd];
  struct xfs_link *sleepq = &chan->sleepq;
  struct xfs_link *t = chan->sleepq.next; /* Really first in q */

  XFSDEB(XDEBMSG, ("xfs_message_wakeup\n"));

  for (; t != sleepq; t = t->next)
    if (t->message->sequence_num == message->sleepers_sequence_num)
      {
	if (t->message->size < size)
	  {
	    printf("XFS PANIC Error: Could not wakeup requestor with opcode = %d properly, to small receive buffer.\n", t->message->opcode);
	    t->error_or_size = ENOMEM;
	  }
	else
	  bcopy(message, t->message, size);
	wakeup((caddr_t) t);
	break;
      }

  return 0;
}

#if defined(__STDC__)
int xfs_message_wakeup_data(int fd,
			    struct xfs_message_wakeup_data *message,
			    u_int size)
#else
int
xfs_message_wakeup_dat(fd, message, size)
     int fd;
     struct xfs_message_wakeup_data *message;
     u_int size;
#endif
{
  struct xfs_channel *chan = &xfs_channel[fd];
  struct xfs_link *sleepq = &chan->sleepq;
  struct xfs_link *t = chan->sleepq.next; /* Really first in q */

  XFSDEB(XDEBMSG, ("xfs_message_wakeup_data\n"));

  for (; t != sleepq; t = t->next)
    if (t->message->sequence_num == message->sleepers_sequence_num)
      {
	if (t->message->size < size)
	  {
	    printf("XFS PANIC Error: Could not wakeup requestor with opcode = %d properly, to small receive buffer.\n", t->message->opcode);
	    t->error_or_size = ENOMEM;
	  }
	else
	  bcopy(message, t->message, size);
	wakeup((caddr_t) t);
	break;
      }

  return 0;
}

/*
 *
 */
#if defined(__STDC__)
static int xfs_uprintf_device(void)
#else
static
int
xfs_uprintf_device()
#endif
{
#if 1
  int i;
  for (i = 0; i < NXFS; i++)
    {
      uprintf("xfs_channel[%d] = {\n", i);
      uprintf("messageq.next = 0x%x ", (u_int) xfs_channel[i].messageq.next);
      uprintf("messageq.prev = 0x%x ", (u_int) xfs_channel[i].messageq.prev);
      uprintf("sleepq.next = 0x%x ", (u_int) xfs_channel[i].sleepq.next);
      uprintf("sleepq.prev = 0x%x ", (u_int) xfs_channel[i].sleepq.prev);
      uprintf("nsequence = %d selecting_proc = 0x%x status = %d\n",
	      xfs_channel[i].nsequence,
	      (u_int) xfs_channel[i].selecting_proc,
	      xfs_channel[i].status);
      uprintf("}\n");
    }
#endif
  return 0;
}

/*
 * Install and uninstall device.
 */
#if defined(__STDC__)
int xfs_install_device(void)
#else
int
xfs_install_device()
#endif
{
  int i;
  for (i = 0; i < NXFS; i++)
    {
      XFSDEB(XDEBDEV, ("before initq(messageq and sleepq)\n"));
      xfs_initq(&xfs_channel[i].messageq);
      xfs_initq(&xfs_channel[i].sleepq);
    }
  return 0;
}

#if defined(__STDC__)
int xfs_uninstall_device(void)
#else
int
xfs_uninstall_device()
#endif
{
  /* Check for open, mounted and active vnodes */
  return 0;
}

#if defined(__STDC__)
int xfs_vdstat_device(void)
#else
int
xfs_vdstat_device()
#endif
{
  return xfs_uprintf_device();
}
