/* GNOME Transcript
 * Copyright (C) 1999-2000 the Free Software Foundation
 * Authors: Jos Miguel Ronquillo
 *          Matias Mutchinick
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */



/**
 * This is a plugin for GNOME Transcript.
 * GNOME Transcript is a SQL Database Client that uses a plugins system 
 * to access diferent database servers.
 * The plugins are handled by libgtrans_ifase a library to designed to 
 * homogeneize access to different sql database servers.
 * This plugins defines the basic functions needed by libgtrans_ifase
 * and by GNOME Transcript to work with PostgreSQL servers, this plugin 
 * was written for and tested only in PostgreSQL 6.5.3 and might
 * not work in any other version of PostgreSQL, how ever it would take little
 * time and effort to port this to other version of PostgreSQL.
 * In order for the plugin to work you will need the shared versions of
 * PostgreSQL libraries, you could find them if available 
 * at http://www.postgresql.com
 *
 * This plugin uses the data types and definitions from the libgtrans_ifase.
 * This are...
 * GTransIFaseConn   : Homogenized connection to database servers.
 * GTransIFaseResult : Homogenized query result.
 *
 * As this library, will be loaded as a plugin, there are two functions (symbols)
 * that are mandatory:
 * Name    : Returns the name of the plugin.
 * Connect : Establish a GTransIFaseConn to the database server.
 *
 * All the other functions needed will be passed pointed at by members
 * in the GTransIFaseConn. These are those functions.
 *  
 * * The function that executes queries, must return a valid
 *   GTransIFaseResult. 
 * conn->Exec = Exec;
 * 
 * * The funcion tha finishes the connection.
 * conn->Disconnect = Disconnect;
 *
 * * The function that checks the status of the connection.
 * conn->Status = Status;
 *
 * * The funcion that returns the connections las error message.
 * conn->ErrorMsg = ErrorMsg;
 *
 * * The funcion that returns a list of the tables. 
 *   (The result mus be a GTransIFaseResult)
 * conn->ListTables = ListTables;
 *
 * * The funcion that returns a list of databases in the same server.
 *   (The result mus be a GTransIFaseResult)
 * conn->ListDb = ListDb;
 *
 * * The function that returns the names of the field type attributes.
 * conn->TableDesignerAttrNames   = TableDesignerAttrNames;
 *
 * * The function that returns the values of the field type attributes.
 * conn->TableDesignerAttrVals    = TableDesignerAttrVals;
 *
 * * The function that returns the names of the field types.
 * conn->TableDesignerFieldTypes  = TableDesignerFieldTypes;
 *
 * * The function that returns the names of the field type masks.
 * conn->TableDesignerTypeMasks   = TableDesignerTypeMasks;
 *
 * * The function that creates the table in the database.
 * conn->TableDesignerCreateTable = TableDesignerCreateTable;
 */

#include <pgsql/libpq-fe.h>
#include <glib.h>
#include <stdlib.h>
#include <stdio.h>
#include <ctype.h>
#include <string.h>
#include <gtrans_ifase/gtrans_ifase.h>
#include "reserved.h"


/* For the locale, not used yet.
 * THis will be replaced for gettext functions */
#define  _(_str_)    _str_
#define N_(_str_)    _str_



gchar *Name (void) {
	return "PostgreSQL_6_5_3";
}



/**
 * The purpose of GNOME Transcript if to provide a maximum of
 * interoperability beetwen the user and the database server,
 * for this reason instead of writting a custom table designer
 * that will work in ok most database servers but leave specific
 * server capabilities, we have decided to include functions in 
 * the plugins to define it and provide all the capabilities that 
 * the actual server provides.
 *
 * This is the way that the plugin functions for the table designer 
 * work:
 *
 * 1.- You must provide functions for defining the table designer.
 * 
 *     1.1  A function that returns a glib doubly linked list (GList *)
 *          with the names of the available field types.
 *          The first item in the list must be an empty string.
 *          Here, this function is named TableDesignerFieldTypes.
 *
 *     1.2  A function that returns a glib doubly linked list (GList *)
 *          with the names of all the available field attributes 
 *          for example (Length, Indices, Precision).
 *          Here this function is TableDesignerAttrNames 
 *
 *     1.3  A function that returns a glib doubly linked list (GList *)
 *          with integer numbers that will work as bit masks used
 *          by the table designer to decide which field type is in title
 *          to which field attributes. This list should have the same
 *          number of elements that the field type list and must start
 *          with a 0 (zero) that corresponds to the empty string in the
 *          type list.
 *          Here this function is TableDesignerAttrMasks 
 *
 *     1.4  A function that returns a glib doubly linked list (GList *)
 *          of glib doubly linked lists with the values available to
 *          each field attribute. This is, first you create separate
 *          GList's one for each field attribute, and each list must
 *          be a list of gchar's of the valid attribute values, this
 *          way the table designer will display a combo with te valid 
 *          options, a link with NULL data, means that this attribute 
 *          is a type-in and anything can be typed in.
 *          Then all this linked lists will be linked in another list.
 *          (You must provide a function this last list).
 *          Here this function is TableDesignerAttrVals 
 *
 * 
 * 2.- Then you must provide function for creating and executing the query
 *     that will create the table defined in the table designer.
 *     The table designer takes the data you entered an creates a GList of
 *     arrays of character strings
 *     Each link of the list reperesents the definition of one field.
 *     And each string of the array, the value entered for a particular 
 *     attribute.
 *     Here this function is TableDesignerCreateTable 
 */


/**
 * ** ABOUT the designer **
 * 
 * Once we have designed our table in the table designer,
 * the designer will returns us 2 things.
 * The name of the table (a gchar *).
 * The definition of the table designed.
 * This definition consist in a glib doubly linked list (GList *),
 * where each link of the list represents the definition of a field.
 * The definition of a field is an arry of string where each string
 * represents a value entered for that field definition. The first
 * string (field[0]) is always the field name, and the second string
 * (field[1]) is the field type. The subsecuent string are the 
 * values entered in the attributes we define in this our plugin
 * in the order we defined.
 * field[2] -> attribute1
 * field[3] -> attribute2
 * field[4] -> attribute3
 * ...
 * These is the data we will use to build the table.
 * So, in the particular case of this plugin each field
 * definition looks like this:
 * field_def[0] -> field's name    
 * field_def[1] -> field's type    
 * field_def[2] -> field's length    
 * field_def[3] -> field's null modifier  
 * field_def[4] -> field's default value 
 * field_def[5] -> field's index type
 * field_def[6] -> field's index name
 */



/**
 * TableDesignerFieldTypes
 *
 * Return a (GList *) of strings of the valid field types.    
 * This is one of the functions pointed at by a member of the conn.
 */
static GList *
TableDesignerFieldTypes()
{
	GList   *list = NULL;
	gint     i;
	gchar   *types[] = { ""         , 
			     "FLOAT4"   ,
			     "FLOAT8"   , 
			     "INT2"     ,
			     "INT4"     , 
			     "INT8"     , 
			     "SERIAL"   ,
			     "MONEY"    ,
			     "CHAR"     ,
			     "TEXT"     ,
			     "VARCHAR"  ,
			     "ABSTIME"  ,
			     "DATE"     ,
			     "DATETIME" ,
			     "INTERVAL" , 
			     "RELTIME"  , 
			     "TIME"     ,
			     "TIMESPAN" ,
			     "TIMESTAMP",
			     "BOOL"     ,
			     "POINT"    ,
			     "LINE"     ,
			     "LSEG"     ,
			     "BOX"      ,
			     "PATH"     , 
			     "POLYGON"  ,
			     "CIRCLE"   ,
			     "CIDR"     ,
			     "INET"     ,
			     NULL        };
	gchar *tmp;
	
	i=0;
	while ( (tmp = types[i++]) != NULL )
		list = g_list_append (list, g_strdup (tmp));
	
	return list;
}





/**
 * TableDesignerTypeMasks
 *
 * Return a (GList *) of integers that are the bit masks
 * for the field arttributes.
 * e.g. 
 * The binary representation of 125 is 1111101, you will see
 * later that this plugin defines 7 attributes for the table designer,
 * a 1 means that a field type is in title to certain attribute.
 * 125 (1111101) as a bit mask, means that this particular field type,
 * the one that this mask belongs to is in title to the 
 * 1,3,4,5,6 and 7 attributes and the 2 attribute is forbidden.
 * This is one of the functions pointed at by a member of the conn.
 */
static GList *
TableDesignerTypeMasks()
{
	GList   *list = NULL;
	gint     i;
	gint     mask[] = { 0, 
			    30,
			    30,
			    30,
			    30,
			    6,
			    24,
			    6,
			    31,
			    30,
			    31,
			    30,
			    30,
			    30,
			    30,
			    30,
			    30,
			    30,
			    30,
			    6,
			    6,
			    6,
			    6,
			    6,
			    6,
			    6,
			    6,
			    30,
			    30,
			    -1 };
	gint tmp;
	
	i=0;
	while ( (tmp = mask[i++]) != -1 )
		list = g_list_append (list,(gpointer) tmp);
	
	return list;
}





/**
 * TableDesignerNullity
 *
 * Return a (GList *) of strings that are the options for the
 * field's NULL attribute.
 */
static GList *
TableDesignerNullity()
{
	GList   *list;
	
	list = g_list_append (NULL, g_strdup(""));
	list = g_list_append (list, g_strdup("NOT NULL"));
	
	return list;
}




/**
 * TableDesignerIndexes
 *
 * Return a (GList *) of strings that are the options for the
 * field's KEY (INDEX) attribute.
 */
static GList *
TableDesignerIndexes()
{
	GList   *list;
	
	list = g_list_append (NULL, g_strdup(""));
	list = g_list_append (list, g_strdup("INDEX"));
	list = g_list_append (list, g_strdup("UNIQUE INDEX"));
	list = g_list_append (list, g_strdup("PRIMARY KEY"));
	
	return list;
}



/**
 * TableDesignerAttrVals
 *
 * Return a (GList *) of (GList *) that are the options for the
 * field's Attributes.
 * This is one of the functions pointed at by a member of the conn.
 */
static GList *
TableDesignerAttrVals()
{
	GList  *list;
	
	list = g_list_append(NULL,NULL);
	list = g_list_append(list,TableDesignerNullity());
	list = g_list_append(list,NULL);
	list = g_list_append(list,TableDesignerIndexes());
	list = g_list_append(list,NULL);
	
	return list;
}




/**
 * TableDesignerAttrNames
 *
 * Return a (GList *) of strings that are the names for the
 * field's attributes.
 * This is one of the functions pointed at by a member of the conn.
 */
static GList *
TableDesignerAttrNames()
{
	GList  *list;
	
	list = g_list_append(NULL,g_strdup("Length"));
	list = g_list_append(list,g_strdup("Null"));
	list = g_list_append(list,g_strdup("Default"));
	list = g_list_append(list,g_strdup("Index"));
	list = g_list_append(list,g_strdup("Index-Name"));
	
	return list;
}





/**
 * Here start the functions that will take the table designer's data
 * turn it into a query and then execute it.
 * First we will do some (or a lott) of error checking.
 *
 * Here I parse a field definition at a time, and return an error
 * message if an invalid vlue has been entered in the table designer,
 * Then I can check more global things, like indexes. 
 */ 

/**
 * These are error types.
 */
enum {
	EVERYT_OK               = 0,
	
	/* Field definition Errors */
	MISSING_FIELD_NAME      = 1,
	INVALID_FIELD_RESERVED  = 2,
	INVALID_FIELD_SYNTAX    = 3,
	MISSING_TYPE            = 4,
	MISSING_LEN             = 5,
	INVALID_LEN_RANGE       = 6,
	INVALID_LEN_SYNTAX      = 7,
	INVALID_INDEX_SYNTAX    = 8,
	INVALID_INDEX_RESERVED  = 9,
	INDEX_NOT_DEFINED       = 10,
	MISSING_INDEX_NAME      = 11,
	INVALID_DEFA_SYNTAX     = 12,
	
	/* Table definition errors */
	NO_FIELDS_DEFINED       = 13,
	MISSING_TABLE_NAME      = 14,
	INVALID_TABLE_RESERVED  = 15,
	INVALID_TABLE_SYNTAX    = 16,
	MULTIPLE_PRIMARY_KEY    = 17
};



/*
 * TableDesignerErrorMsg
 * Here we keep the error messages that must be
 * syncronized with the error types enum.
 */
static gchar * 
TableDesignerErrorMsg[] = {
	"",
	
	/* Field definition error messages */
	N_("Missing field name"),
	N_("Field name is a reserved word"),
	N_("Field name invalid syntax"),
	N_("Missing field type"),
	N_("Field length needed"),
	N_("Invalid field length\nValid range 1 - 4096"),
	N_("Invalid field length\nMust be an integer number"),
	N_("Invalid index name\nSyntax error"),
	N_("Index name is a reserved word"),
	N_("Invalid index name\nIndex not defined"),
	N_("Secondary indices must be named"),
	N_("Invalid default value\nSyntax error"),
	
	/* Table definition error messages */
	N_("Error : No fields defined"),
	N_("Error : Missing table name"),
	N_("Error : Table name is a reserved word"),
	N_("Error : Table name invalid syntax"),
	N_("Error : There can be only one PRIMARY KEY")
};




/**
 * TableDesignerParseInt
 * @str : String representation of an integer number
 *
 * Returns 0 if the string passes the syntax test or 1
 * if it does not.
 * This again is not a function you must provide.
 */
static gint
TableDesignerParseInt(gchar *str)
{
	gint i = 0;
	
	while (isspace(str[i]))	
		i++;
	if (str[i] == '-' || str[i] == '+' )
		i++;
	while (isdigit(str[i]))
		i++;
	
	if (str[i] != 0)
		return 1;
	
	return 0;
}





/**
 * TableDesignerValidateLength
 * @str : A string reperesenting the fields length.
 *
 * If the string does not reperesents an integer,
 * or the integer is out of range, this returns a
 * error code, otherwise returns 0.
 */
static gint
TableDesignerValidateLength(gchar *str)
{
	gint val;
	
	if (TableDesignerParseInt(str))
		return INVALID_LEN_SYNTAX;
	
	val = atoi(str);
	if ( val > 4096 || val < 1  )
		return INVALID_LEN_RANGE;
	
	return 0;
}




/**
 * TableDesignerValidateString
 * @str  : A character string.
 *
 * This function checks that the string wont cause any problems
 * inside the query. 
 * Return 0 if the string is ok and 1 if there is an
 * error.
 */
static gint
TableDesignerValidateString(gchar *str)
{
	gint  i = 0;
	
	while (str[i] != 0){
		
		if (str[i] == '\'')
			return 1;
		
		if (str[i] == '\\'){
			i++;
			if (str[i] == 0)
				return 1;
		}
		i++;
	}
	return 0;
}




/**
 * TableDesignerValidateWordId
 * @word : A string
 *
 * Check if the string is a reserved postgresql word for an
 * identifier.
 * Returns : 1 if word is reserved, otherwise return 0.
 */
gint static
TableDesignerValidateWordId(gchar *word)
{
	gchar *tmp;
	gint   i = 0;
	
	while ((tmp = reserved_id[i++]))
		if (!g_strcasecmp(tmp,word))
			return 1;
	
	return 0;
}





/**
 * TableDesignerValidateWordField
 * @word : A string
 *
 * Check if the string is a reserved postgresql word for a
 * field name.
 * Returns : 1 if word is reserved, otherwise return 0.
 */
gint static
TableDesignerValidateWordField(gchar *word)
{
	gchar *tmp;
	gint   i = 0;
	
	while ((tmp = reserved_field[i++]))
		if (!g_strcasecmp(tmp,word))
			return 1;
	
	return 0;
}





/**
 * TableDesignerValidateWordSyntax
 * @word : A string
 *
 * Check if the string is a valid identifier or field name. 
 * Returns : 1 if word is invalid, otherwise return 0.
 */
gint static
TableDesignerValidateWordSyntax(gchar *word)
{
	gchar  c;
	gint   i = 1;
	
	if (!isalpha(word[0]) && word[0] != '_')
		return 1;
	
	while ((c = word[i++]))		
		if (!isalnum(c) && c != '_')
			return 1;
	
	return 0;
}





/**
 * TableDesignerValidateField
 * @field : An arry of strings that define a field 
 *          in the table designer. (The field definition)
 *
 * Returns an error code if the field definition is wrong
 * and 0 if it is ok.
 * This function checks a lott of stuff, that the required arguments
 * are there, that the args are OK, that args are compatible , etc...
 * (it basicaly checks that the per-field definition is valid)
 */
static gint
TableDesignerValidateField(gchar **field)
{
	gchar *type, *len, *name, *null, *indx, *indx_name, *defa;
	gint   error;
	
	name = field[0];
	type = field[1];
	len  = field[2];
	null = field[3];
	defa = field[4];
	indx = field[5]; 
	indx_name = field[6]; 
	

	/* Name is mandatory */
	if(!name)
		return MISSING_FIELD_NAME;
	
	/* Check for a valid field name */
	if(TableDesignerValidateWordSyntax(name))
		return INVALID_FIELD_SYNTAX;
	
	/* Check that field name is not a reserved word */
	if(TableDesignerValidateWordField(name))
		return INVALID_FIELD_RESERVED;
	
	/* If type is missing we have an error */
	if (!type)
		return MISSING_TYPE;
	
	/* VARCHAR demands field length */
	if (!strcmp(type,"VARCHAR") && !len)
		return MISSING_LEN;
	
	/* Length attribute must be within range */
	if (len)
		if ((error = TableDesignerValidateLength(len)))
			return error;
	
	/* Default value syntax check */
	if (defa)
		if ((error = TableDesignerValidateString(defa)))
			return INVALID_DEFA_SYNTAX;
	
	/* No index-name without index defined */
	if (!indx && indx_name)
		return INDEX_NOT_DEFINED;

	/* No index-name without index defined */
	if (indx)
		if (strcmp(indx,"PRIMARY KEY") && !indx_name)
			return MISSING_INDEX_NAME;
	
	/* Validate index-name syntax */
	if (indx_name)
		if(TableDesignerValidateWordSyntax(indx_name))
			return INVALID_INDEX_SYNTAX;
	
	/* Validate index-name reserved */
	if (indx_name)
		if(TableDesignerValidateWordId(indx_name))
			return INVALID_INDEX_RESERVED;
	
	/* Everithing OK */
	return EVERYT_OK;
}




/**
 * TableDesignerValidateFields
 * @tdef  : A GList of field definitions.
 *
 * Checks one by one the field definitions and return NULL
 * if everiithing is of, returns an error message other wise.
 * This is not a required function if you settle for the server's
 * error message, the error message is mandatory.
 */
static gchar *
TableDesignerValidateFields(GList *tdef)
{
	GList  *tmp;
	gchar **field;
	gint    status = 0 , i = 0;
		
	
	for (tmp = tdef ; tmp != NULL ; tmp = tmp->next , i++){
		
		if ((field = (gchar **)tmp->data) == NULL)
			continue;
			
		status = TableDesignerValidateField(field);
		
		if(status){
			gchar *msg;
			
			msg = g_strdup_printf(_("Error in field definition %i\n%s"),i,
					      TableDesignerErrorMsg[status]);
			
			return msg;
		}
	}
	return NULL;
}




/**
 * TableDesignerValidateFieldNumber
 * @tdef : The GList of field definitions
 *
 * Check tha there is only one field defined as primary key.
 */
static gint
TableDesignerValidateFieldNumber(GList *tdef)
{
	GList  *tmp;
	gint    num = 0;
	
	for (tmp = tdef ; tmp != NULL ; tmp = tmp->next)
		if (tmp->data)
			num++;
	
	if (num == 0)
		return 1;
	
	return 0;
}




/**
 * TableDesignerValidatePrimaryKey
 * @tdef : The GList of field definitions
 *
 * Check tha there is only one field defined as primary key.
 */
static gint
TableDesignerValidatePrimaryKey(GList *tdef)
{
	GList  *tmp;
	gchar **field, *indx;
	gint    keys = 0;
	
	for (tmp = tdef ; tmp != NULL ; tmp = tmp->next){
		
		if ((field = (gchar **)tmp->data) == NULL)
			continue;
			
		indx = field[5];
		if (indx)
			if (!strcmp(indx,"PRIMARY KEY")){
				if (keys == 1)
					return 1;
				keys++;
			}
		}
	
	return 0;
}




/**
 * TableDesignerValidateTable
 * @name : The name for the table
 * @tdef : The GList of field definitions
 *
 * Validate the table definition, syntax, indexes, values, etc...
 */
static gchar *
TableDesignerValidateTable(gchar *tname,
			   GList *tdef)
{
       	/* Table Name */
	if(strlen(tname) == 0)
		return g_strdup(TableDesignerErrorMsg[MISSING_TABLE_NAME]);
	
	/* Must be at least one field defined */
	if(TableDesignerValidateFieldNumber(tdef))
		return g_strdup(TableDesignerErrorMsg[NO_FIELDS_DEFINED]);
	
	/* No more than one primary key */
	if(TableDesignerValidatePrimaryKey(tdef))
		return g_strdup(TableDesignerErrorMsg[MULTIPLE_PRIMARY_KEY]);
		
	/* Is table_name valid for a table name? */
	if(TableDesignerValidateWordSyntax(tname))
		return g_strdup(TableDesignerErrorMsg[INVALID_TABLE_SYNTAX]);
	
	/* Is table name a reserved word? */
	if(TableDesignerValidateWordId(tname))
		return g_strdup(TableDesignerErrorMsg[INVALID_TABLE_RESERVED]);
	
	return TableDesignerValidateFields(tdef);
}





/**
 * Stuff for the table designer query generation and execution
 */

/**
 * TableDesignerCalcFieldSize
 * @field : An arry of strings (field definition).
 *
 * Return the number of characters that this field definition
 * would take inside a query string.
 */
static gint
TableDesignerCalcFieldSize(gchar **field)
{
	gchar *name, *type, *len, *null, *defa, *indx;
	gint   size = 4;
	
	g_return_val_if_fail(field != NULL,0);
	
	
	name = field[0];
	type = field[1];
	len  = field[2];
	null = field[3];
	defa = field[4];
	indx = field[5];
	
	size += strlen(name) + strlen(type);
	
	if(len)
		size += strlen(len) + 2;
	
	if(null)
		size += strlen(null) + 1;
	
	if(defa)
		size += strlen(defa) + 11;
	
	if(indx)
		if (!strcmp(indx,"PRIMARY KEY"))
			size += 12;
	
	return size;
}



/**
 * TableDesignerCalcQuerySize
 * @name  : Tha table name
 * @rows  : A GList of arry of strings (field definitions).
 *
 * Return the number of characters needed to create
 * a query string for the table defined in the table designer.
 */
static gint
TableDesignerCalcQuerySize(gchar *name,
			   GList *tdef)
{
	GList  *tmp;
	gint    size = 20;
	gchar **field;

	size += strlen(name);
	
	for (tmp = tdef ; tmp != NULL ; tmp = tmp->next){
		
		if ((field = (gchar **)tmp->data))
			size += TableDesignerCalcFieldSize(field);
	}
	
	return size;
}






/*
 * *** Regarding Indexation ***
 *
 * Postgres does not let us create a great variety of indices
 * in the "create table" query, so we will create a query for 
 * each index, put them in a GList along with the creation
 * query, start a transaction execute all the querys and 
 * commit the transaction.
 */


/**
 * TableDesignerCreateQuery
 * @tname  : Table name
 * @tdef : A GList of arry of strings (field definitions).
 *
 * Create the query needed to create the table designed in @gtd.
 */
static gchar *
TableDesignerCreateQuery(gchar    *tname,
			 GList    *tdef)
{
	GList   *tmp;
	gchar  **field;
	gchar   *qstr, *fname, *ftype, *flen, *fnull, 
		*fdefa, *findx;
	gint     size, i = 0;
	
	
	size = TableDesignerCalcQuerySize(tname,tdef);
	qstr = g_new0(gchar,size);
	
	strcpy(qstr,"CREATE TABLE ");
	strcat(qstr,tname);
	strcat(qstr," ( ");

	for ( tmp = tdef ; tmp != NULL ; tmp = tmp->next){
		
		/* NULL field, no data, get outta here! */
		if ((field = (gchar **)tmp->data) == NULL)
			continue;
		
		fname = field[0];
		ftype = field[1];
		flen  = field[2];
		fnull = field[3];
		fdefa = field[4];
		findx = field[5];
		
		/* Only add a coma if a field has already been 
		 * parsed into the qstr */
		if (i != 0)
			strcat(qstr,", ");
		
		strcat(qstr,fname);
		strcat(qstr," ");
		strcat(qstr,ftype);
		
		if (flen){
			strcat(qstr,"(");
			strcat(qstr,flen);
			strcat(qstr,")");
		}
		strcat(qstr," ");
		
		if (fnull){
			strcat(qstr,fnull);
			strcat(qstr," ");
		}
		
		if (fdefa){
			strcat(qstr,"DEFAULT \'");
			strcat(qstr,fdefa);
			strcat(qstr,"\' ");
		}
		
		if (findx){
			if (!strcmp(findx,"PRIMARY KEY")){
				strcat(qstr,findx);
				strcat(qstr," ");
			}
		}
		
		i++; /* Fields parsed into qstr */
	}

	strcat(qstr,")");
	
	return qstr;
}





/**
 * TableDesignerCreateIndexQuery
 * @tname : The name of the table to be indexed
 * @fname : The name of the field to be indexed
 * @index : The index type (UNIQUE INDEX, INDEX)
 * @iname : The name of the index to create
 *
 * Creates the query needed to create an @index named @iname
 * for the table @tname on the field @fname.
 *
 * Returns the query string.
 */
static gchar *
TableDesignerCreateIndexQuery(gchar *tname,
			      gchar *fname,
			      gchar *index,
			      gchar *iname)
{
	return g_strconcat("CREATE ",index," ",iname," ON ",tname,"(",fname,")",NULL);
}





/**
 * TableDesignerCreateQueryList
 * @tname  : The name for the table
 * @tdef   : The GList of field definitions
 *           (as passed by the designer)
 *
 * Create a GList with all querys needed to create the table
 * and the proper indices defined in the table designer.
 *
 * Returns : A GList of query strings.
 */
static GList *
TableDesignerCreateQueryList(gchar *tname,
			     GList *tdef)
{
	GList *qlist, *tmp;
	gchar *qstr, **field;
	
	/* Main 'CREATE' query */
	qstr = TableDesignerCreateQuery(tname,tdef);
	qlist = g_list_append(NULL,qstr);
	
	for (tmp = tdef ; tmp != NULL ; tmp = tmp->next){
		
		if((field = tmp->data) == NULL)
			continue;
		
		if (!field[5] || !field[0] || !field[6])
			continue;
		
		/* Primary Key's don't have individual queries
		 * they are defined in the main query */
		if (strcmp(field[5],"PRIMARY KEY")){
			qstr = TableDesignerCreateIndexQuery
				(tname,field[0],field[5],field[6]);
			
			qlist = g_list_append(qlist,qstr);
		}
	}
	
	return qlist;
}





/**
 * TableDesignerExecQueryList
 * @conn  : The connection to the postgres backend, we use 
 *          a PGconn instead of a GTransIFaseConn cause there 
 *          is no need to create GTransIFaseResults
 * @qlist : A GList with all the query strings needed to 
 *          satisfy the designer's definition.
 *
 * Start a transaction, execute all the querys and commit the transaction.
 * If a query errors we inmediately ROLLBACK the transaction and return 
 * the error message.
 *
 * Returns: NULL pointer if evething went OK, error message if
 *          we had an error.
 */
static gchar *
TableDesignerExecQueryList(PGconn *conn,
			   GList  *qlist)
{
	GList    *tmp;
	PGresult *res;
	gchar    *errmsg, *qstr;
	

	res = PQexec(conn,"BEGIN");
	if(PGRES_COMMAND_OK != PQresultStatus(res) || res == NULL){
		errmsg = g_strdup(PQresultErrorMessage(res));
		PQclear(res);
		return errmsg;
	}
	PQclear(res);

	for (tmp = qlist ; tmp != NULL ; tmp = tmp->next){
		
		qstr = (gchar *) tmp->data;
		res = PQexec(conn,qstr);
		if(PGRES_COMMAND_OK != PQresultStatus(res) || res == NULL){
			errmsg = g_strdup(PQresultErrorMessage(res));
			PQclear(res);
			res = PQexec(conn,"ROLLBACK");
			PQclear(res);
			return errmsg;
		}
		PQclear(res);
		
	}
	res = PQexec(conn,"COMMIT");
	PQclear(res);
	
	return NULL;
}




/**
 * TableDesignerFreeQueryList
 * @qlist  : The GList with all the querys
 *
 * Free all the memory used by the querys and the list.
 */
static void
TableDesignerFreeQueryList(GList *qlist)
{
	GList *tmp;
	
	for (tmp = qlist ; tmp != NULL ; tmp = tmp->next)
		g_free(tmp->data);
	
	g_list_free(qlist);
}




/**
 * TableDesignerFreeTableDefs
 * @tdef   : The definition of the table as giver by the 
 *           GTransTableDesigner
 *
 * Free all the memory allocated by the table definition.
 */
static void
TableDesignerFreeTableDefs(GList *tdef)
{
	GList  *tmp;
	gchar **field;
	gint    i;

	for (tmp = tdef ; tmp != NULL ; tmp = tmp->next){

		if ((field = tmp->data) == NULL)
			continue;
		
		/* For all the arrtibutes (5) + field name + field size = 7*/
		for ( i = 0 ; i < 7 ; i++)
		       	if (field[i])
				g_free(field[i]);
		g_free(field);
	}
	g_list_free(tdef);
}




/**
 * TableDesignerCreateTable
 * @conn    : The GTransIFaseConn connection to a database backend
 * @tname   : The name for the table
 * @tdef    : The linked list string arrays (field definitions).
 *
 * Check for errors in the field definition, if everything
 * it ok the table is created and NULL is returned, 
 * otherwise the error message is returned.
 * This funcion is mandatory for the plugin definition.
 * This is one of the functions pointed at by a member of the conn.
 */
static gchar *
TableDesignerCreateTable(GTransIFaseConn  *conn,
			 gchar            *tname,
			 GList            *tdef)
{
	GList     *qlist;
	gchar     *errmsg;
		
        errmsg = TableDesignerValidateTable(tname,tdef);
	if (errmsg)
		return errmsg;
	
	
	qlist = TableDesignerCreateQueryList(tname,tdef);
	errmsg = TableDesignerExecQueryList((PGconn *)conn->conn,qlist);
	TableDesignerFreeQueryList(qlist);
	TableDesignerFreeTableDefs(tdef);

	if (errmsg)
		return errmsg;

	return NULL;
}







/************************************************************************
 * Exec Stuff                                                           *
 ************************************************************************/




/**
 * Exec:
 * @conn  : GTransIFaseConn to a PostgreSQL server
 * @qstr  : SQL query string.
 *
 * Execute a SQL query in the DB server.
 * Return : A GTransIFaseResult.
 */
static GTransIFaseResult * Exec(GTransIFaseConn *conn,
				gchar           *qstr)
{
	GTransIFaseResult *gt_res;
	PGresult          *pg_res; 
	gint               i,j;	


	/* Allocate memory for the GTransIFaseResult result */
	gt_res = gtrans_ifase_result_alloc ();
	
	pg_res = PQexec ((PGconn *)conn->conn, qstr);
	
	/* If the thing errored */
	if ((PGRES_COMMAND_OK != PQresultStatus(pg_res) &&
	     PGRES_TUPLES_OK  != PQresultStatus(pg_res)) ||
	    pg_res == NULL){
		
		gt_res->errormsg = g_strdup(PQresultErrorMessage(pg_res));
		gt_res->status = GTRANS_IFASE_RESULT_ERROR;
		PQclear (pg_res);
		return gt_res;
	}
	
	/* A command returning no tupples */
	if(PGRES_COMMAND_OK == PQresultStatus(pg_res)){
		PQclear (pg_res);
		return gt_res;
	}
	
	
	/* A command returning tupples */
	
	gt_res->type   = GTRANS_IFASE_RESULT_TUPPLES;
	gt_res->n_rows = PQntuples(pg_res);
	gt_res->n_cols = PQnfields(pg_res);
	
	gt_res->fields = g_new0(gchar *,gt_res->n_cols + 1);
	
	/* Copy the field names */
	for (j = 0 ; j < gt_res->n_cols ; j++)
		gt_res->fields[j] = g_strdup(PQfname(pg_res,j));
	gt_res->fields[j] = NULL;

	
	/* Copy the tupples returned */
	for (i = 0 ; i < gt_res->n_rows ; i++){
		
		gchar **row;
		
		row = g_new0(gchar *,gt_res->n_cols);
		
		for (j = 0 ; j < gt_res->n_cols ; j++)
			if (PQgetisnull(pg_res,i,j))
				row[j] = NULL;
			else
				row[j] = g_strdup(PQgetvalue(pg_res,i,j));
		
		gt_res->rows = g_list_append(gt_res->rows,row);
	}
	gt_res->rows = g_list_append(gt_res->rows,NULL);      
	
	/* Free */
	PQclear (pg_res);
	
	return gt_res;
}




/**
 * ListTables:
 * @conn  : GTransIFaseConn to a PostgreSQL server
 *
 * Return : A GTransIFaseResult with the available tables
 * in the database.
 */
static GTransIFaseResult *ListTables (GTransIFaseConn *conn)
{
	g_return_val_if_fail(conn != NULL,NULL);
	
	return Exec (conn,"SELECT tablename FROM pg_tables");
}





/**
 * ListDb:
 * @conn  : GTransIFaseConn to a PostgreSQL server
 *
 * Return : A GTransIFaseResult with the availble databases
 * in the server.
 */
static GTransIFaseResult *ListDb (GTransIFaseConn *conn)
{
	g_return_val_if_fail(conn != NULL,NULL);
	
	return Exec (conn,"SELECT datname FROM pg_database");
}



/**********************************************************************
 * Connection stuff                                                   *
 **********************************************************************/


/**
 * Disconnect:
 * @conn  : GTransIFaseConn to a PostgreSQL server
 *
 * Finich the connection to the server.
 * Also free the GTransIFaseConn.
 */
static void Disconnect (GTransIFaseConn *conn)
{
	g_return_if_fail (conn != NULL);

	PQfinish ((PGconn *)conn->conn);

	if ( conn->host )
		g_free (conn->host);
	if ( conn->db != NULL )
		g_free (conn->db);
	if ( conn->user )
		g_free (conn->user);
	if ( conn->pwd )
		g_free (conn->pwd);
	if ( conn->port )
		g_free (conn->port);
	
	g_free(conn);
}




/**
 * Status:
 * @conn  : GTransIFaseConn to a PostgreSQL server
 *
 * Return 1 if there has been an error with the connection.
 * 0 otherwise.
 */
static gint Status (GTransIFaseConn *conn)
{
	g_return_val_if_fail (conn != NULL, 1);
	
	if( conn->conn != NULL )
		if (CONNECTION_OK == PQstatus ((PGconn *)conn->conn))
			return 0;
	
	return 1;
}






/**
 * ErrorMsg:
 * @conn : GTransIFaseConn to a PostgreSQL server
 *
 * Return the error string of the last error that 
 * happened with the connection.
 */
static gchar *ErrorMsg (GTransIFaseConn *conn)
{
	if (conn == NULL)
		return g_strdup (_("Fatal Error: NULL Connection"));
	
	if (conn->conn == NULL)
		return g_strdup (_("Fatal Error: Connection Impossible"));
	
	return PQerrorMessage ((PGconn*)conn->conn);
}





/**
 * Connect :
 * @host : server where the database backend is running
 * @port : port where the database backend is running
 * @db   : name of the database
 * @user : user name to establish a connection
 * @pwd  : password for the user
 *
 * Establish a connection to a PostgreSQL database server.
 * Returns : A GTransIFaseConn.
 */
GTransIFaseConn *Connect (gchar   *host,
			  gchar   *port,
			  gchar   *db,
			  gchar   *user,
			  gchar   *pwd)
{

	GTransIFaseConn *conn;
	
	conn = gtrans_ifase_conn_alloc();

	if (host)
		conn->host = g_strdup (host);				
	if (port)
		conn->port = g_strdup (port);
	if (db)
		conn->db = g_strdup (db);
	if (user)
		conn->user = g_strdup (user);
	if (pwd)
		conn->pwd = g_strdup (pwd); 

	(PGconn *)(conn->conn) = PQsetdbLogin (host, 
					       port, 
					       NULL, NULL, 
					       db, 
					       user, 
					       pwd);

	
	conn->Connect = NULL;
	conn->Exec = Exec;
	conn->Disconnect = Disconnect;
	conn->Status = Status;
	conn->ErrorMsg = ErrorMsg;
	conn->ListDb = ListDb;
	conn->ListTables = ListTables;
	
	/* Table Designer */
	conn->TableDesignerAttrNames   = TableDesignerAttrNames;
	conn->TableDesignerAttrVals    = TableDesignerAttrVals;
	conn->TableDesignerFieldTypes  = TableDesignerFieldTypes;
	conn->TableDesignerTypeMasks   = TableDesignerTypeMasks;
	conn->TableDesignerCreateTable = TableDesignerCreateTable;
	
	return conn;
}
