// Copyright (c) 2000-2001 Brad Hughes <bhughes@trolltech.com>
//
// Use, modification and distribution is allowed without limitation,
// warranty, or liability of any kind.
//

#ifndef __mainvisual_h
#define __mainvisual_h

class SpectrumAnalyzer;

#include "visual.h"
#include "logscale.h"

#include <qwidget.h>
#include <qarray.h>
#include <qpixmap.h>
#include <qptrlist.h>

#ifdef FFTW
#include <rfftw.h>
#include <fftw.h>
#endif // FFTW

class Buffer;
class Prefs;
class Output;
class VisualNode;

class QTimer;
class QSettings;


class VisualNode
{
public:
    VisualNode(short *l, short *r, unsigned long n, unsigned long o)
	: left(l), right(r), length(n), offset(o)
    {
	// left and right are allocated and then passed to this class
	// the code that allocated left and right should give up all ownership
    }

    ~VisualNode()
    {
	delete [] left;
	delete [] right;
    }

    short *left, *right;
    long length, offset;
};

class VisualBase
{
public:
    // return true if the output should stop
    virtual bool process( VisualNode *node ) = 0;
    virtual void draw( QPainter *, const QColor & ) = 0;
    virtual void resize( const QSize &size ) = 0;
    virtual void configChanged( QSettings & ) = 0;
};

// base class to handle things like frame rate...
class MainVisual : public QWidget, public Visual
{
    Q_OBJECT

public:
    MainVisual( QWidget *parent = 0, const char * = 0 );
    virtual ~MainVisual();

    static Prefs *createPrefs( const QString &visualname,
			       QWidget *parent, const char *name );

    VisualBase *visual() const { return vis; }
    void setVisual( VisualBase *newvis );
    void setVisual( const QString &visualname );

    void add(Buffer *, unsigned long, int, int);
    void prepare();

    void configChanged(QSettings &settings);

    QSize minimumSizeHint() const { return sizeHint(); }
    QSize sizeHint() const { return QSize(4*4*4*2, 3*3*3*2); }

    void paintEvent( QPaintEvent * );
    void resizeEvent( QResizeEvent * );
    void customEvent( QCustomEvent * );

    static QStringList visuals();

    void setFrameRate( int newfps );
    int frameRate() const { return fps; }

public slots:
void timeout();

private:
    VisualBase *vis;
    QPixmap pixmap;
    QPtrList<VisualNode> nodes;
    QTimer *timer;
    bool playing;
    int fps;
};

class StereoScope : public VisualBase
{
public:
    StereoScope();
    virtual ~StereoScope();

    void resize( const QSize &size );
    void configChanged(QSettings &settings);
    bool process( VisualNode *node );
    void draw( QPainter *p, const QColor &back );

protected:
    QColor startColor, targetColor;
    QMemArray<double> magnitudes;
    QSize size;
    bool rubberband;
    double falloff;
    int fps;
};

class MonoScope : public StereoScope
{
public:
    MonoScope();
    virtual ~MonoScope();

    bool process( VisualNode *node );
    void draw( QPainter *p, const QColor &back );
};

#ifdef FFTW
class StereoAnalyzer : public VisualBase
{
public:
    StereoAnalyzer();
    virtual ~StereoAnalyzer();

    void resize( const QSize &size );
    void configChanged(QSettings &settings);
    bool process( VisualNode *node );
    void draw( QPainter *p, const QColor &back );

protected:
    QColor startColor, targetColor;
    QMemArray<QRect> rects;
    QMemArray<double> magnitudes;
    QSize size;
    LogScale scale;
    double scaleFactor, falloff;
    int analyzerBarWidth, fps;
    rfftw_plan plan;
    fftw_real lin[512], rin[512], lout[1024], rout[1024];
};

class MonoAnalyzer : public StereoAnalyzer
{
public:
    MonoAnalyzer();
    virtual ~MonoAnalyzer();

    bool process( VisualNode *node );
    void draw( QPainter *p, const QColor &back );
};

class StereoTopograph : public VisualBase
{
public:
    StereoTopograph();
    virtual ~StereoTopograph();

    void resize( const QSize &size );
    void configChanged(QSettings &settings);
    bool process( VisualNode *node );
    void draw( QPainter *p, const QColor &back );

protected:
    QColor targetColor;
    QMemArray<double> magnitudes;
    QSize size;
    LogScale scale;
    bool clear;
    double scaleFactor, falloff;
    int blockwidth, fps;
    rfftw_plan plan;
    fftw_real lin[512], rin[512], lout[1024], rout[1024];
};

class MonoTopograph : public StereoTopograph
{
public:
    MonoTopograph();
    virtual ~MonoTopograph();

    bool process( VisualNode *node );
    void draw( QPainter *p, const QColor &back );
};

class StereoSpectroscope : public VisualBase
{
public:
    StereoSpectroscope();
    virtual ~StereoSpectroscope();

    void resize( const QSize &size );
    void configChanged(QSettings &settings);
    bool process( VisualNode *node );
    void draw( QPainter *p, const QColor &back );

protected:
    QColor leftColor, rightColor;
    QSize size;
    double edges, cushion, scale;
    fftw_plan plan;
    fftw_complex lin[512], rin[512], lout[1024], rout[1024];
    fftw_real lmag[512],rmag[512],lang[512],rang[512];
    fftw_real angle, rotspeed;
};

class MonoSpectroscope : public StereoSpectroscope
{
public:
    MonoSpectroscope();
    ~MonoSpectroscope();

    bool process( VisualNode *node );
    void draw( QPainter *p, const QColor &back);
};

#endif // FFTW

#endif // __mainvisual_h
