/*
 *  Copyright (C) 2001 Jorn Baayen 
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "galeon.h"
#include "embed.h"
#include "dialog.h"
#include "mozilla.h"
#include "element_props.h"
#include "glade.h"
#include "misc_gui.h"

#include <unistd.h>
#include <string.h>
#include <stdlib.h>
#include <gtk/gtkpixmap.h>
#include <libgnome/libgnome.h>
#include <libgnome/gnome-i18n.h>
#include <libgnomeui/gnome-app.h>
#include <libgnomeui/gnome-app-helper.h>
#include <libgnomeui/gnome-popup-menu.h>
#include <libgnomeui/gnome-stock.h>
#include <glade/glade-xml.h>

typedef struct
{
	GtkWidget *dialog;
	GtkWidget *pixmap;
	GaleonEmbed *embed;
	gchar *imgurl;
} ElementPropsDialog;

/* function prototypes */
void element_properties_close_clicked_cb (GtkWidget *widget,
					  ElementPropsDialog *dialog);
void element_properties_destroy_cb (GtkWidget *widget,
				    ElementPropsDialog *dialog);
gboolean element_longdesc_button_released_cb (GtkWidget *widget,
				              GdkEventButton *event,
				              ElementPropsDialog *dialog);
gboolean element_thumbnail_button_press_event_cb (GtkWidget *widget,
					          GdkEventButton *event,
					          ElementPropsDialog *dialog);

/**
 * element_props_show_dialog: show element props dialog
 */
void
element_props_show_dialog (GaleonEmbed *embed, WrapperContextInfo *info)
{
	GladeXML *gxml;
	ElementPropsDialog *dialog = g_new0 (ElementPropsDialog, 1);
	GtkWidget *tmp;
	gchar *s;
	gint t;

	/* build dialog from glade file */
	gxml = glade_widget_new ("galeon.glade", 
				 "element_properties_dialog",
				 &(dialog->dialog), dialog);

	dialog->embed = embed;

	if (info->context & CONTEXT_LINK)
	{
		tmp = glade_xml_get_widget (gxml, "element_link_page");
		gtk_widget_show (tmp);

		tmp = glade_xml_get_widget (gxml, "element_address_entry");
		if (info->link)
			gtk_entry_set_text (GTK_ENTRY (tmp), info->link);
	
		tmp = glade_xml_get_widget (gxml, "element_open_in_entry");
		if (info->linktarget)
			gtk_entry_set_text (GTK_ENTRY (tmp), info->linktarget);
	
		tmp = glade_xml_get_widget (gxml, "element_target_type_entry");	
		if (info->linktype)
			gtk_entry_set_text (GTK_ENTRY (tmp), info->linktype);
	
		tmp = glade_xml_get_widget (gxml, "element_target_language_entry");
		if (info->linklang)
			gtk_entry_set_text (GTK_ENTRY (tmp), info->linklang);
	
		tmp = glade_xml_get_widget (gxml, "element_relation_entry");
		if (info->linkrel)
			gtk_entry_set_text (GTK_ENTRY (tmp), info->linkrel);
	
		tmp = glade_xml_get_widget (gxml, "element_reversed_entry");
		if (info->linkrev)
			gtk_entry_set_text (GTK_ENTRY (tmp), info->linkrev);

		tmp = glade_xml_get_widget (gxml, "element_title_entry");
		if (info->linkrev)
			gtk_entry_set_text (GTK_ENTRY (tmp), info->linktitle);
	
		tmp = glade_xml_get_widget (gxml, "element_text_entry");	
		if (info->linktext)
			gtk_entry_set_text (GTK_ENTRY (tmp), info->linktext);
	}

	if (info->context & CONTEXT_IMAGE)
	{
		SetPixmapInfo *spinfo;

		tmp = glade_xml_get_widget (gxml, "element_image_page");
		gtk_widget_show (tmp);
	
		tmp = glade_xml_get_widget (gxml, "element_location_entry");
		if (info->img)
			gtk_entry_set_text (GTK_ENTRY (tmp), info->img);
	
		tmp = glade_xml_get_widget (gxml, "element_width_entry");
		s = g_strdup_printf ("%d", info->imgwidth);
		gtk_entry_set_text (GTK_ENTRY (tmp), s);
		g_free (s);
	
		tmp = glade_xml_get_widget (gxml, "element_height_entry");
		s = g_strdup_printf ("%d", info->imgheight);
		gtk_entry_set_text (GTK_ENTRY (tmp), s);
		g_free (s);

		tmp = glade_xml_get_widget (gxml, "element_img_title_entry");
		if (info->imgalt)
			gtk_entry_set_text (GTK_ENTRY (tmp), info->imgtitle);
	
		tmp = glade_xml_get_widget (gxml, "element_alt_entry");
		if (info->imgalt)
			gtk_entry_set_text (GTK_ENTRY (tmp), info->imgalt);

		tmp = glade_xml_get_widget (gxml, "element_longdesc_button");
		gtk_widget_set_sensitive (tmp, (info->imglongdesc != NULL));
		if (info->imglongdesc != NULL)
		{
			gtk_object_set_data_full (GTK_OBJECT (tmp), "url",
						  g_strdup (info->imglongdesc),
						  g_free);
		}

		tmp = glade_xml_get_widget (gxml, "element_thumbnail_eventbox");
		gtk_object_set_data_full (GTK_OBJECT (tmp), "url",
					  g_strdup (info->img), g_free);
		if (info->imgtitle && strlen (info->imgtitle) > 0)
		{
			gtk_object_set_data_full (GTK_OBJECT (tmp), "title",
						  g_strdup (info->imgtitle),
						  g_free);
		}
		else
		{
			gtk_object_set_data_full (GTK_OBJECT (tmp), "title",
						  g_strdup (info->imgalt),
						  g_free);
		}
		gtk_signal_connect (GTK_OBJECT (tmp),
				    "button_press_event",
				    GTK_SIGNAL_FUNC
				    	(element_thumbnail_button_press_event_cb),
				    dialog);

		dialog->pixmap =
			glade_xml_get_widget (gxml, "element_thumbnail_pixmap");
		gtk_widget_hide (dialog->pixmap);
		s = g_strconcat (g_get_tmp_dir(),
				 "/galeon-element-props-XXXXXX", NULL);
		if ((t = mkstemp (s)) < 0)
		{
			g_free (s);
			return;
		}
		close (t);

		dialog->imgurl = g_strdup (info->img);
		spinfo = g_new0 (SetPixmapInfo, 1);
		spinfo->url = dialog->imgurl;
		spinfo->pixmap = dialog->pixmap;
		spinfo->selected_url = dialog->imgurl;

		/* save to tmp location and update pixmap */
		mozilla_save_url (NULL, info->img, s, ACTION_SET_PIXMAP,
				  (gpointer) spinfo);

		g_free (s);
		
	}

	gtk_widget_show (dialog->dialog);
	dialog_set_parent (dialog->dialog, embed->parent_window->wmain);
	gtk_object_unref (GTK_OBJECT (gxml));
}

/**
 * element_properties_close_clicked_cb: close clicked
 */
void
element_properties_close_clicked_cb (GtkWidget *widget,
				     ElementPropsDialog *dialog)
{
	gtk_widget_destroy (dialog->dialog);
}

/**
 * element_properties_destroy_cb: dialog destroyed
 */
void
element_properties_destroy_cb (GtkWidget *widget,
			       ElementPropsDialog *dialog)
{
	dialog->pixmap = NULL;
	g_free (dialog->imgurl);
	dialog->imgurl = NULL;
	g_free (dialog);
}

/**
 * element_longdesc_button_released_cb: "View long desc" button clicked
 */
gboolean
element_longdesc_button_released_cb (GtkWidget *widget,
				     GdkEventButton *event,
				     ElementPropsDialog *dialog)
{
	GaleonEmbed *embed = dialog->embed;

	return_if_not_sane_click (widget, event);

	if (event->button == 3)
	{
		return element_thumbnail_button_press_event_cb 
			(widget, event, dialog);
	}
	else
	{
		gchar *url = (gchar *)
			gtk_object_get_data (GTK_OBJECT (widget), "url");

		if (!url) return FALSE;
		if (!embed_exists (embed)) embed = NULL;

		embed_activate_link_mouse (embed, NULL, url, event);
	}

	return TRUE;
}

/**
 * element_thumbnail_button_press_event_cb: present a nice contextmenu
 * allowing you to save the selected url
 */
gboolean
element_thumbnail_button_press_event_cb (GtkWidget *widget,
					 GdkEventButton *event,
					 ElementPropsDialog *dialog)
{
	gchar *url, *title;
	GaleonEmbed *embed = dialog->embed;

	url = (gchar *) gtk_object_get_data (GTK_OBJECT (widget), "url");
	title = (gchar *) gtk_object_get_data (GTK_OBJECT (widget), "title");

	if (!url) return FALSE;
	if (!embed_exists (embed)) embed = NULL;

	/* contextmenu */
	if (event->button == 3)
	{
		GtkWidget *menu;
		gint action;

		/* context menu */
		static GnomeUIInfo menu_uiinfo[] =
		{
			GNOMEUIINFO_ITEM_STOCK (N_("Open in a new window"),
						NULL, NULL, GNOME_STOCK_MENU_NEW),
			GNOMEUIINFO_ITEM_STOCK (N_("Open in a new tab"),
						NULL, NULL, GNOME_STOCK_MENU_NEW),
			GNOMEUIINFO_ITEM_STOCK (N_("Save as..."),
						NULL, NULL, GNOME_STOCK_MENU_SAVE_AS),
			GNOMEUIINFO_ITEM_STOCK (N_("Copy location"),
						NULL, NULL, GNOME_STOCK_MENU_COPY),
			GNOMEUIINFO_ITEM_STOCK (N_("Add bookmark"),
						NULL, NULL, GNOME_STOCK_MENU_BLANK),
			GNOMEUIINFO_END
		};
	
		/* show context menu */
		menu = misc_gui_new_popup_menu_lock_accels (menu_uiinfo);
		action = gnome_popup_menu_do_popup_modal 
			(GTK_WIDGET (menu), NULL, NULL, event, NULL);

		/* do appropiate action */
		switch (action)
		{
		case 0:
			embed_activate_link (embed, NULL,
					     url, LINKSTATE_NEWWIN);
			break;
		case 1:
			embed_activate_link (embed, NULL,
					     url, LINKSTATE_NEWTAB);
			break;
		case 2:
			/* pass NULL since we dont want the saved
			 * message to show up in the embed */
			embed_save_image (NULL, url, TRUE);
			break;
		case 3:
			if (embed == NULL && all_embeds)
				embed = (GaleonEmbed *)(all_embeds->data);
			if (embed != NULL)
				embed_copy_text_to_clipboard (url, embed);
			break;
		case 4:
			bookmarks_add_bookmark (title, url, NULL,
					GTK_WINDOW (dialog->dialog), 0);
			break;
		default:
			break;
		}

		/* destroy the popup menu */
		gtk_widget_unref (menu);
		
		return TRUE;
	}

	return FALSE;
}
