/*
 * $Id: connections.c,v 0.74 2002/03/29 22:29:48 ceder Exp $
 * Copyright (C) 1991-2002  Lysator Academic Computer Association.
 *
 * This file is part of the LysKOM server.
 * 
 * LysKOM is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by 
 * the Free Software Foundation; either version 1, or (at your option) 
 * any later version.
 * 
 * LysKOM is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with LysKOM; see the file COPYING.  If not, write to
 * Lysator, c/o ISY, Linkoping University, S-581 83 Linkoping, SWEDEN,
 * or the Free Software Foundation, Inc., 675 Mass Ave, Cambridge, 
 * MA 02139, USA.
 *
 * Please mail bug reports to bug-lyskom@lysator.liu.se. 
 */
/*
 * connections.c
 *
 * Denna fil inneh}ller niv}n ovanf|r isc.
 *
 * Created by Willf|r 31/3-90. Mostly written by ceder.
 */


#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <errno.h>
#include <stdio.h>
#include <setjmp.h>
#ifdef HAVE_STRING_H
#  include <string.h>
#endif
#include <sys/types.h>
#include <sys/stat.h>
#ifdef TIME_WITH_SYS_TIME
#include <sys/time.h>
#include <time.h>
#else
#ifdef HAVE_SYS_TIME_H
#include <sys/time.h>
#else
#include <time.h>
#endif
#endif
#include <sys/socket.h>
#include <signal.h>

#include "ldifftime.h"
#include "misc-types.h"
#include "s-string.h"
#include "kom-types.h"
#include "kom-memory.h"
#include "debug.h"
#include "isc-interface.h"
#include "com.h"
#include "async.h"
#include "connections.h"
#include "internal-connections.h"
#include "prot-a-parse-arg.h"
#include "log.h"
#include "lyskomd.h"
#include "services.h"
#include "isc-parse.h"
#include "prot-a.h"
#include "prot-a-parse.h"
#include "server/smalloc.h"
#include "end-of-atomic.h"
#include "send-async.h"
#include "cache.h"
#include "rfc931.h"
#include "param.h"
#include "kom-config.h"
#include "kom-errno.h"
#include "sigflags.h"
#include "server-time.h"
#include "aux-items.h"

IscMaster      * kom_server_mcb    = NULL;
Connection     * active_connection = NULL;

/*
 * This is set TRUE when the server should be closed. It is checked
 * each time around the main loop. It is set if someone with enough
 * privileges issues a `shutdown', or of lyskomd receives a SIGHUP.
 * This not an abort: all data is saved before we exit.
 */
volatile sig_atomic_t go_and_die = 0;

/*
 * The lyskomd server will dump some statistics when it receives a
 * SIGUSR1 signal.  The signal handler sets the do_statistics flag.
 */
volatile sig_atomic_t do_statistics = 0;

jmp_buf 	 parse_env;


const Fnc_descriptor fnc_defs[]={
#include "fnc-def-init.incl"
};

const unsigned long num_fnc_defs = sizeof (fnc_defs) / sizeof (Fnc_descriptor);

unsigned long service_statistics[sizeof (fnc_defs) / sizeof (Fnc_descriptor)];

BUGDECL;



static void
logout_client(Connection *cp)
{
    Connection *real_active_connection;
    int ret;

    if ( active_connection != NULL )
    {
	kom_log("BUGCHK: logout_client(%ld): connection %ld is active.\n",
	    cp->session_no, active_connection->session_no);
    }
    
    switch (cp->magic)
    {
    case CONN_MAGIC_ALLOC:
        break;

    case CONN_MAGIC_FREE:
	kom_log("LOGOUT_CLIENT: Trying to free freed Connection - ignored!\n");
	return;

    default:
	restart_kom("LOGOUT_CLIENT: Bad magic number\n");
    }

    if ( cp->pers_no != 0 )
    {
	int ctr = 0;

	if (active_connection != NULL)
	{
	    kom_log("WNG: logout_client(): active_connection != NULL\n");
	    if (ctr < 100)
		ctr++;
	    else
		kom_log("WNG: won't log the above message more\n");
	}

	real_active_connection = active_connection;
	active_connection = cp;
	logout();
	active_connection = real_active_connection;
    }
    else
    {
#if 0
	/* FIXME (bug 93): send a new async here instead.  This causes the
	   elisp client to say that a secret (or unknown) person has
	   left the system.  */
        async_logout( 0, cp->session_no );
#endif
    }

    switch(cp->protocol)
    {
    case 0:			/* Hasn't yet allocated any protocol. */
	break;
	
    case 'A':
	prot_a_destruct(cp);
	break;
	
    default:
	restart_kom("logout_client(): Bad protocol.\n");
    }

    ret = isc_destroy(kom_server_mcb, cp->isc_session);
    if (ret < 0)
	kom_log("logout_client(): isc_destroyed returned %d\n", ret);
    cp->isc_session = NULL;

    kill_client(cp);		/* Free the Connection */
}

/*
 * This function is part of the shutdown tidy-up sequence.
 */
void
logout_all_clients(void)
{
    Session_no sess = 0;
    Connection *conn;
    
    while ( (sess = traverse_connections (sess)) != 0)
    {
	conn = get_conn_by_number (sess);

	if ( conn == NULL )
	    restart_kom("logout_all_clients(): cant get session %ld.\n",
			sess);
	else
	    logout_client (conn);
    }

    if ( traverse_connections (0) != 0)
	restart_kom("logout_all_clients(): traverse_connections(0) == %ld.\n",
		    traverse_connections(0));
}

    
    

    
/*
 * Call a function in services.c. A pointer to the result is returned.
 * The pointer points to static data which is overwritten on each call.
 */
static Success
call_function(Connection *client,
	      union result_holder *res)
{
    Success	status=FAILURE;	/* OK if the call was successful. */

    if ( active_connection != NULL )
    {
	kom_log("call_function(%ld): active_connection = %ld\n",
	    client->session_no, active_connection->session_no);
    }

    if (client->function == illegal_fnc) 
    {
        err_stat = 0;
	kom_errno = KOM_NOT_IMPL;
	return FAILURE;
    }

    active_connection = client;

    service_statistics[client->function_index]++;

#include "call-switch.incl"

    active_connection = NULL;

    return status;
}


static void
parse_packet(Connection *client)
{
    if ( client->protocol == '\0' ) /* Not known yet. */
    {
	client->protocol = parse_char(client);
	switch(client->protocol)
	{
	case 'A':
	    prot_a_init(client);
	    break;

	default:
	    client->protocol = '\0';
	    isc_puts("%%LysKOM unsupported protocol.\n", client->isc_session);
	    isc_flush(client->isc_session);
	    BUG(("%%%%Unsupported protocol.\n"));
	    longjmp(parse_env, ISC_LOGOUT);
	}
    }

    switch(client->protocol)
    {
    case 'A':
	prot_a_parse_packet(client);
	break;

    default:
	restart_kom("parse_packet(): Bad protocol.\n");
	break;
    }
}

/*
 * Free all parsed areas which are no longer needed. Re-initialize all
 * parse_pos fields so that the parse will expect a new function.
 *
 * This function is called
 *	when a parse error occurs
 *	when a parse is complete and the function has executed.
 */
static void
free_parsed(Connection *client)
{
    s_clear(&client->c_string0);
    s_clear(&client->c_string1);
    client->string0 = EMPTY_STRING; /* So that no one frees it. */
    sfree(client->misc_info_list.misc);
    client->misc_info_list.misc = 0;
    client->misc_info_list.no_of_misc = 0;
    s_clear(&client->aux_item.data);
    s_clear(&client->dummy_aux_item.data);
    sfree( client->c_local_text_no_p);
    client->c_local_text_no_p = NULL;
    client->parse_pos = 0;
    client->fnc_parse_pos = 0;
    client->array_parse_index = 0;
    client->array_parse_parsed_length = 0;
    client->array_parse_pos = 0;
    client->struct_parse_pos = 0;
    client->string_parse_pos = 0;
    client->hunt_parse_pos = 0;
    client->array_hunt_num = 0;
    client->array_hunt_depth = 0;
    sfree(client->num_list.data);
    client->num_list.data = NULL;
    client->num_list.length = 0;
    free_aux_item_list(&client->aux_item_list);
    client->info.highest_aux_no = 0;
}

/*
 * Send a reply to a call.
 */
static void
reply(Connection *client,
      Success status,
      union result_holder *result)
{
    switch(client->protocol)
    {
    case 'A':
	prot_a_reply(client, status, result);
	break;

    default:
	restart_kom("reply(): Bad protocol.\n");
	break;
    }
}


/*
 * Try to parse enough data from client->unparsed to call a function.
 * If more data is needed set client->more_to_parse to FALSE.
 */
static void
parse_unparsed(Connection *client)
{
    String        tmp_str = EMPTY_STRING;
    Success       status;
    union result_holder result;
        
    switch ( setjmp(parse_env) )
    {
    case 0 :
	/* Parse message. If message is complete call function and reply. */
	parse_packet(client);
	status = call_function(client, &result);
	reply(client, status, &result);
	free_parsed(client);
	end_of_atomic(FALSE);
	break;

    case ISC_PROTOCOL_ERR:
	s_clear(&client->string0);
	free_parsed(client);
	isc_puts("%% LysKOM protocol error.\n", client->isc_session);
	isc_flush(client->isc_session);
	BUG(("%%%% Protocol error.\n"));
	s_clear(&client->unparsed);
	client->first_to_parse = 0;
	client->more_to_parse = FALSE;
	end_of_atomic(FALSE);
	break;

    case ISC_MSG_INCOMPLETE:
	client->more_to_parse = FALSE;
	break;

    case ISC_LOGOUT:
	add_to_kill_list(client);
	break;
    }
    
    /* Delete the parsed part of 'unparsed' */
    
    if ( s_substr(&tmp_str, client->unparsed,
		  client->first_to_parse, END_OF_STRING) != OK )
	restart_kom("parse_unparsed: s_substr\n");

    s_clear(&client->unparsed);
    client->unparsed = tmp_str;
    client->first_to_parse = 0;
}


/*
 * There is a message in the event. Parse it.
 */
static void
parse_message(Connection *cp, String tmp_str)
{
    VBUGSTR(tmp_str);

    cp->more_to_parse = TRUE;
    
    if ( s_strcat(&cp->unparsed, tmp_str) != OK )
	restart_kom("parse_message(): s_strcat\n");

    tmp_str = EMPTY_STRING;
    
    /* Parse this packet, but leave the last token if it isn't complete. */

    parse_unparsed(cp);

    return;
}


/*
 * parse data in client->unparsed if there
 * is data that has not yet been taken care of. Return TRUE
 * if there was no data to take care of.
 */
static Bool
parse_forgotten(void)
{
    Bool flg = FALSE;			/* Was there anything? */
    Connection *ci;
    Session_no session_no = 0;

    if ( traverse_connections(0) == 0 )
	return TRUE;

    while ( ( session_no = traverse_connections( session_no ) ) != 0 )
    {
	ci = get_conn_by_number (session_no);

	if ( ci->more_to_parse )
	{
	    parse_unparsed(ci);
	    flg = TRUE;
	}
    }

    return flg ? FALSE : TRUE;
}


/* Return 1 if the named file exists, 0 otherwise */
static int
fexists(const char *filename)
{
    struct stat buf;
    int code;
  
    code = !stat(filename, &buf);
    errno = 0;

    return code;
}


void
dump_statistics(void)
{
    static time_t last_dump = NO_TIME;
    int i;
    FILE *fp;

    if ((fp = fopen(param.statistic_name, "a")) == NULL)
    {
	kom_log("%s: dump_statistics(): can't open file %s\n",
	    __FILE__,
	    param.statistic_name);
	return;
    }

    if (last_dump == NO_TIME)
    {
	fprintf(fp, "RESTART\n");
	last_dump = current_time;
    }
    
    fprintf(fp, "TIME: %s", ctime(&current_time));
    fprintf(fp, "SECONDS: %d\n", (int)ldifftime(current_time, last_dump));
    fprintf(fp, "STATISTICS:");

    /* The last entry corresponds to the dummy entry that is used to
       skip arguments to unimplemented requests.  Skip that, since it
       contains no useful statistics.  */
    for (i = 0; i < num_fnc_defs - 1; i++)
    {
	fprintf(fp, " %d:%lu", fnc_defs[i].function, service_statistics[i]);
	service_statistics[i]=0;
    }

    fprintf(fp, "\n");
    fclose(fp);

    last_dump = current_time;
}

/* List of connections to kill. */

Session_no *kill_list = NULL;
int kill_list_size = 0;

/* Schedule this client for termination. */
void
add_to_kill_list(Connection *conn)
{
    int i;

    for (i = 0; i < kill_list_size; i++)
	if (kill_list[i] == conn->session_no)
	{
	    kom_log("add_to_kill_list(%ld): already present as %d of %d.\n",
		conn->session_no, i, kill_list_size);
	    return;
	}

    if (kill_list == NULL)
    {
	if (kill_list_size != 0)
	    restart_kom("add_to_kill_list(): size = %d\n", kill_list_size);

	kill_list_size = 1;
	kill_list = smalloc(sizeof(Session_no));
    }
    else
    {
	kill_list_size++;
	kill_list = srealloc(kill_list, kill_list_size * sizeof(Session_no));
    }

    kill_list[kill_list_size-1] = conn->session_no;
}

/*
 * check_kill_flg must NEVER be called inside an atomic call!
 */
static void
check_kill_flg(void)
{
    Connection *conn;

    if ( active_connection != NULL )
    {
	restart_kom("check_kill_flg: active_connection == %ld",
		    active_connection->session_no);
    }

    while (kill_list_size > 0)
    {
	--kill_list_size;
	conn = get_conn_by_number (kill_list[kill_list_size]);
	if (conn == NULL)
	{
	    kom_log("check_kill_flg(): Connection %ld doesn't exist.\n",
		kill_list[kill_list_size]);
	}
	else
	{
	    logout_client (conn);
	    end_of_atomic (FALSE);
	}
    }

    if (kill_list != NULL)
    {
	sfree (kill_list);
	kill_list = NULL;
    }
}

static void
login_request(IscEvent  *event)
{
    Connection  * cp;
    const char *realuser;
    char *hostname = NULL;

    /* Supress logins if /etc/nologin exists */
    if (fexists(param.nologin_file))
    {
	isc_puts("%% No logins allowed.\n", event->session);
	isc_flush(event->session);
 	isc_destroy(kom_server_mcb, event->session);
	return;
    }

    hostname = isc_gethostname(event->session->info.tcp.raddr, NULL, 0);
    if (hostname == NULL)
	hostname = isc_getipnum(event->session->info.tcp.raddr, NULL, 0);
    if (hostname == NULL)
	kom_log("WNG: login_request(): unknown hostid.\n");

    /* Get the real user name, as returned by the Ident protocol (rfc 931). */
    realuser = get_real_username(event->session, hostname);
    if (realuser == NULL && param.authentication_level == 2)
    {
	kom_log("Connection from %s rejected - no IDENT available.\n", 
	     hostname);

	isc_puts("%% No IDENT server reachable at your site.\n",
		 event->session);
	isc_flush(event->session);
	isc_destroy(kom_server_mcb, event->session);
	return;
    }

    /* Create a Connection, and link the Connection and the
       isc_session together. */

    cp = new_client();
    cp->isc_session = event->session;
    event->session->udg = cp;

    if (hostname == NULL)
	s_crea_str(&cp->hostname, "unknown");
    else
	s_crea_str(&cp->hostname, hostname);

    if (realuser != NULL)
	s_crea_str(&cp->ident_user, realuser);

    BUG(("\n[Client %lu from %s is connecting]\n", cp->session_no, hostname));
}

static void
logout_request(IscEvent *event)
{
    Connection  * cp;
    
    cp = event->session->udg;
	      
    BUG(("\n[Client %lu from %s", cp->session_no,
	 isc_gethostname(event->session->info.tcp.raddr, NULL, 0)));
    BUG((" is logging out]\n"));

    add_to_kill_list(cp);
}


static void
message_request(IscEvent *event)
{
    Connection *cp;
    String      tmp_str;

    cp = event->session->udg;
    VBUG(("\n[Message from client %lu]\n", cp->session_no));

    /* Pass the message on to the parser */
    tmp_str.string = (unsigned char *)event->msg->buffer;
    tmp_str.len    = event->msg->length;
    parse_message(cp, tmp_str);
}

void
toploop(void)
{
    time_t      last_time;
    IscEvent  * event;
    Bool	pending_input = TRUE; /* Should be TRUE whenever there
				         is or might be pending input
					 from any client in the unparsed
					 buffer. */
    long	timeout = 0;        /* In milliseconds. Start with a
				       small value since we should
				       start garbing right away. */
    while ( !go_and_die )
    {
        if (do_statistics)
	{
	    dump_statistics();
	    do_statistics = 0;
	}

        if (reread_param)
        {
            free_aux_item_definitions();
            initialize_aux_items(param.aux_def_file);
            reread_param = FALSE;
        }

	event = isc_getnextevent(kom_server_mcb,
				 pending_input ? 0 : timeout );

        last_time = current_time;
	current_time = time(NULL);
        if (current_time < last_time)
        {
            kom_log("WARNING: Time is moving in the wrong direction\n");
            /* FIXME (bug 62): Should we take more decisive action here? */
        }

	timeout = 0;

	if ( event == NULL )
	    restart_kom("toploop(): Got NULL event - shouldn't happen...\n");

	switch ( event->event )
	{
	case ISC_EVENT_ERROR:
	    if (errno != EINTR)
	    {
		kom_log("toploop: ISC_EVENT_ERROR (error = '%s'):\n   '%s'\n",
		    strerror(errno), event->msg ? event->msg->buffer
		    : "(unknown)");
	    }
	    break;
	    
	case ISC_EVENT_TIMEOUT:
	    BUG((">"));
#ifndef NDEBUG
	    if (buglevel > 0)
		fflush(stdout);
#endif
	    if ( pending_input == FALSE )
		timeout = end_of_atomic(TRUE); /* Idle. Do some cleaning up. */
	    else
		pending_input = (parse_forgotten() ? FALSE : TRUE);
	    	    
	    break;
	    
	case ISC_EVENT_LOGIN:
	    pending_input = TRUE;
	    login_request(event);
	    break;

	case ISC_EVENT_LOGIN_UNRELOCATED:
	    pending_input = TRUE;
	    BUG(("Connection attempt rejected.\n"));
	    isc_puts("%% No connections left.\n", event->session);
	    isc_flush(event->session);
	    isc_destroy(kom_server_mcb, event->session);

	    async_rejected_connection();
	    break;

	case ISC_EVENT_LOGOUT:
	    pending_input = TRUE;
	    logout_request(event);
	    break;

 	case ISC_EVENT_MESSAGE:
	    pending_input = TRUE;
	    message_request(event);
	    break;

	default:
	    pending_input = TRUE;
	    kom_log("ERROR: toploop(): Unknown ISC_EVENT\n");
	    break;
	}

        isc_dispose(event);

	check_kill_flg();
    }
}
