/* $Header: /usr/src/redhat/BUILD/Linux-PAM-0.59/modules/pam_ncp/RCS/pam_ncp_auth.c,v 1.3 1998/03/04 02:52:07 dwmw2 Exp $ */

/*
 * Copyright David Woodhouse, 1998.  All rights reserved.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 */

/* 
 * $Log: pam_ncp_auth.c,v $
 *
 * Revision 1.5  2000/06/02 17:40:00  vana
 * Tons of new code - passwd, changed auth code
 *
 * Revision 1.4  1999/10/31 15:44:44  vana
 * Couple of new options, new check for group membership
 *
 * Revision 1.3  1998/03/04 02:52:07  dwmw2
 * Oops - licensing has to be GPL as it's using the ncpfs libraries and headers.
 *
 * Revision 1.2  1998/03/04 02:39:12  dwmw2
 * Tidied up (a bit) for initial release.
 *
 * Revision 1.1  1998/03/04 02:21:07  dwmw2
 * Initial revision
 *
 *
 */

#define _GNU_SOURCE
#define _BSD_SOURCE
#define inline __inline__
#include <ncp/nwcalls.h>
#include <ncp/nwnet.h>

#include <features.h>

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <pwd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <errno.h>

#ifndef LINUX 

#include <security/pam_appl.h>

#endif  /* LINUX */

#define _PAM_EXTERN_FUNCTIONS
#include <security/pam_modules.h>

static const char rcsid[] = "$Id: pam_ncp_auth.c,v 1.3 1998/03/04 02:52:07 dwmw2 Exp $ pam_ncp authentication functions. Dave@imladris.demon.co.uk";

/* Define function prototypes */

extern 	int _set_auth_tok(	pam_handle_t *pamh, 
				int flags, int argc, 
				const char **argv	);

extern 	int _set_oldauth_tok(	pam_handle_t *pamh, 
				int flags, int argc, 
				const char **argv	);

extern  int _read_new_pwd(	pam_handle_t *pamh,
				int flags		);

#define QF_VERBOSE	0x0001
#define QF_DEBUG	0x0002
#define QF_NOSU		0x0004
#define QF_NOSUEQ	0x0008
#define QF_AUTOCREATE	0x0010
#define QF_BINDERY	0x0040

static NWCCODE nw_create_conn(	NWCONN_HANDLE *conn,
				const char *server,
				const char *user,
				const char *pwd,
				int qflag) {
        struct ncp_conn_spec spec;
	struct ncp_bindery_object uinfo;
	unsigned char ncp_key[8];
        NWCONN_HANDLE cn;
	long err;

	if (qflag & QF_DEBUG)
		syslog(LOG_DEBUG, "Trying to contact %s/%s\n", server, user);
		
        /* Copy the password into place and convert it to upper case */

	strncpy(spec.server, server, sizeof(spec.server)-1);
	spec.server[sizeof(spec.server)-1] = 0;
	str_upper(spec.server);
	spec.user[0] = 0;
//	strncpy(spec.user, user, sizeof(spec.user)-1);
//	spec.user[sizeof(spec.user)-1] = 0;
	strncpy(spec.password, pwd, sizeof(spec.password)-1);
	spec.password[sizeof(spec.password)-1] = 0;
        str_upper(spec.password);
	spec.uid = ~0;
	spec.login_type = NCP_BINDERY_USER;
        
        /* Attempt to log in. */

	cn = ncp_open(&spec, &err);
	if (!cn) {
		syslog(LOG_WARNING, "%s when trying to open connection\n", strnwerror(err));
		err = PAM_TRY_AGAIN;
		goto fail2;
	}
	if (!(qflag & QF_BINDERY) && NWIsDSServer(cn, NULL)) {
		err = nds_login_auth(cn, user, pwd);
	} else {
		err = ncp_get_encryption_key(cn, ncp_key);
		if (err) {
			syslog(LOG_WARNING, "%s when trying to get encryption key. Doing unencrypted\n", strnwerror(err));
			err = ncp_login_unencrypted(cn, spec.login_type, user, spec.password);
		} else {
			err = ncp_get_bindery_object_id(cn, NCP_BINDERY_USER, user, &uinfo);
			if (err) {
				syslog(LOG_WARNING, "%s when trying to get object ID\n", strnwerror(err));
				err = PAM_USER_UNKNOWN;
				goto fail;
			}
			err = ncp_login_encrypted(cn, &uinfo, ncp_key, spec.password);
		}
	}
	if (err && err != NWE_PASSWORD_EXPIRED) {
		syslog(LOG_WARNING, "%s when trying to login\n", strnwerror(err));
		switch (err) {
			case ERR_NO_SUCH_ENTRY:
			case NWE_SERVER_UNKNOWN:
				err = PAM_USER_UNKNOWN;
				break;
			case NWE_LOGIN_MAX_EXCEEDED:
			case NWE_LOGIN_UNAUTHORIZED_TIME:
			case NWE_LOGIN_UNAUTHORIZED_STATION:
			case NWE_ACCT_DISABLED:
				err = PAM_AUTH_ERR;
				break;
			default:
				err = PAM_AUTH_ERR;
				break;
		}
		goto fail;
	}
#if 0	
	if (err)
		err = PAM_NEW_AUTHTOK_REQD;
#else
	err = 0;
#endif		
        if (qflag & QF_DEBUG)
		syslog(LOG_DEBUG, "User %s/%s was successfully authorized\n", server, user);
	*conn = cn;
	return err;
fail:;
	ncp_close(cn);
fail2:;
	*conn = NULL;
	return err;	
}

static int nw_get_nwid(		NWCONN_HANDLE	conn,
				NWObjectID	*id,
				int qflag) {
	NWCCODE err;

	err = NWCCGetConnInfo(conn, NWCC_INFO_USER_ID, sizeof(*id), id);
	if (err) {
		syslog(LOG_WARNING, "%s when retrieving object ID\n", strnwerror(err));
		return PAM_SYSTEM_ERR;
	}
	return PAM_SUCCESS;
}

static int nw_create_verify_conn(
				NWCONN_HANDLE	*conn,
				NWObjectID	*id,
				const char *server,
				const char *user,
				const char *pwd,
				int qflag,
				const char *group) {
        NWCONN_HANDLE cn;
	NWCCODE err;
	NWObjectID oid;

	err = nw_create_conn(&cn, server, user, pwd, qflag);
	if (err && err != PAM_NEW_AUTHTOK_REQD)
		return err;
	err = nw_get_nwid(cn, &oid, qflag);
	if (err) {
		goto bailout;
	}	
	if (qflag & QF_NOSU) {
		if (oid == 0x00000001) {
			err = PAM_AUTH_ERR;
			syslog(LOG_WARNING, "Access denied for %s/%s because of it is supervisor\n", server, user);
			goto bailout;
		}
		if (qflag & QF_DEBUG)
			syslog(LOG_DEBUG, "User %s/%s passed supervisor check\n", server, user);
	}
	if (qflag & QF_NOSUEQ) {
		nuint8 level;
		
		err = NWGetBinderyAccessLevel(cn, &level, NULL);
		if (err) {
			syslog(LOG_WARNING, "Access denied for %s/%s because of I/O error during object rights verification\n",
					server, user);
			err = PAM_AUTH_ERR;
			goto bailout;
		}
		if ((level >= 0x30) || ((level & 0xF) >= 3)) {
			syslog(LOG_WARNING, "Access denied for %s/%s because of it is supervisor equivalent\n",
					server, user);
			err = PAM_AUTH_ERR;
			goto bailout;
		}
	}
	if (group) {
		Buf_T* buf;
		NWDSContextHandle ctx;
		nbool8 match;
		nuint32 c;
		
		err = NWDSCreateContextHandle(&ctx);
		if (err) {
			syslog(LOG_WARNING, "NWDSCreateContextHandle() failed with %s\n", strnwerror(err));
			err = PAM_SYSTEM_ERR;
			goto bailout;
		}
		c = DCV_XLATE_STRINGS | DCV_TYPELESS_NAMES;
		err = NWDSSetContext(ctx, DCK_FLAGS, &c);
		if (err) {
			syslog(LOG_WARNING, "NWDSSetContext() failed with %s\n", strnwerror(err));
			err = PAM_SYSTEM_ERR;
			NWDSFreeContext(ctx);
			goto bailout;
		}
		err = NWDSAllocBuf(DEFAULT_MESSAGE_LEN, &buf);
		if (err) {
			syslog(LOG_WARNING, "NWDSAllocBuf() failed with %s\n", strnwerror(err));
			err = PAM_SYSTEM_ERR;
			NWDSFreeContext(ctx);
			goto bailout;
		}
		err = NWDSInitBuf(ctx, DSV_COMPARE, buf);
		if (err) {
			syslog(LOG_WARNING, "NWDSInitBuf() failed with %s\n", strnwerror(err));
			err = PAM_SYSTEM_ERR;
			NWDSFreeBuf(buf);
			NWDSFreeContext(ctx);
			goto bailout;
		}
		err = NWDSPutAttrName(ctx, buf, "Group Membership");
		if (err) {
			syslog(LOG_WARNING, "NWDSPutAttrName() failed with %s\n", strnwerror(err));
			err = PAM_SYSTEM_ERR;
			NWDSFreeBuf(buf);
			NWDSFreeContext(ctx);
			goto bailout;
		}
		err = NWDSPutAttrVal(ctx, buf, SYN_DIST_NAME, group);
		if (err) {
			syslog(LOG_WARNING, "NWDSPutAttrVal() failed with %s\n", strnwerror(err));
			err = PAM_SYSTEM_ERR;
			NWDSFreeBuf(buf);
			NWDSFreeContext(ctx);
			goto bailout;
		}
		err = __NWDSCompare(ctx, cn, oid, buf, &match);
		if (err) {
			syslog(LOG_WARNING, "__NWDSCompare() failed with %s\n", strnwerror(err));
			err = PAM_AUTH_ERR;
			NWDSFreeBuf(buf);
			NWDSFreeContext(ctx);
			goto bailout;
		}
		NWDSFreeBuf(buf);
		NWDSFreeContext(ctx);
		if (!match) {
			syslog(LOG_WARNING, "%s is not member of %s\n", user, group);
			err = PAM_AUTH_ERR;
			goto bailout;
		}
	}
	if (id)
		*id = oid;
	if (conn)
		*conn = cn;
	else
		ncp_close(cn);
	return 0;
bailout:;
        /* Close the connection. */
        ncp_close(cn);
        return err;
}

static int nw_attempt_auth(     const char *server,
                                const char *user,
                                const char *pwd,
				int qflag,
				const char *group )
{
	int err;

	err = nw_create_verify_conn(NULL, NULL, server, user, pwd, qflag, group);
        return err;
}

/* The code - what there is of it :) */

PAM_EXTERN
int pam_sm_authenticate(	pam_handle_t *pamh, 
				int flags,
				int argc,
				const char **argv	) 
{
        int retval;
	const char *name;
	char *p;
        int c;

	int qflag = QF_VERBOSE;

	openlog("pam_ncp_auth", LOG_PID, LOG_AUTHPRIV);
	
	/* Get options */
	for (c = 0; c < argc; c++) {
		if (argv[c][0] == '-') {
			int i;
			
			for (i = 1; argv[c][i]; i++) {
				switch (argv[c][i]) {
					case 'v':qflag |= QF_VERBOSE; break;	/* verbose */
					case 'q':qflag &= ~QF_VERBOSE; break;	/* quiet */
					case 'd':qflag |= QF_DEBUG; break;	/* debug */
					case 's':qflag |= QF_NOSU; break;	/* no supervisor */
					case 'S':qflag |= QF_NOSUEQ; break;	/* no supervisor equivalent */
					case 'a':qflag |= QF_AUTOCREATE; break;	/* create account automagically */
					case 'b':qflag |= QF_BINDERY; break;	/* use bindery access */
					default:; /* just silently ignore unknown option... */
				}
			}
		}
	}
	
        /* Get username */
	if ( (retval = pam_get_user( pamh, &name, "login: ") ) != PAM_SUCCESS )
		return retval;


        /* Get password */
	pam_get_item( pamh, PAM_AUTHTOK, (void*) &p );

	if ( !p ) 
		{
			retval = _set_auth_tok( pamh, flags, argc, argv );
			if ( retval != PAM_SUCCESS ) 
				return retval;
 		}
	
	pam_get_item( pamh, PAM_AUTHTOK, (void*) &p );

        /* Find the server name in the configuration. */

        for (c = 0; c < argc; c++) {
                if (!strncmp("server=", argv[c], 7)) {
			const char* server;
			const char* group;
			int rc;
			char sbuf[256];
			
                        server = argv[c] + 7;
			group = strchr(server, '/');
			if (group) {
				if ((size_t)(group - server) < sizeof(sbuf)-1) {
					memcpy(sbuf, server, group-server);
					sbuf[group-server] = 0;
					server = sbuf;
					group = group + 1;
				} else {
					syslog(LOG_ALERT, "Error in configuration file: server name too long!\n");
					continue;
				}
			}
			rc = nw_attempt_auth(server, name, p, qflag, group);
			if (rc == PAM_SUCCESS)
				goto success;
                }
        }

	return PAM_AUTH_ERR;
success:;
	if (qflag & QF_AUTOCREATE) {
		struct passwd* pwd_entry;
		setpwent();
		pwd_entry = getpwnam(name);
		endpwent();
		if (!pwd_entry) {
			FILE* f = fopen("/etc/passwd", "a+");
			struct passwd pwd;
			struct stat stt;
			
			(const char*)pwd.pw_name = name;
			pwd.pw_passwd = "x";
			pwd.pw_uid = 10000;
			pwd.pw_gid = 10000;
			pwd.pw_gecos = "Netware User";
			pwd.pw_dir = "/home/nwuser";
			pwd.pw_shell = "/bin/bash";
			putpwent(&pwd, f);
			fclose(f);
			if (!stat("/etc/shadow", &stt)) {
				f = fopen("/etc/shadow", "a+");
				fprintf(f, "%s:!:10850:0:99999:7:::\n", name);
				fclose(f);
			}
		}
	}
	return PAM_SUCCESS;
}

/* 
 * Does nothing.
 */

PAM_EXTERN
int pam_sm_setcred( pam_handle_t *pamh, 
		    int flags,
		    int argc, 
		    const char **argv)
{
	return PAM_IGNORE;
}

static void nw_cleanup_conn(    pam_handle_t *pamh,
				void *data,
				int error_status)
{
	ncp_close((struct ncp_conn*)data);
}

static int nw_attempt_passwd_prelim(
				pam_handle_t *pamh,
				const char *server,
                                const char *user,
				const char *oldpwd,
				int qflag,
				const char *group,
				int flags )
{
	struct ncp_conn *conn;
	int err;

	err = nw_create_verify_conn(&conn, NULL, server, user, oldpwd, 
					    qflag, group);
	if (err)
		return err;
	pam_set_data(pamh, "pam.ncpfs.passwd.conn", conn, nw_cleanup_conn);
	return 0;
}

static int nw_attempt_passwd_post(
				pam_handle_t *pamh,
				const char *pwd,
				const char *oldpwd,
				int qflag,
				int flags )
{
        struct ncp_conn *conn;
	long err;
	NWObjectID oid;
	char oldpwdup[MAX_DN_BYTES];
	char pwdup[MAX_DN_BYTES];

	err = PAM_AUTHTOK_ERR;
	if (!(flags & PAM_UPDATE_AUTHTOK))
		goto bailout;
	err = pam_get_data(pamh, "pam.ncpfs.passwd.conn", (const void**)&conn);
	if (err)
		return err;
	err = nw_get_nwid(conn, &oid, qflag);
	if (err)
		return err;

	strncpy(oldpwdup, oldpwd, MAX_DN_BYTES);
	pwdup[sizeof(oldpwdup) - 1] = 0;
	str_upper(oldpwdup);

	strncpy(pwdup, pwd, MAX_DN_BYTES);
	pwdup[sizeof(pwdup) - 1] = 0;
	str_upper(pwdup);

	if (qflag & QF_BINDERY) {
		struct ncp_bindery_object uinfo;
		struct ncp_bindery_object u0;
		unsigned char ncp_key[8];
			
		err = ncp_get_bindery_object_name(conn, oid, &u0);
		if (err) {
			syslog(LOG_WARNING, "%s when trying to get object name\n", strnwerror(err));
			err = PAM_USER_UNKNOWN;
			goto bailout;
		}
		err = ncp_get_encryption_key(conn, ncp_key);
		if (err) {
			syslog(LOG_WARNING, "%s when trying to get encryption key\n", strnwerror(err));
			err = PAM_AUTHTOK_ERR;
			goto bailout;
		} else {
			err = ncp_get_bindery_object_id(conn, u0.object_type, u0.object_name, &uinfo);
			if (err) {
				syslog(LOG_WARNING, "%s when trying to get object ID\n", strnwerror(err));
				err = PAM_USER_UNKNOWN;
				goto bailout;
			}
			err = ncp_change_login_passwd(conn, &uinfo, ncp_key, oldpwdup, pwdup);
			if (err) {
				syslog(LOG_WARNING, "%s when trying to change password\n", strnwerror(err));
				err = PAM_AUTHTOK_ERR;
				goto bailout;
			}
		}
	} else {
		NWDSContextHandle ctx;
		nuint32 c;
		char username[MAX_DN_BYTES];

		err = NWDSCreateContextHandle(&ctx);
		if (err) {
			syslog(LOG_WARNING, "NWDSCreateContextHandle() failed with %s\n", strnwerror(err));
			err = PAM_SYSTEM_ERR;
			goto bailout;
		}
		c = DCV_XLATE_STRINGS | DCV_TYPELESS_NAMES;
		err = NWDSSetContext(ctx, DCK_FLAGS, &c);
		if (err) {
			syslog(LOG_WARNING, "NWDSSetContext() failed with %s\n", strnwerror(err));
			err = PAM_SYSTEM_ERR;
			goto bailoutctx;
		}
		NWDSAddConnection(ctx, conn);
		err = NWDSMapIDToName(ctx, conn, oid, username);
		if (err) {
			syslog(LOG_WARNING, "NWDSMapIDToName() failed with %s\n", strnwerror(err));
			err = PAM_USER_UNKNOWN;
			goto bailoutctx;
		}
		if (qflag & QF_DEBUG)
			syslog(LOG_DEBUG, "User has DN %s\n", username);
		err = NWDSChangeObjectPassword(ctx, NDS_PASSWORD, username, oldpwdup, pwdup);
		if (err) {
			syslog(LOG_NOTICE, "NWDSChangeObjectPassword() failed with %s\n", strnwerror(err));
			err = PAM_AUTHTOK_ERR;
			goto bailoutctx;
		}
bailoutctx:;
		NWDSFreeContext(ctx);
	}
bailout:;
        return err;
}

PAM_EXTERN
int pam_sm_chauthtok(	pam_handle_t *pamh, 
			int flags,
			int argc,
			const char **argv	) 
{
        int retval;
	const char *name;
	char *p;
	char *oldpasswd;
        int c;

	int qflag = QF_VERBOSE;
	
	openlog("pam_ncp_auth", LOG_PID, LOG_AUTHPRIV);
	/* Get options */
	for (c = 0; c < argc; c++) {
		if (argv[c][0] == '-') {
			int i;
			
			for (i = 1; argv[c][i]; i++) {
				switch (argv[c][i]) {
					case 'v':qflag |= QF_VERBOSE; break;	/* verbose */
					case 'q':qflag &= ~QF_VERBOSE; break;	/* quiet */
					case 'd':qflag |= QF_DEBUG; break;	/* debug */
					case 's':qflag |= QF_NOSU; break;	/* no supervisor */
					case 'S':qflag |= QF_NOSUEQ; break;	/* no supervisor equivalent */
					case 'a':qflag |= QF_AUTOCREATE; break;	/* create account automagically */
					case 'b':qflag |= QF_BINDERY; break;	/* use bindery access */
					default:; /* just silently ignore unknown option... */
				}
			}
		}
	}
	
        /* Get username */
	if ( (retval = pam_get_user( pamh, &name, "passwd: ") ) != PAM_SUCCESS )
		return retval;

	pam_get_item(pamh, PAM_OLDAUTHTOK, (void*)&oldpasswd);
	if (!oldpasswd) {
		retval = _set_oldauth_tok(pamh, flags, argc, argv);
		if (retval != PAM_SUCCESS)
			return retval;
		pam_get_item(pamh, PAM_OLDAUTHTOK, (void*)&oldpasswd);
	}
	
	
	if (flags & PAM_PRELIM_CHECK) {
	        /* Find the server name in the configuration. */

	        for (c = 0; c < argc; c++) {
        	        if (!strncmp("server=", argv[c], 7)) {
				const char* server;
				const char* group;
				int rc;
				char sbuf[256];
			
                	        server = argv[c] + 7;
				group = strchr(server, '/');
				if (group) {
					if ((size_t)(group - server) < sizeof(sbuf)-1) {
						memcpy(sbuf, server, group-server);
						sbuf[group-server] = 0;
						server = sbuf;
						group = group + 1;
					} else {
						syslog(LOG_ALERT, "Error in configuration file: server name too long!\n");
						continue;
					}
				}
				rc = nw_attempt_passwd_prelim(pamh, server, name, oldpasswd, qflag, group, flags);
				if (rc == PAM_SUCCESS)
					goto success;
	                }
        	}

		return PAM_AUTHTOK_ERR;
	}
	
	if (flags & PAM_UPDATE_AUTHTOK) {
	        /* Get password */
		pam_get_item( pamh, PAM_AUTHTOK, (void*) &p );

		if (!p) {
			retval = _read_new_pwd(pamh, flags);
			if ( retval != PAM_SUCCESS ) 
				return retval;
			pam_get_item( pamh, PAM_AUTHTOK, (void*) &p );
		}
	
		return nw_attempt_passwd_post(pamh, p, oldpasswd, qflag, flags);
	}
	
	return PAM_SYSTEM_ERR;

success:;
	return PAM_SUCCESS;
}



/* static module data */
#ifdef PAM_STATIC
struct pam_module _pam_ncp_auth_modstruct = {
    "pam_ncp_auth",
    pam_sm_authenticate,
    pam_sm_setcred,
    NULL,
    NULL,
    NULL,
    pam_sm_chauthtok,
};
#endif




