/*************************************************************************
 *
 *  $RCSfile: pkgchk_log.cxx,v $
 *
 *  $Revision: 1.7 $
 *
 *  last change: $Author: vg $ $Date: 2003/05/26 08:38:16 $
 *
 *  The Contents of this file are made available subject to the terms of
 *  either of the following licenses
 *
 *         - GNU Lesser General Public License Version 2.1
 *         - Sun Industry Standards Source License Version 1.1
 *
 *  Sun Microsystems Inc., October, 2000
 *
 *  GNU Lesser General Public License Version 2.1
 *  =============================================
 *  Copyright 2000 by Sun Microsystems, Inc.
 *  901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License version 2.1, as published by the Free Software Foundation.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *  MA  02111-1307  USA
 *
 *
 *  Sun Industry Standards Source License Version 1.1
 *  =================================================
 *  The contents of this file are subject to the Sun Industry Standards
 *  Source License Version 1.1 (the "License"); You may not use this file
 *  except in compliance with the License. You may obtain a copy of the
 *  License at http://www.openoffice.org/license.html.
 *
 *  Software provided under this License is provided on an "AS IS" basis,
 *  WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING,
 *  WITHOUT LIMITATION, WARRANTIES THAT THE SOFTWARE IS FREE OF DEFECTS,
 *  MERCHANTABLE, FIT FOR A PARTICULAR PURPOSE, OR NON-INFRINGING.
 *  See the License for the specific provisions governing your rights and
 *  obligations concerning the Software.
 *
 *  The Initial Developer of the Original Code is: Sun Microsystems, Inc.
 *
 *  Copyright: 2002 by Sun Microsystems, Inc.
 *
 *  All Rights Reserved.
 *
 *  Contributor(s): _______________________________________
 *
 *
 ************************************************************************/

#include <stdio.h>

#include "pkgchk_env.h"
#include "rtl/strbuf.hxx"
#include "osl/thread.h"


using namespace ::rtl;
using namespace ::osl;
using namespace ::com::sun::star::uno;

namespace pkgchk
{

//______________________________________________________________________________
void pkgchk_env::log_write( OString const & text ) const
{
    if (! m_log_file_path.getLength())
        return;
    
    if (! m_log_file)
    {
        oslFileHandle file = 0;
        if (osl_File_E_None == osl_openFile(
                m_log_file_path.pData, &file, osl_File_OpenFlag_Write ))
        {
            if (osl_File_E_None != osl_setFilePos( file, osl_Pos_End, 0 ))
            {
                OSL_VERIFY( osl_File_E_None == osl_closeFile( file ) );
                OUString msg(
                    OUSTR("cannot set pos on log file ") + m_log_file_path );
                m_log_file_path = OUString(); // don't try again
                throw RuntimeException( msg, Reference< XInterface >() );
            }
        }
        else if (File::E_None != osl_openFile(
                     m_log_file_path.pData, &file,
                     osl_File_OpenFlag_Create | osl_File_OpenFlag_Write ))
        {
            OUString msg( OUSTR("cannot create log file ") + m_log_file_path );
            m_log_file_path = OUString(); // don't try again
            throw RuntimeException( msg, Reference< XInterface >() );
        }
        m_log_file = file;
        
        // write log stamp
        OStringBuffer buf( 64 );
        buf.append( RTL_CONSTASCII_STRINGPARAM("###### log entry ") );
        TimeValue tSystem, tLocal;
        oslDateTime date_time;
        if (osl_getSystemTime( &tSystem ) &&
            osl_getLocalTimeFromSystemTime( &tSystem, &tLocal ) &&
            osl_getDateTimeFromTimeValue( &tLocal, &date_time ))
        {
            buf.append( (sal_Int32)date_time.Year );
            buf.append( '-' );
            buf.append( (sal_Int32)date_time.Month );
            buf.append( '-' );
            buf.append( (sal_Int32)date_time.Day );
            buf.append( ' ' );
            buf.append( (sal_Int32)date_time.Hours );
            buf.append( ':' );
            buf.append( (sal_Int32)date_time.Minutes );
            buf.append( ':' );
            buf.append( (sal_Int32)date_time.Seconds );
            buf.append( ' ' );
        }
        buf.append( RTL_CONSTASCII_STRINGPARAM("######" LF) );
        log_write( buf.makeStringAndClear() );
    }
    
    file_write( m_log_file, text, m_log_file_path );
}

static OString s_lf( LF );
static OString s_warning( "> warning: " );

//______________________________________________________________________________
void pkgchk_env::log( OUString const & text, bool allow_verbose ) const
{
    OString cstr_text( OUStringToOString( text, osl_getThreadTextEncoding() ) );
    // to stdout if verbose
    if (allow_verbose && m_verbose)
        printf( "%s\n", cstr_text.getStr() );
    // log text
    log_write( cstr_text );
    log_write( s_lf );
    
#if defined DIAG
    TimeValue time;
    OSL_VERIFY( osl_getSystemTime( &time ) );
    double d = time.Seconds - m_start_time.Seconds;
    d += ((double) time.Nanosec / 1000000000.0);
    d -= ((double) m_start_time.Nanosec / 1000000000.0);
    OStringBuffer buf( 64 );
    buf.append( RTL_CONSTASCII_STRINGPARAM("[diag] running time: ") );
    buf.append( d );
    buf.append( RTL_CONSTASCII_STRINGPARAM(" secs" LF) );
    OString msg( buf.makeStringAndClear() );
    // to stdout if verbose
    if (allow_verbose && m_verbose)
        printf( msg.getStr() );
    log_write( msg );
#endif
}

//______________________________________________________________________________
void pkgchk_env::warn( OUString const & text, bool allow_verbose ) const
{
    OString cstr_text( OUStringToOString( text, osl_getThreadTextEncoding() ) );
    // to stderr if verbose
    if (allow_verbose && (m_verbose || m_verbose_on_ignored_errors))
        fprintf( stderr, "> warning: %s\n", cstr_text.getStr() );
    log_write( s_warning );
    log_write( cstr_text );
    log_write( s_lf );
}

//______________________________________________________________________________
void pkgchk_env::err( OUString const & text, bool allow_verbose ) const
{
    if (m_strict_error_handling)
    {
        // breaks process: error is not logged here
        // *All* exceptions are caught by mgr code and logged there
        throw RuntimeException( text, Reference< XInterface >() );
    }
    else
    {
        warn( text, allow_verbose );
    }
}

}
