/*************************************************************************
 *
 *  $RCSfile: fileimpl.cxx,v $
 *
 *  $Revision: 1.3 $
 *
 *  last change: $Author: vg $ $Date: 2003/05/13 12:28:35 $
 *
 *  The Contents of this file are made available subject to the terms of
 *  either of the following licenses
 *
 *         - GNU Lesser General Public License Version 2.1
 *         - Sun Industry Standards Source License Version 1.1
 *
 *  Sun Microsystems Inc., October, 2000
 *
 *  GNU Lesser General Public License Version 2.1
 *  =============================================
 *  Copyright 2000 by Sun Microsystems, Inc.
 *  901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License version 2.1, as published by the Free Software Foundation.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *  MA  02111-1307  USA
 *
 *
 *  Sun Industry Standards Source License Version 1.1
 *  =================================================
 *  The contents of this file are subject to the Sun Industry Standards
 *  Source License Version 1.1 (the "License"); You may not use this file
 *  except in compliance with the License. You may obtain a copy of the
 *  License at http://www.openoffice.org/license.html.
 *
 *  Software provided under this License is provided on an "AS IS" basis,
 *  WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING,
 *  WITHOUT LIMITATION, WARRANTIES THAT THE SOFTWARE IS FREE OF DEFECTS,
 *  MERCHANTABLE, FIT FOR A PARTICULAR PURPOSE, OR NON-INFRINGING.
 *  See the License for the specific provisions governing your rights and
 *  obligations concerning the Software.
 *
 *  The Initial Developer of the Original Code is: Sun Microsystems, Inc.
 *
 *  Copyright: 2000 by Sun Microsystems, Inc.
 *
 *  All Rights Reserved.
 *
 *  Contributor(s): _______________________________________
 *
 *
 ************************************************************************/

/************************************
 Include 
 ***********************************/

#define _WIN32_WINNT 0x0500

#define UNICODE
#include "system.h"
#include <systools\win32\uwinapi.h>

#define _UNICODE
#include <tchar.h>

#ifndef _OSL_DIAGNOSE_H_
#include <osl/diagnose.h>
#endif

#ifndef _OSL_FILE_H_
#include <osl/file.h>
#endif

#ifndef _RTL_USTRING_HXX_
#include <rtl/ustring.hxx>
#endif

#ifndef _FILEIMPL_HXX_
#include "fileimpl.hxx"
#endif

#ifndef _PATH_HELPER_HXX_
#include "path_helper.hxx"
#endif

#include <algorithm>

//------------------------------------------
// Defines
//------------------------------------------

#define ELEMENTS_OF_ARRAY(arr) (sizeof(arr)/(sizeof((arr)[0])))


namespace /* private */
{
    //-------------------------------------
    // forward
    //-------------------------------------
    bool is_floppy_drive(const rtl::OUString& path);
    
    
    //##################################################################
    struct Component
    {       
        Component() : 
            begin_(0),
            end_(0)
        {}

        bool isPresent() const
        {
            return (static_cast<sal_Int32>(end_ - begin_) > 0);
        }        
        
        const sal_Unicode* begin_;
        const sal_Unicode* end_;
    };
                    
    //##################################################################
    struct UNCComponents
    {                        
        Component server_;
        Component share_;
        Component resource_;
    };
    
    //##################################################################            
    // simply checking if this path starts with '\\'
    
    const sal_Unicode* UNC_PREFIX = L"\\\\";
    
    bool is_UNC_path(const rtl::OUString& path)
    {    
        return (0 == wcsncmp(UNC_PREFIX, path.getStr(), wcslen(UNC_PREFIX)));
    }
    
    //##################################################################            
    // Precondition: provided path is a valid UNC path!
    void parse_UNC_path(const rtl::OUString& path, UNCComponents* puncc)
    {                
        if (is_UNC_path(path))
        {            
            const sal_Unicode* pend   = path.getStr() + path.getLength();
            const sal_Unicode* ppos   = path.getStr() + 2;
                    
            puncc->server_.begin_ = ppos;                                    
            while ((ppos < pend) && (*ppos != L'\\'))
                ppos++;
            
            puncc->server_.end_ = ppos;
            
            if (L'\\' == *ppos)    
            {                                                
                puncc->share_.begin_ = ++ppos;                                
                while ((ppos < pend) && (*ppos != L'\\'))
                    ppos++;
                    
                puncc->share_.end_ = ppos;
                                            
                if (L'\\' == *ppos)
                {                                                                                                                                                                               
                    puncc->resource_.begin_ = ++ppos;                                                        
                    while (ppos < pend)
                        ppos++;
                    
                    puncc->resource_.end_ = ppos;
                }                                 
            }
        }                
    }
    
    //##################################################################
    bool is_volume_mount_point(const rtl::OUString& path)
    {
        rtl::OUString p(path);        
        osl::systemPathRemoveSeparator(p);
            
        bool  is_volume_root = false;
        
        if (!is_floppy_drive(p))
        {
            DWORD fattr = GetFileAttributes(p.getStr());
            
            if ((INVALID_FILE_ATTRIBUTES != fattr) && 
                (FILE_ATTRIBUTE_REPARSE_POINT & fattr))
            {
                WIN32_FIND_DATA find_data;
                HANDLE h_find = FindFirstFile(p.getStr(), &find_data);    
                           
                if (IsValidHandle(h_find) &&
                    (FILE_ATTRIBUTE_REPARSE_POINT & find_data.dwFileAttributes) && 
                    (IO_REPARSE_TAG_MOUNT_POINT == find_data.dwReserved0))
                {            
                    is_volume_root = true;
                }        
                if (IsValidHandle(h_find))
                    FindClose(h_find);
            }
        }
        return is_volume_root;
    }
    
    //##################################################################
    // Are we already at the highest possible volume root e.g.
    // 'c:\' or '\\server\share\'
    bool is_path_volume_root(const rtl::OUString& path)
    {                
        bool is_root = false;
        
        if (is_UNC_path(path))
        {
            UNCComponents unc_comp;
            parse_UNC_path(path, &unc_comp);

            OSL_ASSERT(unc_comp.server_.isPresent() && unc_comp.share_.isPresent());
            
            is_root = !unc_comp.resource_.isPresent();
        }
        else // assuming local path
        {
            is_root = osl::systemPathIsLogicalDrivePattern(path);
        }
        return is_root;
    }

    //##################################################################
    bool path_get_parent(rtl::OUString& path)
    {   
        if (!is_path_volume_root(path))
        {
            sal_Int32 i = std::max<sal_Int32>(
                path.lastIndexOf(L'/'), path.lastIndexOf(L'\\'));            
            
            if (-1 < i)
            {
                path = rtl::OUString(path.getStr(), i);
                return true;                    
            }
        }
        return false;
    }
    
    /*******************************************************************
    osl_systemPathGetVolumeRoot 
    ******************************************************************/
     
    void path_travel_to_volume_root(const rtl::OUString& system_path, rtl::OUString& volume_root)
    {
        rtl::OUString sys_path(system_path);
        
        while(!is_volume_mount_point(sys_path) && path_get_parent(sys_path))
            /**/;
        
        volume_root = sys_path;
        osl_systemPathEnsureSeparator(&volume_root.pData);                            
    } 
    
    //#####################################################
    inline bool is_floppy_A_present()
    { return (GetLogicalDrives() & 1); }
    
    //#####################################################
    inline bool is_floppy_B_present()
    { return (GetLogicalDrives() & 2); }
    
    //#####################################################
    // determines if a volume mount point shows to a floppy
    // disk by comparing the unique volume names           
    const LPWSTR FLOPPY_A = L"A:\\";
    const LPWSTR FLOPPY_B = L"B:\\";
    
    bool is_floppy_volume_mount_point(const rtl::OUString& path)
    {                   
        rtl::OUString p(path);
        osl_systemPathEnsureSeparator(&p.pData);
        
        TCHAR vn[51];
        if (GetVolumeNameForVolumeMountPoint(p.getStr(), vn, ELEMENTS_OF_ARRAY(vn)))
        {       
            TCHAR vnfloppy[51];
            if (is_floppy_A_present() && 
                GetVolumeNameForVolumeMountPoint(FLOPPY_A, vnfloppy, ELEMENTS_OF_ARRAY(vnfloppy)) &&
                (0 == wcscmp(vn, vnfloppy)))
                return true;
                    
            if (is_floppy_B_present() &&
                GetVolumeNameForVolumeMountPoint(FLOPPY_B, vnfloppy, ELEMENTS_OF_ARRAY(vnfloppy)) &&
                (0 == wcscmp(vn, vnfloppy)))
                return true;
        }   
        return false;
    }
    
    //################################################
    // we must take into account that even a floppy 
    // drive may be mounted to a directory so checking
    // for the drive letter alone is not sufficient
    // we must compare the unique volume name with 
    // that of the available floppy disks
    const sal_Unicode* FLOPPY_DRV_LETTERS = TEXT("AaBb"); 
       
    bool is_floppy_drive(const rtl::OUString& path)
    {
        const sal_Unicode* pf = path.getStr();
        const sal_Unicode* ps = path.getStr() + 1;        
        return ((wcschr(FLOPPY_DRV_LETTERS, *pf) && (L':' == *ps)) ||
                is_floppy_volume_mount_point(path));                
    }
    
    //#############################################
    UINT get_volume_mount_point_drive_type(const rtl::OUString& path)
    {        
        if (0 == path.getLength())
            return GetDriveType(NULL);
        
        rtl::OUString p(path);
        osl_systemPathEnsureSeparator(&p.pData);
        
        TCHAR vn[51];  
        if (GetVolumeNameForVolumeMountPoint(p.getStr(), vn, ELEMENTS_OF_ARRAY(vn)))
            return GetDriveType(vn);
        
        return DRIVE_NO_ROOT_DIR;  
    }
    
    //#############################################
    oslFileError osl_get_drive_type(const rtl::OUString& path, oslVolumeInfo* pInfo)
    {
        // GetDriveType fails on empty volume mount points
        // see Knowledge Base Q244089
        UINT drive_type;
        if (is_volume_mount_point(path))
            drive_type = get_volume_mount_point_drive_type(path);
        else
	        drive_type = GetDriveType(path.getStr());
    	
	    if (DRIVE_NO_ROOT_DIR == drive_type)
		    return MapError(ERROR_INVALID_DRIVE);
    		    
		pInfo->uValidFields |= osl_VolumeInfo_Mask_Attributes;		

		switch (drive_type)
		{
		case DRIVE_CDROM:
			pInfo->uAttributes |= osl_Volume_Attribute_CompactDisc | osl_Volume_Attribute_Removeable;
			break;
		case DRIVE_REMOVABLE:
			pInfo->uAttributes |= osl_Volume_Attribute_Removeable;
			if (is_floppy_drive(path))
				pInfo->uAttributes |= osl_Volume_Attribute_FloppyDisk;
			break;
		case DRIVE_FIXED:
			pInfo->uAttributes |= osl_Volume_Attribute_FixedDisk;
			break;
		case DRIVE_RAMDISK:
			pInfo->uAttributes |= osl_Volume_Attribute_RAMDisk;
			break;
		case DRIVE_REMOTE:
			pInfo->uAttributes |= osl_Volume_Attribute_Remote;
			break;
		case DRIVE_UNKNOWN:
			pInfo->uAttributes = 0;
			break;		
		default:
			pInfo->uValidFields &= ~osl_VolumeInfo_Mask_Attributes;
			pInfo->uAttributes = 0;
			break;
		}	
		return osl_File_E_None;    
    }
    
    //#############################################
    inline bool is_volume_space_info_request(sal_uInt32 field_mask)
    {
        return (field_mask & 
                (osl_VolumeInfo_Mask_TotalSpace | 
                 osl_VolumeInfo_Mask_UsedSpace  | 
                 osl_VolumeInfo_Mask_FreeSpace));
    }
    
    //#############################################
    void get_volume_space_information(const rtl::OUString& path, oslVolumeInfo *pInfo)
    {
        BOOL ret = GetDiskFreeSpaceEx(
            path.getStr(), 
            (PULARGE_INTEGER)&pInfo->uFreeSpace, 
            (PULARGE_INTEGER)&pInfo->uTotalSpace, 
            NULL);
            
        if (ret)
		{			    			    
			pInfo->uUsedSpace    = pInfo->uTotalSpace - pInfo->uFreeSpace;
			pInfo->uValidFields |= osl_VolumeInfo_Mask_TotalSpace | 
			                       osl_VolumeInfo_Mask_UsedSpace | 
			                       osl_VolumeInfo_Mask_FreeSpace;
		}
    }
    
    //#############################################
    inline bool is_filesystem_attributes_request(sal_uInt32 field_mask)
    {
        return (field_mask & 
                (osl_VolumeInfo_Mask_MaxNameLength | 
	             osl_VolumeInfo_Mask_MaxPathLength | 
	             osl_VolumeInfo_Mask_FileSystemName | 
	             osl_VolumeInfo_Mask_FileSystemCaseHandling));
    }
    
    //#############################################
    inline bool is_drivetype_request(sal_uInt32 field_mask)
    {
        return (field_mask & osl_VolumeInfo_Mask_Attributes);
    }
    
    //#############################################
    oslFileError get_filesystem_attributes(const rtl::OUString& path, sal_uInt32 field_mask, oslVolumeInfo* pInfo)
    {
        pInfo->uAttributes = 0;
        
        oslFileError osl_error = osl_File_E_None;
        
        // osl_get_drive_type must be called first because
        // this function resets osl_VolumeInfo_Mask_Attributes
        // on failure
        if (is_drivetype_request(field_mask))
            osl_error = osl_get_drive_type(path, pInfo);
                
        if ((osl_File_E_None == osl_error) && is_filesystem_attributes_request(field_mask))
        {
            WCHAR vn[MAX_PATH];
	        WCHAR fsn[MAX_PATH];
	        DWORD serial;
	        DWORD mcl;
	        DWORD flags;
        	
	        if (GetVolumeInformation(path.getStr(), vn, MAX_PATH, &serial, &mcl, &flags, fsn, MAX_PATH))
	        {		
		        pInfo->uValidFields   |= osl_VolumeInfo_Mask_MaxNameLength;
		        pInfo->uMaxNameLength  = mcl;
        	
		        pInfo->uValidFields   |= osl_VolumeInfo_Mask_MaxPathLength;
		        pInfo->uMaxPathLength  = MAX_PATH;

		        pInfo->uValidFields   |= osl_VolumeInfo_Mask_FileSystemName;
		        rtl_uString_newFromStr(&pInfo->ustrFileSystemName, fsn);
        		
		        // volumes (even NTFS) will always be considered case
		        // insensitive because the Win32 API is not able to
		        // deal with case sensitive volumes see M$ Knowledge Base
		        // article 100625 that's why we never set the attribute
		        // osl_Volume_Attribute_Case_Sensitive
        		
		        if (flags & FS_CASE_IS_PRESERVED)
		            pInfo->uAttributes |= osl_Volume_Attribute_Case_Is_Preserved;
		            
		        pInfo->uValidFields |= osl_VolumeInfo_Mask_Attributes;
	        }
        }
        return osl_error;
    }
    
} // end namespace private 

       
/*******************************************************************
 osl_getVolumeInformation 
 ******************************************************************/

oslFileError SAL_CALL osl_getVolumeInformation( 
    rtl_uString *ustrURL, oslVolumeInfo *pInfo, sal_uInt32 uFieldMask )
{			
	if (!pInfo)
		return osl_File_E_INVAL;

    rtl::OUString system_path;		
	oslFileError error = _osl_getSystemPathFromFileURL(
	    ustrURL, &system_path.pData, sal_False);
    
    if (osl_File_E_None != error)
        return error;
    
    rtl::OUString volume_root;
    path_travel_to_volume_root(system_path, volume_root);        
    			
	pInfo->uValidFields = 0;
    
    if ((error = get_filesystem_attributes(volume_root, uFieldMask, pInfo)) != osl_File_E_None)
        return error;
            	
	if (is_volume_space_info_request(uFieldMask)) 
	    get_volume_space_information(volume_root, pInfo);
	    	    
	if (uFieldMask & osl_VolumeInfo_Mask_DeviceHandle)
	{
		pInfo->uValidFields |= osl_VolumeInfo_Mask_DeviceHandle;
		osl_getFileURLFromSystemPath(volume_root.pData, (rtl_uString**)&pInfo->pDeviceHandle);
	}
    
	return osl_File_E_None;
}