/*************************************************************************
 *
 *  OpenOffice.org - a multi-platform office productivity suite
 *
 *  $RCSfile: blankdispatcher.cxx,v $
 *
 *  $Revision: 1.34 $
 *
 *  last change: $Author: rt $ $Date: 2005/09/09 01:18:24 $
 *
 *  The Contents of this file are made available subject to
 *  the terms of GNU Lesser General Public License Version 2.1.
 *
 *
 *    GNU Lesser General Public License Version 2.1
 *    =============================================
 *    Copyright 2005 by Sun Microsystems, Inc.
 *    901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License version 2.1, as published by the Free Software Foundation.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *    MA  02111-1307  USA
 *
 ************************************************************************/

//_________________________________________________________________________________________________________________
//	my own includes
//_________________________________________________________________________________________________________________

#include "../loadenv/loadenv.hxx"

#ifndef __FRAMEWORK_DISPATCH_BLANKDISPATCHER_HXX_
#include <dispatch/blankdispatcher.hxx>
#endif

#ifndef __FRAMEWORK_THREADHELP_TRANSACTIONGUARD_HXX_
#include <threadhelp/transactionguard.hxx>
#endif

#ifndef __FRAMEWORK_THREADHELP_READGUARD_HXX_
#include <threadhelp/readguard.hxx>
#endif

#ifndef __FRAMEWORK_THREADHELP_WRITEGUARD_HXX_
#include <threadhelp/writeguard.hxx>
#endif

#ifndef __FRAMEWORK_CLASSES_FRAMELISTANALYZER_HXX_
#include <classes/framelistanalyzer.hxx>
#endif

#ifndef __FRAMEWORK_CLASSES_ARGUMENTANALYZER_HXX_
#include <classes/argumentanalyzer.hxx>
#endif

#ifndef __FRAMEWORK_CLASSES_TASKCREATOR_HXX_
#include <classes/taskcreator.hxx>
#endif

#ifndef __FRAMEWORK_CLASSES_FILTERCACHE_HXX_
#include <classes/filtercache.hxx>
#endif

#ifndef __FRAMEWORK_TARGETS_HXX_
#include <targets.h>
#endif

#ifndef __FRAMEWORK_PROPERTIES_H_
#include <properties.h>
#endif

//_________________________________________________________________________________________________________________
//	interface includes
//_________________________________________________________________________________________________________________

#ifndef _COM_SUN_STAR_UTIL_XMODIFIABLE_HPP_
#include <com/sun/star/util/XModifiable.hpp>
#endif

#ifndef _COM_SUN_STAR_FRAME_XSTORABLE_HPP_
#include <com/sun/star/frame/XStorable.hpp>
#endif

#ifndef _COM_SUN_STAR_AWT_XTOPWINDOW_HPP_
#include <com/sun/star/awt/XTopWindow.hpp>
#endif

#ifndef _COM_SUN_STAR_DOCUMENT_XACTIONLOCKABLE_HPP_
#include <com/sun/star/document/XActionLockable.hpp>
#endif

#ifndef _COM_SUN_STAR_FRAME_XFRAMESSUPPLIER_HPP_
#include <com/sun/star/frame/XFramesSupplier.hpp>
#endif

#ifndef _COM_SUN_STAR_BEANS_XPROPERTYSET_HPP_
#include <com/sun/star/beans/XPropertySet.hpp>
#endif

#ifndef _COM_SUN_STAR_FRAME_XDISPATCH_HPP_
#include <com/sun/star/frame/XDispatch.hpp>
#endif

#ifndef _COM_SUN_STAR_FRAME_XDISPATCHPROVIDER_HPP_
#include <com/sun/star/frame/XDispatchProvider.hpp>
#endif

#include <services/documentlist.hxx>

//_________________________________________________________________________________________________________________
//	includes of other projects
//_________________________________________________________________________________________________________________

//_________________________________________________________________________________________________________________
//	namespace
//_________________________________________________________________________________________________________________

namespace framework{

//_________________________________________________________________________________________________________________
//	non exported const
//_________________________________________________________________________________________________________________

//_________________________________________________________________________________________________________________
//	non exported definitions
//_________________________________________________________________________________________________________________

//_________________________________________________________________________________________________________________
//	declarations
//_________________________________________________________________________________________________________________

/*-************************************************************************************************************//**
    @short      standard ctor
    @descr      These initialize a new instance of this class with needed informations for work.

    @seealso    using at owner

    @param      "xFactory", reference to servicemanager for creation of new services
    @param      "xDesktop", reference to our owner, the Desktop!!!
    @return     -

    @onerror    -
*//*-*************************************************************************************************************/
BlankDispatcher::BlankDispatcher( const css::uno::Reference< css::lang::XMultiServiceFactory >& xFactory ,
                                  const css::uno::Reference< css::frame::XFrame >&              xDesktop, sal_Bool bDef )
		//	Init baseclasses first
        :   BaseDispatcher( xFactory, xDesktop )
        , m_bIsDefaultDispatcher( bDef )
        // Init member
{
}

/*-************************************************************************************************************//**
    @interface  XDispatch
    @short      try to handle(!) or load URL with arguments into new created task
    @descr      We try to handle it without creation of a new task. If this will fail - we start loading of it into
                a new created task. Some started proccesses are asynchron. So we are listener for different callbacks.

    @attention  These listener mechanism is implemented by our base class "BaseDispatcher"! So we don't must look at it ...
                Overwrite dispatch() and reactForLoadingState() only. Base class call this one.

    @seealso    baseclass BaseDispatcher

    @param      "aURL"      , URL to dispatch.
    @param      "lArguments", list of optional arguments.
    @return     -

    @onerror    -
    @threadsafe yes
*//*-*************************************************************************************************************/
void SAL_CALL BlankDispatcher::dispatch( const css::util::URL&                                  aURL,
                                         const css::uno::Sequence< css::beans::PropertyValue >& lArguments ) throw( css::uno::RuntimeException )
{
	/* UNSAFE AREA --------------------------------------------------------------------------------------------- */
    // Register transaction and reject wrong calls
    TransactionGuard aTransaction( m_aTransactionManager, E_HARDEXCEPTIONS );

    /* SAFE AREA ----------------------------------------------------------------------------------------------- */
    // Make snapshot of neccessary members
    ReadGuard aReadLock( m_aLock );
    css::uno::Reference< css::frame::XFrame >              xDesktop             ( m_xOwner.get(), css::uno::UNO_QUERY );
    css::uno::Reference< css::lang::XMultiServiceFactory > xFactory             = m_xFactory;
    sal_Bool                                               bIsDefaultDispatcher = m_bIsDefaultDispatcher;
    aReadLock.unlock();

	/* UNSAFE AREA --------------------------------------------------------------------------------------------- */

    sal_Bool bError = sal_False;
    try
    {
        LoadEnv aEnv(xFactory,
                     xDesktop,
                     aURL.Complete,
                     lArguments,
                     bIsDefaultDispatcher ? ::rtl::OUString::createFromAscii("_default") : ::rtl::OUString::createFromAscii("_blank"),
                     0,
                     LoadEnv::E_ALLOW_CONTENTHANDLER);

        aEnv.startLoading();
        aEnv.waitWhileLoading();
        aEnv.finishLoading();
    }
    catch(const LoadEnvException&)
    {
        LOG_WARNING("BlankDispatcher::dispatch()", "LoadEnvException :-(")
    }
}

/*-************************************************************************************************************//**
    @short          react for return state of dispatch call
    @descr          If we started our dispatch call ... our baseclass call us back to give us some informations about it.
                    So we can react for it. This implmentation create new tasks and try to load something in it.
                    If dispatch was successfully we must set visible state of this task - otherwise task must be deleted.
                    But there exist another mode: Some URLs was handled by an registered content handler.
                    These dispatches works without any tasks or frames!

    @attention      Base class call us back without any set lock or mutex. So we must do it by ourself!
                    BUT ... We don't need any lock here yet ... we don't work on internal member!
                    If YOU change it ... you should think about it again ...

    @seealso        base class BaseDispatcher

    @param          "aURL"       , dispatched URL
    @param          "lDescriptor", optional arguments of original dispatch call
    @param          "xTarget"    , target of dispatch (should be our new created task)
    @param          "bState"     , state of loading proccess (ok or failed)
    @return         -

    @onerror        -
    @threadsafe     yes
*//*-*************************************************************************************************************/
void SAL_CALL BlankDispatcher::reactForLoadingState( const css::util::URL&                                   aURL        ,
                                                     const css::uno::Sequence< css::beans::PropertyValue >&  lDescriptor ,
                                                     const css::uno::Reference< css::frame::XFrame >&        xTarget     ,
                                                           sal_Bool                                          bState      ,
                                                     const css::uno::Any&                                    aAsyncInfo  )
{
	/* UNSAFE AREA --------------------------------------------------------------------------------------------- */
    // Register transaction and reject wrong calls.
    TransactionGuard aTransaction( m_aTransactionManager, E_HARDEXCEPTIONS );

    // We don't need any lock here ... we don't work on internal member here!
    // But if YOU change it ... you should think about it again ...

    if( bState == sal_True )
    {
        implts_enableFrame( xTarget, lDescriptor ); // set visible state!
    }
    else if ( !xTarget->getComponentWindow().is() )
    {
        implts_disableFrame( xTarget ); // dispose it!
    }
}

//*****************************************************************************************************************
void SAL_CALL BlankDispatcher::reactForHandlingState( const css::util::URL&                                  aURL        ,
                                                      const css::uno::Sequence< css::beans::PropertyValue >& lDescriptor ,
                                                            sal_Bool                                         bState      ,
                                                      const css::uno::Any&                                   aAsyncInfo  )
{
	/* UNSAFE AREA --------------------------------------------------------------------------------------------- */
    // Register transaction and reject wrong calls.
    // TransactionGuard aTransaction( m_aTransactionManager, E_HARDEXCEPTIONS );

    /* TODO?!

        I don't know, what I can do here ... URL was handled without using of a task or frame.
        There is nothing to do. Listener are informaed by our baseclass automaticly - frame mustn't be
        initialized or deinitialized.
     */
}

/*-************************************************************************************************************//**
    @short          Tries to find frame that has already loaded the document specified by URL.
    @descr          We search on our desktop container to find any task, which has already loaded
                    given URL. If we found anyone we try to set focus on it to bring it to front.
                    But dont do it in special cases ... like: - hidden loaded documents
                                                              - loaded from template
                                                              - second view of a document ...

    @seealso        method dispatch()

    @param          "aURL"       , dispatched URL
    @param          "lArguments" , optional arguments of original dispatch call
    @param          "xDesktop"   , used for task access
    @return         true for found task
                    false otherwise

    @onerror        We return false.
    @threadsafe     yes
*//*-*************************************************************************************************************/
sal_Bool BlankDispatcher::implts_findAndActivateAlreadyLoadedTask( const css::util::URL&                                   aURL       ,
                                                                   const css::uno::Sequence< css::beans::PropertyValue >&  lArguments ,
                                                                   const css::uno::Reference< css::frame::XFrame >         xDesktop   )
{
    ArgumentAnalyzer aAnalyzer(lArguments, sal_True);

    // break search if new load proccess is a special request!
    sal_Bool bValue = sal_False;
    if(
        ( aAnalyzer.getArgument( E_ASTEMPLATE , bValue ) && bValue )    ||
        ( aAnalyzer.getArgument( E_HIDDEN     , bValue ) && bValue )    ||
        ( aAnalyzer.getArgument( E_OPENNEWVIEW, bValue ) && bValue )
      )
    {
        return sal_False;
    }

    sal_Int16 nNewVersion = 0;  // Must be short! Otherwhise type check will fail ...
    aAnalyzer.getArgument( E_VERSION, nNewVersion );

    // iterate through the tasks
    css::uno::Reference< css::frame::XFrame >           xTask        ;
    css::uno::Reference< css::frame::XFramesSupplier >  xSupplier    ( xDesktop, css::uno::UNO_QUERY );
    css::uno::Reference< css::container::XIndexAccess > xTaskList    ( xSupplier->getFrames(), css::uno::UNO_QUERY );
    sal_Int32 nCount = xTaskList->getCount();
    for( sal_Int32 i=0; i<nCount; ++i )
    {
        css::uno::Any aElement = xTaskList->getByIndex(i);
        if ( !(aElement>>=xTask) || !xTask.is() )
            continue;

        css::uno::Reference< css::frame::XController > xController  ;
        css::uno::Reference< css::frame::XModel >      xModel       ;

        if( xTask.is() )
            xController = xTask->getController();

        if( xController.is() )
			xModel = xController->getModel();

        /*FIXME
            comparing of the old and new URL can fail ...
            if some letters are lower/upper case.
            But it's not allowed to compare e.g. lower case only.
            We must wait for a specialized UCB method, which
            provides the possibility to check, if two URL points
            to the same ressource.
         */
        if (xModel.is())
        {
            ::rtl::OUString sURL = xModel->getURL();
            if (aURL.Main.equals(sURL))
            {
                sal_Bool  bHidden     = sal_False;
                sal_Int16 nOldVersion = 0; // Must be short! Otherwhise type check will fail ...

                aAnalyzer.setArguments(xModel->getArgs(), sal_True);
                aAnalyzer.getArgument( E_HIDDEN , bHidden     );
                aAnalyzer.getArgument( E_VERSION, nOldVersion );

                if(
                    ( bHidden     == sal_False   )   &&
                    ( nNewVersion == nOldVersion )
                )
                {
				    if(aURL.Mark.getLength()>0)
				    {
					    css::uno::Reference< css::frame::XDispatchProvider > xProvider( xTask, css::uno::UNO_QUERY );
					    css::uno::Reference< css::frame::XDispatch > xDispatcher = xProvider->queryDispatch(aURL,SPECIALTARGET_SELF,0);
					    if(xDispatcher.is())
						    xDispatcher->dispatch(aURL,lArguments);
				    }
				    css::uno::Reference< css::awt::XWindow >    xTaskWindow= xTask->getContainerWindow();
				    css::uno::Reference< css::awt::XTopWindow > xTopWindow ( xTaskWindow, css::uno::UNO_QUERY );
				    if( xTopWindow.is() )
				    {
					    xTaskWindow->setVisible( sal_True );
					    xTopWindow->toFront();
				    }
                    // Don't forget to send right event to all registered listener!
                    // These activation was synchron for user ...
                    // If we forget it - he waits for ever.
                    implts_sendResultEvent( xTask, aURL.Complete, sal_True );
                    return sal_True;
                }
            }
        }
    }

    return sal_False;
}

/// @HTML
/** @short  tries to find a frame, which can be recycled for this "_default" dispatch.

    @descr  We iterate on our desktop container through all available tasks and
            recylce tasks only, if they match these criterions:
            <ul>
                <li>old and new document must use the same application module</li>
                <li>old document is not modified or already saved</li>
                <li>old document does not have any still running processes inside</li>
                <li>new request is not a hidden one</li>
                <li>new request don't ask explicitly for a new view</li>
                <li>task is not used by any other concurrent load process</li>
            </ul>

    @param  aURL
            URL for the new requested document

    @param  lArguments
            optional arguments for this new dispatch

    @param  xDesktop
            used to iterate through the list of tasks

    @return A valid reference to a task, which can be used and
            was already locked(!) for this new dispatch request.
            It returns an empty reference if no task could be located,
            which match the search criterions.
 */
/// @NOHTML

css::uno::Reference< css::frame::XFrame > BlankDispatcher::implts_findAndLockRecycleTask( const css::util::URL&                                   aURL       ,
                                                                                          const css::uno::Sequence< css::beans::PropertyValue >&  lArguments ,
                                                                                          const css::uno::Reference< css::frame::XFrame >         xDesktop   )
{
    //------------------------------------------------------------
    // i.a) don't recycle anything for a hidden request
    sal_Bool         bValue       = sal_False;
    ArgumentAnalyzer aArgsAnalyzer(lArguments, sal_True);
    if (aArgsAnalyzer.getArgument(E_HIDDEN, bValue) && bValue)
        return css::uno::Reference< css::frame::XFrame >();

    //------------------------------------------------------------
    // ii) iterate through the tasks
    //     But if no supplier is available ... nothing can be done then.
    css::uno::Reference< css::frame::XFramesSupplier > xSupplier(xDesktop, css::uno::UNO_QUERY);
    if (!xSupplier.is())
        return css::uno::Reference< css::frame::XFrame >();

    //------------------------------------------------------------
    // ii.i) the special backing frame will be recycled everytimes
    FrameListAnalyzer aTasksAnalyzer(xSupplier, css::uno::Reference< css::frame::XFrame >(), FrameListAnalyzer::E_BACKINGCOMPONENT);
    if (aTasksAnalyzer.m_xBackingComponent.is())
        return aTasksAnalyzer.m_xBackingComponent;

    //------------------------------------------------------------
    // ii.ii) new docs/views except in case of backing window -> no recycling
    //        a) "private:factory/" open new documents ...
    //           but we recycle only new documents using the same module!
    //           It would mean, we replace e.g. an empty writer with an empty writer document.
    //           That make no sense. So we should open a new task.
    //        b) OPENNEWVIEW / ASTEMPLATE force creating of a new view ...
    //           So it overwrites the target "_default" and must be opened inside a new task ...
    //           excepting the special backing component exist. It's not allowed to have
    //           an open document against this special one. So we must recycle it.
    if (
        (aURL.Complete.compareToAscii("private:factory/", 16) == COMPARE_EQUAL) ||
        (aArgsAnalyzer.getArgument(E_OPENNEWVIEW, bValue) && bValue           ) ||
        (aArgsAnalyzer.getArgument(E_ASTEMPLATE , bValue) && bValue           )
       )
    {
        return css::uno::Reference< css::frame::XFrame >();
    }

    //------------------------------------------------------------
    // ii.ii) otherwhise we have to look, if the cative task contains the same modul
    //        then the new document will have ... and if it's not already in use or modified.
    css::uno::Reference< css::frame::XFrame > xActive(xSupplier->getActiveFrame(), css::uno::UNO_QUERY);
    if (!xActive.is())
		return css::uno::Reference< css::frame::XFrame >();

    css::uno::Reference< css::frame::XModel >      xModel      ;
    css::uno::Reference< css::frame::XController > xController = xActive->getController();
    if (xController.is())
        xModel = xController->getModel();
    css::uno::Reference< css::util::XModifiable > xModify(xModel, css::uno::UNO_QUERY);
    css::uno::Reference< css::frame::XStorable >  xStore (xModel, css::uno::UNO_QUERY);
    // only new created, not modified and not already stored documents can be used!
    if (!xModify.is() || !xStore.is() || xModify->isModified() || xStore->hasLocation())
        return css::uno::Reference< css::frame::XFrame >();

    FilterCache aCache    ;
    Filter      aNewFilter;
    Filter      aOldFilter;

    // try to get the filter, which should be used for the new document
    ::rtl::OUString sFilterName;
    if (!aArgsAnalyzer.getArgument(E_FILTERNAME, sFilterName) || !aCache.existsFilter(sFilterName))
        return css::uno::Reference< css::frame::XFrame >();
    aNewFilter = aCache.getFilter(sFilterName);

    // change the media descriptor, so we can get the filter of the current loaded document
    aArgsAnalyzer.setArguments(xModel->getArgs(),sal_True);
    sFilterName = ::rtl::OUString();
    if (!aArgsAnalyzer.getArgument(E_FILTERNAME, sFilterName) || !aCache.existsFilter(sFilterName))
        return css::uno::Reference< css::frame::XFrame >();
    aOldFilter = aCache.getFilter(sFilterName);

    // only document with the same module are used
    if (!aOldFilter.sDocumentService.equals(aNewFilter.sDocumentService))
        return css::uno::Reference< css::frame::XFrame >();

    //------------------------------------------------------------
    // ii.iii) look for an optional(!) action lock.
    //         It indicates current using of this task for a
    //         load request - may started in another thread!
    css::uno::Reference< css::document::XActionLockable > xLockCheck(xActive, css::uno::UNO_QUERY);
    if (xLockCheck.is())
    {
        if (xLockCheck->isActionLocked())
            return css::uno::Reference< css::frame::XFrame >();
        // If it seams, that this task is not used by external processes ...
        // reserve it for this new request by setting an action lock!
        xLockCheck->addActionLock();
    }

    //------------------------------------------------------------
    // ii.iv) try to suspend the old view inside this active frame
    //        If it's not successfully - this task can't be recycled realy.
    if (!xController->suspend(sal_True))
    {
        // Attention! Don't forget to remove our own set action lock!
        // Otherwhise this task could not be used by any other request ...
        if (xLockCheck.is())
            xLockCheck->removeActionLock();
        return css::uno::Reference< css::frame::XFrame >();
    }

    // OK - we found a valid task, which can be recycled for this new request
    // And it's locked by us :-)
    return xActive;
}

}		//	namespace framework
