/* $Id: log.c,v 1.79 2001/11/20 04:44:26 roesch Exp $ */
/*
** Copyright (C) 1998,1999,2000,2001 Martin Roesch <roesch@clark.net>
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#include "log.h"

extern OptTreeNode *otn_tmp;    /* global ptr to current rule data */

char *data_dump_buffer;     /* printout buffer for PrintNetData */
int dump_ready;         /* flag to indicate status of printout buffer */
int dump_size;          /* size of printout buffer */
extern u_int32_t event_id;

char *logfile[] =
        { "", "PACKET_FRAG", "PACKET_BOGUS", "PACKET_NONIP", "ARP", "log" };

static unsigned char ezero[6];  /* crap for ARP */

/*
 * Function: OpenLogFile()
 *
 * Purpose: Create the log directory and file to put the packet log into.
 *          This function sucks, I've got to find a better way to do this
 *          this stuff.
 *
 * Arguments: None.
 *
 * Returns: 0 on success, exits on error
 */
int OpenLogFile(int mode, Packet * p)
{
    char log_path[STD_BUF+1]; /* path to log file */
    char log_file[STD_BUF+1]; /* name of log file */
    char proto[5];      /* logged packet protocol */
    char suffix[5];     /* filename suffix */
#ifdef WIN32
    strcpy(suffix,".ids");
#else
    suffix[0] = '\0';
#endif

    /* zero out our buffers */
    bzero((char *) log_path, STD_BUF + 1);
    bzero((char *) log_file, STD_BUF + 1);
    bzero((char *) proto, 5);

    if (mode == GENERIC_LOG || mode == DUMP || mode == BOGUS ||
            mode == NON_IP || mode == ARP)
    {
        snprintf(log_file, STD_BUF, "%s%s/%s",
                chrootdir == NULL ? "" : chrootdir, pv.log_dir, logfile[mode]);

        if((log_ptr = fopen(log_file, "a")) == NULL)
        {
            FatalError("ERROR: OpenLogFile() => fopen(%s) log file: %s\n",
                    log_file, strerror(errno));
        }
        return 0;
    }

    if(otn_tmp != NULL)
    {
        if(otn_tmp->logto != NULL)
        {
            snprintf(log_file, STD_BUF, "%s%s/%s",
                    chrootdir == NULL ? "" : chrootdir,  pv.log_dir, 
                                             otn_tmp->logto);

            if((log_ptr = fopen(log_file, "a")) == NULL)
            {
                FatalError("ERROR: OpenLogFile() => fopen(%s) log file: %s\n", 
                            log_file, strerror(errno));
            }
            return 0;
        }
    }
    /* figure out which way this packet is headed in relation to the homenet */
    if((p->iph->ip_dst.s_addr & pv.netmask) == pv.homenet)
    {
        if((p->iph->ip_src.s_addr & pv.netmask) != pv.homenet)
        {
            snprintf(log_path, STD_BUF, "%s%s/%s",
                    chrootdir == NULL ? "" : chrootdir, pv.log_dir, 
                                             inet_ntoa(p->iph->ip_src));
        }
        else
        {
            if(p->sp >= p->dp)
            {
                snprintf(log_path, STD_BUF, "%s%s/%s",
                        chrootdir == NULL ? "" : chrootdir,  pv.log_dir, 
                                                 inet_ntoa(p->iph->ip_src));
            }
            else
            {
                snprintf(log_path, STD_BUF, "%s%s/%s",
                        chrootdir == NULL ? "" : chrootdir, pv.log_dir, 
                                                 inet_ntoa(p->iph->ip_dst));
            }
        }
    }
    else
    {
        if((p->iph->ip_src.s_addr & pv.netmask) == pv.homenet)
        {
            snprintf(log_path, STD_BUF, "%s%s/%s",
                    chrootdir == NULL ? "" : chrootdir,  pv.log_dir, 
                                             inet_ntoa(p->iph->ip_dst));
        }
        else
        {
            if(p->sp >= p->dp)
            {
                snprintf(log_path, STD_BUF, "%s%s/%s",
                        chrootdir == NULL ? "" : chrootdir, pv.log_dir, 
                                                 inet_ntoa(p->iph->ip_src));
            }
            else
            {
                snprintf(log_path, STD_BUF, "%s%s/%s",
                        chrootdir == NULL ? "" : chrootdir, pv.log_dir, 
                                                 inet_ntoa(p->iph->ip_dst));
            }
        }
    }

    DebugMessage(DEBUG_FLOW, "Creating directory: %s\n", log_path);

    /* build the log directory */
    if(mkdir(log_path, S_IRWXU | S_IRWXG | S_IROTH | S_IXOTH))
    {

        if(errno != EEXIST)
        {
            FatalError("ERROR: OpenLogFile() => mkdir(%s) log directory: %s\n",
                        log_path, strerror(errno));
        }
    }

    DebugMessage(DEBUG_FLOW, "Directory Created!\n");

    /* build the log filename */
    if(p->iph->ip_proto == IPPROTO_TCP ||
            p->iph->ip_proto == IPPROTO_UDP)
    {
        if(p->frag_flag)
        {
            snprintf(log_file, STD_BUF, "%s/IP_FRAG%s", log_path, suffix);
        }
        else
        {
            if(p->sp >= p->dp)
            {
#ifdef WIN32
                snprintf(log_file, STD_BUF, "%s/%s_%d-%d%s", log_path,
                        protocol_names[p->iph->ip_proto], p->sp, p->dp, suffix);
#else
                snprintf(log_file, STD_BUF, "%s/%s:%d-%d%s", log_path,
                        protocol_names[p->iph->ip_proto], p->sp, p->dp, suffix);
#endif
            }
            else
            {
#ifdef WIN32
                snprintf(log_file, STD_BUF, "%s/%s_%d-%d%s", log_path,
                        protocol_names[p->iph->ip_proto], p->dp, p->sp, suffix);
#else
                snprintf(log_file, STD_BUF, "%s/%s:%d-%d%s", log_path,
                        protocol_names[p->iph->ip_proto], p->dp, p->sp, suffix);
#endif
            }
        }
    }
    else
    {
        if(p->frag_flag)
        {
            snprintf(log_file, STD_BUF, "%s/IP_FRAG%s", log_path, suffix);
        }
        else
        {
            if(p->iph->ip_proto == IPPROTO_ICMP)
            {
                snprintf(log_file, STD_BUF, "%s/%s_%s%s", log_path, "ICMP",
                         IcmpFileName(p), suffix);
            }
            else
            {
                snprintf(log_file, STD_BUF, "%s/PROTO%d%s", log_path,
                         p->iph->ip_proto, suffix);
            }
        }
    }

    DebugMessage(DEBUG_FLOW, "Opening file: %s\n", log_file);

    /* finally open the log file */
    if((log_ptr = fopen(log_file, "a")) == NULL)
    {
        FatalError("ERROR: OpenLogFile() => fopen(%s) log file: %s\n",
                log_file, strerror(errno));
    }

    DebugMessage(DEBUG_FLOW, "File opened...\n");

    return 0;
}


/*
 * Function: PrintNetData(FILE *, u_char *,int)
 *
 * Purpose: Do a side by side dump of a buffer, hex dump of buffer bytes on
 *          the left, decoded ASCII on the right.
 *
 * Arguments: fp => ptr to stream to print to
 *            start => pointer to buffer data
 *            len => length of data buffer
 *
 * Returns: void function
 */
void PrintNetData(FILE * fp, u_char * start, const int len)
{
    char *end;          /* ptr to buffer end */
    int i;          /* counter */
    int j;          /* counter */
    int dbuf_size;      /* data buffer size */
    int done;           /* flag */
    char *data;         /* index pointer */
    char *frame_ptr;        /* we use 66 byte frames for a printed line */
    char *d_ptr;        /* data pointer into the frame */
    char *c_ptr;        /* char pointer into the frame */
    char conv[] = "0123456789ABCDEF";   /* xlation lookup table */

    /* initialization */
    done = 0;

    if(start == NULL)
    {
        printf("Got NULL ptr in PrintNetData()\n");
        return;
    }
    /* zero, print a <CR> and get out */
    if(!len)
    {
        fputc('\n', fp);
        return;
    }
    /*
     * if we've already prepared this particular data buffer, just print it
     * out again to save time
     */
    if(dump_ready)
    {
        fwrite(data_dump_buffer, dump_size, 1, fp);
        fflush(fp);
        return;
    }
    end = (char*) (start + (len - 1));    /* set the end of buffer ptr */

    /*if(len > pv.mtus[0])*/
    if(len > 65535)
    {
        if(pv.verbose_flag)
        {
            printf("Got bogus buffer length (%d) for PrintNetData, defaulting to 16 bytes!\n", len);
        }

        if(pv.verbose_bytedump_flag == 1)
        {
            dbuf_size = (FRAME_SIZE + 8) + (FRAME_SIZE + 8) + 1;
        }
        else
        {
            dbuf_size = FRAME_SIZE + FRAME_SIZE + 1;
        }

        /* dbuf_size = 66 + 67; */
        end =  (char*) (start + 15);
    }
    else
    {
        if(pv.verbose_bytedump_flag == 1)
        {
            /* figure out how big the printout data buffer has to be */
            dbuf_size = ((len / 16) * (FRAME_SIZE + 8)) + (FRAME_SIZE + 8) + 1;
        }
        else
        {
            /* figure out how big the printout data buffer has to be */
            dbuf_size = ((len / 16) * FRAME_SIZE) + FRAME_SIZE + 1;
        }

        /* dbuf_size = ((len / 16) * 66) + 67; */
    }

    /* generate the buffer */
    data_dump_buffer = (char *) malloc(dbuf_size);

    /* make sure it got allocated properly */
    if(data_dump_buffer == NULL)
    {
        ErrorMessage("Failed allocating %X bytes! (Length: %X)\n",
                dbuf_size, len);
        perror("PrintNetData()");
        CleanExit(SIGQUIT);
    }
    /* clean it out */
    memset(data_dump_buffer, 0x20, dbuf_size);


    /* set the byte buffer pointer to step thru the data buffer */
    data = (char*) start;

    /* set the frame pointer to the start of the printout buffer */
    frame_ptr = data_dump_buffer;

    /* initialize counters and frame index pointers */
    i = 0;
    j = 0;

    /* loop thru the whole buffer */
    while(!done)
    {
        if(pv.verbose_bytedump_flag == 1)
        {
            d_ptr = frame_ptr + 8;
            c_ptr = (frame_ptr + 8 + C_OFFSET);
            sprintf(frame_ptr, "0x%04X: ", j);
            j += 16;
        }
        else
        {
            d_ptr = frame_ptr;
            c_ptr = (frame_ptr + C_OFFSET);
        }

        /* process 16 bytes per frame */
        for(i = 0; i < 16; i++)
        {
            /*
             * look up the ASCII value of the first nybble of the current
             * data buffer
             */
            *d_ptr = conv[((*data & 0xFF) >> 4)];
            d_ptr++;

            /* look up the second nybble */
            *d_ptr = conv[((*data & 0xFF) & 0x0F)];
            d_ptr++;

            /* put a space in between */
            *d_ptr = 0x20;
            d_ptr++;

            /* print out the char equivalent */
            if(*data > 0x1F && *data < 0x7F)
                *c_ptr = (char) (*data & 0xFF);
            else
                *c_ptr = 0x2E;

            c_ptr++;

            /* increment the pointer or finish up */
            if(data < end)
                data++;
            else
            {
                /* finish up the buffer printout and set the "ready" flags */
                done = 1;
                dump_ready = 1;

                *c_ptr = '\n';
                c_ptr++;
                *c_ptr = '\n';
                c_ptr++;
                *c_ptr = 0;

                dump_size = (int) (c_ptr - data_dump_buffer);
                fwrite(data_dump_buffer, dump_size, 1, fp);
                return;
            }
        }

        *c_ptr = '\n';
        if(pv.verbose_bytedump_flag == 1)
        {
            frame_ptr += (FRAME_SIZE + 8);
        }
        else
        {
            frame_ptr += FRAME_SIZE;
        }
    }
}



/*
 * Function: PrintCharData(FILE *, char *,int)
 *
 * Purpose: Dump the ASCII data from a packet
 *          the left, decoded ASCII on the right.
 *
 * Arguments: fp => ptr to stream to print to
 *            data => pointer to buffer data
 *            data_len => length of data buffer
 *
 * Returns: void function
 */
void PrintCharData(FILE * fp, char *data, int data_len)
{
    int bytes_processed;    /* count of bytes in the data buffer
                 * processed so far */
    int linecount = 0;      /* number of lines in this dump */
    char *index;        /* index pointer into the data buffer */
    char *ddb_ptr;      /* index pointer into the data_dump_buffer */

    /* if there's no data, return */
    if(data == NULL)
    {
        return;
    }
    /* if we've already setup the buffer, just reprint it */
    if(dump_ready)
    {
        fwrite(data_dump_buffer, dump_size, 1, fp);
        fflush(fp);
        return;
    }
    /* setup the pointers and counters */
    bytes_processed = data_len;
    index = data;

    /* allocate a buffer to print the data to */
    data_dump_buffer = (char *) calloc(data_len + (data_len >> 6) + 2, sizeof(char));
    ddb_ptr = data_dump_buffer;

    /* loop thru the bytes in the data buffer */
    while(bytes_processed)
    {
        if(*index > 0x1F && *index < 0x7F)
        {
            *ddb_ptr = *index;
        }
        else
        {
            *ddb_ptr = '.';
        }

        if(++linecount == 64)
        {
            ddb_ptr++;
            *ddb_ptr = '\n';
            linecount = 0;
        }
        ddb_ptr++;
        index++;
        bytes_processed--;
    }

    /* slam a \n on the back */
    ddb_ptr++;
    *ddb_ptr = '\n';
    ddb_ptr++;

    /* setup the globals */
    dump_ready = 1;
    dump_size = (int) (ddb_ptr - data_dump_buffer);
    fwrite(data_dump_buffer, dump_size, 1, fp);
}



/*
 * Function: PrintIPPkt(FILE *, int, Packet *)
 *
 * Purpose: Dump the packet to the stream pointer
 *
 * Arguments: fp => pointer to print data to
 *            type => packet protocol
 *            p => pointer to decoded packet struct
 *
 * Returns: void function
 */
void PrintIPPkt(FILE * fp, int type, Packet * p)
{
    char timestamp[TIMEBUF_SIZE];

    DebugMessage(DEBUG_LOG, "PrintIPPkt type = %d\n", type);

    bzero((char *) timestamp, TIMEBUF_SIZE);
    ts_print((struct timeval *) & p->pkth->ts, timestamp);

    /* dump the timestamp */
    fwrite(timestamp, strlen(timestamp), 1, fp);

    /* dump the ethernet header if we're doing that sort of thing */
    if(pv.show2hdr_flag)
    {
        Print2ndHeader(fp, p);
    }

    /* etc */
    PrintIPHeader(fp, p);

    /* if this isn't a fragment, print the other header info */
    if(!p->frag_flag)
    {
        switch(p->iph->ip_proto)
        {
            case IPPROTO_TCP:
                if(p->tcph != NULL)
                {
                    PrintTCPHeader(fp, p);
                }
                else
                {
                    PrintNetData(fp, (u_char *) (p->iph + (p->iph->ip_hlen << 2)), (p->iph->ip_len - (p->iph->ip_hlen << 2)));
                }

                break;

            case IPPROTO_UDP:
                if(p->udph != NULL)
                {
                    PrintUDPHeader(fp, p);
                }
                else
                {
                    PrintNetData(fp, (u_char *) (p->iph + (p->iph->ip_hlen << 2)), (p->iph->ip_len - (p->iph->ip_hlen << 2)));
                }

                break;

            case IPPROTO_ICMP:
                if(p->icmph != NULL)
                {
                    PrintICMPHeader(fp, p);
                }
                else
                {
                    PrintNetData(fp, (u_char *) (p->iph + (p->iph->ip_hlen << 2)), (p->iph->ip_len - (p->iph->ip_hlen << 2)));
                }

                break;

            default:
                break;
        }
    }
    /* dump the application layer data */
    if(pv.data_flag && !pv.verbose_bytedump_flag)
    {
        if(pv.char_data_flag)
            PrintCharData(fp, (char*) p->data, p->dsize);
        else
            PrintNetData(fp, p->data, p->dsize);
    }
    else if(pv.verbose_bytedump_flag)
    {
        PrintNetData(fp, p->pkt, p->pkth->caplen);
    }

    fprintf(fp, "=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+\n\n");
}




/*
 * Function: OpenAlertSock
 *
 * Purpose:  Connect to UNIX socket for alert logging..
 *
 * Arguments: none..
 *
 * Returns: void function
 */
void OpenAlertSock()
{
    char *srv = UNSOCK_FILE;

    if(access(srv, W_OK))
    {
        ErrorMessage("WARNING: %s file doesn't exist or isn't writable!\n", srv);
    }
    bzero((char *) &alertaddr, sizeof(alertaddr));
    bcopy((const void *) srv, (void *) alertaddr.sun_path, strlen(srv));    /* we trust what we
                                         * define */
    alertaddr.sun_family = AF_UNIX;

    if((alertsd = socket(AF_UNIX, SOCK_DGRAM, 0)) < 0)
    {
        FatalError("socket() call failed: %s", strerror(errno));
    }
}



/****************************************************************************
 *
 * Function: OpenAlertFile(char *)
 *
 * Purpose: Set up the file pointer/file for alerting
 *
 * Arguments: filearg => the filename to open
 *
 * Returns: file handle
 *
 ***************************************************************************/
FILE *OpenAlertFile(char *filearg)
{
    char filename[STD_BUF+1];
    FILE *file;
    char suffix[5];     /* filename suffix */
#ifdef WIN32
    strcpy(suffix,".ids");
#else
    suffix[0] = '\0';
#endif

    if(filearg == NULL)
    {
        if(!pv.daemon_flag)
            snprintf(filename, STD_BUF, "%s%s/alert%s", chrootdir == NULL ?
                     "" : chrootdir, pv.log_dir, suffix);
        else
            snprintf(filename, STD_BUF, "%s%s/%s", chrootdir == NULL ?
                     "" : chrootdir, pv.log_dir, DEFAULT_DAEMON_ALERT_FILE);
    }
    else
    {
        snprintf(filename, STD_BUF, "%s%s", chrootdir == NULL ?
                 "" : chrootdir, filearg);
    }


#ifdef DEBUG
    printf("Opening alert file: %s\n", filename);
#endif

    if((file = fopen(filename, "a")) == NULL)
    {
        FatalError("ERROR in OpenAlertFile() => fopen() alert file %s: %s\n",
                   filename, strerror(errno));
    }
#ifdef WIN32
    /* Do not buffer in WIN32 */
    setvbuf(file, (char *) NULL, _IONBF, (size_t) 0);
#else
    setvbuf(file, (char *) NULL, _IOLBF, (size_t) 0);
#endif

    return file;
}



/*
 *
 * Function: ClearDumpBuf()
 *
 * Purpose: Clear out the buffer that PrintNetData() generates
 *
 * Arguments: None.
 *
 * Returns: void function
 *
 */
void ClearDumpBuf()
{
    if(data_dump_buffer != NULL && dump_ready)
        free(data_dump_buffer);
    else
        return;

    data_dump_buffer = NULL;

    dump_ready = 0;
}


/*
 * Function: FullAlert(Packet *, char *, void *)
 *
 * Purpose: Stub function for compatability
 *
 * Arguments:   p => ptr to packet data
 *            msg => message to send to alert facility
 *            arg => arguments to the alert facility
 *
 * Returns: void function
 */
void FullAlert(Packet * p, char *msg, void *arg, Event *event)
{
    AlertFull(p, msg, alert, event);

    return;
}



/*
 *
 * Function: AlertFull(char *)
 *
 * Purpose: Write a full and informative alert message
 *
 * Arguments: 	p   => packet. (could be NULL)
 * 	            msg => the message to send
 *             file => file pointer to print data to
 *
 * Returns: void function
 *
 */
void AlertFull(Packet * p, char *msg, FILE * file, Event *event)
{
    char timestamp[TIMEBUF_SIZE];

    if(msg != NULL)
    {
        fwrite("[**] ", 5, 1, file);

        if(event != NULL)
        {
                fprintf(file, "[%lu:%lu:%lu] ", 
                        (unsigned long) event->sig_generator,
                        (unsigned long) event->sig_id, 
                        (unsigned long) event->sig_rev);
        }

        if(pv.alert_interface_flag)
        {
            fprintf(file, " <%s> ", PRINT_INTERFACE(pv.interfaces[0]));
            fwrite(msg, strlen(msg), 1, file);
            fwrite(" [**]\n", 6, 1, file);
        }
        else
        {
            fwrite(msg, strlen(msg), 1, file);
            fwrite(" [**]\n", 6, 1, file);
        }
    }
    else
    {
        fwrite("[**] Snort Alert! [**]\n", 23, 1, file);
    }

    if(p && p->iph)
    {
        PrintPriorityData(file, 1);
    }

#ifdef DEBUG
    printf("Logging Alert data!\n");
#endif

    bzero((char *) timestamp, TIMEBUF_SIZE);
    ts_print(p == NULL ? NULL : (struct timeval *) & p->pkth->ts, timestamp);

    /* dump the timestamp */
    fwrite(timestamp, strlen(timestamp), 1, file);

    if(p && p->iph)
    {
        /* print the packet header to the alert file */

        if(pv.show2hdr_flag)
        {
            Print2ndHeader(file, p);
        }

        PrintIPHeader(file, p);

        /* if this isn't a fragment, print the other header info */
        if(!p->frag_flag)
        {
            switch(p->iph->ip_proto)
            {
                case IPPROTO_TCP:
                    PrintTCPHeader(file, p);
                    break;

                case IPPROTO_UDP:
                    PrintUDPHeader(file, p);
                    break;

                case IPPROTO_ICMP:
                    PrintICMPHeader(file, p);
                    break;

                default:
                    break;
            }

            PrintXrefs(file, 1);
        }

        fputc('\n', file);
    } /* End of if(p) */
    else
    {
        fputs("\n\n", file);
    }
            

    return;
}


void FastAlert(Packet * p, char *msg, void *arg, Event *event)
{
    AlertFast(p, msg, alert, event);
    return;
}

/****************************************************************************
 *
 * Function: FastAlert(Packet *, char *)
 *
 * Purpose: Write a minimal alert message to the alert file
 *
 * Arguments: p => pointer to the packet data struct (could be NULL)
 *            msg => the message to print in the alert
 *
 * Returns: void function
 *
 ***************************************************************************/
void AlertFast(Packet * p, char *msg, FILE * file, Event *event)
{
    char timestamp[TIMEBUF_SIZE];

    bzero((char *) timestamp, TIMEBUF_SIZE);
    ts_print(p == NULL ? NULL : (struct timeval *) & p->pkth->ts, timestamp);

    /* dump the timestamp */
    fwrite(timestamp, strlen(timestamp), 1, file);

    if(msg != NULL)
    {
        fwrite(" [**] ", 6, 1, file);

        if(event != NULL)
        {
                fprintf(file, "[%lu:%lu:%lu] ", 
                        (unsigned long) event->sig_generator,
                        (unsigned long) event->sig_id, 
                        (unsigned long) event->sig_rev);
        }

        if(pv.alert_interface_flag)
        {
            fprintf(file, " <%s> ", PRINT_INTERFACE(pv.interfaces[0]));
            fwrite(msg, strlen(msg), 1, file);
        }
        else
        {
            fwrite(msg, strlen(msg), 1, file);
        }

        fwrite(" [**] ", 6, 1, file);
    }

    /* print the packet header to the alert file */
    if(p && p->iph)
    {
        PrintPriorityData(file, 0);

        fprintf(file, "{%s} ", protocol_names[p->iph->ip_proto]);
                

        if(p->frag_flag)
        {
            /* just print the straight IP header */
            fputs(inet_ntoa(p->iph->ip_src), file);
            fwrite(" -> ", 4, 1, file);
            fputs(inet_ntoa(p->iph->ip_dst), file);
        }
        else
        {
            switch(p->iph->ip_proto)
            {
                case IPPROTO_UDP:
                case IPPROTO_TCP:
                    /* print the header complete with port information */
                    fputs(inet_ntoa(p->iph->ip_src), file);
                    fprintf(file, ":%d -> ", p->sp);
                    fputs(inet_ntoa(p->iph->ip_dst), file);
                    fprintf(file, ":%d", p->dp);
                    break;
                case IPPROTO_ICMP:
                default:
                    /* just print the straight IP header */
                    fputs(inet_ntoa(p->iph->ip_src), file);
                    fwrite(" -> ", 4, 1, file);
                    fputs(inet_ntoa(p->iph->ip_dst), file);
            }
        }
    }               /* end of if (p) */

    fputc('\n', file);

    return;
}




/****************************************************************************
 *
 * Function: SyslogAlert(Packet *, char *)
 *
 * Purpose: Send the current alert to syslog
 *
 * Arguments: p => pointer to the packet data struct
 *            msg => the message to print in the alert
 *
 * Returns: void function
 *
 ***************************************************************************/
void SyslogAlert(Packet * p, char *msg, void *arg, Event *event)
{
    char sip[16];
    char dip[16];
    char pri_data[STD_BUF];
    char ip_data[STD_BUF];
    char event_data[STD_BUF];
#define SYSLOG_BUF  1024
    char event_string[SYSLOG_BUF];
    PriorityData *ds_ptr = NULL;  

    bzero(event_string, SYSLOG_BUF);

    if(p && p->iph)
    {
        /*
         * have to do this since inet_ntoa is fucked up and writes to a static
         * memory location
         */
        strlcpy(sip, inet_ntoa(p->iph->ip_src), 16);
        strlcpy(dip, inet_ntoa(p->iph->ip_dst), 16);

        if(event != NULL)
        {
            snprintf(event_data, STD_BUF-1, "[%lu:%lu:%lu] ", 
                    (unsigned long) event->sig_generator,
                    (unsigned long) event->sig_id, 
                    (unsigned long) event->sig_rev);
            strlcat(event_string, event_data, SYSLOG_BUF);
        }

        if(msg != NULL)
        {
            strlcat(event_string, msg, SYSLOG_BUF);
        }
        else
        {
            strlcat(event_string, "ALERT", SYSLOG_BUF);
        }

        if(otn_tmp != NULL)
        {
            ds_ptr = (PriorityData *) otn_tmp->ds_list[PLUGIN_PRIORITY_NUMBER];
        }

        if(ds_ptr != NULL)
        {
            snprintf(pri_data, STD_BUF-1, " [Classification: %s] [Priority: %d]:", 
                    ds_ptr->name, ds_ptr->priority); 

            strlcat(event_string, pri_data, SYSLOG_BUF);
        }

        if((p->iph->ip_proto != IPPROTO_TCP &&
                    p->iph->ip_proto != IPPROTO_UDP) || 
                p->frag_flag)
        {
            if(!pv.alert_interface_flag)
            {
                snprintf(ip_data, STD_BUF-1, " {%s} %s -> %s",  
                        protocol_names[p->iph->ip_proto], sip, dip);
            }
            else
            {
                snprintf(ip_data, STD_BUF-1, " <%s> {%s} %s -> %s",  
                        PRINT_INTERFACE(pv.interfaces[0]), protocol_names[p->iph->ip_proto], sip, dip);
            }
        }
        else
        {
            if(pv.alert_interface_flag)
            {
                snprintf(ip_data, STD_BUF-1, " <%s> {%s} %s:%i -> %s:%i",
                        PRINT_INTERFACE(pv.interfaces[0]),
                        protocol_names[p->iph->ip_proto], sip,
                        p->sp, dip, p->dp);
            }
            else
            {
                snprintf(ip_data, STD_BUF-1, " {%s} %s:%i -> %s:%i",
                        protocol_names[p->iph->ip_proto], sip, p->sp, 
                        dip, p->dp);
            }
        }

        strlcat(event_string, ip_data, SYSLOG_BUF);

        syslog(LOG_AUTH | LOG_ALERT, "%s", event_string);

    }
    else  
    {
        syslog(LOG_AUTH | LOG_ALERT, "%s", msg == NULL ? "ALERT!" : msg);
    }

    return;
}



/****************************************************************************
 *
 * Function: SmbAlert(Packet *, char *)
 *
 * Purpose: Send the current alert to a waiting WinPopup client
 *
 * Arguments: p => pointer to the packet data struct
 *            msg => the message to print in the alert
 *
 * Returns: void function
 *
 ***************************************************************************/
/* DO NOT TURN ON THIS CODE UNLESS YOU'RE REALLY SURE YOU KNOW 
 * WHAT YOU'RE DOING!!
 */ 
#ifdef ENABLE_SMB_ALERTS
void SmbAlert(Packet * p, char *msg, void *arg, Event *event)
{
    char command_line[2048];
    FILE *output;
    FILE *workstations;
    char *tempmsg;
    char workfile[STD_BUF+1];
    char tempwork[STD_BUF+1];
    char timestamp[TIMEBUF_SIZE];
    int msg_str_size;
    char sip[16];
    char dip[16];
    char pri_data[STD_BUF];
    PriorityData *ds_ptr;  

    if(otn_tmp != NULL)
    {
        ds_ptr = (PriorityData *) otn_tmp->ds_list[PLUGIN_PRIORITY_NUMBER];

        if(ds_ptr != NULL)
        {
            snprintf(pri_data, STD_BUF-1, "\n[Classification: %s   "
                    "Priority: %d]\n" , ds_ptr->name, ds_ptr->priority); 
        }
    }

    bzero((char *) timestamp, TIMEBUF_SIZE);

    ts_print(p == NULL ? NULL : (struct timeval *) & p->pkth->ts, timestamp);

    DebugMessage(DEBUG_LOG, "Generating SMB alert!\n");

    /* set the workstation name filename */
    sprintf(workfile, "%s", pv.smbmsg_dir);

    /* message size + IP addrs + ports + pad space */
    msg_str_size = strlen(msg) + 32 + 10 + 150;

    if((tempmsg = (char *) calloc(msg_str_size, sizeof(char))) == NULL)
    {
        FatalError("[!] ERROR: SmbAlert() unable to allocate space for "
                   "tempmsg: %s\n", strerror(errno));
    }

    /* open the message file and the workstation names file */
    if((workstations = fopen(workfile, "r")) != NULL)
    {
        /* clear the read buffers */
        bzero((char *) workfile, STD_BUF + 1);

        if(p && p->iph)
        {
            strncpy(sip, inet_ntoa(p->iph->ip_src), 16);
            strncpy(dip, inet_ntoa(p->iph->ip_dst), 16);

            if(p->frag_flag || 
              (p->iph->ip_proto != IPPROTO_TCP && 
               p->iph->ip_proto != IPPROTO_UDP))
            {
                /* write the alert message into the buffer */
                snprintf(tempmsg, msg_str_size-1,
                        "[**] %s [**]%s%s %s->%s", msg, 
                        ds_ptr!=NULL?pri_data:"\n", timestamp, sip, dip);
            }
            else
            {
                /* write the alert message into the buffer */
                snprintf(tempmsg, msg_str_size-1, 
                        "[**] %s [**]%s%s %s:%d->%s:%d", msg, 
                        ds_ptr!=NULL?pri_data:"\n", timestamp, sip, p->sp, dip, p->dp);
            }
        }
        else
        {
            /*
             * write the alert message into the buffer - this part is for
             * alerts with NULL packets (like portscans)
             */
            snprintf(tempmsg, msg_str_size-1, "[**] %s [**]\n", msg);
        }

        bzero((char *) tempwork, STD_BUF + 1);
        bzero((char *) command_line, 2048);

        /* read in the name of each workstation to send the message to */
        while((fgets(tempwork, STD_BUF, workstations)) != NULL)
        {
            /* if the line isn't blank */
            if(tempwork[0] != 0x0A)
            {
                /* chop the <CR> */
                strip(tempwork);

#ifdef WIN32
                snprintf(command_line, 2047, "start /min net send %s %s",
                        tempwork,tempmsg);

                WinExec(command_line,SW_SHOWMINNOACTIVE);
#else
                /* build the command line */
                snprintf(command_line, 2047,
                        "echo \"%s\" | smbclient -U Snort -M %s", 
                        tempmsg, tempwork);

                /* run the command */
                output = popen(command_line, "r");

                pclose(output);
#endif

                DebugMessage(DEBUG_LOG, "Sending WinPopup alert to: %s\n", 
                             tempwork);
                DebugMessage(DEBUG_LOG, "Command Line: %s\n", command_line);

                bzero((char *) tempwork, STD_BUF + 1);
                bzero((char *) command_line, 2048);
            }
        }

        fclose(workstations);
    }

    free(tempmsg);
}

#endif


/****************************************************************************
 *
 * Function: NoAlert(Packet *, char *)
 *
 * Purpose: Don't alert at all
 *
 * Arguments: p => pointer to the packet data struct
 *            msg => the message to not print in the alert
 *
 * Returns: void function
 *
 ***************************************************************************/
void NoAlert(Packet * p, char *msg, void *arg, Event *event)
{
    return;
}


/****************************************************************************
 *
 * Function: LogPkt(Packet *)
 *
 * Purpose: Log packets that match one of the Snort rules, plus the rules
 *          message
 *
 * Arguments: p => pointer to the packet data structure
 *
 * Returns: void function
 *
 ***************************************************************************/
void LogPkt(Packet * p, char *msg, void *arg, Event *event)
{
    /*
     * some plugins may pass NULL as 'p' structure here (f.e. if they only
     * want to log message
     */
    DebugMessage(DEBUG_LOG,"LogPkt started\n");
    if(p)
        OpenLogFile(0, p);
    else
        OpenLogFile(GENERIC_LOG, p);

    if(msg != NULL)
    {
        fwrite("[**] ", 5, 1, log_ptr);
        fwrite(msg, strlen(msg), 1, log_ptr);
        fwrite(" [**]\n", 6, 1, log_ptr);
    }
    if(p && p->iph)
        PrintIPPkt(log_ptr, p->iph->ip_proto, p);

    fclose(log_ptr);
}



/****************************************************************************
 *
 * Function: LogArpPkt(Packet *)
 *
 * Purpose: Log ARP packets
 *
 * Arguments: p => pointer to the packet data structure
 *
 * Returns: void function
 *
 ***************************************************************************/
void LogArpPkt(Packet * p)
{
    if(p)
    {
        if(pv.logbin_flag)
        {
            /* LogBin(p); */
        }
        else if(!pv.nolog_flag)
        {
            OpenLogFile(ARP, p);

            PrintArpHeader(log_ptr, p);

            fclose(log_ptr);
        }
    }
    else
    {
        /* let's just make sure we don't coredump if we have some logic error in one
         *   of plugins/preprocessors...
         */
        ErrorMessage("WARNING: Null Packet pointer in LogArpPkt. Please report.\n");
    }
}



/****************************************************************************
 *
 * Function: NoLog(Packet *)
 *
 * Purpose: Don't log anything
 *
 * Arguments: p => packet to not log
 *
 * Returns: void function
 *
 ***************************************************************************/
void NoLog(Packet * p, char *msg, void *arg, Event *event)
{
    return;
}

/****************************************************************************
 *
 * Function: Print2ndHeader(FILE *, Packet p)
 *
 * Purpose: Print2ndHeader -- prints second layber  header info.
 *
 * Arguments: fp => file stream to print to
 *
 * Returns: void function
 *
 ***************************************************************************/


void Print2ndHeader(FILE * fp, Packet * p)
{

    switch(datalinks[0]) {
        /* FIXME for multi-threaded support here */
        case DLT_EN10MB:        /* Ethernet */
            if(p && p->eh)
                PrintEthHeader(fp, p);
            break;    
        case DLT_IEEE802:                /* Token Ring */
            if(p && p->trh)
                PrintTrHeader(fp, p);
            break;    
#ifdef DLT_LINUX_SLL        
        case DLT_LINUX_SLL:
            if (p && p->sllh)
                PrintSLLHeader(fp, p);  /* Linux cooked sockets */
            break;
#endif            
        default:
            if(pv.verbose_flag)
                ErrorMessage("Datalink %i type 2nd layer display is not "
                             "supported\n", datalinks[0]);   
                /* FIXME here too */
    }
                

}



/****************************************************************************
 *
 * Function: PrintTrHeader(FILE *, Packet p)
 &
 * Purpose: Print the packet TokenRing header to the specified stream
 *
 * Arguments: fp => file stream to print to
 *
 * Returns: void function
 ***************************************************************************/

void PrintTrHeader(FILE * fp, Packet * p)
{

    fprintf(fp, "%X:%X:%X:%X:%X:%X -> ", p->trh->saddr[0],
            p->trh->saddr[1], p->trh->saddr[2], p->trh->saddr[3],
            p->trh->saddr[4], p->trh->saddr[5]);
    fprintf(fp, "%X:%X:%X:%X:%X:%X\n", p->trh->daddr[0],
            p->trh->daddr[1], p->trh->daddr[2], p->trh->daddr[3],
            p->trh->daddr[4], p->trh->daddr[5]);

    fprintf(fp, "access control:0x%X frame control:0x%X\n", p->trh->ac,
            p->trh->fc);
    if(!p->trhllc)
        return;
    fprintf(fp, "DSAP: 0x%X SSAP 0x%X protoID: %X%X%X Ethertype: %X\n",
            p->trhllc->dsap, p->trhllc->ssap, p->trhllc->protid[0],
            p->trhllc->protid[1], p->trhllc->protid[2], p->trhllc->ethertype);
    if(p->trhmr)
    {
        fprintf(fp, "RIF structure is present:\n");
        fprintf(fp, "bcast: 0x%X length: 0x%X direction: 0x%X largest"
                "fr. size: 0x%X res: 0x%X\n",
                p->trhmr->bcast, p->trhmr->len, p->trhmr->dir, p->trhmr->lf,
                p->trhmr->res);
        fprintf(fp, "rseg -> %X:%X:%X:%X:%X:%X:%X:%X\n",
                p->trhmr->rseg[0], p->trhmr->rseg[1], p->trhmr->rseg[2],
                p->trhmr->rseg[3], p->trhmr->rseg[4], p->trhmr->rseg[5],
                p->trhmr->rseg[6], p->trhmr->rseg[7]);
    }
}


/****************************************************************************
 *
 * Function: PrintEthHeader(FILE *)
 *
 * Purpose: Print the packet Ethernet header to the specified stream
 *
 * Arguments: fp => file stream to print to
 *
 * Returns: void function
 *
 ***************************************************************************/
void PrintEthHeader(FILE * fp, Packet * p)
{
    /* src addr */
    fprintf(fp, "%X:%X:%X:%X:%X:%X -> ", p->eh->ether_src[0],
            p->eh->ether_src[1], p->eh->ether_src[2], p->eh->ether_src[3],
            p->eh->ether_src[4], p->eh->ether_src[5]);

    /* dest addr */
    fprintf(fp, "%X:%X:%X:%X:%X:%X ", p->eh->ether_dst[0],
            p->eh->ether_dst[1], p->eh->ether_dst[2], p->eh->ether_dst[3],
            p->eh->ether_dst[4], p->eh->ether_dst[5]);

    /* protocol and pkt size */
    fprintf(fp, "type:0x%X len:0x%X\n", ntohs(p->eh->ether_type), p->pkth->len);
}


/****************************************************************************
 *
 * Function: PrintSLLHeader(FILE *)
 *
 * Purpose: Print the packet SLL (fake) header to the specified stream (piece
 * partly is borrowed from tcpdump :))
 *
 * Arguments: fp => file stream to print to
 *
 * Returns: void function
 *
 ***************************************************************************/
void PrintSLLHeader(FILE * fp, Packet * p)
{


    switch (ntohs(p->sllh->sll_pkttype)) {
        case LINUX_SLL_HOST:
            (void)fprintf(fp, "< ");
            break;
        case LINUX_SLL_BROADCAST:
            (void)fprintf(fp, "B ");
            break;
        case LINUX_SLL_MULTICAST:
            (void)fprintf(fp, "M ");
            break;
        case LINUX_SLL_OTHERHOST:
            (void)fprintf(fp, "P ");
            break;
        case LINUX_SLL_OUTGOING:
            (void)fprintf(fp, "> ");
            break;
        default:
            (void)fprintf(fp, "? ");
            break;
        }

    /* mac addr */
    fprintf(fp, "l/l len: %i l/l type: 0x%X %X:%X:%X:%X:%X:%X\n", 
            htons(p->sllh->sll_halen), ntohs(p->sllh->sll_hatype),
            p->sllh->sll_addr[0], p->sllh->sll_addr[1], p->sllh->sll_addr[2],
            p->sllh->sll_addr[3], p->sllh->sll_addr[4], p->sllh->sll_addr[5]);

    /* protocol and pkt size */
    fprintf(fp, "pkt type:0x%X proto: 0x%X len:0x%X\n",
                 ntohs(p->sllh->sll_pkttype),
                 ntohs(p->sllh->sll_protocol), p->pkth->len);
}



void PrintArpHeader(FILE * fp, Packet * p)
{
    struct in_addr ip_addr;
    char timestamp[TIMEBUF_SIZE];

    bzero((struct in_addr *) &ip_addr, sizeof(struct in_addr));
    bzero((char *) timestamp, TIMEBUF_SIZE);
    ts_print((struct timeval *) & p->pkth->ts, timestamp);

    /* dump the timestamp */
    fwrite(timestamp, strlen(timestamp), 1, fp);

    if(ntohs(p->ah->ea_hdr.ar_pro) != ETHERNET_TYPE_IP)
    {
        fprintf(fp, "ARP #%d for protocol #%.4X (%d) hardware #%d (%d)\n",
                ntohs(p->ah->ea_hdr.ar_op), ntohs(p->ah->ea_hdr.ar_pro),
                p->ah->ea_hdr.ar_pln, ntohs(p->ah->ea_hdr.ar_hrd),
                p->ah->ea_hdr.ar_hln);

        return;
    }

    switch(ntohs(p->ah->ea_hdr.ar_op))
    {
        case ARPOP_REQUEST:
            bcopy((void *)p->ah->arp_tpa, (void *) &ip_addr, sizeof(ip_addr));
            fprintf(fp, "ARP who-has %s", inet_ntoa(ip_addr));

            if(memcmp((char *) ezero, (char *) p->ah->arp_tha, 6) != 0)
            {
                fprintf(fp, " (%X:%X:%X:%X:%X:%X)", p->ah->arp_tha[0],
                        p->ah->arp_tha[1], p->ah->arp_tha[2], p->ah->arp_tha[3],
                        p->ah->arp_tha[4], p->ah->arp_tha[5]);
            }
            bcopy((void *)p->ah->arp_spa, (void *) &ip_addr, sizeof(ip_addr));

            fprintf(fp, " tell %s", inet_ntoa(ip_addr));

            if(memcmp((char *) p->eh->ether_src, (char *) p->ah->arp_sha, 6) 
                    != 0)
            {
                fprintf(fp, " (%X:%X:%X:%X:%X:%X)", p->ah->arp_sha[0],
                        p->ah->arp_sha[1], p->ah->arp_sha[2], p->ah->arp_sha[3],
                        p->ah->arp_sha[4], p->ah->arp_sha[5]);
            }
            break;

        case ARPOP_REPLY:
            bcopy((void *)p->ah->arp_spa, (void *) &ip_addr, sizeof(ip_addr));
            fprintf(fp, "ARP reply %s", inet_ntoa(ip_addr));

            if(memcmp((char *) p->eh->ether_src, (char *) p->ah->arp_sha, 6) 
                    != 0)
            {
                fprintf(fp, " (%X:%X:%X:%X:%X:%X)", p->ah->arp_sha[0],
                        p->ah->arp_sha[1], p->ah->arp_sha[2], p->ah->arp_sha[3],
                        p->ah->arp_sha[4], p->ah->arp_sha[5]);
            }
            fprintf(fp, " is-at %X:%X:%X:%X:%X:%X", p->ah->arp_sha[0],
                    p->ah->arp_sha[1], p->ah->arp_sha[2], p->ah->arp_sha[3],
                    p->ah->arp_sha[4], p->ah->arp_sha[5]);

            if(memcmp((char *) p->eh->ether_dst, (char *) p->ah->arp_tha, 6) != 0)
            {
                fprintf(fp, " (%X:%X:%X:%X:%X:%X)", p->ah->arp_tha[0],
                        p->ah->arp_tha[1], p->ah->arp_tha[2], p->ah->arp_tha[3],
                        p->ah->arp_tha[4], p->ah->arp_tha[5]);
            }
            break;

        case ARPOP_RREQUEST:
            fprintf(fp, "RARP who-is %X:%X:%X:%X:%X:%X tell %X:%X:%X:%X:%X:%X",
                    p->ah->arp_tha[0], p->ah->arp_tha[1], p->ah->arp_tha[2],
                    p->ah->arp_tha[3], p->ah->arp_tha[4], p->ah->arp_tha[5],
                    p->ah->arp_sha[0], p->ah->arp_sha[1], p->ah->arp_sha[2],
                    p->ah->arp_sha[3], p->ah->arp_sha[4], p->ah->arp_sha[5]);

            break;

        case ARPOP_RREPLY:
            bcopy((void *)p->ah->arp_tpa, (void *) &ip_addr, sizeof(ip_addr));
            fprintf(fp, "RARP reply %X:%X:%X:%X:%X:%X at %s",
                    p->ah->arp_tha[0], p->ah->arp_tha[1], p->ah->arp_tha[2],
                    p->ah->arp_tha[3], p->ah->arp_tha[4], p->ah->arp_tha[5],
                    inet_ntoa(ip_addr));

            break;

        default:
            fprintf(fp, "Unknown operation: %d", ntohs(p->ah->ea_hdr.ar_op));
            break;
    }

    fprintf(fp, "\n\n");

}

/****************************************************************************
 *
 * Function: PrintIPHeader(FILE *)
 *
 * Purpose: Dump the IP header info to the specified stream
 *
 * Arguments: fp => stream to print to
 *
 * Returns: void function
 *
 ***************************************************************************/
void PrintIPHeader(FILE * fp, Packet * p)
{
    if(p->iph == NULL)
    {
        fprintf(fp, "IP header truncated\n");
        return;
    }
    if(p->frag_flag)
    {
        /* just print the straight IP header */
        fputs(inet_ntoa(p->iph->ip_src), fp);
        fwrite(" -> ", 4, 1, fp);
        fputs(inet_ntoa(p->iph->ip_dst), fp);
    }
    else
    {
        if(p->iph->ip_proto != IPPROTO_TCP && p->iph->ip_proto != IPPROTO_UDP)
        {
            /* just print the straight IP header */
            fputs(inet_ntoa(p->iph->ip_src), fp);
            fwrite(" -> ", 4, 1, fp);
            fputs(inet_ntoa(p->iph->ip_dst), fp);
        }
        else
        {
            if(!pv.obfuscation_flag)
            {
                /* print the header complete with port information */
                fputs(inet_ntoa(p->iph->ip_src), fp);
                fprintf(fp, ":%d -> ", p->sp);
                fputs(inet_ntoa(p->iph->ip_dst), fp);
                fprintf(fp, ":%d", p->dp);
            }
            else
            {
                /* print the header complete with port information */
                fprintf(fp, "xxx.xxx.xxx.xxx:%d -> xxx.xxx.xxx.xxx:%d", p->sp, p->dp);
            }
        }
    }

    if(!pv.show2hdr_flag)
    {
        fputc('\n', fp);
    }
    else
    {
        fputc(' ', fp);
    }

    fprintf(fp, "%s TTL:%d TOS:0x%X ID:%d IpLen:%d DgmLen:%d",
            protocol_names[p->iph->ip_proto],
            p->iph->ip_ttl,
            p->iph->ip_tos,
            ntohs(p->iph->ip_id),
            p->iph->ip_hlen << 2, ntohs(p->iph->ip_len));

    /* print the reserved bit if it's set */
    if(p->rf)
    {
        fprintf(fp, " RB");
    }
    /* printf more frags/don't frag bits */
    if(p->df)
        fprintf(fp, " DF");

    if(p->mf)
        fprintf(fp, " MF");

    fputc('\n', fp);

    /* print IP options */
    if(p->ip_option_count != 0)
    {
        PrintIpOptions(fp, p);
    }
    /* print fragment info if necessary */
    if(p->frag_flag)
    {
        fprintf(fp, "Frag Offset: 0x%X   Frag Size: 0x%X",
                (p->frag_offset & 0xFFFF), p->dsize);
        fputc('\n', fp);
    }
}



/****************************************************************************
 *
 * Function: PrintTCPHeader(FILE *)
 *
 * Purpose: Dump the TCP header info to the specified stream
 *
 * Arguments: fp => file stream to print data to
 *
 * Returns: void function
 *
 ***************************************************************************/
void PrintTCPHeader(FILE * fp, Packet * p)
{
    char tcpFlags[9];

    if(p->tcph == NULL)
    {
        fprintf(fp, "TCP header truncated\n");
        return;
    }
    /* print TCP flags */
    CreateTCPFlagString(p, tcpFlags);
    fwrite(tcpFlags, 8, 1, fp); /* We don't care about the NULL */

    /* print other TCP info */
    fprintf(fp, " Seq: 0x%lX  Ack: 0x%lX  Win: 0x%X  TcpLen: %d",
            (u_long) ntohl(p->tcph->th_seq),
            (u_long) ntohl(p->tcph->th_ack),
            ntohs(p->tcph->th_win), p->tcph->th_off << 2);

    if((p->tcph->th_flags & TH_URG) != 0)
    {
        fprintf(fp, "  UrgPtr: 0x%X\n", (u_int16_t) ntohs(p->tcph->th_urp));
    }
    else
    {
        fputc((int) '\n', fp);
    }

    /* dump the TCP options */
    if(p->tcp_option_count != 0)
    {
        PrintTcpOptions(fp, p);
    }
}


void PrintEmbeddedTCPHeader(FILE * fp, Packet * p, int size)
{
    char tcpFlags[9];

    DebugMessage(DEBUG_FLOW, "size is %d\n", size);

    if(size >= 16)
    { 
        CreateTCPFlagString(p, tcpFlags);
        fwrite(tcpFlags, 8, 1, fp); 
        fprintf(fp, " Seq: 0x%lX  Ack: 0x%lX  Win: 0x%X  TcpLen: %d\n",
                (u_long) ntohl(p->tcph->th_seq),
                (u_long) ntohl(p->tcph->th_ack),
                ntohs(p->tcph->th_win), p->tcph->th_off << 2);
    }
    else if(size >= 14)
    { 
        CreateTCPFlagString(p, tcpFlags);
        fwrite(tcpFlags, 8, 1, fp); 
        fprintf(fp, " Seq: 0x%lX  Ack: 0x%lX  TcpLen: %d\n",
                (u_long) ntohl(p->tcph->th_seq), 
                (u_long) ntohl(p->tcph->th_ack),
                p->tcph->th_off << 2);
    }
    else if(size >= 13)
    {
        fprintf(fp, "Seq: 0x%lX  Ack: 0x%lX  TcpLen: %d\n",
                (u_long) ntohl(p->tcph->th_seq), 
                (u_long) ntohl(p->tcph->th_ack), p->tcph->th_off << 2);
    }
    else if(size >= 12)
    {
        fprintf(fp, "Seq: 0x%lX  Ack: 0x%lX\n",
                (u_long) ntohl(p->tcph->th_seq), 
                (u_long) ntohl(p->tcph->th_ack));
    }
    else if(size >= 8)
    {

        fprintf(fp, "Seq: 0x%lX\n",
                (u_long) ntohl(p->tcph->th_seq));
    }
}



/* Input is packet and an nine-byte (including NULL) character array.  Results
 * are put into the character array.
 */
void CreateTCPFlagString(Packet * p, char *flagBuffer)
{
    /* parse TCP flags */
    *flagBuffer++ = (char) ((p->tcph->th_flags & TH_RES1) ? '1' : '*');
    *flagBuffer++ = (char) ((p->tcph->th_flags & TH_RES2) ? '2' : '*');
    *flagBuffer++ = (char) ((p->tcph->th_flags & TH_URG)  ? 'U' : '*');
    *flagBuffer++ = (char) ((p->tcph->th_flags & TH_ACK)  ? 'A' : '*');
    *flagBuffer++ = (char) ((p->tcph->th_flags & TH_PUSH) ? 'P' : '*');
    *flagBuffer++ = (char) ((p->tcph->th_flags & TH_RST)  ? 'R' : '*');
    *flagBuffer++ = (char) ((p->tcph->th_flags & TH_SYN)  ? 'S' : '*');
    *flagBuffer++ = (char) ((p->tcph->th_flags & TH_FIN)  ? 'F' : '*');
    *flagBuffer = '\0';

}


/****************************************************************************
 *
 * Function: PrintUDPHeader(FILE *)
 *
 * Purpose: Dump the UDP header to the specified file stream
 *
 * Arguments: fp => file stream
 *
 * Returns: void function
 *
 ***************************************************************************/
void PrintUDPHeader(FILE * fp, Packet * p)
{

    if(p->udph == NULL)
    {
        fprintf(fp, "UDP header truncated\n");
        return;
    }
    /* not much to do here... */
    fprintf(fp, "Len: %d\n", ntohs(p->udph->uh_len));
}



/****************************************************************************
 *
 * Function: PrintICMPHeader(FILE *)
 *
 * Purpose: Print ICMP header
 *
 * Arguments: fp => file stream
 *
 * Returns: void function
 *
 ***************************************************************************/
void PrintICMPHeader(FILE * fp, Packet * p)
{

    if(p->icmph == NULL)
    {
        fprintf(fp, "ICMP header truncated\n");
        return;
    }

    fprintf(fp, "Type:%d  Code:%d  ", p->icmph->type, p->icmph->code);

    switch(p->icmph->type)
    {
        case ICMP_ECHOREPLY:
            fprintf(fp, "ID:%d  Seq:%d  ", p->icmph->s_icmp_id, 
                    p->icmph->s_icmp_seq);
            fwrite("ECHO REPLY\n", 10, 1, fp);
            break;

        case ICMP_DEST_UNREACH:
            fwrite("DESTINATION UNREACHABLE: ", 25, 1, fp);
            switch(p->icmph->code)
            {
                case ICMP_NET_UNREACH:
                    fwrite("NET UNREACHABLE", 15, 1, fp);
                    break;

                case ICMP_HOST_UNREACH:
                    fwrite("HOST UNREACHABLE", 16, 1, fp);
                    break;

                case ICMP_PROT_UNREACH:
                    fwrite("PROTOCOL UNREACHABLE", 20, 1, fp);
                    break;

                case ICMP_PORT_UNREACH:
                    fwrite("PORT UNREACHABLE", 16, 1, fp);
                    break;

                case ICMP_FRAG_NEEDED:
                    fwrite("FRAGMENTATION NEEDED", 20, 1, fp);
                    break;

                case ICMP_SR_FAILED:
                    fwrite("SOURCE ROUTE FAILED", 19, 1, fp);
                    break;

                case ICMP_NET_UNKNOWN:
                    fwrite("NET UNKNOWN", 11, 1, fp);
                    break;

                case ICMP_HOST_UNKNOWN:
                    fwrite("HOST UNKNOWN", 12, 1, fp);
                    break;

                case ICMP_HOST_ISOLATED:
                    fwrite("HOST ISOLATED", 13, 1, fp);
                    break;

                case ICMP_NET_ANO:
                    fwrite("NET ANO", 7, 1, fp);
                    break;

                case ICMP_HOST_ANO:
                    fwrite("HOST ANO", 8, 1, fp);
                    break;

                case ICMP_NET_UNR_TOS:
                    fwrite("NET UNREACHABLE TOS", 19, 1, fp);
                    break;

                case ICMP_HOST_UNR_TOS:
                    fwrite("HOST UNREACHABLE TOS", 20, 1, fp);
                    break;

                case ICMP_PKT_FILTERED:
                    fwrite("PACKET FILTERED", 15, 1, fp);
                    break;

                case ICMP_PREC_VIOLATION:
                    fwrite("PREC VIOLATION", 14, 1, fp);
                    break;

                case ICMP_PREC_CUTOFF:
                    fwrite("PREC CUTOFF", 12, 1, fp);
                    break;

                default:
                    fwrite("UNKNOWN", 7, 1, fp);
                    break;

            }
            {
                Packet orig_p;
                int orig_iph_size;

                bzero((char *) &orig_p, sizeof(Packet));
                orig_p.iph = p->orig_iph;
                orig_p.tcph = p->orig_tcph;
                orig_p.udph = p->orig_udph;
                orig_p.sp = p->orig_sp;
                orig_p.dp = p->orig_dp;

                if(orig_p.iph != NULL)
                {
                    orig_iph_size = orig_p.iph->ip_hlen << 2;

                    fprintf(fp, "\n** ORIGINAL DATAGRAM DUMP:\n");
                    PrintIPHeader(fp, &orig_p);
                    
                    switch(orig_p.iph->ip_proto)
                    {
                        case IPPROTO_TCP:
                            /* 
                             * we can only guarantee the first 8 bytes of the
                             * tcp header are encapsulated, so lets just print 
                             * them instead of freaking people out all the time
                             *   --MFR
                             */
                            if(orig_p.tcph != NULL)
                                PrintEmbeddedTCPHeader(fp, &orig_p, 
                                                       p->dsize-orig_iph_size);
                            break;

                        case IPPROTO_UDP:
                            if(orig_p.udph != NULL)
                                PrintUDPHeader(fp, &orig_p);
                            break;

                            /* um, isn't it generally considered bad form to 
                             * send an ICMP error message in response to ICMP
                             * traffic??? -MFR
                             */
                        case IPPROTO_ICMP:
                            if(orig_p.icmph != NULL)
                                fprintf(fp, "orig type: %d  code: %d\n", 
                                    orig_p.icmph->type, orig_p.icmph->code);
                            break;

                        default:
                            fprintf(fp, "Protocol: 0x%X (unknown or "
                                    "header truncated)", orig_p.iph->ip_proto);
                            break;
                    }       /* switch */

                    fprintf(fp, "** END OF DUMP");
                }
                else
                {
                    fprintf(fp, "\nORIGINAL DATAGRAM TRUNCATED");
                }
            }
            break;

        case ICMP_SOURCE_QUENCH:
            fwrite("SOURCE QUENCH", 13, 1, fp);
            break;

        case ICMP_REDIRECT:
            fwrite("REDIRECT", 8, 1, fp);
            break;
        case ICMP_ECHO:
            fprintf(fp, "ID:%d   Seq:%d  ", p->icmph->s_icmp_id, 
                    p->icmph->s_icmp_seq);
            fwrite("ECHO", 4, 1, fp);
            break;

        case ICMP_TIME_EXCEEDED:
            fwrite("TTL EXCEEDED", 12, 1, fp);
            break;

        case ICMP_PARAMETERPROB:
            fwrite("PARAMETER PROBLEM", 17, 1, fp);
            break;

        case ICMP_TIMESTAMP:
            fwrite("TIMESTAMP REQUEST", 17, 1, fp);
            break;

        case ICMP_TIMESTAMPREPLY:
            fwrite("TIMESTAMP REPLY", 15, 1, fp);
            break;

        case ICMP_INFO_REQUEST:
            fwrite("INFO REQUEST", 12, 1, fp);
            break;

        case ICMP_INFO_REPLY:
            fwrite("INFO REPLY", 10, 1, fp);
            break;

        case ICMP_ADDRESS:
            fwrite("ADDRESS REQUEST", 15, 1, fp);
            break;

        case ICMP_ADDRESSREPLY:
            fwrite("ADDRESS REPLY", 13, 1, fp);
            break;

        default:
            fwrite("UNKNOWN", 7, 1, fp);

            break;
    }

    putc('\n', fp);

}


void PrintIpOptions(FILE * fp, Packet * p)
{
    int i;
    int j;
    u_long init_offset;
    u_long print_offset;

    init_offset = ftell(fp);

    if(!p->ip_option_count || p->ip_option_count > 40)
        return;

    fprintf(fp, "IP Options (%d) => ", p->ip_option_count);

    for(i = 0; i < (int) p->ip_option_count; i++)
    {
        print_offset = ftell(fp);

        if((print_offset - init_offset) > 60)
        {
            fwrite("\nIP Options => ", 15, 1, fp);
            init_offset = ftell(fp);
        }
            
        switch(p->ip_options[i].code)
        {
            case IPOPT_RR:
                fwrite("RR ", 3, 1, fp);
                break;

            case IPOPT_EOL:
                fwrite("EOL ", 4, 1, fp);
                break;

            case IPOPT_NOP:
                fwrite("NOP ", 4, 1, fp);
                break;

            case IPOPT_TS:
                fwrite("TS ", 3, 1, fp);
                break;

            case IPOPT_SECURITY:
                fwrite("SEC ", 4, 1, fp);
                break;

            case IPOPT_LSRR:
            case IPOPT_LSRR_E:
                fwrite("LSRR ", 5, 1, fp);
                break;

            case IPOPT_SATID:
                fwrite("SID ", 4, 1, fp);
                break;

            case IPOPT_SSRR:
                fwrite("SSRR ", 5, 1, fp);
                break;

            case IPOPT_RTRALT:
                fwrite("RTRALT ", 7, 1, fp);
                break;    

            default:
                fprintf(fp, "Opt %d: ", p->ip_options[i].code);

                if(p->ip_options[i].len)
                {
                    for(j = 0; j < (int)(p->ip_options[i].len - 1); j += 2)
                    {
                        fprintf(fp, "%02X%02X ", p->ip_options[i].data[j], p->ip_options[i].data[j + 1]);
                    }
                }
                break;
        }
    }

    fwrite("\n", 1, 1, fp);
}


void PrintTcpOptions(FILE * fp, Packet * p)
{
    int i;
    int j;
    u_char tmp[5];
    u_long init_offset;
    u_long print_offset;

    init_offset = ftell(fp);

    fprintf(fp, "TCP Options (%d) => ", p->tcp_option_count);

    if(p->tcp_option_count > 40 || !p->tcp_option_count)
        return;

    for(i = 0; i < (int) p->tcp_option_count; i++)
    {
        print_offset = ftell(fp);

        if((print_offset - init_offset) > 60)
        {
            fwrite("\nTCP Options => ", 16, 1, fp);
            init_offset = ftell(fp);
        }
            
        switch(p->tcp_options[i].code)
        {
            case TCPOPT_MAXSEG:
                bzero((char *) tmp, 5);
                fwrite("MSS: ", 5, 1, fp);
                strncpy((char*)tmp, (char*) (p->tcp_options[i].data), 2);
                fprintf(fp, "%u ", EXTRACT_16BITS(tmp));
                break;

            case TCPOPT_EOL:
                fwrite("EOL ", 4, 1, fp);
                break;

            case TCPOPT_NOP:
                fwrite("NOP ", 4, 1, fp);
                break;

            case TCPOPT_WSCALE:
                fprintf(fp, "WS: %u ", p->tcp_options[i].data[0]);
                break;

            case TCPOPT_SACK:
                bzero((char *) tmp, 5);
                memcpy(tmp, p->tcp_options[i].data, 2);
                fprintf(fp, "Sack: %u@", EXTRACT_16BITS(tmp));
                bzero((char *) tmp, 5);
                memcpy(tmp, (p->tcp_options[i].data) + 2, 2);
                fprintf(fp, "%u ", EXTRACT_16BITS(tmp));
                break;

            case TCPOPT_SACKOK:
                fwrite("SackOK ", 7, 1, fp);
                break;

            case TCPOPT_ECHO:
                bzero((char *) tmp, 5);
                memcpy(tmp, p->tcp_options[i].data, 4);
                fprintf(fp, "Echo: %u ", EXTRACT_32BITS(tmp));
                break;

            case TCPOPT_ECHOREPLY:
                bzero((char *) tmp, 5);
                memcpy(tmp, p->tcp_options[i].data, 4);
                fprintf(fp, "Echo Rep: %u ", EXTRACT_32BITS(tmp));
                break;

            case TCPOPT_TIMESTAMP:
                bzero((char *) tmp, 5);
                memcpy(tmp, p->tcp_options[i].data, 4);
                fprintf(fp, "TS: %u ", EXTRACT_32BITS(tmp));
                bzero((char *) tmp, 5);
                memcpy(tmp, (p->tcp_options[i].data) + 4, 4);
                fprintf(fp, "%u ", EXTRACT_32BITS(tmp));
                break;

            case TCPOPT_CC:
                bzero((char *) tmp, 5);
                memcpy(tmp, p->tcp_options[i].data, 4);
                fprintf(fp, "CC %u ", EXTRACT_32BITS(tmp));
                break;

            case TCPOPT_CCNEW:
                bzero((char *) tmp, 5);
                memcpy(tmp, p->tcp_options[i].data, 4);
                fprintf(fp, "CCNEW: %u ", EXTRACT_32BITS(tmp));
                break;

            case TCPOPT_CCECHO:
                bzero((char *) tmp, 5);
                memcpy(tmp, p->tcp_options[i].data, 4);
                fprintf(fp, "CCECHO: %u ", EXTRACT_32BITS(tmp));
                break;

            default:
                if(p->tcp_options[i].len > 2)
                {
                    fprintf(fp, "Opt %d (%d): ", p->tcp_options[i].code,
                            (int) p->tcp_options[i].len);

                    for(j = 0; j < (int) (p->tcp_options[i].len - 1); j += 2)
                    {
                        fprintf(fp, "%02X%02X ", p->tcp_options[i].data[j], 
                                p->tcp_options[i].data[j + 1]);
                    }
                }
                else
                {
                    fprintf(fp, "Opt %d ", p->tcp_options[i].code);
                }
                break;
        }
    }

    fwrite("\n", 1, 1, fp);
}


/*
 * Function: PrintPriorityData(FILE *)
 *
 * Purpose: Prints out priority data associated with an alert
 *
 * Arguments: fp => file descriptor to write the data to
 *            do_newline => tack a \n to the end of the line or not (bool)
 *
 * Returns: void function
 */ 
void PrintPriorityData(FILE *fp, int do_newline)
{
    PriorityData *ds_ptr = NULL;  
    
    if(otn_tmp != NULL)
    {
        ds_ptr = (PriorityData *) otn_tmp->ds_list[PLUGIN_PRIORITY_NUMBER];
    }
    
    if(ds_ptr == NULL)
    {
        return;
    }

    if(do_newline)
    {
        fprintf(fp, "[Classification: %s] [Priority: %d]\n", ds_ptr->name, 
                ds_ptr->priority);
    }
    else
    {
        fprintf(fp, "[Classification: %s] [Priority: %d] ", ds_ptr->name, 
                ds_ptr->priority);
    }

}

        
/*
 * Function: PrintXrefs(FILE *)
 *
 * Purpose: Prints out cross reference data associated with an alert
 *
 * Arguments: fp => file descriptor to write the data to
 *            do_newline => tack a \n to the end of the line or not (bool)
 *
 * Returns: void function
 */ 
void PrintXrefs(FILE *fp, int do_newline)
{
    ReferenceData *ds_ptr = NULL;

    if(otn_tmp != NULL)
    {
        ds_ptr = (ReferenceData *) otn_tmp->ds_list[PLUGIN_REFERENCE_NUMBER];
    }

    if(ds_ptr == NULL)
    {
        return;
    }

    while(ds_ptr != NULL)
    {
        if(do_newline)
        {
            fprintf(fp, "[Xref => %s]\n", ds_ptr->url);
        }
        else
        {
            fprintf(fp, "[Xref => %s]", ds_ptr->url);
        }

        ds_ptr = ds_ptr->next;
    }
}


/****************************************************************************
 *
 * Function: LogBin()
 *
 * Purpose: Log packets in binary (tcpdump) format
 *
 * Arguments: None.
 *
 * Returns: void function
 *
 ***************************************************************************/
void LogBin(Packet * p, char *msg, void *arg, Event *event)
{
    if(p != NULL)
    {
        if(pv.obfuscation_flag)
        {
            if(pv.obfuscation_net == 0)
            {
                if(p->iph != NULL)
                {
                    p->iph->ip_src.s_addr = 0x00000000;
                    p->iph->ip_dst.s_addr = 0x00000000;
                }
            }
            else
            {
                if(p->iph != NULL)
                {
                    if(pv.homenet != 0)
                    {
                        if((p->iph->ip_dst.s_addr & pv.netmask) == pv.homenet)
                        {
                            p->iph->ip_src.s_addr = 
                                pv.obfuscation_net | 
                                (p->iph->ip_src.s_addr & pv.obfuscation_mask);
                        }
                        else
                        {
                            p->iph->ip_dst.s_addr =
                                pv.obfuscation_net | 
                                (p->iph->ip_dst.s_addr & pv.obfuscation_mask);
                        }
                    }
                    else
                    {
                            p->iph->ip_src.s_addr = 
                                pv.obfuscation_net | 
                                (p->iph->ip_src.s_addr & pv.obfuscation_mask);
                            p->iph->ip_dst.s_addr =
                                pv.obfuscation_net | 
                                (p->iph->ip_dst.s_addr & pv.obfuscation_mask);
                    }
                }
            }
        }
        
        pcap_dump((u_char *) dumpd, p->pkth, p->pkt);

        if(!pv.line_buffer_flag)
        {
            fflush((FILE *) dumpd);
        }
    }
}



/****************************************************************************
 *
 * Function: IcmpFileName(Packet *p)
 *
 * Purpose: Set the filename of an ICMP output log according to its type
 *
 * Arguments: p => Packet data struct
 *
 * Returns: the name of the file to set
 *
 ***************************************************************************/
char *IcmpFileName(Packet * p)
{
    if(p->icmph == NULL)
    {
        return "ICMP_TRUNC";
    }

    switch(p->icmph->type)
    {
        case ICMP_ECHOREPLY:
            return "ECHO_REPLY";

        case ICMP_DEST_UNREACH:
            switch(p->icmph->code)
            {
                case ICMP_NET_UNREACH:
                    return "NET_UNRCH";

                case ICMP_HOST_UNREACH:
                    return "HST_UNRCH";

                case ICMP_PROT_UNREACH:
                    return "PROTO_UNRCH";

                case ICMP_PORT_UNREACH:
                    return "PORT_UNRCH";

                case ICMP_FRAG_NEEDED:
                    return "UNRCH_FRAG_NEEDED";

                case ICMP_SR_FAILED:
                    return "UNRCH_SOURCE_ROUTE_FAILED";

                case ICMP_NET_UNKNOWN:
                    return "UNRCH_NETWORK_UNKNOWN";

                case ICMP_HOST_UNKNOWN:
                    return "UNRCH_HOST_UNKNOWN";

                case ICMP_HOST_ISOLATED:
                    return "UNRCH_HOST_ISOLATED";

                case ICMP_NET_ANO:
                    return "UNRCH_NET_ANO";

                case ICMP_HOST_ANO:
                    return "UNRCH_HOST_ANO";

                case ICMP_NET_UNR_TOS:
                    return "UNRCH_NET_UNR_TOS";

                case ICMP_HOST_UNR_TOS:
                    return "UNRCH_HOST_UNR_TOS";

                case ICMP_PKT_FILTERED:
                    return "UNRCH_PACKET_FILT";

                case ICMP_PREC_VIOLATION:
                    return "UNRCH_PREC_VIOL";

                case ICMP_PREC_CUTOFF:
                    return "UNRCH_PREC_CUTOFF";

                default:
                    return "UNKNOWN";

            }

        case ICMP_SOURCE_QUENCH:
            return "SRC_QUENCH";

        case ICMP_REDIRECT:
            return "REDIRECT";

        case ICMP_ECHO:
            return "ECHO";

        case ICMP_TIME_EXCEEDED:
            return "TTL_EXCEED";

        case ICMP_PARAMETERPROB:
            return "PARAM_PROB";

        case ICMP_TIMESTAMP:
            return "TIMESTAMP";

        case ICMP_TIMESTAMPREPLY:
            return "TIMESTAMP_RPL";

        case ICMP_INFO_REQUEST:
            return "INFO_REQ";

        case ICMP_INFO_REPLY:
            return "INFO_RPL";

        case ICMP_ADDRESS:
            return "ADDR";

        case ICMP_ADDRESSREPLY:
            return "ADDR_RPL";

        default:
            return "UNKNOWN";
    }
}



/****************************************************************************
 *
 * Function: InitLogFile(char *logname)
 *
 * Purpose: Initialize the tcpdump log file header
 *
 * Arguments: None.
 *
 * Returns: void function
 *
 ***************************************************************************/
void InitBinLogFile(char *logname)
{
    time_t curr_time;       /* place to stick the clock data */
    struct tm *loc_time;    /* place to stick the adjusted clock data */
    char timebuf[TIMEBUF_SIZE];
    char logdir[STD_BUF];

    bzero((char *) timebuf, TIMEBUF_SIZE);
    curr_time = time(NULL);
    loc_time = localtime(&curr_time);

    strftime(timebuf, TIMEBUF_SIZE-1, "%m%d@%H%M", loc_time);

    if (logname != NULL && strlen(logname) != 0)
    {
        if (strlen(pv.log_dir) + strlen(logname) + 
                strlen(chrootdir == NULL?"":chrootdir) + 2 < sizeof(logdir))
        {
            snprintf (logdir, sizeof(logdir) -1, "%s%s/%s",
                    chrootdir == NULL ? "" : chrootdir, pv.log_dir, logname);
        }
        else
        {
            FatalError("ERROR => InitBinLogFile(logname) %s\n", logname);
        }
    }
    else
    {
        bzero((char *) timebuf, TIMEBUF_SIZE);
        curr_time = time(NULL);
        loc_time = localtime(&curr_time);

        strftime(timebuf, TIMEBUF_SIZE, "%m%d@%H%M", loc_time);

        bzero((char *) logdir, sizeof(logdir));

        if(strlen(pv.log_dir) + strlen(timebuf) + 
                strlen( chrootdir == NULL ? "" : chrootdir) + 12 < sizeof(logdir))
        {
            snprintf(logdir, sizeof(logdir) -1,  "%s%s/snort-%s.log", 
                    chrootdir == NULL ? "" : chrootdir, pv.log_dir, timebuf);
        }
    }

#ifdef DEBUG
    printf("Opening %s\n", logdir);
#endif

    if((dumpd = pcap_dump_open(pd, logdir)) == NULL)
    {
        FatalError("ERROR => InitBinLogFile(%s) pcap_dump_open: %s\n",
                logdir, strerror(errno));
    }
#ifdef DEBUG
    printf("BinLogFile file initialized\n");
#endif

    return;
}



/* This function name is being altered for Win32 because it
   conflicts with a Win32 SDK function name.  However calls to
   this function from within Snort do not need to change because
   SetEvent() is defined in log.h to evaluate to SnortSetEvent()
   on Win32 compiles.
 */
#ifndef WIN32
void SetEvent
#else
void SnortSetEvent
#endif
       (Event *event, u_int32_t generator, u_int32_t id, u_int32_t rev, 
        u_int32_t classification, u_int32_t priority, u_int32_t event_ref)
{
    event->sig_generator = generator;
    event->sig_id = id;
    event->sig_rev = rev;
    event->classification = classification;
    event->priority = priority;
    event->event_id = ++event_id;         /* this one gets set automatically */
    event->event_reference = event_ref;

    return;
}

