/* $Id: sp_priority.c,v 1.5 2001/08/31 06:24:14 roesch Exp $ */

/*
** Copyright (C) 2001 Brian Caswell <bmc@mitre.org>
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

/*
 * sp_priority 
 *
 * Purpose:
 *
 * This plugin allows alerts to be classified and prioritized. The user can 
 * specify what priority each type of rule classification has.  Rules that
 * have a classification will have a default priority set.  Any rule can 
 * set a priority that overrides the default priority. 
 * 
 * Arguments:
 *  
 * As this plugin is based on keywords in rules
 * The rule format is:
 *            
 *      classtype: <rule_type>;
 *         or 
 *      priority: <prioritynumber>; 
 *
 * Effect:
 *
 * This stores the TYPE into a field.  Each type has a default priority set.
 * If a priority exists, then the specific priority will override the default.
 *
 * Here are a few examples.
 *
 * alert TCP any any -> any 80 (msg: "WEB-MISC phf"; flags: A+; \
 *           content: "/cgi-bin/phf"; classtype:attempted-user; )
 *
 * alert TCP any any -> any 80 (msg: "WEB-MISC phf attempt"; flags:A+; \
 *           content: "/cgi-bin/bash"; priority:10;)
 *
 */

#include "sp_priority.h"

extern char *file_name;  /* 
                          * this is the file name from rules.c, generally 
                          * used for error messages
                          */

extern int file_line;    /* 
                          * this is the file line number from rules.c that is
                          * used to indicate file lines for error messages 
                          */


/****************************************************************************
 * 
 * Function: SetupPriority()
 *
 * Purpose: Registers our keywords
 *
 * Arguments: None.
 *
 * Returns: void function
 *
 ****************************************************************************/
void SetupPriority()
{
    /* map the keyword to an initialization/processing function */
    RegisterPlugin("classtype", TypeInit);
    RegisterPlugin("priority", PriorityInit);
    DebugMessage(DEBUG_PLUGIN,"Plugin: Priority Setup\n");
}


/****************************************************************************
 * 
 * Function: TypeInit(char *, OptTreeNode *)
 *
 * Purpose: rule configuration functions
 *`
 * Arguments: data => rule arguments/data
 *            otn => pointer to the current rule option list node
 *
 * Returns: void function
 *
 ****************************************************************************/
void TypeInit(char *data, OptTreeNode *otn, int protocol)
{
    /* 
     * allocate the data structure and attach it to the
     *  rule's data struct list 
     */
    DebugMessage(DEBUG_PLUGIN,"Plugin: Priority Setting up TYPES\n");

    /* whack a new node onto the list */
    NewPriorityNode(otn);

    /* this is where the keyword arguments are processed and placed into the 
       rule option's data structure */
    TypeRuleParseFunction(data, otn);
}

/****************************************************************************
 * 
 * Function: TypeRuleParseFunction(char *, OptTreeNode *)
 *
 * Purpose: This is the function that is used to process the option keyword's
 *          arguments and attach them to the rule's data structures.
 *
 * Arguments: data => argument data
 *            otn => pointer to the current rule's OTN
 *
 * Returns: void function
 *
 ****************************************************************************/
void TypeRuleParseFunction(char *data, OptTreeNode *otn)
{
    PriorityData *ds_ptr;  /* data struct pointer */
    int pri_set = 0;
    ClassTypes *idx;

    DebugMessage(DEBUG_PLUGIN,"Plugin: Priority Parsing TYPE arguements\n");
    /* set the ds pointer to make it easier to reference the option's
       particular data struct */
    ds_ptr = (PriorityData *)otn->ds_list[PLUGIN_PRIORITY_NUMBER];

    /* manipulate the option arguments here */
    while (isspace((int)*data)) data++;

    idx = pv.ct;
    while (idx != NULL)
    {
        if (!strcasecmp(data,idx->type))
        {
            if(!ds_ptr->priority)
            {
                ds_ptr->priority = idx->priority;
                otn->event_data.priority = idx->priority;
            }

            if(ds_ptr->name != NULL)
            {
                free(ds_ptr->name);
            }

            ds_ptr->name = idx->name;
            ds_ptr->type = data;
            ds_ptr->id = idx->id;
            otn->event_data.classification = idx->id;
            pri_set = 1;
            DebugMessage(DEBUG_PLUGIN,"Plugin: ClassType set to %s\n", data);

            return;
        }

        idx = idx->next;
    }

    if(!pri_set)
    {
        ErrorMessage("ERROR %s(%d) => Bad Priority setting \"%s\"\n", 
                     file_name, file_line, data);
    }

}



/****************************************************************************
 * 
 * Function: PriorityInit(char *, OptTreeNode *)
 *
 * Purpose: rule configuration functions
 *
 * Arguments: data => rule arguments/data
 *            otn => pointer to the current rule option list node
 *
 * Returns: void function
 *
 ****************************************************************************/
void PriorityInit(char *data, OptTreeNode *otn, int protocol)
{
    /* 
     * allocate the data structure and attach it to the
     *  rule's data struct list 
     */

    DebugMessage(DEBUG_PLUGIN,"Plugin: Priority setting up PRIORITY\n");

    /* whack a new node onto the list */
    NewPriorityNode(otn);

    /* this is where the keyword arguments are processed and placed into the 
       rule option's data structure */
    PriorityRuleParseFunction(data, otn);
}

/****************************************************************************
 * 
 * Function: PriorityRuleParseFunction(char *, OptTreeNode *)
 *
 * Purpose: This is the function that is used to process the option keyword's
 *          arguments and attach them to the rule's data structures.
 *
 * Arguments: data => argument data
 *            otn => pointer to the current rule's OTN
 *
 * Returns: void function
 *
 ****************************************************************************/
void PriorityRuleParseFunction(char *data, OptTreeNode *otn)
{
    PriorityData *ds_ptr;  /* data struct pointer */
    int priority;

    DebugMessage(DEBUG_PLUGIN,"Plugin: Priority parsing args for PRIORITY\n");

    /* set the ds pointer to make it easier to reference the option's
       particular data struct */
    ds_ptr = (PriorityData *)otn->ds_list[PLUGIN_PRIORITY_NUMBER];

    /* manipulate the option arguments here */
    while (isspace((int)*data)) data++;
 
    priority = atoi(data); 
    ds_ptr->priority = priority;
    ds_ptr->name = strdup("None");

    otn->event_data.priority = priority;

    DebugMessage(DEBUG_PLUGIN,"Plugin: Priority set to %d\n", priority);
}

/****************************************************************************
 * 
 * Function: NewPriorityNode(OptTreeNode *)
 *
 * Purpose: This function adds a new node to the PriorityData linked list.
 *          The code was borrowed from NewNode in sp_pattern_match.c
 *
 * Arguments: otn => pointer to the current rule's OTN
 *
 * Returns: void function
 *
 ****************************************************************************/
void NewPriorityNode(OptTreeNode * otn)
{
    PriorityData *ds_ptr;

    DebugMessage(DEBUG_PLUGIN,"Plugin: Priority creating new PRIORITY node\n");
    ds_ptr = (PriorityData *) otn->ds_list[PLUGIN_PRIORITY_NUMBER];

    if (ds_ptr == NULL)
    {
        DebugMessage(DEBUG_PLUGIN,"Plugin: Priority creating new PRIORITY node\n");
        if ((otn->ds_list[PLUGIN_PRIORITY_NUMBER] = (PriorityData *) 
            calloc(sizeof(PriorityData), sizeof(char))) == NULL)
        {
            FatalError("ERROR => sp_priority NewPriorityNode() calloc failed!\n");
        }
    } 
}
