#include <stdlib.h>
#include <math.h>
#include <stddef.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <GString.h>
#include <gmem.h>
#include <gfile.h>
#include <config.h>
#include "Object.h"
#include "Stream.h"
#include "Array.h"
#include "Dict.h"
#include "XRef.h"
#include "Catalog.h"
#include "Page.h"
#include "GfxFont.h"
#include "PDFDoc.h"
#include "Params.h"
#include "Error.h"

#include "epdf.h"

// FIXME: This definition is a duplicate from ../pdftexcoerce.h, which is
// not usable here because of different macro definitions.
extern integer pdflastpdfboxspec ;

// if pdfTeX should care about a /CropBox or should only use the /MediaBox.
// FIXME: This is needed by ArtCom because we currently cannot specify the
// usage of the /MediaBox in LaTeX with pdftex.def
#define USE_CROPBOX 0

// FIXME: These definitions are duplicates from pdftexd.h
/* #define PDF_PDF_BOX_SPEC_MEDIA 0 */
/* #define PDF_PDF_BOX_SPEC_CROP  1 */
/* #define PDF_PDF_BOX_SPEC_BLEED 2 */
/* #define PDF_PDF_BOX_SPEC_TRIM  3 */
/* #define PDF_PDF_BOX_SPEC_ART   4 */
#define pdfpdfboxspecmedia ( 0 ) 
#define pdfpdfboxspeccrop ( 1 ) 
#define pdfpdfboxspecbleed ( 2 ) 
#define pdfpdfboxspectrim ( 3 ) 
#define pdfpdfboxspecart ( 4 ) 

// The prefix for the PDF keys special to pdfTeX
// This has been registered with Adobe by Hans Hagen
#define pdfkeyprefix "PTEX"

// PdfObject encapsulates the xpdf Object type, and properly 
// frees its resources on destruction.
// Use obj-> to access members of the Object, and
// &obj to get a pointer to the object. 
// It is no longer necessary to call Object::free explicitely.

class PdfObject {
public:
  PdfObject() { /* nothing */ }
  ~PdfObject() { iObject.free(); }
  Object* operator->() { return &iObject; }
  Object* operator&() { return &iObject; }
private: // no copying or assigning
  PdfObject(const PdfObject &);
  void operator=(const PdfObject &);
public:
  Object iObject;
};

// when copying the Resources of the selected page, all objects are copied
// recusively top-down. Indirect objects however are not fetched during
// copying, but get a new object number from pdftex and then will be
// appended into a linked list. Duplicates are checked and removed from the
// list of indirect objects during appending.

enum InObjType {
    objFont,
    objFontDesc,
    objOther
};

struct InObj {
    Ref ref;            // ref in original PDF
    InObjType type;     // object type
    InObj *next;        // next entry in list of indirect objects
    integer num;        // new object number in output PDF
    int fontmap;        // index of font map entry
    integer encoding;   // Encoding for objFont      
    int written;        // has it been written to output PDF?
};

struct UsedEncoding {
    integer encoding;
    GfxFont *font;
    UsedEncoding *next;
};

static InObj *inObjList;
static UsedEncoding *encodingList;
static GBool isInit = gFalse;

// --------------------------------------------------------------------
// Maintain list of open embedded PDF files
// --------------------------------------------------------------------

struct PdfDocument {
    char *file_name;
    PDFDoc *doc;
    XRef *xref;
    InObj *inObjList;
    PdfDocument *next;
};

static PdfDocument *pdfDocuments = 0;

// Returns pointer to PdfDocument record for PDF file.
// Creates a new record if it doesn't exist yet.
// xref is made current for the document.

static PdfDocument *find_add_document(char *file_name)
{
    PdfDocument *p = pdfDocuments;
    while (p && strcmp(p->file_name, file_name) != 0)
        p = p->next;
    if (p) {
        xref = p->xref;
	return p;
    }
    p = new PdfDocument;
    p->file_name = strdup(file_name);
    p->xref = xref = 0;
    GString *docName = new GString(p->file_name);
    p->doc = new PDFDoc(docName);  // takes ownership of docName
    if (!p->doc->isOk() || !p->doc->okToPrint())
        pdftex_fail("xpdf: reading PDF image failed");
    p->inObjList = 0;
    p->next = pdfDocuments;
    pdfDocuments = p;
    return p;
}

// Deallocate a PdfDocument with all its resources
static void delete_document(PdfDocument *pdf_doc)
{
    PdfDocument **p = &pdfDocuments;
    while (*p && *p != pdf_doc)
      p = &((*p)->next);
    // should not happen:
    if (!*p) 
      return;
    // unlink from list
    *p = pdf_doc->next;
    // free pdf_doc's resources
    InObj *r, *n;
    for (r = pdf_doc->inObjList; r != 0; r = n) {
        n = r->next;
        delete r;
    }
    xref = pdf_doc->xref;
    delete pdf_doc->doc;
    xfree(pdf_doc->file_name);
    delete pdf_doc;
}

// --------------------------------------------------------------------

static int addEncoding(GfxFont *gfont)
{
    UsedEncoding *n;
    n = new UsedEncoding;
    n->next = encodingList;
    encodingList = n;
    n->font = gfont;
    n->encoding = pdfnewobjnum();
    return n->encoding;
}

#define addFont(ref, fontmap, encoding) \
        addInObj(objFont, ref, fontmap, encoding)

#define addFontDesc(ref, fontmap) \
        addInObj(objFontDesc, ref, fontmap, 0)

#define addOther(ref) \
        addInObj(objOther, ref, 0, 0)

static int addInObj(InObjType type, Ref ref, int f, integer e)
{
    InObj *p, *q, *n = new InObj;
    if (ref.num == 0)
        pdftex_fail("pdf inclusion: invalid reference");
    n->ref = ref;
    n->type = type;
    n->next = 0;
    n->fontmap = f;
    n->encoding = e;
    n->written = 0;
    if (inObjList == 0)
        inObjList = n;
    else {
        for (p = inObjList; p != 0; p = p->next) {
            if (p->ref.num == ref.num && p->ref.gen == ref.gen) {
                delete n;
                return p->num;
            }
            q = p;
        }
	// it is important to add new objects at the end of the list,
	// because new objects are being added while the list is being
	// written out.
        q->next = n;
    }
    n->num = pdfnewobjnum();
    return n->num;
}

static void copyObject(Object *);

static void copyName(char *s)
{
    pdf_puts("/");
    for (; *s != 0; s++) {
        if (isdigit(*s) || isupper(*s) || islower(*s) || *s == '_' ||
	    *s == '.' || *s == '-' )
            pdfout(*s);
        else
            pdf_printf("#%.2X", *s & 0xFF);
    }
}

static void copyDictEntry(Object *obj, int i)
{
    PdfObject obj1;
    copyName(obj->dictGetKey(i));
    pdf_puts(" ");
    obj->dictGetValNF(i, &obj1);
    copyObject(&obj1);
    pdf_puts("\n");
}

static void copyDict(Object *obj)
{
    int i, l;
    if (!obj->isDict())
        pdftex_fail("pdf inclusion: invalid dict type <%s>", 
                    obj->getTypeName());
    for (i = 0, l = obj->dictGetLength(); i < l; ++i)
        copyDictEntry(obj, i);
}

static void copyFontDict(Object *obj, InObj *r)
{
    int i, k, l;
    char *key;
    if (!obj->isDict())
        pdftex_fail("pdf inclusion: invalid dict type <%s>", 
                    obj->getTypeName());
    pdf_puts("<<\n");
    if (r->type == objFont) { // Font dict
        for (i = 0, l = obj->dictGetLength(); i < l; ++i) {
            key = obj->dictGetKey(i);
            if (strcmp("BaseFont", key) == 0 || 
                strcmp("Encoding", key) == 0)
                continue; // skip original values
            copyDictEntry(obj, i);
        }
        // write new BaseFont and Encoding
        pdf_printf("/BaseFont %i 0 R\n", (int)get_fontname(r->fontmap)); 
        pdf_printf("/Encoding %i 0 R\n", (int)r->encoding); 
    }
    else { // FontDescriptor dict
        for (i = 0, l = obj->dictGetLength(); i < l; ++i) {
            key = obj->dictGetKey(i);
            if (strcmp("FontName", key) == 0 ||
                strncmp("FontFile", key, strlen("FontFile")) == 0)
                continue; // ignore original FontFile/FontName
            if (strcmp("CharSet", key) == 0)
                continue; // ignore CharSet
            copyDictEntry(obj, i);
        }
        // write new FontName and FontFile
        pdf_printf("/FontName %i 0 R\n", (int)get_fontname(r->fontmap)); 
        pdf_printf("/FontFile %i 0 R\n", (int)get_fontfile(r->fontmap));
    }
    pdf_puts(">>");
}

static void copyStream(Stream *str)
{
    int c;
    str->reset();
    if (pdfcrypting) pdfcrypt_initkey();
    while ((c = str->getChar()) != EOF)
        if (pdfcrypting) pdfout(pdfcrypt_byte(c));
        else             pdfout(c);
}

static void copyProcSet(Object *obj)
{
    int i, l;
    PdfObject procset;
    if (!obj->isArray())
        pdftex_fail("pdf inclusion: invalid ProcSet array type <%s>", 
                    obj->getTypeName());
    pdf_puts("/ProcSet [ ");
    for (i = 0, l = obj->arrayGetLength(); i < l; ++i) {
        obj->arrayGet(i, &procset);
        if (!procset->isName())
            pdftex_fail("pdf inclusion: invalid ProcSet entry type <%s>", 
                        procset->getTypeName());
        copyName(procset->getName());
        pdf_puts(" ");
    }
    pdf_puts("]\n");
}

static void copyFont(char *tag, Object *fontRef)
{
    PdfObject fontdict, subtype, basefont, fontdescRef, fontdesc, charset;
    GfxFont *gfont;
    int fontmap;
    // Check whether the font has already been embedded before analysing it.
    InObj *p;
    Ref ref = fontRef->getRef();
    for (p = inObjList; p; p = p->next) {
        if (p->ref.num == ref.num && p->ref.gen == ref.gen) {
	  copyName(tag);
	  pdf_printf(" %d 0 R ", p->num);
	  return;
	}
    }
    fontRef->fetch(&fontdict);
    if (!fontdict->isDict())
        pdftex_fail("pdf inclusion: invalid font dict type <%s>", 
                    fontdict->getTypeName());
    fontdict->dictLookup("Subtype", &subtype);
    if (!subtype->isName())
        pdftex_fail("pdf inclusion: invalid font Subtype entry type <%s>", 
                    subtype->getTypeName());
    /* only handle Type1 fonts; others will be copied */
    if (strcmp(subtype->getName(), "Type1") != 0 ) {
        copyName(tag);
	pdf_puts(" ");
	copyObject(fontRef);
	return;
    }
    fontdict->dictLookup("BaseFont", &basefont);
    if (!basefont->isName())
        pdftex_fail("pdf inclusion: invalid font BaseFont entry type <%s>", 
                    basefont->getTypeName());
    fontmap = lookup_fontmap(basefont->getName());
    if (fontmap >= 0 && is_type1(fontmap) &&
        fontdict->dictLookupNF("FontDescriptor", &fontdescRef) && 
        fontdescRef->isRef() && fontdescRef->fetch(&fontdesc) &&
	fontdesc->isDict()) {
        if (fontdesc->dictLookup("CharSet", &charset) && 
            charset->isString() && is_subsetable(fontmap))
	    mark_glyphs(fontmap, charset->getString()->getCString());
        else
            embed_whole_font(fontmap);
        addFontDesc(fontdescRef->getRef(), fontmap);
    }
    copyName(tag);
    if (fontdesc->isDict()) {
        gfont = new GfxFont(tag, fontRef->getRef(), fontdict->getDict());
        pdf_printf(" %d 0 R ", addFont(fontRef->getRef(), fontmap, 
                                       addEncoding(gfont)));
    }
    else
        pdf_printf(" %d 0 R ", addOther(fontRef->getRef()));
}

static void copyFontResources(Object *obj)
{
    PdfObject fontRef;
    int i, l;
    if (!obj->isDict())
        pdftex_fail("pdf inclusion: invalid font resources dict type <%s>", 
                    obj->getTypeName());
    pdf_puts("/Font << ");
    for (i = 0, l = obj->dictGetLength(); i < l; ++i) {
        obj->dictGetValNF(i, &fontRef);
        if (fontRef->isRef())
            copyFont(obj->dictGetKey(i), &fontRef);
        else
            pdftex_fail("pdf inclusion: invalid font in reference type <%s>", 
                        fontRef->getTypeName());
    }
    pdf_puts(">>\n");
}

static void copyOtherResources(Object *obj, char *key)
{
    PdfObject obj1;
    int i, l;
    if (!obj->isDict())
        pdftex_fail("pdf inclusion: invalid other resources dict type <%s>", 
                    obj->getTypeName());
    copyName(key);
    pdf_puts(" << ");
    for (i = 0, l = obj->dictGetLength(); i < l; ++i) {
        obj->dictGetValNF(i, &obj1);
	copyName(obj->dictGetKey(i));
        if (obj1->isRef())
            pdf_printf(" %d 0 R ", addOther(obj1->getRef()));
        else {
            pdf_puts(" ");
            copyObject(&obj1);
        }
    }
    pdf_puts(">>\n");
}

static char *convertNumToPDF(double n) {
    /* converts double to string; very small and very large numbers are NOT
     * converted to scientific notation.
     * n must be a number or real confirming to the implementation limits of
     * PDF as specified in appendix C.1 of the pdf ref.  
     * These are:
     * maximum value of ints is +2^32
     * maximum value of reals is +2^15
     * smalles values of reals is 1/(2^16)
     */
    static int precision = 6;
    static int fact = (int)1E6;   /* must be 10^precision */
    static double epsilon = 1E-6; /* must be 10^-precision */
    static char buf[64];
    // handle very small values: return 0
    if (fabs(n) < epsilon) {buf[0] = '0'; buf[1] = '\0';}
    else {
        char ints[64];
        int bindex = 0, sindex = 0;
        int ival, fval, i; 
        // handle the sign part if n is negative
        if (n < 0) {
            buf[bindex++] = '-';
            n = -n;    
        }
        // handle the integer part, simply with sprintf
        ival = (int)floor(n);
        n -= ival;
        sprintf(ints, "%d", ival);
        while (ints[sindex] != 0) buf[bindex++] = ints[sindex++];
        // handle the fractional part up to 'precision' digits
        fval = (int)floor(n*fact);
        if (fval) {
            // set a dot
            buf[bindex++] = '.';
            sindex = bindex + precision - 1;
            // fill up trailing zeros with the string terminator NULL
            while (((fval % 10) == 0) && (sindex >= bindex)) {
                buf[sindex--] = '\0';
                fval /= 10;
            }
            // fill up the fractional part back to front
            while (sindex >= bindex) {
                buf[sindex--] = (fval % 10) + '0';
                fval /= 10;
            }
        } else buf[bindex++] = 0;
    }
    return (char *)buf;
}

static void copyObject(Object *obj)
{
    PdfObject obj1;
    int  i, l, c;
    Ref ref;
    char *p;
    GString *s;
    if (obj->isBool()) {
        pdf_printf("%s", obj->getBool() ? "true" : "false");
    }
    else if (obj->isInt()) {
        pdf_printf("%i", obj->getInt());
    }
    else if (obj->isReal()) {
        pdf_printf("%s", convertNumToPDF(obj->getReal()));
    }
    else if (obj->isNum()) {
        pdf_printf("%s", convertNumToPDF(obj->getNum()));
    }
    else if (obj->isString()) {
        s = obj->getString();
        p = s->getCString();
        l = s->getLength();
        if (strlen(p) == l) {
            pdf_puts("(");
            for (; *p != 0; p++) {
                c = *p;
                if (c == '(' || c == ')' || c == '\\')
                    pdf_printf("\\%c", c);
                else if (c < 0x20 || c > 0x7F)
                    pdf_printf("\\%03o", c);
                else
                    pdfout(c);
            }
            pdf_puts(")");
        }
        else {
            pdf_puts("<");
            for (i = 0; i < l; i++) {
                c = s->getChar(i) & 0xFF;
                pdf_printf("%.2x", c);
            }
            pdf_puts(">");
        }
    }
    else if (obj->isName()) {
        copyName(obj->getName());
    }
    else if (obj->isNull()) {
        pdf_puts("null");
    }
    else if (obj->isArray()) {
        pdf_puts("[");
        for (i = 0, l = obj->arrayGetLength(); i < l; ++i) {
            obj->arrayGetNF(i, &obj1);
            if (!obj1->isName())
                pdf_puts(" ");
            copyObject(&obj1);
        }
        pdf_puts("]");
    }
    else if (obj->isDict()) {
        pdf_puts("<<\n");
        copyDict(obj);
        pdf_puts(">>");
    }
    else if (obj->isStream()) {
        obj1->initDict(obj->getStream()->getDict());
        obj->getStream()->getDict()->incRef();
        pdf_puts("<<\n");
        copyDict(&obj1);
        pdf_puts(">>\n");
        pdf_puts("stream\n");
        copyStream(obj->getStream()->getBaseStream());
        pdf_puts("endstream");
    }
    else if (obj->isRef()) {
        ref = obj->getRef();
        if (ref.num == 0) {
            pdftex_warn("pdf inclusion: reference to invalid object was replaced by <null>");
            pdf_puts("null");
        }
        else
            pdf_printf("%d 0 R", addOther(ref));
    }
    else {
        pdftex_fail("pdf inclusion: type <%s> cannot be copied", 
                    obj->getTypeName());
    }
}

static void writeRefs()
{
    PdfObject obj1;
    InObj *r;
    for (r = inObjList; r != 0; r = r->next) {
        if (!r->written) {
	    r->written = 1;
	    zpdfbeginobj(r->num);
	    xref->fetch(r->ref.num, r->ref.gen, &obj1);
	    if (r->type == objFont || r->type == objFontDesc)
	        copyFontDict(&obj1, r);
	    else
                copyObject(&obj1);
	    pdf_puts("\n");
	    pdfendobj();
	}
    }
}

static void writeEncodings()
{
    UsedEncoding *r, *n;
    char *glyphNames[MAX_CHAR_CODE + 1], *s;
    int i;
    for (r = encodingList; r != 0; r = r->next) {
        for (i = 0; i <= MAX_CHAR_CODE; i++)
            if ((s = r->font->getCharName(i)) != 0)
                glyphNames[i] = s;
            else
                glyphNames[i] = notdef;
        write_enc(glyphNames, r->encoding);
    }
    for (r = encodingList; r != 0; r = n) {
        n = r->next;
        delete r->font;
        delete r;
    }
}

// Returns the page number.
integer read_pdf_info(char *image_name, char *page_name, integer page_num)
{
    PdfDocument *pdf_doc;
    Page *page;
    int rotate;
    Rectangle pagebox;
    // initialize
    if (!isInit) {
        // We should better not call xpdf's initParams, which would
        // read $HOME/.xpdfrc to find external font files.  There is
        // no good reason for pdftex to have a hidden dependence on
        // .xpdfrc, and we don't want the output PDF to depend on its
        // contents.  The following four lines replace the call to
        // initParams().
        fontPath = (char **) gmalloc(sizeof(char *));
	fontPath[0] = 0;
	devFontMap = (DevFontMapEntry *) gmalloc(sizeof(DevFontMapEntry));
	devFontMap[0].pdfFont = 0;
	// initParams(xpdfConfigFile);
        errorInit();
        isInit = gTrue;
    }
    // open PDF file
    pdf_doc = find_add_document(image_name);
    epdf_doc = (void *) pdf_doc;
    epdf_num_pages = pdf_doc->doc->getCatalog()->getNumPages();
    if (page_name) {
        // get page by name
        GString name(page_name);
	LinkDest *link = pdf_doc->doc->findDest(&name);
	if (link == 0 || !link->isOk())
	    pdftex_fail("pdf inclusion: invalid destination <%s>",
			page_name);
	Ref ref = link->getPageRef();
	page_num =  pdf_doc->doc->getCatalog()->findPage(ref.num, ref.gen);
	if (page_num == 0)
	    pdftex_fail("pdf inclusion: destination is not a page <%s>",
			page_name);
	delete link;
    } else {
        // get page by number
        if (page_num <= 0 || page_num > epdf_num_pages)
	    pdftex_fail("pdf inclusion: required page does not exist <%i>", 
			epdf_num_pages);
    }
    // get the required page
    page = pdf_doc->doc->getCatalog()->getPage(page_num);

    switch (pdflastpdfboxspec) {
#if USE_CROPBOX
    case pdfpdfboxspeccrop:
        pagebox = page->getLogicalCropBox();
        break;
    
    case pdfpdfboxspecbleed:
        pagebox = page->getLogicalBleedBox();
        break;
    
    case pdfpdfboxspectrim:
        pagebox = page->getLogicalTrimBox();
        break;
    
    case pdfpdfboxspecart:
        pagebox = page->getLogicalArtBox();
        break;
#endif

    default:
        pagebox = page->getLogicalMediaBox();
    }
    epdf_width = pagebox.x2 - pagebox.x1;
    epdf_height = pagebox.y2 - pagebox.y1;
    epdf_orig_x = pagebox.x1;
    epdf_orig_y = pagebox.y1;
    
    rotate = page->getRotate();
    // handle page rotation and adjust dimens as needed
    if (rotate != 0) {
        if (rotate % 90 == 0) {
            // handle only the simple case: multiple of 90s.
            // these are the only values allowed according to the
            // reference (v1.3, p.78).
            // 180 needs no special treatment here
            register float f;
            switch (rotate) {
                case  90: f = epdf_height; epdf_height = epdf_width; epdf_width = f;  break;
                case 270: f = epdf_height; epdf_height = epdf_width; epdf_width = f;  break;
                }
            }
        }
    pdf_doc->xref = xref;
    return page_num;
}

void write_epdf(void)
{
    Page *page;
    PdfObject contents, obj1, obj2;
    Object info;
    char *key;
    int i, l;
    int rotate;
    double scale[6] = {0, 0, 0, 0, 0, 0};
    char mybuf[2048];
    mybuf[0] = '\0';
    PdfDocument *pdf_doc = (PdfDocument *) epdf_doc;
    xref = pdf_doc->xref;
    inObjList = pdf_doc->inObjList;
    encodingList = 0;
    page = pdf_doc->doc->getCatalog()->getPage(epdf_selected_page);
    rotate = page->getRotate();
    Rectangle pagebox;
    // write the Page header
    pdf_puts("/Type /XObject\n");
    pdf_puts("/Subtype /Form\n");
    pdf_puts("/FormType 1\n");

    // write additional information
    convertStringToPDFString (pdf_doc->file_name, mybuf);
    pdf_printf("/%s.FileName (%s)\n", pdfkeyprefix, mybuf);
    pdf_printf("/%s.PageNumber %i\n", pdfkeyprefix, epdf_selected_page);
    pdf_doc->doc->getDocInfoNF(&info);
    if (info.isRef()) {
        // the info dict must be indirect (pdf ref p.61)
        pdf_printf("/%s.InfoDict ", pdfkeyprefix);
        pdf_printf("%d 0 R \n", addOther(info.getRef()));
        }
    
    switch (epdf_page_box) {
#if USE_CROPBOX
    case pdfpdfboxspeccrop:
        pagebox = page->getLogicalCropBox();
        break;
  
    case pdfpdfboxspecbleed:
        pagebox = page->getLogicalBleedBox();
        break;
    
    case pdfpdfboxspectrim:
        pagebox = page->getLogicalTrimBox();
        break;
    
    case pdfpdfboxspecart:
        pagebox = page->getLogicalArtBox();
        break;
#endif
   
    default:
        pagebox = page->getLogicalMediaBox();
    }

    // handle page rotation
    if (rotate != 0) {
        if (rotate % 90 == 0) {
            // this handles only the simple case: multiple of 90s but these
            // are the only values allowed according to the reference
            // (v1.3, p.78).
            // the image is rotated around its center.
            // the /Rotate key is clockwise while the matrix is
            // counterclockwise :-%
            tex_printf (", page is rotated %d degrees", rotate);
            switch (rotate) {
                case  90: scale[1] = -1; scale[2] = 1; scale[5] = pagebox.x1 + pagebox.x2; break;
                case 180: scale[0] = scale[3] = -1;    scale[4] = pagebox.x1 + pagebox.x2; scale[5] = pagebox.y1 + pagebox.y2; break; // width and height are exchanged
                case 270: scale[1] = 1; scale[2] = -1; scale[4] = pagebox.y1 + pagebox.y2; break;
                }
            }
        }
    else {
        scale[0] = scale[3] = 1;
        }

    pdf_printf("/Matrix [%.5g %.5g %.5g %.5g %.5g %.5g]\n",
        scale[0],
        scale[1],
        scale[2],
        scale[3],
        scale[4],
        scale[5]);

    pdf_printf("/BBox [%.5g %.5g %.5g %.5g]\n",
               pagebox.x1,
               pagebox.y1,
               pagebox.x2,
               pagebox.y2);
    
    // write the Resources dictionary
    obj1->initDict(page->getResourceDict());
    page->getResourceDict()->incRef();
    if (!obj1->isDict())
        pdftex_fail("pdf inclusion: invalid resources dict type <%s>", 
                    obj1->getTypeName());
    pdf_puts("/Resources <<\n");
    for (i = 0, l = obj1->dictGetLength(); i < l; ++i) {
        obj1->dictGetVal(i, &obj2);
        key = obj1->dictGetKey(i);
        if (strcmp("Font", key) == 0)
            copyFontResources(&obj2);
        else if (strcmp("ProcSet", key) == 0)
            copyProcSet(&obj2);
        else
            copyOtherResources(&obj2, key);
    }
    pdf_puts(">>\n");
    // write the page contents
    page->getContents(&contents);
    if (contents->isStream()) {
        obj1->initDict(contents->getStream()->getDict());
        contents->getStream()->getDict()->incRef();
        copyDict(&obj1);
        pdf_puts(">>\nstream\n");
        copyStream(contents->getStream()->getBaseStream());
        pdf_puts("endstream\n");
        pdfendobj();
    }
    else if (contents->isArray()) {
        pdfbeginstream();
        for (i = 0, l = contents->arrayGetLength(); i < l; ++i) {
            contents->arrayGet(i, &obj1);
            copyStream(obj1->getStream());
        }
        pdfendstream();
    }
    else
        pdftex_fail("pdf inclusion: invalid page contents type <%s>", 
                    contents->getTypeName());
    // write out all indirect objects
    writeRefs();
    // write out all used encodings (and delete list)
    writeEncodings();
    // save object list, xref
    pdf_doc->inObjList = inObjList;
    pdf_doc->xref = xref;
}

// Called when an image has been written and it's resources in
// image_tab are freed.  We cannot deallocate the PdfDocument yet, as
// future pages of the same document may be embedded. As an
// optimization, we do delete it if it's a one-page document.
void epdf_delete()
{
    PdfDocument *pdf_doc = (PdfDocument *) epdf_doc;
    xref = pdf_doc->xref;
    if (pdf_doc->doc->getCatalog()->getNumPages() == 1)
	delete_document(pdf_doc);
}

// Called when PDF embedding system is finalized.
// Now deallocate all remaining PdfDocuments.
void epdf_check_mem()
{
    if (isInit) {
        PdfDocument *p, *n;
	for (p = pdfDocuments; p; p = n) {
	    n = p->next;
	    delete_document(p);
	}
	// see above for initParams
	gfree(fontPath);
	gfree(devFontMap);
	// freeParams();
        Object::memCheck(errFile);
        gMemReport(errFile);
    }
}
