/*****************************************************************************
 * config.c -- Part of uutraf.c, an UUCP Traffic analyzer and cost estimator *
 *                                                                           *
 * Reads an Taylor-UUCP Log-/Statsfile, and generates a report out of it     *
 *                                                                           *
 * config.c is (c) David Frey, 1995    					     *
 *								             *
 * This program is free software; you can redistribute it and/or modify it   *
 * under the terms of the GNU General Public License as published by the     *
 * Free Software Foundation; either version 2 of the License, or (at your    *
 * option) any later version.                                                *
 *									     *
 * This program is distributed in the hope that it will be useful, but       *
 * WITHOUT ANY WARRANTY; without even the implied warranty of                *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.                      *
 * See the GNU General Public License for more details.                      *
 *									     *
 * You should have received a copy of the GNU General Public License along   *
 * with this program; if not, write to the Free Software Foundation, Inc.,   *
 * 675 Mass Ave, Cambridge, MA 02139, USA.                                   *
 *****************************************************************************/

/* $Log: config.c,v $
 * Revision 1.0  1996/01/01 16:42:07  david
 * Initial revision
 * */

static char RCSId[]="$Id: config.c,v 1.0 1996/01/01 16:42:07 david Rel $"; 

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <malloc.h>
#include <string.h>
#include <time.h>
#include <limits.h>

#include "uutraf.h"

hostentry *BuildHostTable(char *cfgfilename)
/* 
 *  Reads in the config file, which is a table containing the entries 
 *      host cost/min fee/mail fee/(mail*kB) cost/call cost/min
 * , and builds a linked list of host entries.
 */
{
  FILE *cfgfile;
  char line[LINE_MAX+1];
  int  lineno;
  hostentry *hosttab, *ptr;

  hosttab = NULL;
  cfgfile = fopen(cfgfilename,"r");
  if (cfgfile == NULL)
    fprintf(stderr,
	    "%s: Warning: no costs will be calculated due to missing '%s'\n",
	    progname,cfgfilename);
  else
  {
    lineno=1;
    while (fgets(line,LINE_MAX,cfgfile) != NULL)
    {
      if (line[0] != '#') 
      {
	ptr = calloc(1,sizeof(hostentry));
	if (ptr != NULL)
	{
	  /* host cost/min fee/mail fee/(mail*kB) cost/call cost/min */
	  if (sscanf(line,"%8s %f %f %f %f %f",
		     ptr->hostname,
		     &ptr->provlinecostpermin,&ptr->provfeepermail,
		     &ptr->provfeepermailkb, &ptr->pttcostpercall,
		     &ptr->pttlinecostpermin) == EOF)
	  {
	    fprintf(stderr,"%s: Parsing error line line %d of %s\n",
		    progname,lineno,cfgfilename);
	  }
	  else
	  {
	    ptr->next = hosttab; hosttab = ptr;
	  }
	}
	else
	{
	  fprintf(stderr,"%s: Out of memory! (BuildHostTable)\n",progname);
	  exit(1);
	}

	lineno++;
      }
    }
    fclose(cfgfile);
  }
  return hosttab;
}

hostentry *gethostentry(char *hostname)
/*
 * Try to find HOSTNAME and to return its pointer.
 * Create a new entry, when HOSTNAME was not found (and issue a warning,
 * since the host is not recorded in the config file). 
 *
 */
{
  hostentry *ptr, *prevptr;

  ptr = hosttab; prevptr = NULL;
  while ((ptr != NULL) && (strcmp(ptr->hostname,hostname) != 0))
  {
    prevptr = ptr; ptr = ptr->next; 
  }

  if (ptr == NULL)    /* HOSTNAME not found */
  {
    ptr = calloc(1,sizeof(hostentry));
    if (ptr != NULL)
    {
      strcpy(ptr->hostname,hostname);
      ptr->next = hosttab; hosttab = ptr;

      fprintf(stderr,
	      "%s: Warning: no entry for %s. No costs will be calculated.\n",
	      progname,hostname);
    }
    else
    {
      fprintf(stderr,"%s: Out of memory! (GetHostEntry)\n",progname);
      exit(1);
    }
  }

  return ptr;
}
