/* `mecc.c' -- A simple application of the rs255-249 and gf256 library.

   Copyright (C) 1996 Free Software Foundation, Inc.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

   Written by Fung-Chai Lim (fclim@acm.org).

   This file is part of GNU ecc.  */

#ifdef HAVE_CONFIG_H
#if defined (CONFIG_BROKETS)
/* We use <config.h> instead of "config.h" so that a compilation
   using -I. -I$srcdir will use ./config.h rather than $srcdir/config.h
   (which it would do because it found this file in $srcdir).  */
#include <config.h>
#else
#include "config.h"
#endif
#endif

#include <stdio.h>
#include <fcntl.h>

#if HAVE_UNISTD_H
#include <sys/types.h>
#include <unistd.h>
#endif
#ifndef F_OK
#define F_OK	0
#endif

#include <errno.h>
#ifndef errno
extern int errno;
#endif

#include <rsff-f9.h>

extern void error _PROTO_ ((int exitStatus, int errno, char const *format,
			    ...));
extern void process_options _PROTO_ ((int argc, char *argv[], int *encode,
				      int *force, int *verbose));
extern void usage _PROTO_ ((int status));

static void encode_file _PROTO_ ((const char *const infname,
                                  const char *const outfname, int force));
static void decode_file _PROTO_ ((const char *const infname,
                                  const char *const outfname, int force,
				  int verbose));

     void
main (argc, argv)
     int argc;
     char **argv;
{
  int encode;
  int force;
  int verbose;
  char *infname, *outfname;
  extern int optind;

  process_options (argc, argv, &encode, &force, &verbose);

  infname = outfname = NULL;
  if (optind != argc)
    {
      argc -= optind;
      argv += optind;
      if (argc > 2)
	usage (1);
      if (argc == 2)
	outfname = argv[1];
      if (argc > 0)
	infname = argv[0];
    }

  if (encode)
    encode_file (infname, outfname, force);
  else
    decode_file (infname, outfname, force, verbose);
}

#include "mfile.h"

#define MMORE_INPUT(n, file, bytes) \
  do { \
    if ((file)->mode != MREAD) \
      (n) = 0; \
    else { \
      int _left = (file)->lim - (file)->beg; \
      if (_left > (bytes)) \
	(n) = (bytes); \
      else { \
	int _more = mfill ((file), (bytes)); \
	(n) = (_more > (bytes)) ? (bytes) : _more; \
      } \
    } \
  } while (0)

#define MMORE_OUTPUT(output, type, file, bytes) \
  do { \
    if ((file)->mode != MWRITE) \
      (output) = (type) 0; \
    else { \
      if ((file)->beg >= (file)->save) \
        mflush ((file)); \
      (output) = mbuf ((file)); \
    } \
  } while (0)

static struct mfile *min, *mout;

     static void
open_files (infname, outfname, force)
     const char *infname;
     const char *outfname;
     int force;
{
  int fd;

  if (infname == NULL || strEQ (infname, "-"))
    {
      infname = "stdin";
      fd = fileno (stdin);
    }
  else if ((fd = open (infname, O_RDONLY)) < 0)
    error (1, errno, "can't open %s for reading.", infname);
  if ((min = mopen (infname, fd, MREAD)) == NULL)
    exit (1);

  if (outfname == NULL || strEQ (infname, "-"))
    {
      outfname = "stdout";
      fd = fileno (stdout);
    }
  else if (!access (outfname, F_OK) && !force)
    {
      error (0, 0, "Output file %s already exists.", outfname);
      usage (1);
    }
  else if ((fd = open (outfname, O_RDWR | O_CREAT | O_TRUNC, PERM_MODE)) < 0)
    error (1, errno, "can't open %s for writing.", outfname);
  if ((mout = mopen (outfname, fd, MWRITE)) == NULL)
    exit (1);
}

     static void
encode_file (infname, outfname, force)
     const char *const infname;
     const char *const outfname;
     int force;
{
  int i, n;
  GF256 *input, *output;
#ifdef COMPAT_WITH_ECC
  int fsize;
  GF256 save[248];
#endif

  open_files (infname, outfname, force);

  MMORE_OUTPUT (output, GF256 *, mout, 256);
  for (i = 1; i < 249; i++)
    output[i] = 0;

#ifdef COMPAT_WITH_ECC
  fsize = min->st.st_size;
#endif

  do
    {
      MMORE_INPUT (n, min, 248);
      input = (GF256 *) mbuf (min);
      output[0] = 'G';
      for (i = 0; i < n; i++)
	output[i+1] = input[i];
#ifdef COMPAT_WITH_ECC
      if (fsize < 248 + 248 && fsize >= 248)
	for (i = 0; i < 248; i++)
	  save[i] = input[i];
      else if (fsize < 248)
	for (; i < 248; i++)
	  output[i+1] = save[i];
      fsize -= n;
#else
      if (n < 248)
        for (; i < 248; i++)
	  output[i+1] = 0;
#endif /* COMPAT_WITH_ECC */
      output[249] = n;
      mread (min, n);

      rs255_249_encode (&output[1], &output[250]);
      mwrite (mout, 256);
      MMORE_OUTPUT (output, GF256 *, mout, 256);
    }
  while (n == 248);

  mclose (min);
  mclose (mout);
  exit (0);
}

     static void
decode_file (infname, outfname, force, verbose)
     const char *const infname;
     const char *const outfname;
     int force, verbose;
{
  int n, i;
  int block;
  int status;
  int errors;
  GF256 *input, *output;

  open_files (infname, outfname, force);

  block = errors = 0;
  while (1)
    {
      block++;
      MMORE_INPUT (n, min, 256);
      input = mbuf (min);

      if (n <= 0)
	break;
      if (n != 256 || input[0] != 'G')
	error (1, 0, "sync error in block %d.\n", block);

      MMORE_OUTPUT (output, GF256 *, mout, 256);
      for (i = 1; i < 256; i++)
	output[i-1] = input[i];
      mread (min, 256);

      status = rs255_249_decode (&output[0]);

      if (status == RS255_249_E4)
	{
	  errors++;
	  error (0, 0, "unrecoverable error in block %d.", block);
	}
      else if (status != RS255_249_E0 && verbose)
	{
	  int m;

	  m = rs255_249_number_of_errors (status);
	  error (0, 0, "%d byte error in block %d.", m, block);
	}
      
      n = output[248];
      mwrite (mout, n);
    }

  mclose (min);
  mclose (mout);
  exit (errors);
}
