/* 
 * Argument handling and main.
 * Copyright (c) 1995 Markku Rossi.
 *
 * Author: Markku Rossi <mtr@iki.fi> 
 */

/*
 * This file is part of genscript.
 * 
 * Genscript is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * Genscript is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with genscript; see the file COPYING.  If not, write to
 * the Free Software Foundation, 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include "gsint.h"
#include "getopt.h"

/*
 * Prototypes for static functions.
 */

/* Handle options from environment variable <var> */
static void handle_env_options (char *var);

/* Handle options from <argv> array. */
static void handle_options (int argc, char *argv[]);

/* Print usage info. */
static void usage ();

/* Print version info. */
static void version ();


/*
 * Global variables.
 */

char *program;			/* Program's name, used for messages. */
FILE *ofp = NULL;		/* Output file. */
char version_string[256];	/* Genscript's version string. */
char date_string[256];		/* Preformatted time string. */
struct tm run_tm;		/* Time when program is run. */
struct tm mod_tm;		/* Last modification time for current file. */
struct passwd *passwd;		/* Passwd entry for the user running this 
				   program. */

/* Path to our library. */
char *genscript_library = GENSCRIPT_LIBRARY;

/* Library lookup path. */
char libpath[1024];

/* AFM library lookup path. */
char *afm_path = NULL;
char afm_path_buffer[1024];

MediaEntry *media_names = NULL;	/* List of known media. */
MediaEntry *media = NULL;	/* Entry for used media. */
char spooler_command[256];	/* Name of the spooler command. */
char queue_param[16];		/* Parameter for print queue (-P). */
int nl = -1;			/* The newline character: '\n' or '\r'. */
int bs = 8;			/* The backspace character. */

/* Statistics. */
int total_pages = 0;		/* Total number of pages printed. */
int num_truncated_lines = 0;	/* Number of lines truncated. */
int num_missing_chars = 0;	/* Number of unknown characters. */
int missing_chars[256] = {0};	/* Table of unknown characters. */

/* Output media dimensions that are used during PostScript emission. */
int d_page_w = 0;		/* page's width */
int d_page_h = 0;		/* page's height */
int d_header_w = 0;		/* fancy header's width */
int d_header_h = 0;		/* fancy header's height */
int d_output_w = 0;		/* output area's width */
int d_output_h = 0;		/* output area's height  */
int d_output_x_margin = 5;	/* output area's x marginal */
int d_output_y_margin = 5;	/* output area's y marginal */

/* Document needed resources. */
StringHashPtr res_fonts;	/* fonts */

/* Fonts to download. */
StringHashPtr download_fonts;

/* Additional key-value pairs, passed to the generated PostScript code. */
StringHashPtr pagedevice;	/* for setpagedevice */
StringHashPtr statusdict;	/* for statusdict */

/* AFM library handle. */
AFMHandle afm = NULL;


/* Options. */

/* 
 * -1, -2, --columns
 *
 * Number of columns per page.  Default is 1 column.
 */
int num_columns = 1;

/* 
 * -b, --page-header
 * 
 * Set string to be used as a page header.  As a default page header is 
 * constructed from filename, date and page number.
 */
char *page_header = NULL;

/* 
 * -B, --no-header
 *
 * Do not print page headers.
 */

/*
 * -c, --truncate-lines
 *
 * Truncate lines that are longer than the page width.  Default is wrap.
 */
int truncate_lines = 0;

/*
 * -C, --line-numbers
 *
 * Precede each line with its line number.  As a default, do not mark
 * line numbers.
 */
int line_numbers = 0;

/*
 * -d, -P, --queue
 *
 * Name of the queue to which output is put.  Defaults to system's
 * default queue.
 */
char *queue = NULL;
char queue_buf[256];

/* 
 * -e, --special-escapes
 *
 * Enable special escape ('\000') interpretation.
 */
int special_escapes = 0;

/*
 * -f, --font
 *
 * Select body font.
 */
char *Fname = "Courier";
double Fpt = 10.0;
double default_Fpt;		/* Point size of the original font. */
char *default_Fname;		/* Name of the original font. */
int user_body_font_defined = 0;	/* Has user defined new body font? */

double font_widths[256];	/* Width array for body font. */
char font_ctype[256];		/* Font character types. */
int font_is_fixed = 1;		/* Is body font a fixed pitch font? */

/*
 * -F, --header-font
 *
 * Select font to be used to print the standard simple header.
 */
char *HFname = "Courier-Bold";
double HFpt = 10.0;

/*
 * -g, --print-anyway
 *
 * Print document even it contains binary data.  This does nothing
 * since genscript prints files anyway.
 */

/* 
 * -G, --fancy-header
 *
 * Add a fancy header to top of every page.  There are several header styles
 * but the default is 'no fancy header'.
 */
HeaderType header = HDR_SIMPLE;
char *fancy_header_name = NULL;
char fancy_header_default[256];

/* 
 * -i, --indent
 *
 * Indent every line this many characters.
 */
int line_indent = 0;

/* -k XXX */

/* -K XXX */

/* 
 * -L, --lines-per-page
 *
 * Specify how many lines should be printed on a single page.  Normally
 * genscript counts it from font point sizes.
 */
unsigned int lines_per_page = (unsigned int) -1;

/*
 * -m, --mail
 *
 * Send mail notification to user after print job has been completed.
 */
int mail = 0;

/* 
 * -M, --media
 *
 * Name of the output media.  Default is A4.
 */
char *media_name = "A4";
char media_name_buffer[256];

/*
 * -n, --copies
 *
 * Number of copies to print.
 */
int num_copies = 1;

/* 
 * -o, -p, --output-file
 *
 * Leave output to the specified file.  As a default result is spooled to 
 * printer.
 */
char *output_file = OUTPUT_FILE_NONE;

/*
 * -O, --list-missing-characters
 * 
 * List all missing characters.  Default is no listing.
 */
int list_missing_characters = 0;

/* 
 * -q, --quiet
 *
 * Do not tell what we are doing.  Default is to tell something but
 * not --verbose.
 */
int quiet = 0;

/*
 * -r, --landscape
 * -R, --portrait
 *
 * Print with page rotated 90 degrees (landscape mode).  Default is
 * portrait.
 */
int landscape = 0;

/* 
 * -s, --baselineskip
 *
 * Specify baselineskip value that is used when genscript moves to
 * a new line.  Current point movement is font_point_size + baselineskip.
 */
double baselineskip = 1.0;

/* 
 * -t, --title
 * 
 * Title that is printed to the banner page and used in printer queue.
 */
char *title = "genscript output";

/*
 * -T, --tab-size
 *
 * Specify tabulator size.
 */
int tab_size = 8;

/* 
 * -u, --underlay
 *
 * Place text under every page.  Default is no underlay.
 */
double ul_gray = .8;
double ul_ptsize = 200.0;
char *ul_font = "Times-Roman";
char *underlay = NULL;

/*
 * -v, --verbose
 *
 * Tell what we are doing.  Default is no verbose outputs.
 */
int verbose = 0;

/*
 * -X, --encoding
 *
 * Specifies input encoding.  Default is ISO-8859.1.
 */
InputEncoding encoding = ENC_LATIN1;
char *encoding_name = "latin1";
char encoding_name_buffer[256];

/* 
 * -Z, --pass-through-ps-files
 * 
 * Pass through all PostScript files without any modifications.  
 * As a default, don't.
 */
int pass_through_ps_files = 0;

/* 
 * --list-media
 *
 * List all known media.  As a default do not list media names.
 */
int list_media = 0;

/*
 * --list-options
 *
 * Show all options and their values.  Exit successfully.
 */
int list_options = 0;

/* 
 * --page-label-format
 *
 * Format in which page labels are printed, default is "short".
 */
char *page_label_format = "short";
char page_label_format_buf[256];
PageLabelFormat page_label;


/*
 * Static variables.
 */

static struct option long_options[] =
{
  {"columns",		required_argument,	0, 0},
  {"page-header",	required_argument,	0, 'b'},
  {"no-header",		no_argument,		0, 'B'},
  {"truncate-lines",	no_argument,		0, 'c'},
  {"line-numbers",	no_argument,		0, 'C'},
  {"queue",		required_argument,	0, 'd'}, 
  {"setpagedevice",	required_argument,	0, 'D'},
  {"special-escapes",	no_argument,		0, 'e'},
  {"font",		required_argument,	0, 'f'},
  {"header-font",	required_argument,	0, 'F'},
  {"print-anyway",	no_argument,		0, 'g'},
  {"fancy-header",	optional_argument,	0, 'G'},
  {"help", 		no_argument, 		0, 'h'},
  {"indent",		required_argument,	0, 'i'},
  {"lineprinter",	no_argument,		0, 'l'},
  {"lines-per-page",	required_argument,	0, 'L'},
  {"mail",		no_argument,		0, 'm'},
  {"media",		required_argument,	0, 'M'},
  {"copies",		required_argument,	0, 'n'},
  {"newline",		required_argument,	0, 'N'},
  {"output-file",	required_argument,	0, 'p'},
  {"list-missing-characters",	no_argument,	0, 'O'},
  {"quiet",		no_argument,		0, 'q'},
  {"landscape",		no_argument,		0, 'r'},
  {"portrait",		no_argument,		0, 'R'},
  {"baselineskip",	required_argument,	0, 's'},
  {"statusdict",	required_argument,	0, 'S'},
  {"title",		required_argument,	0, 't'},
  {"tab-size",		required_argument,	0, 'T'},
  {"underlay",		required_argument,	0, 'u'},
  {"verbose",		optional_argument,	0, 'v'},
  {"version",		no_argument,		0, 'V'},
  {"encoding",		required_argument,	0, 'X'},
  {"pass-through-ps-files",	no_argument,	0, 'Z'},
  {"list-media",	no_argument,		&list_media, 1},
  {"list-options",	no_argument,		&list_options, 1},
  {"ul-font",		required_argument,	0, 128},
  {"ul-gray",		required_argument,	0, 129},
  {"page-label-format",	required_argument,	0, 130},
  {"download-font",	required_argument,	0, 131},

  {NULL, 0, 0, 0},
};


static struct
{
  char *names[3];
  InputEncoding encoding;
  int nl;
  int bs;
} encodings[] =
{
  {{"latin1", "iso8859", "iso"},	ENC_LATIN1, 		'\n', 8},
  {{"latin2", "iso2", NULL},		ENC_LATIN2, 		'\n', 8},
  {{"ascii", NULL, NULL},		ENC_ASCII, 		'\n', 8},
  {{"asciiscands", NULL, NULL},		ENC_ASCII_SCANDS, 	'\n', 8},
  {{"ibmpc", "pc", "dos"},		ENC_IBMPC, 		'\n', 8},
  {{"mac", NULL, NULL},			ENC_MAC, 		'\r', 8},
  {{"vms", NULL, NULL},			ENC_VMS, 		'\n', 8},
  {{"ps", "PS", NULL},			ENC_PS, 		'\n', 8},

  {{NULL, NULL, NULL}, 0, 0, 0},
};


/* 
 * Global functions.
 */

int 
main (int argc, char *argv[])
{
  InputStream is;
  time_t tim;
  struct tm *tm;
  int i, j, found;
  MediaEntry *mentry;
  char spool_buf[256];
  AFMError afm_error;
  char *cp, *cp2;

  /* Get program's name. */
  program = strrchr (argv[0], '/');
  if (program == NULL)
    program = argv[0];
  else
    program++;

  /* Make getopt_long() to use our modified programname. */
  argv[0] = program;

  /* Create version string. */
  sprintf (version_string, "%s %d.%d (%s)", program, MAJOR_VERSION,
	   MINOR_VERSION, EDITION_STRING);

  /* Create date string. */

  tim = time (NULL);
  tm = localtime (&tim);
  memcpy (&run_tm, tm, sizeof (*tm));

  sprintf (date_string, "%s", asctime (&run_tm));
  i = strlen (date_string);
  date_string[i - 1] = '\0';

  /* Get user's passwd entry. */
  passwd = getpwuid (getuid ());
  if (passwd == NULL)
    fatal ("couldn't get passwd entry for uid=%d: %s", getuid (), 
	   strerror (errno));

  /* Default spooler command name. */
  strcpy (spooler_command, "lpr");
  strcpy (queue_param, "-P");

  /* Default fancy header name. */
  strcpy (fancy_header_default, "enscript");

  /* Fill up build-in libpath. */
  sprintf (libpath, "%s:%s/.genscript", genscript_library, passwd->pw_dir);

  /* Initialize resource sets. */
  res_fonts = strhash_init ();
  download_fonts = strhash_init ();
  pagedevice = strhash_init ();
  statusdict = strhash_init ();

  /*
   * Reading papersize default from /etc/papersize
   */

  /* not yet implemented */

  /* 
   * Read configuration files. 
   */

  /* Global config. */
#define CFG_FILE_NAME "genscript.cfg"
  if (!read_config ("/etc", CFG_FILE_NAME))
    /* Maybe we are not installed yet, let's try ".". */
    if (!read_config ("./lib", CFG_FILE_NAME))
      fatal ("couldn't open config file \"%s\": %s", CFG_FILE_NAME,
	     strerror (errno));

  /* Personal config. */
  (void) read_config (passwd->pw_dir, ".genscriptrc");

  /*
   * Options.
   */

  /* Environment variables. */
  handle_env_options ("ENSCRIPT");
  handle_env_options ("GENSCRIPT");

  /* Command line arguments. */
  handle_options (argc, argv);

  /* 
   * Check options that have some validity conditions.
   */

  /* Input encoding. */

  found = 0;
  for (i = 0; !found && encodings[i].names[0]; i++)
    for (j = 0; j < 3; j++)
      if (encodings[i].names[j] != NULL && MATCH (encodings[i].names[j],
						  encoding_name))
	{
	  /* Found a match for this encoding. */
	  encoding = encodings[i].encoding;
	  encoding_name = encodings[i].names[0];
	  if (nl < 0)
	    nl = encodings[i].nl;
	  bs = encodings[i].bs;
	  found = 1;
	  break;
	}
  if (!found)
    fatal ("unknown encoding: %s", encoding_name);

  /* Fonts. */

  /* Default font for landscape, 2 column printing is Courier 7. */
  if (!user_body_font_defined && landscape && num_columns > 1)
    Fpt = 7.0;

  /* Open AFM library. */
  afm_error = afm_create (afm_path, verbose, &afm);
  if (afm_error != AFM_SUCCESS)
    {
      char buf[256];

      afm_error_to_string (afm_error, buf);
      fatal ("couldn't open AFM library: %s", buf);
    }

  /*
   * Save default Fpt and Fname since special escape 'font' can change 
   * it and later we might want to switch back to the "default" font.
   */
  default_Fpt = Fpt;
  default_Fname = Fname;

  /* Register that document uses at least these fonts. */
  strhash_put (res_fonts, Fname, strlen (Fname) + 1, NULL, NULL);
  strhash_put (res_fonts, HFname, strlen (HFname) + 1, NULL, NULL);

  /* As a default, download both named fonts. */
  strhash_put (download_fonts, Fname, strlen (Fname) + 1, NULL, NULL);
  strhash_put (download_fonts, HFname, strlen (HFname) + 1, NULL, NULL);

  /* Read font's character widths and character types. */
  read_font_info ();

  /* Escape page header string. */
  if (page_header != NULL)
    page_header = escape_string (page_header);

  /* Underlay. */
  if (underlay != NULL)
    {
      strhash_put (res_fonts, ul_font, strlen (ul_font) + 1, NULL, NULL);
      underlay = escape_string (underlay);
    }
  
  /* List media names. */
  if (list_media)
    {
      printf ("known media:\n");
      for (mentry = media_names; mentry; mentry = mentry->next)
	printf ("%s:\tw=%d, h=%d, llx=%d, lly=%d, urx=%d, ury=%d\n",
		mentry->name, mentry->w, mentry->h,
		mentry->llx, mentry->lly, mentry->urx, mentry->ury);
      /* Exit after listing. */
      exit (0);
    }

  /* Output media. */
  for (mentry = media_names; mentry; mentry = mentry->next)
    if (strcmp (media_name, mentry->name) == 0)
      {
	media = mentry;
	break;
      }
  if (media == NULL)
    fatal ("do not know anything about media \"%s\"", media_name);

  /* Page label format. */
  if (MATCH (page_label_format, "short"))
    page_label = LABEL_SHORT;
  else if (MATCH (page_label_format, "long"))
    page_label = LABEL_LONG;
  else
    fatal ("illegal page label format \"%s\"", page_label_format);

  /* 
   * Count output media dimensions.
   */

  if (landscape)
    {
      d_page_w = media->ury - media->lly;
      d_page_h = media->urx - media->llx;
    }
  else
    {
      d_page_w = media->urx - media->llx;
      d_page_h = media->ury - media->lly;
    }


  /* Header. */

  d_header_w = d_page_w;
  switch (header)
    {
    case HDR_NONE:
      d_header_h = 0;
      break;

    case HDR_SIMPLE:
      d_header_h = HFpt * 1.5;
      break;

    case HDR_FANCY:
      d_header_h = 36;
      break;
    }

  d_output_w = d_page_w;
  d_output_h = d_page_h - d_header_h - 2 * d_output_y_margin;

  if (list_options)
    {
#define TF(val) ((val) ? 't' : 'f')

      printf ("libpath=\"%s\"\n", 	libpath);
      printf ("queue=\"%s\"\n", 	queue ? queue : "");
      printf ("queue_param=\"%s\"\n", 	queue_param);
      printf ("verbose=%d\n", 		verbose);
      printf ("num_copies=%d\n", 	num_copies);
      printf ("title=\"%s\"\n", 	title ? title : "");
      printf ("columns=%d\n", 		num_columns);
      printf ("truncate=#%c\n", 	TF (truncate_lines));
      printf ("line_numbers=#%c\n", 	TF (line_numbers));
      printf ("mail=#%c\n", 		TF (mail));
      printf ("quiet=#%c\n", 		TF (quiet));
      printf ("landscape=#%c\n", 	TF (landscape));

      printf ("header=");
      switch (header)
	{
	case HDR_NONE:
	  printf ("none");
	  break;

	case HDR_SIMPLE:
	  printf ("simple");
	  break;

	case HDR_FANCY:
	  printf ("fancy (%s)", fancy_header_name);
	  break;
	}
      printf ("\n");

      printf ("page_header=\"%s\"\n", page_header ? page_header : "");
      printf ("font: name=%s size=%gpt\n", Fname, Fpt);
      printf ("header font: name=%s size=%gpt\n", HFname, HFpt);
      printf ("output_file=%s\n",
	       (output_file == OUTPUT_FILE_NONE
		? "none"
		: (output_file == OUTPUT_FILE_STDOUT
		   ? "stdout"
		   : output_file)));
      printf ("media=%s (w=%d, h=%d, llx=%d, lly=%d, urx=%d, ury=%d)\n",
	       media->name, media->w, media->h, media->llx, media->lly,
	       media->urx,media->ury);
      
      printf ("encoding=%s\n", 		encoding_name);
      printf ("pass_through_ps_files=#%c\n",	TF (pass_through_ps_files));
      printf ("spooler_command=\"%s\"\n", 	spooler_command);
      printf ("special_escapes=#%c\n", 	TF (special_escapes));
      printf ("tab_size=%d\n", 		tab_size);
      printf ("baselineskip=%g\n", 	baselineskip);

      /* statusdict key-value pairs */
      printf ("statusdict: ");
      for (i = strhash_get_first (statusdict, &cp, &j, (void **) &cp2); i;
	   i = strhash_get_next (statusdict, &cp, &j, (void **) &cp2))
	printf ("%s %s ", cp2, cp);
      printf ("\n");
      
      /* setpagedevice key-value pairs */
      printf ("setpagedevice: << ");
      for (i = strhash_get_first (pagedevice, &cp, &j, (void **) &cp2); i;
	   i = strhash_get_next (pagedevice, &cp, &j, (void **) &cp2))
	printf ("/%s %s ", cp, cp2);
      printf (">>\n");

      printf ("nl=%c\n", 		nl == '\n' ? 'n' : 'r');
      printf ("AFM path=%s\n", 		afm_path ? afm_path : "(default)");

      /* Underlay. */
      printf ("underlay=(%s)\n", 	underlay ? underlay : "");
      printf ("ul_gray=%g\n", 		ul_gray);
      printf ("ul_font=%s %gpt\n", 	ul_font, ul_ptsize);

      /* Download fonts. */
      printf ("download-fonts:");
      for (i = strhash_get_first (download_fonts, &cp, &j, (void **) &cp2); i;
	   i = strhash_get_next (download_fonts, &cp, &j, (void **) &cp2))
	printf (" %s", cp);
      printf ("\n");
      exit (0);
    }


  /* Open output file. */
  if (output_file == OUTPUT_FILE_NONE)
    {
      /* Open pipe to printer command. */
      sprintf (spool_buf, "%s %s%s%s", spooler_command,
	       mail ? "-m " : "",
	       queue ? queue_param : "", queue ? queue : "");
      ofp = popen (spool_buf, "w");
      if (ofp == NULL)
	fatal ("couldn't open spooler \"%s\": %s", spool_buf,
	       strerror (errno));
    }
  else if (output_file == OUTPUT_FILE_STDOUT)
    ofp = stdout;
  else
    {
      ofp = fopen (output_file, "w");
      if (ofp == NULL)
	fatal ("couldn't create output file \"%s\": %s", output_file,
	       strerror (errno));
    }


  /* 
   * Process files.
   */

  dump_ps_header ();

  if (optind == argc)
    {
      message (1, "processing stdin\n");
      
      /* stdin's modification time is the current time. */
      memcpy (&mod_tm, &run_tm, sizeof (run_tm));

      is_init (&is);
      is.fp = stdin;
      process_file ("(stdin)", &is);
    }
  else
    {
      for (; optind < argc; optind++)
	{
	  message (1, "processing file %s...\n", argv[optind]);

	  is_init (&is);
	  is.fp = fopen (argv[optind], "r");
	  if (is.fp == NULL)
	    error ("couldn't open input file \"%s\": %s", argv[optind],
		   strerror (errno));
	  else
	    {
	      struct stat stat_st;

	      /* Get modification time. */
	      if (stat (argv[optind], &stat_st) == 0)
		{
		  tim = stat_st.st_mtime;
		  tm = localtime (&tim);
		  memcpy (&mod_tm, tm, sizeof (*tm));

		  process_file (argv[optind], &is);
		}
	      else
		error ("couldn't stat input file \"%s\": %s", argv[optind],
		       strerror (errno));

	      fclose (is.fp);
	    }
	}
    }

  dump_ps_trailer ();

  /* Close output file. */
  if (output_file == OUTPUT_FILE_NONE)
    pclose (ofp);
  else if (output_file != OUTPUT_FILE_STDOUT)
    fclose (ofp);

  /* Tell how things went. */
  message (0, "[ %d pages * %d copy ]", total_pages, num_copies);
  if (output_file == OUTPUT_FILE_NONE)
    message (0, " sent to printer\n");
  else
    message (0, " left in %s\n",
	     output_file == OUTPUT_FILE_STDOUT ? "-" : output_file);
  if (num_truncated_lines)
    message (0, "%d lines were %s\n", num_truncated_lines,
	     truncate_lines ? "truncated" : "wrapped");
  if (num_missing_chars)
    {
      message (0, "%d characters were missing\n", num_missing_chars);
      if (list_missing_characters)
	do_list_missing_characters ();
    }

  /* This is the end. */
  return 0;
}


/*
 * Static functions.
 */


static void
handle_env_options (char *var)
{
  int argc;
  char **argv;
  char *string;
  char *str;
  int i;

  string = getenv (var);
  if (string == NULL)
    return;

  message (1, "handle_env_options(): %s=\"%s\"\n", var, string);

  /* Copy string so we can modify it in place. */
  str = xstrdup (string);

  /*
   * We can count this, each option takes at least 1 character and one 
   * space.  We also need one for program's name and one for the 
   * trailing NULL. 
   */
  argc = (strlen (str) + 1) / 2 + 2;
  argv = xcalloc (argc, sizeof (char *));
  
  /* Set program name. */
  argc = 0;
  argv[argc++] = program;

  /* Split string and set arguments to argv array. */
  i = 0;
  while (str[i])
    {
      /* Skip leading whitespace. */
      for (; str[i] && isspace (str[i]); i++)
	;
      if (!str[i])
	break;

      /* Check for quoted arguments. */
      if (str[i] == '"' || str[i] == '\'')
	{
	  int endch = str[i++];

	  argv[argc++] = str + i;

	  /* Skip until we found the end of the quotation. */
	  for (; str[i] && str[i] != endch; i++)
	    ;
	  if (!str[i])
	    fatal ("syntax error in option string %s=\"%s\":\n\
missing end of quotation: %c", var, string, endch);

	  str[i++] = '\0';
	}
      else
	{
	  argv[argc++] = str + i;

	  /* Skip until whitespace if found. */
	  for (; str[i] && !isspace (str[i]); i++)
	    ;
	  if (str[i])
	    str[i++] = '\0';
	}
    }
  
  /* argv[argc] must be NULL. */
  argv[argc] = NULL;

  message (2, "found following options (argc=%d):\n", argc);
  for (i = 0; i < argc; i++)
    message (2, "%3d = \"%s\"\n", i, argv[i]);

  /* Process options. */
  handle_options (argc, argv);

  /* Check that all got processed. */
  if (optind != argc)
    {
      message (0,
	       "warning: didn't process following options from \
environment variable %s:\n",
	       var);
      for (; optind < argc; optind++)
	message (0, "  option %d = \"%s\"\n", optind, argv[optind]);
    }

  /* Cleanup. */
  xfree (argv);

  /*
   * <str> must not be freed, since some global variables can point to
   * its elements
   */
}


static void
handle_options (int argc, char *argv[])
{
  int c;

  /* Reset optind. */
  optind = 0;

  while (1)
    {
      int option_index = 0;
      const char *cp;

      c = getopt_long (argc, argv,
		       "12b:BcCd:D:ef:F:gGhi:lL:mM:n:N:o:Op:P:qrRs:S:t:T:u:vVX:Z",
		       long_options, &option_index);
      
      if (c == EOF)
	break;

      switch (c)
	{
	case 0:			/* Long option found. */
	  cp = long_options[option_index].name;

	  if (strcmp (cp, "columns") == 0)
	    num_columns = atoi (optarg);
	  break;

	  /* Short options. */

	case '1':		/* one column */
	case '2':		/* two columns */
	  num_columns = c - '0';
	  break;

	case 'b':		/* page header */
	  page_header = optarg;
	  break;

	case 'B':		/* no page headers */
	  header = HDR_NONE;
	  break;

	case 'c':		/* truncate (cut) long lines */
	  truncate_lines = 1;
	  break;

	case 'C':		/* line numbers */
	  line_numbers = 1;
	  break;

	case 'd':		/* specify print queue */
	case 'P':
	  queue = optarg;
	  break;

	case 'D':		/* setpagedevice */
	  parse_key_value_pair (pagedevice, optarg);
	  break;

	case 'e':		/* special escapes */
	  special_escapes = 1;
	  break;

	case 'f':		/* font */
	  if (!parse_font_spec (optarg, &Fname, &Fpt))
	    fatal ("malformed font spec: %s", optarg);
	  user_body_font_defined = 1;
	  break;

	case 'F':		/* header font */
	  if (!parse_font_spec (optarg, &HFname, &HFpt))
	    fatal ("malformed font spec: %s", optarg);
	  break;

	case 'g':		/* print anyway */
	  /* nothing. */
	  break;

	case 'G':		/* fancy header */
	  header = HDR_FANCY;
	  if (optarg)
	    fancy_header_name = optarg;
	  else
	    fancy_header_name = fancy_header_default;

	  if (!file_existsp (fancy_header_name, ".hdr"))
	    fatal ("couldn't find header definition file \"%s.hdr\"",
		   fancy_header_name);
	  break;

	case 'h':		/* help */
	  usage ();
	  exit (0);
	  break;

	case 'i':		/* line indent */
	  line_indent = atoi (optarg);
	  break;

	case 'l':		/* simulate lineprinter */
	  lines_per_page = 65;
	  header = HDR_NONE;
	  landscape = 0;
	  num_columns = 1;
	  break;

	case 'L':		/* lines per page */
	  lines_per_page = atoi (optarg);
	  if (lines_per_page <= 0)
	    fatal ("must print at least one line per each page: %s",
		   argv[optind]);
	  break;

	case 'm':		/* send mail upon completion */
	  mail = 1;
	  break;

	case 'M':		/* select output media */
	  media_name = optarg;
	  break;

	case 'n':		/* num copies */
	  num_copies = atoi (optarg);
	  break;

	case 'N':		/* newline character */
	  if (!(optarg[0] == 'n' || optarg[0] == 'r') || optarg[1] != '\0')
	    {
	      fprintf (stderr, "%s: illegal newline character specifier: \
'%s': expected 'n' or 'r'\n",
		       program, optarg);
	      goto option_error;
	    }
	  if (optarg[0] == 'n')
	    nl = '\n';
	  else
	    nl = '\r';
	  break;

	case 'o':
	case 'p':		/* output file */
	  /* Check output file "-". */
	  if (strcmp (optarg, "-") == 0)
	    output_file = OUTPUT_FILE_STDOUT;
	  else
	    output_file = optarg;
	  break;

	case 'O':		/* list missing characters */
	  list_missing_characters = 1;
	  break;

	case 'q':		/* quiet */
	  quiet = 1;
	  verbose = 0;
	  break;

	case 'r':		/* landscape */
	  landscape = 1;
	  break;

	case 'R':		/* portrait */
	  landscape = 0;
	  break;

	case 's':		/* baselineskip */
	  baselineskip = atof (optarg);
	  break;

	case 'S':		/* statusdict */
	  parse_key_value_pair (statusdict, optarg);
	  break;

	case 't':
	  title = optarg;
	  break;

	case 'T':		/* tabulator size */
	  tab_size = atoi (optarg);
	  if (tab_size <= 0)
	    tab_size = 1;
	  break;

	case 'u':		/* underlay */
	  underlay = optarg;
	  break;

	case 'v':		/* verbose */
	  if (optarg)
	    verbose = atoi (optarg);
	  else
	    verbose++;
	  quiet = 0;
	  break;

	case 'V':		/* version */
	  version ();
	  exit (0);
	  break;

	case 'X':		/* input encoding */
	  encoding_name = optarg;
	  break;

	case 'Z':		/* pass through PostScript files */
	  pass_through_ps_files = 1;
	  break;

	case 128:		/* underlay font */
	  if (!parse_font_spec (optarg, &ul_font, &ul_ptsize))
	    fatal ("malformed font spec: %s", optarg);
	  break;

	case 129:		/* underlay gray */
	  ul_gray = atof (optarg);
	  break;

	case 130:		/* page label format */
	  page_label_format = optarg;
	  break;

	case 131:		/* download font */
	  strhash_put (download_fonts, optarg, strlen (optarg) + 1, NULL,
		       NULL);
	  break;

	case '?':		/* Errors found during getopt_long(). */
	option_error:
	  usage ();
	  exit (1);
	  break;
	  
	default:
	  printf ("Hey!  main() didn't handle option \"%c\" (%d)", c, c);
	  if (optarg)
	    printf (" with arg %s", optarg);
	  printf ("\n");
	  fatal ("This is a bug!");
	  break;
	}
    }
}


static void
usage ()
{
  version ();
  printf ("Usage: %s [options] [file] ...\n\
Options:\n\
  -1, -2, --columns=NUM		specify the number of columns per page\n\
  -b HEADER, --page-header=HEADER\n\
				set page header\n\
  -B, --no-header		no page headers\n\
  -c, --truncate-lines		cut long lines (default is to wrap)\n\
  -C, --line-numbers		precede each line with its line number\n\
  -d QUEUE			print output to queue QUEUE\n\
  -D KEY[:VALUE], --setpagedevice=KEY[:VALUE]\n\
				pass a page device definition to output\n\
  -e, --special-escapes		enable special escapes interpretation\n\
  -f NAME, --font=NAME		use font NAME for body text\n\
  -F NAME, --header-font=NAME	use font NAME for header texts\n\
  -g, --print-anyway		nothing (compatibility option)\n\
  -G, --fancy-header[=NAME]	select fancy page header\n\
  -h, --help			print this help and exit\n\
  -i NUM, --indent=NUM		set line indent to NUM\n\
  -l, --lineprinter		simulate lineprinter:\n\
			  	  --lines-per-page=65, --no-header,\n\
				  --portrait, --columns=1\n\
  -L NUM, --lines-per-page=NUM	specify how many lines are printed on\n\
				each page\n\
  -m, --mail			send mail upon completion\n\
  -M NAME, --media=NAME  	use output media NAME\n\
  -n NUM, --copies=NUM		number of copies\n\
  -N NL, --newline=NL		select the newline character.  Possible\n\
 				values for NL are: n ('\\n') and r ('\\r').\n\
  -o FILE			an alias for option '-p', '--output-file'\n\
  -O, --list-missing-characters	list missing character codes\n\
  -p FILE, --output-file=FILE	leave output to file FILE.  If FILE\n\
				is \"-\", leave output to stdout.\n\
  -P QUEUE, --queue=QUEUE	print output to queue QUEUE\n\
  -q, --quiet			be really quiet\n\
  -r, --landscape		print in landscape mode\n\
  -R, --portrait		print in portrait mode\n\
  -s NUM, --baselineskip=NUM	set baselineskip to NUM\n\
  -S KEY[:VALUE], --statusdict=KEY[:VALUE]\n\
				pass a statusdict definition to output\n\
  -t TITLE, --title=TITLE	set title that is printed on every page\n\
  -T NUM, --tab-size=NUM	set tabulator size to NUM\n\
  -u TEXT, --underlay=TEXT	print TEXT under every page\n\
  -v, --verbose			tell what we are doing\n\
  -V, --version			print the version number of %s and exit\n\
  -X NAME, --encoding=NAME	use input encoding NAME\n\
  -Z, --pass-through-ps-files	pass through PostScript files without any\n\
				modifications\n\
  --download-font=NAME		download font NAME\n\
  --list-media			list names of all known media and exit\n\
  --list-options		list all options and their values\n\
  --page-label-format=FORMAT	set page label format to FORMAT\n\
  --ul-font=NAME		print underlays with font NAME\n\
  --ul-gray=NUM			print underlays with gray value NUM (0 - 1)\n",
	  program, program);
}


static void
version ()
{
  printf ("genscript version %d.%d (%s)\n", MAJOR_VERSION, MINOR_VERSION,
	  EDITION_STRING);
}
