
/*
 * bltUtil.c --
 *
 *	This module implements utility procedures for the BLT
 *	toolkit.
 *
 * Copyright 1991-1997 Bell Labs Innovations for Lucent Technologies.
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted, provided
 * that the above copyright notice appear in all copies and that both that the
 * copyright notice and warranty disclaimer appear in supporting documentation,
 * and that the names of Lucent Technologies any of their entities not be used
 * in advertising or publicity pertaining to distribution of the software
 * without specific, written prior permission.
 *
 * Lucent Technologies disclaims all warranties with regard to this software,
 * including all implied warranties of merchantability and fitness.  In no event
 * shall Lucent Technologies be liable for any special, indirect or
 * consequential damages or any damages whatsoever resulting from loss of use,
 * data or profits, whether in an action of contract, negligence or other
 * tortuous action, arising out of or in connection with the use or performance
 * of this software.  
 */

#include "bltInt.h"
#include <ctype.h>
#include <X11/Xutil.h>
#include <X11/Xproto.h>
#if defined(__STDC__)
#include <stdarg.h>
#else
#include <varargs.h>
#endif

static int FillParse _ANSI_ARGS_((ClientData clientData,
	Tcl_Interp *interp, Tk_Window tkwin, char *value, char *widgRec,
	int flags));
static char *FillPrint _ANSI_ARGS_((ClientData, Tk_Window, char *, int,
	Tcl_FreeProc **));

Tk_CustomOption bltFillOption =
{
    FillParse, FillPrint, (ClientData)0
};

static int PadParse _ANSI_ARGS_((ClientData clientData,
	Tcl_Interp *interp, Tk_Window tkwin, char *value, char *widgRec,
	int offset));
static char *PadPrint _ANSI_ARGS_((ClientData clientData, Tk_Window tkwin,
	char *widgRec, int offset, Tcl_FreeProc **freeProcPtr));

Tk_CustomOption bltPadOption =
{
    PadParse, PadPrint, (ClientData)0
};

static int LengthParse _ANSI_ARGS_((ClientData clientData,
	Tcl_Interp *interp, Tk_Window tkwin, char *value, char *widgRec,
	int flags));
static char *LengthPrint _ANSI_ARGS_((ClientData, Tk_Window, char *, int,
	Tcl_FreeProc **));

Tk_CustomOption bltLengthOption =
{
    LengthParse, LengthPrint, (ClientData)0
};


static int DashesParse _ANSI_ARGS_((ClientData, Tcl_Interp *, Tk_Window,
	char *, char *, int));
static char *DashesPrint _ANSI_ARGS_((ClientData, Tk_Window, char *, int,
	Tcl_FreeProc **));

Tk_CustomOption bltDashesOption =
{
    DashesParse, DashesPrint, (ClientData)0
};


/*
 *----------------------------------------------------------------------
 *
 * Blt_NameOfFill --
 *
 *	Converts the integer representing the fill style into a string.
 *
 *----------------------------------------------------------------------
 */
char *
Blt_NameOfFill(fill)
    Fill fill;
{
    switch (fill) {
    case FILL_X:
	return "x";
    case FILL_Y:
	return "y";
    case FILL_NONE:
	return "none";
    case FILL_BOTH:
	return "both";
    default:
	return "unknown value";
    }
}

/*
 *----------------------------------------------------------------------
 *
 * FillParse --
 *
 *	Converts the fill style string into its numeric representation.
 *
 *	Valid style strings are:
 *
 *	  "none"   Use neither plane.
 * 	  "x"	   X-coordinate plane.
 *	  "y"	   Y-coordinate plane.
 *	  "both"   Use both coordinate planes.
 *
 *----------------------------------------------------------------------
 */
/*ARGSUSED*/
static int
FillParse(clientData, interp, tkwin, value, widgRec, offset)
    ClientData clientData;	/* not used */
    Tcl_Interp *interp;		/* Interpreter to send results back to */
    Tk_Window tkwin;		/* not used */
    char *value;		/* Fill style string */
    char *widgRec;		/* Cubicle structure record */
    int offset;			/* Offset of style in record */
{
    Fill *fillPtr = (Fill *)(widgRec + offset);
    unsigned int length;
    char c;

    c = value[0];
    length = strlen(value);
    if ((c == 'n') && (strncmp(value, "none", length) == 0)) {
	*fillPtr = FILL_NONE;
    } else if ((c == 'x') && (strncmp(value, "x", length) == 0)) {
	*fillPtr = FILL_X;
    } else if ((c == 'y') && (strncmp(value, "y", length) == 0)) {
	*fillPtr = FILL_Y;
    } else if ((c == 'b') && (strncmp(value, "both", length) == 0)) {
	*fillPtr = FILL_BOTH;
    } else {
	Tcl_AppendResult(interp, "bad argument \"", value,
	    "\": should be \"none\", \"x\", \"y\", or \"both\"", (char *)NULL);
	return TCL_ERROR;
    }
    return (TCL_OK);
}

/*
 *----------------------------------------------------------------------
 *
 * FillPrint --
 *
 *	Returns the fill style string based upon the fill flags.
 *
 * Results:
 *	The fill style string is returned.
 *
 *----------------------------------------------------------------------
 */
/*ARGSUSED*/
static char *
FillPrint(clientData, tkwin, widgRec, offset, freeProcPtr)
    ClientData clientData;	/* not used */
    Tk_Window tkwin;		/* not used */
    char *widgRec;		/* Widget structure record */
    int offset;			/* Offset of fill in widget record */
    Tcl_FreeProc **freeProcPtr;	/* not used */
{
    Fill fill = *(Fill *)(widgRec + offset);

    return (Blt_NameOfFill(fill));
}

#ifdef notdef
/*
 *----------------------------------------------------------------------
 *
 * Blt_FlagParse --
 *
 *	Converts the fill style string into its numeric representation.
 *
 *----------------------------------------------------------------------
 */
/*ARGSUSED*/
int
Blt_FlagParse(clientData, interp, tkwin, value, widgRec, offset)
    ClientData clientData;	/* Bit mask to be tested in status word */
    Tcl_Interp *interp;		/* Interpreter to send results back to */
    Tk_Window tkwin;		/* not used */
    char *value;		/* Fill style string */
    char *widgRec;		/* Cubicle structure record */
    int offset;			/* Offset of style in record */
{
    unsigned int mask = (unsigned int)clientData;	/* Bit to be tested */
    int *flagPtr = (int *)(widgRec + offset);
    int bool;

    if (Tcl_GetBoolean(interp, value, &bool) != TCL_OK) {
	return TCL_ERROR;
    }
    if (bool) {
	*flagPtr |= mask;
    } else {
	*flagPtr &= ~mask;
    }
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * Blt_FlagPrint --
 *
 *	Returns the fill style string based upon the fill flags.
 *
 * Results:
 *	The fill style string is returned.
 *
 *----------------------------------------------------------------------
 */
/*ARGSUSED*/
char *
Blt_FlagPrint(clientData, tkwin, widgRec, offset, freeProcPtr)
    ClientData clientData;	/* Bit mask to be test in status word */
    Tk_Window tkwin;		/* not used */
    char *widgRec;		/* Widget structure record */
    int offset;			/* Offset of fill in widget record */
    Tcl_FreeProc **freeProcPtr;	/* Unused */
{
    unsigned int mask = (unsigned int)clientData;	/* Bit to be tested */
    unsigned int bool = *(unsigned int *)(widgRec + offset);

    return (bool & mask) ? "1" : "0";
}
#endif

/*
 *----------------------------------------------------------------------
 *
 * Blt_GetLength --
 *
 *	Like Tk_GetPixels, but doesn't allow negative pixel values.
 *
 * Results:
 *	A standard Tcl result.
 *
 *----------------------------------------------------------------------
 */
/*ARGSUSED*/
int
Blt_GetLength(interp, tkwin, string, valuePtr)
    Tcl_Interp *interp;
    Tk_Window tkwin;
    char *string;
    int *valuePtr;
{
    int length;

    if (Tk_GetPixels(interp, tkwin, string, &length) != TCL_OK) {
	return TCL_ERROR;
    }
    if (length < 0) {
	Tcl_AppendResult(interp, "can't have negative screen distance \"", string,
	    "\"", (char *)NULL);
	return TCL_ERROR;
    }
    *valuePtr = length;
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * LengthParse --
 *
 *	Like TK_CONFIG_PIXELS, but adds an extra check for negative
 *	values.
 *
 *----------------------------------------------------------------------
 */
/*ARGSUSED*/
static int
LengthParse(clientData, interp, tkwin, value, widgRec, offset)
    ClientData clientData;	/* not used */
    Tcl_Interp *interp;		/* Interpreter to send results back to */
    Tk_Window tkwin;		/* Window */
    char *value;		/* Pixel value string */
    char *widgRec;		/* Widget record */
    int offset;			/* Offset of pixel size in record */
{
    int *lengthPtr = (int *)(widgRec + offset);

    return (Blt_GetLength(interp, tkwin, value, lengthPtr));
}

/*
 *----------------------------------------------------------------------
 *
 * LengthPrint --
 *
 *	Returns the string representing the positive pixel size.
 *
 * Results:
 *	The pixel size string is returned.
 *
 *----------------------------------------------------------------------
 */
/*ARGSUSED*/
static char *
LengthPrint(clientData, tkwin, widgRec, offset, freeProcPtr)
    ClientData clientData;	/* not used */
    Tk_Window tkwin;		/* not used */
    char *widgRec;		/* Widget structure record */
    int offset;			/* Offset in widget record */
    Tcl_FreeProc **freeProcPtr;	/* not used */
{
    int length = *(int *)(widgRec + offset);
    char *result;
    char string[200];

    sprintf(string, "%d", length);
    result = strdup(string);
    assert(result);
    *freeProcPtr = (Tcl_FreeProc *)free;
    return (result);
}

/*
 *----------------------------------------------------------------------
 *
 * PadParse --
 *
 *	Convert a string into two pad values.  The string may be in one of
 *	the following forms:
 *
 *	    n    - n is a non-negative integer. This sets both
 *		   pad values to n.
 *	  {n m}  - both n and m are non-negative integers. side1
 *		   is set to n, side2 is set to m.
 *
 * Results:
 *	If the string is successfully converted, TCL_OK is returned.
 *	Otherwise, TCL_ERROR is returned and an error message is left in
 *	interp->result.
 *
 * Side Effects:
 *	The padding structure passed is updated with the new values.
 *
 *----------------------------------------------------------------------
 */
/*ARGSUSED*/
static int
PadParse(clientData, interp, tkwin, value, widgRec, offset)
    ClientData clientData;	/* not used */
    Tcl_Interp *interp;		/* Interpreter to send results back to */
    Tk_Window tkwin;		/* Window */
    char *value;		/* Pixel value string */
    char *widgRec;		/* Widget record */
    int offset;			/* Offset of pad in widget */
{
    Pad *padPtr = (Pad *)(widgRec + offset);
    int numElem;
    int pad, result;
    char **padArr;

    if (Tcl_SplitList(interp, value, &numElem, &padArr) != TCL_OK) {
	return TCL_ERROR;
    }
    result = TCL_ERROR;
    if ((numElem < 1) || (numElem > 2)) {
	Tcl_AppendResult(interp, "wrong # elements in padding list",
	    (char *)NULL);
	goto error;
    }
    if (Blt_GetLength(interp, tkwin, padArr[0], &pad) != TCL_OK) {
	goto error;
    }
    padPtr->side1 = pad;
    if ((numElem > 1) &&
	(Blt_GetLength(interp, tkwin, padArr[1], &pad) != TCL_OK)) {
	goto error;
    }
    padPtr->side2 = pad;
    result = TCL_OK;

  error:
    free((char *)padArr);
    return (result);
}

/*
 *----------------------------------------------------------------------
 *
 * PadPrint --
 *
 *	Converts the two pad values into a Tcl list.  Each pad has two
 *	pixel values.  For vertical pads, they represent the top and bottom
 *	margins.  For horizontal pads, they're the left and right margins.
 *	All pad values are non-negative integers.
 *
 * Results:
 *	The padding list is returned.
 *
 *----------------------------------------------------------------------
 */
/*ARGSUSED*/
static char *
PadPrint(clientData, tkwin, widgRec, offset, freeProcPtr)
    ClientData clientData;	/* not used */
    Tk_Window tkwin;		/* not used */
    char *widgRec;		/* Structure record */
    int offset;			/* Offset of pad in record */
    Tcl_FreeProc **freeProcPtr;	/* not used */
{
    Pad *padPtr = (Pad *)(widgRec + offset);
    char *result;
    char string[200];

    sprintf(string, "%d %d", padPtr->side1, padPtr->side2);
    result = strdup(string);
    if (result == NULL) {
	return "out of memory";
    }
    *freeProcPtr = (Tcl_FreeProc *)free;
    return (result);
}

/*
 *----------------------------------------------------------------------
 *
 * GetDashes --
 *
 *	Converts a Tcl list of dash values into a dash list ready for
 *	use with XSetDashes.
 *
 * 	A valid list dash values can have zero through 11 elements
 *	(PostScript limit).  Values must be between 1 and 255. Although
 *	a list of 0 (like the empty string) means no dashes.
 *
 * Results:
 *	A standard Tcl result. If the list represented a valid dash
 *	list TCL_OK is returned and *dashesPtr* will contain the
 *	valid dash list. Otherwise, TCL_ERROR is returned and
 *	interp->result will contain an error message.
 *
 *
 *----------------------------------------------------------------------
 */
static int
GetDashes(interp, string, dashesPtr)
    Tcl_Interp *interp;
    char *string;
    Dashes *dashesPtr;
{
    int numValues;
    char **strArr;
    long int value;
    register int i;

    numValues = 0;
    if ((string == NULL) || (*string == '\0')) {
	dashesPtr->numValues = 0;
	return TCL_OK;
    }
    if (Tcl_SplitList(interp, string, &numValues, &strArr) != TCL_OK) {
	return TCL_ERROR;
    }
    if (numValues > 11) {	/* This is the postscript limit */
	Tcl_AppendResult(interp, "too many values in dash list", (char *)NULL);
	goto badDashList;
    }
    for (i = 0; i < numValues; i++) {
	if (Blt_ExprLong(interp, strArr[i], &value) != TCL_OK) {
	    goto badDashList;
	}
	if ((value < 1) || (value > 255)) {
	    /* Backward compatibility: Allow list of 0 to turn off dashes */
	    if ((value == 0) && (numValues == 1)) {
		break;
	    }
	    Tcl_AppendResult(interp, "dash value \"", strArr[i],
		"\" is out of range", (char *)NULL);
	    goto badDashList;
	}
	dashesPtr->valueArr[i] = (unsigned char)value;
    }
    dashesPtr->valueArr[i] = '\0';
    dashesPtr->numValues = i;
    if (numValues > 0) {
	free((char *)strArr);
    }
    return TCL_OK;

  badDashList:
    if (numValues > 0) {
	free((char *)strArr);
    }
    return TCL_ERROR;
}

/*
 *----------------------------------------------------------------------
 *
 * DashesParse --
 *
 *	Convert the list of dash values into a dashes array.
 *
 * Results:
 *	The return value is a standard Tcl result.
 *
 * Side Effects:
 *	The Dashes structure is updated with the new dash list.
 *
 *----------------------------------------------------------------------
 */
/*ARGSUSED*/
static int
DashesParse(clientData, interp, tkwin, value, widgRec, offset)
    ClientData clientData;	/* not used */
    Tcl_Interp *interp;		/* Interpreter to send results back to */
    Tk_Window tkwin;		/* not used */
    char *value;		/* New dash value list */
    char *widgRec;		/* Widget record */
    int offset;			/* offset to Dashes structure */
{
    Dashes *dashesPtr = (Dashes *)(widgRec + offset);

    return (GetDashes(interp, value, dashesPtr));
}

/*
 *----------------------------------------------------------------------
 *
 * DashesPrint --
 *
 *	Convert the dashes array into a list of values.
 *
 * Results:
 *	The string representing the dashes returned.
 *
 *----------------------------------------------------------------------
 */
/*ARGSUSED*/
static char *
DashesPrint(clientData, tkwin, widgRec, offset, freeProcPtr)
    ClientData clientData;	/* not used */
    Tk_Window tkwin;		/* not used */
    char *widgRec;		/* Widget record */
    int offset;			/* offset of Dashes in record */
    Tcl_FreeProc **freeProcPtr;	/* Memory deallocation scheme to use */
{
    Dashes *dashesPtr = (Dashes *)(widgRec + offset);
    Tcl_DString dStr;
    register int i;
    char *result;
    char string[200];

    if (dashesPtr->numValues == 0) {
	return "";
    }
    Tcl_DStringInit(&dStr);
    for (i = 0; i < dashesPtr->numValues; i++) {
	sprintf(string, "%d", (int)dashesPtr->valueArr[i]);
	Tcl_DStringAppendElement(&dStr, string);
    }
    result = Tcl_DStringValue(&dStr);
    if (result == dStr.staticSpace) {
	result = strdup(result);
    } 
    *freeProcPtr = (Tcl_FreeProc *)free;
    return (result);
}

#ifndef NDEBUG
    
void
Blt_Assert(testExpr, fileName, lineNumber)
    char *testExpr;
    char *fileName;
    int lineNumber;
{
    fprintf(stderr, "line %d of %s: Assert \"%s\" failed",
	    lineNumber, fileName, testExpr);
    abort();
}

#endif


/*
 *----------------------------------------------------------------------
 *
 * Blt_ConfigModified --
 *
 *      Given the configuration specifications and one or more option 
 *	patterns (terminated by a NULL), indicate if any of the matching
 *	configuration options has been reset.
 *
 * Results:
 *      Returns 1 if one of the options has changed, 0 otherwise.
 *
 *----------------------------------------------------------------------
 */
int
Blt_ConfigModified TCL_VARARGS_DEF(Tk_ConfigSpec *, arg1)
{
    va_list argList;
    Tk_ConfigSpec *specs;
    register Tk_ConfigSpec *specPtr;
    register char *option;

    specs = TCL_VARARGS_START(Tk_ConfigSpec *, arg1, argList);
    for (;;) {
	option = va_arg(argList, char *);
	if (option == NULL) {
	    break;
	}
	for (specPtr = specs; specPtr->type != TK_CONFIG_END; specPtr++) {
	    if ((Tcl_StringMatch(specPtr->argvName, option)) &&
		(specPtr->specFlags & TK_CONFIG_OPTION_SPECIFIED)) {
		va_end(argList);
		return 1;
	    }
	}
    }
    va_end(argList);
    return 0;
}

/*
 *----------------------------------------------------------------------
 *
 * OpSearch --
 *
 *      Performs a binary search on the array of command operation
 *      specifications to find a partial, anchored match for the
 *      given operation string.
 *
 * Results:
 *	If the string matches unambiguously the index of the specification
 *	in the array is returned.  If the string does not match, even
 *	as an abbreviation, any operation, -1 is returned.  If the string
 *	matches, but ambiguously -2 is returned.
 *
 *----------------------------------------------------------------------
 */
static int
OpSearch(specArr, numSpecs, operation)
    Blt_OpSpec specArr[];
    int numSpecs;
    char *operation;		/* Name of minor operation to search for */
{
    Blt_OpSpec *specPtr;
    char c;
    register int high, low, median;
    register int compare, length;

    low = 0;
    high = numSpecs - 1;
    c = operation[0];
    length = strlen(operation);
    while (low <= high) {
	median = (low + high) >> 1;
	specPtr = specArr + median;

	/* Test the first character */
	compare = c - specPtr->name[0];
	if (!compare) {
	    /* Now test the entire string */
	    compare = strncmp(operation, specPtr->name, length);
	    if ((compare == 0) && (length < specPtr->minChars)) {
		return -2;	/* Ambiguous operation name */
	    }
	}
	if (compare < 0) {
	    high = median - 1;
	} else if (compare > 0) {
	    low = median + 1;
	} else {
	    return (median);	/* Op found. */
	}
    }
    return -1;			/* Can't find operation */
}

/*
 *----------------------------------------------------------------------
 *
 * Blt_LookupOperation --
 *
 *      Find the command operation given a string name.  This is useful
 *      where a group of command operations have the same argument
 *      signature.
 *
 * Results:
 *      If found, a pointer to the procedure (function pointer) is
 *      returned.  Otherwise NULL is returned and an error message
 *      containing a list of the possible commands is returned in
 *      interp->result.
 *
 *----------------------------------------------------------------------
 */
Blt_Operation
Blt_LookupOperation(interp, numSpecs, specArr, argIndex, numArgs, argArr)
    Tcl_Interp *interp;		/* Interpreter to report errors to */
    int numSpecs;		/* Number of specifications in array */
    Blt_OpSpec specArr[];	/* Operation specification array */
    Blt_OpIndex argIndex;	/* Index of the operation name argument */
    int numArgs;		/* Number of arguments in the argument vector.
				 * This includes any prefixed arguments */
    char *argArr[];		/* Argument vector */
{
    Blt_OpSpec *specPtr;
    char *string;
    register int i;
    register int specIndex;

    if (numArgs <= argIndex) {	/* No operation argument */
	Tcl_AppendResult(interp, "wrong # args: ", (char *)NULL);
      usage:
	Tcl_AppendResult(interp, "should be one of...", (char *)NULL);
	for (specIndex = 0; specIndex < numSpecs; specIndex++) {
	    Tcl_AppendResult(interp, "\n  ", (char *)NULL);
	    for (i = 0; i < argIndex; i++) {
		Tcl_AppendResult(interp, argArr[i], " ", (char *)NULL);
	    }
	    specPtr = specArr + specIndex;
	    Tcl_AppendResult(interp, specPtr->name, " ", specPtr->usage,
		(char *)NULL);
	}
	return NULL;
    }
    string = argArr[argIndex];
    specIndex = OpSearch(specArr, numSpecs, string);

    if (specIndex == -2) {
	char c;
	int length;

	Tcl_AppendResult(interp, "ambiguous", (char *)NULL);
	if (argIndex > 2) {
	    Tcl_AppendResult(interp, " ", argArr[argIndex - 1], (char *)NULL);
	}
	Tcl_AppendResult(interp, " operation \"", string, "\" matches:",
	    (char *)NULL);

	c = string[0];
	length = strlen(string);
	for (specIndex = 0; specIndex < numSpecs; specIndex++) {
	    specPtr = specArr + specIndex;
	    if ((c == specPtr->name[0]) &&
		(strncmp(string, specPtr->name, length) == 0)) {
		Tcl_AppendResult(interp, " ", specPtr->name, (char *)NULL);
	    }
	}
	return NULL;

    } else if (specIndex == -1) {	/* Can't find operation, display help  */
	Tcl_AppendResult(interp, "bad", (char *)NULL);
	if (argIndex > 2) {
	    Tcl_AppendResult(interp, " ", argArr[argIndex - 1], (char *)NULL);
	}
	Tcl_AppendResult(interp, " operation \"", string, "\": ", (char *)NULL);
	goto usage;
    }
    specPtr = specArr + specIndex;
    if ((numArgs < specPtr->minArgs) || ((specPtr->maxArgs > 0) &&
	    (numArgs > specPtr->maxArgs))) {
	Tcl_AppendResult(interp, "wrong # args: should be \"", (char *)NULL);
	for (i = 0; i < argIndex; i++) {
	    Tcl_AppendResult(interp, argArr[i], " ", (char *)NULL);
	}
	Tcl_AppendResult(interp, specPtr->name, " ", specPtr->usage, "\"",
	    (char *)NULL);
	return NULL;
    }
    return (specPtr->proc);
}

#ifndef HAVE_STRDUP
/*
 *----------------------------------------------------------------------
 *
 * strdup --
 *
 *      Create a copy of the string from heap storage.
 *
 * Results:
 *      Returns a pointer to the need string copy.
 *
 *----------------------------------------------------------------------
 */
char *
strdup(string)
    char *string;
{
    char *newPtr;

    newPtr = (char *)malloc(sizeof(char) * (strlen(string) + 1));
    if (newPtr != NULL) {
	strcpy(newPtr, string);
    }
    return (newPtr);
}

#endif /*HAVE_STRDUP*/

#ifdef notdef
#ifndef HAVE_STRCASECMP

static unsigned char lcase[] =
{
    '\000', '\001', '\002', '\003', '\004', '\005', '\006', '\007',
    '\010', '\011', '\012', '\013', '\014', '\015', '\016', '\017',
    '\020', '\021', '\022', '\023', '\024', '\025', '\026', '\027',
    '\030', '\031', '\032', '\033', '\034', '\035', '\036', '\037',
    '\040', '\041', '\042', '\043', '\044', '\045', '\046', '\047',
    '\050', '\051', '\052', '\053', '\054', '\055', '\056', '\057',
    '\060', '\061', '\062', '\063', '\064', '\065', '\066', '\067',
    '\070', '\071', '\072', '\073', '\074', '\075', '\076', '\077',
    '\100', '\141', '\142', '\143', '\144', '\145', '\146', '\147',
    '\150', '\151', '\152', '\153', '\154', '\155', '\156', '\157',
    '\160', '\161', '\162', '\163', '\164', '\165', '\166', '\167',
    '\170', '\171', '\172', '\133', '\134', '\135', '\136', '\137',
    '\140', '\141', '\142', '\143', '\144', '\145', '\146', '\147',
    '\150', '\151', '\152', '\153', '\154', '\155', '\156', '\157',
    '\160', '\161', '\162', '\163', '\164', '\165', '\166', '\167',
    '\170', '\171', '\172', '\173', '\174', '\175', '\176', '\177',
    '\200', '\201', '\202', '\203', '\204', '\205', '\206', '\207',
    '\210', '\211', '\212', '\213', '\214', '\215', '\216', '\217',
    '\220', '\221', '\222', '\223', '\224', '\225', '\226', '\227',
    '\230', '\231', '\232', '\233', '\234', '\235', '\236', '\237',
    '\240', '\241', '\242', '\243', '\244', '\245', '\246', '\247',
    '\250', '\251', '\252', '\253', '\254', '\255', '\256', '\257',
    '\260', '\261', '\262', '\263', '\264', '\265', '\266', '\267',
    '\270', '\271', '\272', '\273', '\274', '\275', '\276', '\277',
    '\300', '\341', '\342', '\343', '\344', '\345', '\346', '\347',
    '\350', '\351', '\352', '\353', '\354', '\355', '\356', '\357',
    '\360', '\361', '\362', '\363', '\364', '\365', '\366', '\367',
    '\370', '\371', '\372', '\333', '\334', '\335', '\336', '\337',
    '\340', '\341', '\342', '\343', '\344', '\345', '\346', '\347',
    '\350', '\351', '\352', '\353', '\354', '\355', '\356', '\357',
    '\360', '\361', '\362', '\363', '\364', '\365', '\366', '\367',
    '\370', '\371', '\372', '\373', '\374', '\375', '\376', '\377',
};

/*
 *----------------------------------------------------------------------
 *
 * strcasecmp --
 *
 *      Compare two strings, disregarding case.
 *
 * Results:
 *      Returns a signed integer representing the following:
 *
 *	zero      - two strings are equal
 *	negative  - first string is less than second
 *	positive  - first string is greater than second
 *
 *----------------------------------------------------------------------
 */
int
strcasecmp(str1, str2)
    CONST char *str1;
    CONST char *str2;
{
    unsigned char *s = (unsigned char *)str1;
    unsigned char *t = (unsigned char *)str2;

    for ( /* empty */ ; (lcase[*s] == lcase[*t]); s++, t++) {
	if (*s == '\0') {
	    return 0;
	}
    }
    return (lcase[*s] - lcase[*t]);
}

#ifdef notdef
/*
 *----------------------------------------------------------------------
 *
 * strncasecmp --
 *
 *      Compare two strings, disregarding case, up to a given length.
 *
 * Results:
 *      Returns a signed integer representing the following:
 *
 *	zero      - two strings are equal
 *	negative  - first string is less than second
 *	positive  - first string is greater than second
 *
 *----------------------------------------------------------------------
 */
int
strncasecmp(str1, str2, length)
    CONST char *str1;
    CONST char *str2;
    size_t length;
{
    register unsigned char *s = (unsigned char *)str1;
    register unsigned char *t = (unsigned char *)str2;

    for ( /* empty */ ; (length > 0); s++, t++, length--) {
	if (lcase[*s] != lcase[*t]) {
	    return (lcase[*s] - lcase[*t]);
	}
	if (*s == '\0') {
	    return 0;
	}
    }
    return 0;
}

#endif

#endif /*HAVE_STRCASECMP*/
#endif /* notdef */

/*
 *----------------------------------------------------------------------
 *
 * Blt_AppendDoubleElement --
 *
 *      Convenience routine to append a double precision value to
 * 	interp->result as a separate element.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      List representation of double precision number is appended
 *	to interp->result.
 *
 *----------------------------------------------------------------------
 */
void
Blt_AppendDoubleElement(interp, value)
    Tcl_Interp *interp;		/* Interpreter to append numeric string */
    double value;
{
    char string[TCL_DOUBLE_SPACE + 1];

    Tcl_PrintDouble(interp, value, string);
    Tcl_AppendElement(interp, string);
}

/*
 *----------------------------------------------------------------------
 *
 * Blt_AppendIntElement --
 *
 *      Convenience routine to append an integer value to interp->result
 *	as a separate list element.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      List representation of integer is appended to interp->result.
 *
 *----------------------------------------------------------------------
 */
void
Blt_AppendIntElement(interp, value)
    Tcl_Interp *interp;		/* Interpreter to append numeric string */
    int value;
{
    char string[200];

    sprintf(string, "%d", value);
    Tcl_AppendElement(interp, string);
}

#if (TCL_MAJOR_VERSION >= 8)
/*
 *----------------------------------------------------------------------
 *
 * Blt_ExprDouble --
 *
 *      Same functionality as pre-8.0 Tcl_ExprDouble.
 *
 * Results:
 *      A standard Tcl result.
 *
 * Side effects:
 *      The interpreter result is cleared.
 *
 *----------------------------------------------------------------------
 */
int
Blt_ExprDouble(interp, string, valuePtr)
    Tcl_Interp *interp;		/* Interpreter to append numeric string */
    char *string;
    double *valuePtr;
{
    if (Tcl_ExprDouble(interp, string, valuePtr) != TCL_OK) {
	return TCL_ERROR;
    }
    Tcl_ResetResult(interp);
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * Blt_ExprLong --
 *
 *      Same functionality as pre-8.0 Tcl_ExprLong.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      The interpreter result is cleared.
 *
 *----------------------------------------------------------------------
 */
int
Blt_ExprLong(interp, string, valuePtr)
    Tcl_Interp *interp;		/* Interpreter to append numeric string */
    char *string;
    long *valuePtr;
{
    if (Tcl_ExprLong(interp, string, valuePtr) != TCL_OK) {
	return TCL_ERROR;
    }
    Tcl_ResetResult(interp);
    return TCL_OK;
}
#endif


/*
 *----------------------------------------------------------------------
 *
 * Blt_GetPrivateGCFromDrawable --
 *
 *      Like Tk_GetGC, but doesn't share the GC with any other widget.
 *	This is needed because the certain GC parameters (like dashes)
 *	can not be set via XCreateGC, therefore there is no way for
 *	Tk's hashing mechanism to recognize that two such GCs differ.
 *
 * Results:
 *      A new GC is returned.
 *
 *----------------------------------------------------------------------
 */
GC
Blt_GetPrivateGCFromDrawable(tkwin, drawable, gcMask, valuePtr)
    Tk_Window tkwin;
    Drawable drawable;
    unsigned long gcMask;
    XGCValues *valuePtr;
{
    Pixmap pixmap;
    GC newGC;

    pixmap = None;
    if (drawable == None) {
	Drawable root;
	int depth;

	root = RootWindow(Tk_Display(tkwin), Tk_ScreenNumber(tkwin));
	depth = Tk_Depth(tkwin);

	if (depth == DefaultDepth(Tk_Display(tkwin), Tk_ScreenNumber(tkwin))) {
	    drawable = root;
	} else {
	    pixmap = Tk_GetPixmap(Tk_Display(tkwin), root, 1, 1, depth);
	    drawable = pixmap;
	}
    }
    newGC = XCreateGC(Tk_Display(tkwin), drawable, gcMask, valuePtr);
    if (pixmap != None) {
	Tk_FreePixmap(Tk_Display(tkwin), pixmap);
    }
    return (newGC);
}


/*
 *----------------------------------------------------------------------
 *
 * Blt_GetPrivateGC --
 *
 *      Like Tk_GetGC, but doesn't share the GC with any other widget.
 *	This is needed because the certain GC parameters (like dashes)
 *	can not be set via XCreateGC, therefore there is no way for
 *	Tk's hashing mechanism to recognize that two such GCs differ.
 *
 * Results:
 *      A new GC is returned.
 *
 *----------------------------------------------------------------------
 */
GC
Blt_GetPrivateGC(tkwin, gcMask, valuePtr)
    Tk_Window tkwin;
    unsigned long gcMask;
    XGCValues *valuePtr;
{
    return Blt_GetPrivateGCFromDrawable(tkwin, Tk_WindowId(tkwin), gcMask, 
	valuePtr);
}

void
Blt_FreePrivateGC(display, gc)
    Display *display;
    GC gc;
{
    Tk_FreeXId(display, (XID) XGContextFromGC(gc));
    XFreeGC(display, gc);
}

/* ARGSUSED */
static int
XGetImageErrorProc(clientData, errEventPtr)
    ClientData clientData;
    XErrorEvent *errEventPtr;
{
    int *errorPtr = (int *)clientData;

    *errorPtr = TCL_ERROR;
    return 0;
}

int
Blt_GetSnapshot(tkwin, drawable, width, height, colorTablePtr, colorPtrPtr, 
	dataPtrPtr)
    Tk_Window tkwin;
    Drawable drawable;
    int width, height;
    Tcl_HashTable *colorTablePtr;
    XColor **colorPtrPtr;
    unsigned long **dataPtrPtr;
{
    XImage *imagePtr;
    XColor *colorPtr, *colorArr;
    unsigned long *dataPtr, *dataArr;
    Tcl_HashEntry *hPtr;
    Tcl_HashSearch cursor;
    int numColors;
    unsigned long pixel;
    unsigned int numPixels;
    int x, y;
    int isNew;
    int result = TCL_OK;
    Tk_ErrorHandler errHandler;

    errHandler = Tk_CreateErrorHandler(Tk_Display(tkwin), BadMatch,
	X_GetImage, -1, XGetImageErrorProc, (ClientData)&result);
    imagePtr = XGetImage(Tk_Display(tkwin), drawable, 0, 0, width, height,
	AllPlanes, ZPixmap);
    Tk_DeleteErrorHandler(errHandler);
    XSync(Tk_Display(tkwin), False);
    if (result != TCL_OK) {
	return -1;
    }

    /*
     * First allocate an array of pixels big enough to hold the image.
     */
    numPixels = width * height;
    dataArr = (unsigned long *) malloc(sizeof(unsigned long) * numPixels);
    assert(dataArr);
    /*
     * Fill the array with each pixel of the image. At the same time, build 
     * up a hashtable of the pixels used.
     */

    Tcl_InitHashTable(colorTablePtr, TCL_ONE_WORD_KEYS);
    dataPtr = dataArr;
    for (y = 0; y < height; y++) {
	for (x = 0; x < width; x++) {
	    pixel = XGetPixel(imagePtr, x, y);
	    hPtr = Tcl_CreateHashEntry(colorTablePtr, (char *)pixel, &isNew);
	    if (isNew) {
		Tcl_SetHashValue(hPtr, (char *)pixel);
	    }
	    *dataPtr++  = pixel;
	}
    }
    XDestroyImage(imagePtr);

    /* 
     * Convert the hashtable of pixels into an array of XColors so that we 
     * can call XQueryColors with it. XQueryColors will convert the pixels 
     * into their RGB values.
     */
    numColors = Blt_ListGetLength(colorTablePtr);
    colorArr = (XColor *)malloc(sizeof(XColor) * numColors);

    colorPtr = colorArr;
    for (hPtr = Tcl_FirstHashEntry(colorTablePtr, &cursor); hPtr != NULL;
	hPtr = Tcl_NextHashEntry(&cursor)) {
	colorPtr->pixel = (unsigned long)Tcl_GetHashValue(hPtr);
	Tcl_SetHashValue(hPtr, (char *)colorPtr);
	colorPtr++;
    }
    XQueryColors(Tk_Display(tkwin), Tk_Colormap(tkwin), colorArr, numColors);

    *dataPtrPtr = dataArr;
    *colorPtrPtr = colorArr;
    return numPixels;
}

/*
 *----------------------------------------------------------------------
 *
 * Blt_SnapPhoto --
 *
 *      Takes a snapshot of an X drawable (pixmap or window) and 
 *	writes it to an existing Tk_Image. 
 *
 *	The trick here is to efficiently convert the pixel values
 *	(indices into the color table) into RGB color values.  In the
 *	days of 8-bit displays, it was simpler to get RGB values for
 *	all 256 indices into the colormap.  Instead we'll build a
 *	hashtable of unique pixels and from that an array of pixels to
 *	pass to XQueryColors.  
 *
 * Results:
 *      A standard Tcl result.
 *
 * Side Effects:
 *	The named Tk_Photo is updated with the XImage.
 *
 *----------------------------------------------------------------------
 */

int
Blt_SnapPhoto(interp, tkwin, drawable, width, height, photoName)
    Tcl_Interp *interp;
    Tk_Window tkwin;
    Drawable drawable;
    int width, height;
    char *photoName;
{
    Tk_PhotoHandle photo;	/* The photo image to write into. */
    Tk_PhotoImageBlock block;
    XColor *colorPtr, *colorArr;
    unsigned long *dataArr;
    PixelColor *pixelPtr, *pixelArr;
    Tcl_HashTable colorTable;
    Tcl_HashEntry *hPtr;
    register int i;
    int numPixels;

    photo = Tk_FindPhoto(photoName);
    if (photo == NULL) {
	Tcl_AppendResult(interp, "image \"", photoName, "\" doesn't",
	    " exist or is not a photo image", (char *)NULL);
	return TCL_ERROR;
    }

    numPixels = Blt_GetSnapshot(tkwin, drawable, width, height, &colorTable,
	&colorArr, &dataArr);
    if (numPixels < 0) {
	Tcl_AppendResult(interp, 
		"can't grab window or pixmap (possibly obscured?)", (char *)NULL);
	return TCL_ERROR;	/* Can't grab window image */
    }
    /*  
     * Go through the array of pixels, replacing each pixel of the
     * image with its RGB value.  
     */
    pixelPtr = pixelArr = (PixelColor *)malloc(sizeof(PixelColor) * numPixels);
    for (i = 0; i < numPixels; i++) {
	hPtr = Tcl_FindHashEntry(&colorTable, (char *)dataArr[i]);
	colorPtr = (XColor *)Tcl_GetHashValue(hPtr);
	pixelPtr->Red = (unsigned char)(colorPtr->red >> 8);
	pixelPtr->Green = (unsigned char)(colorPtr->green >> 8);
	pixelPtr->Blue = (unsigned char)(colorPtr->blue >> 8);
	pixelPtr++;
    }
    free((char *)dataArr);
    free((char *)colorArr);
    Tcl_DeleteHashTable(&colorTable);

    /* 
     * Finally, we're ready to call the Tk_Photo routines. They'll take
     * the RGB array we've processed to build the Tk image of the snapshot.
     */
    Tk_PhotoGetImage(photo, &block);
    Tk_PhotoExpand(photo, width, height);
    block.width = width;
    block.height = height;
    block.pixelSize = sizeof(PixelColor);
    block.pitch = sizeof(PixelColor) * width;
    block.offset[0] = Tk_Offset(RGB, red);
    block.offset[1] = Tk_Offset(RGB, green);
    block.offset[2] = Tk_Offset(RGB, blue);

    block.pixelPtr = (unsigned char *)pixelArr;
    Tk_PhotoPutBlock(photo, &block, 0, 0, width, height);
    free((char *)block.pixelPtr);

    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * Blt_FindChild --
 *
 *      Searches the parent for a named child window.
 *
 *	Oddly enough, the isn't in Tk's C API, but only in it's Tcl API.
 *	It's easy enough to do here, if you peek into the Tk_FakeWin
 *	structure.
 *
 * Results:
 *      The child Tk_Window. If the named child can't be found, NULL
 *	is returned.
 *
 *----------------------------------------------------------------------
 */

#define Tk_FirstChild(tkwin)	((Tk_Window)(((Tk_FakeWin *) (tkwin))->dummy2))
#define Tk_NextChild(tkwin)	((Tk_Window)(((Tk_FakeWin *) (tkwin))->dummy4))


/*LINTLIBRARY*/
Tk_Window
Blt_FindChild(parent, name)
    Tk_Window parent;
    char *name;
{
    register Tk_Window tkwin;

    for (tkwin = Tk_FirstChild(parent); tkwin != NULL; 
	 tkwin = Tk_NextChild(tkwin)) {
	if (strcmp(name, Tk_Name(tkwin)) == 0) {
	    return (tkwin);
	}
    }
    return NULL;
}
    
/*
 *----------------------------------------------------------------------
 *
 * Blt_FindChild --
 *
 *      Searches the parent for a named child window.
 *
 *	Oddly enough, the isn't in Tk's C API, but only in it's Tcl API.
 *	It's easy enough to do here, if you peek into the Tk_FakeWin
 *	structure.
 *
 * Results:
 *      The child Tk_Window. If the named child can't be found, NULL
 *	is returned.
 *
 *----------------------------------------------------------------------
 */

#define Tk_FirstChild(tkwin)	((Tk_Window)(((Tk_FakeWin *) (tkwin))->dummy2))
#define Tk_NextChild(tkwin)	((Tk_Window)(((Tk_FakeWin *) (tkwin))->dummy4))
    
/*
 *----------------------------------------------------------------------
 *
 * Blt_Toplevel --
 *
 *      Climbs up the widget hierarchy to find the top level window of
 *      the window given.
 *
 * Results:
 *      Returns the Tk_Window of the toplevel widget. 
 *
 *----------------------------------------------------------------------
 */
Tk_Window
Blt_Toplevel(tkwin)
    register Tk_Window tkwin;
{
    while (!Tk_IsTopLevel(tkwin)) {
	tkwin = Tk_Parent(tkwin);
    }
    return tkwin;
}

/*
 *----------------------------------------------------------------------
 *
 * Blt_XWindowId --
 *
 *      Returns the XID for the Tk_Window given.  Starting in Tk 8.0, 
 *	the toplevel widgets are wrapped by another window.  Currently
 *	There's no way to get at that window, other than what is done
 *	here.  Query the X window hierarchy and grab the parent.
 *
 * Results:
 *      Returns the X Window ID of the widget.  If it's a toplevel, then
 *	the XID of the wrapper is returned.
 *
 *----------------------------------------------------------------------
 */
Window
Blt_XWindowId(tkwin)
    Tk_Window tkwin;
{
    Window window;

    window = Tk_WindowId(tkwin);
#if (TK_MAJOR_VERSION >= 8)
    if (Tk_IsTopLevel(tkwin)) {
	Window root, parent;
	Window *childrenPtr;
	unsigned int numChildren;

	parent = None;
	if (XQueryTree(Tk_Display(tkwin), Tk_WindowId(tkwin), &root, &parent, 
		       &childrenPtr, &numChildren) > 0) {
	    XFree(childrenPtr);
	    window = parent;
	}
    }
#endif /* TK_MAJOR_VERSION >= 8 */
    return window;
}

/*
 *----------------------------------------------------------------------
 *
 * Blt_ConfigureWidgetComponent --
 *
 *      Configures a component of a widget.  This is useful for
 *	multiple components which aren't uniquely identified by
 *	an Tk_Window (not like a widget).
 *
 *	This is really a dumb hack to work around the limitations of
 *	the Tk resource database.  It creates a temporary window,
 *	needed to call Tk_ConfigureWidget, by the name of the
 *	component.  This lets us, for example, set resources for axes
 *	of the graph widget. The graph really has only one window, but
 *	its convenient to specify components in a hierarchy of options.
 *
 *		*graph.x.logScale yes
 *		*graph.Axis.logScale yes
 *		*graph.temperature.scaleSymbols yes
 *		*graph.Element.scaleSymbols yes
 *
 * Results:
 *      A standard Tcl result.
 *
 * Side Effects:
 *	Temporary window is generated merely to call with 
 *	Tk_ConfigureWidget.
 *
 *---------------------------------------------------------------------- */
int
Blt_ConfigureWidgetComponent(interp, parent, name, class, specsPtr, argc, argv, 
	widgRec, flags)
    Tcl_Interp *interp;
    Tk_Window parent;		/* Window to associate with component */
    char name[];		/* Name of component */
    char class[];
    Tk_ConfigSpec *specsPtr;
    int argc;
    char *argv[];
    char *widgRec;
    int flags;
{
    Tk_Window tkwin;
    int result;
    char *tempName;
    int temporary = 0;

    tempName = strdup(name);

    /* Window name can't start with an upper case letter */
    tempName[0] = tolower(name[0]);

    /* 
     * Create component if a child window by the component's name
     * doesn't already exist. 
     */
    tkwin = Blt_FindChild(parent, tempName);
    if (tkwin == NULL) {
	tkwin = Tk_CreateWindow(interp, parent, tempName, (char *)NULL);
	temporary = 1;
    }
    assert(Tk_Depth(tkwin) == Tk_Depth(parent));
    free(tempName);

    if (tkwin == NULL) {
	return TCL_ERROR;
    }
    Tk_SetClass(tkwin, class);
    result = Tk_ConfigureWidget(interp, tkwin, specsPtr, argc, argv, widgRec,
	flags);
    if (temporary) {
	Tk_DestroyWindow(tkwin);
    }
    return (result);
}



