/* Mapping tables for JIS0212 handling.
   Copyright (C) 1997 Free Software Foundation, Inc.
   This file is part of the GNU C Library.
   Contributed by Ulrich Drepper <drepper@cygnus.com>, 1997.

   The GNU C Library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   The GNU C Library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with the GNU C Library; see the file COPYING.LIB.  If not,
   write to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.  */

#include <inttypes.h>
#include <wchar.h>

#include <jis0212.h>

/* The defined JIS 0201 positions contains large gaps.  It's better to
   avoid using one large table with such big gaps.  Instead we match each
   char in a table resulting in an index to a table with the UCS.  The
   following command can be used to generate the table:

   egrep '^0x[[:xdigit:]][[:xdigit:]][[:xdigit:]][[:xdigit:]]' \
     .../eastasia/jis/jis0212.txt | perl tab.pl

   where tab.pl is

   ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
   $first=$last=$idx=0;
   while (<>) {
     local($jis,$ucs,%rest) = split;
     local($u)=hex($ucs);
     local($j)=hex($jis);
     local($k) = int(($j - 0x2121) / 256) * 94 + (($j - 0x2121) % 256);
     if ($k - $last > 6) {
       if ($last != 0) {
	 printf (" { start: %#06x, end: %#06x, idx: %5d },\n",
		 $first, $last, $idx);
	 $idx += $last - $first + 1;
       }
       $first=$k;
     }
     $last=$k;
   }
   printf (" { start: %#06x, end: %#06x, idx: %5d }\n",
	   $first, $last, $idx);
   ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
*/
const struct jisx0212_idx jisx0212_to_ucs_idx[] =
{
 { start: 0x006c, end: 0x0076, idx:     0 },
 { start: 0x007f, end: 0x0081, idx:    11 },
 { start: 0x00a8, end: 0x00ae, idx:    14 },
 { start: 0x0216, end: 0x0231, idx:    21 },
 { start: 0x0255, end: 0x0261, idx:    49 },
 { start: 0x0285, end: 0x0291, idx:    62 },
 { start: 0x02f0, end: 0x02ff, idx:    75 },
 { start: 0x0310, end: 0x031f, idx:    91 },
 { start: 0x034e, end: 0x03a4, idx:   107 },
 { start: 0x03ac, end: 0x0402, idx:   194 },
 { start: 0x0582, end: 0x1c2a, idx:   281 },
 { start: 0 },
};


/* The table which the UCS values indexed by the table above can be
   generated using the following code:

   egrep '^0x[[:xdigit:]][[:xdigit:]][[:xdigit:]][[:xdigit:]]' \
     .../eastasia/jis/jis0212.txt | perl tab.pl

   where tab.pl is:

   ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
   $first=$last=$idx=0;

   sub fmt {
     printf ("\n ") if (($n % 8) == 0);
     ++$n;
     my($val) = pop(@_);
     printf (" %#06x,", $val);
   }

   while (<>) {
     local($jis,$ucs,%rest) = split;
     local($u)=hex($ucs);
     local($j)=hex($jis);
     local($k) = int(($j - 0x2121) / 256) * 94 + (($j - 0x2121) % 256);
     if ($k - $last > 6) {
       if ($last != 0) {
	 $idx += $last - $first + 1;
       }
       $first=$k;
     } else {
       for ($m = $last + 1; $m < $k; ++$m) {
         fmt (0);
       }
     }
     fmt ($u);
     $last=$k;
   }
   ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
*/
const uint16_t jisx0212_to_ucs[] =
{
  0x02d8, 0x02c7, 0x00b8, 0x02d9, 0x02dd, 0x00af, 0x02db, 0x02da,
  0x007e, 0x0384, 0x0385, 0x00a1, 0x00a6, 0x00bf, 0x00ba, 0x00aa,
  0x00a9, 0x00ae, 0x2122, 0x00a4, 0x2116, 0x0386, 0x0388, 0x0389,
  0x038a, 0x03aa, 000000, 0x038c, 000000, 0x038e, 0x03ab, 000000,
  0x038f, 000000, 000000, 000000, 000000, 0x03ac, 0x03ad, 0x03ae,
  0x03af, 0x03ca, 0x0390, 0x03cc, 0x03c2, 0x03cd, 0x03cb, 0x03b0,
  0x03ce, 0x0402, 0x0403, 0x0404, 0x0405, 0x0406, 0x0407, 0x0408,
  0x0409, 0x040a, 0x040b, 0x040c, 0x040e, 0x040f, 0x0452, 0x0453,
  0x0454, 0x0455, 0x0456, 0x0457, 0x0458, 0x0459, 0x045a, 0x045b,
  0x045c, 0x045e, 0x045f, 0x00c6, 0x0110, 000000, 0x0126, 000000,
  0x0132, 000000, 0x0141, 0x013f, 000000, 0x014a, 0x00d8, 0x0152,
  000000, 0x0166, 0x00de, 0x00e6, 0x0111, 0x00f0, 0x0127, 0x0131,
  0x0133, 0x0138, 0x0142, 0x0140, 0x0149, 0x014b, 0x00f8, 0x0153,
  0x00df, 0x0167, 0x00fe, 0x00c1, 0x00c0, 0x00c4, 0x00c2, 0x0102,
  0x01cd, 0x0100, 0x0104, 0x00c5, 0x00c3, 0x0106, 0x0108, 0x010c,
  0x00c7, 0x010a, 0x010e, 0x00c9, 0x00c8, 0x00cb, 0x00ca, 0x011a,
  0x0116, 0x0112, 0x0118, 000000, 0x011c, 0x011e, 0x0122, 0x0120,
  0x0124, 0x00cd, 0x00cc, 0x00cf, 0x00ce, 0x01cf, 0x0130, 0x012a,
  0x012e, 0x0128, 0x0134, 0x0136, 0x0139, 0x013d, 0x013b, 0x0143,
  0x0147, 0x0145, 0x00d1, 0x00d3, 0x00d2, 0x00d6, 0x00d4, 0x01d1,
  0x0150, 0x014c, 0x00d5, 0x0154, 0x0158, 0x0156, 0x015a, 0x015c,
  0x0160, 0x015e, 0x0164, 0x0162, 0x00da, 0x00d9, 0x00dc, 0x00db,
  0x016c, 0x01d3, 0x0170, 0x016a, 0x0172, 0x016e, 0x0168, 0x01d7,
  0x01db, 0x01d9, 0x01d5, 0x0174, 0x00dd, 0x0178, 0x0176, 0x0179,
  0x017d, 0x017b, 0x00e1, 0x00e0, 0x00e4, 0x00e2, 0x0103, 0x01ce,
  0x0101, 0x0105, 0x00e5, 0x00e3, 0x0107, 0x0109, 0x010d, 0x00e7,
  0x010b, 0x010f, 0x00e9, 0x00e8, 0x00eb, 0x00ea, 0x011b, 0x0117,
  0x0113, 0x0119, 0x01f5, 0x011d, 0x011f, 000000, 0x0121, 0x0125,
  0x00ed, 0x00ec, 0x00ef, 0x00ee, 0x01d0, 000000, 0x012b, 0x012f,
  0x0129, 0x0135, 0x0137, 0x013a, 0x013e, 0x013c, 0x0144, 0x0148,
  0x0146, 0x00f1, 0x00f3, 0x00f2, 0x00f6, 0x00f4, 0x01d2, 0x0151,
  0x014d, 0x00f5, 0x0155, 0x0159, 0x0157, 0x015b, 0x015d, 0x0161,
  0x015f, 0x0165, 0x0163, 0x00fa, 0x00f9, 0x00fc, 0x00fb, 0x016d,
  0x01d4, 0x0171, 0x016b, 0x0173, 0x016f, 0x0169, 0x01d8, 0x01dc,
  0x01da, 0x01d6, 0x0175, 0x00fd, 0x00ff, 0x0177, 0x017a, 0x017e,
  0x017c, 0x4e02, 0x4e04, 0x4e05, 0x4e0c, 0x4e12, 0x4e1f, 0x4e23,
  0x4e24, 0x4e28, 0x4e2b, 0x4e2e, 0x4e2f, 0x4e30, 0x4e35, 0x4e40,
  0x4e41, 0x4e44, 0x4e47, 0x4e51, 0x4e5a, 0x4e5c, 0x4e63, 0x4e68,
  0x4e69, 0x4e74, 0x4e75, 0x4e79, 0x4e7f, 0x4e8d, 0x4e96, 0x4e97,
  0x4e9d, 0x4eaf, 0x4eb9, 0x4ec3, 0x4ed0, 0x4eda, 0x4edb, 0x4ee0,
  0x4ee1, 0x4ee2, 0x4ee8, 0x4eef, 0x4ef1, 0x4ef3, 0x4ef5, 0x4efd,
  0x4efe, 0x4eff, 0x4f00, 0x4f02, 0x4f03, 0x4f08, 0x4f0b, 0x4f0c,
  0x4f12, 0x4f15, 0x4f16, 0x4f17, 0x4f19, 0x4f2e, 0x4f31, 0x4f60,
  0x4f33, 0x4f35, 0x4f37, 0x4f39, 0x4f3b, 0x4f3e, 0x4f40, 0x4f42,
  0x4f48, 0x4f49, 0x4f4b, 0x4f4c, 0x4f52, 0x4f54, 0x4f56, 0x4f58,
  0x4f5f, 0x4f63, 0x4f6a, 0x4f6c, 0x4f6e, 0x4f71, 0x4f77, 0x4f78,
  0x4f79, 0x4f7a, 0x4f7d, 0x4f7e, 0x4f81, 0x4f82, 0x4f84, 0x4f85,
  0x4f89, 0x4f8a, 0x4f8c, 0x4f8e, 0x4f90, 0x4f92, 0x4f93, 0x4f94,
  0x4f97, 0x4f99, 0x4f9a, 0x4f9e, 0x4f9f, 0x4fb2, 0x4fb7, 0x4fb9,
  0x4fbb, 0x4fbc, 0x4fbd, 0x4fbe, 0x4fc0, 0x4fc1, 0x4fc5, 0x4fc6,
  0x4fc8, 0x4fc9, 0x4fcb, 0x4fcc, 0x4fcd, 0x4fcf, 0x4fd2, 0x4fdc,
  0x4fe0, 0x4fe2, 0x4ff0, 0x4ff2, 0x4ffc, 0x4ffd, 0x4fff, 0x5000,
  0x5001, 0x5004, 0x5007, 0x500a, 0x500c, 0x500e, 0x5010, 0x5013,
  0x5017, 0x5018, 0x501b, 0x501c, 0x501d, 0x501e, 0x5022, 0x5027,
  0x502e, 0x5030, 0x5032, 0x5033, 0x5035, 0x5040, 0x5041, 0x5042,
  0x5045, 0x5046, 0x504a, 0x504c, 0x504e, 0x5051, 0x5052, 0x5053,
  0x5057, 0x5059, 0x505f, 0x5060, 0x5062, 0x5063, 0x5066, 0x5067,
  0x506a, 0x506d, 0x5070, 0x5071, 0x503b, 0x5081, 0x5083, 0x5084,
  0x5086, 0x508a, 0x508e, 0x508f, 0x5090, 0x5092, 0x5093, 0x5094,
  0x5096, 0x509b, 0x509c, 0x509e, 0x509f, 0x50a0, 0x50a1, 0x50a2,
  0x50aa, 0x50af, 0x50b0, 0x50b9, 0x50ba, 0x50bd, 0x50c0, 0x50c3,
  0x50c4, 0x50c7, 0x50cc, 0x50ce, 0x50d0, 0x50d3, 0x50d4, 0x50d8,
  0x50dc, 0x50dd, 0x50df, 0x50e2, 0x50e4, 0x50e6, 0x50e8, 0x50e9,
  0x50ef, 0x50f1, 0x50f6, 0x50fa, 0x50fe, 0x5103, 0x5106, 0x5107,
  0x5108, 0x510b, 0x510c, 0x510d, 0x510e, 0x50f2, 0x5110, 0x5117,
  0x5119, 0x511b, 0x511c, 0x511d, 0x511e, 0x5123, 0x5127, 0x5128,
  0x512c, 0x512d, 0x512f, 0x5131, 0x5133, 0x5134, 0x5135, 0x5138,
  0x5139, 0x5142, 0x514a, 0x514f, 0x5153, 0x5155, 0x5157, 0x5158,
  0x515f, 0x5164, 0x5166, 0x517e, 0x5183, 0x5184, 0x518b, 0x518e,
  0x5198, 0x519d, 0x51a1, 0x51a3, 0x51ad, 0x51b8, 0x51ba, 0x51bc,
  0x51be, 0x51bf, 0x51c2, 0x51c8, 0x51cf, 0x51d1, 0x51d2, 0x51d3,
  0x51d5, 0x51d8, 0x51de, 0x51e2, 0x51e5, 0x51ee, 0x51f2, 0x51f3,
  0x51f4, 0x51f7, 0x5201, 0x5202, 0x5205, 0x5212, 0x5213, 0x5215,
  0x5216, 0x5218, 0x5222, 0x5228, 0x5231, 0x5232, 0x5235, 0x523c,
  0x5245, 0x5249, 0x5255, 0x5257, 0x5258, 0x525a, 0x525c, 0x525f,
  0x5260, 0x5261, 0x5266, 0x526e, 0x5277, 0x5278, 0x5279, 0x5280,
  0x5282, 0x5285, 0x528a, 0x528c, 0x5293, 0x5295, 0x5296, 0x5297,
  0x5298, 0x529a, 0x529c, 0x52a4, 0x52a5, 0x52a6, 0x52a7, 0x52af,
  0x52b0, 0x52b6, 0x52b7, 0x52b8, 0x52ba, 0x52bb, 0x52bd, 0x52c0,
  0x52c4, 0x52c6, 0x52c8, 0x52cc, 0x52cf, 0x52d1, 0x52d4, 0x52d6,
  0x52db, 0x52dc, 0x52e1, 0x52e5, 0x52e8, 0x52e9, 0x52ea, 0x52ec,
  0x52f0, 0x52f1, 0x52f4, 0x52f6, 0x52f7, 0x5300, 0x5303, 0x530a,
  0x530b, 0x530c, 0x5311, 0x5313, 0x5318, 0x531b, 0x531c, 0x531e,
  0x531f, 0x5325, 0x5327, 0x5328, 0x5329, 0x532b, 0x532c, 0x532d,
  0x5330, 0x5332, 0x5335, 0x533c, 0x533d, 0x533e, 0x5342, 0x534c,
  0x534b, 0x5359, 0x535b, 0x5361, 0x5363, 0x5365, 0x536c, 0x536d,
  0x5372, 0x5379, 0x537e, 0x5383, 0x5387, 0x5388, 0x538e, 0x5393,
  0x5394, 0x5399, 0x539d, 0x53a1, 0x53a4, 0x53aa, 0x53ab, 0x53af,
  0x53b2, 0x53b4, 0x53b5, 0x53b7, 0x53b8, 0x53ba, 0x53bd, 0x53c0,
  0x53c5, 0x53cf, 0x53d2, 0x53d3, 0x53d5, 0x53da, 0x53dd, 0x53de,
  0x53e0, 0x53e6, 0x53e7, 0x53f5, 0x5402, 0x5413, 0x541a, 0x5421,
  0x5427, 0x5428, 0x542a, 0x542f, 0x5431, 0x5434, 0x5435, 0x5443,
  0x5444, 0x5447, 0x544d, 0x544f, 0x545e, 0x5462, 0x5464, 0x5466,
  0x5467, 0x5469, 0x546b, 0x546d, 0x546e, 0x5474, 0x547f, 0x5481,
  0x5483, 0x5485, 0x5488, 0x5489, 0x548d, 0x5491, 0x5495, 0x5496,
  0x549c, 0x549f, 0x54a1, 0x54a6, 0x54a7, 0x54a9, 0x54aa, 0x54ad,
  0x54ae, 0x54b1, 0x54b7, 0x54b9, 0x54ba, 0x54bb, 0x54bf, 0x54c6,
  0x54ca, 0x54cd, 0x54ce, 0x54e0, 0x54ea, 0x54ec, 0x54ef, 0x54f6,
  0x54fc, 0x54fe, 0x54ff, 0x5500, 0x5501, 0x5505, 0x5508, 0x5509,
  0x550c, 0x550d, 0x550e, 0x5515, 0x552a, 0x552b, 0x5532, 0x5535,
  0x5536, 0x553b, 0x553c, 0x553d, 0x5541, 0x5547, 0x5549, 0x554a,
  0x554d, 0x5550, 0x5551, 0x5558, 0x555a, 0x555b, 0x555e, 0x5560,
  0x5561, 0x5564, 0x5566, 0x557f, 0x5581, 0x5582, 0x5586, 0x5588,
  0x558e, 0x558f, 0x5591, 0x5592, 0x5593, 0x5594, 0x5597, 0x55a3,
  0x55a4, 0x55ad, 0x55b2, 0x55bf, 0x55c1, 0x55c3, 0x55c6, 0x55c9,
  0x55cb, 0x55cc, 0x55ce, 0x55d1, 0x55d2, 0x55d3, 0x55d7, 0x55d8,
  0x55db, 0x55de, 0x55e2, 0x55e9, 0x55f6, 0x55ff, 0x5605, 0x5608,
  0x560a, 0x560d, 0x560e, 0x560f, 0x5610, 0x5611, 0x5612, 0x5619,
  0x562c, 0x5630, 0x5633, 0x5635, 0x5637, 0x5639, 0x563b, 0x563c,
  0x563d, 0x563f, 0x5640, 0x5641, 0x5643, 0x5644, 0x5646, 0x5649,
  0x564b, 0x564d, 0x564f, 0x5654, 0x565e, 0x5660, 0x5661, 0x5662,
  0x5663, 0x5666, 0x5669, 0x566d, 0x566f, 0x5671, 0x5672, 0x5675,
  0x5684, 0x5685, 0x5688, 0x568b, 0x568c, 0x5695, 0x5699, 0x569a,
  0x569d, 0x569e, 0x569f, 0x56a6, 0x56a7, 0x56a8, 0x56a9, 0x56ab,
  0x56ac, 0x56ad, 0x56b1, 0x56b3, 0x56b7, 0x56be, 0x56c5, 0x56c9,
  0x56ca, 0x56cb, 0x56cf, 0x56d0, 0x56cc, 0x56cd, 0x56d9, 0x56dc,
  0x56dd, 0x56df, 0x56e1, 0x56e4, 0x56e5, 0x56e6, 0x56e7, 0x56e8,
  0x56f1, 0x56eb, 0x56ed, 0x56f6, 0x56f7, 0x5701, 0x5702, 0x5707,
  0x570a, 0x570c, 0x5711, 0x5715, 0x571a, 0x571b, 0x571d, 0x5720,
  0x5722, 0x5723, 0x5724, 0x5725, 0x5729, 0x572a, 0x572c, 0x572e,
  0x572f, 0x5733, 0x5734, 0x573d, 0x573e, 0x573f, 0x5745, 0x5746,
  0x574c, 0x574d, 0x5752, 0x5762, 0x5765, 0x5767, 0x5768, 0x576b,
  0x576d, 0x576e, 0x576f, 0x5770, 0x5771, 0x5773, 0x5774, 0x5775,
  0x5777, 0x5779, 0x577a, 0x577b, 0x577c, 0x577e, 0x5781, 0x5783,
  0x578c, 0x5794, 0x5797, 0x5799, 0x579a, 0x579c, 0x579d, 0x579e,
  0x579f, 0x57a1, 0x5795, 0x57a7, 0x57a8, 0x57a9, 0x57ac, 0x57b8,
  0x57bd, 0x57c7, 0x57c8, 0x57cc, 0x57cf, 0x57d5, 0x57dd, 0x57de,
  0x57e4, 0x57e6, 0x57e7, 0x57e9, 0x57ed, 0x57f0, 0x57f5, 0x57f6,
  0x57f8, 0x57fd, 0x57fe, 0x57ff, 0x5803, 0x5804, 0x5808, 0x5809,
  0x57e1, 0x580c, 0x580d, 0x581b, 0x581e, 0x581f, 0x5820, 0x5826,
  0x5827, 0x582d, 0x5832, 0x5839, 0x583f, 0x5849, 0x584c, 0x584d,
  0x584f, 0x5850, 0x5855, 0x585f, 0x5861, 0x5864, 0x5867, 0x5868,
  0x5878, 0x587c, 0x587f, 0x5880, 0x5881, 0x5887, 0x5888, 0x5889,
  0x588a, 0x588c, 0x588d, 0x588f, 0x5890, 0x5894, 0x5896, 0x589d,
  0x58a0, 0x58a1, 0x58a2, 0x58a6, 0x58a9, 0x58b1, 0x58b2, 0x58c4,
  0x58bc, 0x58c2, 0x58c8, 0x58cd, 0x58ce, 0x58d0, 0x58d2, 0x58d4,
  0x58d6, 0x58da, 0x58dd, 0x58e1, 0x58e2, 0x58e9, 0x58f3, 0x5905,
  0x5906, 0x590b, 0x590c, 0x5912, 0x5913, 0x5914, 0x8641, 0x591d,
  0x5921, 0x5923, 0x5924, 0x5928, 0x592f, 0x5930, 0x5933, 0x5935,
  0x5936, 0x593f, 0x5943, 0x5946, 0x5952, 0x5953, 0x5959, 0x595b,
  0x595d, 0x595e, 0x595f, 0x5961, 0x5963, 0x596b, 0x596d, 0x596f,
  0x5972, 0x5975, 0x5976, 0x5979, 0x597b, 0x597c, 0x598b, 0x598c,
  0x598e, 0x5992, 0x5995, 0x5997, 0x599f, 0x59a4, 0x59a7, 0x59ad,
  0x59ae, 0x59af, 0x59b0, 0x59b3, 0x59b7, 0x59ba, 0x59bc, 0x59c1,
  0x59c3, 0x59c4, 0x59c8, 0x59ca, 0x59cd, 0x59d2, 0x59dd, 0x59de,
  0x59df, 0x59e3, 0x59e4, 0x59e7, 0x59ee, 0x59ef, 0x59f1, 0x59f2,
  0x59f4, 0x59f7, 0x5a00, 0x5a04, 0x5a0c, 0x5a0d, 0x5a0e, 0x5a12,
  0x5a13, 0x5a1e, 0x5a23, 0x5a24, 0x5a27, 0x5a28, 0x5a2a, 0x5a2d,
  0x5a30, 0x5a44, 0x5a45, 0x5a47, 0x5a48, 0x5a4c, 0x5a50, 0x5a55,
  0x5a5e, 0x5a63, 0x5a65, 0x5a67, 0x5a6d, 0x5a77, 0x5a7a, 0x5a7b,
  0x5a7e, 0x5a8b, 0x5a90, 0x5a93, 0x5a96, 0x5a99, 0x5a9c, 0x5a9e,
  0x5a9f, 0x5aa0, 0x5aa2, 0x5aa7, 0x5aac, 0x5ab1, 0x5ab2, 0x5ab3,
  0x5ab5, 0x5ab8, 0x5aba, 0x5abb, 0x5abf, 0x5ac4, 0x5ac6, 0x5ac8,
  0x5acf, 0x5ada, 0x5adc, 0x5ae0, 0x5ae5, 0x5aea, 0x5aee, 0x5af5,
  0x5af6, 0x5afd, 0x5b00, 0x5b01, 0x5b08, 0x5b17, 0x5b34, 0x5b19,
  0x5b1b, 0x5b1d, 0x5b21, 0x5b25, 0x5b2d, 0x5b38, 0x5b41, 0x5b4b,
  0x5b4c, 0x5b52, 0x5b56, 0x5b5e, 0x5b68, 0x5b6e, 0x5b6f, 0x5b7c,
  0x5b7d, 0x5b7e, 0x5b7f, 0x5b81, 0x5b84, 0x5b86, 0x5b8a, 0x5b8e,
  0x5b90, 0x5b91, 0x5b93, 0x5b94, 0x5b96, 0x5ba8, 0x5ba9, 0x5bac,
  0x5bad, 0x5baf, 0x5bb1, 0x5bb2, 0x5bb7, 0x5bba, 0x5bbc, 0x5bc0,
  0x5bc1, 0x5bcd, 0x5bcf, 0x5bd6, 0x5bd7, 0x5bd8, 0x5bd9, 0x5bda,
  0x5be0, 0x5bef, 0x5bf1, 0x5bf4, 0x5bfd, 0x5c0c, 0x5c17, 0x5c1e,
  0x5c1f, 0x5c23, 0x5c26, 0x5c29, 0x5c2b, 0x5c2c, 0x5c2e, 0x5c30,
  0x5c32, 0x5c35, 0x5c36, 0x5c59, 0x5c5a, 0x5c5c, 0x5c62, 0x5c63,
  0x5c67, 0x5c68, 0x5c69, 0x5c6d, 0x5c70, 0x5c74, 0x5c75, 0x5c7a,
  0x5c7b, 0x5c7c, 0x5c7d, 0x5c87, 0x5c88, 0x5c8a, 0x5c8f, 0x5c92,
  0x5c9d, 0x5c9f, 0x5ca0, 0x5ca2, 0x5ca3, 0x5ca6, 0x5caa, 0x5cb2,
  0x5cb4, 0x5cb5, 0x5cba, 0x5cc9, 0x5ccb, 0x5cd2, 0x5cdd, 0x5cd7,
  0x5cee, 0x5cf1, 0x5cf2, 0x5cf4, 0x5d01, 0x5d06, 0x5d0d, 0x5d12,
  0x5d2b, 0x5d23, 0x5d24, 0x5d26, 0x5d27, 0x5d31, 0x5d34, 0x5d39,
  0x5d3d, 0x5d3f, 0x5d42, 0x5d43, 0x5d46, 0x5d48, 0x5d55, 0x5d51,
  0x5d59, 0x5d4a, 0x5d5f, 0x5d60, 0x5d61, 0x5d62, 0x5d64, 0x5d6a,
  0x5d6d, 0x5d70, 0x5d79, 0x5d7a, 0x5d7e, 0x5d7f, 0x5d81, 0x5d83,
  0x5d88, 0x5d8a, 0x5d92, 0x5d93, 0x5d94, 0x5d95, 0x5d99, 0x5d9b,
  0x5d9f, 0x5da0, 0x5da7, 0x5dab, 0x5db0, 0x5db4, 0x5db8, 0x5db9,
  0x5dc3, 0x5dc7, 0x5dcb, 0x5dd0, 0x5dce, 0x5dd8, 0x5dd9, 0x5de0,
  0x5de4, 0x5de9, 0x5df8, 0x5df9, 0x5e00, 0x5e07, 0x5e0d, 0x5e12,
  0x5e14, 0x5e15, 0x5e18, 0x5e1f, 0x5e20, 0x5e2e, 0x5e28, 0x5e32,
  0x5e35, 0x5e3e, 0x5e4b, 0x5e50, 0x5e49, 0x5e51, 0x5e56, 0x5e58,
  0x5e5b, 0x5e5c, 0x5e5e, 0x5e68, 0x5e6a, 0x5e6b, 0x5e6c, 0x5e6d,
  0x5e6e, 0x5e70, 0x5e80, 0x5e8b, 0x5e8e, 0x5ea2, 0x5ea4, 0x5ea5,
  0x5ea8, 0x5eaa, 0x5eac, 0x5eb1, 0x5eb3, 0x5ebd, 0x5ebe, 0x5ebf,
  0x5ec6, 0x5ecc, 0x5ecb, 0x5ece, 0x5ed1, 0x5ed2, 0x5ed4, 0x5ed5,
  0x5edc, 0x5ede, 0x5ee5, 0x5eeb, 0x5f02, 0x5f06, 0x5f07, 0x5f08,
  0x5f0e, 0x5f19, 0x5f1c, 0x5f1d, 0x5f21, 0x5f22, 0x5f23, 0x5f24,
  0x5f28, 0x5f2b, 0x5f2c, 0x5f2e, 0x5f30, 0x5f34, 0x5f36, 0x5f3b,
  0x5f3d, 0x5f3f, 0x5f40, 0x5f44, 0x5f45, 0x5f47, 0x5f4d, 0x5f50,
  0x5f54, 0x5f58, 0x5f5b, 0x5f60, 0x5f63, 0x5f64, 0x5f67, 0x5f6f,
  0x5f72, 0x5f74, 0x5f75, 0x5f78, 0x5f7a, 0x5f7d, 0x5f7e, 0x5f89,
  0x5f8d, 0x5f8f, 0x5f96, 0x5f9c, 0x5f9d, 0x5fa2, 0x5fa7, 0x5fab,
  0x5fa4, 0x5fac, 0x5faf, 0x5fb0, 0x5fb1, 0x5fb8, 0x5fc4, 0x5fc7,
  0x5fc8, 0x5fc9, 0x5fcb, 0x5fd0, 0x5fd1, 0x5fd2, 0x5fd3, 0x5fd4,
  0x5fde, 0x5fe1, 0x5fe2, 0x5fe8, 0x5fe9, 0x5fea, 0x5fec, 0x5fed,
  0x5fee, 0x5fef, 0x5ff2, 0x5ff3, 0x5ff6, 0x5ffa, 0x5ffc, 0x6007,
  0x600a, 0x600d, 0x6013, 0x6014, 0x6017, 0x6018, 0x601a, 0x601f,
  0x6024, 0x602d, 0x6033, 0x6035, 0x6040, 0x6047, 0x6048, 0x6049,
  0x604c, 0x6051, 0x6054, 0x6056, 0x6057, 0x605d, 0x6061, 0x6067,
  0x6071, 0x607e, 0x607f, 0x6082, 0x6086, 0x6088, 0x608a, 0x608e,
  0x6091, 0x6093, 0x6095, 0x6098, 0x609d, 0x609e, 0x60a2, 0x60a4,
  0x60a5, 0x60a8, 0x60b0, 0x60b1, 0x60b7, 0x60bb, 0x60be, 0x60c2,
  0x60c4, 0x60c8, 0x60c9, 0x60ca, 0x60cb, 0x60ce, 0x60cf, 0x60d4,
  0x60d5, 0x60d9, 0x60db, 0x60dd, 0x60de, 0x60e2, 0x60e5, 0x60f2,
  0x60f5, 0x60f8, 0x60fc, 0x60fd, 0x6102, 0x6107, 0x610a, 0x610c,
  0x6110, 0x6111, 0x6112, 0x6113, 0x6114, 0x6116, 0x6117, 0x6119,
  0x611c, 0x611e, 0x6122, 0x612a, 0x612b, 0x6130, 0x6131, 0x6135,
  0x6136, 0x6137, 0x6139, 0x6141, 0x6145, 0x6146, 0x6149, 0x615e,
  0x6160, 0x616c, 0x6172, 0x6178, 0x617b, 0x617c, 0x617f, 0x6180,
  0x6181, 0x6183, 0x6184, 0x618b, 0x618d, 0x6192, 0x6193, 0x6197,
  0x6198, 0x619c, 0x619d, 0x619f, 0x61a0, 0x61a5, 0x61a8, 0x61aa,
  0x61ad, 0x61b8, 0x61b9, 0x61bc, 0x61c0, 0x61c1, 0x61c2, 0x61ce,
  0x61cf, 0x61d5, 0x61dc, 0x61dd, 0x61de, 0x61df, 0x61e1, 0x61e2,
  0x61e7, 0x61e9, 0x61e5, 0x61ec, 0x61ed, 0x61ef, 0x6201, 0x6203,
  0x6204, 0x6207, 0x6213, 0x6215, 0x621c, 0x6220, 0x6222, 0x6223,
  0x6227, 0x6229, 0x622b, 0x6239, 0x623d, 0x6242, 0x6243, 0x6244,
  0x6246, 0x624c, 0x6250, 0x6251, 0x6252, 0x6254, 0x6256, 0x625a,
  0x625c, 0x6264, 0x626d, 0x626f, 0x6273, 0x627a, 0x627d, 0x628d,
  0x628e, 0x628f, 0x6290, 0x62a6, 0x62a8, 0x62b3, 0x62b6, 0x62b7,
  0x62ba, 0x62be, 0x62bf, 0x62c4, 0x62ce, 0x62d5, 0x62d6, 0x62da,
  0x62ea, 0x62f2, 0x62f4, 0x62fc, 0x62fd, 0x6303, 0x6304, 0x630a,
  0x630b, 0x630d, 0x6310, 0x6313, 0x6316, 0x6318, 0x6329, 0x632a,
  0x632d, 0x6335, 0x6336, 0x6339, 0x633c, 0x6341, 0x6342, 0x6343,
  0x6344, 0x6346, 0x634a, 0x634b, 0x634e, 0x6352, 0x6353, 0x6354,
  0x6358, 0x635b, 0x6365, 0x6366, 0x636c, 0x636d, 0x6371, 0x6374,
  0x6375, 0x6378, 0x637c, 0x637d, 0x637f, 0x6382, 0x6384, 0x6387,
  0x638a, 0x6390, 0x6394, 0x6395, 0x6399, 0x639a, 0x639e, 0x63a4,
  0x63a6, 0x63ad, 0x63ae, 0x63af, 0x63bd, 0x63c1, 0x63c5, 0x63c8,
  0x63ce, 0x63d1, 0x63d3, 0x63d4, 0x63d5, 0x63dc, 0x63e0, 0x63e5,
  0x63ea, 0x63ec, 0x63f2, 0x63f3, 0x63f5, 0x63f8, 0x63f9, 0x6409,
  0x640a, 0x6410, 0x6412, 0x6414, 0x6418, 0x641e, 0x6420, 0x6422,
  0x6424, 0x6425, 0x6429, 0x642a, 0x642f, 0x6430, 0x6435, 0x643d,
  0x643f, 0x644b, 0x644f, 0x6451, 0x6452, 0x6453, 0x6454, 0x645a,
  0x645b, 0x645c, 0x645d, 0x645f, 0x6460, 0x6461, 0x6463, 0x646d,
  0x6473, 0x6474, 0x647b, 0x647d, 0x6485, 0x6487, 0x648f, 0x6490,
  0x6491, 0x6498, 0x6499, 0x649b, 0x649d, 0x649f, 0x64a1, 0x64a3,
  0x64a6, 0x64a8, 0x64ac, 0x64b3, 0x64bd, 0x64be, 0x64bf, 0x64c4,
  0x64c9, 0x64ca, 0x64cb, 0x64cc, 0x64ce, 0x64d0, 0x64d1, 0x64d5,
  0x64d7, 0x64e4, 0x64e5, 0x64e9, 0x64ea, 0x64ed, 0x64f0, 0x64f5,
  0x64f7, 0x64fb, 0x64ff, 0x6501, 0x6504, 0x6508, 0x6509, 0x650a,
  0x650f, 0x6513, 0x6514, 0x6516, 0x6519, 0x651b, 0x651e, 0x651f,
  0x6522, 0x6526, 0x6529, 0x652e, 0x6531, 0x653a, 0x653c, 0x653d,
  0x6543, 0x6547, 0x6549, 0x6550, 0x6552, 0x6554, 0x655f, 0x6560,
  0x6567, 0x656b, 0x657a, 0x657d, 0x6581, 0x6585, 0x658a, 0x6592,
  0x6595, 0x6598, 0x659d, 0x65a0, 0x65a3, 0x65a6, 0x65ae, 0x65b2,
  0x65b3, 0x65b4, 0x65bf, 0x65c2, 0x65c8, 0x65c9, 0x65ce, 0x65d0,
  0x65d4, 0x65d6, 0x65d8, 0x65df, 0x65f0, 0x65f2, 0x65f4, 0x65f5,
  0x65f9, 0x65fe, 0x65ff, 0x6600, 0x6604, 0x6608, 0x6609, 0x660d,
  0x6611, 0x6612, 0x6615, 0x6616, 0x661d, 0x661e, 0x6621, 0x6622,
  0x6623, 0x6624, 0x6626, 0x6629, 0x662a, 0x662b, 0x662c, 0x662e,
  0x6630, 0x6631, 0x6633, 0x6639, 0x6637, 0x6640, 0x6645, 0x6646,
  0x664a, 0x664c, 0x6651, 0x664e, 0x6657, 0x6658, 0x6659, 0x665b,
  0x665c, 0x6660, 0x6661, 0x66fb, 0x666a, 0x666b, 0x666c, 0x667e,
  0x6673, 0x6675, 0x667f, 0x6677, 0x6678, 0x6679, 0x667b, 0x6680,
  0x667c, 0x668b, 0x668c, 0x668d, 0x6690, 0x6692, 0x6699, 0x669a,
  0x669b, 0x669c, 0x669f, 0x66a0, 0x66a4, 0x66ad, 0x66b1, 0x66b2,
  0x66b5, 0x66bb, 0x66bf, 0x66c0, 0x66c2, 0x66c3, 0x66c8, 0x66cc,
  0x66ce, 0x66cf, 0x66d4, 0x66db, 0x66df, 0x66e8, 0x66eb, 0x66ec,
  0x66ee, 0x66fa, 0x6705, 0x6707, 0x670e, 0x6713, 0x6719, 0x671c,
  0x6720, 0x6722, 0x6733, 0x673e, 0x6745, 0x6747, 0x6748, 0x674c,
  0x6754, 0x6755, 0x675d, 0x6766, 0x676c, 0x676e, 0x6774, 0x6776,
  0x677b, 0x6781, 0x6784, 0x678e, 0x678f, 0x6791, 0x6793, 0x6796,
  0x6798, 0x6799, 0x679b, 0x67b0, 0x67b1, 0x67b2, 0x67b5, 0x67bb,
  0x67bc, 0x67bd, 0x67f9, 0x67c0, 0x67c2, 0x67c3, 0x67c5, 0x67c8,
  0x67c9, 0x67d2, 0x67d7, 0x67d9, 0x67dc, 0x67e1, 0x67e6, 0x67f0,
  0x67f2, 0x67f6, 0x67f7, 0x6852, 0x6814, 0x6819, 0x681d, 0x681f,
  0x6828, 0x6827, 0x682c, 0x682d, 0x682f, 0x6830, 0x6831, 0x6833,
  0x683b, 0x683f, 0x6844, 0x6845, 0x684a, 0x684c, 0x6855, 0x6857,
  0x6858, 0x685b, 0x686b, 0x686e, 0x686f, 0x6870, 0x6871, 0x6872,
  0x6875, 0x6879, 0x687a, 0x687b, 0x687c, 0x6882, 0x6884, 0x6886,
  0x6888, 0x6896, 0x6898, 0x689a, 0x689c, 0x68a1, 0x68a3, 0x68a5,
  0x68a9, 0x68aa, 0x68ae, 0x68b2, 0x68bb, 0x68c5, 0x68c8, 0x68cc,
  0x68cf, 0x68d0, 0x68d1, 0x68d3, 0x68d6, 0x68d9, 0x68dc, 0x68dd,
  0x68e5, 0x68e8, 0x68ea, 0x68eb, 0x68ec, 0x68ed, 0x68f0, 0x68f1,
  0x68f5, 0x68f6, 0x68fb, 0x68fc, 0x68fd, 0x6906, 0x6909, 0x690a,
  0x6910, 0x6911, 0x6913, 0x6916, 0x6917, 0x6931, 0x6933, 0x6935,
  0x6938, 0x693b, 0x6942, 0x6945, 0x6949, 0x694e, 0x6957, 0x695b,
  0x6963, 0x6964, 0x6965, 0x6966, 0x6968, 0x6969, 0x696c, 0x6970,
  0x6971, 0x6972, 0x697a, 0x697b, 0x697f, 0x6980, 0x698d, 0x6992,
  0x6996, 0x6998, 0x69a1, 0x69a5, 0x69a6, 0x69a8, 0x69ab, 0x69ad,
  0x69af, 0x69b7, 0x69b8, 0x69ba, 0x69bc, 0x69c5, 0x69c8, 0x69d1,
  0x69d6, 0x69d7, 0x69e2, 0x69e5, 0x69ee, 0x69ef, 0x69f1, 0x69f3,
  0x69f5, 0x69fe, 0x6a00, 0x6a01, 0x6a03, 0x6a0f, 0x6a11, 0x6a15,
  0x6a1a, 0x6a1d, 0x6a20, 0x6a24, 0x6a28, 0x6a30, 0x6a32, 0x6a34,
  0x6a37, 0x6a3b, 0x6a3e, 0x6a3f, 0x6a45, 0x6a46, 0x6a49, 0x6a4a,
  0x6a4e, 0x6a50, 0x6a51, 0x6a52, 0x6a55, 0x6a56, 0x6a5b, 0x6a64,
  0x6a67, 0x6a6a, 0x6a71, 0x6a73, 0x6a7e, 0x6a81, 0x6a83, 0x6a86,
  0x6a87, 0x6a89, 0x6a8b, 0x6a91, 0x6a9b, 0x6a9d, 0x6a9e, 0x6a9f,
  0x6aa5, 0x6aab, 0x6aaf, 0x6ab0, 0x6ab1, 0x6ab4, 0x6abd, 0x6abe,
  0x6abf, 0x6ac6, 0x6ac9, 0x6ac8, 0x6acc, 0x6ad0, 0x6ad4, 0x6ad5,
  0x6ad6, 0x6adc, 0x6add, 0x6ae4, 0x6ae7, 0x6aec, 0x6af0, 0x6af1,
  0x6af2, 0x6afc, 0x6afd, 0x6b02, 0x6b03, 0x6b06, 0x6b07, 0x6b09,
  0x6b0f, 0x6b10, 0x6b11, 0x6b17, 0x6b1b, 0x6b1e, 0x6b24, 0x6b28,
  0x6b2b, 0x6b2c, 0x6b2f, 0x6b35, 0x6b36, 0x6b3b, 0x6b3f, 0x6b46,
  0x6b4a, 0x6b4d, 0x6b52, 0x6b56, 0x6b58, 0x6b5d, 0x6b60, 0x6b67,
  0x6b6b, 0x6b6e, 0x6b70, 0x6b75, 0x6b7d, 0x6b7e, 0x6b82, 0x6b85,
  0x6b97, 0x6b9b, 0x6b9f, 0x6ba0, 0x6ba2, 0x6ba3, 0x6ba8, 0x6ba9,
  0x6bac, 0x6bad, 0x6bae, 0x6bb0, 0x6bb8, 0x6bb9, 0x6bbd, 0x6bbe,
  0x6bc3, 0x6bc4, 0x6bc9, 0x6bcc, 0x6bd6, 0x6bda, 0x6be1, 0x6be3,
  0x6be6, 0x6be7, 0x6bee, 0x6bf1, 0x6bf7, 0x6bf9, 0x6bff, 0x6c02,
  0x6c04, 0x6c05, 0x6c09, 0x6c0d, 0x6c0e, 0x6c10, 0x6c12, 0x6c19,
  0x6c1f, 0x6c26, 0x6c27, 0x6c28, 0x6c2c, 0x6c2e, 0x6c33, 0x6c35,
  0x6c36, 0x6c3a, 0x6c3b, 0x6c3f, 0x6c4a, 0x6c4b, 0x6c4d, 0x6c4f,
  0x6c52, 0x6c54, 0x6c59, 0x6c5b, 0x6c5c, 0x6c6b, 0x6c6d, 0x6c6f,
  0x6c74, 0x6c76, 0x6c78, 0x6c79, 0x6c7b, 0x6c85, 0x6c86, 0x6c87,
  0x6c89, 0x6c94, 0x6c95, 0x6c97, 0x6c98, 0x6c9c, 0x6c9f, 0x6cb0,
  0x6cb2, 0x6cb4, 0x6cc2, 0x6cc6, 0x6ccd, 0x6ccf, 0x6cd0, 0x6cd1,
  0x6cd2, 0x6cd4, 0x6cd6, 0x6cda, 0x6cdc, 0x6ce0, 0x6ce7, 0x6ce9,
  0x6ceb, 0x6cec, 0x6cee, 0x6cf2, 0x6cf4, 0x6d04, 0x6d07, 0x6d0a,
  0x6d0e, 0x6d0f, 0x6d11, 0x6d13, 0x6d1a, 0x6d26, 0x6d27, 0x6d28,
  0x6c67, 0x6d2e, 0x6d2f, 0x6d31, 0x6d39, 0x6d3c, 0x6d3f, 0x6d57,
  0x6d5e, 0x6d5f, 0x6d61, 0x6d65, 0x6d67, 0x6d6f, 0x6d70, 0x6d7c,
  0x6d82, 0x6d87, 0x6d91, 0x6d92, 0x6d94, 0x6d96, 0x6d97, 0x6d98,
  0x6daa, 0x6dac, 0x6db4, 0x6db7, 0x6db9, 0x6dbd, 0x6dbf, 0x6dc4,
  0x6dc8, 0x6dca, 0x6dce, 0x6dcf, 0x6dd6, 0x6ddb, 0x6ddd, 0x6ddf,
  0x6de0, 0x6de2, 0x6de5, 0x6de9, 0x6def, 0x6df0, 0x6df4, 0x6df6,
  0x6dfc, 0x6e00, 0x6e04, 0x6e1e, 0x6e22, 0x6e27, 0x6e32, 0x6e36,
  0x6e39, 0x6e3b, 0x6e3c, 0x6e44, 0x6e45, 0x6e48, 0x6e49, 0x6e4b,
  0x6e4f, 0x6e51, 0x6e52, 0x6e53, 0x6e54, 0x6e57, 0x6e5c, 0x6e5d,
  0x6e5e, 0x6e62, 0x6e63, 0x6e68, 0x6e73, 0x6e7b, 0x6e7d, 0x6e8d,
  0x6e93, 0x6e99, 0x6ea0, 0x6ea7, 0x6ead, 0x6eae, 0x6eb1, 0x6eb3,
  0x6ebb, 0x6ebf, 0x6ec0, 0x6ec1, 0x6ec3, 0x6ec7, 0x6ec8, 0x6eca,
  0x6ecd, 0x6ece, 0x6ecf, 0x6eeb, 0x6eed, 0x6eee, 0x6ef9, 0x6efb,
  0x6efd, 0x6f04, 0x6f08, 0x6f0a, 0x6f0c, 0x6f0d, 0x6f16, 0x6f18,
  0x6f1a, 0x6f1b, 0x6f26, 0x6f29, 0x6f2a, 0x6f2f, 0x6f30, 0x6f33,
  0x6f36, 0x6f3b, 0x6f3c, 0x6f2d, 0x6f4f, 0x6f51, 0x6f52, 0x6f53,
  0x6f57, 0x6f59, 0x6f5a, 0x6f5d, 0x6f5e, 0x6f61, 0x6f62, 0x6f68,
  0x6f6c, 0x6f7d, 0x6f7e, 0x6f83, 0x6f87, 0x6f88, 0x6f8b, 0x6f8c,
  0x6f8d, 0x6f90, 0x6f92, 0x6f93, 0x6f94, 0x6f96, 0x6f9a, 0x6f9f,
  0x6fa0, 0x6fa5, 0x6fa6, 0x6fa7, 0x6fa8, 0x6fae, 0x6faf, 0x6fb0,
  0x6fb5, 0x6fb6, 0x6fbc, 0x6fc5, 0x6fc7, 0x6fc8, 0x6fca, 0x6fda,
  0x6fde, 0x6fe8, 0x6fe9, 0x6ff0, 0x6ff5, 0x6ff9, 0x6ffc, 0x6ffd,
  0x7000, 0x7005, 0x7006, 0x7007, 0x700d, 0x7017, 0x7020, 0x7023,
  0x702f, 0x7034, 0x7037, 0x7039, 0x703c, 0x7043, 0x7044, 0x7048,
  0x7049, 0x704a, 0x704b, 0x7054, 0x7055, 0x705d, 0x705e, 0x704e,
  0x7064, 0x7065, 0x706c, 0x706e, 0x7075, 0x7076, 0x707e, 0x7081,
  0x7085, 0x7086, 0x7094, 0x7095, 0x7096, 0x7097, 0x7098, 0x709b,
  0x70a4, 0x70ab, 0x70b0, 0x70b1, 0x70b4, 0x70b7, 0x70ca, 0x70d1,
  0x70d3, 0x70d4, 0x70d5, 0x70d6, 0x70d8, 0x70dc, 0x70e4, 0x70fa,
  0x7103, 0x7104, 0x7105, 0x7106, 0x7107, 0x710b, 0x710c, 0x710f,
  0x711e, 0x7120, 0x712b, 0x712d, 0x712f, 0x7130, 0x7131, 0x7138,
  0x7141, 0x7145, 0x7146, 0x7147, 0x714a, 0x714b, 0x7150, 0x7152,
  0x7157, 0x715a, 0x715c, 0x715e, 0x7160, 0x7168, 0x7179, 0x7180,
  0x7185, 0x7187, 0x718c, 0x7192, 0x719a, 0x719b, 0x71a0, 0x71a2,
  0x71af, 0x71b0, 0x71b2, 0x71b3, 0x71ba, 0x71bf, 0x71c0, 0x71c1,
  0x71c4, 0x71cb, 0x71cc, 0x71d3, 0x71d6, 0x71d9, 0x71da, 0x71dc,
  0x71f8, 0x71fe, 0x7200, 0x7207, 0x7208, 0x7209, 0x7213, 0x7217,
  0x721a, 0x721d, 0x721f, 0x7224, 0x722b, 0x722f, 0x7234, 0x7238,
  0x7239, 0x7241, 0x7242, 0x7243, 0x7245, 0x724e, 0x724f, 0x7250,
  0x7253, 0x7255, 0x7256, 0x725a, 0x725c, 0x725e, 0x7260, 0x7263,
  0x7268, 0x726b, 0x726e, 0x726f, 0x7271, 0x7277, 0x7278, 0x727b,
  0x727c, 0x727f, 0x7284, 0x7289, 0x728d, 0x728e, 0x7293, 0x729b,
  0x72a8, 0x72ad, 0x72ae, 0x72b1, 0x72b4, 0x72be, 0x72c1, 0x72c7,
  0x72c9, 0x72cc, 0x72d5, 0x72d6, 0x72d8, 0x72df, 0x72e5, 0x72f3,
  0x72f4, 0x72fa, 0x72fb, 0x72fe, 0x7302, 0x7304, 0x7305, 0x7307,
  0x730b, 0x730d, 0x7312, 0x7313, 0x7318, 0x7319, 0x731e, 0x7322,
  0x7324, 0x7327, 0x7328, 0x732c, 0x7331, 0x7332, 0x7335, 0x733a,
  0x733b, 0x733d, 0x7343, 0x734d, 0x7350, 0x7352, 0x7356, 0x7358,
  0x735d, 0x735e, 0x735f, 0x7360, 0x7366, 0x7367, 0x7369, 0x736b,
  0x736c, 0x736e, 0x736f, 0x7371, 0x7377, 0x7379, 0x737c, 0x7380,
  0x7381, 0x7383, 0x7385, 0x7386, 0x738e, 0x7390, 0x7393, 0x7395,
  0x7397, 0x7398, 0x739c, 0x739e, 0x739f, 0x73a0, 0x73a2, 0x73a5,
  0x73a6, 0x73aa, 0x73ab, 0x73ad, 0x73b5, 0x73b7, 0x73b9, 0x73bc,
  0x73bd, 0x73bf, 0x73c5, 0x73c6, 0x73c9, 0x73cb, 0x73cc, 0x73cf,
  0x73d2, 0x73d3, 0x73d6, 0x73d9, 0x73dd, 0x73e1, 0x73e3, 0x73e6,
  0x73e7, 0x73e9, 0x73f4, 0x73f5, 0x73f7, 0x73f9, 0x73fa, 0x73fb,
  0x73fd, 0x73ff, 0x7400, 0x7401, 0x7404, 0x7407, 0x740a, 0x7411,
  0x741a, 0x741b, 0x7424, 0x7426, 0x7428, 0x7429, 0x742a, 0x742b,
  0x742c, 0x742d, 0x742e, 0x742f, 0x7430, 0x7431, 0x7439, 0x7440,
  0x7443, 0x7444, 0x7446, 0x7447, 0x744b, 0x744d, 0x7451, 0x7452,
  0x7457, 0x745d, 0x7462, 0x7466, 0x7467, 0x7468, 0x746b, 0x746d,
  0x746e, 0x7471, 0x7472, 0x7480, 0x7481, 0x7485, 0x7486, 0x7487,
  0x7489, 0x748f, 0x7490, 0x7491, 0x7492, 0x7498, 0x7499, 0x749a,
  0x749c, 0x749f, 0x74a0, 0x74a1, 0x74a3, 0x74a6, 0x74a8, 0x74a9,
  0x74aa, 0x74ab, 0x74ae, 0x74af, 0x74b1, 0x74b2, 0x74b5, 0x74b9,
  0x74bb, 0x74bf, 0x74c8, 0x74c9, 0x74cc, 0x74d0, 0x74d3, 0x74d8,
  0x74da, 0x74db, 0x74de, 0x74df, 0x74e4, 0x74e8, 0x74ea, 0x74eb,
  0x74ef, 0x74f4, 0x74fa, 0x74fb, 0x74fc, 0x74ff, 0x7506, 0x7512,
  0x7516, 0x7517, 0x7520, 0x7521, 0x7524, 0x7527, 0x7529, 0x752a,
  0x752f, 0x7536, 0x7539, 0x753d, 0x753e, 0x753f, 0x7540, 0x7543,
  0x7547, 0x7548, 0x754e, 0x7550, 0x7552, 0x7557, 0x755e, 0x755f,
  0x7561, 0x756f, 0x7571, 0x7579, 0x757a, 0x757b, 0x757c, 0x757d,
  0x757e, 0x7581, 0x7585, 0x7590, 0x7592, 0x7593, 0x7595, 0x7599,
  0x759c, 0x75a2, 0x75a4, 0x75b4, 0x75ba, 0x75bf, 0x75c0, 0x75c1,
  0x75c4, 0x75c6, 0x75cc, 0x75ce, 0x75cf, 0x75d7, 0x75dc, 0x75df,
  0x75e0, 0x75e1, 0x75e4, 0x75e7, 0x75ec, 0x75ee, 0x75ef, 0x75f1,
  0x75f9, 0x7600, 0x7602, 0x7603, 0x7604, 0x7607, 0x7608, 0x760a,
  0x760c, 0x760f, 0x7612, 0x7613, 0x7615, 0x7616, 0x7619, 0x761b,
  0x761c, 0x761d, 0x761e, 0x7623, 0x7625, 0x7626, 0x7629, 0x762d,
  0x7632, 0x7633, 0x7635, 0x7638, 0x7639, 0x763a, 0x763c, 0x764a,
  0x7640, 0x7641, 0x7643, 0x7644, 0x7645, 0x7649, 0x764b, 0x7655,
  0x7659, 0x765f, 0x7664, 0x7665, 0x766d, 0x766e, 0x766f, 0x7671,
  0x7674, 0x7681, 0x7685, 0x768c, 0x768d, 0x7695, 0x769b, 0x769c,
  0x769d, 0x769f, 0x76a0, 0x76a2, 0x76a3, 0x76a4, 0x76a5, 0x76a6,
  0x76a7, 0x76a8, 0x76aa, 0x76ad, 0x76bd, 0x76c1, 0x76c5, 0x76c9,
  0x76cb, 0x76cc, 0x76ce, 0x76d4, 0x76d9, 0x76e0, 0x76e6, 0x76e8,
  0x76ec, 0x76f0, 0x76f1, 0x76f6, 0x76f9, 0x76fc, 0x7700, 0x7706,
  0x770a, 0x770e, 0x7712, 0x7714, 0x7715, 0x7717, 0x7719, 0x771a,
  0x771c, 0x7722, 0x7728, 0x772d, 0x772e, 0x772f, 0x7734, 0x7735,
  0x7736, 0x7739, 0x773d, 0x773e, 0x7742, 0x7745, 0x7746, 0x774a,
  0x774d, 0x774e, 0x774f, 0x7752, 0x7756, 0x7757, 0x775c, 0x775e,
  0x775f, 0x7760, 0x7762, 0x7764, 0x7767, 0x776a, 0x776c, 0x7770,
  0x7772, 0x7773, 0x7774, 0x777a, 0x777d, 0x7780, 0x7784, 0x778c,
  0x778d, 0x7794, 0x7795, 0x7796, 0x779a, 0x779f, 0x77a2, 0x77a7,
  0x77aa, 0x77ae, 0x77af, 0x77b1, 0x77b5, 0x77be, 0x77c3, 0x77c9,
  0x77d1, 0x77d2, 0x77d5, 0x77d9, 0x77de, 0x77df, 0x77e0, 0x77e4,
  0x77e6, 0x77ea, 0x77ec, 0x77f0, 0x77f1, 0x77f4, 0x77f8, 0x77fb,
  0x7805, 0x7806, 0x7809, 0x780d, 0x780e, 0x7811, 0x781d, 0x7821,
  0x7822, 0x7823, 0x782d, 0x782e, 0x7830, 0x7835, 0x7837, 0x7843,
  0x7844, 0x7847, 0x7848, 0x784c, 0x784e, 0x7852, 0x785c, 0x785e,
  0x7860, 0x7861, 0x7863, 0x7864, 0x7868, 0x786a, 0x786e, 0x787a,
  0x787e, 0x788a, 0x788f, 0x7894, 0x7898, 0x78a1, 0x789d, 0x789e,
  0x789f, 0x78a4, 0x78a8, 0x78ac, 0x78ad, 0x78b0, 0x78b1, 0x78b2,
  0x78b3, 0x78bb, 0x78bd, 0x78bf, 0x78c7, 0x78c8, 0x78c9, 0x78cc,
  0x78ce, 0x78d2, 0x78d3, 0x78d5, 0x78d6, 0x78e4, 0x78db, 0x78df,
  0x78e0, 0x78e1, 0x78e6, 0x78ea, 0x78f2, 0x78f3, 0x7900, 0x78f6,
  0x78f7, 0x78fa, 0x78fb, 0x78ff, 0x7906, 0x790c, 0x7910, 0x791a,
  0x791c, 0x791e, 0x791f, 0x7920, 0x7925, 0x7927, 0x7929, 0x792d,
  0x7931, 0x7934, 0x7935, 0x793b, 0x793d, 0x793f, 0x7944, 0x7945,
  0x7946, 0x794a, 0x794b, 0x794f, 0x7951, 0x7954, 0x7958, 0x795b,
  0x795c, 0x7967, 0x7969, 0x796b, 0x7972, 0x7979, 0x797b, 0x797c,
  0x797e, 0x798b, 0x798c, 0x7991, 0x7993, 0x7994, 0x7995, 0x7996,
  0x7998, 0x799b, 0x799c, 0x79a1, 0x79a8, 0x79a9, 0x79ab, 0x79af,
  0x79b1, 0x79b4, 0x79b8, 0x79bb, 0x79c2, 0x79c4, 0x79c7, 0x79c8,
  0x79ca, 0x79cf, 0x79d4, 0x79d6, 0x79da, 0x79dd, 0x79de, 0x79e0,
  0x79e2, 0x79e5, 0x79ea, 0x79eb, 0x79ed, 0x79f1, 0x79f8, 0x79fc,
  0x7a02, 0x7a03, 0x7a07, 0x7a09, 0x7a0a, 0x7a0c, 0x7a11, 0x7a15,
  0x7a1b, 0x7a1e, 0x7a21, 0x7a27, 0x7a2b, 0x7a2d, 0x7a2f, 0x7a30,
  0x7a34, 0x7a35, 0x7a38, 0x7a39, 0x7a3a, 0x7a44, 0x7a45, 0x7a47,
  0x7a48, 0x7a4c, 0x7a55, 0x7a56, 0x7a59, 0x7a5c, 0x7a5d, 0x7a5f,
  0x7a60, 0x7a65, 0x7a67, 0x7a6a, 0x7a6d, 0x7a75, 0x7a78, 0x7a7e,
  0x7a80, 0x7a82, 0x7a85, 0x7a86, 0x7a8a, 0x7a8b, 0x7a90, 0x7a91,
  0x7a94, 0x7a9e, 0x7aa0, 0x7aa3, 0x7aac, 0x7ab3, 0x7ab5, 0x7ab9,
  0x7abb, 0x7abc, 0x7ac6, 0x7ac9, 0x7acc, 0x7ace, 0x7ad1, 0x7adb,
  0x7ae8, 0x7ae9, 0x7aeb, 0x7aec, 0x7af1, 0x7af4, 0x7afb, 0x7afd,
  0x7afe, 0x7b07, 0x7b14, 0x7b1f, 0x7b23, 0x7b27, 0x7b29, 0x7b2a,
  0x7b2b, 0x7b2d, 0x7b2e, 0x7b2f, 0x7b30, 0x7b31, 0x7b34, 0x7b3d,
  0x7b3f, 0x7b40, 0x7b41, 0x7b47, 0x7b4e, 0x7b55, 0x7b60, 0x7b64,
  0x7b66, 0x7b69, 0x7b6a, 0x7b6d, 0x7b6f, 0x7b72, 0x7b73, 0x7b77,
  0x7b84, 0x7b89, 0x7b8e, 0x7b90, 0x7b91, 0x7b96, 0x7b9b, 0x7b9e,
  0x7ba0, 0x7ba5, 0x7bac, 0x7baf, 0x7bb0, 0x7bb2, 0x7bb5, 0x7bb6,
  0x7bba, 0x7bbb, 0x7bbc, 0x7bbd, 0x7bc2, 0x7bc5, 0x7bc8, 0x7bca,
  0x7bd4, 0x7bd6, 0x7bd7, 0x7bd9, 0x7bda, 0x7bdb, 0x7be8, 0x7bea,
  0x7bf2, 0x7bf4, 0x7bf5, 0x7bf8, 0x7bf9, 0x7bfa, 0x7bfc, 0x7bfe,
  0x7c01, 0x7c02, 0x7c03, 0x7c04, 0x7c06, 0x7c09, 0x7c0b, 0x7c0c,
  0x7c0e, 0x7c0f, 0x7c19, 0x7c1b, 0x7c20, 0x7c25, 0x7c26, 0x7c28,
  0x7c2c, 0x7c31, 0x7c33, 0x7c34, 0x7c36, 0x7c39, 0x7c3a, 0x7c46,
  0x7c4a, 0x7c55, 0x7c51, 0x7c52, 0x7c53, 0x7c59, 0x7c5a, 0x7c5b,
  0x7c5c, 0x7c5d, 0x7c5e, 0x7c61, 0x7c63, 0x7c67, 0x7c69, 0x7c6d,
  0x7c6e, 0x7c70, 0x7c72, 0x7c79, 0x7c7c, 0x7c7d, 0x7c86, 0x7c87,
  0x7c8f, 0x7c94, 0x7c9e, 0x7ca0, 0x7ca6, 0x7cb0, 0x7cb6, 0x7cb7,
  0x7cba, 0x7cbb, 0x7cbc, 0x7cbf, 0x7cc4, 0x7cc7, 0x7cc8, 0x7cc9,
  0x7ccd, 0x7ccf, 0x7cd3, 0x7cd4, 0x7cd5, 0x7cd7, 0x7cd9, 0x7cda,
  0x7cdd, 0x7ce6, 0x7ce9, 0x7ceb, 0x7cf5, 0x7d03, 0x7d07, 0x7d08,
  0x7d09, 0x7d0f, 0x7d11, 0x7d12, 0x7d13, 0x7d16, 0x7d1d, 0x7d1e,
  0x7d23, 0x7d26, 0x7d2a, 0x7d2d, 0x7d31, 0x7d3c, 0x7d3d, 0x7d3e,
  0x7d40, 0x7d41, 0x7d47, 0x7d48, 0x7d4d, 0x7d51, 0x7d53, 0x7d57,
  0x7d59, 0x7d5a, 0x7d5c, 0x7d5d, 0x7d65, 0x7d67, 0x7d6a, 0x7d70,
  0x7d78, 0x7d7a, 0x7d7b, 0x7d7f, 0x7d81, 0x7d82, 0x7d83, 0x7d85,
  0x7d86, 0x7d88, 0x7d8b, 0x7d8c, 0x7d8d, 0x7d91, 0x7d96, 0x7d97,
  0x7d9d, 0x7d9e, 0x7da6, 0x7da7, 0x7daa, 0x7db3, 0x7db6, 0x7db7,
  0x7db9, 0x7dc2, 0x7dc3, 0x7dc4, 0x7dc5, 0x7dc6, 0x7dcc, 0x7dcd,
  0x7dce, 0x7dd7, 0x7dd9, 0x7e00, 0x7de2, 0x7de5, 0x7de6, 0x7dea,
  0x7deb, 0x7ded, 0x7df1, 0x7df5, 0x7df6, 0x7df9, 0x7dfa, 0x7e08,
  0x7e10, 0x7e11, 0x7e15, 0x7e17, 0x7e1c, 0x7e1d, 0x7e20, 0x7e27,
  0x7e28, 0x7e2c, 0x7e2d, 0x7e2f, 0x7e33, 0x7e36, 0x7e3f, 0x7e44,
  0x7e45, 0x7e47, 0x7e4e, 0x7e50, 0x7e52, 0x7e58, 0x7e5f, 0x7e61,
  0x7e62, 0x7e65, 0x7e6b, 0x7e6e, 0x7e6f, 0x7e73, 0x7e78, 0x7e7e,
  0x7e81, 0x7e86, 0x7e87, 0x7e8a, 0x7e8d, 0x7e91, 0x7e95, 0x7e98,
  0x7e9a, 0x7e9d, 0x7e9e, 0x7f3c, 0x7f3b, 0x7f3d, 0x7f3e, 0x7f3f,
  0x7f43, 0x7f44, 0x7f47, 0x7f4f, 0x7f52, 0x7f53, 0x7f5b, 0x7f5c,
  0x7f5d, 0x7f61, 0x7f63, 0x7f64, 0x7f65, 0x7f66, 0x7f6d, 0x7f71,
  0x7f7d, 0x7f7e, 0x7f7f, 0x7f80, 0x7f8b, 0x7f8d, 0x7f8f, 0x7f90,
  0x7f91, 0x7f96, 0x7f97, 0x7f9c, 0x7fa1, 0x7fa2, 0x7fa6, 0x7faa,
  0x7fad, 0x7fb4, 0x7fbc, 0x7fbf, 0x7fc0, 0x7fc3, 0x7fc8, 0x7fce,
  0x7fcf, 0x7fdb, 0x7fdf, 0x7fe3, 0x7fe5, 0x7fe8, 0x7fec, 0x7fee,
  0x7fef, 0x7ff2, 0x7ffa, 0x7ffd, 0x7ffe, 0x7fff, 0x8007, 0x8008,
  0x800a, 0x800d, 0x800e, 0x800f, 0x8011, 0x8013, 0x8014, 0x8016,
  0x801d, 0x801e, 0x801f, 0x8020, 0x8024, 0x8026, 0x802c, 0x802e,
  0x8030, 0x8034, 0x8035, 0x8037, 0x8039, 0x803a, 0x803c, 0x803e,
  0x8040, 0x8044, 0x8060, 0x8064, 0x8066, 0x806d, 0x8071, 0x8075,
  0x8081, 0x8088, 0x808e, 0x809c, 0x809e, 0x80a6, 0x80a7, 0x80ab,
  0x80b8, 0x80b9, 0x80c8, 0x80cd, 0x80cf, 0x80d2, 0x80d4, 0x80d5,
  0x80d7, 0x80d8, 0x80e0, 0x80ed, 0x80ee, 0x80f0, 0x80f2, 0x80f3,
  0x80f6, 0x80f9, 0x80fa, 0x80fe, 0x8103, 0x810b, 0x8116, 0x8117,
  0x8118, 0x811c, 0x811e, 0x8120, 0x8124, 0x8127, 0x812c, 0x8130,
  0x8135, 0x813a, 0x813c, 0x8145, 0x8147, 0x814a, 0x814c, 0x8152,
  0x8157, 0x8160, 0x8161, 0x8167, 0x8168, 0x8169, 0x816d, 0x816f,
  0x8177, 0x8181, 0x8190, 0x8184, 0x8185, 0x8186, 0x818b, 0x818e,
  0x8196, 0x8198, 0x819b, 0x819e, 0x81a2, 0x81ae, 0x81b2, 0x81b4,
  0x81bb, 0x81cb, 0x81c3, 0x81c5, 0x81ca, 0x81ce, 0x81cf, 0x81d5,
  0x81d7, 0x81db, 0x81dd, 0x81de, 0x81e1, 0x81e4, 0x81eb, 0x81ec,
  0x81f0, 0x81f1, 0x81f2, 0x81f5, 0x81f6, 0x81f8, 0x81f9, 0x81fd,
  0x81ff, 0x8200, 0x8203, 0x820f, 0x8213, 0x8214, 0x8219, 0x821a,
  0x821d, 0x8221, 0x8222, 0x8228, 0x8232, 0x8234, 0x823a, 0x8243,
  0x8244, 0x8245, 0x8246, 0x824b, 0x824e, 0x824f, 0x8251, 0x8256,
  0x825c, 0x8260, 0x8263, 0x8267, 0x826d, 0x8274, 0x827b, 0x827d,
  0x827f, 0x8280, 0x8281, 0x8283, 0x8284, 0x8287, 0x8289, 0x828a,
  0x828e, 0x8291, 0x8294, 0x8296, 0x8298, 0x829a, 0x829b, 0x82a0,
  0x82a1, 0x82a3, 0x82a4, 0x82a7, 0x82a8, 0x82a9, 0x82aa, 0x82ae,
  0x82b0, 0x82b2, 0x82b4, 0x82b7, 0x82ba, 0x82bc, 0x82be, 0x82bf,
  0x82c6, 0x82d0, 0x82d5, 0x82da, 0x82e0, 0x82e2, 0x82e4, 0x82e8,
  0x82ea, 0x82ed, 0x82ef, 0x82f6, 0x82f7, 0x82fd, 0x82fe, 0x8300,
  0x8301, 0x8307, 0x8308, 0x830a, 0x830b, 0x8354, 0x831b, 0x831d,
  0x831e, 0x831f, 0x8321, 0x8322, 0x832c, 0x832d, 0x832e, 0x8330,
  0x8333, 0x8337, 0x833a, 0x833c, 0x833d, 0x8342, 0x8343, 0x8344,
  0x8347, 0x834d, 0x834e, 0x8351, 0x8355, 0x8356, 0x8357, 0x8370,
  0x8378, 0x837d, 0x837f, 0x8380, 0x8382, 0x8384, 0x8386, 0x838d,
  0x8392, 0x8394, 0x8395, 0x8398, 0x8399, 0x839b, 0x839c, 0x839d,
  0x83a6, 0x83a7, 0x83a9, 0x83ac, 0x83be, 0x83bf, 0x83c0, 0x83c7,
  0x83c9, 0x83cf, 0x83d0, 0x83d1, 0x83d4, 0x83dd, 0x8353, 0x83e8,
  0x83ea, 0x83f6, 0x83f8, 0x83f9, 0x83fc, 0x8401, 0x8406, 0x840a,
  0x840f, 0x8411, 0x8415, 0x8419, 0x83ad, 0x842f, 0x8439, 0x8445,
  0x8447, 0x8448, 0x844a, 0x844d, 0x844f, 0x8451, 0x8452, 0x8456,
  0x8458, 0x8459, 0x845a, 0x845c, 0x8460, 0x8464, 0x8465, 0x8467,
  0x846a, 0x8470, 0x8473, 0x8474, 0x8476, 0x8478, 0x847c, 0x847d,
  0x8481, 0x8485, 0x8492, 0x8493, 0x8495, 0x849e, 0x84a6, 0x84a8,
  0x84a9, 0x84aa, 0x84af, 0x84b1, 0x84b4, 0x84ba, 0x84bd, 0x84be,
  0x84c0, 0x84c2, 0x84c7, 0x84c8, 0x84cc, 0x84cf, 0x84d3, 0x84dc,
  0x84e7, 0x84ea, 0x84ef, 0x84f0, 0x84f1, 0x84f2, 0x84f7, 0x8532,
  0x84fa, 0x84fb, 0x84fd, 0x8502, 0x8503, 0x8507, 0x850c, 0x850e,
  0x8510, 0x851c, 0x851e, 0x8522, 0x8523, 0x8524, 0x8525, 0x8527,
  0x852a, 0x852b, 0x852f, 0x8533, 0x8534, 0x8536, 0x853f, 0x8546,
  0x854f, 0x8550, 0x8551, 0x8552, 0x8553, 0x8556, 0x8559, 0x855c,
  0x855d, 0x855e, 0x855f, 0x8560, 0x8561, 0x8562, 0x8564, 0x856b,
  0x856f, 0x8579, 0x857a, 0x857b, 0x857d, 0x857f, 0x8581, 0x8585,
  0x8586, 0x8589, 0x858b, 0x858c, 0x858f, 0x8593, 0x8598, 0x859d,
  0x859f, 0x85a0, 0x85a2, 0x85a5, 0x85a7, 0x85b4, 0x85b6, 0x85b7,
  0x85b8, 0x85bc, 0x85bd, 0x85be, 0x85bf, 0x85c2, 0x85c7, 0x85ca,
  0x85cb, 0x85ce, 0x85ad, 0x85d8, 0x85da, 0x85df, 0x85e0, 0x85e6,
  0x85e8, 0x85ed, 0x85f3, 0x85f6, 0x85fc, 0x85ff, 0x8600, 0x8604,
  0x8605, 0x860d, 0x860e, 0x8610, 0x8611, 0x8612, 0x8618, 0x8619,
  0x861b, 0x861e, 0x8621, 0x8627, 0x8629, 0x8636, 0x8638, 0x863a,
  0x863c, 0x863d, 0x8640, 0x8642, 0x8646, 0x8652, 0x8653, 0x8656,
  0x8657, 0x8658, 0x8659, 0x865d, 0x8660, 0x8661, 0x8662, 0x8663,
  0x8664, 0x8669, 0x866c, 0x866f, 0x8675, 0x8676, 0x8677, 0x867a,
  0x868d, 0x8691, 0x8696, 0x8698, 0x869a, 0x869c, 0x86a1, 0x86a6,
  0x86a7, 0x86a8, 0x86ad, 0x86b1, 0x86b3, 0x86b4, 0x86b5, 0x86b7,
  0x86b8, 0x86b9, 0x86bf, 0x86c0, 0x86c1, 0x86c3, 0x86c5, 0x86d1,
  0x86d2, 0x86d5, 0x86d7, 0x86da, 0x86dc, 0x86e0, 0x86e3, 0x86e5,
  0x86e7, 0x8688, 0x86fa, 0x86fc, 0x86fd, 0x8704, 0x8705, 0x8707,
  0x870b, 0x870e, 0x870f, 0x8710, 0x8713, 0x8714, 0x8719, 0x871e,
  0x871f, 0x8721, 0x8723, 0x8728, 0x872e, 0x872f, 0x8731, 0x8732,
  0x8739, 0x873a, 0x873c, 0x873d, 0x873e, 0x8740, 0x8743, 0x8745,
  0x874d, 0x8758, 0x875d, 0x8761, 0x8764, 0x8765, 0x876f, 0x8771,
  0x8772, 0x877b, 0x8783, 0x8784, 0x8785, 0x8786, 0x8787, 0x8788,
  0x8789, 0x878b, 0x878c, 0x8790, 0x8793, 0x8795, 0x8797, 0x8798,
  0x8799, 0x879e, 0x87a0, 0x87a3, 0x87a7, 0x87ac, 0x87ad, 0x87ae,
  0x87b1, 0x87b5, 0x87be, 0x87bf, 0x87c1, 0x87c8, 0x87c9, 0x87ca,
  0x87ce, 0x87d5, 0x87d6, 0x87d9, 0x87da, 0x87dc, 0x87df, 0x87e2,
  0x87e3, 0x87e4, 0x87ea, 0x87eb, 0x87ed, 0x87f1, 0x87f3, 0x87f8,
  0x87fa, 0x87ff, 0x8801, 0x8803, 0x8806, 0x8809, 0x880a, 0x880b,
  0x8810, 0x8819, 0x8812, 0x8813, 0x8814, 0x8818, 0x881a, 0x881b,
  0x881c, 0x881e, 0x881f, 0x8828, 0x882d, 0x882e, 0x8830, 0x8832,
  0x8835, 0x883a, 0x883c, 0x8841, 0x8843, 0x8845, 0x8848, 0x8849,
  0x884a, 0x884b, 0x884e, 0x8851, 0x8855, 0x8856, 0x8858, 0x885a,
  0x885c, 0x885f, 0x8860, 0x8864, 0x8869, 0x8871, 0x8879, 0x887b,
  0x8880, 0x8898, 0x889a, 0x889b, 0x889c, 0x889f, 0x88a0, 0x88a8,
  0x88aa, 0x88ba, 0x88bd, 0x88be, 0x88c0, 0x88ca, 0x88cb, 0x88cc,
  0x88cd, 0x88ce, 0x88d1, 0x88d2, 0x88d3, 0x88db, 0x88de, 0x88e7,
  0x88ef, 0x88f0, 0x88f1, 0x88f5, 0x88f7, 0x8901, 0x8906, 0x890d,
  0x890e, 0x890f, 0x8915, 0x8916, 0x8918, 0x8919, 0x891a, 0x891c,
  0x8920, 0x8926, 0x8927, 0x8928, 0x8930, 0x8931, 0x8932, 0x8935,
  0x8939, 0x893a, 0x893e, 0x8940, 0x8942, 0x8945, 0x8946, 0x8949,
  0x894f, 0x8952, 0x8957, 0x895a, 0x895b, 0x895c, 0x8961, 0x8962,
  0x8963, 0x896b, 0x896e, 0x8970, 0x8973, 0x8975, 0x897a, 0x897b,
  0x897c, 0x897d, 0x8989, 0x898d, 0x8990, 0x8994, 0x8995, 0x899b,
  0x899c, 0x899f, 0x89a0, 0x89a5, 0x89b0, 0x89b4, 0x89b5, 0x89b6,
  0x89b7, 0x89bc, 0x89d4, 0x89d5, 0x89d6, 0x89d7, 0x89d8, 0x89e5,
  0x89e9, 0x89eb, 0x89ed, 0x89f1, 0x89f3, 0x89f6, 0x89f9, 0x89fd,
  0x89ff, 0x8a04, 0x8a05, 0x8a07, 0x8a0f, 0x8a11, 0x8a12, 0x8a14,
  0x8a15, 0x8a1e, 0x8a20, 0x8a22, 0x8a24, 0x8a26, 0x8a2b, 0x8a2c,
  0x8a2f, 0x8a35, 0x8a37, 0x8a3d, 0x8a3e, 0x8a40, 0x8a43, 0x8a45,
  0x8a47, 0x8a49, 0x8a4d, 0x8a4e, 0x8a53, 0x8a56, 0x8a57, 0x8a58,
  0x8a5c, 0x8a5d, 0x8a61, 0x8a65, 0x8a67, 0x8a75, 0x8a76, 0x8a77,
  0x8a79, 0x8a7a, 0x8a7b, 0x8a7e, 0x8a7f, 0x8a80, 0x8a83, 0x8a86,
  0x8a8b, 0x8a8f, 0x8a90, 0x8a92, 0x8a96, 0x8a97, 0x8a99, 0x8a9f,
  0x8aa7, 0x8aa9, 0x8aae, 0x8aaf, 0x8ab3, 0x8ab6, 0x8ab7, 0x8abb,
  0x8abe, 0x8ac3, 0x8ac6, 0x8ac8, 0x8ac9, 0x8aca, 0x8ad1, 0x8ad3,
  0x8ad4, 0x8ad5, 0x8ad7, 0x8add, 0x8adf, 0x8aec, 0x8af0, 0x8af4,
  0x8af5, 0x8af6, 0x8afc, 0x8aff, 0x8b05, 0x8b06, 0x8b0b, 0x8b11,
  0x8b1c, 0x8b1e, 0x8b1f, 0x8b0a, 0x8b2d, 0x8b30, 0x8b37, 0x8b3c,
  0x8b42, 0x8b43, 0x8b44, 0x8b45, 0x8b46, 0x8b48, 0x8b52, 0x8b53,
  0x8b54, 0x8b59, 0x8b4d, 0x8b5e, 0x8b63, 0x8b6d, 0x8b76, 0x8b78,
  0x8b79, 0x8b7c, 0x8b7e, 0x8b81, 0x8b84, 0x8b85, 0x8b8b, 0x8b8d,
  0x8b8f, 0x8b94, 0x8b95, 0x8b9c, 0x8b9e, 0x8b9f, 0x8c38, 0x8c39,
  0x8c3d, 0x8c3e, 0x8c45, 0x8c47, 0x8c49, 0x8c4b, 0x8c4f, 0x8c51,
  0x8c53, 0x8c54, 0x8c57, 0x8c58, 0x8c5b, 0x8c5d, 0x8c59, 0x8c63,
  0x8c64, 0x8c66, 0x8c68, 0x8c69, 0x8c6d, 0x8c73, 0x8c75, 0x8c76,
  0x8c7b, 0x8c7e, 0x8c86, 0x8c87, 0x8c8b, 0x8c90, 0x8c92, 0x8c93,
  0x8c99, 0x8c9b, 0x8c9c, 0x8ca4, 0x8cb9, 0x8cba, 0x8cc5, 0x8cc6,
  0x8cc9, 0x8ccb, 0x8ccf, 0x8cd6, 0x8cd5, 0x8cd9, 0x8cdd, 0x8ce1,
  0x8ce8, 0x8cec, 0x8cef, 0x8cf0, 0x8cf2, 0x8cf5, 0x8cf7, 0x8cf8,
  0x8cfe, 0x8cff, 0x8d01, 0x8d03, 0x8d09, 0x8d12, 0x8d17, 0x8d1b,
  0x8d65, 0x8d69, 0x8d6c, 0x8d6e, 0x8d7f, 0x8d82, 0x8d84, 0x8d88,
  0x8d8d, 0x8d90, 0x8d91, 0x8d95, 0x8d9e, 0x8d9f, 0x8da0, 0x8da6,
  0x8dab, 0x8dac, 0x8daf, 0x8db2, 0x8db5, 0x8db7, 0x8db9, 0x8dbb,
  0x8dc0, 0x8dc5, 0x8dc6, 0x8dc7, 0x8dc8, 0x8dca, 0x8dce, 0x8dd1,
  0x8dd4, 0x8dd5, 0x8dd7, 0x8dd9, 0x8de4, 0x8de5, 0x8de7, 0x8dec,
  0x8df0, 0x8dbc, 0x8df1, 0x8df2, 0x8df4, 0x8dfd, 0x8e01, 0x8e04,
  0x8e05, 0x8e06, 0x8e0b, 0x8e11, 0x8e14, 0x8e16, 0x8e20, 0x8e21,
  0x8e22, 0x8e23, 0x8e26, 0x8e27, 0x8e31, 0x8e33, 0x8e36, 0x8e37,
  0x8e38, 0x8e39, 0x8e3d, 0x8e40, 0x8e41, 0x8e4b, 0x8e4d, 0x8e4e,
  0x8e4f, 0x8e54, 0x8e5b, 0x8e5c, 0x8e5d, 0x8e5e, 0x8e61, 0x8e62,
  0x8e69, 0x8e6c, 0x8e6d, 0x8e6f, 0x8e70, 0x8e71, 0x8e79, 0x8e7a,
  0x8e7b, 0x8e82, 0x8e83, 0x8e89, 0x8e90, 0x8e92, 0x8e95, 0x8e9a,
  0x8e9b, 0x8e9d, 0x8e9e, 0x8ea2, 0x8ea7, 0x8ea9, 0x8ead, 0x8eae,
  0x8eb3, 0x8eb5, 0x8eba, 0x8ebb, 0x8ec0, 0x8ec1, 0x8ec3, 0x8ec4,
  0x8ec7, 0x8ecf, 0x8ed1, 0x8ed4, 0x8edc, 0x8ee8, 0x8eee, 0x8ef0,
  0x8ef1, 0x8ef7, 0x8ef9, 0x8efa, 0x8eed, 0x8f00, 0x8f02, 0x8f07,
  0x8f08, 0x8f0f, 0x8f10, 0x8f16, 0x8f17, 0x8f18, 0x8f1e, 0x8f20,
  0x8f21, 0x8f23, 0x8f25, 0x8f27, 0x8f28, 0x8f2c, 0x8f2d, 0x8f2e,
  0x8f34, 0x8f35, 0x8f36, 0x8f37, 0x8f3a, 0x8f40, 0x8f41, 0x8f43,
  0x8f47, 0x8f4f, 0x8f51, 0x8f52, 0x8f53, 0x8f54, 0x8f55, 0x8f58,
  0x8f5d, 0x8f5e, 0x8f65, 0x8f9d, 0x8fa0, 0x8fa1, 0x8fa4, 0x8fa5,
  0x8fa6, 0x8fb5, 0x8fb6, 0x8fb8, 0x8fbe, 0x8fc0, 0x8fc1, 0x8fc6,
  0x8fca, 0x8fcb, 0x8fcd, 0x8fd0, 0x8fd2, 0x8fd3, 0x8fd5, 0x8fe0,
  0x8fe3, 0x8fe4, 0x8fe8, 0x8fee, 0x8ff1, 0x8ff5, 0x8ff6, 0x8ffb,
  0x8ffe, 0x9002, 0x9004, 0x9008, 0x900c, 0x9018, 0x901b, 0x9028,
  0x9029, 0x902f, 0x902a, 0x902c, 0x902d, 0x9033, 0x9034, 0x9037,
  0x903f, 0x9043, 0x9044, 0x904c, 0x905b, 0x905d, 0x9062, 0x9066,
  0x9067, 0x906c, 0x9070, 0x9074, 0x9079, 0x9085, 0x9088, 0x908b,
  0x908c, 0x908e, 0x9090, 0x9095, 0x9097, 0x9098, 0x9099, 0x909b,
  0x90a0, 0x90a1, 0x90a2, 0x90a5, 0x90b0, 0x90b2, 0x90b3, 0x90b4,
  0x90b6, 0x90bd, 0x90cc, 0x90be, 0x90c3, 0x90c4, 0x90c5, 0x90c7,
  0x90c8, 0x90d5, 0x90d7, 0x90d8, 0x90d9, 0x90dc, 0x90dd, 0x90df,
  0x90e5, 0x90d2, 0x90f6, 0x90eb, 0x90ef, 0x90f0, 0x90f4, 0x90fe,
  0x90ff, 0x9100, 0x9104, 0x9105, 0x9106, 0x9108, 0x910d, 0x9110,
  0x9114, 0x9116, 0x9117, 0x9118, 0x911a, 0x911c, 0x911e, 0x9120,
  0x9125, 0x9122, 0x9123, 0x9127, 0x9129, 0x912e, 0x912f, 0x9131,
  0x9134, 0x9136, 0x9137, 0x9139, 0x913a, 0x913c, 0x913d, 0x9143,
  0x9147, 0x9148, 0x914f, 0x9153, 0x9157, 0x9159, 0x915a, 0x915b,
  0x9161, 0x9164, 0x9167, 0x916d, 0x9174, 0x9179, 0x917a, 0x917b,
  0x9181, 0x9183, 0x9185, 0x9186, 0x918a, 0x918e, 0x9191, 0x9193,
  0x9194, 0x9195, 0x9198, 0x919e, 0x91a1, 0x91a6, 0x91a8, 0x91ac,
  0x91ad, 0x91ae, 0x91b0, 0x91b1, 0x91b2, 0x91b3, 0x91b6, 0x91bb,
  0x91bc, 0x91bd, 0x91bf, 0x91c2, 0x91c3, 0x91c5, 0x91d3, 0x91d4,
  0x91d7, 0x91d9, 0x91da, 0x91de, 0x91e4, 0x91e5, 0x91e9, 0x91ea,
  0x91ec, 0x91ed, 0x91ee, 0x91ef, 0x91f0, 0x91f1, 0x91f7, 0x91f9,
  0x91fb, 0x91fd, 0x9200, 0x9201, 0x9204, 0x9205, 0x9206, 0x9207,
  0x9209, 0x920a, 0x920c, 0x9210, 0x9212, 0x9213, 0x9216, 0x9218,
  0x921c, 0x921d, 0x9223, 0x9224, 0x9225, 0x9226, 0x9228, 0x922e,
  0x922f, 0x9230, 0x9233, 0x9235, 0x9236, 0x9238, 0x9239, 0x923a,
  0x923c, 0x923e, 0x9240, 0x9242, 0x9243, 0x9246, 0x9247, 0x924a,
  0x924d, 0x924e, 0x924f, 0x9251, 0x9258, 0x9259, 0x925c, 0x925d,
  0x9260, 0x9261, 0x9265, 0x9267, 0x9268, 0x9269, 0x926e, 0x926f,
  0x9270, 0x9275, 0x9276, 0x9277, 0x9278, 0x9279, 0x927b, 0x927c,
  0x927d, 0x927f, 0x9288, 0x9289, 0x928a, 0x928d, 0x928e, 0x9292,
  0x9297, 0x9299, 0x929f, 0x92a0, 0x92a4, 0x92a5, 0x92a7, 0x92a8,
  0x92ab, 0x92af, 0x92b2, 0x92b6, 0x92b8, 0x92ba, 0x92bb, 0x92bc,
  0x92bd, 0x92bf, 0x92c0, 0x92c1, 0x92c2, 0x92c3, 0x92c5, 0x92c6,
  0x92c7, 0x92c8, 0x92cb, 0x92cc, 0x92cd, 0x92ce, 0x92d0, 0x92d3,
  0x92d5, 0x92d7, 0x92d8, 0x92d9, 0x92dc, 0x92dd, 0x92df, 0x92e0,
  0x92e1, 0x92e3, 0x92e5, 0x92e7, 0x92e8, 0x92ec, 0x92ee, 0x92f0,
  0x92f9, 0x92fb, 0x92ff, 0x9300, 0x9302, 0x9308, 0x930d, 0x9311,
  0x9314, 0x9315, 0x931c, 0x931d, 0x931e, 0x931f, 0x9321, 0x9324,
  0x9325, 0x9327, 0x9329, 0x932a, 0x9333, 0x9334, 0x9336, 0x9337,
  0x9347, 0x9348, 0x9349, 0x9350, 0x9351, 0x9352, 0x9355, 0x9357,
  0x9358, 0x935a, 0x935e, 0x9364, 0x9365, 0x9367, 0x9369, 0x936a,
  0x936d, 0x936f, 0x9370, 0x9371, 0x9373, 0x9374, 0x9376, 0x937a,
  0x937d, 0x937f, 0x9380, 0x9381, 0x9382, 0x9388, 0x938a, 0x938b,
  0x938d, 0x938f, 0x9392, 0x9395, 0x9398, 0x939b, 0x939e, 0x93a1,
  0x93a3, 0x93a4, 0x93a6, 0x93a8, 0x93ab, 0x93b4, 0x93b5, 0x93b6,
  0x93ba, 0x93a9, 0x93c1, 0x93c4, 0x93c5, 0x93c6, 0x93c7, 0x93c9,
  0x93ca, 0x93cb, 0x93cc, 0x93cd, 0x93d3, 0x93d9, 0x93dc, 0x93de,
  0x93df, 0x93e2, 0x93e6, 0x93e7, 0x93f9, 0x93f7, 0x93f8, 0x93fa,
  0x93fb, 0x93fd, 0x9401, 0x9402, 0x9404, 0x9408, 0x9409, 0x940d,
  0x940e, 0x940f, 0x9415, 0x9416, 0x9417, 0x941f, 0x942e, 0x942f,
  0x9431, 0x9432, 0x9433, 0x9434, 0x943b, 0x943f, 0x943d, 0x9443,
  0x9445, 0x9448, 0x944a, 0x944c, 0x9455, 0x9459, 0x945c, 0x945f,
  0x9461, 0x9463, 0x9468, 0x946b, 0x946d, 0x946e, 0x946f, 0x9471,
  0x9472, 0x9484, 0x9483, 0x9578, 0x9579, 0x957e, 0x9584, 0x9588,
  0x958c, 0x958d, 0x958e, 0x959d, 0x959e, 0x959f, 0x95a1, 0x95a6,
  0x95a9, 0x95ab, 0x95ac, 0x95b4, 0x95b6, 0x95ba, 0x95bd, 0x95bf,
  0x95c6, 0x95c8, 0x95c9, 0x95cb, 0x95d0, 0x95d1, 0x95d2, 0x95d3,
  0x95d9, 0x95da, 0x95dd, 0x95de, 0x95df, 0x95e0, 0x95e4, 0x95e6,
  0x961d, 0x961e, 0x9622, 0x9624, 0x9625, 0x9626, 0x962c, 0x9631,
  0x9633, 0x9637, 0x9638, 0x9639, 0x963a, 0x963c, 0x963d, 0x9641,
  0x9652, 0x9654, 0x9656, 0x9657, 0x9658, 0x9661, 0x966e, 0x9674,
  0x967b, 0x967c, 0x967e, 0x967f, 0x9681, 0x9682, 0x9683, 0x9684,
  0x9689, 0x9691, 0x9696, 0x969a, 0x969d, 0x969f, 0x96a4, 0x96a5,
  0x96a6, 0x96a9, 0x96ae, 0x96af, 0x96b3, 0x96ba, 0x96ca, 0x96d2,
  0x5db2, 0x96d8, 0x96da, 0x96dd, 0x96de, 0x96df, 0x96e9, 0x96ef,
  0x96f1, 0x96fa, 0x9702, 0x9703, 0x9705, 0x9709, 0x971a, 0x971b,
  0x971d, 0x9721, 0x9722, 0x9723, 0x9728, 0x9731, 0x9733, 0x9741,
  0x9743, 0x974a, 0x974e, 0x974f, 0x9755, 0x9757, 0x9758, 0x975a,
  0x975b, 0x9763, 0x9767, 0x976a, 0x976e, 0x9773, 0x9776, 0x9777,
  0x9778, 0x977b, 0x977d, 0x977f, 0x9780, 0x9789, 0x9795, 0x9796,
  0x9797, 0x9799, 0x979a, 0x979e, 0x979f, 0x97a2, 0x97ac, 0x97ae,
  0x97b1, 0x97b2, 0x97b5, 0x97b6, 0x97b8, 0x97b9, 0x97ba, 0x97bc,
  0x97be, 0x97bf, 0x97c1, 0x97c4, 0x97c5, 0x97c7, 0x97c9, 0x97ca,
  0x97cc, 0x97cd, 0x97ce, 0x97d0, 0x97d1, 0x97d4, 0x97d7, 0x97d8,
  0x97d9, 0x97dd, 0x97de, 0x97e0, 0x97db, 0x97e1, 0x97e4, 0x97ef,
  0x97f1, 0x97f4, 0x97f7, 0x97f8, 0x97fa, 0x9807, 0x980a, 0x9819,
  0x980d, 0x980e, 0x9814, 0x9816, 0x981c, 0x981e, 0x9820, 0x9823,
  0x9826, 0x982b, 0x982e, 0x982f, 0x9830, 0x9832, 0x9833, 0x9835,
  0x9825, 0x983e, 0x9844, 0x9847, 0x984a, 0x9851, 0x9852, 0x9853,
  0x9856, 0x9857, 0x9859, 0x985a, 0x9862, 0x9863, 0x9865, 0x9866,
  0x986a, 0x986c, 0x98ab, 0x98ad, 0x98ae, 0x98b0, 0x98b4, 0x98b7,
  0x98b8, 0x98ba, 0x98bb, 0x98bf, 0x98c2, 0x98c5, 0x98c8, 0x98cc,
  0x98e1, 0x98e3, 0x98e5, 0x98e6, 0x98e7, 0x98ea, 0x98f3, 0x98f6,
  0x9902, 0x9907, 0x9908, 0x9911, 0x9915, 0x9916, 0x9917, 0x991a,
  0x991b, 0x991c, 0x991f, 0x9922, 0x9926, 0x9927, 0x992b, 0x9931,
  0x9932, 0x9933, 0x9934, 0x9935, 0x9939, 0x993a, 0x993b, 0x993c,
  0x9940, 0x9941, 0x9946, 0x9947, 0x9948, 0x994d, 0x994e, 0x9954,
  0x9958, 0x9959, 0x995b, 0x995c, 0x995e, 0x995f, 0x9960, 0x999b,
  0x999d, 0x999f, 0x99a6, 0x99b0, 0x99b1, 0x99b2, 0x99b5, 0x99b9,
  0x99ba, 0x99bd, 0x99bf, 0x99c3, 0x99c9, 0x99d3, 0x99d4, 0x99d9,
  0x99da, 0x99dc, 0x99de, 0x99e7, 0x99ea, 0x99eb, 0x99ec, 0x99f0,
  0x99f4, 0x99f5, 0x99f9, 0x99fd, 0x99fe, 0x9a02, 0x9a03, 0x9a04,
  0x9a0b, 0x9a0c, 0x9a10, 0x9a11, 0x9a16, 0x9a1e, 0x9a20, 0x9a22,
  0x9a23, 0x9a24, 0x9a27, 0x9a2d, 0x9a2e, 0x9a33, 0x9a35, 0x9a36,
  0x9a38, 0x9a47, 0x9a41, 0x9a44, 0x9a4a, 0x9a4b, 0x9a4c, 0x9a4e,
  0x9a51, 0x9a54, 0x9a56, 0x9a5d, 0x9aaa, 0x9aac, 0x9aae, 0x9aaf,
  0x9ab2, 0x9ab4, 0x9ab5, 0x9ab6, 0x9ab9, 0x9abb, 0x9abe, 0x9abf,
  0x9ac1, 0x9ac3, 0x9ac6, 0x9ac8, 0x9ace, 0x9ad0, 0x9ad2, 0x9ad5,
  0x9ad6, 0x9ad7, 0x9adb, 0x9adc, 0x9ae0, 0x9ae4, 0x9ae5, 0x9ae7,
  0x9ae9, 0x9aec, 0x9af2, 0x9af3, 0x9af5, 0x9af9, 0x9afa, 0x9afd,
  0x9aff, 0x9b00, 0x9b01, 0x9b02, 0x9b03, 0x9b04, 0x9b05, 0x9b08,
  0x9b09, 0x9b0b, 0x9b0c, 0x9b0d, 0x9b0e, 0x9b10, 0x9b12, 0x9b16,
  0x9b19, 0x9b1b, 0x9b1c, 0x9b20, 0x9b26, 0x9b2b, 0x9b2d, 0x9b33,
  0x9b34, 0x9b35, 0x9b37, 0x9b39, 0x9b3a, 0x9b3d, 0x9b48, 0x9b4b,
  0x9b4c, 0x9b55, 0x9b56, 0x9b57, 0x9b5b, 0x9b5e, 0x9b61, 0x9b63,
  0x9b65, 0x9b66, 0x9b68, 0x9b6a, 0x9b6b, 0x9b6c, 0x9b6d, 0x9b6e,
  0x9b73, 0x9b75, 0x9b77, 0x9b78, 0x9b79, 0x9b7f, 0x9b80, 0x9b84,
  0x9b85, 0x9b86, 0x9b87, 0x9b89, 0x9b8a, 0x9b8b, 0x9b8d, 0x9b8f,
  0x9b90, 0x9b94, 0x9b9a, 0x9b9d, 0x9b9e, 0x9ba6, 0x9ba7, 0x9ba9,
  0x9bac, 0x9bb0, 0x9bb1, 0x9bb2, 0x9bb7, 0x9bb8, 0x9bbb, 0x9bbc,
  0x9bbe, 0x9bbf, 0x9bc1, 0x9bc7, 0x9bc8, 0x9bce, 0x9bd0, 0x9bd7,
  0x9bd8, 0x9bdd, 0x9bdf, 0x9be5, 0x9be7, 0x9bea, 0x9beb, 0x9bef,
  0x9bf3, 0x9bf7, 0x9bf8, 0x9bf9, 0x9bfa, 0x9bfd, 0x9bff, 0x9c00,
  0x9c02, 0x9c0b, 0x9c0f, 0x9c11, 0x9c16, 0x9c18, 0x9c19, 0x9c1a,
  0x9c1c, 0x9c1e, 0x9c22, 0x9c23, 0x9c26, 0x9c27, 0x9c28, 0x9c29,
  0x9c2a, 0x9c31, 0x9c35, 0x9c36, 0x9c37, 0x9c3d, 0x9c41, 0x9c43,
  0x9c44, 0x9c45, 0x9c49, 0x9c4a, 0x9c4e, 0x9c4f, 0x9c50, 0x9c53,
  0x9c54, 0x9c56, 0x9c58, 0x9c5b, 0x9c5d, 0x9c5e, 0x9c5f, 0x9c63,
  0x9c69, 0x9c6a, 0x9c5c, 0x9c6b, 0x9c68, 0x9c6e, 0x9c70, 0x9c72,
  0x9c75, 0x9c77, 0x9c7b, 0x9ce6, 0x9cf2, 0x9cf7, 0x9cf9, 0x9d0b,
  0x9d02, 0x9d11, 0x9d17, 0x9d18, 0x9d1c, 0x9d1d, 0x9d1e, 0x9d2f,
  0x9d30, 0x9d32, 0x9d33, 0x9d34, 0x9d3a, 0x9d3c, 0x9d45, 0x9d3d,
  0x9d42, 0x9d43, 0x9d47, 0x9d4a, 0x9d53, 0x9d54, 0x9d5f, 0x9d63,
  0x9d62, 0x9d65, 0x9d69, 0x9d6a, 0x9d6b, 0x9d70, 0x9d76, 0x9d77,
  0x9d7b, 0x9d7c, 0x9d7e, 0x9d83, 0x9d84, 0x9d86, 0x9d8a, 0x9d8d,
  0x9d8e, 0x9d92, 0x9d93, 0x9d95, 0x9d96, 0x9d97, 0x9d98, 0x9da1,
  0x9daa, 0x9dac, 0x9dae, 0x9db1, 0x9db5, 0x9db9, 0x9dbc, 0x9dbf,
  0x9dc3, 0x9dc7, 0x9dc9, 0x9dca, 0x9dd4, 0x9dd5, 0x9dd6, 0x9dd7,
  0x9dda, 0x9dde, 0x9ddf, 0x9de0, 0x9de5, 0x9de7, 0x9de9, 0x9deb,
  0x9dee, 0x9df0, 0x9df3, 0x9df4, 0x9dfe, 0x9e0a, 0x9e02, 0x9e07,
  0x9e0e, 0x9e10, 0x9e11, 0x9e12, 0x9e15, 0x9e16, 0x9e19, 0x9e1c,
  0x9e1d, 0x9e7a, 0x9e7b, 0x9e7c, 0x9e80, 0x9e82, 0x9e83, 0x9e84,
  0x9e85, 0x9e87, 0x9e8e, 0x9e8f, 0x9e96, 0x9e98, 0x9e9b, 0x9e9e,
  0x9ea4, 0x9ea8, 0x9eac, 0x9eae, 0x9eaf, 0x9eb0, 0x9eb3, 0x9eb4,
  0x9eb5, 0x9ec6, 0x9ec8, 0x9ecb, 0x9ed5, 0x9edf, 0x9ee4, 0x9ee7,
  0x9eec, 0x9eed, 0x9eee, 0x9ef0, 0x9ef1, 0x9ef2, 0x9ef5, 0x9ef8,
  0x9eff, 0x9f02, 0x9f03, 0x9f09, 0x9f0f, 0x9f10, 0x9f11, 0x9f12,
  0x9f14, 0x9f16, 0x9f17, 0x9f19, 0x9f1a, 0x9f1b, 0x9f1f, 0x9f22,
  0x9f26, 0x9f2a, 0x9f2b, 0x9f2f, 0x9f31, 0x9f32, 0x9f34, 0x9f37,
  0x9f39, 0x9f3a, 0x9f3c, 0x9f3d, 0x9f3f, 0x9f41, 0x9f43, 0x9f44,
  0x9f45, 0x9f46, 0x9f47, 0x9f53, 0x9f55, 0x9f56, 0x9f57, 0x9f58,
  0x9f5a, 0x9f5d, 0x9f5e, 0x9f68, 0x9f69, 0x9f6d, 0x9f6e, 0x9f6f,
  0x9f70, 0x9f71, 0x9f73, 0x9f75, 0x9f7a, 0x9f7d, 0x9f8f, 0x9f90,
  0x9f91, 0x9f92, 0x9f94, 0x9f96, 0x9f97, 0x9f9e, 0x9fa1, 0x9fa2,
  0x9fa3, 0x9fa5
};


/* The UCS char covered by JIS X0212 are sparsely distributed.  So we
   use again a two-step mapping table.  The index table can be computed
   using

   egrep '^0x[[:xdigit:]][[:xdigit:]][[:xdigit:]][[:xdigit:]]' \
     ../eastasia/jis/jis0212.txt | awk '{ print $2, $1 }' | sort -u |
   perl tab.pl

   where tab.pl is

   ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
   $first=$last=$idx=0;
   while (<>) {
     local($ucs,$jis,%rest) = split;
     local($u)=hex($ucs);
     local($j)=hex($jis);
     if ($u - $last > 6) {
       if ($last != 0) {
	 printf ("  { start: %#06x, end: %#06x, idx: %5d },\n",
		 $first, $last, $idx);
	 $idx += $last - $first + 1;
       }
       $first=$u;
     }
     $last=$u;
   }
   printf ("  { start: %#06x, end: %#06x, idx: %5d }, \n",
	   $first, $last, $idx);
   ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
*/
const struct jisx0212_idx jisx0212_from_ucs_idx[] =
{
  { start: 0x007e, end: 0x007e, idx:     0 },
  { start: 0x00a1, end: 0x00af, idx:     1 },
  { start: 0x00b8, end: 0x017e, idx:    16 },
  { start: 0x01cd, end: 0x01dc, idx:   215 },
  { start: 0x01f5, end: 0x01f5, idx:   231 },
  { start: 0x02c7, end: 0x02c7, idx:   232 },
  { start: 0x02d8, end: 0x02dd, idx:   233 },
  { start: 0x0384, end: 0x0390, idx:   239 },
  { start: 0x03aa, end: 0x03b0, idx:   252 },
  { start: 0x03c2, end: 0x03c2, idx:   259 },
  { start: 0x03ca, end: 0x03ce, idx:   260 },
  { start: 0x0402, end: 0x040f, idx:   265 },
  { start: 0x0452, end: 0x045f, idx:   279 },
  { start: 0x2116, end: 0x2116, idx:   293 },
  { start: 0x2122, end: 0x2122, idx:   294 },
  { start: 0x4e02, end: 0x4e05, idx:   295 },
  { start: 0x4e0c, end: 0x4e12, idx:   299 },
  { start: 0x4e1f, end: 0x4e35, idx:   306 },
  { start: 0x4e40, end: 0x4e47, idx:   329 },
  { start: 0x4e51, end: 0x4e51, idx:   337 },
  { start: 0x4e5a, end: 0x4e5c, idx:   338 },
  { start: 0x4e63, end: 0x4e69, idx:   341 },
  { start: 0x4e74, end: 0x4e7f, idx:   348 },
  { start: 0x4e8d, end: 0x4e8d, idx:   360 },
  { start: 0x4e96, end: 0x4e9d, idx:   361 },
  { start: 0x4eaf, end: 0x4eaf, idx:   369 },
  { start: 0x4eb9, end: 0x4eb9, idx:   370 },
  { start: 0x4ec3, end: 0x4ec3, idx:   371 },
  { start: 0x4ed0, end: 0x4ed0, idx:   372 },
  { start: 0x4eda, end: 0x4ee8, idx:   373 },
  { start: 0x4eef, end: 0x4ef5, idx:   388 },
  { start: 0x4efd, end: 0x4f19, idx:   395 },
  { start: 0x4f2e, end: 0x4f58, idx:   424 },
  { start: 0x4f5f, end: 0x4f63, idx:   467 },
  { start: 0x4f6a, end: 0x4f9f, idx:   472 },
  { start: 0x4fb2, end: 0x4fd2, idx:   526 },
  { start: 0x4fdc, end: 0x4fe2, idx:   559 },
  { start: 0x4ff0, end: 0x4ff2, idx:   566 },
  { start: 0x4ffc, end: 0x5027, idx:   569 },
  { start: 0x502e, end: 0x5071, idx:   613 },
  { start: 0x5081, end: 0x50a2, idx:   681 },
  { start: 0x50aa, end: 0x50b0, idx:   715 },
  { start: 0x50b9, end: 0x5110, idx:   722 },
  { start: 0x5117, end: 0x5139, idx:   810 },
  { start: 0x5142, end: 0x5142, idx:   845 },
  { start: 0x514a, end: 0x5158, idx:   846 },
  { start: 0x515f, end: 0x5166, idx:   861 },
  { start: 0x517e, end: 0x5184, idx:   869 },
  { start: 0x518b, end: 0x518e, idx:   876 },
  { start: 0x5198, end: 0x51a3, idx:   880 },
  { start: 0x51ad, end: 0x51ad, idx:   892 },
  { start: 0x51b8, end: 0x51c8, idx:   893 },
  { start: 0x51cf, end: 0x51e5, idx:   910 },
  { start: 0x51ee, end: 0x51f7, idx:   933 },
  { start: 0x5201, end: 0x5205, idx:   943 },
  { start: 0x5212, end: 0x5218, idx:   948 },
  { start: 0x5222, end: 0x5228, idx:   955 },
  { start: 0x5231, end: 0x5235, idx:   962 },
  { start: 0x523c, end: 0x523c, idx:   967 },
  { start: 0x5245, end: 0x5249, idx:   968 },
  { start: 0x5255, end: 0x5266, idx:   973 },
  { start: 0x526e, end: 0x526e, idx:   991 },
  { start: 0x5277, end: 0x5279, idx:   992 },
  { start: 0x5280, end: 0x528c, idx:   995 },
  { start: 0x5293, end: 0x529c, idx:  1008 },
  { start: 0x52a4, end: 0x52a7, idx:  1018 },
  { start: 0x52af, end: 0x52f7, idx:  1022 },
  { start: 0x5300, end: 0x5303, idx:  1095 },
  { start: 0x530a, end: 0x5335, idx:  1099 },
  { start: 0x533c, end: 0x5342, idx:  1143 },
  { start: 0x534b, end: 0x534c, idx:  1150 },
  { start: 0x5359, end: 0x5365, idx:  1152 },
  { start: 0x536c, end: 0x5372, idx:  1165 },
  { start: 0x5379, end: 0x53c5, idx:  1172 },
  { start: 0x53cf, end: 0x53e7, idx:  1249 },
  { start: 0x53f5, end: 0x53f5, idx:  1274 },
  { start: 0x5402, end: 0x5402, idx:  1275 },
  { start: 0x5413, end: 0x5413, idx:  1276 },
  { start: 0x541a, end: 0x541a, idx:  1277 },
  { start: 0x5421, end: 0x5435, idx:  1278 },
  { start: 0x5443, end: 0x544f, idx:  1299 },
  { start: 0x545e, end: 0x5474, idx:  1312 },
  { start: 0x547f, end: 0x54bf, idx:  1335 },
  { start: 0x54c6, end: 0x54ce, idx:  1400 },
  { start: 0x54e0, end: 0x54e0, idx:  1409 },
  { start: 0x54ea, end: 0x54ef, idx:  1410 },
  { start: 0x54f6, end: 0x550e, idx:  1416 },
  { start: 0x5515, end: 0x5515, idx:  1441 },
  { start: 0x552a, end: 0x552b, idx:  1442 },
  { start: 0x5532, end: 0x5551, idx:  1444 },
  { start: 0x5558, end: 0x5566, idx:  1476 },
  { start: 0x557f, end: 0x5597, idx:  1491 },
  { start: 0x55a3, end: 0x55a4, idx:  1516 },
  { start: 0x55ad, end: 0x55b2, idx:  1518 },
  { start: 0x55bf, end: 0x55e2, idx:  1524 },
  { start: 0x55e9, end: 0x55e9, idx:  1560 },
  { start: 0x55f6, end: 0x55f6, idx:  1561 },
  { start: 0x55ff, end: 0x5612, idx:  1562 },
  { start: 0x5619, end: 0x5619, idx:  1582 },
  { start: 0x562c, end: 0x5654, idx:  1583 },
  { start: 0x565e, end: 0x5675, idx:  1624 },
  { start: 0x5684, end: 0x568c, idx:  1648 },
  { start: 0x5695, end: 0x569f, idx:  1657 },
  { start: 0x56a6, end: 0x56b7, idx:  1668 },
  { start: 0x56be, end: 0x56be, idx:  1686 },
  { start: 0x56c5, end: 0x56d0, idx:  1687 },
  { start: 0x56d9, end: 0x56f7, idx:  1699 },
  { start: 0x5701, end: 0x5734, idx:  1730 },
  { start: 0x573d, end: 0x5752, idx:  1782 },
  { start: 0x5762, end: 0x5783, idx:  1804 },
  { start: 0x578c, end: 0x578c, idx:  1838 },
  { start: 0x5794, end: 0x57ac, idx:  1839 },
  { start: 0x57b8, end: 0x57bd, idx:  1864 },
  { start: 0x57c7, end: 0x57d5, idx:  1870 },
  { start: 0x57dd, end: 0x580d, idx:  1885 },
  { start: 0x581b, end: 0x5832, idx:  1934 },
  { start: 0x5839, end: 0x583f, idx:  1958 },
  { start: 0x5849, end: 0x5855, idx:  1965 },
  { start: 0x585f, end: 0x5868, idx:  1978 },
  { start: 0x5878, end: 0x5896, idx:  1988 },
  { start: 0x589d, end: 0x58a9, idx:  2019 },
  { start: 0x58b1, end: 0x58b2, idx:  2032 },
  { start: 0x58bc, end: 0x58e2, idx:  2034 },
  { start: 0x58e9, end: 0x58e9, idx:  2073 },
  { start: 0x58f3, end: 0x58f3, idx:  2074 },
  { start: 0x5905, end: 0x5914, idx:  2075 },
  { start: 0x591d, end: 0x5928, idx:  2091 },
  { start: 0x592f, end: 0x5936, idx:  2103 },
  { start: 0x593f, end: 0x5946, idx:  2111 },
  { start: 0x5952, end: 0x5963, idx:  2119 },
  { start: 0x596b, end: 0x597c, idx:  2137 },
  { start: 0x598b, end: 0x5997, idx:  2155 },
  { start: 0x599f, end: 0x59d2, idx:  2168 },
  { start: 0x59dd, end: 0x59e7, idx:  2220 },
  { start: 0x59ee, end: 0x59f7, idx:  2231 },
  { start: 0x5a00, end: 0x5a04, idx:  2241 },
  { start: 0x5a0c, end: 0x5a13, idx:  2246 },
  { start: 0x5a1e, end: 0x5a30, idx:  2254 },
  { start: 0x5a44, end: 0x5a55, idx:  2273 },
  { start: 0x5a5e, end: 0x5a6d, idx:  2291 },
  { start: 0x5a77, end: 0x5a7e, idx:  2307 },
  { start: 0x5a8b, end: 0x5ac8, idx:  2315 },
  { start: 0x5acf, end: 0x5acf, idx:  2377 },
  { start: 0x5ada, end: 0x5aee, idx:  2378 },
  { start: 0x5af5, end: 0x5af6, idx:  2399 },
  { start: 0x5afd, end: 0x5b01, idx:  2401 },
  { start: 0x5b08, end: 0x5b08, idx:  2406 },
  { start: 0x5b17, end: 0x5b25, idx:  2407 },
  { start: 0x5b2d, end: 0x5b2d, idx:  2422 },
  { start: 0x5b34, end: 0x5b38, idx:  2423 },
  { start: 0x5b41, end: 0x5b41, idx:  2428 },
  { start: 0x5b4b, end: 0x5b56, idx:  2429 },
  { start: 0x5b5e, end: 0x5b5e, idx:  2441 },
  { start: 0x5b68, end: 0x5b6f, idx:  2442 },
  { start: 0x5b7c, end: 0x5b96, idx:  2450 },
  { start: 0x5ba8, end: 0x5bc1, idx:  2477 },
  { start: 0x5bcd, end: 0x5bcf, idx:  2503 },
  { start: 0x5bd6, end: 0x5be0, idx:  2506 },
  { start: 0x5bef, end: 0x5bf4, idx:  2517 },
  { start: 0x5bfd, end: 0x5bfd, idx:  2523 },
  { start: 0x5c0c, end: 0x5c0c, idx:  2524 },
  { start: 0x5c17, end: 0x5c17, idx:  2525 },
  { start: 0x5c1e, end: 0x5c36, idx:  2526 },
  { start: 0x5c59, end: 0x5c7d, idx:  2551 },
  { start: 0x5c87, end: 0x5c92, idx:  2588 },
  { start: 0x5c9d, end: 0x5caa, idx:  2600 },
  { start: 0x5cb2, end: 0x5cba, idx:  2614 },
  { start: 0x5cc9, end: 0x5ccb, idx:  2623 },
  { start: 0x5cd2, end: 0x5cdd, idx:  2626 },
  { start: 0x5cee, end: 0x5cf4, idx:  2638 },
  { start: 0x5d01, end: 0x5d06, idx:  2645 },
  { start: 0x5d0d, end: 0x5d12, idx:  2651 },
  { start: 0x5d23, end: 0x5d4a, idx:  2657 },
  { start: 0x5d51, end: 0x5d70, idx:  2697 },
  { start: 0x5d79, end: 0x5d8a, idx:  2729 },
  { start: 0x5d92, end: 0x5da0, idx:  2747 },
  { start: 0x5da7, end: 0x5db9, idx:  2762 },
  { start: 0x5dc3, end: 0x5dd0, idx:  2781 },
  { start: 0x5dd8, end: 0x5dd9, idx:  2795 },
  { start: 0x5de0, end: 0x5de9, idx:  2797 },
  { start: 0x5df8, end: 0x5df9, idx:  2807 },
  { start: 0x5e00, end: 0x5e00, idx:  2809 },
  { start: 0x5e07, end: 0x5e18, idx:  2810 },
  { start: 0x5e1f, end: 0x5e20, idx:  2828 },
  { start: 0x5e28, end: 0x5e35, idx:  2830 },
  { start: 0x5e3e, end: 0x5e3e, idx:  2844 },
  { start: 0x5e49, end: 0x5e5e, idx:  2845 },
  { start: 0x5e68, end: 0x5e70, idx:  2867 },
  { start: 0x5e80, end: 0x5e80, idx:  2876 },
  { start: 0x5e8b, end: 0x5e8e, idx:  2877 },
  { start: 0x5ea2, end: 0x5eb3, idx:  2881 },
  { start: 0x5ebd, end: 0x5ebf, idx:  2899 },
  { start: 0x5ec6, end: 0x5ed5, idx:  2902 },
  { start: 0x5edc, end: 0x5ede, idx:  2918 },
  { start: 0x5ee5, end: 0x5eeb, idx:  2921 },
  { start: 0x5f02, end: 0x5f0e, idx:  2928 },
  { start: 0x5f19, end: 0x5f67, idx:  2941 },
  { start: 0x5f6f, end: 0x5f7e, idx:  3020 },
  { start: 0x5f89, end: 0x5f8f, idx:  3036 },
  { start: 0x5f96, end: 0x5fb1, idx:  3043 },
  { start: 0x5fb8, end: 0x5fb8, idx:  3071 },
  { start: 0x5fc4, end: 0x5fd4, idx:  3072 },
  { start: 0x5fde, end: 0x5ffc, idx:  3089 },
  { start: 0x6007, end: 0x6024, idx:  3120 },
  { start: 0x602d, end: 0x6035, idx:  3150 },
  { start: 0x6040, end: 0x6040, idx:  3159 },
  { start: 0x6047, end: 0x6067, idx:  3160 },
  { start: 0x6071, end: 0x6071, idx:  3193 },
  { start: 0x607e, end: 0x60a8, idx:  3194 },
  { start: 0x60b0, end: 0x60e5, idx:  3237 },
  { start: 0x60f2, end: 0x6122, idx:  3291 },
  { start: 0x612a, end: 0x6139, idx:  3340 },
  { start: 0x6141, end: 0x6149, idx:  3356 },
  { start: 0x615e, end: 0x6160, idx:  3365 },
  { start: 0x616c, end: 0x6184, idx:  3368 },
  { start: 0x618b, end: 0x61ad, idx:  3393 },
  { start: 0x61b8, end: 0x61c2, idx:  3428 },
  { start: 0x61ce, end: 0x61d5, idx:  3439 },
  { start: 0x61dc, end: 0x61ef, idx:  3447 },
  { start: 0x6201, end: 0x6207, idx:  3467 },
  { start: 0x6213, end: 0x6215, idx:  3474 },
  { start: 0x621c, end: 0x622b, idx:  3477 },
  { start: 0x6239, end: 0x625c, idx:  3493 },
  { start: 0x6264, end: 0x6264, idx:  3529 },
  { start: 0x626d, end: 0x6273, idx:  3530 },
  { start: 0x627a, end: 0x627d, idx:  3537 },
  { start: 0x628d, end: 0x6290, idx:  3541 },
  { start: 0x62a6, end: 0x62a8, idx:  3545 },
  { start: 0x62b3, end: 0x62c4, idx:  3548 },
  { start: 0x62ce, end: 0x62ce, idx:  3566 },
  { start: 0x62d5, end: 0x62da, idx:  3567 },
  { start: 0x62ea, end: 0x62ea, idx:  3573 },
  { start: 0x62f2, end: 0x62f4, idx:  3574 },
  { start: 0x62fc, end: 0x6318, idx:  3577 },
  { start: 0x6329, end: 0x632d, idx:  3606 },
  { start: 0x6335, end: 0x635b, idx:  3611 },
  { start: 0x6365, end: 0x63a6, idx:  3650 },
  { start: 0x63ad, end: 0x63af, idx:  3716 },
  { start: 0x63bd, end: 0x63d5, idx:  3719 },
  { start: 0x63dc, end: 0x63f9, idx:  3744 },
  { start: 0x6409, end: 0x6435, idx:  3774 },
  { start: 0x643d, end: 0x643f, idx:  3819 },
  { start: 0x644b, end: 0x6463, idx:  3822 },
  { start: 0x646d, end: 0x6474, idx:  3847 },
  { start: 0x647b, end: 0x647d, idx:  3855 },
  { start: 0x6485, end: 0x6487, idx:  3858 },
  { start: 0x648f, end: 0x6491, idx:  3861 },
  { start: 0x6498, end: 0x64ac, idx:  3864 },
  { start: 0x64b3, end: 0x64b3, idx:  3885 },
  { start: 0x64bd, end: 0x64d7, idx:  3886 },
  { start: 0x64e4, end: 0x6531, idx:  3913 },
  { start: 0x653a, end: 0x6549, idx:  3991 },
  { start: 0x6550, end: 0x6554, idx:  4007 },
  { start: 0x655f, end: 0x6560, idx:  4012 },
  { start: 0x6567, end: 0x656b, idx:  4014 },
  { start: 0x657a, end: 0x658a, idx:  4019 },
  { start: 0x6592, end: 0x65a6, idx:  4036 },
  { start: 0x65ae, end: 0x65b4, idx:  4057 },
  { start: 0x65bf, end: 0x65d8, idx:  4064 },
  { start: 0x65df, end: 0x65df, idx:  4090 },
  { start: 0x65f0, end: 0x6616, idx:  4091 },
  { start: 0x661d, end: 0x6639, idx:  4130 },
  { start: 0x6640, end: 0x6661, idx:  4159 },
  { start: 0x666a, end: 0x666c, idx:  4193 },
  { start: 0x6673, end: 0x6680, idx:  4196 },
  { start: 0x668b, end: 0x6692, idx:  4210 },
  { start: 0x6699, end: 0x66a4, idx:  4218 },
  { start: 0x66ad, end: 0x66d4, idx:  4230 },
  { start: 0x66db, end: 0x66df, idx:  4270 },
  { start: 0x66e8, end: 0x66ee, idx:  4275 },
  { start: 0x66fa, end: 0x66fb, idx:  4282 },
  { start: 0x6705, end: 0x6707, idx:  4284 },
  { start: 0x670e, end: 0x6722, idx:  4287 },
  { start: 0x6733, end: 0x6733, idx:  4308 },
  { start: 0x673e, end: 0x673e, idx:  4309 },
  { start: 0x6745, end: 0x674c, idx:  4310 },
  { start: 0x6754, end: 0x6755, idx:  4318 },
  { start: 0x675d, end: 0x675d, idx:  4320 },
  { start: 0x6766, end: 0x6784, idx:  4321 },
  { start: 0x678e, end: 0x679b, idx:  4352 },
  { start: 0x67b0, end: 0x67c9, idx:  4366 },
  { start: 0x67d2, end: 0x67e6, idx:  4392 },
  { start: 0x67f0, end: 0x67f9, idx:  4413 },
  { start: 0x6814, end: 0x681f, idx:  4423 },
  { start: 0x6827, end: 0x6833, idx:  4435 },
  { start: 0x683b, end: 0x685b, idx:  4448 },
  { start: 0x686b, end: 0x6888, idx:  4481 },
  { start: 0x6896, end: 0x68b2, idx:  4511 },
  { start: 0x68bb, end: 0x68bb, idx:  4540 },
  { start: 0x68c5, end: 0x68dd, idx:  4541 },
  { start: 0x68e5, end: 0x68fd, idx:  4566 },
  { start: 0x6906, end: 0x6917, idx:  4591 },
  { start: 0x6931, end: 0x693b, idx:  4609 },
  { start: 0x6942, end: 0x694e, idx:  4620 },
  { start: 0x6957, end: 0x695b, idx:  4633 },
  { start: 0x6963, end: 0x6972, idx:  4638 },
  { start: 0x697a, end: 0x6980, idx:  4654 },
  { start: 0x698d, end: 0x6998, idx:  4661 },
  { start: 0x69a1, end: 0x69af, idx:  4673 },
  { start: 0x69b7, end: 0x69bc, idx:  4688 },
  { start: 0x69c5, end: 0x69c8, idx:  4694 },
  { start: 0x69d1, end: 0x69d7, idx:  4698 },
  { start: 0x69e2, end: 0x69e5, idx:  4705 },
  { start: 0x69ee, end: 0x69f5, idx:  4709 },
  { start: 0x69fe, end: 0x6a03, idx:  4717 },
  { start: 0x6a0f, end: 0x6a28, idx:  4723 },
  { start: 0x6a30, end: 0x6a5b, idx:  4749 },
  { start: 0x6a64, end: 0x6a6a, idx:  4793 },
  { start: 0x6a71, end: 0x6a73, idx:  4800 },
  { start: 0x6a7e, end: 0x6a91, idx:  4803 },
  { start: 0x6a9b, end: 0x6ab4, idx:  4823 },
  { start: 0x6abd, end: 0x6abf, idx:  4849 },
  { start: 0x6ac6, end: 0x6add, idx:  4852 },
  { start: 0x6ae4, end: 0x6af2, idx:  4876 },
  { start: 0x6afc, end: 0x6b3f, idx:  4891 },
  { start: 0x6b46, end: 0x6b60, idx:  4959 },
  { start: 0x6b67, end: 0x6b75, idx:  4986 },
  { start: 0x6b7d, end: 0x6b85, idx:  5001 },
  { start: 0x6b97, end: 0x6bb0, idx:  5010 },
  { start: 0x6bb8, end: 0x6bcc, idx:  5036 },
  { start: 0x6bd6, end: 0x6bda, idx:  5057 },
  { start: 0x6be1, end: 0x6be7, idx:  5062 },
  { start: 0x6bee, end: 0x6c12, idx:  5069 },
  { start: 0x6c19, end: 0x6c1f, idx:  5106 },
  { start: 0x6c26, end: 0x6c3f, idx:  5113 },
  { start: 0x6c4a, end: 0x6c5c, idx:  5139 },
  { start: 0x6c67, end: 0x6c7b, idx:  5158 },
  { start: 0x6c85, end: 0x6c89, idx:  5179 },
  { start: 0x6c94, end: 0x6c9f, idx:  5184 },
  { start: 0x6cb0, end: 0x6cb4, idx:  5196 },
  { start: 0x6cc2, end: 0x6cc6, idx:  5201 },
  { start: 0x6ccd, end: 0x6ce0, idx:  5206 },
  { start: 0x6ce7, end: 0x6cf4, idx:  5226 },
  { start: 0x6d04, end: 0x6d13, idx:  5240 },
  { start: 0x6d1a, end: 0x6d1a, idx:  5256 },
  { start: 0x6d26, end: 0x6d31, idx:  5257 },
  { start: 0x6d39, end: 0x6d3f, idx:  5269 },
  { start: 0x6d57, end: 0x6d57, idx:  5276 },
  { start: 0x6d5e, end: 0x6d67, idx:  5277 },
  { start: 0x6d6f, end: 0x6d70, idx:  5287 },
  { start: 0x6d7c, end: 0x6d87, idx:  5289 },
  { start: 0x6d91, end: 0x6d98, idx:  5301 },
  { start: 0x6daa, end: 0x6dac, idx:  5309 },
  { start: 0x6db4, end: 0x6dcf, idx:  5312 },
  { start: 0x6dd6, end: 0x6e04, idx:  5340 },
  { start: 0x6e1e, end: 0x6e27, idx:  5387 },
  { start: 0x6e32, end: 0x6e3c, idx:  5397 },
  { start: 0x6e44, end: 0x6e68, idx:  5408 },
  { start: 0x6e73, end: 0x6e73, idx:  5445 },
  { start: 0x6e7b, end: 0x6e7d, idx:  5446 },
  { start: 0x6e8d, end: 0x6e99, idx:  5449 },
  { start: 0x6ea0, end: 0x6ea0, idx:  5462 },
  { start: 0x6ea7, end: 0x6eb3, idx:  5463 },
  { start: 0x6ebb, end: 0x6ecf, idx:  5476 },
  { start: 0x6eeb, end: 0x6eee, idx:  5497 },
  { start: 0x6ef9, end: 0x6efd, idx:  5501 },
  { start: 0x6f04, end: 0x6f0d, idx:  5506 },
  { start: 0x6f16, end: 0x6f1b, idx:  5516 },
  { start: 0x6f26, end: 0x6f3c, idx:  5522 },
  { start: 0x6f4f, end: 0x6f6c, idx:  5545 },
  { start: 0x6f7d, end: 0x6fbc, idx:  5575 },
  { start: 0x6fc5, end: 0x6fca, idx:  5639 },
  { start: 0x6fda, end: 0x6fde, idx:  5645 },
  { start: 0x6fe8, end: 0x6fe9, idx:  5650 },
  { start: 0x6ff0, end: 0x700d, idx:  5652 },
  { start: 0x7017, end: 0x7017, idx:  5682 },
  { start: 0x7020, end: 0x7023, idx:  5683 },
  { start: 0x702f, end: 0x703c, idx:  5687 },
  { start: 0x7043, end: 0x7055, idx:  5701 },
  { start: 0x705d, end: 0x7065, idx:  5720 },
  { start: 0x706c, end: 0x706e, idx:  5729 },
  { start: 0x7075, end: 0x7076, idx:  5732 },
  { start: 0x707e, end: 0x7086, idx:  5734 },
  { start: 0x7094, end: 0x709b, idx:  5743 },
  { start: 0x70a4, end: 0x70a4, idx:  5751 },
  { start: 0x70ab, end: 0x70b7, idx:  5752 },
  { start: 0x70ca, end: 0x70ca, idx:  5765 },
  { start: 0x70d1, end: 0x70dc, idx:  5766 },
  { start: 0x70e4, end: 0x70e4, idx:  5778 },
  { start: 0x70fa, end: 0x70fa, idx:  5779 },
  { start: 0x7103, end: 0x710f, idx:  5780 },
  { start: 0x711e, end: 0x7120, idx:  5793 },
  { start: 0x712b, end: 0x7131, idx:  5796 },
  { start: 0x7138, end: 0x7138, idx:  5803 },
  { start: 0x7141, end: 0x7160, idx:  5804 },
  { start: 0x7168, end: 0x7168, idx:  5836 },
  { start: 0x7179, end: 0x7179, idx:  5837 },
  { start: 0x7180, end: 0x7192, idx:  5838 },
  { start: 0x719a, end: 0x71a2, idx:  5857 },
  { start: 0x71af, end: 0x71b3, idx:  5866 },
  { start: 0x71ba, end: 0x71c4, idx:  5871 },
  { start: 0x71cb, end: 0x71cc, idx:  5882 },
  { start: 0x71d3, end: 0x71dc, idx:  5884 },
  { start: 0x71f8, end: 0x7200, idx:  5894 },
  { start: 0x7207, end: 0x7209, idx:  5903 },
  { start: 0x7213, end: 0x7224, idx:  5906 },
  { start: 0x722b, end: 0x7239, idx:  5924 },
  { start: 0x7241, end: 0x7245, idx:  5939 },
  { start: 0x724e, end: 0x7293, idx:  5944 },
  { start: 0x729b, end: 0x729b, idx:  6014 },
  { start: 0x72a8, end: 0x72b4, idx:  6015 },
  { start: 0x72be, end: 0x72cc, idx:  6028 },
  { start: 0x72d5, end: 0x72d8, idx:  6043 },
  { start: 0x72df, end: 0x72e5, idx:  6047 },
  { start: 0x72f3, end: 0x7343, idx:  6054 },
  { start: 0x734d, end: 0x7386, idx:  6135 },
  { start: 0x738e, end: 0x73ad, idx:  6193 },
  { start: 0x73b5, end: 0x73e9, idx:  6225 },
  { start: 0x73f4, end: 0x740a, idx:  6278 },
  { start: 0x7411, end: 0x7411, idx:  6301 },
  { start: 0x741a, end: 0x741b, idx:  6302 },
  { start: 0x7424, end: 0x7431, idx:  6304 },
  { start: 0x7439, end: 0x7439, idx:  6318 },
  { start: 0x7440, end: 0x7472, idx:  6319 },
  { start: 0x7480, end: 0x74bf, idx:  6370 },
  { start: 0x74c8, end: 0x74ff, idx:  6434 },
  { start: 0x7506, end: 0x7506, idx:  6490 },
  { start: 0x7512, end: 0x7517, idx:  6491 },
  { start: 0x7520, end: 0x752f, idx:  6497 },
  { start: 0x7536, end: 0x7557, idx:  6513 },
  { start: 0x755e, end: 0x7561, idx:  6547 },
  { start: 0x756f, end: 0x7571, idx:  6551 },
  { start: 0x7579, end: 0x7585, idx:  6554 },
  { start: 0x7590, end: 0x75a4, idx:  6567 },
  { start: 0x75b4, end: 0x75cf, idx:  6588 },
  { start: 0x75d7, end: 0x75f1, idx:  6616 },
  { start: 0x75f9, end: 0x75f9, idx:  6643 },
  { start: 0x7600, end: 0x764b, idx:  6644 },
  { start: 0x7655, end: 0x7665, idx:  6720 },
  { start: 0x766d, end: 0x7674, idx:  6737 },
  { start: 0x7681, end: 0x7685, idx:  6745 },
  { start: 0x768c, end: 0x768d, idx:  6750 },
  { start: 0x7695, end: 0x76ad, idx:  6752 },
  { start: 0x76bd, end: 0x76d9, idx:  6777 },
  { start: 0x76e0, end: 0x7784, idx:  6806 },
  { start: 0x778c, end: 0x778d, idx:  6971 },
  { start: 0x7794, end: 0x77b5, idx:  6973 },
  { start: 0x77be, end: 0x77c9, idx:  7007 },
  { start: 0x77d1, end: 0x77fb, idx:  7019 },
  { start: 0x7805, end: 0x7811, idx:  7062 },
  { start: 0x781d, end: 0x7823, idx:  7075 },
  { start: 0x782d, end: 0x7837, idx:  7082 },
  { start: 0x7843, end: 0x7852, idx:  7093 },
  { start: 0x785c, end: 0x786e, idx:  7109 },
  { start: 0x787a, end: 0x787e, idx:  7128 },
  { start: 0x788a, end: 0x78b3, idx:  7133 },
  { start: 0x78bb, end: 0x78bf, idx:  7175 },
  { start: 0x78c7, end: 0x78ea, idx:  7180 },
  { start: 0x78f2, end: 0x7910, idx:  7216 },
  { start: 0x791a, end: 0x795c, idx:  7247 },
  { start: 0x7967, end: 0x796b, idx:  7314 },
  { start: 0x7972, end: 0x7972, idx:  7319 },
  { start: 0x7979, end: 0x797e, idx:  7320 },
  { start: 0x798b, end: 0x79a1, idx:  7326 },
  { start: 0x79a8, end: 0x79bb, idx:  7349 },
  { start: 0x79c2, end: 0x79f1, idx:  7369 },
  { start: 0x79f8, end: 0x7a3a, idx:  7417 },
  { start: 0x7a44, end: 0x7a4c, idx:  7484 },
  { start: 0x7a55, end: 0x7a6d, idx:  7493 },
  { start: 0x7a75, end: 0x7a94, idx:  7518 },
  { start: 0x7a9e, end: 0x7aa3, idx:  7550 },
  { start: 0x7aac, end: 0x7aac, idx:  7556 },
  { start: 0x7ab3, end: 0x7abc, idx:  7557 },
  { start: 0x7ac6, end: 0x7ad1, idx:  7567 },
  { start: 0x7adb, end: 0x7adb, idx:  7579 },
  { start: 0x7ae8, end: 0x7af4, idx:  7580 },
  { start: 0x7afb, end: 0x7afe, idx:  7593 },
  { start: 0x7b07, end: 0x7b07, idx:  7597 },
  { start: 0x7b14, end: 0x7b14, idx:  7598 },
  { start: 0x7b1f, end: 0x7b34, idx:  7599 },
  { start: 0x7b3d, end: 0x7b47, idx:  7621 },
  { start: 0x7b4e, end: 0x7b4e, idx:  7632 },
  { start: 0x7b55, end: 0x7b55, idx:  7633 },
  { start: 0x7b60, end: 0x7b77, idx:  7634 },
  { start: 0x7b84, end: 0x7ba5, idx:  7658 },
  { start: 0x7bac, end: 0x7bca, idx:  7692 },
  { start: 0x7bd4, end: 0x7bdb, idx:  7723 },
  { start: 0x7be8, end: 0x7bea, idx:  7731 },
  { start: 0x7bf2, end: 0x7c0f, idx:  7734 },
  { start: 0x7c19, end: 0x7c3a, idx:  7764 },
  { start: 0x7c46, end: 0x7c4a, idx:  7798 },
  { start: 0x7c51, end: 0x7c72, idx:  7803 },
  { start: 0x7c79, end: 0x7c7d, idx:  7837 },
  { start: 0x7c86, end: 0x7c87, idx:  7842 },
  { start: 0x7c8f, end: 0x7c94, idx:  7844 },
  { start: 0x7c9e, end: 0x7ca6, idx:  7850 },
  { start: 0x7cb0, end: 0x7cdd, idx:  7859 },
  { start: 0x7ce6, end: 0x7ceb, idx:  7905 },
  { start: 0x7cf5, end: 0x7cf5, idx:  7911 },
  { start: 0x7d03, end: 0x7d16, idx:  7912 },
  { start: 0x7d1d, end: 0x7d31, idx:  7932 },
  { start: 0x7d3c, end: 0x7d5d, idx:  7953 },
  { start: 0x7d65, end: 0x7d70, idx:  7987 },
  { start: 0x7d78, end: 0x7d9e, idx:  7999 },
  { start: 0x7da6, end: 0x7daa, idx:  8038 },
  { start: 0x7db3, end: 0x7db9, idx:  8043 },
  { start: 0x7dc2, end: 0x7dce, idx:  8050 },
  { start: 0x7dd7, end: 0x7dd9, idx:  8063 },
  { start: 0x7de2, end: 0x7e00, idx:  8066 },
  { start: 0x7e08, end: 0x7e08, idx:  8097 },
  { start: 0x7e10, end: 0x7e20, idx:  8098 },
  { start: 0x7e27, end: 0x7e36, idx:  8115 },
  { start: 0x7e3f, end: 0x7e47, idx:  8131 },
  { start: 0x7e4e, end: 0x7e58, idx:  8140 },
  { start: 0x7e5f, end: 0x7e9e, idx:  8151 },
  { start: 0x7f3b, end: 0x7f47, idx:  8215 },
  { start: 0x7f4f, end: 0x7f53, idx:  8228 },
  { start: 0x7f5b, end: 0x7f66, idx:  8233 },
  { start: 0x7f6d, end: 0x7f71, idx:  8245 },
  { start: 0x7f7d, end: 0x7f80, idx:  8250 },
  { start: 0x7f8b, end: 0x7fad, idx:  8254 },
  { start: 0x7fb4, end: 0x7fb4, idx:  8289 },
  { start: 0x7fbc, end: 0x7fcf, idx:  8290 },
  { start: 0x7fdb, end: 0x7ff2, idx:  8310 },
  { start: 0x7ffa, end: 0x7fff, idx:  8334 },
  { start: 0x8007, end: 0x8016, idx:  8340 },
  { start: 0x801d, end: 0x8044, idx:  8356 },
  { start: 0x8060, end: 0x8066, idx:  8396 },
  { start: 0x806d, end: 0x8075, idx:  8403 },
  { start: 0x8081, end: 0x8081, idx:  8412 },
  { start: 0x8088, end: 0x808e, idx:  8413 },
  { start: 0x809c, end: 0x809e, idx:  8420 },
  { start: 0x80a6, end: 0x80ab, idx:  8423 },
  { start: 0x80b8, end: 0x80b9, idx:  8429 },
  { start: 0x80c8, end: 0x80d8, idx:  8431 },
  { start: 0x80e0, end: 0x80e0, idx:  8448 },
  { start: 0x80ed, end: 0x8103, idx:  8449 },
  { start: 0x810b, end: 0x810b, idx:  8472 },
  { start: 0x8116, end: 0x813c, idx:  8473 },
  { start: 0x8145, end: 0x8157, idx:  8512 },
  { start: 0x8160, end: 0x816f, idx:  8531 },
  { start: 0x8177, end: 0x8177, idx:  8547 },
  { start: 0x8181, end: 0x81a2, idx:  8548 },
  { start: 0x81ae, end: 0x81b4, idx:  8582 },
  { start: 0x81bb, end: 0x81bb, idx:  8589 },
  { start: 0x81c3, end: 0x81e4, idx:  8590 },
  { start: 0x81eb, end: 0x8203, idx:  8624 },
  { start: 0x820f, end: 0x8228, idx:  8649 },
  { start: 0x8232, end: 0x823a, idx:  8675 },
  { start: 0x8243, end: 0x826d, idx:  8684 },
  { start: 0x8274, end: 0x8274, idx:  8727 },
  { start: 0x827b, end: 0x82bf, idx:  8728 },
  { start: 0x82c6, end: 0x82c6, idx:  8797 },
  { start: 0x82d0, end: 0x82ef, idx:  8798 },
  { start: 0x82f6, end: 0x830b, idx:  8830 },
  { start: 0x831b, end: 0x8322, idx:  8852 },
  { start: 0x832c, end: 0x8357, idx:  8860 },
  { start: 0x8370, end: 0x8370, idx:  8904 },
  { start: 0x8378, end: 0x8386, idx:  8905 },
  { start: 0x838d, end: 0x839d, idx:  8920 },
  { start: 0x83a6, end: 0x83ad, idx:  8937 },
  { start: 0x83be, end: 0x83c0, idx:  8945 },
  { start: 0x83c7, end: 0x83d4, idx:  8948 },
  { start: 0x83dd, end: 0x83dd, idx:  8962 },
  { start: 0x83e8, end: 0x83ea, idx:  8963 },
  { start: 0x83f6, end: 0x8419, idx:  8966 },
  { start: 0x842f, end: 0x842f, idx:  9002 },
  { start: 0x8439, end: 0x8439, idx:  9003 },
  { start: 0x8445, end: 0x8485, idx:  9004 },
  { start: 0x8492, end: 0x8495, idx:  9069 },
  { start: 0x849e, end: 0x849e, idx:  9073 },
  { start: 0x84a6, end: 0x84d3, idx:  9074 },
  { start: 0x84dc, end: 0x84dc, idx:  9120 },
  { start: 0x84e7, end: 0x8510, idx:  9121 },
  { start: 0x851c, end: 0x8536, idx:  9163 },
  { start: 0x853f, end: 0x853f, idx:  9190 },
  { start: 0x8546, end: 0x8546, idx:  9191 },
  { start: 0x854f, end: 0x8564, idx:  9192 },
  { start: 0x856b, end: 0x856f, idx:  9214 },
  { start: 0x8579, end: 0x85ad, idx:  9219 },
  { start: 0x85b4, end: 0x85ce, idx:  9272 },
  { start: 0x85d8, end: 0x8605, idx:  9299 },
  { start: 0x860d, end: 0x8629, idx:  9345 },
  { start: 0x8636, end: 0x8646, idx:  9374 },
  { start: 0x8652, end: 0x867a, idx:  9391 },
  { start: 0x8688, end: 0x86c5, idx:  9432 },
  { start: 0x86d1, end: 0x86e7, idx:  9494 },
  { start: 0x86fa, end: 0x86fd, idx:  9517 },
  { start: 0x8704, end: 0x8732, idx:  9521 },
  { start: 0x8739, end: 0x8745, idx:  9568 },
  { start: 0x874d, end: 0x874d, idx:  9581 },
  { start: 0x8758, end: 0x8765, idx:  9582 },
  { start: 0x876f, end: 0x8772, idx:  9596 },
  { start: 0x877b, end: 0x877b, idx:  9600 },
  { start: 0x8783, end: 0x87b5, idx:  9601 },
  { start: 0x87be, end: 0x87c1, idx:  9652 },
  { start: 0x87c8, end: 0x87ce, idx:  9656 },
  { start: 0x87d5, end: 0x881f, idx:  9663 },
  { start: 0x8828, end: 0x8869, idx:  9738 },
  { start: 0x8871, end: 0x8871, idx:  9804 },
  { start: 0x8879, end: 0x8880, idx:  9805 },
  { start: 0x8898, end: 0x88a0, idx:  9813 },
  { start: 0x88a8, end: 0x88aa, idx:  9822 },
  { start: 0x88ba, end: 0x88c0, idx:  9825 },
  { start: 0x88ca, end: 0x88d3, idx:  9832 },
  { start: 0x88db, end: 0x88de, idx:  9842 },
  { start: 0x88e7, end: 0x88e7, idx:  9846 },
  { start: 0x88ef, end: 0x88f7, idx:  9847 },
  { start: 0x8901, end: 0x8906, idx:  9856 },
  { start: 0x890d, end: 0x8928, idx:  9862 },
  { start: 0x8930, end: 0x8963, idx:  9890 },
  { start: 0x896b, end: 0x897d, idx:  9942 },
  { start: 0x8989, end: 0x89a5, idx:  9961 },
  { start: 0x89b0, end: 0x89bc, idx:  9990 },
  { start: 0x89d4, end: 0x89d8, idx: 10003 },
  { start: 0x89e5, end: 0x8a07, idx: 10008 },
  { start: 0x8a0f, end: 0x8a15, idx: 10043 },
  { start: 0x8a1e, end: 0x8a67, idx: 10050 },
  { start: 0x8a75, end: 0x8a9f, idx: 10124 },
  { start: 0x8aa7, end: 0x8aca, idx: 10167 },
  { start: 0x8ad1, end: 0x8adf, idx: 10203 },
  { start: 0x8aec, end: 0x8b11, idx: 10218 },
  { start: 0x8b1c, end: 0x8b1f, idx: 10256 },
  { start: 0x8b2d, end: 0x8b30, idx: 10260 },
  { start: 0x8b37, end: 0x8b63, idx: 10264 },
  { start: 0x8b6d, end: 0x8b6d, idx: 10309 },
  { start: 0x8b76, end: 0x8b95, idx: 10310 },
  { start: 0x8b9c, end: 0x8b9f, idx: 10342 },
  { start: 0x8c38, end: 0x8c3e, idx: 10346 },
  { start: 0x8c45, end: 0x8c7e, idx: 10353 },
  { start: 0x8c86, end: 0x8c9c, idx: 10411 },
  { start: 0x8ca4, end: 0x8ca4, idx: 10434 },
  { start: 0x8cb9, end: 0x8cba, idx: 10435 },
  { start: 0x8cc5, end: 0x8ce1, idx: 10437 },
  { start: 0x8ce8, end: 0x8d09, idx: 10466 },
  { start: 0x8d12, end: 0x8d1b, idx: 10500 },
  { start: 0x8d65, end: 0x8d6e, idx: 10510 },
  { start: 0x8d7f, end: 0x8d95, idx: 10520 },
  { start: 0x8d9e, end: 0x8dd9, idx: 10543 },
  { start: 0x8de4, end: 0x8df4, idx: 10603 },
  { start: 0x8dfd, end: 0x8e16, idx: 10620 },
  { start: 0x8e20, end: 0x8e27, idx: 10646 },
  { start: 0x8e31, end: 0x8e41, idx: 10654 },
  { start: 0x8e4b, end: 0x8e54, idx: 10671 },
  { start: 0x8e5b, end: 0x8e62, idx: 10681 },
  { start: 0x8e69, end: 0x8e71, idx: 10689 },
  { start: 0x8e79, end: 0x8e7b, idx: 10698 },
  { start: 0x8e82, end: 0x8e89, idx: 10701 },
  { start: 0x8e90, end: 0x8ec7, idx: 10709 },
  { start: 0x8ecf, end: 0x8ed4, idx: 10765 },
  { start: 0x8edc, end: 0x8edc, idx: 10771 },
  { start: 0x8ee8, end: 0x8f08, idx: 10772 },
  { start: 0x8f0f, end: 0x8f47, idx: 10805 },
  { start: 0x8f4f, end: 0x8f5e, idx: 10862 },
  { start: 0x8f65, end: 0x8f65, idx: 10878 },
  { start: 0x8f9d, end: 0x8fa6, idx: 10879 },
  { start: 0x8fb5, end: 0x8fd5, idx: 10889 },
  { start: 0x8fe0, end: 0x900c, idx: 10922 },
  { start: 0x9018, end: 0x901b, idx: 10967 },
  { start: 0x9028, end: 0x9037, idx: 10971 },
  { start: 0x903f, end: 0x9044, idx: 10987 },
  { start: 0x904c, end: 0x904c, idx: 10993 },
  { start: 0x905b, end: 0x9079, idx: 10994 },
  { start: 0x9085, end: 0x90a5, idx: 11025 },
  { start: 0x90b0, end: 0x90b6, idx: 11058 },
  { start: 0x90bd, end: 0x90f6, idx: 11065 },
  { start: 0x90fe, end: 0x9148, idx: 11123 },
  { start: 0x914f, end: 0x916d, idx: 11198 },
  { start: 0x9174, end: 0x91c5, idx: 11229 },
  { start: 0x91d3, end: 0x9251, idx: 11311 },
  { start: 0x9258, end: 0x927f, idx: 11438 },
  { start: 0x9288, end: 0x92f0, idx: 11478 },
  { start: 0x92f9, end: 0x9315, idx: 11583 },
  { start: 0x931c, end: 0x932a, idx: 11612 },
  { start: 0x9333, end: 0x9337, idx: 11627 },
  { start: 0x9347, end: 0x9349, idx: 11632 },
  { start: 0x9350, end: 0x93ab, idx: 11635 },
  { start: 0x93b4, end: 0x93ba, idx: 11727 },
  { start: 0x93c1, end: 0x93e7, idx: 11734 },
  { start: 0x93f7, end: 0x9417, idx: 11773 },
  { start: 0x941f, end: 0x941f, idx: 11806 },
  { start: 0x942e, end: 0x9434, idx: 11807 },
  { start: 0x943b, end: 0x944c, idx: 11814 },
  { start: 0x9455, end: 0x9472, idx: 11832 },
  { start: 0x9483, end: 0x9484, idx: 11862 },
  { start: 0x9578, end: 0x958e, idx: 11864 },
  { start: 0x959d, end: 0x95ac, idx: 11887 },
  { start: 0x95b4, end: 0x95bf, idx: 11903 },
  { start: 0x95c6, end: 0x95e6, idx: 11915 },
  { start: 0x961d, end: 0x9641, idx: 11948 },
  { start: 0x9652, end: 0x9658, idx: 11985 },
  { start: 0x9661, end: 0x9661, idx: 11992 },
  { start: 0x966e, end: 0x9674, idx: 11993 },
  { start: 0x967b, end: 0x9689, idx: 12000 },
  { start: 0x9691, end: 0x96b3, idx: 12015 },
  { start: 0x96ba, end: 0x96ba, idx: 12050 },
  { start: 0x96ca, end: 0x96ca, idx: 12051 },
  { start: 0x96d2, end: 0x96df, idx: 12052 },
  { start: 0x96e9, end: 0x96f1, idx: 12066 },
  { start: 0x96fa, end: 0x96fa, idx: 12075 },
  { start: 0x9702, end: 0x9709, idx: 12076 },
  { start: 0x971a, end: 0x9728, idx: 12084 },
  { start: 0x9731, end: 0x9733, idx: 12099 },
  { start: 0x9741, end: 0x9743, idx: 12102 },
  { start: 0x974a, end: 0x975b, idx: 12105 },
  { start: 0x9763, end: 0x9780, idx: 12123 },
  { start: 0x9789, end: 0x9789, idx: 12153 },
  { start: 0x9795, end: 0x97a2, idx: 12154 },
  { start: 0x97ac, end: 0x97e4, idx: 12168 },
  { start: 0x97ef, end: 0x97fa, idx: 12225 },
  { start: 0x9807, end: 0x9835, idx: 12237 },
  { start: 0x983e, end: 0x984a, idx: 12284 },
  { start: 0x9851, end: 0x985a, idx: 12297 },
  { start: 0x9862, end: 0x986c, idx: 12307 },
  { start: 0x98ab, end: 0x98cc, idx: 12318 },
  { start: 0x98e1, end: 0x98ea, idx: 12352 },
  { start: 0x98f3, end: 0x98f6, idx: 12362 },
  { start: 0x9902, end: 0x9908, idx: 12366 },
  { start: 0x9911, end: 0x9960, idx: 12373 },
  { start: 0x999b, end: 0x999f, idx: 12453 },
  { start: 0x99a6, end: 0x99a6, idx: 12458 },
  { start: 0x99b0, end: 0x99c9, idx: 12459 },
  { start: 0x99d3, end: 0x99de, idx: 12485 },
  { start: 0x99e7, end: 0x9a04, idx: 12497 },
  { start: 0x9a0b, end: 0x9a16, idx: 12527 },
  { start: 0x9a1e, end: 0x9a38, idx: 12539 },
  { start: 0x9a41, end: 0x9a56, idx: 12566 },
  { start: 0x9a5d, end: 0x9a5d, idx: 12588 },
  { start: 0x9aaa, end: 0x9b3d, idx: 12589 },
  { start: 0x9b48, end: 0x9b4c, idx: 12737 },
  { start: 0x9b55, end: 0x9b9e, idx: 12742 },
  { start: 0x9ba6, end: 0x9bd0, idx: 12816 },
  { start: 0x9bd7, end: 0x9c02, idx: 12859 },
  { start: 0x9c0b, end: 0x9c2a, idx: 12903 },
  { start: 0x9c31, end: 0x9c7b, idx: 12935 },
  { start: 0x9ce6, end: 0x9ce6, idx: 13010 },
  { start: 0x9cf2, end: 0x9cf9, idx: 13011 },
  { start: 0x9d02, end: 0x9d02, idx: 13019 },
  { start: 0x9d0b, end: 0x9d1e, idx: 13020 },
  { start: 0x9d2f, end: 0x9d4a, idx: 13040 },
  { start: 0x9d53, end: 0x9d54, idx: 13068 },
  { start: 0x9d5f, end: 0x9d98, idx: 13070 },
  { start: 0x9da1, end: 0x9da1, idx: 13128 },
  { start: 0x9daa, end: 0x9dca, idx: 13129 },
  { start: 0x9dd4, end: 0x9df4, idx: 13162 },
  { start: 0x9dfe, end: 0x9e1d, idx: 13195 },
  { start: 0x9e7a, end: 0x9e87, idx: 13227 },
  { start: 0x9e8e, end: 0x9e8f, idx: 13241 },
  { start: 0x9e96, end: 0x9eb5, idx: 13243 },
  { start: 0x9ec6, end: 0x9ecb, idx: 13275 },
  { start: 0x9ed5, end: 0x9ed5, idx: 13281 },
  { start: 0x9edf, end: 0x9ef8, idx: 13282 },
  { start: 0x9eff, end: 0x9f47, idx: 13308 },
  { start: 0x9f53, end: 0x9f5e, idx: 13381 },
  { start: 0x9f68, end: 0x9f7d, idx: 13393 },
  { start: 0x9f8f, end: 0x9f97, idx: 13415 },
  { start: 0x9f9e, end: 0x9fa5, idx: 13424 },
  { start: 0 }
};


/* The values corresponding to the table can be computed using

    egrep '^0x[[:xdigit:]][[:xdigit:]][[:xdigit:]][[:xdigit:]]' \
      .../eastasia/jis/jis0212.txt | awk '{ print $2, $1 }' |
    sort -u | perl tab.pl

   where tab.pl contains:

   ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
   $first=$last=$idx=0;

   sub fmt {
     printf ("\n ") if (($n % 6) == 0);
     ++$n;
     my($val) = pop(@_);
     printf (" \"\\x%02x\\x%02x\",",
	     $val < 256 ? $val : int($val / 256),
	     $val < 256 ? 0 : $val % 256);
   }

   while (<>) {
     local($ucs,$jis,%rest) = split;
     local($u)=hex($ucs);
     local($j)=hex($jis);
     if ($u - $last > 6) {
       if ($last != 0) {
	 $idx += $last - $first + 1;
       }
       $first=$u;
     } else {
       for ($m = $last + 1; $m < $u; ++$m) {
	 fmt (0);
       }
     }
     fmt ($u);
     $last=$u;
   }
   ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
*/
const char jisx0212_from_ucs[][2] =
{
  "\x7e\x00", "\xa1\x00", "\x00\x00", "\x00\x00", "\xa4\x00", "\x00\x00",
  "\xa6\x00", "\x00\x00", "\x00\x00", "\xa9\x00", "\xaa\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\xae\x00", "\xaf\x00", "\xb8\x00", "\x00\x00",
  "\xba\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\xbf\x00",
  "\xc0\x00", "\xc1\x00", "\xc2\x00", "\xc3\x00", "\xc4\x00", "\xc5\x00",
  "\xc6\x00", "\xc7\x00", "\xc8\x00", "\xc9\x00", "\xca\x00", "\xcb\x00",
  "\xcc\x00", "\xcd\x00", "\xce\x00", "\xcf\x00", "\x00\x00", "\xd1\x00",
  "\xd2\x00", "\xd3\x00", "\xd4\x00", "\xd5\x00", "\xd6\x00", "\x00\x00",
  "\xd8\x00", "\xd9\x00", "\xda\x00", "\xdb\x00", "\xdc\x00", "\xdd\x00",
  "\xde\x00", "\xdf\x00", "\xe0\x00", "\xe1\x00", "\xe2\x00", "\xe3\x00",
  "\xe4\x00", "\xe5\x00", "\xe6\x00", "\xe7\x00", "\xe8\x00", "\xe9\x00",
  "\xea\x00", "\xeb\x00", "\xec\x00", "\xed\x00", "\xee\x00", "\xef\x00",
  "\xf0\x00", "\xf1\x00", "\xf2\x00", "\xf3\x00", "\xf4\x00", "\xf5\x00",
  "\xf6\x00", "\x00\x00", "\xf8\x00", "\xf9\x00", "\xfa\x00", "\xfb\x00",
  "\xfc\x00", "\xfd\x00", "\xfe\x00", "\xff\x00", "\x01\x00", "\x01\x01",
  "\x01\x02", "\x01\x03", "\x01\x04", "\x01\x05", "\x01\x06", "\x01\x07",
  "\x01\x08", "\x01\x09", "\x01\x0a", "\x01\x0b", "\x01\x0c", "\x01\x0d",
  "\x01\x0e", "\x01\x0f", "\x01\x10", "\x01\x11", "\x01\x12", "\x01\x13",
  "\x00\x00", "\x00\x00", "\x01\x16", "\x01\x17", "\x01\x18", "\x01\x19",
  "\x01\x1a", "\x01\x1b", "\x01\x1c", "\x01\x1d", "\x01\x1e", "\x01\x1f",
  "\x01\x20", "\x01\x21", "\x01\x22", "\x00\x00", "\x01\x24", "\x01\x25",
  "\x01\x26", "\x01\x27", "\x01\x28", "\x01\x29", "\x01\x2a", "\x01\x2b",
  "\x00\x00", "\x00\x00", "\x01\x2e", "\x01\x2f", "\x01\x30", "\x01\x31",
  "\x01\x32", "\x01\x33", "\x01\x34", "\x01\x35", "\x01\x36", "\x01\x37",
  "\x01\x38", "\x01\x39", "\x01\x3a", "\x01\x3b", "\x01\x3c", "\x01\x3d",
  "\x01\x3e", "\x01\x3f", "\x01\x40", "\x01\x41", "\x01\x42", "\x01\x43",
  "\x01\x44", "\x01\x45", "\x01\x46", "\x01\x47", "\x01\x48", "\x01\x49",
  "\x01\x4a", "\x01\x4b", "\x01\x4c", "\x01\x4d", "\x00\x00", "\x00\x00",
  "\x01\x50", "\x01\x51", "\x01\x52", "\x01\x53", "\x01\x54", "\x01\x55",
  "\x01\x56", "\x01\x57", "\x01\x58", "\x01\x59", "\x01\x5a", "\x01\x5b",
  "\x01\x5c", "\x01\x5d", "\x01\x5e", "\x01\x5f", "\x01\x60", "\x01\x61",
  "\x01\x62", "\x01\x63", "\x01\x64", "\x01\x65", "\x01\x66", "\x01\x67",
  "\x01\x68", "\x01\x69", "\x01\x6a", "\x01\x6b", "\x01\x6c", "\x01\x6d",
  "\x01\x6e", "\x01\x6f", "\x01\x70", "\x01\x71", "\x01\x72", "\x01\x73",
  "\x01\x74", "\x01\x75", "\x01\x76", "\x01\x77", "\x01\x78", "\x01\x79",
  "\x01\x7a", "\x01\x7b", "\x01\x7c", "\x01\x7d", "\x01\x7e", "\x01\xcd",
  "\x01\xce", "\x01\xcf", "\x01\xd0", "\x01\xd1", "\x01\xd2", "\x01\xd3",
  "\x01\xd4", "\x01\xd5", "\x01\xd6", "\x01\xd7", "\x01\xd8", "\x01\xd9",
  "\x01\xda", "\x01\xdb", "\x01\xdc", "\x01\xf5", "\x02\xc7", "\x02\xd8",
  "\x02\xd9", "\x02\xda", "\x02\xdb", "\x00\x00", "\x02\xdd", "\x03\x84",
  "\x03\x85", "\x03\x86", "\x00\x00", "\x03\x88", "\x03\x89", "\x03\x8a",
  "\x00\x00", "\x03\x8c", "\x00\x00", "\x03\x8e", "\x03\x8f", "\x03\x90",
  "\x03\xaa", "\x03\xab", "\x03\xac", "\x03\xad", "\x03\xae", "\x03\xaf",
  "\x03\xb0", "\x03\xc2", "\x03\xca", "\x03\xcb", "\x03\xcc", "\x03\xcd",
  "\x03\xce", "\x04\x02", "\x04\x03", "\x04\x04", "\x04\x05", "\x04\x06",
  "\x04\x07", "\x04\x08", "\x04\x09", "\x04\x0a", "\x04\x0b", "\x04\x0c",
  "\x00\x00", "\x04\x0e", "\x04\x0f", "\x04\x52", "\x04\x53", "\x04\x54",
  "\x04\x55", "\x04\x56", "\x04\x57", "\x04\x58", "\x04\x59", "\x04\x5a",
  "\x04\x5b", "\x04\x5c", "\x00\x00", "\x04\x5e", "\x04\x5f", "\x21\x16",
  "\x21\x22", "\x4e\x02", "\x00\x00", "\x4e\x04", "\x4e\x05", "\x4e\x0c",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x4e\x12",
  "\x4e\x1f", "\x00\x00", "\x00\x00", "\x00\x00", "\x4e\x23", "\x4e\x24",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x4e\x28", "\x00\x00", "\x00\x00",
  "\x4e\x2b", "\x00\x00", "\x00\x00", "\x4e\x2e", "\x4e\x2f", "\x4e\x30",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x4e\x35", "\x4e\x40",
  "\x4e\x41", "\x00\x00", "\x00\x00", "\x4e\x44", "\x00\x00", "\x00\x00",
  "\x4e\x47", "\x4e\x51", "\x4e\x5a", "\x00\x00", "\x4e\x5c", "\x4e\x63",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x4e\x68", "\x4e\x69",
  "\x4e\x74", "\x4e\x75", "\x00\x00", "\x00\x00", "\x00\x00", "\x4e\x79",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x4e\x7f",
  "\x4e\x8d", "\x4e\x96", "\x4e\x97", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x4e\x9d", "\x4e\xaf", "\x4e\xb9", "\x4e\xc3",
  "\x4e\xd0", "\x4e\xda", "\x4e\xdb", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x4e\xe0", "\x4e\xe1", "\x4e\xe2", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x4e\xe8", "\x4e\xef", "\x00\x00",
  "\x4e\xf1", "\x00\x00", "\x4e\xf3", "\x00\x00", "\x4e\xf5", "\x4e\xfd",
  "\x4e\xfe", "\x4e\xff", "\x4f\x00", "\x00\x00", "\x4f\x02", "\x4f\x03",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x4f\x08", "\x00\x00",
  "\x00\x00", "\x4f\x0b", "\x4f\x0c", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x4f\x12", "\x00\x00", "\x00\x00", "\x4f\x15",
  "\x4f\x16", "\x4f\x17", "\x00\x00", "\x4f\x19", "\x4f\x2e", "\x00\x00",
  "\x00\x00", "\x4f\x31", "\x00\x00", "\x4f\x33", "\x00\x00", "\x4f\x35",
  "\x00\x00", "\x4f\x37", "\x00\x00", "\x4f\x39", "\x00\x00", "\x4f\x3b",
  "\x00\x00", "\x00\x00", "\x4f\x3e", "\x00\x00", "\x4f\x40", "\x00\x00",
  "\x4f\x42", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x4f\x48", "\x4f\x49", "\x00\x00", "\x4f\x4b", "\x4f\x4c", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x4f\x52", "\x00\x00",
  "\x4f\x54", "\x00\x00", "\x4f\x56", "\x00\x00", "\x4f\x58", "\x4f\x5f",
  "\x4f\x60", "\x00\x00", "\x00\x00", "\x4f\x63", "\x4f\x6a", "\x00\x00",
  "\x4f\x6c", "\x00\x00", "\x4f\x6e", "\x00\x00", "\x00\x00", "\x4f\x71",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x4f\x77",
  "\x4f\x78", "\x4f\x79", "\x4f\x7a", "\x00\x00", "\x00\x00", "\x4f\x7d",
  "\x4f\x7e", "\x00\x00", "\x00\x00", "\x4f\x81", "\x4f\x82", "\x00\x00",
  "\x4f\x84", "\x4f\x85", "\x00\x00", "\x00\x00", "\x00\x00", "\x4f\x89",
  "\x4f\x8a", "\x00\x00", "\x4f\x8c", "\x00\x00", "\x4f\x8e", "\x00\x00",
  "\x4f\x90", "\x00\x00", "\x4f\x92", "\x4f\x93", "\x4f\x94", "\x00\x00",
  "\x00\x00", "\x4f\x97", "\x00\x00", "\x4f\x99", "\x4f\x9a", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x4f\x9e", "\x4f\x9f", "\x4f\xb2", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x4f\xb7", "\x00\x00", "\x4f\xb9",
  "\x00\x00", "\x4f\xbb", "\x4f\xbc", "\x4f\xbd", "\x4f\xbe", "\x00\x00",
  "\x4f\xc0", "\x4f\xc1", "\x00\x00", "\x00\x00", "\x00\x00", "\x4f\xc5",
  "\x4f\xc6", "\x00\x00", "\x4f\xc8", "\x4f\xc9", "\x00\x00", "\x4f\xcb",
  "\x4f\xcc", "\x4f\xcd", "\x00\x00", "\x4f\xcf", "\x00\x00", "\x00\x00",
  "\x4f\xd2", "\x4f\xdc", "\x00\x00", "\x00\x00", "\x00\x00", "\x4f\xe0",
  "\x00\x00", "\x4f\xe2", "\x4f\xf0", "\x00\x00", "\x4f\xf2", "\x4f\xfc",
  "\x4f\xfd", "\x00\x00", "\x4f\xff", "\x50\x00", "\x50\x01", "\x00\x00",
  "\x00\x00", "\x50\x04", "\x00\x00", "\x00\x00", "\x50\x07", "\x00\x00",
  "\x00\x00", "\x50\x0a", "\x00\x00", "\x50\x0c", "\x00\x00", "\x50\x0e",
  "\x00\x00", "\x50\x10", "\x00\x00", "\x00\x00", "\x50\x13", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x50\x17", "\x50\x18", "\x00\x00", "\x00\x00",
  "\x50\x1b", "\x50\x1c", "\x50\x1d", "\x50\x1e", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x50\x22", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x50\x27", "\x50\x2e", "\x00\x00", "\x50\x30", "\x00\x00", "\x50\x32",
  "\x50\x33", "\x00\x00", "\x50\x35", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x50\x3b", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x50\x40", "\x50\x41", "\x50\x42", "\x00\x00", "\x00\x00",
  "\x50\x45", "\x50\x46", "\x00\x00", "\x00\x00", "\x00\x00", "\x50\x4a",
  "\x00\x00", "\x50\x4c", "\x00\x00", "\x50\x4e", "\x00\x00", "\x00\x00",
  "\x50\x51", "\x50\x52", "\x50\x53", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x50\x57", "\x00\x00", "\x50\x59", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x50\x5f", "\x50\x60", "\x00\x00", "\x50\x62",
  "\x50\x63", "\x00\x00", "\x00\x00", "\x50\x66", "\x50\x67", "\x00\x00",
  "\x00\x00", "\x50\x6a", "\x00\x00", "\x00\x00", "\x50\x6d", "\x00\x00",
  "\x00\x00", "\x50\x70", "\x50\x71", "\x50\x81", "\x00\x00", "\x50\x83",
  "\x50\x84", "\x00\x00", "\x50\x86", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x50\x8a", "\x00\x00", "\x00\x00", "\x00\x00", "\x50\x8e", "\x50\x8f",
  "\x50\x90", "\x00\x00", "\x50\x92", "\x50\x93", "\x50\x94", "\x00\x00",
  "\x50\x96", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x50\x9b",
  "\x50\x9c", "\x00\x00", "\x50\x9e", "\x50\x9f", "\x50\xa0", "\x50\xa1",
  "\x50\xa2", "\x50\xaa", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x50\xaf", "\x50\xb0", "\x50\xb9", "\x50\xba", "\x00\x00", "\x00\x00",
  "\x50\xbd", "\x00\x00", "\x00\x00", "\x50\xc0", "\x00\x00", "\x00\x00",
  "\x50\xc3", "\x50\xc4", "\x00\x00", "\x00\x00", "\x50\xc7", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x50\xcc", "\x00\x00", "\x50\xce",
  "\x00\x00", "\x50\xd0", "\x00\x00", "\x00\x00", "\x50\xd3", "\x50\xd4",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x50\xd8", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x50\xdc", "\x50\xdd", "\x00\x00", "\x50\xdf", "\x00\x00",
  "\x00\x00", "\x50\xe2", "\x00\x00", "\x50\xe4", "\x00\x00", "\x50\xe6",
  "\x00\x00", "\x50\xe8", "\x50\xe9", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x50\xef", "\x00\x00", "\x50\xf1", "\x50\xf2",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x50\xf6", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x50\xfa", "\x00\x00", "\x00\x00", "\x00\x00", "\x50\xfe",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x51\x03", "\x00\x00",
  "\x00\x00", "\x51\x06", "\x51\x07", "\x51\x08", "\x00\x00", "\x00\x00",
  "\x51\x0b", "\x51\x0c", "\x51\x0d", "\x51\x0e", "\x00\x00", "\x51\x10",
  "\x51\x17", "\x00\x00", "\x51\x19", "\x00\x00", "\x51\x1b", "\x51\x1c",
  "\x51\x1d", "\x51\x1e", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x51\x23", "\x00\x00", "\x00\x00", "\x00\x00", "\x51\x27", "\x51\x28",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x51\x2c", "\x51\x2d", "\x00\x00",
  "\x51\x2f", "\x00\x00", "\x51\x31", "\x00\x00", "\x51\x33", "\x51\x34",
  "\x51\x35", "\x00\x00", "\x00\x00", "\x51\x38", "\x51\x39", "\x51\x42",
  "\x51\x4a", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x51\x4f",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x51\x53", "\x00\x00", "\x51\x55",
  "\x00\x00", "\x51\x57", "\x51\x58", "\x51\x5f", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x51\x64", "\x00\x00", "\x51\x66", "\x51\x7e",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x51\x83", "\x51\x84",
  "\x51\x8b", "\x00\x00", "\x00\x00", "\x51\x8e", "\x51\x98", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x51\x9d", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x51\xa1", "\x00\x00", "\x51\xa3", "\x51\xad", "\x51\xb8",
  "\x00\x00", "\x51\xba", "\x00\x00", "\x51\xbc", "\x00\x00", "\x51\xbe",
  "\x51\xbf", "\x00\x00", "\x00\x00", "\x51\xc2", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x51\xc8", "\x51\xcf", "\x00\x00",
  "\x51\xd1", "\x51\xd2", "\x51\xd3", "\x00\x00", "\x51\xd5", "\x00\x00",
  "\x00\x00", "\x51\xd8", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x51\xde", "\x00\x00", "\x00\x00", "\x00\x00", "\x51\xe2",
  "\x00\x00", "\x00\x00", "\x51\xe5", "\x51\xee", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x51\xf2", "\x51\xf3", "\x51\xf4", "\x00\x00", "\x00\x00",
  "\x51\xf7", "\x52\x01", "\x52\x02", "\x00\x00", "\x00\x00", "\x52\x05",
  "\x52\x12", "\x52\x13", "\x00\x00", "\x52\x15", "\x52\x16", "\x00\x00",
  "\x52\x18", "\x52\x22", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x52\x28", "\x52\x31", "\x52\x32", "\x00\x00", "\x00\x00",
  "\x52\x35", "\x52\x3c", "\x52\x45", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x52\x49", "\x52\x55", "\x00\x00", "\x52\x57", "\x52\x58", "\x00\x00",
  "\x52\x5a", "\x00\x00", "\x52\x5c", "\x00\x00", "\x00\x00", "\x52\x5f",
  "\x52\x60", "\x52\x61", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x52\x66", "\x52\x6e", "\x52\x77", "\x52\x78", "\x52\x79", "\x52\x80",
  "\x00\x00", "\x52\x82", "\x00\x00", "\x00\x00", "\x52\x85", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x52\x8a", "\x00\x00", "\x52\x8c",
  "\x52\x93", "\x00\x00", "\x52\x95", "\x52\x96", "\x52\x97", "\x52\x98",
  "\x00\x00", "\x52\x9a", "\x00\x00", "\x52\x9c", "\x52\xa4", "\x52\xa5",
  "\x52\xa6", "\x52\xa7", "\x52\xaf", "\x52\xb0", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x52\xb6", "\x52\xb7", "\x52\xb8",
  "\x00\x00", "\x52\xba", "\x52\xbb", "\x00\x00", "\x52\xbd", "\x00\x00",
  "\x00\x00", "\x52\xc0", "\x00\x00", "\x00\x00", "\x00\x00", "\x52\xc4",
  "\x00\x00", "\x52\xc6", "\x00\x00", "\x52\xc8", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x52\xcc", "\x00\x00", "\x00\x00", "\x52\xcf", "\x00\x00",
  "\x52\xd1", "\x00\x00", "\x00\x00", "\x52\xd4", "\x00\x00", "\x52\xd6",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x52\xdb", "\x52\xdc",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x52\xe1", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x52\xe5", "\x00\x00", "\x00\x00", "\x52\xe8",
  "\x52\xe9", "\x52\xea", "\x00\x00", "\x52\xec", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x52\xf0", "\x52\xf1", "\x00\x00", "\x00\x00", "\x52\xf4",
  "\x00\x00", "\x52\xf6", "\x52\xf7", "\x53\x00", "\x00\x00", "\x00\x00",
  "\x53\x03", "\x53\x0a", "\x53\x0b", "\x53\x0c", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x53\x11", "\x00\x00", "\x53\x13", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x53\x18", "\x00\x00", "\x00\x00",
  "\x53\x1b", "\x53\x1c", "\x00\x00", "\x53\x1e", "\x53\x1f", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x53\x25", "\x00\x00",
  "\x53\x27", "\x53\x28", "\x53\x29", "\x00\x00", "\x53\x2b", "\x53\x2c",
  "\x53\x2d", "\x00\x00", "\x00\x00", "\x53\x30", "\x00\x00", "\x53\x32",
  "\x00\x00", "\x00\x00", "\x53\x35", "\x53\x3c", "\x53\x3d", "\x53\x3e",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x53\x42", "\x53\x4b", "\x53\x4c",
  "\x53\x59", "\x00\x00", "\x53\x5b", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x53\x61", "\x00\x00", "\x53\x63", "\x00\x00",
  "\x53\x65", "\x53\x6c", "\x53\x6d", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x53\x72", "\x53\x79", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x53\x7e", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x53\x83", "\x00\x00", "\x00\x00", "\x00\x00", "\x53\x87", "\x53\x88",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x53\x8e",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x53\x93", "\x53\x94",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x53\x99", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x53\x9d", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x53\xa1", "\x00\x00", "\x00\x00", "\x53\xa4", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x53\xaa", "\x53\xab", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x53\xaf", "\x00\x00", "\x00\x00", "\x53\xb2",
  "\x00\x00", "\x53\xb4", "\x53\xb5", "\x00\x00", "\x53\xb7", "\x53\xb8",
  "\x00\x00", "\x53\xba", "\x00\x00", "\x00\x00", "\x53\xbd", "\x00\x00",
  "\x00\x00", "\x53\xc0", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x53\xc5", "\x53\xcf", "\x00\x00", "\x00\x00", "\x53\xd2", "\x53\xd3",
  "\x00\x00", "\x53\xd5", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x53\xda", "\x00\x00", "\x00\x00", "\x53\xdd", "\x53\xde", "\x00\x00",
  "\x53\xe0", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x53\xe6", "\x53\xe7", "\x53\xf5", "\x54\x02", "\x54\x13", "\x54\x1a",
  "\x54\x21", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x54\x27", "\x54\x28", "\x00\x00", "\x54\x2a", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x54\x2f", "\x00\x00", "\x54\x31", "\x00\x00",
  "\x00\x00", "\x54\x34", "\x54\x35", "\x54\x43", "\x54\x44", "\x00\x00",
  "\x00\x00", "\x54\x47", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x54\x4d", "\x00\x00", "\x54\x4f", "\x54\x5e", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x54\x62", "\x00\x00", "\x54\x64", "\x00\x00",
  "\x54\x66", "\x54\x67", "\x00\x00", "\x54\x69", "\x00\x00", "\x54\x6b",
  "\x00\x00", "\x54\x6d", "\x54\x6e", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x54\x74", "\x54\x7f", "\x00\x00", "\x54\x81",
  "\x00\x00", "\x54\x83", "\x00\x00", "\x54\x85", "\x00\x00", "\x00\x00",
  "\x54\x88", "\x54\x89", "\x00\x00", "\x00\x00", "\x00\x00", "\x54\x8d",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x54\x91", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x54\x95", "\x54\x96", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x54\x9c", "\x00\x00", "\x00\x00", "\x54\x9f",
  "\x00\x00", "\x54\xa1", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x54\xa6", "\x54\xa7", "\x00\x00", "\x54\xa9", "\x54\xaa", "\x00\x00",
  "\x00\x00", "\x54\xad", "\x54\xae", "\x00\x00", "\x00\x00", "\x54\xb1",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x54\xb7",
  "\x00\x00", "\x54\xb9", "\x54\xba", "\x54\xbb", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x54\xbf", "\x54\xc6", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x54\xca", "\x00\x00", "\x00\x00", "\x54\xcd", "\x54\xce", "\x54\xe0",
  "\x54\xea", "\x00\x00", "\x54\xec", "\x00\x00", "\x00\x00", "\x54\xef",
  "\x54\xf6", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x54\xfc", "\x00\x00", "\x54\xfe", "\x54\xff", "\x55\x00", "\x55\x01",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x55\x05", "\x00\x00", "\x00\x00",
  "\x55\x08", "\x55\x09", "\x00\x00", "\x00\x00", "\x55\x0c", "\x55\x0d",
  "\x55\x0e", "\x55\x15", "\x55\x2a", "\x55\x2b", "\x55\x32", "\x00\x00",
  "\x00\x00", "\x55\x35", "\x55\x36", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x55\x3b", "\x55\x3c", "\x55\x3d", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x55\x41", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x55\x47", "\x00\x00", "\x55\x49", "\x55\x4a", "\x00\x00",
  "\x00\x00", "\x55\x4d", "\x00\x00", "\x00\x00", "\x55\x50", "\x55\x51",
  "\x55\x58", "\x00\x00", "\x55\x5a", "\x55\x5b", "\x00\x00", "\x00\x00",
  "\x55\x5e", "\x00\x00", "\x55\x60", "\x55\x61", "\x00\x00", "\x00\x00",
  "\x55\x64", "\x00\x00", "\x55\x66", "\x55\x7f", "\x00\x00", "\x55\x81",
  "\x55\x82", "\x00\x00", "\x00\x00", "\x00\x00", "\x55\x86", "\x00\x00",
  "\x55\x88", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x55\x8e", "\x55\x8f", "\x00\x00", "\x55\x91", "\x55\x92", "\x55\x93",
  "\x55\x94", "\x00\x00", "\x00\x00", "\x55\x97", "\x55\xa3", "\x55\xa4",
  "\x55\xad", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x55\xb2",
  "\x55\xbf", "\x00\x00", "\x55\xc1", "\x00\x00", "\x55\xc3", "\x00\x00",
  "\x00\x00", "\x55\xc6", "\x00\x00", "\x00\x00", "\x55\xc9", "\x00\x00",
  "\x55\xcb", "\x55\xcc", "\x00\x00", "\x55\xce", "\x00\x00", "\x00\x00",
  "\x55\xd1", "\x55\xd2", "\x55\xd3", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x55\xd7", "\x55\xd8", "\x00\x00", "\x00\x00", "\x55\xdb", "\x00\x00",
  "\x00\x00", "\x55\xde", "\x00\x00", "\x00\x00", "\x00\x00", "\x55\xe2",
  "\x55\xe9", "\x55\xf6", "\x55\xff", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x56\x05", "\x00\x00", "\x00\x00", "\x56\x08",
  "\x00\x00", "\x56\x0a", "\x00\x00", "\x00\x00", "\x56\x0d", "\x56\x0e",
  "\x56\x0f", "\x56\x10", "\x56\x11", "\x56\x12", "\x56\x19", "\x56\x2c",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x56\x30", "\x00\x00", "\x00\x00",
  "\x56\x33", "\x00\x00", "\x56\x35", "\x00\x00", "\x56\x37", "\x00\x00",
  "\x56\x39", "\x00\x00", "\x56\x3b", "\x56\x3c", "\x56\x3d", "\x00\x00",
  "\x56\x3f", "\x56\x40", "\x56\x41", "\x00\x00", "\x56\x43", "\x56\x44",
  "\x00\x00", "\x56\x46", "\x00\x00", "\x00\x00", "\x56\x49", "\x00\x00",
  "\x56\x4b", "\x00\x00", "\x56\x4d", "\x00\x00", "\x56\x4f", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x56\x54", "\x56\x5e", "\x00\x00",
  "\x56\x60", "\x56\x61", "\x56\x62", "\x56\x63", "\x00\x00", "\x00\x00",
  "\x56\x66", "\x00\x00", "\x00\x00", "\x56\x69", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x56\x6d", "\x00\x00", "\x56\x6f", "\x00\x00", "\x56\x71",
  "\x56\x72", "\x00\x00", "\x00\x00", "\x56\x75", "\x56\x84", "\x56\x85",
  "\x00\x00", "\x00\x00", "\x56\x88", "\x00\x00", "\x00\x00", "\x56\x8b",
  "\x56\x8c", "\x56\x95", "\x00\x00", "\x00\x00", "\x00\x00", "\x56\x99",
  "\x56\x9a", "\x00\x00", "\x00\x00", "\x56\x9d", "\x56\x9e", "\x56\x9f",
  "\x56\xa6", "\x56\xa7", "\x56\xa8", "\x56\xa9", "\x00\x00", "\x56\xab",
  "\x56\xac", "\x56\xad", "\x00\x00", "\x00\x00", "\x00\x00", "\x56\xb1",
  "\x00\x00", "\x56\xb3", "\x00\x00", "\x00\x00", "\x00\x00", "\x56\xb7",
  "\x56\xbe", "\x56\xc5", "\x00\x00", "\x00\x00", "\x00\x00", "\x56\xc9",
  "\x56\xca", "\x56\xcb", "\x56\xcc", "\x56\xcd", "\x00\x00", "\x56\xcf",
  "\x56\xd0", "\x56\xd9", "\x00\x00", "\x00\x00", "\x56\xdc", "\x56\xdd",
  "\x00\x00", "\x56\xdf", "\x00\x00", "\x56\xe1", "\x00\x00", "\x00\x00",
  "\x56\xe4", "\x56\xe5", "\x56\xe6", "\x56\xe7", "\x56\xe8", "\x00\x00",
  "\x00\x00", "\x56\xeb", "\x00\x00", "\x56\xed", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x56\xf1", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x56\xf6", "\x56\xf7", "\x57\x01", "\x57\x02", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x57\x07", "\x00\x00", "\x00\x00", "\x57\x0a",
  "\x00\x00", "\x57\x0c", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x57\x11", "\x00\x00", "\x00\x00", "\x00\x00", "\x57\x15", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x57\x1a", "\x57\x1b", "\x00\x00",
  "\x57\x1d", "\x00\x00", "\x00\x00", "\x57\x20", "\x00\x00", "\x57\x22",
  "\x57\x23", "\x57\x24", "\x57\x25", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x57\x29", "\x57\x2a", "\x00\x00", "\x57\x2c", "\x00\x00", "\x57\x2e",
  "\x57\x2f", "\x00\x00", "\x00\x00", "\x00\x00", "\x57\x33", "\x57\x34",
  "\x57\x3d", "\x57\x3e", "\x57\x3f", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x57\x45", "\x57\x46", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x57\x4c", "\x57\x4d", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x57\x52", "\x57\x62", "\x00\x00",
  "\x00\x00", "\x57\x65", "\x00\x00", "\x57\x67", "\x57\x68", "\x00\x00",
  "\x00\x00", "\x57\x6b", "\x00\x00", "\x57\x6d", "\x57\x6e", "\x57\x6f",
  "\x57\x70", "\x57\x71", "\x00\x00", "\x57\x73", "\x57\x74", "\x57\x75",
  "\x00\x00", "\x57\x77", "\x00\x00", "\x57\x79", "\x57\x7a", "\x57\x7b",
  "\x57\x7c", "\x00\x00", "\x57\x7e", "\x00\x00", "\x00\x00", "\x57\x81",
  "\x00\x00", "\x57\x83", "\x57\x8c", "\x57\x94", "\x57\x95", "\x00\x00",
  "\x57\x97", "\x00\x00", "\x57\x99", "\x57\x9a", "\x00\x00", "\x57\x9c",
  "\x57\x9d", "\x57\x9e", "\x57\x9f", "\x00\x00", "\x57\xa1", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x57\xa7", "\x57\xa8",
  "\x57\xa9", "\x00\x00", "\x00\x00", "\x57\xac", "\x57\xb8", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x57\xbd", "\x57\xc7", "\x57\xc8",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x57\xcc", "\x00\x00", "\x00\x00",
  "\x57\xcf", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x57\xd5", "\x57\xdd", "\x57\xde", "\x00\x00", "\x00\x00", "\x57\xe1",
  "\x00\x00", "\x00\x00", "\x57\xe4", "\x00\x00", "\x57\xe6", "\x57\xe7",
  "\x00\x00", "\x57\xe9", "\x00\x00", "\x00\x00", "\x00\x00", "\x57\xed",
  "\x00\x00", "\x00\x00", "\x57\xf0", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x57\xf5", "\x57\xf6", "\x00\x00", "\x57\xf8", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x57\xfd", "\x57\xfe", "\x57\xff",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x58\x03", "\x58\x04", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x58\x08", "\x58\x09", "\x00\x00", "\x00\x00",
  "\x58\x0c", "\x58\x0d", "\x58\x1b", "\x00\x00", "\x00\x00", "\x58\x1e",
  "\x58\x1f", "\x58\x20", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x58\x26", "\x58\x27", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x58\x2d", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x58\x32", "\x58\x39", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x58\x3f", "\x58\x49", "\x00\x00", "\x00\x00",
  "\x58\x4c", "\x58\x4d", "\x00\x00", "\x58\x4f", "\x58\x50", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x58\x55", "\x58\x5f", "\x00\x00",
  "\x58\x61", "\x00\x00", "\x00\x00", "\x58\x64", "\x00\x00", "\x00\x00",
  "\x58\x67", "\x58\x68", "\x58\x78", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x58\x7c", "\x00\x00", "\x00\x00", "\x58\x7f", "\x58\x80", "\x58\x81",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x58\x87",
  "\x58\x88", "\x58\x89", "\x58\x8a", "\x00\x00", "\x58\x8c", "\x58\x8d",
  "\x00\x00", "\x58\x8f", "\x58\x90", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x58\x94", "\x00\x00", "\x58\x96", "\x58\x9d", "\x00\x00", "\x00\x00",
  "\x58\xa0", "\x58\xa1", "\x58\xa2", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x58\xa6", "\x00\x00", "\x00\x00", "\x58\xa9", "\x58\xb1", "\x58\xb2",
  "\x58\xbc", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x58\xc2", "\x00\x00", "\x58\xc4", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x58\xc8", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x58\xcd",
  "\x58\xce", "\x00\x00", "\x58\xd0", "\x00\x00", "\x58\xd2", "\x00\x00",
  "\x58\xd4", "\x00\x00", "\x58\xd6", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x58\xda", "\x00\x00", "\x00\x00", "\x58\xdd", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x58\xe1", "\x58\xe2", "\x58\xe9", "\x58\xf3", "\x59\x05",
  "\x59\x06", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x59\x0b",
  "\x59\x0c", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x59\x12", "\x59\x13", "\x59\x14", "\x59\x1d", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x59\x21", "\x00\x00", "\x59\x23", "\x59\x24", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x59\x28", "\x59\x2f", "\x59\x30", "\x00\x00",
  "\x00\x00", "\x59\x33", "\x00\x00", "\x59\x35", "\x59\x36", "\x59\x3f",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x59\x43", "\x00\x00", "\x00\x00",
  "\x59\x46", "\x59\x52", "\x59\x53", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x59\x59", "\x00\x00", "\x59\x5b", "\x00\x00",
  "\x59\x5d", "\x59\x5e", "\x59\x5f", "\x00\x00", "\x59\x61", "\x00\x00",
  "\x59\x63", "\x59\x6b", "\x00\x00", "\x59\x6d", "\x00\x00", "\x59\x6f",
  "\x00\x00", "\x00\x00", "\x59\x72", "\x00\x00", "\x00\x00", "\x59\x75",
  "\x59\x76", "\x00\x00", "\x00\x00", "\x59\x79", "\x00\x00", "\x59\x7b",
  "\x59\x7c", "\x59\x8b", "\x59\x8c", "\x00\x00", "\x59\x8e", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x59\x92", "\x00\x00", "\x00\x00", "\x59\x95",
  "\x00\x00", "\x59\x97", "\x59\x9f", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x59\xa4", "\x00\x00", "\x00\x00", "\x59\xa7", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x59\xad", "\x59\xae",
  "\x59\xaf", "\x59\xb0", "\x00\x00", "\x00\x00", "\x59\xb3", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x59\xb7", "\x00\x00", "\x00\x00", "\x59\xba",
  "\x00\x00", "\x59\xbc", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x59\xc1", "\x00\x00", "\x59\xc3", "\x59\xc4", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x59\xc8", "\x00\x00", "\x59\xca", "\x00\x00", "\x00\x00",
  "\x59\xcd", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x59\xd2",
  "\x59\xdd", "\x59\xde", "\x59\xdf", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x59\xe3", "\x59\xe4", "\x00\x00", "\x00\x00", "\x59\xe7", "\x59\xee",
  "\x59\xef", "\x00\x00", "\x59\xf1", "\x59\xf2", "\x00\x00", "\x59\xf4",
  "\x00\x00", "\x00\x00", "\x59\xf7", "\x5a\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x5a\x04", "\x5a\x0c", "\x5a\x0d", "\x5a\x0e", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x5a\x12", "\x5a\x13", "\x5a\x1e", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x5a\x23", "\x5a\x24", "\x00\x00",
  "\x00\x00", "\x5a\x27", "\x5a\x28", "\x00\x00", "\x5a\x2a", "\x00\x00",
  "\x00\x00", "\x5a\x2d", "\x00\x00", "\x00\x00", "\x5a\x30", "\x5a\x44",
  "\x5a\x45", "\x00\x00", "\x5a\x47", "\x5a\x48", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x5a\x4c", "\x00\x00", "\x00\x00", "\x00\x00", "\x5a\x50",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x5a\x55", "\x5a\x5e",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x5a\x63", "\x00\x00",
  "\x5a\x65", "\x00\x00", "\x5a\x67", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x5a\x6d", "\x5a\x77", "\x00\x00", "\x00\x00",
  "\x5a\x7a", "\x5a\x7b", "\x00\x00", "\x00\x00", "\x5a\x7e", "\x5a\x8b",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x5a\x90", "\x00\x00",
  "\x00\x00", "\x5a\x93", "\x00\x00", "\x00\x00", "\x5a\x96", "\x00\x00",
  "\x00\x00", "\x5a\x99", "\x00\x00", "\x00\x00", "\x5a\x9c", "\x00\x00",
  "\x5a\x9e", "\x5a\x9f", "\x5a\xa0", "\x00\x00", "\x5a\xa2", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x5a\xa7", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x5a\xac", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x5a\xb1", "\x5a\xb2", "\x5a\xb3", "\x00\x00", "\x5a\xb5",
  "\x00\x00", "\x00\x00", "\x5a\xb8", "\x00\x00", "\x5a\xba", "\x5a\xbb",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x5a\xbf", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x5a\xc4", "\x00\x00", "\x5a\xc6", "\x00\x00",
  "\x5a\xc8", "\x5a\xcf", "\x5a\xda", "\x00\x00", "\x5a\xdc", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x5a\xe0", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x5a\xe5", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x5a\xea", "\x00\x00", "\x00\x00", "\x00\x00", "\x5a\xee", "\x5a\xf5",
  "\x5a\xf6", "\x5a\xfd", "\x00\x00", "\x00\x00", "\x5b\x00", "\x5b\x01",
  "\x5b\x08", "\x5b\x17", "\x00\x00", "\x5b\x19", "\x00\x00", "\x5b\x1b",
  "\x00\x00", "\x5b\x1d", "\x00\x00", "\x00\x00", "\x00\x00", "\x5b\x21",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x5b\x25", "\x5b\x2d", "\x5b\x34",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x5b\x38", "\x5b\x41", "\x5b\x4b",
  "\x5b\x4c", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x5b\x52", "\x00\x00", "\x00\x00", "\x00\x00", "\x5b\x56", "\x5b\x5e",
  "\x5b\x68", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x5b\x6e", "\x5b\x6f", "\x5b\x7c", "\x5b\x7d", "\x5b\x7e", "\x5b\x7f",
  "\x00\x00", "\x5b\x81", "\x00\x00", "\x00\x00", "\x5b\x84", "\x00\x00",
  "\x5b\x86", "\x00\x00", "\x00\x00", "\x00\x00", "\x5b\x8a", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x5b\x8e", "\x00\x00", "\x5b\x90", "\x5b\x91",
  "\x00\x00", "\x5b\x93", "\x5b\x94", "\x00\x00", "\x5b\x96", "\x5b\xa8",
  "\x5b\xa9", "\x00\x00", "\x00\x00", "\x5b\xac", "\x5b\xad", "\x00\x00",
  "\x5b\xaf", "\x00\x00", "\x5b\xb1", "\x5b\xb2", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x5b\xb7", "\x00\x00", "\x00\x00", "\x5b\xba",
  "\x00\x00", "\x5b\xbc", "\x00\x00", "\x00\x00", "\x00\x00", "\x5b\xc0",
  "\x5b\xc1", "\x5b\xcd", "\x00\x00", "\x5b\xcf", "\x5b\xd6", "\x5b\xd7",
  "\x5b\xd8", "\x5b\xd9", "\x5b\xda", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x5b\xe0", "\x5b\xef", "\x00\x00", "\x5b\xf1",
  "\x00\x00", "\x00\x00", "\x5b\xf4", "\x5b\xfd", "\x5c\x0c", "\x5c\x17",
  "\x5c\x1e", "\x5c\x1f", "\x00\x00", "\x00\x00", "\x00\x00", "\x5c\x23",
  "\x00\x00", "\x00\x00", "\x5c\x26", "\x00\x00", "\x00\x00", "\x5c\x29",
  "\x00\x00", "\x5c\x2b", "\x5c\x2c", "\x00\x00", "\x5c\x2e", "\x00\x00",
  "\x5c\x30", "\x00\x00", "\x5c\x32", "\x00\x00", "\x00\x00", "\x5c\x35",
  "\x5c\x36", "\x5c\x59", "\x5c\x5a", "\x00\x00", "\x5c\x5c", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x5c\x62", "\x5c\x63",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x5c\x67", "\x5c\x68", "\x5c\x69",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x5c\x6d", "\x00\x00", "\x00\x00",
  "\x5c\x70", "\x00\x00", "\x00\x00", "\x00\x00", "\x5c\x74", "\x5c\x75",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x5c\x7a", "\x5c\x7b",
  "\x5c\x7c", "\x5c\x7d", "\x5c\x87", "\x5c\x88", "\x00\x00", "\x5c\x8a",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x5c\x8f", "\x00\x00",
  "\x00\x00", "\x5c\x92", "\x5c\x9d", "\x00\x00", "\x5c\x9f", "\x5c\xa0",
  "\x00\x00", "\x5c\xa2", "\x5c\xa3", "\x00\x00", "\x00\x00", "\x5c\xa6",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x5c\xaa", "\x5c\xb2", "\x00\x00",
  "\x5c\xb4", "\x5c\xb5", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x5c\xba", "\x5c\xc9", "\x00\x00", "\x5c\xcb", "\x5c\xd2", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x5c\xd7", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x5c\xdd", "\x5c\xee", "\x00\x00",
  "\x00\x00", "\x5c\xf1", "\x5c\xf2", "\x00\x00", "\x5c\xf4", "\x5d\x01",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x5d\x06", "\x5d\x0d",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x5d\x12", "\x5d\x23",
  "\x5d\x24", "\x00\x00", "\x5d\x26", "\x5d\x27", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x5d\x2b", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x5d\x31", "\x00\x00", "\x00\x00", "\x5d\x34", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x5d\x39", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x5d\x3d", "\x00\x00", "\x5d\x3f", "\x00\x00", "\x00\x00",
  "\x5d\x42", "\x5d\x43", "\x00\x00", "\x00\x00", "\x5d\x46", "\x00\x00",
  "\x5d\x48", "\x00\x00", "\x5d\x4a", "\x5d\x51", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x5d\x55", "\x00\x00", "\x00\x00", "\x00\x00", "\x5d\x59",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x5d\x5f",
  "\x5d\x60", "\x5d\x61", "\x5d\x62", "\x00\x00", "\x5d\x64", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x5d\x6a", "\x00\x00",
  "\x00\x00", "\x5d\x6d", "\x00\x00", "\x00\x00", "\x5d\x70", "\x5d\x79",
  "\x5d\x7a", "\x00\x00", "\x00\x00", "\x00\x00", "\x5d\x7e", "\x5d\x7f",
  "\x00\x00", "\x5d\x81", "\x00\x00", "\x5d\x83", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x5d\x88", "\x00\x00", "\x5d\x8a", "\x5d\x92",
  "\x5d\x93", "\x5d\x94", "\x5d\x95", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x5d\x99", "\x00\x00", "\x5d\x9b", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x5d\x9f", "\x5d\xa0", "\x5d\xa7", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x5d\xab", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x5d\xb0",
  "\x00\x00", "\x5d\xb2", "\x00\x00", "\x5d\xb4", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x5d\xb8", "\x5d\xb9", "\x5d\xc3", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x5d\xc7", "\x00\x00", "\x00\x00", "\x00\x00", "\x5d\xcb",
  "\x00\x00", "\x00\x00", "\x5d\xce", "\x00\x00", "\x5d\xd0", "\x5d\xd8",
  "\x5d\xd9", "\x5d\xe0", "\x00\x00", "\x00\x00", "\x00\x00", "\x5d\xe4",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x5d\xe9", "\x5d\xf8",
  "\x5d\xf9", "\x5e\x00", "\x5e\x07", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x5e\x0d", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x5e\x12", "\x00\x00", "\x5e\x14", "\x5e\x15", "\x00\x00",
  "\x00\x00", "\x5e\x18", "\x5e\x1f", "\x5e\x20", "\x5e\x28", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x5e\x2e", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x5e\x32", "\x00\x00", "\x00\x00", "\x5e\x35",
  "\x5e\x3e", "\x5e\x49", "\x00\x00", "\x5e\x4b", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x5e\x50", "\x5e\x51", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x5e\x56", "\x00\x00", "\x5e\x58", "\x00\x00",
  "\x00\x00", "\x5e\x5b", "\x5e\x5c", "\x00\x00", "\x5e\x5e", "\x5e\x68",
  "\x00\x00", "\x5e\x6a", "\x5e\x6b", "\x5e\x6c", "\x5e\x6d", "\x5e\x6e",
  "\x00\x00", "\x5e\x70", "\x5e\x80", "\x5e\x8b", "\x00\x00", "\x00\x00",
  "\x5e\x8e", "\x5e\xa2", "\x00\x00", "\x5e\xa4", "\x5e\xa5", "\x00\x00",
  "\x00\x00", "\x5e\xa8", "\x00\x00", "\x5e\xaa", "\x00\x00", "\x5e\xac",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x5e\xb1", "\x00\x00",
  "\x5e\xb3", "\x5e\xbd", "\x5e\xbe", "\x5e\xbf", "\x5e\xc6", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x5e\xcb", "\x5e\xcc", "\x00\x00",
  "\x5e\xce", "\x00\x00", "\x00\x00", "\x5e\xd1", "\x5e\xd2", "\x00\x00",
  "\x5e\xd4", "\x5e\xd5", "\x5e\xdc", "\x00\x00", "\x5e\xde", "\x5e\xe5",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x5e\xeb",
  "\x5f\x02", "\x00\x00", "\x00\x00", "\x00\x00", "\x5f\x06", "\x5f\x07",
  "\x5f\x08", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x5f\x0e", "\x5f\x19", "\x00\x00", "\x00\x00", "\x5f\x1c", "\x5f\x1d",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x5f\x21", "\x5f\x22", "\x5f\x23",
  "\x5f\x24", "\x00\x00", "\x00\x00", "\x00\x00", "\x5f\x28", "\x00\x00",
  "\x00\x00", "\x5f\x2b", "\x5f\x2c", "\x00\x00", "\x5f\x2e", "\x00\x00",
  "\x5f\x30", "\x00\x00", "\x00\x00", "\x00\x00", "\x5f\x34", "\x00\x00",
  "\x5f\x36", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x5f\x3b",
  "\x00\x00", "\x5f\x3d", "\x00\x00", "\x5f\x3f", "\x5f\x40", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x5f\x44", "\x5f\x45", "\x00\x00", "\x5f\x47",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x5f\x4d",
  "\x00\x00", "\x00\x00", "\x5f\x50", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x5f\x54", "\x00\x00", "\x00\x00", "\x00\x00", "\x5f\x58", "\x00\x00",
  "\x00\x00", "\x5f\x5b", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x5f\x60", "\x00\x00", "\x00\x00", "\x5f\x63", "\x5f\x64", "\x00\x00",
  "\x00\x00", "\x5f\x67", "\x5f\x6f", "\x00\x00", "\x00\x00", "\x5f\x72",
  "\x00\x00", "\x5f\x74", "\x5f\x75", "\x00\x00", "\x00\x00", "\x5f\x78",
  "\x00\x00", "\x5f\x7a", "\x00\x00", "\x00\x00", "\x5f\x7d", "\x5f\x7e",
  "\x5f\x89", "\x00\x00", "\x00\x00", "\x00\x00", "\x5f\x8d", "\x00\x00",
  "\x5f\x8f", "\x5f\x96", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x5f\x9c", "\x5f\x9d", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x5f\xa2", "\x00\x00", "\x5f\xa4", "\x00\x00", "\x00\x00",
  "\x5f\xa7", "\x00\x00", "\x00\x00", "\x00\x00", "\x5f\xab", "\x5f\xac",
  "\x00\x00", "\x00\x00", "\x5f\xaf", "\x5f\xb0", "\x5f\xb1", "\x5f\xb8",
  "\x5f\xc4", "\x00\x00", "\x00\x00", "\x5f\xc7", "\x5f\xc8", "\x5f\xc9",
  "\x00\x00", "\x5f\xcb", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x5f\xd0", "\x5f\xd1", "\x5f\xd2", "\x5f\xd3", "\x5f\xd4", "\x5f\xde",
  "\x00\x00", "\x00\x00", "\x5f\xe1", "\x5f\xe2", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x5f\xe8", "\x5f\xe9", "\x5f\xea",
  "\x00\x00", "\x5f\xec", "\x5f\xed", "\x5f\xee", "\x5f\xef", "\x00\x00",
  "\x00\x00", "\x5f\xf2", "\x5f\xf3", "\x00\x00", "\x00\x00", "\x5f\xf6",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x5f\xfa", "\x00\x00", "\x5f\xfc",
  "\x60\x07", "\x00\x00", "\x00\x00", "\x60\x0a", "\x00\x00", "\x00\x00",
  "\x60\x0d", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x60\x13", "\x60\x14", "\x00\x00", "\x00\x00", "\x60\x17", "\x60\x18",
  "\x00\x00", "\x60\x1a", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x60\x1f", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x60\x24",
  "\x60\x2d", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x60\x33", "\x00\x00", "\x60\x35", "\x60\x40", "\x60\x47", "\x60\x48",
  "\x60\x49", "\x00\x00", "\x00\x00", "\x60\x4c", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x60\x51", "\x00\x00", "\x00\x00", "\x60\x54",
  "\x00\x00", "\x60\x56", "\x60\x57", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x60\x5d", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x60\x61", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x60\x67", "\x60\x71", "\x60\x7e", "\x60\x7f", "\x00\x00", "\x00\x00",
  "\x60\x82", "\x00\x00", "\x00\x00", "\x00\x00", "\x60\x86", "\x00\x00",
  "\x60\x88", "\x00\x00", "\x60\x8a", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x60\x8e", "\x00\x00", "\x00\x00", "\x60\x91", "\x00\x00", "\x60\x93",
  "\x00\x00", "\x60\x95", "\x00\x00", "\x00\x00", "\x60\x98", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x60\x9d", "\x60\x9e", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x60\xa2", "\x00\x00", "\x60\xa4", "\x60\xa5",
  "\x00\x00", "\x00\x00", "\x60\xa8", "\x60\xb0", "\x60\xb1", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x60\xb7", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x60\xbb", "\x00\x00", "\x00\x00", "\x60\xbe",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x60\xc2", "\x00\x00", "\x60\xc4",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x60\xc8", "\x60\xc9", "\x60\xca",
  "\x60\xcb", "\x00\x00", "\x00\x00", "\x60\xce", "\x60\xcf", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x60\xd4", "\x60\xd5", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x60\xd9", "\x00\x00", "\x60\xdb", "\x00\x00",
  "\x60\xdd", "\x60\xde", "\x00\x00", "\x00\x00", "\x00\x00", "\x60\xe2",
  "\x00\x00", "\x00\x00", "\x60\xe5", "\x60\xf2", "\x00\x00", "\x00\x00",
  "\x60\xf5", "\x00\x00", "\x00\x00", "\x60\xf8", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x60\xfc", "\x60\xfd", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x61\x02", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x61\x07", "\x00\x00", "\x00\x00", "\x61\x0a", "\x00\x00", "\x61\x0c",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x61\x10", "\x61\x11", "\x61\x12",
  "\x61\x13", "\x61\x14", "\x00\x00", "\x61\x16", "\x61\x17", "\x00\x00",
  "\x61\x19", "\x00\x00", "\x00\x00", "\x61\x1c", "\x00\x00", "\x61\x1e",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x61\x22", "\x61\x2a", "\x61\x2b",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x61\x30", "\x61\x31",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x61\x35", "\x61\x36", "\x61\x37",
  "\x00\x00", "\x61\x39", "\x61\x41", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x61\x45", "\x61\x46", "\x00\x00", "\x00\x00", "\x61\x49", "\x61\x5e",
  "\x00\x00", "\x61\x60", "\x61\x6c", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x61\x72", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x61\x78", "\x00\x00", "\x00\x00", "\x61\x7b",
  "\x61\x7c", "\x00\x00", "\x00\x00", "\x61\x7f", "\x61\x80", "\x61\x81",
  "\x00\x00", "\x61\x83", "\x61\x84", "\x61\x8b", "\x00\x00", "\x61\x8d",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x61\x92", "\x61\x93",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x61\x97", "\x61\x98", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x61\x9c", "\x61\x9d", "\x00\x00", "\x61\x9f",
  "\x61\xa0", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x61\xa5",
  "\x00\x00", "\x00\x00", "\x61\xa8", "\x00\x00", "\x61\xaa", "\x00\x00",
  "\x00\x00", "\x61\xad", "\x61\xb8", "\x61\xb9", "\x00\x00", "\x00\x00",
  "\x61\xbc", "\x00\x00", "\x00\x00", "\x00\x00", "\x61\xc0", "\x61\xc1",
  "\x61\xc2", "\x61\xce", "\x61\xcf", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x61\xd5", "\x61\xdc", "\x61\xdd", "\x61\xde",
  "\x61\xdf", "\x00\x00", "\x61\xe1", "\x61\xe2", "\x00\x00", "\x00\x00",
  "\x61\xe5", "\x00\x00", "\x61\xe7", "\x00\x00", "\x61\xe9", "\x00\x00",
  "\x00\x00", "\x61\xec", "\x61\xed", "\x00\x00", "\x61\xef", "\x62\x01",
  "\x00\x00", "\x62\x03", "\x62\x04", "\x00\x00", "\x00\x00", "\x62\x07",
  "\x62\x13", "\x00\x00", "\x62\x15", "\x62\x1c", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x62\x20", "\x00\x00", "\x62\x22", "\x62\x23", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x62\x27", "\x00\x00", "\x62\x29", "\x00\x00",
  "\x62\x2b", "\x62\x39", "\x00\x00", "\x00\x00", "\x00\x00", "\x62\x3d",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x62\x42", "\x62\x43",
  "\x62\x44", "\x00\x00", "\x62\x46", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x62\x4c", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x62\x50", "\x62\x51", "\x62\x52", "\x00\x00", "\x62\x54", "\x00\x00",
  "\x62\x56", "\x00\x00", "\x00\x00", "\x00\x00", "\x62\x5a", "\x00\x00",
  "\x62\x5c", "\x62\x64", "\x62\x6d", "\x00\x00", "\x62\x6f", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x62\x73", "\x62\x7a", "\x00\x00", "\x00\x00",
  "\x62\x7d", "\x62\x8d", "\x62\x8e", "\x62\x8f", "\x62\x90", "\x62\xa6",
  "\x00\x00", "\x62\xa8", "\x62\xb3", "\x00\x00", "\x00\x00", "\x62\xb6",
  "\x62\xb7", "\x00\x00", "\x00\x00", "\x62\xba", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x62\xbe", "\x62\xbf", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x62\xc4", "\x62\xce", "\x62\xd5", "\x62\xd6", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x62\xda", "\x62\xea", "\x62\xf2", "\x00\x00",
  "\x62\xf4", "\x62\xfc", "\x62\xfd", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x63\x03", "\x63\x04", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x63\x0a", "\x63\x0b", "\x00\x00",
  "\x63\x0d", "\x00\x00", "\x00\x00", "\x63\x10", "\x00\x00", "\x00\x00",
  "\x63\x13", "\x00\x00", "\x00\x00", "\x63\x16", "\x00\x00", "\x63\x18",
  "\x63\x29", "\x63\x2a", "\x00\x00", "\x00\x00", "\x63\x2d", "\x63\x35",
  "\x63\x36", "\x00\x00", "\x00\x00", "\x63\x39", "\x00\x00", "\x00\x00",
  "\x63\x3c", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x63\x41",
  "\x63\x42", "\x63\x43", "\x63\x44", "\x00\x00", "\x63\x46", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x63\x4a", "\x63\x4b", "\x00\x00", "\x00\x00",
  "\x63\x4e", "\x00\x00", "\x00\x00", "\x00\x00", "\x63\x52", "\x63\x53",
  "\x63\x54", "\x00\x00", "\x00\x00", "\x00\x00", "\x63\x58", "\x00\x00",
  "\x00\x00", "\x63\x5b", "\x63\x65", "\x63\x66", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x63\x6c", "\x63\x6d", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x63\x71", "\x00\x00", "\x00\x00", "\x63\x74",
  "\x63\x75", "\x00\x00", "\x00\x00", "\x63\x78", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x63\x7c", "\x63\x7d", "\x00\x00", "\x63\x7f", "\x00\x00",
  "\x00\x00", "\x63\x82", "\x00\x00", "\x63\x84", "\x00\x00", "\x00\x00",
  "\x63\x87", "\x00\x00", "\x00\x00", "\x63\x8a", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x63\x90", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x63\x94", "\x63\x95", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x63\x99", "\x63\x9a", "\x00\x00", "\x00\x00", "\x00\x00", "\x63\x9e",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x63\xa4",
  "\x00\x00", "\x63\xa6", "\x63\xad", "\x63\xae", "\x63\xaf", "\x63\xbd",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x63\xc1", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x63\xc5", "\x00\x00", "\x00\x00", "\x63\xc8", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x63\xce", "\x00\x00",
  "\x00\x00", "\x63\xd1", "\x00\x00", "\x63\xd3", "\x63\xd4", "\x63\xd5",
  "\x63\xdc", "\x00\x00", "\x00\x00", "\x00\x00", "\x63\xe0", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x63\xe5", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x63\xea", "\x00\x00", "\x63\xec", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x63\xf2", "\x63\xf3",
  "\x00\x00", "\x63\xf5", "\x00\x00", "\x00\x00", "\x63\xf8", "\x63\xf9",
  "\x64\x09", "\x64\x0a", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x64\x10", "\x00\x00", "\x64\x12", "\x00\x00", "\x64\x14",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x64\x18", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x64\x1e", "\x00\x00", "\x64\x20",
  "\x00\x00", "\x64\x22", "\x00\x00", "\x64\x24", "\x64\x25", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x64\x29", "\x64\x2a", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x64\x2f", "\x64\x30", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x64\x35", "\x64\x3d", "\x00\x00", "\x64\x3f",
  "\x64\x4b", "\x00\x00", "\x00\x00", "\x00\x00", "\x64\x4f", "\x00\x00",
  "\x64\x51", "\x64\x52", "\x64\x53", "\x64\x54", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x64\x5a", "\x64\x5b", "\x64\x5c",
  "\x64\x5d", "\x00\x00", "\x64\x5f", "\x64\x60", "\x64\x61", "\x00\x00",
  "\x64\x63", "\x64\x6d", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x64\x73", "\x64\x74", "\x64\x7b", "\x00\x00", "\x64\x7d",
  "\x64\x85", "\x00\x00", "\x64\x87", "\x64\x8f", "\x64\x90", "\x64\x91",
  "\x64\x98", "\x64\x99", "\x00\x00", "\x64\x9b", "\x00\x00", "\x64\x9d",
  "\x00\x00", "\x64\x9f", "\x00\x00", "\x64\xa1", "\x00\x00", "\x64\xa3",
  "\x00\x00", "\x00\x00", "\x64\xa6", "\x00\x00", "\x64\xa8", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x64\xac", "\x64\xb3", "\x64\xbd", "\x64\xbe",
  "\x64\xbf", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x64\xc4",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x64\xc9", "\x64\xca",
  "\x64\xcb", "\x64\xcc", "\x00\x00", "\x64\xce", "\x00\x00", "\x64\xd0",
  "\x64\xd1", "\x00\x00", "\x00\x00", "\x00\x00", "\x64\xd5", "\x00\x00",
  "\x64\xd7", "\x64\xe4", "\x64\xe5", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x64\xe9", "\x64\xea", "\x00\x00", "\x00\x00", "\x64\xed", "\x00\x00",
  "\x00\x00", "\x64\xf0", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x64\xf5", "\x00\x00", "\x64\xf7", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x64\xfb", "\x00\x00", "\x00\x00", "\x00\x00", "\x64\xff", "\x00\x00",
  "\x65\x01", "\x00\x00", "\x00\x00", "\x65\x04", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x65\x08", "\x65\x09", "\x65\x0a", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x65\x0f", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x65\x13", "\x65\x14", "\x00\x00", "\x65\x16", "\x00\x00", "\x00\x00",
  "\x65\x19", "\x00\x00", "\x65\x1b", "\x00\x00", "\x00\x00", "\x65\x1e",
  "\x65\x1f", "\x00\x00", "\x00\x00", "\x65\x22", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x65\x26", "\x00\x00", "\x00\x00", "\x65\x29", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x65\x2e", "\x00\x00", "\x00\x00",
  "\x65\x31", "\x65\x3a", "\x00\x00", "\x65\x3c", "\x65\x3d", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x65\x43", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x65\x47", "\x00\x00", "\x65\x49", "\x65\x50",
  "\x00\x00", "\x65\x52", "\x00\x00", "\x65\x54", "\x65\x5f", "\x65\x60",
  "\x65\x67", "\x00\x00", "\x00\x00", "\x00\x00", "\x65\x6b", "\x65\x7a",
  "\x00\x00", "\x00\x00", "\x65\x7d", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x65\x81", "\x00\x00", "\x00\x00", "\x00\x00", "\x65\x85", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x65\x8a", "\x65\x92", "\x00\x00",
  "\x00\x00", "\x65\x95", "\x00\x00", "\x00\x00", "\x65\x98", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x65\x9d", "\x00\x00", "\x00\x00",
  "\x65\xa0", "\x00\x00", "\x00\x00", "\x65\xa3", "\x00\x00", "\x00\x00",
  "\x65\xa6", "\x65\xae", "\x00\x00", "\x00\x00", "\x00\x00", "\x65\xb2",
  "\x65\xb3", "\x65\xb4", "\x65\xbf", "\x00\x00", "\x00\x00", "\x65\xc2",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x65\xc8",
  "\x65\xc9", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x65\xce",
  "\x00\x00", "\x65\xd0", "\x00\x00", "\x00\x00", "\x00\x00", "\x65\xd4",
  "\x00\x00", "\x65\xd6", "\x00\x00", "\x65\xd8", "\x65\xdf", "\x65\xf0",
  "\x00\x00", "\x65\xf2", "\x00\x00", "\x65\xf4", "\x65\xf5", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x65\xf9", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x65\xfe", "\x65\xff", "\x66\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x66\x04", "\x00\x00", "\x00\x00", "\x00\x00", "\x66\x08",
  "\x66\x09", "\x00\x00", "\x00\x00", "\x00\x00", "\x66\x0d", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x66\x11", "\x66\x12", "\x00\x00", "\x00\x00",
  "\x66\x15", "\x66\x16", "\x66\x1d", "\x66\x1e", "\x00\x00", "\x00\x00",
  "\x66\x21", "\x66\x22", "\x66\x23", "\x66\x24", "\x00\x00", "\x66\x26",
  "\x00\x00", "\x00\x00", "\x66\x29", "\x66\x2a", "\x66\x2b", "\x66\x2c",
  "\x00\x00", "\x66\x2e", "\x00\x00", "\x66\x30", "\x66\x31", "\x00\x00",
  "\x66\x33", "\x00\x00", "\x00\x00", "\x00\x00", "\x66\x37", "\x00\x00",
  "\x66\x39", "\x66\x40", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x66\x45", "\x66\x46", "\x00\x00", "\x00\x00", "\x00\x00", "\x66\x4a",
  "\x00\x00", "\x66\x4c", "\x00\x00", "\x66\x4e", "\x00\x00", "\x00\x00",
  "\x66\x51", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x66\x57", "\x66\x58", "\x66\x59", "\x00\x00", "\x66\x5b", "\x66\x5c",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x66\x60", "\x66\x61", "\x66\x6a",
  "\x66\x6b", "\x66\x6c", "\x66\x73", "\x00\x00", "\x66\x75", "\x00\x00",
  "\x66\x77", "\x66\x78", "\x66\x79", "\x00\x00", "\x66\x7b", "\x66\x7c",
  "\x00\x00", "\x66\x7e", "\x66\x7f", "\x66\x80", "\x66\x8b", "\x66\x8c",
  "\x66\x8d", "\x00\x00", "\x00\x00", "\x66\x90", "\x00\x00", "\x66\x92",
  "\x66\x99", "\x66\x9a", "\x66\x9b", "\x66\x9c", "\x00\x00", "\x00\x00",
  "\x66\x9f", "\x66\xa0", "\x00\x00", "\x00\x00", "\x00\x00", "\x66\xa4",
  "\x66\xad", "\x00\x00", "\x00\x00", "\x00\x00", "\x66\xb1", "\x66\xb2",
  "\x00\x00", "\x00\x00", "\x66\xb5", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x66\xbb", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x66\xbf", "\x66\xc0", "\x00\x00", "\x66\xc2", "\x66\xc3", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x66\xc8", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x66\xcc", "\x00\x00", "\x66\xce", "\x66\xcf", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x66\xd4", "\x66\xdb", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x66\xdf", "\x66\xe8", "\x00\x00", "\x00\x00",
  "\x66\xeb", "\x66\xec", "\x00\x00", "\x66\xee", "\x66\xfa", "\x66\xfb",
  "\x67\x05", "\x00\x00", "\x67\x07", "\x67\x0e", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x67\x13", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x67\x19", "\x00\x00", "\x00\x00", "\x67\x1c",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x67\x20", "\x00\x00", "\x67\x22",
  "\x67\x33", "\x67\x3e", "\x67\x45", "\x00\x00", "\x67\x47", "\x67\x48",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x67\x4c", "\x67\x54", "\x67\x55",
  "\x67\x5d", "\x67\x66", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x67\x6c", "\x00\x00", "\x67\x6e", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x67\x74", "\x00\x00", "\x67\x76",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x67\x7b", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x67\x81", "\x00\x00",
  "\x00\x00", "\x67\x84", "\x67\x8e", "\x67\x8f", "\x00\x00", "\x67\x91",
  "\x00\x00", "\x67\x93", "\x00\x00", "\x00\x00", "\x67\x96", "\x00\x00",
  "\x67\x98", "\x67\x99", "\x00\x00", "\x67\x9b", "\x67\xb0", "\x67\xb1",
  "\x67\xb2", "\x00\x00", "\x00\x00", "\x67\xb5", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x67\xbb", "\x67\xbc", "\x67\xbd",
  "\x00\x00", "\x00\x00", "\x67\xc0", "\x00\x00", "\x67\xc2", "\x67\xc3",
  "\x00\x00", "\x67\xc5", "\x00\x00", "\x00\x00", "\x67\xc8", "\x67\xc9",
  "\x67\xd2", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x67\xd7",
  "\x00\x00", "\x67\xd9", "\x00\x00", "\x00\x00", "\x67\xdc", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x67\xe1", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x67\xe6", "\x67\xf0", "\x00\x00", "\x67\xf2",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x67\xf6", "\x67\xf7", "\x00\x00",
  "\x67\xf9", "\x68\x14", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x68\x19", "\x00\x00", "\x00\x00", "\x00\x00", "\x68\x1d", "\x00\x00",
  "\x68\x1f", "\x68\x27", "\x68\x28", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x68\x2c", "\x68\x2d", "\x00\x00", "\x68\x2f", "\x68\x30", "\x68\x31",
  "\x00\x00", "\x68\x33", "\x68\x3b", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x68\x3f", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x68\x44",
  "\x68\x45", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x68\x4a",
  "\x00\x00", "\x68\x4c", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x68\x52", "\x00\x00", "\x00\x00", "\x68\x55", "\x00\x00",
  "\x68\x57", "\x68\x58", "\x00\x00", "\x00\x00", "\x68\x5b", "\x68\x6b",
  "\x00\x00", "\x00\x00", "\x68\x6e", "\x68\x6f", "\x68\x70", "\x68\x71",
  "\x68\x72", "\x00\x00", "\x00\x00", "\x68\x75", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x68\x79", "\x68\x7a", "\x68\x7b", "\x68\x7c", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x68\x82", "\x00\x00",
  "\x68\x84", "\x00\x00", "\x68\x86", "\x00\x00", "\x68\x88", "\x68\x96",
  "\x00\x00", "\x68\x98", "\x00\x00", "\x68\x9a", "\x00\x00", "\x68\x9c",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x68\xa1", "\x00\x00",
  "\x68\xa3", "\x00\x00", "\x68\xa5", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x68\xa9", "\x68\xaa", "\x00\x00", "\x00\x00", "\x00\x00", "\x68\xae",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x68\xb2", "\x68\xbb", "\x68\xc5",
  "\x00\x00", "\x00\x00", "\x68\xc8", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x68\xcc", "\x00\x00", "\x00\x00", "\x68\xcf", "\x68\xd0", "\x68\xd1",
  "\x00\x00", "\x68\xd3", "\x00\x00", "\x00\x00", "\x68\xd6", "\x00\x00",
  "\x00\x00", "\x68\xd9", "\x00\x00", "\x00\x00", "\x68\xdc", "\x68\xdd",
  "\x68\xe5", "\x00\x00", "\x00\x00", "\x68\xe8", "\x00\x00", "\x68\xea",
  "\x68\xeb", "\x68\xec", "\x68\xed", "\x00\x00", "\x00\x00", "\x68\xf0",
  "\x68\xf1", "\x00\x00", "\x00\x00", "\x00\x00", "\x68\xf5", "\x68\xf6",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x68\xfb", "\x68\xfc",
  "\x68\xfd", "\x69\x06", "\x00\x00", "\x00\x00", "\x69\x09", "\x69\x0a",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x69\x10",
  "\x69\x11", "\x00\x00", "\x69\x13", "\x00\x00", "\x00\x00", "\x69\x16",
  "\x69\x17", "\x69\x31", "\x00\x00", "\x69\x33", "\x00\x00", "\x69\x35",
  "\x00\x00", "\x00\x00", "\x69\x38", "\x00\x00", "\x00\x00", "\x69\x3b",
  "\x69\x42", "\x00\x00", "\x00\x00", "\x69\x45", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x69\x49", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x69\x4e", "\x69\x57", "\x00\x00", "\x00\x00", "\x00\x00", "\x69\x5b",
  "\x69\x63", "\x69\x64", "\x69\x65", "\x69\x66", "\x00\x00", "\x69\x68",
  "\x69\x69", "\x00\x00", "\x00\x00", "\x69\x6c", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x69\x70", "\x69\x71", "\x69\x72", "\x69\x7a", "\x69\x7b",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x69\x7f", "\x69\x80", "\x69\x8d",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x69\x92", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x69\x96", "\x00\x00", "\x69\x98", "\x69\xa1",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x69\xa5", "\x69\xa6", "\x00\x00",
  "\x69\xa8", "\x00\x00", "\x00\x00", "\x69\xab", "\x00\x00", "\x69\xad",
  "\x00\x00", "\x69\xaf", "\x69\xb7", "\x69\xb8", "\x00\x00", "\x69\xba",
  "\x00\x00", "\x69\xbc", "\x69\xc5", "\x00\x00", "\x00\x00", "\x69\xc8",
  "\x69\xd1", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x69\xd6",
  "\x69\xd7", "\x69\xe2", "\x00\x00", "\x00\x00", "\x69\xe5", "\x69\xee",
  "\x69\xef", "\x00\x00", "\x69\xf1", "\x00\x00", "\x69\xf3", "\x00\x00",
  "\x69\xf5", "\x69\xfe", "\x00\x00", "\x6a\x00", "\x6a\x01", "\x00\x00",
  "\x6a\x03", "\x6a\x0f", "\x00\x00", "\x6a\x11", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x6a\x15", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x6a\x1a", "\x00\x00", "\x00\x00", "\x6a\x1d", "\x00\x00", "\x00\x00",
  "\x6a\x20", "\x00\x00", "\x00\x00", "\x00\x00", "\x6a\x24", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x6a\x28", "\x6a\x30", "\x00\x00", "\x6a\x32",
  "\x00\x00", "\x6a\x34", "\x00\x00", "\x00\x00", "\x6a\x37", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x6a\x3b", "\x00\x00", "\x00\x00", "\x6a\x3e",
  "\x6a\x3f", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x6a\x45", "\x6a\x46", "\x00\x00", "\x00\x00", "\x6a\x49", "\x6a\x4a",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x6a\x4e", "\x00\x00", "\x6a\x50",
  "\x6a\x51", "\x6a\x52", "\x00\x00", "\x00\x00", "\x6a\x55", "\x6a\x56",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x6a\x5b", "\x6a\x64",
  "\x00\x00", "\x00\x00", "\x6a\x67", "\x00\x00", "\x00\x00", "\x6a\x6a",
  "\x6a\x71", "\x00\x00", "\x6a\x73", "\x6a\x7e", "\x00\x00", "\x00\x00",
  "\x6a\x81", "\x00\x00", "\x6a\x83", "\x00\x00", "\x00\x00", "\x6a\x86",
  "\x6a\x87", "\x00\x00", "\x6a\x89", "\x00\x00", "\x6a\x8b", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x6a\x91", "\x6a\x9b",
  "\x00\x00", "\x6a\x9d", "\x6a\x9e", "\x6a\x9f", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x6a\xa5", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x6a\xab", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x6a\xaf", "\x6a\xb0", "\x6a\xb1", "\x00\x00", "\x00\x00",
  "\x6a\xb4", "\x6a\xbd", "\x6a\xbe", "\x6a\xbf", "\x6a\xc6", "\x00\x00",
  "\x6a\xc8", "\x6a\xc9", "\x00\x00", "\x00\x00", "\x6a\xcc", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x6a\xd0", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x6a\xd4", "\x6a\xd5", "\x6a\xd6", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x6a\xdc", "\x6a\xdd", "\x6a\xe4", "\x00\x00",
  "\x00\x00", "\x6a\xe7", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x6a\xec", "\x00\x00", "\x00\x00", "\x00\x00", "\x6a\xf0", "\x6a\xf1",
  "\x6a\xf2", "\x6a\xfc", "\x6a\xfd", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x6b\x02", "\x6b\x03", "\x00\x00", "\x00\x00", "\x6b\x06",
  "\x6b\x07", "\x00\x00", "\x6b\x09", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x6b\x0f", "\x6b\x10", "\x6b\x11", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x6b\x17", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x6b\x1b", "\x00\x00", "\x00\x00", "\x6b\x1e",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x6b\x24",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x6b\x28", "\x00\x00", "\x00\x00",
  "\x6b\x2b", "\x6b\x2c", "\x00\x00", "\x00\x00", "\x6b\x2f", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x6b\x35", "\x6b\x36",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x6b\x3b", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x6b\x3f", "\x6b\x46", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x6b\x4a", "\x00\x00", "\x00\x00", "\x6b\x4d", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x6b\x52", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x6b\x56", "\x00\x00", "\x6b\x58", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x6b\x5d", "\x00\x00", "\x00\x00", "\x6b\x60",
  "\x6b\x67", "\x00\x00", "\x00\x00", "\x00\x00", "\x6b\x6b", "\x00\x00",
  "\x00\x00", "\x6b\x6e", "\x00\x00", "\x6b\x70", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x6b\x75", "\x6b\x7d", "\x6b\x7e", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x6b\x82", "\x00\x00", "\x00\x00", "\x6b\x85",
  "\x6b\x97", "\x00\x00", "\x00\x00", "\x00\x00", "\x6b\x9b", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x6b\x9f", "\x6b\xa0", "\x00\x00", "\x6b\xa2",
  "\x6b\xa3", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x6b\xa8",
  "\x6b\xa9", "\x00\x00", "\x00\x00", "\x6b\xac", "\x6b\xad", "\x6b\xae",
  "\x00\x00", "\x6b\xb0", "\x6b\xb8", "\x6b\xb9", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x6b\xbd", "\x6b\xbe", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x6b\xc3", "\x6b\xc4", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x6b\xc9", "\x00\x00", "\x00\x00", "\x6b\xcc", "\x6b\xd6",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x6b\xda", "\x6b\xe1", "\x00\x00",
  "\x6b\xe3", "\x00\x00", "\x00\x00", "\x6b\xe6", "\x6b\xe7", "\x6b\xee",
  "\x00\x00", "\x00\x00", "\x6b\xf1", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x6b\xf7", "\x00\x00", "\x6b\xf9", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x6b\xff", "\x00\x00",
  "\x00\x00", "\x6c\x02", "\x00\x00", "\x6c\x04", "\x6c\x05", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x6c\x09", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x6c\x0d", "\x6c\x0e", "\x00\x00", "\x6c\x10", "\x00\x00", "\x6c\x12",
  "\x6c\x19", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x6c\x1f", "\x6c\x26", "\x6c\x27", "\x6c\x28", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x6c\x2c", "\x00\x00", "\x6c\x2e", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x6c\x33", "\x00\x00", "\x6c\x35", "\x6c\x36",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x6c\x3a", "\x6c\x3b", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x6c\x3f", "\x6c\x4a", "\x6c\x4b", "\x00\x00",
  "\x6c\x4d", "\x00\x00", "\x6c\x4f", "\x00\x00", "\x00\x00", "\x6c\x52",
  "\x00\x00", "\x6c\x54", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x6c\x59", "\x00\x00", "\x6c\x5b", "\x6c\x5c", "\x6c\x67", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x6c\x6b", "\x00\x00", "\x6c\x6d", "\x00\x00",
  "\x6c\x6f", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x6c\x74",
  "\x00\x00", "\x6c\x76", "\x00\x00", "\x6c\x78", "\x6c\x79", "\x00\x00",
  "\x6c\x7b", "\x6c\x85", "\x6c\x86", "\x6c\x87", "\x00\x00", "\x6c\x89",
  "\x6c\x94", "\x6c\x95", "\x00\x00", "\x6c\x97", "\x6c\x98", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x6c\x9c", "\x00\x00", "\x00\x00", "\x6c\x9f",
  "\x6c\xb0", "\x00\x00", "\x6c\xb2", "\x00\x00", "\x6c\xb4", "\x6c\xc2",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x6c\xc6", "\x6c\xcd", "\x00\x00",
  "\x6c\xcf", "\x6c\xd0", "\x6c\xd1", "\x6c\xd2", "\x00\x00", "\x6c\xd4",
  "\x00\x00", "\x6c\xd6", "\x00\x00", "\x00\x00", "\x00\x00", "\x6c\xda",
  "\x00\x00", "\x6c\xdc", "\x00\x00", "\x00\x00", "\x00\x00", "\x6c\xe0",
  "\x6c\xe7", "\x00\x00", "\x6c\xe9", "\x00\x00", "\x6c\xeb", "\x6c\xec",
  "\x00\x00", "\x6c\xee", "\x00\x00", "\x00\x00", "\x00\x00", "\x6c\xf2",
  "\x00\x00", "\x6c\xf4", "\x6d\x04", "\x00\x00", "\x00\x00", "\x6d\x07",
  "\x00\x00", "\x00\x00", "\x6d\x0a", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x6d\x0e", "\x6d\x0f", "\x00\x00", "\x6d\x11", "\x00\x00", "\x6d\x13",
  "\x6d\x1a", "\x6d\x26", "\x6d\x27", "\x6d\x28", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x6d\x2e", "\x6d\x2f", "\x00\x00",
  "\x6d\x31", "\x6d\x39", "\x00\x00", "\x00\x00", "\x6d\x3c", "\x00\x00",
  "\x00\x00", "\x6d\x3f", "\x6d\x57", "\x6d\x5e", "\x6d\x5f", "\x00\x00",
  "\x6d\x61", "\x00\x00", "\x00\x00", "\x00\x00", "\x6d\x65", "\x00\x00",
  "\x6d\x67", "\x6d\x6f", "\x6d\x70", "\x6d\x7c", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x6d\x82", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x6d\x87", "\x6d\x91", "\x6d\x92", "\x00\x00",
  "\x6d\x94", "\x00\x00", "\x6d\x96", "\x6d\x97", "\x6d\x98", "\x6d\xaa",
  "\x00\x00", "\x6d\xac", "\x6d\xb4", "\x00\x00", "\x00\x00", "\x6d\xb7",
  "\x00\x00", "\x6d\xb9", "\x00\x00", "\x00\x00", "\x00\x00", "\x6d\xbd",
  "\x00\x00", "\x6d\xbf", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x6d\xc4", "\x00\x00", "\x00\x00", "\x00\x00", "\x6d\xc8", "\x00\x00",
  "\x6d\xca", "\x00\x00", "\x00\x00", "\x00\x00", "\x6d\xce", "\x6d\xcf",
  "\x6d\xd6", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x6d\xdb",
  "\x00\x00", "\x6d\xdd", "\x00\x00", "\x6d\xdf", "\x6d\xe0", "\x00\x00",
  "\x6d\xe2", "\x00\x00", "\x00\x00", "\x6d\xe5", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x6d\xe9", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x6d\xef", "\x6d\xf0", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x6d\xf4", "\x00\x00", "\x6d\xf6", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x6d\xfc", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x6e\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x6e\x04", "\x6e\x1e",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x6e\x22", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x6e\x27", "\x6e\x32", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x6e\x36", "\x00\x00", "\x00\x00", "\x6e\x39", "\x00\x00",
  "\x6e\x3b", "\x6e\x3c", "\x6e\x44", "\x6e\x45", "\x00\x00", "\x00\x00",
  "\x6e\x48", "\x6e\x49", "\x00\x00", "\x6e\x4b", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x6e\x4f", "\x00\x00", "\x6e\x51", "\x6e\x52", "\x6e\x53",
  "\x6e\x54", "\x00\x00", "\x00\x00", "\x6e\x57", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x6e\x5c", "\x6e\x5d", "\x6e\x5e", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x6e\x62", "\x6e\x63", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x6e\x68", "\x6e\x73", "\x6e\x7b", "\x00\x00",
  "\x6e\x7d", "\x6e\x8d", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x6e\x93", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x6e\x99", "\x6e\xa0", "\x6e\xa7", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x6e\xad", "\x6e\xae", "\x00\x00",
  "\x00\x00", "\x6e\xb1", "\x00\x00", "\x6e\xb3", "\x6e\xbb", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x6e\xbf", "\x6e\xc0", "\x6e\xc1", "\x00\x00",
  "\x6e\xc3", "\x00\x00", "\x00\x00", "\x00\x00", "\x6e\xc7", "\x6e\xc8",
  "\x00\x00", "\x6e\xca", "\x00\x00", "\x00\x00", "\x6e\xcd", "\x6e\xce",
  "\x6e\xcf", "\x6e\xeb", "\x00\x00", "\x6e\xed", "\x6e\xee", "\x6e\xf9",
  "\x00\x00", "\x6e\xfb", "\x00\x00", "\x6e\xfd", "\x6f\x04", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x6f\x08", "\x00\x00", "\x6f\x0a", "\x00\x00",
  "\x6f\x0c", "\x6f\x0d", "\x6f\x16", "\x00\x00", "\x6f\x18", "\x00\x00",
  "\x6f\x1a", "\x6f\x1b", "\x6f\x26", "\x00\x00", "\x00\x00", "\x6f\x29",
  "\x6f\x2a", "\x00\x00", "\x00\x00", "\x6f\x2d", "\x00\x00", "\x6f\x2f",
  "\x6f\x30", "\x00\x00", "\x00\x00", "\x6f\x33", "\x00\x00", "\x00\x00",
  "\x6f\x36", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x6f\x3b",
  "\x6f\x3c", "\x6f\x4f", "\x00\x00", "\x6f\x51", "\x6f\x52", "\x6f\x53",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x6f\x57", "\x00\x00", "\x6f\x59",
  "\x6f\x5a", "\x00\x00", "\x00\x00", "\x6f\x5d", "\x6f\x5e", "\x00\x00",
  "\x00\x00", "\x6f\x61", "\x6f\x62", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x6f\x68", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x6f\x6c", "\x6f\x7d", "\x6f\x7e", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x6f\x83", "\x00\x00", "\x00\x00", "\x00\x00", "\x6f\x87",
  "\x6f\x88", "\x00\x00", "\x00\x00", "\x6f\x8b", "\x6f\x8c", "\x6f\x8d",
  "\x00\x00", "\x00\x00", "\x6f\x90", "\x00\x00", "\x6f\x92", "\x6f\x93",
  "\x6f\x94", "\x00\x00", "\x6f\x96", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x6f\x9a", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x6f\x9f",
  "\x6f\xa0", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x6f\xa5",
  "\x6f\xa6", "\x6f\xa7", "\x6f\xa8", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x6f\xae", "\x6f\xaf", "\x6f\xb0", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x6f\xb5", "\x6f\xb6", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x6f\xbc", "\x6f\xc5",
  "\x00\x00", "\x6f\xc7", "\x6f\xc8", "\x00\x00", "\x6f\xca", "\x6f\xda",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x6f\xde", "\x6f\xe8", "\x6f\xe9",
  "\x6f\xf0", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x6f\xf5",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x6f\xf9", "\x00\x00", "\x00\x00",
  "\x6f\xfc", "\x6f\xfd", "\x00\x00", "\x00\x00", "\x70\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x70\x05", "\x70\x06", "\x70\x07",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x70\x0d",
  "\x70\x17", "\x70\x20", "\x00\x00", "\x00\x00", "\x70\x23", "\x70\x2f",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x70\x34", "\x00\x00",
  "\x00\x00", "\x70\x37", "\x00\x00", "\x70\x39", "\x00\x00", "\x00\x00",
  "\x70\x3c", "\x70\x43", "\x70\x44", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x70\x48", "\x70\x49", "\x70\x4a", "\x70\x4b", "\x00\x00", "\x00\x00",
  "\x70\x4e", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x70\x54", "\x70\x55", "\x70\x5d", "\x70\x5e", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x70\x64", "\x70\x65", "\x70\x6c",
  "\x00\x00", "\x70\x6e", "\x70\x75", "\x70\x76", "\x70\x7e", "\x00\x00",
  "\x00\x00", "\x70\x81", "\x00\x00", "\x00\x00", "\x00\x00", "\x70\x85",
  "\x70\x86", "\x70\x94", "\x70\x95", "\x70\x96", "\x70\x97", "\x70\x98",
  "\x00\x00", "\x00\x00", "\x70\x9b", "\x70\xa4", "\x70\xab", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x70\xb0", "\x70\xb1", "\x00\x00",
  "\x00\x00", "\x70\xb4", "\x00\x00", "\x00\x00", "\x70\xb7", "\x70\xca",
  "\x70\xd1", "\x00\x00", "\x70\xd3", "\x70\xd4", "\x70\xd5", "\x70\xd6",
  "\x00\x00", "\x70\xd8", "\x00\x00", "\x00\x00", "\x00\x00", "\x70\xdc",
  "\x70\xe4", "\x70\xfa", "\x71\x03", "\x71\x04", "\x71\x05", "\x71\x06",
  "\x71\x07", "\x00\x00", "\x00\x00", "\x00\x00", "\x71\x0b", "\x71\x0c",
  "\x00\x00", "\x00\x00", "\x71\x0f", "\x71\x1e", "\x00\x00", "\x71\x20",
  "\x71\x2b", "\x00\x00", "\x71\x2d", "\x00\x00", "\x71\x2f", "\x71\x30",
  "\x71\x31", "\x71\x38", "\x71\x41", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x71\x45", "\x71\x46", "\x71\x47", "\x00\x00", "\x00\x00", "\x71\x4a",
  "\x71\x4b", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x71\x50",
  "\x00\x00", "\x71\x52", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x71\x57", "\x00\x00", "\x00\x00", "\x71\x5a", "\x00\x00", "\x71\x5c",
  "\x00\x00", "\x71\x5e", "\x00\x00", "\x71\x60", "\x71\x68", "\x71\x79",
  "\x71\x80", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x71\x85",
  "\x00\x00", "\x71\x87", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x71\x8c", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x71\x92", "\x71\x9a", "\x71\x9b", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x71\xa0", "\x00\x00", "\x71\xa2", "\x71\xaf", "\x71\xb0",
  "\x00\x00", "\x71\xb2", "\x71\xb3", "\x71\xba", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x71\xbf", "\x71\xc0", "\x71\xc1", "\x00\x00",
  "\x00\x00", "\x71\xc4", "\x71\xcb", "\x71\xcc", "\x71\xd3", "\x00\x00",
  "\x00\x00", "\x71\xd6", "\x00\x00", "\x00\x00", "\x71\xd9", "\x71\xda",
  "\x00\x00", "\x71\xdc", "\x71\xf8", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x71\xfe", "\x00\x00", "\x72\x00", "\x72\x07",
  "\x72\x08", "\x72\x09", "\x72\x13", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x72\x17", "\x00\x00", "\x00\x00", "\x72\x1a", "\x00\x00", "\x00\x00",
  "\x72\x1d", "\x00\x00", "\x72\x1f", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x72\x24", "\x72\x2b", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x72\x2f", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x72\x34",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x72\x38", "\x72\x39", "\x72\x41",
  "\x72\x42", "\x72\x43", "\x00\x00", "\x72\x45", "\x72\x4e", "\x72\x4f",
  "\x72\x50", "\x00\x00", "\x00\x00", "\x72\x53", "\x00\x00", "\x72\x55",
  "\x72\x56", "\x00\x00", "\x00\x00", "\x00\x00", "\x72\x5a", "\x00\x00",
  "\x72\x5c", "\x00\x00", "\x72\x5e", "\x00\x00", "\x72\x60", "\x00\x00",
  "\x00\x00", "\x72\x63", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x72\x68", "\x00\x00", "\x00\x00", "\x72\x6b", "\x00\x00", "\x00\x00",
  "\x72\x6e", "\x72\x6f", "\x00\x00", "\x72\x71", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x72\x77", "\x72\x78", "\x00\x00",
  "\x00\x00", "\x72\x7b", "\x72\x7c", "\x00\x00", "\x00\x00", "\x72\x7f",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x72\x84", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x72\x89", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x72\x8d", "\x72\x8e", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x72\x93", "\x72\x9b", "\x72\xa8", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x72\xad", "\x72\xae", "\x00\x00", "\x00\x00",
  "\x72\xb1", "\x00\x00", "\x00\x00", "\x72\xb4", "\x72\xbe", "\x00\x00",
  "\x00\x00", "\x72\xc1", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x72\xc7", "\x00\x00", "\x72\xc9", "\x00\x00", "\x00\x00",
  "\x72\xcc", "\x72\xd5", "\x72\xd6", "\x00\x00", "\x72\xd8", "\x72\xdf",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x72\xe5",
  "\x72\xf3", "\x72\xf4", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x72\xfa", "\x72\xfb", "\x00\x00", "\x00\x00", "\x72\xfe",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x73\x02", "\x00\x00", "\x73\x04",
  "\x73\x05", "\x00\x00", "\x73\x07", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x73\x0b", "\x00\x00", "\x73\x0d", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x73\x12", "\x73\x13", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x73\x18", "\x73\x19", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x73\x1e", "\x00\x00", "\x00\x00", "\x00\x00", "\x73\x22",
  "\x00\x00", "\x73\x24", "\x00\x00", "\x00\x00", "\x73\x27", "\x73\x28",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x73\x2c", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x73\x31", "\x73\x32", "\x00\x00", "\x00\x00",
  "\x73\x35", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x73\x3a",
  "\x73\x3b", "\x00\x00", "\x73\x3d", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x73\x43", "\x73\x4d", "\x00\x00", "\x00\x00",
  "\x73\x50", "\x00\x00", "\x73\x52", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x73\x56", "\x00\x00", "\x73\x58", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x73\x5d", "\x73\x5e", "\x73\x5f", "\x73\x60", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x73\x66", "\x73\x67",
  "\x00\x00", "\x73\x69", "\x00\x00", "\x73\x6b", "\x73\x6c", "\x00\x00",
  "\x73\x6e", "\x73\x6f", "\x00\x00", "\x73\x71", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x73\x77", "\x00\x00", "\x73\x79",
  "\x00\x00", "\x00\x00", "\x73\x7c", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x73\x80", "\x73\x81", "\x00\x00", "\x73\x83", "\x00\x00", "\x73\x85",
  "\x73\x86", "\x73\x8e", "\x00\x00", "\x73\x90", "\x00\x00", "\x00\x00",
  "\x73\x93", "\x00\x00", "\x73\x95", "\x00\x00", "\x73\x97", "\x73\x98",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x73\x9c", "\x00\x00", "\x73\x9e",
  "\x73\x9f", "\x73\xa0", "\x00\x00", "\x73\xa2", "\x00\x00", "\x00\x00",
  "\x73\xa5", "\x73\xa6", "\x00\x00", "\x00\x00", "\x00\x00", "\x73\xaa",
  "\x73\xab", "\x00\x00", "\x73\xad", "\x73\xb5", "\x00\x00", "\x73\xb7",
  "\x00\x00", "\x73\xb9", "\x00\x00", "\x00\x00", "\x73\xbc", "\x73\xbd",
  "\x00\x00", "\x73\xbf", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x73\xc5", "\x73\xc6", "\x00\x00", "\x00\x00", "\x73\xc9",
  "\x00\x00", "\x73\xcb", "\x73\xcc", "\x00\x00", "\x00\x00", "\x73\xcf",
  "\x00\x00", "\x00\x00", "\x73\xd2", "\x73\xd3", "\x00\x00", "\x00\x00",
  "\x73\xd6", "\x00\x00", "\x00\x00", "\x73\xd9", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x73\xdd", "\x00\x00", "\x00\x00", "\x00\x00", "\x73\xe1",
  "\x00\x00", "\x73\xe3", "\x00\x00", "\x00\x00", "\x73\xe6", "\x73\xe7",
  "\x00\x00", "\x73\xe9", "\x73\xf4", "\x73\xf5", "\x00\x00", "\x73\xf7",
  "\x00\x00", "\x73\xf9", "\x73\xfa", "\x73\xfb", "\x00\x00", "\x73\xfd",
  "\x00\x00", "\x73\xff", "\x74\x00", "\x74\x01", "\x00\x00", "\x00\x00",
  "\x74\x04", "\x00\x00", "\x00\x00", "\x74\x07", "\x00\x00", "\x00\x00",
  "\x74\x0a", "\x74\x11", "\x74\x1a", "\x74\x1b", "\x74\x24", "\x00\x00",
  "\x74\x26", "\x00\x00", "\x74\x28", "\x74\x29", "\x74\x2a", "\x74\x2b",
  "\x74\x2c", "\x74\x2d", "\x74\x2e", "\x74\x2f", "\x74\x30", "\x74\x31",
  "\x74\x39", "\x74\x40", "\x00\x00", "\x00\x00", "\x74\x43", "\x74\x44",
  "\x00\x00", "\x74\x46", "\x74\x47", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x74\x4b", "\x00\x00", "\x74\x4d", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x74\x51", "\x74\x52", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x74\x57", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x74\x5d", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x74\x62",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x74\x66", "\x74\x67", "\x74\x68",
  "\x00\x00", "\x00\x00", "\x74\x6b", "\x00\x00", "\x74\x6d", "\x74\x6e",
  "\x00\x00", "\x00\x00", "\x74\x71", "\x74\x72", "\x74\x80", "\x74\x81",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x74\x85", "\x74\x86", "\x74\x87",
  "\x00\x00", "\x74\x89", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x74\x8f", "\x74\x90", "\x74\x91", "\x74\x92", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x74\x98", "\x74\x99",
  "\x74\x9a", "\x00\x00", "\x74\x9c", "\x00\x00", "\x00\x00", "\x74\x9f",
  "\x74\xa0", "\x74\xa1", "\x00\x00", "\x74\xa3", "\x00\x00", "\x00\x00",
  "\x74\xa6", "\x00\x00", "\x74\xa8", "\x74\xa9", "\x74\xaa", "\x74\xab",
  "\x00\x00", "\x00\x00", "\x74\xae", "\x74\xaf", "\x00\x00", "\x74\xb1",
  "\x74\xb2", "\x00\x00", "\x00\x00", "\x74\xb5", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x74\xb9", "\x00\x00", "\x74\xbb", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x74\xbf", "\x74\xc8", "\x74\xc9", "\x00\x00", "\x00\x00",
  "\x74\xcc", "\x00\x00", "\x00\x00", "\x00\x00", "\x74\xd0", "\x00\x00",
  "\x00\x00", "\x74\xd3", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x74\xd8", "\x00\x00", "\x74\xda", "\x74\xdb", "\x00\x00", "\x00\x00",
  "\x74\xde", "\x74\xdf", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x74\xe4", "\x00\x00", "\x00\x00", "\x00\x00", "\x74\xe8", "\x00\x00",
  "\x74\xea", "\x74\xeb", "\x00\x00", "\x00\x00", "\x00\x00", "\x74\xef",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x74\xf4", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x74\xfa", "\x74\xfb",
  "\x74\xfc", "\x00\x00", "\x00\x00", "\x74\xff", "\x75\x06", "\x75\x12",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x75\x16", "\x75\x17", "\x75\x20",
  "\x75\x21", "\x00\x00", "\x00\x00", "\x75\x24", "\x00\x00", "\x00\x00",
  "\x75\x27", "\x00\x00", "\x75\x29", "\x75\x2a", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x75\x2f", "\x75\x36", "\x00\x00", "\x00\x00",
  "\x75\x39", "\x00\x00", "\x00\x00", "\x00\x00", "\x75\x3d", "\x75\x3e",
  "\x75\x3f", "\x75\x40", "\x00\x00", "\x00\x00", "\x75\x43", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x75\x47", "\x75\x48", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x75\x4e", "\x00\x00", "\x75\x50",
  "\x00\x00", "\x75\x52", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x75\x57", "\x75\x5e", "\x75\x5f", "\x00\x00", "\x75\x61", "\x75\x6f",
  "\x00\x00", "\x75\x71", "\x75\x79", "\x75\x7a", "\x75\x7b", "\x75\x7c",
  "\x75\x7d", "\x75\x7e", "\x00\x00", "\x00\x00", "\x75\x81", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x75\x85", "\x75\x90", "\x00\x00", "\x75\x92",
  "\x75\x93", "\x00\x00", "\x75\x95", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x75\x99", "\x00\x00", "\x00\x00", "\x75\x9c", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x75\xa2", "\x00\x00", "\x75\xa4",
  "\x75\xb4", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x75\xba", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x75\xbf",
  "\x75\xc0", "\x75\xc1", "\x00\x00", "\x00\x00", "\x75\xc4", "\x00\x00",
  "\x75\xc6", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x75\xcc", "\x00\x00", "\x75\xce", "\x75\xcf", "\x75\xd7", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x75\xdc", "\x00\x00", "\x00\x00",
  "\x75\xdf", "\x75\xe0", "\x75\xe1", "\x00\x00", "\x00\x00", "\x75\xe4",
  "\x00\x00", "\x00\x00", "\x75\xe7", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x75\xec", "\x00\x00", "\x75\xee", "\x75\xef", "\x00\x00",
  "\x75\xf1", "\x75\xf9", "\x76\x00", "\x00\x00", "\x76\x02", "\x76\x03",
  "\x76\x04", "\x00\x00", "\x00\x00", "\x76\x07", "\x76\x08", "\x00\x00",
  "\x76\x0a", "\x00\x00", "\x76\x0c", "\x00\x00", "\x00\x00", "\x76\x0f",
  "\x00\x00", "\x00\x00", "\x76\x12", "\x76\x13", "\x00\x00", "\x76\x15",
  "\x76\x16", "\x00\x00", "\x00\x00", "\x76\x19", "\x00\x00", "\x76\x1b",
  "\x76\x1c", "\x76\x1d", "\x76\x1e", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x76\x23", "\x00\x00", "\x76\x25", "\x76\x26", "\x00\x00",
  "\x00\x00", "\x76\x29", "\x00\x00", "\x00\x00", "\x00\x00", "\x76\x2d",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x76\x32", "\x76\x33",
  "\x00\x00", "\x76\x35", "\x00\x00", "\x00\x00", "\x76\x38", "\x76\x39",
  "\x76\x3a", "\x00\x00", "\x76\x3c", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x76\x40", "\x76\x41", "\x00\x00", "\x76\x43", "\x76\x44", "\x76\x45",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x76\x49", "\x76\x4a", "\x76\x4b",
  "\x76\x55", "\x00\x00", "\x00\x00", "\x00\x00", "\x76\x59", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x76\x5f", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x76\x64", "\x76\x65", "\x76\x6d",
  "\x76\x6e", "\x76\x6f", "\x00\x00", "\x76\x71", "\x00\x00", "\x00\x00",
  "\x76\x74", "\x76\x81", "\x00\x00", "\x00\x00", "\x00\x00", "\x76\x85",
  "\x76\x8c", "\x76\x8d", "\x76\x95", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x76\x9b", "\x76\x9c", "\x76\x9d", "\x00\x00",
  "\x76\x9f", "\x76\xa0", "\x00\x00", "\x76\xa2", "\x76\xa3", "\x76\xa4",
  "\x76\xa5", "\x76\xa6", "\x76\xa7", "\x76\xa8", "\x00\x00", "\x76\xaa",
  "\x00\x00", "\x00\x00", "\x76\xad", "\x76\xbd", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x76\xc1", "\x00\x00", "\x00\x00", "\x00\x00", "\x76\xc5",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x76\xc9", "\x00\x00", "\x76\xcb",
  "\x76\xcc", "\x00\x00", "\x76\xce", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x76\xd4", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x76\xd9", "\x76\xe0", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x76\xe6", "\x00\x00", "\x76\xe8", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x76\xec", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x76\xf0", "\x76\xf1", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x76\xf6", "\x00\x00", "\x00\x00", "\x76\xf9", "\x00\x00", "\x00\x00",
  "\x76\xfc", "\x00\x00", "\x00\x00", "\x00\x00", "\x77\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x77\x06", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x77\x0a", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x77\x0e", "\x00\x00", "\x00\x00", "\x00\x00", "\x77\x12", "\x00\x00",
  "\x77\x14", "\x77\x15", "\x00\x00", "\x77\x17", "\x00\x00", "\x77\x19",
  "\x77\x1a", "\x00\x00", "\x77\x1c", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x77\x22", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x77\x28", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x77\x2d", "\x77\x2e", "\x77\x2f", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x77\x34", "\x77\x35", "\x77\x36", "\x00\x00",
  "\x00\x00", "\x77\x39", "\x00\x00", "\x00\x00", "\x00\x00", "\x77\x3d",
  "\x77\x3e", "\x00\x00", "\x00\x00", "\x00\x00", "\x77\x42", "\x00\x00",
  "\x00\x00", "\x77\x45", "\x77\x46", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x77\x4a", "\x00\x00", "\x00\x00", "\x77\x4d", "\x77\x4e", "\x77\x4f",
  "\x00\x00", "\x00\x00", "\x77\x52", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x77\x56", "\x77\x57", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x77\x5c", "\x00\x00", "\x77\x5e", "\x77\x5f", "\x77\x60", "\x00\x00",
  "\x77\x62", "\x00\x00", "\x77\x64", "\x00\x00", "\x00\x00", "\x77\x67",
  "\x00\x00", "\x00\x00", "\x77\x6a", "\x00\x00", "\x77\x6c", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x77\x70", "\x00\x00", "\x77\x72", "\x77\x73",
  "\x77\x74", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x77\x7a", "\x00\x00", "\x00\x00", "\x77\x7d", "\x00\x00", "\x00\x00",
  "\x77\x80", "\x00\x00", "\x00\x00", "\x00\x00", "\x77\x84", "\x77\x8c",
  "\x77\x8d", "\x77\x94", "\x77\x95", "\x77\x96", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x77\x9a", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x77\x9f", "\x00\x00", "\x00\x00", "\x77\xa2", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x77\xa7", "\x00\x00", "\x00\x00", "\x77\xaa",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x77\xae", "\x77\xaf", "\x00\x00",
  "\x77\xb1", "\x00\x00", "\x00\x00", "\x00\x00", "\x77\xb5", "\x77\xbe",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x77\xc3", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x77\xc9", "\x77\xd1",
  "\x77\xd2", "\x00\x00", "\x00\x00", "\x77\xd5", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x77\xd9", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x77\xde", "\x77\xdf", "\x77\xe0", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x77\xe4", "\x00\x00", "\x77\xe6", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x77\xea", "\x00\x00", "\x77\xec", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x77\xf0", "\x77\xf1", "\x00\x00", "\x00\x00", "\x77\xf4", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x77\xf8", "\x00\x00", "\x00\x00", "\x77\xfb",
  "\x78\x05", "\x78\x06", "\x00\x00", "\x00\x00", "\x78\x09", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x78\x0d", "\x78\x0e", "\x00\x00", "\x00\x00",
  "\x78\x11", "\x78\x1d", "\x00\x00", "\x00\x00", "\x00\x00", "\x78\x21",
  "\x78\x22", "\x78\x23", "\x78\x2d", "\x78\x2e", "\x00\x00", "\x78\x30",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x78\x35", "\x00\x00",
  "\x78\x37", "\x78\x43", "\x78\x44", "\x00\x00", "\x00\x00", "\x78\x47",
  "\x78\x48", "\x00\x00", "\x00\x00", "\x00\x00", "\x78\x4c", "\x00\x00",
  "\x78\x4e", "\x00\x00", "\x00\x00", "\x00\x00", "\x78\x52", "\x78\x5c",
  "\x00\x00", "\x78\x5e", "\x00\x00", "\x78\x60", "\x78\x61", "\x00\x00",
  "\x78\x63", "\x78\x64", "\x00\x00", "\x00\x00", "\x00\x00", "\x78\x68",
  "\x00\x00", "\x78\x6a", "\x00\x00", "\x00\x00", "\x00\x00", "\x78\x6e",
  "\x78\x7a", "\x00\x00", "\x00\x00", "\x00\x00", "\x78\x7e", "\x78\x8a",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x78\x8f", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x78\x94", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x78\x98", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x78\x9d", "\x78\x9e", "\x78\x9f", "\x00\x00", "\x78\xa1", "\x00\x00",
  "\x00\x00", "\x78\xa4", "\x00\x00", "\x00\x00", "\x00\x00", "\x78\xa8",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x78\xac", "\x78\xad", "\x00\x00",
  "\x00\x00", "\x78\xb0", "\x78\xb1", "\x78\xb2", "\x78\xb3", "\x78\xbb",
  "\x00\x00", "\x78\xbd", "\x00\x00", "\x78\xbf", "\x78\xc7", "\x78\xc8",
  "\x78\xc9", "\x00\x00", "\x00\x00", "\x78\xcc", "\x00\x00", "\x78\xce",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x78\xd2", "\x78\xd3", "\x00\x00",
  "\x78\xd5", "\x78\xd6", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x78\xdb", "\x00\x00", "\x00\x00", "\x00\x00", "\x78\xdf", "\x78\xe0",
  "\x78\xe1", "\x00\x00", "\x00\x00", "\x78\xe4", "\x00\x00", "\x78\xe6",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x78\xea", "\x78\xf2", "\x78\xf3",
  "\x00\x00", "\x00\x00", "\x78\xf6", "\x78\xf7", "\x00\x00", "\x00\x00",
  "\x78\xfa", "\x78\xfb", "\x00\x00", "\x00\x00", "\x00\x00", "\x78\xff",
  "\x79\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x79\x06", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x79\x0c", "\x00\x00", "\x00\x00", "\x00\x00", "\x79\x10", "\x79\x1a",
  "\x00\x00", "\x79\x1c", "\x00\x00", "\x79\x1e", "\x79\x1f", "\x79\x20",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x79\x25", "\x00\x00",
  "\x79\x27", "\x00\x00", "\x79\x29", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x79\x2d", "\x00\x00", "\x00\x00", "\x00\x00", "\x79\x31", "\x00\x00",
  "\x00\x00", "\x79\x34", "\x79\x35", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x79\x3b", "\x00\x00", "\x79\x3d", "\x00\x00",
  "\x79\x3f", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x79\x44",
  "\x79\x45", "\x79\x46", "\x00\x00", "\x00\x00", "\x00\x00", "\x79\x4a",
  "\x79\x4b", "\x00\x00", "\x00\x00", "\x00\x00", "\x79\x4f", "\x00\x00",
  "\x79\x51", "\x00\x00", "\x00\x00", "\x79\x54", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x79\x58", "\x00\x00", "\x00\x00", "\x79\x5b", "\x79\x5c",
  "\x79\x67", "\x00\x00", "\x79\x69", "\x00\x00", "\x79\x6b", "\x79\x72",
  "\x79\x79", "\x00\x00", "\x79\x7b", "\x79\x7c", "\x00\x00", "\x79\x7e",
  "\x79\x8b", "\x79\x8c", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x79\x91", "\x00\x00", "\x79\x93", "\x79\x94", "\x79\x95", "\x79\x96",
  "\x00\x00", "\x79\x98", "\x00\x00", "\x00\x00", "\x79\x9b", "\x79\x9c",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x79\xa1", "\x79\xa8",
  "\x79\xa9", "\x00\x00", "\x79\xab", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x79\xaf", "\x00\x00", "\x79\xb1", "\x00\x00", "\x00\x00", "\x79\xb4",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x79\xb8", "\x00\x00", "\x00\x00",
  "\x79\xbb", "\x79\xc2", "\x00\x00", "\x79\xc4", "\x00\x00", "\x00\x00",
  "\x79\xc7", "\x79\xc8", "\x00\x00", "\x79\xca", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x79\xcf", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x79\xd4", "\x00\x00", "\x79\xd6", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x79\xda", "\x00\x00", "\x00\x00", "\x79\xdd", "\x79\xde",
  "\x00\x00", "\x79\xe0", "\x00\x00", "\x79\xe2", "\x00\x00", "\x00\x00",
  "\x79\xe5", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x79\xea",
  "\x79\xeb", "\x00\x00", "\x79\xed", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x79\xf1", "\x79\xf8", "\x00\x00", "\x00\x00", "\x00\x00", "\x79\xfc",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x7a\x02",
  "\x7a\x03", "\x00\x00", "\x00\x00", "\x00\x00", "\x7a\x07", "\x00\x00",
  "\x7a\x09", "\x7a\x0a", "\x00\x00", "\x7a\x0c", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x7a\x11", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x7a\x15", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x7a\x1b", "\x00\x00", "\x00\x00", "\x7a\x1e", "\x00\x00", "\x00\x00",
  "\x7a\x21", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x7a\x27", "\x00\x00", "\x00\x00", "\x00\x00", "\x7a\x2b", "\x00\x00",
  "\x7a\x2d", "\x00\x00", "\x7a\x2f", "\x7a\x30", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x7a\x34", "\x7a\x35", "\x00\x00", "\x00\x00", "\x7a\x38",
  "\x7a\x39", "\x7a\x3a", "\x7a\x44", "\x7a\x45", "\x00\x00", "\x7a\x47",
  "\x7a\x48", "\x00\x00", "\x00\x00", "\x00\x00", "\x7a\x4c", "\x7a\x55",
  "\x7a\x56", "\x00\x00", "\x00\x00", "\x7a\x59", "\x00\x00", "\x00\x00",
  "\x7a\x5c", "\x7a\x5d", "\x00\x00", "\x7a\x5f", "\x7a\x60", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x7a\x65", "\x00\x00", "\x7a\x67",
  "\x00\x00", "\x00\x00", "\x7a\x6a", "\x00\x00", "\x00\x00", "\x7a\x6d",
  "\x7a\x75", "\x00\x00", "\x00\x00", "\x7a\x78", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x7a\x7e", "\x00\x00", "\x7a\x80",
  "\x00\x00", "\x7a\x82", "\x00\x00", "\x00\x00", "\x7a\x85", "\x7a\x86",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x7a\x8a", "\x7a\x8b", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x7a\x90", "\x7a\x91", "\x00\x00",
  "\x00\x00", "\x7a\x94", "\x7a\x9e", "\x00\x00", "\x7a\xa0", "\x00\x00",
  "\x00\x00", "\x7a\xa3", "\x7a\xac", "\x7a\xb3", "\x00\x00", "\x7a\xb5",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x7a\xb9", "\x00\x00", "\x7a\xbb",
  "\x7a\xbc", "\x7a\xc6", "\x00\x00", "\x00\x00", "\x7a\xc9", "\x00\x00",
  "\x00\x00", "\x7a\xcc", "\x00\x00", "\x7a\xce", "\x00\x00", "\x00\x00",
  "\x7a\xd1", "\x7a\xdb", "\x7a\xe8", "\x7a\xe9", "\x00\x00", "\x7a\xeb",
  "\x7a\xec", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x7a\xf1",
  "\x00\x00", "\x00\x00", "\x7a\xf4", "\x7a\xfb", "\x00\x00", "\x7a\xfd",
  "\x7a\xfe", "\x7b\x07", "\x7b\x14", "\x7b\x1f", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x7b\x23", "\x00\x00", "\x00\x00", "\x00\x00", "\x7b\x27",
  "\x00\x00", "\x7b\x29", "\x7b\x2a", "\x7b\x2b", "\x00\x00", "\x7b\x2d",
  "\x7b\x2e", "\x7b\x2f", "\x7b\x30", "\x7b\x31", "\x00\x00", "\x00\x00",
  "\x7b\x34", "\x7b\x3d", "\x00\x00", "\x7b\x3f", "\x7b\x40", "\x7b\x41",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x7b\x47",
  "\x7b\x4e", "\x7b\x55", "\x7b\x60", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x7b\x64", "\x00\x00", "\x7b\x66", "\x00\x00", "\x00\x00", "\x7b\x69",
  "\x7b\x6a", "\x00\x00", "\x00\x00", "\x7b\x6d", "\x00\x00", "\x7b\x6f",
  "\x00\x00", "\x00\x00", "\x7b\x72", "\x7b\x73", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x7b\x77", "\x7b\x84", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x7b\x89", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x7b\x8e", "\x00\x00", "\x7b\x90", "\x7b\x91", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x7b\x96", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x7b\x9b", "\x00\x00", "\x00\x00", "\x7b\x9e", "\x00\x00",
  "\x7b\xa0", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x7b\xa5",
  "\x7b\xac", "\x00\x00", "\x00\x00", "\x7b\xaf", "\x7b\xb0", "\x00\x00",
  "\x7b\xb2", "\x00\x00", "\x00\x00", "\x7b\xb5", "\x7b\xb6", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x7b\xba", "\x7b\xbb", "\x7b\xbc", "\x7b\xbd",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x7b\xc2", "\x00\x00",
  "\x00\x00", "\x7b\xc5", "\x00\x00", "\x00\x00", "\x7b\xc8", "\x00\x00",
  "\x7b\xca", "\x7b\xd4", "\x00\x00", "\x7b\xd6", "\x7b\xd7", "\x00\x00",
  "\x7b\xd9", "\x7b\xda", "\x7b\xdb", "\x7b\xe8", "\x00\x00", "\x7b\xea",
  "\x7b\xf2", "\x00\x00", "\x7b\xf4", "\x7b\xf5", "\x00\x00", "\x00\x00",
  "\x7b\xf8", "\x7b\xf9", "\x7b\xfa", "\x00\x00", "\x7b\xfc", "\x00\x00",
  "\x7b\xfe", "\x00\x00", "\x00\x00", "\x7c\x01", "\x7c\x02", "\x7c\x03",
  "\x7c\x04", "\x00\x00", "\x7c\x06", "\x00\x00", "\x00\x00", "\x7c\x09",
  "\x00\x00", "\x7c\x0b", "\x7c\x0c", "\x00\x00", "\x7c\x0e", "\x7c\x0f",
  "\x7c\x19", "\x00\x00", "\x7c\x1b", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x7c\x20", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x7c\x25", "\x7c\x26", "\x00\x00", "\x7c\x28", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x7c\x2c", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x7c\x31", "\x00\x00", "\x7c\x33", "\x7c\x34", "\x00\x00", "\x7c\x36",
  "\x00\x00", "\x00\x00", "\x7c\x39", "\x7c\x3a", "\x7c\x46", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x7c\x4a", "\x7c\x51", "\x7c\x52", "\x7c\x53",
  "\x00\x00", "\x7c\x55", "\x00\x00", "\x00\x00", "\x00\x00", "\x7c\x59",
  "\x7c\x5a", "\x7c\x5b", "\x7c\x5c", "\x7c\x5d", "\x7c\x5e", "\x00\x00",
  "\x00\x00", "\x7c\x61", "\x00\x00", "\x7c\x63", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x7c\x67", "\x00\x00", "\x7c\x69", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x7c\x6d", "\x7c\x6e", "\x00\x00", "\x7c\x70", "\x00\x00",
  "\x7c\x72", "\x7c\x79", "\x00\x00", "\x00\x00", "\x7c\x7c", "\x7c\x7d",
  "\x7c\x86", "\x7c\x87", "\x7c\x8f", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x7c\x94", "\x7c\x9e", "\x00\x00", "\x7c\xa0", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x7c\xa6", "\x7c\xb0",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x7c\xb6",
  "\x7c\xb7", "\x00\x00", "\x00\x00", "\x7c\xba", "\x7c\xbb", "\x7c\xbc",
  "\x00\x00", "\x00\x00", "\x7c\xbf", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x7c\xc4", "\x00\x00", "\x00\x00", "\x7c\xc7", "\x7c\xc8",
  "\x7c\xc9", "\x00\x00", "\x00\x00", "\x00\x00", "\x7c\xcd", "\x00\x00",
  "\x7c\xcf", "\x00\x00", "\x00\x00", "\x00\x00", "\x7c\xd3", "\x7c\xd4",
  "\x7c\xd5", "\x00\x00", "\x7c\xd7", "\x00\x00", "\x7c\xd9", "\x7c\xda",
  "\x00\x00", "\x00\x00", "\x7c\xdd", "\x7c\xe6", "\x00\x00", "\x00\x00",
  "\x7c\xe9", "\x00\x00", "\x7c\xeb", "\x7c\xf5", "\x7d\x03", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x7d\x07", "\x7d\x08", "\x7d\x09", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x7d\x0f", "\x00\x00",
  "\x7d\x11", "\x7d\x12", "\x7d\x13", "\x00\x00", "\x00\x00", "\x7d\x16",
  "\x7d\x1d", "\x7d\x1e", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x7d\x23", "\x00\x00", "\x00\x00", "\x7d\x26", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x7d\x2a", "\x00\x00", "\x00\x00", "\x7d\x2d", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x7d\x31", "\x7d\x3c", "\x7d\x3d", "\x7d\x3e",
  "\x00\x00", "\x7d\x40", "\x7d\x41", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x7d\x47", "\x7d\x48", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x7d\x4d", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x7d\x51", "\x00\x00", "\x7d\x53", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x7d\x57", "\x00\x00", "\x7d\x59", "\x7d\x5a", "\x00\x00", "\x7d\x5c",
  "\x7d\x5d", "\x7d\x65", "\x00\x00", "\x7d\x67", "\x00\x00", "\x00\x00",
  "\x7d\x6a", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x7d\x70", "\x7d\x78", "\x00\x00", "\x7d\x7a", "\x7d\x7b", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x7d\x7f", "\x00\x00", "\x7d\x81", "\x7d\x82",
  "\x7d\x83", "\x00\x00", "\x7d\x85", "\x7d\x86", "\x00\x00", "\x7d\x88",
  "\x00\x00", "\x00\x00", "\x7d\x8b", "\x7d\x8c", "\x7d\x8d", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x7d\x91", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x7d\x96", "\x7d\x97", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x7d\x9d", "\x7d\x9e", "\x7d\xa6", "\x7d\xa7",
  "\x00\x00", "\x00\x00", "\x7d\xaa", "\x7d\xb3", "\x00\x00", "\x00\x00",
  "\x7d\xb6", "\x7d\xb7", "\x00\x00", "\x7d\xb9", "\x7d\xc2", "\x7d\xc3",
  "\x7d\xc4", "\x7d\xc5", "\x7d\xc6", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x7d\xcc", "\x7d\xcd", "\x7d\xce", "\x7d\xd7",
  "\x00\x00", "\x7d\xd9", "\x7d\xe2", "\x00\x00", "\x00\x00", "\x7d\xe5",
  "\x7d\xe6", "\x00\x00", "\x00\x00", "\x00\x00", "\x7d\xea", "\x7d\xeb",
  "\x00\x00", "\x7d\xed", "\x00\x00", "\x00\x00", "\x00\x00", "\x7d\xf1",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x7d\xf5", "\x7d\xf6", "\x00\x00",
  "\x00\x00", "\x7d\xf9", "\x7d\xfa", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x7e\x00", "\x7e\x08", "\x7e\x10", "\x7e\x11",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x7e\x15", "\x00\x00", "\x7e\x17",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x7e\x1c", "\x7e\x1d",
  "\x00\x00", "\x00\x00", "\x7e\x20", "\x7e\x27", "\x7e\x28", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x7e\x2c", "\x7e\x2d", "\x00\x00", "\x7e\x2f",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x7e\x33", "\x00\x00", "\x00\x00",
  "\x7e\x36", "\x7e\x3f", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x7e\x44", "\x7e\x45", "\x00\x00", "\x7e\x47", "\x7e\x4e", "\x00\x00",
  "\x7e\x50", "\x00\x00", "\x7e\x52", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x7e\x58", "\x7e\x5f", "\x00\x00", "\x7e\x61",
  "\x7e\x62", "\x00\x00", "\x00\x00", "\x7e\x65", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x7e\x6b", "\x00\x00", "\x00\x00",
  "\x7e\x6e", "\x7e\x6f", "\x00\x00", "\x00\x00", "\x00\x00", "\x7e\x73",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x7e\x78", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x7e\x7e", "\x00\x00",
  "\x00\x00", "\x7e\x81", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x7e\x86", "\x7e\x87", "\x00\x00", "\x00\x00", "\x7e\x8a", "\x00\x00",
  "\x00\x00", "\x7e\x8d", "\x00\x00", "\x00\x00", "\x00\x00", "\x7e\x91",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x7e\x95", "\x00\x00", "\x00\x00",
  "\x7e\x98", "\x00\x00", "\x7e\x9a", "\x00\x00", "\x00\x00", "\x7e\x9d",
  "\x7e\x9e", "\x7f\x3b", "\x7f\x3c", "\x7f\x3d", "\x7f\x3e", "\x7f\x3f",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x7f\x43", "\x7f\x44", "\x00\x00",
  "\x00\x00", "\x7f\x47", "\x7f\x4f", "\x00\x00", "\x00\x00", "\x7f\x52",
  "\x7f\x53", "\x7f\x5b", "\x7f\x5c", "\x7f\x5d", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x7f\x61", "\x00\x00", "\x7f\x63", "\x7f\x64", "\x7f\x65",
  "\x7f\x66", "\x7f\x6d", "\x00\x00", "\x00\x00", "\x00\x00", "\x7f\x71",
  "\x7f\x7d", "\x7f\x7e", "\x7f\x7f", "\x7f\x80", "\x7f\x8b", "\x00\x00",
  "\x7f\x8d", "\x00\x00", "\x7f\x8f", "\x7f\x90", "\x7f\x91", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x7f\x96", "\x7f\x97", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x7f\x9c", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x7f\xa1", "\x7f\xa2", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x7f\xa6", "\x00\x00", "\x00\x00", "\x00\x00", "\x7f\xaa",
  "\x00\x00", "\x00\x00", "\x7f\xad", "\x7f\xb4", "\x7f\xbc", "\x00\x00",
  "\x00\x00", "\x7f\xbf", "\x7f\xc0", "\x00\x00", "\x00\x00", "\x7f\xc3",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x7f\xc8", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x7f\xce", "\x7f\xcf",
  "\x7f\xdb", "\x00\x00", "\x00\x00", "\x00\x00", "\x7f\xdf", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x7f\xe3", "\x00\x00", "\x7f\xe5", "\x00\x00",
  "\x00\x00", "\x7f\xe8", "\x00\x00", "\x00\x00", "\x00\x00", "\x7f\xec",
  "\x00\x00", "\x7f\xee", "\x7f\xef", "\x00\x00", "\x00\x00", "\x7f\xf2",
  "\x7f\xfa", "\x00\x00", "\x00\x00", "\x7f\xfd", "\x7f\xfe", "\x7f\xff",
  "\x80\x07", "\x80\x08", "\x00\x00", "\x80\x0a", "\x00\x00", "\x00\x00",
  "\x80\x0d", "\x80\x0e", "\x80\x0f", "\x00\x00", "\x80\x11", "\x00\x00",
  "\x80\x13", "\x80\x14", "\x00\x00", "\x80\x16", "\x80\x1d", "\x80\x1e",
  "\x80\x1f", "\x80\x20", "\x00\x00", "\x00\x00", "\x00\x00", "\x80\x24",
  "\x00\x00", "\x80\x26", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x80\x2c", "\x00\x00", "\x80\x2e", "\x00\x00", "\x80\x30",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x80\x34", "\x80\x35", "\x00\x00",
  "\x80\x37", "\x00\x00", "\x80\x39", "\x80\x3a", "\x00\x00", "\x80\x3c",
  "\x00\x00", "\x80\x3e", "\x00\x00", "\x80\x40", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x80\x44", "\x80\x60", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x80\x64", "\x00\x00", "\x80\x66", "\x80\x6d", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x80\x71", "\x00\x00", "\x00\x00", "\x00\x00", "\x80\x75",
  "\x80\x81", "\x80\x88", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x80\x8e", "\x80\x9c", "\x00\x00", "\x80\x9e", "\x80\xa6",
  "\x80\xa7", "\x00\x00", "\x00\x00", "\x00\x00", "\x80\xab", "\x80\xb8",
  "\x80\xb9", "\x80\xc8", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x80\xcd", "\x00\x00", "\x80\xcf", "\x00\x00", "\x00\x00", "\x80\xd2",
  "\x00\x00", "\x80\xd4", "\x80\xd5", "\x00\x00", "\x80\xd7", "\x80\xd8",
  "\x80\xe0", "\x80\xed", "\x80\xee", "\x00\x00", "\x80\xf0", "\x00\x00",
  "\x80\xf2", "\x80\xf3", "\x00\x00", "\x00\x00", "\x80\xf6", "\x00\x00",
  "\x00\x00", "\x80\xf9", "\x80\xfa", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x80\xfe", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x81\x03",
  "\x81\x0b", "\x81\x16", "\x81\x17", "\x81\x18", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x81\x1c", "\x00\x00", "\x81\x1e", "\x00\x00", "\x81\x20",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x81\x24", "\x00\x00", "\x00\x00",
  "\x81\x27", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x81\x2c",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x81\x30", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x81\x35", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x81\x3a", "\x00\x00", "\x81\x3c", "\x81\x45", "\x00\x00",
  "\x81\x47", "\x00\x00", "\x00\x00", "\x81\x4a", "\x00\x00", "\x81\x4c",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x81\x52",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x81\x57", "\x81\x60",
  "\x81\x61", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x81\x67", "\x81\x68", "\x81\x69", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x81\x6d", "\x00\x00", "\x81\x6f", "\x81\x77", "\x81\x81", "\x00\x00",
  "\x00\x00", "\x81\x84", "\x81\x85", "\x81\x86", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x81\x8b", "\x00\x00", "\x00\x00", "\x81\x8e",
  "\x00\x00", "\x81\x90", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x81\x96", "\x00\x00", "\x81\x98", "\x00\x00", "\x00\x00",
  "\x81\x9b", "\x00\x00", "\x00\x00", "\x81\x9e", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x81\xa2", "\x81\xae", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x81\xb2", "\x00\x00", "\x81\xb4", "\x81\xbb", "\x81\xc3", "\x00\x00",
  "\x81\xc5", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x81\xca",
  "\x81\xcb", "\x00\x00", "\x00\x00", "\x81\xce", "\x81\xcf", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x81\xd5", "\x00\x00",
  "\x81\xd7", "\x00\x00", "\x00\x00", "\x00\x00", "\x81\xdb", "\x00\x00",
  "\x81\xdd", "\x81\xde", "\x00\x00", "\x00\x00", "\x81\xe1", "\x00\x00",
  "\x00\x00", "\x81\xe4", "\x81\xeb", "\x81\xec", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x81\xf0", "\x81\xf1", "\x81\xf2", "\x00\x00", "\x00\x00",
  "\x81\xf5", "\x81\xf6", "\x00\x00", "\x81\xf8", "\x81\xf9", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x81\xfd", "\x00\x00", "\x81\xff", "\x82\x00",
  "\x00\x00", "\x00\x00", "\x82\x03", "\x82\x0f", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x82\x13", "\x82\x14", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x82\x19", "\x82\x1a", "\x00\x00", "\x00\x00", "\x82\x1d",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x82\x21", "\x82\x22", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x82\x28", "\x82\x32",
  "\x00\x00", "\x82\x34", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x82\x3a", "\x82\x43", "\x82\x44", "\x82\x45", "\x82\x46",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x82\x4b", "\x00\x00",
  "\x00\x00", "\x82\x4e", "\x82\x4f", "\x00\x00", "\x82\x51", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x82\x56", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x82\x5c", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x82\x60", "\x00\x00", "\x00\x00", "\x82\x63", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x82\x67", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x82\x6d", "\x82\x74", "\x82\x7b", "\x00\x00",
  "\x82\x7d", "\x00\x00", "\x82\x7f", "\x82\x80", "\x82\x81", "\x00\x00",
  "\x82\x83", "\x82\x84", "\x00\x00", "\x00\x00", "\x82\x87", "\x00\x00",
  "\x82\x89", "\x82\x8a", "\x00\x00", "\x00\x00", "\x00\x00", "\x82\x8e",
  "\x00\x00", "\x00\x00", "\x82\x91", "\x00\x00", "\x00\x00", "\x82\x94",
  "\x00\x00", "\x82\x96", "\x00\x00", "\x82\x98", "\x00\x00", "\x82\x9a",
  "\x82\x9b", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x82\xa0",
  "\x82\xa1", "\x00\x00", "\x82\xa3", "\x82\xa4", "\x00\x00", "\x00\x00",
  "\x82\xa7", "\x82\xa8", "\x82\xa9", "\x82\xaa", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x82\xae", "\x00\x00", "\x82\xb0", "\x00\x00", "\x82\xb2",
  "\x00\x00", "\x82\xb4", "\x00\x00", "\x00\x00", "\x82\xb7", "\x00\x00",
  "\x00\x00", "\x82\xba", "\x00\x00", "\x82\xbc", "\x00\x00", "\x82\xbe",
  "\x82\xbf", "\x82\xc6", "\x82\xd0", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x82\xd5", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x82\xda", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x82\xe0", "\x00\x00", "\x82\xe2", "\x00\x00", "\x82\xe4", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x82\xe8", "\x00\x00", "\x82\xea", "\x00\x00",
  "\x00\x00", "\x82\xed", "\x00\x00", "\x82\xef", "\x82\xf6", "\x82\xf7",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x82\xfd",
  "\x82\xfe", "\x00\x00", "\x83\x00", "\x83\x01", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x83\x07", "\x83\x08", "\x00\x00",
  "\x83\x0a", "\x83\x0b", "\x83\x1b", "\x00\x00", "\x83\x1d", "\x83\x1e",
  "\x83\x1f", "\x00\x00", "\x83\x21", "\x83\x22", "\x83\x2c", "\x83\x2d",
  "\x83\x2e", "\x00\x00", "\x83\x30", "\x00\x00", "\x00\x00", "\x83\x33",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x83\x37", "\x00\x00", "\x00\x00",
  "\x83\x3a", "\x00\x00", "\x83\x3c", "\x83\x3d", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x83\x42", "\x83\x43", "\x83\x44", "\x00\x00",
  "\x00\x00", "\x83\x47", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x83\x4d", "\x83\x4e", "\x00\x00", "\x00\x00", "\x83\x51",
  "\x00\x00", "\x83\x53", "\x83\x54", "\x83\x55", "\x83\x56", "\x83\x57",
  "\x83\x70", "\x83\x78", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x83\x7d", "\x00\x00", "\x83\x7f", "\x83\x80", "\x00\x00", "\x83\x82",
  "\x00\x00", "\x83\x84", "\x00\x00", "\x83\x86", "\x83\x8d", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x83\x92", "\x00\x00", "\x83\x94",
  "\x83\x95", "\x00\x00", "\x00\x00", "\x83\x98", "\x83\x99", "\x00\x00",
  "\x83\x9b", "\x83\x9c", "\x83\x9d", "\x83\xa6", "\x83\xa7", "\x00\x00",
  "\x83\xa9", "\x00\x00", "\x00\x00", "\x83\xac", "\x83\xad", "\x83\xbe",
  "\x83\xbf", "\x83\xc0", "\x83\xc7", "\x00\x00", "\x83\xc9", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x83\xcf", "\x83\xd0",
  "\x83\xd1", "\x00\x00", "\x00\x00", "\x83\xd4", "\x83\xdd", "\x83\xe8",
  "\x00\x00", "\x83\xea", "\x83\xf6", "\x00\x00", "\x83\xf8", "\x83\xf9",
  "\x00\x00", "\x00\x00", "\x83\xfc", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x84\x01", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x84\x06", "\x00\x00", "\x00\x00", "\x00\x00", "\x84\x0a", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x84\x0f", "\x00\x00", "\x84\x11",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x84\x15", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x84\x19", "\x84\x2f", "\x84\x39", "\x84\x45", "\x00\x00",
  "\x84\x47", "\x84\x48", "\x00\x00", "\x84\x4a", "\x00\x00", "\x00\x00",
  "\x84\x4d", "\x00\x00", "\x84\x4f", "\x00\x00", "\x84\x51", "\x84\x52",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x84\x56", "\x00\x00", "\x84\x58",
  "\x84\x59", "\x84\x5a", "\x00\x00", "\x84\x5c", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x84\x60", "\x00\x00", "\x00\x00", "\x00\x00", "\x84\x64",
  "\x84\x65", "\x00\x00", "\x84\x67", "\x00\x00", "\x00\x00", "\x84\x6a",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x84\x70",
  "\x00\x00", "\x00\x00", "\x84\x73", "\x84\x74", "\x00\x00", "\x84\x76",
  "\x00\x00", "\x84\x78", "\x00\x00", "\x00\x00", "\x00\x00", "\x84\x7c",
  "\x84\x7d", "\x00\x00", "\x00\x00", "\x00\x00", "\x84\x81", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x84\x85", "\x84\x92", "\x84\x93", "\x00\x00",
  "\x84\x95", "\x84\x9e", "\x84\xa6", "\x00\x00", "\x84\xa8", "\x84\xa9",
  "\x84\xaa", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x84\xaf",
  "\x00\x00", "\x84\xb1", "\x00\x00", "\x00\x00", "\x84\xb4", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x84\xba", "\x00\x00",
  "\x00\x00", "\x84\xbd", "\x84\xbe", "\x00\x00", "\x84\xc0", "\x00\x00",
  "\x84\xc2", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x84\xc7",
  "\x84\xc8", "\x00\x00", "\x00\x00", "\x00\x00", "\x84\xcc", "\x00\x00",
  "\x00\x00", "\x84\xcf", "\x00\x00", "\x00\x00", "\x00\x00", "\x84\xd3",
  "\x84\xdc", "\x84\xe7", "\x00\x00", "\x00\x00", "\x84\xea", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x84\xef", "\x84\xf0", "\x84\xf1",
  "\x84\xf2", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x84\xf7",
  "\x00\x00", "\x00\x00", "\x84\xfa", "\x84\xfb", "\x00\x00", "\x84\xfd",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x85\x02", "\x85\x03",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x85\x07", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x85\x0c", "\x00\x00", "\x85\x0e", "\x00\x00",
  "\x85\x10", "\x85\x1c", "\x00\x00", "\x85\x1e", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x85\x22", "\x85\x23", "\x85\x24", "\x85\x25", "\x00\x00",
  "\x85\x27", "\x00\x00", "\x00\x00", "\x85\x2a", "\x85\x2b", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x85\x2f", "\x00\x00", "\x00\x00", "\x85\x32",
  "\x85\x33", "\x85\x34", "\x00\x00", "\x85\x36", "\x85\x3f", "\x85\x46",
  "\x85\x4f", "\x85\x50", "\x85\x51", "\x85\x52", "\x85\x53", "\x00\x00",
  "\x00\x00", "\x85\x56", "\x00\x00", "\x00\x00", "\x85\x59", "\x00\x00",
  "\x00\x00", "\x85\x5c", "\x85\x5d", "\x85\x5e", "\x85\x5f", "\x85\x60",
  "\x85\x61", "\x85\x62", "\x00\x00", "\x85\x64", "\x85\x6b", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x85\x6f", "\x85\x79", "\x85\x7a", "\x85\x7b",
  "\x00\x00", "\x85\x7d", "\x00\x00", "\x85\x7f", "\x00\x00", "\x85\x81",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x85\x85", "\x85\x86", "\x00\x00",
  "\x00\x00", "\x85\x89", "\x00\x00", "\x85\x8b", "\x85\x8c", "\x00\x00",
  "\x00\x00", "\x85\x8f", "\x00\x00", "\x00\x00", "\x00\x00", "\x85\x93",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x85\x98", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x85\x9d", "\x00\x00", "\x85\x9f",
  "\x85\xa0", "\x00\x00", "\x85\xa2", "\x00\x00", "\x00\x00", "\x85\xa5",
  "\x00\x00", "\x85\xa7", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x85\xad", "\x85\xb4", "\x00\x00", "\x85\xb6", "\x85\xb7",
  "\x85\xb8", "\x00\x00", "\x00\x00", "\x00\x00", "\x85\xbc", "\x85\xbd",
  "\x85\xbe", "\x85\xbf", "\x00\x00", "\x00\x00", "\x85\xc2", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x85\xc7", "\x00\x00", "\x00\x00",
  "\x85\xca", "\x85\xcb", "\x00\x00", "\x00\x00", "\x85\xce", "\x85\xd8",
  "\x00\x00", "\x85\xda", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x85\xdf", "\x85\xe0", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x85\xe6", "\x00\x00", "\x85\xe8", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x85\xed", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x85\xf3", "\x00\x00", "\x00\x00", "\x85\xf6",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x85\xfc",
  "\x00\x00", "\x00\x00", "\x85\xff", "\x86\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x86\x04", "\x86\x05", "\x86\x0d", "\x86\x0e", "\x00\x00",
  "\x86\x10", "\x86\x11", "\x86\x12", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x86\x18", "\x86\x19", "\x00\x00", "\x86\x1b",
  "\x00\x00", "\x00\x00", "\x86\x1e", "\x00\x00", "\x00\x00", "\x86\x21",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x86\x27",
  "\x00\x00", "\x86\x29", "\x86\x36", "\x00\x00", "\x86\x38", "\x00\x00",
  "\x86\x3a", "\x00\x00", "\x86\x3c", "\x86\x3d", "\x00\x00", "\x00\x00",
  "\x86\x40", "\x86\x41", "\x86\x42", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x86\x46", "\x86\x52", "\x86\x53", "\x00\x00", "\x00\x00", "\x86\x56",
  "\x86\x57", "\x86\x58", "\x86\x59", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x86\x5d", "\x00\x00", "\x00\x00", "\x86\x60", "\x86\x61", "\x86\x62",
  "\x86\x63", "\x86\x64", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x86\x69", "\x00\x00", "\x00\x00", "\x86\x6c", "\x00\x00", "\x00\x00",
  "\x86\x6f", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x86\x75", "\x86\x76", "\x86\x77", "\x00\x00", "\x00\x00", "\x86\x7a",
  "\x86\x88", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x86\x8d",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x86\x91", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x86\x96", "\x00\x00", "\x86\x98", "\x00\x00",
  "\x86\x9a", "\x00\x00", "\x86\x9c", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x86\xa1", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x86\xa6", "\x86\xa7", "\x86\xa8", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x86\xad", "\x00\x00", "\x00\x00", "\x00\x00", "\x86\xb1",
  "\x00\x00", "\x86\xb3", "\x86\xb4", "\x86\xb5", "\x00\x00", "\x86\xb7",
  "\x86\xb8", "\x86\xb9", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x86\xbf", "\x86\xc0", "\x86\xc1", "\x00\x00", "\x86\xc3",
  "\x00\x00", "\x86\xc5", "\x86\xd1", "\x86\xd2", "\x00\x00", "\x00\x00",
  "\x86\xd5", "\x00\x00", "\x86\xd7", "\x00\x00", "\x00\x00", "\x86\xda",
  "\x00\x00", "\x86\xdc", "\x00\x00", "\x00\x00", "\x00\x00", "\x86\xe0",
  "\x00\x00", "\x00\x00", "\x86\xe3", "\x00\x00", "\x86\xe5", "\x00\x00",
  "\x86\xe7", "\x86\xfa", "\x00\x00", "\x86\xfc", "\x86\xfd", "\x87\x04",
  "\x87\x05", "\x00\x00", "\x87\x07", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x87\x0b", "\x00\x00", "\x00\x00", "\x87\x0e", "\x87\x0f", "\x87\x10",
  "\x00\x00", "\x00\x00", "\x87\x13", "\x87\x14", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x87\x19", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x87\x1e", "\x87\x1f", "\x00\x00", "\x87\x21", "\x00\x00",
  "\x87\x23", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x87\x28",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x87\x2e",
  "\x87\x2f", "\x00\x00", "\x87\x31", "\x87\x32", "\x87\x39", "\x87\x3a",
  "\x00\x00", "\x87\x3c", "\x87\x3d", "\x87\x3e", "\x00\x00", "\x87\x40",
  "\x00\x00", "\x00\x00", "\x87\x43", "\x00\x00", "\x87\x45", "\x87\x4d",
  "\x87\x58", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x87\x5d",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x87\x61", "\x00\x00", "\x00\x00",
  "\x87\x64", "\x87\x65", "\x87\x6f", "\x00\x00", "\x87\x71", "\x87\x72",
  "\x87\x7b", "\x87\x83", "\x87\x84", "\x87\x85", "\x87\x86", "\x87\x87",
  "\x87\x88", "\x87\x89", "\x00\x00", "\x87\x8b", "\x87\x8c", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x87\x90", "\x00\x00", "\x00\x00", "\x87\x93",
  "\x00\x00", "\x87\x95", "\x00\x00", "\x87\x97", "\x87\x98", "\x87\x99",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x87\x9e", "\x00\x00",
  "\x87\xa0", "\x00\x00", "\x00\x00", "\x87\xa3", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x87\xa7", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x87\xac", "\x87\xad", "\x87\xae", "\x00\x00", "\x00\x00", "\x87\xb1",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x87\xb5", "\x87\xbe", "\x87\xbf",
  "\x00\x00", "\x87\xc1", "\x87\xc8", "\x87\xc9", "\x87\xca", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x87\xce", "\x87\xd5", "\x87\xd6", "\x00\x00",
  "\x00\x00", "\x87\xd9", "\x87\xda", "\x00\x00", "\x87\xdc", "\x00\x00",
  "\x00\x00", "\x87\xdf", "\x00\x00", "\x00\x00", "\x87\xe2", "\x87\xe3",
  "\x87\xe4", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x87\xea", "\x87\xeb", "\x00\x00", "\x87\xed", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x87\xf1", "\x00\x00", "\x87\xf3", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x87\xf8", "\x00\x00", "\x87\xfa", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x87\xff", "\x00\x00", "\x88\x01",
  "\x00\x00", "\x88\x03", "\x00\x00", "\x00\x00", "\x88\x06", "\x00\x00",
  "\x00\x00", "\x88\x09", "\x88\x0a", "\x88\x0b", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x88\x10", "\x00\x00", "\x88\x12", "\x88\x13",
  "\x88\x14", "\x00\x00", "\x00\x00", "\x00\x00", "\x88\x18", "\x88\x19",
  "\x88\x1a", "\x88\x1b", "\x88\x1c", "\x00\x00", "\x88\x1e", "\x88\x1f",
  "\x88\x28", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x88\x2d",
  "\x88\x2e", "\x00\x00", "\x88\x30", "\x00\x00", "\x88\x32", "\x00\x00",
  "\x00\x00", "\x88\x35", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x88\x3a", "\x00\x00", "\x88\x3c", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x88\x41", "\x00\x00", "\x88\x43", "\x00\x00", "\x88\x45",
  "\x00\x00", "\x00\x00", "\x88\x48", "\x88\x49", "\x88\x4a", "\x88\x4b",
  "\x00\x00", "\x00\x00", "\x88\x4e", "\x00\x00", "\x00\x00", "\x88\x51",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x88\x55", "\x88\x56", "\x00\x00",
  "\x88\x58", "\x00\x00", "\x88\x5a", "\x00\x00", "\x88\x5c", "\x00\x00",
  "\x00\x00", "\x88\x5f", "\x88\x60", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x88\x64", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x88\x69",
  "\x88\x71", "\x88\x79", "\x00\x00", "\x88\x7b", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x88\x80", "\x88\x98", "\x00\x00", "\x88\x9a",
  "\x88\x9b", "\x88\x9c", "\x00\x00", "\x00\x00", "\x88\x9f", "\x88\xa0",
  "\x88\xa8", "\x00\x00", "\x88\xaa", "\x88\xba", "\x00\x00", "\x00\x00",
  "\x88\xbd", "\x88\xbe", "\x00\x00", "\x88\xc0", "\x88\xca", "\x88\xcb",
  "\x88\xcc", "\x88\xcd", "\x88\xce", "\x00\x00", "\x00\x00", "\x88\xd1",
  "\x88\xd2", "\x88\xd3", "\x88\xdb", "\x00\x00", "\x00\x00", "\x88\xde",
  "\x88\xe7", "\x88\xef", "\x88\xf0", "\x88\xf1", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x88\xf5", "\x00\x00", "\x88\xf7", "\x89\x01", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x89\x06", "\x89\x0d", "\x89\x0e",
  "\x89\x0f", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x89\x15", "\x89\x16", "\x00\x00", "\x89\x18", "\x89\x19", "\x89\x1a",
  "\x00\x00", "\x89\x1c", "\x00\x00", "\x00\x00", "\x00\x00", "\x89\x20",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x89\x26",
  "\x89\x27", "\x89\x28", "\x89\x30", "\x89\x31", "\x89\x32", "\x00\x00",
  "\x00\x00", "\x89\x35", "\x00\x00", "\x00\x00", "\x00\x00", "\x89\x39",
  "\x89\x3a", "\x00\x00", "\x00\x00", "\x00\x00", "\x89\x3e", "\x00\x00",
  "\x89\x40", "\x00\x00", "\x89\x42", "\x00\x00", "\x00\x00", "\x89\x45",
  "\x89\x46", "\x00\x00", "\x00\x00", "\x89\x49", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x89\x4f", "\x00\x00", "\x00\x00",
  "\x89\x52", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x89\x57",
  "\x00\x00", "\x00\x00", "\x89\x5a", "\x89\x5b", "\x89\x5c", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x89\x61", "\x89\x62", "\x89\x63",
  "\x89\x6b", "\x00\x00", "\x00\x00", "\x89\x6e", "\x00\x00", "\x89\x70",
  "\x00\x00", "\x00\x00", "\x89\x73", "\x00\x00", "\x89\x75", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x89\x7a", "\x89\x7b", "\x89\x7c",
  "\x89\x7d", "\x89\x89", "\x00\x00", "\x00\x00", "\x00\x00", "\x89\x8d",
  "\x00\x00", "\x00\x00", "\x89\x90", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x89\x94", "\x89\x95", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x89\x9b", "\x89\x9c", "\x00\x00", "\x00\x00", "\x89\x9f",
  "\x89\xa0", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x89\xa5",
  "\x89\xb0", "\x00\x00", "\x00\x00", "\x00\x00", "\x89\xb4", "\x89\xb5",
  "\x89\xb6", "\x89\xb7", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x89\xbc", "\x89\xd4", "\x89\xd5", "\x89\xd6", "\x89\xd7", "\x89\xd8",
  "\x89\xe5", "\x00\x00", "\x00\x00", "\x00\x00", "\x89\xe9", "\x00\x00",
  "\x89\xeb", "\x00\x00", "\x89\xed", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x89\xf1", "\x00\x00", "\x89\xf3", "\x00\x00", "\x00\x00", "\x89\xf6",
  "\x00\x00", "\x00\x00", "\x89\xf9", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x89\xfd", "\x00\x00", "\x89\xff", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x8a\x04", "\x8a\x05", "\x00\x00", "\x8a\x07", "\x8a\x0f",
  "\x00\x00", "\x8a\x11", "\x8a\x12", "\x00\x00", "\x8a\x14", "\x8a\x15",
  "\x8a\x1e", "\x00\x00", "\x8a\x20", "\x00\x00", "\x8a\x22", "\x00\x00",
  "\x8a\x24", "\x00\x00", "\x8a\x26", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x8a\x2b", "\x8a\x2c", "\x00\x00", "\x00\x00", "\x8a\x2f",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x8a\x35",
  "\x00\x00", "\x8a\x37", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x8a\x3d", "\x8a\x3e", "\x00\x00", "\x8a\x40", "\x00\x00",
  "\x00\x00", "\x8a\x43", "\x00\x00", "\x8a\x45", "\x00\x00", "\x8a\x47",
  "\x00\x00", "\x8a\x49", "\x00\x00", "\x00\x00", "\x00\x00", "\x8a\x4d",
  "\x8a\x4e", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x8a\x53",
  "\x00\x00", "\x00\x00", "\x8a\x56", "\x8a\x57", "\x8a\x58", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x8a\x5c", "\x8a\x5d", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x8a\x61", "\x00\x00", "\x00\x00", "\x00\x00", "\x8a\x65",
  "\x00\x00", "\x8a\x67", "\x8a\x75", "\x8a\x76", "\x8a\x77", "\x00\x00",
  "\x8a\x79", "\x8a\x7a", "\x8a\x7b", "\x00\x00", "\x00\x00", "\x8a\x7e",
  "\x8a\x7f", "\x8a\x80", "\x00\x00", "\x00\x00", "\x8a\x83", "\x00\x00",
  "\x00\x00", "\x8a\x86", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x8a\x8b", "\x00\x00", "\x00\x00", "\x00\x00", "\x8a\x8f", "\x8a\x90",
  "\x00\x00", "\x8a\x92", "\x00\x00", "\x00\x00", "\x00\x00", "\x8a\x96",
  "\x8a\x97", "\x00\x00", "\x8a\x99", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x8a\x9f", "\x8a\xa7", "\x00\x00", "\x8a\xa9",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x8a\xae", "\x8a\xaf",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x8a\xb3", "\x00\x00", "\x00\x00",
  "\x8a\xb6", "\x8a\xb7", "\x00\x00", "\x00\x00", "\x00\x00", "\x8a\xbb",
  "\x00\x00", "\x00\x00", "\x8a\xbe", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x8a\xc3", "\x00\x00", "\x00\x00", "\x8a\xc6", "\x00\x00",
  "\x8a\xc8", "\x8a\xc9", "\x8a\xca", "\x8a\xd1", "\x00\x00", "\x8a\xd3",
  "\x8a\xd4", "\x8a\xd5", "\x00\x00", "\x8a\xd7", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x8a\xdd", "\x00\x00", "\x8a\xdf",
  "\x8a\xec", "\x00\x00", "\x00\x00", "\x00\x00", "\x8a\xf0", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x8a\xf4", "\x8a\xf5", "\x8a\xf6", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x8a\xfc", "\x00\x00",
  "\x00\x00", "\x8a\xff", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x8b\x05", "\x8b\x06", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x8b\x0a", "\x8b\x0b", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x8b\x11", "\x8b\x1c", "\x00\x00", "\x8b\x1e", "\x8b\x1f",
  "\x8b\x2d", "\x00\x00", "\x00\x00", "\x8b\x30", "\x8b\x37", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x8b\x3c", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x8b\x42", "\x8b\x43", "\x8b\x44",
  "\x8b\x45", "\x8b\x46", "\x00\x00", "\x8b\x48", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x8b\x4d", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x8b\x52", "\x8b\x53", "\x8b\x54", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x8b\x59", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x8b\x5e", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x8b\x63", "\x8b\x6d", "\x8b\x76", "\x00\x00", "\x8b\x78", "\x8b\x79",
  "\x00\x00", "\x00\x00", "\x8b\x7c", "\x00\x00", "\x8b\x7e", "\x00\x00",
  "\x00\x00", "\x8b\x81", "\x00\x00", "\x00\x00", "\x8b\x84", "\x8b\x85",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x8b\x8b",
  "\x00\x00", "\x8b\x8d", "\x00\x00", "\x8b\x8f", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x8b\x94", "\x8b\x95", "\x8b\x9c", "\x00\x00",
  "\x8b\x9e", "\x8b\x9f", "\x8c\x38", "\x8c\x39", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x8c\x3d", "\x8c\x3e", "\x8c\x45", "\x00\x00", "\x8c\x47",
  "\x00\x00", "\x8c\x49", "\x00\x00", "\x8c\x4b", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x8c\x4f", "\x00\x00", "\x8c\x51", "\x00\x00", "\x8c\x53",
  "\x8c\x54", "\x00\x00", "\x00\x00", "\x8c\x57", "\x8c\x58", "\x8c\x59",
  "\x00\x00", "\x8c\x5b", "\x00\x00", "\x8c\x5d", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x8c\x63", "\x8c\x64", "\x00\x00",
  "\x8c\x66", "\x00\x00", "\x8c\x68", "\x8c\x69", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x8c\x6d", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x8c\x73", "\x00\x00", "\x8c\x75", "\x8c\x76", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x8c\x7b", "\x00\x00", "\x00\x00",
  "\x8c\x7e", "\x8c\x86", "\x8c\x87", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x8c\x8b", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x8c\x90",
  "\x00\x00", "\x8c\x92", "\x8c\x93", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x8c\x99", "\x00\x00", "\x8c\x9b", "\x8c\x9c",
  "\x8c\xa4", "\x8c\xb9", "\x8c\xba", "\x8c\xc5", "\x8c\xc6", "\x00\x00",
  "\x00\x00", "\x8c\xc9", "\x00\x00", "\x8c\xcb", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x8c\xcf", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x8c\xd5", "\x8c\xd6", "\x00\x00", "\x00\x00", "\x8c\xd9",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x8c\xdd", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x8c\xe1", "\x8c\xe8", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x8c\xec", "\x00\x00", "\x00\x00", "\x8c\xef", "\x8c\xf0", "\x00\x00",
  "\x8c\xf2", "\x00\x00", "\x00\x00", "\x8c\xf5", "\x00\x00", "\x8c\xf7",
  "\x8c\xf8", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x8c\xfe", "\x8c\xff", "\x00\x00", "\x8d\x01", "\x00\x00", "\x8d\x03",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x8d\x09",
  "\x8d\x12", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x8d\x17",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x8d\x1b", "\x8d\x65", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x8d\x69", "\x00\x00", "\x00\x00", "\x8d\x6c",
  "\x00\x00", "\x8d\x6e", "\x8d\x7f", "\x00\x00", "\x00\x00", "\x8d\x82",
  "\x00\x00", "\x8d\x84", "\x00\x00", "\x00\x00", "\x00\x00", "\x8d\x88",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x8d\x8d", "\x00\x00",
  "\x00\x00", "\x8d\x90", "\x8d\x91", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x8d\x95", "\x8d\x9e", "\x8d\x9f", "\x8d\xa0", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x8d\xa6", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x8d\xab", "\x8d\xac", "\x00\x00", "\x00\x00",
  "\x8d\xaf", "\x00\x00", "\x00\x00", "\x8d\xb2", "\x00\x00", "\x00\x00",
  "\x8d\xb5", "\x00\x00", "\x8d\xb7", "\x00\x00", "\x8d\xb9", "\x00\x00",
  "\x8d\xbb", "\x8d\xbc", "\x00\x00", "\x00\x00", "\x00\x00", "\x8d\xc0",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x8d\xc5", "\x8d\xc6",
  "\x8d\xc7", "\x8d\xc8", "\x00\x00", "\x8d\xca", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x8d\xce", "\x00\x00", "\x00\x00", "\x8d\xd1", "\x00\x00",
  "\x00\x00", "\x8d\xd4", "\x8d\xd5", "\x00\x00", "\x8d\xd7", "\x00\x00",
  "\x8d\xd9", "\x8d\xe4", "\x8d\xe5", "\x00\x00", "\x8d\xe7", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x8d\xec", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x8d\xf0", "\x8d\xf1", "\x8d\xf2", "\x00\x00", "\x8d\xf4",
  "\x8d\xfd", "\x00\x00", "\x00\x00", "\x00\x00", "\x8e\x01", "\x00\x00",
  "\x00\x00", "\x8e\x04", "\x8e\x05", "\x8e\x06", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x8e\x0b", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x8e\x11", "\x00\x00", "\x00\x00", "\x8e\x14",
  "\x00\x00", "\x8e\x16", "\x8e\x20", "\x8e\x21", "\x8e\x22", "\x8e\x23",
  "\x00\x00", "\x00\x00", "\x8e\x26", "\x8e\x27", "\x8e\x31", "\x00\x00",
  "\x8e\x33", "\x00\x00", "\x00\x00", "\x8e\x36", "\x8e\x37", "\x8e\x38",
  "\x8e\x39", "\x00\x00", "\x00\x00", "\x00\x00", "\x8e\x3d", "\x00\x00",
  "\x00\x00", "\x8e\x40", "\x8e\x41", "\x8e\x4b", "\x00\x00", "\x8e\x4d",
  "\x8e\x4e", "\x8e\x4f", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x8e\x54", "\x8e\x5b", "\x8e\x5c", "\x8e\x5d", "\x8e\x5e", "\x00\x00",
  "\x00\x00", "\x8e\x61", "\x8e\x62", "\x8e\x69", "\x00\x00", "\x00\x00",
  "\x8e\x6c", "\x8e\x6d", "\x00\x00", "\x8e\x6f", "\x8e\x70", "\x8e\x71",
  "\x8e\x79", "\x8e\x7a", "\x8e\x7b", "\x8e\x82", "\x8e\x83", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x8e\x89", "\x8e\x90",
  "\x00\x00", "\x8e\x92", "\x00\x00", "\x00\x00", "\x8e\x95", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x8e\x9a", "\x8e\x9b", "\x00\x00",
  "\x8e\x9d", "\x8e\x9e", "\x00\x00", "\x00\x00", "\x00\x00", "\x8e\xa2",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x8e\xa7", "\x00\x00",
  "\x8e\xa9", "\x00\x00", "\x00\x00", "\x00\x00", "\x8e\xad", "\x8e\xae",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x8e\xb3", "\x00\x00",
  "\x8e\xb5", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x8e\xba",
  "\x8e\xbb", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x8e\xc0",
  "\x8e\xc1", "\x00\x00", "\x8e\xc3", "\x8e\xc4", "\x00\x00", "\x00\x00",
  "\x8e\xc7", "\x8e\xcf", "\x00\x00", "\x8e\xd1", "\x00\x00", "\x00\x00",
  "\x8e\xd4", "\x8e\xdc", "\x8e\xe8", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x8e\xed", "\x8e\xee", "\x00\x00", "\x8e\xf0", "\x8e\xf1",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x8e\xf7",
  "\x00\x00", "\x8e\xf9", "\x8e\xfa", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x8f\x00", "\x00\x00", "\x8f\x02", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x8f\x07", "\x8f\x08", "\x8f\x0f",
  "\x8f\x10", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x8f\x16", "\x8f\x17", "\x8f\x18", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x8f\x1e", "\x00\x00", "\x8f\x20", "\x8f\x21",
  "\x00\x00", "\x8f\x23", "\x00\x00", "\x8f\x25", "\x00\x00", "\x8f\x27",
  "\x8f\x28", "\x00\x00", "\x00\x00", "\x00\x00", "\x8f\x2c", "\x8f\x2d",
  "\x8f\x2e", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x8f\x34", "\x8f\x35", "\x8f\x36", "\x8f\x37", "\x00\x00", "\x00\x00",
  "\x8f\x3a", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x8f\x40", "\x8f\x41", "\x00\x00", "\x8f\x43", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x8f\x47", "\x8f\x4f", "\x00\x00", "\x8f\x51", "\x8f\x52",
  "\x8f\x53", "\x8f\x54", "\x8f\x55", "\x00\x00", "\x00\x00", "\x8f\x58",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x8f\x5d", "\x8f\x5e",
  "\x8f\x65", "\x8f\x9d", "\x00\x00", "\x00\x00", "\x8f\xa0", "\x8f\xa1",
  "\x00\x00", "\x00\x00", "\x8f\xa4", "\x8f\xa5", "\x8f\xa6", "\x8f\xb5",
  "\x8f\xb6", "\x00\x00", "\x8f\xb8", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x8f\xbe", "\x00\x00", "\x8f\xc0", "\x8f\xc1",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x8f\xc6", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x8f\xca", "\x8f\xcb", "\x00\x00", "\x8f\xcd",
  "\x00\x00", "\x00\x00", "\x8f\xd0", "\x00\x00", "\x8f\xd2", "\x8f\xd3",
  "\x00\x00", "\x8f\xd5", "\x8f\xe0", "\x00\x00", "\x00\x00", "\x8f\xe3",
  "\x8f\xe4", "\x00\x00", "\x00\x00", "\x00\x00", "\x8f\xe8", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x8f\xee", "\x00\x00",
  "\x00\x00", "\x8f\xf1", "\x00\x00", "\x00\x00", "\x00\x00", "\x8f\xf5",
  "\x8f\xf6", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x8f\xfb",
  "\x00\x00", "\x00\x00", "\x8f\xfe", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x90\x02", "\x00\x00", "\x90\x04", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x90\x08", "\x00\x00", "\x00\x00", "\x00\x00", "\x90\x0c", "\x90\x18",
  "\x00\x00", "\x00\x00", "\x90\x1b", "\x90\x28", "\x90\x29", "\x90\x2a",
  "\x00\x00", "\x90\x2c", "\x90\x2d", "\x00\x00", "\x90\x2f", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x90\x33", "\x90\x34", "\x00\x00", "\x00\x00",
  "\x90\x37", "\x90\x3f", "\x00\x00", "\x00\x00", "\x00\x00", "\x90\x43",
  "\x90\x44", "\x90\x4c", "\x90\x5b", "\x00\x00", "\x90\x5d", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x90\x62", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x90\x66", "\x90\x67", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x90\x6c", "\x00\x00", "\x00\x00", "\x00\x00", "\x90\x70",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x90\x74", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x90\x79", "\x90\x85", "\x00\x00", "\x00\x00",
  "\x90\x88", "\x00\x00", "\x00\x00", "\x90\x8b", "\x90\x8c", "\x00\x00",
  "\x90\x8e", "\x00\x00", "\x90\x90", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x90\x95", "\x00\x00", "\x90\x97", "\x90\x98", "\x90\x99",
  "\x00\x00", "\x90\x9b", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x90\xa0", "\x90\xa1", "\x90\xa2", "\x00\x00", "\x00\x00", "\x90\xa5",
  "\x90\xb0", "\x00\x00", "\x90\xb2", "\x90\xb3", "\x90\xb4", "\x00\x00",
  "\x90\xb6", "\x90\xbd", "\x90\xbe", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x90\xc3", "\x90\xc4", "\x90\xc5", "\x00\x00", "\x90\xc7",
  "\x90\xc8", "\x00\x00", "\x00\x00", "\x00\x00", "\x90\xcc", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x90\xd2", "\x00\x00",
  "\x00\x00", "\x90\xd5", "\x00\x00", "\x90\xd7", "\x90\xd8", "\x90\xd9",
  "\x00\x00", "\x00\x00", "\x90\xdc", "\x90\xdd", "\x00\x00", "\x90\xdf",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x90\xe5",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x90\xeb",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x90\xef", "\x90\xf0", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x90\xf4", "\x00\x00", "\x90\xf6", "\x90\xfe",
  "\x90\xff", "\x91\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x91\x04",
  "\x91\x05", "\x91\x06", "\x00\x00", "\x91\x08", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x91\x0d", "\x00\x00", "\x00\x00", "\x91\x10",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x91\x14", "\x00\x00", "\x91\x16",
  "\x91\x17", "\x91\x18", "\x00\x00", "\x91\x1a", "\x00\x00", "\x91\x1c",
  "\x00\x00", "\x91\x1e", "\x00\x00", "\x91\x20", "\x00\x00", "\x91\x22",
  "\x91\x23", "\x00\x00", "\x91\x25", "\x00\x00", "\x91\x27", "\x00\x00",
  "\x91\x29", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x91\x2e",
  "\x91\x2f", "\x00\x00", "\x91\x31", "\x00\x00", "\x00\x00", "\x91\x34",
  "\x00\x00", "\x91\x36", "\x91\x37", "\x00\x00", "\x91\x39", "\x91\x3a",
  "\x00\x00", "\x91\x3c", "\x91\x3d", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x91\x43", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x91\x47", "\x91\x48", "\x91\x4f", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x91\x53", "\x00\x00", "\x00\x00", "\x00\x00", "\x91\x57", "\x00\x00",
  "\x91\x59", "\x91\x5a", "\x91\x5b", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x91\x61", "\x00\x00", "\x00\x00", "\x91\x64",
  "\x00\x00", "\x00\x00", "\x91\x67", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x91\x6d", "\x91\x74", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x91\x79", "\x91\x7a", "\x91\x7b", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x91\x81", "\x00\x00",
  "\x91\x83", "\x00\x00", "\x91\x85", "\x91\x86", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x91\x8a", "\x00\x00", "\x00\x00", "\x00\x00", "\x91\x8e",
  "\x00\x00", "\x00\x00", "\x91\x91", "\x00\x00", "\x91\x93", "\x91\x94",
  "\x91\x95", "\x00\x00", "\x00\x00", "\x91\x98", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x91\x9e", "\x00\x00", "\x00\x00",
  "\x91\xa1", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x91\xa6",
  "\x00\x00", "\x91\xa8", "\x00\x00", "\x00\x00", "\x00\x00", "\x91\xac",
  "\x91\xad", "\x91\xae", "\x00\x00", "\x91\xb0", "\x91\xb1", "\x91\xb2",
  "\x91\xb3", "\x00\x00", "\x00\x00", "\x91\xb6", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x91\xbb", "\x91\xbc", "\x91\xbd", "\x00\x00",
  "\x91\xbf", "\x00\x00", "\x00\x00", "\x91\xc2", "\x91\xc3", "\x00\x00",
  "\x91\xc5", "\x91\xd3", "\x91\xd4", "\x00\x00", "\x00\x00", "\x91\xd7",
  "\x00\x00", "\x91\xd9", "\x91\xda", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x91\xde", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x91\xe4", "\x91\xe5", "\x00\x00", "\x00\x00", "\x00\x00", "\x91\xe9",
  "\x91\xea", "\x00\x00", "\x91\xec", "\x91\xed", "\x91\xee", "\x91\xef",
  "\x91\xf0", "\x91\xf1", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x91\xf7", "\x00\x00", "\x91\xf9", "\x00\x00", "\x91\xfb",
  "\x00\x00", "\x91\xfd", "\x00\x00", "\x00\x00", "\x92\x00", "\x92\x01",
  "\x00\x00", "\x00\x00", "\x92\x04", "\x92\x05", "\x92\x06", "\x92\x07",
  "\x00\x00", "\x92\x09", "\x92\x0a", "\x00\x00", "\x92\x0c", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x92\x10", "\x00\x00", "\x92\x12", "\x92\x13",
  "\x00\x00", "\x00\x00", "\x92\x16", "\x00\x00", "\x92\x18", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x92\x1c", "\x92\x1d", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x92\x23", "\x92\x24", "\x92\x25",
  "\x92\x26", "\x00\x00", "\x92\x28", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x92\x2e", "\x92\x2f", "\x92\x30", "\x00\x00",
  "\x00\x00", "\x92\x33", "\x00\x00", "\x92\x35", "\x92\x36", "\x00\x00",
  "\x92\x38", "\x92\x39", "\x92\x3a", "\x00\x00", "\x92\x3c", "\x00\x00",
  "\x92\x3e", "\x00\x00", "\x92\x40", "\x00\x00", "\x92\x42", "\x92\x43",
  "\x00\x00", "\x00\x00", "\x92\x46", "\x92\x47", "\x00\x00", "\x00\x00",
  "\x92\x4a", "\x00\x00", "\x00\x00", "\x92\x4d", "\x92\x4e", "\x92\x4f",
  "\x00\x00", "\x92\x51", "\x92\x58", "\x92\x59", "\x00\x00", "\x00\x00",
  "\x92\x5c", "\x92\x5d", "\x00\x00", "\x00\x00", "\x92\x60", "\x92\x61",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x92\x65", "\x00\x00", "\x92\x67",
  "\x92\x68", "\x92\x69", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x92\x6e", "\x92\x6f", "\x92\x70", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x92\x75", "\x92\x76", "\x92\x77", "\x92\x78", "\x92\x79",
  "\x00\x00", "\x92\x7b", "\x92\x7c", "\x92\x7d", "\x00\x00", "\x92\x7f",
  "\x92\x88", "\x92\x89", "\x92\x8a", "\x00\x00", "\x00\x00", "\x92\x8d",
  "\x92\x8e", "\x00\x00", "\x00\x00", "\x00\x00", "\x92\x92", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x92\x97", "\x00\x00", "\x92\x99",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x92\x9f",
  "\x92\xa0", "\x00\x00", "\x00\x00", "\x00\x00", "\x92\xa4", "\x92\xa5",
  "\x00\x00", "\x92\xa7", "\x92\xa8", "\x00\x00", "\x00\x00", "\x92\xab",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x92\xaf", "\x00\x00", "\x00\x00",
  "\x92\xb2", "\x00\x00", "\x00\x00", "\x00\x00", "\x92\xb6", "\x00\x00",
  "\x92\xb8", "\x00\x00", "\x92\xba", "\x92\xbb", "\x92\xbc", "\x92\xbd",
  "\x00\x00", "\x92\xbf", "\x92\xc0", "\x92\xc1", "\x92\xc2", "\x92\xc3",
  "\x00\x00", "\x92\xc5", "\x92\xc6", "\x92\xc7", "\x92\xc8", "\x00\x00",
  "\x00\x00", "\x92\xcb", "\x92\xcc", "\x92\xcd", "\x92\xce", "\x00\x00",
  "\x92\xd0", "\x00\x00", "\x00\x00", "\x92\xd3", "\x00\x00", "\x92\xd5",
  "\x00\x00", "\x92\xd7", "\x92\xd8", "\x92\xd9", "\x00\x00", "\x00\x00",
  "\x92\xdc", "\x92\xdd", "\x00\x00", "\x92\xdf", "\x92\xe0", "\x92\xe1",
  "\x00\x00", "\x92\xe3", "\x00\x00", "\x92\xe5", "\x00\x00", "\x92\xe7",
  "\x92\xe8", "\x00\x00", "\x00\x00", "\x00\x00", "\x92\xec", "\x00\x00",
  "\x92\xee", "\x00\x00", "\x92\xf0", "\x92\xf9", "\x00\x00", "\x92\xfb",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x92\xff", "\x93\x00", "\x00\x00",
  "\x93\x02", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x93\x08", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x93\x0d",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x93\x11", "\x00\x00", "\x00\x00",
  "\x93\x14", "\x93\x15", "\x93\x1c", "\x93\x1d", "\x93\x1e", "\x93\x1f",
  "\x00\x00", "\x93\x21", "\x00\x00", "\x00\x00", "\x93\x24", "\x93\x25",
  "\x00\x00", "\x93\x27", "\x00\x00", "\x93\x29", "\x93\x2a", "\x93\x33",
  "\x93\x34", "\x00\x00", "\x93\x36", "\x93\x37", "\x93\x47", "\x93\x48",
  "\x93\x49", "\x93\x50", "\x93\x51", "\x93\x52", "\x00\x00", "\x00\x00",
  "\x93\x55", "\x00\x00", "\x93\x57", "\x93\x58", "\x00\x00", "\x93\x5a",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x93\x5e", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x93\x64", "\x93\x65", "\x00\x00",
  "\x93\x67", "\x00\x00", "\x93\x69", "\x93\x6a", "\x00\x00", "\x00\x00",
  "\x93\x6d", "\x00\x00", "\x93\x6f", "\x93\x70", "\x93\x71", "\x00\x00",
  "\x93\x73", "\x93\x74", "\x00\x00", "\x93\x76", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x93\x7a", "\x00\x00", "\x00\x00", "\x93\x7d", "\x00\x00",
  "\x93\x7f", "\x93\x80", "\x93\x81", "\x93\x82", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x93\x88", "\x00\x00", "\x93\x8a",
  "\x93\x8b", "\x00\x00", "\x93\x8d", "\x00\x00", "\x93\x8f", "\x00\x00",
  "\x00\x00", "\x93\x92", "\x00\x00", "\x00\x00", "\x93\x95", "\x00\x00",
  "\x00\x00", "\x93\x98", "\x00\x00", "\x00\x00", "\x93\x9b", "\x00\x00",
  "\x00\x00", "\x93\x9e", "\x00\x00", "\x00\x00", "\x93\xa1", "\x00\x00",
  "\x93\xa3", "\x93\xa4", "\x00\x00", "\x93\xa6", "\x00\x00", "\x93\xa8",
  "\x93\xa9", "\x00\x00", "\x93\xab", "\x93\xb4", "\x93\xb5", "\x93\xb6",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x93\xba", "\x93\xc1", "\x00\x00",
  "\x00\x00", "\x93\xc4", "\x93\xc5", "\x93\xc6", "\x93\xc7", "\x00\x00",
  "\x93\xc9", "\x93\xca", "\x93\xcb", "\x93\xcc", "\x93\xcd", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x93\xd3", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x93\xd9", "\x00\x00",
  "\x00\x00", "\x93\xdc", "\x00\x00", "\x93\xde", "\x93\xdf", "\x00\x00",
  "\x00\x00", "\x93\xe2", "\x00\x00", "\x00\x00", "\x00\x00", "\x93\xe6",
  "\x93\xe7", "\x93\xf7", "\x93\xf8", "\x93\xf9", "\x93\xfa", "\x93\xfb",
  "\x00\x00", "\x93\xfd", "\x00\x00", "\x00\x00", "\x00\x00", "\x94\x01",
  "\x94\x02", "\x00\x00", "\x94\x04", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x94\x08", "\x94\x09", "\x00\x00", "\x00\x00", "\x00\x00", "\x94\x0d",
  "\x94\x0e", "\x94\x0f", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x94\x15", "\x94\x16", "\x94\x17", "\x94\x1f", "\x94\x2e",
  "\x94\x2f", "\x00\x00", "\x94\x31", "\x94\x32", "\x94\x33", "\x94\x34",
  "\x94\x3b", "\x00\x00", "\x94\x3d", "\x00\x00", "\x94\x3f", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x94\x43", "\x00\x00", "\x94\x45", "\x00\x00",
  "\x00\x00", "\x94\x48", "\x00\x00", "\x94\x4a", "\x00\x00", "\x94\x4c",
  "\x94\x55", "\x00\x00", "\x00\x00", "\x00\x00", "\x94\x59", "\x00\x00",
  "\x00\x00", "\x94\x5c", "\x00\x00", "\x00\x00", "\x94\x5f", "\x00\x00",
  "\x94\x61", "\x00\x00", "\x94\x63", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x94\x68", "\x00\x00", "\x00\x00", "\x94\x6b", "\x00\x00",
  "\x94\x6d", "\x94\x6e", "\x94\x6f", "\x00\x00", "\x94\x71", "\x94\x72",
  "\x94\x83", "\x94\x84", "\x95\x78", "\x95\x79", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x95\x7e", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x95\x84", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x95\x88", "\x00\x00", "\x00\x00", "\x00\x00", "\x95\x8c", "\x95\x8d",
  "\x95\x8e", "\x95\x9d", "\x95\x9e", "\x95\x9f", "\x00\x00", "\x95\xa1",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x95\xa6", "\x00\x00",
  "\x00\x00", "\x95\xa9", "\x00\x00", "\x95\xab", "\x95\xac", "\x95\xb4",
  "\x00\x00", "\x95\xb6", "\x00\x00", "\x00\x00", "\x00\x00", "\x95\xba",
  "\x00\x00", "\x00\x00", "\x95\xbd", "\x00\x00", "\x95\xbf", "\x95\xc6",
  "\x00\x00", "\x95\xc8", "\x95\xc9", "\x00\x00", "\x95\xcb", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x95\xd0", "\x95\xd1", "\x95\xd2",
  "\x95\xd3", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x95\xd9", "\x95\xda", "\x00\x00", "\x00\x00", "\x95\xdd", "\x95\xde",
  "\x95\xdf", "\x95\xe0", "\x00\x00", "\x00\x00", "\x00\x00", "\x95\xe4",
  "\x00\x00", "\x95\xe6", "\x96\x1d", "\x96\x1e", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x96\x22", "\x00\x00", "\x96\x24", "\x96\x25", "\x96\x26",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x96\x2c",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x96\x31", "\x00\x00",
  "\x96\x33", "\x00\x00", "\x00\x00", "\x00\x00", "\x96\x37", "\x96\x38",
  "\x96\x39", "\x96\x3a", "\x00\x00", "\x96\x3c", "\x96\x3d", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x96\x41", "\x96\x52", "\x00\x00", "\x96\x54",
  "\x00\x00", "\x96\x56", "\x96\x57", "\x96\x58", "\x96\x61", "\x96\x6e",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x96\x74",
  "\x96\x7b", "\x96\x7c", "\x00\x00", "\x96\x7e", "\x96\x7f", "\x00\x00",
  "\x96\x81", "\x96\x82", "\x96\x83", "\x96\x84", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x96\x89", "\x96\x91", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x96\x96", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x96\x9a", "\x00\x00", "\x00\x00", "\x96\x9d", "\x00\x00", "\x96\x9f",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x96\xa4", "\x96\xa5",
  "\x96\xa6", "\x00\x00", "\x00\x00", "\x96\xa9", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x96\xae", "\x96\xaf", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x96\xb3", "\x96\xba", "\x96\xca", "\x96\xd2", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x96\xd8", "\x00\x00",
  "\x96\xda", "\x00\x00", "\x00\x00", "\x96\xdd", "\x96\xde", "\x96\xdf",
  "\x96\xe9", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x96\xef", "\x00\x00", "\x96\xf1", "\x96\xfa", "\x97\x02", "\x97\x03",
  "\x00\x00", "\x97\x05", "\x00\x00", "\x00\x00", "\x00\x00", "\x97\x09",
  "\x97\x1a", "\x97\x1b", "\x00\x00", "\x97\x1d", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x97\x21", "\x97\x22", "\x97\x23", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x97\x28", "\x97\x31", "\x00\x00", "\x97\x33",
  "\x97\x41", "\x00\x00", "\x97\x43", "\x97\x4a", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x97\x4e", "\x97\x4f", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x97\x55", "\x00\x00", "\x97\x57", "\x97\x58",
  "\x00\x00", "\x97\x5a", "\x97\x5b", "\x97\x63", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x97\x67", "\x00\x00", "\x00\x00", "\x97\x6a", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x97\x6e", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x97\x73", "\x00\x00", "\x00\x00", "\x97\x76", "\x97\x77",
  "\x97\x78", "\x00\x00", "\x00\x00", "\x97\x7b", "\x00\x00", "\x97\x7d",
  "\x00\x00", "\x97\x7f", "\x97\x80", "\x97\x89", "\x97\x95", "\x97\x96",
  "\x97\x97", "\x00\x00", "\x97\x99", "\x97\x9a", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x97\x9e", "\x97\x9f", "\x00\x00", "\x00\x00", "\x97\xa2",
  "\x97\xac", "\x00\x00", "\x97\xae", "\x00\x00", "\x00\x00", "\x97\xb1",
  "\x97\xb2", "\x00\x00", "\x00\x00", "\x97\xb5", "\x97\xb6", "\x00\x00",
  "\x97\xb8", "\x97\xb9", "\x97\xba", "\x00\x00", "\x97\xbc", "\x00\x00",
  "\x97\xbe", "\x97\xbf", "\x00\x00", "\x97\xc1", "\x00\x00", "\x00\x00",
  "\x97\xc4", "\x97\xc5", "\x00\x00", "\x97\xc7", "\x00\x00", "\x97\xc9",
  "\x97\xca", "\x00\x00", "\x97\xcc", "\x97\xcd", "\x97\xce", "\x00\x00",
  "\x97\xd0", "\x97\xd1", "\x00\x00", "\x00\x00", "\x97\xd4", "\x00\x00",
  "\x00\x00", "\x97\xd7", "\x97\xd8", "\x97\xd9", "\x00\x00", "\x97\xdb",
  "\x00\x00", "\x97\xdd", "\x97\xde", "\x00\x00", "\x97\xe0", "\x97\xe1",
  "\x00\x00", "\x00\x00", "\x97\xe4", "\x97\xef", "\x00\x00", "\x97\xf1",
  "\x00\x00", "\x00\x00", "\x97\xf4", "\x00\x00", "\x00\x00", "\x97\xf7",
  "\x97\xf8", "\x00\x00", "\x97\xfa", "\x98\x07", "\x00\x00", "\x00\x00",
  "\x98\x0a", "\x00\x00", "\x00\x00", "\x98\x0d", "\x98\x0e", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x98\x14", "\x00\x00",
  "\x98\x16", "\x00\x00", "\x00\x00", "\x98\x19", "\x00\x00", "\x00\x00",
  "\x98\x1c", "\x00\x00", "\x98\x1e", "\x00\x00", "\x98\x20", "\x00\x00",
  "\x00\x00", "\x98\x23", "\x00\x00", "\x98\x25", "\x98\x26", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x98\x2b", "\x00\x00", "\x00\x00",
  "\x98\x2e", "\x98\x2f", "\x98\x30", "\x00\x00", "\x98\x32", "\x98\x33",
  "\x00\x00", "\x98\x35", "\x98\x3e", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x98\x44", "\x00\x00", "\x00\x00", "\x98\x47",
  "\x00\x00", "\x00\x00", "\x98\x4a", "\x98\x51", "\x98\x52", "\x98\x53",
  "\x00\x00", "\x00\x00", "\x98\x56", "\x98\x57", "\x00\x00", "\x98\x59",
  "\x98\x5a", "\x98\x62", "\x98\x63", "\x00\x00", "\x98\x65", "\x98\x66",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x98\x6a", "\x00\x00", "\x98\x6c",
  "\x98\xab", "\x00\x00", "\x98\xad", "\x98\xae", "\x00\x00", "\x98\xb0",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x98\xb4", "\x00\x00", "\x00\x00",
  "\x98\xb7", "\x98\xb8", "\x00\x00", "\x98\xba", "\x98\xbb", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x98\xbf", "\x00\x00", "\x00\x00", "\x98\xc2",
  "\x00\x00", "\x00\x00", "\x98\xc5", "\x00\x00", "\x00\x00", "\x98\xc8",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x98\xcc", "\x98\xe1", "\x00\x00",
  "\x98\xe3", "\x00\x00", "\x98\xe5", "\x98\xe6", "\x98\xe7", "\x00\x00",
  "\x00\x00", "\x98\xea", "\x98\xf3", "\x00\x00", "\x00\x00", "\x98\xf6",
  "\x99\x02", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x99\x07",
  "\x99\x08", "\x99\x11", "\x00\x00", "\x00\x00", "\x00\x00", "\x99\x15",
  "\x99\x16", "\x99\x17", "\x00\x00", "\x00\x00", "\x99\x1a", "\x99\x1b",
  "\x99\x1c", "\x00\x00", "\x00\x00", "\x99\x1f", "\x00\x00", "\x00\x00",
  "\x99\x22", "\x00\x00", "\x00\x00", "\x00\x00", "\x99\x26", "\x99\x27",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x99\x2b", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x99\x31", "\x99\x32", "\x99\x33",
  "\x99\x34", "\x99\x35", "\x00\x00", "\x00\x00", "\x00\x00", "\x99\x39",
  "\x99\x3a", "\x99\x3b", "\x99\x3c", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x99\x40", "\x99\x41", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x99\x46", "\x99\x47", "\x99\x48", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x99\x4d", "\x99\x4e", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x99\x54", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x99\x58", "\x99\x59", "\x00\x00", "\x99\x5b", "\x99\x5c", "\x00\x00",
  "\x99\x5e", "\x99\x5f", "\x99\x60", "\x99\x9b", "\x00\x00", "\x99\x9d",
  "\x00\x00", "\x99\x9f", "\x99\xa6", "\x99\xb0", "\x99\xb1", "\x99\xb2",
  "\x00\x00", "\x00\x00", "\x99\xb5", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x99\xb9", "\x99\xba", "\x00\x00", "\x00\x00", "\x99\xbd", "\x00\x00",
  "\x99\xbf", "\x00\x00", "\x00\x00", "\x00\x00", "\x99\xc3", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x99\xc9", "\x99\xd3",
  "\x99\xd4", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x99\xd9",
  "\x99\xda", "\x00\x00", "\x99\xdc", "\x00\x00", "\x99\xde", "\x99\xe7",
  "\x00\x00", "\x00\x00", "\x99\xea", "\x99\xeb", "\x99\xec", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x99\xf0", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x99\xf4", "\x99\xf5", "\x00\x00", "\x00\x00", "\x00\x00", "\x99\xf9",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x99\xfd", "\x99\xfe", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x9a\x02", "\x9a\x03", "\x9a\x04", "\x9a\x0b",
  "\x9a\x0c", "\x00\x00", "\x00\x00", "\x00\x00", "\x9a\x10", "\x9a\x11",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x9a\x16", "\x9a\x1e",
  "\x00\x00", "\x9a\x20", "\x00\x00", "\x9a\x22", "\x9a\x23", "\x9a\x24",
  "\x00\x00", "\x00\x00", "\x9a\x27", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x9a\x2d", "\x9a\x2e", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x9a\x33", "\x00\x00", "\x9a\x35", "\x9a\x36",
  "\x00\x00", "\x9a\x38", "\x9a\x41", "\x00\x00", "\x00\x00", "\x9a\x44",
  "\x00\x00", "\x00\x00", "\x9a\x47", "\x00\x00", "\x00\x00", "\x9a\x4a",
  "\x9a\x4b", "\x9a\x4c", "\x00\x00", "\x9a\x4e", "\x00\x00", "\x00\x00",
  "\x9a\x51", "\x00\x00", "\x00\x00", "\x9a\x54", "\x00\x00", "\x9a\x56",
  "\x9a\x5d", "\x9a\xaa", "\x00\x00", "\x9a\xac", "\x00\x00", "\x9a\xae",
  "\x9a\xaf", "\x00\x00", "\x00\x00", "\x9a\xb2", "\x00\x00", "\x9a\xb4",
  "\x9a\xb5", "\x9a\xb6", "\x00\x00", "\x00\x00", "\x9a\xb9", "\x00\x00",
  "\x9a\xbb", "\x00\x00", "\x00\x00", "\x9a\xbe", "\x9a\xbf", "\x00\x00",
  "\x9a\xc1", "\x00\x00", "\x9a\xc3", "\x00\x00", "\x00\x00", "\x9a\xc6",
  "\x00\x00", "\x9a\xc8", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x9a\xce", "\x00\x00", "\x9a\xd0", "\x00\x00", "\x9a\xd2",
  "\x00\x00", "\x00\x00", "\x9a\xd5", "\x9a\xd6", "\x9a\xd7", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x9a\xdb", "\x9a\xdc", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x9a\xe0", "\x00\x00", "\x00\x00", "\x00\x00", "\x9a\xe4",
  "\x9a\xe5", "\x00\x00", "\x9a\xe7", "\x00\x00", "\x9a\xe9", "\x00\x00",
  "\x00\x00", "\x9a\xec", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x9a\xf2", "\x9a\xf3", "\x00\x00", "\x9a\xf5", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x9a\xf9", "\x9a\xfa", "\x00\x00", "\x00\x00",
  "\x9a\xfd", "\x00\x00", "\x9a\xff", "\x9b\x00", "\x9b\x01", "\x9b\x02",
  "\x9b\x03", "\x9b\x04", "\x9b\x05", "\x00\x00", "\x00\x00", "\x9b\x08",
  "\x9b\x09", "\x00\x00", "\x9b\x0b", "\x9b\x0c", "\x9b\x0d", "\x9b\x0e",
  "\x00\x00", "\x9b\x10", "\x00\x00", "\x9b\x12", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x9b\x16", "\x00\x00", "\x00\x00", "\x9b\x19", "\x00\x00",
  "\x9b\x1b", "\x9b\x1c", "\x00\x00", "\x00\x00", "\x00\x00", "\x9b\x20",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x9b\x26",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x9b\x2b", "\x00\x00",
  "\x9b\x2d", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x9b\x33", "\x9b\x34", "\x9b\x35", "\x00\x00", "\x9b\x37", "\x00\x00",
  "\x9b\x39", "\x9b\x3a", "\x00\x00", "\x00\x00", "\x9b\x3d", "\x9b\x48",
  "\x00\x00", "\x00\x00", "\x9b\x4b", "\x9b\x4c", "\x9b\x55", "\x9b\x56",
  "\x9b\x57", "\x00\x00", "\x00\x00", "\x00\x00", "\x9b\x5b", "\x00\x00",
  "\x00\x00", "\x9b\x5e", "\x00\x00", "\x00\x00", "\x9b\x61", "\x00\x00",
  "\x9b\x63", "\x00\x00", "\x9b\x65", "\x9b\x66", "\x00\x00", "\x9b\x68",
  "\x00\x00", "\x9b\x6a", "\x9b\x6b", "\x9b\x6c", "\x9b\x6d", "\x9b\x6e",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x9b\x73", "\x00\x00",
  "\x9b\x75", "\x00\x00", "\x9b\x77", "\x9b\x78", "\x9b\x79", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x9b\x7f", "\x9b\x80",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x9b\x84", "\x9b\x85", "\x9b\x86",
  "\x9b\x87", "\x00\x00", "\x9b\x89", "\x9b\x8a", "\x9b\x8b", "\x00\x00",
  "\x9b\x8d", "\x00\x00", "\x9b\x8f", "\x9b\x90", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x9b\x94", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x9b\x9a", "\x00\x00", "\x00\x00", "\x9b\x9d", "\x9b\x9e",
  "\x9b\xa6", "\x9b\xa7", "\x00\x00", "\x9b\xa9", "\x00\x00", "\x00\x00",
  "\x9b\xac", "\x00\x00", "\x00\x00", "\x00\x00", "\x9b\xb0", "\x9b\xb1",
  "\x9b\xb2", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x9b\xb7",
  "\x9b\xb8", "\x00\x00", "\x00\x00", "\x9b\xbb", "\x9b\xbc", "\x00\x00",
  "\x9b\xbe", "\x9b\xbf", "\x00\x00", "\x9b\xc1", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x9b\xc7", "\x9b\xc8", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x9b\xce", "\x00\x00",
  "\x9b\xd0", "\x9b\xd7", "\x9b\xd8", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x9b\xdd", "\x00\x00", "\x9b\xdf", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x9b\xe5", "\x00\x00", "\x9b\xe7",
  "\x00\x00", "\x00\x00", "\x9b\xea", "\x9b\xeb", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x9b\xef", "\x00\x00", "\x00\x00", "\x00\x00", "\x9b\xf3",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x9b\xf7", "\x9b\xf8", "\x9b\xf9",
  "\x9b\xfa", "\x00\x00", "\x00\x00", "\x9b\xfd", "\x00\x00", "\x9b\xff",
  "\x9c\x00", "\x00\x00", "\x9c\x02", "\x9c\x0b", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x9c\x0f", "\x00\x00", "\x9c\x11", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x9c\x16", "\x00\x00", "\x9c\x18", "\x9c\x19",
  "\x9c\x1a", "\x00\x00", "\x9c\x1c", "\x00\x00", "\x9c\x1e", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x9c\x22", "\x9c\x23", "\x00\x00", "\x00\x00",
  "\x9c\x26", "\x9c\x27", "\x9c\x28", "\x9c\x29", "\x9c\x2a", "\x9c\x31",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x9c\x35", "\x9c\x36", "\x9c\x37",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x9c\x3d",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x9c\x41", "\x00\x00", "\x9c\x43",
  "\x9c\x44", "\x9c\x45", "\x00\x00", "\x00\x00", "\x00\x00", "\x9c\x49",
  "\x9c\x4a", "\x00\x00", "\x00\x00", "\x00\x00", "\x9c\x4e", "\x9c\x4f",
  "\x9c\x50", "\x00\x00", "\x00\x00", "\x9c\x53", "\x9c\x54", "\x00\x00",
  "\x9c\x56", "\x00\x00", "\x9c\x58", "\x00\x00", "\x00\x00", "\x9c\x5b",
  "\x9c\x5c", "\x9c\x5d", "\x9c\x5e", "\x9c\x5f", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x9c\x63", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x9c\x68", "\x9c\x69", "\x9c\x6a", "\x9c\x6b", "\x00\x00", "\x00\x00",
  "\x9c\x6e", "\x00\x00", "\x9c\x70", "\x00\x00", "\x9c\x72", "\x00\x00",
  "\x00\x00", "\x9c\x75", "\x00\x00", "\x9c\x77", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x9c\x7b", "\x9c\xe6", "\x9c\xf2", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x9c\xf7", "\x00\x00", "\x9c\xf9", "\x9d\x02",
  "\x9d\x0b", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x9d\x11", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x9d\x17", "\x9d\x18", "\x00\x00", "\x00\x00", "\x00\x00", "\x9d\x1c",
  "\x9d\x1d", "\x9d\x1e", "\x9d\x2f", "\x9d\x30", "\x00\x00", "\x9d\x32",
  "\x9d\x33", "\x9d\x34", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x9d\x3a", "\x00\x00", "\x9d\x3c", "\x9d\x3d", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x9d\x42", "\x9d\x43", "\x00\x00",
  "\x9d\x45", "\x00\x00", "\x9d\x47", "\x00\x00", "\x00\x00", "\x9d\x4a",
  "\x9d\x53", "\x9d\x54", "\x9d\x5f", "\x00\x00", "\x00\x00", "\x9d\x62",
  "\x9d\x63", "\x00\x00", "\x9d\x65", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x9d\x69", "\x9d\x6a", "\x9d\x6b", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x9d\x70", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x9d\x76", "\x9d\x77", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x9d\x7b", "\x9d\x7c", "\x00\x00", "\x9d\x7e", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x9d\x83", "\x9d\x84", "\x00\x00", "\x9d\x86",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x9d\x8a", "\x00\x00", "\x00\x00",
  "\x9d\x8d", "\x9d\x8e", "\x00\x00", "\x00\x00", "\x00\x00", "\x9d\x92",
  "\x9d\x93", "\x00\x00", "\x9d\x95", "\x9d\x96", "\x9d\x97", "\x9d\x98",
  "\x9d\xa1", "\x9d\xaa", "\x00\x00", "\x9d\xac", "\x00\x00", "\x9d\xae",
  "\x00\x00", "\x00\x00", "\x9d\xb1", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x9d\xb5", "\x00\x00", "\x00\x00", "\x00\x00", "\x9d\xb9", "\x00\x00",
  "\x00\x00", "\x9d\xbc", "\x00\x00", "\x00\x00", "\x9d\xbf", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x9d\xc3", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x9d\xc7", "\x00\x00", "\x9d\xc9", "\x9d\xca", "\x9d\xd4", "\x9d\xd5",
  "\x9d\xd6", "\x9d\xd7", "\x00\x00", "\x00\x00", "\x9d\xda", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x9d\xde", "\x9d\xdf", "\x9d\xe0", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x9d\xe5", "\x00\x00", "\x9d\xe7",
  "\x00\x00", "\x9d\xe9", "\x00\x00", "\x9d\xeb", "\x00\x00", "\x00\x00",
  "\x9d\xee", "\x00\x00", "\x9d\xf0", "\x00\x00", "\x00\x00", "\x9d\xf3",
  "\x9d\xf4", "\x9d\xfe", "\x00\x00", "\x00\x00", "\x00\x00", "\x9e\x02",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x9e\x07", "\x00\x00",
  "\x00\x00", "\x9e\x0a", "\x00\x00", "\x00\x00", "\x00\x00", "\x9e\x0e",
  "\x00\x00", "\x9e\x10", "\x9e\x11", "\x9e\x12", "\x00\x00", "\x00\x00",
  "\x9e\x15", "\x9e\x16", "\x00\x00", "\x00\x00", "\x9e\x19", "\x00\x00",
  "\x00\x00", "\x9e\x1c", "\x9e\x1d", "\x9e\x7a", "\x9e\x7b", "\x9e\x7c",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x9e\x80", "\x00\x00", "\x9e\x82",
  "\x9e\x83", "\x9e\x84", "\x9e\x85", "\x00\x00", "\x9e\x87", "\x9e\x8e",
  "\x9e\x8f", "\x9e\x96", "\x00\x00", "\x9e\x98", "\x00\x00", "\x00\x00",
  "\x9e\x9b", "\x00\x00", "\x00\x00", "\x9e\x9e", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x9e\xa4", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x9e\xa8", "\x00\x00", "\x00\x00", "\x00\x00", "\x9e\xac",
  "\x00\x00", "\x9e\xae", "\x9e\xaf", "\x9e\xb0", "\x00\x00", "\x00\x00",
  "\x9e\xb3", "\x9e\xb4", "\x9e\xb5", "\x9e\xc6", "\x00\x00", "\x9e\xc8",
  "\x00\x00", "\x00\x00", "\x9e\xcb", "\x9e\xd5", "\x9e\xdf", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x9e\xe4", "\x00\x00", "\x00\x00",
  "\x9e\xe7", "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x9e\xec",
  "\x9e\xed", "\x9e\xee", "\x00\x00", "\x9e\xf0", "\x9e\xf1", "\x9e\xf2",
  "\x00\x00", "\x00\x00", "\x9e\xf5", "\x00\x00", "\x00\x00", "\x9e\xf8",
  "\x9e\xff", "\x00\x00", "\x00\x00", "\x9f\x02", "\x9f\x03", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x9f\x09", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x00\x00", "\x9f\x0f", "\x9f\x10",
  "\x9f\x11", "\x9f\x12", "\x00\x00", "\x9f\x14", "\x00\x00", "\x9f\x16",
  "\x9f\x17", "\x00\x00", "\x9f\x19", "\x9f\x1a", "\x9f\x1b", "\x00\x00",
  "\x00\x00", "\x00\x00", "\x9f\x1f", "\x00\x00", "\x00\x00", "\x9f\x22",
  "\x00\x00", "\x00\x00", "\x00\x00", "\x9f\x26", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x9f\x2a", "\x9f\x2b", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x9f\x2f", "\x00\x00", "\x9f\x31", "\x9f\x32", "\x00\x00", "\x9f\x34",
  "\x00\x00", "\x00\x00", "\x9f\x37", "\x00\x00", "\x9f\x39", "\x9f\x3a",
  "\x00\x00", "\x9f\x3c", "\x9f\x3d", "\x00\x00", "\x9f\x3f", "\x00\x00",
  "\x9f\x41", "\x00\x00", "\x9f\x43", "\x9f\x44", "\x9f\x45", "\x9f\x46",
  "\x9f\x47", "\x9f\x53", "\x00\x00", "\x9f\x55", "\x9f\x56", "\x9f\x57",
  "\x9f\x58", "\x00\x00", "\x9f\x5a", "\x00\x00", "\x00\x00", "\x9f\x5d",
  "\x9f\x5e", "\x9f\x68", "\x9f\x69", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x9f\x6d", "\x9f\x6e", "\x9f\x6f", "\x9f\x70", "\x9f\x71", "\x00\x00",
  "\x9f\x73", "\x00\x00", "\x9f\x75", "\x00\x00", "\x00\x00", "\x00\x00",
  "\x00\x00", "\x9f\x7a", "\x00\x00", "\x00\x00", "\x9f\x7d", "\x9f\x8f",
  "\x9f\x90", "\x9f\x91", "\x9f\x92", "\x00\x00", "\x9f\x94", "\x00\x00",
  "\x9f\x96", "\x9f\x97", "\x9f\x9e", "\x00\x00", "\x00\x00", "\x9f\xa1",
  "\x9f\xa2", "\x9f\xa3", "\x00\x00", "\x9f\xa5"
};
