/* 
 * tkUnixScrollbar.c --
 *
 *	This file implements the Unix specific portion of the scrollbar
 *	widget.
 *
 * Copyright (c) 1996 by Sun Microsystems, Inc.
 *
 * See the file "license.terms" for information on usage and redistribution
 * of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 *
 * SCCS: @(#) tkUnixScrlbr.c 1.8 96/12/10 20:05:07
 */
/*
 * TkSTEP modifications Copyright (c) Alfredo K. Kojima
 * - surgery performed and implanted into Tk8.0 by Steve Murray
 */

#include "tkScrollbar.h"

/*
 * Minimum slider length, in pixels (designed to make sure that the slider
 * is always easy to grab with the mouse).
 */

#define MIN_SLIDER_LENGTH	16

/*
 * Declaration of Unix specific scrollbar structure.
 */

typedef struct UnixScrollbar {
    TkScrollbar info;		/* Generic scrollbar info. */
    GC troughGC;		/* For drawing trough's stippled part. */
    GC troughBackGC;            /* For drawing trough. */
    GC copyGC;			/* Used for copying from pixmap onto screen. */
} UnixScrollbar;

/*
 * The class procedure table for the scrollbar widget.
 */

TkClassProcs tkpScrollbarProcs = { 
    NULL,			/* createProc. */
    NULL,			/* geometryProc. */
    NULL			/* modalProc. */
};


/*
 *----------------------------------------------------------------------
 *
 * TkpCreateScrollbar --
 *
 *	Allocate a new TkScrollbar structure.
 *
 * Results:
 *	Returns a newly allocated TkScrollbar structure.
 *
 * Side effects:
 *	Registers an event handler for the widget.
 *
 *----------------------------------------------------------------------
 */

TkScrollbar *
TkpCreateScrollbar(tkwin)
    Tk_Window tkwin;
{
    UnixScrollbar *scrollPtr = (UnixScrollbar *)ckalloc(sizeof(UnixScrollbar));
    scrollPtr->troughGC = None;
    scrollPtr->copyGC = None;

    Tk_CreateEventHandler(tkwin,
	    ExposureMask|StructureNotifyMask|FocusChangeMask,
	    TkScrollbarEventProc, (ClientData) scrollPtr);

    return (TkScrollbar *) scrollPtr;
}

/*
 *--------------------------------------------------------------
 *
 * TkpDisplayScrollbar --
 *
 *	This procedure redraws the contents of a scrollbar window.
 *	It is invoked as a do-when-idle handler, so it only runs
 *	when there's nothing else for the application to do.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Information appears on the screen.
 *
 *--------------------------------------------------------------
 */

void
TkpDisplayScrollbar(clientData)
    ClientData clientData;	/* Information about window. */
{
    register TkScrollbar *scrollPtr = (TkScrollbar *) clientData;
    UnixScrollbar *unixScrollPtr = (UnixScrollbar *)scrollPtr;
    register Tk_Window tkwin = scrollPtr->tkwin;
    Tk_3DBorder border;
    int relief, width, elementBorderWidth, sliderSize, delta;
    int arrowl = scrollPtr->arrowLength, arrsize;
    int buttonSize;
    Pixmap pixmap;
    int empty=0;

    if ((scrollPtr->tkwin == NULL) || !Tk_IsMapped(tkwin)) {
	goto done;
    }

    if ((scrollPtr->firstFraction <= 0.0) && (scrollPtr->lastFraction >= 1.0)) {
        empty = 1;
    }

    if (scrollPtr->vertical) {
	width = Tk_Width(tkwin) - 2*scrollPtr->inset;
    } else {
	width = Tk_Height(tkwin) - 2*scrollPtr->inset;
    }
    elementBorderWidth = scrollPtr->elementBorderWidth;
    if (elementBorderWidth < 0) {
	elementBorderWidth = scrollPtr->borderWidth;
    }

    if (scrollPtr->bump == None) {
        MakePixmaps((ClientData)scrollPtr);
    }

    /*
     * In order to avoid screen flashes, this procedure redraws
     * the scrollbar in a pixmap, then copies the pixmap to the
     * screen in a single operation.  This means that there's no
     * point in time where the on-sreen image has been cleared.
     */

    pixmap = Tk_GetPixmap(scrollPtr->display, Tk_WindowId(tkwin),
	    Tk_Width(tkwin), Tk_Height(tkwin), Tk_Depth(tkwin));
    
    sliderSize = scrollPtr->sliderLast - scrollPtr->sliderFirst;
    if (scrollPtr->highlightWidth != 0) {
	GC gc;

	if (scrollPtr->flags & GOT_FOCUS) {
	    gc = Tk_GCForColor(scrollPtr->highlightColorPtr, pixmap);
	} else {
	    gc = Tk_GCForColor(scrollPtr->highlightBgColorPtr, pixmap);
	}
	Tk_DrawFocusHighlight(tkwin, gc, scrollPtr->highlightWidth, pixmap);
    }

    XFillRectangle(scrollPtr->display, pixmap, unixScrollPtr->troughBackGC,
	scrollPtr->highlightWidth, scrollPtr->highlightWidth,
 	Tk_Width(tkwin) - 2*scrollPtr->highlightWidth,
  	Tk_Height(tkwin) - 2*scrollPtr->highlightWidth);
    
    XDrawRectangle(scrollPtr->display, pixmap, 
	Tk_3DBorderGC(scrollPtr->tkwin,scrollPtr->bgBorder,TK_3D_DARK2_GC),
 	scrollPtr->highlightWidth, scrollPtr->highlightWidth,
 	Tk_Width(tkwin) - 2*scrollPtr->highlightWidth-1,
 	Tk_Height(tkwin) - 2*scrollPtr->highlightWidth-1);

    buttonSize = arrowl+1;
    if (empty) {
	XFillRectangle(scrollPtr->display, pixmap, unixScrollPtr->troughGC,
		   scrollPtr->inset, scrollPtr->inset,
		   (unsigned) (Tk_Width(tkwin) - 2*scrollPtr->inset),
		   (unsigned) (Tk_Height(tkwin) - 2*scrollPtr->inset));
    } else {
	if (scrollPtr->vertical) {
	    XFillRectangle(scrollPtr->display, pixmap, unixScrollPtr->troughGC,
			   scrollPtr->inset, scrollPtr->inset,
			   (unsigned) (Tk_Width(tkwin) - 2*scrollPtr->inset),
			   (unsigned) (Tk_Height(tkwin) - 2*buttonSize -
				       2*scrollPtr->inset));
	} else {
	    XFillRectangle(scrollPtr->display, pixmap, unixScrollPtr->troughGC,
			   scrollPtr->inset+(arrowl+1)*2, scrollPtr->inset,
			   (unsigned) (Tk_Width(tkwin) - 2*buttonSize -
				       2*scrollPtr->inset),
			   (unsigned) (Tk_Height(tkwin) - 2*scrollPtr->inset));
	}
    }    

    /* 
     * don't draw slider/arrows if everything is visible
     */
    if (empty) {
	goto docopy;
    }    
#ifdef ENABLE_STEP    
    if (scrollPtr->noArrows) {
	goto doSlider;
    }
#endif    

    if (scrollPtr->activeField == TOP_ARROW) {
	border = scrollPtr->activeBorder;
	relief = scrollPtr->activeField == TOP_ARROW ? scrollPtr->activeRelief
		: TK_RELIEF_RAISED;
        delta = (relief == TK_RELIEF_SUNKEN) ? (scrollPtr->borderWidth)/2+1 : 0;

    } else {
	border = scrollPtr->bgBorder;
	relief = TK_RELIEF_RAISED;
        delta = 0;
    }

    arrsize = scrollPtr->arrowLength - elementBorderWidth*2-3;
    if (scrollPtr->vertical) {
	Tk_Fill3DRectangle(tkwin, pixmap, border, scrollPtr->inset,
			   Tk_Height(tkwin) - 2*buttonSize -
			   scrollPtr->inset+1, arrowl, arrowl,
			   elementBorderWidth, relief);
	DrawArrow(scrollPtr, pixmap, Tk_Width(tkwin)/2 - 1 + delta,
		  Tk_Height(tkwin) - buttonSize*2 -
		  scrollPtr->inset + buttonSize/2 + delta,
		  arrsize, arrsize, ARROW_UP);
    } else {
	Tk_Fill3DRectangle(tkwin, pixmap, border,
			   scrollPtr->inset, scrollPtr->inset,
			   arrowl, arrowl, elementBorderWidth, relief);
	DrawArrow(scrollPtr, pixmap, scrollPtr->inset + buttonSize/2 + delta-1,
		  (Tk_Height(tkwin)+1)/2 - 1 + delta,
		  arrsize, arrsize, ARROW_LEFT);
    }

    /*
     * Display the bottom or right arrow.
     */

    if (scrollPtr->activeField == BOTTOM_ARROW) {
	border = scrollPtr->activeBorder;
	relief = scrollPtr->activeField == BOTTOM_ARROW
		? scrollPtr->activeRelief : TK_RELIEF_RAISED;
        delta = (relief == TK_RELIEF_SUNKEN) ? (scrollPtr->borderWidth)/2+1 : 0;

    } else {
	border = scrollPtr->bgBorder;
	relief = TK_RELIEF_RAISED;
        delta = 0;
    }

    if (scrollPtr->vertical) {
	Tk_Fill3DRectangle(tkwin, pixmap, border, scrollPtr->inset,
	 	Tk_Height(tkwin)-buttonSize-scrollPtr->inset+1,
		arrowl, arrowl,
		elementBorderWidth, relief);
	DrawArrow(scrollPtr, pixmap, Tk_Width(tkwin)/2 - 1 + delta,
		  Tk_Height(tkwin) - buttonSize + buttonSize/2 - 
		  scrollPtr->inset+ delta, arrsize, arrsize, ARROW_DOWN);
    } else {
	Tk_Fill3DRectangle(tkwin, pixmap, border,
	 	scrollPtr->inset + buttonSize, scrollPtr->inset, 
		arrowl, arrowl, elementBorderWidth, relief);
	DrawArrow(scrollPtr, pixmap, 
		  scrollPtr->inset + buttonSize + buttonSize/2 + delta-1,
		  (Tk_Height(tkwin)+1)/2 - 1 + delta,
		  arrsize, arrsize, ARROW_RIGHT);
    }


#ifdef ENABLE_STEP
    doSlider:
#endif    

    /*
     * Display the slider.
     */

    if (scrollPtr->activeField == SLIDER) {
	border = scrollPtr->activeBorder;
	relief = TK_RELIEF_RAISED;
    } else {
	border = scrollPtr->bgBorder;
	relief = TK_RELIEF_RAISED;
    }
    if (scrollPtr->vertical) {
	Tk_Fill3DRectangle(tkwin, pixmap, border,
		scrollPtr->inset, scrollPtr->sliderFirst,
		width, sliderSize,
		elementBorderWidth, relief);
	if ((sliderSize > SB_BUMP_HEIGHT*2) && (width > SB_BUMP_WIDTH)) {
	    XCopyArea(scrollPtr->display, scrollPtr->bump, pixmap,
		unixScrollPtr->copyGC, 0, 0, SB_BUMP_WIDTH, SB_BUMP_HEIGHT,
		(Tk_Width(tkwin) - SB_BUMP_WIDTH)/2,
		scrollPtr->sliderFirst + (sliderSize - 1 - SB_BUMP_HEIGHT)/2);
	} else {
	    int bump_size;
	    
	    bump_size = width - 2*(scrollPtr->inset+4);
	    if ((sliderSize > bump_size+2) && (bump_size > 1)) {
		Tk_Fill3DRectangle(tkwin, pixmap, border,
		    (Tk_Width(tkwin) - bump_size)/2,
		    scrollPtr->sliderFirst + (sliderSize - bump_size)/2,
		    bump_size, bump_size, 2, TK_RELIEF_SUNKEN);
	    }	    
	}	
    } else {
	Tk_Fill3DRectangle(tkwin, pixmap, border,
		scrollPtr->sliderFirst, scrollPtr->inset,
		sliderSize, width,
		elementBorderWidth, relief);
	if ((sliderSize > SB_BUMP_HEIGHT*2) && (width > SB_BUMP_HEIGHT)) {
	    XCopyArea(scrollPtr->display, scrollPtr->bump, pixmap,
		unixScrollPtr->copyGC, 0, 0, SB_BUMP_WIDTH, SB_BUMP_HEIGHT,
		scrollPtr->sliderFirst + (sliderSize - SB_BUMP_HEIGHT)/2,
		(Tk_Height(tkwin) - 1 - SB_BUMP_WIDTH)/2);
	} else {
	    int bump_size;
	    
	    bump_size = width - 2*(scrollPtr->inset+4);	    
	    if ((sliderSize > bump_size+2) && (bump_size > 1)) {
		Tk_Fill3DRectangle(tkwin, pixmap, border,
		   scrollPtr->sliderFirst + (sliderSize - bump_size)/2-1, 
		   (Tk_Height(tkwin) - bump_size)/2,
		   bump_size, bump_size, 2, TK_RELIEF_SUNKEN);
	    }
	}
    }

    /*
     * Copy the information from the off-screen pixmap onto the screen,
     * then delete the pixmap.
     */

    docopy:

    XCopyArea(scrollPtr->display, pixmap, Tk_WindowId(tkwin),
	    ((UnixScrollbar*)scrollPtr)->copyGC, 0, 0,
	    (unsigned) Tk_Width(tkwin), (unsigned) Tk_Height(tkwin), 0, 0);
    Tk_FreePixmap(scrollPtr->display, pixmap);

    done:
    scrollPtr->flags &= ~REDRAW_PENDING;
}

/*
 *----------------------------------------------------------------------
 *
 * TkpComputeScrollbarGeometry --
 *
 *	After changes in a scrollbar's size or configuration, this
 *	procedure recomputes various geometry information used in
 *	displaying the scrollbar.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The scrollbar will be displayed differently.
 *
 *----------------------------------------------------------------------
 */

extern void
TkpComputeScrollbarGeometry(scrollPtr)
    register TkScrollbar *scrollPtr;	/* Scrollbar whose geometry may
					 * have changed. */
{
    int width, fieldLength, minLength;

    if (scrollPtr->highlightWidth < 0) {
	scrollPtr->highlightWidth = 0;
    }
    scrollPtr->inset = scrollPtr->highlightWidth + 2;
    width = (scrollPtr->vertical) ? Tk_Width(scrollPtr->tkwin)
	    : Tk_Height(scrollPtr->tkwin);
#ifdef ENABLE_STEP
    if (scrollPtr->noArrows) {
        scrollPtr->arrowLength = -1;
    } else
#endif
    scrollPtr->arrowLength = width - 2*scrollPtr->inset;
    fieldLength = (scrollPtr->vertical ? Tk_Height(scrollPtr->tkwin)
	    : Tk_Width(scrollPtr->tkwin))
	    - 2*(scrollPtr->arrowLength + 1 + scrollPtr->inset);

    minLength = width - 2*scrollPtr->inset;
    if (minLength < MIN_SLIDER_LENGTH)
        minLength = MIN_SLIDER_LENGTH;

    if (fieldLength < 0) {
	fieldLength = 0;
    }
    scrollPtr->sliderFirst = fieldLength*scrollPtr->firstFraction;
    scrollPtr->sliderLast = fieldLength*scrollPtr->lastFraction;

    /*
     * Adjust the slider so that some piece of it is always
     * displayed in the scrollbar and so that it has at least
     * a minimal width (so it can be grabbed with the mouse).
     */

    if (scrollPtr->sliderFirst > (fieldLength - 2*scrollPtr->borderWidth)) {
	scrollPtr->sliderFirst = fieldLength - 2*scrollPtr->borderWidth;
    }
    if (scrollPtr->sliderFirst < 0) {
	scrollPtr->sliderFirst = 0;
    }
    if (scrollPtr->sliderLast < scrollPtr->sliderFirst + minLength) {
	scrollPtr->sliderLast = scrollPtr->sliderFirst + minLength;
    }
    if (scrollPtr->sliderLast > fieldLength) {
	scrollPtr->sliderLast = fieldLength;
    }
    if (scrollPtr->vertical) {
        scrollPtr->sliderFirst += scrollPtr->inset;
        scrollPtr->sliderLast += scrollPtr->inset;
    } else {
        scrollPtr->sliderFirst += (scrollPtr->arrowLength+1)*2 + scrollPtr->inset;
        scrollPtr->sliderLast += (scrollPtr->arrowLength+1)*2 + scrollPtr->inset;
    }

    /*
     * Register the desired geometry for the window (leave enough space
     * for the two arrows plus a minimum-size slider, plus border around
     * the whole window, if any).  Then arrange for the window to be
     * redisplayed.
     */

    if (scrollPtr->vertical) {
	Tk_GeometryRequest(scrollPtr->tkwin,
		scrollPtr->width + 2*scrollPtr->inset,
		2*(scrollPtr->arrowLength + scrollPtr->borderWidth
		+ scrollPtr->inset + 1));
    } else {
	Tk_GeometryRequest(scrollPtr->tkwin,
		2*(scrollPtr->arrowLength + scrollPtr->borderWidth
		+ scrollPtr->inset + 1), scrollPtr->width + 2*scrollPtr->inset);
    }
    Tk_SetInternalBorder(scrollPtr->tkwin, scrollPtr->inset);
}

/*
 *----------------------------------------------------------------------
 *
 * TkpDestroyScrollbar --
 *
 *	Free data structures associated with the scrollbar control.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Frees the GCs associated with the scrollbar.
 *
 *----------------------------------------------------------------------
 */

void
TkpDestroyScrollbar(scrollPtr)
    TkScrollbar *scrollPtr;
{
    UnixScrollbar *unixScrollPtr = (UnixScrollbar *)scrollPtr;

    if (unixScrollPtr->troughGC != None) {
	Tk_FreeGC(scrollPtr->display, unixScrollPtr->troughGC);
    }
    if (unixScrollPtr->copyGC != None) {
	Tk_FreeGC(scrollPtr->display, unixScrollPtr->copyGC);
    }
    if (unixScrollPtr->troughBackGC != None) {
	Tk_FreeGC(scrollPtr->display, unixScrollPtr->troughBackGC); 
    }
    if (scrollPtr->bump != None) {
	Tk_FreePixmap(scrollPtr->display, scrollPtr->bump);
    }
    if (scrollPtr->stipple != None) {
	Tk_FreeBitmap(scrollPtr->display, scrollPtr->stipple);
    }    
    if (scrollPtr->darkPtr != NULL) {
	Tk_FreeColor(scrollPtr->darkPtr);
    }    
}

/*
 *----------------------------------------------------------------------
 *
 * TkpConfigureScrollbar --
 *
 *	This procedure is called after the generic code has finished
 *	processing configuration options, in order to configure
 *	platform specific options.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Configuration info may get changed.
 *
 *----------------------------------------------------------------------
 */

void
TkpConfigureScrollbar(scrollPtr)
    register TkScrollbar *scrollPtr;	/* Information about widget;  may or
					 * may not already have values for
					 * some fields. */
{
    XGCValues gcValues;
    XColor newcolor;
    GC new;
    UnixScrollbar *unixScrollPtr = (UnixScrollbar *) scrollPtr;


    if (scrollPtr->stipple==None) {
	scrollPtr->stipple = Tk_GetBitmap((Tcl_Interp *)NULL, scrollPtr->tkwin,
		Tk_GetUid("gray50"));
	if (scrollPtr->stipple == None) {
	    panic("ScrollBar couldn't allocate bitmap for trough");
	}
    }

    Tk_SetBackgroundFromBorder(scrollPtr->tkwin, scrollPtr->bgBorder);

    /* Allocate colour for trough base stipple. */

    newcolor.red = (60 * (int) scrollPtr->troughColorPtr->red)/100;
    newcolor.green =(60 * (int) scrollPtr->troughColorPtr->green)/100;
    newcolor.blue = (60 * (int) scrollPtr->troughColorPtr->blue)/100;
    
    if (scrollPtr->darkPtr != NULL) {
	Tk_FreeColor(scrollPtr->darkPtr);
    }    
    scrollPtr->darkPtr = Tk_GetColorByValue(scrollPtr->tkwin,
	&newcolor);

    gcValues.background = scrollPtr->troughColorPtr->pixel;
    gcValues.foreground = scrollPtr->darkPtr->pixel;   
    gcValues.stipple = scrollPtr->stipple;
    gcValues.fill_style = FillOpaqueStippled;

    new = Tk_GetGC(scrollPtr->tkwin, GCForeground|GCBackground|GCStipple|
	GCFillStyle, &gcValues);

    if (unixScrollPtr->troughGC != None) {
	Tk_FreeGC(scrollPtr->display, unixScrollPtr->troughGC);
    } 
    unixScrollPtr->troughGC = new;
    gcValues.foreground = scrollPtr->troughColorPtr->pixel;
    new = Tk_GetGC(scrollPtr->tkwin, GCForeground, &gcValues);
/*    if (unixScrollPtr->troughBackGC != None) {
	Tk_FreeGC(scrollPtr->display, unixScrollPtr->troughBackGC);
    }
*/
    unixScrollPtr->troughBackGC = new;
    if (unixScrollPtr->copyGC == None) {
	gcValues.graphics_exposures = False;
	unixScrollPtr->copyGC = Tk_GetGC(scrollPtr->tkwin, GCGraphicsExposures,
	    &gcValues);
    }
    if (Tk_IsMapped(scrollPtr->tkwin)&&(scrollPtr->flags & RECONFIGURE)) {
	scrollPtr->flags &= ~RECONFIGURE;
	if (scrollPtr->bump != None) {
	    Tk_FreePixmap(scrollPtr->display, scrollPtr->bump);
	}
	MakePixmaps(scrollPtr);
    }
}

/*
 *--------------------------------------------------------------
 *
 * TkpScrollbarPosition --
 *
 *	Determine the scrollbar element corresponding to a
 *	given position.
 *
 * Results:
 *	One of TOP_ARROW, TOP_GAP, etc., indicating which element
 *	of the scrollbar covers the position given by (x, y).  If
 *	(x,y) is outside the scrollbar entirely, then OUTSIDE is
 *	returned.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

int
TkpScrollbarPosition(scrollPtr, x, y)
    register TkScrollbar *scrollPtr;	/* Scrollbar widget record. */
    int x, y;				/* Coordinates within scrollPtr's
					 * window. */
{
    int length, width, tmp;

    if (scrollPtr->vertical) {
	length = Tk_Height(scrollPtr->tkwin);
	width = Tk_Width(scrollPtr->tkwin);
    } else {
	tmp = x;
	x = y;
	y = tmp;
	length = Tk_Width(scrollPtr->tkwin);
	width = Tk_Height(scrollPtr->tkwin);
    }

    if ((x < scrollPtr->inset) || (x >= (width - scrollPtr->inset))
	    || (y < scrollPtr->inset) || (y >= (length - scrollPtr->inset))) {
	return OUTSIDE;
    }

    /*
     * All of the calculations in this procedure mirror those in
     * TkpDisplayScrollbar.  Be sure to keep the two consistent.
     */

    if (scrollPtr->vertical) {	
	if (y < scrollPtr->sliderFirst) {
	    return TOP_GAP;
	}
	if (y < scrollPtr->sliderLast) {	    
	    return SLIDER;
	}
#ifdef ENABLE_STEP
	if (scrollPtr->noArrows || 
	    y < (length - ((scrollPtr->arrowLength+1)*2 + scrollPtr->inset))) {
#else	
	if (y < (length - ((scrollPtr->arrowLength+1)*2 + scrollPtr->inset))) {
#endif	    
	    return BOTTOM_GAP;
	}
#ifdef ENABLE_STEP	
	if (y < (length - (scrollPtr->arrowLength +1+ scrollPtr->inset))
	    && !scrollPtr->noArrows) {
#else
	if (y < (length - (scrollPtr->arrowLength+1+ scrollPtr->inset))) {
#endif	    
	    return TOP_ARROW;
	}	
	return BOTTOM_ARROW;
    } else {
#ifdef ENABLE_STEP
	if (!scrollPtr->noArrows && 
	    y < (scrollPtr->arrowLength + 1+scrollPtr->inset)) {
#else	
	if (y < (scrollPtr->arrowLength + 1+scrollPtr->inset)) {
#endif	    
	    return TOP_ARROW;
	}
#ifdef ENABLE_STEP
	if (!scrollPtr->noArrows &&
	    y < ((scrollPtr->arrowLength+1)*2 + scrollPtr->inset)) {
#else	    
	if (y < ((scrollPtr->arrowLength+1)*2 + scrollPtr->inset)) {
#endif	    
	    return BOTTOM_ARROW;
	}
	if (y < (scrollPtr->sliderFirst)) {
	    return TOP_GAP;
	}
	if (y < (scrollPtr->sliderLast)) {
	    return SLIDER;
	}
	return BOTTOM_GAP;
    }
}


/*
 * DrawArrow --
 *   	Draws an antialiased arrow with direction dir and size w x h. 
 *      It's coordinates are relative to the center of the arrow. 
 */
void DrawArrow(scrollPtr, drawable, x, y, w, h, dir)
    TkScrollbar   *scrollPtr;
    Drawable	drawable;
    int 	x, y;	/* position relative to middle of arrow */
    int 	w, h;
    char	dir;
{
    int i, hw;
    float d, s;
    GC 	core, border;

    core = Tk_3DBorderGC(scrollPtr->tkwin,scrollPtr->bgBorder,TK_3D_DARK2_GC);
    border = Tk_3DBorderGC(scrollPtr->tkwin,scrollPtr->bgBorder,TK_3D_DARK_GC);

    if (h<3) h=3;
    if (w<3) w=3;
    if (dir<2) { /* vertical */
	hw = w/2;
	d = ((float)w/2)/(float)h;
	y -= h/2;	
    } else {
	hw = h/2;
	d = ((float)h/2)/(float)w;
	x -= w/2;
    }    
    s=0;
    switch (dir) {
     case ARROW_DOWN:
	s=h*d+0.5;
	d=-d;
     case ARROW_UP:
	for(i = 0; i < h; i++) {
	    s += d;	
	    XDrawLine(scrollPtr->display, drawable, core,
		      (unsigned)(x-(int)(s-0.5)), y+i,
		      (unsigned)(x+(int)(s-0.5)), y+i);
	    /* anti-aliasing */	
	    if ((int)(s+0.5)!=(int)s) {
		XDrawPoint(scrollPtr->display, drawable, 
			   border, (unsigned)(x+(int)s), y+i);
		XDrawPoint(scrollPtr->display, drawable,
			   border, (unsigned)(x-(int)s), y+i);
	    }
	}
	break;
     case ARROW_RIGHT:
	s=h*d+0.5;
	d=-d;
     case ARROW_LEFT:
	for(i = 0; i < h; i++) {
	    s += d;	
	    XDrawLine(scrollPtr->display, drawable,
		      core, x+i,(unsigned)(y-(int)(s-0.5)),
		      x+i,(unsigned)(y+(int)(s-0.5)));
	    /* anti-aliasing */	
	    if ((int)(s+0.5)!=(int)s) {
		XDrawPoint(scrollPtr->display, drawable,
			   border, x+i,(unsigned)(y+(int)s));
		XDrawPoint(scrollPtr->display, drawable,
			   border, x+i,(unsigned)(y-(int)s));
	    }
	}
	break;
    }
}
