/*  VER 104  TAB P   $Id: sys.c,v 1.2 1997/08/14 13:39:46 src Exp $
 *
 *  handle the local sys or newsfeeds file
 *
 *  copyright 1996, 1997 Egil Kvaleberg, egil@kvaleberg.no
 *  the GNU General Public License applies
 */

#include "common.h"
#include "proto.h"
#include "options.h"
#include "news.h"

/* 
 *  globals for sys
 */
typedef struct sys_grp {
    struct sys_grp *next;
    char not;
    char name[1]; /* extend as required... */
} SYS_GRP;

SYS_GRP *sysgrp = 0;
SYS_GRP *sys_me = 0;

/*
 *  add group in sys-file
 *  new items are inserted at the top of the list, 
 *  so the list will appear in the opposite sequence of the original
 */
static 
    sys_group(char *group,int me,int not)
{
    int n; 
    SYS_GRP *sp;
    SYS_GRP **spp;

    n = sizeof(SYS_GRP) + strlen(group);

    if (!(sp = malloc(n))) {
	log_msg(L_ERRno,"out of memory");
	unlock_exit(1);
    }

    /* fill in structure */
    sp->next = 0; 
    sp->not = not;
    strcpy(sp->name,group);

    /* insert at top of list */
    spp = (me) ? &sys_me : &sysgrp;
    sp->next = *spp;
    *spp = sp;
}

/*
 *  for C News:
 *  check if group name matches pattern
 *  return match "worth" if it does, or 0 if not
 */
static 
sys_match(char *group,char *pattern)
{
    int n;

    if (pattern[0]=='a' && pattern[1]=='l' && pattern[2]=='l') {
	switch (pattern[3]) {
	case '\0': /* e.g. "rec.humor" with pattern "all" */
	    return 1; 
	case '.':  /* e.g. "rec.humor" with pattern "all.something" */
	    group = strchr(group,'.');
	    n = sys_match(group ? group+1 : "",pattern+4);
	    return n ? n+99 : 0;
	}
    }

    /* non-wildcard case */
    while (*group) {
	if (*pattern== '\\' ) 
	    ++pattern;
	if (*group == '.') {
	    switch (*pattern) {
	    case '\0': /* e.g. "rec.humor" with pattern "rec" */
		return 1; 
	    case '.':  /* e.g. "rec.humor" with pattern "rec.all" */
		n = sys_match(group+1,pattern+1);
		return n ? n+100 : 0;
	    default:
		/* e.g. "rec.humor" with pattern "recsomething" */
		return 0;
	    }
	}
	if (*group++ != *pattern++) return 0;
    }

    /* group exhausted */
    switch (*pattern) {
    case '\0': 
	return 100; /* exact match */
    case '.':
	/* "alt.humor" when pattern is "alt.humor.all" does not match */
    default:
	return 0; /* no match */
    }
}

/*
 *  is group allowed in sys-file?
 *
 *  Behaviour for INN, from newsfeeds(5):
 *
 *  For example,  to  receive  all
 *  ``comp''  groups,  but  only  comp.sources.unix within the
 *  sources newsgroups, the following set of patterns  can  be
 *  used:
 *    comp.*,!comp.sources.*,comp.sources.unix
 *  There  are	three  things to note about this example.  The
 *  first is that the trailing ``.*'' is required.  The second
 *  is	that,  again, the result of the last match is the most
 *  important.	The third is that ``comp.sources.*'' could  be
 *  written  as  ``comp.sources*'' but this would not have the
 *  same effect if there were a ``comp.sources-only'' group.
 */
static int 
sys_allow_grp(char *group, SYS_GRP *sp)
{
    int match;

    if (inn_opt) {
	/*
	 * INN:
	 * the last match is the most important
	 * since our version of the list is stored with the last 
	 * item first, we can get away by simply returning the 
	 * status of *our* first match.
	 */
	for (; sp; sp = sp->next) {
	    if (wildmat(group, sp->name)) {
		/* if !-flag, then not allowed, otherwise all right */
		return !(sp->not);
	    }
	}
	return 0; /* default to a no-no */
    } else {
	/*
	 * C News:
	 * the order is not significant
	 * when more than one pattern matches:
	 * if the longest matched pattern is longer than the longest
	 * mismatched pattern, it is a match. otherwise, it is not.
	 * longest is measured as number of words, "all" counting 
	 * sligthly less than other words
	 */
	int n;
	int best_n = 0;
	int result = 0;

	for (; sp; sp = sp->next) {
	    if (n=sys_match(group,sp->name)) {
		if (sp->not) {
		    if (n >= best_n) {
			best_n = n;
			result = 0;
		    }
		} else {
		    if (n > best_n) {
			best_n = n;
			result = 1;
		    }
		}
	    }
	}
	return result;
    }
}

/*
 *  is group allowed in sys-file?
 */
int 
sys_allow(char *group)
{
    if (!sys_allow_grp(group, sys_me)) {
	log_msg(L_DEBUG4,"group %s not accepted by ME",group);
	return 0;
    }
    if (!sys_allow_grp(group, sysgrp)) { 
	log_msg(L_DEBUG4,"group %s disallowed by sys",group);
	return 0;
    }
    return 1;
}

/*
 *  pick an item from a C News style sys file
 *  items are:
 *	text
 *	"!" 
 *	"/" 
 *	"," 
 *	":" 
 *	"#" 
 *	"\n"
 */
static void 
sys_item(char *item)
{
    static int state = 0;
    static int is_me = 0;

    switch (state) {
	/* 
	 *  state: site
	 */
    case 0:
    default:		   
	switch (*item) {
	case '#':
	    state = 1;
	    break;
	case '\n':
	    break;
	case '!':
	case '/':
	case ',':
	case ':':
	    /* error */
	    log_msg(L_ERRno,"bad syntax in sys");
	    state = 1;
	    break;
	default:
	    if (strcmp(item,"ME") == 0) {
		log_msg(L_DEBUG4,"found ME sys-entry");
		state = 2;
		is_me = 1;
	    } else if (strcmp(item,spoolname) == 0) {
		log_msg(L_DEBUG,"found sys-entry for %s",item);
		state = 2;
		is_me = 0;
	    } else {
		/* BUG: C News allows own system name instead of ME */
		log_msg(L_DEBUG4,"skipping sys-entry for %s",item);
		state = 1;
	    }
	    break;
	}
	break;

	/* 
	 *  state: skip rest of line
	 */
    case 1:		   
	switch (*item) {
	default:
	    break;
	case '\n':
	    state = 0;
	    break;
	}
	break;

	/* 
	 *  state: ignore exclusions
	 */
    case 2:
	switch (*item) {
	case ':':
	    state = 3;
	    break;
	case '#':
	    state = 1;
	    break;
	case '\n':
	    state = 0;
	    break;
	default:
	case '!':
	case ',':
	    /* ignore */
	    break;
	}
	break;

	/* 
	 *  state: groups
	 */
    case 3:
    case 4: /* seen a '!' */
	switch (*item) {
	case ':': /* ignore flags */
	case '/': /* ignore distlist */
	case '#':
	    state = 1;
	    break;
	case '\n':
	    state = 0;
	    break;
	case '!':
	    state = 4;
	    break;
	case ',':
	    state = 3; 
	    break;
	default:
	    log_msg(L_DEBUG4,"sys group %s%s",(state==4?"!":""),item);
	    sys_group(item,is_me,state==4);
	    state = 3;
	    break;
	}
    }
}

/*
 *  pick a line from a C News style sys file
 *
 *  lines may be continued by using a trailing slash
 *
 *  format is:
 *	site/exclusions:grouplist/distlist:flags:cmd
 *
 *  site is (short) name of spool
 *  exclusions is things in patch we don't want to be bothered with
 *  grouplist is:
 *	group
 *	group,grouplist
 *	!group
 *	all		    matches everything (like *)
 *	something.all	    matches something.*
 *  distlist is a list of distributions
 *
 *  returns offset required when tokens continue
 */
static int 
sys_line(char *line, int offset)
{
    char what[2];
    int n;
    char *p;
    char *start = offset ? line : 0;

    /* quite rare case first: overflowed line continuation */
    if (line[0]=='\n' && !line[1] && offset >= 1) {
	for (n = 0; n < offset; ++n) {
	    if (line[offset-(n+1)] != '\\') break;
	}
	if (n & 1) {
	    line[--offset] = '\0';
	    return offset;
	}
    }

    for (p = line+offset; ;++p) {
	switch (*p) {
	case '#':
	case '/':
	case '!':
	case ':':
	case ',':
	case '\n':
	    /* separator */
	    what[0] = *p;
	    what[1] = '\0';
	    if (start) {
		*p = '\0';
		sys_item(start);
		*p = what[0];
		start = 0;
	    }
	    sys_item(what);
	    break;
	case ' ':
	case '\t':
	    *p = '\0';
	    if (start) sys_item(start);
	    start = 0; /* start collecting again */
	    break;
	case '\\':
	    switch (p[1]) {
	    case '\n': 
		/* line continuation */
		if (!p[2]) {
		    if (start) {
			/* token already collected */
			*p = '\0';
			if (start != line) strcpy(line,start);
			return strlen(line); /* offset */
		    } else {
			return 0;
		    }
		} else {
		    strcpy(p,p+2);
		    --p;
		    /* completely ignore backslash followed by newline */
		    break;
		}
		break;
	    case '\0': 
		/* long line */
		if (!start) start = p;
		if (start != line) strcpy(line,start);
		return strlen(line); /* offset */
	    default:
		if (!start) start = p;
		/* always include the next character as part of the token */
		++p;
		break;
	    }
	    break;
	case '\0':
	    if (start) {
		/* very long line, and token already collected */
		if (start != line) strcpy(line,start);
		return strlen(line); /* offset */
	    } 
	    return 0;
	default:
	    if (!start) start = p;
	    break;
	}
    }
}

/*
 *  load local sys or newsfeeds file
 *
 *  lines may be continued by using a trailing slash
 *
 *  the C News format is:
 *	site/exclusions:grouplist/distlist:flags:cmds
 *  the INN format is:
 *	site/exclusions:grouplist/distlist:flags:param
 *
 *  site is (short) name of spool
 *  exclusions is things in patch we don't want to be bothered with
 *  grouplist is:
 *	group
 *	group,grouplist
 *	!group
 *	group/distlist
 *  group is:
 *	groupelem	    matches groupelem.*
 *	groupelem.groupelem
 *  groupelem is: 
 *	all		    matches everything (C News)
 *	*		    matches everything (INN)
 *	something	    matches something
 *  distlist is a list of distributions that we simply ignore
 *  since distributions are quietly going out of fashion anyway
 */
void 
load_sys(void)
{
    FILE *f;
    char sysname[PATH_MAX];
    char buf[BUFSIZ];
    int offset;

    progtitle("read sys");

    if (inn_opt) {
	build_alt_filename(sysname,news_home(),NEWSFEEDS,newsfeeds);
    } else {
	build_alt_filename(sysname,news_home(),C_SYS_FILE,newsfeeds);
    }

    if (!(f = fopen(sysname,"r"))) {
	log_msg(L_ERRno,"can't open \"%s\"",sysname);
	unlock_exit(8);
    }
    log_msg(L_DEBUG,"reading %s",sysname);
    if (!is_regular(f,sysname)) {
	fclose(f);
	unlock_exit(8);
    }
    /* load local sys file */
    offset = 0;
    while (fgets(buf+offset,BUFSIZ-offset,f)) {
	offset = sys_line(buf,offset);
	if (offset >= BUFSIZ-3) {
	    log_msg(L_ERRno,"token too long in \"%s\"",sysname);
	    unlock_exit(8);
	}
    }
    fclose(f);
}
