/********************************************************************\
 * Data.h -- the main data structure of the program                 *
 * Copyright (C) 1997 Robin D. Clark                                *
 * Copyright (C) 1997 Linas Vepstas                                 *
 *                                                                  *
 * This program is free software; you can redistribute it and/or    *
 * modify it under the terms of the GNU General Public License as   *
 * published by the Free Software Foundation; either version 2 of   *
 * the License, or (at your option) any later version.              *
 *                                                                  *
 * This program is distributed in the hope that it will be useful,  *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of   *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the    *
 * GNU General Public License for more details.                     *
 *                                                                  *
 * You should have received a copy of the GNU General Public License*
 * along with this program; if not, write to the Free Software      *
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.        *
 *                                                                  *
 *   Author: Rob Clark                                              *
 * Internet: rclark@cs.hmc.edu                                      *
 *  Address: 609 8th Street                                         *
 *           Huntington Beach, CA 92648-4632                        *
\********************************************************************/

#ifndef __XACC_ACCOUNT_GROUP_H__
#define __XACC_ACCOUNT_GROUP_H__

#include "config.h"

#include "Account.h"

/** STRUCTS *********************************************************/
typedef struct _account_group {
  /* The flags: */
  unsigned int saved : 1;
  unsigned int new   : 1;
  
  Account *parent;                 /* back-pointer to parent */

  int      numAcc;                 /* number of accounts in array */
  Account  **account;              /* array of account pointers   */

  /* cached parameters */
  double balance;

} AccountGroup;

/** PROTOTYPES ******************************************************/
AccountGroup    *mallocAccountGroup( void );
void    freeAccountGroup( AccountGroup *account_group );

/*
 * The xaccConcatGroups() subroutine will move all accounts
 * from the "from" group to the "to" group
 *
 * The xaccMergeAccounts() subroutine will go through a group,
 * merging all accounts that have the same name and description.
 * This function is useful when importing Quicken(TM) files.
 */
void    xaccConcatGroups (AccountGroup *to, AccountGroup *from);
void    xaccMergeAccounts (AccountGroup *grp);

/*
 * The xaccAccountGroupNotSaved() subroutine will return 
 *    a non-zero value if any account in the group or in
 *    any subgroup hasn't been saved.
 *
 * The xaccAccountGroupMarkSaved() subroutine will mark
 *    the entire group as having been saved.
 */
int     xaccAccountGroupNotSaved  (AccountGroup *grp);
void    xaccAccountGroupMarkSaved (AccountGroup *grp);

Account *getAccount( AccountGroup *grp, int num );
Account *removeAccount( AccountGroup *grp, int num );
int     insertAccount( AccountGroup *grp, Account *acc );

/*
 * The xaccRemoveAccount() subroutine will remove the indicated
 * account from its parent account group. It will NOT free the
 * associated memory or otherwise alter the account: the account
 * can now be reparented to a new location.
 * Note, however, that it will mark the old parents as having 
 * been modified.
 *
 * The xaccRemoveGroup() subroutine will remove the indicated
 * account group from its parent account. It will NOT free the
 * associated memory or otherwise alter the account group: the 
 * account group can now be reparented to a new location.
 * Note, however, that it will mark the old parents as having 
 * been modified.
 */
void    xaccRemoveAccount (Account *);
void    xaccRemoveGroup (AccountGroup *);
int     xaccInsertSubAccount( Account *parent, Account *child );

/*
 * The xaccGetNumAccounts() subroutine returns the number
 * of accounts, including subaccounts, in the account group
 */
int     xaccGetNumAccounts (AccountGroup *grp);

/* 
 * The xaccGetAccountFromID() subroutine fetches the account
 *    with the indicated account id from the collection of accounts
 *    in the indicated AccountGroup.  It returns NULL if the 
 *    account was not found.
 *
 * The xaccGetPeerAccountFromID() subroutine fetches the account
 *    with the indicated account id from the collection of accounts
 *    in the same AccountGroup anchor group. It returns NULL if the
 *    account was not found.
 *
 * The xaccGetAccountFromName() subroutine fetches the
 *    account by name from the collection of accounts
 *    in the indicated AccountGroup group.
 *
 * The xaccGetPeerAccountFromName() subroutine fetches the
 *    account by name from the collection of accounts
 *    in the same AccountGroup anchor group.
 */

Account *xaccGetAccountFromID       (AccountGroup *, int);
Account *xaccGetPeerAccountFromID   (Account *, int);
Account *xaccGetAccountFromName     (AccountGroup *, char *);
Account *xaccGetPeerAccountFromName (Account *, char *);

/*
 * The xaccRecomputeGroupBalance() subroutine recursively totals
 * up the balances of all accounts in a group.
 */

void xaccRecomputeGroupBalance (AccountGroup *);

/*
 * The xaccGetRootGroupOfAcct () subroutine will find the topmost 
 * (root) group to which this account belongs.
 */

AccountGroup * xaccGetRootGroupOfAcct (Account *);

/* The xaccConsolidateGrpTrans() subroutine scans through
 * all of the transactions in an account, and compares them.
 * if any of them are exact duplicates, the duplicates are removed.
 * duplicates may occur when accounts from multiple sources are 
 * merged.  Note that this can be a dangerous operation to perform 
 *
 * Note that this suborutine merely walks the acount group
 * tree, and calls ConsolidateTransacations on each account
 */

void xaccConsolidateGrpTransactions (AccountGroup *);


/** GLOBALS *********************************************************/
extern AccountGroup *topgroup;

#endif /* __XACC_ACCOUNT_GROUP_H__ */
