/*
**          Copyright (c) Massachusetts Institute of Technology 1994.
**          All Rights Reserved.
**          Unpublished rights reserved under the copyright laws of
**          the United States.
**
** THIS MATERIAL IS PROVIDED AS IS, WITH ABSOLUTELY NO WARRANTY EXPRESSED
** OR IMPLIED.  ANY USE IS AT YOUR OWN RISK.
**
** This code is distributed freely and may be used freely under the 
** following conditions:
**
**     1. This notice may not be removed or altered.
**
**     2. This code may not be re-distributed or modified
**        without permission from MIT (contact 
**        lclint-request@larch.lcs.mit.edu.)  
**
**        Modification and re-distribution are encouraged,
**        but we want to keep track of changes and
**        distribution sites.
*/
/*
** sRefSetStack.c
**
** based on list_template.c
**
** where T has T_equal (or change this) and T_unparse
*/

# include "basic.h"

sRefSetStack
sRefSetStack_new ()
{
  sRefSetStack s = (sRefSetStack) dmalloc (sizeof (_sRefSetStack));
  
  s->elements = sRefSet_new ();
  s->allElements = sRefSet_undefined;
  s->thisbranch = sRefSetStackNULL;
  s->lastbranch = sRefSetStackNULL;
  s->parent = sRefSetStackNULL;
  s->lexlevel = fileScope;
  return (s);
}

sRefSetStack
sRefSetStack_branch (sRefSetStack s)
{
  sRefSetStack t = sRefSetStack_new ();

  t->parent = s;
  s->thisbranch = t;
  t->lexlevel = s->lexlevel + 1;

  if (s->lexlevel == fileScope)
    {
      t->allElements = t->elements;
    }
  else
    {
      t->allElements = s->allElements; 
    }

  /* llmsg (message ("sRefSetStack_branch %d: allElements = %s",
     t->lexlevel,
     sRefSet_unparse (t->allElements)));
     */

  return t;
}

sRefSetStack
sRefSetStack_trueBranch (sRefSetStack s)
{
  sRefSetStack t = sRefSetStack_new ();

  t->parent = s;
  s->thisbranch = t;
  t->lexlevel = s->lexlevel;
  t->allElements = sRefSet_newCopy (s->allElements);

  /* llmsg (message ("sRefSetStack_trueBranch %d: allElements = %s",
		  t->lexlevel,
		  sRefSet_unparse (t->allElements))); */
  return t;
}

sRefSetStack
sRefSetStack_altBranch (sRefSetStack s)
{
  sRefSetStack t = sRefSetStack_new ();
  sRefSetStack p = s->parent;

  llassert (p != sRefSetStackNULL);
  llassert (p->thisbranch == s);
  llassert (p->lastbranch == sRefSetStackNULL);

  p->lastbranch = s;
  p->thisbranch = t;
  t->parent = p;
  t->lexlevel = p->lexlevel;
  t->allElements = sRefSet_newCopy (p->allElements);

  /* llmsg (message ("sRefSetStack_altBranch %d: allElements = %s / %s",
		  t->lexlevel,
		  sRefSet_unparse (t->allElements),
		  sRefSet_unparse (p->lastbranch->allElements))); */
  return t;
}

sRefSetStack
sRefSetStack_pop (sRefSetStack s)
{
  sRefSetStack p = s->parent;
  
  llassert (p != sRefSetStackNULL);  
  llassert (p->thisbranch == s);

  p->thisbranch = sRefSetStackNULL;
  
  if (p->lexlevel <= paramsScope) /* global or parameter scope */
    {
    }
  else
    {
      p->elements = sRefSet_levelUnion (p->elements, s->elements, p->lexlevel);
      sRefSet_levelPrune (p->allElements, p->lexlevel);
    }

  sRefSetStack_free (s);

  /*
  llerror (message ("sRefSetStack_pop %d: allElements = %s",
		  p->lexlevel,
		  sRefSet_unparse (p->allElements)));

		  */
  return p;
}

sRefSetStack
sRefSetStack_popBranches (sRefSetStack alt)
{
  sRefSetStack p = alt->parent;
  sRefSetStack this = p->lastbranch;

  llassert (p != sRefSetStackNULL);
  llassert (p->thisbranch == alt);
  llassert (this != sRefSetStackNULL);

  /* llmsg (message ("sRefSetStack_popBranches %d: allElements = %s / %s + %s / %s",
		  p->lexlevel,
		  sRefSet_unparse (p->allElements),
		  sRefSet_unparse (alt->elements),
		  sRefSet_unparse (this->elements),
		  sRefSet_unparse (p->elements))); */

  p->elements = sRefSet_levelUnion (p->elements, 
				    sRefSet_intersect (alt->elements, this->elements),
				    p->lexlevel);
  p->allElements = sRefSet_union (p->allElements, p->elements);
  
  /* llmsg (message ("sRefSetStack_popBranches %d: allElements => %s / %s",
		p->lexlevel,
		sRefSet_unparse (p->allElements),
		sRefSet_unparse (p->elements))); */
  
  sRefSetStack_freeAll (alt);
  sRefSetStack_freeAll (this);

  p->thisbranch = sRefSetStackNULL;
  p->lastbranch = sRefSetStackNULL;

  return p;
}

sRefSetStack
sRefSetStack_popTrueBranch (sRefSetStack s)
{
  sRefSetStack p = s->parent;

  llassert (p != sRefSetStackNULL);
  llassert (p->thisbranch == s);
  llassert (p->lastbranch == sRefSetStackNULL);

  sRefSetStack_freeAll (s);

  /*
  llerror (message ("sRefSetStack_popTrueBranch %d: allElements = %s",
		  p->lexlevel,
		  sRefSet_unparse (p->allElements)));
		  */

  return p;
}

void sRefSetStack_push (sRefSetStack s, sRef el)
{
  /* printf ("push: %s\n", sRef_unparse (el)); */
  s->elements = sRefSet_insert (s->elements, el);

  if (s->lexlevel > fileScope && (s->elements != s->allElements))
    {
      s->allElements = sRefSet_insert (s->allElements, el);
    }

  /* printf ("==> %s\n", sRefSet_unparse (s->elements)); */
  /* printf ("==> %s\n", sRefSet_unparse (s->allElements)); */
}

sRefSet sRefSetStack_top (sRefSetStack s)
{
  return (s->allElements);
}

cstring
sRefSetStack_unparse (sRefSetStack s)
{
  if (s != sRefSetStack_undefined)
    {
      return (message ("[ %s ]", sRefSet_unparse (s->elements)));
    }
  else
    {
      return (message ("[ <undef> ]"));
    }
}

void
sRefSetStack_free (sRefSetStack s)
{
  sRefSet_free (s->elements); 
  free (s);
}

void
sRefSetStack_freeAll (sRefSetStack s)
{
  sRefSet_free (s->elements); 
  sRefSet_free (s->allElements); 
  free (s);
}
