/*
 * file devices.c
 *
 * Routines for Pointers device processing
 *
 * Joystick, mouse and trackball
 *
 * original idea from Chris Sharp <sharp@uk.ibm.com>
 *
 */

#define __DEVICES_C_

/* #define USE_X11_JOYEVENTS */ /* just done (if selected) in makefile */

#include "xmame.h"

#include <stdio.h>
#include <stdlib.h>
#include <fcntl.h>

#ifdef svgalib

#include <vga.h>
#include <vgagl.h>
#include <vgamouse.h>
#include <signal.h>

#endif

#ifdef USE_JOYSTICK

#ifdef X11_JOYSTICK
/* standard X input extensions based joystick */
#include <X11/extensions/XI.h>
#include <X11/extensions/XInput.h>
void x11_poll_joystick (void);
struct joystick {
	int buttons;
	int x;
	int y;
} joy_data, joy_orig;
XDevice *xdevice;

#elif  LIN_FM_TOWNS
/* FM-TOWNS game pad for linux */
#include <linux/joystick.h>
#include "pad.h"
void fm_town_poll_joystick (void);
extern struct JS_DATA_TYPE joy_data,joy_orig;

#elif  I386_JOYSTICK
void i386_poll_joystick (void);
/* specific joystick for PC clones */
#ifdef netbsd_i386
#include <machine/joystick.h>
extern struct joystick joy_data,joy_orig;
#else
#include <linux/joystick.h>
extern struct JS_DATA_TYPE joy_data,joy_orig;
#endif

#else
#error "USE_JOYSTICK needs a joystick type definition. Abort"
#endif

/* if USE_JOYSTICK */
#endif

#define MOUSE_THRESHOLD	0
#define JOY_THRESHOLD	0

#define TAN_22_5	106
#define TAN_67_5	618

#define abs(x) ( ( (x) >= 0 )? (x) : -(x) )
/* INLINE int abs(int x) {if (x>=0) return (x); return -(x); } */

/* joystick specific variables */

extern int joy_fd;
int joy_up, joy_down, joy_left, joy_right;
int joy_b1, joy_b2, joy_b3, joy_b4;

void (*poll_joy_func)(void);

/* mouse variables */
int mouse_x, mouse_y, mouse_ox, mouse_oy;
int mouse_up, mouse_down, mouse_left, mouse_right;
int mouse_b1, mouse_b2, mouse_b3;

/* trackball variables */
int trak_display_width;
int trak_display_height;
int trak_x;
int trak_y;
int prev_trak_x;
int prev_trak_y;

/*
 ******************************* Mouse routines *********************
 */

/* 
 * NOTE:
 * 
 * Due to the fact that in X-World mouse and trakball movements are 
 * coded as a MotionNotify event, we have a unique, centralized point to
 * evaluate these event. then propper variables are setted according data
 */

#ifndef svgalib
void process_mouse_event(XEvent *event) {
    mouse_x = trak_x = event->xbutton.x_root;	
    mouse_y = trak_y =event->xbutton.y_root;	
    mouse_b1 = ( event->xbutton.state & Button1Mask ) ? 1 : 0;
    mouse_b2 = ( event->xbutton.state & Button2Mask ) ? 2 : 0;
    mouse_b3 = ( event->xbutton.state & Button3Mask ) ? 4 : 0;
    return;
}
#endif

int sysdep_mouse_init(void) {
	mouse_x = mouse_ox = 0;
	mouse_y = mouse_oy = 0;
	mouse_up = mouse_down = mouse_left = mouse_right = 0;
	mouse_b1 = mouse_b2 = mouse_b3 = 0;
	if(use_mouse || use_trakball ) {
	   fprintf(stderr,"Mouse/Trakball Selected: Joystick (if any) and keyCursors Disabled\n");
	   use_joystick = 0 ;
	}
	return OSD_OK;
}

void svgalib_mouse_init() {
/* must be invoked in vga init routine */
#ifdef svgalib
   int fd;
   fd=mouse_init_return_fd("/dev/mouse",vga_getmousetype(),MOUSE_DEFAULTSAMPLERATE);
   if (fd<0) {
       perror("mouse_init");
       fprintf(stderr,"SVGALib: failed to open mouse device %d\n",fd);
       use_mouse=0;
   } else {
       fprintf(stderr,"Mouse/Trakball Selected: Joystick (if any) and keyCursors Disabled\n");
       use_joystick = 0 ;
   }
   /* fix ranges and initial position of mouse */
    mouse_setxrange(0,1000);
    mouse_setyrange(0,1000);
    mouse_x = mouse_ox = 500;
    mouse_y = mouse_oy = 500;
    mouse_setposition(mouse_x,mouse_y);
#endif
   return;
}

void sysdep_mouse_close(void) {
	if(!use_mouse) return;
	if (!use_trakball ) {	/* done by sysdep_trak_close() too */
#ifndef svgalib
	    XUngrabPointer(display,CurrentTime);
#else
	mouse_close();
#endif
	    return;
	}
}

/* evaluate mouse movements and button status */
void osd_poll_mouse(void) {
    int mouse_dx, mouse_dy;
    int mouse_tan;
    static int mouse_poll_count=0;

#ifdef svgalib
    /* in non-X11 mouse and trakball poll are not event-driven, so ask driver */
    /* 
	well, this is not really true: mouse_update() calls a hidden event parser
        but....
    */
    mouse_update();
    mouse_x=mouse_getx();
    mouse_y=mouse_gety();
    mouse_tan = mouse_getbutton();
    mouse_b1 = (MOUSE_LEFTBUTTON & mouse_tan)?   1 : 0;
    mouse_b2 = (MOUSE_RIGHTBUTTON & mouse_tan)?  2 : 0;
    mouse_b3 = (MOUSE_MIDDLEBUTTON & mouse_tan)? 4 : 0;
#endif

    ++mouse_poll_count;

    if( mouse_poll_count == 2 )
	{
        mouse_left = mouse_right = mouse_up = mouse_down = 0;

        mouse_dx = mouse_x - mouse_ox;
        mouse_dy = mouse_y - mouse_oy;

        /* Only register mouse movement over a certain threshold */
        if( ( abs(mouse_dx) > MOUSE_THRESHOLD ) || ( abs(mouse_dy) > MOUSE_THRESHOLD ) )
        {
            /* Pre-check for division by zero */
            if( ! mouse_dy ) mouse_dy++;
            /* Calculate the mouse delta tangent using fixed pt. */
            mouse_tan = (mouse_dx * 256)/mouse_dy;

            /* Approximate the mouse direction to a joystick direction */
            if( mouse_dy <= 0 ) { 
		if     ( mouse_tan<-TAN_67_5) {                mouse_right = 1;}
                else if( mouse_tan<-TAN_22_5) {mouse_up = 1;   mouse_right = 1;}
                else if( mouse_tan<TAN_22_5 ) {mouse_up = 1;                   }
                else if( mouse_tan<TAN_67_5 ) {mouse_up = 1;   mouse_left = 1; }
                else 			      {		       mouse_left = 1; }
            } else {
                if     ( mouse_tan<-TAN_67_5) {                mouse_left = 1; }
                else if( mouse_tan<-TAN_22_5) {mouse_down = 1; mouse_left = 1; }
                else if( mouse_tan<TAN_22_5 ) {mouse_down = 1; 		}
                else if( mouse_tan<TAN_67_5 ) {mouse_down = 1; mouse_right = 1;}
                else 			      {		       mouse_right = 1;}
            }
        }
	/* 
          in MSDOS, reset cursor position at every check. 
	  not done in X due to lost of Xrequests 
        */
        /* position_mouse( MOUSE_CENTRE_X, MOUSE_CENTRE_Y ); */
        mouse_ox = mouse_x;
        mouse_oy = mouse_y;
        mouse_poll_count = 0;
	}
}

int osd_mouse_pressed(int code)
{
    switch (code) {
    	case OSD_JOY_LEFT: return mouse_left; break;
    	case OSD_JOY_RIGHT: return mouse_right; break;
    	case OSD_JOY_UP: return mouse_up; break;
    	case OSD_JOY_DOWN: return mouse_down; break;
    	case OSD_JOY_FIRE1: return (mouse_b1); break;
    	case OSD_JOY_FIRE2: return (mouse_b2); break;
    	case OSD_JOY_FIRE3: return (mouse_b3); break;
    	case OSD_JOY_FIRE: return (mouse_b1 || mouse_b2 || mouse_b3); break;
    	default: return 0; break;
    }
}

/*
 ******************************* TrakBall routines *********************
 */

int sysdep_trakball_init(void) {
	if (!use_trakball) return OSD_OK;
	fprintf(stderr,"Trackball emulation via mouse enabled\n");
	/* just simply initialize variables */
	trak_x = prev_trak_x = 0;
	trak_y = prev_trak_y = 0;
	trak_display_width = 1;
	trak_display_height = 1;
	return OSD_OK;
}

void sysdep_trakball_close(void) { /* called from osd_close_display() */
	/* ungrabb mouse pointer */
	if (!use_trakball) return;
#ifndef svgalib
	XUngrabPointer(display,CurrentTime);
#else
	mouse_close();
#endif
	return;
}

void sysdep_poll_trakball(void) {
#ifndef svgalib
	Window root,child;
	int root_x, root_y, pos_x, pos_y;
	unsigned int  keys_buttons;
	if (XQueryPointer(display,window, &root,&child, &root_x,&root_y, &pos_x,&pos_y,&keys_buttons) ){
	    trak_x = pos_x;
	    trak_y = pos_y;
	}
#else
	trak_x = mouse_getx();
	trak_y = mouse_gety();
#endif
	return;
}

int osd_trak_pressed(int code)
{
    if (!use_trakball) return 0;
    /* really in msdos code returns mouse codes. no sense, but... 
       in fact makes mouse and trakball work the same */
    switch (code) {
    	case OSD_JOY_LEFT:  return mouse_left;  break;
    	case OSD_JOY_RIGHT: return mouse_right; break;
    	case OSD_JOY_UP:    return mouse_up;    break;
    	case OSD_JOY_DOWN:  return mouse_down;  break;
    	case OSD_JOY_FIRE1: return (mouse_b1);  break;
    	case OSD_JOY_FIRE2: return (mouse_b2);  break;
    	case OSD_JOY_FIRE3: return (mouse_b3);  break;
    	case OSD_JOY_FIRE:  return (mouse_b1 || mouse_b2 || mouse_b3); break;
    	default: return 0; break;
    }
    return 0;
}

int osd_trak_read(int axis)
{
	if (!use_trakball) return(NO_TRAK);
	if (axis==X_AXIS) return trak_x - prev_trak_x;
	if (axis==Y_AXIS) return trak_y - prev_trak_y;
	return 0; /* never should occurs */
}

void osd_trak_center_x(void) {
	prev_trak_x = trak_x;
}

void osd_trak_center_y(void) {
	prev_trak_y = trak_y;
}

/*
 ************************** FM TOWNS PAD specifics routines *************
 */
#ifdef LIN_FM_TOWNS
/*
 * Towns Pad control module for Xmame
 *
 * Author : Osamu KURATI
 * Version : 29 apr 1997 0.000
 */

/*
 * Joy Stick Code
 * Up : 1
 * Down       : 2
 * Left       : 4
 * Right      : 8
 * A  : 10
 * B  : 20
 *
 *
 * PAD bit number
 * up         : 0
 * down               : 1
 * left               : 2
 * right      : 3
 * A          : 4
 * B          : 5
 * RUN                : 6
 * SELECT     : 7
 */

static int iPad;
static unsigned long lPadLastButton = 0;

void PadOpen(char *pdevname)
{
  if ((iPad = open(pdevname, O_NONBLOCK | O_RDONLY)) < 0){
    fprintf(stderr,"Unable to open FM-TOWNS game pad. Joystick disabled\n");
    use_joystick=0;
  }
  lPadLastButton = 0;
  return;
}

int Pad()
{
  struct pad_event ev;
  if (read(iPad, &ev, sizeof ev) == sizeof ev){
    lPadLastButton = ev.buttons;
  }
  return((int) lPadLastButton & 0xff);
}

void PadClose() { close(iPad); }

/* TOWNS_PAD */
#endif


/*
 ******************************* JoyStick routines *********************
 */

int  sysdep_joy_initvars(void)
{
    joy_up   = joy_down  = 0;
    joy_left = joy_right = 0 ;
    joy_b1 = joy_b2 = 0;
    joy_b3 = joy_b4 = 0;
    if(!use_joystick) return OSD_OK;
#ifdef USE_JOYSTICK

#ifdef I386_JOYSTICK
    poll_joy_func = i386_poll_joystick;
#elif LIN_FM_TOWNS
    PadOpen(towns_pad_dev); 
    poll_joy_func = fm_town_poll_joystick;
#elif X11_JOYSTICK
    /* cannot init joystick till display has been opened ... */
    /* x11_joystick_init() */
    poll_joy_func = x11_poll_joystick;
#else
#error "USE_JOYSTICK option needs to specify joystick type"
#endif

#endif
    return OSD_OK;
}

void sysdep_joy_close(void) {
#ifdef LIN_FM_TOWNS
    if (use_joystick)       PadClose();
#else
    /* just done in sysdep_exit() */
#endif
    return;
}


#ifndef svgalib

void process_x11_joy_event(XEvent *event) {
#ifdef USE_X11_JOYEVENTS
/* does not run yet, don't know why :-( */
#ifdef X11_JOYSTICK
    XDeviceButtonEvent *dbe;
    XDeviceMotionEvent *dme;
    /* remember that event types are not harcoded: we evaluated it in XOpenDevice() */
    /* hack: we'll suppose that:
	 first_axis is allways equals 0. 
	 device_id is joystic's id
       in a real program, should be checked... 
     */
    if (!use_joystick) return;
    if ( (event->type==devicebuttonpress) || (event->type==devicebuttonrelease) ) {
	dbe=(XDeviceButtonEvent *) event;	
	/* evaluate button state */
	joy_data.buttons=dbe->device_state;
	/* and now X-Y position remember that uses relative mode */
	if (! swapjoyaxis) 
	     { 
		joy_data.x=joy_orig.x+( dbe->axis_data[0] >> joyfilter); 
		joy_data.y=joy_orig.y+( dbe->axis_data[1] >> joyfilter); 
	} else { 
		joy_data.x=joy_orig.x+( dbe->axis_data[1] >> joyfilter); 
		joy_data.y=joy_orig.y+( dbe->axis_data[0] >> joyfilter); 
	}
    }
    if ( (event->type==devicemotionnotify) ) {
	dme=(XDeviceMotionEvent *) event;	
	/* evaluate button state */
	joy_data.buttons=dme->device_state;
	if (! swapjoyaxis) 
	     { joy_data.x=joy_orig.x+dme->axis_data[0]; joy_data.y=joy_orig.y+dme->axis_data[1]; } 
	else { joy_data.x=joy_orig.x+dme->axis_data[1]; joy_data.y=joy_orig.y+dme->axis_data[0]; }
    }
    joy_b1 = ( joy_data.buttons & Button1Mask ) ? 1 : 0;
    joy_b2 = ( joy_data.buttons & Button2Mask ) ? 2 : 0;
    joy_b3 = ( joy_data.buttons & Button3Mask ) ? 4 : 0;
    joy_b4 = ( joy_data.buttons & Button4Mask ) ? 8 : 0;
#endif
#endif
    return;
}

/* notdef svgalib */
#endif

int x11_joystick_init() {
#ifndef X11_JOYSTICK
	if (!use_joystick) return OSD_OK;
#else
	int 		i,j,k;
	int 		result;
	XDeviceInfoPtr 	list,slist;
	XAnyClassPtr 	any;
	XButtonInfoPtr 	binfo;
	XValuatorInfoPtr vinfo;
	XAxisInfoPtr    ainfo;
	XInputClassInfo *classptr;
	XEventClass 	xeventlist[8];
	int 		xeventcount;
	

	/* query server for input extensions */
	result =XQueryExtension(display,"XInputExtension",&i,&j,&k);
	if(!result) {
	    fprintf(stderr,"Your server doesn't support XInput Extensions\n");
	    fprintf(stderr,"X11 Joystick disabled\n");
	    use_joystick=0;
	    return OSD_OK;
	}
	/* now get input device list and locate desired device */
	list = XListInputDevices(display,&result);
	if (!list ) {
	    fprintf(stderr,"No extended input devices found !!\n");
	    fprintf(stderr,"X11 Joystick disabled\n");
	    use_joystick=0;
	    return OSD_OK;
	}
	slist=list;
	for(i=j=0;i<result;i++,list++) 
		if ( ! strcmp(x11joyname,list->name)  ) { j=1; break; }
	if (!j) {
	    fprintf(stderr,"Cannot locate device \"%s\" in available devices\n",x11joyname);
	    fprintf(stderr,"X11 Joystick disabled\n");
	    use_joystick=0;
	    XFreeDeviceList(slist);
	    return OSD_OK;
	}
	/* test for correct device ( search at least two buttons and two axis */
	any = (XAnyClassPtr)(list->inputclassinfo);
	result=0;
	for(j=0;j<list->num_classes;j++) {
	    switch(any->class) {
		case ButtonClass:
			binfo=(XButtonInfoPtr) any;
			if ((num_x_buttons=binfo->num_buttons)>=2) result |= 0x01;
			fprintf(stderr,"%s: %d buttons\n",x11joyname,num_x_buttons);
			joy_data.buttons=joy_orig.buttons=0;
			break;
		case ValuatorClass:
			vinfo=(XValuatorInfoPtr) any;
			if (vinfo->num_axes>=2) result |= 0x02;
			fprintf(stderr,"%s: %d axes\n",x11joyname,vinfo->num_axes);
			ainfo=vinfo->axes;
			if (!swapjoyaxis ) {
			  joy_data.x=joy_orig.x=(ainfo->max_value-ainfo->min_value)/2;
			  ainfo++; /* next axis */
			  joy_data.y=joy_orig.y=(ainfo->max_value-ainfo->min_value)/2;
			} else {
			  joy_data.y=joy_orig.y=(ainfo->max_value-ainfo->min_value)/2;
			  ainfo++; /* next axis */
			  joy_data.x=joy_orig.x=(ainfo->max_value-ainfo->min_value)/2;
			}
			break;
		case KeyClass: /* no sense to use a extended key device */
			fprintf(stderr,"%s: Ingnoring KeyClass info\n",x11joyname);
		default: break;  /* unnknown class: ignore */
	    }
	    any = (XAnyClassPtr) ((char *) any+any->length);
	}
	if (result != 0x03 ) {
	    fprintf(stderr,"Your selected X11 device \"%s\"doesn't match X-Mame requirements\n",x11joyname);
	    fprintf(stderr,"X11 Joystick disabled\n");
	    use_joystick=0;
	    XFreeDeviceList(slist);
	    return OSD_OK;
	}
	/* once located, try to open */	
	if ( ! (xdevice=XOpenDevice(display,list->id) ) ) {
	    fprintf(stderr,"XDeviceOpen error\n");
	    use_joystick=0;
	    XFreeDeviceList(slist);
	    return OSD_OK;
	} 
	/* buscamos los eventos asociados que necesitamos */
	/* realmente el bucle for y la sentencia switch no son necesarias, pues
	   en XInput.h se buscan automaticamente los elementos de cada dato, pero
	   lo pongo de ejemplo para si en el futuro se quieren chequear la existencia
           de una determinada clase antes de pedir eventos. Nosotros sabemos a 
	   priori que no deberia fallar....
	*/
	xeventcount=0;
	for (i=0,classptr=xdevice->classes;i<xdevice->num_classes;i++,classptr++ ) {
	    switch(classptr->input_class) {
		case KeyClass: break;
		case ButtonClass:
	    		DeviceButtonPress(xdevice,devicebuttonpress,xeventlist[xeventcount]);
			if (devicebuttonpress) xeventcount++;
	    		DeviceButtonRelease(xdevice,devicebuttonrelease,xeventlist[xeventcount]);
			if (devicebuttonrelease) xeventcount++;
			break;
		case ValuatorClass:
	    		DeviceMotionNotify(xdevice,devicemotionnotify,xeventlist[xeventcount]);
			if (devicemotionnotify) xeventcount++;
	    		DeviceButtonMotion(xdevice,devicebuttonmotion,xeventlist[xeventcount]);
			if (devicebuttonmotion) xeventcount++;
			break;
		case FocusClass: break;
		case ProximityClass: break;
		case OtherClass: break;
		default: break;
	    }
	}
#if 0
	/* 
	NOTE: don't know why but these two items don't work in my linux
	XInputExtension Joystick module. still working on it ...
	*/

	/* force relative motion report */
	XSetDeviceMode(display,xdevice,Relative);
	/* set starting point of joystick (to force joy to be centered) */
	if(!swapjoyaxis) {
	   XSetDeviceValuators(display,xdevice,&(joy_data.x),0,1);
	   XSetDeviceValuators(display,xdevice,&(joy_data.y),1,1);
	}else{
	   XSetDeviceValuators(display,xdevice,&(joy_data.y),0,1);
	   XSetDeviceValuators(display,xdevice,&(joy_data.x),1,1);
	}
#endif
#ifdef USE_X11_JOYEVENTS
	/* not sure why don't recognize all events type. still working... */
	XSelectExtensionEvent(display,window,xeventlist,xeventcount);
	fprintf(stderr,"X11PointerDevice: Using X-Window Events\n");
#else
	fprintf(stderr,"X11PointerDevice: Using Demand QueryState\n");
#endif
	fprintf(stderr,"Found and installed X11 pointer device \"%s\"\n",x11joyname);
	/* and finaly free requested device list */
	XFreeDeviceList(slist);
#endif
	return OSD_OK;
}

void osd_poll_joystick (void)
{
    /* in msdos.c is coded like this. Not sure if correct .... */
    if (use_mouse)    { osd_poll_mouse(); return; }
    if (use_joystick) { (*poll_joy_func)(); return; }

}

int osd_joy_pressed (int joycode)
{
    /* if joystick not implemented , all variables set to zero */
    if (use_mouse) 	    return osd_mouse_pressed(joycode);
    if (use_trakball) 	    return osd_trak_pressed(joycode);
    switch (joycode)
    {
    	case OSD_JOY_LEFT:  return joy_left;  break;
    	case OSD_JOY_RIGHT: return joy_right; break;
    	case OSD_JOY_UP:    return joy_up;    break;
    	case OSD_JOY_DOWN:  return joy_down;  break;
    	case OSD_JOY_FIRE1: return joy_b1;    break;
    	case OSD_JOY_FIRE2: return joy_b2;    break;
    	case OSD_JOY_FIRE3: return joy_b3;    break;
    	case OSD_JOY_FIRE4: return joy_b4;    break;
    	case OSD_JOY_FIRE:  return (joy_b1 || joy_b2 || joy_b3 || joy_b4); break;
    	default:            return FALSE;     break;
    }
}

/* return the name of a joystick button */
const char *osd_joy_name(int joycode)
{
  static char *joynames[] = { "LEFT", "RIGHT", "UP", "DOWN", "Button A",
			      "Button B", "Button C", "Button D", "Any Button" };
  static char *nonedefined = "None";

  if (joycode && joycode <= OSD_MAX_JOY) return (char*)joynames[joycode-1];
  else return (char *)nonedefined;
}

/*
 * given a new x an y joystick axis value convert it to a move definition
 */

void evaluate_joy_moves(void) {
#ifdef USE_JOYSTICK
    int joy_dx,joy_dy,joy_tan;
    static int joy_poll_count=0;
    if (++joy_poll_count==2) {
	/* code rewritten by jantonio@dit.upm.es Aug-1997 */
        joy_left = joy_right = joy_up = joy_down = 0;
	joy_dx=joy_data.x-joy_orig.x;
	joy_dy=joy_data.y-joy_orig.y;
        
	/* Only register joystick movement over a certain threshold */
        if( ( abs(joy_dx) > JOY_THRESHOLD ) || ( abs(joy_dy) > JOY_THRESHOLD ) )
        {
            /* Pre-check for division by zero */
            if( ! joy_dy ) joy_dy++;
            /* Calculate the joystick delta tangent using fixed pt. */
            joy_tan = (joy_dx * 256)/joy_dy;

            /* evaluate joystick direction */
            if( joy_dy <= 0 ) { 
	        if     ( joy_tan < -TAN_67_5 ) {               joy_right = 1; }
                else if( joy_tan < -TAN_22_5 ) { joy_up = 1;   joy_right = 1; }
                else if( joy_tan < TAN_22_5  ) { joy_up = 1;                  }
                else if( joy_tan < TAN_67_5  ) { joy_up = 1;   joy_left = 1;  }
                 else 			       { 	       joy_left = 1;  }
            } else {
                if     ( joy_tan < -TAN_67_5 ) {               joy_left = 1;  }
                else if( joy_tan < -TAN_22_5 ) { joy_down = 1; joy_left = 1;  }
                else if( joy_tan < TAN_22_5  ) { joy_down = 1; 		      }
                else if( joy_tan < TAN_67_5  ) { joy_down = 1; joy_right = 1; }
                else 			       { 	       joy_right = 1; }
            }
        }
        joy_poll_count = 0;
    } /* if joy_poll_count */
#endif
    return;
}

/*
 * Linux FM-TOWNS game pad driver based joystick emulation routine
 */
void fm_town_poll_joystick(void) {
#if ( defined USE_JOYSTICK ) && ( defined LIN_FM_TOWNS )
      int res = Pad();

      /* get value of buttons */
      joy_b1 = (res>>4) & 0x01;
      joy_b2 = (res>>4) & 0x02;
      joy_b3 = (res>>4) & 0x04;
      joy_b4 = (res>>4) & 0x08;

      joy_up    = res & 0x01;
      joy_down  = res & 0x02;
      joy_left  = res & 0x04;
      joy_right = res & 0x08;

      evaluate_joy_moves();
#endif
	return;
}

/* 
 * Routine to manage joystick via X-Windows Input Extensions
 * should work in any X-Server that supports them
 */
void x11_poll_joystick(void) {
#if ( defined USE_JOYSTICK ) && (defined X11_JOYSTICK )

#ifndef USE_X11_JOYEVENTS
	/* perform a roudtrip query to joy device to ask state */
	XDeviceState    *xstate;
	XInputClass     *any;
	XValuatorState  *vinfo;
	XButtonState    *binfo;
	int j;
	xstate = XQueryDeviceState(display,xdevice);
	any = (XInputClass *)(xstate->data);
	for(j=0;j<xstate->num_classes;j++) {
	    switch(any->class) {
		case ButtonClass:
			binfo=(XButtonState *) any;
			joy_data.buttons=(int)binfo->buttons[0];
			break;
		case ValuatorClass:
			vinfo=(XValuatorState *) any;
			if (!swapjoyaxis ) {
			   joy_data.x = joy_orig.x + ( *(vinfo->valuators) >> joyfilter);
			   joy_data.y = joy_orig.y + ( *(vinfo->valuators+1) >> joyfilter);
			} else {
			   joy_data.y= joy_orig.y + ( *(vinfo->valuators) >> joyfilter);
			   joy_data.x= joy_orig.x + ( *(vinfo->valuators+1) >> joyfilter);
			}
			break;
		case KeyClass: /* no sense to use a extended key device */
		default: break;  /* unknown class: ignore */
	    }
	    any = (XInputClass *) ((char *) any+any->length);
	}
	XFreeDeviceState(xstate);
/* no idea why, but my Xserver starts in 2nd bit to encoding button masks */
#if 0
        joy_b1=joy_data.buttons & 0x01; joy_b2=joy_data.buttons & 0x02;
        joy_b3=joy_data.buttons & 0x04; joy_b4=joy_data.buttons & 0x08;
#else
        joy_b1=joy_data.buttons & 0x02; joy_b2=joy_data.buttons & 0x04;
        joy_b3=joy_data.buttons & 0x08; joy_b4=joy_data.buttons & 0x10;
#endif

#endif 
	evaluate_joy_moves();
#endif
	return;
}

/* 
 * Routine to manage PC clones mouse via standard driver 
 */
void i386_poll_joystick (void)
{
#if ( defined USE_JOYSTICK ) && ( defined  I386_JOYSTICK )
        int res;
#ifdef netbsd_i386
	res = read(joy_fd,&joy_data,sizeof(struct joystick) );
	if (res == sizeof(struct joystick )) {
#else
	res = read(joy_fd,&joy_data,sizeof(struct JS_DATA_TYPE) );
	if (res == sizeof(struct JS_DATA_TYPE )) {
#endif
	/* get value of buttons */
#ifdef netbsd_i386
	    joy_b1 = joy_data.b1;
	    joy_b2 = joy_data.b2;
#else
	    joy_b1 = joy_data.buttons & 0x01;	
	    joy_b2 = joy_data.buttons & 0x02;	
	    joy_b3 = joy_data.buttons & 0x04;	
	    joy_b4 = joy_data.buttons & 0x08;	
#endif
            /* now parse movements */
	    if (! swapjoyaxis) {
	        joy_data.x >>=joyfilter;
	        joy_data.y >>=joyfilter;
	    } else {
	        int a = (joy_data.y >> joyfilter);
	        joy_data.y = (joy_data.x >> joyfilter);
	        joy_data.x = a;
	    }
	/* evaluate joystick movements */
	    evaluate_joy_moves();
	} /* if read ok */
#endif
	return;
}

