/****************************************************************************/
/*                                                                          */
/* IMG* Image Processing Tools Library                                      */
/* Program:   imgFFTDOG.c                                                   */
/* Author:    Simon A.J. Winder                                             */
/* Date:      Thu Oct 20 20:45:38 1994                                      */
/* Copyright (C) 1994 Simon A.J. Winder                                     */
/*                                                                          */
/****************************************************************************/

#include "tools.h"

#define PRGNAME "FFTDOG"
#define ERROR(e) imgError(e,PRGNAME)

void make_filter(it_image *);
void real_multiply(it_image *,it_image *);

static double kc,rc,ks,rs;

int main(int argc,char **argv)
{
  it_image *in1,*f1;
  int width,height;
  double aa,pk;

  IFHELP
    {
      fprintf(stderr,"img%s - Fourier difference of Gaussian filter\n",
	      PRGNAME);
      fprintf(stderr,"img%s [rc rs [kc ks]]\n",
	      PRGNAME);
      fprintf(stderr,"  stdin: Complex\n");
      fprintf(stderr,"  stdout: Complex\n");
      fprintf(stderr,"  rc, rs are the centre and surround radii at e^(-0.5) (pixels)\n");
      fprintf(stderr,"  kc, ks are the centre and surround gain constants\n");
      exit(0);
    }

  imgStart(PRGNAME);

  rc=1.0;
  rs=3.0;
  if(argc>5)
    ERROR("invalid arguments");
  if(argc>1)
    {
      if(argc<3)
	ERROR("too few arguments");
      rc=atof(argv[1]);   /* e^(-0.5) radius */
      rs=atof(argv[2]);
    }
  rc=rc/M_SQRT2;
  rs=rs/M_SQRT2;
  if(argc>3)
    {
      if(argc<5)
	ERROR("too few arguments");
      kc=atof(argv[3]);
      ks=atof(argv[4]);
    }
  else
    {
      aa=rs*rs-rc*rc;
      pk=pow(rs/rc,-2.0*rc*rc/aa)-pow(rs/rc,-2.0*rs*rs/aa);
      kc=1.0/(pk*M_PI*rc*rc);
      ks=1.0/(pk*M_PI*rs*rs);
      fprintf(stderr,"kc = %g, ks = %g.\n",kc,ks);
    }

  /* Loop for all images */
  do {
    in1=i_read_image_file(stdin,IT_COMPLEX,IM_FRAGMENT);
    if(in1==NULL)
      ERROR("can't import image file");
    width=in1->width;
    height=in1->height;

    if((f1=i_create_image(width,height,IT_FLOAT,IM_FRAGMENT))==NULL)
      ERROR("out of memory");

    make_filter(f1);
    real_multiply(in1,f1);
    i_destroy_image(f1);
    i_write_image_file(stdout,in1,IF_BINARY);
    i_destroy_image(in1);
  } while(!feof(stdin));

  imgFinish(PRGNAME);
  return(0);
}

void make_filter(it_image *f)
{
  double u,v,phwc,phhc,rsqd,cc,cs,xc,xs;
  int x,y,width,height,halfwidth,halfheight;
  it_float *flt_ptr;

  width=f->width;
  height=f->height;
  halfwidth=width/2;
  halfheight=height/2;
  phwc=M_PI/((double) halfwidth);
  phhc=M_PI/((double) halfheight);
  cc=kc*M_PI*rc*rc;
  cs=ks*M_PI*rs*rs;
  xc= -rc*rc/4.0;
  xs= -rs*rs/4.0;

  for(y=0;y<height;y++)
    {
      flt_ptr=im_float_row(f,y);
      for(x=0;x<width;x++)
	{
	  u=phwc*(double) ((x<halfwidth)?(x):(x-width));
	  v=phhc*(double) ((y<halfheight)?(y):(y-height));
	  rsqd=u*u+v*v;
	  *flt_ptr++= cc*exp(xc*rsqd)-cs*exp(xs*rsqd);
	}
    }
}

void real_multiply(it_image *c,it_image *f)
{
  int x,y,width;
  it_float *flt_ptr;
  it_complex *cpx_ptr;
  
  width=c->width;
  for(y=0;y<(c->height);y++)
    {
      flt_ptr=im_float_row(f,y);
      cpx_ptr=im_complex_row(c,y);
      for(x=0;x<width;x++)
	{
	  cpx_ptr->Im *= *flt_ptr;
	  (cpx_ptr++)->Re *= *flt_ptr++;
	}
    }
}
/* Version 1.0 (Oct 1994) */
/* Version 1.1 (Nov 1994) */
