/************************************************************
 *                                                          *
 *  Permission is hereby granted  to  any  individual   or  *
 *  institution   for  use,  copying, or redistribution of  *
 *  the xgobi code and associated documentation,  provided  *
 *  that   such  code  and documentation are not sold  for  *
 *  profit and the  following copyright notice is retained  *
 *  in the code and documentation:                          *
 *        Copyright (c) 1990, ..., 1996 Bellcore            *
 *                                                          *
 *  We welcome your questions and comments, and request     *
 *  that you share any modifications with us.               *
 *                                                          *
 *    Deborah F. Swayne            Dianne Cook              *
 *   dfs@research.att.com       dicook@iastate.edu          *
 *      (973) 360-8423    www.public.iastate.edu/~dicook/   *
 *                                                          *
 *                    Andreas Buja                          *
 *                andreas@research.att.com                  *
 *              www.research.att.com/~andreas/              *
 *                                                          *
 ************************************************************/

#include <stdio.h>
#include "xincludes.h"
#include "xgobitypes.h"
#include "xgobivars.h"
#include "xgobiexterns.h"

extern xgobidata xgobi;

static Widget var_mouse, var_mouse_tform, var_mouse_tour, var_mouse_corr;
static Widget var_mouse_oblique, var_mouse_dotplot;
static int cursor_in_panel = 0 ;
static int last_touched = 1 ;

/* ARGSUSED */
XtEventHandler
set_var_tform_label(w, xg, evnt)
  Widget w;
  xgobidata *xg;
  XEvent *evnt;
{
  if (evnt->type == EnterNotify)
  {
    XtUnmapWidget(var_mouse);
    XtUnmapWidget(var_mouse_dotplot);
    XtUnmapWidget(var_mouse_oblique);
    XtUnmapWidget(var_mouse_tour);
    XtUnmapWidget(var_mouse_corr);
    XtMapWidget(var_mouse_tform);
  }
  else if (evnt->type == LeaveNotify)
  {
    XtUnmapWidget(var_mouse_tform);
    if (xg->is_touring && !xg->is_backtracking && !xg->is_local_scan)
      XtMapWidget(var_mouse_tour);
    else if (xg->is_spinning && xg->is_spin_type.oblique)
      XtMapWidget(var_mouse_oblique);
    else if (xg->is_dotplotting)
      XtMapWidget(var_mouse_dotplot);
    else if (xg->is_corr_touring && !(corr_backtracking(xg)))
      XtMapWidget(var_mouse_corr);
    else if (xg->is_xyplotting ||
        (xg->is_spinning && !xg->is_spin_type.oblique))
      XtMapWidget(var_mouse);
  }
}

void
set_varsel_label(xg)
/*
 * Set the label above the variable selection panel.
*/
  xgobidata *xg;
{
  if (xg->is_touring)
  {
    XtUnmapWidget(var_mouse);
    XtUnmapWidget(var_mouse_oblique);
    XtUnmapWidget(var_mouse_dotplot);
    XtUnmapWidget(var_mouse_corr);
    if (xg->is_backtracking || xg->is_local_scan)
      XtUnmapWidget(var_mouse_tour);
    else
      XtMapWidget(var_mouse_tour);
  }
  else if (xg->is_corr_touring)
  {
    XtUnmapWidget(var_mouse);
    XtUnmapWidget(var_mouse_oblique);
    XtUnmapWidget(var_mouse_dotplot);
    XtUnmapWidget(var_mouse_tour);
    if (corr_backtracking(xg))
      XtUnmapWidget(var_mouse_corr);
    else
      XtMapWidget(var_mouse_corr);
  }
  else if (xg->is_spinning && xg->is_spin_type.oblique)
  {
    XtUnmapWidget(var_mouse);
    XtUnmapWidget(var_mouse_dotplot);
    XtUnmapWidget(var_mouse_tour);
    XtUnmapWidget(var_mouse_corr);
    XtMapWidget(var_mouse_oblique);
  }
  else if (xg->is_xyplotting ||
       (xg->is_spinning && !xg->is_spin_type.oblique))
  {
    XtUnmapWidget(var_mouse_tour);
    XtUnmapWidget(var_mouse_corr);
    XtUnmapWidget(var_mouse_dotplot);
    XtUnmapWidget(var_mouse_oblique);
    XtMapWidget(var_mouse);
  }
  else if (xg->is_dotplotting)
  {
    XtUnmapWidget(var_mouse_tour);
    XtUnmapWidget(var_mouse_corr);
    XtUnmapWidget(var_mouse_oblique);
    XtUnmapWidget(var_mouse);
    XtMapWidget(var_mouse_dotplot);
  }

}

void
make_varboxes(xg)
  xgobidata *xg;
{
  int j;

  for (j=0; j<xg->ncols-1; j++)
  {
    xg->varboxw[j] = XtVaCreateWidget("VarForm",
      boxWidgetClass, xg->var_panel,
      NULL);
    if (mono) set_mono(xg->varboxw[j]);
  }
  /*
   * Create a variable box for the variable that can
   * be defined in brushing,
  */
  xg->varboxw[xg->ncols-1] = XtVaCreateWidget("VarForm",
    boxWidgetClass, xg->var_panel,
    XtNmappedWhenManaged, (Boolean) False,
    NULL);
  if (mono) set_mono(xg->varboxw[xg->ncols-1]);

  for (j=0; j<xg->ncols; j++)
  {
    xg->varlabw[j] = XtVaCreateManagedWidget("VarLabel",
      menuButtonWidgetClass, xg->varboxw[j],
      XtNlabel, (String) xg->collab[j],
      XtNleftBitmap, (Pixmap) NULL,  /* Don't want menu icons here */
      NULL);
    if (mono) set_mono(xg->varlabw[j]);

    add_menupb_help(&xg->nhelpids.menupb,
      xg->varlabw[j], "VarTransform");
    XtAddEventHandler(xg->varlabw[j],
      ButtonPressMask, FALSE,
      (XtEventHandler) add_tform_menu, (XtPointer) xg);
    XtAddEventHandler(xg->varlabw[j],
      EnterWindowMask | LeaveWindowMask, FALSE,
      (XtEventHandler) set_var_tform_label, (XtPointer) xg);
  }

  for (j=0; j<xg->ncols; j++)
  {
    xg->vardraww[j] = XtVaCreateManagedWidget("VarWindow",
      labelWidgetClass, xg->varboxw[j],
      XtNfromVert, (Widget) xg->varlabw[j],
      XtNlabel, (String) "",
      NULL);
    if (mono) set_mono(xg->vardraww[j]);

    add_pb_help(&xg->nhelpids.pb,
      xg->vardraww[j], "VarPanel");

    XtAddEventHandler(xg->vardraww[j],
      ExposureMask, FALSE,
      (XtEventHandler) varexpose, (XtPointer) xg);
    XtAddEventHandler(xg->vardraww[j],
      ButtonPressMask | ShiftMask | ControlMask | KeyPressMask | 
      KeyReleaseMask, FALSE,
      (XtEventHandler) varselect, (XtPointer) xg);
  }

  xg->varchosen[0] = True;
  xg->varchosen[1] = True;
  for (j=2; j<xg->ncols; j++)
    xg->varchosen[j] = False;
}

void
make_var_mouses(xg)
  xgobidata *xg;
{
/*
 * VarMouseLabel: the variable selection mouse label for use in
 * xyplotting and axis rotation and correlation tour (var_mouse),
 * grand tour (var_mouse_tour),
 * oblique rotation (var_mouse_oblique),
 * and the var transformation mouse label (var_mouse_tform).
*/
  Arg args[5];

  XtSetArg(args[0], XtNfromVert, xg->var_panel);
  XtSetArg(args[1], XtNtop, XtChainBottom);
  XtSetArg(args[2], XtNbottom, XtChainBottom);

  XtSetArg(args[3], XtNlabel, "L: Select X,  M: Y ");
  var_mouse = XtCreateManagedWidget("VarMouseLabel",
    labelWidgetClass, xg->box2, args, 4);
  if (mono) set_mono(var_mouse);

/* The remainder are not initially mapped */
  XtSetArg(args[3], XtNmappedWhenManaged, False);

  XtSetArg(args[4], XtNlabel, "M: Select Y ");
  var_mouse_dotplot = XtCreateManagedWidget("VarMouseLabel",
    labelWidgetClass, xg->box2, args, 5);
  if (mono) set_mono(var_mouse_dotplot);

  XtSetArg(args[4], XtNlabel, "L/M: Select Variable ");
  var_mouse_oblique = XtCreateManagedWidget("VarMouseLabel",
    labelWidgetClass, xg->box2, args, 5);
  if (mono) set_mono(var_mouse_oblique);

  XtSetArg(args[4], XtNlabel, "L/M: Toggle (see Help)");
  var_mouse_tour = XtCreateManagedWidget("VarMouseLabel",
    labelWidgetClass, xg->box2, args, 5);
  if (mono) set_mono(var_mouse_tour);

  XtSetArg(args[4], XtNlabel, "L: X, M: Y (see Help)");
  var_mouse_corr = XtCreateManagedWidget("VarMouseLabel",
    labelWidgetClass, xg->box2, args, 5);
  if (mono) set_mono(var_mouse_corr);

  XtSetArg(args[4], XtNlabel, "L: Transform Variable ");
  var_mouse_tform = XtCreateManagedWidget("VarMouseLabel",
    labelWidgetClass, xg->box2, args, 5);
  if (mono) set_mono(var_mouse_tform);
}

void
make_varpanel(xg)
  xgobidata *xg;
{
  int j;
  Dimension width, height;

/*
 * VarPanel: contains all variable selection widgets.
*/
  XtVaGetValues(xg->workspace, XtNheight, &height, NULL);

  xg->var_panel = XtVaCreateManagedWidget("VarPanel",
    boxWidgetClass, xg->box2,
    XtNheight, (Dimension) height,
    /*XtNfromVert, (Widget) var_mouse,*/
    NULL);
  if (mono) set_mono(xg->var_panel);

  xg->varboxw = (Widget *) XtMalloc(
    (Cardinal) xg->ncols * sizeof(Widget));
  xg->varlabw = (Widget *) XtMalloc(
    (Cardinal) xg->ncols * sizeof(Widget));
  xg->vardraww = (Widget *) XtMalloc(
    (Cardinal) xg->ncols * sizeof(Widget));
  xg->varchosen = (Boolean *) XtMalloc(
    (Cardinal) xg->ncols * sizeof(Boolean));

  make_varboxes(xg);
  XtManageChildren(xg->varboxw, (unsigned int) xg->ncols);

/*
 * Force the variable drawing area to be square; assign the
 * value of radius which is used in drawing the lines.
*/
  XtVaGetValues(xg->vardraww[0], XtNwidth, &width, NULL);
  xg->radius = width/2 - 1;

  for (j=0; j<xg->ncols; j++)
    XtVaSetValues(xg->vardraww[j], XtNheight, (Dimension) width, NULL);

  make_var_mouses(xg);
}

void
draw_last_touched(xg)
/*
 * If the cursor is inside the var_panel widget, then draw the
 * circle indicating which variable is last_touched.
*/
  xgobidata *xg;
{
  if (cursor_in_panel)
  {
    XDrawArc(display, XtWindow(xg->vardraww[last_touched]),
      varpanel_copy_GC,
      (int) (xg->radius - 5), (int) (xg->radius - 5),
      (unsigned int) 10, (unsigned int) 10,
      0, 360*64);
  }
}

void
get_vlab_colors(xg, vlab_colors)
  xgobidata *xg;
  WidgetColors *vlab_colors;
{
  int j;

  XtVaGetValues(xg->varlabw[0],
    XtNforeground, &vlab_colors->fg,
    XtNbackground, &vlab_colors->bg,
    XtNborderColor, &vlab_colors->border, NULL);

  for (j=1; j<xg->ncols_used; j++)
  {
    if (xg->varchosen[j] == False)
    {
      XtVaGetValues(xg->varlabw[j],
        XtNforeground, &vlab_colors->fg,
        XtNbackground, &vlab_colors->bg,
        XtNborderColor, &vlab_colors->border, NULL);
      break;
    }
  }
}

void
refresh_vlab(varno, xg)
/*
 * this routine will check what highlighting the variable has
 * and will change it or leave it the same according to whether
 * caller wants a highlighted or a dehighlighted label.
*/
  int varno;
  xgobidata *xg;
{
  static int init = 0;
  Boolean chosen = xg->varchosen[varno];
  static WidgetColors vlab_colors;

  if (!init)
  {
    get_vlab_colors(xg, &vlab_colors);
    init = 1;
  }

  if (chosen)
  {
    XtVaSetValues(xg->varlabw[varno],
      XtNforeground, vlab_colors.bg,
      XtNbackground, vlab_colors.fg, NULL);
  }
  else
  {
    XtVaSetValues(xg->varlabw[varno],
      XtNforeground, vlab_colors.fg,
      XtNbackground, vlab_colors.bg, NULL);
  }
}

void
refresh_vbox(xg, varno, do_label)
  xgobidata *xg;
  int varno;
  int do_label;
/*
 * the purpose of this function is to refresh the variable bars
 * appropriately when changing between modes
 * and to refresh the labels.
*/
{
  int x, y;
  Widget vwsp = xg->vardraww[varno];
  Window vwin = XtWindow(vwsp);
  unsigned int r = xg->radius;
  XGCValues *gcv, gcv_inst;

  XGetGCValues(display, varpanel_xor_GC, GCCapStyle|GCJoinStyle, &gcv_inst);
  gcv = &gcv_inst;
  XClearWindow(display, vwin);

  if (xg->varchosen[varno])
    XSetLineAttributes(display, varpanel_xor_GC, 2, LineSolid,
      gcv->cap_style, gcv->join_style);

  XDrawArc(display, vwin, varpanel_xor_GC, 0, 0,
    (unsigned int) 2*r, (unsigned int) 2*r,
    0, 360*64);

  if (xg->is_xyplotting)
  {
    if (xg->varchosen[varno])
    {
      if (varno == xg->xy_vars.x)
        ChooseX(vwin, r, r);
      else if (varno == xg->xy_vars.y)
        ChooseY(vwin, r, r);
    }
  }

  else if (xg->is_dotplotting)
  {
    if (xg->varchosen[varno])
    {
      if (varno == xg->dotplot_vars.y)
        ChooseY(vwin, r, r);
    }
  }

  else if (xg->is_spinning)
  {
    if (xg->varchosen[varno])
    {
      if (xg->is_spin_type.yaxis)   /* Using vertical axis */
      {
        if (varno == xg->spin_vars.x)
          ChooseX(vwin, r, r * xg->ocost.y/PRECISION2);
        else if (varno == xg->spin_vars.z)
          ChooseX(vwin, r, r * xg->osint.y/PRECISION2);
        else if (varno == xg->spin_vars.y)
          ChooseY(vwin, r, r);
      }
      else if (xg->is_spin_type.xaxis)  /* Using horizontal axis */
      {
        if (varno == xg->spin_vars.x)
          ChooseX(vwin, r, r);
        else if (varno == xg->spin_vars.z)
          ChooseY(vwin, r, r * xg->osint.x/PRECISION2);
        else if (varno == xg->spin_vars.y)
          ChooseY(vwin, r, r * xg->ocost.x/PRECISION2);
      }
      else if (xg->is_spin_type.oblique)
      {
        float fr = (float) r;
        if (varno == xg->spin_vars.x)
        {
          xg->xax.x = INT(fr * xg->Rmat0[0][0]);
          xg->xax.y = INT(fr * xg->Rmat0[1][0]);
          ChooseVar(vwin, r, xg->xax.x, xg->xax.y);
        }
        else if (varno == xg->spin_vars.y)
        {
          xg->yax.x = INT(fr * xg->Rmat0[0][1]);
          xg->yax.y = INT(fr * xg->Rmat0[1][1]);
          ChooseVar(vwin, r, xg->yax.x, xg->yax.y);
        }
        else if (varno == xg->spin_vars.z)
        {
          xg->zax.x = INT(fr * xg->Rmat0[0][2]);
          xg->zax.y = INT(fr * xg->Rmat0[1][2]);
          ChooseVar(vwin, r, xg->zax.x, xg->zax.y);
        }
      }
      if (varno == last_touched)
        draw_last_touched(xg);
    }
  }
  else if (xg->is_touring)
  {
    x = INT(FLOAT(r) * xg->uold[0][varno]);
    y = INT(FLOAT(r) * xg->uold[1][varno]);
    ChooseVar(vwin, r, x, y);
    if (varno == xg->manip_var)
        draw_manip_var(xg);
    draw_frozen_var(xg);
  }
  else if (xg->is_corr_touring)
  {
    calc_var_xy(xg, (int) r, varno, &x, &y);
    ChooseVar(vwin, r, x, y);
    if (varno == xg->corr_xmanip_var || varno == xg->corr_ymanip_var)
      draw_cmanip_var(xg);
    draw_cfrozen_var(xg);
  }

  if (xg->varchosen[varno])
    XSetLineAttributes(display, varpanel_xor_GC, 0, LineSolid,
      gcv->cap_style, gcv->join_style);

  if (do_label)
    refresh_vlab(varno, xg);
}

void
refresh_vboxes(xg)
  xgobidata *xg;
{
  int j;
  int do_label = 1;

  for (j=0; j<xg->ncols_used; j++)
    refresh_vbox(xg, j, do_label);
  if (xg->is_spinning)
    draw_last_touched(xg);
  if (xg->is_touring)
  {
    draw_manip_var(xg);
    draw_frozen_var(xg);
  }
  if (xg->is_corr_touring)
  {
    draw_cmanip_var(xg);
    draw_cfrozen_var(xg);
  }
}

int
xy_varselect(varno, event, xg)
/*
 * This is used only during simple xy plotting.
*/
  int varno;
  XButtonEvent *event;
  xgobidata *xg;
{
  int prev;
  int newvar = 1;
/*
 * If it's a left click ...
*/
  if (event->button == 1)
  {
    /*
     * If it's the current y variable, switch the two.
    */
    if (varno == xg->xy_vars.y)
    {
      xg->xy_vars.y = xg->xy_vars.x;
      xg->xy_vars.x = varno;
      refresh_vbox(xg, xg->xy_vars.y, 0);
      refresh_vbox(xg, xg->xy_vars.x, 0);
    }
    /*
     * Otherwise choose the new x.
    */
    else
    {
      prev = xg->xy_vars.x;
      xg->varchosen[prev] = False;
      xg->xy_vars.x = varno;
      xg->varchosen[varno] = True;
      refresh_vbox(xg, prev, 1);
      refresh_vbox(xg, xg->xy_vars.x, 1);
    }
  }
/*
 * If it's a middle click ...
*/
  else if (event->button == 2)
  {
    /*
     * If it's the current x variable, switch the two.
    */
    if (varno == xg->xy_vars.x)
    {
      xg->xy_vars.x = xg->xy_vars.y;
      xg->xy_vars.y = varno;
      refresh_vbox(xg, xg->xy_vars.y, 0);
      refresh_vbox(xg, xg->xy_vars.x, 0);
    }
    /*
     * Otherwise choose the new y.
    */
    else
    {
      prev = xg->xy_vars.y;
      xg->varchosen[prev] = False;
      xg->xy_vars.y = varno;
      xg->varchosen[varno] = True;
      refresh_vbox(xg, prev, 1);
      refresh_vbox(xg, xg->xy_vars.y, 1);
    }
  }
  return(newvar);
}

/********** Rotation **************/

/* ARGSUSED */
XtActionProc
ShowTarget(w, event, params, nparams)
  Widget w;
  XEvent *event;
  String *params;
  Cardinal nparams;
{
/*
 * If the cursor is outside the variable selection panel, check
 * for its entry.
*/
  XCrossingEvent *evnt = (XCrossingEvent *) event;

  if (xgobi.is_spinning &&
    !cursor_in_panel &&
    evnt->type == EnterNotify)
  {
    cursor_in_panel = 1;
    draw_last_touched(&xgobi);
  }
}

/* ARGSUSED */
XtActionProc
DontShowTarget(w, event, params, nparams)
  Widget w;
  XEvent *event;
  String *params;
  Cardinal nparams;
{
/*
 * If the cursor is inside the variable selection panel, look for
 * LeaveNotify events.  The detail member of the XCrossing Event
 * structure is NotifyInferior if the cursor has entered the window
 * belonging to a child widget of the window receiving the event.
*/
  XCrossingEvent *evnt = (XCrossingEvent *) event;

  if (xgobi.is_spinning &&
    cursor_in_panel &&
    evnt->type == LeaveNotify &&
    evnt->detail != NotifyInferior)
  {
    cursor_in_panel = 0;
    refresh_vbox(&xgobi, last_touched, 0);
  }
}

/*
void
track_cursor(event, varwindow, xg)
  XEvent *event;
  Window varwindow;
  xgobidata *xg;
{
  XCrossingEvent *evnt = (XCrossingEvent *) event;
  if (!cursor_in_panel)
  {
    if (evnt->type == EnterNotify &&
      evnt->window == varwindow)
    {
      cursor_in_panel = 1;
      draw_last_touched(xg);
    }
  }
  else
  {
    if (evnt->type == LeaveNotify &&
      evnt->window == varwindow &&
      evnt->detail != NotifyInferior)
    {
      cursor_in_panel = 0;
      refresh_vbox(xg, last_touched, 0);
    }
  }
}
*/

void
reset_last_touched(xg)
  xgobidata *xg;
{
/*
 * Reset the last_touched variable when moving between
 * y axis and x axis rotation.
*/
  if (xg->is_spin_type.yaxis)
  {
    if (last_touched == xg->spin_vars.y)
      last_touched = xg->spin_vars.x;
  }
  else if (xg->is_spin_type.xaxis)
  {
    if (last_touched == xg->spin_vars.x)
      last_touched = xg->spin_vars.y;
  }
}

void
spin_var_lines(xg)
  xgobidata *xg;
{
  Window vwin;
  XGCValues *gcv, gcv_inst;
  XGetGCValues(display, varpanel_xor_GC, GCCapStyle|GCJoinStyle, &gcv_inst);
  gcv = &gcv_inst;

  XSetLineAttributes(display, varpanel_xor_GC, 2, LineSolid,
    gcv->cap_style, gcv->join_style);
    
  if (xg->is_spin_type.yaxis)   /* Using vertical axis */
  {
    vwin = XtWindow(xg->vardraww[xg->spin_vars.x]);
    ChooseX(vwin, xg->radius,
        xg->radius * xg->ocost.y/PRECISION2);
    ChooseX(vwin, xg->radius,
        xg->radius * xg->icost.y/PRECISION2);

    vwin = XtWindow(xg->vardraww[xg->spin_vars.z]);
    ChooseX(vwin, xg->radius,
        xg->radius * xg->osint.y/PRECISION2);
    ChooseX(vwin, xg->radius,
        xg->radius * xg->isint.y/PRECISION2);

    xg->ocost.y = xg->icost.y;
    xg->osint.y = xg->isint.y;
  }
  /*
   * Using horizontal axis in the plane of the screen
  */
  else if (xg->is_spin_type.xaxis)
  {
    vwin = XtWindow(xg->vardraww[xg->spin_vars.y]);
    ChooseY(vwin, xg->radius,
        xg->radius * xg->ocost.x/PRECISION2);
    ChooseY(vwin, xg->radius,
        xg->radius * xg->icost.x/PRECISION2);

    vwin = XtWindow(xg->vardraww[xg->spin_vars.z]);
    ChooseY(vwin, xg->radius,
        xg->radius * xg->osint.x/PRECISION2);
    ChooseY(vwin, xg->radius,
        xg->radius * xg->isint.x/PRECISION2);

    xg->ocost.x = xg->icost.x;
    xg->osint.x = xg->isint.x;
  }

  XSetLineAttributes(display, varpanel_xor_GC, 0, LineSolid,
    gcv->cap_style, gcv->join_style);

  draw_last_touched(xg);
}

int
spin_varsel_oblique(varno, xg)
  int varno;
  xgobidata *xg;
{
  int prev, newvar = 1;

  if (varno == xg->spin_vars.x ||
    varno == xg->spin_vars.y ||
    varno == xg->spin_vars.z)
   {
    prev = last_touched;
    last_touched = varno;
    refresh_vbox(xg, prev, 0);
    newvar = 0;
  }
  else
  {
    if (last_touched == xg->spin_vars.x)
      xg->spin_vars.x = varno;
    else if (last_touched == xg->spin_vars.y)
      xg->spin_vars.y = varno;
    else if (last_touched == xg->spin_vars.z)
      xg->spin_vars.z = varno;

    prev = last_touched;
    xg->varchosen[prev] = False;
    refresh_vbox(xg, prev, 1);

    last_touched = varno;
    xg->varchosen[varno] = True;
  }
  return(newvar);
}

int
spin_varsel_vert(varno, event, xg)
  int varno;
  XButtonEvent *event;
  xgobidata *xg;
{
/*
 * Here x and z are plotted horizontally; y is plotted vertically.
*/
  int prev, newvar = 1;

  if (event->button == 1)
  {
  /*
   * Is it already a chosen x variable?  If so, set last_touched = varno.
  */
    if (varno == xg->spin_vars.x || varno == xg->spin_vars.z)
      last_touched = varno;
    else if (varno == xg->spin_vars.y)
    {
    /*
     * Or is it the chosen y variable?  If so,
     * exchange y and last_touched.
    */
      if (last_touched == xg->spin_vars.x)
      {
        xg->spin_vars.y = xg->spin_vars.x;
        last_touched = xg->spin_vars.x = varno;
      }
      else if (last_touched == xg->spin_vars.z)
      {
        xg->spin_vars.y = xg->spin_vars.z;
        last_touched = xg->spin_vars.z = varno;
      }
    }
/*
 * Else, the new variable replaces last_touched.
*/
    else
    {  /* last_touched <- varno */
      if (last_touched == xg->spin_vars.x)
      {
        prev = xg->spin_vars.x;
        last_touched = xg->spin_vars.x = varno;
      }
      else if (last_touched == xg->spin_vars.z)
      {
        prev = xg->spin_vars.z;
        last_touched = xg->spin_vars.z = varno;
      }
      xg->varchosen[prev] = False;
      refresh_vbox(xg, prev, 1);

      xg->varchosen[varno] = True;
    }
  }
  else if (event->button == 2)
  {
    /*
     * Is it already a chosen y variable?  If so, ignore.
    */
    if (varno == xg->spin_vars.y)
      newvar = 0;
    /*
     * Or is it a chosen x variable?  If so,
     * exchange y for the chosen x.
    */
    else if (varno == xg->spin_vars.x)
    {
      last_touched = xg->spin_vars.x = xg->spin_vars.y;
      xg->spin_vars.y = varno;
    }
    else if (varno == xg->spin_vars.z)
    {
      last_touched = xg->spin_vars.z = xg->spin_vars.y;
      xg->spin_vars.y = varno;
    }
    else
    {
      prev = xg->spin_vars.y;
      xg->spin_vars.y = varno;

      xg->varchosen[prev] = False;
      refresh_vbox(xg, prev, 1);
      xg->varchosen[varno] = True;
    }
  }
  return(newvar);
}

int
spin_varsel_horiz(varno, event, xg)
  int varno;
  XButtonEvent *event;
  xgobidata *xg;
{
  int prev, newvar = 1;

  if (event->button == 1)
  {
    /*
     * Is it already the chosen x variable?  If so,
     * ignore the click.
    */
    if (varno == xg->spin_vars.x)
        newvar = 0;
    else if (varno == xg->spin_vars.y)
    {
    /*
     * Or is it a current y?  If so, exchange the x and the y.
    */
      last_touched = xg->spin_vars.y = xg->spin_vars.x;;
      xg->spin_vars.x = varno;
    }
    else if (varno == xg->spin_vars.z)
    {
      last_touched = xg->spin_vars.z = xg->spin_vars.x;
      xg->spin_vars.x = varno;
    }
    else
    {
      prev = xg->spin_vars.x;
      xg->spin_vars.x = varno;
      xg->varchosen[prev] = False;
      refresh_vbox(xg, prev, 1);
      xg->varchosen[varno] = True;
    }
  }
  else if (event->button == 2)
  {
/*
 * Is it already a chosen y variable?  If so, exchange y and z.
*/
    if (varno == xg->spin_vars.y || varno == xg->spin_vars.z)
      last_touched = varno;
/*
 * Or is it the x variable?  If so, exchange x and last_touched.
*/
    else if (varno == xg->spin_vars.x)
    {
      if (last_touched == xg->spin_vars.y)
      {
        xg->spin_vars.x = xg->spin_vars.y;
        last_touched = xg->spin_vars.y = varno;
      }
      else if (last_touched == xg->spin_vars.z)
      {
        xg->spin_vars.x = xg->spin_vars.z;
        last_touched = xg->spin_vars.z = varno;
      }
    }
/*
 * Else, the new variable replaces last_touched.
*/
    else
    {
      if (last_touched == xg->spin_vars.y)
      {
        prev = xg->spin_vars.y;
        last_touched = xg->spin_vars.y = varno;
      }
      else if (last_touched == xg->spin_vars.z)
      {
        prev = xg->spin_vars.z;
        last_touched = xg->spin_vars.z = varno;
      }
      xg->varchosen[prev] = False;
      refresh_vbox(xg, prev, 1);
      xg->varchosen[varno] = True;
    }
  }
  return(newvar);
}

int
spin_varselect(varno, event, xg)
/*
 * This is the routine called during rotation to handle changes in
 * variables entering the plot.  Call refresh_vbox() with instructions
 * to call refresh_vlab().
*/
  int varno;
  XButtonEvent *event;
  xgobidata *xg;
{
  int newvar = 1;

  if (xg->is_spin_type.yaxis)
    newvar = spin_varsel_vert(varno, event, xg);
  else if (xg->is_spin_type.xaxis)
    newvar = spin_varsel_horiz(varno, event, xg);
  else if (xg->is_spin_type.oblique)
    newvar = spin_varsel_oblique(varno, xg);

  if (newvar)
    refresh_vboxes(xg);

  return(newvar);
}

void
draw_ob_var_lines(xg)
  xgobidata *xg;
{
  Window vwin;
  int r = xg->radius;
  float fr = (float) xg->radius;
  XGCValues *gcv, gcv_inst;
  XGetGCValues(display, varpanel_xor_GC, GCCapStyle|GCJoinStyle, &gcv_inst);
  gcv = &gcv_inst;

  XSetLineAttributes(display, varpanel_xor_GC, 2, LineSolid,
    gcv->cap_style, gcv->join_style);

  vwin = XtWindow(xg->vardraww[xg->spin_vars.x]);
  ChooseVar(vwin, r, xg->xax.x, xg->xax.y);
  xg->xax.x = INT(fr * xg->Rmat0[0][0]);
  xg->xax.y = INT(fr * xg->Rmat0[1][0]);
  ChooseVar(vwin, r, xg->xax.x, xg->xax.y);

  vwin = XtWindow(xg->vardraww[xg->spin_vars.y]);
  ChooseVar(vwin, r, xg->yax.x, xg->yax.y);
  xg->yax.x = INT(fr * xg->Rmat0[0][1]);
  xg->yax.y = INT(fr * xg->Rmat0[1][1]);
  ChooseVar(vwin, r, xg->yax.x, xg->yax.y);

  vwin = XtWindow(xg->vardraww[xg->spin_vars.z]);
  ChooseVar(vwin, r, xg->zax.x, xg->zax.y);
  xg->zax.x = INT(fr * xg->Rmat0[0][2]);
  xg->zax.y = INT(fr * xg->Rmat0[1][2]);
  ChooseVar(vwin, r, xg->zax.x, xg->zax.y);

  XSetLineAttributes(display, varpanel_xor_GC, 0, LineSolid,
    gcv->cap_style, gcv->join_style);

  draw_last_touched(xg);
}

/********** Grand Tour ************/

void
tour_var_lines(xg)
  xgobidata *xg;
{
  int j;
  int x, y;
  int r = xg->radius;
  float fr = (float) xg->radius;
  Window vwin;
  XGCValues *gcv, gcv_inst;
  XGetGCValues(display, varpanel_xor_GC, GCCapStyle|GCJoinStyle, &gcv_inst);
  gcv = &gcv_inst;

  if (xg->is_touring)
  {
    for (j=0; j<xg->ncols_used; j++)
    {
      if (xg->varchosen[j])
        XSetLineAttributes(display, varpanel_xor_GC, 2, LineSolid,
          gcv->cap_style, gcv->join_style);
    
      vwin = XtWindow(xg->vardraww[j]);
      x = INT(fr * xg->uold[0][j]);
      y = INT(fr * xg->uold[1][j]);
      ChooseVar(vwin, r, x, y);
      x = INT(fr * xg->u[0][j]);
      y = INT(fr * xg->u[1][j]);
      ChooseVar(vwin, r, x, y);
      xg->uold[0][j] = xg->u[0][j];
      xg->uold[1][j] = xg->u[1][j];

      if (xg->varchosen[j])
        XSetLineAttributes(display, varpanel_xor_GC, 0, LineSolid,
          gcv->cap_style, gcv->join_style);
    }
  }

}

int
add_variable(xg, varno)
  xgobidata *xg;
  int varno;
{
  int found, i, j;

  xg->varchosen[varno] = True;
  found = 0;
  if (varno < xg->tour_vars[0])
  {
    for (j=xg->numvars_t; j>0; j--)
      xg->tour_vars[j] = xg->tour_vars[j-1];
    xg->tour_vars[0] = varno;
    xg->numvars_t += 1;
    found = 1;
  }
  i = 0;
  while (!found && i < xg->numvars_t-1 )
  {
    if ( varno > xg->tour_vars[i]  &&
         varno < xg->tour_vars[i+1] )
    {
      for (j=xg->numvars_t; j>i; j--)
        xg->tour_vars[j+1] = xg->tour_vars[j];
       xg->tour_vars[i+1] = varno;
       xg->numvars_t += 1;
       found = 1;
    }
    i++;
  }
  if (!found && (varno > xg->tour_vars[xg->numvars_t-1]))
  {
    xg->tour_vars[xg->numvars_t] = varno;
    xg->numvars_t += 1;
    found = 1;
  }

  return(found);
}

void
remove_variable(xg, varno)
  xgobidata *xg;
  int varno;
{
  int i, j, found, itmp;

  /* for frozen variables we want this to remain true -
     if this routine is ever called for another purpose this line will 
     need to be changed */
  xg->varchosen[varno] = True;
  i = 0;
  found = 0;
  while (!found)
  {
    if (varno == xg->tour_vars[i])
    {
      itmp = i;

      xg->numvars_t -= 1;
      for (j=itmp; j<xg->numvars_t; j++)
        xg->tour_vars[j] = xg->tour_vars[j+1];

      found = 1;
      refresh_vbox(xg, varno, 1);
    }
    i++;
  } /* end while */

}

int
tour_varselect(varno, event, xg)
/*
 * This routine is called during grand touring to handle changes
 * in variables included in plot. It calls refresh_vbox() with
 * further call to refresh_vlab() to handle variable box refreshes
 * and label highlighting.
*/
  int varno;
  XButtonEvent *event;
  xgobidata *xg;
{
  int newvar = 0;
  int i, j;
  int selected = 0;
  int found, itmp;

  if (event->state == 0)
  {
    xg->run_tour_proc = False;
    for (i=0; i<xg->numvars_t; i++)
      if (varno == xg->tour_vars[i])
        selected = 1;
    /*
     * deselect variable if numvars_t will not be reduced to 1
    */
    if (selected)
    {
      if (xg->numvars_t > 2)
      {
        xg->varchosen[varno] = False;
        i = 0;
        found = 0;
        while (!found)
        {
          if (varno == xg->tour_vars[i])
          {
            itmp = i;
            /*
             * Make current position the new starting plane
             * and make the new ending plane the current position
             * with 0 in indx'th place of tour_vars.
            */
            init_basis(xg);
  /*
   * This is necessary to prevent problems when deselecting either variable
   * X or Y in the first projection. Now if either of the two variables
   * is deselected the first variable in the tour_vars that is not
   * contributing to the current projection becomes the new X or Y variable
  */
            if (xg->nhist_list == 1 && xg->step == 0.0)
            {
              check_deselection(varno, xg);
            }
  /*
   * all other cases of deselection
  */
            else
            {
  /*
   * generate new ending basis with deselected variable zeroed out, only
   * necessary in the case of not being princ_comp of optimz because these
   * cases are dealt with later.
  */
              if (!(xg->is_pp_optimz || xg->is_princ_comp))
              {
                for (i=0; i<2; i++)
                  xg->u1[i][varno] = 0.0;
              }
            }
            /*
             * now refresh vboxes and labels and tour_vars
            */
            xg->numvars_t -= 1;
            for (j=itmp; j<xg->numvars_t; j++)
              xg->tour_vars[j] = xg->tour_vars[j+1];

            /* dfs, testing */
             if (xg->is_princ_comp)
               refresh_all_var_labels(xg);
            /*              */

            found = 1;
            refresh_vbox(xg, varno, 1);
          }
          i++;
        } /* end while */
  
  /*
   * keep_sphered data updated to active variables
  */
        if (xg->is_princ_comp && xg->ncols_used >= 2)
        {
          if (update_vc_active_and_do_svd(xg))
            spherize_data(xg);
          else
            copy_tform_to_sphered(xg);
        }  
  /*
   * Cater for the case of principal components or optimize is on, when
   * the behaviour should be to completely blank out the variable
   * immediately, rather than fade it out.
  */
        if (xg->is_princ_comp || xg->is_pp_optimz)
        {
          xg->u[0][varno] = 0.0;
          norm(xg->u[0], xg->ncols_used);
          xg->u[1][varno] = 0.0;
          norm(xg->u[1], xg->ncols_used);
          gram_schmidt(xg->u[0], xg->u[1], xg->ncols_used);
          init_basis(xg);
          copy_basis(xg->u0, xg->u1, xg->ncols_used);
          copy_u0_to_pd0(xg);
          copy_basis(xg->u0, xg->v1, xg->ncols_used);
          update_lims(xg);
          update_world(xg);
          zero_tau(xg);
          zero_princ_angles(xg);
          world_to_plane(xg);
          plane_to_screen(xg);
          plot_once(xg);
/* dfs, testing */
          reset_var_labels(xg, !xg->is_princ_comp);
          reset_one_label(xg, varno, 1);
          tour_var_lines(xg);
          xg->new_direction_flag = True;
        }
  
  /* recalculate section variables */
        if (xg->is_tour_section)
        {
          set_tour_section_eps(0., xg, 0);
          tour_section_calcs(xg, 0);
        }
  /*
   * reset pp_plot if in projection pursuit, because the scale changes
   * depending on variables included.
  */
        if (xg->is_pp)
        {
          reset_pp_plot();
          xg->recalc_max_min = True;
        }
        /*
         * zero tau to force new tour from current position
        */
        zero_tau(xg);
        if (!xg->is_princ_comp)
        {
          set_fading_var(2);
          set_sens_localscan(xg, 0);
          reset_backtrack_cmd(xg, 0, 0, 0, 0);
          if (xg->numvars_t == 2)
          {
            set_ready_to_stop_now(0);
            set_counting_to_stop(1);
          }
        }
      }
      if (xg->tour_cont_fact == infinite)
        xg->new_direction_flag = True;
    }
  
    else  /* if (!selected) */
    { /* add variable */
      if (!var_frozen(xg, varno)) /* don't add if var is frozen */
      {
        found = add_variable(xg, varno);
  
      if (xg->is_princ_comp && xg->ncols_used >= 2)
      {
        if (update_vc_active_and_do_svd(xg))
          spherize_data(xg);
        else
          copy_tform_to_sphered(xg);
      }
      if (xg->is_princ_comp && check_singular_vc())
      {
        i = 0;
        found = 0;
        /* take the variable out of the sphered group - 
         * warning message is generated. awful programming
         * but this is the way that it is set up currently.
        */
        while (!found)
        {
          if (varno == xg->tour_vars[i])
          {
            itmp = i;
            xg->numvars_t -= 1;
            for (j=itmp; j<xg->numvars_t; j++)
              xg->tour_vars[j] = xg->tour_vars[j+1];
            found = 1;
          }
          i++;
        } /* end while */
        if (update_vc_active_and_do_svd(xg))
          spherize_data(xg);
        else
          copy_tform_to_sphered(xg);
      }
      else
      {
        if (xg->is_princ_comp)
        {
          init_basis(xg);
          update_lims(xg);
          update_world(xg);
          world_to_plane(xg);
          plane_to_screen(xg);
          plot_once(xg);
          tour_var_lines(xg);
/* dfs, testing */
          /*reset_one_label(xg, varno, 0);*/
          reset_var_labels(xg, !xg->is_princ_comp);
        }
        refresh_vbox(xg, varno, 1);
          
    /* recalculate section variables */
        if (xg->is_tour_section)
        {
          set_tour_section_eps(0., xg, 0);
          tour_section_calcs(xg, 0);
        }
    
        if (xg->is_pp)
        {
          reset_pp_plot();
          xg->recalc_max_min = 1;
        }
    
        /*
         * zero tau to force new tour from current position
        */
        zero_tau(xg);
    
        if (xg->is_pp_optimz)
          xg->new_direction_flag = True;
    
  	if (!found)
          (void) fprintf(stderr, "error: didn't find indexed variable \n");
    
        if (xg->tour_cont_fact == infinite)
          xg->new_direction_flag = True;
      }
      }
    }
    if (tour_on(xg) == True)
      start_tour_proc(xg);
  }
  else if (event->state == 1)
  {
    set_manip_var(xg, varno);
  }
  else if (event->state == 4)
  {
    set_frozen_var(xg, varno);
  }
  
  return(newvar);
}

void
set_varpanel_for_receive_tour(xg)
  xgobidata *xg;
{
  int j, var;
  void refresh_vlab();

  if (xg->tour_link_state == receive)  /* If receive was just turned on ... */
  {
    for (j=0; j<xg->ncols_used; j++)
    {
      if (xg->varchosen[j])
      {
        xg->varchosen[j] = False;
        refresh_vbox(xg, j, True);
      }
    }
  }
  else  /* else if receive was just turned off ... */
  {
    for (j=0; j<xg->numvars_t; j++)
    {
      var = xg->tour_vars[j];
      xg->varchosen[var] = True;
      refresh_vbox(xg, j, True);
    }
  }
}

void
map_group_var(xg)
  xgobidata *xg;
{
  XtVaSetValues(xg->varlabw[xg->ncols-1],
    XtNlabel, (String) xg->collab_tform[xg->ncols-1],
    NULL);
  XtMapWidget(xg->varboxw[xg->ncols-1]);
}

void
carry_spin_vars(xg)
  xgobidata *xg;
{
  int j;

  if (xg->is_dotplotting)
  {
    xg->spin_vars.y = xg->dotplot_vars.y;
    for (j=0; j<xg->ncols; j++)
      if (j != xg->spin_vars.y)
      {
        xg->spin_vars.x = j;
        break;
      }
    for (j=0; j<xg->ncols; j++)
      if (j != xg->spin_vars.x && j != xg->spin_vars.y)
      {
        xg->spin_vars.z = j;
        break;
      }
  }
  else if (xg->is_xyplotting)
  {
    xg->spin_vars.x = xg->xy_vars.x;
    xg->spin_vars.y = xg->xy_vars.y;
    for (j=0; j<xg->ncols; j++)
      if (j != xg->spin_vars.x && j != xg->spin_vars.y)
      {
        xg->spin_vars.z = j;
        break;
      }
  }
  else if (xg->is_touring)
  {
    xg->spin_vars.x = xg->tour_vars[0];
    xg->spin_vars.z = xg->tour_vars[2];
    xg->spin_vars.y = xg->tour_vars[1];
  }
  else if (xg->is_corr_touring)
  {
    xg->spin_vars.x = xg->corr_xvars[0];
    xg->spin_vars.z = xg->corr_xvars[1];
    xg->spin_vars.y = xg->corr_yvars[0];
  }

  /*
   * The target variable may have been deselected.
  */
  if (last_touched != xg->spin_vars.x &&
    last_touched != xg->spin_vars.y &&
    last_touched != xg->spin_vars.z)
  {
    last_touched = xg->spin_vars.z;
  }

  reinit_spin(xg);
  reset_last_touched(xg);
}

void
carry_tour_vars(xg)
  xgobidata *xg;
{
  int i, inc;

  if (xg->is_dotplotting)
  {
    inc = 0;
    for (i=0; i<xg->numvars_t; i++)
    {
      if (xg->tour_vars[i] == xg->dotplot_vars.y)
      {
        inc = 1;
        break;
      }
    }

    if (inc)
    {
      if (xg->tour_vars[1] == xg->dotplot_vars.y)
        ;
      else
      {
        for (i=0; i<xg->numvars_t; i++)
        {
          if (xg->tour_vars[i] == xg->dotplot_vars.y)
          {
            xg->tour_vars[i] = xg->tour_vars[1];
            xg->tour_vars[1] = xg->dotplot_vars.y;
            break;
          }
        }
      }
    } else {
      xg->tour_vars[xg->numvars_t] = xg->tour_vars[1];
      xg->tour_vars[1] = xg->dotplot_vars.y;
      xg->numvars_t++;
    }
  }

  else if (xg->is_xyplotting)
  {
  /*
   * First see if xy_vars.x is included.
  */
    inc = 0;
    for (i=0; i<xg->numvars_t; i++)
    {
      if (xg->tour_vars[i] == xg->xy_vars.x)
      {
        inc = 1;
        break;
      }
    }

    if (inc)
    {
      if (xg->tour_vars[0] == xg->xy_vars.x)
        ;
      else
      {
        for (i=0; i<xg->numvars_t; i++)
        {
          if (xg->tour_vars[i] == xg->xy_vars.x)
          {
            xg->tour_vars[i] = xg->tour_vars[0];
            xg->tour_vars[0] = xg->xy_vars.x;
            break;
          }
        }
      }
    } else {
      xg->tour_vars[xg->numvars_t] = xg->tour_vars[0];
      xg->tour_vars[0] = xg->xy_vars.x;
      xg->numvars_t++;
    }

  /*
   * Then see if xy_vars.y is included.
  */
    inc = 0;
    for (i=0; i<xg->numvars_t; i++)
    {
      if (xg->tour_vars[i] == xg->xy_vars.y)
      {
        inc = 1;
        break;
      }
    }

    if (inc)
    {
      if (xg->tour_vars[1] == xg->xy_vars.y)
        ;
      else
      {
        for (i=0; i<xg->numvars_t; i++)
        {
          if (xg->tour_vars[i] == xg->xy_vars.y)
          {
            xg->tour_vars[i] = xg->tour_vars[1];
            xg->tour_vars[1] = xg->xy_vars.y;
            break;
          }
        }
      }
    } else {
      xg->tour_vars[xg->numvars_t] = xg->tour_vars[1];
      xg->tour_vars[1] = xg->xy_vars.y;
      xg->numvars_t++;
    }
  }

  else if (xg->is_spinning)
  {
    /*
     * Switch to three variables ...
    */
    xg->tour_vars[0] = xg->spin_vars.x;
    xg->tour_vars[1] = xg->spin_vars.y;
    xg->tour_vars[2] = xg->spin_vars.z;
    xg->numvars_t = 3;
    reinit_tour(xg);
  }
  else if (xg->is_corr_touring)
  {
    xg->numvars_t = 0;

    for (i=0; i<xg->ncorr_xvars; i++)
    {
      xg->tour_vars[i] = xg->corr_xvars[i];
      xg->numvars_t++ ;
    }
    for (i=0;i<xg->ncorr_yvars;i++)
    {
      xg->tour_vars[i+xg->ncorr_xvars] = xg->corr_yvars[i];
      xg->numvars_t++ ;
    }
    reinit_tour(xg);
  }
}

void
carry_xyplot_vars(xg)
  xgobidata *xg;
{
/*
 * When entering XYPlot from another mode with carry_vars
 * set to True, do this ...
*/
  if (xg->is_dotplotting)
  {
    if (xg->xy_vars.y == xg->dotplot_vars.y)
      ;
    else if (xg->xy_vars.x == xg->dotplot_vars.y)
    {
      xg->xy_vars.x = xg->xy_vars.y;
      xg->xy_vars.y = xg->dotplot_vars.y;
    }
  }
  else if (xg->is_spinning)
  {
    xg->xy_vars.y = xg->spin_vars.y;
    xg->xy_vars.x = xg->spin_vars.x;
  }
  else if (xg->is_touring)
  {
    xg->xy_vars.y = xg->tour_vars[1];
    xg->xy_vars.x = xg->tour_vars[0];
  }
  else if (xg->is_corr_touring)
  {
    xg->xy_vars.x = xg->corr_xvars[0];
    xg->xy_vars.y = xg->corr_yvars[0];
  }
}

void
carry_dotplot_vars(xg)
  xgobidata *xg;
{
/*
 * When entering DotPlot from another mode with carry_vars
 * set to True, do this ...
*/
  if (xg->is_xyplotting)
    xg->dotplot_vars.y = xg->xy_vars.y;
  else if (xg->is_spinning)
    xg->dotplot_vars.y = xg->spin_vars.y;
  else if (xg->is_touring)
    xg->dotplot_vars.y = xg->tour_vars[1];
  else if (xg->is_corr_touring)
    xg->dotplot_vars.y = xg->corr_yvars[0];
}

void
carry_corr_vars(xg)
  xgobidata *xg;
{
  int i, inc;

  if (xg->is_dotplotting)
  {
    inc = 0;
    for (i=0; i<xg->ncorr_yvars; i++)
    {
      if (xg->corr_xvars[i] == xg->dotplot_vars.y)
      {
        inc = 1;
        break;
      }
    }

    if (inc)
    {
      if (i != 0)
      {
        xg->corr_yvars[i] = xg->corr_yvars[0];
        xg->corr_yvars[0] = xg->dotplot_vars.y;
      }
    } else {
      xg->corr_yvars[xg->ncorr_yvars] = xg->corr_yvars[0];
      xg->corr_yvars[0] = xg->dotplot_vars.y;
      xg->ncorr_yvars++;
    }
  }

  else if (xg->is_xyplotting)
  {
  /*
   * First see if xy_vars.x is included.
  */
    inc = 0;
    for (i=0; i<xg->ncorr_xvars; i++)
    {
      if (xg->corr_xvars[i] == xg->xy_vars.x)
      {
        inc = 1;
        break;
      }
    }

    if (inc)
    {
      if (i != 0)
      {
        xg->corr_xvars[i] = xg->corr_xvars[0];
        xg->corr_xvars[0] = xg->xy_vars.x;
      }
    } else {
      xg->corr_xvars[xg->ncorr_xvars] = xg->corr_xvars[0];
      xg->corr_xvars[0] = xg->xy_vars.x;
      xg->ncorr_xvars++;
    }

  /*
   * Then see if xy_vars.y is included.
  */
    inc = 0;
    for (i=0; i<xg->ncorr_yvars; i++)
    {
      if (xg->corr_yvars[i] == xg->xy_vars.y)
      {
        inc = 1;
        break;
      }
    }

    if (inc)
    {
      if (i != 0)
      {
        xg->corr_yvars[i] = xg->corr_yvars[0];
        xg->corr_yvars[0] = xg->xy_vars.y;
      }
    } else {
      xg->corr_yvars[xg->ncorr_yvars] = xg->corr_yvars[0];
      xg->corr_yvars[0] = xg->xy_vars.y;
      xg->ncorr_yvars++;
    }
  }

  else if (xg->is_spinning)
  {
    /*
     * Switch to three variables ...
    */
    xg->corr_xvars[0] = xg->spin_vars.x;
    xg->corr_xvars[1] = xg->spin_vars.z;
    xg->corr_yvars[0] = xg->spin_vars.y;
    xg->ncorr_xvars = 2;
    xg->ncorr_yvars = 1;
    reinit_corr(xg);
  }
  else if (xg->is_touring)
  {
    /* take variable with largest y contribution to be single y variable */
    float max;

    max = xg->tour_vars[0];
    for (i=0; i<xg->numvars_t; i++)
      if (xg->u[1][i] > max)
        max = xg->tour_vars[i];
    xg->corr_yvars[0] = max;
    xg->ncorr_yvars = 1;    

    xg->ncorr_xvars = 0;
    for (i=0; i<xg->numvars_t; i++)
      if (xg->tour_vars[i] != xg->corr_yvars[0])
        xg->corr_xvars[xg->ncorr_xvars++] = xg->tour_vars[i];
    reinit_corr(xg);
  }
}

/********** For New Data **********/

void
destroy_varsel_widgets(ncols_prev, xg)
  int ncols_prev;
  xgobidata *xg;
{
  int j;

  XtUnmapWidget(xg->var_panel);
  XtUnmanageChild(xg->var_panel);
  /*
   * Destroy all the variable boxes.
  */
  /*XtUnmanageChildren(xg->varboxw, ncols_prev);*/
  for (j=0; j<ncols_prev; j++)
    XtDestroyWidget(xg->varboxw[j]);
}

void
reset_var_panel(xg)
  xgobidata *xg;
{
  int j;

  /*
   * Reallocate the variable panel and menus for the new ncols.
  */
  xg->varboxw = (Widget *)
    XtRealloc((char *) xg->varboxw,
    (Cardinal) xg->ncols * sizeof(Widget));
  xg->varlabw = (Widget *)
    XtRealloc((char *) xg->varlabw,
    (Cardinal) xg->ncols * sizeof(Widget));
  xg->vardraww = (Widget *)
    XtRealloc((char *) xg->vardraww,
    (Cardinal) xg->ncols * sizeof(Widget));
  xg->varchosen = (Boolean *)
    XtRealloc((char *) xg->varchosen,
    (Cardinal) xg->ncols * sizeof(Boolean));

  /*
   * Create the new variable boxes.
  */
  make_varboxes(xg);

  for (j=0; j<xg->ncols; j++)
  {
    XtVaSetValues(xg->vardraww[j],
      XtNwidth, (Dimension) (2*(xg->radius+1)),
      XtNheight, (Dimension) (2*(xg->radius+1)), NULL);
    /* This line is important! */
    XtRealizeWidget(xg->varboxw[j]);
  }
  XtManageChildren(xg->varboxw, (unsigned int) xg->ncols);

  XtMapWidget(xg->var_panel);

  for (j=0; j<xg->ncols_used; j++)
    refresh_vbox(xg, j, 1);
}
