//============================================================================
//
//    SSSS    tt          lll  lll
//   SS  SS   tt           ll   ll
//   SS     tttttt  eeee   ll   ll   aaaa    "An Atari 2600 VCS Emulator"
//    SSSS    tt   ee  ee  ll   ll      aa
//       SS   tt   eeeeee  ll   ll   aaaaa   Copyright (c) 1995,1996,1997
//   SS  SS   tt   ee      ll   ll  aa  aa         Bradford W. Mott
//    SSSS     ttt  eeeee llll llll  aaaaa
//
//============================================================================

/**
  Objects of this class are VCS-Script interpreters.  They can read
  in a VCS-Script file, parse it, and evalute it.

  The parser has been hacked together pretty quickly and I'm sure it
  has several problems, however, it seems to behave pretty good
  on "correct" input.

  @author  Bradford W. Mott
  @version $Id: VCSScrpt.cxx,v 1.1 1997/05/17 19:04:13 bwmott Exp $
*/

#include <fstream.h>
#include <string.h>
#include "Error.hxx"
#include "Props.hxx"
#include "VCSScrpt.hxx"

//============================================================================
// Construct VCSScript object
//============================================================================
VCSScript::VCSScript()
{
  myStream = 0;
}

//============================================================================
// Destructor
//============================================================================
VCSScript::~VCSScript()
{
}

//============================================================================
// Evaluate the specified script using the supplied environment
//============================================================================
void VCSScript::evaluate(istream& stream, Properties& env)
{
  myStream = &stream;

  // Make sure this is a VCS Script file
  char buffer[256];
  myStream->getline(buffer, 256);

  if(strcmp(buffer, "VCS-Script") != 0)
  {
    Error err;
    err.message() << "The file doesn't seem to be a VCS-Script file!";
    Throw(err);
  }

  // Initialize state
  myLineNumber = 2;
  myTokenValid = false;

  // Run the code!
  run(env);
}

//============================================================================
// Get the next token but don't discard it.  Answer true iff a valid 
// token was read.
//============================================================================
bool VCSScript::peekToken(char* token)
{
  // If we have a valid token in myToken then return it
  if(myTokenValid)
  {
    strcpy(token, myToken);
    return true;
  }
  else
  {
    char c;

    // This new token is not valid until we're finished reading it in
    myTokenValid = false;

    // Read the first character that we're interested in looking at
    for(;;)
    {
      // Get the next non-whitespace character
      do
      {
        myStream->get(c);

        // Increment the line counter
        if(c == '\n')
        {
          ++myLineNumber;
        }

      } while(((c == ' ') || (c == '\t') || (c == '\n')) && myStream->good());

      // If the stream is nolonger good then answer false
      if(!myStream->good())
      {
        return false;
      }

      // Have we found a comment?
      if(c == ';')
      {
        // Yes! so forget everything on this line...
        while(myStream->good())
        {
          myStream->get(c);

          // Increment the line counter
          if(c == '\n')
          {
            ++myLineNumber;
            break;
          }
        }

        // If the stream is nolonger good then answer false
        if(!myStream->good())
        {
          return false;
        }
      }
      else
      {
        // We now have an interesting non-whitespace character in 'c'
        break;
      }
    }

    // Based on the first character of the token read the token
    if(c == '(')
    {
      strcpy(token, "(");
    }
    else if(c == ')')
    {
      strcpy(token, ")");
    }
    else if(c == '\'')
    {
      strcpy(token, "'");
    }
    else if(c == '\"')
    {
      for(int t = 0; myStream->good() ; ++t)
      {
        myStream->get(c);

        if(c == '\"')
        {
          token[t] = 0;
          break;
        }
        else if(c == '\n')
        {
          Error err;
          err.message() << "VCS-Script: Nonterminated string literal on line "
              << myLineNumber;
          Throw(err);
        }
        else
        {
          token[t] = c;
        }
      }

      // If the stream is nolonger good then answer false
      if(!myStream->good())
      {
        return false;
      }
    }
    else
    {
      int t = 0;
      token[t++] = c;

      while((myStream->peek() != ')') && (myStream->peek() != '(')
          && (myStream->peek() != '\'') && (myStream->peek() != '\"') 
          && (myStream->peek() != ' ') && (myStream->peek() != '\t') 
          && (myStream->peek() != '\n') && myStream->good())
      {   
        myStream->get(c);
        token[t++] = c;
      }
      token[t++] = 0;

      // If the stream is nolonger good then answer false
      if(!myStream->good())
      {
        return false;
      }
    }

    strcpy(myToken, token);
    myTokenValid = true;
    return true;
  }
}

//============================================================================
// Set token to the next token and remove the token.  
// Answer true if everything is okay.
//============================================================================
bool VCSScript::nextToken(char* token)
{
  // Get the next token
  bool result = peekToken(token);

  // Indicate that the current token has been consumed
  myTokenValid = false;

  return result;
}


//============================================================================
// Skip the next list and any sub-lists within that list
//============================================================================
void VCSScript::skip(int level)
{
  char token[256];

  // Skip all of the nested lists
  for(;;)
  {
    nextToken(token);
    if(strcmp(token, "(") == 0)
    {
      ++level;
    }
    else if(strcmp(token, ")") == 0)
    {
      --level;
      if(level == 0)
        break;
    }
  }
}

//============================================================================
// Evalute the boolean expression that follows
//============================================================================
bool VCSScript::evaluateBooleanExpression(Properties& env)
{
  char token[256];

  nextToken(token);

  if(strcmp(token, "(") == 0)
  {
    nextToken(token);

    if(strcmp(token, "or") == 0)
    {
      for(;;)
      {
        peekToken(token);
        if(strcmp(token, ")") == 0)
        {
          nextToken(token);
          return false;
        }
        else if(evaluateBooleanExpression(env))
        {
          // We've found a true expression so skip the rest of them
          skip(1);
          return true;
        } 
      } 
    }
    else if(strcmp(token, "member") == 0)
    {
      char variable[256];

      nextToken(variable);
      nextToken(token);
      if(strcmp(token, "'") != 0)
      {
        Error err;
        err.message() << "VCS-Script: Unquoted list on line " << myLineNumber;
        Throw(err);
      }

      nextToken(token);
      if(strcmp(token, "(") != 0)
      {
        Error err;
        err.message() << "VCS-Script: Bad member statement on line " 
            << myLineNumber;
        Throw(err);
      }

      bool result;
      for(;;)
      {
        nextToken(token);
        if(strcmp(token, ")") == 0)
        {
          result = false;
          break;
        }

        // TODO! should be case insensitive??
        if(strcmp(env.get(variable), token) == 0)
        {
          // Skip all the other items in the list
          skip(1);

          result = true;
          break; 
        }
      }

      nextToken(token);
      if(strcmp(token, ")") != 0)
      {
        Error err;
        err.message() << "VCS-Script: Bad member statement on line " 
            << myLineNumber;
        Throw(err);
      }
      return result;
    }
  }
  else
  {
    Error err;
    err.message() << "VCS-Script: Bad boolean expression on line " 
        << myLineNumber;
    Throw(err);
  } 
  return false;
}

//============================================================================
// Run the next statement
//============================================================================
void VCSScript::run(Properties& env)
{
  char token[256];

  // If no more tokens then return
  if(!nextToken(token))
  {
    return;
  }

  if(strcmp(token, "(") == 0)
  {
    nextToken(token);

    if(strcmp(token, "set!") == 0)
    {
      char variable[256];
      char value[256];
      nextToken(variable);
      nextToken(value);

      env.set(variable, value);
//      cout << variable << " = " << value << endl;

      nextToken(token);
      if(strcmp(token, ")") != 0)
      {
        Error err;
        err.message() << "VCS-Script: Bad set! statement on line " 
            << myLineNumber;
        Throw(err);
      }
    }
    else if(strcmp(token, "if") == 0)
    {
      int line = myLineNumber;
      if(evaluateBooleanExpression(env))
      {
        // Expression true so run the statement inside the if
        run(env);
      }
      else
      {
        // Expression false so skip the statement inside the if
        skip();      
      }

      nextToken(token);
      if(strcmp(token, ")") != 0)
      {
        Error err;
        err.message() << "VCS-Script: Bad if statement on line " << line;
        Throw(err);
      }
    }
    else if(strcmp(token, "begin") == 0)
    {
      for(;;)
      {
        peekToken(token);

        if(strcmp(token, ")") == 0)
        {
          nextToken(token);
          return;
        }
        else
        {
          run(env);
        }
      }
    }
    else if(strcmp(token, ")") == 0)
    {
      return; 
    }
    else
    {
      Error err;
      err.message() << "VCS-Script: Illegal function on line " << myLineNumber;
      Throw(err);
    }
  }
  else
  {
    Error err;
    err.message() << "VCS-Script: Bad token on line " << myLineNumber;
    Throw(err);
  }
}

