/********************************************************************************
 * Copyright (c) Des Herriott 1993, 1994
 *               Erik Kunze   1995, 1996, 1997
 *
 * Permission to use, distribute, and sell this software and its documentation
 * for any purpose is hereby granted without fee, provided that the above
 * copyright notice appear in all copies and that both that copyright notice and
 * this permission notice appear in supporting documentation, and that the name
 * of the copyright holder not be used in advertising or publicity pertaining to
 * distribution of the software without specific, written prior permission.  The
 * copyright holder makes no representations about the suitability of this
 * software for any purpose.  It is provided "as is" without express or implied
 * warranty. THE CODE MAY NOT BE MODIFIED OR REUSED WITHOUT PERMISSION!
 *
 * THE COPYRIGHT HOLDER DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO
 * EVENT SHALL THE COPYRIGHT HOLDER BE LIABLE FOR ANY SPECIAL, INDIRECT OR
 * CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE,
 * DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER
 * TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
 * PERFORMANCE OF THIS SOFTWARE.
 *
 * Author: Des Herriott
 *         Erik Kunze
 *
 * changed by EKU
 *******************************************************************************/
#ifndef lint
static char rcsid[] = "$Id: util.c,v 3.11 1997/09/13 16:04:54 erik Rel $";
#endif

#include <stdio.h>
#include <stdarg.h>
#include <stdlib.h>
#include <string.h>
#include <sys/time.h>
#include <ctype.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/stat.h>
#include "z80.h"
#include "resource.h"
#include "mem.h"
#include "main.h"
#include "util.h"

struct onQuitRecord {
	void (*f)();
	struct onQuitRecord *next;
};




static struct onQuitRecord *onQuitList = NULL;

FILE *
Fopen(char *name, char *amode)
{
	FILE *fp = NULL;
	char *buf;
	char *path, *element;
	if (!name || !name[0])
	{
		return NULL;
	}
	if (name[0] == '/' || !strncmp(name, "./", 2) || !strncmp(name, "../", 3))
	{
		return (fopen(name, amode));
	}
	path = Strdup(GETCFG(libDir), "Fopen");
	element = strtok(path, ":");
	while (element)
	{
		buf = (char *)Malloc(strlen(name) + strlen(element) + 2, "Fopen");
		(void)sprintf(buf, "%s/%s", element, name);
		fp = fopen(buf, amode);
		free(buf);
		if (fp)
		{
			break;
		}
		element = strtok(NULL, ":");
	}
	free(path);
	return fp;
}

void *
Malloc(long size, char *function)
{
	void *p;
	if (!(p = calloc(size, 1)))
	{
		Msg(M_FATAL, "malloc failed in %s()", function);
	}
	return p;
}

char *
Strdup(char *src, char *function)
{
	char *dst;
	if (!(dst = strdup(src)))
	{
		Msg(M_FATAL, "strdup failed in %s()", function);
	}
	return dst;
}

void
OnQuit(void (*fn)())
{
	struct onQuitRecord *p;
	p = (struct onQuitRecord *)Malloc(sizeof (struct onQuitRecord), "OnQuit");
	p->f = fn;
	p->next = onQuitList;
	onQuitList = p;
}

void
Quit(int status)
{
	struct onQuitRecord *p = onQuitList;
	while (p)
	{
		p->f();
		p = p->next;
	}
	exit(status);
}

#ifdef nec_ews

void
Msg(va_alist)
va_dcl
{
	va_list args;
	int errorLevel;
	char *fmt;
	va_start(args);
	errorLevel = (int)va_arg(args, int);
	fmt = (char *)va_arg(args, char *);
#else
void
Msg(int errorLevel, ...)
{
	va_list args;
	char *fmt;
	va_start(args, errorLevel);
	fmt = (char *)va_arg(args, char *);
#endif
	if (!GETCFG(quiet)
		|| errorLevel == M_ERR || errorLevel == M_PERR || errorLevel == M_FATAL)
	{
		(void)fprintf(stderr, "%s: ", ProgName);
		switch(errorLevel)
		{
			case M_WARN:
			case M_PWARN:
				(void)fprintf(stderr, "warning: ");
				break;
			case M_ERR:
			case M_PERR:
				(void)fprintf(stderr, "error: ");
				break;
			case M_FATAL:
				(void)fprintf(stderr, "fatal: ");
				break;
			case M_DEBUG:
				(void)fprintf(stderr, "debug: ");
				break;
		}
		(void)vfprintf(stderr, fmt, args);
		(void)putc('\n', stderr);
		if (errorLevel == M_PWARN || errorLevel == M_PERR)
		{
			perror("  -> reason");
		}
	}
	if (errorLevel == M_FATAL)
	{
		Quit(1);
	}
}

void
IntFrequency(int ms)
{
	struct itimerval itv;
	itv.it_interval.tv_sec = 0;
	itv.it_interval.tv_usec = ms;
	itv.it_value.tv_sec = 0;
	itv.it_value.tv_usec = ms;
	(void)setitimer(ITIMER_REAL, &itv, NULL);
}

int
ClassifyDescriptor(int fd)
{
	struct stat buf;
	if (isatty(fd))
	{
		return FD_TTY;
	}
	if (!fstat(fd, &buf))
	{
		if (S_ISREG(buf.st_mode))
		{
			return FD_FILE;
		}
		else if (S_ISFIFO(buf.st_mode))
		{
			return FD_PIPE;
		}
	}
	Msg(M_WARN, "don't know where fd %d is coming from", fd);
	return FD_UNKNOWN;
}

void
SetBlocking(int fd, int mode)
{
	int status;
	if  ((status = fcntl(fd, F_GETFL)) != -1)
	{
#ifdef O_NONBLOCK
		status = mode ? status | O_NONBLOCK : status & ~O_NONBLOCK;
#else
		status = mode ?	status | O_NDELAY : status & ~O_NDELAY;
#endif
		if (fcntl(fd, F_SETFL, status) != -1)
		{
			return;
		}
	}
	Msg(M_PERR, "couldn't make fd <%d> %sblocking", fd, mode ? "non" : "");
}

char *
GetBaseName(char *filename)
{
	char *s;
	if (filename)
	{
		if ((s = strrchr(filename, '/')))
		{
			return ++s;
		}
		else
		{
			return filename;
		}
	}
	return ("-none-");
}

int
GetFileType(char *fname)
{
	static struct { char *extension; int type;} types[] = {
		{ "sna", FT_SNA },
		{ "z80", FT_Z80 },
		{ "slt", FT_SLT },
		{ "tap", FT_TAP },
		{ "voc", FT_VOC },
		{ "dsk", FT_DSK },
		{ "mdr", FT_MDR }
	};
	char *p;
	int i;
	if ((p = strrchr(fname, '.')))
	{
		p++;
		for (i = 0; i < (sizeof(types) / sizeof(types[0])); i++)
		{
			if (!strncasecmp(p, types[i].extension, 4))
			{
				return types[i].type;
			}
		}
	}
	return FT_UNKNOWN;
}

#ifdef NEED_STRCASECMP
int
strcasecmp(char *s1, char *s2)
{
	char c1, c2;
	for ( ; *s1 && *s2; s1++, s2++)
	{
		c1 = isupper(*s1) ? tolower(*s1) : *s1;
		c2 = isupper(*s2) ? tolower(*s2) : *s2;
		if (c1 != c2)
		{
			return (c1 - c2);
		}
	}
	if (*s1 || *s2)
	{
		return (*s1 - *s2);
	}
	return 0;
}
int
strncasecmp(char *s1, char *s2, int n)
{
	int i;
	char c1, c2;
	for (i = 0; *s1 && *s2 && i < n; s1++, s2++, i++)
	{
		c1 = isupper(*s1) ? tolower(*s1) : *s1;
		c2 = isupper(*s2) ? tolower(*s2) : *s2;
		if (c1 != c2)
		{
			return (c1 - c2);
		}
	}
	if ((i < n) && (*s1 || *s2))
	{
		return (*s1 - *s2);
	}
	return 0;
}
#endif

void
DestroyList(llist *head, void (*pDestroy)(void *))
{
	llist *pNext;
	while(head)
	{
		if (head->item)
		{
			if (pDestroy)
			{
				pDestroy(head->item);
			}
			free(head->item);
		}
		pNext = head->next;
		free(head);
		head = pNext;
	}
}

llist *
AppendElemList(llist *head, void *item)
{
	llist **ppHead = &head;
	while(*ppHead)
	{
		ppHead = &((*ppHead)->next);
	}
	*ppHead = Malloc(sizeof(llist), "AppendList");
	(*ppHead)->item = item;
	(*ppHead)->next = NULL;
	return head;
}

llist *
PrependElemList(llist *head, void *item)
{
	llist *pHead;
	pHead = Malloc(sizeof(llist), "PrependList");
	pHead->item = item;
	pHead->next = head;
	return pHead;
}

llist *
RemoveElemList(void *item, int (*pCompare)(void *, void *), llist *head,
			   void (*pDestroy)(void *))
{
	llist **ppHead = &head;
	llist *pElem;
	while (*ppHead && (*pCompare)((*ppHead)->item, item))
	{
		ppHead = &((*ppHead)->next);
	}
	if (*ppHead)
	{
		pElem = *ppHead;
		*ppHead = (*ppHead)->next;
		if (pDestroy)
		{
			pDestroy(pElem);
		}
		free(pElem);
	}
	return head;
}

llist *
SortedInsertList(void *item, int (*pCompare)(void *, void *), llist *head)
{
	llist **ppHead = &head;
	llist *pElem = Malloc(sizeof(llist), "SortedInsertList");
	pElem->item = item;
	pElem->next = NULL;
	while (*ppHead && (*pCompare)(item, (*ppHead)->item) >= 0)
	{
		ppHead = &(*ppHead)->next;
	}
	if (*ppHead)
	{
		pElem->next = *ppHead;
	}
	*ppHead = pElem;
	return head;
}

void *
RetrieveElemList(llist *head, int which)
{
	while(head && which)
	{
		head = head->next;
		which--;
	}
	return (head ? head->item : NULL);
}

