;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;                                                                       ;;
;;;                Centre for Speech Technology Research                  ;;
;;;                     University of Edinburgh, UK                       ;;
;;;                       Copyright (c) 1996,1997                         ;;
;;;                        All Rights Reserved.                           ;;
;;;                                                                       ;;
;;;  Permission to use, copy, modify, distribute this software and its    ;;
;;;  documentation for research, educational and individual use only, is  ;;
;;;  hereby granted without fee, subject to the following conditions:     ;;
;;;   1. The code must retain the above copyright notice, this list of    ;;
;;;      conditions and the following disclaimer.                         ;;
;;;   2. Any modifications must be clearly marked as such.                ;;
;;;   3. Original authors' names are not deleted.                         ;;
;;;  This software may not be used for commercial purposes without        ;;
;;;  specific prior written permission from the authors.                  ;;
;;;                                                                       ;;
;;;  THE UNIVERSITY OF EDINBURGH AND THE CONTRIBUTORS TO THIS WORK        ;;
;;;  DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING      ;;
;;;  ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT   ;;
;;;  SHALL THE UNIVERSITY OF EDINBURGH NOR THE CONTRIBUTORS BE LIABLE     ;;
;;;  FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES    ;;
;;;  WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN   ;;
;;;  AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,          ;;
;;;  ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF       ;;
;;;  THIS SOFTWARE.                                                       ;;
;;;                                                                       ;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;  General Festival Scheme specific functions
;;;  Including definitions of various standard variables.

(defvar hush_startup nil
  "hush_startup
  If set to non-nil, the copyright banner is not displayed at start up.")

(defvar readline_histsize 256
  "readline_histsize
  The number of line so be saved in the users history file when a 
  Festival session ends.  The histfile is \".festival_history\" in the
  users home directory.  Note this value is only checked when the 
  command interpreter is started, hence this should be set in a user's
  \".festivalrc\" or system init file.  Reseting it at the command
  interpreter will have no effect.")

(defvar ! nil
  "!
  In interactive mode, this variable's value is the return value of the
  previously evaluated expression.")

(defvar Parameter nil
  "Parameter
  An assoc-list of parameters and values for various parts of the speech
  synthesis system.  This is used by the functions Parameter.set 
  Parameter.def and Parameter.get as well as internal C++ functions.")

;; will be set automatically on start-up
(defvar festival_version "unknown"
  "festival_version
 A string containing the current version number of the system.")

;; will be set automatically on start-up
(defvar festival_version_number '(x x x)
  "festival_version_number
 A list of major, minor and subminor version numbers of the current
 system.  e.g. (1 0 12).")

(define (Parameter.set name val)
"(Parameter.set NAME VAL)
  Set parameter NAME to VAL (deleting any previous setting)"
  (let ((pair (assoc name Parameter)))
   (if pair
       (set-car! (cdr pair) val)
       (set! Parameter (cons (list name val) Parameter)))
   val))

(define (Parameter.def name val)
"(Parameter.def NAME VAL)
  Set parameter NAME to VAL if not already set"
  (let ((pair (assoc name Parameter)))
   (if pair
       (car (cdr pair))
       (and
	(set! Parameter (cons (list name val) Parameter))
	val))))

(define (Parameter.get name)
"(Parameter.get NAME)
  Get parameter NAME's value (nil if unset)"
  (let ((pair (assoc name Parameter)))
    (if pair
	(car (cdr pair))
	nil)))

(define (apply_method method utt)
"(apply_method METHOD UTT)
Apply the appropriate function to utt defined in parameter."
  (let ((method_val (Parameter.get method)))
    (cond
     ((null method_val)
      nil)   ;; should be an error, but I'll let you off at present
     ((and (symbol? method_val) (symbol-bound? method_val))
      (apply (symbol-value method_val) (list utt)))
     ((not (symbol? method_val))
      (apply method_val (list utt)))
     (t      ;; again is probably an error
      nil))))

(defvar ufoutfp nil)

;;;  Feature Function Functions
(define (utt.features utt stream func_list)
"(utt.features UTT STREAMNAME FUNCLIST)
  Get vectors of feature values for each item in STREAMNAME in UTT.
  If ufoutfp is non-nil write them to that file else return them as a list.
  [see Features]"
  (if ufoutfp
      (and 
       (mapcar
	(lambda (s)
	  (mapcar 
	   (lambda (f)
	     (format ufoutfp "%l " (utt.streamitem.feat utt s f)))
	   func_list)
	  (terpri ufoutfp))
	(utt.stream utt stream)) t)
      (mapcar (lambda (s) (mapcar (lambda (f)
				    (utt.streamitem.feat utt s f))
				  func_list))
	      (utt.stream utt stream))))

(define (dump_feats stream features names outfile)
"(dump_feats STREAMNAME FEATURES NAMES OUTFILE)
  Dump FEATURES of STREAMNAME in each utterances in NAMES to OUTFILE.
  Utterances are created by calling test_seg on each member of NAMES."
  (set! ufoutfp (fopen outfile "w"))
  (mapcar
   (lambda (name)
     (print name)
     (utt.features (test_seg name) stream features))
   names)
  (fclose ufoutfp)
  (set! ufoutfp nil))

(define (utt.type utt)
"(utt.type UTT)
  Returns the type of UTT."
  (intern (utt.streamitem.feat utt (car (utt.stream utt 'IForm)) 'type)))

(define (utt.save.segs utt filename)
"(utt.save.segs UTT FILE)
  Save segments of UTT in a FILE in xlabel format."
  (let ((fd (fopen filename "w")))
    (format fd "#\n")
    (mapcar
     (lambda (info)
       (format fd "%2.4f 100 %s\n" (car info) (car (cdr info))))
     (utt.features utt 'Segment '(end name)))
    (fclose fd)
    utt))

(define (utt.save.stream utt streamname filename)
"(utt.save.stream UTT STREAMNAME FILE)
  Save items in STREAMNAME of UTT in a FILE in xlabel format."
  (let ((fd (fopen filename "w")))
    (format fd "#\n")
    (mapcar
     (lambda (info)
       (format fd "%2.4f 100 %s\n" (car info) (car (cdr info))))
     (utt.features utt streamname '(end name)))
    (fclose fd)
    utt))

(define (utt.save.units utt filename)
"(utt.save.units UTT FILE)
  Save selected unit descriptions of UTT in a FILE in xlabel format."
  (let ((fd (fopen filename "w")))
    (format fd "#\n")
    (mapcar
     (lambda (info)
       (format fd "%2.4f 100 %l\n" 
	       (car info) 
	       (ducs.unit (car (cdr info)))))
     (utt.features utt 'Segment '(end unit_id)))
    (fclose fd)
    utt))

(define (utt.resynth labfile f0file)
"(utt.resynth LABFILE F0FILE)
Resynthesize an utterance from a label file and F0 file (in any format
supported by the Speech Tool Library).   This loads, synthesizes and
plays the utterance."
   (utt.play (utt.synth (utt.load.segf0 labfile f0file))))

(define (utt.stream.present utt streamname)
"(utt.stream.present UTT STREAMNAME)
Returns t if UTT caontaisn a stream call STREAMNAME, nil otherwise."
  (if (member name (utt.streamnames utt))
      t
      nil))

(define (stream.set_duration streamitem duration)
"(stream.set_duration STREAMITEM DURATION)
Sets STREAMITEMS duration to DURATION.  Note care should be take with
this as duration is derived from the end feature.  It is the user's
responsibility to ensure that durations do not cause end times through a
stream to cease to be unordered."
  (streamitem.set_end
   streamitem
   (+ duration
      (if (streamitem.prev streamitem)
	  (streamitem.end streamitem)
	  0))))

(defvar server_max_clients 10
  "server_max_clients
In server mode, the maximum number of clients supported at any one
time.  When more that this number of clients attach simulaneous
the last ones are denied access.  Default value is 10.
[see Server/client API]")

(defvar server_port 1314
  "server_port
In server mode the inet port number the server will wait for connects
on.  The default value is 1314. [see Server/client API]")

(defvar server_log_file t
  "server_log_file
If set to t server log information is printed to standard output
of the server process.  If set to nil no output is given.  If set
to anything else the value is used as the name of file to which
server log information is appended.  Note this value is checked at
server start time, there is no way a client may change this.
[see Server/client API]")

(defvar server_passwd nil
  "server_passwd
If non-nil clients must send this passwd to the server followed by
a newline before they can get a connection.  It would be normal
to set this for the particular server task.
[see Server/client API]")

(defvar server_access_list '(localhost)
  "server_access_list
If non-nil this is the exhaustive list of machines and domains
from which clients may access the server.  This is a list of REGEXs
that client host must match.  Remember to add the backslashes before
the dots. [see Server/client API]")

(defvar server_deny_list nil
  "server_deny_list
If non-nil this is a list of machines which are to be denied access
to the server absolutely, irrespective of any other control features.
The list is a list of REGEXs that are used to matched the client hostname.
This list is checked first, then server_access_list, then passwd.
[see Server/client API]")

(define (def_feature_docstring fname fdoc)
"(def_feature_docstring FEATURENAME FEATUREDOC)
As some feature are used directly of stream items with no
accompanying feature function, the features are just values on the feature
list.  This function also those features to have an accompanying
documentation string."
  (let ((fff (assoc fname ff_docstrings)))
    (cond
     (fff  ;; replace what's already there
      (set-cdr! fff fdoc))
     (t
      (set! ff_docstrings (cons (cons fname fdoc) ff_docstrings))))
    t))

(define (linear_regression utt item model)
  "(linear_regression UTT ITEM MODEL)
Use linear regression MODEL on UTT and ITEM.  MODEL consists of a list
of features, weights and optional map list.  E.g. ((Intercept 100)
(tobi_accent 10 (H* !H*)))."
  (let ((intercept (if (equal? 'Intercept (car (car model))) 
                       (car (cdr (car model))) 0))
        (mm (if (equal? 'Intercept (car (car model))) 
                (cdr model) model)))
  (apply + 
   (cons intercept
   (mapcar
    (lambda (f)
     (let ((ff (utt.streamitem.feat utt item (car f))))
      (if (car (cdr (cdr f)))
         (if (member_string ff (car (cdr (cdr f))))
           (car (cdr f))
           0)
         (* (parse-number ff) (car (cdr f))))))
    mm)))))

(defvar help
 "The Festival Speech Synthesizer System: Help

Getting Help
  (doc '<SYMBOL>)   displays help on <SYMBOL>
  (manual nil)      displays manual in local netscape
  C-c               return to top level
  C-d or (quit)     Exit Festival
(If compiled with readline)
  M-h               desplays help on current symbol  
  M-s               speaks help on current symbol  
  M-m               displays relevant manula page in local netscape
  TAB               Command, symbol and filename completion
  C-p or up-arrow   Previous command
  C-b or left-arrow Move back one character
  C-f or right-arrow 
                    Move forward one character
  Normal Emacs commands work for editing command line

Doing stuff
  (SayText TEXT)      Synthesize text, text should be surrounded by
                      double quotes
  (tts FILENAME nil)  Say contexts of file, FILENAME should be 
                      surrounded by double quotes
  (voice_rab_diphone) Select voice (Britsh Male)
  (voice_kd_diphone)  Select voice (American Male)
")

(define (festival_warranty)
"(festival_warranty)
  Display Festival's copyright and warranty. [see Copying]"
 (format t
   (string-append
    "    The Festival Speech Synthesis System: "
    festival_version
"
                Centre for Speech Technology Research                  
                     University of Edinburgh, UK                       
                       Copyright (c) 1996,1997                         
                        All Rights Reserved.                           
                                                                       
  Permission to use, copy, modify, distribute this software and its            
  documentation for research, educational and individual use only, is  
  hereby granted without fee, subject to the following conditions:     
   1. The code must retain the above copyright notice, this list of    
      conditions and the following disclaimer.                         
   2. Any modifications must be clearly marked as such.                
   3. Original authors' names are not deleted.                         
  This software may not be used for commercial purposes without        
  specific prior written permission from the authors.                  
                                  
  THE UNIVERSITY OF EDINBURGH AND THE CONTRIBUTORS TO THIS WORK        
  DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING      
  ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT   
  SHALL THE UNIVERSITY OF EDINBURGH NOR THE CONTRIBUTORS BE LIABLE     
  FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES    
  WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN   
  AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,          
  ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF       
  THIS SOFTWARE.                                                       
")))

(define (intro)
"(intro)
 Synthesize an introduction to the Festival Speech Synthesis System."
  (tts (path-append libdir "../examples/intro.text") nil))

(define (intro-spanish)
"(intro-spanish)
 Synthesize an introduction to the Festival Speech Synthesis System
 in spanish.  Spanish voice must already be selected for this."
  (tts (path-append libdir "../examples/spintro.text") nil))

(define (na_play FILENAME)
"(play_wave FILENAME)
Play given wavefile"
 (utt.play (utt.synth (eval (list 'Utterance 'Wave FILENAME)))))

;;; Some autoload commands
(autoload manual-sym "festdoc" "Show appropriate manual section for symbol.")
(autoload manual "festdoc" "Show manual section.")

(autoload display "display" "Graphically display utterance.")

(autoload festtest "festtest" "Run tests of Festival.")

(defvar diphone_module_hooks nil
  "diphone_module_hooks
  A function or list of functions that will be applied to the utterance
  at the start of the diphone module.  It can be used to map segment 
  names to those that will be used by the diphone database itself.
  Typical use specifies _ and $ for consonant clusters and syllable 
  boundaries, mapping to dark ll's etc.  Reduction and tap type 
  phenomena should probabaly be done by post lexical rules though the 
  distinction is not a clear one.")

(def_feature_docstring
  'Segment.diphone_phone_name
  "Segment.diphone_phone_name
  This is produced by the diphone module to contain the desired phone
  name for the desired diphone.  This adds things like _ if part of 
  a consonant or $ to denote syllable boundaries.  These are generated
  on a per voice basis by function(s) specified by diphone_module_hooks.
  Identification of dark ll's etc. may also be included.  Note this is not
  necessarily the name of the diphone selected as if it is not found
  some of these characters will be removed and fall back values will be
  used.")

(def_feature_docstring
  'Syllable.stress
  "Syllable.stress
  The lexical stress of the syllable as specified from the lexicon entry
  corresponding to the word related to this syllable.")

;;;
;;;  I treid some tests on the resulting speed both runtime and loadtime
;;;  but compiled files don't seem to make any significant difference
;;;
(define (compile_library)
  "(compile_library)
Compile all the scheme files in the library directory."
  (mapcar
   (lambda (file)
     (format t "compile ... %s\n" file)
     (compile-file (string-before file ".scm")))
   (list
     "synthesis.scm" "siod.scm" "init.scm" "lexicons.scm"
     "festival.scm" "gsw_diphone.scm" "intonation.scm" "duration.scm"
     "pos.scm" "phrase.scm" "don_diphone.scm" "rab_diphone.scm"
     "voices.scm" "tts.scm" "festdoc.scm" "languages.scm" "token.scm"
     "mbrola.scm" "display.scm" "postlex.scm" "tokenpos.scm"
     "festtest.scm" "cslush.scm" "ducs_cluster.scm" "sucs.scm"
     "web.scm" "cart_aux.scm"
     "lts_nrl.scm" "lts_nrl_us.scm" "email-mode.scm"
     "mrpa_phones.scm" "radio_phones.scm" "holmes_phones.scm"
     "mrpa_durs.scm" "klatt_durs.scm" "gswdurtreeZ.scm"
     "tobi.scm" "f2bf0lr.scm"))
  t)

(provide 'festival)
