;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;                                                                       ;;
;;;                Centre for Speech Technology Research                  ;;
;;;                     University of Edinburgh, UK                       ;;
;;;                       Copyright (c) 1996,1997                         ;;
;;;                        All Rights Reserved.                           ;;
;;;                                                                       ;;
;;;  Permission to use, copy, modify, distribute this software and its    ;;
;;;  documentation for research, educational and individual use only, is  ;;
;;;  hereby granted without fee, subject to the following conditions:     ;;
;;;   1. The code must retain the above copyright notice, this list of    ;;
;;;      conditions and the following disclaimer.                         ;;
;;;   2. Any modifications must be clearly marked as such.                ;;
;;;   3. Original authors' names are not deleted.                         ;;
;;;  This software may not be used for commercial purposes without        ;;
;;;  specific prior written permission from the authors.                  ;;
;;;                                                                       ;;
;;;  THE UNIVERSITY OF EDINBURGH AND THE CONTRIBUTORS TO THIS WORK        ;;
;;;  DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING      ;;
;;;  ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT   ;;
;;;  SHALL THE UNIVERSITY OF EDINBURGH NOR THE CONTRIBUTORS BE LIABLE     ;;
;;;  FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES    ;;
;;;  WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN   ;;
;;;  AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,          ;;
;;;  ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF       ;;
;;;  THIS SOFTWARE.                                                       ;;
;;;                                                                       ;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;;  Definition of various lexicons
;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;;  If there exists a sudirectory of the lib-path called dicts then that 
;;;  is used as the lexicon directory by default.  If it doesn't exist 
;;;  we set lexdir to the directory in CSTR where our lexicons are.  
;;;  In non-CSTR installations where lexicons are not in lib/dicts, 
;;;  you should set lexdir in sitevars.scm

(if (probe_file (path-append libdir "dicts"))
    (defvar lexdir (path-append libdir "dicts/")
      "lexdir
  The directory where the lexicon(s) are, by default.")
  ;; else we'll guess we're in the CSTR filespace
    (defvar lexdir (path-as-directory "/projects/festival/lib/dicts/")
      "lexdir
  The directory where the lexicon(s) are, by default."))

(require 'lts_nrl)    ;; the letter to sound rules

(define (setup_cstr_lex)
"(setup_cstr_lexicon)
Define and setup the CSTR lexicon.  The CSTR lexicon consists
of about 25,000 entries in the mrpa phone set.  A large number of
specific local entries are also added to the addenda."
  (if (not (member_string "mrpa" (lex.list)))
      (begin
	(lex.create "mrpa")
	(lex.set.compile.file (path-append lexdir "cstrlex.out"))
	(lex.set.phoneset "mrpa")
	(lex.set.lts.method 'lts_rules)
	(lex.set.lts.ruleset 'nrl)
	(mrpa_addenda)
	(lex.add.entry
	 '("previous" nil (((p r ii) 1) ((v ii) 0) ((@ s) 0))))
	(lex.add.entry
	 '("audio" () (((oo d) 1) ((ii) 0) ((ou) 0))))
	(lex.add.entry
	 '("modules" () (((m o d) 1) ((uu l s) 0))))
	)))

(define (setup_oald_lex)
"(setup_oald_lexicon)
Define and setup the CUVOALD lexicon.  This is derived from the
Computer Users Version of the Oxford Advanced Learners' Dictionary
of Current English.  It contains around 70,000 entries and includes
part of speech information."
  (if (not (member_string "oald" (lex.list)))
      (begin
	(lex.create "oald")
	(lex.set.compile.file (path-append lexdir "cuvoald710-0.2.out"))
	(lex.set.phoneset "mrpa")
	(lex.set.lts.method 'lts_rules)
	(lex.set.lts.ruleset 'nrl)
	(lex.add.entry 
	 '("a" dt (((@) 0))))
	(lex.add.entry 
	 '("a" n (((ei) 1))))
	(lex.add.entry
	 '("us" prp (((uh s) 0)) ((pos "Qx*"))))
	(lex.add.entry
	 '("'s" pos (((@ z) 0))))
	(lex.add.entry
	 '("'s" n (((@ z) 0))))
	(lex.add.entry 
	 '("the" dt (((dh @) 0))))
	(lex.add.entry
	 '("taylor" n (((t ei) 1) ((l @) 0))))
	(lex.add.entry
	 '("who" prp ((( h uu ) 0))))
	(mrpa_addenda))))

(define (setup_moby_lex)
"(setup_moby_lexicon)
Define and setup the MOBY lexicon.  This is derived from the public
domain version of the Moby (TM) Pronunciator II lexicon.  It can be
converted automatically to British English mrpa phoneset which of
course is sub-optimal.  It contains around 120,000 entries and has part
of speech information for homographs."
  (if (not (member_string "moby" (lex.list)))
      (begin
	(lex.create "moby")
	;  (lex.set.compile.file (path-append lexdir "mobylex.out"))
	(lex.set.compile.file "/home/awb/src/mobypron/mobylex.out")
	(lex.set.phoneset "mrpa")
	(lex.set.lts.method 'lts_rules)
	(lex.set.lts.ruleset 'nrl)
	(lex.add.entry 
	 '("a" dt (((@) 0))))
	(lex.add.entry 
	 '("the" dt (((dh @) 0))))
	(lex.add.entry
	 '("taylor" n (((t ei) 1) ((l @) 0))))
	(lex.add.entry
	 '("who" prp ((( h uu ) 0))))
	(mrpa_addenda))))

(define (setup_beep_lex)
  "(setup_beep_lex)
Lexicon derived from the British English Example Pronunciation dictionary
(BEEP) from Tony Robinson  ajr@eng.cam.ac.uk.  Around 160,000 entries."
  (if (not (member_string "beep" (lex.list)))
    (begin
      (lex.create "beep")
      (lex.set.compile.file (path-append lexdir "beep_lex.out"))
      (lex.set.phoneset "mrpa")
      (lex.set.lts.method 'lts_rules)
      (lex.set.lts.ruleset 'nrl)
      (lex.add.entry
       '("taylor" nil (((t ei) 1) ((l @) 0))))
      (mrpa_addenda))))

;;; The nrl letter to sound rules produce mrpa phone set so we need
;;; to do some fancy things to make them work for American English
(define (f2b_lts word features)
"(f2b_lts WORD FEATURES)
Letter to sound rule system for f2b (American English), uses the NRL
LTS ruleset and maps the result to the radio phone set."
 '("unknown" nil (((ah n) 0) ((n ow n) 1)))
)

(define (setup_cmu_lex)
  "(setup_cmu_lex)
Lexicon derived from the CMU lexicon (cmudict-0.1), around 100,000 entries,
in the radio phoneset (sort of darpa-like)."
  (if (not (member_string "cmu" (lex.list)))
      (begin
	(lex.create "cmu")
	(lex.set.compile.file (path-append lexdir "cmu_lex.out"))
	(lex.set.phoneset "radio")
	(require 'lts_nrl_us)    ;; US English letter to sound rules
	(lex.set.lts.method 'lts_rules)
	(lex.set.lts.ruleset 'nrl_us)
	(lex.add.entry 
	 '("a" dt (((ey) 0))))
	(lex.add.entry 
	 '("a" n (((ey) 1))))
	(lex.add.entry 
	 '("t" n (((t iy) 1))))
	(lex.add.entry 
	 '("w" n (((d ah b) 1) ((ah l) 0) ((y uw) 1))))
	(lex.add.entry
	 '("'s" pos (((ax z) 0))))
	(lex.add.entry
	 '("bought" v (((b ao t) 1))))
	(lex.add.entry
	 '("edinburgh" n (((eh d) 1) ((ah n) 0) ((b ax ) 0) ((r ow) 0))))
	(cmu_addenda))))

(define (mrpa_addenda)
"(mrpa_addenda)
Add a whole host of various entries to the current lexicon with
mrpa phones."
  (lex.add.entry 
   '("algorithm" n (((a l g) 1) ((o) 0) ((r i th m) 0))))
  (lex.add.entry 
   '("algorithms" n (((a l g) 1) ((o) 0) ((r i th m z) 0))))
  (lex.add.entry 
   '("algorithmic" n (((a l g) 1) ((o) 0) ((r i th) 1) ((m i k) 0))))
  (lex.add.entry 
   '("alices" n (((a l) 1) ((i s) 0) ((i z) 0))))
  (lex.add.entry 
   '( "Angeles" n (((a n) 1) ((jh i) 0) ((l ii z) 0))))
  (lex.add.entry 
   '( "atr" n ((( ei ) 1) (( t ii ) 1) (( a a ) 1))))
  (lex.add.entry 
   '( "att" n ((( ei ) 1) (( t ii ) 1) (( a n d ) 0) (( t ii ) 1))))
  (lex.add.entry 
   '( "awb" n ((( ei ) 1) ((d uh) 1) ((b @ l) 0) ((y uu) 0) ((b ii) 1))))
  (lex.add.entry
   '("cdrom" n (((s ii) 1) ((d ii) 1) ((r o m) 1))))
  (lex.add.entry
   '("cdroms" n (((s ii) 1) ((d ii) 1) ((r o m z) 1))))
  (lex.add.entry 
   '("cepstra" n (((k e p) 1) ((s t r @) 0))))
  (lex.add.entry 
   '("cepstral" n (((k e p) 1) ((s t r @ l) 0))))
  (lex.add.entry 
   '("cepstrum" n (((k e p) 1) ((s t r @ m) 0))))
  (lex.add.entry 
   '("co" nil (((k ou) 1))))
  (lex.add.entry 
   '( "cstr" n ((( s ii ) 1) (( e s ) 1) (( t ii ) 1) (( aa ) 1)) ))
  (lex.add.entry
   '("database" n (((d ei) 1) ((t @) 0) ((b ei s) 1))))
  (lex.add.entry
   '("databases" n (((d ei) 1) ((t @) 0) ((b ei s) 1) ((i z) 0))))
  (lex.add.entry
   '("diphone" n (((d ai) 1) ((f ou n) 0))))
  (lex.add.entry
   '("diphones" n (((d ai) 1) ((f ou n s) 0))))
  (lex.add.entry 
   '( "edinburgh" n ((( e d ) 1) (( i n ) 0) ((b r @) 0))))
  (lex.add.entry 
   '( "email" n ((( ii ) 1) (( m ei l) 0))))
  (lex.add.entry 
   '( "emailed" n ((( ii ) 1) (( m ei l d) 0))))
  (lex.add.entry 
   '( "emacs" n ((( ii ) 1) (( m a k s) 0))))
  (lex.add.entry 
   '("globally" a (((g l ou b) 1) ((@ l) 0) ((ii) 0))))
  (lex.add.entry
   '("grave" n (((g r ei v) 1)) ((pos "Kj%"))))
  (lex.add.entry
   '("graves" j (((g r ei v z) 1)) ((pos "Kj%"))))
  (lex.add.entry
   '("greece" n (((g r ii s) 1)) ((pos "Nm%"))))
  (lex.add.entry 
   '("hong" j (((h o ng) 1))))
  (lex.add.entry 
   '("hz" n (((h @@ t z) 1))))
  (lex.add.entry 
   '("innovative" j (((i n) 1) ((@) 0) ((v ei t) 1) ((i v) 0))))
  (lex.add.entry 
   '("job" n (((jh o b) 1))))
  (lex.add.entry
   '("jobs" n (((jh o b z) 1))))
  (lex.add.entry 
   '( "Jr" n (((jh uu n) 1) ((i@) 0)) ((pos "K6%" "OA%"))))
  (lex.add.entry 
   '("kong" n (((k o ng) 1))))
  (lex.add.entry 
   '("khz" n (((k i) 1) ((l ou) 0) ((h @@ t z) 1))))
  (lex.add.entry
   '("labor" n (((l ei) 1) ((b @) 0))))
  (lex.add.entry 
   '( "Los" n ((( l o s) 1))))
  (lex.add.entry 
   '("lower" v (((l ou) 1) ((@) 0))))
  (lex.add.entry 
   '("lowered" v (((l ou) 1) ((@ d) 0))))
  (lex.add.entry 
   '("lowering" v (((l ou) 1) ((@ r) 0) (( i ng ) 0))))
  (lex.add.entry 
   '( "mbrola" n (((e m) 0) ((b r ou l ) 1) (( @ ) 0))))
  (lex.add.entry 
   '("mhz" n (((m e) 1) ((g @) 0) ((h @@ t z) 1))))
  (lex.add.entry 
   '("minute" n (((m i n) 1) ((i t) 0))))
  (lex.add.entry 
   '("ms" n (((e m) 1) ((e s) 1))))
  (lex.add.entry 
   '("no" dt (((n ou) 1))))
  (lex.add.entry 
   '("pault" n ((( p oo l) 1) ((t ii) 1))))
  (lex.add.entry 
   '("put" v (((p u t) 1))))
  (lex.add.entry 
   '("putting" v (((p u t) 1) (( i ng) 0))))
  (lex.add.entry 
   '( "psola" n ((( p i ) 0) (( s ou  ) 1) (( l @ ) 0 ))))
  (lex.add.entry 
   '("reuter" n (((r oi) 1) ((t @@) 0))))
  (lex.add.entry 
   '("reuters" n (((r oi) 1) ((t @@ s) 0))))
  (lex.add.entry
   '("row" v (((r ou) 1))))
  (lex.add.entry
   '("row" n (((r ou) 1))))
  (lex.add.entry 
   '( "San" n ((( s a n) 1))))
  (lex.add.entry
   '("second" n (((s e k) 1) ((@ n d) 0))))
  (lex.add.entry
   '("seconds" n (((s e k) 1) ((@ n d z) 0))))
  (lex.add.entry
   '("sixteenth" n (((s i k) 1) ((s t ii n th) 1)) ((pos "K6%" "OA%"))))
  (lex.add.entry
   '("sony" n (((s ou) 1) ((n ii) 1))))
  (lex.add.entry
   '("SSML" v (((e s) 1) ((e s) 1) ((e m) 1) ((e l) 0))))
  (lex.add.entry 
   '("sun" n (((s uh n) 1))))
  (lex.add.entry
   '("synthesise" v (((s i n th) 1) ((@ s) 0) ((ai z) 0))))
  (lex.add.entry
   '("synthesised" v (((s i n th) 1) ((@ s) 0) ((ai z d) 0))))
  (lex.add.entry
   '("synthesizer" n (((s i n th) 1) ((@ s) 0) ((ai z) 0) ((@) 0))))
  (lex.add.entry
   '("synthesiser" n (((s i n th) 1) ((@ s) 0) ((ai z) 0) ((@) 0))))
  (lex.add.entry
   '("synthesizers" n (((s i n th) 1) ((@ s) 0) ((ai z) 0) ((@ s) 0))))
  (lex.add.entry
   '("synthesisers" n (((s i n th) 1) ((@ s) 0) ((ai z) 0) ((@ s) 0))))
  (lex.add.entry 
   '( "tobi" n ((( t ou ) 1) (( b ii ) 0))))
  (lex.add.entry 
   '("todays" n (((t @ d) 0) ((d ei s) 1))))
  (lex.add.entry 
   '( "tts" n ((( t ii ) 1) (( t ii ) 1) (( e s ) 1))))
  (lex.add.entry 
   '( "ulaw" n ((( m y uu ) 1) (( l oo ) 0))))
  (lex.add.entry
   '("waveform" n  (((w ei v) 1) ((f oo m) 0))))
  (lex.add.entry
   '("winds" n (((w i n d z) 1)) ))
  (lex.add.entry
   '("wind" v (((w ai n d) 1)) ))
  (lex.add.entry 
   '("within" a (((w i dh) 0) ((i n) 1)) ((pos "Pu*" "T-*"))))
  (lex.add.entry 
   '("*" n (((a s) 1) ((t @ r) 0) ((i s k) 0)) ((pos "K6%"))))
  (lex.add.entry 
   '("%" n (((p @ s) 1) ((e n t) 1)) ((pos "K9%"))))
  (lex.add.entry 
   '("&" n (((a m p) 1) ((@@ s) 0) ((a n d) 0))))
  (lex.add.entry 
   '("$" n (((d o l) 1) ((@) 0))))
  (lex.add.entry 
   '("#" n (((h a sh) 1))))
  (lex.add.entry 
   '("@" n (((a t) 1))))
  (lex.add.entry 
   '("+" n (((p l uh s) 0)) ((pos "K7%" "OA%" "T-%"))))
  (lex.add.entry 
   '("^" n (((k a r) 1) ((e t) 0)) ((pos "K6$"))))
  (lex.add.entry 
   '("~" n (((t i l d) 1) ((@) 0)) ((pos "K6$"))))
  (lex.add.entry 
   '("=" n (((ii k) 1) ((w @ l) 0))))
  (lex.add.entry 
   '("/" n (((s l a sh) 1))))
  (lex.add.entry 
   '("\\" n (((b a k) 1) ((s l a sh) 1))))
  (lex.add.entry 
   '("_" n (((uh n d) 1) ((@ s) 0) ((k oo) 1))))
  (lex.add.entry 
   '("|" n (((v @@ t) 1) ((i k l) 0) ((b aa) 1))))
  (lex.add.entry 
   '(">" n ((( g r ei ) 1) ((t @@) 0) ((dh a n) 1))))
  (lex.add.entry 
   '("<" n ((( l e s) 1) ((dh a n) 1))))
  (lex.add.entry 
   '("[" n ((( l e f t) 1) ((b r a k) 1) ((i t) 0))))
  (lex.add.entry 
   '("]" n ((( r ai t) 1) ((b r a k) 1) ((i t) 0))))
  (lex.add.entry 
   '(" " n (((s p ei s) 1))))
  (lex.add.entry 
   '("\t" n (((t a b) 1))))
  (lex.add.entry 
   '("\n" n (((n y uu) 1) ((l ai n) 1))))
  (lex.add.entry '("." punc nil))
  (lex.add.entry '("'" punc nil))
  (lex.add.entry '(":" punc nil))
  (lex.add.entry '(";" punc nil))
  (lex.add.entry '("," punc nil))
  (lex.add.entry '("-" punc nil))
  (lex.add.entry '("\"" punc nil))
  (lex.add.entry '("`" punc nil))
  (lex.add.entry '("?" punc nil))
  (lex.add.entry '("!" punc nil))
  )

(define (cmu_addenda)
  "(cmu_addenda)
Add entries to the current lexicon (radio/darpa).  These are basically
words that are not in the CMU lexicon."
  (lex.add.entry
   '("non-intoxicating" () 
      (((n aa n) 1) ((ih n t) 0) ((aa k) 1) ((s ih k) 0) ((ey t) 1) ((ih ng) 0))))
  (lex.add.entry 
   '("AT&T" n (((ey) 1) ((t iy) 1) ((ah n d) 0) ((t iy) 1))))
  (lex.add.entry 
   '("cepstra" n (((k eh p) 1) ((s t r aa) 0))))
  (lex.add.entry 
   '("cepstral" n (((k eh p) 1) ((s t r ah l) 0))))
  (lex.add.entry 
   '("cepstrum" n (((k eh p) 1) ((s t r ah m) 0))))
  (lex.add.entry
   '("diphone" n (((d ay) 1) ((f ow n) 0))))
  (lex.add.entry
   '("diphones" n (((d ay) 1) ((f ow n s) 0))))
  (lex.add.entry
   '("Dr" n (((d aa k) 1) ((t er r) 0))))
  (lex.add.entry
   '("emacs" n (((iy) 1) ((m ae k s) 0))))
  (lex.add.entry 
   '("hz" n (((hh eh r t z) 1))))
  (lex.add.entry 
   '("khz" n (((k ih) 1) ((l ax) 0) ((hh eh r t z) 1))))
  (lex.add.entry
   '("lived" v (((l ih v d) 1))))
  (lex.add.entry
   '("Ltd" n (((l ih m) 1) ((ah t) 0) ((ah d) 0))))
  (lex.add.entry
   '("Mrs" n (((m ih s) 1) ((ah s) 0))))
  (lex.add.entry 
   '("mhz" n (((m eh) 1) ((g ax) 0) ((hh eh r t z) 1))))
  (lex.add.entry
   '("NASA" n (((n ae) 1) ((s ae) 1))))
  (lex.add.entry
   '("pasadena" n (((p ae s) 1) ((ae d) 0) ((iy n) 1) ((ax) 0))))
  (lex.add.entry
   '("reuters" n (((r oy t) 1) ((er r z) 0))))
  (lex.add.entry
   '("Jan" n (((jh ae n y) 1) ((uw) 0) ((eh r) 1) ((iy) 0))))
  (lex.add.entry
   '("Feb" n (((f eh b) 1) ((r ax) 0) ((er r) 0) ((iy) 0))))
  (lex.add.entry
   '("February" n (((f eh b) 1) ((r ax) 0) ((er r) 0) ((iy) 0))))
  (lex.add.entry
   '("Mar" n (((m aa r ch) 0))))
  (lex.add.entry
   '("Apr" n (((ey p) 1) ((r ah l) 0))))
  (lex.add.entry
   '("Jun" n  (((jh uw n) 1))))
  (lex.add.entry
   '("Jul" n  (((jh uw l) 1) ((ay) 1))))
  (lex.add.entry
   '("Aug" n  (((aa g) 1) ((ah s t) 0))))
  (lex.add.entry
   '("Sep" n (((s eh p) 0) ((t eh m b) 1) ((er r) 0))))
  (lex.add.entry
   '("Sept" n (((s eh p) 0) ((t eh m b) 1) ((er r) 0))))
  (lex.add.entry
   '("Oct" n (((aa k) 0) ((t ow b) 1) ((er r) 0))))
  (lex.add.entry
   '("Nov" n (((n ow v) 0) ((eh m b) 1) ((er r) 0))))
  (lex.add.entry
   '("Dec" n (((d ih s) 0) ((eh m b) 1) ((er r) 0))))
  (lex.add.entry
   '("'s" pos (((ax z) 0))))
  (lex.add.entry 
   '("*" n (((ae s) 1) ((t er r) 0) ((ih s k) 0)) ((pos "K6%"))))
  (lex.add.entry 
   '("%" n (((p er r) 1) ((s eh n t) 1)) ((pos "K9%"))))
  (lex.add.entry 
   '("&" n (((ae m p) 1) ((er r s) 0) ((ae n d) 0))))
  (lex.add.entry 
   '("$" n (((d aa l) 1) ((er r) 0))))
  (lex.add.entry 
   '("#" n (((h ae sh) 1))))
  (lex.add.entry 
   '("@" n (((ae t) 1))))
  (lex.add.entry 
   '("+" n (((p l ah s) 0)) ((pos "K7%" "OA%" "T-%"))))
  (lex.add.entry 
   '("^" n (((k eh r) 1) ((eh t) 0)) ((pos "K6$"))))
  (lex.add.entry 
   '("~" n (((t ih l d) 1) ((ah) 0)) ((pos "K6$"))))
  (lex.add.entry 
   '("=" n (((iy k) 1) ((w ah l) 0))))
  (lex.add.entry 
   '("/" n (((s l ae sh) 1))))
  (lex.add.entry 
   '("\\" n (((b ae k) 1) ((s l ae sh) 1))))
  (lex.add.entry 
   '("_" n (((ah n d) 1) ((er r s) 0) ((k ao r) 1))))
  (lex.add.entry 
   '("|" n (((v er r t) 1) ((ih k) 0) ((ah l) 0) ((b aa r) 1))))
  (lex.add.entry 
   '(">" n ((( g r ey ) 1) ((t er r) 0) ((dh ae n) 1))))
  (lex.add.entry 
   '("<" n ((( l eh s) 1) ((dh ae n) 1))))
  (lex.add.entry 
   '("[" n ((( l eh f t) 1) ((b r ae k) 1) ((ih t) 0))))
  (lex.add.entry 
   '("]" n ((( r ay t) 1) ((b r ae k) 1) ((ih t) 0))))
  (lex.add.entry 
   '(" " n (((s p ey s) 1))))
  (lex.add.entry 
   '("\t" n (((t ae b) 1))))
  (lex.add.entry 
   '("\n" n (((n uw) 1) ((l ay n) 1))))
  (lex.add.entry '("." punc nil))
  (lex.add.entry '("'" punc nil))
  (lex.add.entry '(":" punc nil))
  (lex.add.entry '(";" punc nil))
  (lex.add.entry '("," punc nil))
  (lex.add.entry '("-" punc nil))
  (lex.add.entry '("\"" punc nil))
  (lex.add.entry '("`" punc nil))
  (lex.add.entry '("?" punc nil))
  (lex.add.entry '("!" punc nil))
  (lex.add.entry
   '(before ()
	    (((b iy) 0) ((f ao r) 1))))
  )

(define (lex_user_unknown_word word feats)
  "(lex_user_unknown_word WORD FEATS)
Function called by lexicon when 'function type letter to sound rules
is defined.  It is the user's responsibility to defined this function
themselves when they want to deal with unknown words themselves."
  (error "lex_user_unknown_word: has not been defined by user"))

(provide 'lexicons)

