/*************************************************************************/
/*                                                                       */
/*                Centre for Speech Technology Research                  */
/*                     University of Edinburgh, UK                       */
/*                       Copyright (c) 1996,1997                         */
/*                        All Rights Reserved.                           */
/*                                                                       */
/*  Permission to use, copy, modify, distribute this software and its    */
/*  documentation for research, educational and individual use only, is  */
/*  hereby granted without fee, subject to the following conditions:     */
/*   1. The code must retain the above copyright notice, this list of    */
/*      conditions and the following disclaimer.                         */
/*   2. Any modifications must be clearly marked as such.                */
/*   3. Original authors' names are not deleted.                         */
/*  This software may not be used for commercial purposes without        */
/*  specific prior written permission from the authors.                  */
/*                                                                       */
/*  THE UNIVERSITY OF EDINBURGH AND THE CONTRIBUTORS TO THIS WORK        */
/*  DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING      */
/*  ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT   */
/*  SHALL THE UNIVERSITY OF EDINBURGH NOR THE CONTRIBUTORS BE LIABLE     */
/*  FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES    */
/*  WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN   */
/*  AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,          */
/*  ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF       */
/*  THIS SOFTWARE.                                                       */
/*                                                                       */
/*************************************************************************/
/*                     Author :  Alan W Black                            */
/*                     Date   :  April 1996                              */
/*-----------------------------------------------------------------------*/
/*                                                                       */
/* Phone Name sybsystem, allowing definitions of PhoneSets               */
/* by arbitrary features (and mapping)                                   */
/*                                                                       */
/*=======================================================================*/
#include <stdio.h>
#include "EST_unix.h"
#include "festival.h"
#include "festivalP.h"

static void check_phoneset(void);
static void ps_add_def(const EST_String &name, PhoneSet *ps);
static LISP lisp_select_phoneset(LISP phoneset);

LISP phone_set_list = NULL;

PhoneSet *current_phoneset = NULL;

int Phone::match_features(Phone *foreign)
{
    // Try to match the features of the foreign phone with this one
    EST_TBI *f;

    for (f=features.list.head(); f != 0; f=next(f))
    {	
	if ( features.list(f).v != foreign->val(features.list(f).k))
	    return FALSE;
    }

    return TRUE;

}

PhoneSet::~PhoneSet()
{
    gc_unprotect(&silences);
    gc_unprotect(&map);
    gc_unprotect(&feature_defs);
    gc_unprotect(&phones);
}

Phone * PhoneSet::member(const EST_String &phone) const
{
    LISP p = siod_assoc_str(phone,phones);
    if (p != 0)
	return get_c_phone(car(cdr(p)));
    else 
    {
	cerr << "Phone \"" << phone << "\" not member of " << 
	    psetname << endl;
	return 0;
    }
}

const char *PhoneSet::phnum(const int n) const
{
    // return the nth phone name
    int i;
    LISP p;

    // Should use nth for this, but I want to controll the error case
    for (i=0,p=phones; p != NIL; p=cdr(p),i++)
    {
	if (i == n)
	    return get_c_string(car(car(p)));
    }

    cerr << "Phone (phnum) " << n << " to large, not that many members in " << 
	psetname << endl;
    festival_error();
    return NULL;
}

int PhoneSet::add_phone(Phone *phone)
{
    // Add phone (deleting existing one with warning)
    LISP lpair;

    lpair = siod_assoc_str(phone->phone_name(),phones);

    if (lpair == NIL)
    {
	phones = cons(make_param_lisp(phone->phone_name(),phone_cons(phone)),
		      phones);
	return TRUE;
    }
    else
	return FALSE;
}

void PhoneSet::set_feature(const EST_String &name, LISP vals)
{
    LISP lpair;

    lpair = siod_assoc_str(name, feature_defs);

    if (lpair == NIL)
	feature_defs = cons(make_param_lisp(name,vals),feature_defs);
    else
    {
	cerr << "PhoneSet: replacing feature definition of " <<
	    name << " PhoneSet " << psetname << endl;
	CAR(cdr(lpair)) = vals;
    }
}

int PhoneSet::phnum(const char *phone) const
{
    // Return a unique number for this phone, i.e. position in list
    int i;
    LISP p;

    for (i=0,p=phones; p != NIL; p=cdr(p),i++)
    {
	if (streq(phone,get_c_string(car(car(p)))))
	    return i;
    }

    cerr << "Phone \"" << phone << "\" not member of " << 
	psetname << endl;
    festival_error();

    return -1;
}

Phone *PhoneSet::find_matched_phone(Phone *foreign)
{
    // find a phone in the current set that matches the features
    // in foreign (from a different phoneset)
    LISP p;

    for (p=phones; p != NIL; p=cdr(p))
    {
	if (get_c_phone(car(cdr(car(p))))->match_features(foreign))
	    return get_c_phone(car(cdr(car(p))));
    }

    // could try harder 

    cerr << "Cannot map phoneme " << *foreign << endl;
    festival_error();

    return 0;

}

int PhoneSet::is_silence(const EST_String &ph) const
{
    // TRUE is ph is silence
    
    return (siod_member_str(ph,silences) != NIL);

}

void PhoneSet::set_silences(LISP sils)
{
    silences=sils;
}

void PhoneSet::set_map(LISP m)
{
    map=m;
}

LISP make_phoneset(LISP args,LISP env)
{
    // define a new phoneme set 
    (void)env;
    PhoneSet *ps = new PhoneSet;
    Phone *phone;
    LISP f,p,pv;
    LISP name, features, phones;
    EST_String feat,val;
    int num_feats;

    name = car(args);
    features = car(cdr(args));
    phones = car(cdr(cdr(args)));

    ps->set_phone_set_name(get_c_string(name));
    // Define the phonetic features
    num_feats = siod_llength(features);
    for (f=features; f != NIL; f=cdr(f))
	ps->set_feature(get_c_string(car(car(f))),cdr(car(f)));

    // Define the phones
    for (p=phones; p != NIL; p=cdr(p))
    {
	if (siod_llength(cdr(car(p))) != num_feats)
	{
	    cerr << "Wrong number of phone features for "
		<< get_c_string(car(car(p))) << " in " <<
		    get_c_string(name) << endl;
	    festival_error();
	}
	phone = new Phone;
	phone->set_phone_name(get_c_string(car(car(p))));
	for (pv=cdr(car(p)),f=features; f != NIL; pv=cdr(pv),f=cdr(f))
	{
	    feat = get_c_string(car(car(f)));
	    val = get_c_string(car(pv));
	    if (ps->feat_val(feat,val))
		phone->add_feat(feat,val);
	    else
	    {
		cerr << "Phone " << phone->phone_name() << 
		    " has invalid value "
		    << get_c_string(car(pv)) << " for feature "
			<< feat << endl;
		festival_error();
	    }
	}
	if (ps->add_phone(phone) == FALSE)
	{
	    cerr << "Phone " << phone->phone_name() << 
		" multiply defined " << endl;
	    festival_error();
	}
    }

    ps_add_def(ps->phone_set_name(),ps);
    current_phoneset = ps;  // selects this one as current 

    return NIL;
}

static LISP lisp_set_silence(LISP silences)
{
    // Set list of names as silences for current phoneset

    check_phoneset();
    current_phoneset->set_silences(silences);
    return silences;
}

PhoneSet *phoneset_name_to_set(const EST_String &name)
{
    LISP lpair = siod_assoc_str(name,phone_set_list);

    if (lpair == NIL)
    {
	cerr << "Phoneset " << name << " not defined" << endl;
	festival_error();
    }
    
	return (PhoneSet *)PTRVAL(car(cdr(lpair)));

}

static LISP lisp_select_phoneset(LISP phoneset)
{
    // Select named phoneset and make it current
    EST_String name = get_c_string(phoneset);
    LISP lpair;

    lpair = siod_assoc_str(name,phone_set_list);

    if (lpair == NIL)
    {
	cerr << "Phoneset " << name << " not defined" << endl;
	festival_error();
    }
    else
	current_phoneset = (PhoneSet *)PTRVAL(car(cdr(lpair)));

    return phoneset;

}

static void ps_add_def(const EST_String &name, PhoneSet *ps)
{
    //  Add phoneset to list of phonesets
    LISP lpair;
    PhoneSet *old_ps;

    lpair = siod_assoc_str(name,phone_set_list);
    
    if (lpair == NIL)
    {
	phone_set_list = cons(cons(rintern(name),
				   cons(MKPTR(ps),NIL)),
			      phone_set_list);
	gc_protect(&phone_set_list);
    }
    else
    {
	cwarn << "Phoneset \"" << name << "\" redefined" << endl;
	old_ps = (PhoneSet *)PTRVAL(car(cdr(lpair)));
	delete old_ps;
	PTRVAL(car(cdr(lpair))) = ps;
    }

    return;
}

static void check_phoneset(void)
{
    // check if there is a phoneset defined 
    
    if (current_phoneset == NULL)
    {
	cerr << "No phoneset currently selected";
	festival_error();
    }
}

static EST_Val ff_ph_feature(EST_Utterance &u,EST_Stream_Item &s,
			     const EST_String &name)
{
    // This function is called for all phone feature 
    // It looks at the name ff_feature_name to find out the
    // the actual name used to call this feature and removed the
    // ph_ prefix and uses the remainder as the phone feature name
    (void)u;
    EST_String fname, rrr;
    Phone *phone_def;

    if (!name.contains("ph_",0))
    {
	cerr << "Not a phone feature function " << name << endl;
	festival_error();
    }

    check_phoneset();

    fname = name.after("ph_");
    phone_def = current_phoneset->member(s.name());
    if (phone_def == 0)
    {
	cerr << "Phone " << s.name() << " not in phone set " <<
	    current_phoneset->phone_set_name() << endl;
	festival_error();
    }

    rrr = phone_def->val(fname,"");
    if (rrr == "") 
    {
	cerr << "Phone " << s.name() << " does not have feature " <<
	    fname << endl;
	festival_error();
    }

    return EST_Val(rrr);

}

static PhoneSet *find_phoneset(EST_String name)
{
    // get the phone set from the phone set list
    LISP lpair;

    lpair = siod_assoc_str(name,phone_set_list);

    if (lpair == NIL)
    {
	cerr << "Phoneset " << name << " not defined" << endl;
	festival_error();
    }
	return (PhoneSet *)PTRVAL(car(cdr(lpair)));

}

const EST_String &map_phone(const EST_String &fromphonename, const EST_String &fromsetname,
			const EST_String &tosetname)
{
    PhoneSet *fromset, *toset;
    Phone *fromphone, *tophone;

    fromset = find_phoneset(fromsetname);
    toset = find_phoneset(tosetname);

    // should check specific matches in fromset first
    fromphone = fromset->member(fromphonename);
    if (fromphone == 0)
	festival_error();
    tophone = toset->find_matched_phone(fromphone);

    return tophone->phone_name();
}

int ph_is_silence(const EST_String &ph)
{
    // TRUE if this phone is silence

    check_phoneset();
    return current_phoneset->is_silence(ph);

}

EST_String ph_silence(void)
{
    // return the first silence in the current_phoneset
    EST_String s;

    check_phoneset();
    
    if (current_phoneset->get_silences() == NIL)
    {
	cerr << "No silences set for " << current_phoneset->phone_set_name()
	    << endl;
	return "sil";
    }
    else
	return get_c_string(car(current_phoneset->get_silences()));

}

int ph_is_vowel(const EST_String &ph)
{
    // TRUE if this phone is a vowel, assumes the feature vc is used

    return (ph_feat(ph,"vc") == "+");
}

int ph_is_consonant(const EST_String &ph)
{
    // TRUE if this phone is a consonant, assumes the feature vc is used

    return ((ph_feat(ph,"vc") == "-") &&
	    !(ph_is_silence(ph)));
}

int ph_is_liquid(const EST_String &ph)
{
    // TRUE if this phone is a liquid

    return (ph_feat(ph,"ctype") == "l");
}

int ph_is_stop(const EST_String &ph)
{
    // TRUE if this phone is a stop

    return (ph_feat(ph,"ctype") == "s");
}

int ph_is_fricative(const EST_String &ph)
{
    // TRUE if this phone is a stop

    return (ph_feat(ph,"ctype") == "f");
}

int ph_is_nasal(const EST_String &ph)
{
    // TRUE if this phone is a nasal

    return (ph_feat(ph,"ctype") == "n");
}

int ph_is_obstruent(const EST_String &ph)
{
    // TRUE if this phone is a obstruent
    EST_String v = ph_feat(ph,"ctype");

    return ((v == "s") || (v == "f") || (v == "a"));
}

int ph_is_sonorant(const EST_String &ph)
{
    // TRUE if this phone is a sonorant

    return !ph_is_obstruent(ph);
}

int ph_is_voiced(const EST_String &ph)
{
    // TRUE if this phone is a sonorant

    return (ph_feat(ph,"cvox") == "+");
}

int ph_is_syllabic(const EST_String &ph)
{
    // TRUE if this phone is a syllabic consonant (or vowel)
    // Yes I know we just don't have this ...

    return (ph_feat(ph,"vc") == "+");
}

const EST_String &ph_feat(const EST_String &ph,const EST_String &feat)
{
    // Values for this phone -- error is phone of feat doesn't exist
    Phone *phone_def;
    EST_String rrr;

    check_phoneset();
    phone_def = current_phoneset->member(ph);
    if (phone_def == 0)
    {
	cerr << "Phone " << ph << " not in phone set " <<
	    current_phoneset->phone_set_name() << endl;
	festival_error();
    }

    return phone_def->val(feat,"");

}

int ph_sonority(const EST_String &ph)
{
    // assumes standard phone features
    Phone *p;

    check_phoneset();
    p = current_phoneset->member(ph);
    if (p == 0)
	return 1;
    
    if (p->val("vc") == "+")
	return 5;
    else if (p->val("ctype") == "l") // || glide 
        return 4;
    else if (p->val("ctype") == "n")
        return 3;
    else if (p->val("cvox") == "+") // voiced obstruents (stop fric affric)
        return 2;
    else
        return 1;

}

LISP l_phoneset(LISP options)
{
    //  Return Lisp form of current phone set 
    LISP description=NIL;

    check_phoneset();

    if ((options == NIL) ||
	(siod_member_str("silences",options)))
    {
	description = cons(make_param_lisp("silences",
					   current_phoneset->get_silences()),
			   description);
    }
    if ((options == NIL) ||
	(siod_member_str("phones",options)))
    {
	LISP phones = current_phoneset->get_phones();
	LISP features = current_phoneset->get_feature_defs();
	LISP p,f,p_desc=NIL,f_desc=NIL;
	
	for (p=phones; p != NIL; p=cdr(p))
	{
	    f_desc = NIL;
	    for (f=features; f != NIL; f=cdr(f))
	    {
		f_desc = cons(rintern(ph_feat(get_c_string(car(car(p))),
						get_c_string(car(car(f))))),
			      f_desc);
	    }
	    p_desc = cons(cons(car(car(p)),f_desc),p_desc);
	}
	description = cons(make_param_lisp("phones",p_desc),description);
    }
    if ((options == NIL) ||
	(siod_member_str("features",options)))
    {
	description = cons(make_param_lisp("features",
					   current_phoneset->get_feature_defs()),
			   description);
    }
    if ((options == NIL) ||
	(siod_member_str("name",options)))
    {
	description = cons(make_param_str("name",
					  current_phoneset->phone_set_name()),
			   description);
    }

    return description;
}

LISP l_phoneset_list()
{
    LISP phonesets = NIL;
    LISP l;

    for (l=phone_set_list; l != NIL; l=cdr(l))
	phonesets = cons(car(car(l)),phonesets);

    return phonesets;
}

void festival_phoneset_funcs(void)
{
    // define Lisp accessor functions 

    init_fsubr("defPhoneSet",make_phoneset,
 "(defPhoneSet PHONESETNAME FEATURES PHONEDEFS)\n\
  Define a new phoneset named PHONESETNAME.  Each phone is described with a\n\
  set of features as described in FEATURES.  Some of these FEATURES may\n\
  be significant in various parts of the system.  Copying an existing\n\
  description is a good start. [see Phonesets]");
    init_subr_1("PhoneSet.select",lisp_select_phoneset,
 "(PhoneSet.select PHONESETNAME)\n\
  Select PHONESETNAME as current phoneset. [see Phonesets]");
    init_subr_1("PhoneSet.silences",lisp_set_silence,
 "(PhoneSet.silences LIST)\n\
  Declare LIST of phones as silences.  The first in the list should be\n\
  the \"most\" silent. [see Phonesets]");
    init_subr_1("PhoneSet.description",l_phoneset,
 "(Phoneset.description OPTIONS)\n\
  Returns a lisp for of the current phoneme set.  Options is a list of\n\
  parts of the definition you require.  OPTIONS may include, silences,\n\
  phones, features and/or name.  If nil all are returned.");
    init_subr_0("PhoneSet.list",l_phoneset_list,
 "(Phoneset.list)\n\
  List the names of all currently defined Phonesets.");
    // All feature functions starting with "ph_"
    festival_def_ff_pref("ph_","Segment",ff_ph_feature,
    "Segment.ph_*\n\
  Access phoneset features for a segment.  This definition covers multiple\n\
  feature functions where ph_ may be extended with any features that\n\
  are defined in the phoneset (e.g. vc, vlng, cplace etc.).");

}

