 /************************************************************************/
 /*                                                                      */
 /*                Centre for Speech Technology Research                 */
 /*                     University of Edinburgh, UK                      */
 /*                       Copyright (c) 1996,1997                        */
 /*                        All Rights Reserved.                          */
 /*                                                                      */
 /*  Permission to use, copy, modify, distribute this software and its   */
 /*  documentation for research, educational and individual use only, is */
 /*  hereby granted without fee, subject to the following conditions:    */
 /*   1. The code must retain the above copyright notice, this list of   */
 /*      conditions and the following disclaimer.                        */
 /*   2. Any modifications must be clearly marked as such.               */
 /*   3. Original authors' names are not deleted.                        */
 /*  This software may not be used for commercial purposes without       */
 /*  specific prior written permission from the authors.                 */
 /*                                                                      */
 /*  THE UNIVERSITY OF EDINBURGH AND THE CONTRIBUTORS TO THIS WORK       */
 /*  DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING     */
 /*  ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT  */
 /*  SHALL THE UNIVERSITY OF EDINBURGH NOR THE CONTRIBUTORS BE LIABLE    */
 /*  FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES   */
 /*  WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN  */
 /*  AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,         */
 /*  ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF      */
 /*  THIS SOFTWARE.                                                      */
 /*                                                                      */
 /************************************************************************/
 /*                 Author: Richard Caley (rjc@cstr.ed.ac.uk)            */
 /*                   Date: Thu Mar 13 1997                              */
 /************************************************************************/
 /*                                                                      */
 /* Databases which consist of indivisual files in the filesystem,       */
 /* textual cataloges and indecies and so on.                            */
 /*                                                                      */
 /************************************************************************/

#ifndef __RAWUNITDATABASE_H__
#define __RAWUNITDATABASE_H__

#include "UnitDatabase.h"

#define RUC_NAME_LENGTH (10)
#define RUC_FILENAME_LENGTH (10)
#define RUC_MAX_SEGMENTS_PER_UNIT (10)

class RawUnitCatalogue : public UnitCatalogue {

public:

  class Itterator : UnitCatalogue::Itterator {
  public:
    Itterator(void);
    virtual ~Itterator(void);
    virtual UnitName next(void);
    virtual int at_end(void);
  };

  typedef struct Entry {
    char name[RUC_NAME_LENGTH+1];
    char filename[RUC_FILENAME_LENGTH+1];
    int n_segments;
    float segments[RUC_MAX_SEGMENTS_PER_UNIT];
  } Entry;

private:

  static const int s_line_length;

  EST_TStringHash<Entry *> p_entries;
  int p_num_entries;

  RawUnitCatalogue(void);
  virtual ~RawUnitCatalogue(void);

  EST_read_status fill_from(FILE *stream);
  EST_write_status write_to(FILE *stream);

  UnitCatalogue::Itterator *itterator(void);
  struct Entry *lookup(EST_String name); 

public:
  virtual int num_entries (void) {return p_num_entries; };
  virtual int has_entry(EST_String name);

  friend class RawUnitDatabase;
};

class RawUnitDatabase : public UnitDatabase {

public:
  struct available_data {
    ContentType contents;
    EST_Pathname directory;
    EST_String extension;
    struct UnitDatabase::wave_information_s wave_info;
    struct UnitDatabase::coefficient_information_s coef_info;
    int is_waveform(void) { return UnitDatabase::is_waveform(contents); };
  };

private:

  static const char s_magic[];
  static const int s_magic_size;

  EST_Pathname p_catalogue_filename;
  int p_sample_rate;

  int p_num_data;
  struct available_data *p_data;

  int process_data_included(ValueType value);
  int process_indexes(ValueType value);
public:
  RawUnitDatabase(void);
  virtual ~RawUnitDatabase(void);

  static bool recognise(FILE *stream);
  static UnitDatabase *create(void);

  virtual EST_String subtype_name(void) const {return "Raw";};

  virtual EST_read_status fill_from(FILE *stream);
  virtual EST_write_status write_to(FILE *stream);

  virtual EST_StrVector wave_names(void) const;
  virtual EST_StrVector coefficient_names(void) const;
  virtual EST_StrVector unit_names(void) const;

  virtual struct wave_information_s *wave_information(ContentType content);
  virtual struct coefficient_information_s *coefficient_information(ContentType content);

  virtual EST_Wave *get_wave(ContentType content, Unit &unit);
  virtual EST_Track *get_coefficients(ContentType content,  EST_ContourType format, Unit &unit);
  virtual EST_TVector<float> *get_segments(Unit &unit);

  virtual void property_names(EST_TList<EST_String> &list) const;
  virtual ValueType property(EST_String property) const;
  virtual int set_property(EST_String property, ValueType value);

  virtual void print_description(FILE *stream);

};

#endif
