(* Styles are common display attributes *)
open Htmlfmt
open Fonts

(* Configuration *)
let default_family   = ref "helvetica"
and default_size     = ref 12
and default_bgcolor  = ref "white"
and default_fgcolor  = ref "black"

let table = (Hashtbl.create 37 : (string, gattr list) Hashtbl.t)
let fonttable = Hashtbl.create 37

let get s = 
  let fontattrs = 
   try Hashtbl.find fonttable s with Not_found -> []
  and otherattrs =
   try Hashtbl.find table s with Not_found -> []
  in
  let attrs =  List.map (fun fi -> Font fi) fontattrs @ otherattrs 
  in
   if attrs = [] then raise Not_found else attrs

let set_font name attrs =
  Hashtbl.remove fonttable name;
  Hashtbl.add fonttable name attrs;
  if name = "default" then 
    Fonts.default := Fonts.merge !Fonts.default attrs

let init family slant =
  set_font "default" [Family family;  Weight "medium"; Slant "r";  FontIndex 3];
  set_font "header1" [Family family;  Weight "bold"; Slant "r"; FontIndex 7];
  set_font "header2" [Family family;  Weight "bold"; Slant "r"; FontIndex 6];
  set_font "header3" [Family family;  Weight "medium"; Slant slant; FontIndex 5];
  set_font "header4" [Family family;  Weight "bold"; Slant "r"; FontIndex 4];
  set_font "header5" [Family family;  Weight "medium"; Slant slant; FontIndex 4];
  set_font "header6" [Family family;  Weight "bold"; Slant "r"; FontIndex 4];
  set_font "bold" [ Weight "bold"];
  set_font "italic" [ Slant slant];
  (* should be a fixed font. Since we have newlines, spacing should be 0 *)
  set_font "verbatim" [Family "courier"];
  set_font "fixed" [Family "courier"];
  Hashtbl.add table "verbatim" [Spacing 0];
  Hashtbl.add table "anchor" [FgColor "blue"; Underlined];
  Hashtbl.add table "header1" [Justification "center"; Spacing 20];
  Hashtbl.add table "header2" [Justification "center"; Spacing 10];
  Hashtbl.add table "header3" [Justification "left"; Spacing 10];
  Hashtbl.add table "header4" [Justification "left"; Spacing 5];
  Hashtbl.add table "header5" [Justification "left"];
  Hashtbl.add table "header6" [Justification "left"]


(* For the preferences *)
let get_font =  Hashtbl.find fonttable


