
package jde.debugger;

import jde.debugger.spec.*;
import jde.debugger.expr.*;

import com.sun.jdi.*;
import com.sun.jdi.event.*;
import com.sun.jdi.connect.*;
import com.sun.jdi.request.*;

import java.io.*;
import java.net.*;
import java.util.*;

/**
 * Application.java
 * <p>
 * Is the most important class of the jdebug: it holds control of what and
 * how happens to the application, as it reacts to commands it receives from
 * the jde side, and the events it receives from the debugee vm side.
 * <p>
 * Created: Wed Jul  7 16:16:59 1999
 *
 * @author Amit Kumar
 * @since 0.1
 */

public class Application implements Protocol {

    /**
     * Some classes require a unique ID with which to refer to objects
     * they are tracking: for instance eventRequestSpecs, which need a spec
     * ID with which to identify the specs, and identifiableSpecRequests
     * (in ApplicationCommands)
     * <p>
     * This variable keeps a monotonically increasing count, and can be
     * used to get a new id, using {@link #getNewID}
     */
    private long id = 0;
    public Long getNewID() {
	// synchronize on any static object
	synchronized (Jdebug.my_id) {
	    return new Long(id++);
	}
    }

    /** The ID that uniquely identifies this application in jdebug. */
    final Integer my_id;	

    /** Represents the jde. {@link Jdebug} manages the jdebug link to jde. */
    final Jdebug jdebug;

    /** Socket connection to do i/o */
    Socket mySocket = null;

    /**
     * The {@link EventHandler} manages the events received from the
     * debugee vm
     */
    EventHandler eventHandler;

    /** The debugee VM */
    VirtualMachine vm;	        

    /**
     * {@link jde.debugger.spec.EventRequestSpecList} is responsible for
     * keeping track of the events that the user is interested in. To do
     * this, it maintains a list of "eventRequestSpec"s.
     *
     * @see jde.debugger.spec.EventRequestSpecList
     * @see jde.debugger.spec.EventRequestSpec
     */
    EventRequestSpecList eventRequestSpecs;

    /**
     * A collection of the commands for which the reply (result/error) has
     * not been sent.
     * @see Jdebug#pendingCommands
     */
    Collection pendingCommands;

    /**
     * A store of all the objects jde knows about.
     * @see ObjectStore
     */
    ObjectStore store;

    /* Command handlers */
    GeneralCommands generalCommands;
    SpecCommands specCommands;
    ThreadCommands threadCommands;

    /**
     * keeps track of the state of the application: exceptions/error messages
     * will not be raised if we're shutting down.
     */
    private boolean shuttingDown = false;
    private boolean isShuttingDown() { return shuttingDown; }
     

    /**
     * The constructor.
     *
     * @param jdebug {@link Jdebug} class
     * @param app_id My identifying identifier
     */
    public Application(Jdebug jdebug, Integer app_id) {
        this.jdebug = jdebug;
        this.my_id = app_id;

        pendingCommands = Collections.synchronizedSet(new HashSet());

	store = new ObjectStore(this);
    }

    /**
     * Start off
     *
     * @param connectType Specifies how to connect to the application. We're
     * only supporting 'launch' right now.
     * @param args A list of arguments: i does NOT contain the app_id at
     * index 0, that has already been stripped out.
     */
    public int initialize(String connectType, List args) 
        throws JDEException {

	// note that ioport makes sense only for launching connectors. the
	// other connectors connect to an app that's already managing it's
	// own i/o.
	int ioport = -1;
	
	// create a connection according to the kind of connection
	// required: whether launch, or attach etc. currently we're
	// only supporting launching of debuggee vms. in the future,
	// during the handshake, jde could request for some other
	// kind of connectors, say, ATTACH, with corresponding
	// arguments. then we'll simply need to call this constructor
	// with a connectType=ATTACH, and it will try to do the Right
	// Thing.

	// for launch, we need to know the command line of the app to
	// launch. for example, if this is the command line usually used
	// to launch the app test.Test:
	// java test.Test "Some phrase" 123 word
	// the command that comes from jde should be something like:
	// -1 32 launch 2 test.Test "Some phrase" 123 word
	// the list of args we get in this function, then, will be:
	// (test.Test), (Some phrase), (123), (word)
	// we have to be careful never to lose the fact that "Some phrase"
	// has an embedded space that needs to be passed correctly to the
	// app.
	if (connectType.equals(LAUNCH)) {
	    // this is the connector that launches a debuggee vm
	    String connectSpec = "com.sun.jdi.CommandLineLaunch";

	    // check if this kind of connector is, indeed,
	    // available. if not, throw an exception.
	    Connector connector;
	    if ((connector = getConnector(connectSpec)) == null) 
		throw new JDEException("No such connector is avaliable: "+connectSpec);

	    ioport = launch((LaunchingConnector)connector, args);
	}
	// attaching through a socket/shared memory
	else if (connectType.equals(ATTACH_SOCKET)
		 || connectType.equals(ATTACH_SHMEM)) {

	    // the attaching connector...
	    String connectSpec = null;
	    if (connectType.equals(ATTACH_SOCKET))
		connectSpec = "com.sun.jdi.SocketAttach";
	    else
		connectSpec = "com.sun.jdi.SharedMemoryAttach";

	    Connector connector;
	    if ((connector = getConnector(connectSpec)) == null) 
		throw new JDEException("No such connector is available: "+connectSpec);

	    if (connectType.equals(ATTACH_SOCKET))
		attachSocket((AttachingConnector)connector, args);
	    else
		attachShmem((AttachingConnector)connector, args);
	}
	// listening
	else if (connectType.equals(LISTEN_SOCKET)
		 || connectType.equals(LISTEN_SHMEM)) {

	    // the listening connector...
	    String connectSpec = null;
	    if (connectType.equals(LISTEN_SOCKET))
		connectSpec = "com.sun.jdi.SocketListen";
	    else
		connectSpec = "com.sun.jdi.SharedMemoryListen";

	    Connector connector;
	    if ((connector = getConnector(connectSpec)) == null) 
		throw new JDEException("No such connector is available: "+connectSpec);

	    if (connectType.equals(LISTEN_SOCKET))
		listenSocket((ListeningConnector)connector, args);
	    else
		listenShmem((ListeningConnector)connector, args);
	}

	eventHandler = new EventHandler(this);

	eventRequestSpecs = new EventRequestSpecList(this);

	// we need to raise all class prepare events to 
	// make sure we resolve the corresponding specs.
	ClassPrepareRequest cprequest =
	    vm.eventRequestManager().createClassPrepareRequest();
	// this (hack?) is used to identify if the user itself specified
	// a class prepare request, or the event was raised because of
	// this request.
	cprequest.putProperty("default", "default");
	cprequest.setSuspendPolicy(EventRequest.SUSPEND_ALL);
	cprequest.enable();

	// set up command handling classes.
	generalCommands =
	    new GeneralCommands(this, store);
	specCommands =
	    new SpecCommands(this,eventRequestSpecs,store);
	threadCommands =
	    new ThreadCommands(this, store);

	return ioport;
    }

    public VirtualMachine getVM() { return vm; }

    public Integer getId() { return my_id; }

    public ObjectStore getStore() { return store; }

    public EventQueue getEventQueue() { return vm.eventQueue(); }

    public void shutdown() {
	// we need to be a little considerate to the output. given
	// there are all kinds of threads running, we make sure we
	// wait for all the output to complete.

	shuttingDown = true;
	
	// isolate the process first
	Process process = null;
	if (vm != null) 
	    process = vm.process();
	try {
	    if (vm != null) {
		vm.dispose();
		vm = null;
		eventHandler.shutdown();
	    }
	    mySocket.close();
	} catch (Exception ex) {
	    // do nothing
	} finally {
	    if (process != null) {
		process.destroy();
		// XXX sun's jdb implementation works a lot to make sure
		// the stderr and stdout are dumped before killing
		// things. i'm not sure how important it is, or even how
		// well it works (given i can't test it)
		// sooo, if the reader finds bugs with the output handling
		// on finish, lemme know.
	    }
	    jdebug.removeApplication(my_id);
	}
    }




    /*
     *
     * CONNECTING TO THE DEBUGEE VM
     *
     */


    
    private Connector getConnector(String name) {
        List connectors = Bootstrap.virtualMachineManager().allConnectors();
        Iterator iter = connectors.iterator();
        while (iter.hasNext()) {
            Connector connector = (Connector)iter.next();
            if (connector.name().equals(name)) {
                return connector;
            }
        }
        return null;
    }

    /**
     * launches a new vm,and returns the port no. of the app i/o
     */
    private int launch(LaunchingConnector connector, List args)
	throws JDEException {

	// first set up the argument map. a table that describes the
	// different keys should be in the public jpda documentation.
	// internally, it is at
	// http://jbug/jbug/doc/conninv.html
	Map argumentMap = connector.defaultArguments();
	
	Connector.Argument mainArg =
	    (Connector.Argument)argumentMap.get("main");

	// compose the command line
	String commandLine = "";
	String quote =((Connector.Argument)argumentMap.get("quote")).value();

	// check if there are special launch options we need to process
	if (args.size() == 0)
	    throw new JDEException("Insufficient arguments");
	
	String executable = "java";
	// be careful with the loop here....
	while ((args.size() >0)
	       && args.get(0).toString().startsWith("-")) {
	    String arg = args.remove(0).toString().toLowerCase();
	    if (arg.equals("-use_executable")) {
		if (args.size() == 0)
		    throw new JDEException("Missing argument to 'use_executable'");
		executable = args.remove(0).toString();
		Connector.Argument vmexecArg =
		    (Connector.Argument)argumentMap.get("vmexec");
		vmexecArg.setValue(executable);
		continue;
	    } else {
		args.add(0, arg);
		break;
	    }
	}
	
	if (args.size() == 0)
	    throw new JDEException("Missing arguments: no class specified?");
	
	// take care of spaces too! so quote everything.
	Iterator iterator = args.iterator();
	while(iterator.hasNext()) {
	  // commandLine += quote + iterator.next() + quote + " ";
	  String arg = (String)iterator.next();
	  if (arg.equalsIgnoreCase("-classic")) {
	    Connector.Argument optionsArg =
	      (Connector.Argument)argumentMap.get("options");
	    String options = optionsArg.value();
	    options = "-classic" + " " + options;
	    optionsArg.setValue(options);
	    signal(MESSAGE, "VM options: '" + options + "'");
	  }
	  else
	    commandLine += quote + arg + quote + " ";
	}
	mainArg.setValue(commandLine);
	
	//	    signal(DEBUG, "Command line: '"+executable+" "+commandLine+"'");

	try {
	    vm = connector.launch(argumentMap);
	    signal(MESSAGE, "Launched VM " + vm.description());
	} catch (IOException ex) {
	    Debug.printIf(ex);
	    throw new JDEException("Unable to launch: "+ex);
	} catch (IllegalConnectorArgumentsException ex) {
	    throw new JDEException("Invalid or inconsistent connector arguments for connector '"+connector+"'");
	} catch (VMStartException ex) {
	    dumpFailedAppStreams(ex.process());
	    throw new JDEException(ex.getMessage());
	}

	int port = manageAppStreams(vm.process());
	return port;
    }

    /** attaches to a vm via a socket */
    private void attachSocket(AttachingConnector connector, List args)
	throws JDEException {

	if (args.size() < 1)
	    throw new JDEException("Missing arguments: specify at least the port");

	try {
	    Map argumentMap = connector.defaultArguments();

	    while ((args.size() > 0)
		   && args.get(0).toString().startsWith("-")) {
		String arg = args.remove(0).toString().toLowerCase();
		if (arg.equals("-host")) {
		    if (args.size() == 0)
			throw new JDEException("Missing argument to 'host'");
		    String host = args.remove(0).toString();
		    Connector.Argument hostArg =
			(Connector.Argument)argumentMap.get("host");
		    hostArg.setValue(host);
		} else if (arg.equals("-port")) {
		    if (args.size() == 0)
			throw new JDEException("Missing argument to 'port'");
		    String port = args.remove(0).toString();
		    Connector.Argument portArg =
			(Connector.Argument)argumentMap.get("port");
		    portArg.setValue(port);
		} else {
		    args.add(0, arg);
		    break;
		}
	    }

	    vm = connector.attach(argumentMap);
	    signal(MESSAGE, "Attached VM (socket) " + vm.description());

	} catch (IOException ex) {
	    Debug.printIf(ex);
	    throw new JDEException("Error occured managing application streams");
	} catch (IllegalConnectorArgumentsException ex) {
	    throw new JDEException("Illegal connector arguments for connector '"+connector);
	}
    }

    
    /** attaches to a vm via shared memory */
    private void attachShmem(AttachingConnector connector, List args)
	throws JDEException {

	if (args.size() < 1)
	    throw new JDEException("Missing name");

	try {
	    Map argumentMap = connector.defaultArguments();

	    Connector.Argument nameArg =
		(Connector.Argument)argumentMap.get("name");
	    nameArg.setValue(args.remove(0).toString());

	    vm = connector.attach(argumentMap);
	    signal(MESSAGE, "Attached VM (shmem) " + vm.description());

	} catch (IOException ex) {
	    Debug.printIf(ex);
	    throw new JDEException("Error occured managing application streams");
	} catch (IllegalConnectorArgumentsException ex) {
	    throw new JDEException("Illegal connector arguments for connector '"+connector);
	}
    }

    /** listenes to a vm via a socket */
    private void listenSocket(final ListeningConnector connector, List args)
	throws JDEException {

	if (args.size() < 1)
	    throw new JDEException("Missing arguments: specify at least the port");

	try {
	    final Map argumentMap = connector.defaultArguments();

	    final String port = args.remove(0).toString();
	    Connector.Argument portArg =
		(Connector.Argument)argumentMap.get("port");
	    portArg.setValue(port);

	    connector.startListening(argumentMap);
	    /*	    Thread tmp = new Thread("Waiting for an accept") {
		    public void run() {
		    try { */
			    vm = connector.accept(argumentMap);
			    connector.stopListening(argumentMap);
			    /*
			    signal(CONNECTED_TO_VM, port);
			} catch (IOException ex) {
			    Debug.printIf(ex);
			    signal(ERROR, "Error occured managing application streams");
			} catch(IllegalConnectorArgumentsException ex) {
			    signal(ERROR, "Illegal connector arguments for connector '"+connector);
			}
		    }
		};
		tmp.start();*/
	} catch (IOException ex) {
	    Debug.printIf(ex);
	    throw new JDEException("Error occured managing application streams");
	} catch(IllegalConnectorArgumentsException ex) {
	    throw new JDEException("Illegal connector arguments for connector '"+connector);
	}
    }

    
    /** listenes to a vm via shared memory */
    private void listenShmem(ListeningConnector connector, List args)
	throws JDEException {

	if (args.size() < 1)
	    throw new JDEException("Missing name");

	try {
	    Map argumentMap = connector.defaultArguments();

	    Connector.Argument nameArg =
		(Connector.Argument)argumentMap.get("name");
	    nameArg.setValue(args.remove(0).toString());

	    connector.startListening(argumentMap);
	    vm = connector.accept(argumentMap);
	    connector.stopListening(argumentMap);
	} catch (IOException ex) {
	    Debug.printIf(ex);
	    throw new JDEException("Error occured managing application streams");
	} catch (IllegalConnectorArgumentsException ex) {
	    throw new JDEException("Illegal connector arguments for connector '"+connector);
	}
    }

    
    /*
     * FUNCTIONS TO MANAGE IO STREAMS
     */

    

    private int manageAppStreams(final Process process)
	throws JDEException {
	ServerSocket ss = null;
	try {
	    ss = new ServerSocket(0);
	} catch (IOException ex) {
	    throw new JDEException("Unable to create a server socket");
	}

	final ServerSocket sstmp = ss;
	final int port = ss.getLocalPort();
	Thread thr = new Thread("Temporary Thread for App #"+my_id) {
		public void run() {
		    try {
			mySocket = sstmp.accept();
			sstmp.close();
		    } catch (IOException ex) {
			shutdown();
		    }
		    processRemoteInput(process.getOutputStream());
		    displayRemoteOutput(process.getInputStream());
		    displayRemoteOutput(process.getErrorStream());
		}
	    };
	    thr.start();
	return port;
    }


    /** after an app fails, dump its output */
    private void dumpFailedAppStreams(Process process) {
	try {
	    dumpStream(process.getInputStream());
	    dumpStream(process.getErrorStream());
	} catch (IOException ex) {
	    // ignore since this was a last ditch after a failure anyways.
	}
    }

    /**
     * Create a thread that reads the input from standard input
     * and sends it to the process
     */
    private void processRemoteInput(final OutputStream stream) {
        Thread thr = new Thread("Input Processor for App #"+my_id) {
            public void run() {
                try {
                    PrintStream out = new PrintStream(stream, true);
                    BufferedReader in =
                    new BufferedReader(new InputStreamReader(mySocket.getInputStream()));
                    String line;
                    while ((line = in.readLine()) != null) {
                        out.println(line);
                        out.flush();
                    }
		    if (!Application.this.isShuttingDown()) {
			try {
			    mySocket.close();
			    signal(MESSAGE, "Application I/O closed");
			} catch (Exception ex) {
			    signal(MESSAGE, "Couldn't close application I/O");
			}
		    }
			
                } catch (IOException ex) {
		    if (!Application.this.isShuttingDown()) {
			try {
			    mySocket.close();
			    signal(ERROR, "Input error; application I/O closed");
			} catch (Exception e) {
			    signal(ERROR, "Input error; couldn't close application I/O");
			}
		    }
                }
            }
        };
        thr.setPriority(Thread.MAX_PRIORITY-1);
        thr.start();
    }

    /**	
     *	Create a Thread that will retrieve and display any output.
     *	Needs to be high priority, else debugger may exit before
     *	it can be displayed.
     */
    private void displayRemoteOutput(final InputStream stream) {
	Thread thr = new Thread("Output Processor for App #"+my_id) { 
	    public void run() {
                try {
                    dumpStream(stream);
                } catch (IOException ex) {
		    if (!Application.this.isShuttingDown()) {
			try {
			    mySocket.close();
			    signal(ERROR, "Output error; application I/O closed");
			} catch (Exception e) {
			    signal(ERROR, "Output error; couldn't close application I/O");
			}
		    }
                }
	    }
	};
	thr.setPriority(Thread.MAX_PRIORITY-1);
	thr.start();
    }

    private void dumpStream(InputStream stream)
	throws IOException {
	if ((mySocket == null) || (mySocket.getOutputStream() == null))
	    return;
        PrintStream outStream = new PrintStream(mySocket.getOutputStream());
        BufferedReader in = 
            new BufferedReader(new InputStreamReader(stream));
        String line;
        while ((line = in.readLine()) != null) {
            outStream.println(line);
	    outStream.flush();
        }
	if (!Application.this.isShuttingDown()) {
	    try {
		mySocket.close();
		signal(MESSAGE, "Application I/O closed");
	    } catch (Exception ex) {
		signal(MESSAGE, "Couldn't close application I/O");
	    }
	}
    }



    
    /*
     *
     * USEFUL FUNCTIONS
     *
     */


    
    /**
     * Return a list of ReferenceType objects for all
     * currently loaded classes and interfaces whose name
     * matches the given pattern.  The pattern syntax is
     * open to some future revision, but currently consists
     * of a fully-qualified class name in which the first
     * component may optionally be a "*" character, designating
     * an arbitrary prefix.
     */
    public List findClassesMatchingPattern(String pattern)
	throws JDEException {
	if (vm == null) return null;
	List result = new ArrayList();  //### Is default size OK?
	if (pattern.startsWith("*.")) {
	    // Wildcard matches any leading package name.
	    pattern = pattern.substring(1);
	    List classes = vm.allClasses();
	    Iterator iter = classes.iterator();
	    while (iter.hasNext()) {
		ReferenceType type = ((ReferenceType)iter.next());
		if (type.name().endsWith(pattern)) {
		    result.add(type);
		}
	    }
	    return result;
	} else {
            // It's a class name.
            return vm.classesByName(pattern);
	}
    }



    /*
     *
     * HANDLING JDE COMMANDS
     *
     */
    


    /**
     * Primary function called by {@link Jdebug} when a command specific to
     * a vm/app is received from jde.
     *
     * @param cmd_id Identifies a command uniquely. See {@link
     * Jdebug#pendingCommands}
     * @param command The command
     * @param args And its arguments
     */
    public void handleCommand(Integer cmd_id, String command, List args) {

	synchronized (pendingCommands) {
	    // first check if we already got a command with this cmd_id that
	    // we haven't handled yet.
	    if (pendingCommands.contains(cmd_id)) {
		// return an "invalid" command id to indicate that the 
		// command id itself is bad
		signalCommandError(new Integer(-1), "Duplicate command id '"+cmd_id+"'");
	    }
	}

	try {
	    command = command.toLowerCase();
	    // running a loaded class
	    if (command.equals(RUN)) {
		generalCommands.doRun(cmd_id, args);
	    }
	    // finish up
	    else if (command.equals(FINISH)) {
		generalCommands.doFinish(cmd_id, args);
	    }
	    // expression evaluation
	    else if (command.equals(EVALUATE)) {
		generalCommands.doEvaluate(cmd_id, args);
	    }
	    // get info about an object
	    else if (command.equals(GET_OBJECT)) {
		generalCommands.doGetObject(cmd_id, args);
	    }
	    // get info about an array
	    else if (command.equals(GET_ARRAY)) {
		generalCommands.doGetArray(cmd_id, args);
	    }
	    // get the value of a string
	    else if (command.equals(GET_STRING)) {
		generalCommands.doGetString(cmd_id, args);
	    }
	    // get info about local variables
	    else if (command.equals(GET_LOCALS)) {
		generalCommands.doGetLocals(cmd_id, args);
	    }
	    // get all the loaded classes
	    else if (command.equals(GET_LOADED_CLASSES)) {
		generalCommands.doGetLoadedClasses(cmd_id, args);
	    }
	    // get path information
	    else if (command.equals(GET_PATH_INFORMATION)) {
		generalCommands.doGetPathInformation(cmd_id, args);
	    }
	    // get class preparation notification
	    else if (command.equals(TRACE_CLASSES)) {
		generalCommands.doTraceClasses(cmd_id, args);
	    }
	    // cancel one of the above
	    else if (command.equals(CANCEL_TRACE_CLASSES)) {
		generalCommands.doCancelTraceClasses(cmd_id, args);
	    }
	    // exceptions...
	    else if (command.equals(TRACE_EXCEPTIONS)) {
		specCommands.doTraceExceptions(cmd_id, args);
	    }
	    // watchpoints...
	    else if (command.equals(WATCH)) {
		specCommands.doWatch(cmd_id, args);
	    }
	    // breakpoint commands
	    else if (command.equals(BREAK)) {
		specCommands.doBreak(cmd_id, args);
	    }
	    // clear a breakpoint
	    else if (command.equals(CLEAR)) {
		specCommands.doClear(cmd_id, args);
	    }
	    // do a step
	    else if (command.equals(STEP)) {
		specCommands.doStep(cmd_id, args);
	    }
	    // request for tracing methods
	    else if (command.equals(TRACE_METHODS)) {
		generalCommands.doTraceMethods(cmd_id, args);
	    }
	    // cancel a tracing request
	    else if (command.equals(CANCEL_TRACE_METHODS)) {
		generalCommands.doCancelTraceMethods(cmd_id, args);
	    }
	    // suspend a thread, threadgroup or class
	    else if (command.equals(SUSPEND)) {
		threadCommands.doSuspend(cmd_id, args);
	    }
	    // resume a thread, threadgroup or class
	    else if (command.equals(RESUME)) {
		threadCommands.doResume(cmd_id, args);
	    }
	    // interrupt a thread 
	    else if (command.equals(INTERRUPT)) {
		threadCommands.doInterrupt(cmd_id, args);
	    }
	    // kill a thread using a throwable object
	    else if (command.equals(KILL_THREAD)) {
		threadCommands.doKillThread(cmd_id, args);
	    }
	    // get all threads
	    else if (command.equals(GET_THREADS)) {
		threadCommands.doGetThreads(cmd_id, args);
	    }
	    // get detailed info on one thread
	    else if (command.equals(GET_THREAD)) {
		threadCommands.doGetThread(cmd_id, args);
	    }
	    // get info about an object
	    else if (command.equals(GET_OBJECT_MONITORS)) {
		threadCommands.doGetObjectMonitors(cmd_id, args);
	    }
	    // trace thread start/deaths
	    else if (command.equals(TRACE_THREADS)) {
		threadCommands.doTraceThreads(cmd_id, args);
	    }
	    // cancel a thread start/death trace request
	    else if (command.equals(CANCEL_TRACE_THREADS)) {
		threadCommands.doCancelTraceThreads(cmd_id, args);
	    }
	    // if it's not supported
	    else {
		signalCommandError(cmd_id, "'"+command+"' is not supported");
	    }
	} catch (JDEException ex) {
	    // a jde exception was raised. the kind of error is already
	    // in there.
	    Debug.printIf(ex);
	    signalCommandError(cmd_id, ex);
	    return;
	} catch (Exception ex) {
	    Debug.printIf(ex);
	    signalCommandError(cmd_id, "An unspecified error occured: '"+ex.toString()+"'");
	    return;
	} finally {
	    synchronized (pendingCommands) {
		// once the command is done, remove it from the pending
		// commands list. if an exception is raised, it should be
		// removed from the pending list there.
		pendingCommands.remove(cmd_id);
	    }
	}
    }
    

    /* spec related functions */

    
    /**
     * This method is executed whenever a new reference type is prepared.
     * If any outstanding specs match, they get resolved in the process
     *
     * @see EventRequestSpecList#resolve(ReferenceType)
     */
    public void resolve(ReferenceType ref) {
	eventRequestSpecs.resolve(ref);
    }

    /**
     * Inform jde on a successful spec resolution
     */
    public void informJDEInstallSuccessful(EventRequestSpec spec) {
	signal(SPEC_RESOLVED, spec.getID());
    }
    
    /**
     * Removes a Spec from the specList, and informs jde.
     * If there is an error while resolving a spec, indicating that it
     * cannot be resolved (ie even at a later time when more classes are
     * prepared), this function is called to remove it from the list, and
     * inform the jde about this error
     */
    public void removeSpecAndInformJDE(EventRequestSpec spec, String problem) {
	if (spec instanceof BreakpointSpec) {
	    signal(INVALID+BREAK, new LispForm(spec.getID()
					       +" \""+problem+"\""));
	} else if (spec instanceof WatchpointSpec) {
	    signal(INVALID+WATCH, new LispForm(spec.getID()
					       +" \""+problem+"\""));
	} else if (spec instanceof ExceptionSpec) {
	    signal(INVALID+TRACE_EXCEPTIONS,
		   new LispForm(spec.getID()+" \""+problem+"\""));
	}
	eventRequestSpecs.delete(spec);
    }

	    
    /* shortcut to jde output commands */

    public void signal(String type, Object obj){
	jdebug.signal(my_id, type, obj);
    }

    public void signalCommandResult(Integer cmd_id) {
	jdebug.signalCommandResult(my_id, cmd_id);
    }

    public void signalCommandResult(Integer cmd_id, Object obj) {
	jdebug.signalCommandResult(my_id, cmd_id, obj);
    }

    public void signalCommandError(Integer cmd_id, String error) {
	jdebug.signalCommandError(my_id, cmd_id, error);
    }

    /**
     * if a jde exception was caused somewhere (causing an error to be sent
     * back for the command), we can use the string in the exception to
     * create the error string
     */
    public void signalCommandError(Integer cmd_id, JDEException ex) {
	jdebug.signalCommandError(my_id, cmd_id, ex.getMessage());
    }
     
} // Application
