#include "seaview.h"
#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>


/* allongement maximal prevu pour les seqs suite a edition */
#define MAX_SEQ_ALLONG 5000
#define MINI_MAX_LENGTH  10000 /* pour depart avec fichier vide */

/* 
Purpose
resource	argument	type		default			exemple

Default file format for saving operations
save		-save		{mase, phylip,	mase			phylip
			    clustal, msf, fasta}

Control of printout options
printoutblock	N/A		integer		10			3
printoutcpl	N/A		integer		80			90
	[this is the paper width, not the # of residues desired on each line]
printoutlpp	N/A		integer		60			90
Name of help file
helpfile	N/A		string		"seaview.help"

Standard coloring of protein sequences (<= 8 colors + white for gaps)
(colors are red, green, yellow, blue, violet, pink, black, gold, and orange)
stdcolorgroups	N/A		string		EDQNHRKBZ,ILMV,APSGT,FY,WC
						      BZDE,ACWY,FGHIK,LMNPQRSTV
Alternate coloring of protein sequences
altcolorgroups	N/A		string		\0	AC,DEFGHIK,LMNPQRS,WY,TV

Faster but less smooth writing
fast		-fast		no-value	False

Residues colored on background
inverted	-inverted	no-value	False

Need to reduce fonts
reducefonts     -reducefonts    no-value        False

N/A		filename
*/	

FL_OBJECT *create_dna_scroller(int x, int y, int w, int h, int dbl_buff);
FL_FORM *create_the_form(char **seq, char **seqname, 
		char **comments, FL_OBJECT **the_scroller, int double_buffer,
		int reducefonts, int inverted,
		known_format default_format, int numb_dnacolors,
		FL_COLOR *dnacolors, 
		int numb_stdprotcolors, FL_COLOR *stdprotcolors,
		int numb_altprotcolors, FL_COLOR *altprotcolors,
		color_choice curr_color_choice, char *progname);
int prep_custom_colors(FL_COLOR *colors, char *customcolors, 
	int max_colors);
color_choice prep_aa_color_code(char *list_std, char *list_alt, 
	int maxprotcolors, int *numb_stdprotcolors, int *numb_altprotcolors);
void handle_props_pup(FL_OBJECT *ob, int item);
void deplacer_grp_seqs(SEA_VIEW *view, int target);
int taille_from_rang(int);
void viewer_callback(FL_OBJECT *ob, long which);
void del_gap_only_sites(SEA_VIEW *view);
void get_menu_taille_valeur(FL_OBJECT *menu, int *taille, int *valeur);
void init_menu_taille_valeur(FL_OBJECT *menu, int taille, int valeur, 
	int offset);
void update_menu_taille_valeur(FL_OBJECT *menu, int taille, int valeur, 
	int offset);
void reference_toggle(SEA_VIEW *view, int on);
void handle_mouse(SEA_VIEW *view, FL_Coord mx, FL_Coord my, 
	int *p_selecting_seqs, int *p_sel_seq_move, int *p_modifying_segment);
int handle_draw(SEA_VIEW *view);
void handle_keyboard(SEA_VIEW *view, int key);
void handle_push(SEA_VIEW *view, FL_Coord mx, FL_Coord my, int key, 
	int *p_modifying_segment, int *p_selecting_seqs, int *p_sel_seq_move);
int delete_gaps_before(SEA_VIEW *view, int numseq, int numsite, int total);

/* external functions */
void draw_comment_lines(FL_OBJECT *ob, SEA_VIEW *view);
extern void *memccpy(void *s1, const void *s2, int c, size_t n);


/* variables globales */
/* pour controle parametres option printout */
#define PRINTOUT_LABEL_SIZE 30
static char printout_label_block[PRINTOUT_LABEL_SIZE] = "block size=";
static char printout_label_cpl[PRINTOUT_LABEL_SIZE] = "chars/line=";
static char printout_label_lpp[PRINTOUT_LABEL_SIZE] = "lines/page=";
static int printout_block;
static int printout_cpl;
static int printout_lpp;
/* pour help file */
static char help_file[100];


int main(int argc, char *argv[])
{
FL_OBJECT *the_scroller;
FL_FORM *my_form;
char *masename, *progname;
static char *seq[MAXNSEQS], *seqname[MAXNSEQS], *comments[MAXNSEQS];
known_format defaultformat;
static FL_COLOR dnacolors[] =
	{ FL_WHITE, FL_RED, FL_GREEN, FL_YELLOW, FL_DOGERBLUE };
int numb_dnacolors = sizeof(dnacolors) / sizeof(FL_COLOR);
static FL_COLOR protcolors[] =
	{ FL_WHITE, FL_RED, FL_GREEN, FL_YELLOW, FL_DOGERBLUE, FL_DARKVIOLET,
	FL_DEEPPINK, FL_BLACK, FL_DARKGOLD, FL_DARKORANGE };
int max_protcolors = sizeof(protcolors) / sizeof(FL_COLOR);
static const char def_stdcolorgroups[] = "EDQNHRKBZ,ILMV,APSGT,FY,WC";
int numb_stdprotcolors, numb_altprotcolors, i;
color_choice curr_color_choice;

static FL_CMD_OPT cmdopt[] =  /* correspondance arguments <-> resources */
{
/* nom arg, nom resource, type arg, valeur ssi type arg = XrmoptionNoArg */
	{"-fast", "*.fast", XrmoptionNoArg, "True"},
	{"-inverted", "*.inverted", XrmoptionNoArg, "True"},
	{"-reducefonts", "*.reducefonts", XrmoptionNoArg, "True"},
	{"-save", "*.save", XrmoptionSepArg, 0}
};
static int quick_and_dirty, reducefonts, inverted; 
static char save_format[10];
static char stdcolorgroups[50], altcolorgroups[50];
static char customdnacolors[200];
static char customprotcolors[300];
#define DEF_FORMAT "mase"
static const char *possible_formats[] = 
			{DEF_FORMAT, "phylip", "clustal", "msf", "fasta"};
const int nbr_formats = sizeof(possible_formats)/sizeof(char *);
/* association resources lues avec variables du prog */
static FL_resource res[] = 
{
/* nom resource, classe (semble inutile), type, addresse ou mettre valeur,
	valeur par defaut, pour type FL_STRING: taille du champ pour valeur */
	{"fast", "", FL_BOOL, &quick_and_dirty, "0" },
	{"reducefonts", "", FL_BOOL, &reducefonts, "0" },
	{"inverted", "", FL_BOOL, &inverted, "0" },
	{"save", "", FL_STRING, save_format, DEF_FORMAT, sizeof(save_format) },
	{"stdcolorgroups", "", FL_STRING, stdcolorgroups, def_stdcolorgroups,
		sizeof(stdcolorgroups) },
	{"altcolorgroups", "", FL_STRING, altcolorgroups, "", 
		sizeof(altcolorgroups) },
	{"printoutblock", "", FL_INT, &printout_block, "10", 0},
	{"printoutcpl", "", FL_INT, &printout_cpl, "80", 0},
	{"printoutlpp", "", FL_INT, &printout_lpp, "60", 0},
	{"helpfile", "", FL_STRING, help_file, "seaview.help", 
		sizeof(help_file) },
	{"dnacolornames", "", FL_STRING, customdnacolors, "", 
		sizeof(customdnacolors) },
	{"protcolornames", "", FL_STRING, customprotcolors, "", 
		sizeof(customprotcolors) }
};
#undef DEF_FORMAT


progname = (char *) extract_filename( (const char *) argv[0] );
/* init + reads arguments and resources */
fl_display = fl_initialize(&argc, argv, "Seaview", cmdopt, 
	sizeof(cmdopt) / sizeof(FL_CMD_OPT) );

/* access to read arguments and resources */
fl_get_app_resources(res, sizeof(res) / sizeof(FL_resource) );
/* converts saving format from string value to known_format enum value */
for( i=0; i < nbr_formats; i++) 
	if(strcmp(save_format, possible_formats[i]) == 0) break;
if( i >= nbr_formats ) {
	fprintf(stderr,
		"invalid value for save argument/resource: %s\n",save_format);
	exit(1);
	}
defaultformat = (known_format) i;
/* process custom color settings */
prep_custom_colors(dnacolors, customdnacolors, numb_dnacolors);
max_protcolors = prep_custom_colors(protcolors, customprotcolors, 
	max_protcolors);
/* process resource-read stdcolorgroups and altcolorgroups */
curr_color_choice = prep_aa_color_code(stdcolorgroups, altcolorgroups, 
		max_protcolors, &numb_stdprotcolors, &numb_altprotcolors);

/* process file_name argument */
if(argc >= 2) masename = argv[1];
else	masename = NULL;

fl_set_goodies_font(FL_BOLD_STYLE, FL_MEDIUM_SIZE );

my_form = create_the_form(seq, seqname, comments, &the_scroller, 
	!quick_and_dirty, reducefonts, inverted,
	defaultformat, numb_dnacolors, dnacolors, 
	numb_stdprotcolors, protcolors,
	numb_altprotcolors, protcolors,
	curr_color_choice, progname);
if(masename != NULL)
	load_alignment_file(the_scroller, masename, NULL, NULL, defaultformat);
fl_do_forms();
return 0;
}


#ifdef _AIX
/* sur IBM RISC __filbuf est en fait _filbuf utilise a l'interieur de xforms */
int __filbuf(FILE *fich)
{
return _filbuf(fich);
}
#endif

void out_of_memory(void)
{
fprintf(stderr,"Error: Not enough memory!\n");
exit(1);
}


void my_watch_cursor(Window win)
{
fl_set_cursor(win, XC_watch); 
XFlush(fl_get_display());
}


const char *extract_filename( const char *fname)
{
char *p, *q;
q = (char *)fname;
if(q == NULL) return "";
do	{ 
#ifdef __VMS
	p = strchr(q,']'); 
	if(p == NULL) p = strchr(q,':'); 
#else
	p = strchr(q,'/'); 
#endif
	if(p != NULL) q = p+1; 
	}
while 	(p != NULL);
return q;
}


void allonge_seqs(char **seq, int totseqs, int maxlen, int *eachlength)
{
int num, l;
char *newseq;
for(num=0; num<totseqs; num++) {
	l = eachlength[num];
	newseq = (char *)malloc(maxlen+1);
	if(newseq==NULL) out_of_memory();
	memcpy(newseq,seq[num],l+1);
	free(seq[num]); seq[num] = newseq;
	}
}

/* memoire pour contenir coloriage standard */
static int color_for_aa_gaps;
static char std_aminoacids[30];
static int std_aa_color[30];
/* memoire pour contenir coloriage alternatif */
static char alt_aminoacids[30] = "";
static int alt_aa_color[30];
/* pointeurs vers coloriage courant */
static char *current_aminoacids;
static int *current_aa_color;


int decode_color_scheme(char *list,  char *aminoacids, int *aa_color,
	int maxprotcolors)
{
int nbr_colors = 1, current = 0, i;
char *p;
aminoacids[0] = 0;
p=strtok(list,",");
while( p!= NULL && nbr_colors < maxprotcolors) {
	strcat(aminoacids,p);
	for(i=0; i < (int) strlen(p); i++)
		aa_color[current++] = nbr_colors;
	nbr_colors++;
	p=strtok(NULL,",");
	}
return nbr_colors;
}


int prep_custom_colors(FL_COLOR *colors, char *customcolors, 
	int max_colors)
{
char *nom;
static int next_color = FL_FREE_COL1;
int rank = 0, r, g, b;

if(*customcolors == 0) return max_colors;
nom = strtok(customcolors, ",");
while(nom != NULL && rank < max_colors - 1) {
	r = g = b = -1;
	sscanf(nom, "%d%d%d", &r, &g, &b);
	if( r>=0 && g>=0 && b>=0 )
		fl_mapcolor(next_color, r, g, b);
	else
		fl_mapcolorname(next_color, nom);
	colors[++rank] = next_color++;
	nom = strtok(NULL, ",");
	}
return rank + 1;
}


color_choice prep_aa_color_code(char *list_std, char *list_alt, 
	int maxprotcolors, int *numb_stdprotcolors, int *numb_altprotcolors)
{
/* couleur pour gaps = 1ere couleur connue (comptee a partir de 0) */
color_for_aa_gaps = 0;
current_aminoacids = std_aminoacids;
current_aa_color = std_aa_color;
/* decodage du coloriage standard des proteines */
*numb_stdprotcolors = 
	decode_color_scheme(list_std, std_aminoacids, std_aa_color,
		maxprotcolors);
if(*list_alt == 0) return NO_ALT_COLORS;
/* decodage du coloriage alternatif des proteines */
*numb_altprotcolors = 
	decode_color_scheme(list_alt, alt_aminoacids, alt_aa_color,
		maxprotcolors);
return USING_STANDARD_COLORS;
}


void set_aa_color_mode(color_choice choice)
{
if(choice == USING_ALT_COLORS) {
	current_aminoacids = alt_aminoacids;
	current_aa_color = alt_aa_color;
	}
else	{
	current_aminoacids = std_aminoacids;
	current_aa_color = std_aa_color;
	}
}


int get_color_for_aa( int key )
/* returns the color # used to display character key in protein */
{
char *pos;
pos=strchr(current_aminoacids, key); 
return ( pos == NULL ? 
	color_for_aa_gaps : current_aa_color[pos - current_aminoacids] );
}


int get_color_for_base( int key )
/* returns the color # used to display character key in DNA */
{
static char bases[]="ACGTU";
static int base_color[] = {1,2,3,4,4};
char *pos;
pos=strchr(bases, key); 
return ( pos == NULL ? 0 : base_color[pos - bases] );
}


char ***prepcolseqs(char **seq, int totseqs, int maxlen, int *eachlength,
	int (*calc_color_function)( int ), int numb_gc)
{
int num, l, col, i;
char ***colseq;
if(totseqs == 0 || numb_gc == 1) return NULL;
colseq=(char ***)malloc(totseqs*sizeof(char **));
if(colseq == NULL) out_of_memory();
for(num=0; num<totseqs; num++) {
	colseq[num] = (char **)malloc(numb_gc * sizeof(char *));
	if(colseq[num] == NULL) out_of_memory();
	l = eachlength[num];
	for(col=0; col < numb_gc; col++) {
		colseq[num][col]=(char *)malloc(maxlen+1);
		if(colseq[num][col] == NULL) out_of_memory();
		memset(colseq[num][col], ' ', l);
		colseq[num][col][l]=0;
		}
	for(i=0; i<l; i++) {
		col = calc_color_function( seq[num][i] );
		colseq[num][col][i] = seq[num][i];
		}
	}
return colseq;
}


char ***prepcolseqs_by_difference(char **seq, int totseqs, int ref_seq0,
	int maxlen, int *eachlength,
	int (*calc_color_function)( int ), int numb_gc)
{
int num, l, col, i;
char ***colseq;
if(totseqs == 0) return NULL;
colseq = (char ***)malloc(totseqs*sizeof(char **));
if(colseq == NULL) return NULL;
for(num=0; num<totseqs; num++) { /* allocation memoire */
	colseq[num] = (char **)malloc(numb_gc * sizeof(char *));
	if(colseq[num] == NULL) return NULL;
	l = eachlength[num];
	for(col=0; col < numb_gc; col++) {
		colseq[num][col]=(char *)malloc(maxlen+1);
		if(colseq[num][col] == NULL) return NULL;
		memset(colseq[num][col], ' ', l);
		colseq[num][col][l]=0;
		}
	}
for(i=0; i<eachlength[ref_seq0]; i++) { /* coloration seq de reference */
	col = calc_color_function( seq[ref_seq0][i] );
	colseq[ref_seq0][col][i] = seq[ref_seq0][i];
	}
for(num=0; num<totseqs; num++) { /* coloration des autres sequences */
	if(num == ref_seq0) continue;
	l = eachlength[num];
	for(i=0; i<l; i++) {
		if(seq[num][i] != seq[ref_seq0][i]) {
			col = calc_color_function( seq[num][i] );
			colseq[num][col][i] = seq[num][i];
			}
		else	{
			colseq[num][0][i] = '.';
			}
		}
	}
return colseq;
}


void draw_cursor(FL_OBJECT *ob, int on_off, int site, int seq, 
	int cursor_in_comment)
{
SEA_VIEW *view = (SEA_VIEW *) ( (FL_OBJECT *)ob->u_vdata )->u_vdata;
int x, y, cursor_x, cursor_y, c, max_curs_coord;
char *debut, *fin;
FL_COLOR background, foreground;
static char lettre[]="A";
static char cursor_coord[100];

if( (!cursor_in_comment) && (seq == 0 || site > view->each_length[seq-1] + 1) ) 
	return;
cursor_x = site - view->first_site;
if(cursor_in_comment) {
	if(view->tot_comment_lines == 0 || !view->show_comment_lines) return;
	if(cursor_x < 0 || cursor_x >= view->tot_sites ) return;
	if(seq <= 0 || seq > view->tot_comment_lines ) return;
	y = view->y_seq + (seq + view->pos_first_comment_line - 2) * 	
		view->line_height;
	x = view->x_seq + cursor_x * view->char_width;
	lettre[0] = view->comment_line[seq-1][site-1];
	}
else	{
	if(view->tot_seqs == 0) return;
	cursor_y = seq - view->first_seq;
	if(cursor_x < 0 || cursor_x >= view->tot_sites ) return;
	if(cursor_y < 0 || cursor_y >= view->tot_lines ) return;
	y = view->y_seq + cursor_y * view->line_height;
	x = view->x_seq + cursor_x * view->char_width;
	lettre[0] = view->sequence[seq-1][site-1];
	}
if(view->numb_gc == 1) { /* cas noir et blanc */
	if(on_off) {
		foreground = FL_WHITE;
		background = FL_BLACK;
		}
	else	{
		foreground = FL_BLACK;
		background = FL_WHITE;
		}
	}
else if( (!cursor_in_comment) && view->inverted_colors) { 
	/* cas inverted colors */
	if(site <= view->each_length[seq - 1]) {
		for(c=0; c<view->numb_gc; c++)
			if(view->col_seq[seq-1][c][site-1] != ' ') break;
		if(c == 0) background = ob->col1;
		else background = view->curr_colors[c];
		}
	else
		background = ob->col1;
	if(on_off) {
		foreground = background;
		background = FL_BLACK;
		}
	else	{
		if(view->active_region == NULL || 
				view->region_line[site - 1] == 'X')
			foreground = FL_BLACK;
		else
			foreground = ob->col2;
		}
	}
else if(cursor_in_comment) { /* dans les lignes comment sur ecran couleur */
	if(on_off) {
		foreground = FL_WHITE; background = FL_BLACK;
		}
	else	{
		foreground = FL_BLACK; background = ob->col1;
		}
	}
else	{ /* cas colored letters */
	if( site <= view->each_length[seq - 1]) {
		for(c=0; c<view->numb_gc; c++)
			if(view->col_seq[seq-1][c][site-1] != ' ') break;
		lettre[0] = view->col_seq[seq-1][c][site-1];
		foreground = view->curr_colors[c];
		}
	if(on_off) {
		background = FL_BLACK;
		}
	else	{
		if( (!cursor_in_comment) && view->active_region != NULL && 
				view->region_line[site - 1] == 'X')
			background = ob->col2;
		else
			background = ob->col1;
		}
	}
fl_rectangle(TRUE, x, 
	y - view->line_height/2, view->char_width, view->line_height, 
	background);
fl_drw_text_cursor(FL_ALIGN_LEFT,x,y,0,0,foreground,ob->lstyle,ob->lsize,
	lettre, 0, -1);
if(on_off) {
/* ecriture des coordonnees du curseur */
	max_curs_coord = view->tot_sites - 12;
	cursor_coord[max_curs_coord] = 0;
	x = view->x_seq + 6 * view->char_width;
	y = view->y_seq - view->line_height;
	fl_rectangle(TRUE, x , y - view->line_height/2, 
		(max_curs_coord ) * view->char_width, 
		view->line_height, ob->col1);
	if(!cursor_in_comment) {
		debut = view->sequence[seq-1]; fin = debut + site; c = 0;
		do { if( *(debut++) != '-' ) c++; } while (debut < fin);
		sprintf(cursor_coord, "Seq:%d Pos:%d|%d [%s]", seq, 
			site, c, view->seqname[seq - 1]);
		}
	else
		sprintf(cursor_coord, "Pos:%d", site);
	fl_drw_text_cursor(FL_ALIGN_LEFT, x, y, 0, 0, 
		view->namecolor, FL_FIXED_STYLE, ob->lsize, cursor_coord,0, -1);
	}
}


void draw_seq_names(FL_OBJECT *ob, SEA_VIEW *view)
{
int x, y, num, debut, fin;
FL_COLOR couleur;
static char trunc_name[100];

x = view->x_name; y = view->y_name;
if(view->draw_names == -1) { /* ecrire tous les noms */
	fl_rectangle(TRUE, ob->x, ob->y,  /* fond pour les noms */
		view->x_seq - ob->x,
		ob->h, ob->col1);
	debut = view->first_seq - 1; 
	fin = view->first_seq - 2 + view->tot_lines;
	if(fin >= view->tot_seqs) fin = view->tot_seqs - 1;
/* write region name */
	if(view->active_region != NULL) {
		memccpy(trunc_name, view->active_region->name, 0, 
			view->wid_names);
		trunc_name[view->wid_names] = 0;
		fl_drw_text_cursor(FL_ALIGN_LEFT, view->x_name, 
		view->y_name + FL_min(view->tot_lines, view->tot_seqs) * 
			view->line_height, 0, 0, view->region_color,
		ob->lstyle, ob->lsize, trunc_name, 0, -1);
		}
	}
else if(view->draw_names == -2) { /* ecrire tous les noms mais rien qu'eux */
	debut = view->first_seq - 1; 
	fin = view->first_seq - 2 + view->tot_lines;
	if(fin >= view->tot_seqs) fin = view->tot_seqs - 1;
	fl_rectangle(TRUE, ob->x,   /* fond pour les noms */
		y - view->line_height/2, view->x_seq - ob->x,
		(fin - debut + 1) * view->line_height, ob->col1);
	}
else	{ /* ecrire un seul nom */
	debut = fin = view->draw_names - 1;
	y += view->line_height * (view->draw_names - view->first_seq);
	if( !view->sel_seqs[view->draw_names - 1] ) fl_rectangle(TRUE, 
		x, y - view->line_height/2, 
		(view->wid_names+1) * view->char_width, 
		view->line_height, ob->col1);
	}
if(view->tot_seqs > 0) { /* nbre de seqs selectionnees */
	fl_rectangle(TRUE, x, 
		view->y_name - view->line_height - view->line_height/2, 
		(view->wid_names+1) * view->char_width, view->line_height, 
		ob->col1);
	sprintf(trunc_name, "sel=%d", view->tot_sel_seqs);
	fl_drw_text_cursor(FL_ALIGN_LEFT, x, view->y_name - view->line_height, 
		0, 0, view->namecolor, FL_FIXED_STYLE, view->DNA_obj->lsize, 
		trunc_name, 0, -1);
	}
/* le(s) nom(s) a ecrire */
for(num = debut; num <= fin; num++) {
	if(view->sel_seqs[num]) {
		fl_rectangle(TRUE, 
			x, y - view->line_height/2, 
			(view->wid_names+1) * view->char_width, 
			view->line_height, FL_BLACK);
		couleur = FL_WHITE;
		}
	else
		couleur = view->namecolor;
	memccpy(trunc_name, view->seqname[num], 0, view->wid_names);
	trunc_name[view->wid_names] = 0;
	fl_drw_text_cursor(FL_ALIGN_LEFT, x, y, 0, 0,
		couleur, FL_FIXED_STYLE, ob->lsize,
		trunc_name, 0, -1);
	y += view->line_height;
	}
}


void draw_header(FL_OBJECT *ob, SEA_VIEW *view)
{
int l_line, c;
static char site_line[300];

if(view->mod_seq) return;
if(view->tot_seqs == 0) return;
if(view->active_region != NULL) draw_region_line(ob, view);
/* write site numbers */
l_line = view->tot_sites;
if(l_line < 5) return;
memset(site_line,' ',l_line);
sprintf(site_line,"%d",view->first_site); c=strlen(site_line); site_line[c]=' ';
sprintf(site_line + l_line - 5,"%5d",view->first_site + l_line - 1);
fl_rectangle(1, view->x_seq, 
	view->y_seq - view->line_height - view->line_height/2,
	(view->tot_sites + 1) * view->char_width, view->line_height, ob->col1);
fl_drw_text_cursor(FL_ALIGN_LEFT,view->x_seq, view->y_seq - view->line_height,
	0,0, view->namecolor, FL_FIXED_STYLE, ob->lsize, site_line, 0, -1);
}  


void draw_dna_seqs(FL_OBJECT *ob, SEA_VIEW *view)
{
int nline, c, offset, x, y, l_line;
char lastchar, *lastpos, *pos;
int debut, fin;

if(view->mod_seq == 0) { /* draw all lines */
/* clear the background for drawing seqs */
	fl_rectangle(TRUE, 
		view->x_seq, 
		ob->y, (view->tot_sites + 1) * view->char_width, 
		ob->h, ob->col1);
	debut = view->first_seq - 1;
	fin = FL_min(view->first_seq + view->tot_lines - 1, view->tot_seqs) - 1;
	}
else if(view->mod_seq == -1) { /* draw selected lines */
	debut = view->first_seq - 1;
	fin = FL_min(view->first_seq + view->tot_lines - 1, view->tot_seqs) - 1;
	while(fin >= debut && !view->sel_seqs[fin]) fin--;
	if(debut > fin) return;
	while(!view->sel_seqs[debut]) debut++;
	y = view->y_seq + (debut - view->first_seq + 1) * view->line_height - 
		view->line_height/2;
	for(nline = debut; nline <= fin; nline++) {
		if(view->sel_seqs[nline])
			fl_rectangle(TRUE, view->x_seq, y, 
				(view->tot_sites + 1) * view->char_width, 
				view->line_height, ob->col1);
		y += view->line_height;
		}
	}
else	{ /* draw just line # view->mod_seq */
/* clear the background of the modified seq */
	fl_rectangle(TRUE, view->x_seq, view->y_seq + 
		(view->mod_seq - view->first_seq) * view->line_height -
		view->line_height/2, (view->tot_sites + 1) * view->char_width, 
		view->line_height, ob->col1);
	debut = fin = view->mod_seq - 1;
	}
/* draw special background for region sites */
draw_region_background(view, debut, fin); 

/* write sequences */
x = view->x_seq;
y = view->y_seq + (debut - (view->first_seq - 1)) * view->line_height;
offset = view->first_site - 1;
for(nline = debut; nline <= fin; nline++) {
	if( (view->mod_seq != -1 || view->sel_seqs[nline]) &&
		view->each_length[nline] > offset) {
		l_line = FL_min( view->tot_sites, 
				view->each_length[nline] - offset );
		for(c=0; c < view->numb_gc; c++) { /* write DNA seqs */
			if(view->numb_gc > 1) 
				pos = view->col_seq[nline][c];
			else
				pos = view->sequence[nline];
			lastpos = pos + offset + l_line;
			lastchar = *lastpos;
			*lastpos = 0;
			fl_drw_text_cursor(FL_ALIGN_LEFT, x, y, 0, 0,
				view->curr_colors[c], 
				ob->lstyle, ob->lsize, pos + offset, 0, -1);
			*lastpos = lastchar;
			}
		}
	y += view->line_height;
	}
draw_header(ob, view);
}  


void draw_dna_seqs_inverted(FL_OBJECT *ob, SEA_VIEW *view)
{
int nline, c, offset, x, y, l_line, xx, yy, firstline, lastline, use_region,
	debut, fin, der_site;
char lastchar, *lastpos, *pos;
list_segments *segment, *first_segment;
FL_COLOR save_col0;

if(view->mod_seq <= 0 ) { /* tout redessiner */
	fl_rectangle(TRUE, view->x_seq, ob->y, ob->w - view->x_seq + ob->x, 
		ob->h, ob->col1);
	}
save_col0 = view->curr_colors[0];
view->curr_colors[0] = ob->col1;
/* write sequences */
x = view->x_seq;
offset= view->first_site - 1;
if(view->mod_seq <= 0) {
	firstline = view->first_seq-1;
	lastline = FL_min(view->first_seq + view->tot_lines -1, view->tot_seqs);
	}
else	{
	firstline = view->mod_seq - 1;
	lastline = firstline + 1;
	}
use_region = (view->active_region != NULL &&
		(first_segment = view->active_region->list) != NULL);
if(use_region) {
	do	{
		if(first_segment->fin >= view->first_site) break;
		first_segment = first_segment->next;
		}
	while(first_segment != NULL);
	}
y = view->y_seq + (firstline - view->first_seq + 1) * view->line_height; 
yy = y - view->line_height/2;
for(nline=firstline; nline < lastline; /* ecriture des fonds de couleur */
		nline++, y += view->line_height, yy += view->line_height ) {
	l_line = ( offset + view->tot_sites < view->each_length[nline] ? 
		view->tot_sites : view->each_length[nline] - offset);
	l_line = FL_max(l_line, 0); /* nbre de caract a ecrire */
	if(l_line > 0) {
		for(c = 0; c < view->numb_gc; c++) { 
			xx = x; 
			pos = view->col_seq[nline][c] + offset;
			lastpos = pos + l_line;
			while(pos < lastpos) {
				if( *(pos++) != ' ' ) {
					fl_rectangle(1, xx, yy, 
						view->char_width, 
						view->line_height, 
						view->curr_colors[c]);
					}
				xx += view->char_width;
				}
			}
		}
	if(l_line < view->tot_sites) /* au dela de la fin de la seq */
	 	fl_rectangle(1, x + l_line * view->char_width, 
			yy,
			view->char_width * (view->tot_sites - l_line), 
			view->line_height, ob->col1);
	if(l_line == 0) continue;
/*ecriture des seqs: regions en noir et autres en col2 */
	pos = view->sequence[nline] + offset; 
	lastpos = pos + l_line;
	lastchar = *lastpos;
	*lastpos = 0;
	fl_drw_text_cursor(FL_ALIGN_LEFT,x,y,0,0,
		( use_region ? ob->col2 : FL_BLACK ), 
		ob->lstyle, ob->lsize, 
		pos, 0, -1);
	*lastpos = lastchar;
	if(use_region) { /* re-ecrire les regions en noir */
		der_site = view->first_site + l_line - 1;
		segment = first_segment;
		while( segment  != NULL ) {
			if(segment->debut > der_site) break;
			debut = FL_max(segment->debut, view->first_site);
			fin = FL_min(segment->fin, der_site);
			pos = view->sequence[nline] + debut - 1; 
			lastpos = view->sequence[nline] + fin;
			lastchar = *lastpos;
			*lastpos = 0;
			fl_drw_text_cursor(FL_ALIGN_LEFT,
				x + (debut - offset - 1)*view->char_width,
				y, 0, 0,
				FL_BLACK, ob->lstyle, ob->lsize, pos, 0, -1);
			*lastpos = lastchar;
			segment = segment->next;
			}
		}
	}
view->curr_colors[0] = save_col0;
draw_header(ob, view);
}


void set_tot_lines(SEA_VIEW *view, int new_val)
{
int l;
double x;
view->tot_lines = new_val;
l = view->tot_seqs - view->tot_lines + 1;
if(l<1) l=1;
if(view->first_seq > l) {
	view->first_seq = l;
	}
if(view->tot_seqs > 0) {
	x = view->tot_lines / ( (double) view->tot_seqs ); 
	if(x>1) x=1;
	}
else	x = 1;
fl_set_slider_size(view->vertsli,x);
fl_set_slider_bounds(view->vertsli, 1, l);
fl_set_slider_value(view->vertsli, view->first_seq);
}


int compute_size_params(FL_OBJECT *mygroup, int force_recompute)
{
static int old_w = 0;
static int old_h = 0;
SEA_VIEW *myview = (SEA_VIEW *)mygroup->u_vdata;
FL_OBJECT *ob = myview->DNA_obj;
double x;
int l, tot_lines;

if( !force_recompute && ob->w == old_w && ob->h == old_h) return FALSE;
fl_freeze_form(mygroup->form);
old_w = ob->w; old_h = ob->h;
myview->x_name = myview->char_width/2 + ob->x;
myview->y_name = myview->line_height/2 + 2 + myview->line_height + ob->y;
myview->x_seq = (myview->wid_names +2) * myview->char_width + ob->x;
myview->y_seq = myview->y_name;
myview->tot_sites = ( ob->w - myview->x_seq + ob->x ) / myview->char_width;
tot_lines = ( ob->h - 4 ) / myview->line_height  - 1;
if(myview->active_region != NULL) tot_lines--;
if(myview->show_comment_lines) {
	tot_lines -= myview->tot_comment_lines;
	myview->pos_first_comment_line = 
		FL_min(tot_lines, myview->tot_seqs) + 1;
	if(myview->active_region != NULL) ++(myview->pos_first_comment_line);
	}
if(myview->double_buffer) {
	myview->x_name -= ob->x;
	myview->x_seq -= ob->x;
	myview->y_name -= ob->y;
	myview->y_seq -= ob->y;
	}
/* init sliders bounds and size */
fl_set_slider_value(myview->horsli, 1); /* transitoire */
if(myview->tot_seqs > 0) {
	x = ( (double) myview->tot_sites ) / ( myview->seq_length + 3 ); 
	if(x>1) x=1;
	}
else	x = 1;
fl_set_slider_size(myview->horsli,x);
l = myview->seq_length - myview->tot_sites+3;
if(l<1) l=1;
fl_set_slider_bounds(myview->horsli,1,l);
if(myview->first_site > l) {
	myview->first_site = l;
	}
fl_set_slider_value(myview->horsli, myview->first_site);
set_tot_lines(myview, tot_lines);
fl_unfreeze_form(mygroup->form);
return TRUE;
}


void init_dna_scroller(FL_OBJECT *scroller, char *sequence[], char *seqname[],
  int totseqs, char **comments, const char *masename, int protein, char *header)
{
int i, ascend, descent, l, mainpop;
FL_OBJECT *myfree;
SEA_VIEW *myview;
list_regions *mylregion;
myview = (SEA_VIEW *)scroller->u_vdata;
myfree = myview->DNA_obj;

myfree->wantkey = FL_KEY_TAB; /* accept keys + 4 arrows */
myview->sequence = sequence;
myview->comments = comments;
myview->header = header;
myview->seqname = seqname;
myview->tot_seqs = totseqs;
myview->seq_length = 0;
myview->protein = protein;
myview->allow_seq_edit = FALSE;
mainpop = myview->bouton_props->u_ldata;
fl_setpup_mode(mainpop, 40, FL_PUP_BOX);
fl_setpup_mode(mainpop, 50, FL_PUP_GRAY);
fl_setpup_mode(mainpop, 51, FL_PUP_RADIO);
fl_setpup_mode(mainpop, 52, FL_PUP_RADIO);
fl_setpup_mode(mainpop, 53, FL_PUP_RADIO);
myview->hjkl = FALSE;
myview->modif_but_not_saved = FALSE;
if(masename != NULL) {
	myview->masename = (char *)malloc(strlen(masename)+1);
	if(myview->masename == NULL) out_of_memory();
	strcpy(myview->masename, masename);
	}
else	myview->masename = NULL;
if(totseqs > 0) {
	myview->each_length = (int *)malloc(totseqs * sizeof(int));
	if(myview->each_length == NULL) out_of_memory();
	}
for(i=0; i<totseqs; i++) {
	l=strlen(sequence[i]);
	myview->each_length[i] = l;
	if(l > myview->seq_length) myview->seq_length = l;
	}
myview->max_seq_length = myview->seq_length + MAX_SEQ_ALLONG;
myview->max_seq_length = FL_max(myview->max_seq_length, MINI_MAX_LENGTH);
myview->wid_names = 0;
for(i=0; i<totseqs; i++) {
	l=strlen(seqname[i]);
	while(l > 0 && seqname[i][l-1] == ' ') l--; seqname[i][l] = 0;
	if(l > myview->wid_names) myview->wid_names = l;
	}
if(totseqs > 0 && myview->wid_names < 10) myview->wid_names = 10;
if(myview->wid_names > 20) myview->wid_names = 20;
allonge_seqs(sequence, totseqs, myview->max_seq_length, myview->each_length);
if(myview->numb_dnacolors > 1 && myview->bouton_props != NULL) { 
	/* DNA/prot modes useful only with colors */
	fl_setpup_mode(mainpop, 31, FL_PUP_RADIO);
	fl_setpup_mode(mainpop, 32, FL_PUP_RADIO);
	if(myview->alt_colors != NO_ALT_COLORS)
		fl_setpup_mode(mainpop, 33, FL_PUP_RADIO);
	if(protein) {
		if(myview->alt_colors == USING_ALT_COLORS) {
			fl_setpup_mode(mainpop, 33, FL_PUP_CHECK);
			set_aa_color_mode(USING_ALT_COLORS);
			myview->numb_gc = myview->numb_altprotcolors;
			myview->curr_colors = myview->altprotcolors;
			}
		else	{
			fl_setpup_mode(mainpop, 32, FL_PUP_CHECK);
			set_aa_color_mode(USING_STANDARD_COLORS);
			myview->numb_gc = myview->numb_stdprotcolors;
			myview->curr_colors = myview->stdprotcolors;
			}
		}
	else	{
		fl_setpup_mode(mainpop, 31, FL_PUP_CHECK);
		myview->numb_gc = myview->numb_dnacolors;
		myview->curr_colors = myview->dnacolors;
		}
	}
if(myview->numb_gc > 1) {
	myview->col_seq = prepcolseqs(sequence, totseqs, myview->max_seq_length,
		myview->each_length, 
		( myview->protein ? get_color_for_aa : get_color_for_base ), 
		myview->numb_gc);
	myview->alt_col_seq = NULL;
	}
myview->first_seq = myview->first_site = 1;
myview->line_height=fl_get_char_height(myfree->lstyle, myfree->lsize,
			&ascend,&descent);
myview->char_width = fl_get_string_width(myfree->lstyle, myfree->lsize,
		"W",1);
myview->draw_names = -1;
myview->mod_cursor = FALSE;
myview->mod_seq =0;
myview->multipl->u_ldata = 0;
if(myview->tot_seqs == 0)
	myview->cursor_seq = myview->cursor_site = 0;
else
	myview->cursor_seq = myview->cursor_site = 1;
myview->cursor_in_comment = FALSE;
myview->tot_sel_seqs = 0;
if(myview->menu_edit != NULL) {
	fl_set_menu_item_mode(myview->menu_edit, RENAME_SEQ, FL_PUP_GRAY);
	fl_set_menu_item_mode(myview->menu_edit, EDIT_COMMENTS, FL_PUP_GRAY);
	fl_set_menu_item_mode(myview->menu_edit, DUPLICATE_SEQ, FL_PUP_GRAY);
	fl_set_menu_item_mode(myview->menu_edit, DELETE_SEQ, FL_PUP_GRAY);
	fl_set_menu_item_mode(myview->menu_edit, COMPLEMENT_SEQ, FL_PUP_GRAY);
	fl_set_menu_item_mode(myview->menu_edit, REVERSE_SEQ, FL_PUP_GRAY);
	fl_set_menu_item_mode(myview->menu_edit, EXCHANGE_UT, FL_PUP_GRAY);
	fl_set_menu_item_mode(myview->menu_edit, ALIGN_SEQS, FL_PUP_GRAY);
	fl_set_menu_item_mode(myview->menu_edit, DOT_PLOT, FL_PUP_GRAY);
	fl_set_menu_item_mode(myview->menu_edit, CONSENSUS_SEQ, FL_PUP_GRAY);
	fl_set_menu_item_mode(myview->menu_edit, DELETE_GAP_ONLY_SITES, 
		FL_PUP_NONE);
	}
if(myview->tot_seqs > 0) {
	myview->sel_seqs = (int *)calloc(myview->tot_seqs, sizeof(int));
	if(myview->sel_seqs == NULL) out_of_memory();
	}
/* interpreter les regions du header du fichier mase */
myview->regions = parse_regions_from_header(header);
/* initialiser les regions et leur menu avec pas de region active */
myview->mod_region_line = FALSE;
if(myview->tot_seqs > 0) {
	myview->region_line = (char *)malloc(myview->max_seq_length + 1);
	if(myview->region_line == NULL) out_of_memory();
	}
myview->region_length = 0;
myview->active_region = NULL;
if(myview->menu_regions != NULL) {
	fl_clear_menu(myview->menu_regions);
	if(myview->tot_seqs > 0) {
		fl_set_menu(myview->menu_regions,
		   	"Create set|Save set|Rename set|"
			"Hide set|Delete set%l");
		fl_set_menu_item_mode(myview->menu_regions, 2, FL_PUP_GRAY);
		fl_set_menu_item_mode(myview->menu_regions, 3, FL_PUP_GRAY);
		fl_set_menu_item_mode(myview->menu_regions, 4, FL_PUP_GRAY);
		fl_set_menu_item_mode(myview->menu_regions, 5, FL_PUP_GRAY);
		}
	mylregion = myview->regions;
	i = 0;
	while(mylregion != NULL) {
		i++;
		fl_addto_menu(myview->menu_regions, mylregion->element->name);
		mylregion = mylregion->next;
		}
	init_menu_taille_valeur(myview->menu_regions, i, 0, 5);
	}
/* interpreter les species sets du fichier mase */
myview->numb_species_sets = parse_species_sets_from_header(header,
	myview->list_species_sets, myview->name_species_sets, myview->tot_seqs);
if(myview->menu_species != NULL) {
	fl_clear_menu(myview->menu_species);
	if(myview->tot_seqs > 0) {
		fl_set_menu(myview->menu_species,
			 "Create group|Delete group%l");
		fl_set_menu_item_mode(myview->menu_species, 1, FL_PUP_GRAY);
		fl_set_menu_item_mode(myview->menu_species, 2, FL_PUP_GRAY);
		}
	for(i=0; i < myview->numb_species_sets; i++) {
		fl_addto_menu(myview->menu_species, 
			myview->name_species_sets[i]);
		}
	init_menu_taille_valeur(myview->menu_species, 
		myview->numb_species_sets, 0, 2);
	}
/* interpreter les comment lines du header */
myview->tot_comment_lines = parse_comment_lines_from_header(header,
	&(myview->comment_line), &(myview->comment_name), 
	&(myview->comment_length) , myview->max_seq_length);
myview->show_comment_lines = (myview->tot_comment_lines > 0);
myview->active_comment_line = 0;
update_menu_footers(myview);
compute_size_params(scroller, TRUE);
fl_set_focus_object(scroller->form, myview->DNA_obj);
}


int mod_multipl(FL_OBJECT *obj, int newval)
{
static char label[16] = "mult=";
if(newval == 0) {
	obj->u_ldata = 0;
	fl_set_object_label(obj,"");
	}
else	{
	sprintf(label+5,"%d",newval);
	fl_set_object_label(obj,label);
	obj->u_ldata = newval;
	}
return newval;
}


void rename_sequence(SEA_VIEW *view)
{
char *new_name;
const char *rep;
int num;
if(view->tot_sel_seqs != 1) return; /* only by security, should not happen */
num = 0;
while( ! view->sel_seqs[num] ) num++;
rep = fl_show_input("Rename:",view->seqname[num]);
if(rep == NULL) return;
new_name = (char *)malloc(strlen(rep)+1);
if(new_name == NULL) return;
free(view->seqname[num]);
strcpy(new_name, rep);
view->seqname[num] = new_name;
if(num+1 >= view->first_seq && num+1 <= view->first_seq+view->tot_lines-1) {
	view->draw_names = num + 1;
	fl_redraw_object(view->DNA_obj);
	}
view->modif_but_not_saved = TRUE;
}


void del_gap_only_sites(SEA_VIEW *view)
{
int position, numseq, allgaps, inrun, debut, count;

view->seq_length = 0;
for(numseq = 0; numseq < view->tot_seqs; numseq++) 
	if(view->each_length[numseq] > view->seq_length)
		view->seq_length = view->each_length[numseq];
inrun = FALSE;
position = -1;
while( ++position <= view->seq_length) {
	for(numseq = 0; numseq < view->tot_seqs; numseq++) 
		if(position < view->each_length[numseq] &&
			view->sequence[numseq][position] != '-') break;
	allgaps = (numseq >= view->tot_seqs);
	if(position >= view->seq_length) allgaps = FALSE;
	if(inrun == allgaps) continue;
	if(allgaps && !inrun) {
		inrun = TRUE;
		debut = position;
		}
	else	{
		inrun = FALSE;
		count = position - debut;
		for(numseq = 1; numseq <= view->tot_seqs; numseq++)
			delete_gaps_before(view, numseq, debut + count + 1, 
				count);
		view->seq_length -= count;
		if(view->cursor_site > position) view->cursor_site -= count;
		if(view->regions != NULL) delete_region_part(view, 
			debut + count + 1, count);
		if(view->tot_comment_lines > 0)	delete_in_all_comments(count, 
			debut + count + 1, view);
		position -= count;
		}
	}
view->seq_length = 0;
for(numseq = 0; numseq < view->tot_seqs; numseq++) 
	if(view->each_length[numseq] > view->seq_length)
		view->seq_length = view->each_length[numseq];	
}


int delete_selected_seqs(SEA_VIEW *view)
{
void **temp_data;
int new_tot_seqs, i, j, c, *temp_int, numset;
new_tot_seqs = view->tot_seqs - view->tot_sel_seqs;
temp_data = (void **)malloc(new_tot_seqs * sizeof(void *));
if(temp_data == NULL) return TRUE;
temp_int = (int *)malloc(new_tot_seqs * sizeof(int));
if(temp_int == NULL) return TRUE;

for(i = 0, j = 0; i<  view->tot_seqs; i++) /* process sequences */
	if( ! view->sel_seqs[i] ) temp_data[j++] = (void *) view->sequence[i];
	else free(view->sequence[i]);
memcpy(view->sequence , temp_data, new_tot_seqs * sizeof(char *) );

for(i = 0, j = 0; i<  view->tot_seqs; i++) /* process comments */
	if( ! view->sel_seqs[i] ) temp_data[j++] = (void *) view->comments[i];
	else free(view->comments[i]);
memcpy(view->comments , temp_data, new_tot_seqs * sizeof(char *) );

for(i = 0, j = 0; i<  view->tot_seqs; i++) /* process seq names */
	if( ! view->sel_seqs[i] ) temp_data[j++] = (void *) view->seqname[i];
	else free(view->seqname[i]);
memcpy(view->seqname , temp_data, new_tot_seqs * sizeof(char *) );

for(i = 0, j = 0; i<  view->tot_seqs; i++) /* process seq lengths */
	if( ! view->sel_seqs[i] ) temp_int[j++]= view->each_length[i];
memcpy(view->each_length , temp_int, new_tot_seqs * sizeof(int) );

if(view->numb_gc > 1) { /* process color-coded sequences */
	for(i = 0, j = 0; i<  view->tot_seqs; i++)
		if( ! view->sel_seqs[i] ) temp_data[j++] = 
						(void *) view->col_seq[i];
		else 	{
			for(c=0; c<view->numb_gc; c++)
				free(view->col_seq[i][c]);
			free(view->col_seq[i]);
			}
	memcpy(view->col_seq , temp_data, new_tot_seqs * sizeof(char **) );
   	}
for(numset = 0; numset < view->numb_species_sets; numset++) {
/* process species sets */
	for(i = 0, j = 0; i < view->tot_seqs; i++) { 
		if( ! view->sel_seqs[i] ) 
			temp_int[j++]= view->list_species_sets[numset][i];
		}
	memcpy(view->list_species_sets[numset], temp_int, 
		new_tot_seqs * sizeof(int) );
	}

if(! view->cursor_in_comment) view->cursor_seq = 1;
memset(view->sel_seqs, 0, new_tot_seqs * sizeof(int));
view->tot_seqs = new_tot_seqs;
select_deselect_seq(view, 0);
if(view->menu_species != NULL) {
	numset = 0;
	while(numset < view->numb_species_sets) { /* check for empty sets */
		for(i = 0, j = 0; i < view->tot_seqs; i++) 
			if( view->list_species_sets[numset][i] ) j++;
		if( j == 0 ) { /* set became empty => remove it */
			int taille, valeur;
			free(view->list_species_sets[numset]);
			free(view->name_species_sets[numset]);
			memcpy(view->list_species_sets + numset,
			view->list_species_sets + numset + 1,
			(view->numb_species_sets - numset - 1)*sizeof(int *) );
			memcpy(view->name_species_sets + numset,
			view->name_species_sets + numset + 1,
			(view->numb_species_sets - numset - 1)*sizeof(char *) );
			fl_delete_menu_item(view->menu_species, numset + 3);
			(view->numb_species_sets)--;
			get_menu_taille_valeur(view->menu_species, &taille,
				&valeur);
			init_menu_taille_valeur(view->menu_species, taille - 1,
				0, 2);
			}
		else	numset++;
		}
	}
view->modif_but_not_saved = TRUE;
return FALSE;
}


void vh_sliders_callback(FL_OBJECT * ob, long which)
{
SEA_VIEW *view = (SEA_VIEW *) ( (FL_OBJECT *)ob->u_vdata )->u_vdata;
int old, new_val;
new_val = fl_get_slider_value(ob);
if (which == 0) { /* vertical */
      	old = view->first_seq;
      	view->first_seq = new_val;
	}
else	{ /* horizontal */
      	old = view->first_site;
      	view->first_site = new_val;
	view->draw_names = 0;
	}
if(old != new_val) fl_redraw_object(view->DNA_obj);
}


void lrdu_button_callback(FL_OBJECT *ob, long which)
{
SEA_VIEW *view = (SEA_VIEW *) ( (FL_OBJECT *)ob->u_vdata )->u_vdata;
FL_OBJECT *sli;
int *pval, step, signe, newval;
double mini, maxi;

if(which <= 3) { /* mouvement horizontal */
	sli = view->horsli;
	pval = &view->first_site;
	}
else	{ /* mouvement vertical */
	sli = view->vertsli;
	pval = &view->first_seq;
	}
if( which % 2 == 0) { /* mouvement par pas de 1 */
	step = 1;
	}
else	{ /* mouvement par "page" */
	if(which <= 3) /* page horizontale */
		step = view->tot_sites * 0.90;
	else	 /* page verticale */
		step = view->tot_lines - 2;
	}
if( which % 4 <= 1) /* mouvement de diminution */
	signe = -1;
else	 /* mouvement d'augmentation */
	signe = 1;
newval = *pval + signe * step;
fl_get_slider_bounds(sli, &mini, &maxi);
if(newval < mini) newval = mini;
if(newval > maxi) newval = maxi;
if(newval != *pval) {
	*pval = newval;
	fl_set_slider_value(sli, *pval);
	if(which <= 3)  view->draw_names = 0;
	fl_redraw_object(view->DNA_obj);
	}
}


void handle_props_pup(FL_OBJECT *ob, int item)
/* creation-modif de tous les menus et sousmenus du bouton Props */
{
static int mainpop = -1, clustal_opt_pop = -1, consensus_opt_pop = -1;
int sizepop, formatpop, colorpop, dnakeyspop;
Window fenetre = FL_ObjWin(ob);
if( item == -1) { /* creation du menu et des sous-menus */
	sizepop = fl_defpup(fenetre,	
		"Tiny%r1%x11|Small%r1%x12|Normal%r1%x13|Medium%r1%x14|"
		"Large%r1%x15|Huge%r1%x16");
	fl_setpup_shadow(sizepop, FALSE);
	fl_setpup_softedge(sizepop, TRUE);
	formatpop = fl_defpup(fenetre,
	    "Save as Mase%r2%x21|Save as Phylip%r2%x22|Save as Clustal%r2%x23|"
	    "Save as MSF%r2%x24|Save as fasta%r2%x25");
	fl_setpup_shadow(formatpop, FALSE);
	fl_setpup_softedge(formatpop, TRUE);
	colorpop = fl_defpup(fenetre,
		"DNA/RNA colors%r3%x31|Protein colors%r3%x32|"
		"Alt. colors%r3%x33%l|Inverted colors%b%x34");
	fl_setpup_shadow(colorpop, FALSE);
	fl_setpup_softedge(colorpop, TRUE);
	dnakeyspop = fl_defpup(fenetre,
		"hjkl<space> => GATCN%r4%x51|"
		"hjkl<space> => TCGAN%r4%x52|hjkl<space> => ACGTN%r4%x53");
	fl_setpup_shadow(dnakeyspop, FALSE);
	fl_setpup_softedge(dnakeyspop, TRUE);
	handle_props_pup(ob, 57); /* creation du sous menu clustal_opt_pop */
	handle_props_pup(ob, 61); /* creation du sous menu consensus_opt_pop */
	mainpop = fl_defpup(fenetre,
		"Fontsize%m%x10|Save format%m%x20|Colors%m%x30|"
		"Allow seq. edition%b%x40|"
		"<Alt>R by Reference%b%x41|"
		"DNA keys%d%m%x50|"
		"Clustalw options%m%x55|"
		"Consensus option%m%x60", 
		sizepop, formatpop, colorpop, dnakeyspop, clustal_opt_pop,
		consensus_opt_pop);
	fl_setpup_shadow(mainpop, FALSE);
	fl_setpup_softedge(mainpop, TRUE);
	ob->u_ldata = mainpop; /* mainpop stocke dans l_data de bouton Props */
	}
else if(item == 57) { /* creation-modif du popup clustal_opt_pop */
	char tmp[100];
	FL_OBJECT *scroller;
	SEA_VIEW *view;
	scroller = (FL_OBJECT *) ob->u_vdata;
	view = (SEA_VIEW *)scroller->u_vdata;
	if(clustal_opt_pop != -1) fl_freepup(clustal_opt_pop);
	if(view->clustal_options != NULL) {
		strcpy(tmp, view->clustal_options);
		strcat(tmp, "%B%x56%l|");
		}
	else
		tmp[0] = 0;
	strcat(tmp, "Edit options%x57");
	clustal_opt_pop = fl_defpup(fenetre, tmp);
	fl_setpup_shadow(clustal_opt_pop, FALSE);
	fl_setpup_softedge(clustal_opt_pop, TRUE);
	if(mainpop != -1) fl_setpup_submenu(mainpop, 55, clustal_opt_pop);
	}
else if(item == 61) { /* creation-modif du popup consensus_opt_pop */
	char tmp[100];
	FL_OBJECT *scroller;
	SEA_VIEW *view;
	scroller = (FL_OBJECT *) ob->u_vdata;
	view = (SEA_VIEW *)scroller->u_vdata;
	if(consensus_opt_pop != -1) fl_freepup(consensus_opt_pop);
	sprintf(tmp, "%d", view->consensus_threshold);
	strcat(tmp, " %% %t%l|Edit value%x61");
	consensus_opt_pop = fl_defpup(fenetre, tmp);
	fl_setpup_shadow(consensus_opt_pop, FALSE);
	fl_setpup_softedge(consensus_opt_pop, TRUE);
	if(mainpop != -1) fl_setpup_submenu(mainpop, 60, consensus_opt_pop);
	}
}


int taille_from_rang(int rang)
{
static int taille[] = { FL_TINY_SIZE, FL_SMALL_SIZE,  FL_NORMAL_SIZE,  FL_MEDIUM_SIZE,  FL_LARGE_SIZE, FL_HUGE_SIZE };
return taille[rang - 11];
}


void props_button_callback(FL_OBJECT *ob, long toto)
{
FL_OBJECT *scroller;
SEA_VIEW *view;
int ascend, descent, reponse, previous_format, mainpop;
mainpop = ob->u_ldata;
reponse = fl_dopup(mainpop);
if(reponse == -1) return;
scroller = (FL_OBJECT *) ob->u_vdata;
view = (SEA_VIEW *)scroller->u_vdata;
if(reponse >= 11 && reponse <= 16) {  /* changing character size */
	if(view->DNA_obj->lsize == taille_from_rang(reponse) ) return;
	my_watch_cursor(view->dnawin);
	fl_freeze_form(ob->form); /* absolument necessaire */
	fl_set_object_lsize( view->DNA_obj, taille_from_rang(reponse));
	view->line_height=fl_get_char_height(view->DNA_obj->lstyle, 
		view->DNA_obj->lsize, &ascend,&descent);
	view->char_width = fl_get_string_width(view->DNA_obj->lstyle, 
		view->DNA_obj->lsize, "W",1);
	compute_size_params(scroller, TRUE);
	fl_unfreeze_form(ob->form);
	fl_reset_cursor(view->dnawin);
	}
else if(reponse >= 21 && reponse <= 25) {
/* changing file saving format */
	previous_format = view->format_for_save + 21;
	view->format_for_save = (known_format)(reponse - 21);
	if(previous_format != reponse) {
		fl_setpup_mode(mainpop, reponse, FL_PUP_CHECK);
		fl_set_menu_item_mode(view->menu_file,SAVE,FL_PUP_GRAY);
		if(view->masename != NULL) {
			free(view->masename);
			view->masename = NULL;
			}
		}
	}
else if( reponse >= 31 && reponse <= 33 && view->numb_gc > 1 ) { 
/* changing DNA / protein / Alternate protein mode */
	int num, col;
	my_watch_cursor(view->dnawin);
	view->protein = (reponse >= 32);
	if(view->numb_gc > 1) { /* free col_seq */
		for(num = 0; num < view->tot_seqs; num++) {
			for(col = 0; col < view->numb_gc; col++) 
				free(view->col_seq[num][col]);
			free(view->col_seq[num]);
			}
		}
	if(reponse == 31) {
		view->numb_gc = view->numb_dnacolors;
		view->curr_colors = view->dnacolors;
		}
	else if(reponse == 32) {
		set_aa_color_mode(USING_STANDARD_COLORS);
		view->numb_gc = view->numb_stdprotcolors;
		view->curr_colors = view->stdprotcolors;
		}
	else if(reponse == 33) {
		set_aa_color_mode(USING_ALT_COLORS);
		view->numb_gc = view->numb_altprotcolors;
		view->curr_colors = view->altprotcolors;
		}
	view->col_seq = prepcolseqs(view->sequence, view->tot_seqs, 
		view->max_seq_length, view->each_length, 
		(view->protein ? get_color_for_aa : get_color_for_base),
		view->numb_gc);
	fl_redraw_object(view->DNA_obj);
	if(view->alt_colors != NO_ALT_COLORS) {
		if(reponse >= 32)  view->alt_colors = 
			(color_choice)(reponse - 31);
		}
	fl_setpup_mode(mainpop, reponse, FL_PUP_CHECK);
	fl_reset_cursor(view->dnawin);
	}
else if (reponse == 34 && view->numb_gc > 1 ) { /* toggle inverted_colors */
	int style;
	view->inverted_colors = !view->inverted_colors;
	if(view->inverted_colors) {
		style = FL_FIXED_STYLE;
		}
	else	{
		style = FL_FIXEDBOLD_STYLE;
		}
	fl_setpup_mode(mainpop, 34, 
		(view->inverted_colors ? FL_PUP_CHECK : FL_PUP_BOX) );
	fl_set_object_lstyle(view->DNA_obj, style);
	}
else if(reponse == 40) { /* toggle sequence edit allowed */
	view->allow_seq_edit = !view->allow_seq_edit;
	if( view->allow_seq_edit ) {
		fl_setpup_mode(mainpop, 50, FL_PUP_NONE);
		}
	else	{
		fl_setpup_mode(mainpop, 50, FL_PUP_GRAY);
		}
	}
else if(reponse == 41) { /* toggle view by reference */
	int status = fl_getpup_mode(mainpop, 41) & FL_PUP_CHECK;
	reference_toggle(view, status);
	}
else if(reponse >= 51 && reponse <= 53) { /* dna keys */
	fl_setpup_mode(mainpop, 51, FL_PUP_RADIO);
	fl_setpup_mode(mainpop, 52, FL_PUP_RADIO);
	fl_setpup_mode(mainpop, 53, FL_PUP_RADIO);
	if( ! view->hjkl ) {
		fl_setpup_mode(mainpop, reponse, FL_PUP_CHECK);
		view->hjkl = reponse - 50;
		}
	else if(view->hjkl + 50 == reponse) {
		view->hjkl = 0;
		}
	else	{
		fl_setpup_mode(mainpop, reponse, FL_PUP_CHECK);
		view->hjkl = reponse - 50;
		}
	}
else if(reponse == 56) { /* toggle use of clustal options */
	/* rien a faire car passage ds le menu le fait tout seul */
	}
else if(reponse == 57) { /* edit clustal options */
	const char *rep;
	int l; 
	rep = fl_show_input("Enter clustal options "
#ifdef OLD_CLUSTALW
	"(ex: /gapopen=5)", 
#else
	"(ex: -gapopen=5)", 
#endif
		(view->clustal_options == NULL ? "": view->clustal_options) );
	if(rep == NULL) return;
	if(view->clustal_options != NULL) free(view->clustal_options);
	l = strlen(rep);
	while(l > 0 && rep[l-1] == ' ') l--;
	if(l == 0) view->clustal_options = NULL;
	else	{
		char *p, *q;
		view->clustal_options = malloc(l+1);
		/* copy rep to clustal_options */
		p = (char *)rep - 1; q = view->clustal_options;
		while(*(++p) != 0) *(q++) = *p;
		*q = 0;
		}
	handle_props_pup(ob, reponse);
	}
else if(reponse == 61) { /* edit consensus value */
	const char *rep;
	char tmp[50];
	int value; char *p, *q;
	sprintf(tmp, "%d %%", view->consensus_threshold);
	rep = fl_show_input("Set consensus threshold value in %", tmp );
	if(rep == NULL) return;
	value = -1;
	sscanf(rep, "%d", &value);
	if(value <= 0 || value > 100) return;
	view->consensus_threshold = value;
	handle_props_pup(ob, reponse);
	}
}


void edit_menu_callback(FL_OBJECT *ob, long which)
{
FL_OBJECT *scroller;
SEA_VIEW *view;
int reponse = fl_get_menu(ob);
if(reponse == -1 ) return;
scroller = (FL_OBJECT *) ob->u_vdata;
view = (SEA_VIEW *)scroller->u_vdata;
if(reponse == RENAME_SEQ) { /* rename the selected sequence */
	rename_sequence(view);
	}
else if(reponse == EDIT_COMMENTS) {
	edit_comments_dialog(scroller);
	}
else if(reponse == DELETE_SEQ) { /* delete selected sequences from alignment */
	char temp[100];
	sprintf(temp,"Confirm request of deletion of %d sequence(s)",
		view->tot_sel_seqs);
	if( fl_show_question(temp, 0) ) {
		if( delete_selected_seqs(view) )
			fl_show_alert("Not enough memory for this operation",
				"","",TRUE);
		else	{
			compute_size_params(scroller, TRUE);
			update_menu_footers(view);
			fl_redraw_object(view->DNA_obj);
			fl_redraw_object(view->vertsli);
			}
		}
	}
else if(reponse == CREATE_SEQ) { /* create a new sequence */
	char *newname;
	newname = (char *) fl_show_input("Name of the new sequence?", "");
	if(newname == NULL || strlen(newname) == 0) return;
	add_seq_to_align(scroller, newname, "-", 1);
	}
else if(reponse == LOAD_SEQ) { /* load a new sequence */
	load_seq_dialog(scroller);
	}
else if(reponse == DUPLICATE_SEQ || reponse == COMPLEMENT_SEQ ||
		 reponse == REVERSE_SEQ) { 
	int num, lenseq, old_first_seq;
	char *newseq, *p, *q, newname[100];
	if(view->tot_sel_seqs != 1 ) return; /* par securite */
	if(reponse != DUPLICATE_SEQ && view->protein) return; /* par securite */
	for(num = 0; num < view->tot_seqs; num++)
		if(view->sel_seqs[num]) break;
	lenseq = view->each_length[num];
	newseq = (char *)malloc(lenseq + 1);
	if(newseq == NULL) {
		fl_show_alert("Not enough memory",
			"to create a new sequence", "", TRUE);
		fl_set_menu_item_mode(ob, reponse, FL_PUP_GRAY);
		return;
		}
	if(reponse == DUPLICATE_SEQ) {
		strcpy(newseq, view->sequence[num]);
		}
	else	{
		p = view->sequence[num]; q = newseq + lenseq - 1;
		while( *p != 0) {
			if(reponse == COMPLEMENT_SEQ) *q = complement_base(*p);
			else *q = *p;
			p++; q--;
			}
 		newseq[lenseq] = 0;
		}
	if(reponse == COMPLEMENT_SEQ) strcpy(newname,"C_");
	else if(reponse == REVERSE_SEQ) strcpy(newname,"R_");
	else	strcpy(newname,"D_");
	strcat(newname,view->seqname[num]);
	fl_freeze_form(scroller->form);
	old_first_seq = view->first_seq;
	add_seq_to_align(scroller, newname , newseq, lenseq);
	free(newseq);	 
/* placer la nouvelle seq apres celle de depart */
	view->sel_seqs[num] = FALSE;
	view->sel_seqs[view->tot_seqs - 1] = TRUE;
	deplacer_grp_seqs( view, FL_min(num + 2, view->tot_seqs) );
/* montrer les 2 seqs concernees */
	if(old_first_seq > num + 1) view->first_seq = 
		FL_min(num + 1, view->tot_seqs - view->tot_lines + 1);
	else if(old_first_seq + view->tot_lines - 1 < num + 2) 
		view->first_seq = FL_min(num + 2, 
			view->tot_seqs - view->tot_lines + 1);
	else
		view->first_seq = old_first_seq;
	fl_set_slider_value(view->vertsli, view->first_seq);
	fl_unfreeze_form(scroller->form);
	}
else if(reponse == EXCHANGE_UT) { /* exchange Us and Ts */
	int num, col; char *p, *q;
	if(view->tot_sel_seqs == 0 || view->protein) return; /* par securite */
	my_watch_cursor(view->dnawin);
	col = get_color_for_base('T');
	for(num = 0; num < view->tot_seqs; num++) {
		if( ! view->sel_seqs[num] ) continue;
		p = view->sequence[num] - 1;
		q = view->col_seq[num][col] - 1;
		while( *(++p) != 0 ) {
			++q;
			if( *p == 'U' ) {
			   	*p = *q = 'T'; 
			   	}
			else if( *p == 'T' ) {
			   	*p = *q = 'U'; 
			   	}
			}
		}
	view->modif_but_not_saved = TRUE;
	view->draw_names = 0;
	fl_redraw_object(view->DNA_obj);
	fl_reset_cursor(view->dnawin);
	}
else if(reponse == ALIGN_SEQS) { /* align selected sites */
	my_watch_cursor(view->dnawin);
	align_selected_parts(view);
	fl_redraw_object(view->DNA_obj);
	fl_redraw_object(view->horsli);
	fl_reset_cursor(view->dnawin);
	}
else if(reponse == DOT_PLOT) { /* dot plot */
	int num1, num2;
	extern void show_dot_plot(char *seq1, char *seq2, char *seqname1, 
		char *seqname2, int l1, int l2, int maxseqlength, 
		void *seaview_data);

	if(view->tot_sel_seqs != 2) return;
	for(num1 = 0; num1 < view->tot_seqs; num1++)
		if(view->sel_seqs[num1]) break;
	for(num2 = num1 + 1; num2 < view->tot_seqs; num2++)
		if(view->sel_seqs[num2]) break;
	show_dot_plot(view->sequence[num1], view->sequence[num2],
		view->seqname[num1], view->seqname[num2],
		view->each_length[num1], view->each_length[num2],
		view->max_seq_length, (void *)view);
	}
else if (reponse == CONSENSUS_SEQ) {
	char *newseq, newname[100];
	int old_total, *tmp, new_pos, i, old_first_seq;

	if(view->tot_sel_seqs <= 1 ) return; /* par securite */
	newseq = cre_consensus(view, newname);
	if(newseq == NULL) {
		fl_show_alert("Not enough memory",
			"to create a new sequence", "", TRUE);
		fl_set_menu_item_mode(ob, reponse, FL_PUP_GRAY);
		return;
		}
	fl_freeze_form(scroller->form);
	old_first_seq = view->first_seq;
	old_total = view->tot_seqs;
	add_seq_to_align(scroller, newname , newseq, strlen(newseq));
	free(newseq);	
	for(i=0; i < view->tot_seqs; i++)
		if(view->sel_seqs[i]) new_pos = i;
	new_pos += 2;
	if(view->tot_seqs == old_total || new_pos == view->tot_seqs) {
		fl_unfreeze_form(scroller->form);
		return; 
		}
/* placer la nouvelle seq apres la derniere des selectionnees */
	tmp = (int *)calloc(view->tot_seqs, sizeof(int));
	if(tmp == NULL) {
		fl_unfreeze_form(scroller->form);
		return; 
		}
	memcpy(tmp, view->sel_seqs, view->tot_seqs * sizeof(int) );
	memset(view->sel_seqs, 0, view->tot_seqs * sizeof(int) );
	view->sel_seqs[view->tot_seqs - 1] = TRUE;
	old_total = view->tot_sel_seqs;
	view->tot_sel_seqs = 1;
	deplacer_grp_seqs( view, new_pos );
	memcpy(view->sel_seqs, tmp, view->tot_seqs * sizeof(int) );
	view->tot_sel_seqs = old_total;
	free(tmp);
/* montrer la seq concernee */
	if(old_first_seq > new_pos ) view->first_seq = 
		FL_min(new_pos - 2,  1);
	else if(old_first_seq + view->tot_lines - 1 < new_pos) 
		view->first_seq = FL_min(new_pos - 2, 
			view->tot_seqs - view->tot_lines + 1);
	else
		view->first_seq = old_first_seq;
	fl_set_slider_value(view->vertsli, view->first_seq);
	fl_unfreeze_form(scroller->form);
	}
else if(reponse == DELETE_GAP_ONLY_SITES) {
	if( !fl_show_question("Confirm remove all gap_containing sites?", 0) ) 
		return;
	my_watch_cursor(view->dnawin);
	del_gap_only_sites(view);
	compute_size_params(scroller, TRUE);
	fl_redraw_object(view->DNA_obj);
	fl_redraw_object(view->horsli);
	fl_reset_cursor(view->dnawin);
	}
}


void set_and_show_new_cursor_site(SEA_VIEW *view, int new_pos, int center,
	int force_redraw)
{
int old_pos;
old_pos = view->cursor_site;
if(new_pos != old_pos)
	view->cursor_site = new_pos;
if(new_pos >= view->first_site && new_pos < view->first_site +
	view->tot_sites - 1) {
	if( !force_redraw && 
		( (view->cursor_in_comment && view->mod_comment_line == 0) ||
		(!view->cursor_in_comment && view->mod_seq == 0) ) )
		view->mod_cursor = TRUE; 
	}
else	{
	if(center) 
		view->first_site = 
			view->cursor_site - view->tot_sites/2;
	else	{
		if(new_pos >= old_pos) 
			view->first_site = view->cursor_site + 10 - 
				view->tot_sites;
		else
			view->first_site = view->cursor_site - 10;
		}
	if(view->first_site + view->tot_sites - 1 >
		view->seq_length + 1 )
		view->first_site = view->seq_length - view->tot_sites + 2;
	if(view->first_site <=0 ) 
		view->first_site = 1;
	fl_set_slider_value(view->horsli,view->first_site);
	view->mod_seq = 0;
	view->mod_comment_line = 0;
	}
view->draw_names = 0;
fl_redraw_object( view->DNA_obj );
}


void set_and_show_new_cursor_seq(SEA_VIEW *view, int new_pos)
{
if(view->cursor_in_comment) {
	if(view->comment_length[new_pos - 1] + 1 < view->cursor_site) {
		fl_ringbell(0); 
		return;
		}
	if(new_pos == view->cursor_seq) return;
	view->cursor_seq = new_pos;
	view->mod_cursor = TRUE;
	view->draw_names = 0;
	fl_redraw_object( view->DNA_obj );
	return;
	}
if(view->each_length[new_pos - 1] + 1 < view->cursor_site) {
	fl_ringbell(0); return;
	}
if(new_pos != view->cursor_seq || new_pos < view->first_seq ||
	view->cursor_site != view->old_cursor_site ||
	new_pos >= view->first_seq + view->tot_lines) {
	view->cursor_seq = new_pos;
	if(new_pos >= view->first_seq && new_pos < view->first_seq +
		view->tot_lines) {
		view->mod_cursor = TRUE;
		view->draw_names = 0;
		}
	else	{
		view->first_seq = view->cursor_seq - view->tot_lines/2;
		if(view->first_seq + view->tot_lines >= 
			view->tot_seqs )
			view->first_seq = view->tot_seqs - view->tot_lines + 1;
		if(view->first_seq <=0 ) 
			view->first_seq = 1;
		fl_set_slider_value(view->vertsli,view->first_seq);
		view->draw_names = -1;
		}
	fl_redraw_object( view->DNA_obj );
	}
}


void goto_callback(FL_OBJECT *ob, long which)
{
FL_OBJECT *scroller;
SEA_VIEW *view;
int l, maxi, num, new_pos = -1, numerique = TRUE;
char *p, *q, target[50];
static char *upname;
static int l_upname = 0;

if(which == 0) { /* appele par bouton goto */
	ob = (FL_OBJECT *)ob->u_vdata;
	}
scroller = (FL_OBJECT *)ob->u_vdata;
view = (SEA_VIEW *)scroller->u_vdata;
if(view->tot_seqs == 0) return;
p = (char *)fl_get_input(ob);
q = p - 1; while(*(++q) != 0) {
	if(!isdigit(*q)) numerique = FALSE;
	}
if(numerique) { /* aller a une position de l'alignement */
	sscanf(p,"%d",&new_pos);
	if(view->cursor_in_comment)
		maxi = view->comment_length[view->cursor_seq - 1];
	else
		maxi = view->each_length[view->cursor_seq - 1];
	if( new_pos <= 0 || new_pos > maxi ) {
		fl_set_input(ob,"");
		fl_ringbell(0);
		}
	else	{
		set_and_show_new_cursor_site(view, new_pos, TRUE, FALSE);
		}
	}
else	{ /* recherche d'une seq par son nom */
	l = strlen(p);
	if(l > sizeof(target) - 1) l = sizeof(target) - 1;
	strncpy(target, p, l); target[l] = 0;
	q = target - 1; while (*(++q) != 0) *q = toupper(*q);
	if(view->wid_names > l_upname) {
		if(l_upname > 0) free(upname);
		upname = (char *)malloc(view->wid_names + 1);
		l_upname = view->wid_names;
		}
	for(num = 0; num < view->tot_seqs; num++) {
		strncpy(upname, view->seqname[num], l_upname);
		upname[l_upname] = 0;
		q = upname - 1; while (*(++q) != 0) *q = toupper(*q);
		if(strstr(upname, target) != NULL) break;
		}
	if(num >= view->tot_seqs) {
		fl_ringbell(0);
		return;
		}
	if(view->cursor_site < view->first_site || 
		view->cursor_site >= view->first_site + view->tot_sites)
		view->cursor_site = view->first_site;
	set_and_show_new_cursor_seq(view, num + 1);
	}
fl_set_focus_object(ob->form, view->DNA_obj);
}


char *search_with_gaps(char *target, char *debut)
{
char *cherche, *trouve = debut - 1;
do	{
	debut = trouve + 1;
	trouve = NULL;
	cherche = target;
	do	{
		while( *debut == '-' ) debut++;
		if(trouve == NULL) trouve = debut;
		if ( toupper(*debut) != *cherche ) break;
		cherche++; debut++;
		}
	while( *cherche != 0 );
	}
while( *trouve != 0  && *cherche != 0);
return ( *cherche == 0 ? trouve : NULL );
}


void search_callback(FL_OBJECT *ob, long which)
{
FL_OBJECT *scroller;
SEA_VIEW *view;
char target[50], *pos, *debut;
int new_pos;
int l = -1;

if(which == 1) { /* appele par champ input */
	pos = (char *)fl_get_input(ob) - 1;
	scroller = (FL_OBJECT *) ob->u_vdata;
	}
else	{ /* appele par bouton search */
	pos = (char *)fl_get_input((FL_OBJECT *)ob->u_vdata) - 1;
	scroller = (FL_OBJECT *)( (FL_OBJECT *)ob->u_vdata )->u_vdata;
	}
view = (SEA_VIEW *)scroller->u_vdata;
if(view->tot_seqs == 0) return;
while( *++pos && l+1 < sizeof(target) ) 
	target[++l] = toupper(*pos);
target[++l] = 0;
if( l == 0 ) return;
if(view->cursor_in_comment)
	debut = view->comment_line[view->cursor_seq - 1] + view->cursor_site;
else
	debut = view->sequence[view->cursor_seq - 1] + view->cursor_site;
pos = search_with_gaps(target, debut);
if(pos == NULL) fl_ringbell(0);
else	{
	if(view->cursor_in_comment)
		new_pos = pos - view->comment_line[view->cursor_seq - 1] + 1;
	else
		new_pos = pos - view->sequence[view->cursor_seq - 1] + 1;
	set_and_show_new_cursor_site(view, new_pos, TRUE, FALSE);
	}
fl_set_focus_object(ob->form,view->DNA_obj);
}


void free_alignment(SEA_VIEW *view)
{
int num, c;
if(view->header!=NULL) { free(view->header); view->header = NULL; }
for(num = 0; num < view->tot_seqs; num++) {
	free(view->sequence[num]);
	free(view->seqname[num]);
	if(view->comments[num] != NULL) free(view->comments[num]);
	if(view->numb_gc > 1) {
		for(c = 0; c < view->numb_gc; c++) free(view->col_seq[num][c]);
		free(view->col_seq[num]);
		}
	}
if( view->numb_gc > 1 && view->tot_seqs > 0 ) free(view->col_seq);
if(view->masename != NULL) {
	free(view->masename);
	view->masename = NULL;
	}
if( view->tot_seqs >= 1 ) {
	free(view->each_length);
	free(view->sel_seqs);
	free(view->region_line);
	}
while (view->regions != NULL)
	delete_region(view, 1);
for(num = 0; num < view->numb_species_sets; num++) {
	free(view->list_species_sets[num]);
	free(view->name_species_sets[num]);
	}
view->numb_species_sets = 0;
view->tot_seqs = 0;
view->tot_sel_seqs = 0;
free_region(view->active_region);
view->active_region = NULL;
if(view->menu_file != NULL) {
	fl_set_menu_item_mode(view->menu_file,SAVE,FL_PUP_GRAY);
	fl_set_menu_item_mode(view->menu_file,SAVE_AS,FL_PUP_GRAY);
	fl_set_menu_item_mode(view->menu_file,SAVE_REGIONS,FL_PUP_GRAY);
	}
if(view->tot_comment_lines > 0) {
	for(num = 0; num < view->tot_comment_lines; num++) {
		free(view->comment_name[num]);
		free(view->comment_line[num]);
		}
	free(view->comment_name);
	free(view->comment_line);
	free(view->comment_length);
	view->tot_comment_lines = 0;
	view->show_comment_lines = FALSE;
	}
}


void load_alignment_file(FL_OBJECT *scroller, char *filename, char *message, 
	char *pattern, known_format file_format)
{
SEA_VIEW *view;
char *err_message;
int protein;
view = (SEA_VIEW *)scroller->u_vdata;
if(view->tot_seqs > 0 && view->modif_but_not_saved) {
	if( ! fl_show_question("Alignment was modified but not saved\n"
		"Do you want to read a new alignment?", 0) ) return;
	}
if(filename == NULL) {
	filename = (char *)fl_show_fselector(message,"",pattern,"");
	if(filename==NULL) return;
	}
my_watch_cursor(view->dnawin);
if(view->alt_col_seq != NULL) {
	reference_toggle(view, FALSE);
	}
free_alignment(view);
if(file_format == MASE_FORMAT)
	view->tot_seqs = read_mase_seqs_header(filename,view->sequence,
		view->seqname, view->comments, &view->header, MAXNSEQS,
		&err_message);
else if(file_format == FASTA_FORMAT)
	view->tot_seqs = read_fasta_align(filename,view->sequence,
		view->seqname, view->comments, &view->header, MAXNSEQS,
		&err_message);
else if(file_format == PHYLIP_FORMAT)
	view->tot_seqs = read_phylip_align(filename, view->sequence,
		view->seqname, view->comments, &view->header, MAXNSEQS,
		&err_message);
else if(file_format == CLUSTAL_FORMAT)
	view->tot_seqs = read_clustal_align(filename, view->sequence,
		view->seqname, view->comments, &view->header, MAXNSEQS,
		&err_message);
else if(file_format == MSF_FORMAT)
	view->tot_seqs = read_msf_align(filename, view->sequence,
		view->seqname, view->comments, &view->header, MAXNSEQS,
		&err_message);
if(view->tot_seqs == 0) {
	fl_reset_cursor(view->dnawin);
	fl_show_alert("Error while reading file", filename,
		err_message, TRUE);
	fl_redraw_object(view->DNA_obj);
	return;
	}
fl_freeze_form(scroller->form);
protein = is_a_protein_seq(view->sequence[0]);
	{ char *q, *p = filename;
	while ( (q = strchr(p, '/')) != NULL ) p = q + 1;
	fl_wintitle(view->dnawin, p);
	}
if(view->menu_file != NULL) {
	if(file_format == view->format_for_save) {
		fl_set_menu_item_mode(view->menu_file,SAVE,FL_PUP_NONE);
		fl_set_menu_item_mode(view->menu_file,SAVE_AS,FL_PUP_NONE);
		}
	else	{
		fl_set_menu_item_mode(view->menu_file,SAVE_AS,FL_PUP_NONE);
		if(view->masename != NULL) {
			free(view->masename);
			view->masename = filename = NULL;
			}
		}
	}
init_dna_scroller(scroller, view->sequence, view->seqname, 
	view->tot_seqs, view->comments, filename, protein,
	view->header);
fl_unfreeze_form(scroller->form);
fl_redraw_object(view->DNA_obj);
fl_reset_cursor(view->dnawin);
return;
}

void printout_appbutton_callback(int *target_value, char *target_label,
	const char *message)
/* changer la valeur d'une option et l'afficher sur le bouton correspondant */
{
char *p, defaut[10];
int old_val, new_val, lu;
FL_FORM *fsel_form;
FL_OBJECT *obj;

old_val = *target_value;
sprintf(defaut, "%d", old_val);
p = (char *)fl_show_input(message, defaut);
if(p == NULL) return;
lu = sscanf(p, "%d", &new_val);
if( lu != 1 || new_val == old_val || new_val <= 0) return;
fsel_form = fl_get_fselector_form();
obj = fsel_form->last;
while (obj != NULL) { /* rechercher le bouton par son label */
	if( strcmp(obj->label, target_label) == 0) break;
	obj = obj->prev;
	}
if(obj == NULL) return;
p = strchr(target_label, '=') + 1;
sprintf(p, "%d", new_val);
fl_set_object_label(obj, target_label);
fl_redraw_object(obj);
*target_value = new_val;
}

/* les 3 callbacks des boutons du fselector pour option printout */
void printout_block_callback(void *p)
{
printout_appbutton_callback(&printout_block, printout_label_block,
	"Enter desired printout block size");
}

void printout_cpl_callback(void *p)
{
printout_appbutton_callback(&printout_cpl, printout_label_cpl,
	"Enter desired number of residues per line");
}

void printout_lpp_callback(void *p)
{
printout_appbutton_callback(&printout_lpp, printout_label_lpp,
	"Enter desired number of lines per page");
}


int mainwin_close_callback(FL_FORM *form, void *data)
{
SEA_VIEW *view = (SEA_VIEW *)data;
if(view->modif_but_not_saved) {
	if( ! fl_show_question(
		"Alignment was modified but not saved\n"
		"Do you want to quit anyway?", 0) ) return FL_IGNORE;
	}
exit(0);
}


void file_menu_callback(FL_OBJECT *ob, long which)
{
FL_OBJECT *scroller;
SEA_VIEW *view;
static char *pattern[] = {"*.mase", "*", "*.aln", "*.msf", "*"};
static char *typename[] = {"Mase", "Phylip", "Clustal", "MSF", "Fasta"};
static char fsel_message[50];
const char *filename;
int reponse=fl_get_menu(ob);
if(reponse == -1 ) return;
scroller = (FL_OBJECT *) ob->u_vdata;
view = (SEA_VIEW *)scroller->u_vdata;
if(reponse == QUIT) { /* quit */
	mainwin_close_callback(ob->form, view);
	}
else if(reponse == OPEN_MASE) { 
	load_alignment_file(scroller, NULL, "Choose a .mase file", 
	"*.mase", MASE_FORMAT);
	}
else if(reponse == OPEN_PHYLIP )	{  
	load_alignment_file(scroller, NULL,
		"Choose a Phylip file" ,"*",  PHYLIP_FORMAT );
	}
else if(reponse == OPEN_FASTA )	{  
	load_alignment_file(scroller, NULL,
		"Choose a Fasta file" ,"*",  FASTA_FORMAT );
	}
else if(reponse == OPEN_CLUSTAL )	{  
	load_alignment_file(scroller, NULL,
		"Choose a Clustal file" ,"*.aln",  CLUSTAL_FORMAT );
	}
else if(reponse == OPEN_MSF )	{  
	load_alignment_file(scroller, NULL,
		"Choose an MSF file" ,"*.msf",  MSF_FORMAT );
	}
else if(reponse == SAVE || reponse == SAVE_AS) 	{ 
	char *err;
	if(reponse == SAVE_AS) { /* Save as */
		FILE *in;
		sprintf(fsel_message,"Choose a %s file name", 
			typename[view->format_for_save]);
		filename = fl_show_fselector(fsel_message, "",
			pattern[view->format_for_save], 
			extract_filename(view->masename));
		if(filename==NULL) return;
		in = fopen(filename,"r");
		if(in != NULL) {
			char tmp[200];
			fclose(in);
			sprintf(tmp, "This file already exists\n%s\n"
				"Do you want to overwrite it?", filename);
			if( ! fl_show_question(tmp, 0) )
				return;
			}
		}
	else 	filename = view->masename;
	my_watch_cursor(view->dnawin);
	save_active_region(view, TRUE);
	err = save_alignment_or_region(filename, view->sequence, view->comments,
		view->header, view->seqname, view->tot_seqs, view->each_length,
		view->regions, NULL, view->format_for_save,
		view->numb_species_sets, view->list_species_sets,
		view->name_species_sets, NULL, 0, view->protein,
		view->tot_comment_lines, view->comment_name, 
		view->comment_line);
	fl_reset_cursor(view->dnawin);
	if(err != NULL) fl_show_alert(err, "", "", TRUE);
	else 	{
		if(reponse == SAVE_AS) {
			if(view->masename != NULL) free(view->masename);
			view->masename=(char *)malloc(strlen(filename)+1);
			if(view->masename == NULL) out_of_memory();
			strcpy(view->masename,filename);
			{ char *q, *p = (char *)filename;
			while ( (q = strchr(p, '/')) != NULL ) p = q + 1;
			fl_wintitle(view->dnawin, p);
			}
			fl_set_menu_item_mode(ob, SAVE, FL_PUP_NONE);
			}
		view->modif_but_not_saved = FALSE;
		}
	}
else if(reponse == SAVE_REGIONS) {  /* save current regions choice */
	char *err;
	static char regions_only_filename[200];
	static int first = TRUE;
	if(first) {
	 	strcpy(regions_only_filename, "regions");
		if( strchr(pattern[view->format_for_save],'.') != NULL)
			strcat(regions_only_filename, 
				pattern[view->format_for_save]+1);
		first = FALSE;
		}
	if(view->active_region == NULL) return;
	sprintf(fsel_message,"Choose a %s file name", 
			typename[view->format_for_save]);
	filename = fl_show_fselector(fsel_message,"",
			pattern[view->format_for_save], 
			regions_only_filename);
	if(filename==NULL) return;
	my_watch_cursor(view->dnawin);
	err = save_alignment_or_region(filename, view->sequence, view->comments,
		view->header, view->seqname, view->tot_seqs, view->each_length,
		NULL, view->active_region, view->format_for_save,
		0, NULL, NULL, view->sel_seqs, view->tot_sel_seqs, 
		view->protein, 0, NULL, NULL);
	fl_reset_cursor(view->dnawin);
	if(err != NULL) fl_show_alert(err, "", "", TRUE);
	else strcpy(regions_only_filename, extract_filename(filename));
	}
else if(reponse == PRINTOUT && view->tot_seqs > 0) {
	int anerr;
	FILE *in;
	char *p;
	static char save_label_block[PRINTOUT_LABEL_SIZE];
	static char save_label_cpl[PRINTOUT_LABEL_SIZE];
	static char save_label_lpp[PRINTOUT_LABEL_SIZE];

	p = strchr(printout_label_block, '='); 
	sprintf(p+1, "%d", printout_block);
	fl_add_fselector_appbutton(printout_label_block, 
		printout_block_callback, NULL);
	strcpy(save_label_block, printout_label_block);
	p = strchr(printout_label_cpl, '='); sprintf(p+1, "%d", printout_cpl);
	fl_add_fselector_appbutton(printout_label_cpl, printout_cpl_callback, NULL);
	strcpy(save_label_cpl, printout_label_cpl);
	p = strchr(printout_label_lpp, '='); sprintf(p+1, "%d", printout_lpp);
	fl_add_fselector_appbutton(printout_label_lpp, printout_lpp_callback, NULL);
	strcpy(save_label_lpp, printout_label_lpp);
	
	filename = (char *) fl_show_fselector("Enter a printout file name",
			 "", "", "");
	
	fl_remove_fselector_appbutton(save_label_lpp);
	fl_remove_fselector_appbutton(save_label_cpl);
	fl_remove_fselector_appbutton(save_label_block);

	if(filename==NULL) return;
	in = fopen(filename,"r");
	if(in != NULL) {
		char tmp[200];
		fclose(in);
		sprintf(tmp, "This file already exists\n%s\n"
				"Do you want to overwrite it?", filename);
		if( ! fl_show_question(tmp, 0) )
			return;
		}
	my_watch_cursor(view->dnawin);
	if( view->alt_col_seq != NULL ) {
		for(anerr = 0; anerr < view->tot_seqs; anerr++)
			if(view->sel_seqs[anerr]) break;
		}
	else	anerr = -1;
	anerr = prepare_printout(filename, view->sequence, view->tot_seqs,
		view->seqname, view->each_length, printout_cpl,
		printout_block, printout_lpp, view->masename, anerr);
	fl_reset_cursor(view->dnawin);
	if( anerr ) fl_show_alert("Error while writing to file",
		filename, "", TRUE);
	}
}

typedef struct {
	FL_OBJECT *br_item, *box;
	FILE *in;
	} browser_struct;

void browser_titres_callback(FL_OBJECT *ob, long which)
{
int reponse;
static char line[200];
char *p;
browser_struct *br_data;
FL_Coord x, y, w, h;
int charwidth, newwidth, maxl;

reponse = fl_get_browser(ob);
if(reponse == 0) return;
br_data = (browser_struct *)(ob->u_vdata);
p = (char *)fl_get_browser_line(ob, reponse);
rewind(br_data->in);
fl_hide_object(br_data->box);
fl_freeze_form(ob->form);
fl_clear_browser(br_data->br_item);
while( fgets(line, sizeof(line), br_data->in) != NULL) {
	if(strncmp(line, ">>>", 3) != 0) continue;
	line[ strlen(line) - 1 ] = 0;
	if(strcmp(line+3, p) != 0) continue;
	maxl = 0;
	while( fgets(line, sizeof(line), br_data->in) != NULL && 
			strncmp(line, ">>>", 3) != 0 ) {
		line[ strlen(line) - 1 ] = 0;
		if(*line == 0) strcpy(line," ");
		fl_add_browser_line(br_data->br_item, line);
		maxl = FL_max(maxl, (int)strlen(line));
		}
	/* agrandir fenetre si pas assez large */
	fl_get_browser_dimension(br_data->br_item, &x, &y, &w, &h);
	charwidth = fl_get_char_width(FL_FIXED_STYLE, FL_MEDIUM_SIZE);
	maxl++;
	if( w < maxl * charwidth) {
		newwidth = ob->form->w + (maxl * charwidth - w);
		fl_set_form_size(ob->form, newwidth, ob->form->h );
		}
	fl_set_object_label(br_data->box, p);
	break;
	}
fl_show_object(br_data->box);
fl_unfreeze_form(ob->form);
}


void browser_ok_callback(FL_OBJECT *ob, long which)
{
fl_hide_form(ob->form);
}

int simple_close_callback(FL_FORM *form, void *data)
{
return FL_OK;
}


void free_colseqs_by_difference(char ***alt_col_seq, int total, int numb_gc)
{
int num, c;

if(alt_col_seq == NULL) return;
for(num = 0; num < total; num++) {
	for(c = 0; c < numb_gc; c++) free(alt_col_seq[num][c]);
	free(alt_col_seq[num]);
	}
if( total > 0 ) free(alt_col_seq);
}


void reference_toggle(SEA_VIEW *view, int on)
{
int menu;
char ***tmp;
static int old_pos;

if(view->numb_gc == 1) return;
menu = view->bouton_props->u_ldata;
if(on) { /* tenter de passer en mode par reference */
	if(view ->inverted_colors || view->tot_sel_seqs != 1 || 
			view->numb_gc == 1) {
		fl_setpup_mode(menu, 41, FL_PUP_BOX);
		return;
		}
	my_watch_cursor(view->dnawin);
	for(old_pos = 0; old_pos < view->tot_seqs; old_pos++)
		if(view->sel_seqs[old_pos]) break;
	deplacer_grp_seqs(view, 1);
	view->first_seq = 1;
	fl_set_slider_value(view->vertsli, 1);
	view->alt_col_seq = prepcolseqs_by_difference(view->sequence, 
		view->tot_seqs, 0, 
		view->max_seq_length,
		view->each_length, 
		( view->protein ? get_color_for_aa : get_color_for_base ), 
		view->numb_gc);
	fl_reset_cursor(view->dnawin);
	if(view->alt_col_seq == NULL) {
		fl_redraw_object(view->DNA_obj);
		fl_setpup_mode(menu, 41, FL_PUP_BOX);
		return;
		}
	tmp = view->alt_col_seq;
	view->alt_col_seq = view->col_seq;
	view->col_seq = tmp;		
	fl_redraw_object(view->DNA_obj);
	fl_setpup_mode(menu, 30, FL_PUP_GRAY);
	fl_setpup_mode(menu, 41, FL_PUP_CHECK);
	fl_set_object_lcol(view->menu_edit, FL_INACTIVE_COL);
	fl_deactivate_object(view->menu_edit);
	fl_set_object_lcol(view->menu_species, FL_INACTIVE_COL);
	fl_deactivate_object(view->menu_species);
	}
else	{ /* retour mode normal */
	tmp = view->alt_col_seq;
	view->alt_col_seq = view->col_seq;
	view->col_seq = tmp;		
	free_colseqs_by_difference(view->alt_col_seq, view->tot_seqs, 
		view->numb_gc);
	view->alt_col_seq = NULL;
	deplacer_grp_seqs(view, old_pos + 1);
	fl_redraw_object(view->DNA_obj);
	fl_setpup_mode(menu, 30, FL_PUP_NONE);
	fl_setpup_mode(menu, 41, FL_PUP_BOX);
	fl_set_object_lcol(view->menu_edit, FL_BLACK);
	fl_activate_object(view->menu_edit);
	fl_set_object_lcol(view->menu_species, FL_BLACK);
	fl_activate_object(view->menu_species);
	}
}


void help_callback(FL_OBJECT *ob, long which)
{
static browser_struct browser_data;
static int first = TRUE;
static FL_FORM *help_form;

if(first) {
	char line[85];
	FILE *in;
	FL_OBJECT *browser_titres, *browser_item, *ok_button, *br_item_label;
	int titres_width, fin_titres;
	in = open_path(help_file);
	if(in == NULL) {
		fl_show_alert("Help file",help_file,
		   "not found in PATH directories nor in current directory", 
			TRUE);
		return;
		}
	first = FALSE;
	help_form = fl_bgn_form(FL_FLAT_BOX, 885, 325);
	titres_width = fl_get_string_width(FL_NORMAL_STYLE, FL_SMALL_SIZE,
 		"To reduce button labels", 23) * 1.2;
	browser_titres = 
	   	fl_add_browser(FL_HOLD_BROWSER, 5, 20, titres_width, 275, 
		"Choose help item");
	fl_set_object_callback(browser_titres, browser_titres_callback, 0);
	fl_set_object_lalign(browser_titres, FL_ALIGN_TOP);
	fl_set_object_color(browser_titres, FL_MCOL, browser_titres->col2);
  	fl_set_object_resize(browser_titres,FL_RESIZE_Y);
	fl_set_object_gravity(browser_titres,FL_NorthWest,FL_SouthWest);
	ok_button = fl_add_button(FL_NORMAL_BUTTON,5,300,titres_width,20,"OK");
	fl_set_object_callback(ok_button, browser_ok_callback, 0);
  	fl_set_object_resize(ok_button,FL_RESIZE_NONE);
	fl_set_object_gravity(ok_button,FL_SouthWest,FL_SouthWest);
	fin_titres = titres_width + 10;
	browser_item = 
	   	fl_add_browser(FL_NORMAL_BROWSER, fin_titres, 20, 
		880 - fin_titres, 300, "");
	fl_set_object_lalign(browser_item, FL_ALIGN_TOP);
	fl_set_object_color(browser_item, FL_MCOL, browser_item->col2);
	fl_set_browser_fontsize(browser_item, FL_MEDIUM_SIZE);
	fl_set_browser_fontstyle(browser_item, FL_FIXED_STYLE);
  	fl_set_object_resize(browser_item,FL_RESIZE_ALL);
	fl_set_object_gravity(browser_item,FL_NorthWest,FL_SouthEast);
	br_item_label = fl_add_box(FL_NO_BOX, fin_titres, 0, 
		880 - fin_titres, 20,"Help Info");
	fl_set_object_lalign(br_item_label, FL_ALIGN_CENTER);
  	fl_set_object_resize(br_item_label,FL_RESIZE_X);
	fl_set_object_gravity(br_item_label,FL_NorthWest,FL_NorthEast);
	browser_data.box = br_item_label;
	browser_data.br_item = browser_item;
	browser_data.in = in;
	browser_titres->u_vdata = &browser_data;
	fl_end_form();
	fl_set_form_atclose(help_form, simple_close_callback, NULL);
	while( fgets(line, sizeof(line), in) != NULL) {
		if(strncmp(line, ">>>", 3) != 0) continue;
		line[ strlen(line) - 1 ] = 0;
		fl_add_browser_line(browser_titres, line+3);
		}
	}
if(help_form->visible /* <=> window is mapped under X */) {
	my_open_and_raise_form(help_form);
	}
else 
	fl_show_form(help_form, FL_PLACE_FREE, FL_FULLBORDER, "Help");
}


void my_open_and_raise_form(FL_FORM *form)
{
XMapRaised(fl_get_display(), form->window); 
/* marche mal si 2 fois de suite !!
fl_winshow(form->window);
fl_redraw_form(form); 
*/
}


int insert_gaps_at(SEA_VIEW *view, int seq, int site, int total)
{
char *pos, **psequence;
int l, c, gapcolor, *plength;
if(view->cursor_in_comment) {
	psequence = view->comment_line;
	plength = view->comment_length;
	}
else	{
	psequence = view->sequence;
	plength = view->each_length;
	gapcolor = ( view->protein ? 
		get_color_for_aa('-') : get_color_for_base('-') );
	}
l = plength[seq-1];
if(site > l + 1) return total;
if( l + total > view->max_seq_length) total = view->max_seq_length - l;
pos = psequence[seq-1] + site - 1;
memmove(pos+total, pos, l - site + 2);
memset(pos, '-', total);
if( (!view->cursor_in_comment) && view->numb_gc > 1) {
	for (c=0; c<view->numb_gc; c++) {
	   pos= &view->col_seq[seq-1][c][site-1];
	   memmove(pos+total, pos, l - site + 2);
	   memset(pos,' ',total);
	   }
	memset(view->col_seq[seq-1][gapcolor] + site - 1, '-', total);
	}
plength[seq-1] += total;
view->modif_but_not_saved = TRUE;
return total;
}


int delete_gaps_before(SEA_VIEW *view, int numseq, int numsite, int total)
{
char *site, *finseq, **psequence;
int c, count = -1, l, retval, *plength;

psequence = view->sequence;
plength = view->each_length;
site = psequence[numseq-1] + numsite - 1;
finseq = psequence[numseq-1] + plength[numseq-1] - 1;
do	{ site--; count++; }
while ( count < total && site >= psequence[numseq-1] && 
	( view->allow_seq_edit || *site == '-' || site > finseq) );
if(count == 0) return 0;
/* ne rien faire si on efface au dela de la fin de la seq */
if(numsite - count > plength[numseq-1]) return count;
l = plength[numseq-1];
retval = count;
if(numsite > l) { /* effacer depuis au dela fin jusqu'a interieur seq */
	count -= (numsite - l - 1);
	numsite = l + 1;
	}
site = psequence[numseq-1] + numsite - 1;
memmove(site-count, site, l - numsite + 2);
if( view->numb_gc > 1) {
	for (c=0; c < view->numb_gc; c++) {
		site= view->col_seq[numseq-1][c] + numsite - 1;
		memmove(site-count,site, l - numsite + 2);
		}
	}
plength[numseq-1] -= count;
view->modif_but_not_saved = TRUE;
return retval;
}


void adjust_menu_edit_modes(SEA_VIEW *view)
{
if(view->menu_edit != NULL) {
	if(view->tot_sel_seqs != 0) {
		fl_set_menu_item_mode(view->menu_edit,DELETE_SEQ,FL_PUP_NONE);
		fl_set_menu_item_mode(view->menu_edit,CONSENSUS_SEQ,
			FL_PUP_NONE);
		if(!view->protein)fl_set_menu_item_mode(view->menu_edit,
			EXCHANGE_UT,FL_PUP_NONE);
		}
	else	{
		fl_set_menu_item_mode(view->menu_edit,DELETE_SEQ,FL_PUP_GRAY);
		fl_set_menu_item_mode(view->menu_edit,EXCHANGE_UT,FL_PUP_GRAY);
		}
	if(view->tot_sel_seqs == 1) {
		fl_set_menu_item_mode(view->menu_edit,RENAME_SEQ,FL_PUP_NONE);
		fl_set_menu_item_mode(view->menu_edit,DUPLICATE_SEQ,
			FL_PUP_NONE);
		fl_set_menu_item_mode(view->menu_edit,EDIT_COMMENTS,
			FL_PUP_NONE);
		if(!view->protein) {
	      		fl_set_menu_item_mode(view->menu_edit,COMPLEMENT_SEQ,
				FL_PUP_NONE);
			fl_set_menu_item_mode(view->menu_edit,REVERSE_SEQ,
				FL_PUP_NONE);
			}
		}
	else	{
		fl_set_menu_item_mode(view->menu_edit,RENAME_SEQ,FL_PUP_GRAY);
		fl_set_menu_item_mode(view->menu_edit,DUPLICATE_SEQ,
			FL_PUP_GRAY);
		fl_set_menu_item_mode(view->menu_edit,EDIT_COMMENTS,
			FL_PUP_GRAY);
		fl_set_menu_item_mode(view->menu_edit,COMPLEMENT_SEQ,
			FL_PUP_GRAY);
		fl_set_menu_item_mode(view->menu_edit,REVERSE_SEQ,FL_PUP_GRAY);
		}
	if(view->tot_sel_seqs == 2) 
		fl_set_menu_item_mode(view->menu_edit,DOT_PLOT,FL_PUP_NONE);
	else
		fl_set_menu_item_mode(view->menu_edit,DOT_PLOT,FL_PUP_GRAY);
	if(view->tot_sel_seqs >= 2) {
		fl_set_menu_item_mode(view->menu_edit,ALIGN_SEQS,FL_PUP_NONE);
		fl_set_menu_item_mode(view->menu_edit,CONSENSUS_SEQ,
			FL_PUP_NONE);
		}
	else	{
		fl_set_menu_item_mode(view->menu_edit,ALIGN_SEQS,FL_PUP_GRAY);
		fl_set_menu_item_mode(view->menu_edit,CONSENSUS_SEQ,
			FL_PUP_GRAY);
		}
	}
}


void select_deselect_seq(SEA_VIEW *view, int new_seq)
{
/* new_seq = # seq a select/deselect; si 0: tout deselectionner; 
   si -1: tout selectionner 
   si -2: ne pas changer la selection mais ajuster l'interface selon son etat
*/
if(new_seq > 0) { /* traiter une sequence */
	view->sel_seqs[new_seq-1] = !view->sel_seqs[new_seq-1];
	if(view->sel_seqs[new_seq-1])
		++view->tot_sel_seqs;
	else
		--view->tot_sel_seqs;
	}
else if(new_seq == 0)	{ /* tout deselectionner */
	view->tot_sel_seqs = 0;
	memset(view->sel_seqs, 0, view->tot_seqs * sizeof(int));
	}
else if(new_seq == -1)	{ /* tout selectionner */
	int i;
	view->tot_sel_seqs = view->tot_seqs;
	for(i=0; i < view->tot_seqs; i++) view->sel_seqs[i] = TRUE;
	}
adjust_menu_edit_modes(view);
if(view->menu_species != NULL) {
	int taille, valeur;
	if(view->tot_sel_seqs == 0)
		fl_set_menu_item_mode(view->menu_species, 1, FL_PUP_GRAY);
	else
		fl_set_menu_item_mode(view->menu_species, 1, FL_PUP_NONE);
	fl_set_menu_item_mode(view->menu_species, 2, FL_PUP_GRAY);
	get_menu_taille_valeur(view->menu_species, &taille, &valeur);
	if(valeur != 0) 
		update_menu_taille_valeur(view->menu_species, taille, 0, 2);
	}
}


void deplacer_grp_seqs(SEA_VIEW *view, int target)
{
/* deplacer toutes les seqs selectionnees pour positionner la premiere
d'entre elles en position target */
int *new_rank, *old_rank, old, new_val, numset;
char **aux;
-- target;
new_rank = (int *)malloc(view->tot_seqs * sizeof(int));
old_rank = (int *)malloc(view->tot_seqs * sizeof(int));
aux = (char **)malloc( view->tot_seqs * sizeof(char *) );
if(new_rank == NULL || old_rank == NULL || aux == NULL) out_of_memory();
/* compute old_rank[new_val] = old */
new_val = -1;
/* place first all non selected seqs */
for(old = 0; old < view->tot_seqs; old++) {
	if(!view->sel_seqs[old]) old_rank[++new_val] = old;
	}
/* allocate room for selected seqs */
if(target + view->tot_sel_seqs > view->tot_seqs)
	target = view->tot_seqs - view->tot_sel_seqs;
old = view->tot_seqs - view->tot_sel_seqs - target;
if(old != 0)
	memmove(old_rank + target + view->tot_sel_seqs, old_rank + target,
		old * sizeof(int));
/* insert selected seqs */
for(old = 0; old < view->tot_seqs; old++)
	if(view->sel_seqs[old]) old_rank[target++] = old;
/* compute new_rank[old] = new_val */
for(new_val = 0; new_val < view->tot_seqs; new_val++)
	new_rank[old_rank[new_val]] = new_val;
/* displace all sequence order-dependent ingredients */
/* deplacer la position du curseur */
if(!view->cursor_in_comment) {
	view->cursor_seq = new_rank[view->cursor_seq - 1] + 1;
	view->old_cursor_seq = view->cursor_seq;
	}
/* deplacer les seqs */
for(old = 0; old < view->tot_seqs; old++)
	aux[new_rank[old]] = view->sequence[old];
memcpy(view->sequence, aux, view->tot_seqs * sizeof(char *) );
/* deplacer les noms */
for(old = 0; old < view->tot_seqs; old++)
	aux[new_rank[old]] = view->seqname[old];
memcpy(view->seqname, aux, view->tot_seqs * sizeof(char *) );
/* deplacer les commentaires */
for(old = 0; old < view->tot_seqs; old++)
	aux[new_rank[old]] = view->comments[old];
memcpy(view->comments, aux, view->tot_seqs * sizeof(char *) );
/* deplacer les seqs en couleurs */
if(view->numb_gc > 1) {
	for(old = 0; old < view->tot_seqs; old++)
		aux[new_rank[old]] = (char *) view->col_seq[old];
	memcpy(view->col_seq, aux, view->tot_seqs * sizeof(char *) );
	}
/* deplacer les sequences selectionnees */
for(old = 0; old < view->tot_seqs; old++)
	old_rank[new_rank[old]] = view->sel_seqs[old];
memcpy(view->sel_seqs, old_rank, view->tot_seqs * sizeof(int) );
/* deplacer les longueurs de sequences */
for(old = 0; old < view->tot_seqs; old++)
	old_rank[new_rank[old]] = view->each_length[old];
memcpy(view->each_length, old_rank, view->tot_seqs * sizeof(int) );
/* process species sets */
for(numset = 0; numset < view->numb_species_sets; numset++) {
	for(old = 0; old < view->tot_seqs; old++)
		old_rank[new_rank[old]] = view->list_species_sets[numset][old];
	memcpy(view->list_species_sets[numset], old_rank, 
		view->tot_seqs * sizeof(int) );
	}
free(aux); free(old_rank); free(new_rank);
view->modif_but_not_saved = TRUE;
}


void update_current_seq_length(int newlength, SEA_VIEW *view)
{
double x; int l;
if(newlength > view->seq_length) {
	view->seq_length = 
		( newlength+20 < view->max_seq_length ? 
		newlength+20 : view->max_seq_length );
	l = view->seq_length - view->tot_sites+3;
	if(l<1) l=1;
	fl_set_slider_bounds(view->horsli,1,l);
	x = ( (double) view->tot_sites ) / 
		( view->seq_length + 3 ) ;
	if(x>1) x=1;
	fl_set_slider_size(view->horsli,x);
	}
}


int insert_char_in_seq( int key, int total, SEA_VIEW *view)
/* to insert the typed key in the sequence at cursor location if it is visible
returns # of inserted chars if ok, 0 if error (= cursor not visible or 
max seq size is reached) 
*/
{
char *pos;
int l, c, *plength;
if(view->cursor_in_comment) {
	if( view->cursor_seq < 1 ||
	   view->cursor_seq >= view->tot_comment_lines ||
	   view->cursor_site < view->first_site ||
	   view->cursor_site >= view->first_site + view->tot_sites ) return 0;
	l = view->comment_length[view->cursor_seq - 1];
	}
else	{
	if( view->cursor_seq < view->first_seq ||
	   view->cursor_seq >=view->first_seq+view->tot_lines ||
	   view->cursor_site < view->first_site ||
	   view->cursor_site >= view->first_site + view->tot_sites ) return 0;
	l = view->each_length[view->cursor_seq-1];
	}
if(view->cursor_site > l + 1) return 0;
if( l + total > view->max_seq_length) total = view->max_seq_length - l;
if(total <= 0) return 0;
if(view->cursor_in_comment) 
	pos = view->comment_line[view->cursor_seq - 1] + view->cursor_site - 1;
else
	pos = view->sequence[view->cursor_seq - 1] + view->cursor_site - 1;
memmove(pos+total, pos, l - view->cursor_site + 2);
key = toupper(key);
memset(pos, key, total);
if( (!view->cursor_in_comment) && view->numb_gc > 1) {
	for (c=0; c<view->numb_gc; c++) {
	   pos= &view->col_seq[view->cursor_seq-1][c][view->cursor_site-1];
	   memmove(pos+total, pos, l - view->cursor_site + 2);
	   memset(pos, ' ' ,total);
	   }
	c = (view->protein ? get_color_for_aa(key) : get_color_for_base(key) );
	memset(view->col_seq[view->cursor_seq-1][c] + view->cursor_site - 1, 
		key, total);
	}
if(view->cursor_in_comment) 
	plength = &(view->comment_length[view->cursor_seq-1]);
else
	plength = &(view->each_length[view->cursor_seq-1]);
(*plength) += total;
update_current_seq_length(*plength, view);
view->modif_but_not_saved = TRUE;
return total;
}


/*  The routine that does drawing */
int DNAview_handler(FL_OBJECT * ob, int event, FL_Coord mx, FL_Coord my,
		   int key, void *xev)
{
SEA_VIEW *view;
/* numero courant de la derniere seq selectionnee pendant selection de seqs
par glissement de la souris 
*/
static int selecting_seqs = 0, sel_seq_move = 0;
static int modifying_segment = 0;

if(ob->u_vdata == NULL) return 0;
view = (SEA_VIEW *) ( ( (FL_OBJECT *)ob->u_vdata )->u_vdata );

    switch (event)
    {
    case FL_DBLCLICK:
	{
	int new_seq;
	if(view->double_buffer) {
		mx -= view->DNA_obj->x;
		my -= view->DNA_obj->y;
		}
	new_seq = (my + view->line_height/2 - view->y_seq)/view->line_height + 
		view->first_seq;
	if( new_seq < view->first_seq || new_seq > view->tot_seqs ||
		new_seq >= view->first_seq + view->tot_lines ) break;
	if(mx < view->x_name || mx >= view->x_seq - view->char_width ||
		key != 1) break;
	/* double click sur nom de seq: selection de toutes les seqs */
	if(view->alt_col_seq != NULL) break;
   	if(view->multipl->u_ldata > 0) mod_multipl(view->multipl,0);
	select_deselect_seq(view, -1);
	selecting_seqs = 0;
	view->draw_names = -2;
	fl_redraw_object(ob);
	break;
	}
    case FL_SHORTCUT: /* <Alt>S or <Alt>R */
     if(key == FL_ALT_VAL + 'S' )
	{
	char *err;
	if(view->masename == NULL) break;
	my_watch_cursor(view->dnawin);
	save_active_region(view, TRUE);
	err = save_alignment_or_region(view->masename, view->sequence, 
		view->comments, 
		view->header, view->seqname, view->tot_seqs, view->each_length,
		view->regions, NULL, view->format_for_save,
		view->numb_species_sets, view->list_species_sets,
		view->name_species_sets, NULL, 0, view->protein,
		view->tot_comment_lines, view->comment_name,
		view->comment_line);
	fl_reset_cursor(view->dnawin);
	if(err != NULL) fl_show_alert(err, "", "", TRUE);
	else view->modif_but_not_saved = FALSE;
   	if(view->multipl->u_ldata > 0) mod_multipl(view->multipl,0);
	}
     else if(key == FL_ALT_VAL + 'R' )
	{
	int status, mainpop;
	mainpop = view->bouton_props->u_ldata;
	status = fl_getpup_mode(mainpop, 41) & FL_PUP_CHECK;
	reference_toggle(view, !status);
	}
     break;
    case FL_DRAW:
	return handle_draw(view);

    case FL_PUSH: /* key: 1=bouton gauche, 2=centre, 3=droit de la souris */
	handle_push(view, mx, my, key, &modifying_segment, &selecting_seqs, 
		&sel_seq_move);
	break;
    case FL_MOUSE: /* mouvement avec souris enfoncee */
	handle_mouse(view, mx, my, &selecting_seqs, &sel_seq_move, 
		&modifying_segment);
	break;
     case FL_RELEASE:
     case FL_LEAVE:
	if(selecting_seqs) {
		if(sel_seq_move) {
			select_deselect_seq(view, sel_seq_move);
			view->draw_names = sel_seq_move;
			fl_redraw_object(ob);
			}
		else			
			select_deselect_seq(view, -2);
		selecting_seqs = 0;
		}
	else if(modifying_segment) {
		end_change_segment(view);
		view->draw_names = 0;
		fl_redraw_object(ob);
		modifying_segment = 0;
		}
	break;
     case FL_KEYBOARD:
	handle_keyboard(view, key);
	break;
    }
    return 0;
}


void handle_mouse(SEA_VIEW *view, FL_Coord mx, FL_Coord my, 
	int *p_selecting_seqs, int *p_sel_seq_move, int *p_modifying_segment)
{ /* mouvement avec souris enfoncee */
int debut, fin, step, num, new_seq, new_site;

if(*p_selecting_seqs != 0) {
	if(view->double_buffer) my -= view->DNA_obj->y;
	new_seq = (my + view->line_height/2 - 
		view->y_seq)/view->line_height + view->first_seq;
	if(new_seq == *p_selecting_seqs) return;
	if( new_seq < view->first_seq || new_seq > view->tot_seqs ||
		new_seq >= view->first_seq + view->tot_lines ) return;
	if(!view->sel_seqs[new_seq - 1]) 
		{ debut= new_seq; fin = *p_selecting_seqs; }
	else
		{ debut= *p_selecting_seqs; fin = new_seq; }
	if(debut < fin) step = 1;
	else	step = -1;
	*p_selecting_seqs = new_seq;
	for(num = debut; num != fin; num += step) {
		new_seq = debut + fin - step - num - 1;
		if(view->sel_seqs[new_seq]) {
			view->sel_seqs[new_seq] = FALSE;
			--(view->tot_sel_seqs);
			}
		else	{
			view->sel_seqs[new_seq] = TRUE;
			++(view->tot_sel_seqs);
			}
		if(*p_sel_seq_move == new_seq + 1) *p_sel_seq_move = 0;
		}
	if(*p_sel_seq_move) {
		if( view->sel_seqs[*p_sel_seq_move - 1] ) {
			view->sel_seqs[*p_sel_seq_move - 1] = FALSE;
			--(view->tot_sel_seqs);
			}
		else	{
			view->sel_seqs[*p_sel_seq_move - 1] = TRUE;
			++(view->tot_sel_seqs);
			}
		*p_sel_seq_move = 0;
		}
	view->draw_names = -2;
	fl_redraw_object(view->DNA_obj);
	}
else if(*p_modifying_segment != 0) {
	if(view->double_buffer) mx -= view->DNA_obj->x;
	new_site = (mx - view->x_seq )/view->char_width + 
		view->first_site;
	if(new_site == *p_modifying_segment) return;
	if( new_site < view->first_site || 
		new_site > view->first_site + view->tot_sites ||
		new_site > view->region_length ) return;
	if( continue_change_segment(view, new_site) ) {
		*p_modifying_segment = new_site;
		view->draw_names = 0;
		view->mod_region_line = TRUE;
		fl_redraw_object(view->DNA_obj);
		}
	}
}


int handle_draw(SEA_VIEW *view)
{
int newsize;

/* returns TRUE if window size was changed by user */
newsize = compute_size_params( (FL_OBJECT *) view->DNA_obj->u_vdata, FALSE ); 
if(view->draw_names) { /* soit tous (<= -1) soit un seul ( >= 1) */
	draw_seq_names(view->DNA_obj, view);
	if(view->draw_names > 0 || view->draw_names == -2){
		/* si > 0 ou -2, ne pas ecrire les seqs*/
		view->draw_names = -1;
		return 0;
		}
	}
if(view->mod_cursor) {
	/* effacer old_cursor en ecrivant dessus */
	draw_cursor(view->DNA_obj, FALSE, view->old_cursor_site, 
		view->old_cursor_seq, view->old_cursor_in_comment);
	view->mod_cursor = FALSE;
	}
else if(view->mod_region_line) {
	draw_region_line(view->DNA_obj, view);
	view->mod_region_line = FALSE;
	}
else if(view->mod_comment_line) {
	draw_comment_lines(view->DNA_obj, view);
	view->mod_comment_line = FALSE;
	}
else	{
	if(view->inverted_colors)
		    draw_dna_seqs_inverted(view->DNA_obj, view);
	else
		    draw_dna_seqs(view->DNA_obj, view);
	draw_comment_lines(view->DNA_obj, view);
	}
view->mod_seq = 0;
view->draw_names = -1;
draw_cursor(view->DNA_obj, TRUE , view->cursor_site, view->cursor_seq, 
	view->cursor_in_comment);
view->old_cursor_seq = view->cursor_seq;
view->old_cursor_site = view->cursor_site;
view->old_cursor_in_comment = view->cursor_in_comment;
return newsize; /* to call this object's callback */
}


void handle_push(SEA_VIEW *view, FL_Coord mx, FL_Coord my, int key, 
	int *p_modifying_segment, int *p_selecting_seqs, int *p_sel_seq_move)
/* key: 1=bouton gauche, 2=centre, 3=droit de la souris */
{
int new_site, new_seq, new_line;

if(view->multipl->u_ldata > 0) mod_multipl(view->multipl,0);
if(view->double_buffer) {
	mx -= view->DNA_obj->x;
	my -= view->DNA_obj->y;
	}
new_seq = (my + view->line_height/2 - view->y_seq)/view->line_height + 
	view->first_seq;
new_line = new_seq - view->first_seq + 1;
new_site = (mx - view->x_seq )/view->char_width + 
	view->first_site;
if(view->active_region != NULL && 
  new_seq == view->first_seq + FL_min(view->tot_lines,view->tot_seqs) &&
	new_site >= view->first_site && 
	new_site < view->first_site + view->tot_sites && 
	new_site <= view->region_length ) {
/* work with segments: extend, or create, or delete */
	if(key == 2) { /* middle button:extend left neighbor segment */
		new_seq = extend_segment_at_left(view, new_site);
		if(new_seq) fl_ringbell(0);
		else	{
			view->draw_names = 0;
			fl_redraw_object(view->DNA_obj);
			}
		}
	else if(key == 3) { /* right button=>delete segment */
		new_seq = suppr_segment(view->active_region, new_site,
			view->region_line);
		if(new_seq) fl_ringbell(0);
		else	{
			view->draw_names = 0;
			fl_redraw_object(view->DNA_obj);
			}
		}
	else	{ /* left button=>extend or create segment */
		new_seq = begin_change_segment(view, new_site);
		if(new_seq) {
			view->mod_region_line = TRUE;
			*p_modifying_segment = new_site;
			}
		view->draw_names = 0;
		fl_redraw_object(view->DNA_obj);
		}
	return;
	}
if( new_line >= view->pos_first_comment_line &&
		new_line < view->pos_first_comment_line + 
		view->tot_comment_lines ) {
/* dans les comment lines */
	int num, old;
	if(key != 1) return;
	num = new_line - view->pos_first_comment_line + 1;
	if(mx >= view->x_name && mx < view->x_seq - view->char_width) {
	/* click sur nom de comment line: selection/deselection */
		old = view->active_comment_line;
		if( old != 0 ) {
			view->draw_names = 0;
			view->active_comment_line = 0;
			view->mod_comment_line = old;
			fl_redraw_object(view->DNA_obj);
			}
		if(old == num)
			view->active_comment_line = 0;
		else
			view->active_comment_line = num;
		view->draw_names = 0;
		view->mod_comment_line = num;
		update_menu_footers(view);
		fl_redraw_object(view->DNA_obj);
		}
	else if( new_site >= view->first_site && 
		new_site < view->first_site + view->tot_sites &&
		new_site <= view->comment_length[num - 1] + 1 ) {
		/* click sur comment: positionnement du curseur */
		view->cursor_site = new_site;
		view->cursor_seq = num;
		view->draw_names = 0;
		view->mod_cursor = TRUE;
		view->cursor_in_comment = TRUE;
		fl_redraw_object(view->DNA_obj);
		}
	return;
	}
	
if( new_seq < view->first_seq || new_seq > view->tot_seqs ||
	new_seq >= view->first_seq + view->tot_lines ) return;
if(mx >= view->x_name && mx < view->x_seq - view->char_width) {
/* click sur nom de seq: selection/deselection */
	if(view->alt_col_seq != NULL) return;
	if(key == 1) {
		*p_selecting_seqs = new_seq;
		*p_sel_seq_move = new_seq;
		return;
		}
	else if(key == 3) {
		select_deselect_seq(view, 0);
		view->draw_names = -2;
		}
	else	{ /* milieu: depl des seqs selectionnees */
		if(view->tot_sel_seqs == 0 || 
			view->sel_seqs[new_seq - 1]) 
			{ fl_ringbell(0); return; }
		deplacer_grp_seqs(view, new_seq);
		}
	fl_redraw_object(view->DNA_obj);
	return;
	}
if(key != 1)  return;
/* click sur seq: positionnement du curseur */
if( new_site >= view->first_site && 
	new_site < view->first_site + view->tot_sites &&
	new_site <= view->each_length[new_seq-1] + 1 ) {
	view->cursor_site = new_site;
	view->cursor_seq = new_seq;
	view->draw_names = 0;
	view->mod_cursor = TRUE;
	view->cursor_in_comment = FALSE;
	fl_redraw_object(view->DNA_obj);
	}
return;
}


void handle_keyboard(SEA_VIEW *view, int key)
{
int new_pos, multipl, num;

multipl = view->multipl->u_ldata;
if(multipl == 0) multipl = 1;
if(key == XK_Right) { /* right arrow */		
	new_pos = view->cursor_site + multipl;
    if(view->cursor_in_comment) {
	if(new_pos > view->comment_length[view->cursor_seq-1]+1)
	   new_pos = view->comment_length[view->cursor_seq-1]+1;
	}
    else {
	if(new_pos > view->each_length[view->cursor_seq-1] + 1) 
	   new_pos = view->each_length[view->cursor_seq-1] + 1;
	}
    set_and_show_new_cursor_site(view, new_pos,FALSE,FALSE);
    }
else if(key == XK_Left) { /* left arrow */		
	new_pos = FL_max(1, view->cursor_site - multipl);
	set_and_show_new_cursor_site(view, new_pos,FALSE,FALSE);
	}
else if(key == XK_Up) { /* up arrow */
	new_pos = FL_max(1, view->cursor_seq - multipl);
	set_and_show_new_cursor_seq(view, new_pos);
	}
else if(key == XK_Down){ /* down arrow */
	new_pos = view->cursor_seq + multipl;
	if(view->cursor_in_comment) {
		if(new_pos > view->tot_comment_lines) 
			new_pos = view->tot_comment_lines;
		}
	else	{
		if(new_pos > view->tot_seqs) 
			new_pos = view->tot_seqs;
		}
	set_and_show_new_cursor_seq(view, new_pos);
	}
else if(view->cursor_in_comment && 
	view->active_comment_line == view->cursor_seq) {
	if(view->alt_col_seq != NULL) return;
	if( isprint(key) ) 
		insert_char_in_comment(key, 1, view);
	else if( key == 0x7f || key == 0x8) /* del or BS */
		delete_char_in_comment(view, 1, 
			view->active_comment_line, 
			view->cursor_site, FALSE);
	else
		return;
	}
else if(key=='>' || key=='<' || key=='[' || key==']') {
	int oldpos;
	int upper_step=50, bracket_step=5 ;
	double max_w, min_w;
	fl_get_slider_bounds(view->horsli, &min_w, &max_w);
	oldpos=fl_get_slider_value(view->horsli);
	new_pos = oldpos;
	upper_step *= multipl;
	bracket_step *= multipl;
	if(key=='>') {
		new_pos=oldpos+upper_step;
		if(new_pos>max_w) new_pos=max_w;
		}
	else if(key=='<') {
		new_pos=oldpos-upper_step;
		if(new_pos<1) new_pos=1;
		}
	else if(key==']') {
		new_pos=oldpos+bracket_step;
		if(new_pos>max_w) new_pos=max_w;
		}
	else if(key=='[') {
		new_pos=oldpos-bracket_step;
		if(new_pos<1) new_pos=1;
		}
	if(new_pos!=oldpos) {
		fl_set_slider_value(view->horsli,new_pos);
		view->draw_names = 0;
		view->first_site = new_pos;
		fl_redraw_object(view->DNA_obj);
		}
	}
else if(key == 0x7f || key == 0x8 ) { /* delete or backspace */
	int count, count_each, debut, fin, test;
	    if(view->multipl->u_ldata > 0)
		mod_multipl(view->multipl,0);
	if(view->cursor_in_comment)
		test = view->cursor_seq < 1 ||
	   		view->cursor_seq > view->tot_comment_lines;
	else
		test = view->cursor_seq < view->first_seq ||
	   		view->cursor_seq >= view->first_seq+view->tot_lines;
	if( test ||
	   view->cursor_site < view->first_site ||
	   view->cursor_site >=view->first_site+view->tot_sites)
		{ fl_ringbell(0); return; }
	if(view->cursor_in_comment) {
		if( delete_char_in_comment(view, multipl, 
			view->cursor_seq, view->cursor_site,
			TRUE) != multipl) fl_ringbell(0);
		return;
		}
	if(view->alt_col_seq != NULL) return;
	if(view->tot_sel_seqs > 1 &&
		view->sel_seqs[view->cursor_seq - 1])
		{ debut = 1; fin = view->tot_seqs; test = TRUE;}
	else
		{ debut = fin = view->cursor_seq; test = FALSE;}
	for(num = debut; num<=fin; num++) 
		{
		if(test && !view->sel_seqs[num-1]) continue;
		count_each = delete_gaps_before(view,
			num, 
			view->cursor_site, multipl);
		if(count_each < multipl) fl_ringbell(0);
		if(num == view->cursor_seq) count = count_each;
		}
	/* si ttes seqs selectionnees, traiter aussi regions et comments */
	if(count_each == multipl && 
		(!view->cursor_in_comment) &&
		view->tot_sel_seqs == view->tot_seqs ) {
		if(view->regions != NULL)
			delete_region_part(view, 
				view->cursor_site,multipl);
		if(view->tot_comment_lines > 0)
			delete_in_all_comments(multipl,
				view->cursor_site, view);
		}
	new_pos = view->cursor_site - count;
	if(new_pos <= 0) new_pos = 1;
	if(view->cursor_in_comment) {
		view->mod_comment_line = view->cursor_seq;
		}
	else	{
		if(view->tot_sel_seqs > 1 && 
			view->tot_sel_seqs != view->tot_seqs &&
			view->sel_seqs[view->cursor_seq - 1])
			view->mod_seq = -1;
		else if(view->tot_sel_seqs <= 1 || 
			!view->sel_seqs[view->cursor_seq - 1])
			view->mod_seq = view->cursor_seq;
		}
	set_and_show_new_cursor_site(view, new_pos, 
		FALSE,TRUE);
	}
else if(key == '_' ) { /* del gap in all but current seq(s) */
	int count_each;
	    if(view->multipl->u_ldata > 0)
		mod_multipl(view->multipl,0);
	if(view->cursor_in_comment) return;
	if( view->cursor_seq < view->first_seq ||
	   	view->cursor_seq >=view->first_seq+view->tot_lines ||
	  	view->cursor_site < view->first_site ||
	   	view->cursor_site >= 
	   	view->first_site + view->tot_sites ||
	   	view->tot_sel_seqs == view->tot_seqs )
		 { fl_ringbell(0); return; }
	if(view->alt_col_seq != NULL) return;
	for( num = 1; num <= view->tot_seqs; num++) {
		if(num == view->cursor_seq || 
			(view->sel_seqs[view->cursor_seq-1] &&
			view->sel_seqs[num-1] ) ) continue;
		count_each = delete_gaps_before(view, 
		    	num, view->cursor_site, multipl);
		if(count_each < multipl) {
			fl_ringbell(0);
			return;
			}
		}
	if(count_each == multipl && view->regions != NULL)
		delete_region_part(view, view->cursor_site, multipl);
	if(count_each == multipl && view->tot_comment_lines > 0)
		delete_in_all_comments(multipl, view->cursor_site, view);
	new_pos = view->cursor_site - multipl;
	if(new_pos <= 0) new_pos = 1;
	set_and_show_new_cursor_site(view, new_pos, FALSE, TRUE);
	}
else if( key == '-' || (key == ' ' && !view->hjkl)
			  /* gap key = - or space */
	|| key == '+' ) { /* insert gap in other seqs key */
	int newlength = 0, count = 0, count_each, debut, fin, test;
	    if(view->multipl->u_ldata > 0)
		mod_multipl(view->multipl,0);
	if(view->cursor_in_comment && key == '+') return;
	if(view->cursor_in_comment)
		test = FALSE;
	else
		test = view->cursor_seq < view->first_seq ||
	   		view->cursor_seq >=view->first_seq+view->tot_lines;
	if( test || view->cursor_site < view->first_site ||
	   	view->cursor_site >= 
		view->first_site + view->tot_sites )
		 { fl_ringbell(0); return; }
	if(view->alt_col_seq != NULL) return;
	if(key != '+') { /* gap key */
	    if(view->tot_sel_seqs > 1 && 
		(!view->cursor_in_comment) &&
		view->sel_seqs[view->cursor_seq - 1])
		{ debut = 1; fin = view->tot_seqs; test = TRUE;}
	    else
		{ debut = fin = view->cursor_seq; test = FALSE;}
	    for(num = debut; num<=fin; num++) 
		{
		if(test && !view->sel_seqs[num-1]) continue;
		count_each = insert_gaps_at(view, num, 
			view->cursor_site, multipl);
		if(count_each < multipl) fl_ringbell(0);
		if(num == view->cursor_seq) count = count_each;
		if(view->cursor_in_comment) {
		   if(newlength < view->comment_length[num-1])
			newlength = view->comment_length[num-1];
		     }
		else {
		   if(newlength < view->each_length[num-1])
			   newlength = view->each_length[num-1];
		     }
		}
/* si ttes seqs selectionnees, traiter aussi regions et comments */
	    if(count_each == multipl && 
		(!view->cursor_in_comment) &&
		view->tot_sel_seqs == view->tot_seqs) {
		if(view->regions != NULL)
			insert_region_part(view, view->cursor_site, multipl);
		if(view->tot_comment_lines > 0)
			insert_gap_all_comments(multipl,view->cursor_site, 
				view);
		}
	    }
	else	{ /* + ==> gap in other sequences */
		if(view->tot_sel_seqs == view->tot_seqs) {
			fl_ringbell(0); return;
			}
		for( num = 1; num <= view->tot_seqs; num++) {
			if(num == view->cursor_seq || 
			     (view->sel_seqs[view->cursor_seq-1] &&
			      view->sel_seqs[num-1] ) ) continue;
			count_each = insert_gaps_at(view, 
			    num, view->cursor_site, multipl);
			if(count_each < multipl) {
				fl_ringbell(0); return;
				}
			if(newlength < view->each_length[num-1])
			   	newlength = view->each_length[num-1];
			}
		count = multipl;
		if(count_each == multipl && 
			view->regions != NULL)
			insert_region_part(view, view->cursor_site, multipl);
		if(count_each == multipl && 
			view->tot_comment_lines > 0) {
			insert_gap_all_comments(multipl,view->cursor_site, 
				view);
			   }
		}
	new_pos = view->cursor_site + count;
	if(view->cursor_in_comment) {
	 	if(new_pos> view->comment_length[view->cursor_seq-1]+1)
	    		new_pos= view->comment_length[view->cursor_seq-1]+1;
	 	}
	else 	{
	 	if(new_pos > view->each_length[view->cursor_seq-1] + 1)
	   		 new_pos = view->each_length[view->cursor_seq-1] + 1;
	 	}
	if(view->cursor_in_comment)
		view->mod_comment_line = view->cursor_seq;
	else if(key != '+' ) {
		if(view->tot_sel_seqs > 1 && 
			view->tot_sel_seqs != view->tot_seqs &&
			view->sel_seqs[view->cursor_seq - 1])
			view->mod_seq = -1;
		else if(view->tot_sel_seqs <= 1 ||
			!view->sel_seqs[view->cursor_seq - 1] )
			view->mod_seq = view->cursor_seq;
		}
	update_current_seq_length(newlength, view);
	set_and_show_new_cursor_site(view, new_pos, FALSE,TRUE);
	}
else if( key >= '0' && key <= '9' ) { /* multiplicateur */
	multipl = view->multipl->u_ldata * 10;
	multipl += (key - '0');
	mod_multipl(view->multipl, multipl);
	return;
	}
else if( view->allow_seq_edit && (view->alt_col_seq == NULL) &&
	(!view->cursor_in_comment) &&
	( (key>='a' && key<='z') || (key>='A' && key<='Z') || key == ' ' ) ) {
	if(view->hjkl) {
		static char typedkey[]= "hjklHJKL ";
		static char dnaequivs[3][10]={
			"GATCGATCN", "TCGATCAGN", "ACGTACGTN"};
		char *p;
		p = strchr(typedkey, key);
		if(p != NULL) 
			key = *( dnaequivs[view->hjkl - 1] + (p - typedkey) );
		}
	if(key == ' ') num = 0;
	else	num = insert_char_in_seq(key, multipl, view);
	if( num == 0 ) fl_ringbell(0);
	else 	{
		view->mod_seq = view->cursor_seq;
		set_and_show_new_cursor_site(view, 
			view->cursor_site + num, FALSE, TRUE);
		}
	}
if(view->multipl->u_ldata > 0) mod_multipl(view->multipl, 0);
}


FL_OBJECT *create_dna_scroller(int x, int y, int w, int h, int double_buffer)
{
  FL_OBJECT *dna_group, *obj;
  int wmultipl, x_pos;
  SEA_VIEW *myscroller = (SEA_VIEW *)calloc(1,sizeof(SEA_VIEW));
  if(myscroller == NULL) out_of_memory();
  dna_group = fl_bgn_group();
/* fleche bas */
  myscroller->down_arrow = 
	obj = fl_add_button(FL_TOUCH_BUTTON,x+5,y+h-3-20,20,20,"@2->");
  obj->u_vdata = (void *)dna_group;
  fl_set_object_callback(obj,lrdu_button_callback,6);
  fl_set_object_resize(obj,FL_RESIZE_NONE);
  fl_set_object_gravity(obj,FL_SouthWest,FL_SouthWest);
/* screen move bas */
  myscroller->down_screen_move = 
	obj = fl_add_button(FL_TOUCH_BUTTON,x+5,y+h-3-22-20,20,20,"@2>>");
  obj->u_vdata = (void *)dna_group;
  fl_set_object_callback(obj,lrdu_button_callback,7);
  fl_set_object_resize(obj,FL_RESIZE_NONE);
  fl_set_object_gravity(obj,FL_SouthWest,FL_SouthWest);
/* ascenc. vertical */
  myscroller->vertsli = obj = fl_add_slider(
			FL_VERT_SLIDER,x+5,y+26+22+22,20,h-118,"");
  obj->u_vdata = (void *)dna_group;
  fl_set_object_callback(obj,vh_sliders_callback,0);
  fl_set_object_resize(obj,FL_RESIZE_Y);
  fl_set_object_gravity(obj,FL_NorthWest,FL_SouthWest);
  fl_set_slider_bounds(obj,1,1);
  fl_set_slider_size(obj,1);
  fl_set_slider_value(obj,1);
  fl_set_slider_step(obj,1);
  fl_set_slider_return(obj,FL_RETURN_END_CHANGED);
/* screen move haut */
  myscroller->up_screen_move = 
	obj = fl_add_button(FL_TOUCH_BUTTON,x+5,y+26+22,20,20,"@8>>");
  obj->u_vdata = (void *)dna_group;
  fl_set_object_callback(obj,lrdu_button_callback,5);
  fl_set_object_resize(obj,FL_RESIZE_NONE);
  fl_set_object_gravity(obj,FL_NorthWest,FL_NorthWest);
/* fleche haut */
  myscroller->up_arrow = 
	obj = fl_add_button(FL_TOUCH_BUTTON,x+5,y+26,20,20,"@8->");
  obj->u_vdata = (void *)dna_group;
  fl_set_object_callback(obj,lrdu_button_callback,4);
  fl_set_object_resize(obj,FL_RESIZE_NONE);
  fl_set_object_gravity(obj,FL_NorthWest,FL_NorthWest);
/* valeur du multiplicateur */
  wmultipl = fl_get_string_width(FL_NORMAL_STYLE, FL_NORMAL_SIZE,"mult=9999",9);
  x_pos = 5;
  myscroller->multipl = obj = fl_add_box(FL_FLAT_BOX,
	x+x_pos, y, wmultipl, 20, "");
  x_pos += wmultipl + 5;
  fl_set_object_lstyle(obj, FL_NORMAL_STYLE);
  fl_set_object_lsize(obj,FL_NORMAL_SIZE);
  fl_set_object_lalign(obj,FL_ALIGN_CENTER);
  fl_set_object_resize(obj,FL_RESIZE_NONE);
  fl_set_object_gravity(obj,FL_NorthWest,FL_NorthWest);
  obj->u_vdata = (void *)dna_group;
/* fleche gauche */
  myscroller->left_arrow = 
	obj = fl_add_button(FL_TOUCH_BUTTON,x+x_pos,y,20,20,"@<-");
  x_pos += 22;
  obj->u_vdata = (void *)dna_group;
  fl_set_object_callback(obj,lrdu_button_callback,0);
  fl_set_object_resize(obj,FL_RESIZE_NONE);
  fl_set_object_gravity(obj,FL_NorthWest,FL_NorthWest);
/* screen move gauche */
  myscroller->left_screen_move = 
	obj = fl_add_button(FL_TOUCH_BUTTON,x+x_pos,y,20,20,"@<<");
  x_pos += 22;
  obj->u_vdata = (void *)dna_group;
  fl_set_object_callback(obj,lrdu_button_callback,1);
  fl_set_object_resize(obj,FL_RESIZE_NONE);
  fl_set_object_gravity(obj,FL_NorthWest,FL_NorthWest);
/* ascens. horizontal */  
  myscroller->horsli = obj = fl_add_slider(
			FL_HOR_SLIDER,x+x_pos,y,w-x_pos-2*24,20,"");
  obj->u_vdata = (void *)dna_group;
  fl_set_object_resize(obj,FL_RESIZE_NONE);
  fl_set_object_gravity(obj,FL_NorthWest,FL_NorthEast);
  fl_set_object_callback(obj,vh_sliders_callback,1);
  fl_set_slider_bounds(obj,1,1);
  fl_set_slider_size(obj,1);
  fl_set_slider_value(obj,1);
  fl_set_slider_step(obj,1);
  fl_set_slider_return(obj,FL_RETURN_END_CHANGED);
/* screen move a droite */
  myscroller->right_screen_move = 
	obj = fl_add_button(FL_TOUCH_BUTTON,x+w - 23 - 22,y,20,20,"@>>");
  obj->u_vdata = (void *)dna_group;
  fl_set_object_callback(obj,lrdu_button_callback,3);
  fl_set_object_resize(obj,FL_RESIZE_NONE);
  fl_set_object_gravity(obj,FL_NorthEast,FL_NorthEast);
/* fleche droite */  
  myscroller->right_arrow = 
	obj = fl_add_button(FL_TOUCH_BUTTON,x+w - 23,y,20,20,"@->");
  obj->u_vdata = (void *)dna_group;
  fl_set_object_callback(obj,lrdu_button_callback,2);
  fl_set_object_resize(obj,FL_RESIZE_NONE);
  fl_set_object_gravity(obj,FL_NorthEast,FL_NorthEast);

/* cadre pour noms + sequences */  
  obj = fl_add_frame(FL_DOWN_FRAME,
	x+30,y -2+30,w - 30 - FL_BOUND_WIDTH,h - 35,"");
  fl_set_object_resize(obj,FL_RESIZE_ALL);
  fl_set_object_gravity(obj,FL_NorthWest,FL_SouthEast);
/* noms + sequences */  
  myscroller->DNA_obj = obj = fl_add_free(FL_INPUT_FREE,
 	x+30,y-2+30,w - 30 - FL_BOUND_WIDTH,h - 35,"",  DNAview_handler);
  obj->click_timeout = FL_CLICK_TIMEOUT; /* allow for double click events */
  obj->u_vdata = (void *)dna_group;
  fl_set_object_boxtype(obj,FL_FLAT_BOX);
  fl_set_object_lstyle(obj,FL_FIXEDBOLD_STYLE);
  fl_set_object_resize(obj,FL_RESIZE_ALL);
  fl_set_object_gravity(obj,FL_NorthWest,FL_SouthEast);
  fl_set_object_shortcut(obj,"#s#r",FALSE);
  fl_set_object_callback(obj, viewer_callback, 0);
/* 
attention: change les coord d'ecriture dans l'objet, marge n'est plus comptee 
*/
  myscroller->double_buffer = double_buffer;
  if(double_buffer) 
  	fl_set_object_dblbuffer(obj, TRUE); 

  fl_end_group();
  dna_group->u_vdata = (void *)myscroller;
  myscroller->menu_regions = myscroller->menu_file = myscroller->menu_edit = 
		myscroller->bouton_props = myscroller->menu_species = NULL;
  return dna_group;
}


FL_FORM *create_the_form(char **seq, char **seqname, 
		char **comments, FL_OBJECT **the_scroller, int double_buffer,
		int reducefonts, int inverted,
		known_format default_format, int numb_dnacolors,
		FL_COLOR *dnacolors, 
		int numb_stdprotcolors, FL_COLOR *stdprotcolors,
		int numb_altprotcolors, FL_COLOR *altprotcolors,
		color_choice curr_color_choice, char *progname)
{
/* a cause bug dans double-buffering, il est important que l'objet
dnascroller soit le dernier cree dans la form!!! */
FL_FORM *my_form;
FL_OBJECT *obj, *bouton_search, *champ_search, *bouton_goto, *champ_goto,
	*menu_file, *bouton_props, *menu_regions, *menu_edit, *menu_species,
	*menu_footers, *bouton_help;
int menuid, menu_width, fin_menu, seq_char_size;
SEA_VIEW *view;
/* pour trouver les valeurs courantes de police boutons et larg bordures */
static int buttonFontSize, menuFontSize, borderWidth;
static FL_resource res[] = 
{
/* nom resource, classe (semble inutile), type, addresse ou mettre valeur,
	valeur par defaut, pour type FL_STRING: taille du champ pour valeur */
	{"menuFontSize", "", FL_INT, &menuFontSize, "10" },
	{"buttonFontSize", "", FL_INT, &buttonFontSize, "10" },
	{"borderWidth", "", FL_INT, &borderWidth, "3" }
};
fl_get_app_resources( res, sizeof(res) / sizeof(FL_resource) );
/* calcul larg necessaire pour bouton/menu avec le plus large nom */
buttonFontSize = FL_max(buttonFontSize, menuFontSize);
menu_width = fl_get_string_width(
	FL_BOLD_STYLE, buttonFontSize, "Species", 7) + 4 * borderWidth;

my_form = fl_bgn_form(FL_FLAT_BOX,1000,700);
/* menu File */
menu_file = obj = fl_add_menu(FL_PULLDOWN_MENU,5,5,menu_width,25,"File");
fl_set_menu(obj,
"Open Mase|Open Phylip|Open Clustal|Open MSF|Open Fasta%l|"
"<Alt>S Save|Save as...|Save current sites%l|Prepare printout%l|"
"Quit");
fl_set_menu_item_mode(obj,SAVE,FL_PUP_GRAY);
fl_set_menu_item_mode(obj,SAVE_AS,FL_PUP_GRAY);
fl_set_menu_item_mode(obj,SAVE_REGIONS,FL_PUP_GRAY);
fl_set_object_callback(obj,file_menu_callback, 0);
fl_set_object_boxtype(obj,FL_UP_BOX);
fl_set_object_resize(obj,FL_RESIZE_NONE);
fl_set_object_gravity(obj,FL_NorthWest,FL_NorthWest);
fin_menu = 5 + menu_width + 5;

/* menu Props */
bouton_props = obj = fl_add_button(FL_MENU_BUTTON, fin_menu, 5,
	 menu_width + 5, 25, "Props");
fl_set_object_callback(obj, props_button_callback, 0);
fl_set_object_resize(obj,FL_RESIZE_NONE);
fl_set_object_gravity(obj,FL_NorthWest,FL_NorthWest);
fl_set_object_lalign(obj,FL_ALIGN_INSIDE | FL_ALIGN_LEFT);
fl_set_object_lstyle(obj,FL_BOLD_STYLE);
fin_menu += menu_width + 5 + 5;

/* menu regions */
menu_regions = obj = fl_add_menu(FL_PULLDOWN_MENU,fin_menu, 5,menu_width,25,
		"Sites");
fl_set_object_callback(obj,regions_menu_callback, 0);
fl_set_object_boxtype(obj,FL_UP_BOX);
fl_set_object_resize(obj,FL_RESIZE_NONE);
fl_set_object_gravity(obj,FL_NorthWest,FL_NorthWest);
fin_menu += menu_width + 5;

/* menu species */
menu_species = obj = fl_add_menu(FL_PULLDOWN_MENU,fin_menu, 5,menu_width,25,
		"Species");
fl_set_object_callback(obj,species_menu_callback, 0);
fl_set_object_boxtype(obj,FL_UP_BOX);
fl_set_object_resize(obj,FL_RESIZE_NONE);
fl_set_object_gravity(obj,FL_NorthWest,FL_NorthWest);
fin_menu += menu_width + 5;

/* menu footers */
menu_footers = obj = fl_add_menu(FL_PULLDOWN_MENU,fin_menu, 5,menu_width,25,
		"Footers");
fl_set_menu(obj, "Show footers|Create footer|Delete footer");
fl_set_menu_item_mode(obj, SHOW_HIDE_FOOTERS, FL_PUP_GRAY);
fl_set_object_callback(obj, footers_menu_callback, 0);
fl_set_object_boxtype(obj, FL_UP_BOX);
fl_set_object_resize(obj, FL_RESIZE_NONE);
fl_set_object_gravity(obj, FL_NorthWest, FL_NorthWest);
fin_menu += menu_width + 5;

/* bouton search + champ Search */
bouton_search = obj = fl_add_button(FL_NORMAL_BUTTON, fin_menu , 5, 
		menu_width, 25, "Search:");
fl_set_object_callback(obj, search_callback, 0);
fl_set_object_resize(obj,FL_RESIZE_NONE);
fl_set_object_gravity(obj,FL_NorthWest,FL_NorthWest);
fin_menu += menu_width + 5;
champ_search = obj = fl_add_input(FL_NORMAL_INPUT,fin_menu, 5, 80, 25, "");
#ifdef sun /*par securite car ne marche pas sur dec-alpha */
/*
fl_set_object_lsize(obj,FL_MEDIUM_SIZE);
*/
#endif
fl_set_object_callback(obj, search_callback, 1);
bouton_search->u_vdata = (void *)( champ_search ) ;
fl_set_object_resize(obj,FL_RESIZE_NONE);
fl_set_object_gravity(obj,FL_NorthWest,FL_NorthWest);
fin_menu += 80 + 5;

/*  bouton + champ Goto */
bouton_goto = obj = fl_add_button(FL_NORMAL_BUTTON, fin_menu , 5, 
		menu_width, 25, "Goto:");
fl_set_object_callback(obj, goto_callback, 0);
fl_set_object_resize(obj,FL_RESIZE_NONE);
fl_set_object_gravity(obj,FL_NorthWest,FL_NorthWest);
fin_menu += menu_width + 5;
champ_goto = obj = fl_add_input(FL_NORMAL_INPUT,fin_menu, 5, 80, 25, "");
bouton_goto->u_vdata = (void *)( champ_goto ) ;
fl_set_object_callback(obj, goto_callback, 1);
fl_set_input_return(obj, FL_RETURN_END);
fl_set_object_resize(obj,FL_RESIZE_NONE);
fl_set_object_gravity(obj,FL_NorthWest,FL_NorthWest);
fin_menu += 80 + 5;

/* menu Edit */
menu_edit = obj = fl_add_menu(FL_PULLDOWN_MENU, fin_menu, 5,
		menu_width, 25, "Edit");
fl_set_menu(obj,
   "Rename sequence|Edit comments|Delete sequence(s)|Create sequence|"
   "Load sequence|Duplicate sequence|"
   "Complement sequence|Reverse sequence|Exchange Us and Ts|Align sites|"
   "Dot plot|Consensus sequence|Del. gap-only sites");
fl_set_object_callback(obj, edit_menu_callback, 0);
fl_set_object_boxtype(obj,FL_UP_BOX);
fl_set_object_resize(obj,FL_RESIZE_NONE);
fl_set_object_gravity(obj,FL_NorthWest,FL_NorthWest);
fin_menu += menu_width + 5;

/* bouton help */
bouton_help = obj = fl_add_button(FL_NORMAL_BUTTON, fin_menu , 5, menu_width , 
	25, "Help");
fl_set_object_callback(obj, help_callback, 0);
fl_set_object_resize(obj,FL_RESIZE_NONE);
fl_set_object_gravity(obj,FL_NorthWest,FL_NorthWest);
fin_menu += menu_width + 5;

/* label aide-memoire */
	{ char label[] = "><][-+_";
	int w;
	w = fl_get_string_width(bouton_help->lstyle, bouton_help->lsize, 
		label, strlen(label)) + 8;
	obj = fl_add_text(FL_NORMAL_TEXT, 
		my_form->w - w, 5, w, bouton_help->h, label);
	fl_set_object_lsize(obj, bouton_help->lsize);
	fl_set_object_lstyle(obj, bouton_help->lstyle);
	fl_set_object_resize(obj,FL_RESIZE_NONE);
	fl_set_object_gravity(obj,FL_NorthEast,FL_NorthEast);
	}

/* tout le groupe scroller */
*the_scroller = create_dna_scroller(0, 35, 1000, 665, double_buffer);
view = (SEA_VIEW *)((*the_scroller)->u_vdata);
fl_end_form();
fl_set_form_atclose(my_form, mainwin_close_callback, view);

/* initialisation independantes des sequences */
view->menu_regions = menu_regions;
view->menu_species = menu_species;
view->menu_file = menu_file;
view->menu_edit = menu_edit;
view->menu_footers = menu_footers;
view->bouton_props = bouton_props;
menu_file->u_vdata = (void *)( *the_scroller );
bouton_props->u_vdata = (void *)( *the_scroller ) ;
menu_regions->u_vdata = (void *)( *the_scroller ) ;
menu_species->u_vdata = (void *)( *the_scroller ) ;
menu_edit->u_vdata = (void *)( *the_scroller ) ;
menu_footers->u_vdata = (void *)( *the_scroller ) ;
champ_search->u_vdata = (void *)( *the_scroller ) ;
champ_goto->u_vdata = (void *)( *the_scroller ) ;
view->format_for_save = default_format; /* default format for saving */
view->reducefonts = reducefonts;
view->tot_seqs = 0;
view->first_seq = 1; view->tot_sites = 1;
view->numb_gc = 1;
view->line_height = 1;
view->char_width = 1;
view->draw_names = -1;
view->mod_seq = 0;
view->mod_comment_line = 0;
view->consensus_threshold = 60;

fl_show_form(my_form, FL_PLACE_CENTER | FL_FREE_SIZE, FL_FULLBORDER,  
		progname );
view->dnawin = fl_winget();

/* creation du menu Props et de ses sous-menus 
DOIT ETRE FAIT APRES fl_show_form !!!!!!!!!!!!!!!!!
sinon ne marche pas quand visual n'est pas celui par defaut
*/
handle_props_pup(bouton_props, -1); 
menuid = bouton_props->u_ldata; 
/* taille par defaut des lettres des sequences */
seq_char_size = (reducefonts ? 13 /* NORMAL */: 14 /* MEDIUM */ );
fl_set_object_lsize(view->DNA_obj, taille_from_rang(seq_char_size) );
fl_setpup_mode(menuid, seq_char_size, FL_PUP_CHECK);
fl_setpup_mode(menuid, view->format_for_save + 21, FL_PUP_CHECK);
if ( fl_get_visual_depth() > 1) { /* 1 en B&W, 8 en couleurs sun */
	if(curr_color_choice == NO_ALT_COLORS)
		fl_setpup_mode(menuid, 33, FL_PUP_GRAY);
	/* couleur du fond pour seqs */
  	fl_set_object_color(view->DNA_obj, FL_MCOL, FL_COL1); 
	view->region_color = FL_WHITE;
	view->inverted_colors = inverted;
	if(inverted) {
		fl_setpup_mode(menuid, 34, FL_PUP_CHECK);
		view->DNA_obj->lstyle = FL_FIXED_STYLE;
		}
	}
else	{ /* the Black and White case */
	numb_dnacolors = numb_stdprotcolors = numb_altprotcolors = 1;
	dnacolors[0] = FL_BLACK;
	fl_setpup_mode(menuid, 30, FL_PUP_GRAY);
	fl_setpup_mode(menuid, 41, FL_PUP_GRAY);
	/* couleur du fond pour seqs */
  	fl_set_object_color(view->DNA_obj, FL_WHITE, FL_WHITE); 
	view->region_color = FL_BLACK;
	view->numb_gc = 1;
	view->curr_colors = dnacolors;
	view->inverted_colors = FALSE;
	}
view->dnacolors = dnacolors;
view->numb_dnacolors = numb_dnacolors;
view->stdprotcolors = stdprotcolors;
view->numb_stdprotcolors = numb_stdprotcolors;
view->altprotcolors = altprotcolors;
view->numb_altprotcolors = numb_altprotcolors;
view->namecolor = FL_BLACK;
view->alt_colors = curr_color_choice;
view->clustal_options = NULL;
view->show_comment_lines = FALSE;
view->tot_comment_lines = 0;
init_dna_scroller(*the_scroller, seq, seqname, 0, comments, NULL,
	FALSE, NULL);
return my_form;
}


void viewer_callback(FL_OBJECT *ob, long which)
{
FL_OBJECT *dna_group;
SEA_VIEW *view;
dna_group = (FL_OBJECT *)ob->u_vdata;
view = (SEA_VIEW *)dna_group->u_vdata;
fl_redraw_object(view->horsli);
fl_redraw_object(view->vertsli);
}
